/*
 *  $Id: graph_sfuncs.c 22340 2019-07-25 10:23:59Z yeti-dn $
 *  Copyright (C) 2019 David Necas (Yeti).
 *  E-mail: yeti@gwyddion.net.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 */

#include "config.h"
#include <stdlib.h>
#include <string.h>
#include <libgwyddion/gwymacros.h>
#include <libprocess/gwyprocess.h>
#include <libgwydgets/gwygraphmodel.h>
#include <libgwydgets/gwycombobox.h>
#include <libgwydgets/gwynullstore.h>
#include <libgwydgets/gwycheckboxes.h>
#include <libgwydgets/gwystock.h>
#include <libgwydgets/gwydgetutils.h>
#include <libgwymodule/gwymodule-graph.h>
#include <app/gwymoduleutils.h>
#include <app/gwyapp.h>
#include "../process/preview.h"

typedef enum {
    GWY_SF_DH                     = 0,
    GWY_SF_CDH                    = 1,
    GWY_SF_DA                     = 2,
    GWY_SF_CDA                    = 3,
    GWY_SF_ACF                    = 4,
    GWY_SF_HHCF                   = 5,
    GWY_SF_PSDF                   = 6,
    GWY_SF_NFUNCTIONS
} GwySFOutputType;

enum {
    MIN_RESOLUTION = 4,
    MAX_RESOLUTION = 16384
};

typedef struct {
    gint curve;
    gboolean all;
    GwySFOutputType output_type;
    gint resolution;
    gdouble oversample;
    gboolean fixres;
    GwyAppDataId target_graph;
} SFuncsArgs;

typedef struct {
    SFuncsArgs *args;
    GwyGraphModel *parent_gmodel;
    GtkWidget *dialogue;
    GtkWidget *graph;
    GtkWidget *curve;
    GtkWidget *all;
    GtkWidget *fixres;
    GtkObject *resolution;
    GtkObject *oversample;
    GtkWidget *output_type;
    GtkWidget *target_graph;
} SFuncsControls;

static gboolean module_register              (void);
static void     graph_sfuncs                 (GwyGraph *graph);
static void     graph_sfuncs_dialogue        (GwyContainer *data,
                                              GwyGraphModel *parent_gmodel,
                                              SFuncsArgs *args);
static void     curve_changed                (GtkComboBox *combo,
                                              SFuncsControls *controls);
static void     all_changed                  (GtkToggleButton *toggle,
                                              SFuncsControls *controls);
static void     fixres_changed               (GtkToggleButton *toggle,
                                              SFuncsControls *controls);
static void     resolution_changed           (SFuncsControls *controls,
                                              GtkAdjustment *adj);
static void     oversample_changed           (SFuncsControls *controls,
                                              GtkAdjustment *adj);
static void     output_type_changed          (GtkComboBox *combo,
                                              SFuncsControls *controls);
static void     target_graph_changed         (SFuncsControls *controls);
static void     update_sensitivity           (SFuncsControls *controls);
static void     preview                      (SFuncsControls *controls);
static void     set_graph_model_units        (GwyGraphModel *gmodel,
                                              GwyGraphModel *sourcegmodel,
                                              GwySFOutputType type);
static gboolean sfunction_has_native_sampling(GwySFOutputType type);
static void     load_args                    (GwyContainer *settings,
                                              SFuncsArgs *args);
static void     save_args                    (GwyContainer *settings,
                                              SFuncsArgs *args);

static const GwyEnum sf_types[] =  {
    { N_("Height distribution"),         GWY_SF_DH,   },
    { N_("Cum. height distribution"),    GWY_SF_CDH,  },
    { N_("Distribution of angles"),      GWY_SF_DA,   },
    { N_("Cum. distribution of angles"), GWY_SF_CDA,  },
    { N_("ACF"),                         GWY_SF_ACF,  },
    { N_("HHCF"),                        GWY_SF_HHCF, },
    { N_("PSDF"),                        GWY_SF_PSDF, },
};

static const SFuncsArgs sfuncs_defaults = {
    0, FALSE,
    GWY_SF_DH,
    120, FALSE,
    4.0,
    GWY_APP_DATA_ID_NONE,
};

static GwyAppDataId target_id = GWY_APP_DATA_ID_NONE;

static GwyModuleInfo module_info = {
    GWY_MODULE_ABI_VERSION,
    &module_register,
    N_("Calculates one-dimensional statistical functions (height "
       "distribution, correlations, PSDF)."),
    "Yeti <yeti@gwyddion.net>",
    "1.0",
    "David Nečas (Yeti)",
    "2019",
};

GWY_MODULE_QUERY2(module_info, graph_sfuncs)

static gboolean
module_register(void)
{
    gwy_graph_func_register("graph_sfuncs",
                            (GwyGraphFunc)&graph_sfuncs,
                            N_("/Statistical _Functions..."),
                            NULL,
                            GWY_MENU_FLAG_GRAPH,
                            N_("Calculate 1D statistical functions"));

    return TRUE;
}

static void
graph_sfuncs(GwyGraph *graph)
{
    GwyContainer *data;
    SFuncsArgs args;

    gwy_app_data_browser_get_current(GWY_APP_CONTAINER, &data, 0);
    load_args(gwy_app_settings_get(), &args);
    graph_sfuncs_dialogue(data, gwy_graph_get_model(graph), &args);
    save_args(gwy_app_settings_get(), &args);
}

static void
graph_sfuncs_dialogue(GwyContainer *data, GwyGraphModel *parent_gmodel,
                      SFuncsArgs *args)
{
    GtkWidget *dialogue, *hbox, *table;
    GwyGraphModel *gmodel;
    SFuncsControls controls;
    gint row, response;

    gwy_clear(&controls, 1);
    controls.args = args;
    controls.parent_gmodel = parent_gmodel;
    gmodel = gwy_graph_model_new();
    set_graph_model_units(gmodel, parent_gmodel, args->output_type);

    dialogue = gtk_dialog_new_with_buttons(_("Statistical Functions"),
                                           NULL, 0, NULL);
    controls.dialogue = dialogue;
    gtk_dialog_add_button(GTK_DIALOG(dialogue), GTK_STOCK_CANCEL,
                          GTK_RESPONSE_CANCEL);
    gtk_dialog_add_button(GTK_DIALOG(dialogue), GTK_STOCK_OK, GTK_RESPONSE_OK);
    gwy_help_add_to_graph_dialog(GTK_DIALOG(dialogue), GWY_HELP_DEFAULT);
    gtk_dialog_set_default_response(GTK_DIALOG(dialogue), GTK_RESPONSE_OK);

    hbox = gtk_hbox_new(FALSE, 2);
    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialogue)->vbox), hbox,
                       TRUE, TRUE, 0);

    /* Parameters */
    table = gtk_table_new(11, 3, FALSE);
    gtk_table_set_row_spacings(GTK_TABLE(table), 2);
    gtk_table_set_col_spacings(GTK_TABLE(table), 6);
    gtk_container_set_border_width(GTK_CONTAINER(table), 4);
    gtk_box_pack_start(GTK_BOX(hbox), table, FALSE, FALSE, 0);
    row = 0;

    controls.curve = gwy_combo_box_graph_curve_new(G_CALLBACK(curve_changed),
                                                   &controls,
                                                   controls.parent_gmodel,
                                                   args->curve);
    gwy_table_attach_adjbar(table, row++, _("_Graph curve:"), NULL,
                            GTK_OBJECT(controls.curve),
                            GWY_HSCALE_WIDGET_NO_EXPAND);

    controls.all = gtk_check_button_new_with_mnemonic(_("_All curves"));
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(controls.all), args->all);
    gtk_table_attach(GTK_TABLE(table), controls.all,
                     0, 2, row, row+1, GTK_FILL, 0, 0, 0);
    g_signal_connect(controls.all, "toggled",
                     G_CALLBACK(all_changed), &controls);
    row++;

    controls.output_type
        = gwy_enum_combo_box_new(sf_types, G_N_ELEMENTS(sf_types),
                                 G_CALLBACK(output_type_changed), &controls,
                                 args->output_type, TRUE);
    gwy_table_attach_adjbar(table, row++, _("_Quantity:"), NULL,
                            GTK_OBJECT(controls.output_type),
                            GWY_HSCALE_WIDGET_NO_EXPAND);

    controls.resolution = gtk_adjustment_new(args->resolution,
                                            MIN_RESOLUTION, MAX_RESOLUTION,
                                            1, 10, 0);
    gwy_table_attach_adjbar(GTK_WIDGET(table), row,
                            _("_Fixed resolution:"), NULL,
                            controls.resolution,
                            GWY_HSCALE_CHECK | GWY_HSCALE_SQRT);
    g_signal_connect_swapped(controls.resolution, "value-changed",
                             G_CALLBACK(resolution_changed), &controls);
    controls.fixres = gwy_table_hscale_get_check(controls.resolution);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(controls.fixres),
                                 args->fixres);
    g_signal_connect(controls.fixres, "toggled",
                     G_CALLBACK(fixres_changed), &controls);
    row++;

    controls.oversample = gtk_adjustment_new(args->oversample, 1.0, 16.0,
                                             0.01, 1.0, 0);
    gwy_table_attach_adjbar(GTK_WIDGET(table), row++,
                            _("O_versampling:"), "×",
                            controls.oversample, GWY_HSCALE_SQRT);
    g_signal_connect_swapped(controls.oversample, "value-changed",
                             G_CALLBACK(oversample_changed), &controls);

    controls.target_graph = create_target_graph(&args->target_graph,
                                                GTK_WIDGET(table), row++,
                                                gmodel);
    g_signal_connect_swapped(controls.target_graph, "changed",
                             G_CALLBACK(target_graph_changed), &controls);
    row++;

    /* Graph */
    controls.graph = gwy_graph_new(gmodel);
    g_object_set(gmodel, "label-visible", FALSE, NULL);
    g_object_unref(gmodel);
    gtk_widget_set_size_request(controls.graph, 480, 300);

    gwy_graph_enable_user_input(GWY_GRAPH(controls.graph), FALSE);
    gtk_box_pack_start(GTK_BOX(hbox), controls.graph, TRUE, TRUE, 0);

    curve_changed(GTK_COMBO_BOX(controls.curve), &controls);
    update_sensitivity(&controls);

    gtk_widget_show_all(dialogue);
    response = gtk_dialog_run(GTK_DIALOG(dialogue));
    if (response == GTK_RESPONSE_OK) {
        g_object_set(gmodel, "label-visible", TRUE, NULL);
        gwy_app_add_graph_or_curves(gmodel, data, &args->target_graph, 1);
    }
    gtk_widget_destroy(dialogue);
}

static void
curve_changed(GtkComboBox *combo, SFuncsControls *controls)
{
    SFuncsArgs *args = controls->args;

    args->curve = gwy_enum_combo_box_get_active(combo);
    preview(controls);
}

static void
all_changed(GtkToggleButton *toggle, SFuncsControls *controls)
{
    SFuncsArgs *args = controls->args;
    args->all = gtk_toggle_button_get_active(toggle);
    update_sensitivity(controls);
    preview(controls);
}

static void
fixres_changed(GtkToggleButton *toggle, SFuncsControls *controls)
{
    SFuncsArgs *args = controls->args;
    args->fixres = gtk_toggle_button_get_active(toggle);
    preview(controls);
}

static void
resolution_changed(SFuncsControls *controls, GtkAdjustment *adj)
{
    SFuncsArgs *args = controls->args;
    args->resolution = gwy_adjustment_get_int(adj);
    preview(controls);
}

static void
oversample_changed(SFuncsControls *controls, GtkAdjustment *adj)
{
    SFuncsArgs *args = controls->args;
    args->oversample = gtk_adjustment_get_value(adj);
    preview(controls);
}

static void
output_type_changed(GtkComboBox *combo, SFuncsControls *controls)
{
    SFuncsArgs *args = controls->args;
    args->output_type = gwy_enum_combo_box_get_active(combo);
    update_sensitivity(controls);
    preview(controls);
}

static void
target_graph_changed(SFuncsControls *controls)
{
    GwyDataChooser *chooser = GWY_DATA_CHOOSER(controls->target_graph);
    gwy_data_chooser_get_active_id(chooser, &controls->args->target_graph);
}

static void
update_sensitivity(SFuncsControls *controls)
{
    SFuncsArgs *args = controls->args;
    gboolean sens;

    sens = !args->all;
    gwy_table_hscale_set_sensitive(GTK_OBJECT(controls->curve), sens);

    sens = !sfunction_has_native_sampling(args->output_type);
    gwy_table_hscale_set_sensitive(controls->resolution, sens);
    /* TOOD */
}

static void
oversample_curve(const gdouble *xdata, const gdouble *ydata, guint ndata,
                 GwyDataLine *oversampled)
{
    guint i, j, nover;
    gdouble xfrom, xto;
    gdouble *d;

    if (ndata == 1)
        gwy_data_line_fill(oversampled, ydata[0]);

    xfrom = xdata[0];
    xto = xdata[ndata-1];
    nover = gwy_data_line_get_res(oversampled);
    d = gwy_data_line_get_data(oversampled);
    for (i = j = 0; i < nover; i++) {
        gdouble t, x = i/(nover - 1.0)*(xto - xfrom) + xfrom;

        while (j < ndata && xdata[j] < x)
            j++;

        if (j == 0)
            d[i] = ydata[0];
        else if (j == ndata)
            d[i] = ydata[ndata-1];
        else if (xdata[j-1] == xdata[j])
            d[i] = 0.5*(ydata[j-1] + ydata[j]);
        else {
            t = (x - xdata[j-1])/(xdata[j] - xdata[j-1]);
            d[i] = t*ydata[j] + (1.0 - t)*ydata[j-1];
        }
    }
}

static void
calculate_stats(GwyGraphCurveModel *gcmodel,
                GwySIUnit *xunit, GwySIUnit *yunit,
                const SFuncsArgs *args,
                GwyDataLine *dline)
{
    GwySFOutputType output_type = args->output_type;
    GwyDataLine *oversampled;
    const gdouble *xdata, *ydata;
    guint ndata, i, j, nover;
    gdouble xfrom, xto;
    gdouble *d, *diffdata = NULL;

    ndata = gwy_graph_curve_model_get_ndata(gcmodel);
    xdata = gwy_graph_curve_model_get_xdata(gcmodel);
    ydata = gwy_graph_curve_model_get_ydata(gcmodel);
    nover = GWY_ROUND(ndata*args->oversample);

    xfrom = xdata[0];
    xto = xdata[ndata-1];
    if (xto == xfrom) {
        if (xto) {
            xto += 1e-9*fabs(xto);
            xfrom -= 1e-9*fabs(xfrom);
        }
        else {
            xfrom = -1e-9;
            xto = 1e-9;
        }
    }

    oversampled = gwy_data_line_new(nover, xto - xfrom, FALSE);
    gwy_si_unit_assign(gwy_data_line_get_si_unit_x(oversampled), xunit);
    gwy_si_unit_assign(gwy_data_line_get_si_unit_y(oversampled), yunit);

    /* Oversample derivatives, not values for DA and CDA. */
    if (output_type == GWY_SF_DA || output_type == GWY_SF_CDA) {
        if (ndata == 1)
            diffdata = g_new0(gdouble, 1);
        else {
            diffdata = g_new0(gdouble, ndata-1);
            for (i = j = 0; i < ndata-1; i++) {
                /* Cannot handle infinite derivatives. */
                if (xdata[i] == xdata[j+1])
                    continue;

                diffdata[j] = (ydata[j+1] - ydata[j])/(xdata[j+1] - xdata[j]);
                j++;
            }
            ndata = j;
        }
        ydata = diffdata;

        gwy_si_unit_divide(yunit, xunit,
                           gwy_data_line_get_si_unit_y(oversampled));
    }

    oversample_curve(xdata, ydata, ndata, oversampled);

    if (output_type == GWY_SF_DH || output_type == GWY_SF_DA) {
        gwy_data_line_distribution(oversampled, dline, 0.0, 0.0, TRUE,
                                   args->fixres ? args->resolution : -1);
    }
    else if (output_type == GWY_SF_CDH || output_type == GWY_SF_CDA) {
        gwy_data_line_distribution(oversampled, dline, 0.0, 0.0, TRUE,
                                   args->fixres ? args->resolution : -1);
        d = gwy_data_line_get_data(dline);
        ndata = gwy_data_line_get_res(dline);
        for (i = 1; i < ndata; i++)
            d[i] += d[i-1];
        gwy_data_line_multiply(dline, 1.0/d[ndata-1]);
        gwy_si_unit_set_from_string(gwy_data_line_get_si_unit_y(dline), NULL);
    }
    else if (output_type == GWY_SF_ACF) {
        gwy_data_line_add(oversampled, -gwy_data_line_get_avg(oversampled));
        gwy_data_line_acf(oversampled, dline);
    }
    else if (output_type == GWY_SF_HHCF) {
        gwy_data_line_add(oversampled, -gwy_data_line_get_avg(oversampled));
        gwy_data_line_hhcf(oversampled, dline);
    }
    else if (output_type == GWY_SF_PSDF) {
        gwy_data_line_add(oversampled, -gwy_data_line_get_avg(oversampled));
        /* Interpolation is ignored. */
        gwy_data_line_psdf(oversampled, dline, GWY_WINDOWING_HANN,
                           GWY_INTERPOLATION_LINEAR);
    }
    else {
        g_assert_not_reached();
    }

    g_object_unref(oversampled);
    g_free(diffdata);
}

static void
preview(SFuncsControls *controls)
{
    SFuncsArgs *args = controls->args;
    GwyGraphCurveModel *gcmodel;
    GwyGraphModel *gmodel;
    GwySIUnit *xunit, *yunit;
    const gchar *xlabel, *ylabel, *title;
    GwyDataLine *dline;
    gint i, ifrom, ito;
    gchar *s;

    dline = gwy_data_line_new(1, 1.0, FALSE);
    g_object_get(controls->parent_gmodel,
                 "si-unit-x", &xunit,
                 "si-unit-y", &yunit,
                 NULL);

    gmodel = gwy_graph_get_model(GWY_GRAPH(controls->graph));
    gwy_graph_model_remove_all_curves(gmodel);
    title = gettext(gwy_enum_to_string(args->output_type,
                                       sf_types, G_N_ELEMENTS(sf_types)));

    if (args->all) {
        ifrom = 0;
        ito = gwy_graph_model_get_n_curves(controls->parent_gmodel);
    }
    else {
        ifrom = args->curve;
        ito = ifrom + 1;
    }

    for (i = ifrom; i < ito; i++) {
        gcmodel = gwy_graph_model_get_curve(controls->parent_gmodel, i);
        calculate_stats(gcmodel, xunit, yunit, args, dline);
        gcmodel = gwy_graph_curve_model_new();
        gwy_graph_curve_model_set_data_from_dataline(gcmodel, dline, 0, 0);
        g_object_set(gcmodel, "mode", GWY_GRAPH_CURVE_LINE, NULL);
        if (args->all) {
            s = g_strdup_printf("%s %d", title, i+1);
            g_object_set(gcmodel,
                         "color", gwy_graph_get_preset_color(i),
                         "description", s,
                         NULL);
            g_free(s);
        }
        else
            g_object_set(gcmodel, "description", title, NULL);
        gwy_graph_model_add_curve(gmodel, gcmodel);
        g_object_unref(gcmodel);
    }

    gwy_graph_model_set_units_from_data_line(gmodel, dline);
    g_object_unref(dline);
    g_object_unref(xunit);
    g_object_unref(yunit);

    switch (args->output_type) {
        case GWY_SF_DH:
        xlabel = "z";
        ylabel = "ρ";
        break;

        case GWY_SF_CDH:
        xlabel = "z";
        ylabel = "D";
        break;

        case GWY_SF_DA:
        xlabel = "tan β";
        ylabel = "ρ";
        break;

        case GWY_SF_CDA:
        xlabel = "tan β";
        ylabel = "D";
        break;

        case GWY_SF_ACF:
        xlabel = "τ";
        ylabel = "G";
        break;

        case GWY_SF_HHCF:
        xlabel = "τ";
        ylabel = "H";
        break;

        case GWY_SF_PSDF:
        xlabel = "k";
        ylabel = "W<sub>1</sub>";
        break;

        default:
        g_assert_not_reached();
        break;
    }

    g_object_set(gmodel,
                 "title", title,
                 "axis-label-bottom", xlabel,
                 "axis-label-left", ylabel,
                 NULL);
}

/* This is a bit silly but it solves a chicken-and-egg problem.  We do not
 * want to run preview() before GUI was set up.  But we need correct units
 * before setting the target graph widget, otherwise it will get filtered to
 * some nonsense and render remembering GwyAppDataId pointless. */
static void
set_graph_model_units(GwyGraphModel *gmodel,
                      GwyGraphModel *sourcegmodel,
                      GwySFOutputType type)
{
    GwySIUnit *unit, *xunit, *yunit;

    g_object_get(sourcegmodel, "si-unit-x", &xunit, "si-unit-y", &yunit, NULL);
    unit = gwy_si_unit_new(NULL);

    if (type == GWY_SF_DH || type == GWY_SF_CDH)
        gwy_si_unit_assign(unit, yunit);
    else if (type == GWY_SF_DA || type == GWY_SF_CDA)
        gwy_si_unit_divide(yunit, xunit, unit);
    else if (type == GWY_SF_ACF || type == GWY_SF_HHCF)
        gwy_si_unit_assign(unit, xunit);
    else if (type == GWY_SF_PSDF)
        gwy_si_unit_power(xunit, -1, unit);
    else {
        g_assert_not_reached();
    }
    g_object_set(gmodel, "si-unit-x", unit, NULL);

    if (type == GWY_SF_DH)
        gwy_si_unit_power(yunit, -1, unit);
    else if (type == GWY_SF_DA)
        gwy_si_unit_divide(xunit, yunit, unit);
    else if (type == GWY_SF_CDH || type == GWY_SF_CDA)
        gwy_si_unit_set_from_string(unit, NULL);
    else if (type == GWY_SF_ACF || type == GWY_SF_HHCF)
        gwy_si_unit_power(yunit, 2, unit);
    else if (type == GWY_SF_PSDF)
        gwy_si_unit_power_multiply(yunit, 2, xunit, 1, unit);
    else {
        g_assert_not_reached();
    }
    g_object_set(gmodel, "si-unit-y", unit, NULL);

    g_object_unref(xunit);
    g_object_unref(yunit);
    g_object_unref(unit);
}

static gboolean
sfunction_has_native_sampling(GwySFOutputType type)
{
    return (type == GWY_SF_ACF || type == GWY_SF_HHCF || type == GWY_SF_PSDF);
}

static const gchar all_key[]           = "/module/graph_sfuncs/all";
static const gchar fixres_key[]        = "/module/graph_sfuncs/fixres";
static const gchar output_type_key[]   = "/module/graph_sfuncs/output_type";
static const gchar oversample_key[]    = "/module/graph_sfuncs/oversample";
static const gchar resolution_key[]    = "/module/graph_sfuncs/resolution";

static void
sanitize_args(SFuncsArgs *args)
{
    args->output_type = CLAMP(args->output_type,
                              GWY_SF_DH, GWY_SF_NFUNCTIONS-1);
    args->resolution = CLAMP(args->resolution, MIN_RESOLUTION, MAX_RESOLUTION);
    args->oversample = CLAMP(args->oversample, 1.0, 16.0);
    args->all = !!args->all;
    args->fixres = !!args->fixres;
    gwy_app_data_id_verify_graph(&args->target_graph);
}

static void
load_args(GwyContainer *settings, SFuncsArgs *args)
{
    *args = sfuncs_defaults;

    gwy_container_gis_int32_by_name(settings, resolution_key,
                                    &args->resolution);
    gwy_container_gis_double_by_name(settings, oversample_key,
                                     &args->oversample);
    gwy_container_gis_enum_by_name(settings, output_type_key,
                                   &args->output_type);
    gwy_container_gis_boolean_by_name(settings, fixres_key, &args->fixres);
    gwy_container_gis_boolean_by_name(settings, all_key, &args->all);
    args->target_graph = target_id;
    sanitize_args(args);
}

static void
save_args(GwyContainer *settings, SFuncsArgs *args)
{
    target_id = args->target_graph;
    gwy_container_set_int32_by_name(settings, resolution_key,
                                    args->resolution);
    gwy_container_set_double_by_name(settings, oversample_key,
                                     args->oversample);
    gwy_container_set_enum_by_name(settings, output_type_key,
                                   args->output_type);
    gwy_container_set_boolean_by_name(settings, fixres_key, args->fixres);
    gwy_container_set_boolean_by_name(settings, all_key, args->all);
}

/* vim: set cin et ts=4 sw=4 cino=>1s,e0,n0,f0,{0,}0,^0,\:1s,=0,g1s,h0,t0,+1s,c3,(0,u0 : */
