/**
* @license Apache-2.0
*
* Copyright (c) 2018 The Stdlib Authors.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

'use strict';

// MODULES //

var tape = require( 'tape' );
var semver = require( 'semver' );
var IS_WINDOWS = require( '@stdlib/assert/is-windows' );
var IS_NODE = require( '@stdlib/assert/is-node' );
var VERSION = require( '@stdlib/process/node-version' );
var PINF = require( '@stdlib/constants/float64/pinf' );
var NINF = require( '@stdlib/constants/float64/ninf' );
var MAX_UINT32 = require( '@stdlib/constants/uint32/max' );
var isnan = require( '@stdlib/math/base/assert/is-nan' );
var isNumber = require( '@stdlib/assert/is-number' ).isPrimitive;
var lpad = require( '@stdlib/string/left-pad' );
var pow = require( '@stdlib/math/base/special/pow' );
var toBinaryString = require( './../../../../float64/base/to-binary-string' );
var randu = require( '@stdlib/random/base/randu' );
var round = require( '@stdlib/math/base/special/round' );
var setLowWord = require( './../lib' );


// VARIABLES //

var opts;


// TESTS //

tape( 'main export is a function', function test( t ) {
	t.ok( true, __filename );
	t.equal( typeof setLowWord, 'function', 'main export is a function' );
	t.end();
});

tape( 'the function returns a number', function test( t ) {
	var y = setLowWord( pow( 2, 53 ), 5 );
	t.equal( typeof y, 'number', 'returns a number' );
	t.end();
});

tape( 'the function sets the lower order word of a double-precision floating-point number', function test( t ) {
	var actual;
	var values;
	var high;
	var low;
	var x;
	var v;
	var y;
	var i;
	var j;

	values = [
		5.0,
		pow( 2.0, 53 ),
		1.0e308,
		-1.0e308,
		-3.14,
		1.0e-324,
		4.94e-324,
		1.234567890123456789,
		-4.94e-324,
		6.333333333333333333e-310,
		-0.0,
		0.0,
		100.0,
		1.0/10.0,
		0.625,
		1.0/3.0,
		5.0e-240,
		-5.0e-240,
		10.0,
		15.0,
		-10.0,
		-15.0,
		pow( 2.0, -42 ),
		-pow( 2.0, 100 ),
		1.0,
		-1.0,
		1.5,
		1111111111111.111111111,
		-1111111111111.111111111,
		pow( 2.0, 54 ),
		pow( 2.0, 53 ) + 1.0,
		pow( 2.0, 53 ) + 2.0,
		pow( 2.0, 55 ),
		pow( 2.0, 56 ) - 1.0,
		-pow( 2.0, 57 ) + 5.0,
		3.0*pow( 2.0, 53 ),
		8.0*pow( 2.0, 54 )
	];

	// For each value, replace the lower order bits with random integers and test that the returned values match expectations...
	for ( i = 0; i < values.length; i++ ) {
		v = values[ i ];
		x = toBinaryString( v );
		high = x.substring( 0, 32 );
		for ( j = 0; j < 10; j++ ) {
			// Generate a random lower order word:
			low = round( randu()*MAX_UINT32 );

			// Generate a new double:
			y = setLowWord( v, low );

			// Convert to a binary string:
			low = low.toString( 2 );
			low = lpad( low, 32, '0' );

			// Lower order bits:
			actual = toBinaryString( y ).substring( 32 );
			t.equal( actual, low, 'returned double has the same lower order word as ' + v );

			// higher order bits:
			actual = toBinaryString( y ).substring( 0, 32 );
			t.equal( actual, high, 'returned double has the same higher order word as ' + v );
		}
	}
	t.end();
});

opts = {
	'skip': IS_NODE && IS_WINDOWS && VERSION && semver.lt( VERSION, '0.11.0' )
};
tape( 'if provided `+infinity`, setting the lower order bits will return `NaN`', opts, function test( t ) {
	var low;
	var y;

	low = pow( 2.0, 14 );
	y = setLowWord( PINF, low );

	t.equal( isnan( y ), true, 'returns NaN' );
	t.end();
});

opts = {
	'skip': !(IS_NODE && IS_WINDOWS && VERSION && semver.lt( VERSION, '0.11.0' ))
};
tape( 'if provided `+infinity`, setting the lower order bits will return `NaN` (non-compliant IEEE 754 behavior possible on Windows for Node <v0.11.0)', opts, function test( t ) {
	var bool;
	var low;
	var y;

	low = pow( 2.0, 14 );
	y = setLowWord( PINF, low );

	t.equal( isNumber( y ), true, 'returns a number' );
	t.equal( y.toString(), 'NaN', 'serializes as NaN' );

	bool = isnan( y );
	if ( bool ) {
		t.equal( bool, true, 'treated as canonical NaN (IEEE 754 compliant)' );
	} else {
		t.equal( bool, false, 'not IEEE 754 compliant' );
	}
	t.end();
});

opts = {
	'skip': IS_NODE && IS_WINDOWS && VERSION && semver.lt( VERSION, '0.11.0' )
};
tape( 'if provided `-infinity`, setting the lower order bits will return `NaN`', opts, function test( t ) {
	var low;
	var y;

	low = pow( 2.0, 14 );
	y = setLowWord( NINF, low );

	t.equal( isnan( y ), true, 'returns NaN' );
	t.end();
});

opts = {
	'skip': !(IS_NODE && IS_WINDOWS && VERSION && semver.lt( VERSION, '0.11.0' ))
};
tape( 'if provided `-infinity`, setting the lower order bits will return `NaN` (non-compliant IEEE 754 behavior possible on Windows for Node <v0.11.0)', opts, function test( t ) {
	var bool;
	var low;
	var y;

	low = pow( 2.0, 14 );
	y = setLowWord( NINF, low );

	t.equal( isNumber( y ), true, 'returns a number' );
	t.equal( y.toString(), 'NaN', 'serializes as NaN' );

	bool = isnan( y );
	if ( bool ) {
		t.equal( bool, true, 'treated as canonical NaN (IEEE 754 compliant)' );
	} else {
		t.equal( bool, false, 'not IEEE 754 compliant' );
	}
	t.end();
});

tape( 'if provided `NaN`, setting the lower order bits will return `NaN`', function test( t ) {
	var y = setLowWord( NaN, pow( 2.0, 14 ) );
	t.equal( isnan( y ), true, 'returns NaN' );
	t.end();
});
