#/
# @license Apache-2.0
#
# Copyright (c) 2018 The Stdlib Authors.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#    http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#/

# VARIABLES #

# Define the command-line options to use when invoking the shellcheck executable:
SHELLCHECK_FLAGS ?=


# RULES #

#/
# Checks whether [shellcheck][1] is installed.
#
# [1]: https://github.com/koalaman/shellcheck
#
# @private
#
# @example
# make check-shellcheck
#/
check-shellcheck:
ifeq (, $(shell command -v $(SHELLCHECK) 2>/dev/null))
	$(QUIET) echo ''
	$(QUIET) echo 'shellcheck is not installed. Please install shellcheck and try again.'
	$(QUIET) echo 'For install instructions, see https://github.com/koalaman/shellcheck'
	$(QUIET) echo 'and the project development guide.'
	$(QUIET) echo ''
	$(QUIET) exit 1
else
	$(QUIET) echo 'shellcheck is installed.'
	$(QUIET) exit 0
endif

.PHONY: check-shellcheck

#/
# Lints shell script files using [shellcheck][1].
#
# ## Notes
#
# -   This rule is useful when wanting to glob for files, irrespective of context, for a particular directory in order to lint all contained shell script files.
#
# [1]: https://github.com/koalaman/shellcheck
#
# @private
# @param {string} [SHELL_FILTER] - file path pattern (e.g., `.*/_tools/.*`)
# @param {*} [FAST_FAIL] - flag indicating whether to stop linting upon encountering a lint error
#
# @example
# make shellcheck
#
# @example
# make shellcheck SHELL_FILTER=".*/_tools/.*"
#/
shellcheck:
ifeq ($(FAIL_FAST), true)
	$(QUIET) $(FIND_SHELL_CMD) | grep '^[\/]\|^[a-zA-Z]:[/\]' | while read -r file; do \
		echo ''; \
		echo "Linting file: $$file"; \
		$(SHELLCHECK) $(SHELLCHECK_FLAGS) $$file || exit 1; \
	done
else
	$(QUIET) $(FIND_SHELL_CMD) | grep '^[\/]\|^[a-zA-Z]:[/\]' | while read -r file; do \
		echo ''; \
		echo "Linting file: $$file"; \
		$(SHELLCHECK) $(SHELLCHECK_FLAGS) $$file || echo 'Linting failed.'; \
	done
endif

.PHONY: shellcheck

#/
# Lints a specified list of shell script files using [shellcheck][1].
#
# ## Notes
#
# -   This rule is useful when wanting to lint a list of shell script files generated by some other command (e.g., a list of changed shell script files obtained via `git diff`).
#
# [1]: https://github.com/koalaman/shellcheck
#
# @private
# @param {string} FILES - list of shell script file paths
# @param {*} [FAST_FAIL] - flag indicating whether to stop linting upon encountering a lint error
#
# @example
# make shellcheck-files FILES='/foo/file.sh /bar/file.sh'
#/
shellcheck-files:
ifeq ($(FAIL_FAST), true)
	$(QUIET) for file in $(FILES); do \
		echo ''; \
		echo "Linting file: $$file"; \
		$(SHELLCHECK) $(SHELLCHECK_FLAGS) $$file || exit 1; \
	done
else
	$(QUIET) for file in $(FILES); do \
		echo ''; \
		echo "Linting file: $$file"; \
		$(SHELLCHECK) $(SHELLCHECK_FLAGS) $$file || echo 'Linting failed.'; \
	done
endif

.PHONY: shellcheck-files
