/* $Id$
 *
 * Abstract base class for all nodes of the abstract syntax tree.
 *
 * Copyright (C) 2007-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */


#ifndef __AST_NODE_HPP_INCLUDED
#define __AST_NODE_HPP_INCLUDED

#include <ostream>
#include "frontend/visitor/Visitor.hpp"
#include "frontend/ast/Location.hpp"
#include "util/MiscUtil.hpp"
#include "util/GarbageCollect.hpp"

namespace ast {

//! This is the abstract base class for all AST Nodes.
/** All AST nodes need to derive from this class
 */
class AstNode : public util::GarbageCollect {
public:
	//! c'tor
	/** @param loc source code location of the AST node. */
	AstNode(Location loc) : location(loc), number(-1) {}

	//! Accept a Visitor.
 	/** All leaf AST nodes need to implement this method.
         *
         *  @param visitor the Visitor that can visit this node.
         */
	virtual void accept(Visitor& visitor) = 0;

	/** Put a textual representation of the AstNode on the stream.
	 *  @param stream stream to put the textual representation to.
	 */
	virtual void put(std::ostream &stream) const = 0;


	/** source location of the AST node. */
	Location location;

	/** unique node number for Dot graphs */
	int number;

protected:
	/** d'tor, not used in this interface. */
	virtual ~AstNode() {}
};

/** operator to write node to a stream.
 *
 *  @param stream stream to write the node to.
 *  @param node node that should get written to stream.
 *  @return modified stream.
 */
std::ostream&
operator<<(std::ostream &stream, const AstNode &node);


/** operator to write node to a stream.
 *
 *  @param stream stream to write the node to.
 *  @param node node that should get written to stream. If NULL, (null) gets
 *         written.
 *  @return modified stream.
 */
std::ostream&
operator<<(std::ostream &stream, const AstNode *node);


}; /* namespace ast */

#endif /* __AST_NODE_HPP_INCLUDED */
