/**
 * Neuroimaging plugin for Orthanc
 * Copyright (C) 2021-2024 Sebastien Jodogne, ICTEAM UCLouvain, Belgium
 *
 * This program is free software: you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 **/

// AUTOGENERATED FILE, DON'T MODIFY BY HAND

#include <gtest/gtest.h>

#include "../Framework/DicomInstancesCollection.h"

#include <DicomFormat/DicomMap.h>

static void CheckConsistency(const nifti_image& nifti)
{
  ASSERT_EQ(nifti.nx, nifti.dim[1]);
  ASSERT_EQ(nifti.ny, nifti.dim[2]);
  ASSERT_EQ(nifti.nz, nifti.dim[3]);
  ASSERT_EQ(nifti.nt, nifti.dim[4]);
  ASSERT_EQ(nifti.nu, nifti.dim[5]);
  ASSERT_EQ(nifti.nv, nifti.dim[6]);
  ASSERT_EQ(nifti.nw, nifti.dim[7]);
  ASSERT_EQ(nifti.dx, nifti.pixdim[1]);
  ASSERT_EQ(nifti.dy, nifti.pixdim[2]);
  ASSERT_EQ(nifti.dz, nifti.pixdim[3]);
  ASSERT_EQ(nifti.dt, nifti.pixdim[4]);
  ASSERT_EQ(nifti.du, nifti.pixdim[5]);
  ASSERT_EQ(nifti.dv, nifti.pixdim[6]);
  ASSERT_EQ(nifti.dw, nifti.pixdim[7]);
  ASSERT_FLOAT_EQ(nifti.qoffset_x, nifti.sto_xyz.m[0][3]);
  ASSERT_FLOAT_EQ(nifti.qoffset_y, nifti.sto_xyz.m[1][3]);
  ASSERT_FLOAT_EQ(nifti.qoffset_z, nifti.sto_xyz.m[2][3]);
  ASSERT_FLOAT_EQ(0.0, nifti.sto_xyz.m[3][0]);
  ASSERT_FLOAT_EQ(0.0, nifti.sto_xyz.m[3][1]);
  ASSERT_FLOAT_EQ(0.0, nifti.sto_xyz.m[3][2]);
  ASSERT_FLOAT_EQ(1.0, nifti.sto_xyz.m[3][3]);
  ASSERT_EQ(nifti.xyz_units, NIFTI_UNITS_MM);
  ASSERT_EQ(nifti.time_units, NIFTI_UNITS_SEC);
}

static void LoadInstance_0(Neuro::DicomInstancesCollection& target)
{
  // ('brainix', 'SOUS - 702/IM-0001-0001.dcm')
  Orthanc::DicomMap tags;
  tags.SetValue(0x0008, 0x0060, "MR", false);
  tags.SetValue(0x0008, 0x0070, "Philips Medical Systems", false);
  tags.SetValue(0x0020, 0x0013, "1", false);
  tags.SetValue(0x0018, 0x0080, "449.998992919921", false);
  tags.SetValue(0x0018, 0x0081, "10.0", false);
  tags.SetValue(0x0018, 0x1312, "ROW", false);
  tags.SetValue(0x0008, 0x0016, "1.2.840.10008.5.1.4.1.1.4", false);
  tags.SetValue(0x0028, 0x0002, "1", false);
  tags.SetValue(0x0028, 0x0004, "MONOCHROME2", false);
  tags.SetValue(0x0028, 0x0010, "512", false);
  tags.SetValue(0x0028, 0x0011, "512", false);
  tags.SetValue(0x0028, 0x0100, "16", false);
  tags.SetValue(0x0028, 0x0101, "12", false);
  tags.SetValue(0x0028, 0x0102, "11", false);
  tags.SetValue(0x0028, 0x0103, "0", false);
  tags.SetValue(0x0008, 0x0032, "143958.890000", false);
  tags.SetValue(0x0018, 0x0050, "5.0", false);
  tags.SetValue(0x0018, 0x0088, "6.0", false);
  tags.SetValue(0x0020, 0x0032, "-123.48501954041\\-123.85898304080\\83.2371420264244", false);
  tags.SetValue(0x0020, 0x0037, "0.99971222877502\\7.8810308973E-12\\0.02398800104856\\-0.0017278126906\\0.99740260839462\\0.07200747728347", false);
  tags.SetValue(0x0028, 0x0030, "0.46875\\0.46875", false);
  std::unique_ptr<Neuro::InputDicomInstance> dicom(new Neuro::InputDicomInstance(tags));
  target.AddInstance(dicom.release(), "nope");
}

TEST(Autogenerated, Test_0)
{
  Neuro::DicomInstancesCollection instances;
  LoadInstance_0(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 512);
  ASSERT_NEAR(nifti.pixdim[1], 0.468750, 0.00001);
  ASSERT_EQ(nifti.dim[2], 512);
  ASSERT_NEAR(nifti.pixdim[2], 0.468750, 0.00001);
  ASSERT_EQ(nifti.dim[3], 1);
  ASSERT_NEAR(nifti.pixdim[3], 6.000000, 0.00001);
  ASSERT_EQ(nifti.ndim, 3);
  ASSERT_EQ(nifti.dim[0], 3);
  ASSERT_EQ(nifti.nvox, 262144u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(54));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(54));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(54));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -0.468615, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], -0.000810, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], 0.143554, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 123.898880, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 0.467532, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], 0.432169, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], -115.050117, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.011244, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], -0.033754, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 5.982694, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], 100.485184, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.000432, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, -0.999279, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, 0.036035, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=10;Time=143958.890");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadInstance_1(Neuro::DicomInstancesCollection& target)
{
  // ('vix', 'IM-0001-0001.dcm')
  Orthanc::DicomMap tags;
  tags.SetValue(0x0008, 0x0060, "CT", false);
  tags.SetValue(0x0008, 0x0070, "SIEMENS", false);
  tags.SetValue(0x0020, 0x0013, "1", false);
  tags.SetValue(0x0028, 0x1052, "-1024", false);
  tags.SetValue(0x0028, 0x1053, "1", false);
  tags.SetValue(0x0008, 0x0016, "1.2.840.10008.5.1.4.1.1.2", false);
  tags.SetValue(0x0028, 0x0002, "1", false);
  tags.SetValue(0x0028, 0x0004, "MONOCHROME2", false);
  tags.SetValue(0x0028, 0x0010, "512", false);
  tags.SetValue(0x0028, 0x0011, "512", false);
  tags.SetValue(0x0028, 0x0100, "16", false);
  tags.SetValue(0x0028, 0x0101, "12", false);
  tags.SetValue(0x0028, 0x0102, "11", false);
  tags.SetValue(0x0028, 0x0103, "1", false);
  tags.SetValue(0x0008, 0x0032, "101812.967223", false);
  tags.SetValue(0x0018, 0x0050, "1", false);
  tags.SetValue(0x0020, 0x0032, "-126.798828125\\-210.798828125\\-41.5", false);
  tags.SetValue(0x0020, 0x0037, "1\\0\\0\\0\\1\\0", false);
  tags.SetValue(0x0028, 0x0030, "0.40234375\\0.40234375", false);
  std::unique_ptr<Neuro::InputDicomInstance> dicom(new Neuro::InputDicomInstance(tags));
  target.AddInstance(dicom.release(), "nope");
}

TEST(Autogenerated, Test_1)
{
  Neuro::DicomInstancesCollection instances;
  LoadInstance_1(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 512);
  ASSERT_NEAR(nifti.pixdim[1], 0.402344, 0.00001);
  ASSERT_EQ(nifti.dim[2], 512);
  ASSERT_NEAR(nifti.pixdim[2], 0.402344, 0.00001);
  ASSERT_EQ(nifti.dim[3], 1);
  ASSERT_NEAR(nifti.pixdim[3], 1.000000, 0.00001);
  ASSERT_EQ(nifti.ndim, 3);
  ASSERT_EQ(nifti.dim[0], 3);
  ASSERT_EQ(nifti.nvox, 262144u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 4);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(0));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(0));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(0));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, -1024.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -0.402344, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 126.798828, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 0.402344, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], 5.201172, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 1.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -41.500000, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, 1.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, 0.000000, 0.0001);
  ASSERT_STREQ(nifti.descrip, "Time=101812.967");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadInstance_2(Neuro::DicomInstancesCollection& target)
{
  // ('dcm_qa', 'In/Orientation/ax/axasc35/MR.1.3.12.2.1107.5.2.32.35131.2014031012493950715786673')
  Orthanc::DicomMap tags;
  tags.SetValue(0x0008, 0x0060, "MR", false);
  tags.SetValue(0x0008, 0x0070, "SIEMENS", false);
  tags.SetValue(0x0020, 0x0013, "1", false);
  tags.SetValue(0x0018, 0x0080, "3000", false);
  tags.SetValue(0x0018, 0x0081, "30", false);
  tags.SetValue(0x0018, 0x1312, "COL", false);
  tags.SetValue(0x0019, 0x1029, "0.0\\70.00000001\\142.50000002\\215.0\\285.0\\357.50000001\\429.99999999\\500.0\\572.50000001\\645.00000002\\715.0\\787.50000001\\860.00000002\\932.5\\1002.5\\1075.00000001\\1147.50000002\\1217.5\\1290.00000001\\1362.50000002\\1432.5\\1505.0\\1577.50000001\\1647.50000002\\1720.0\\1792.50000001\\1862.50000002\\1935.0\\2007.50000001\\2077.50000001\\2149.99999999\\2222.5\\2295.00000001\\2365.00000002\\2437.5", false);
  tags.SetValue(0x0008, 0x0016, "1.2.840.10008.5.1.4.1.1.4", false);
  tags.SetValue(0x0028, 0x0002, "1", false);
  tags.SetValue(0x0028, 0x0004, "MONOCHROME2", false);
  tags.SetValue(0x0028, 0x0010, "384", false);
  tags.SetValue(0x0028, 0x0011, "384", false);
  tags.SetValue(0x0028, 0x0100, "16", false);
  tags.SetValue(0x0028, 0x0101, "12", false);
  tags.SetValue(0x0028, 0x0102, "11", false);
  tags.SetValue(0x0028, 0x0103, "0", false);
  tags.SetValue(0x0008, 0x0032, "134935.305000", false);
  tags.SetValue(0x0018, 0x0050, "3", false);
  tags.SetValue(0x0018, 0x0088, "3.6000000030835", false);
  tags.SetValue(0x0020, 0x0032, "-624\\-661.82658862211\\-6.5255017698948", false);
  tags.SetValue(0x0020, 0x0037, "1\\-1e-016\\0\\1e-016\\0.99415096409965\\-0.1079993545339", false);
  tags.SetValue(0x0028, 0x0030, "3.25\\3.25", false);
  std::unique_ptr<Neuro::InputDicomInstance> dicom(new Neuro::InputDicomInstance(tags));
  dicom->GetCSAHeader().AddTag("NumberOfImagesInMosaic", "US").AddValue("35");
  dicom->GetCSAHeader().AddTag("PhaseEncodingDirectionPositive", "IS").AddValue("1");
  dicom->GetCSAHeader().AddTag("SliceNormalVector", "FD").AddValue("0.0").AddValue("0.10799944").AddValue("0.99415095");
  target.AddInstance(dicom.release(), "nope");
}

TEST(Autogenerated, Test_2)
{
  Neuro::DicomInstancesCollection instances;
  LoadInstance_2(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 64);
  ASSERT_NEAR(nifti.pixdim[1], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 64);
  ASSERT_NEAR(nifti.pixdim[2], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 35);
  ASSERT_NEAR(nifti.pixdim[3], 3.600000, 0.00001);
  ASSERT_EQ(nifti.ndim, 3);
  ASSERT_EQ(nifti.dim[0], 3);
  ASSERT_EQ(nifti.nvox, 143360u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(57));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(57));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(57));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 1);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -3.250000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 104.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 3.230991, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], -0.388798, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], -58.684311, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 0.350998, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 3.578943, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -84.798035, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, 0.998537, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, 0.054079, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=30;Time=134935.305;phase=1");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadInstance_3(Neuro::DicomInstancesCollection& target)
{
  // ('dcm_qa', 'In/Orientation/ax/axasc35/MR.1.3.12.2.1107.5.2.32.35131.2014031012494230872886774')
  Orthanc::DicomMap tags;
  tags.SetValue(0x0008, 0x0060, "MR", false);
  tags.SetValue(0x0008, 0x0070, "SIEMENS", false);
  tags.SetValue(0x0020, 0x0013, "2", false);
  tags.SetValue(0x0018, 0x0080, "3000", false);
  tags.SetValue(0x0018, 0x0081, "30", false);
  tags.SetValue(0x0018, 0x1312, "COL", false);
  tags.SetValue(0x0019, 0x1029, "0.0\\72.50000001\\144.99999999\\217.5\\287.50000001\\359.99999999\\432.5\\502.5\\575.00000001\\647.49999999\\717.5\\790.00000001\\862.49999999\\932.5\\1005.0\\1077.49999998\\1147.49999999\\1220.0\\1292.50000001\\1362.49999999\\1435.0\\1507.50000001\\1579.99999999\\1649.99999999\\1722.5\\1794.99999998\\1864.99999999\\1937.5\\2010.00000001\\2079.99999999\\2152.5\\2225.00000001\\2294.99999998\\2367.49999999\\2440.0", false);
  tags.SetValue(0x0008, 0x0016, "1.2.840.10008.5.1.4.1.1.4", false);
  tags.SetValue(0x0028, 0x0002, "1", false);
  tags.SetValue(0x0028, 0x0004, "MONOCHROME2", false);
  tags.SetValue(0x0028, 0x0010, "384", false);
  tags.SetValue(0x0028, 0x0011, "384", false);
  tags.SetValue(0x0028, 0x0100, "16", false);
  tags.SetValue(0x0028, 0x0101, "12", false);
  tags.SetValue(0x0028, 0x0102, "11", false);
  tags.SetValue(0x0028, 0x0103, "0", false);
  tags.SetValue(0x0008, 0x0032, "134938.315000", false);
  tags.SetValue(0x0018, 0x0050, "3", false);
  tags.SetValue(0x0018, 0x0088, "3.6000000030835", false);
  tags.SetValue(0x0020, 0x0032, "-624\\-661.82658862211\\-6.5255017698948", false);
  tags.SetValue(0x0020, 0x0037, "1\\-1e-016\\0\\1e-016\\0.99415096409965\\-0.1079993545339", false);
  tags.SetValue(0x0028, 0x0030, "3.25\\3.25", false);
  std::unique_ptr<Neuro::InputDicomInstance> dicom(new Neuro::InputDicomInstance(tags));
  dicom->GetCSAHeader().AddTag("NumberOfImagesInMosaic", "US").AddValue("35");
  dicom->GetCSAHeader().AddTag("PhaseEncodingDirectionPositive", "IS").AddValue("1");
  dicom->GetCSAHeader().AddTag("SliceNormalVector", "FD").AddValue("0.0").AddValue("0.10799944").AddValue("0.99415095");
  target.AddInstance(dicom.release(), "nope");
}

TEST(Autogenerated, Test_3)
{
  Neuro::DicomInstancesCollection instances;
  LoadInstance_3(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 64);
  ASSERT_NEAR(nifti.pixdim[1], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 64);
  ASSERT_NEAR(nifti.pixdim[2], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 35);
  ASSERT_NEAR(nifti.pixdim[3], 3.600000, 0.00001);
  ASSERT_EQ(nifti.ndim, 3);
  ASSERT_EQ(nifti.dim[0], 3);
  ASSERT_EQ(nifti.nvox, 143360u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(57));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(57));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(57));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 1);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -3.250000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 104.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 3.230991, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], -0.388798, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], -58.684311, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 0.350998, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 3.578943, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -84.798035, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, 0.998537, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, 0.054079, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=30;Time=134938.315;phase=1");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadInstance_4(Neuro::DicomInstancesCollection& target)
{
  // ('dcm_qa', 'In/Orientation/ax/axasc36/MR.1.3.12.2.1107.5.2.32.35131.2014031012525641770887330')
  Orthanc::DicomMap tags;
  tags.SetValue(0x0008, 0x0060, "MR", false);
  tags.SetValue(0x0008, 0x0070, "SIEMENS", false);
  tags.SetValue(0x0020, 0x0013, "1", false);
  tags.SetValue(0x0018, 0x0080, "3000", false);
  tags.SetValue(0x0018, 0x0081, "30", false);
  tags.SetValue(0x0018, 0x1312, "COL", false);
  tags.SetValue(0x0019, 0x1029, "0.0\\70.00000001\\139.99999998\\209.99999999\\280.0\\350.00000001\\417.49999998\\487.49999999\\557.5\\627.5\\697.49999998\\767.49999999\\837.49999999\\907.5\\977.49999998\\1044.99999998\\1114.99999999\\1185.0\\1255.0\\1324.99999998\\1394.99999999\\1465.0\\1535.0\\1604.99999998\\1672.49999999\\1742.49999999\\1812.5\\1882.50000001\\1952.49999998\\2022.49999999\\2092.5\\2162.50000001\\2232.49999998\\2299.99999999\\2370.0\\2440.0", false);
  tags.SetValue(0x0008, 0x0016, "1.2.840.10008.5.1.4.1.1.4", false);
  tags.SetValue(0x0028, 0x0002, "1", false);
  tags.SetValue(0x0028, 0x0004, "MONOCHROME2", false);
  tags.SetValue(0x0028, 0x0010, "384", false);
  tags.SetValue(0x0028, 0x0011, "384", false);
  tags.SetValue(0x0028, 0x0100, "16", false);
  tags.SetValue(0x0028, 0x0101, "12", false);
  tags.SetValue(0x0028, 0x0102, "11", false);
  tags.SetValue(0x0028, 0x0103, "0", false);
  tags.SetValue(0x0008, 0x0032, "135252.445000", false);
  tags.SetValue(0x0018, 0x0050, "3", false);
  tags.SetValue(0x0018, 0x0088, "3.6000000571183", false);
  tags.SetValue(0x0020, 0x0032, "-624\\-662.02098940946\\-8.3149762571995", false);
  tags.SetValue(0x0020, 0x0037, "1\\-1e-016\\0\\1e-016\\0.99415096409965\\-0.1079993545339", false);
  tags.SetValue(0x0028, 0x0030, "3.25\\3.25", false);
  std::unique_ptr<Neuro::InputDicomInstance> dicom(new Neuro::InputDicomInstance(tags));
  dicom->GetCSAHeader().AddTag("NumberOfImagesInMosaic", "US").AddValue("36");
  dicom->GetCSAHeader().AddTag("PhaseEncodingDirectionPositive", "IS").AddValue("1");
  dicom->GetCSAHeader().AddTag("SliceNormalVector", "FD").AddValue("0.0").AddValue("0.10799921").AddValue("0.99415098");
  target.AddInstance(dicom.release(), "nope");
}

TEST(Autogenerated, Test_4)
{
  Neuro::DicomInstancesCollection instances;
  LoadInstance_4(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 64);
  ASSERT_NEAR(nifti.pixdim[1], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 64);
  ASSERT_NEAR(nifti.pixdim[2], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 36);
  ASSERT_NEAR(nifti.pixdim[3], 3.600000, 0.00001);
  ASSERT_EQ(nifti.ndim, 3);
  ASSERT_EQ(nifti.dim[0], 3);
  ASSERT_EQ(nifti.nvox, 147456u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(57));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(57));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(57));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 1);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -3.250000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 104.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 3.230991, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], -0.388798, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], -58.489914, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 0.350998, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 3.578943, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -86.587509, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, 0.998537, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, 0.054079, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=30;Time=135252.445;phase=1");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadInstance_5(Neuro::DicomInstancesCollection& target)
{
  // ('dcm_qa', 'In/Orientation/ax/axasc36/MR.1.3.12.2.1107.5.2.32.35131.2014031012525922908387440')
  Orthanc::DicomMap tags;
  tags.SetValue(0x0008, 0x0060, "MR", false);
  tags.SetValue(0x0008, 0x0070, "SIEMENS", false);
  tags.SetValue(0x0020, 0x0013, "2", false);
  tags.SetValue(0x0018, 0x0080, "3000", false);
  tags.SetValue(0x0018, 0x0081, "30", false);
  tags.SetValue(0x0018, 0x1312, "COL", false);
  tags.SetValue(0x0019, 0x1029, "0.0\\70.00000001\\140.00000001\\210.00000002\\277.5\\347.5\\417.50000001\\487.50000002\\557.50000002\\627.5\\697.50000001\\767.50000002\\837.50000002\\905.0\\975.00000001\\1045.00000001\\1115.00000002\\1185.0\\1255.0\\1325.00000001\\1395.00000002\\1465.0\\1532.5\\1602.50000001\\1672.50000002\\1742.50000002\\1812.5\\1882.50000001\\1952.50000001\\2022.50000002\\2092.5\\2160.0\\2230.00000001\\2300.00000002\\2370.00000002\\2440.0", false);
  tags.SetValue(0x0008, 0x0016, "1.2.840.10008.5.1.4.1.1.4", false);
  tags.SetValue(0x0028, 0x0002, "1", false);
  tags.SetValue(0x0028, 0x0004, "MONOCHROME2", false);
  tags.SetValue(0x0028, 0x0010, "384", false);
  tags.SetValue(0x0028, 0x0011, "384", false);
  tags.SetValue(0x0028, 0x0100, "16", false);
  tags.SetValue(0x0028, 0x0101, "12", false);
  tags.SetValue(0x0028, 0x0102, "11", false);
  tags.SetValue(0x0028, 0x0103, "0", false);
  tags.SetValue(0x0008, 0x0032, "135255.457500", false);
  tags.SetValue(0x0018, 0x0050, "3", false);
  tags.SetValue(0x0018, 0x0088, "3.6000000571183", false);
  tags.SetValue(0x0020, 0x0032, "-624\\-662.02098940946\\-8.3149762571995", false);
  tags.SetValue(0x0020, 0x0037, "1\\-1e-016\\0\\1e-016\\0.99415096409965\\-0.1079993545339", false);
  tags.SetValue(0x0028, 0x0030, "3.25\\3.25", false);
  std::unique_ptr<Neuro::InputDicomInstance> dicom(new Neuro::InputDicomInstance(tags));
  dicom->GetCSAHeader().AddTag("NumberOfImagesInMosaic", "US").AddValue("36");
  dicom->GetCSAHeader().AddTag("PhaseEncodingDirectionPositive", "IS").AddValue("1");
  dicom->GetCSAHeader().AddTag("SliceNormalVector", "FD").AddValue("0.0").AddValue("0.10799921").AddValue("0.99415098");
  target.AddInstance(dicom.release(), "nope");
}

TEST(Autogenerated, Test_5)
{
  Neuro::DicomInstancesCollection instances;
  LoadInstance_5(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 64);
  ASSERT_NEAR(nifti.pixdim[1], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 64);
  ASSERT_NEAR(nifti.pixdim[2], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 36);
  ASSERT_NEAR(nifti.pixdim[3], 3.600000, 0.00001);
  ASSERT_EQ(nifti.ndim, 3);
  ASSERT_EQ(nifti.dim[0], 3);
  ASSERT_EQ(nifti.nvox, 147456u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(57));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(57));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(57));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 1);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -3.250000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 104.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 3.230991, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], -0.388798, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], -58.489914, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 0.350998, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 3.578943, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -86.587509, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, 0.998537, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, 0.054079, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=30;Time=135255.457;phase=1");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadInstance_6(Neuro::DicomInstancesCollection& target)
{
  // ('dcm_qa', 'In/Orientation/ax/axasc36b/MR.1.3.12.2.1107.5.2.32.35131.2014031012542072126387788')
  Orthanc::DicomMap tags;
  tags.SetValue(0x0008, 0x0060, "MR", false);
  tags.SetValue(0x0008, 0x0070, "SIEMENS", false);
  tags.SetValue(0x0020, 0x0013, "1", false);
  tags.SetValue(0x0018, 0x0080, "3000", false);
  tags.SetValue(0x0018, 0x0081, "30", false);
  tags.SetValue(0x0018, 0x1312, "COL", false);
  tags.SetValue(0x0019, 0x1029, "0.0\\70.00000001\\139.99999998\\209.99999999\\280.0\\350.00000001\\419.99999998\\487.49999999\\557.5\\627.5\\697.49999998\\767.49999999\\837.49999999\\907.5\\977.50000001\\1047.49999999\\1114.99999999\\1185.0\\1255.0\\1324.99999998\\1394.99999999\\1465.0\\1535.0\\1604.99999998\\1672.49999999\\1742.49999999\\1812.5\\1882.50000001\\1952.49999998\\2022.49999999\\2092.5\\2162.50000001\\2232.49999998\\2299.99999999\\2370.0\\2440.0", false);
  tags.SetValue(0x0008, 0x0016, "1.2.840.10008.5.1.4.1.1.4", false);
  tags.SetValue(0x0028, 0x0002, "1", false);
  tags.SetValue(0x0028, 0x0004, "MONOCHROME2", false);
  tags.SetValue(0x0028, 0x0010, "384", false);
  tags.SetValue(0x0028, 0x0011, "384", false);
  tags.SetValue(0x0028, 0x0100, "16", false);
  tags.SetValue(0x0028, 0x0101, "12", false);
  tags.SetValue(0x0028, 0x0102, "11", false);
  tags.SetValue(0x0028, 0x0103, "0", false);
  tags.SetValue(0x0008, 0x0032, "135416.225000", false);
  tags.SetValue(0x0018, 0x0050, "3", false);
  tags.SetValue(0x0018, 0x0088, "3.6000000571183", false);
  tags.SetValue(0x0020, 0x0032, "-624\\-662.02098940946\\-8.3149762571995", false);
  tags.SetValue(0x0020, 0x0037, "1\\-1e-016\\0\\1e-016\\0.99415096409965\\-0.1079993545339", false);
  tags.SetValue(0x0028, 0x0030, "3.25\\3.25", false);
  std::unique_ptr<Neuro::InputDicomInstance> dicom(new Neuro::InputDicomInstance(tags));
  dicom->GetCSAHeader().AddTag("NumberOfImagesInMosaic", "US").AddValue("36");
  dicom->GetCSAHeader().AddTag("PhaseEncodingDirectionPositive", "IS").AddValue("1");
  dicom->GetCSAHeader().AddTag("SliceNormalVector", "FD").AddValue("0.0").AddValue("0.10799921").AddValue("0.99415098");
  target.AddInstance(dicom.release(), "nope");
}

TEST(Autogenerated, Test_6)
{
  Neuro::DicomInstancesCollection instances;
  LoadInstance_6(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 64);
  ASSERT_NEAR(nifti.pixdim[1], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 64);
  ASSERT_NEAR(nifti.pixdim[2], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 36);
  ASSERT_NEAR(nifti.pixdim[3], 3.600000, 0.00001);
  ASSERT_EQ(nifti.ndim, 3);
  ASSERT_EQ(nifti.dim[0], 3);
  ASSERT_EQ(nifti.nvox, 147456u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(57));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(57));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(57));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 1);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -3.250000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 104.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 3.230991, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], -0.388798, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], -58.489914, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 0.350998, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 3.578943, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -86.587509, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, 0.998537, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, 0.054079, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=30;Time=135416.225;phase=1");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadInstance_7(Neuro::DicomInstancesCollection& target)
{
  // ('dcm_qa', 'In/Orientation/ax/axasc36b/MR.1.3.12.2.1107.5.2.32.35131.2014031012542352754587892')
  Orthanc::DicomMap tags;
  tags.SetValue(0x0008, 0x0060, "MR", false);
  tags.SetValue(0x0008, 0x0070, "SIEMENS", false);
  tags.SetValue(0x0020, 0x0013, "2", false);
  tags.SetValue(0x0018, 0x0080, "3000", false);
  tags.SetValue(0x0018, 0x0081, "30", false);
  tags.SetValue(0x0018, 0x1312, "COL", false);
  tags.SetValue(0x0019, 0x1029, "0.0\\70.00000001\\140.00000001\\210.00000002\\280.0\\347.5\\417.50000001\\487.50000002\\557.50000002\\627.5\\697.50000001\\767.50000002\\837.50000002\\907.5\\975.00000001\\1045.00000001\\1115.00000002\\1185.0\\1255.0\\1325.00000001\\1395.00000002\\1465.00000003\\1535.0\\1602.50000001\\1672.50000002\\1742.50000002\\1812.5\\1882.50000001\\1952.50000001\\2022.50000002\\2092.5\\2162.50000001\\2230.00000001\\2300.00000002\\2370.00000002\\2440.0", false);
  tags.SetValue(0x0008, 0x0016, "1.2.840.10008.5.1.4.1.1.4", false);
  tags.SetValue(0x0028, 0x0002, "1", false);
  tags.SetValue(0x0028, 0x0004, "MONOCHROME2", false);
  tags.SetValue(0x0028, 0x0010, "384", false);
  tags.SetValue(0x0028, 0x0011, "384", false);
  tags.SetValue(0x0028, 0x0100, "16", false);
  tags.SetValue(0x0028, 0x0101, "12", false);
  tags.SetValue(0x0028, 0x0102, "11", false);
  tags.SetValue(0x0028, 0x0103, "0", false);
  tags.SetValue(0x0008, 0x0032, "135419.237500", false);
  tags.SetValue(0x0018, 0x0050, "3", false);
  tags.SetValue(0x0018, 0x0088, "3.6000000571183", false);
  tags.SetValue(0x0020, 0x0032, "-624\\-662.02098940946\\-8.3149762571995", false);
  tags.SetValue(0x0020, 0x0037, "1\\-1e-016\\0\\1e-016\\0.99415096409965\\-0.1079993545339", false);
  tags.SetValue(0x0028, 0x0030, "3.25\\3.25", false);
  std::unique_ptr<Neuro::InputDicomInstance> dicom(new Neuro::InputDicomInstance(tags));
  dicom->GetCSAHeader().AddTag("NumberOfImagesInMosaic", "US").AddValue("36");
  dicom->GetCSAHeader().AddTag("PhaseEncodingDirectionPositive", "IS").AddValue("1");
  dicom->GetCSAHeader().AddTag("SliceNormalVector", "FD").AddValue("0.0").AddValue("0.10799921").AddValue("0.99415098");
  target.AddInstance(dicom.release(), "nope");
}

TEST(Autogenerated, Test_7)
{
  Neuro::DicomInstancesCollection instances;
  LoadInstance_7(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 64);
  ASSERT_NEAR(nifti.pixdim[1], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 64);
  ASSERT_NEAR(nifti.pixdim[2], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 36);
  ASSERT_NEAR(nifti.pixdim[3], 3.600000, 0.00001);
  ASSERT_EQ(nifti.ndim, 3);
  ASSERT_EQ(nifti.dim[0], 3);
  ASSERT_EQ(nifti.nvox, 147456u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(57));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(57));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(57));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 1);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -3.250000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 104.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 3.230991, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], -0.388798, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], -58.489914, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 0.350998, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 3.578943, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -86.587509, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, 0.998537, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, 0.054079, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=30;Time=135419.237;phase=1");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadInstance_8(Neuro::DicomInstancesCollection& target)
{
  // ('dcm_qa', 'In/Orientation/ax/axdesc35/MR.1.3.12.2.1107.5.2.32.35131.2014031012504272932486891')
  Orthanc::DicomMap tags;
  tags.SetValue(0x0008, 0x0060, "MR", false);
  tags.SetValue(0x0008, 0x0070, "SIEMENS", false);
  tags.SetValue(0x0020, 0x0013, "1", false);
  tags.SetValue(0x0018, 0x0080, "3000", false);
  tags.SetValue(0x0018, 0x0081, "30", false);
  tags.SetValue(0x0018, 0x1312, "COL", false);
  tags.SetValue(0x0019, 0x1029, "2437.5\\2364.99999999\\2295.00000001\\2222.5\\2149.99999999\\2080.00000002\\2007.50000001\\1935.0\\1864.99999999\\1792.50000001\\1720.0\\1649.99999999\\1577.50000001\\1505.0\\1435.0\\1362.50000002\\1290.00000001\\1220.0\\1147.49999999\\1075.00000001\\1002.5\\932.5\\860.00000002\\787.50000001\\717.5\\644.99999999\\572.50000001\\502.5\\429.99999999\\357.50000001\\287.50000001\\215.0\\142.50000002\\72.50000001\\0.0", false);
  tags.SetValue(0x0008, 0x0016, "1.2.840.10008.5.1.4.1.1.4", false);
  tags.SetValue(0x0028, 0x0002, "1", false);
  tags.SetValue(0x0028, 0x0004, "MONOCHROME2", false);
  tags.SetValue(0x0028, 0x0010, "384", false);
  tags.SetValue(0x0028, 0x0011, "384", false);
  tags.SetValue(0x0028, 0x0100, "16", false);
  tags.SetValue(0x0028, 0x0101, "12", false);
  tags.SetValue(0x0028, 0x0102, "11", false);
  tags.SetValue(0x0028, 0x0103, "0", false);
  tags.SetValue(0x0008, 0x0032, "135041.527500", false);
  tags.SetValue(0x0018, 0x0050, "3", false);
  tags.SetValue(0x0018, 0x0088, "3.6000000030835", false);
  tags.SetValue(0x0020, 0x0032, "-624\\-661.82658862211\\-6.5255017698948", false);
  tags.SetValue(0x0020, 0x0037, "1\\-1e-016\\0\\1e-016\\0.99415096409965\\-0.1079993545339", false);
  tags.SetValue(0x0028, 0x0030, "3.25\\3.25", false);
  std::unique_ptr<Neuro::InputDicomInstance> dicom(new Neuro::InputDicomInstance(tags));
  dicom->GetCSAHeader().AddTag("NumberOfImagesInMosaic", "US").AddValue("35");
  dicom->GetCSAHeader().AddTag("PhaseEncodingDirectionPositive", "IS").AddValue("1");
  dicom->GetCSAHeader().AddTag("SliceNormalVector", "FD").AddValue("0.0").AddValue("0.10799944").AddValue("0.99415095");
  target.AddInstance(dicom.release(), "nope");
}

TEST(Autogenerated, Test_8)
{
  Neuro::DicomInstancesCollection instances;
  LoadInstance_8(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 64);
  ASSERT_NEAR(nifti.pixdim[1], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 64);
  ASSERT_NEAR(nifti.pixdim[2], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 35);
  ASSERT_NEAR(nifti.pixdim[3], 3.600000, 0.00001);
  ASSERT_EQ(nifti.ndim, 3);
  ASSERT_EQ(nifti.dim[0], 3);
  ASSERT_EQ(nifti.nvox, 143360u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(57));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(57));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(57));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 2);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -3.250000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 104.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 3.230991, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], -0.388798, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], -58.684311, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 0.350998, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 3.578943, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -84.798035, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, 0.998537, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, 0.054079, 0.0001);
  ASSERT_TRUE(std::string(nifti.descrip) == "TE=30;Time=135041.527;phase=1" ||
              std::string(nifti.descrip) == "TE=30;Time=135041.528;phase=1");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadInstance_9(Neuro::DicomInstancesCollection& target)
{
  // ('dcm_qa', 'In/Orientation/ax/axdesc35/MR.1.3.12.2.1107.5.2.32.35131.2014031012504554260286994')
  Orthanc::DicomMap tags;
  tags.SetValue(0x0008, 0x0060, "MR", false);
  tags.SetValue(0x0008, 0x0070, "SIEMENS", false);
  tags.SetValue(0x0020, 0x0013, "2", false);
  tags.SetValue(0x0018, 0x0080, "3000", false);
  tags.SetValue(0x0018, 0x0081, "30", false);
  tags.SetValue(0x0018, 0x1312, "COL", false);
  tags.SetValue(0x0019, 0x1029, "2437.5\\2364.99999999\\2294.99999998\\2222.5\\2149.99999999\\2079.99999999\\2007.49999998\\1935.0\\1862.49999999\\1792.49999998\\1720.0\\1647.49999999\\1577.49999998\\1505.0\\1432.5\\1362.49999999\\1289.99999998\\1217.5\\1147.49999999\\1074.99999998\\1002.5\\932.5\\859.99999999\\787.50000001\\717.5\\644.99999999\\572.49999998\\502.5\\429.99999999\\357.49999998\\285.0\\215.0\\142.49999999\\69.99999998\\0.0", false);
  tags.SetValue(0x0008, 0x0016, "1.2.840.10008.5.1.4.1.1.4", false);
  tags.SetValue(0x0028, 0x0002, "1", false);
  tags.SetValue(0x0028, 0x0004, "MONOCHROME2", false);
  tags.SetValue(0x0028, 0x0010, "384", false);
  tags.SetValue(0x0028, 0x0011, "384", false);
  tags.SetValue(0x0028, 0x0100, "16", false);
  tags.SetValue(0x0028, 0x0101, "12", false);
  tags.SetValue(0x0028, 0x0102, "11", false);
  tags.SetValue(0x0028, 0x0103, "0", false);
  tags.SetValue(0x0008, 0x0032, "135044.540000", false);
  tags.SetValue(0x0018, 0x0050, "3", false);
  tags.SetValue(0x0018, 0x0088, "3.6000000030835", false);
  tags.SetValue(0x0020, 0x0032, "-624\\-661.82658862211\\-6.5255017698948", false);
  tags.SetValue(0x0020, 0x0037, "1\\-1e-016\\0\\1e-016\\0.99415096409965\\-0.1079993545339", false);
  tags.SetValue(0x0028, 0x0030, "3.25\\3.25", false);
  std::unique_ptr<Neuro::InputDicomInstance> dicom(new Neuro::InputDicomInstance(tags));
  dicom->GetCSAHeader().AddTag("NumberOfImagesInMosaic", "US").AddValue("35");
  dicom->GetCSAHeader().AddTag("PhaseEncodingDirectionPositive", "IS").AddValue("1");
  dicom->GetCSAHeader().AddTag("SliceNormalVector", "FD").AddValue("0.0").AddValue("0.10799944").AddValue("0.99415095");
  target.AddInstance(dicom.release(), "nope");
}

TEST(Autogenerated, Test_9)
{
  Neuro::DicomInstancesCollection instances;
  LoadInstance_9(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 64);
  ASSERT_NEAR(nifti.pixdim[1], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 64);
  ASSERT_NEAR(nifti.pixdim[2], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 35);
  ASSERT_NEAR(nifti.pixdim[3], 3.600000, 0.00001);
  ASSERT_EQ(nifti.ndim, 3);
  ASSERT_EQ(nifti.dim[0], 3);
  ASSERT_EQ(nifti.nvox, 143360u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(57));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(57));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(57));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 2);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -3.250000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 104.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 3.230991, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], -0.388798, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], -58.684311, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 0.350998, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 3.578943, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -84.798035, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, 0.998537, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, 0.054079, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=30;Time=135044.540;phase=1");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadInstance_10(Neuro::DicomInstancesCollection& target)
{
  // ('dcm_qa', 'In/Orientation/ax/axdesc36/MR.1.3.12.2.1107.5.2.32.35131.2014031012533345318387559')
  Orthanc::DicomMap tags;
  tags.SetValue(0x0008, 0x0060, "MR", false);
  tags.SetValue(0x0008, 0x0070, "SIEMENS", false);
  tags.SetValue(0x0020, 0x0013, "1", false);
  tags.SetValue(0x0018, 0x0080, "3000", false);
  tags.SetValue(0x0018, 0x0081, "30", false);
  tags.SetValue(0x0018, 0x1312, "COL", false);
  tags.SetValue(0x0019, 0x1029, "2439.99999997\\2372.5\\2302.49999999\\2232.49999998\\2162.49999998\\2092.5\\2022.49999999\\1952.49999998\\1882.49999998\\1812.5\\1745.0\\1674.99999999\\1604.99999998\\1534.99999997\\1465.0\\1394.99999999\\1324.99999998\\1254.99999998\\1185.0\\1117.49999999\\1047.49999999\\977.49999998\\907.5\\837.49999999\\767.49999999\\697.49999998\\627.49999997\\557.5\\489.99999999\\419.99999998\\349.99999998\\280.0\\209.99999999\\139.99999998\\69.99999998\\0.0", false);
  tags.SetValue(0x0008, 0x0016, "1.2.840.10008.5.1.4.1.1.4", false);
  tags.SetValue(0x0028, 0x0002, "1", false);
  tags.SetValue(0x0028, 0x0004, "MONOCHROME2", false);
  tags.SetValue(0x0028, 0x0010, "384", false);
  tags.SetValue(0x0028, 0x0011, "384", false);
  tags.SetValue(0x0028, 0x0100, "16", false);
  tags.SetValue(0x0028, 0x0101, "12", false);
  tags.SetValue(0x0028, 0x0102, "11", false);
  tags.SetValue(0x0028, 0x0103, "0", false);
  tags.SetValue(0x0008, 0x0032, "135332.235000", false);
  tags.SetValue(0x0018, 0x0050, "3", false);
  tags.SetValue(0x0018, 0x0088, "3.6000000571183", false);
  tags.SetValue(0x0020, 0x0032, "-624\\-662.02098940946\\-8.3149762571995", false);
  tags.SetValue(0x0020, 0x0037, "1\\-1e-016\\0\\1e-016\\0.99415096409965\\-0.1079993545339", false);
  tags.SetValue(0x0028, 0x0030, "3.25\\3.25", false);
  std::unique_ptr<Neuro::InputDicomInstance> dicom(new Neuro::InputDicomInstance(tags));
  dicom->GetCSAHeader().AddTag("NumberOfImagesInMosaic", "US").AddValue("36");
  dicom->GetCSAHeader().AddTag("PhaseEncodingDirectionPositive", "IS").AddValue("1");
  dicom->GetCSAHeader().AddTag("SliceNormalVector", "FD").AddValue("0.0").AddValue("0.10799921").AddValue("0.99415098");
  target.AddInstance(dicom.release(), "nope");
}

TEST(Autogenerated, Test_10)
{
  Neuro::DicomInstancesCollection instances;
  LoadInstance_10(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 64);
  ASSERT_NEAR(nifti.pixdim[1], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 64);
  ASSERT_NEAR(nifti.pixdim[2], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 36);
  ASSERT_NEAR(nifti.pixdim[3], 3.600000, 0.00001);
  ASSERT_EQ(nifti.ndim, 3);
  ASSERT_EQ(nifti.dim[0], 3);
  ASSERT_EQ(nifti.nvox, 147456u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(57));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(57));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(57));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 2);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -3.250000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 104.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 3.230991, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], -0.388798, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], -58.489914, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 0.350998, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 3.578943, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -86.587509, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, 0.998537, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, 0.054079, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=30;Time=135332.235;phase=1");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadInstance_11(Neuro::DicomInstancesCollection& target)
{
  // ('dcm_qa', 'In/Orientation/ax/axdesc36/MR.1.3.12.2.1107.5.2.32.35131.2014031012533626317387666')
  Orthanc::DicomMap tags;
  tags.SetValue(0x0008, 0x0060, "MR", false);
  tags.SetValue(0x0008, 0x0070, "SIEMENS", false);
  tags.SetValue(0x0020, 0x0013, "2", false);
  tags.SetValue(0x0018, 0x0080, "3000", false);
  tags.SetValue(0x0018, 0x0081, "30", false);
  tags.SetValue(0x0018, 0x1312, "COL", false);
  tags.SetValue(0x0019, 0x1029, "2440.0\\2370.0\\2300.00000002\\2232.50000001\\2162.50000001\\2092.5\\2022.49999999\\1952.50000001\\1882.50000001\\1812.5\\1742.49999999\\1672.50000002\\1605.00000001\\1535.0\\1465.0\\1395.00000002\\1325.00000001\\1255.0\\1185.0\\1114.99999999\\1045.00000001\\977.50000001\\907.5\\837.49999999\\767.50000002\\697.50000001\\627.5\\557.5\\487.50000002\\417.50000001\\350.00000001\\280.0\\209.99999999\\140.00000001\\70.00000001\\0.0", false);
  tags.SetValue(0x0008, 0x0016, "1.2.840.10008.5.1.4.1.1.4", false);
  tags.SetValue(0x0028, 0x0002, "1", false);
  tags.SetValue(0x0028, 0x0004, "MONOCHROME2", false);
  tags.SetValue(0x0028, 0x0010, "384", false);
  tags.SetValue(0x0028, 0x0011, "384", false);
  tags.SetValue(0x0028, 0x0100, "16", false);
  tags.SetValue(0x0028, 0x0101, "12", false);
  tags.SetValue(0x0028, 0x0102, "11", false);
  tags.SetValue(0x0028, 0x0103, "0", false);
  tags.SetValue(0x0008, 0x0032, "135335.247500", false);
  tags.SetValue(0x0018, 0x0050, "3", false);
  tags.SetValue(0x0018, 0x0088, "3.6000000571183", false);
  tags.SetValue(0x0020, 0x0032, "-624\\-662.02098940946\\-8.3149762571995", false);
  tags.SetValue(0x0020, 0x0037, "1\\-1e-016\\0\\1e-016\\0.99415096409965\\-0.1079993545339", false);
  tags.SetValue(0x0028, 0x0030, "3.25\\3.25", false);
  std::unique_ptr<Neuro::InputDicomInstance> dicom(new Neuro::InputDicomInstance(tags));
  dicom->GetCSAHeader().AddTag("NumberOfImagesInMosaic", "US").AddValue("36");
  dicom->GetCSAHeader().AddTag("PhaseEncodingDirectionPositive", "IS").AddValue("1");
  dicom->GetCSAHeader().AddTag("SliceNormalVector", "FD").AddValue("0.0").AddValue("0.10799921").AddValue("0.99415098");
  target.AddInstance(dicom.release(), "nope");
}

TEST(Autogenerated, Test_11)
{
  Neuro::DicomInstancesCollection instances;
  LoadInstance_11(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 64);
  ASSERT_NEAR(nifti.pixdim[1], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 64);
  ASSERT_NEAR(nifti.pixdim[2], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 36);
  ASSERT_NEAR(nifti.pixdim[3], 3.600000, 0.00001);
  ASSERT_EQ(nifti.ndim, 3);
  ASSERT_EQ(nifti.dim[0], 3);
  ASSERT_EQ(nifti.nvox, 147456u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(57));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(57));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(57));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 2);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -3.250000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 104.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 3.230991, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], -0.388798, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], -58.489914, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 0.350998, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 3.578943, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -86.587509, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, 0.998537, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, 0.054079, 0.0001);
  ASSERT_TRUE(std::string(nifti.descrip) == "TE=30;Time=135335.247;phase=1" ||
              std::string(nifti.descrip) == "TE=30;Time=135335.248;phase=1");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadInstance_12(Neuro::DicomInstancesCollection& target)
{
  // ('dcm_qa', 'In/Orientation/ax/axint35/MR.1.3.12.2.1107.5.2.32.35131.2014031012511448225287110')
  Orthanc::DicomMap tags;
  tags.SetValue(0x0008, 0x0060, "MR", false);
  tags.SetValue(0x0008, 0x0070, "SIEMENS", false);
  tags.SetValue(0x0020, 0x0013, "1", false);
  tags.SetValue(0x0018, 0x0080, "3000", false);
  tags.SetValue(0x0018, 0x0081, "30", false);
  tags.SetValue(0x0018, 0x1312, "COL", false);
  tags.SetValue(0x0019, 0x1029, "0.0\\1292.50000001\\72.50000001\\1362.50000002\\145.00000002\\1435.0\\215.0\\1507.50000001\\287.50000001\\1577.50000001\\360.00000002\\1649.99999999\\429.99999999\\1722.5\\502.5\\1792.50000001\\575.00000001\\1864.99999999\\645.00000002\\1937.5\\717.5\\2007.50000001\\790.00000001\\2080.00000002\\860.00000002\\2152.5\\932.5\\2222.5\\1005.0\\2295.00000001\\1075.00000001\\2367.49999999\\1147.49999999\\2437.5\\1220.0", false);
  tags.SetValue(0x0008, 0x0016, "1.2.840.10008.5.1.4.1.1.4", false);
  tags.SetValue(0x0028, 0x0002, "1", false);
  tags.SetValue(0x0028, 0x0004, "MONOCHROME2", false);
  tags.SetValue(0x0028, 0x0010, "384", false);
  tags.SetValue(0x0028, 0x0011, "384", false);
  tags.SetValue(0x0028, 0x0100, "16", false);
  tags.SetValue(0x0028, 0x0101, "12", false);
  tags.SetValue(0x0028, 0x0102, "11", false);
  tags.SetValue(0x0028, 0x0103, "0", false);
  tags.SetValue(0x0008, 0x0032, "135110.870000", false);
  tags.SetValue(0x0018, 0x0050, "3", false);
  tags.SetValue(0x0018, 0x0088, "3.6000000030835", false);
  tags.SetValue(0x0020, 0x0032, "-624\\-661.82658862211\\-6.5255017698948", false);
  tags.SetValue(0x0020, 0x0037, "1\\-1e-016\\0\\1e-016\\0.99415096409965\\-0.1079993545339", false);
  tags.SetValue(0x0028, 0x0030, "3.25\\3.25", false);
  std::unique_ptr<Neuro::InputDicomInstance> dicom(new Neuro::InputDicomInstance(tags));
  dicom->GetCSAHeader().AddTag("NumberOfImagesInMosaic", "US").AddValue("35");
  dicom->GetCSAHeader().AddTag("PhaseEncodingDirectionPositive", "IS").AddValue("1");
  dicom->GetCSAHeader().AddTag("SliceNormalVector", "FD").AddValue("0.0").AddValue("0.10799944").AddValue("0.99415095");
  target.AddInstance(dicom.release(), "nope");
}

TEST(Autogenerated, Test_12)
{
  Neuro::DicomInstancesCollection instances;
  LoadInstance_12(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 64);
  ASSERT_NEAR(nifti.pixdim[1], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 64);
  ASSERT_NEAR(nifti.pixdim[2], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 35);
  ASSERT_NEAR(nifti.pixdim[3], 3.600000, 0.00001);
  ASSERT_EQ(nifti.ndim, 3);
  ASSERT_EQ(nifti.dim[0], 3);
  ASSERT_EQ(nifti.nvox, 143360u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(57));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(57));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(57));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 3);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -3.250000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 104.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 3.230991, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], -0.388798, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], -58.684311, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 0.350998, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 3.578943, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -84.798035, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, 0.998537, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, 0.054079, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=30;Time=135110.870;phase=1");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadInstance_13(Neuro::DicomInstancesCollection& target)
{
  // ('dcm_qa', 'In/Orientation/ax/axint35/MR.1.3.12.2.1107.5.2.32.35131.2014031012511729322987214')
  Orthanc::DicomMap tags;
  tags.SetValue(0x0008, 0x0060, "MR", false);
  tags.SetValue(0x0008, 0x0070, "SIEMENS", false);
  tags.SetValue(0x0020, 0x0013, "2", false);
  tags.SetValue(0x0018, 0x0080, "3000", false);
  tags.SetValue(0x0018, 0x0081, "30", false);
  tags.SetValue(0x0018, 0x1312, "COL", false);
  tags.SetValue(0x0019, 0x1029, "0.0\\1289.99999998\\72.49999998\\1362.49999999\\142.49999999\\1435.0\\215.0\\1505.0\\287.50000001\\1577.49999998\\357.49999998\\1649.99999999\\429.99999999\\1720.0\\502.5\\1792.49999998\\574.99999998\\1864.99999999\\644.99999999\\1935.0\\717.5\\2007.50000001\\789.99999998\\2079.99999999\\859.99999999\\2152.5\\932.5\\2222.5\\1005.0\\2294.99999998\\1074.99999998\\2367.49999999\\1147.49999999\\2437.5\\1220.0", false);
  tags.SetValue(0x0008, 0x0016, "1.2.840.10008.5.1.4.1.1.4", false);
  tags.SetValue(0x0028, 0x0002, "1", false);
  tags.SetValue(0x0028, 0x0004, "MONOCHROME2", false);
  tags.SetValue(0x0028, 0x0010, "384", false);
  tags.SetValue(0x0028, 0x0011, "384", false);
  tags.SetValue(0x0028, 0x0100, "16", false);
  tags.SetValue(0x0028, 0x0101, "12", false);
  tags.SetValue(0x0028, 0x0102, "11", false);
  tags.SetValue(0x0028, 0x0103, "0", false);
  tags.SetValue(0x0008, 0x0032, "135113.882500", false);
  tags.SetValue(0x0018, 0x0050, "3", false);
  tags.SetValue(0x0018, 0x0088, "3.6000000030835", false);
  tags.SetValue(0x0020, 0x0032, "-624\\-661.82658862211\\-6.5255017698948", false);
  tags.SetValue(0x0020, 0x0037, "1\\-1e-016\\0\\1e-016\\0.99415096409965\\-0.1079993545339", false);
  tags.SetValue(0x0028, 0x0030, "3.25\\3.25", false);
  std::unique_ptr<Neuro::InputDicomInstance> dicom(new Neuro::InputDicomInstance(tags));
  dicom->GetCSAHeader().AddTag("NumberOfImagesInMosaic", "US").AddValue("35");
  dicom->GetCSAHeader().AddTag("PhaseEncodingDirectionPositive", "IS").AddValue("1");
  dicom->GetCSAHeader().AddTag("SliceNormalVector", "FD").AddValue("0.0").AddValue("0.10799944").AddValue("0.99415095");
  target.AddInstance(dicom.release(), "nope");
}

TEST(Autogenerated, Test_13)
{
  Neuro::DicomInstancesCollection instances;
  LoadInstance_13(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 64);
  ASSERT_NEAR(nifti.pixdim[1], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 64);
  ASSERT_NEAR(nifti.pixdim[2], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 35);
  ASSERT_NEAR(nifti.pixdim[3], 3.600000, 0.00001);
  ASSERT_EQ(nifti.ndim, 3);
  ASSERT_EQ(nifti.dim[0], 3);
  ASSERT_EQ(nifti.nvox, 143360u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(57));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(57));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(57));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 3);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -3.250000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 104.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 3.230991, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], -0.388798, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], -58.684311, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 0.350998, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 3.578943, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -84.798035, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, 0.998537, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, 0.054079, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=30;Time=135113.883;phase=1");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadInstance_14(Neuro::DicomInstancesCollection& target)
{
  // ('dcm_qa', 'In/Orientation/ax/axint36/MR.1.3.12.2.1107.5.2.32.35131.2014031012544786586988013')
  Orthanc::DicomMap tags;
  tags.SetValue(0x0008, 0x0060, "MR", false);
  tags.SetValue(0x0008, 0x0070, "SIEMENS", false);
  tags.SetValue(0x0020, 0x0013, "1", false);
  tags.SetValue(0x0018, 0x0080, "3000", false);
  tags.SetValue(0x0018, 0x0081, "30", false);
  tags.SetValue(0x0018, 0x1312, "COL", false);
  tags.SetValue(0x0019, 0x1029, "1255.0\\0.0\\1325.00000001\\70.00000001\\1394.99999999\\140.00000001\\1465.0\\209.99999999\\1535.0\\280.0\\1605.00000001\\350.00000001\\1672.50000002\\417.50000001\\1742.49999999\\487.49999999\\1812.5\\557.5\\1882.50000001\\627.5\\1952.50000001\\697.50000001\\2022.49999999\\767.49999999\\2092.5\\837.49999999\\2162.50000001\\907.5\\2232.50000001\\977.50000001\\2299.99999999\\1045.00000001\\2370.0\\1114.99999999\\2440.0\\1185.0", false);
  tags.SetValue(0x0008, 0x0016, "1.2.840.10008.5.1.4.1.1.4", false);
  tags.SetValue(0x0028, 0x0002, "1", false);
  tags.SetValue(0x0028, 0x0004, "MONOCHROME2", false);
  tags.SetValue(0x0028, 0x0010, "384", false);
  tags.SetValue(0x0028, 0x0011, "384", false);
  tags.SetValue(0x0028, 0x0100, "16", false);
  tags.SetValue(0x0028, 0x0101, "12", false);
  tags.SetValue(0x0028, 0x0102, "11", false);
  tags.SetValue(0x0028, 0x0103, "0", false);
  tags.SetValue(0x0008, 0x0032, "135444.722500", false);
  tags.SetValue(0x0018, 0x0050, "3", false);
  tags.SetValue(0x0018, 0x0088, "3.6000000571183", false);
  tags.SetValue(0x0020, 0x0032, "-624\\-662.02098940946\\-8.3149762571995", false);
  tags.SetValue(0x0020, 0x0037, "1\\-1e-016\\0\\1e-016\\0.99415096409965\\-0.1079993545339", false);
  tags.SetValue(0x0028, 0x0030, "3.25\\3.25", false);
  std::unique_ptr<Neuro::InputDicomInstance> dicom(new Neuro::InputDicomInstance(tags));
  dicom->GetCSAHeader().AddTag("NumberOfImagesInMosaic", "US").AddValue("36");
  dicom->GetCSAHeader().AddTag("PhaseEncodingDirectionPositive", "IS").AddValue("1");
  dicom->GetCSAHeader().AddTag("SliceNormalVector", "FD").AddValue("0.0").AddValue("0.10799921").AddValue("0.99415098");
  target.AddInstance(dicom.release(), "nope");
}

TEST(Autogenerated, Test_14)
{
  Neuro::DicomInstancesCollection instances;
  LoadInstance_14(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 64);
  ASSERT_NEAR(nifti.pixdim[1], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 64);
  ASSERT_NEAR(nifti.pixdim[2], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 36);
  ASSERT_NEAR(nifti.pixdim[3], 3.600000, 0.00001);
  ASSERT_EQ(nifti.ndim, 3);
  ASSERT_EQ(nifti.dim[0], 3);
  ASSERT_EQ(nifti.nvox, 147456u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(57));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(57));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(57));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 5);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -3.250000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 104.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 3.230991, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], -0.388798, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], -58.489914, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 0.350998, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 3.578943, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -86.587509, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, 0.998537, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, 0.054079, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=30;Time=135444.723;phase=1");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadInstance_15(Neuro::DicomInstancesCollection& target)
{
  // ('dcm_qa', 'In/Orientation/ax/axint36/MR.1.3.12.2.1107.5.2.32.35131.2014031012545067891488118')
  Orthanc::DicomMap tags;
  tags.SetValue(0x0008, 0x0060, "MR", false);
  tags.SetValue(0x0008, 0x0070, "SIEMENS", false);
  tags.SetValue(0x0020, 0x0013, "2", false);
  tags.SetValue(0x0018, 0x0080, "3000", false);
  tags.SetValue(0x0018, 0x0081, "30", false);
  tags.SetValue(0x0018, 0x1312, "COL", false);
  tags.SetValue(0x0019, 0x1029, "1254.99999998\\0.0\\1324.99999998\\69.99999998\\1394.99999999\\139.99999998\\1465.0\\209.99999999\\1532.5\\280.0\\1602.49999998\\347.5\\1672.49999999\\417.49999998\\1742.49999999\\487.49999999\\1812.5\\557.5\\1882.49999998\\627.5\\1952.49999998\\697.49999998\\2022.49999999\\767.49999999\\2092.5\\837.49999999\\2160.0\\905.0\\2229.99999998\\974.99999998\\2299.99999999\\1044.99999998\\2370.0\\1114.99999999\\2440.0\\1185.0", false);
  tags.SetValue(0x0008, 0x0016, "1.2.840.10008.5.1.4.1.1.4", false);
  tags.SetValue(0x0028, 0x0002, "1", false);
  tags.SetValue(0x0028, 0x0004, "MONOCHROME2", false);
  tags.SetValue(0x0028, 0x0010, "384", false);
  tags.SetValue(0x0028, 0x0011, "384", false);
  tags.SetValue(0x0028, 0x0100, "16", false);
  tags.SetValue(0x0028, 0x0101, "12", false);
  tags.SetValue(0x0028, 0x0102, "11", false);
  tags.SetValue(0x0028, 0x0103, "0", false);
  tags.SetValue(0x0008, 0x0032, "135447.735000", false);
  tags.SetValue(0x0018, 0x0050, "3", false);
  tags.SetValue(0x0018, 0x0088, "3.6000000571183", false);
  tags.SetValue(0x0020, 0x0032, "-624\\-662.02098940946\\-8.3149762571995", false);
  tags.SetValue(0x0020, 0x0037, "1\\-1e-016\\0\\1e-016\\0.99415096409965\\-0.1079993545339", false);
  tags.SetValue(0x0028, 0x0030, "3.25\\3.25", false);
  std::unique_ptr<Neuro::InputDicomInstance> dicom(new Neuro::InputDicomInstance(tags));
  dicom->GetCSAHeader().AddTag("NumberOfImagesInMosaic", "US").AddValue("36");
  dicom->GetCSAHeader().AddTag("PhaseEncodingDirectionPositive", "IS").AddValue("1");
  dicom->GetCSAHeader().AddTag("SliceNormalVector", "FD").AddValue("0.0").AddValue("0.10799921").AddValue("0.99415098");
  target.AddInstance(dicom.release(), "nope");
}

TEST(Autogenerated, Test_15)
{
  Neuro::DicomInstancesCollection instances;
  LoadInstance_15(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 64);
  ASSERT_NEAR(nifti.pixdim[1], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 64);
  ASSERT_NEAR(nifti.pixdim[2], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 36);
  ASSERT_NEAR(nifti.pixdim[3], 3.600000, 0.00001);
  ASSERT_EQ(nifti.ndim, 3);
  ASSERT_EQ(nifti.dim[0], 3);
  ASSERT_EQ(nifti.nvox, 147456u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(57));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(57));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(57));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 5);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -3.250000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 104.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 3.230991, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], -0.388798, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], -58.489914, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 0.350998, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 3.578943, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -86.587509, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, 0.998537, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, 0.054079, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=30;Time=135447.735;phase=1");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadInstance_16(Neuro::DicomInstancesCollection& target)
{
  // ('dcm_qa', 'In/Orientation/axmb/AxAsc36mb2a/jpg1.dcm')
  Orthanc::DicomMap tags;
  tags.SetValue(0x0008, 0x0060, "MR", false);
  tags.SetValue(0x0008, 0x0070, "SIEMENS", false);
  tags.SetValue(0x0020, 0x0013, "1", false);
  tags.SetValue(0x0018, 0x0080, "3000", false);
  tags.SetValue(0x0018, 0x0081, "34", false);
  tags.SetValue(0x0018, 0x1312, "COL", false);
  tags.SetValue(0x0019, 0x1029, "0.0\\137.49999998\\277.5\\417.49999998\\557.5\\694.99999998\\834.99999999\\974.99999998\\1114.99999999\\1254.99999998\\1392.49999999\\1532.5\\1672.49999999\\1812.5\\1949.99999998\\2090.0\\2229.99999998\\2370.0\\2509.99999998\\2647.49999999\\2787.49999998\\2927.49999999\\3067.49999998\\3204.99999999\\3345.0\\3484.99999999\\3625.0\\3764.99999998\\3902.5\\4042.49999998\\4182.5\\4322.49999998\\4459.99999999\\4599.99999998\\4739.99999999\\4879.99999998", false);
  tags.SetValue(0x0008, 0x0016, "1.2.840.10008.5.1.4.1.1.4", false);
  tags.SetValue(0x0028, 0x0002, "1", false);
  tags.SetValue(0x0028, 0x0004, "MONOCHROME2", false);
  tags.SetValue(0x0028, 0x0010, "516", false);
  tags.SetValue(0x0028, 0x0011, "516", false);
  tags.SetValue(0x0028, 0x0100, "16", false);
  tags.SetValue(0x0028, 0x0101, "12", false);
  tags.SetValue(0x0028, 0x0102, "11", false);
  tags.SetValue(0x0028, 0x0103, "0", false);
  tags.SetValue(0x0008, 0x0032, "140149.417500", false);
  tags.SetValue(0x0018, 0x0050, "3", false);
  tags.SetValue(0x0018, 0x0088, "3.599999959714", false);
  tags.SetValue(0x0020, 0x0032, "-696.0000243187\\-737.45290897958\\51.565712817043", false);
  tags.SetValue(0x0020, 0x0037, "1\\2e-016\\0\\-2e-016\\0.98388503666185\\-0.178802222114", false);
  tags.SetValue(0x0028, 0x0030, "2.6976745128632\\2.6976745128632", false);
  std::unique_ptr<Neuro::InputDicomInstance> dicom(new Neuro::InputDicomInstance(tags));
  dicom->GetCSAHeader().AddTag("NumberOfImagesInMosaic", "US").AddValue("36");
  dicom->GetCSAHeader().AddTag("PhaseEncodingDirectionPositive", "IS").AddValue("1");
  dicom->GetCSAHeader().AddTag("SliceNormalVector", "FD").AddValue("0.0").AddValue("0.17880235").AddValue("0.98388501");
  target.AddInstance(dicom.release(), "nope");
}

TEST(Autogenerated, Test_16)
{
  Neuro::DicomInstancesCollection instances;
  LoadInstance_16(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 86);
  ASSERT_NEAR(nifti.pixdim[1], 2.697675, 0.00001);
  ASSERT_EQ(nifti.dim[2], 86);
  ASSERT_NEAR(nifti.pixdim[2], 2.697675, 0.00001);
  ASSERT_EQ(nifti.dim[3], 36);
  ASSERT_NEAR(nifti.pixdim[3], 3.600000, 0.00001);
  ASSERT_EQ(nifti.ndim, 3);
  ASSERT_EQ(nifti.dim[0], 3);
  ASSERT_EQ(nifti.nvox, 266256u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(57));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(57));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(57));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 1);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -2.697675, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 115.999977, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 2.654202, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], -0.643688, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], -58.807571, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 0.482350, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 3.541986, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -93.139343, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, -0.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, 0.995963, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, 0.089763, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=34;Time=140149.418;phase=1");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadInstance_17(Neuro::DicomInstancesCollection& target)
{
  // ('dcm_qa', 'In/Orientation/axmb/AxAsc36mb2a/jpg2.dcm')
  Orthanc::DicomMap tags;
  tags.SetValue(0x0008, 0x0060, "MR", false);
  tags.SetValue(0x0008, 0x0070, "SIEMENS", false);
  tags.SetValue(0x0020, 0x0013, "2", false);
  tags.SetValue(0x0018, 0x0080, "3000", false);
  tags.SetValue(0x0018, 0x0081, "34", false);
  tags.SetValue(0x0018, 0x1312, "COL", false);
  tags.SetValue(0x0019, 0x1029, "0.0\\137.49999998\\277.5\\417.49999998\\557.5\\697.49999998\\834.99999999\\975.00000001\\1114.99999999\\1255.0\\1392.49999999\\1532.5\\1672.49999999\\1812.5\\1952.49999998\\2090.0\\2229.99999998\\2370.0\\0.0\\137.49999998\\277.5\\417.49999998\\557.5\\697.49999998\\834.99999999\\975.00000001\\1114.99999999\\1255.0\\1392.49999999\\1532.5\\1672.49999999\\1812.5\\1952.49999998\\2090.0\\2229.99999998\\2370.0", false);
  tags.SetValue(0x0008, 0x0016, "1.2.840.10008.5.1.4.1.1.4", false);
  tags.SetValue(0x0028, 0x0002, "1", false);
  tags.SetValue(0x0028, 0x0004, "MONOCHROME2", false);
  tags.SetValue(0x0028, 0x0010, "516", false);
  tags.SetValue(0x0028, 0x0011, "516", false);
  tags.SetValue(0x0028, 0x0100, "16", false);
  tags.SetValue(0x0028, 0x0101, "12", false);
  tags.SetValue(0x0028, 0x0102, "11", false);
  tags.SetValue(0x0028, 0x0103, "0", false);
  tags.SetValue(0x0008, 0x0032, "140203.975000", false);
  tags.SetValue(0x0018, 0x0050, "3", false);
  tags.SetValue(0x0018, 0x0088, "3.599999959714", false);
  tags.SetValue(0x0020, 0x0032, "-696.0000243187\\-737.45290897958\\51.565712817043", false);
  tags.SetValue(0x0020, 0x0037, "1\\2e-016\\0\\-2e-016\\0.98388503666185\\-0.178802222114", false);
  tags.SetValue(0x0028, 0x0030, "2.6976745128632\\2.6976745128632", false);
  std::unique_ptr<Neuro::InputDicomInstance> dicom(new Neuro::InputDicomInstance(tags));
  dicom->GetCSAHeader().AddTag("NumberOfImagesInMosaic", "US").AddValue("36");
  dicom->GetCSAHeader().AddTag("PhaseEncodingDirectionPositive", "IS").AddValue("1");
  dicom->GetCSAHeader().AddTag("SliceNormalVector", "FD").AddValue("0.0").AddValue("0.17880235").AddValue("0.98388501");
  target.AddInstance(dicom.release(), "nope");
}

TEST(Autogenerated, Test_17)
{
  Neuro::DicomInstancesCollection instances;
  LoadInstance_17(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 86);
  ASSERT_NEAR(nifti.pixdim[1], 2.697675, 0.00001);
  ASSERT_EQ(nifti.dim[2], 86);
  ASSERT_NEAR(nifti.pixdim[2], 2.697675, 0.00001);
  ASSERT_EQ(nifti.dim[3], 36);
  ASSERT_NEAR(nifti.pixdim[3], 3.600000, 0.00001);
  ASSERT_EQ(nifti.ndim, 3);
  ASSERT_EQ(nifti.dim[0], 3);
  ASSERT_EQ(nifti.nvox, 266256u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(57));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(57));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(57));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -2.697675, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 115.999977, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 2.654202, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], -0.643688, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], -58.807571, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 0.482350, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 3.541986, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -93.139343, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, -0.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, 0.995963, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, 0.089763, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=34;Time=140203.975;phase=1;mb=2");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadInstance_18(Neuro::DicomInstancesCollection& target)
{
  // ('dcm_qa', 'In/Orientation/cor/corasc35/MR.1.3.12.2.1107.5.2.32.35131.2014031012570555283988916')
  Orthanc::DicomMap tags;
  tags.SetValue(0x0008, 0x0060, "MR", false);
  tags.SetValue(0x0008, 0x0070, "SIEMENS", false);
  tags.SetValue(0x0020, 0x0013, "1", false);
  tags.SetValue(0x0018, 0x0080, "3000", false);
  tags.SetValue(0x0018, 0x0081, "30", false);
  tags.SetValue(0x0018, 0x1312, "ROW", false);
  tags.SetValue(0x0019, 0x1029, "0.0\\72.50000001\\144.99999999\\215.0\\287.50000001\\359.99999999\\429.99999999\\502.5\\575.00000001\\644.99999999\\717.5\\790.00000001\\859.99999999\\932.5\\1005.0\\1075.00000001\\1147.49999999\\1220.0\\1290.00000001\\1362.49999999\\1435.0\\1507.50000001\\1577.49999998\\1649.99999999\\1722.5\\1792.50000001\\1864.99999999\\1937.5\\2007.50000001\\2079.99999999\\2152.5\\2222.5\\2295.00000001\\2367.49999999\\2437.5", false);
  tags.SetValue(0x0008, 0x0016, "1.2.840.10008.5.1.4.1.1.4", false);
  tags.SetValue(0x0028, 0x0002, "1", false);
  tags.SetValue(0x0028, 0x0004, "MONOCHROME2", false);
  tags.SetValue(0x0028, 0x0010, "384", false);
  tags.SetValue(0x0028, 0x0011, "384", false);
  tags.SetValue(0x0028, 0x0100, "16", false);
  tags.SetValue(0x0028, 0x0101, "12", false);
  tags.SetValue(0x0028, 0x0102, "11", false);
  tags.SetValue(0x0028, 0x0103, "0", false);
  tags.SetValue(0x0008, 0x0032, "135701.907500", false);
  tags.SetValue(0x0018, 0x0050, "3", false);
  tags.SetValue(0x0018, 0x0088, "3.6000000917836", false);
  tags.SetValue(0x0020, 0x0032, "-624\\-37.655651722011\\623.83806791559", false);
  tags.SetValue(0x0020, 0x0037, "1\\0\\0\\0\\-0.1529858224513\\-0.988228383588", false);
  tags.SetValue(0x0028, 0x0030, "3.25\\3.25", false);
  std::unique_ptr<Neuro::InputDicomInstance> dicom(new Neuro::InputDicomInstance(tags));
  dicom->GetCSAHeader().AddTag("NumberOfImagesInMosaic", "US").AddValue("35");
  dicom->GetCSAHeader().AddTag("PhaseEncodingDirectionPositive", "IS").AddValue("1");
  dicom->GetCSAHeader().AddTag("SliceNormalVector", "FD").AddValue("0.0").AddValue("0.98822844").AddValue("-0.15298548");
  target.AddInstance(dicom.release(), "nope");
}

TEST(Autogenerated, Test_18)
{
  Neuro::DicomInstancesCollection instances;
  LoadInstance_18(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 64);
  ASSERT_NEAR(nifti.pixdim[1], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 64);
  ASSERT_NEAR(nifti.pixdim[2], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 35);
  ASSERT_NEAR(nifti.pixdim[3], 3.600000, 0.00001);
  ASSERT_EQ(nifti.ndim, 3);
  ASSERT_EQ(nifti.dim[0], 3);
  ASSERT_EQ(nifti.nvox, 143360u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(54));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(54));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(54));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 1);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -3.250000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 104.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], -0.497204, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], -3.557622, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], 148.532135, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 3.211742, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], -0.550749, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -92.380424, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, 0.650774, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, 0.759271, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=30;Time=135701.908;phase=1");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadInstance_19(Neuro::DicomInstancesCollection& target)
{
  // ('dcm_qa', 'In/Orientation/cor/corasc35/MR.1.3.12.2.1107.5.2.32.35131.2014031012570836467089021')
  Orthanc::DicomMap tags;
  tags.SetValue(0x0008, 0x0060, "MR", false);
  tags.SetValue(0x0008, 0x0070, "SIEMENS", false);
  tags.SetValue(0x0020, 0x0013, "2", false);
  tags.SetValue(0x0018, 0x0080, "3000", false);
  tags.SetValue(0x0018, 0x0081, "30", false);
  tags.SetValue(0x0018, 0x1312, "ROW", false);
  tags.SetValue(0x0019, 0x1029, "0.0\\72.49999998\\142.49999999\\215.0\\287.49999998\\357.49999998\\429.99999999\\502.49999997\\572.49999998\\644.99999999\\717.5\\789.99999998\\859.99999999\\932.5\\1004.99999998\\1074.99999998\\1147.49999999\\1220.0\\1289.99999998\\1362.49999999\\1435.0\\1504.99999998\\1577.49999998\\1649.99999999\\1720.0\\1792.49999998\\1864.99999999\\1935.0\\2007.49999998\\2079.99999999\\2152.5\\2222.49999997\\2294.99999998\\2367.49999999\\2437.5", false);
  tags.SetValue(0x0008, 0x0016, "1.2.840.10008.5.1.4.1.1.4", false);
  tags.SetValue(0x0028, 0x0002, "1", false);
  tags.SetValue(0x0028, 0x0004, "MONOCHROME2", false);
  tags.SetValue(0x0028, 0x0010, "384", false);
  tags.SetValue(0x0028, 0x0011, "384", false);
  tags.SetValue(0x0028, 0x0100, "16", false);
  tags.SetValue(0x0028, 0x0101, "12", false);
  tags.SetValue(0x0028, 0x0102, "11", false);
  tags.SetValue(0x0028, 0x0103, "0", false);
  tags.SetValue(0x0008, 0x0032, "135704.920000", false);
  tags.SetValue(0x0018, 0x0050, "3", false);
  tags.SetValue(0x0018, 0x0088, "3.6000000917836", false);
  tags.SetValue(0x0020, 0x0032, "-624\\-37.655651722011\\623.83806791559", false);
  tags.SetValue(0x0020, 0x0037, "1\\0\\0\\0\\-0.1529858224513\\-0.988228383588", false);
  tags.SetValue(0x0028, 0x0030, "3.25\\3.25", false);
  std::unique_ptr<Neuro::InputDicomInstance> dicom(new Neuro::InputDicomInstance(tags));
  dicom->GetCSAHeader().AddTag("NumberOfImagesInMosaic", "US").AddValue("35");
  dicom->GetCSAHeader().AddTag("PhaseEncodingDirectionPositive", "IS").AddValue("1");
  dicom->GetCSAHeader().AddTag("SliceNormalVector", "FD").AddValue("0.0").AddValue("0.98822844").AddValue("-0.15298548");
  target.AddInstance(dicom.release(), "nope");
}

TEST(Autogenerated, Test_19)
{
  Neuro::DicomInstancesCollection instances;
  LoadInstance_19(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 64);
  ASSERT_NEAR(nifti.pixdim[1], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 64);
  ASSERT_NEAR(nifti.pixdim[2], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 35);
  ASSERT_NEAR(nifti.pixdim[3], 3.600000, 0.00001);
  ASSERT_EQ(nifti.ndim, 3);
  ASSERT_EQ(nifti.dim[0], 3);
  ASSERT_EQ(nifti.nvox, 143360u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(54));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(54));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(54));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 1);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -3.250000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 104.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], -0.497204, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], -3.557622, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], 148.532135, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 3.211742, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], -0.550749, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -92.380424, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, 0.650774, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, 0.759271, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=30;Time=135704.920;phase=1");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadInstance_20(Neuro::DicomInstancesCollection& target)
{
  // ('dcm_qa', 'In/Orientation/cor/corasc36/MR.1.3.12.2.1107.5.2.32.35131.2014031012553857441288245')
  Orthanc::DicomMap tags;
  tags.SetValue(0x0008, 0x0060, "MR", false);
  tags.SetValue(0x0008, 0x0070, "SIEMENS", false);
  tags.SetValue(0x0020, 0x0013, "1", false);
  tags.SetValue(0x0018, 0x0080, "3000", false);
  tags.SetValue(0x0018, 0x0081, "30", false);
  tags.SetValue(0x0018, 0x1312, "ROW", false);
  tags.SetValue(0x0019, 0x1029, "0.0\\67.5\\137.50000001\\207.50000002\\277.5\\347.5\\417.50000001\\487.50000002\\557.5\\627.5\\695.00000001\\765.00000001\\834.99999999\\905.0\\975.00000001\\1045.00000001\\1114.99999999\\1185.0\\1255.0\\1322.50000001\\1392.50000002\\1462.49999999\\1532.5\\1602.50000001\\1672.50000002\\1742.49999999\\1812.5\\1882.50000001\\1950.00000001\\2020.00000002\\2090.0\\2160.0\\2230.00000001\\2300.00000002\\2370.0\\2440.0", false);
  tags.SetValue(0x0008, 0x0016, "1.2.840.10008.5.1.4.1.1.4", false);
  tags.SetValue(0x0028, 0x0002, "1", false);
  tags.SetValue(0x0028, 0x0004, "MONOCHROME2", false);
  tags.SetValue(0x0028, 0x0010, "384", false);
  tags.SetValue(0x0028, 0x0011, "384", false);
  tags.SetValue(0x0028, 0x0100, "16", false);
  tags.SetValue(0x0028, 0x0101, "12", false);
  tags.SetValue(0x0028, 0x0102, "11", false);
  tags.SetValue(0x0028, 0x0103, "0", false);
  tags.SetValue(0x0008, 0x0032, "135534.370000", false);
  tags.SetValue(0x0018, 0x0050, "3", false);
  tags.SetValue(0x0018, 0x0088, "3.6000000014324", false);
  tags.SetValue(0x0020, 0x0032, "-624\\-39.434460315761\\624.11344232813", false);
  tags.SetValue(0x0020, 0x0037, "1\\0\\0\\0\\-0.1529858224513\\-0.988228383588", false);
  tags.SetValue(0x0028, 0x0030, "3.25\\3.25", false);
  std::unique_ptr<Neuro::InputDicomInstance> dicom(new Neuro::InputDicomInstance(tags));
  dicom->GetCSAHeader().AddTag("NumberOfImagesInMosaic", "US").AddValue("36");
  dicom->GetCSAHeader().AddTag("PhaseEncodingDirectionPositive", "IS").AddValue("1");
  dicom->GetCSAHeader().AddTag("SliceNormalVector", "FD").AddValue("0.0").AddValue("0.98822836").AddValue("-0.152986");
  target.AddInstance(dicom.release(), "nope");
}

TEST(Autogenerated, Test_20)
{
  Neuro::DicomInstancesCollection instances;
  LoadInstance_20(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 64);
  ASSERT_NEAR(nifti.pixdim[1], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 64);
  ASSERT_NEAR(nifti.pixdim[2], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 36);
  ASSERT_NEAR(nifti.pixdim[3], 3.600000, 0.00001);
  ASSERT_EQ(nifti.ndim, 3);
  ASSERT_EQ(nifti.dim[0], 3);
  ASSERT_EQ(nifti.nvox, 147456u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(54));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(54));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(54));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 1);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -3.250000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 104.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], -0.497204, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], -3.557622, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], 150.310944, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 3.211742, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], -0.550749, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -92.105034, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, 0.650774, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, 0.759271, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=30;Time=135534.370;phase=1");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadInstance_21(Neuro::DicomInstancesCollection& target)
{
  // ('dcm_qa', 'In/Orientation/cor/corasc36/MR.1.3.12.2.1107.5.2.32.35131.2014031012554137824288349')
  Orthanc::DicomMap tags;
  tags.SetValue(0x0008, 0x0060, "MR", false);
  tags.SetValue(0x0008, 0x0070, "SIEMENS", false);
  tags.SetValue(0x0020, 0x0013, "2", false);
  tags.SetValue(0x0018, 0x0080, "3000", false);
  tags.SetValue(0x0018, 0x0081, "30", false);
  tags.SetValue(0x0018, 0x1312, "ROW", false);
  tags.SetValue(0x0019, 0x1029, "0.0\\70.00000001\\139.99999998\\209.99999999\\280.0\\350.00000001\\419.99999998\\489.99999999\\560.0\\627.5\\697.50000001\\767.49999999\\837.49999999\\907.5\\977.50000001\\1047.49999999\\1117.49999999\\1185.0\\1255.0\\1324.99999998\\1394.99999999\\1465.0\\1535.0\\1604.99999998\\1674.99999999\\1745.0\\1812.5\\1882.50000001\\1952.49999998\\2022.49999999\\2092.5\\2162.50000001\\2232.49999998\\2302.49999999\\2372.5\\2440.0", false);
  tags.SetValue(0x0008, 0x0016, "1.2.840.10008.5.1.4.1.1.4", false);
  tags.SetValue(0x0028, 0x0002, "1", false);
  tags.SetValue(0x0028, 0x0004, "MONOCHROME2", false);
  tags.SetValue(0x0028, 0x0010, "384", false);
  tags.SetValue(0x0028, 0x0011, "384", false);
  tags.SetValue(0x0028, 0x0100, "16", false);
  tags.SetValue(0x0028, 0x0101, "12", false);
  tags.SetValue(0x0028, 0x0102, "11", false);
  tags.SetValue(0x0028, 0x0103, "0", false);
  tags.SetValue(0x0008, 0x0032, "135537.380000", false);
  tags.SetValue(0x0018, 0x0050, "3", false);
  tags.SetValue(0x0018, 0x0088, "3.6000000014324", false);
  tags.SetValue(0x0020, 0x0032, "-624\\-39.434460315761\\624.11344232813", false);
  tags.SetValue(0x0020, 0x0037, "1\\0\\0\\0\\-0.1529858224513\\-0.988228383588", false);
  tags.SetValue(0x0028, 0x0030, "3.25\\3.25", false);
  std::unique_ptr<Neuro::InputDicomInstance> dicom(new Neuro::InputDicomInstance(tags));
  dicom->GetCSAHeader().AddTag("NumberOfImagesInMosaic", "US").AddValue("36");
  dicom->GetCSAHeader().AddTag("PhaseEncodingDirectionPositive", "IS").AddValue("1");
  dicom->GetCSAHeader().AddTag("SliceNormalVector", "FD").AddValue("0.0").AddValue("0.98822836").AddValue("-0.152986");
  target.AddInstance(dicom.release(), "nope");
}

TEST(Autogenerated, Test_21)
{
  Neuro::DicomInstancesCollection instances;
  LoadInstance_21(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 64);
  ASSERT_NEAR(nifti.pixdim[1], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 64);
  ASSERT_NEAR(nifti.pixdim[2], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 36);
  ASSERT_NEAR(nifti.pixdim[3], 3.600000, 0.00001);
  ASSERT_EQ(nifti.ndim, 3);
  ASSERT_EQ(nifti.dim[0], 3);
  ASSERT_EQ(nifti.nvox, 147456u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(54));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(54));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(54));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 1);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -3.250000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 104.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], -0.497204, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], -3.557622, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], 150.310944, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 3.211742, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], -0.550749, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -92.105034, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, 0.650774, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, 0.759271, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=30;Time=135537.380;phase=1");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadInstance_22(Neuro::DicomInstancesCollection& target)
{
  // ('dcm_qa', 'In/Orientation/cor/cordesc35/MR.1.3.12.2.1107.5.2.32.35131.2014031012573612281789140')
  Orthanc::DicomMap tags;
  tags.SetValue(0x0008, 0x0060, "MR", false);
  tags.SetValue(0x0008, 0x0070, "SIEMENS", false);
  tags.SetValue(0x0020, 0x0013, "1", false);
  tags.SetValue(0x0018, 0x0080, "3000", false);
  tags.SetValue(0x0018, 0x0081, "30", false);
  tags.SetValue(0x0018, 0x1312, "ROW", false);
  tags.SetValue(0x0019, 0x1029, "2437.5\\2367.50000002\\2295.00000001\\2222.5\\2152.50000003\\2080.00000002\\2007.50000001\\1937.5\\1865.00000002\\1792.50000001\\1722.5\\1650.00000002\\1577.50000001\\1507.50000001\\1435.00000003\\1362.50000002\\1292.50000001\\1220.0\\1147.50000002\\1077.50000001\\1005.0\\932.50000002\\860.00000002\\790.00000001\\717.50000003\\645.00000002\\575.00000001\\502.5\\430.00000002\\360.00000002\\287.50000001\\215.00000003\\145.00000002\\72.50000001\\0.0", false);
  tags.SetValue(0x0008, 0x0016, "1.2.840.10008.5.1.4.1.1.4", false);
  tags.SetValue(0x0028, 0x0002, "1", false);
  tags.SetValue(0x0028, 0x0004, "MONOCHROME2", false);
  tags.SetValue(0x0028, 0x0010, "384", false);
  tags.SetValue(0x0028, 0x0011, "384", false);
  tags.SetValue(0x0028, 0x0100, "16", false);
  tags.SetValue(0x0028, 0x0101, "12", false);
  tags.SetValue(0x0028, 0x0102, "11", false);
  tags.SetValue(0x0028, 0x0103, "0", false);
  tags.SetValue(0x0008, 0x0032, "135734.797500", false);
  tags.SetValue(0x0018, 0x0050, "3", false);
  tags.SetValue(0x0018, 0x0088, "3.6000000917836", false);
  tags.SetValue(0x0020, 0x0032, "-624\\-37.655651722011\\623.83806791559", false);
  tags.SetValue(0x0020, 0x0037, "1\\0\\0\\0\\-0.1529858224513\\-0.988228383588", false);
  tags.SetValue(0x0028, 0x0030, "3.25\\3.25", false);
  std::unique_ptr<Neuro::InputDicomInstance> dicom(new Neuro::InputDicomInstance(tags));
  dicom->GetCSAHeader().AddTag("NumberOfImagesInMosaic", "US").AddValue("35");
  dicom->GetCSAHeader().AddTag("PhaseEncodingDirectionPositive", "IS").AddValue("1");
  dicom->GetCSAHeader().AddTag("SliceNormalVector", "FD").AddValue("0.0").AddValue("0.98822844").AddValue("-0.15298548");
  target.AddInstance(dicom.release(), "nope");
}

TEST(Autogenerated, Test_22)
{
  Neuro::DicomInstancesCollection instances;
  LoadInstance_22(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 64);
  ASSERT_NEAR(nifti.pixdim[1], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 64);
  ASSERT_NEAR(nifti.pixdim[2], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 35);
  ASSERT_NEAR(nifti.pixdim[3], 3.600000, 0.00001);
  ASSERT_EQ(nifti.ndim, 3);
  ASSERT_EQ(nifti.dim[0], 3);
  ASSERT_EQ(nifti.nvox, 143360u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(54));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(54));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(54));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 2);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -3.250000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 104.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], -0.497204, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], -3.557622, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], 148.532135, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 3.211742, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], -0.550749, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -92.380424, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, 0.650774, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, 0.759271, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=30;Time=135734.797;phase=1");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadInstance_23(Neuro::DicomInstancesCollection& target)
{
  // ('dcm_qa', 'In/Orientation/cor/cordesc35/MR.1.3.12.2.1107.5.2.32.35131.2014031012573893053289241')
  Orthanc::DicomMap tags;
  tags.SetValue(0x0008, 0x0060, "MR", false);
  tags.SetValue(0x0008, 0x0070, "SIEMENS", false);
  tags.SetValue(0x0020, 0x0013, "2", false);
  tags.SetValue(0x0018, 0x0080, "3000", false);
  tags.SetValue(0x0018, 0x0081, "30", false);
  tags.SetValue(0x0018, 0x1312, "ROW", false);
  tags.SetValue(0x0019, 0x1029, "2437.5\\2367.49999999\\2295.00000001\\2222.5\\2152.5\\2080.00000002\\2007.50000001\\1937.5\\1864.99999999\\1792.50000001\\1720.0\\1649.99999999\\1577.50000001\\1505.0\\1435.0\\1362.49999999\\1290.00000001\\1220.0\\1147.49999999\\1075.00000001\\1005.0\\932.5\\860.00000002\\790.00000001\\717.5\\644.99999999\\575.00000001\\502.5\\429.99999999\\357.50000001\\287.50000001\\215.0\\142.50000002\\72.50000001\\0.0", false);
  tags.SetValue(0x0008, 0x0016, "1.2.840.10008.5.1.4.1.1.4", false);
  tags.SetValue(0x0028, 0x0002, "1", false);
  tags.SetValue(0x0028, 0x0004, "MONOCHROME2", false);
  tags.SetValue(0x0028, 0x0010, "384", false);
  tags.SetValue(0x0028, 0x0011, "384", false);
  tags.SetValue(0x0028, 0x0100, "16", false);
  tags.SetValue(0x0028, 0x0101, "12", false);
  tags.SetValue(0x0028, 0x0102, "11", false);
  tags.SetValue(0x0028, 0x0103, "0", false);
  tags.SetValue(0x0008, 0x0032, "135737.810000", false);
  tags.SetValue(0x0018, 0x0050, "3", false);
  tags.SetValue(0x0018, 0x0088, "3.6000000917836", false);
  tags.SetValue(0x0020, 0x0032, "-624\\-37.655651722011\\623.83806791559", false);
  tags.SetValue(0x0020, 0x0037, "1\\0\\0\\0\\-0.1529858224513\\-0.988228383588", false);
  tags.SetValue(0x0028, 0x0030, "3.25\\3.25", false);
  std::unique_ptr<Neuro::InputDicomInstance> dicom(new Neuro::InputDicomInstance(tags));
  dicom->GetCSAHeader().AddTag("NumberOfImagesInMosaic", "US").AddValue("35");
  dicom->GetCSAHeader().AddTag("PhaseEncodingDirectionPositive", "IS").AddValue("1");
  dicom->GetCSAHeader().AddTag("SliceNormalVector", "FD").AddValue("0.0").AddValue("0.98822844").AddValue("-0.15298548");
  target.AddInstance(dicom.release(), "nope");
}

TEST(Autogenerated, Test_23)
{
  Neuro::DicomInstancesCollection instances;
  LoadInstance_23(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 64);
  ASSERT_NEAR(nifti.pixdim[1], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 64);
  ASSERT_NEAR(nifti.pixdim[2], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 35);
  ASSERT_NEAR(nifti.pixdim[3], 3.600000, 0.00001);
  ASSERT_EQ(nifti.ndim, 3);
  ASSERT_EQ(nifti.dim[0], 3);
  ASSERT_EQ(nifti.nvox, 143360u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(54));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(54));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(54));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 2);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -3.250000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 104.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], -0.497204, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], -3.557622, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], 148.532135, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 3.211742, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], -0.550749, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -92.380424, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, 0.650774, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, 0.759271, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=30;Time=135737.810;phase=1");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadInstance_24(Neuro::DicomInstancesCollection& target)
{
  // ('dcm_qa', 'In/Orientation/cor/cordesc36/MR.1.3.12.2.1107.5.2.32.35131.2014031012560686746788470')
  Orthanc::DicomMap tags;
  tags.SetValue(0x0008, 0x0060, "MR", false);
  tags.SetValue(0x0008, 0x0070, "SIEMENS", false);
  tags.SetValue(0x0020, 0x0013, "1", false);
  tags.SetValue(0x0018, 0x0080, "3000", false);
  tags.SetValue(0x0018, 0x0081, "30", false);
  tags.SetValue(0x0018, 0x1312, "ROW", false);
  tags.SetValue(0x0019, 0x1029, "2440.0\\2370.0\\2302.49999999\\2232.50000001\\2162.50000001\\2092.5\\2022.49999999\\1952.50000001\\1882.50000001\\1812.5\\1742.49999999\\1675.00000002\\1605.00000001\\1535.0\\1465.0\\1394.99999999\\1325.00000001\\1255.0\\1185.0\\1117.49999999\\1047.50000002\\977.50000001\\907.5\\837.49999999\\767.50000002\\697.50000001\\627.5\\557.5\\489.99999999\\420.00000001\\350.00000001\\280.0\\209.99999999\\140.00000001\\70.00000001\\0.0", false);
  tags.SetValue(0x0008, 0x0016, "1.2.840.10008.5.1.4.1.1.4", false);
  tags.SetValue(0x0028, 0x0002, "1", false);
  tags.SetValue(0x0028, 0x0004, "MONOCHROME2", false);
  tags.SetValue(0x0028, 0x0010, "384", false);
  tags.SetValue(0x0028, 0x0011, "384", false);
  tags.SetValue(0x0028, 0x0100, "16", false);
  tags.SetValue(0x0028, 0x0101, "12", false);
  tags.SetValue(0x0028, 0x0102, "11", false);
  tags.SetValue(0x0028, 0x0103, "0", false);
  tags.SetValue(0x0008, 0x0032, "135604.842500", false);
  tags.SetValue(0x0018, 0x0050, "3", false);
  tags.SetValue(0x0018, 0x0088, "3.6000000014324", false);
  tags.SetValue(0x0020, 0x0032, "-624\\-39.434460315761\\624.11344232813", false);
  tags.SetValue(0x0020, 0x0037, "1\\0\\0\\0\\-0.1529858224513\\-0.988228383588", false);
  tags.SetValue(0x0028, 0x0030, "3.25\\3.25", false);
  std::unique_ptr<Neuro::InputDicomInstance> dicom(new Neuro::InputDicomInstance(tags));
  dicom->GetCSAHeader().AddTag("NumberOfImagesInMosaic", "US").AddValue("36");
  dicom->GetCSAHeader().AddTag("PhaseEncodingDirectionPositive", "IS").AddValue("1");
  dicom->GetCSAHeader().AddTag("SliceNormalVector", "FD").AddValue("0.0").AddValue("0.98822836").AddValue("-0.152986");
  target.AddInstance(dicom.release(), "nope");
}

TEST(Autogenerated, Test_24)
{
  Neuro::DicomInstancesCollection instances;
  LoadInstance_24(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 64);
  ASSERT_NEAR(nifti.pixdim[1], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 64);
  ASSERT_NEAR(nifti.pixdim[2], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 36);
  ASSERT_NEAR(nifti.pixdim[3], 3.600000, 0.00001);
  ASSERT_EQ(nifti.ndim, 3);
  ASSERT_EQ(nifti.dim[0], 3);
  ASSERT_EQ(nifti.nvox, 147456u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(54));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(54));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(54));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 2);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -3.250000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 104.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], -0.497204, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], -3.557622, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], 150.310944, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 3.211742, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], -0.550749, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -92.105034, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, 0.650774, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, 0.759271, 0.0001);
  ASSERT_TRUE(std::string(nifti.descrip) == "TE=30;Time=135604.842;phase=1" ||
              std::string(nifti.descrip) == "TE=30;Time=135604.843;phase=1");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadInstance_25(Neuro::DicomInstancesCollection& target)
{
  // ('dcm_qa', 'In/Orientation/cor/cordesc36/MR.1.3.12.2.1107.5.2.32.35131.2014031012560968177888575')
  Orthanc::DicomMap tags;
  tags.SetValue(0x0008, 0x0060, "MR", false);
  tags.SetValue(0x0008, 0x0070, "SIEMENS", false);
  tags.SetValue(0x0020, 0x0013, "2", false);
  tags.SetValue(0x0018, 0x0080, "3000", false);
  tags.SetValue(0x0018, 0x0081, "30", false);
  tags.SetValue(0x0018, 0x1312, "ROW", false);
  tags.SetValue(0x0019, 0x1029, "2440.0\\2370.0\\2299.99999999\\2232.49999998\\2162.50000001\\2092.5\\2022.49999999\\1952.49999998\\1882.49999998\\1812.5\\1742.49999999\\1672.49999999\\1604.99999998\\1535.0\\1465.0\\1394.99999999\\1324.99999998\\1255.0\\1185.0\\1114.99999999\\1044.99999998\\977.49999998\\907.5\\837.49999999\\767.49999999\\697.49999998\\627.5\\557.5\\487.49999999\\417.49999998\\350.00000001\\280.0\\209.99999999\\139.99999998\\69.99999998\\0.0", false);
  tags.SetValue(0x0008, 0x0016, "1.2.840.10008.5.1.4.1.1.4", false);
  tags.SetValue(0x0028, 0x0002, "1", false);
  tags.SetValue(0x0028, 0x0004, "MONOCHROME2", false);
  tags.SetValue(0x0028, 0x0010, "384", false);
  tags.SetValue(0x0028, 0x0011, "384", false);
  tags.SetValue(0x0028, 0x0100, "16", false);
  tags.SetValue(0x0028, 0x0101, "12", false);
  tags.SetValue(0x0028, 0x0102, "11", false);
  tags.SetValue(0x0028, 0x0103, "0", false);
  tags.SetValue(0x0008, 0x0032, "135607.855000", false);
  tags.SetValue(0x0018, 0x0050, "3", false);
  tags.SetValue(0x0018, 0x0088, "3.6000000014324", false);
  tags.SetValue(0x0020, 0x0032, "-624\\-39.434460315761\\624.11344232813", false);
  tags.SetValue(0x0020, 0x0037, "1\\0\\0\\0\\-0.1529858224513\\-0.988228383588", false);
  tags.SetValue(0x0028, 0x0030, "3.25\\3.25", false);
  std::unique_ptr<Neuro::InputDicomInstance> dicom(new Neuro::InputDicomInstance(tags));
  dicom->GetCSAHeader().AddTag("NumberOfImagesInMosaic", "US").AddValue("36");
  dicom->GetCSAHeader().AddTag("PhaseEncodingDirectionPositive", "IS").AddValue("1");
  dicom->GetCSAHeader().AddTag("SliceNormalVector", "FD").AddValue("0.0").AddValue("0.98822836").AddValue("-0.152986");
  target.AddInstance(dicom.release(), "nope");
}

TEST(Autogenerated, Test_25)
{
  Neuro::DicomInstancesCollection instances;
  LoadInstance_25(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 64);
  ASSERT_NEAR(nifti.pixdim[1], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 64);
  ASSERT_NEAR(nifti.pixdim[2], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 36);
  ASSERT_NEAR(nifti.pixdim[3], 3.600000, 0.00001);
  ASSERT_EQ(nifti.ndim, 3);
  ASSERT_EQ(nifti.dim[0], 3);
  ASSERT_EQ(nifti.nvox, 147456u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(54));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(54));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(54));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 2);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -3.250000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 104.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], -0.497204, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], -3.557622, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], 150.310944, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 3.211742, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], -0.550749, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -92.105034, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, 0.650774, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, 0.759271, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=30;Time=135607.855;phase=1");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadInstance_26(Neuro::DicomInstancesCollection& target)
{
  // ('dcm_qa', 'In/Orientation/cor/corint35/MR.1.3.12.2.1107.5.2.32.35131.2014031012580590504189357')
  Orthanc::DicomMap tags;
  tags.SetValue(0x0008, 0x0060, "MR", false);
  tags.SetValue(0x0008, 0x0070, "SIEMENS", false);
  tags.SetValue(0x0020, 0x0013, "1", false);
  tags.SetValue(0x0018, 0x0080, "3000", false);
  tags.SetValue(0x0018, 0x0081, "30", false);
  tags.SetValue(0x0018, 0x1312, "ROW", false);
  tags.SetValue(0x0019, 0x1029, "0.0\\1290.00000001\\70.00000001\\1362.50000002\\142.50000002\\1432.50000002\\215.00000003\\1505.0\\285.0\\1577.50000001\\357.50000001\\1647.50000002\\430.00000002\\1720.0\\502.5\\1792.50000001\\572.50000001\\1865.00000002\\645.00000002\\1935.00000003\\717.50000003\\2007.50000001\\787.50000001\\2080.00000002\\860.00000002\\2150.00000002\\932.50000002\\2222.5\\1002.5\\2295.00000001\\1075.00000001\\2365.00000002\\1147.50000002\\2437.5\\1217.50000003", false);
  tags.SetValue(0x0008, 0x0016, "1.2.840.10008.5.1.4.1.1.4", false);
  tags.SetValue(0x0028, 0x0002, "1", false);
  tags.SetValue(0x0028, 0x0004, "MONOCHROME2", false);
  tags.SetValue(0x0028, 0x0010, "384", false);
  tags.SetValue(0x0028, 0x0011, "384", false);
  tags.SetValue(0x0028, 0x0100, "16", false);
  tags.SetValue(0x0028, 0x0101, "12", false);
  tags.SetValue(0x0028, 0x0102, "11", false);
  tags.SetValue(0x0028, 0x0103, "0", false);
  tags.SetValue(0x0008, 0x0032, "135801.485000", false);
  tags.SetValue(0x0018, 0x0050, "3", false);
  tags.SetValue(0x0018, 0x0088, "3.6000000917836", false);
  tags.SetValue(0x0020, 0x0032, "-624\\-37.655651722011\\623.83806791559", false);
  tags.SetValue(0x0020, 0x0037, "1\\0\\0\\0\\-0.1529858224513\\-0.988228383588", false);
  tags.SetValue(0x0028, 0x0030, "3.25\\3.25", false);
  std::unique_ptr<Neuro::InputDicomInstance> dicom(new Neuro::InputDicomInstance(tags));
  dicom->GetCSAHeader().AddTag("NumberOfImagesInMosaic", "US").AddValue("35");
  dicom->GetCSAHeader().AddTag("PhaseEncodingDirectionPositive", "IS").AddValue("1");
  dicom->GetCSAHeader().AddTag("SliceNormalVector", "FD").AddValue("0.0").AddValue("0.98822844").AddValue("-0.15298548");
  target.AddInstance(dicom.release(), "nope");
}

TEST(Autogenerated, Test_26)
{
  Neuro::DicomInstancesCollection instances;
  LoadInstance_26(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 64);
  ASSERT_NEAR(nifti.pixdim[1], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 64);
  ASSERT_NEAR(nifti.pixdim[2], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 35);
  ASSERT_NEAR(nifti.pixdim[3], 3.600000, 0.00001);
  ASSERT_EQ(nifti.ndim, 3);
  ASSERT_EQ(nifti.dim[0], 3);
  ASSERT_EQ(nifti.nvox, 143360u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(54));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(54));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(54));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 3);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -3.250000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 104.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], -0.497204, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], -3.557622, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], 148.532135, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 3.211742, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], -0.550749, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -92.380424, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, 0.650774, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, 0.759271, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=30;Time=135801.485;phase=1");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadInstance_27(Neuro::DicomInstancesCollection& target)
{
  // ('dcm_qa', 'In/Orientation/cor/corint35/MR.1.3.12.2.1107.5.2.32.35131.2014031012580871608689461')
  Orthanc::DicomMap tags;
  tags.SetValue(0x0008, 0x0060, "MR", false);
  tags.SetValue(0x0008, 0x0070, "SIEMENS", false);
  tags.SetValue(0x0020, 0x0013, "2", false);
  tags.SetValue(0x0018, 0x0080, "3000", false);
  tags.SetValue(0x0018, 0x0081, "30", false);
  tags.SetValue(0x0018, 0x1312, "ROW", false);
  tags.SetValue(0x0019, 0x1029, "0.0\\1290.00000001\\70.00000001\\1362.49999999\\142.50000002\\1432.5\\215.0\\1505.0\\285.0\\1577.50000001\\357.50000001\\1647.49999999\\429.99999999\\1720.0\\500.0\\1792.50000001\\572.50000001\\1862.49999999\\644.99999999\\1935.0\\715.0\\2007.50000001\\787.50000001\\2077.50000001\\860.00000002\\2149.99999999\\929.99999999\\2222.5\\1002.5\\2292.50000001\\1075.00000001\\2364.99999999\\1147.49999999\\2437.5\\1217.5", false);
  tags.SetValue(0x0008, 0x0016, "1.2.840.10008.5.1.4.1.1.4", false);
  tags.SetValue(0x0028, 0x0002, "1", false);
  tags.SetValue(0x0028, 0x0004, "MONOCHROME2", false);
  tags.SetValue(0x0028, 0x0010, "384", false);
  tags.SetValue(0x0028, 0x0011, "384", false);
  tags.SetValue(0x0028, 0x0100, "16", false);
  tags.SetValue(0x0028, 0x0101, "12", false);
  tags.SetValue(0x0028, 0x0102, "11", false);
  tags.SetValue(0x0028, 0x0103, "0", false);
  tags.SetValue(0x0008, 0x0032, "135804.497500", false);
  tags.SetValue(0x0018, 0x0050, "3", false);
  tags.SetValue(0x0018, 0x0088, "3.6000000917836", false);
  tags.SetValue(0x0020, 0x0032, "-624\\-37.655651722011\\623.83806791559", false);
  tags.SetValue(0x0020, 0x0037, "1\\0\\0\\0\\-0.1529858224513\\-0.988228383588", false);
  tags.SetValue(0x0028, 0x0030, "3.25\\3.25", false);
  std::unique_ptr<Neuro::InputDicomInstance> dicom(new Neuro::InputDicomInstance(tags));
  dicom->GetCSAHeader().AddTag("NumberOfImagesInMosaic", "US").AddValue("35");
  dicom->GetCSAHeader().AddTag("PhaseEncodingDirectionPositive", "IS").AddValue("1");
  dicom->GetCSAHeader().AddTag("SliceNormalVector", "FD").AddValue("0.0").AddValue("0.98822844").AddValue("-0.15298548");
  target.AddInstance(dicom.release(), "nope");
}

TEST(Autogenerated, Test_27)
{
  Neuro::DicomInstancesCollection instances;
  LoadInstance_27(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 64);
  ASSERT_NEAR(nifti.pixdim[1], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 64);
  ASSERT_NEAR(nifti.pixdim[2], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 35);
  ASSERT_NEAR(nifti.pixdim[3], 3.600000, 0.00001);
  ASSERT_EQ(nifti.ndim, 3);
  ASSERT_EQ(nifti.dim[0], 3);
  ASSERT_EQ(nifti.nvox, 143360u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(54));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(54));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(54));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 3);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -3.250000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 104.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], -0.497204, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], -3.557622, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], 148.532135, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 3.211742, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], -0.550749, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -92.380424, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, 0.650774, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, 0.759271, 0.0001);
  ASSERT_TRUE(std::string(nifti.descrip) == "TE=30;Time=135804.497;phase=1" ||
              std::string(nifti.descrip) == "TE=30;Time=135804.498;phase=1");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadInstance_28(Neuro::DicomInstancesCollection& target)
{
  // ('dcm_qa', 'In/Orientation/cor/corint36/MR.1.3.12.2.1107.5.2.32.35131.2014031012563584014288694')
  Orthanc::DicomMap tags;
  tags.SetValue(0x0008, 0x0060, "MR", false);
  tags.SetValue(0x0008, 0x0070, "SIEMENS", false);
  tags.SetValue(0x0020, 0x0013, "1", false);
  tags.SetValue(0x0018, 0x0080, "3000", false);
  tags.SetValue(0x0018, 0x0081, "30", false);
  tags.SetValue(0x0018, 0x1312, "ROW", false);
  tags.SetValue(0x0019, 0x1029, "1255.0\\0.0\\1325.00000001\\70.00000001\\1394.99999999\\139.99999998\\1465.0\\209.99999999\\1535.0\\280.0\\1602.50000001\\347.5\\1672.49999999\\417.50000001\\1742.49999999\\487.49999999\\1812.5\\557.5\\1882.50000001\\627.5\\1952.49999998\\697.50000001\\2022.49999999\\767.49999999\\2092.5\\837.49999999\\2160.0\\907.5\\2230.00000001\\975.00000001\\2299.99999999\\1044.99999998\\2370.0\\1114.99999999\\2440.0\\1185.0", false);
  tags.SetValue(0x0008, 0x0016, "1.2.840.10008.5.1.4.1.1.4", false);
  tags.SetValue(0x0028, 0x0002, "1", false);
  tags.SetValue(0x0028, 0x0004, "MONOCHROME2", false);
  tags.SetValue(0x0028, 0x0010, "384", false);
  tags.SetValue(0x0028, 0x0011, "384", false);
  tags.SetValue(0x0028, 0x0100, "16", false);
  tags.SetValue(0x0028, 0x0101, "12", false);
  tags.SetValue(0x0028, 0x0102, "11", false);
  tags.SetValue(0x0028, 0x0103, "0", false);
  tags.SetValue(0x0008, 0x0032, "135633.507500", false);
  tags.SetValue(0x0018, 0x0050, "3", false);
  tags.SetValue(0x0018, 0x0088, "3.6000000014324", false);
  tags.SetValue(0x0020, 0x0032, "-624\\-39.434460315761\\624.11344232813", false);
  tags.SetValue(0x0020, 0x0037, "1\\0\\0\\0\\-0.1529858224513\\-0.988228383588", false);
  tags.SetValue(0x0028, 0x0030, "3.25\\3.25", false);
  std::unique_ptr<Neuro::InputDicomInstance> dicom(new Neuro::InputDicomInstance(tags));
  dicom->GetCSAHeader().AddTag("NumberOfImagesInMosaic", "US").AddValue("36");
  dicom->GetCSAHeader().AddTag("PhaseEncodingDirectionPositive", "IS").AddValue("1");
  dicom->GetCSAHeader().AddTag("SliceNormalVector", "FD").AddValue("0.0").AddValue("0.98822836").AddValue("-0.152986");
  target.AddInstance(dicom.release(), "nope");
}

TEST(Autogenerated, Test_28)
{
  Neuro::DicomInstancesCollection instances;
  LoadInstance_28(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 64);
  ASSERT_NEAR(nifti.pixdim[1], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 64);
  ASSERT_NEAR(nifti.pixdim[2], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 36);
  ASSERT_NEAR(nifti.pixdim[3], 3.600000, 0.00001);
  ASSERT_EQ(nifti.ndim, 3);
  ASSERT_EQ(nifti.dim[0], 3);
  ASSERT_EQ(nifti.nvox, 147456u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(54));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(54));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(54));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 5);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -3.250000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 104.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], -0.497204, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], -3.557622, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], 150.310944, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 3.211742, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], -0.550749, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -92.105034, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, 0.650774, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, 0.759271, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=30;Time=135633.508;phase=1");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadInstance_29(Neuro::DicomInstancesCollection& target)
{
  // ('dcm_qa', 'In/Orientation/cor/corint36/MR.1.3.12.2.1107.5.2.32.35131.2014031012563864907288801')
  Orthanc::DicomMap tags;
  tags.SetValue(0x0008, 0x0060, "MR", false);
  tags.SetValue(0x0008, 0x0070, "SIEMENS", false);
  tags.SetValue(0x0020, 0x0013, "2", false);
  tags.SetValue(0x0018, 0x0080, "3000", false);
  tags.SetValue(0x0018, 0x0081, "30", false);
  tags.SetValue(0x0018, 0x1312, "ROW", false);
  tags.SetValue(0x0019, 0x1029, "1254.99999998\\0.0\\1324.99999998\\69.99999998\\1394.99999999\\139.99999998\\1462.49999999\\207.49999999\\1532.49999997\\277.5\\1602.49999998\\347.49999997\\1672.49999999\\417.49999998\\1742.49999999\\487.49999999\\1812.5\\557.5\\1882.49999998\\627.49999997\\1952.49999998\\697.49999998\\2022.49999999\\767.49999999\\2090.0\\834.99999999\\2159.99999997\\905.0\\2229.99999998\\974.99999998\\2299.99999999\\1044.99999998\\2370.0\\1114.99999999\\2439.99999997\\1185.0", false);
  tags.SetValue(0x0008, 0x0016, "1.2.840.10008.5.1.4.1.1.4", false);
  tags.SetValue(0x0028, 0x0002, "1", false);
  tags.SetValue(0x0028, 0x0004, "MONOCHROME2", false);
  tags.SetValue(0x0028, 0x0010, "384", false);
  tags.SetValue(0x0028, 0x0011, "384", false);
  tags.SetValue(0x0028, 0x0100, "16", false);
  tags.SetValue(0x0028, 0x0101, "12", false);
  tags.SetValue(0x0028, 0x0102, "11", false);
  tags.SetValue(0x0028, 0x0103, "0", false);
  tags.SetValue(0x0008, 0x0032, "135636.520000", false);
  tags.SetValue(0x0018, 0x0050, "3", false);
  tags.SetValue(0x0018, 0x0088, "3.6000000014324", false);
  tags.SetValue(0x0020, 0x0032, "-624\\-39.434460315761\\624.11344232813", false);
  tags.SetValue(0x0020, 0x0037, "1\\0\\0\\0\\-0.1529858224513\\-0.988228383588", false);
  tags.SetValue(0x0028, 0x0030, "3.25\\3.25", false);
  std::unique_ptr<Neuro::InputDicomInstance> dicom(new Neuro::InputDicomInstance(tags));
  dicom->GetCSAHeader().AddTag("NumberOfImagesInMosaic", "US").AddValue("36");
  dicom->GetCSAHeader().AddTag("PhaseEncodingDirectionPositive", "IS").AddValue("1");
  dicom->GetCSAHeader().AddTag("SliceNormalVector", "FD").AddValue("0.0").AddValue("0.98822836").AddValue("-0.152986");
  target.AddInstance(dicom.release(), "nope");
}

TEST(Autogenerated, Test_29)
{
  Neuro::DicomInstancesCollection instances;
  LoadInstance_29(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 64);
  ASSERT_NEAR(nifti.pixdim[1], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 64);
  ASSERT_NEAR(nifti.pixdim[2], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 36);
  ASSERT_NEAR(nifti.pixdim[3], 3.600000, 0.00001);
  ASSERT_EQ(nifti.ndim, 3);
  ASSERT_EQ(nifti.dim[0], 3);
  ASSERT_EQ(nifti.nvox, 147456u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(54));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(54));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(54));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 5);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -3.250000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 104.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], -0.497204, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], -3.557622, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], 150.310944, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 3.211742, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], -0.550749, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -92.105034, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, 0.650774, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, 0.759271, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=30;Time=135636.520;phase=1");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadInstance_30(Neuro::DicomInstancesCollection& target)
{
  // ('dcm_qa', 'In/Orientation/sag/sagasc35/MR.1.3.12.2.1107.5.2.32.35131.2014031013000537156690252')
  Orthanc::DicomMap tags;
  tags.SetValue(0x0008, 0x0060, "MR", false);
  tags.SetValue(0x0008, 0x0070, "SIEMENS", false);
  tags.SetValue(0x0020, 0x0013, "1", false);
  tags.SetValue(0x0018, 0x0080, "3000", false);
  tags.SetValue(0x0018, 0x0081, "30", false);
  tags.SetValue(0x0018, 0x1312, "ROW", false);
  tags.SetValue(0x0019, 0x1029, "0.0\\72.50000001\\142.50000002\\215.0\\287.50000001\\357.50000001\\430.00000002\\502.5\\575.00000001\\645.00000002\\717.5\\790.00000001\\860.00000002\\932.5\\1005.0\\1075.00000001\\1147.50000002\\1220.0\\1290.00000001\\1362.50000002\\1435.0\\1505.0\\1577.50000001\\1650.00000002\\1720.0\\1792.50000001\\1865.00000002\\1935.0\\2007.50000001\\2080.00000002\\2152.5\\2222.5\\2295.00000001\\2367.50000002\\2437.5", false);
  tags.SetValue(0x0008, 0x0016, "1.2.840.10008.5.1.4.1.1.4", false);
  tags.SetValue(0x0028, 0x0002, "1", false);
  tags.SetValue(0x0028, 0x0004, "MONOCHROME2", false);
  tags.SetValue(0x0028, 0x0010, "384", false);
  tags.SetValue(0x0028, 0x0011, "384", false);
  tags.SetValue(0x0028, 0x0100, "16", false);
  tags.SetValue(0x0028, 0x0101, "12", false);
  tags.SetValue(0x0028, 0x0102, "11", false);
  tags.SetValue(0x0028, 0x0103, "0", false);
  tags.SetValue(0x0008, 0x0032, "140000.990000", false);
  tags.SetValue(0x0018, 0x0050, "3", false);
  tags.SetValue(0x0018, 0x0088, "3.6000000448788", false);
  tags.SetValue(0x0020, 0x0032, "-61.200000762939\\-660.3196144104\\598.57627105713", false);
  tags.SetValue(0x0020, 0x0037, "0\\1\\0\\0\\0\\-1", false);
  tags.SetValue(0x0028, 0x0030, "3.25\\3.25", false);
  std::unique_ptr<Neuro::InputDicomInstance> dicom(new Neuro::InputDicomInstance(tags));
  dicom->GetCSAHeader().AddTag("NumberOfImagesInMosaic", "US").AddValue("35");
  dicom->GetCSAHeader().AddTag("PhaseEncodingDirectionPositive", "IS").AddValue("1");
  dicom->GetCSAHeader().AddTag("SliceNormalVector", "FD").AddValue("1.0").AddValue("0.0").AddValue("0.0");
  target.AddInstance(dicom.release(), "nope");
}

TEST(Autogenerated, Test_30)
{
  Neuro::DicomInstancesCollection instances;
  LoadInstance_30(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 64);
  ASSERT_NEAR(nifti.pixdim[1], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 64);
  ASSERT_NEAR(nifti.pixdim[2], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 35);
  ASSERT_NEAR(nifti.pixdim[3], 3.600000, 0.00001);
  ASSERT_EQ(nifti.ndim, 3);
  ASSERT_EQ(nifti.dim[0], 3);
  ASSERT_EQ(nifti.nvox, 143360u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(54));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(54));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(54));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 1);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -3.600000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 61.200001, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], -3.250000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], 140.319641, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 3.250000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -126.173706, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.500000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, -0.500000, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, -0.500000, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=30;Time=140000.990;phase=1");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadInstance_31(Neuro::DicomInstancesCollection& target)
{
  // ('dcm_qa', 'In/Orientation/sag/sagasc35/MR.1.3.12.2.1107.5.2.32.35131.2014031013000818402490359')
  Orthanc::DicomMap tags;
  tags.SetValue(0x0008, 0x0060, "MR", false);
  tags.SetValue(0x0008, 0x0070, "SIEMENS", false);
  tags.SetValue(0x0020, 0x0013, "2", false);
  tags.SetValue(0x0018, 0x0080, "3000", false);
  tags.SetValue(0x0018, 0x0081, "30", false);
  tags.SetValue(0x0018, 0x1312, "ROW", false);
  tags.SetValue(0x0019, 0x1029, "0.0\\72.50000001\\142.49999999\\215.0\\287.50000001\\357.49999998\\429.99999999\\502.5\\572.50000001\\644.99999999\\717.5\\787.50000001\\859.99999999\\932.5\\1002.5\\1075.00000001\\1147.49999999\\1217.5\\1290.00000001\\1362.49999999\\1435.0\\1505.0\\1577.49999998\\1649.99999999\\1720.0\\1792.50000001\\1864.99999999\\1935.0\\2007.50000001\\2079.99999999\\2149.99999999\\2222.5\\2294.99999998\\2364.99999999\\2437.5", false);
  tags.SetValue(0x0008, 0x0016, "1.2.840.10008.5.1.4.1.1.4", false);
  tags.SetValue(0x0028, 0x0002, "1", false);
  tags.SetValue(0x0028, 0x0004, "MONOCHROME2", false);
  tags.SetValue(0x0028, 0x0010, "384", false);
  tags.SetValue(0x0028, 0x0011, "384", false);
  tags.SetValue(0x0028, 0x0100, "16", false);
  tags.SetValue(0x0028, 0x0101, "12", false);
  tags.SetValue(0x0028, 0x0102, "11", false);
  tags.SetValue(0x0028, 0x0103, "0", false);
  tags.SetValue(0x0008, 0x0032, "140004.002500", false);
  tags.SetValue(0x0018, 0x0050, "3", false);
  tags.SetValue(0x0018, 0x0088, "3.6000000448788", false);
  tags.SetValue(0x0020, 0x0032, "-61.200000762939\\-660.3196144104\\598.57627105713", false);
  tags.SetValue(0x0020, 0x0037, "0\\1\\0\\0\\0\\-1", false);
  tags.SetValue(0x0028, 0x0030, "3.25\\3.25", false);
  std::unique_ptr<Neuro::InputDicomInstance> dicom(new Neuro::InputDicomInstance(tags));
  dicom->GetCSAHeader().AddTag("NumberOfImagesInMosaic", "US").AddValue("35");
  dicom->GetCSAHeader().AddTag("PhaseEncodingDirectionPositive", "IS").AddValue("1");
  dicom->GetCSAHeader().AddTag("SliceNormalVector", "FD").AddValue("1.0").AddValue("0.0").AddValue("0.0");
  target.AddInstance(dicom.release(), "nope");
}

TEST(Autogenerated, Test_31)
{
  Neuro::DicomInstancesCollection instances;
  LoadInstance_31(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 64);
  ASSERT_NEAR(nifti.pixdim[1], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 64);
  ASSERT_NEAR(nifti.pixdim[2], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 35);
  ASSERT_NEAR(nifti.pixdim[3], 3.600000, 0.00001);
  ASSERT_EQ(nifti.ndim, 3);
  ASSERT_EQ(nifti.dim[0], 3);
  ASSERT_EQ(nifti.nvox, 143360u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(54));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(54));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(54));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 1);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -3.600000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 61.200001, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], -3.250000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], 140.319641, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 3.250000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -126.173706, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.500000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, -0.500000, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, -0.500000, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=30;Time=140004.003;phase=1");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadInstance_32(Neuro::DicomInstancesCollection& target)
{
  // ('dcm_qa', 'In/Orientation/sag/sagasc36/MR.1.3.12.2.1107.5.2.32.35131.2014031012583942473089577')
  Orthanc::DicomMap tags;
  tags.SetValue(0x0008, 0x0060, "MR", false);
  tags.SetValue(0x0008, 0x0070, "SIEMENS", false);
  tags.SetValue(0x0020, 0x0013, "1", false);
  tags.SetValue(0x0018, 0x0080, "3000", false);
  tags.SetValue(0x0018, 0x0081, "30", false);
  tags.SetValue(0x0018, 0x1312, "ROW", false);
  tags.SetValue(0x0019, 0x1029, "0.0\\69.99999998\\139.99999998\\209.99999999\\277.5\\347.5\\417.49999998\\487.49999999\\557.5\\627.5\\697.49999998\\767.49999999\\837.49999999\\905.0\\974.99999998\\1044.99999998\\1114.99999999\\1185.0\\1255.0\\1324.99999998\\1394.99999999\\1465.0\\1532.5\\1602.49999998\\1672.49999999\\1742.49999999\\1812.5\\1882.49999998\\1952.49999998\\2022.49999999\\2090.0\\2160.0\\2229.99999998\\2299.99999999\\2370.0\\2440.0", false);
  tags.SetValue(0x0008, 0x0016, "1.2.840.10008.5.1.4.1.1.4", false);
  tags.SetValue(0x0028, 0x0002, "1", false);
  tags.SetValue(0x0028, 0x0004, "MONOCHROME2", false);
  tags.SetValue(0x0028, 0x0010, "384", false);
  tags.SetValue(0x0028, 0x0011, "384", false);
  tags.SetValue(0x0028, 0x0100, "16", false);
  tags.SetValue(0x0028, 0x0101, "12", false);
  tags.SetValue(0x0028, 0x0102, "11", false);
  tags.SetValue(0x0028, 0x0103, "0", false);
  tags.SetValue(0x0008, 0x0032, "135835.760000", false);
  tags.SetValue(0x0018, 0x0050, "3", false);
  tags.SetValue(0x0018, 0x0088, "3.6", false);
  tags.SetValue(0x0020, 0x0032, "-63\\-660.3196144104\\598.57627105713", false);
  tags.SetValue(0x0020, 0x0037, "0\\1\\0\\0\\0\\-1", false);
  tags.SetValue(0x0028, 0x0030, "3.25\\3.25", false);
  std::unique_ptr<Neuro::InputDicomInstance> dicom(new Neuro::InputDicomInstance(tags));
  dicom->GetCSAHeader().AddTag("NumberOfImagesInMosaic", "US").AddValue("36");
  dicom->GetCSAHeader().AddTag("PhaseEncodingDirectionPositive", "IS").AddValue("1");
  dicom->GetCSAHeader().AddTag("SliceNormalVector", "FD").AddValue("1.0").AddValue("0.0").AddValue("0.0");
  target.AddInstance(dicom.release(), "nope");
}

TEST(Autogenerated, Test_32)
{
  Neuro::DicomInstancesCollection instances;
  LoadInstance_32(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 64);
  ASSERT_NEAR(nifti.pixdim[1], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 64);
  ASSERT_NEAR(nifti.pixdim[2], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 36);
  ASSERT_NEAR(nifti.pixdim[3], 3.600000, 0.00001);
  ASSERT_EQ(nifti.ndim, 3);
  ASSERT_EQ(nifti.dim[0], 3);
  ASSERT_EQ(nifti.nvox, 147456u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(54));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(54));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(54));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 1);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -3.600000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 63.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], -3.250000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], 140.319641, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 3.250000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -126.173706, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.500000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, -0.500000, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, -0.500000, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=30;Time=135835.760;phase=1");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadInstance_33(Neuro::DicomInstancesCollection& target)
{
  // ('dcm_qa', 'In/Orientation/sag/sagasc36/MR.1.3.12.2.1107.5.2.32.35131.2014031012584223181889687')
  Orthanc::DicomMap tags;
  tags.SetValue(0x0008, 0x0060, "MR", false);
  tags.SetValue(0x0008, 0x0070, "SIEMENS", false);
  tags.SetValue(0x0020, 0x0013, "2", false);
  tags.SetValue(0x0018, 0x0080, "3000", false);
  tags.SetValue(0x0018, 0x0081, "30", false);
  tags.SetValue(0x0018, 0x1312, "ROW", false);
  tags.SetValue(0x0019, 0x1029, "0.0\\70.00000001\\137.50000001\\207.50000002\\277.5\\347.5\\417.50000001\\487.50000002\\557.5\\627.5\\697.50000001\\765.00000001\\835.00000002\\905.0\\975.00000001\\1045.00000001\\1115.00000002\\1185.0\\1255.0\\1325.00000001\\1392.50000002\\1462.49999999\\1532.5\\1602.50000001\\1672.50000002\\1742.50000002\\1812.5\\1882.50000001\\1952.50000001\\2020.00000002\\2090.0\\2160.0\\2230.00000001\\2300.00000002\\2370.0\\2440.0", false);
  tags.SetValue(0x0008, 0x0016, "1.2.840.10008.5.1.4.1.1.4", false);
  tags.SetValue(0x0028, 0x0002, "1", false);
  tags.SetValue(0x0028, 0x0004, "MONOCHROME2", false);
  tags.SetValue(0x0028, 0x0010, "384", false);
  tags.SetValue(0x0028, 0x0011, "384", false);
  tags.SetValue(0x0028, 0x0100, "16", false);
  tags.SetValue(0x0028, 0x0101, "12", false);
  tags.SetValue(0x0028, 0x0102, "11", false);
  tags.SetValue(0x0028, 0x0103, "0", false);
  tags.SetValue(0x0008, 0x0032, "135838.772500", false);
  tags.SetValue(0x0018, 0x0050, "3", false);
  tags.SetValue(0x0018, 0x0088, "3.6", false);
  tags.SetValue(0x0020, 0x0032, "-63\\-660.3196144104\\598.57627105713", false);
  tags.SetValue(0x0020, 0x0037, "0\\1\\0\\0\\0\\-1", false);
  tags.SetValue(0x0028, 0x0030, "3.25\\3.25", false);
  std::unique_ptr<Neuro::InputDicomInstance> dicom(new Neuro::InputDicomInstance(tags));
  dicom->GetCSAHeader().AddTag("NumberOfImagesInMosaic", "US").AddValue("36");
  dicom->GetCSAHeader().AddTag("PhaseEncodingDirectionPositive", "IS").AddValue("1");
  dicom->GetCSAHeader().AddTag("SliceNormalVector", "FD").AddValue("1.0").AddValue("0.0").AddValue("0.0");
  target.AddInstance(dicom.release(), "nope");
}

TEST(Autogenerated, Test_33)
{
  Neuro::DicomInstancesCollection instances;
  LoadInstance_33(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 64);
  ASSERT_NEAR(nifti.pixdim[1], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 64);
  ASSERT_NEAR(nifti.pixdim[2], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 36);
  ASSERT_NEAR(nifti.pixdim[3], 3.600000, 0.00001);
  ASSERT_EQ(nifti.ndim, 3);
  ASSERT_EQ(nifti.dim[0], 3);
  ASSERT_EQ(nifti.nvox, 147456u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(54));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(54));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(54));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 1);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -3.600000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 63.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], -3.250000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], 140.319641, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 3.250000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -126.173706, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.500000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, -0.500000, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, -0.500000, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=30;Time=135838.772;phase=1");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadInstance_34(Neuro::DicomInstancesCollection& target)
{
  // ('dcm_qa', 'In/Orientation/sag/sagdesc35/MR.1.3.12.2.1107.5.2.32.35131.2014031013003590507090477')
  Orthanc::DicomMap tags;
  tags.SetValue(0x0008, 0x0060, "MR", false);
  tags.SetValue(0x0008, 0x0070, "SIEMENS", false);
  tags.SetValue(0x0020, 0x0013, "1", false);
  tags.SetValue(0x0018, 0x0080, "3000", false);
  tags.SetValue(0x0018, 0x0081, "30", false);
  tags.SetValue(0x0018, 0x1312, "ROW", false);
  tags.SetValue(0x0019, 0x1029, "2437.5\\2365.00000002\\2295.00000001\\2222.5\\2149.99999999\\2080.00000002\\2007.50000001\\1935.0\\1862.50000002\\1792.50000001\\1720.0\\1647.50000002\\1577.50000001\\1505.0\\1432.5\\1362.50000002\\1290.00000001\\1217.5\\1147.50000002\\1075.00000001\\1002.5\\932.5\\860.00000002\\787.50000001\\717.5\\645.00000002\\572.50000001\\502.5\\430.00000002\\357.50000001\\285.0\\215.0\\142.50000002\\70.00000001\\0.0", false);
  tags.SetValue(0x0008, 0x0016, "1.2.840.10008.5.1.4.1.1.4", false);
  tags.SetValue(0x0028, 0x0002, "1", false);
  tags.SetValue(0x0028, 0x0004, "MONOCHROME2", false);
  tags.SetValue(0x0028, 0x0010, "384", false);
  tags.SetValue(0x0028, 0x0011, "384", false);
  tags.SetValue(0x0028, 0x0100, "16", false);
  tags.SetValue(0x0028, 0x0101, "12", false);
  tags.SetValue(0x0028, 0x0102, "11", false);
  tags.SetValue(0x0028, 0x0103, "0", false);
  tags.SetValue(0x0008, 0x0032, "140034.397500", false);
  tags.SetValue(0x0018, 0x0050, "3", false);
  tags.SetValue(0x0018, 0x0088, "3.6000000448788", false);
  tags.SetValue(0x0020, 0x0032, "-61.200000762939\\-660.3196144104\\598.57627105713", false);
  tags.SetValue(0x0020, 0x0037, "0\\1\\0\\0\\0\\-1", false);
  tags.SetValue(0x0028, 0x0030, "3.25\\3.25", false);
  std::unique_ptr<Neuro::InputDicomInstance> dicom(new Neuro::InputDicomInstance(tags));
  dicom->GetCSAHeader().AddTag("NumberOfImagesInMosaic", "US").AddValue("35");
  dicom->GetCSAHeader().AddTag("PhaseEncodingDirectionPositive", "IS").AddValue("1");
  dicom->GetCSAHeader().AddTag("SliceNormalVector", "FD").AddValue("1.0").AddValue("0.0").AddValue("0.0");
  target.AddInstance(dicom.release(), "nope");
}

TEST(Autogenerated, Test_34)
{
  Neuro::DicomInstancesCollection instances;
  LoadInstance_34(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 64);
  ASSERT_NEAR(nifti.pixdim[1], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 64);
  ASSERT_NEAR(nifti.pixdim[2], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 35);
  ASSERT_NEAR(nifti.pixdim[3], 3.600000, 0.00001);
  ASSERT_EQ(nifti.ndim, 3);
  ASSERT_EQ(nifti.dim[0], 3);
  ASSERT_EQ(nifti.nvox, 143360u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(54));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(54));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(54));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 2);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -3.600000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 61.200001, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], -3.250000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], 140.319641, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 3.250000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -126.173706, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.500000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, -0.500000, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, -0.500000, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=30;Time=140034.397;phase=1");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadInstance_35(Neuro::DicomInstancesCollection& target)
{
  // ('dcm_qa', 'In/Orientation/sag/sagdesc35/MR.1.3.12.2.1107.5.2.32.35131.2014031013003871821190579')
  Orthanc::DicomMap tags;
  tags.SetValue(0x0008, 0x0060, "MR", false);
  tags.SetValue(0x0008, 0x0070, "SIEMENS", false);
  tags.SetValue(0x0020, 0x0013, "2", false);
  tags.SetValue(0x0018, 0x0080, "3000", false);
  tags.SetValue(0x0018, 0x0081, "30", false);
  tags.SetValue(0x0018, 0x1312, "ROW", false);
  tags.SetValue(0x0019, 0x1029, "2437.5\\2364.99999999\\2292.50000001\\2222.5\\2149.99999999\\2077.49999998\\2007.50000001\\1935.0\\1862.49999999\\1792.50000001\\1720.0\\1647.49999999\\1577.49999998\\1505.0\\1432.5\\1362.49999999\\1290.00000001\\1217.5\\1144.99999999\\1074.99999998\\1002.5\\929.99999999\\859.99999999\\787.50000001\\715.0\\644.99999999\\572.50000001\\500.0\\429.99999999\\357.49999998\\285.0\\215.0\\142.49999999\\70.00000001\\0.0", false);
  tags.SetValue(0x0008, 0x0016, "1.2.840.10008.5.1.4.1.1.4", false);
  tags.SetValue(0x0028, 0x0002, "1", false);
  tags.SetValue(0x0028, 0x0004, "MONOCHROME2", false);
  tags.SetValue(0x0028, 0x0010, "384", false);
  tags.SetValue(0x0028, 0x0011, "384", false);
  tags.SetValue(0x0028, 0x0100, "16", false);
  tags.SetValue(0x0028, 0x0101, "12", false);
  tags.SetValue(0x0028, 0x0102, "11", false);
  tags.SetValue(0x0028, 0x0103, "0", false);
  tags.SetValue(0x0008, 0x0032, "140037.410000", false);
  tags.SetValue(0x0018, 0x0050, "3", false);
  tags.SetValue(0x0018, 0x0088, "3.6000000448788", false);
  tags.SetValue(0x0020, 0x0032, "-61.200000762939\\-660.3196144104\\598.57627105713", false);
  tags.SetValue(0x0020, 0x0037, "0\\1\\0\\0\\0\\-1", false);
  tags.SetValue(0x0028, 0x0030, "3.25\\3.25", false);
  std::unique_ptr<Neuro::InputDicomInstance> dicom(new Neuro::InputDicomInstance(tags));
  dicom->GetCSAHeader().AddTag("NumberOfImagesInMosaic", "US").AddValue("35");
  dicom->GetCSAHeader().AddTag("PhaseEncodingDirectionPositive", "IS").AddValue("1");
  dicom->GetCSAHeader().AddTag("SliceNormalVector", "FD").AddValue("1.0").AddValue("0.0").AddValue("0.0");
  target.AddInstance(dicom.release(), "nope");
}

TEST(Autogenerated, Test_35)
{
  Neuro::DicomInstancesCollection instances;
  LoadInstance_35(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 64);
  ASSERT_NEAR(nifti.pixdim[1], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 64);
  ASSERT_NEAR(nifti.pixdim[2], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 35);
  ASSERT_NEAR(nifti.pixdim[3], 3.600000, 0.00001);
  ASSERT_EQ(nifti.ndim, 3);
  ASSERT_EQ(nifti.dim[0], 3);
  ASSERT_EQ(nifti.nvox, 143360u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(54));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(54));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(54));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 2);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -3.600000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 61.200001, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], -3.250000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], 140.319641, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 3.250000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -126.173706, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.500000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, -0.500000, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, -0.500000, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=30;Time=140037.410;phase=1");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadInstance_36(Neuro::DicomInstancesCollection& target)
{
  // ('dcm_qa', 'In/Orientation/sag/sagdesc36/MR.1.3.12.2.1107.5.2.32.35131.2014031012590727219489809')
  Orthanc::DicomMap tags;
  tags.SetValue(0x0008, 0x0060, "MR", false);
  tags.SetValue(0x0008, 0x0070, "SIEMENS", false);
  tags.SetValue(0x0020, 0x0013, "1", false);
  tags.SetValue(0x0018, 0x0080, "3000", false);
  tags.SetValue(0x0018, 0x0081, "30", false);
  tags.SetValue(0x0018, 0x1312, "ROW", false);
  tags.SetValue(0x0019, 0x1029, "2440.0\\2370.00000002\\2300.00000002\\2230.00000001\\2162.50000001\\2092.5\\2022.50000002\\1952.50000001\\1882.50000001\\1812.5\\1742.50000002\\1672.50000002\\1602.50000001\\1535.0\\1465.0\\1395.00000002\\1325.00000001\\1255.0\\1185.0\\1115.00000002\\1045.00000001\\975.00000001\\907.5\\837.50000002\\767.50000002\\697.50000001\\627.5\\557.50000002\\487.50000002\\417.50000001\\347.5\\280.0\\210.00000002\\140.00000001\\70.00000001\\0.0", false);
  tags.SetValue(0x0008, 0x0016, "1.2.840.10008.5.1.4.1.1.4", false);
  tags.SetValue(0x0028, 0x0002, "1", false);
  tags.SetValue(0x0028, 0x0004, "MONOCHROME2", false);
  tags.SetValue(0x0028, 0x0010, "384", false);
  tags.SetValue(0x0028, 0x0011, "384", false);
  tags.SetValue(0x0028, 0x0100, "16", false);
  tags.SetValue(0x0028, 0x0101, "12", false);
  tags.SetValue(0x0028, 0x0102, "11", false);
  tags.SetValue(0x0028, 0x0103, "0", false);
  tags.SetValue(0x0008, 0x0032, "135905.272500", false);
  tags.SetValue(0x0018, 0x0050, "3", false);
  tags.SetValue(0x0018, 0x0088, "3.6", false);
  tags.SetValue(0x0020, 0x0032, "-63\\-660.3196144104\\598.57627105713", false);
  tags.SetValue(0x0020, 0x0037, "0\\1\\0\\0\\0\\-1", false);
  tags.SetValue(0x0028, 0x0030, "3.25\\3.25", false);
  std::unique_ptr<Neuro::InputDicomInstance> dicom(new Neuro::InputDicomInstance(tags));
  dicom->GetCSAHeader().AddTag("NumberOfImagesInMosaic", "US").AddValue("36");
  dicom->GetCSAHeader().AddTag("PhaseEncodingDirectionPositive", "IS").AddValue("1");
  dicom->GetCSAHeader().AddTag("SliceNormalVector", "FD").AddValue("1.0").AddValue("0.0").AddValue("0.0");
  target.AddInstance(dicom.release(), "nope");
}

TEST(Autogenerated, Test_36)
{
  Neuro::DicomInstancesCollection instances;
  LoadInstance_36(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 64);
  ASSERT_NEAR(nifti.pixdim[1], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 64);
  ASSERT_NEAR(nifti.pixdim[2], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 36);
  ASSERT_NEAR(nifti.pixdim[3], 3.600000, 0.00001);
  ASSERT_EQ(nifti.ndim, 3);
  ASSERT_EQ(nifti.dim[0], 3);
  ASSERT_EQ(nifti.nvox, 147456u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(54));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(54));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(54));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 2);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -3.600000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 63.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], -3.250000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], 140.319641, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 3.250000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -126.173706, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.500000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, -0.500000, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, -0.500000, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=30;Time=135905.272;phase=1");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadInstance_37(Neuro::DicomInstancesCollection& target)
{
  // ('dcm_qa', 'In/Orientation/sag/sagdesc36/MR.1.3.12.2.1107.5.2.32.35131.201403101259108388189913')
  Orthanc::DicomMap tags;
  tags.SetValue(0x0008, 0x0060, "MR", false);
  tags.SetValue(0x0008, 0x0070, "SIEMENS", false);
  tags.SetValue(0x0020, 0x0013, "2", false);
  tags.SetValue(0x0018, 0x0080, "3000", false);
  tags.SetValue(0x0018, 0x0081, "30", false);
  tags.SetValue(0x0018, 0x1312, "ROW", false);
  tags.SetValue(0x0019, 0x1029, "2440.0\\2370.0\\2299.99999999\\2230.00000001\\2160.0\\2090.0\\2022.49999999\\1952.49999998\\1882.50000001\\1812.5\\1742.49999999\\1672.49999999\\1602.50000001\\1532.5\\1462.49999999\\1394.99999999\\1325.00000001\\1255.0\\1185.0\\1114.99999999\\1045.00000001\\975.00000001\\905.0\\834.99999999\\767.49999999\\697.50000001\\627.5\\557.5\\487.49999999\\417.50000001\\347.5\\277.5\\207.49999999\\139.99999998\\70.00000001\\0.0", false);
  tags.SetValue(0x0008, 0x0016, "1.2.840.10008.5.1.4.1.1.4", false);
  tags.SetValue(0x0028, 0x0002, "1", false);
  tags.SetValue(0x0028, 0x0004, "MONOCHROME2", false);
  tags.SetValue(0x0028, 0x0010, "384", false);
  tags.SetValue(0x0028, 0x0011, "384", false);
  tags.SetValue(0x0028, 0x0100, "16", false);
  tags.SetValue(0x0028, 0x0101, "12", false);
  tags.SetValue(0x0028, 0x0102, "11", false);
  tags.SetValue(0x0028, 0x0103, "0", false);
  tags.SetValue(0x0008, 0x0032, "135908.285000", false);
  tags.SetValue(0x0018, 0x0050, "3", false);
  tags.SetValue(0x0018, 0x0088, "3.6", false);
  tags.SetValue(0x0020, 0x0032, "-63\\-660.3196144104\\598.57627105713", false);
  tags.SetValue(0x0020, 0x0037, "0\\1\\0\\0\\0\\-1", false);
  tags.SetValue(0x0028, 0x0030, "3.25\\3.25", false);
  std::unique_ptr<Neuro::InputDicomInstance> dicom(new Neuro::InputDicomInstance(tags));
  dicom->GetCSAHeader().AddTag("NumberOfImagesInMosaic", "US").AddValue("36");
  dicom->GetCSAHeader().AddTag("PhaseEncodingDirectionPositive", "IS").AddValue("1");
  dicom->GetCSAHeader().AddTag("SliceNormalVector", "FD").AddValue("1.0").AddValue("0.0").AddValue("0.0");
  target.AddInstance(dicom.release(), "nope");
}

TEST(Autogenerated, Test_37)
{
  Neuro::DicomInstancesCollection instances;
  LoadInstance_37(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 64);
  ASSERT_NEAR(nifti.pixdim[1], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 64);
  ASSERT_NEAR(nifti.pixdim[2], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 36);
  ASSERT_NEAR(nifti.pixdim[3], 3.600000, 0.00001);
  ASSERT_EQ(nifti.ndim, 3);
  ASSERT_EQ(nifti.dim[0], 3);
  ASSERT_EQ(nifti.nvox, 147456u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(54));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(54));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(54));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 2);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -3.600000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 63.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], -3.250000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], 140.319641, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 3.250000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -126.173706, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.500000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, -0.500000, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, -0.500000, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=30;Time=135908.285;phase=1");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadInstance_38(Neuro::DicomInstancesCollection& target)
{
  // ('dcm_qa', 'In/Orientation/sag/sagint35/MR.1.3.12.2.1107.5.2.32.35131.2014031013010980862390698')
  Orthanc::DicomMap tags;
  tags.SetValue(0x0008, 0x0060, "MR", false);
  tags.SetValue(0x0008, 0x0070, "SIEMENS", false);
  tags.SetValue(0x0020, 0x0013, "1", false);
  tags.SetValue(0x0018, 0x0080, "3000", false);
  tags.SetValue(0x0018, 0x0081, "30", false);
  tags.SetValue(0x0018, 0x1312, "ROW", false);
  tags.SetValue(0x0019, 0x1029, "0.0\\1292.50000001\\72.50000001\\1362.49999999\\144.99999999\\1435.0\\215.0\\1507.50000001\\287.50000001\\1577.50000001\\359.99999999\\1649.99999999\\432.5\\1722.5\\502.5\\1795.00000001\\575.00000001\\1864.99999999\\647.49999999\\1937.5\\717.5\\2010.00000001\\790.00000001\\2079.99999999\\862.49999999\\2152.5\\932.5\\2225.00000001\\1005.0\\2295.00000001\\1077.50000001\\2367.49999999\\1147.49999999\\2440.0\\1220.0", false);
  tags.SetValue(0x0008, 0x0016, "1.2.840.10008.5.1.4.1.1.4", false);
  tags.SetValue(0x0028, 0x0002, "1", false);
  tags.SetValue(0x0028, 0x0004, "MONOCHROME2", false);
  tags.SetValue(0x0028, 0x0010, "384", false);
  tags.SetValue(0x0028, 0x0011, "384", false);
  tags.SetValue(0x0028, 0x0100, "16", false);
  tags.SetValue(0x0028, 0x0101, "12", false);
  tags.SetValue(0x0028, 0x0102, "11", false);
  tags.SetValue(0x0028, 0x0103, "0", false);
  tags.SetValue(0x0008, 0x0032, "140105.875000", false);
  tags.SetValue(0x0018, 0x0050, "3", false);
  tags.SetValue(0x0018, 0x0088, "3.6000000448788", false);
  tags.SetValue(0x0020, 0x0032, "-61.200000762939\\-660.3196144104\\598.57627105713", false);
  tags.SetValue(0x0020, 0x0037, "0\\1\\0\\0\\0\\-1", false);
  tags.SetValue(0x0028, 0x0030, "3.25\\3.25", false);
  std::unique_ptr<Neuro::InputDicomInstance> dicom(new Neuro::InputDicomInstance(tags));
  dicom->GetCSAHeader().AddTag("NumberOfImagesInMosaic", "US").AddValue("35");
  dicom->GetCSAHeader().AddTag("PhaseEncodingDirectionPositive", "IS").AddValue("1");
  dicom->GetCSAHeader().AddTag("SliceNormalVector", "FD").AddValue("1.0").AddValue("0.0").AddValue("0.0");
  target.AddInstance(dicom.release(), "nope");
}

TEST(Autogenerated, Test_38)
{
  Neuro::DicomInstancesCollection instances;
  LoadInstance_38(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 64);
  ASSERT_NEAR(nifti.pixdim[1], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 64);
  ASSERT_NEAR(nifti.pixdim[2], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 35);
  ASSERT_NEAR(nifti.pixdim[3], 3.600000, 0.00001);
  ASSERT_EQ(nifti.ndim, 3);
  ASSERT_EQ(nifti.dim[0], 3);
  ASSERT_EQ(nifti.nvox, 143360u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(54));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(54));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(54));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 3);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -3.600000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 61.200001, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], -3.250000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], 140.319641, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 3.250000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -126.173706, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.500000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, -0.500000, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, -0.500000, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=30;Time=140105.875;phase=1");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadInstance_39(Neuro::DicomInstancesCollection& target)
{
  // ('dcm_qa', 'In/Orientation/sag/sagint35/MR.1.3.12.2.1107.5.2.32.35131.2014031013011261838990799')
  Orthanc::DicomMap tags;
  tags.SetValue(0x0008, 0x0060, "MR", false);
  tags.SetValue(0x0008, 0x0070, "SIEMENS", false);
  tags.SetValue(0x0020, 0x0013, "2", false);
  tags.SetValue(0x0018, 0x0080, "3000", false);
  tags.SetValue(0x0018, 0x0081, "30", false);
  tags.SetValue(0x0018, 0x1312, "ROW", false);
  tags.SetValue(0x0019, 0x1029, "0.0\\1292.49999998\\72.49999998\\1362.49999999\\144.99999999\\1435.0\\215.0\\1507.49999998\\287.49999998\\1577.49999998\\359.99999999\\1649.99999999\\429.99999999\\1722.49999997\\502.5\\1792.49999998\\574.99999998\\1864.99999999\\644.99999999\\1937.5\\717.5\\2007.49999998\\789.99999998\\2079.99999999\\859.99999999\\2152.5\\932.5\\2222.49999997\\1004.99999998\\2294.99999998\\1077.49999998\\2367.49999999\\1147.49999999\\2437.5\\1220.0", false);
  tags.SetValue(0x0008, 0x0016, "1.2.840.10008.5.1.4.1.1.4", false);
  tags.SetValue(0x0028, 0x0002, "1", false);
  tags.SetValue(0x0028, 0x0004, "MONOCHROME2", false);
  tags.SetValue(0x0028, 0x0010, "384", false);
  tags.SetValue(0x0028, 0x0011, "384", false);
  tags.SetValue(0x0028, 0x0100, "16", false);
  tags.SetValue(0x0028, 0x0101, "12", false);
  tags.SetValue(0x0028, 0x0102, "11", false);
  tags.SetValue(0x0028, 0x0103, "0", false);
  tags.SetValue(0x0008, 0x0032, "140108.887500", false);
  tags.SetValue(0x0018, 0x0050, "3", false);
  tags.SetValue(0x0018, 0x0088, "3.6000000448788", false);
  tags.SetValue(0x0020, 0x0032, "-61.200000762939\\-660.3196144104\\598.57627105713", false);
  tags.SetValue(0x0020, 0x0037, "0\\1\\0\\0\\0\\-1", false);
  tags.SetValue(0x0028, 0x0030, "3.25\\3.25", false);
  std::unique_ptr<Neuro::InputDicomInstance> dicom(new Neuro::InputDicomInstance(tags));
  dicom->GetCSAHeader().AddTag("NumberOfImagesInMosaic", "US").AddValue("35");
  dicom->GetCSAHeader().AddTag("PhaseEncodingDirectionPositive", "IS").AddValue("1");
  dicom->GetCSAHeader().AddTag("SliceNormalVector", "FD").AddValue("1.0").AddValue("0.0").AddValue("0.0");
  target.AddInstance(dicom.release(), "nope");
}

TEST(Autogenerated, Test_39)
{
  Neuro::DicomInstancesCollection instances;
  LoadInstance_39(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 64);
  ASSERT_NEAR(nifti.pixdim[1], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 64);
  ASSERT_NEAR(nifti.pixdim[2], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 35);
  ASSERT_NEAR(nifti.pixdim[3], 3.600000, 0.00001);
  ASSERT_EQ(nifti.ndim, 3);
  ASSERT_EQ(nifti.dim[0], 3);
  ASSERT_EQ(nifti.nvox, 143360u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(54));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(54));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(54));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 3);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -3.600000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 61.200001, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], -3.250000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], 140.319641, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 3.250000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -126.173706, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.500000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, -0.500000, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, -0.500000, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=30;Time=140108.888;phase=1");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadInstance_40(Neuro::DicomInstancesCollection& target)
{
  // ('dcm_qa', 'In/Orientation/sag/sagint36/MR.1.3.12.2.1107.5.2.32.35131.2014031012593442716690029')
  Orthanc::DicomMap tags;
  tags.SetValue(0x0008, 0x0060, "MR", false);
  tags.SetValue(0x0008, 0x0070, "SIEMENS", false);
  tags.SetValue(0x0020, 0x0013, "1", false);
  tags.SetValue(0x0018, 0x0080, "3000", false);
  tags.SetValue(0x0018, 0x0081, "30", false);
  tags.SetValue(0x0018, 0x1312, "ROW", false);
  tags.SetValue(0x0019, 0x1029, "1255.0\\0.0\\1325.00000001\\70.00000001\\1395.00000002\\140.00000001\\1465.0\\210.00000002\\1535.0\\280.0\\1605.00000001\\350.00000001\\1675.00000002\\420.00000001\\1745.00000002\\490.00000002\\1812.5\\557.50000002\\1882.50000001\\627.5\\1952.50000001\\697.50000001\\2022.50000002\\767.50000002\\2092.5\\837.50000002\\2162.50000001\\907.5\\2232.50000001\\977.50000001\\2302.50000002\\1047.50000002\\2372.5\\1117.50000002\\2440.0\\1185.0", false);
  tags.SetValue(0x0008, 0x0016, "1.2.840.10008.5.1.4.1.1.4", false);
  tags.SetValue(0x0028, 0x0002, "1", false);
  tags.SetValue(0x0028, 0x0004, "MONOCHROME2", false);
  tags.SetValue(0x0028, 0x0010, "384", false);
  tags.SetValue(0x0028, 0x0011, "384", false);
  tags.SetValue(0x0028, 0x0100, "16", false);
  tags.SetValue(0x0028, 0x0101, "12", false);
  tags.SetValue(0x0028, 0x0102, "11", false);
  tags.SetValue(0x0028, 0x0103, "0", false);
  tags.SetValue(0x0008, 0x0032, "135931.837500", false);
  tags.SetValue(0x0018, 0x0050, "3", false);
  tags.SetValue(0x0018, 0x0088, "3.6", false);
  tags.SetValue(0x0020, 0x0032, "-63\\-660.3196144104\\598.57627105713", false);
  tags.SetValue(0x0020, 0x0037, "0\\1\\0\\0\\0\\-1", false);
  tags.SetValue(0x0028, 0x0030, "3.25\\3.25", false);
  std::unique_ptr<Neuro::InputDicomInstance> dicom(new Neuro::InputDicomInstance(tags));
  dicom->GetCSAHeader().AddTag("NumberOfImagesInMosaic", "US").AddValue("36");
  dicom->GetCSAHeader().AddTag("PhaseEncodingDirectionPositive", "IS").AddValue("1");
  dicom->GetCSAHeader().AddTag("SliceNormalVector", "FD").AddValue("1.0").AddValue("0.0").AddValue("0.0");
  target.AddInstance(dicom.release(), "nope");
}

TEST(Autogenerated, Test_40)
{
  Neuro::DicomInstancesCollection instances;
  LoadInstance_40(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 64);
  ASSERT_NEAR(nifti.pixdim[1], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 64);
  ASSERT_NEAR(nifti.pixdim[2], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 36);
  ASSERT_NEAR(nifti.pixdim[3], 3.600000, 0.00001);
  ASSERT_EQ(nifti.ndim, 3);
  ASSERT_EQ(nifti.dim[0], 3);
  ASSERT_EQ(nifti.nvox, 147456u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(54));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(54));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(54));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 5);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -3.600000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 63.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], -3.250000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], 140.319641, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 3.250000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -126.173706, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.500000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, -0.500000, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, -0.500000, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=30;Time=135931.837;phase=1");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadInstance_41(Neuro::DicomInstancesCollection& target)
{
  // ('dcm_qa', 'In/Orientation/sag/sagint36/MR.1.3.12.2.1107.5.2.32.35131.2014031012593723427590139')
  Orthanc::DicomMap tags;
  tags.SetValue(0x0008, 0x0060, "MR", false);
  tags.SetValue(0x0008, 0x0070, "SIEMENS", false);
  tags.SetValue(0x0020, 0x0013, "2", false);
  tags.SetValue(0x0018, 0x0080, "3000", false);
  tags.SetValue(0x0018, 0x0081, "30", false);
  tags.SetValue(0x0018, 0x1312, "ROW", false);
  tags.SetValue(0x0019, 0x1029, "1255.0\\0.0\\1325.00000001\\70.00000001\\1394.99999999\\139.99999998\\1465.0\\209.99999999\\1535.0\\280.0\\1605.00000001\\350.00000001\\1674.99999999\\420.00000001\\1742.49999999\\487.49999999\\1812.5\\557.5\\1882.50000001\\627.5\\1952.49999998\\697.50000001\\2022.49999999\\767.49999999\\2092.5\\837.49999999\\2162.50000001\\907.5\\2232.50000001\\977.50000001\\2299.99999999\\1047.49999999\\2370.0\\1114.99999999\\2440.0\\1185.0", false);
  tags.SetValue(0x0008, 0x0016, "1.2.840.10008.5.1.4.1.1.4", false);
  tags.SetValue(0x0028, 0x0002, "1", false);
  tags.SetValue(0x0028, 0x0004, "MONOCHROME2", false);
  tags.SetValue(0x0028, 0x0010, "384", false);
  tags.SetValue(0x0028, 0x0011, "384", false);
  tags.SetValue(0x0028, 0x0100, "16", false);
  tags.SetValue(0x0028, 0x0101, "12", false);
  tags.SetValue(0x0028, 0x0102, "11", false);
  tags.SetValue(0x0028, 0x0103, "0", false);
  tags.SetValue(0x0008, 0x0032, "135934.850000", false);
  tags.SetValue(0x0018, 0x0050, "3", false);
  tags.SetValue(0x0018, 0x0088, "3.6", false);
  tags.SetValue(0x0020, 0x0032, "-63\\-660.3196144104\\598.57627105713", false);
  tags.SetValue(0x0020, 0x0037, "0\\1\\0\\0\\0\\-1", false);
  tags.SetValue(0x0028, 0x0030, "3.25\\3.25", false);
  std::unique_ptr<Neuro::InputDicomInstance> dicom(new Neuro::InputDicomInstance(tags));
  dicom->GetCSAHeader().AddTag("NumberOfImagesInMosaic", "US").AddValue("36");
  dicom->GetCSAHeader().AddTag("PhaseEncodingDirectionPositive", "IS").AddValue("1");
  dicom->GetCSAHeader().AddTag("SliceNormalVector", "FD").AddValue("1.0").AddValue("0.0").AddValue("0.0");
  target.AddInstance(dicom.release(), "nope");
}

TEST(Autogenerated, Test_41)
{
  Neuro::DicomInstancesCollection instances;
  LoadInstance_41(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 64);
  ASSERT_NEAR(nifti.pixdim[1], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 64);
  ASSERT_NEAR(nifti.pixdim[2], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 36);
  ASSERT_NEAR(nifti.pixdim[3], 3.600000, 0.00001);
  ASSERT_EQ(nifti.ndim, 3);
  ASSERT_EQ(nifti.dim[0], 3);
  ASSERT_EQ(nifti.nvox, 147456u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(54));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(54));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(54));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 5);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -3.600000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 63.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], -3.250000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], 140.319641, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 3.250000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -126.173706, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.500000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, -0.500000, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, -0.500000, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=30;Time=135934.850;phase=1");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadInstance_42(Neuro::DicomInstancesCollection& target)
{
  // ('dcm_qa', 'In/TotalReadoutTime/PF_PAT_POS_PFOV_PEres_INTERP_test2/58PF_NOPAT_NOPOS_PERES100_ES0P59_BW2222_AP_0016/PF_PAT_POS_BW_INTERP_TEST.MR.HEAD_HARMS.0016.0001.2017.09.20.16.16.44.39766.120117828.IMA')
  Orthanc::DicomMap tags;
  tags.SetValue(0x0008, 0x0060, "MR", false);
  tags.SetValue(0x0008, 0x0070, "SIEMENS", false);
  tags.SetValue(0x0020, 0x0013, "1", false);
  tags.SetValue(0x0018, 0x0080, "6100", false);
  tags.SetValue(0x0018, 0x0081, "60", false);
  tags.SetValue(0x0018, 0x1312, "COL", false);
  tags.SetValue(0x0019, 0x1029, "3039.99999998\\0.0\\3139.99999998\\99.99999998\\3242.49999999\\202.49999998\\3342.5\\304.99999999\\3444.99999998\\405.0\\3547.49999999\\507.49999998\\3647.49999999\\607.49999998\\3750.0\\709.99999999\\3849.99999998\\810.0\\3952.49999998\\912.49999998\\4052.49999999\\1012.49999998\\4155.0\\1114.99999999\\4254.99999998\\1215.0\\4357.49999998\\1317.49999998\\4457.49999999\\1417.49999998\\4560.0\\1519.99999999\\4659.99999997\\1620.0\\4762.49999998\\1722.49999997\\4862.49999999\\1822.49999998\\4965.0\\1924.99999999\\5067.49999998\\2027.5\\5167.49999998\\2127.49999997\\5269.99999999\\2229.99999998\\5370.0\\2329.99999999\\5472.49999997\\2432.5\\5572.49999998\\2532.5\\5674.99999999\\2634.99999998\\5774.99999999\\2734.99999999\\5877.49999997\\2837.49999999\\5977.49999998\\2937.5", false);
  tags.SetValue(0x0008, 0x0016, "1.2.840.10008.5.1.4.1.1.4", false);
  tags.SetValue(0x0028, 0x0002, "1", false);
  tags.SetValue(0x0028, 0x0004, "MONOCHROME2", false);
  tags.SetValue(0x0028, 0x0010, "720", false);
  tags.SetValue(0x0028, 0x0011, "720", false);
  tags.SetValue(0x0028, 0x0100, "16", false);
  tags.SetValue(0x0028, 0x0101, "16", false);
  tags.SetValue(0x0028, 0x0102, "15", false);
  tags.SetValue(0x0028, 0x0103, "0", false);
  tags.SetValue(0x0008, 0x0032, "153509.335000", false);
  tags.SetValue(0x0018, 0x0050, "2.4000000953674", false);
  tags.SetValue(0x0018, 0x0088, "2.3999999741376", false);
  tags.SetValue(0x0020, 0x0032, "-866.17921113968\\-872.62231063843\\-64.073608398438", false);
  tags.SetValue(0x0020, 0x0037, "1\\0\\0\\0\\1\\0", false);
  tags.SetValue(0x0028, 0x0030, "2.4000000953674\\2.4000000953674", false);
  std::unique_ptr<Neuro::InputDicomInstance> dicom(new Neuro::InputDicomInstance(tags));
  dicom->GetCSAHeader().AddTag("NumberOfImagesInMosaic", "US").AddValue("60");
  dicom->GetCSAHeader().AddTag("PhaseEncodingDirectionPositive", "IS").AddValue("1");
  dicom->GetCSAHeader().AddTag("SliceNormalVector", "FD").AddValue("0.0").AddValue("0.0").AddValue("1.0");
  target.AddInstance(dicom.release(), "nope");
}

TEST(Autogenerated, Test_42)
{
  Neuro::DicomInstancesCollection instances;
  LoadInstance_42(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 90);
  ASSERT_NEAR(nifti.pixdim[1], 2.400000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 90);
  ASSERT_NEAR(nifti.pixdim[2], 2.400000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 60);
  ASSERT_NEAR(nifti.pixdim[3], 2.400000, 0.00001);
  ASSERT_EQ(nifti.ndim, 3);
  ASSERT_EQ(nifti.dim[0], 3);
  ASSERT_EQ(nifti.nvox, 486000u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(57));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(57));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(57));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 5);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -2.400000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 110.179169, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 2.400000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], -96.977722, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 2.400000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -64.073608, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, 1.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, 0.000000, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=60;Time=153509.335;phase=1");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadInstance_43(Neuro::DicomInstancesCollection& target)
{
  // ('dcm_qa', 'In/TotalReadoutTime/PF_PAT_POS_PFOV_PEres_INTERP_test2/58PF_NOPAT_NOPOS_PERES100_ES0P59_BW2222_PA_0017/PF_PAT_POS_BW_INTERP_TEST.MR.HEAD_HARMS.0017.0001.2017.09.20.16.16.44.39766.120121926.IMA')
  Orthanc::DicomMap tags;
  tags.SetValue(0x0008, 0x0060, "MR", false);
  tags.SetValue(0x0008, 0x0070, "SIEMENS", false);
  tags.SetValue(0x0020, 0x0013, "1", false);
  tags.SetValue(0x0018, 0x0080, "6100", false);
  tags.SetValue(0x0018, 0x0081, "60", false);
  tags.SetValue(0x0018, 0x1312, "COL", false);
  tags.SetValue(0x0019, 0x1029, "3040.00000001\\0.0\\3139.99999998\\100.00000001\\3242.49999999\\202.49999998\\3342.5\\304.99999999\\3445.00000001\\405.0\\3547.49999999\\507.50000001\\3647.49999999\\607.49999998\\3750.0\\709.99999999\\3850.00000001\\810.0\\3952.49999998\\912.50000001\\4052.49999999\\1012.49999998\\4155.0\\1114.99999999\\4255.0\\1215.0\\4357.49999998\\1317.5\\4457.49999999\\1417.49999998\\4560.0\\1519.99999999\\4660.0\\1620.0\\4762.49999998\\1722.5\\4862.49999999\\1822.49999998\\4965.0\\1924.99999999\\5067.5\\2027.5\\5167.49999998\\2127.5\\5269.99999999\\2229.99999998\\5370.0\\2329.99999999\\5472.5\\2432.5\\5572.49999998\\2532.5\\5674.99999999\\2634.99999998\\5774.99999999\\2734.99999999\\5877.5\\2837.49999999\\5977.50000001\\2937.5", false);
  tags.SetValue(0x0008, 0x0016, "1.2.840.10008.5.1.4.1.1.4", false);
  tags.SetValue(0x0028, 0x0002, "1", false);
  tags.SetValue(0x0028, 0x0004, "MONOCHROME2", false);
  tags.SetValue(0x0028, 0x0010, "720", false);
  tags.SetValue(0x0028, 0x0011, "720", false);
  tags.SetValue(0x0028, 0x0100, "16", false);
  tags.SetValue(0x0028, 0x0101, "16", false);
  tags.SetValue(0x0028, 0x0102, "15", false);
  tags.SetValue(0x0028, 0x0103, "0", false);
  tags.SetValue(0x0008, 0x0032, "153612.390000", false);
  tags.SetValue(0x0018, 0x0050, "2.4000000953674", false);
  tags.SetValue(0x0018, 0x0088, "2.3999999741376", false);
  tags.SetValue(0x0020, 0x0032, "-866.17921113968\\-872.62231063843\\-64.073608398438", false);
  tags.SetValue(0x0020, 0x0037, "1\\0\\0\\0\\1\\0", false);
  tags.SetValue(0x0028, 0x0030, "2.4000000953674\\2.4000000953674", false);
  std::unique_ptr<Neuro::InputDicomInstance> dicom(new Neuro::InputDicomInstance(tags));
  dicom->GetCSAHeader().AddTag("NumberOfImagesInMosaic", "US").AddValue("60");
  dicom->GetCSAHeader().AddTag("PhaseEncodingDirectionPositive", "IS").AddValue("0");
  dicom->GetCSAHeader().AddTag("SliceNormalVector", "FD").AddValue("0.0").AddValue("0.0").AddValue("1.0");
  target.AddInstance(dicom.release(), "nope");
}

TEST(Autogenerated, Test_43)
{
  Neuro::DicomInstancesCollection instances;
  LoadInstance_43(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 90);
  ASSERT_NEAR(nifti.pixdim[1], 2.400000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 90);
  ASSERT_NEAR(nifti.pixdim[2], 2.400000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 60);
  ASSERT_NEAR(nifti.pixdim[3], 2.400000, 0.00001);
  ASSERT_EQ(nifti.ndim, 3);
  ASSERT_EQ(nifti.dim[0], 3);
  ASSERT_EQ(nifti.nvox, 486000u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(57));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(57));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(57));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 5);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -2.400000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 110.179169, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 2.400000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], -96.977722, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 2.400000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -64.073608, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, 1.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, 0.000000, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=60;Time=153612.390;phase=0");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadInstance_44(Neuro::DicomInstancesCollection& target)
{
  // ('dcm_qa', 'In/TotalReadoutTime/PF_PAT_POS_PFOV_PEres_INTERP_test2/78PF_NOPAT_NOPOS_PERES100_ES0P59_BW2222_AP_0018/PF_PAT_POS_BW_INTERP_TEST.MR.HEAD_HARMS.0018.0001.2017.09.20.16.16.44.39766.120126024.IMA')
  Orthanc::DicomMap tags;
  tags.SetValue(0x0008, 0x0060, "MR", false);
  tags.SetValue(0x0008, 0x0070, "SIEMENS", false);
  tags.SetValue(0x0020, 0x0013, "1", false);
  tags.SetValue(0x0018, 0x0080, "6100", false);
  tags.SetValue(0x0018, 0x0081, "60", false);
  tags.SetValue(0x0018, 0x1312, "COL", false);
  tags.SetValue(0x0019, 0x1029, "3040.00000001\\0.0\\3142.50000002\\102.50000001\\3242.49999999\\202.50000001\\3345.0\\304.99999999\\3445.00000001\\405.0\\3547.50000002\\507.50000001\\3649.99999999\\610.00000002\\3750.0\\709.99999999\\3852.50000001\\812.5\\3952.50000001\\912.50000001\\4054.99999999\\1015.00000001\\4155.0\\1114.99999999\\4257.50000001\\1217.5\\4357.50000001\\1317.5\\4459.99999999\\1420.00000001\\4560.0\\1520.00000002\\4662.50000001\\1622.5\\4762.50000001\\1722.5\\4864.99999999\\1825.00000001\\4965.0\\1925.00000002\\5067.5\\2027.5\\5167.50000001\\2130.0\\5270.00000002\\2230.00000001\\5372.5\\2332.50000002\\5472.5\\2432.5\\5575.00000001\\2535.0\\5675.00000002\\2635.00000001\\5777.5\\2737.50000002\\5877.5\\2837.49999999\\5980.00000001\\2940.0", false);
  tags.SetValue(0x0008, 0x0016, "1.2.840.10008.5.1.4.1.1.4", false);
  tags.SetValue(0x0028, 0x0002, "1", false);
  tags.SetValue(0x0028, 0x0004, "MONOCHROME2", false);
  tags.SetValue(0x0028, 0x0010, "720", false);
  tags.SetValue(0x0028, 0x0011, "720", false);
  tags.SetValue(0x0028, 0x0100, "16", false);
  tags.SetValue(0x0028, 0x0101, "16", false);
  tags.SetValue(0x0028, 0x0102, "15", false);
  tags.SetValue(0x0028, 0x0103, "0", false);
  tags.SetValue(0x0008, 0x0032, "153702.160000", false);
  tags.SetValue(0x0018, 0x0050, "2.4000000953674", false);
  tags.SetValue(0x0018, 0x0088, "2.3999999741376", false);
  tags.SetValue(0x0020, 0x0032, "-866.17921113968\\-872.62231063843\\-64.073608398438", false);
  tags.SetValue(0x0020, 0x0037, "1\\0\\0\\0\\1\\0", false);
  tags.SetValue(0x0028, 0x0030, "2.4000000953674\\2.4000000953674", false);
  std::unique_ptr<Neuro::InputDicomInstance> dicom(new Neuro::InputDicomInstance(tags));
  dicom->GetCSAHeader().AddTag("NumberOfImagesInMosaic", "US").AddValue("60");
  dicom->GetCSAHeader().AddTag("PhaseEncodingDirectionPositive", "IS").AddValue("1");
  dicom->GetCSAHeader().AddTag("SliceNormalVector", "FD").AddValue("0.0").AddValue("0.0").AddValue("1.0");
  target.AddInstance(dicom.release(), "nope");
}

TEST(Autogenerated, Test_44)
{
  Neuro::DicomInstancesCollection instances;
  LoadInstance_44(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 90);
  ASSERT_NEAR(nifti.pixdim[1], 2.400000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 90);
  ASSERT_NEAR(nifti.pixdim[2], 2.400000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 60);
  ASSERT_NEAR(nifti.pixdim[3], 2.400000, 0.00001);
  ASSERT_EQ(nifti.ndim, 3);
  ASSERT_EQ(nifti.dim[0], 3);
  ASSERT_EQ(nifti.nvox, 486000u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(57));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(57));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(57));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 5);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -2.400000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 110.179169, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 2.400000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], -96.977722, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 2.400000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -64.073608, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, 1.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, 0.000000, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=60;Time=153702.160;phase=1");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadInstance_45(Neuro::DicomInstancesCollection& target)
{
  // ('dcm_qa', 'In/TotalReadoutTime/PF_PAT_POS_PFOV_PEres_INTERP_test2/78PF_NOPAT_NOPOS_PERES100_ES0P59_BW2222_PA_0019/PF_PAT_POS_BW_INTERP_TEST.MR.HEAD_HARMS.0019.0001.2017.09.20.16.16.44.39766.120130122.IMA')
  Orthanc::DicomMap tags;
  tags.SetValue(0x0008, 0x0060, "MR", false);
  tags.SetValue(0x0008, 0x0070, "SIEMENS", false);
  tags.SetValue(0x0020, 0x0013, "1", false);
  tags.SetValue(0x0018, 0x0080, "6100", false);
  tags.SetValue(0x0018, 0x0081, "60", false);
  tags.SetValue(0x0018, 0x1312, "COL", false);
  tags.SetValue(0x0019, 0x1029, "3040.00000001\\0.0\\3140.00000001\\100.00000001\\3242.50000002\\202.50000001\\3342.5\\302.50000002\\3445.00000001\\405.0\\3545.00000001\\505.0\\3647.50000002\\607.50000001\\3747.5\\707.50000002\\3850.00000001\\810.0\\3950.00000001\\910.0\\4052.50000002\\1012.50000001\\4152.5\\1112.50000002\\4255.0\\1215.0\\4355.00000001\\1315.0\\4457.50000002\\1417.50000001\\4557.50000002\\1520.00000002\\4660.0\\1620.00000002\\4762.50000001\\1722.5\\4862.50000002\\1822.50000001\\4965.0\\1925.00000002\\5065.0\\2025.00000002\\5167.50000001\\2127.5\\5267.50000002\\2227.50000001\\5370.00000002\\2330.00000002\\5470.0\\2430.00000002\\5572.50000001\\2532.5\\5672.50000002\\2632.50000001\\5775.00000002\\2735.00000002\\5875.0\\2835.00000002\\5977.50000001\\2937.5", false);
  tags.SetValue(0x0008, 0x0016, "1.2.840.10008.5.1.4.1.1.4", false);
  tags.SetValue(0x0028, 0x0002, "1", false);
  tags.SetValue(0x0028, 0x0004, "MONOCHROME2", false);
  tags.SetValue(0x0028, 0x0010, "720", false);
  tags.SetValue(0x0028, 0x0011, "720", false);
  tags.SetValue(0x0028, 0x0100, "16", false);
  tags.SetValue(0x0028, 0x0101, "16", false);
  tags.SetValue(0x0028, 0x0102, "15", false);
  tags.SetValue(0x0028, 0x0103, "0", false);
  tags.SetValue(0x0008, 0x0032, "153738.497500", false);
  tags.SetValue(0x0018, 0x0050, "2.4000000953674", false);
  tags.SetValue(0x0018, 0x0088, "2.3999999741376", false);
  tags.SetValue(0x0020, 0x0032, "-866.17921113968\\-872.62231063843\\-64.073608398438", false);
  tags.SetValue(0x0020, 0x0037, "1\\0\\0\\0\\1\\0", false);
  tags.SetValue(0x0028, 0x0030, "2.4000000953674\\2.4000000953674", false);
  std::unique_ptr<Neuro::InputDicomInstance> dicom(new Neuro::InputDicomInstance(tags));
  dicom->GetCSAHeader().AddTag("NumberOfImagesInMosaic", "US").AddValue("60");
  dicom->GetCSAHeader().AddTag("PhaseEncodingDirectionPositive", "IS").AddValue("0");
  dicom->GetCSAHeader().AddTag("SliceNormalVector", "FD").AddValue("0.0").AddValue("0.0").AddValue("1.0");
  target.AddInstance(dicom.release(), "nope");
}

TEST(Autogenerated, Test_45)
{
  Neuro::DicomInstancesCollection instances;
  LoadInstance_45(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 90);
  ASSERT_NEAR(nifti.pixdim[1], 2.400000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 90);
  ASSERT_NEAR(nifti.pixdim[2], 2.400000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 60);
  ASSERT_NEAR(nifti.pixdim[3], 2.400000, 0.00001);
  ASSERT_EQ(nifti.ndim, 3);
  ASSERT_EQ(nifti.dim[0], 3);
  ASSERT_EQ(nifti.nvox, 486000u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(57));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(57));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(57));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 5);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -2.400000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 110.179169, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 2.400000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], -96.977722, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 2.400000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -64.073608, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, 1.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, 0.000000, 0.0001);
  ASSERT_TRUE(std::string(nifti.descrip) == "TE=60;Time=153738.497;phase=0" ||
              std::string(nifti.descrip) == "TE=60;Time=153738.498;phase=0");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadInstance_46(Neuro::DicomInstancesCollection& target)
{
  // ('dcm_qa', 'In/TotalReadoutTime/PF_PAT_POS_PFOV_PEres_INTERP_test2/NOPF_NOPAT_100POS_PERES100_ES0P59_BW2222_AP_0014/PF_PAT_POS_BW_INTERP_TEST.MR.HEAD_HARMS.0014.0001.2017.09.20.16.16.44.39766.120109632.IMA')
  Orthanc::DicomMap tags;
  tags.SetValue(0x0008, 0x0060, "MR", false);
  tags.SetValue(0x0008, 0x0070, "SIEMENS", false);
  tags.SetValue(0x0020, 0x0013, "1", false);
  tags.SetValue(0x0018, 0x0080, "10730", false);
  tags.SetValue(0x0018, 0x0081, "111.2", false);
  tags.SetValue(0x0018, 0x1312, "COL", false);
  tags.SetValue(0x0019, 0x1029, "5355.00000001\\0.0\\5532.5\\177.49999999\\5709.99999999\\355.00000001\\5889.99999998\\535.0\\6067.5\\712.49999999\\6245.0\\892.49999999\\6424.99999999\\1070.00000001\\6602.50000001\\1247.5\\6782.5\\1427.49999999\\6959.99999999\\1605.00000001\\7137.50000001\\1785.0\\7317.5\\1962.49999999\\7495.0\\2139.99999998\\7674.99999999\\2320.00000001\\7852.50000001\\2497.5\\8030.0\\2677.49999999\\8209.99999999\\2855.00000001\\8387.50000001\\3032.5\\8567.5\\3212.49999999\\8745.0\\3389.99999998\\8922.49999999\\3570.00000001\\9102.50000001\\3747.5\\9280.0\\3924.99999999\\9459.99999999\\4105.00000001\\9637.50000001\\4282.5\\9815.0\\4462.49999999\\9995.0\\4639.99999998\\10172.49999999\\4817.5\\10352.50000001\\4997.5\\10530.0\\5174.99999999", false);
  tags.SetValue(0x0008, 0x0016, "1.2.840.10008.5.1.4.1.1.4", false);
  tags.SetValue(0x0028, 0x0002, "1", false);
  tags.SetValue(0x0028, 0x0004, "MONOCHROME2", false);
  tags.SetValue(0x0028, 0x0010, "720", false);
  tags.SetValue(0x0028, 0x0011, "720", false);
  tags.SetValue(0x0028, 0x0100, "16", false);
  tags.SetValue(0x0028, 0x0101, "16", false);
  tags.SetValue(0x0028, 0x0102, "15", false);
  tags.SetValue(0x0028, 0x0103, "0", false);
  tags.SetValue(0x0008, 0x0032, "153309.137500", false);
  tags.SetValue(0x0018, 0x0050, "2.4000000953674", false);
  tags.SetValue(0x0018, 0x0088, "2.3999999741376", false);
  tags.SetValue(0x0020, 0x0032, "-866.17921113968\\-872.62231063843\\-64.073608398438", false);
  tags.SetValue(0x0020, 0x0037, "1\\0\\0\\0\\1\\0", false);
  tags.SetValue(0x0028, 0x0030, "2.4000000953674\\2.4000000953674", false);
  std::unique_ptr<Neuro::InputDicomInstance> dicom(new Neuro::InputDicomInstance(tags));
  dicom->GetCSAHeader().AddTag("NumberOfImagesInMosaic", "US").AddValue("60");
  dicom->GetCSAHeader().AddTag("PhaseEncodingDirectionPositive", "IS").AddValue("1");
  dicom->GetCSAHeader().AddTag("SliceNormalVector", "FD").AddValue("0.0").AddValue("0.0").AddValue("1.0");
  target.AddInstance(dicom.release(), "nope");
}

TEST(Autogenerated, Test_46)
{
  Neuro::DicomInstancesCollection instances;
  LoadInstance_46(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 90);
  ASSERT_NEAR(nifti.pixdim[1], 2.400000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 90);
  ASSERT_NEAR(nifti.pixdim[2], 2.400000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 60);
  ASSERT_NEAR(nifti.pixdim[3], 2.400000, 0.00001);
  ASSERT_EQ(nifti.ndim, 3);
  ASSERT_EQ(nifti.dim[0], 3);
  ASSERT_EQ(nifti.nvox, 486000u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(57));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(57));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(57));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 5);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -2.400000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 110.179169, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 2.400000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], -96.977722, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 2.400000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -64.073608, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, 1.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, 0.000000, 0.0001);
  ASSERT_TRUE(std::string(nifti.descrip) == "TE=1.1e+02;Time=153309.138;phase=1" ||
              std::string(nifti.descrip) == "TE=1.1e+002;Time=153309.138;phase=1");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadInstance_47(Neuro::DicomInstancesCollection& target)
{
  // ('dcm_qa', 'In/TotalReadoutTime/PF_PAT_POS_PFOV_PEres_INTERP_test2/NOPF_NOPAT_100POS_PERES100_ES0P59_BW2222_PA_0015/PF_PAT_POS_BW_INTERP_TEST.MR.HEAD_HARMS.0015.0001.2017.09.20.16.16.44.39766.120113730.IMA')
  Orthanc::DicomMap tags;
  tags.SetValue(0x0008, 0x0060, "MR", false);
  tags.SetValue(0x0008, 0x0070, "SIEMENS", false);
  tags.SetValue(0x0020, 0x0013, "1", false);
  tags.SetValue(0x0018, 0x0080, "10730", false);
  tags.SetValue(0x0018, 0x0081, "111.2", false);
  tags.SetValue(0x0018, 0x1312, "COL", false);
  tags.SetValue(0x0019, 0x1029, "5354.99999998\\0.0\\5535.0\\179.99999999\\5712.49999999\\357.49999998\\5889.99999998\\537.50000001\\6070.00000001\\715.0\\6247.5\\892.49999999\\6427.49999999\\1072.49999998\\6604.99999998\\1250.0\\6782.5\\1429.99999999\\6962.49999999\\1607.49999998\\7139.99999998\\1785.0\\7320.00000001\\1965.0\\7497.5\\2142.49999999\\7674.99999999\\2322.49999998\\7854.99999998\\2500.0\\8032.5\\2677.49999999\\8212.49999999\\2857.49999998\\8389.99999998\\3035.0\\8567.5\\3215.0\\8747.5\\3392.49999999\\8924.99999999\\3570.00000001\\9104.99999998\\3750.0\\9282.5\\3927.49999999\\9459.99999999\\4107.49999998\\9639.99999998\\4285.0\\9817.5\\4462.49999999\\9997.5\\4642.49999999\\10174.99999999\\4820.00000001\\10352.50000001\\4997.5\\10532.5\\5177.49999999", false);
  tags.SetValue(0x0008, 0x0016, "1.2.840.10008.5.1.4.1.1.4", false);
  tags.SetValue(0x0028, 0x0002, "1", false);
  tags.SetValue(0x0028, 0x0004, "MONOCHROME2", false);
  tags.SetValue(0x0028, 0x0010, "720", false);
  tags.SetValue(0x0028, 0x0011, "720", false);
  tags.SetValue(0x0028, 0x0100, "16", false);
  tags.SetValue(0x0028, 0x0101, "16", false);
  tags.SetValue(0x0028, 0x0102, "15", false);
  tags.SetValue(0x0028, 0x0103, "0", false);
  tags.SetValue(0x0008, 0x0032, "153352.892500", false);
  tags.SetValue(0x0018, 0x0050, "2.4000000953674", false);
  tags.SetValue(0x0018, 0x0088, "2.3999999741376", false);
  tags.SetValue(0x0020, 0x0032, "-866.17921113968\\-872.62231063843\\-64.073608398438", false);
  tags.SetValue(0x0020, 0x0037, "1\\0\\0\\0\\1\\0", false);
  tags.SetValue(0x0028, 0x0030, "2.4000000953674\\2.4000000953674", false);
  std::unique_ptr<Neuro::InputDicomInstance> dicom(new Neuro::InputDicomInstance(tags));
  dicom->GetCSAHeader().AddTag("NumberOfImagesInMosaic", "US").AddValue("60");
  dicom->GetCSAHeader().AddTag("PhaseEncodingDirectionPositive", "IS").AddValue("0");
  dicom->GetCSAHeader().AddTag("SliceNormalVector", "FD").AddValue("0.0").AddValue("0.0").AddValue("1.0");
  target.AddInstance(dicom.release(), "nope");
}

TEST(Autogenerated, Test_47)
{
  Neuro::DicomInstancesCollection instances;
  LoadInstance_47(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 90);
  ASSERT_NEAR(nifti.pixdim[1], 2.400000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 90);
  ASSERT_NEAR(nifti.pixdim[2], 2.400000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 60);
  ASSERT_NEAR(nifti.pixdim[3], 2.400000, 0.00001);
  ASSERT_EQ(nifti.ndim, 3);
  ASSERT_EQ(nifti.dim[0], 3);
  ASSERT_EQ(nifti.nvox, 486000u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(57));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(57));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(57));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 5);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -2.400000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 110.179169, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 2.400000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], -96.977722, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 2.400000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -64.073608, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, 1.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, 0.000000, 0.0001);
  ASSERT_TRUE(std::string(nifti.descrip) == "TE=1.1e+02;Time=153352.892;phase=0" ||
              std::string(nifti.descrip) == "TE=1.1e+002;Time=153352.892;phase=0");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadInstance_48(Neuro::DicomInstancesCollection& target)
{
  // ('dcm_qa', 'In/TotalReadoutTime/PF_PAT_POS_PFOV_PEres_INTERP_test2/NOPF_NOPAT_50POS_PERES100_ES0P59_BW2222_AP_0012/PF_PAT_POS_BW_INTERP_TEST.MR.HEAD_HARMS.0012.0001.2017.09.20.16.16.44.39766.120101436.IMA')
  Orthanc::DicomMap tags;
  tags.SetValue(0x0008, 0x0060, "MR", false);
  tags.SetValue(0x0008, 0x0070, "SIEMENS", false);
  tags.SetValue(0x0020, 0x0013, "1", false);
  tags.SetValue(0x0018, 0x0080, "8320", false);
  tags.SetValue(0x0018, 0x0081, "84", false);
  tags.SetValue(0x0018, 0x1312, "COL", false);
  tags.SetValue(0x0019, 0x1029, "4149.99999999\\0.0\\4287.50000001\\137.49999998\\4424.99999999\\274.99999999\\4565.0\\415.00000001\\4702.49999998\\552.49999999\\4840.0\\690.0\\4979.99999998\\829.99999999\\5117.49999999\\967.5\\5255.0\\1104.99999998\\5394.99999999\\1245.0\\5532.5\\1382.50000001\\5669.99999998\\1519.99999999\\5810.0\\1660.0\\5947.49999998\\1797.49999999\\6084.99999999\\1935.0\\6225.00000001\\2074.99999998\\6362.49999999\\2212.49999999\\6500.0\\2350.00000001\\6639.99999998\\2489.99999999\\6777.5\\2627.5\\6915.00000001\\2764.99999998\\7054.99999999\\2905.0\\7192.5\\3042.49999998\\7329.99999999\\3179.99999999\\7470.0\\3320.00000001\\7607.49999998\\3457.49999999\\7745.0\\3595.0\\7884.99999998\\3734.99999999\\8022.49999999\\3872.5\\8160.0\\4009.99999998", false);
  tags.SetValue(0x0008, 0x0016, "1.2.840.10008.5.1.4.1.1.4", false);
  tags.SetValue(0x0028, 0x0002, "1", false);
  tags.SetValue(0x0028, 0x0004, "MONOCHROME2", false);
  tags.SetValue(0x0028, 0x0010, "720", false);
  tags.SetValue(0x0028, 0x0011, "720", false);
  tags.SetValue(0x0028, 0x0100, "16", false);
  tags.SetValue(0x0028, 0x0101, "16", false);
  tags.SetValue(0x0028, 0x0102, "15", false);
  tags.SetValue(0x0028, 0x0103, "0", false);
  tags.SetValue(0x0008, 0x0032, "153031.312500", false);
  tags.SetValue(0x0018, 0x0050, "2.4000000953674", false);
  tags.SetValue(0x0018, 0x0088, "2.3999999741376", false);
  tags.SetValue(0x0020, 0x0032, "-866.17921113968\\-872.62231063843\\-64.073608398438", false);
  tags.SetValue(0x0020, 0x0037, "1\\0\\0\\0\\1\\0", false);
  tags.SetValue(0x0028, 0x0030, "2.4000000953674\\2.4000000953674", false);
  std::unique_ptr<Neuro::InputDicomInstance> dicom(new Neuro::InputDicomInstance(tags));
  dicom->GetCSAHeader().AddTag("NumberOfImagesInMosaic", "US").AddValue("60");
  dicom->GetCSAHeader().AddTag("PhaseEncodingDirectionPositive", "IS").AddValue("1");
  dicom->GetCSAHeader().AddTag("SliceNormalVector", "FD").AddValue("0.0").AddValue("0.0").AddValue("1.0");
  target.AddInstance(dicom.release(), "nope");
}

TEST(Autogenerated, Test_48)
{
  Neuro::DicomInstancesCollection instances;
  LoadInstance_48(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 90);
  ASSERT_NEAR(nifti.pixdim[1], 2.400000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 90);
  ASSERT_NEAR(nifti.pixdim[2], 2.400000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 60);
  ASSERT_NEAR(nifti.pixdim[3], 2.400000, 0.00001);
  ASSERT_EQ(nifti.ndim, 3);
  ASSERT_EQ(nifti.dim[0], 3);
  ASSERT_EQ(nifti.nvox, 486000u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(57));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(57));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(57));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 5);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -2.400000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 110.179169, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 2.400000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], -96.977722, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 2.400000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -64.073608, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, 1.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, 0.000000, 0.0001);
  ASSERT_TRUE(std::string(nifti.descrip) == "TE=84;Time=153031.312;phase=1" ||
              std::string(nifti.descrip) == "TE=84;Time=153031.313;phase=1");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadInstance_49(Neuro::DicomInstancesCollection& target)
{
  // ('dcm_qa', 'In/TotalReadoutTime/PF_PAT_POS_PFOV_PEres_INTERP_test2/NOPF_NOPAT_50POS_PERES100_ES0P59_BW2222_PA_0013/PF_PAT_POS_BW_INTERP_TEST.MR.HEAD_HARMS.0013.0001.2017.09.20.16.16.44.39766.120105534.IMA')
  Orthanc::DicomMap tags;
  tags.SetValue(0x0008, 0x0060, "MR", false);
  tags.SetValue(0x0008, 0x0070, "SIEMENS", false);
  tags.SetValue(0x0020, 0x0013, "1", false);
  tags.SetValue(0x0018, 0x0080, "8320", false);
  tags.SetValue(0x0018, 0x0081, "84", false);
  tags.SetValue(0x0018, 0x1312, "COL", false);
  tags.SetValue(0x0019, 0x1029, "4149.99999999\\0.0\\4287.49999998\\137.49999998\\4427.49999999\\277.5\\4565.0\\414.99999998\\4702.49999998\\552.49999999\\4842.5\\692.5\\4979.99999998\\829.99999999\\5117.49999999\\967.5\\5257.49999998\\1107.49999998\\5394.99999999\\1245.0\\5532.5\\1382.49999998\\5672.49999999\\1522.49999999\\5810.0\\1660.0\\5947.49999998\\1797.49999999\\6087.49999999\\1937.5\\6224.99999998\\2074.99999998\\6362.49999999\\2212.49999999\\6500.0\\2352.49999998\\6639.99999998\\2489.99999999\\6777.5\\2627.5\\6914.99999998\\2767.49999999\\7054.99999999\\2905.0\\7192.5\\3042.49999998\\7329.99999999\\3182.5\\7470.0\\3319.99999998\\7607.49999998\\3457.49999999\\7745.0\\3597.5\\7884.99999998\\3734.99999999\\8022.49999999\\3872.5\\8160.0\\4012.49999998", false);
  tags.SetValue(0x0008, 0x0016, "1.2.840.10008.5.1.4.1.1.4", false);
  tags.SetValue(0x0028, 0x0002, "1", false);
  tags.SetValue(0x0028, 0x0004, "MONOCHROME2", false);
  tags.SetValue(0x0028, 0x0010, "720", false);
  tags.SetValue(0x0028, 0x0011, "720", false);
  tags.SetValue(0x0028, 0x0100, "16", false);
  tags.SetValue(0x0028, 0x0101, "16", false);
  tags.SetValue(0x0028, 0x0102, "15", false);
  tags.SetValue(0x0028, 0x0103, "0", false);
  tags.SetValue(0x0008, 0x0032, "153129.910000", false);
  tags.SetValue(0x0018, 0x0050, "2.4000000953674", false);
  tags.SetValue(0x0018, 0x0088, "2.3999999741376", false);
  tags.SetValue(0x0020, 0x0032, "-866.17921113968\\-872.62231063843\\-64.073608398438", false);
  tags.SetValue(0x0020, 0x0037, "1\\0\\0\\0\\1\\0", false);
  tags.SetValue(0x0028, 0x0030, "2.4000000953674\\2.4000000953674", false);
  std::unique_ptr<Neuro::InputDicomInstance> dicom(new Neuro::InputDicomInstance(tags));
  dicom->GetCSAHeader().AddTag("NumberOfImagesInMosaic", "US").AddValue("60");
  dicom->GetCSAHeader().AddTag("PhaseEncodingDirectionPositive", "IS").AddValue("0");
  dicom->GetCSAHeader().AddTag("SliceNormalVector", "FD").AddValue("0.0").AddValue("0.0").AddValue("1.0");
  target.AddInstance(dicom.release(), "nope");
}

TEST(Autogenerated, Test_49)
{
  Neuro::DicomInstancesCollection instances;
  LoadInstance_49(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 90);
  ASSERT_NEAR(nifti.pixdim[1], 2.400000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 90);
  ASSERT_NEAR(nifti.pixdim[2], 2.400000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 60);
  ASSERT_NEAR(nifti.pixdim[3], 2.400000, 0.00001);
  ASSERT_EQ(nifti.ndim, 3);
  ASSERT_EQ(nifti.dim[0], 3);
  ASSERT_EQ(nifti.nvox, 486000u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(57));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(57));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(57));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 5);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -2.400000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 110.179169, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 2.400000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], -96.977722, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 2.400000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -64.073608, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, 1.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, 0.000000, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=84;Time=153129.910;phase=0");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadInstance_50(Neuro::DicomInstancesCollection& target)
{
  // ('dcm_qa', 'In/TotalReadoutTime/PF_PAT_POS_PFOV_PEres_INTERP_test2/NOPF_NOPAT_NOPOS_PERES100_ES0P59_BW2222_200PFOV_AP_0034/PF_PAT_POS_BW_INTERP_TEST.MR.HEAD_HARMS.0034.0001.2017.09.20.16.16.44.39766.120194668.IMA')
  Orthanc::DicomMap tags;
  tags.SetValue(0x0008, 0x0060, "MR", false);
  tags.SetValue(0x0008, 0x0070, "SIEMENS", false);
  tags.SetValue(0x0020, 0x0013, "1", false);
  tags.SetValue(0x0018, 0x0080, "10730", false);
  tags.SetValue(0x0018, 0x0081, "111.2", false);
  tags.SetValue(0x0018, 0x1312, "COL", false);
  tags.SetValue(0x0019, 0x1029, "5355.00000001\\0.0\\5532.5\\177.50000002\\5712.50000002\\357.50000001\\5890.00000001\\535.0\\6070.00000001\\715.00000003\\6247.5\\892.50000002\\6425.00000002\\1070.00000001\\6605.00000001\\1250.0\\6782.5\\1427.50000002\\6960.00000002\\1607.50000001\\7140.00000001\\1785.0\\7317.5\\1962.50000002\\7497.5\\2142.50000002\\7675.00000002\\2320.00000001\\7852.50000001\\2500.0\\8032.5\\2677.50000002\\8210.00000002\\2855.00000001\\8390.00000001\\3035.0\\8567.5\\3212.50000002\\8745.00000002\\3392.50000002\\8925.00000002\\3570.00000001\\9102.50000001\\3747.5\\9282.5\\3927.50000002\\9460.00000002\\4105.00000001\\9637.50000001\\4285.0\\9817.5\\4462.50000002\\9995.00000002\\4640.00000001\\10175.00000002\\4820.00000001\\10352.50000001\\4997.5\\10530.0\\5177.50000002", false);
  tags.SetValue(0x0008, 0x0016, "1.2.840.10008.5.1.4.1.1.4", false);
  tags.SetValue(0x0028, 0x0002, "1", false);
  tags.SetValue(0x0028, 0x0004, "MONOCHROME2", false);
  tags.SetValue(0x0028, 0x0010, "1440", false);
  tags.SetValue(0x0028, 0x0011, "720", false);
  tags.SetValue(0x0028, 0x0100, "16", false);
  tags.SetValue(0x0028, 0x0101, "16", false);
  tags.SetValue(0x0028, 0x0102, "15", false);
  tags.SetValue(0x0028, 0x0103, "0", false);
  tags.SetValue(0x0008, 0x0032, "160838.905000", false);
  tags.SetValue(0x0018, 0x0050, "2.4000000953674", false);
  tags.SetValue(0x0018, 0x0088, "2.3999999741376", false);
  tags.SetValue(0x0020, 0x0032, "-866.17921113968\\-1736.6223449707\\-64.073608398438", false);
  tags.SetValue(0x0020, 0x0037, "1\\0\\0\\0\\1\\0", false);
  tags.SetValue(0x0028, 0x0030, "2.4000000953674\\2.4000000953674", false);
  std::unique_ptr<Neuro::InputDicomInstance> dicom(new Neuro::InputDicomInstance(tags));
  dicom->GetCSAHeader().AddTag("NumberOfImagesInMosaic", "US").AddValue("60");
  dicom->GetCSAHeader().AddTag("PhaseEncodingDirectionPositive", "IS").AddValue("1");
  dicom->GetCSAHeader().AddTag("SliceNormalVector", "FD").AddValue("0.0").AddValue("0.0").AddValue("1.0");
  target.AddInstance(dicom.release(), "nope");
}

TEST(Autogenerated, Test_50)
{
  Neuro::DicomInstancesCollection instances;
  LoadInstance_50(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 90);
  ASSERT_NEAR(nifti.pixdim[1], 2.400000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 180);
  ASSERT_NEAR(nifti.pixdim[2], 2.400000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 60);
  ASSERT_NEAR(nifti.pixdim[3], 2.400000, 0.00001);
  ASSERT_EQ(nifti.ndim, 3);
  ASSERT_EQ(nifti.dim[0], 3);
  ASSERT_EQ(nifti.nvox, 972000u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(57));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(57));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(57));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 5);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -2.400000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 110.179169, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 2.400000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], -204.977753, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 2.400000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -64.073608, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, 1.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, 0.000000, 0.0001);
  ASSERT_TRUE(std::string(nifti.descrip) == "TE=1.1e+02;Time=160838.905;phase=1" ||
              std::string(nifti.descrip) == "TE=1.1e+002;Time=160838.905;phase=1");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadInstance_51(Neuro::DicomInstancesCollection& target)
{
  // ('dcm_qa', 'In/TotalReadoutTime/PF_PAT_POS_PFOV_PEres_INTERP_test2/NOPF_NOPAT_NOPOS_PERES100_ES0P59_BW2222_200PFOV_PA_0035/PF_PAT_POS_BW_INTERP_TEST.MR.HEAD_HARMS.0035.0001.2017.09.20.16.16.44.39766.120198766.IMA')
  Orthanc::DicomMap tags;
  tags.SetValue(0x0008, 0x0060, "MR", false);
  tags.SetValue(0x0008, 0x0070, "SIEMENS", false);
  tags.SetValue(0x0020, 0x0013, "1", false);
  tags.SetValue(0x0018, 0x0080, "10730", false);
  tags.SetValue(0x0018, 0x0081, "111.2", false);
  tags.SetValue(0x0018, 0x1312, "COL", false);
  tags.SetValue(0x0019, 0x1029, "5354.99999998\\0.0\\5532.5\\177.49999999\\5712.49999999\\357.49999998\\5889.99999998\\535.0\\6067.5\\712.49999999\\6247.5\\892.49999999\\6424.99999999\\1069.99999998\\6604.99999998\\1250.0\\6782.5\\1427.49999999\\6959.99999999\\1604.99999998\\7139.99999998\\1785.0\\7317.5\\1962.49999999\\7497.5\\2142.49999999\\7674.99999999\\2319.99999998\\7852.49999998\\2497.5\\8032.5\\2677.49999999\\8209.99999999\\2854.99999998\\8389.99999998\\3035.0\\8567.5\\3212.49999999\\8745.0\\3389.99999998\\8924.99999999\\3569.99999998\\9102.49999998\\3747.5\\9280.0\\3927.49999999\\9459.99999999\\4104.99999998\\9637.49999998\\4282.5\\9817.5\\4462.49999999\\9995.0\\4639.99999998\\10172.49999999\\4819.99999998\\10352.49999998\\4997.5\\10530.0\\5174.99999999", false);
  tags.SetValue(0x0008, 0x0016, "1.2.840.10008.5.1.4.1.1.4", false);
  tags.SetValue(0x0028, 0x0002, "1", false);
  tags.SetValue(0x0028, 0x0004, "MONOCHROME2", false);
  tags.SetValue(0x0028, 0x0010, "1440", false);
  tags.SetValue(0x0028, 0x0011, "720", false);
  tags.SetValue(0x0028, 0x0100, "16", false);
  tags.SetValue(0x0028, 0x0101, "16", false);
  tags.SetValue(0x0028, 0x0102, "15", false);
  tags.SetValue(0x0028, 0x0103, "0", false);
  tags.SetValue(0x0008, 0x0032, "160919.177500", false);
  tags.SetValue(0x0018, 0x0050, "2.4000000953674", false);
  tags.SetValue(0x0018, 0x0088, "2.3999999741376", false);
  tags.SetValue(0x0020, 0x0032, "-866.17921113968\\-1736.6223449707\\-64.073608398438", false);
  tags.SetValue(0x0020, 0x0037, "1\\0\\0\\0\\1\\0", false);
  tags.SetValue(0x0028, 0x0030, "2.4000000953674\\2.4000000953674", false);
  std::unique_ptr<Neuro::InputDicomInstance> dicom(new Neuro::InputDicomInstance(tags));
  dicom->GetCSAHeader().AddTag("NumberOfImagesInMosaic", "US").AddValue("60");
  dicom->GetCSAHeader().AddTag("PhaseEncodingDirectionPositive", "IS").AddValue("0");
  dicom->GetCSAHeader().AddTag("SliceNormalVector", "FD").AddValue("0.0").AddValue("0.0").AddValue("1.0");
  target.AddInstance(dicom.release(), "nope");
}

TEST(Autogenerated, Test_51)
{
  Neuro::DicomInstancesCollection instances;
  LoadInstance_51(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 90);
  ASSERT_NEAR(nifti.pixdim[1], 2.400000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 180);
  ASSERT_NEAR(nifti.pixdim[2], 2.400000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 60);
  ASSERT_NEAR(nifti.pixdim[3], 2.400000, 0.00001);
  ASSERT_EQ(nifti.ndim, 3);
  ASSERT_EQ(nifti.dim[0], 3);
  ASSERT_EQ(nifti.nvox, 972000u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(57));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(57));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(57));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 5);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -2.400000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 110.179169, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 2.400000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], -204.977753, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 2.400000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -64.073608, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, 1.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, 0.000000, 0.0001);
  ASSERT_TRUE(std::string(nifti.descrip) == "TE=1.1e+02;Time=160919.177;phase=0" ||
              std::string(nifti.descrip) == "TE=1.1e+002;Time=160919.177;phase=0");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadInstance_52(Neuro::DicomInstancesCollection& target)
{
  // ('dcm_qa', 'In/TotalReadoutTime/PF_PAT_POS_PFOV_PEres_INTERP_test2/NOPF_NOPAT_NOPOS_PERES100_ES0P59_BW2222_AP_0008/PF_PAT_POS_BW_INTERP_TEST.MR.HEAD_HARMS.0008.0001.2017.09.20.16.16.44.39766.120085044.IMA')
  Orthanc::DicomMap tags;
  tags.SetValue(0x0008, 0x0060, "MR", false);
  tags.SetValue(0x0008, 0x0070, "SIEMENS", false);
  tags.SetValue(0x0020, 0x0013, "1", false);
  tags.SetValue(0x0018, 0x0080, "6100", false);
  tags.SetValue(0x0018, 0x0081, "60", false);
  tags.SetValue(0x0018, 0x1312, "COL", false);
  tags.SetValue(0x0019, 0x1029, "3040.00000001\\0.0\\3140.00000001\\100.00000001\\3242.50000002\\202.50000001\\3342.5\\305.00000002\\3445.00000001\\405.0\\3547.50000002\\507.50000001\\3647.50000002\\607.50000001\\3750.0\\710.00000002\\3850.00000001\\810.0\\3952.50000001\\912.50000001\\4052.50000002\\1012.50000001\\4155.0\\1115.00000002\\4255.0\\1215.00000003\\4357.50000001\\1317.5\\4457.50000002\\1417.50000001\\4560.0\\1520.00000002\\4660.0\\1620.00000002\\4762.50000001\\1722.5\\4862.50000002\\1822.50000001\\4965.00000003\\1925.00000002\\5067.5\\2027.50000003\\5167.50000001\\2127.5\\5270.00000002\\2230.00000001\\5370.00000002\\2330.00000002\\5472.5\\2432.50000002\\5572.50000001\\2532.5\\5675.00000002\\2635.00000001\\5775.00000002\\2735.00000002\\5877.5\\2837.50000002\\5977.50000001\\2937.5", false);
  tags.SetValue(0x0008, 0x0016, "1.2.840.10008.5.1.4.1.1.4", false);
  tags.SetValue(0x0028, 0x0002, "1", false);
  tags.SetValue(0x0028, 0x0004, "MONOCHROME2", false);
  tags.SetValue(0x0028, 0x0010, "720", false);
  tags.SetValue(0x0028, 0x0011, "720", false);
  tags.SetValue(0x0028, 0x0100, "16", false);
  tags.SetValue(0x0028, 0x0101, "16", false);
  tags.SetValue(0x0028, 0x0102, "15", false);
  tags.SetValue(0x0028, 0x0103, "0", false);
  tags.SetValue(0x0008, 0x0032, "152433.090000", false);
  tags.SetValue(0x0018, 0x0050, "2.4000000953674", false);
  tags.SetValue(0x0018, 0x0088, "2.3999999741376", false);
  tags.SetValue(0x0020, 0x0032, "-866.17921113968\\-872.62231063843\\-64.073608398438", false);
  tags.SetValue(0x0020, 0x0037, "1\\0\\0\\0\\1\\0", false);
  tags.SetValue(0x0028, 0x0030, "2.4000000953674\\2.4000000953674", false);
  std::unique_ptr<Neuro::InputDicomInstance> dicom(new Neuro::InputDicomInstance(tags));
  dicom->GetCSAHeader().AddTag("NumberOfImagesInMosaic", "US").AddValue("60");
  dicom->GetCSAHeader().AddTag("PhaseEncodingDirectionPositive", "IS").AddValue("1");
  dicom->GetCSAHeader().AddTag("SliceNormalVector", "FD").AddValue("0.0").AddValue("0.0").AddValue("1.0");
  target.AddInstance(dicom.release(), "nope");
}

TEST(Autogenerated, Test_52)
{
  Neuro::DicomInstancesCollection instances;
  LoadInstance_52(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 90);
  ASSERT_NEAR(nifti.pixdim[1], 2.400000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 90);
  ASSERT_NEAR(nifti.pixdim[2], 2.400000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 60);
  ASSERT_NEAR(nifti.pixdim[3], 2.400000, 0.00001);
  ASSERT_EQ(nifti.ndim, 3);
  ASSERT_EQ(nifti.dim[0], 3);
  ASSERT_EQ(nifti.nvox, 486000u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(57));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(57));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(57));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 5);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -2.400000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 110.179169, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 2.400000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], -96.977722, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 2.400000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -64.073608, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, 1.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, 0.000000, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=60;Time=152433.090;phase=1");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadInstance_53(Neuro::DicomInstancesCollection& target)
{
  // ('dcm_qa', 'In/TotalReadoutTime/PF_PAT_POS_PFOV_PEres_INTERP_test2/NOPF_NOPAT_NOPOS_PERES100_ES0P59_BW2222_INTERP_AP_0028/PF_PAT_POS_BW_INTERP_TEST.MR.HEAD_HARMS.0028.0001.2017.09.20.16.16.44.39766.120168028.IMA')
  Orthanc::DicomMap tags;
  tags.SetValue(0x0008, 0x0060, "MR", false);
  tags.SetValue(0x0008, 0x0070, "SIEMENS", false);
  tags.SetValue(0x0020, 0x0013, "1", false);
  tags.SetValue(0x0018, 0x0080, "6100", false);
  tags.SetValue(0x0018, 0x0081, "60", false);
  tags.SetValue(0x0018, 0x1312, "COL", false);
  tags.SetValue(0x0019, 0x1029, "3040.00000001\\0.0\\3140.00000001\\100.00000001\\3242.49999999\\202.50000001\\3342.5\\302.49999999\\3445.00000001\\405.0\\3545.00000001\\505.0\\3647.49999999\\607.50000001\\3747.5\\707.50000002\\3850.00000001\\810.0\\3950.00000001\\910.0\\4052.49999999\\1012.50000001\\4152.5\\1114.99999999\\4255.0\\1215.0\\4357.50000001\\1317.5\\4457.50000002\\1417.50000001\\4560.0\\1520.00000002\\4660.0\\1620.0\\4762.50000001\\1722.5\\4862.50000002\\1822.50000001\\4965.0\\1925.00000002\\5065.0\\2024.99999999\\5167.50000001\\2127.5\\5267.50000002\\2227.50000001\\5370.0\\2330.00000002\\5470.0\\2429.99999999\\5572.50000001\\2532.5\\5672.50000002\\2635.00000001\\5774.99999999\\2735.00000002\\5877.5\\2837.49999999\\5977.50000001\\2937.5", false);
  tags.SetValue(0x0008, 0x0016, "1.2.840.10008.5.1.4.1.1.4", false);
  tags.SetValue(0x0028, 0x0002, "1", false);
  tags.SetValue(0x0028, 0x0004, "MONOCHROME2", false);
  tags.SetValue(0x0028, 0x0010, "1440", false);
  tags.SetValue(0x0028, 0x0011, "1440", false);
  tags.SetValue(0x0028, 0x0100, "16", false);
  tags.SetValue(0x0028, 0x0101, "16", false);
  tags.SetValue(0x0028, 0x0102, "15", false);
  tags.SetValue(0x0028, 0x0103, "0", false);
  tags.SetValue(0x0008, 0x0032, "155248.285000", false);
  tags.SetValue(0x0018, 0x0050, "2.4000000953674", false);
  tags.SetValue(0x0018, 0x0088, "2.3999999741376", false);
  tags.SetValue(0x0020, 0x0032, "-866.17921113968\\-872.62231063843\\-64.073608398438", false);
  tags.SetValue(0x0020, 0x0037, "1\\0\\0\\0\\1\\0", false);
  tags.SetValue(0x0028, 0x0030, "1.2000000476837\\1.2000000476837", false);
  std::unique_ptr<Neuro::InputDicomInstance> dicom(new Neuro::InputDicomInstance(tags));
  dicom->GetCSAHeader().AddTag("NumberOfImagesInMosaic", "US").AddValue("60");
  dicom->GetCSAHeader().AddTag("PhaseEncodingDirectionPositive", "IS").AddValue("1");
  dicom->GetCSAHeader().AddTag("SliceNormalVector", "FD").AddValue("0.0").AddValue("0.0").AddValue("1.0");
  target.AddInstance(dicom.release(), "nope");
}

TEST(Autogenerated, Test_53)
{
  Neuro::DicomInstancesCollection instances;
  LoadInstance_53(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 180);
  ASSERT_NEAR(nifti.pixdim[1], 1.200000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 180);
  ASSERT_NEAR(nifti.pixdim[2], 1.200000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 60);
  ASSERT_NEAR(nifti.pixdim[3], 2.400000, 0.00001);
  ASSERT_EQ(nifti.ndim, 3);
  ASSERT_EQ(nifti.dim[0], 3);
  ASSERT_EQ(nifti.nvox, 1944000u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(57));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(57));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(57));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 5);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -1.200000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 110.179169, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 1.200000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], -98.177719, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 2.400000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -64.073608, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, 1.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, 0.000000, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=60;Time=155248.285;phase=1");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadInstance_54(Neuro::DicomInstancesCollection& target)
{
  // ('dcm_qa', 'In/TotalReadoutTime/PF_PAT_POS_PFOV_PEres_INTERP_test2/NOPF_NOPAT_NOPOS_PERES100_ES0P59_BW2222_INTERP_PA_0029/PF_PAT_POS_BW_INTERP_TEST.MR.HEAD_HARMS.0029.0001.2017.09.20.16.16.44.39766.120172126.IMA')
  Orthanc::DicomMap tags;
  tags.SetValue(0x0008, 0x0060, "MR", false);
  tags.SetValue(0x0008, 0x0070, "SIEMENS", false);
  tags.SetValue(0x0020, 0x0013, "1", false);
  tags.SetValue(0x0018, 0x0080, "6100", false);
  tags.SetValue(0x0018, 0x0081, "60", false);
  tags.SetValue(0x0018, 0x1312, "COL", false);
  tags.SetValue(0x0019, 0x1029, "3040.00000001\\0.0\\3142.50000002\\102.50000001\\3242.50000002\\202.50000001\\3345.0\\305.00000002\\3445.00000001\\405.0\\3547.50000002\\507.50000001\\3647.50000002\\607.50000001\\3750.0\\710.00000002\\3850.00000001\\810.0\\3952.50000001\\912.50000001\\4052.50000002\\1012.50000001\\4155.0\\1115.00000002\\4255.0\\1217.5\\4357.50000001\\1317.5\\4460.00000002\\1420.00000001\\4560.0\\1520.00000002\\4662.50000001\\1622.5\\4762.50000001\\1722.5\\4865.00000002\\1825.00000001\\4965.00000003\\1925.00000002\\5067.5\\2027.50000003\\5167.50000001\\2127.5\\5270.00000002\\2230.00000001\\5370.00000002\\2330.00000002\\5472.5\\2432.50000002\\5572.50000001\\2532.5\\5675.00000002\\2635.00000001\\5775.00000002\\2735.00000002\\5877.5\\2837.50000002\\5980.00000001\\2940.0", false);
  tags.SetValue(0x0008, 0x0016, "1.2.840.10008.5.1.4.1.1.4", false);
  tags.SetValue(0x0028, 0x0002, "1", false);
  tags.SetValue(0x0028, 0x0004, "MONOCHROME2", false);
  tags.SetValue(0x0028, 0x0010, "1440", false);
  tags.SetValue(0x0028, 0x0011, "1440", false);
  tags.SetValue(0x0028, 0x0100, "16", false);
  tags.SetValue(0x0028, 0x0101, "16", false);
  tags.SetValue(0x0028, 0x0102, "15", false);
  tags.SetValue(0x0028, 0x0103, "0", false);
  tags.SetValue(0x0008, 0x0032, "155321.340000", false);
  tags.SetValue(0x0018, 0x0050, "2.4000000953674", false);
  tags.SetValue(0x0018, 0x0088, "2.3999999741376", false);
  tags.SetValue(0x0020, 0x0032, "-866.17921113968\\-872.62231063843\\-64.073608398438", false);
  tags.SetValue(0x0020, 0x0037, "1\\0\\0\\0\\1\\0", false);
  tags.SetValue(0x0028, 0x0030, "1.2000000476837\\1.2000000476837", false);
  std::unique_ptr<Neuro::InputDicomInstance> dicom(new Neuro::InputDicomInstance(tags));
  dicom->GetCSAHeader().AddTag("NumberOfImagesInMosaic", "US").AddValue("60");
  dicom->GetCSAHeader().AddTag("PhaseEncodingDirectionPositive", "IS").AddValue("0");
  dicom->GetCSAHeader().AddTag("SliceNormalVector", "FD").AddValue("0.0").AddValue("0.0").AddValue("1.0");
  target.AddInstance(dicom.release(), "nope");
}

TEST(Autogenerated, Test_54)
{
  Neuro::DicomInstancesCollection instances;
  LoadInstance_54(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 180);
  ASSERT_NEAR(nifti.pixdim[1], 1.200000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 180);
  ASSERT_NEAR(nifti.pixdim[2], 1.200000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 60);
  ASSERT_NEAR(nifti.pixdim[3], 2.400000, 0.00001);
  ASSERT_EQ(nifti.ndim, 3);
  ASSERT_EQ(nifti.dim[0], 3);
  ASSERT_EQ(nifti.nvox, 1944000u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(57));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(57));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(57));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 5);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -1.200000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 110.179169, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 1.200000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], -98.177719, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 2.400000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -64.073608, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, 1.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, 0.000000, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=60;Time=155321.340;phase=0");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadInstance_55(Neuro::DicomInstancesCollection& target)
{
  // ('dcm_qa', 'In/TotalReadoutTime/PF_PAT_POS_PFOV_PEres_INTERP_test2/NOPF_NOPAT_NOPOS_PERES100_ES0P59_BW2222_PA_0009/PF_PAT_POS_BW_INTERP_TEST.MR.HEAD_HARMS.0009.0001.2017.09.20.16.16.44.39766.120089142.IMA')
  Orthanc::DicomMap tags;
  tags.SetValue(0x0008, 0x0060, "MR", false);
  tags.SetValue(0x0008, 0x0070, "SIEMENS", false);
  tags.SetValue(0x0020, 0x0013, "1", false);
  tags.SetValue(0x0018, 0x0080, "6100", false);
  tags.SetValue(0x0018, 0x0081, "60", false);
  tags.SetValue(0x0018, 0x1312, "COL", false);
  tags.SetValue(0x0019, 0x1029, "3040.00000001\\0.0\\3142.50000002\\102.50000001\\3242.49999999\\202.50000001\\3345.0\\304.99999999\\3445.00000001\\405.0\\3547.50000002\\507.50000001\\3647.50000002\\607.50000001\\3750.0\\710.00000002\\3850.00000001\\810.0\\3952.50000001\\912.50000001\\4052.50000002\\1012.50000001\\4155.0\\1115.00000002\\4255.0\\1215.0\\4357.50000001\\1317.5\\4457.50000002\\1420.00000001\\4560.0\\1520.00000002\\4662.50000001\\1622.5\\4762.50000001\\1722.5\\4865.00000002\\1825.00000001\\4965.0\\1925.00000002\\5067.5\\2027.5\\5167.50000001\\2127.5\\5270.00000002\\2230.00000001\\5370.0\\2330.00000002\\5472.5\\2432.5\\5572.50000001\\2532.5\\5675.00000002\\2635.00000001\\5774.99999999\\2735.00000002\\5877.5\\2837.49999999\\5977.50000001\\2940.0", false);
  tags.SetValue(0x0008, 0x0016, "1.2.840.10008.5.1.4.1.1.4", false);
  tags.SetValue(0x0028, 0x0002, "1", false);
  tags.SetValue(0x0028, 0x0004, "MONOCHROME2", false);
  tags.SetValue(0x0028, 0x0010, "720", false);
  tags.SetValue(0x0028, 0x0011, "720", false);
  tags.SetValue(0x0028, 0x0100, "16", false);
  tags.SetValue(0x0028, 0x0101, "16", false);
  tags.SetValue(0x0028, 0x0102, "15", false);
  tags.SetValue(0x0028, 0x0103, "0", false);
  tags.SetValue(0x0008, 0x0032, "152556.220000", false);
  tags.SetValue(0x0018, 0x0050, "2.4000000953674", false);
  tags.SetValue(0x0018, 0x0088, "2.3999999741376", false);
  tags.SetValue(0x0020, 0x0032, "-866.17921113968\\-872.62231063843\\-64.073608398438", false);
  tags.SetValue(0x0020, 0x0037, "1\\0\\0\\0\\1\\0", false);
  tags.SetValue(0x0028, 0x0030, "2.4000000953674\\2.4000000953674", false);
  std::unique_ptr<Neuro::InputDicomInstance> dicom(new Neuro::InputDicomInstance(tags));
  dicom->GetCSAHeader().AddTag("NumberOfImagesInMosaic", "US").AddValue("60");
  dicom->GetCSAHeader().AddTag("PhaseEncodingDirectionPositive", "IS").AddValue("0");
  dicom->GetCSAHeader().AddTag("SliceNormalVector", "FD").AddValue("0.0").AddValue("0.0").AddValue("1.0");
  target.AddInstance(dicom.release(), "nope");
}

TEST(Autogenerated, Test_55)
{
  Neuro::DicomInstancesCollection instances;
  LoadInstance_55(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 90);
  ASSERT_NEAR(nifti.pixdim[1], 2.400000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 90);
  ASSERT_NEAR(nifti.pixdim[2], 2.400000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 60);
  ASSERT_NEAR(nifti.pixdim[3], 2.400000, 0.00001);
  ASSERT_EQ(nifti.ndim, 3);
  ASSERT_EQ(nifti.dim[0], 3);
  ASSERT_EQ(nifti.nvox, 486000u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(57));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(57));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(57));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 5);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -2.400000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 110.179169, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 2.400000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], -96.977722, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 2.400000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -64.073608, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, 1.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, 0.000000, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=60;Time=152556.220;phase=0");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadInstance_56(Neuro::DicomInstancesCollection& target)
{
  // ('dcm_qa', 'In/TotalReadoutTime/PF_PAT_POS_PFOV_PEres_INTERP_test2/NOPF_NOPAT_NOPOS_PERES100_ES0P60_BW2222_LR_0031/PF_PAT_POS_BW_INTERP_TEST.MR.HEAD_HARMS.0031.0001.2017.09.20.16.16.44.39766.120182374.IMA')
  Orthanc::DicomMap tags;
  tags.SetValue(0x0008, 0x0060, "MR", false);
  tags.SetValue(0x0008, 0x0070, "SIEMENS", false);
  tags.SetValue(0x0020, 0x0013, "1", false);
  tags.SetValue(0x0018, 0x0080, "6100", false);
  tags.SetValue(0x0018, 0x0081, "60", false);
  tags.SetValue(0x0018, 0x1312, "ROW", false);
  tags.SetValue(0x0019, 0x1029, "3040.00000001\\0.0\\3142.49999999\\102.50000001\\3245.0\\204.99999999\\3345.0\\304.99999999\\3447.49999998\\407.5\\3547.49999999\\507.50000001\\3649.99999999\\609.99999999\\3750.0\\709.99999999\\3852.50000001\\812.5\\3952.49999998\\912.50000001\\4054.99999999\\1014.99999998\\4155.0\\1114.99999999\\4257.50000001\\1217.5\\4357.49999998\\1317.5\\4459.99999999\\1419.99999998\\4560.0\\1519.99999999\\4662.50000001\\1622.5\\4762.49999998\\1725.00000001\\4864.99999999\\1824.99999998\\4967.5\\1927.49999999\\5067.5\\2027.5\\5169.99999998\\2130.0\\5269.99999999\\2229.99999998\\5372.5\\2332.49999999\\5472.5\\2432.5\\5574.99999998\\2535.0\\5674.99999999\\2634.99999998\\5777.5\\2737.49999999\\5877.5\\2837.49999999\\5979.99999998\\2940.0", false);
  tags.SetValue(0x0008, 0x0016, "1.2.840.10008.5.1.4.1.1.4", false);
  tags.SetValue(0x0028, 0x0002, "1", false);
  tags.SetValue(0x0028, 0x0004, "MONOCHROME2", false);
  tags.SetValue(0x0028, 0x0010, "720", false);
  tags.SetValue(0x0028, 0x0011, "720", false);
  tags.SetValue(0x0028, 0x0100, "16", false);
  tags.SetValue(0x0028, 0x0101, "16", false);
  tags.SetValue(0x0028, 0x0102, "15", false);
  tags.SetValue(0x0028, 0x0103, "0", false);
  tags.SetValue(0x0008, 0x0032, "155859.327500", false);
  tags.SetValue(0x0018, 0x0050, "2.4000000953674", false);
  tags.SetValue(0x0018, 0x0088, "2.3999999741376", false);
  tags.SetValue(0x0020, 0x0032, "-866.17921113545\\-872.62231064266\\-64.073608398438", false);
  tags.SetValue(0x0020, 0x0037, "1\\4.896e-012\\0\\-4.896e-012\\1\\0", false);
  tags.SetValue(0x0028, 0x0030, "2.4000000953674\\2.4000000953674", false);
  std::unique_ptr<Neuro::InputDicomInstance> dicom(new Neuro::InputDicomInstance(tags));
  dicom->GetCSAHeader().AddTag("NumberOfImagesInMosaic", "US").AddValue("60");
  dicom->GetCSAHeader().AddTag("PhaseEncodingDirectionPositive", "IS").AddValue("0");
  dicom->GetCSAHeader().AddTag("SliceNormalVector", "FD").AddValue("0.0").AddValue("0.0").AddValue("1.0");
  target.AddInstance(dicom.release(), "nope");
}

TEST(Autogenerated, Test_56)
{
  Neuro::DicomInstancesCollection instances;
  LoadInstance_56(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 90);
  ASSERT_NEAR(nifti.pixdim[1], 2.400000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 90);
  ASSERT_NEAR(nifti.pixdim[2], 2.400000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 60);
  ASSERT_NEAR(nifti.pixdim[3], 2.400000, 0.00001);
  ASSERT_EQ(nifti.ndim, 3);
  ASSERT_EQ(nifti.dim[0], 3);
  ASSERT_EQ(nifti.nvox, 486000u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(54));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(54));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(54));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 5);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -2.400000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 110.179169, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 2.400000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], -96.977722, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 2.400000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -64.073608, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, -0.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, 1.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, 0.000000, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=60;Time=155859.328;phase=0");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadInstance_57(Neuro::DicomInstancesCollection& target)
{
  // ('dcm_qa', 'In/TotalReadoutTime/PF_PAT_POS_PFOV_PEres_INTERP_test2/NOPF_NOPAT_NOPOS_PERES100_ES0P60_BW2222_RL_0030/PF_PAT_POS_BW_INTERP_TEST.MR.HEAD_HARMS.0030.0001.2017.09.20.16.16.44.39766.120178276.IMA')
  Orthanc::DicomMap tags;
  tags.SetValue(0x0008, 0x0060, "MR", false);
  tags.SetValue(0x0008, 0x0070, "SIEMENS", false);
  tags.SetValue(0x0020, 0x0013, "1", false);
  tags.SetValue(0x0018, 0x0080, "6100", false);
  tags.SetValue(0x0018, 0x0081, "60", false);
  tags.SetValue(0x0018, 0x1312, "ROW", false);
  tags.SetValue(0x0019, 0x1029, "3040.00000001\\0.0\\3142.49999999\\102.50000001\\3242.49999999\\202.49999998\\3345.0\\304.99999999\\3445.00000001\\405.0\\3547.49999999\\507.50000001\\3647.49999999\\607.49999998\\3750.0\\709.99999999\\3850.00000001\\810.0\\3952.49999998\\912.50000001\\4054.99999999\\1014.99999998\\4155.0\\1114.99999999\\4257.50000001\\1217.5\\4357.49999998\\1317.5\\4459.99999999\\1419.99999998\\4560.0\\1519.99999999\\4662.50000001\\1622.5\\4762.49999998\\1722.5\\4864.99999999\\1824.99999998\\4965.0\\1924.99999999\\5067.5\\2027.5\\5167.49999998\\2127.5\\5269.99999999\\2229.99999998\\5370.0\\2329.99999999\\5472.5\\2432.5\\5572.49999998\\2535.0\\5674.99999999\\2634.99999998\\5777.5\\2737.49999999\\5877.5\\2837.49999999\\5979.99999998\\2940.0", false);
  tags.SetValue(0x0008, 0x0016, "1.2.840.10008.5.1.4.1.1.4", false);
  tags.SetValue(0x0028, 0x0002, "1", false);
  tags.SetValue(0x0028, 0x0004, "MONOCHROME2", false);
  tags.SetValue(0x0028, 0x0010, "720", false);
  tags.SetValue(0x0028, 0x0011, "720", false);
  tags.SetValue(0x0028, 0x0100, "16", false);
  tags.SetValue(0x0028, 0x0101, "16", false);
  tags.SetValue(0x0028, 0x0102, "15", false);
  tags.SetValue(0x0028, 0x0103, "0", false);
  tags.SetValue(0x0008, 0x0032, "155814.140000", false);
  tags.SetValue(0x0018, 0x0050, "2.4000000953674", false);
  tags.SetValue(0x0018, 0x0088, "2.3999999741376", false);
  tags.SetValue(0x0020, 0x0032, "-866.17921113968\\-872.62231063843\\-64.073608398438", false);
  tags.SetValue(0x0020, 0x0037, "1\\0\\0\\0\\1\\0", false);
  tags.SetValue(0x0028, 0x0030, "2.4000000953674\\2.4000000953674", false);
  std::unique_ptr<Neuro::InputDicomInstance> dicom(new Neuro::InputDicomInstance(tags));
  dicom->GetCSAHeader().AddTag("NumberOfImagesInMosaic", "US").AddValue("60");
  dicom->GetCSAHeader().AddTag("PhaseEncodingDirectionPositive", "IS").AddValue("1");
  dicom->GetCSAHeader().AddTag("SliceNormalVector", "FD").AddValue("0.0").AddValue("0.0").AddValue("1.0");
  target.AddInstance(dicom.release(), "nope");
}

TEST(Autogenerated, Test_57)
{
  Neuro::DicomInstancesCollection instances;
  LoadInstance_57(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 90);
  ASSERT_NEAR(nifti.pixdim[1], 2.400000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 90);
  ASSERT_NEAR(nifti.pixdim[2], 2.400000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 60);
  ASSERT_NEAR(nifti.pixdim[3], 2.400000, 0.00001);
  ASSERT_EQ(nifti.ndim, 3);
  ASSERT_EQ(nifti.dim[0], 3);
  ASSERT_EQ(nifti.nvox, 486000u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(54));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(54));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(54));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 5);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -2.400000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 110.179169, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 2.400000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], -96.977722, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 2.400000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -64.073608, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, 1.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, 0.000000, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=60;Time=155814.140;phase=1");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadInstance_58(Neuro::DicomInstancesCollection& target)
{
  // ('dcm_qa', 'In/TotalReadoutTime/PF_PAT_POS_PFOV_PEres_INTERP_test2/NOPF_NOPAT_NOPOS_PERES100_ES1P00_BW1112_AP_0026/PF_PAT_POS_BW_INTERP_TEST.MR.HEAD_HARMS.0026.0001.2017.09.20.16.16.44.39766.120159832.IMA')
  Orthanc::DicomMap tags;
  tags.SetValue(0x0008, 0x0060, "MR", false);
  tags.SetValue(0x0008, 0x0070, "SIEMENS", false);
  tags.SetValue(0x0020, 0x0013, "1", false);
  tags.SetValue(0x0018, 0x0080, "9280", false);
  tags.SetValue(0x0018, 0x0081, "95.4", false);
  tags.SetValue(0x0018, 0x1312, "COL", false);
  tags.SetValue(0x0019, 0x1029, "4630.0\\0.0\\4785.0\\155.0\\4940.0\\310.0\\5092.5\\462.49999999\\5247.5\\617.49999999\\5402.5\\772.49999999\\5554.99999999\\927.49999999\\5709.99999999\\1079.99999999\\5864.99999999\\1234.99999999\\6019.99999999\\1389.99999998\\6172.49999999\\1542.50000001\\6327.49999998\\1697.50000001\\6482.49999998\\1852.50000001\\6637.49999998\\2007.50000001\\6790.00000001\\2160.0\\6945.00000001\\2315.0\\7100.00000001\\2470.0\\7255.0\\2625.0\\7407.5\\2777.5\\7562.5\\2932.5\\7717.5\\3087.49999999\\7870.0\\3242.49999999\\8024.99999999\\3394.99999999\\8179.99999999\\3549.99999999\\8334.99999999\\3704.99999999\\8487.49999999\\3857.49999998\\8642.49999999\\4012.49999998\\8797.49999999\\4167.50000001\\8952.49999998\\4322.50000001\\9105.00000001\\4475.00000001", false);
  tags.SetValue(0x0008, 0x0016, "1.2.840.10008.5.1.4.1.1.4", false);
  tags.SetValue(0x0028, 0x0002, "1", false);
  tags.SetValue(0x0028, 0x0004, "MONOCHROME2", false);
  tags.SetValue(0x0028, 0x0010, "720", false);
  tags.SetValue(0x0028, 0x0011, "720", false);
  tags.SetValue(0x0028, 0x0100, "16", false);
  tags.SetValue(0x0028, 0x0101, "16", false);
  tags.SetValue(0x0028, 0x0102, "15", false);
  tags.SetValue(0x0028, 0x0103, "0", false);
  tags.SetValue(0x0008, 0x0032, "154900.915000", false);
  tags.SetValue(0x0018, 0x0050, "2.4000000953674", false);
  tags.SetValue(0x0018, 0x0088, "2.3999999741376", false);
  tags.SetValue(0x0020, 0x0032, "-866.17921113968\\-872.62231063843\\-64.073608398438", false);
  tags.SetValue(0x0020, 0x0037, "1\\0\\0\\0\\1\\0", false);
  tags.SetValue(0x0028, 0x0030, "2.4000000953674\\2.4000000953674", false);
  std::unique_ptr<Neuro::InputDicomInstance> dicom(new Neuro::InputDicomInstance(tags));
  dicom->GetCSAHeader().AddTag("NumberOfImagesInMosaic", "US").AddValue("60");
  dicom->GetCSAHeader().AddTag("PhaseEncodingDirectionPositive", "IS").AddValue("1");
  dicom->GetCSAHeader().AddTag("SliceNormalVector", "FD").AddValue("0.0").AddValue("0.0").AddValue("1.0");
  target.AddInstance(dicom.release(), "nope");
}

TEST(Autogenerated, Test_58)
{
  Neuro::DicomInstancesCollection instances;
  LoadInstance_58(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 90);
  ASSERT_NEAR(nifti.pixdim[1], 2.400000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 90);
  ASSERT_NEAR(nifti.pixdim[2], 2.400000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 60);
  ASSERT_NEAR(nifti.pixdim[3], 2.400000, 0.00001);
  ASSERT_EQ(nifti.ndim, 3);
  ASSERT_EQ(nifti.dim[0], 3);
  ASSERT_EQ(nifti.nvox, 486000u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(57));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(57));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(57));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 5);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -2.400000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 110.179169, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 2.400000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], -96.977722, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 2.400000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -64.073608, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, 1.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, 0.000000, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=95;Time=154900.915;phase=1");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadInstance_59(Neuro::DicomInstancesCollection& target)
{
  // ('dcm_qa', 'In/TotalReadoutTime/PF_PAT_POS_PFOV_PEres_INTERP_test2/NOPF_NOPAT_NOPOS_PERES100_ES1P00_BW1112_PA_0027/PF_PAT_POS_BW_INTERP_TEST.MR.HEAD_HARMS.0027.0001.2017.09.20.16.16.44.39766.120163930.IMA')
  Orthanc::DicomMap tags;
  tags.SetValue(0x0008, 0x0060, "MR", false);
  tags.SetValue(0x0008, 0x0070, "SIEMENS", false);
  tags.SetValue(0x0020, 0x0013, "1", false);
  tags.SetValue(0x0018, 0x0080, "9280", false);
  tags.SetValue(0x0018, 0x0081, "95.4", false);
  tags.SetValue(0x0018, 0x1312, "COL", false);
  tags.SetValue(0x0019, 0x1029, "4630.0\\0.0\\4782.5\\152.5\\4937.5\\307.5\\5092.5\\462.49999999\\5247.5\\617.49999999\\5399.99999999\\769.99999999\\5554.99999999\\925.00000002\\5709.99999999\\1080.00000002\\5864.99999999\\1235.00000002\\6017.50000002\\1387.50000001\\6172.50000002\\1542.50000001\\6327.50000001\\1697.50000001\\6480.00000001\\1852.50000001\\6635.00000001\\2005.0\\6790.00000001\\2160.0\\6945.00000001\\2315.0\\7097.5\\2467.5\\7252.5\\2622.5\\7407.5\\2777.5\\7562.5\\2932.5\\7715.0\\3084.99999999\\7870.0\\3239.99999999\\8024.99999999\\3394.99999999\\8179.99999999\\3550.00000002\\8332.49999999\\3702.50000001\\8487.50000002\\3857.50000001\\8642.50000002\\4012.50000001\\8795.00000001\\4165.00000001\\8950.00000001\\4320.00000001\\9105.00000001\\4475.00000001", false);
  tags.SetValue(0x0008, 0x0016, "1.2.840.10008.5.1.4.1.1.4", false);
  tags.SetValue(0x0028, 0x0002, "1", false);
  tags.SetValue(0x0028, 0x0004, "MONOCHROME2", false);
  tags.SetValue(0x0028, 0x0010, "720", false);
  tags.SetValue(0x0028, 0x0011, "720", false);
  tags.SetValue(0x0028, 0x0100, "16", false);
  tags.SetValue(0x0028, 0x0101, "16", false);
  tags.SetValue(0x0028, 0x0102, "15", false);
  tags.SetValue(0x0028, 0x0103, "0", false);
  tags.SetValue(0x0008, 0x0032, "155026.907500", false);
  tags.SetValue(0x0018, 0x0050, "2.4000000953674", false);
  tags.SetValue(0x0018, 0x0088, "2.3999999741376", false);
  tags.SetValue(0x0020, 0x0032, "-866.17921113968\\-872.62231063843\\-64.073608398438", false);
  tags.SetValue(0x0020, 0x0037, "1\\0\\0\\0\\1\\0", false);
  tags.SetValue(0x0028, 0x0030, "2.4000000953674\\2.4000000953674", false);
  std::unique_ptr<Neuro::InputDicomInstance> dicom(new Neuro::InputDicomInstance(tags));
  dicom->GetCSAHeader().AddTag("NumberOfImagesInMosaic", "US").AddValue("60");
  dicom->GetCSAHeader().AddTag("PhaseEncodingDirectionPositive", "IS").AddValue("0");
  dicom->GetCSAHeader().AddTag("SliceNormalVector", "FD").AddValue("0.0").AddValue("0.0").AddValue("1.0");
  target.AddInstance(dicom.release(), "nope");
}

TEST(Autogenerated, Test_59)
{
  Neuro::DicomInstancesCollection instances;
  LoadInstance_59(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 90);
  ASSERT_NEAR(nifti.pixdim[1], 2.400000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 90);
  ASSERT_NEAR(nifti.pixdim[2], 2.400000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 60);
  ASSERT_NEAR(nifti.pixdim[3], 2.400000, 0.00001);
  ASSERT_EQ(nifti.ndim, 3);
  ASSERT_EQ(nifti.dim[0], 3);
  ASSERT_EQ(nifti.nvox, 486000u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(57));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(57));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(57));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 5);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -2.400000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 110.179169, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 2.400000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], -96.977722, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 2.400000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -64.073608, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, 1.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, 0.000000, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=95;Time=155026.908;phase=0");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadInstance_60(Neuro::DicomInstancesCollection& target)
{
  // ('dcm_qa', 'In/TotalReadoutTime/PF_PAT_POS_PFOV_PEres_INTERP_test2/NOPF_NOPAT_NOPOS_PERES100_ES1P00_BW2222_AP_0024/PF_PAT_POS_BW_INTERP_TEST.MR.HEAD_HARMS.0024.0001.2017.09.20.16.16.44.39766.120151636.IMA')
  Orthanc::DicomMap tags;
  tags.SetValue(0x0008, 0x0060, "MR", false);
  tags.SetValue(0x0008, 0x0070, "SIEMENS", false);
  tags.SetValue(0x0020, 0x0013, "1", false);
  tags.SetValue(0x0018, 0x0080, "9280", false);
  tags.SetValue(0x0018, 0x0081, "95.4", false);
  tags.SetValue(0x0018, 0x1312, "COL", false);
  tags.SetValue(0x0019, 0x1029, "4630.0\\0.0\\4782.5\\152.5\\4937.5\\307.5\\5092.5\\462.49999999\\5245.0\\617.49999999\\5399.99999999\\769.99999999\\5554.99999999\\924.99999999\\5709.99999999\\1079.99999999\\5862.49999999\\1232.49999998\\6017.49999999\\1387.49999998\\6172.49999999\\1542.49999998\\6327.49999998\\1697.49999998\\6479.99999998\\1849.99999998\\6634.99999998\\2005.0\\6789.99999998\\2160.0\\6944.99999998\\2315.0\\7097.5\\2467.5\\7252.5\\2622.5\\7407.5\\2777.5\\7560.0\\2932.5\\7715.0\\3084.99999999\\7870.0\\3239.99999999\\8024.99999999\\3394.99999999\\8177.49999999\\3547.49999999\\8332.49999999\\3702.49999998\\8487.49999999\\3857.49999998\\8642.49999999\\4012.49999998\\8794.99999998\\4164.99999998\\8949.99999998\\4319.99999998\\9104.99999998\\4474.99999998", false);
  tags.SetValue(0x0008, 0x0016, "1.2.840.10008.5.1.4.1.1.4", false);
  tags.SetValue(0x0028, 0x0002, "1", false);
  tags.SetValue(0x0028, 0x0004, "MONOCHROME2", false);
  tags.SetValue(0x0028, 0x0010, "720", false);
  tags.SetValue(0x0028, 0x0011, "720", false);
  tags.SetValue(0x0028, 0x0100, "16", false);
  tags.SetValue(0x0028, 0x0101, "16", false);
  tags.SetValue(0x0028, 0x0102, "15", false);
  tags.SetValue(0x0028, 0x0103, "0", false);
  tags.SetValue(0x0008, 0x0032, "154534.577500", false);
  tags.SetValue(0x0018, 0x0050, "2.4000000953674", false);
  tags.SetValue(0x0018, 0x0088, "2.3999999741376", false);
  tags.SetValue(0x0020, 0x0032, "-866.17921113968\\-872.62231063843\\-64.073608398438", false);
  tags.SetValue(0x0020, 0x0037, "1\\0\\0\\0\\1\\0", false);
  tags.SetValue(0x0028, 0x0030, "2.4000000953674\\2.4000000953674", false);
  std::unique_ptr<Neuro::InputDicomInstance> dicom(new Neuro::InputDicomInstance(tags));
  dicom->GetCSAHeader().AddTag("NumberOfImagesInMosaic", "US").AddValue("60");
  dicom->GetCSAHeader().AddTag("PhaseEncodingDirectionPositive", "IS").AddValue("1");
  dicom->GetCSAHeader().AddTag("SliceNormalVector", "FD").AddValue("0.0").AddValue("0.0").AddValue("1.0");
  target.AddInstance(dicom.release(), "nope");
}

TEST(Autogenerated, Test_60)
{
  Neuro::DicomInstancesCollection instances;
  LoadInstance_60(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 90);
  ASSERT_NEAR(nifti.pixdim[1], 2.400000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 90);
  ASSERT_NEAR(nifti.pixdim[2], 2.400000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 60);
  ASSERT_NEAR(nifti.pixdim[3], 2.400000, 0.00001);
  ASSERT_EQ(nifti.ndim, 3);
  ASSERT_EQ(nifti.dim[0], 3);
  ASSERT_EQ(nifti.nvox, 486000u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(57));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(57));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(57));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 5);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -2.400000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 110.179169, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 2.400000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], -96.977722, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 2.400000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -64.073608, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, 1.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, 0.000000, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=95;Time=154534.578;phase=1");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadInstance_61(Neuro::DicomInstancesCollection& target)
{
  // ('dcm_qa', 'In/TotalReadoutTime/PF_PAT_POS_PFOV_PEres_INTERP_test2/NOPF_NOPAT_NOPOS_PERES100_ES1P00_BW2222_PA_0025/PF_PAT_POS_BW_INTERP_TEST.MR.HEAD_HARMS.0025.0001.2017.09.20.16.16.44.39766.120155734.IMA')
  Orthanc::DicomMap tags;
  tags.SetValue(0x0008, 0x0060, "MR", false);
  tags.SetValue(0x0008, 0x0070, "SIEMENS", false);
  tags.SetValue(0x0020, 0x0013, "1", false);
  tags.SetValue(0x0018, 0x0080, "9280", false);
  tags.SetValue(0x0018, 0x0081, "95.4", false);
  tags.SetValue(0x0018, 0x1312, "COL", false);
  tags.SetValue(0x0019, 0x1029, "4630.0\\0.0\\4785.0\\155.0\\4937.5\\310.0\\5092.5\\462.49999999\\5247.5\\617.49999999\\5402.5\\772.49999999\\5554.99999999\\924.99999999\\5709.99999999\\1079.99999999\\5864.99999999\\1234.99999999\\6019.99999999\\1389.99999998\\6172.49999999\\1542.49999998\\6327.49999998\\1697.49999998\\6482.49999998\\1852.50000001\\6637.49999998\\2007.50000001\\6790.00000001\\2160.0\\6945.00000001\\2315.0\\7100.00000001\\2470.0\\7252.5\\2625.0\\7407.5\\2777.5\\7562.5\\2932.5\\7717.5\\3087.49999999\\7870.0\\3239.99999999\\8024.99999999\\3394.99999999\\8179.99999999\\3549.99999999\\8334.99999999\\3704.99999999\\8487.49999999\\3857.49999998\\8642.49999999\\4012.49999998\\8797.49999999\\4167.49999998\\8952.49999998\\4322.49999998\\9104.99999998\\4475.00000001", false);
  tags.SetValue(0x0008, 0x0016, "1.2.840.10008.5.1.4.1.1.4", false);
  tags.SetValue(0x0028, 0x0002, "1", false);
  tags.SetValue(0x0028, 0x0004, "MONOCHROME2", false);
  tags.SetValue(0x0028, 0x0010, "720", false);
  tags.SetValue(0x0028, 0x0011, "720", false);
  tags.SetValue(0x0028, 0x0100, "16", false);
  tags.SetValue(0x0028, 0x0101, "16", false);
  tags.SetValue(0x0028, 0x0102, "15", false);
  tags.SetValue(0x0028, 0x0103, "0", false);
  tags.SetValue(0x0008, 0x0032, "154613.292500", false);
  tags.SetValue(0x0018, 0x0050, "2.4000000953674", false);
  tags.SetValue(0x0018, 0x0088, "2.3999999741376", false);
  tags.SetValue(0x0020, 0x0032, "-866.17921113968\\-872.62231063843\\-64.073608398438", false);
  tags.SetValue(0x0020, 0x0037, "1\\0\\0\\0\\1\\0", false);
  tags.SetValue(0x0028, 0x0030, "2.4000000953674\\2.4000000953674", false);
  std::unique_ptr<Neuro::InputDicomInstance> dicom(new Neuro::InputDicomInstance(tags));
  dicom->GetCSAHeader().AddTag("NumberOfImagesInMosaic", "US").AddValue("60");
  dicom->GetCSAHeader().AddTag("PhaseEncodingDirectionPositive", "IS").AddValue("0");
  dicom->GetCSAHeader().AddTag("SliceNormalVector", "FD").AddValue("0.0").AddValue("0.0").AddValue("1.0");
  target.AddInstance(dicom.release(), "nope");
}

TEST(Autogenerated, Test_61)
{
  Neuro::DicomInstancesCollection instances;
  LoadInstance_61(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 90);
  ASSERT_NEAR(nifti.pixdim[1], 2.400000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 90);
  ASSERT_NEAR(nifti.pixdim[2], 2.400000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 60);
  ASSERT_NEAR(nifti.pixdim[3], 2.400000, 0.00001);
  ASSERT_EQ(nifti.ndim, 3);
  ASSERT_EQ(nifti.dim[0], 3);
  ASSERT_EQ(nifti.nvox, 486000u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(57));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(57));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(57));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 5);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -2.400000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 110.179169, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 2.400000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], -96.977722, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 2.400000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -64.073608, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, 1.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, 0.000000, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=95;Time=154613.293;phase=0");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadInstance_62(Neuro::DicomInstancesCollection& target)
{
  // ('dcm_qa', 'In/TotalReadoutTime/PF_PAT_POS_PFOV_PEres_INTERP_test2/NOPF_NOPAT_NOPOS_PERES50_ES0P59_BW2222_AP_0020/PF_PAT_POS_BW_INTERP_TEST.MR.HEAD_HARMS.0020.0001.2017.09.20.16.16.44.39766.120134220.IMA')
  Orthanc::DicomMap tags;
  tags.SetValue(0x0008, 0x0060, "MR", false);
  tags.SetValue(0x0008, 0x0070, "SIEMENS", false);
  tags.SetValue(0x0020, 0x0013, "1", false);
  tags.SetValue(0x0018, 0x0080, "6100", false);
  tags.SetValue(0x0018, 0x0081, "60", false);
  tags.SetValue(0x0018, 0x1312, "COL", false);
  tags.SetValue(0x0019, 0x1029, "3040.00000001\\0.0\\3142.49999999\\102.50000001\\3242.49999999\\202.49999998\\3345.0\\304.99999999\\3445.00000001\\405.0\\3547.49999999\\507.50000001\\3647.49999999\\607.49999998\\3750.0\\709.99999999\\3850.00000001\\810.0\\3952.49999998\\912.50000001\\4052.49999999\\1012.49999998\\4155.0\\1114.99999999\\4255.0\\1215.0\\4357.49999998\\1317.5\\4457.49999999\\1419.99999998\\4560.0\\1519.99999999\\4662.50000001\\1622.5\\4762.49999998\\1722.5\\4864.99999999\\1824.99999998\\4965.0\\1924.99999999\\5067.5\\2027.5\\5167.50000001\\2127.5\\5269.99999999\\2230.00000001\\5370.0\\2329.99999999\\5472.5\\2432.5\\5572.50000001\\2532.5\\5674.99999999\\2635.00000001\\5774.99999999\\2734.99999999\\5877.5\\2837.49999999\\5977.50000001\\2937.5", false);
  tags.SetValue(0x0008, 0x0016, "1.2.840.10008.5.1.4.1.1.4", false);
  tags.SetValue(0x0028, 0x0002, "1", false);
  tags.SetValue(0x0028, 0x0004, "MONOCHROME2", false);
  tags.SetValue(0x0028, 0x0010, "720", false);
  tags.SetValue(0x0028, 0x0011, "720", false);
  tags.SetValue(0x0028, 0x0100, "16", false);
  tags.SetValue(0x0028, 0x0101, "16", false);
  tags.SetValue(0x0028, 0x0102, "15", false);
  tags.SetValue(0x0028, 0x0103, "0", false);
  tags.SetValue(0x0008, 0x0032, "154118.387500", false);
  tags.SetValue(0x0018, 0x0050, "2.4000000953674", false);
  tags.SetValue(0x0018, 0x0088, "2.3999999741376", false);
  tags.SetValue(0x0020, 0x0032, "-866.17921113968\\-872.62231063843\\-64.073608398438", false);
  tags.SetValue(0x0020, 0x0037, "1\\0\\0\\0\\1\\0", false);
  tags.SetValue(0x0028, 0x0030, "2.4000000953674\\2.4000000953674", false);
  std::unique_ptr<Neuro::InputDicomInstance> dicom(new Neuro::InputDicomInstance(tags));
  dicom->GetCSAHeader().AddTag("NumberOfImagesInMosaic", "US").AddValue("60");
  dicom->GetCSAHeader().AddTag("PhaseEncodingDirectionPositive", "IS").AddValue("1");
  dicom->GetCSAHeader().AddTag("SliceNormalVector", "FD").AddValue("0.0").AddValue("0.0").AddValue("1.0");
  target.AddInstance(dicom.release(), "nope");
}

TEST(Autogenerated, Test_62)
{
  Neuro::DicomInstancesCollection instances;
  LoadInstance_62(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 90);
  ASSERT_NEAR(nifti.pixdim[1], 2.400000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 90);
  ASSERT_NEAR(nifti.pixdim[2], 2.400000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 60);
  ASSERT_NEAR(nifti.pixdim[3], 2.400000, 0.00001);
  ASSERT_EQ(nifti.ndim, 3);
  ASSERT_EQ(nifti.dim[0], 3);
  ASSERT_EQ(nifti.nvox, 486000u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(57));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(57));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(57));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 5);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -2.400000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 110.179169, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 2.400000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], -96.977722, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 2.400000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -64.073608, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, 1.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, 0.000000, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=60;Time=154118.388;phase=1");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadInstance_63(Neuro::DicomInstancesCollection& target)
{
  // ('dcm_qa', 'In/TotalReadoutTime/PF_PAT_POS_PFOV_PEres_INTERP_test2/NOPF_NOPAT_NOPOS_PERES50_ES0P59_BW2222_PA_0021/PF_PAT_POS_BW_INTERP_TEST.MR.HEAD_HARMS.0021.0001.2017.09.20.16.16.44.39766.120138318.IMA')
  Orthanc::DicomMap tags;
  tags.SetValue(0x0008, 0x0060, "MR", false);
  tags.SetValue(0x0008, 0x0070, "SIEMENS", false);
  tags.SetValue(0x0020, 0x0013, "1", false);
  tags.SetValue(0x0018, 0x0080, "6100", false);
  tags.SetValue(0x0018, 0x0081, "60", false);
  tags.SetValue(0x0018, 0x1312, "COL", false);
  tags.SetValue(0x0019, 0x1029, "3040.00000001\\0.0\\3140.00000001\\100.00000001\\3242.50000002\\202.50000001\\3345.0\\305.00000002\\3445.00000001\\405.0\\3547.50000002\\507.50000001\\3647.50000002\\607.50000001\\3750.0\\710.00000002\\3850.00000001\\810.0\\3952.50000001\\912.50000001\\4052.50000002\\1012.50000001\\4155.0\\1115.00000002\\4255.0\\1215.0\\4357.50000001\\1317.5\\4457.50000002\\1417.50000001\\4560.0\\1520.00000002\\4660.0\\1620.0\\4762.50000001\\1722.5\\4862.50000002\\1825.00000001\\4965.0\\1925.00000002\\5067.5\\2027.5\\5167.50000001\\2127.5\\5270.00000002\\2230.00000001\\5370.0\\2330.00000002\\5472.5\\2432.5\\5572.50000001\\2532.5\\5675.00000002\\2635.00000001\\5774.99999999\\2735.00000002\\5877.5\\2837.49999999\\5977.50000001\\2937.5", false);
  tags.SetValue(0x0008, 0x0016, "1.2.840.10008.5.1.4.1.1.4", false);
  tags.SetValue(0x0028, 0x0002, "1", false);
  tags.SetValue(0x0028, 0x0004, "MONOCHROME2", false);
  tags.SetValue(0x0028, 0x0010, "720", false);
  tags.SetValue(0x0028, 0x0011, "720", false);
  tags.SetValue(0x0028, 0x0100, "16", false);
  tags.SetValue(0x0028, 0x0101, "16", false);
  tags.SetValue(0x0028, 0x0102, "15", false);
  tags.SetValue(0x0028, 0x0103, "0", false);
  tags.SetValue(0x0008, 0x0032, "154152.187500", false);
  tags.SetValue(0x0018, 0x0050, "2.4000000953674", false);
  tags.SetValue(0x0018, 0x0088, "2.3999999741376", false);
  tags.SetValue(0x0020, 0x0032, "-866.17921113968\\-872.62231063843\\-64.073608398438", false);
  tags.SetValue(0x0020, 0x0037, "1\\0\\0\\0\\1\\0", false);
  tags.SetValue(0x0028, 0x0030, "2.4000000953674\\2.4000000953674", false);
  std::unique_ptr<Neuro::InputDicomInstance> dicom(new Neuro::InputDicomInstance(tags));
  dicom->GetCSAHeader().AddTag("NumberOfImagesInMosaic", "US").AddValue("60");
  dicom->GetCSAHeader().AddTag("PhaseEncodingDirectionPositive", "IS").AddValue("0");
  dicom->GetCSAHeader().AddTag("SliceNormalVector", "FD").AddValue("0.0").AddValue("0.0").AddValue("1.0");
  target.AddInstance(dicom.release(), "nope");
}

TEST(Autogenerated, Test_63)
{
  Neuro::DicomInstancesCollection instances;
  LoadInstance_63(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 90);
  ASSERT_NEAR(nifti.pixdim[1], 2.400000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 90);
  ASSERT_NEAR(nifti.pixdim[2], 2.400000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 60);
  ASSERT_NEAR(nifti.pixdim[3], 2.400000, 0.00001);
  ASSERT_EQ(nifti.ndim, 3);
  ASSERT_EQ(nifti.dim[0], 3);
  ASSERT_EQ(nifti.nvox, 486000u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(57));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(57));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(57));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 5);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -2.400000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 110.179169, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 2.400000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], -96.977722, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 2.400000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -64.073608, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, 1.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, 0.000000, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=60;Time=154152.188;phase=0");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadInstance_64(Neuro::DicomInstancesCollection& target)
{
  // ('dcm_qa', 'In/TotalReadoutTime/PF_PAT_POS_PFOV_PEres_INTERP_test2/NOPF_NOPAT_NOPOS_PERES75_ES0P59_BW2222_AP_0022/PF_PAT_POS_BW_INTERP_TEST.MR.HEAD_HARMS.0022.0001.2017.09.20.16.16.44.39766.120142416.IMA')
  Orthanc::DicomMap tags;
  tags.SetValue(0x0008, 0x0060, "MR", false);
  tags.SetValue(0x0008, 0x0070, "SIEMENS", false);
  tags.SetValue(0x0020, 0x0013, "1", false);
  tags.SetValue(0x0018, 0x0080, "6100", false);
  tags.SetValue(0x0018, 0x0081, "60", false);
  tags.SetValue(0x0018, 0x1312, "COL", false);
  tags.SetValue(0x0019, 0x1029, "3040.00000001\\0.0\\3142.50000002\\102.50000001\\3242.49999999\\202.50000001\\3345.0\\304.99999999\\3445.00000001\\405.0\\3547.50000002\\507.50000001\\3647.49999999\\607.50000001\\3750.0\\709.99999999\\3850.00000001\\810.0\\3952.50000001\\912.50000001\\4052.49999999\\1012.50000001\\4155.0\\1114.99999999\\4255.0\\1215.0\\4357.50000001\\1317.5\\4457.49999999\\1417.50000001\\4560.0\\1519.99999999\\4660.0\\1622.5\\4762.50000001\\1722.5\\4864.99999999\\1825.00000001\\4965.0\\1925.00000002\\5067.5\\2027.5\\5167.50000001\\2127.5\\5269.99999999\\2230.00000001\\5370.0\\2330.00000002\\5472.5\\2432.5\\5572.50000001\\2532.5\\5675.00000002\\2635.00000001\\5774.99999999\\2735.00000002\\5877.5\\2837.49999999\\5977.50000001\\2937.5", false);
  tags.SetValue(0x0008, 0x0016, "1.2.840.10008.5.1.4.1.1.4", false);
  tags.SetValue(0x0028, 0x0002, "1", false);
  tags.SetValue(0x0028, 0x0004, "MONOCHROME2", false);
  tags.SetValue(0x0028, 0x0010, "720", false);
  tags.SetValue(0x0028, 0x0011, "720", false);
  tags.SetValue(0x0028, 0x0100, "16", false);
  tags.SetValue(0x0028, 0x0101, "16", false);
  tags.SetValue(0x0028, 0x0102, "15", false);
  tags.SetValue(0x0028, 0x0103, "0", false);
  tags.SetValue(0x0008, 0x0032, "154329.567500", false);
  tags.SetValue(0x0018, 0x0050, "2.4000000953674", false);
  tags.SetValue(0x0018, 0x0088, "2.3999999741376", false);
  tags.SetValue(0x0020, 0x0032, "-866.17921113968\\-872.62231063843\\-64.073608398438", false);
  tags.SetValue(0x0020, 0x0037, "1\\0\\0\\0\\1\\0", false);
  tags.SetValue(0x0028, 0x0030, "2.4000000953674\\2.4000000953674", false);
  std::unique_ptr<Neuro::InputDicomInstance> dicom(new Neuro::InputDicomInstance(tags));
  dicom->GetCSAHeader().AddTag("NumberOfImagesInMosaic", "US").AddValue("60");
  dicom->GetCSAHeader().AddTag("PhaseEncodingDirectionPositive", "IS").AddValue("1");
  dicom->GetCSAHeader().AddTag("SliceNormalVector", "FD").AddValue("0.0").AddValue("0.0").AddValue("1.0");
  target.AddInstance(dicom.release(), "nope");
}

TEST(Autogenerated, Test_64)
{
  Neuro::DicomInstancesCollection instances;
  LoadInstance_64(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 90);
  ASSERT_NEAR(nifti.pixdim[1], 2.400000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 90);
  ASSERT_NEAR(nifti.pixdim[2], 2.400000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 60);
  ASSERT_NEAR(nifti.pixdim[3], 2.400000, 0.00001);
  ASSERT_EQ(nifti.ndim, 3);
  ASSERT_EQ(nifti.dim[0], 3);
  ASSERT_EQ(nifti.nvox, 486000u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(57));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(57));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(57));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 5);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -2.400000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 110.179169, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 2.400000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], -96.977722, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 2.400000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -64.073608, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, 1.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, 0.000000, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=60;Time=154329.568;phase=1");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadInstance_65(Neuro::DicomInstancesCollection& target)
{
  // ('dcm_qa', 'In/TotalReadoutTime/PF_PAT_POS_PFOV_PEres_INTERP_test2/NOPF_NOPAT_NOPOS_PERES75_ES0P59_BW2222_PA_0023/PF_PAT_POS_BW_INTERP_TEST.MR.HEAD_HARMS.0023.0001.2017.09.20.16.16.44.39766.120147538.IMA')
  Orthanc::DicomMap tags;
  tags.SetValue(0x0008, 0x0060, "MR", false);
  tags.SetValue(0x0008, 0x0070, "SIEMENS", false);
  tags.SetValue(0x0020, 0x0013, "1", false);
  tags.SetValue(0x0018, 0x0080, "6100", false);
  tags.SetValue(0x0018, 0x0081, "60", false);
  tags.SetValue(0x0018, 0x1312, "COL", false);
  tags.SetValue(0x0019, 0x1029, "3040.00000001\\0.0\\3142.50000002\\102.50000001\\3242.49999999\\202.50000001\\3345.0\\304.99999999\\3445.00000001\\407.5\\3547.50000002\\507.50000001\\3649.99999999\\610.00000002\\3750.0\\709.99999999\\3852.50000001\\812.5\\3952.50000001\\912.50000001\\4054.99999999\\1015.00000001\\4155.0\\1115.00000002\\4257.50000001\\1217.5\\4357.50000001\\1317.5\\4459.99999999\\1420.00000001\\4560.0\\1520.00000002\\4662.50000001\\1622.5\\4762.50000001\\1722.5\\4865.00000002\\1825.00000001\\4965.0\\1927.50000002\\5067.5\\2027.5\\5170.00000001\\2130.0\\5270.00000002\\2230.00000001\\5372.5\\2332.50000002\\5472.5\\2432.5\\5575.00000001\\2535.0\\5675.00000002\\2635.00000001\\5777.5\\2737.50000002\\5877.5\\2837.49999999\\5980.00000001\\2940.0", false);
  tags.SetValue(0x0008, 0x0016, "1.2.840.10008.5.1.4.1.1.4", false);
  tags.SetValue(0x0028, 0x0002, "1", false);
  tags.SetValue(0x0028, 0x0004, "MONOCHROME2", false);
  tags.SetValue(0x0028, 0x0010, "720", false);
  tags.SetValue(0x0028, 0x0011, "720", false);
  tags.SetValue(0x0028, 0x0100, "16", false);
  tags.SetValue(0x0028, 0x0101, "16", false);
  tags.SetValue(0x0028, 0x0102, "15", false);
  tags.SetValue(0x0028, 0x0103, "0", false);
  tags.SetValue(0x0008, 0x0032, "154358.315000", false);
  tags.SetValue(0x0018, 0x0050, "2.4000000953674", false);
  tags.SetValue(0x0018, 0x0088, "2.3999999741376", false);
  tags.SetValue(0x0020, 0x0032, "-866.17921113968\\-872.62231063843\\-64.073608398438", false);
  tags.SetValue(0x0020, 0x0037, "1\\0\\0\\0\\1\\0", false);
  tags.SetValue(0x0028, 0x0030, "2.4000000953674\\2.4000000953674", false);
  std::unique_ptr<Neuro::InputDicomInstance> dicom(new Neuro::InputDicomInstance(tags));
  dicom->GetCSAHeader().AddTag("NumberOfImagesInMosaic", "US").AddValue("60");
  dicom->GetCSAHeader().AddTag("PhaseEncodingDirectionPositive", "IS").AddValue("0");
  dicom->GetCSAHeader().AddTag("SliceNormalVector", "FD").AddValue("0.0").AddValue("0.0").AddValue("1.0");
  target.AddInstance(dicom.release(), "nope");
}

TEST(Autogenerated, Test_65)
{
  Neuro::DicomInstancesCollection instances;
  LoadInstance_65(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 90);
  ASSERT_NEAR(nifti.pixdim[1], 2.400000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 90);
  ASSERT_NEAR(nifti.pixdim[2], 2.400000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 60);
  ASSERT_NEAR(nifti.pixdim[3], 2.400000, 0.00001);
  ASSERT_EQ(nifti.ndim, 3);
  ASSERT_EQ(nifti.dim[0], 3);
  ASSERT_EQ(nifti.nvox, 486000u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(57));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(57));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(57));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 5);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -2.400000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 110.179169, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 2.400000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], -96.977722, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 2.400000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -64.073608, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, 1.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, 0.000000, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=60;Time=154358.315;phase=0");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadInstance_66(Neuro::DicomInstancesCollection& target)
{
  // ('dcm_qa', 'In/TotalReadoutTime/PF_PAT_POS_PFOV_PEres_INTERP_test2/NOPF_P2_NOPOS_PERES100_ES0P59_BW2222_AP_0010/PF_PAT_POS_BW_INTERP_TEST.MR.HEAD_HARMS.0010.0001.2017.09.20.16.16.44.39766.120093240.IMA')
  Orthanc::DicomMap tags;
  tags.SetValue(0x0008, 0x0060, "MR", false);
  tags.SetValue(0x0008, 0x0070, "SIEMENS", false);
  tags.SetValue(0x0020, 0x0013, "1", false);
  tags.SetValue(0x0018, 0x0080, "6100", false);
  tags.SetValue(0x0018, 0x0081, "60", false);
  tags.SetValue(0x0018, 0x1312, "COL", false);
  tags.SetValue(0x0019, 0x1029, "3040.00000001\\0.0\\3140.00000001\\100.00000001\\3242.49999999\\202.50000001\\3342.5\\302.50000002\\3445.00000001\\405.0\\3545.00000001\\505.0\\3647.49999999\\607.50000001\\3750.0\\709.99999999\\3850.00000001\\810.0\\3952.50000001\\912.50000001\\4052.50000002\\1012.50000001\\4155.0\\1115.00000002\\4255.0\\1215.0\\4357.50000001\\1317.5\\4457.50000002\\1417.50000001\\4560.0\\1520.00000002\\4660.0\\1620.0\\4762.50000001\\1722.5\\4862.50000002\\1822.50000001\\4965.0\\1925.00000002\\5065.0\\2024.99999999\\5167.50000001\\2127.5\\5267.50000002\\2230.00000001\\5370.0\\2330.00000002\\5472.5\\2432.5\\5572.50000001\\2532.5\\5675.00000002\\2635.00000001\\5774.99999999\\2735.00000002\\5877.5\\2837.49999999\\5977.50000001\\2937.5", false);
  tags.SetValue(0x0008, 0x0016, "1.2.840.10008.5.1.4.1.1.4", false);
  tags.SetValue(0x0028, 0x0002, "1", false);
  tags.SetValue(0x0028, 0x0004, "MONOCHROME2", false);
  tags.SetValue(0x0028, 0x0010, "720", false);
  tags.SetValue(0x0028, 0x0011, "720", false);
  tags.SetValue(0x0028, 0x0100, "16", false);
  tags.SetValue(0x0028, 0x0101, "16", false);
  tags.SetValue(0x0028, 0x0102, "15", false);
  tags.SetValue(0x0028, 0x0103, "0", false);
  tags.SetValue(0x0008, 0x0032, "152737.565000", false);
  tags.SetValue(0x0018, 0x0050, "2.4000000953674", false);
  tags.SetValue(0x0018, 0x0088, "2.3999999741376", false);
  tags.SetValue(0x0020, 0x0032, "-866.17921113968\\-872.62231063843\\-64.073608398438", false);
  tags.SetValue(0x0020, 0x0037, "1\\0\\0\\0\\1\\0", false);
  tags.SetValue(0x0028, 0x0030, "2.4000000953674\\2.4000000953674", false);
  std::unique_ptr<Neuro::InputDicomInstance> dicom(new Neuro::InputDicomInstance(tags));
  dicom->GetCSAHeader().AddTag("NumberOfImagesInMosaic", "US").AddValue("60");
  dicom->GetCSAHeader().AddTag("PhaseEncodingDirectionPositive", "IS").AddValue("1");
  dicom->GetCSAHeader().AddTag("SliceNormalVector", "FD").AddValue("0.0").AddValue("0.0").AddValue("1.0");
  target.AddInstance(dicom.release(), "nope");
}

TEST(Autogenerated, Test_66)
{
  Neuro::DicomInstancesCollection instances;
  LoadInstance_66(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 90);
  ASSERT_NEAR(nifti.pixdim[1], 2.400000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 90);
  ASSERT_NEAR(nifti.pixdim[2], 2.400000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 60);
  ASSERT_NEAR(nifti.pixdim[3], 2.400000, 0.00001);
  ASSERT_EQ(nifti.ndim, 3);
  ASSERT_EQ(nifti.dim[0], 3);
  ASSERT_EQ(nifti.nvox, 486000u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(57));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(57));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(57));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 5);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -2.400000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 110.179169, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 2.400000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], -96.977722, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 2.400000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -64.073608, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, 1.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, 0.000000, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=60;Time=152737.565;phase=1");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadInstance_67(Neuro::DicomInstancesCollection& target)
{
  // ('dcm_qa', 'In/TotalReadoutTime/PF_PAT_POS_PFOV_PEres_INTERP_test2/NOPF_P2_NOPOS_PERES100_ES0P59_BW2222_PA_0011/PF_PAT_POS_BW_INTERP_TEST.MR.HEAD_HARMS.0011.0001.2017.09.20.16.16.44.39766.120097338.IMA')
  Orthanc::DicomMap tags;
  tags.SetValue(0x0008, 0x0060, "MR", false);
  tags.SetValue(0x0008, 0x0070, "SIEMENS", false);
  tags.SetValue(0x0020, 0x0013, "1", false);
  tags.SetValue(0x0018, 0x0080, "6100", false);
  tags.SetValue(0x0018, 0x0081, "60", false);
  tags.SetValue(0x0018, 0x1312, "COL", false);
  tags.SetValue(0x0019, 0x1029, "3040.00000001\\0.0\\3142.50000002\\102.50000001\\3242.49999999\\202.50000001\\3345.0\\304.99999999\\3445.00000001\\405.0\\3547.50000002\\507.50000001\\3647.50000002\\610.00000002\\3750.0\\710.00000002\\3852.50000001\\812.5\\3952.50000001\\912.50000001\\4054.99999999\\1015.00000001\\4155.0\\1115.00000002\\4257.50000001\\1217.5\\4357.50000001\\1317.5\\4460.00000002\\1420.00000001\\4560.0\\1520.00000002\\4662.50000001\\1622.5\\4762.50000001\\1722.5\\4865.00000002\\1825.00000001\\4965.0\\1925.00000002\\5067.5\\2027.5\\5167.50000001\\2127.5\\5270.00000002\\2230.00000001\\5372.5\\2332.50000002\\5472.5\\2432.5\\5575.00000001\\2535.0\\5675.00000002\\2635.00000001\\5777.5\\2737.50000002\\5877.5\\2837.49999999\\5980.00000001\\2940.0", false);
  tags.SetValue(0x0008, 0x0016, "1.2.840.10008.5.1.4.1.1.4", false);
  tags.SetValue(0x0028, 0x0002, "1", false);
  tags.SetValue(0x0028, 0x0004, "MONOCHROME2", false);
  tags.SetValue(0x0028, 0x0010, "720", false);
  tags.SetValue(0x0028, 0x0011, "720", false);
  tags.SetValue(0x0028, 0x0100, "16", false);
  tags.SetValue(0x0028, 0x0101, "16", false);
  tags.SetValue(0x0028, 0x0102, "15", false);
  tags.SetValue(0x0028, 0x0103, "0", false);
  tags.SetValue(0x0008, 0x0032, "152834.595000", false);
  tags.SetValue(0x0018, 0x0050, "2.4000000953674", false);
  tags.SetValue(0x0018, 0x0088, "2.3999999741376", false);
  tags.SetValue(0x0020, 0x0032, "-866.17921113968\\-872.62231063843\\-64.073608398438", false);
  tags.SetValue(0x0020, 0x0037, "1\\0\\0\\0\\1\\0", false);
  tags.SetValue(0x0028, 0x0030, "2.4000000953674\\2.4000000953674", false);
  std::unique_ptr<Neuro::InputDicomInstance> dicom(new Neuro::InputDicomInstance(tags));
  dicom->GetCSAHeader().AddTag("NumberOfImagesInMosaic", "US").AddValue("60");
  dicom->GetCSAHeader().AddTag("PhaseEncodingDirectionPositive", "IS").AddValue("0");
  dicom->GetCSAHeader().AddTag("SliceNormalVector", "FD").AddValue("0.0").AddValue("0.0").AddValue("1.0");
  target.AddInstance(dicom.release(), "nope");
}

TEST(Autogenerated, Test_67)
{
  Neuro::DicomInstancesCollection instances;
  LoadInstance_67(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 90);
  ASSERT_NEAR(nifti.pixdim[1], 2.400000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 90);
  ASSERT_NEAR(nifti.pixdim[2], 2.400000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 60);
  ASSERT_NEAR(nifti.pixdim[3], 2.400000, 0.00001);
  ASSERT_EQ(nifti.ndim, 3);
  ASSERT_EQ(nifti.dim[0], 3);
  ASSERT_EQ(nifti.nvox, 486000u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(57));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(57));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(57));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 5);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -2.400000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 110.179169, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 2.400000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], -96.977722, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 2.400000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -64.073608, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, 1.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, 0.000000, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=60;Time=152834.595;phase=0");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadInstance_68(Neuro::DicomInstancesCollection& target)
{
  // ('dcm_qa', 'In/TotalReadoutTime/PF_PAT_POS_PFOV_PEres_INTERP_test2/NOPF_P4_NOPOS_PERES100_ES0P59_BW2222_AP_0032/PF_PAT_POS_BW_INTERP_TEST.MR.HEAD_HARMS.0032.0001.2017.09.20.16.16.44.39766.120186472.IMA')
  Orthanc::DicomMap tags;
  tags.SetValue(0x0008, 0x0060, "MR", false);
  tags.SetValue(0x0008, 0x0070, "SIEMENS", false);
  tags.SetValue(0x0020, 0x0013, "1", false);
  tags.SetValue(0x0018, 0x0080, "6100", false);
  tags.SetValue(0x0018, 0x0081, "60", false);
  tags.SetValue(0x0018, 0x1312, "COL", false);
  tags.SetValue(0x0019, 0x1029, "3040.00000001\\0.0\\3140.00000001\\100.00000001\\3242.50000002\\202.50000001\\3342.5\\302.50000002\\3445.00000001\\405.0\\3545.00000001\\505.0\\3647.50000002\\607.50000001\\3747.50000003\\707.50000002\\3850.00000001\\810.00000003\\3950.00000001\\910.0\\4052.50000002\\1012.50000001\\4152.50000003\\1112.50000002\\4255.0\\1215.00000003\\4357.50000001\\1317.5\\4457.50000002\\1417.50000001\\4560.00000003\\1520.00000002\\4660.0\\1620.00000002\\4762.50000001\\1722.5\\4862.50000002\\1822.50000001\\4965.00000003\\1925.00000002\\5065.0\\2025.00000002\\5167.50000001\\2127.5\\5267.50000002\\2227.50000001\\5370.00000002\\2330.00000002\\5470.0\\2430.00000002\\5572.50000001\\2532.5\\5672.50000002\\2632.50000001\\5775.00000002\\2735.00000002\\5877.5\\2837.50000002\\5977.50000001\\2937.5", false);
  tags.SetValue(0x0008, 0x0016, "1.2.840.10008.5.1.4.1.1.4", false);
  tags.SetValue(0x0028, 0x0002, "1", false);
  tags.SetValue(0x0028, 0x0004, "MONOCHROME2", false);
  tags.SetValue(0x0028, 0x0010, "720", false);
  tags.SetValue(0x0028, 0x0011, "720", false);
  tags.SetValue(0x0028, 0x0100, "16", false);
  tags.SetValue(0x0028, 0x0101, "16", false);
  tags.SetValue(0x0028, 0x0102, "15", false);
  tags.SetValue(0x0028, 0x0103, "0", false);
  tags.SetValue(0x0008, 0x0032, "160241.307500", false);
  tags.SetValue(0x0018, 0x0050, "2.4000000953674", false);
  tags.SetValue(0x0018, 0x0088, "2.3999999741376", false);
  tags.SetValue(0x0020, 0x0032, "-866.17921113968\\-872.62231063843\\-64.073608398438", false);
  tags.SetValue(0x0020, 0x0037, "1\\0\\0\\0\\1\\0", false);
  tags.SetValue(0x0028, 0x0030, "2.4000000953674\\2.4000000953674", false);
  std::unique_ptr<Neuro::InputDicomInstance> dicom(new Neuro::InputDicomInstance(tags));
  dicom->GetCSAHeader().AddTag("NumberOfImagesInMosaic", "US").AddValue("60");
  dicom->GetCSAHeader().AddTag("PhaseEncodingDirectionPositive", "IS").AddValue("1");
  dicom->GetCSAHeader().AddTag("SliceNormalVector", "FD").AddValue("0.0").AddValue("0.0").AddValue("1.0");
  target.AddInstance(dicom.release(), "nope");
}

TEST(Autogenerated, Test_68)
{
  Neuro::DicomInstancesCollection instances;
  LoadInstance_68(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 90);
  ASSERT_NEAR(nifti.pixdim[1], 2.400000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 90);
  ASSERT_NEAR(nifti.pixdim[2], 2.400000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 60);
  ASSERT_NEAR(nifti.pixdim[3], 2.400000, 0.00001);
  ASSERT_EQ(nifti.ndim, 3);
  ASSERT_EQ(nifti.dim[0], 3);
  ASSERT_EQ(nifti.nvox, 486000u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(57));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(57));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(57));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 5);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -2.400000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 110.179169, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 2.400000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], -96.977722, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 2.400000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -64.073608, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, 1.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, 0.000000, 0.0001);
  ASSERT_TRUE(std::string(nifti.descrip) == "TE=60;Time=160241.307;phase=1" ||
              std::string(nifti.descrip) == "TE=60;Time=160241.308;phase=1");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadInstance_69(Neuro::DicomInstancesCollection& target)
{
  // ('dcm_qa', 'In/TotalReadoutTime/PF_PAT_POS_PFOV_PEres_INTERP_test2/NOPF_P4_NOPOS_PERES100_ES0P59_BW2222_PA_0033/PF_PAT_POS_BW_INTERP_TEST.MR.HEAD_HARMS.0033.0001.2017.09.20.16.16.44.39766.120190570.IMA')
  Orthanc::DicomMap tags;
  tags.SetValue(0x0008, 0x0060, "MR", false);
  tags.SetValue(0x0008, 0x0070, "SIEMENS", false);
  tags.SetValue(0x0020, 0x0013, "1", false);
  tags.SetValue(0x0018, 0x0080, "6100", false);
  tags.SetValue(0x0018, 0x0081, "60", false);
  tags.SetValue(0x0018, 0x1312, "COL", false);
  tags.SetValue(0x0019, 0x1029, "3040.00000001\\0.0\\3140.00000001\\100.00000001\\3242.49999999\\202.50000001\\3342.5\\302.49999999\\3445.00000001\\405.0\\3545.00000001\\505.0\\3647.49999999\\607.50000001\\3747.5\\707.49999999\\3850.00000001\\810.0\\3952.50000001\\912.50000001\\4052.49999999\\1012.50000001\\4155.0\\1114.99999999\\4255.0\\1215.0\\4357.50000001\\1317.5\\4457.49999999\\1417.50000001\\4560.0\\1519.99999999\\4660.0\\1620.0\\4762.50000001\\1722.5\\4862.49999999\\1822.50000001\\4965.0\\1924.99999999\\5065.0\\2024.99999999\\5167.50000001\\2127.5\\5267.50000002\\2227.50000001\\5370.0\\2330.00000002\\5472.5\\2432.5\\5572.50000001\\2532.5\\5674.99999999\\2635.00000001\\5774.99999999\\2735.00000002\\5877.5\\2837.49999999\\5977.50000001\\2937.5", false);
  tags.SetValue(0x0008, 0x0016, "1.2.840.10008.5.1.4.1.1.4", false);
  tags.SetValue(0x0028, 0x0002, "1", false);
  tags.SetValue(0x0028, 0x0004, "MONOCHROME2", false);
  tags.SetValue(0x0028, 0x0010, "720", false);
  tags.SetValue(0x0028, 0x0011, "720", false);
  tags.SetValue(0x0028, 0x0100, "16", false);
  tags.SetValue(0x0028, 0x0101, "16", false);
  tags.SetValue(0x0028, 0x0102, "15", false);
  tags.SetValue(0x0028, 0x0103, "0", false);
  tags.SetValue(0x0008, 0x0032, "160410.412500", false);
  tags.SetValue(0x0018, 0x0050, "2.4000000953674", false);
  tags.SetValue(0x0018, 0x0088, "2.3999999741376", false);
  tags.SetValue(0x0020, 0x0032, "-866.17921113968\\-872.62231063843\\-64.073608398438", false);
  tags.SetValue(0x0020, 0x0037, "1\\0\\0\\0\\1\\0", false);
  tags.SetValue(0x0028, 0x0030, "2.4000000953674\\2.4000000953674", false);
  std::unique_ptr<Neuro::InputDicomInstance> dicom(new Neuro::InputDicomInstance(tags));
  dicom->GetCSAHeader().AddTag("NumberOfImagesInMosaic", "US").AddValue("60");
  dicom->GetCSAHeader().AddTag("PhaseEncodingDirectionPositive", "IS").AddValue("0");
  dicom->GetCSAHeader().AddTag("SliceNormalVector", "FD").AddValue("0.0").AddValue("0.0").AddValue("1.0");
  target.AddInstance(dicom.release(), "nope");
}

TEST(Autogenerated, Test_69)
{
  Neuro::DicomInstancesCollection instances;
  LoadInstance_69(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 90);
  ASSERT_NEAR(nifti.pixdim[1], 2.400000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 90);
  ASSERT_NEAR(nifti.pixdim[2], 2.400000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 60);
  ASSERT_NEAR(nifti.pixdim[3], 2.400000, 0.00001);
  ASSERT_EQ(nifti.ndim, 3);
  ASSERT_EQ(nifti.dim[0], 3);
  ASSERT_EQ(nifti.nvox, 486000u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(57));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(57));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(57));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 5);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -2.400000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 110.179169, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 2.400000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], -96.977722, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 2.400000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -64.073608, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, 1.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, 0.000000, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=60;Time=160410.413;phase=0");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadInstance_70(Neuro::DicomInstancesCollection& target)
{
  // ('dcm_qa_nih', 'In/20180918GE/mr_0004/axial_epi_fmri_interleaved_i_to_s-00001.dcm')
  Orthanc::DicomMap tags;
  tags.SetValue(0x0008, 0x0060, "MR", false);
  tags.SetValue(0x0008, 0x0070, "GE MEDICAL SYSTEMS", false);
  tags.SetValue(0x0020, 0x0013, "1", false);
  tags.SetValue(0x0018, 0x0080, "5000", false);
  tags.SetValue(0x0018, 0x0081, "30", false);
  tags.SetValue(0x0018, 0x1312, "COL", false);
  tags.SetValue(0x0008, 0x0016, "1.2.840.10008.5.1.4.1.1.4", false);
  tags.SetValue(0x0028, 0x0002, "1", false);
  tags.SetValue(0x0028, 0x0004, "MONOCHROME2", false);
  tags.SetValue(0x0028, 0x0010, "64", false);
  tags.SetValue(0x0028, 0x0011, "64", false);
  tags.SetValue(0x0028, 0x0100, "16", false);
  tags.SetValue(0x0028, 0x0101, "16", false);
  tags.SetValue(0x0028, 0x0102, "15", false);
  tags.SetValue(0x0028, 0x0103, "1", false);
  tags.SetValue(0x0008, 0x0032, "114815", false);
  tags.SetValue(0x0018, 0x0050, "3", false);
  tags.SetValue(0x0018, 0x0088, "5", false);
  tags.SetValue(0x0020, 0x0032, "-118.125\\-143.344\\-17.6875", false);
  tags.SetValue(0x0020, 0x0037, "1.00000\\-0.00000\\0.00000\\-0.00000\\1.00000\\0.00000", false);
  tags.SetValue(0x0028, 0x0030, "3.75\\3.75", false);
  std::unique_ptr<Neuro::InputDicomInstance> dicom(new Neuro::InputDicomInstance(tags));
  target.AddInstance(dicom.release(), "nope");
}

TEST(Autogenerated, Test_70)
{
  Neuro::DicomInstancesCollection instances;
  LoadInstance_70(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 64);
  ASSERT_NEAR(nifti.pixdim[1], 3.750000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 64);
  ASSERT_NEAR(nifti.pixdim[2], 3.750000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 1);
  ASSERT_NEAR(nifti.pixdim[3], 5.000000, 0.00001);
  ASSERT_EQ(nifti.ndim, 3);
  ASSERT_EQ(nifti.dim[0], 3);
  ASSERT_EQ(nifti.nvox, 4096u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 4);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(57));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(57));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(57));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -3.750000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 118.125000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 3.750000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], -92.906006, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 5.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -17.687500, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, 1.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, 0.000000, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=30;Time=114815.000");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadInstance_71(Neuro::DicomInstancesCollection& target)
{
  // ('dcm_qa_nih', 'In/20180918GE/mr_0004/axial_epi_fmri_interleaved_i_to_s-00002.dcm')
  Orthanc::DicomMap tags;
  tags.SetValue(0x0008, 0x0060, "MR", false);
  tags.SetValue(0x0008, 0x0070, "GE MEDICAL SYSTEMS", false);
  tags.SetValue(0x0020, 0x0013, "2", false);
  tags.SetValue(0x0018, 0x0080, "5000", false);
  tags.SetValue(0x0018, 0x0081, "30", false);
  tags.SetValue(0x0018, 0x1312, "COL", false);
  tags.SetValue(0x0008, 0x0016, "1.2.840.10008.5.1.4.1.1.4", false);
  tags.SetValue(0x0028, 0x0002, "1", false);
  tags.SetValue(0x0028, 0x0004, "MONOCHROME2", false);
  tags.SetValue(0x0028, 0x0010, "64", false);
  tags.SetValue(0x0028, 0x0011, "64", false);
  tags.SetValue(0x0028, 0x0100, "16", false);
  tags.SetValue(0x0028, 0x0101, "16", false);
  tags.SetValue(0x0028, 0x0102, "15", false);
  tags.SetValue(0x0028, 0x0103, "1", false);
  tags.SetValue(0x0008, 0x0032, "114815", false);
  tags.SetValue(0x0018, 0x0050, "3", false);
  tags.SetValue(0x0018, 0x0088, "5", false);
  tags.SetValue(0x0020, 0x0032, "-118.125\\-143.344\\-12.6875", false);
  tags.SetValue(0x0020, 0x0037, "1.00000\\-0.00000\\0.00000\\-0.00000\\1.00000\\0.00000", false);
  tags.SetValue(0x0028, 0x0030, "3.75\\3.75", false);
  std::unique_ptr<Neuro::InputDicomInstance> dicom(new Neuro::InputDicomInstance(tags));
  target.AddInstance(dicom.release(), "nope");
}

TEST(Autogenerated, Test_71)
{
  Neuro::DicomInstancesCollection instances;
  LoadInstance_71(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 64);
  ASSERT_NEAR(nifti.pixdim[1], 3.750000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 64);
  ASSERT_NEAR(nifti.pixdim[2], 3.750000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 1);
  ASSERT_NEAR(nifti.pixdim[3], 5.000000, 0.00001);
  ASSERT_EQ(nifti.ndim, 3);
  ASSERT_EQ(nifti.dim[0], 3);
  ASSERT_EQ(nifti.nvox, 4096u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 4);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(57));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(57));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(57));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -3.750000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 118.125000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 3.750000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], -92.906006, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 5.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -12.687500, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, 1.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, 0.000000, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=30;Time=114815.000");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadInstance_72(Neuro::DicomInstancesCollection& target)
{
  // ('dcm_qa_nih', 'In/20180918GE/mr_0005/axial_epi_fmri_sequential_i_to_s-00001.dcm')
  Orthanc::DicomMap tags;
  tags.SetValue(0x0008, 0x0060, "MR", false);
  tags.SetValue(0x0008, 0x0070, "GE MEDICAL SYSTEMS", false);
  tags.SetValue(0x0020, 0x0013, "1", false);
  tags.SetValue(0x0018, 0x0080, "5000", false);
  tags.SetValue(0x0018, 0x0081, "30", false);
  tags.SetValue(0x0018, 0x1312, "COL", false);
  tags.SetValue(0x0008, 0x0016, "1.2.840.10008.5.1.4.1.1.4", false);
  tags.SetValue(0x0028, 0x0002, "1", false);
  tags.SetValue(0x0028, 0x0004, "MONOCHROME2", false);
  tags.SetValue(0x0028, 0x0010, "64", false);
  tags.SetValue(0x0028, 0x0011, "64", false);
  tags.SetValue(0x0028, 0x0100, "16", false);
  tags.SetValue(0x0028, 0x0101, "16", false);
  tags.SetValue(0x0028, 0x0102, "15", false);
  tags.SetValue(0x0028, 0x0103, "1", false);
  tags.SetValue(0x0008, 0x0032, "115224", false);
  tags.SetValue(0x0018, 0x0050, "3", false);
  tags.SetValue(0x0018, 0x0088, "5", false);
  tags.SetValue(0x0020, 0x0032, "-118.125\\-143.344\\-17.6875", false);
  tags.SetValue(0x0020, 0x0037, "1.00000\\-0.00000\\0.00000\\-0.00000\\1.00000\\0.00000", false);
  tags.SetValue(0x0028, 0x0030, "3.75\\3.75", false);
  std::unique_ptr<Neuro::InputDicomInstance> dicom(new Neuro::InputDicomInstance(tags));
  target.AddInstance(dicom.release(), "nope");
}

TEST(Autogenerated, Test_72)
{
  Neuro::DicomInstancesCollection instances;
  LoadInstance_72(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 64);
  ASSERT_NEAR(nifti.pixdim[1], 3.750000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 64);
  ASSERT_NEAR(nifti.pixdim[2], 3.750000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 1);
  ASSERT_NEAR(nifti.pixdim[3], 5.000000, 0.00001);
  ASSERT_EQ(nifti.ndim, 3);
  ASSERT_EQ(nifti.dim[0], 3);
  ASSERT_EQ(nifti.nvox, 4096u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 4);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(57));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(57));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(57));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -3.750000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 118.125000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 3.750000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], -92.906006, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 5.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -17.687500, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, 1.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, 0.000000, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=30;Time=115224.000");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadInstance_73(Neuro::DicomInstancesCollection& target)
{
  // ('dcm_qa_nih', 'In/20180918GE/mr_0005/axial_epi_fmri_sequential_i_to_s-00002.dcm')
  Orthanc::DicomMap tags;
  tags.SetValue(0x0008, 0x0060, "MR", false);
  tags.SetValue(0x0008, 0x0070, "GE MEDICAL SYSTEMS", false);
  tags.SetValue(0x0020, 0x0013, "2", false);
  tags.SetValue(0x0018, 0x0080, "5000", false);
  tags.SetValue(0x0018, 0x0081, "30", false);
  tags.SetValue(0x0018, 0x1312, "COL", false);
  tags.SetValue(0x0008, 0x0016, "1.2.840.10008.5.1.4.1.1.4", false);
  tags.SetValue(0x0028, 0x0002, "1", false);
  tags.SetValue(0x0028, 0x0004, "MONOCHROME2", false);
  tags.SetValue(0x0028, 0x0010, "64", false);
  tags.SetValue(0x0028, 0x0011, "64", false);
  tags.SetValue(0x0028, 0x0100, "16", false);
  tags.SetValue(0x0028, 0x0101, "16", false);
  tags.SetValue(0x0028, 0x0102, "15", false);
  tags.SetValue(0x0028, 0x0103, "1", false);
  tags.SetValue(0x0008, 0x0032, "115224", false);
  tags.SetValue(0x0018, 0x0050, "3", false);
  tags.SetValue(0x0018, 0x0088, "5", false);
  tags.SetValue(0x0020, 0x0032, "-118.125\\-143.344\\-12.6875", false);
  tags.SetValue(0x0020, 0x0037, "1.00000\\-0.00000\\0.00000\\-0.00000\\1.00000\\0.00000", false);
  tags.SetValue(0x0028, 0x0030, "3.75\\3.75", false);
  std::unique_ptr<Neuro::InputDicomInstance> dicom(new Neuro::InputDicomInstance(tags));
  target.AddInstance(dicom.release(), "nope");
}

TEST(Autogenerated, Test_73)
{
  Neuro::DicomInstancesCollection instances;
  LoadInstance_73(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 64);
  ASSERT_NEAR(nifti.pixdim[1], 3.750000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 64);
  ASSERT_NEAR(nifti.pixdim[2], 3.750000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 1);
  ASSERT_NEAR(nifti.pixdim[3], 5.000000, 0.00001);
  ASSERT_EQ(nifti.ndim, 3);
  ASSERT_EQ(nifti.dim[0], 3);
  ASSERT_EQ(nifti.nvox, 4096u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 4);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(57));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(57));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(57));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -3.750000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 118.125000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 3.750000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], -92.906006, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 5.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -12.687500, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, 1.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, 0.000000, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=30;Time=115224.000");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadInstance_74(Neuro::DicomInstancesCollection& target)
{
  // ('dcm_qa_nih', 'In/20180918GE/mr_0006/axial_epi_fmri_interleaved_s_to_i-00001.dcm')
  Orthanc::DicomMap tags;
  tags.SetValue(0x0008, 0x0060, "MR", false);
  tags.SetValue(0x0008, 0x0070, "GE MEDICAL SYSTEMS", false);
  tags.SetValue(0x0020, 0x0013, "1", false);
  tags.SetValue(0x0018, 0x0080, "5000", false);
  tags.SetValue(0x0018, 0x0081, "30", false);
  tags.SetValue(0x0018, 0x1312, "COL", false);
  tags.SetValue(0x0008, 0x0016, "1.2.840.10008.5.1.4.1.1.4", false);
  tags.SetValue(0x0028, 0x0002, "1", false);
  tags.SetValue(0x0028, 0x0004, "MONOCHROME2", false);
  tags.SetValue(0x0028, 0x0010, "64", false);
  tags.SetValue(0x0028, 0x0011, "64", false);
  tags.SetValue(0x0028, 0x0100, "16", false);
  tags.SetValue(0x0028, 0x0101, "16", false);
  tags.SetValue(0x0028, 0x0102, "15", false);
  tags.SetValue(0x0028, 0x0103, "1", false);
  tags.SetValue(0x0008, 0x0032, "115418", false);
  tags.SetValue(0x0018, 0x0050, "3", false);
  tags.SetValue(0x0018, 0x0088, "5", false);
  tags.SetValue(0x0020, 0x0032, "-118.125\\-143.344\\52.3125", false);
  tags.SetValue(0x0020, 0x0037, "1.00000\\-0.00000\\0.00000\\-0.00000\\1.00000\\0.00000", false);
  tags.SetValue(0x0028, 0x0030, "3.75\\3.75", false);
  std::unique_ptr<Neuro::InputDicomInstance> dicom(new Neuro::InputDicomInstance(tags));
  target.AddInstance(dicom.release(), "nope");
}

TEST(Autogenerated, Test_74)
{
  Neuro::DicomInstancesCollection instances;
  LoadInstance_74(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 64);
  ASSERT_NEAR(nifti.pixdim[1], 3.750000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 64);
  ASSERT_NEAR(nifti.pixdim[2], 3.750000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 1);
  ASSERT_NEAR(nifti.pixdim[3], 5.000000, 0.00001);
  ASSERT_EQ(nifti.ndim, 3);
  ASSERT_EQ(nifti.dim[0], 3);
  ASSERT_EQ(nifti.nvox, 4096u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 4);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(57));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(57));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(57));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -3.750000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 118.125000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 3.750000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], -92.906006, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 5.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], 52.312500, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, 1.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, 0.000000, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=30;Time=115418.000");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadInstance_75(Neuro::DicomInstancesCollection& target)
{
  // ('dcm_qa_nih', 'In/20180918GE/mr_0006/axial_epi_fmri_interleaved_s_to_i-00002.dcm')
  Orthanc::DicomMap tags;
  tags.SetValue(0x0008, 0x0060, "MR", false);
  tags.SetValue(0x0008, 0x0070, "GE MEDICAL SYSTEMS", false);
  tags.SetValue(0x0020, 0x0013, "2", false);
  tags.SetValue(0x0018, 0x0080, "5000", false);
  tags.SetValue(0x0018, 0x0081, "30", false);
  tags.SetValue(0x0018, 0x1312, "COL", false);
  tags.SetValue(0x0008, 0x0016, "1.2.840.10008.5.1.4.1.1.4", false);
  tags.SetValue(0x0028, 0x0002, "1", false);
  tags.SetValue(0x0028, 0x0004, "MONOCHROME2", false);
  tags.SetValue(0x0028, 0x0010, "64", false);
  tags.SetValue(0x0028, 0x0011, "64", false);
  tags.SetValue(0x0028, 0x0100, "16", false);
  tags.SetValue(0x0028, 0x0101, "16", false);
  tags.SetValue(0x0028, 0x0102, "15", false);
  tags.SetValue(0x0028, 0x0103, "1", false);
  tags.SetValue(0x0008, 0x0032, "115418", false);
  tags.SetValue(0x0018, 0x0050, "3", false);
  tags.SetValue(0x0018, 0x0088, "5", false);
  tags.SetValue(0x0020, 0x0032, "-118.125\\-143.344\\47.3125", false);
  tags.SetValue(0x0020, 0x0037, "1.00000\\-0.00000\\0.00000\\-0.00000\\1.00000\\0.00000", false);
  tags.SetValue(0x0028, 0x0030, "3.75\\3.75", false);
  std::unique_ptr<Neuro::InputDicomInstance> dicom(new Neuro::InputDicomInstance(tags));
  target.AddInstance(dicom.release(), "nope");
}

TEST(Autogenerated, Test_75)
{
  Neuro::DicomInstancesCollection instances;
  LoadInstance_75(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 64);
  ASSERT_NEAR(nifti.pixdim[1], 3.750000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 64);
  ASSERT_NEAR(nifti.pixdim[2], 3.750000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 1);
  ASSERT_NEAR(nifti.pixdim[3], 5.000000, 0.00001);
  ASSERT_EQ(nifti.ndim, 3);
  ASSERT_EQ(nifti.dim[0], 3);
  ASSERT_EQ(nifti.nvox, 4096u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 4);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(57));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(57));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(57));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -3.750000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 118.125000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 3.750000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], -92.906006, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 5.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], 47.312500, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, 1.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, 0.000000, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=30;Time=115418.000");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadInstance_76(Neuro::DicomInstancesCollection& target)
{
  // ('dcm_qa_nih', 'In/20180918GE/mr_0007/axial_epi_fmri_sequential_s_to_i-00001.dcm')
  Orthanc::DicomMap tags;
  tags.SetValue(0x0008, 0x0060, "MR", false);
  tags.SetValue(0x0008, 0x0070, "GE MEDICAL SYSTEMS", false);
  tags.SetValue(0x0020, 0x0013, "1", false);
  tags.SetValue(0x0018, 0x0080, "5000", false);
  tags.SetValue(0x0018, 0x0081, "30", false);
  tags.SetValue(0x0018, 0x1312, "COL", false);
  tags.SetValue(0x0008, 0x0016, "1.2.840.10008.5.1.4.1.1.4", false);
  tags.SetValue(0x0028, 0x0002, "1", false);
  tags.SetValue(0x0028, 0x0004, "MONOCHROME2", false);
  tags.SetValue(0x0028, 0x0010, "64", false);
  tags.SetValue(0x0028, 0x0011, "64", false);
  tags.SetValue(0x0028, 0x0100, "16", false);
  tags.SetValue(0x0028, 0x0101, "16", false);
  tags.SetValue(0x0028, 0x0102, "15", false);
  tags.SetValue(0x0028, 0x0103, "1", false);
  tags.SetValue(0x0008, 0x0032, "115551", false);
  tags.SetValue(0x0018, 0x0050, "3", false);
  tags.SetValue(0x0018, 0x0088, "5", false);
  tags.SetValue(0x0020, 0x0032, "-118.125\\-143.344\\52.3125", false);
  tags.SetValue(0x0020, 0x0037, "1.00000\\-0.00000\\0.00000\\-0.00000\\1.00000\\0.00000", false);
  tags.SetValue(0x0028, 0x0030, "3.75\\3.75", false);
  std::unique_ptr<Neuro::InputDicomInstance> dicom(new Neuro::InputDicomInstance(tags));
  target.AddInstance(dicom.release(), "nope");
}

TEST(Autogenerated, Test_76)
{
  Neuro::DicomInstancesCollection instances;
  LoadInstance_76(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 64);
  ASSERT_NEAR(nifti.pixdim[1], 3.750000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 64);
  ASSERT_NEAR(nifti.pixdim[2], 3.750000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 1);
  ASSERT_NEAR(nifti.pixdim[3], 5.000000, 0.00001);
  ASSERT_EQ(nifti.ndim, 3);
  ASSERT_EQ(nifti.dim[0], 3);
  ASSERT_EQ(nifti.nvox, 4096u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 4);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(57));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(57));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(57));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -3.750000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 118.125000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 3.750000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], -92.906006, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 5.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], 52.312500, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, 1.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, 0.000000, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=30;Time=115551.000");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadInstance_77(Neuro::DicomInstancesCollection& target)
{
  // ('dcm_qa_nih', 'In/20180918GE/mr_0007/axial_epi_fmri_sequential_s_to_i-00002.dcm')
  Orthanc::DicomMap tags;
  tags.SetValue(0x0008, 0x0060, "MR", false);
  tags.SetValue(0x0008, 0x0070, "GE MEDICAL SYSTEMS", false);
  tags.SetValue(0x0020, 0x0013, "2", false);
  tags.SetValue(0x0018, 0x0080, "5000", false);
  tags.SetValue(0x0018, 0x0081, "30", false);
  tags.SetValue(0x0018, 0x1312, "COL", false);
  tags.SetValue(0x0008, 0x0016, "1.2.840.10008.5.1.4.1.1.4", false);
  tags.SetValue(0x0028, 0x0002, "1", false);
  tags.SetValue(0x0028, 0x0004, "MONOCHROME2", false);
  tags.SetValue(0x0028, 0x0010, "64", false);
  tags.SetValue(0x0028, 0x0011, "64", false);
  tags.SetValue(0x0028, 0x0100, "16", false);
  tags.SetValue(0x0028, 0x0101, "16", false);
  tags.SetValue(0x0028, 0x0102, "15", false);
  tags.SetValue(0x0028, 0x0103, "1", false);
  tags.SetValue(0x0008, 0x0032, "115551", false);
  tags.SetValue(0x0018, 0x0050, "3", false);
  tags.SetValue(0x0018, 0x0088, "5", false);
  tags.SetValue(0x0020, 0x0032, "-118.125\\-143.344\\47.3125", false);
  tags.SetValue(0x0020, 0x0037, "1.00000\\-0.00000\\0.00000\\-0.00000\\1.00000\\0.00000", false);
  tags.SetValue(0x0028, 0x0030, "3.75\\3.75", false);
  std::unique_ptr<Neuro::InputDicomInstance> dicom(new Neuro::InputDicomInstance(tags));
  target.AddInstance(dicom.release(), "nope");
}

TEST(Autogenerated, Test_77)
{
  Neuro::DicomInstancesCollection instances;
  LoadInstance_77(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 64);
  ASSERT_NEAR(nifti.pixdim[1], 3.750000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 64);
  ASSERT_NEAR(nifti.pixdim[2], 3.750000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 1);
  ASSERT_NEAR(nifti.pixdim[3], 5.000000, 0.00001);
  ASSERT_EQ(nifti.ndim, 3);
  ASSERT_EQ(nifti.dim[0], 3);
  ASSERT_EQ(nifti.nvox, 4096u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 4);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(57));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(57));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(57));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -3.750000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 118.125000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 3.750000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], -92.906006, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 5.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], 47.312500, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, 1.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, 0.000000, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=30;Time=115551.000");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadInstance_78(Neuro::DicomInstancesCollection& target)
{
  // ('dcm_qa_nih', 'In/20180918Si/mr_0003/epi_pe_ap-00001.dcm')
  Orthanc::DicomMap tags;
  tags.SetValue(0x0008, 0x0060, "MR", false);
  tags.SetValue(0x0008, 0x0070, "SIEMENS", false);
  tags.SetValue(0x0020, 0x0013, "1", false);
  tags.SetValue(0x0018, 0x0080, "2435.37", false);
  tags.SetValue(0x0018, 0x0081, "50", false);
  tags.SetValue(0x0018, 0x1312, "COL", false);
  tags.SetValue(0x0019, 0x1029, "0.0\\1450.00000001\\482.50000001\\1932.50000001\\967.50000001", false);
  tags.SetValue(0x0008, 0x0016, "1.2.840.10008.5.1.4.1.1.4", false);
  tags.SetValue(0x0028, 0x0002, "1", false);
  tags.SetValue(0x0028, 0x0004, "MONOCHROME2", false);
  tags.SetValue(0x0028, 0x0010, "216", false);
  tags.SetValue(0x0028, 0x0011, "216", false);
  tags.SetValue(0x0028, 0x0100, "16", false);
  tags.SetValue(0x0028, 0x0101, "12", false);
  tags.SetValue(0x0028, 0x0102, "11", false);
  tags.SetValue(0x0028, 0x0103, "0", false);
  tags.SetValue(0x0008, 0x0032, "122458.102500", false);
  tags.SetValue(0x0018, 0x0050, "3", false);
  tags.SetValue(0x0018, 0x0088, "12", false);
  tags.SetValue(0x0020, 0x0032, "-324\\-324\\16", false);
  tags.SetValue(0x0020, 0x0037, "1\\0\\0\\0\\1\\0", false);
  tags.SetValue(0x0028, 0x0030, "3\\3", false);
  std::unique_ptr<Neuro::InputDicomInstance> dicom(new Neuro::InputDicomInstance(tags));
  dicom->GetCSAHeader().AddTag("NumberOfImagesInMosaic", "US").AddValue("5");
  dicom->GetCSAHeader().AddTag("PhaseEncodingDirectionPositive", "IS").AddValue("1");
  dicom->GetCSAHeader().AddTag("SliceNormalVector", "FD").AddValue("0.0").AddValue("0.0").AddValue("1.0");
  target.AddInstance(dicom.release(), "nope");
}

TEST(Autogenerated, Test_78)
{
  Neuro::DicomInstancesCollection instances;
  LoadInstance_78(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 72);
  ASSERT_NEAR(nifti.pixdim[1], 3.000000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 72);
  ASSERT_NEAR(nifti.pixdim[2], 3.000000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 5);
  ASSERT_NEAR(nifti.pixdim[3], 12.000000, 0.00001);
  ASSERT_EQ(nifti.ndim, 3);
  ASSERT_EQ(nifti.dim[0], 3);
  ASSERT_EQ(nifti.nvox, 25920u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(57));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(57));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(57));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 3);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -3.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 108.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 3.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], -105.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 12.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], 16.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, 1.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, 0.000000, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=50;Time=122458.102;phase=1");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadInstance_79(Neuro::DicomInstancesCollection& target)
{
  // ('dcm_qa_nih', 'In/20180918Si/mr_0003/epi_pe_ap-00002.dcm')
  Orthanc::DicomMap tags;
  tags.SetValue(0x0008, 0x0060, "MR", false);
  tags.SetValue(0x0008, 0x0070, "SIEMENS", false);
  tags.SetValue(0x0020, 0x0013, "2", false);
  tags.SetValue(0x0018, 0x0080, "2435.37", false);
  tags.SetValue(0x0018, 0x0081, "50", false);
  tags.SetValue(0x0018, 0x1312, "COL", false);
  tags.SetValue(0x0019, 0x1029, "0.0\\1450.0\\482.5\\1932.5\\967.5", false);
  tags.SetValue(0x0008, 0x0016, "1.2.840.10008.5.1.4.1.1.4", false);
  tags.SetValue(0x0028, 0x0002, "1", false);
  tags.SetValue(0x0028, 0x0004, "MONOCHROME2", false);
  tags.SetValue(0x0028, 0x0010, "216", false);
  tags.SetValue(0x0028, 0x0011, "216", false);
  tags.SetValue(0x0028, 0x0100, "16", false);
  tags.SetValue(0x0028, 0x0101, "12", false);
  tags.SetValue(0x0028, 0x0102, "11", false);
  tags.SetValue(0x0028, 0x0103, "0", false);
  tags.SetValue(0x0008, 0x0032, "122500.537500", false);
  tags.SetValue(0x0018, 0x0050, "3", false);
  tags.SetValue(0x0018, 0x0088, "12", false);
  tags.SetValue(0x0020, 0x0032, "-324\\-324\\16", false);
  tags.SetValue(0x0020, 0x0037, "1\\0\\0\\0\\1\\0", false);
  tags.SetValue(0x0028, 0x0030, "3\\3", false);
  std::unique_ptr<Neuro::InputDicomInstance> dicom(new Neuro::InputDicomInstance(tags));
  dicom->GetCSAHeader().AddTag("NumberOfImagesInMosaic", "US").AddValue("5");
  dicom->GetCSAHeader().AddTag("PhaseEncodingDirectionPositive", "IS").AddValue("1");
  dicom->GetCSAHeader().AddTag("SliceNormalVector", "FD").AddValue("0.0").AddValue("0.0").AddValue("1.0");
  target.AddInstance(dicom.release(), "nope");
}

TEST(Autogenerated, Test_79)
{
  Neuro::DicomInstancesCollection instances;
  LoadInstance_79(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 72);
  ASSERT_NEAR(nifti.pixdim[1], 3.000000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 72);
  ASSERT_NEAR(nifti.pixdim[2], 3.000000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 5);
  ASSERT_NEAR(nifti.pixdim[3], 12.000000, 0.00001);
  ASSERT_EQ(nifti.ndim, 3);
  ASSERT_EQ(nifti.dim[0], 3);
  ASSERT_EQ(nifti.nvox, 25920u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(57));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(57));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(57));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 3);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -3.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 108.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 3.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], -105.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 12.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], 16.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, 1.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, 0.000000, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=50;Time=122500.538;phase=1");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadInstance_80(Neuro::DicomInstancesCollection& target)
{
  // ('dcm_qa_nih', 'In/20180918Si/mr_0004/epi_pe_pa-00001.dcm')
  Orthanc::DicomMap tags;
  tags.SetValue(0x0008, 0x0060, "MR", false);
  tags.SetValue(0x0008, 0x0070, "SIEMENS", false);
  tags.SetValue(0x0020, 0x0013, "1", false);
  tags.SetValue(0x0018, 0x0080, "2435.37", false);
  tags.SetValue(0x0018, 0x0081, "50", false);
  tags.SetValue(0x0018, 0x1312, "COL", false);
  tags.SetValue(0x0019, 0x1029, "0.0\\1447.49999999\\482.5\\1929.99999999\\965.0", false);
  tags.SetValue(0x0008, 0x0016, "1.2.840.10008.5.1.4.1.1.4", false);
  tags.SetValue(0x0028, 0x0002, "1", false);
  tags.SetValue(0x0028, 0x0004, "MONOCHROME2", false);
  tags.SetValue(0x0028, 0x0010, "216", false);
  tags.SetValue(0x0028, 0x0011, "216", false);
  tags.SetValue(0x0028, 0x0100, "16", false);
  tags.SetValue(0x0028, 0x0101, "12", false);
  tags.SetValue(0x0028, 0x0102, "11", false);
  tags.SetValue(0x0028, 0x0103, "0", false);
  tags.SetValue(0x0008, 0x0032, "122654.517500", false);
  tags.SetValue(0x0018, 0x0050, "3", false);
  tags.SetValue(0x0018, 0x0088, "12", false);
  tags.SetValue(0x0020, 0x0032, "-324\\-324\\16", false);
  tags.SetValue(0x0020, 0x0037, "1\\0\\0\\0\\1\\0", false);
  tags.SetValue(0x0028, 0x0030, "3\\3", false);
  std::unique_ptr<Neuro::InputDicomInstance> dicom(new Neuro::InputDicomInstance(tags));
  dicom->GetCSAHeader().AddTag("NumberOfImagesInMosaic", "US").AddValue("5");
  dicom->GetCSAHeader().AddTag("PhaseEncodingDirectionPositive", "IS").AddValue("0");
  dicom->GetCSAHeader().AddTag("SliceNormalVector", "FD").AddValue("0.0").AddValue("0.0").AddValue("1.0");
  target.AddInstance(dicom.release(), "nope");
}

TEST(Autogenerated, Test_80)
{
  Neuro::DicomInstancesCollection instances;
  LoadInstance_80(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 72);
  ASSERT_NEAR(nifti.pixdim[1], 3.000000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 72);
  ASSERT_NEAR(nifti.pixdim[2], 3.000000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 5);
  ASSERT_NEAR(nifti.pixdim[3], 12.000000, 0.00001);
  ASSERT_EQ(nifti.ndim, 3);
  ASSERT_EQ(nifti.dim[0], 3);
  ASSERT_EQ(nifti.nvox, 25920u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(57));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(57));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(57));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 3);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -3.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 108.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 3.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], -105.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 12.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], 16.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, 1.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, 0.000000, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=50;Time=122654.518;phase=0");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadInstance_81(Neuro::DicomInstancesCollection& target)
{
  // ('dcm_qa_nih', 'In/20180918Si/mr_0004/epi_pe_pa-00002.dcm')
  Orthanc::DicomMap tags;
  tags.SetValue(0x0008, 0x0060, "MR", false);
  tags.SetValue(0x0008, 0x0070, "SIEMENS", false);
  tags.SetValue(0x0020, 0x0013, "2", false);
  tags.SetValue(0x0018, 0x0080, "2435.37", false);
  tags.SetValue(0x0018, 0x0081, "50", false);
  tags.SetValue(0x0018, 0x1312, "COL", false);
  tags.SetValue(0x0019, 0x1029, "0.0\\1447.49999999\\482.5\\1932.5\\965.0", false);
  tags.SetValue(0x0008, 0x0016, "1.2.840.10008.5.1.4.1.1.4", false);
  tags.SetValue(0x0028, 0x0002, "1", false);
  tags.SetValue(0x0028, 0x0004, "MONOCHROME2", false);
  tags.SetValue(0x0028, 0x0010, "216", false);
  tags.SetValue(0x0028, 0x0011, "216", false);
  tags.SetValue(0x0028, 0x0100, "16", false);
  tags.SetValue(0x0028, 0x0101, "12", false);
  tags.SetValue(0x0028, 0x0102, "11", false);
  tags.SetValue(0x0028, 0x0103, "0", false);
  tags.SetValue(0x0008, 0x0032, "122656.952500", false);
  tags.SetValue(0x0018, 0x0050, "3", false);
  tags.SetValue(0x0018, 0x0088, "12", false);
  tags.SetValue(0x0020, 0x0032, "-324\\-324\\16", false);
  tags.SetValue(0x0020, 0x0037, "1\\0\\0\\0\\1\\0", false);
  tags.SetValue(0x0028, 0x0030, "3\\3", false);
  std::unique_ptr<Neuro::InputDicomInstance> dicom(new Neuro::InputDicomInstance(tags));
  dicom->GetCSAHeader().AddTag("NumberOfImagesInMosaic", "US").AddValue("5");
  dicom->GetCSAHeader().AddTag("PhaseEncodingDirectionPositive", "IS").AddValue("0");
  dicom->GetCSAHeader().AddTag("SliceNormalVector", "FD").AddValue("0.0").AddValue("0.0").AddValue("1.0");
  target.AddInstance(dicom.release(), "nope");
}

TEST(Autogenerated, Test_81)
{
  Neuro::DicomInstancesCollection instances;
  LoadInstance_81(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 72);
  ASSERT_NEAR(nifti.pixdim[1], 3.000000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 72);
  ASSERT_NEAR(nifti.pixdim[2], 3.000000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 5);
  ASSERT_NEAR(nifti.pixdim[3], 12.000000, 0.00001);
  ASSERT_EQ(nifti.ndim, 3);
  ASSERT_EQ(nifti.dim[0], 3);
  ASSERT_EQ(nifti.nvox, 25920u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(57));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(57));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(57));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 3);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -3.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 108.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 3.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], -105.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 12.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], 16.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, 1.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, 0.000000, 0.0001);
  ASSERT_TRUE(std::string(nifti.descrip) == "TE=50;Time=122656.952;phase=0" ||
              std::string(nifti.descrip) == "TE=50;Time=122656.953;phase=0");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadInstance_82(Neuro::DicomInstancesCollection& target)
{
  // ('dcm_qa_nih', 'In/20180918Si/mr_0005/epi_pe_rl-00001.dcm')
  Orthanc::DicomMap tags;
  tags.SetValue(0x0008, 0x0060, "MR", false);
  tags.SetValue(0x0008, 0x0070, "SIEMENS", false);
  tags.SetValue(0x0020, 0x0013, "1", false);
  tags.SetValue(0x0018, 0x0080, "2435.37", false);
  tags.SetValue(0x0018, 0x0081, "50", false);
  tags.SetValue(0x0018, 0x1312, "ROW", false);
  tags.SetValue(0x0019, 0x1029, "0.0\\1450.0\\482.5\\1932.5\\967.5", false);
  tags.SetValue(0x0008, 0x0016, "1.2.840.10008.5.1.4.1.1.4", false);
  tags.SetValue(0x0028, 0x0002, "1", false);
  tags.SetValue(0x0028, 0x0004, "MONOCHROME2", false);
  tags.SetValue(0x0028, 0x0010, "216", false);
  tags.SetValue(0x0028, 0x0011, "216", false);
  tags.SetValue(0x0028, 0x0100, "16", false);
  tags.SetValue(0x0028, 0x0101, "12", false);
  tags.SetValue(0x0028, 0x0102, "11", false);
  tags.SetValue(0x0028, 0x0103, "0", false);
  tags.SetValue(0x0008, 0x0032, "122812.085000", false);
  tags.SetValue(0x0018, 0x0050, "3", false);
  tags.SetValue(0x0018, 0x0088, "12", false);
  tags.SetValue(0x0020, 0x0032, "-324.00000000007\\-323.99999999993\\16", false);
  tags.SetValue(0x0020, 0x0037, "1\\-2.07e-013\\0\\2.07e-013\\1\\0", false);
  tags.SetValue(0x0028, 0x0030, "3\\3", false);
  std::unique_ptr<Neuro::InputDicomInstance> dicom(new Neuro::InputDicomInstance(tags));
  dicom->GetCSAHeader().AddTag("NumberOfImagesInMosaic", "US").AddValue("5");
  dicom->GetCSAHeader().AddTag("PhaseEncodingDirectionPositive", "IS").AddValue("1");
  dicom->GetCSAHeader().AddTag("SliceNormalVector", "FD").AddValue("0.0").AddValue("0.0").AddValue("1.0");
  target.AddInstance(dicom.release(), "nope");
}

TEST(Autogenerated, Test_82)
{
  Neuro::DicomInstancesCollection instances;
  LoadInstance_82(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 72);
  ASSERT_NEAR(nifti.pixdim[1], 3.000000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 72);
  ASSERT_NEAR(nifti.pixdim[2], 3.000000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 5);
  ASSERT_NEAR(nifti.pixdim[3], 12.000000, 0.00001);
  ASSERT_EQ(nifti.ndim, 3);
  ASSERT_EQ(nifti.dim[0], 3);
  ASSERT_EQ(nifti.nvox, 25920u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(54));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(54));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(54));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 3);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -3.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 108.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 3.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], -105.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 12.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], 16.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, 1.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, 0.000000, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=50;Time=122812.085;phase=1");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadInstance_83(Neuro::DicomInstancesCollection& target)
{
  // ('dcm_qa_nih', 'In/20180918Si/mr_0005/epi_pe_rl-00002.dcm')
  Orthanc::DicomMap tags;
  tags.SetValue(0x0008, 0x0060, "MR", false);
  tags.SetValue(0x0008, 0x0070, "SIEMENS", false);
  tags.SetValue(0x0020, 0x0013, "2", false);
  tags.SetValue(0x0018, 0x0080, "2435.37", false);
  tags.SetValue(0x0018, 0x0081, "50", false);
  tags.SetValue(0x0018, 0x1312, "ROW", false);
  tags.SetValue(0x0019, 0x1029, "0.0\\1450.0\\485.0\\1932.5\\967.5", false);
  tags.SetValue(0x0008, 0x0016, "1.2.840.10008.5.1.4.1.1.4", false);
  tags.SetValue(0x0028, 0x0002, "1", false);
  tags.SetValue(0x0028, 0x0004, "MONOCHROME2", false);
  tags.SetValue(0x0028, 0x0010, "216", false);
  tags.SetValue(0x0028, 0x0011, "216", false);
  tags.SetValue(0x0028, 0x0100, "16", false);
  tags.SetValue(0x0028, 0x0101, "12", false);
  tags.SetValue(0x0028, 0x0102, "11", false);
  tags.SetValue(0x0028, 0x0103, "0", false);
  tags.SetValue(0x0008, 0x0032, "122814.520000", false);
  tags.SetValue(0x0018, 0x0050, "3", false);
  tags.SetValue(0x0018, 0x0088, "12", false);
  tags.SetValue(0x0020, 0x0032, "-324.00000000007\\-323.99999999993\\16", false);
  tags.SetValue(0x0020, 0x0037, "1\\-2.07e-013\\0\\2.07e-013\\1\\0", false);
  tags.SetValue(0x0028, 0x0030, "3\\3", false);
  std::unique_ptr<Neuro::InputDicomInstance> dicom(new Neuro::InputDicomInstance(tags));
  dicom->GetCSAHeader().AddTag("NumberOfImagesInMosaic", "US").AddValue("5");
  dicom->GetCSAHeader().AddTag("PhaseEncodingDirectionPositive", "IS").AddValue("1");
  dicom->GetCSAHeader().AddTag("SliceNormalVector", "FD").AddValue("0.0").AddValue("0.0").AddValue("1.0");
  target.AddInstance(dicom.release(), "nope");
}

TEST(Autogenerated, Test_83)
{
  Neuro::DicomInstancesCollection instances;
  LoadInstance_83(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 72);
  ASSERT_NEAR(nifti.pixdim[1], 3.000000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 72);
  ASSERT_NEAR(nifti.pixdim[2], 3.000000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 5);
  ASSERT_NEAR(nifti.pixdim[3], 12.000000, 0.00001);
  ASSERT_EQ(nifti.ndim, 3);
  ASSERT_EQ(nifti.dim[0], 3);
  ASSERT_EQ(nifti.nvox, 25920u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(54));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(54));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(54));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 3);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -3.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 108.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 3.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], -105.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 12.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], 16.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, 1.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, 0.000000, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=50;Time=122814.520;phase=1");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadInstance_84(Neuro::DicomInstancesCollection& target)
{
  // ('dcm_qa_nih', 'In/20180918Si/mr_0006/epi_pe_lr-00001.dcm')
  Orthanc::DicomMap tags;
  tags.SetValue(0x0008, 0x0060, "MR", false);
  tags.SetValue(0x0008, 0x0070, "SIEMENS", false);
  tags.SetValue(0x0020, 0x0013, "1", false);
  tags.SetValue(0x0018, 0x0080, "2435.37", false);
  tags.SetValue(0x0018, 0x0081, "50", false);
  tags.SetValue(0x0018, 0x1312, "ROW", false);
  tags.SetValue(0x0019, 0x1029, "0.0\\1447.49999999\\482.5\\1932.5\\965.0", false);
  tags.SetValue(0x0008, 0x0016, "1.2.840.10008.5.1.4.1.1.4", false);
  tags.SetValue(0x0028, 0x0002, "1", false);
  tags.SetValue(0x0028, 0x0004, "MONOCHROME2", false);
  tags.SetValue(0x0028, 0x0010, "216", false);
  tags.SetValue(0x0028, 0x0011, "216", false);
  tags.SetValue(0x0028, 0x0100, "16", false);
  tags.SetValue(0x0028, 0x0101, "12", false);
  tags.SetValue(0x0028, 0x0102, "11", false);
  tags.SetValue(0x0028, 0x0103, "0", false);
  tags.SetValue(0x0008, 0x0032, "123002.285000", false);
  tags.SetValue(0x0018, 0x0050, "3", false);
  tags.SetValue(0x0018, 0x0088, "12", false);
  tags.SetValue(0x0020, 0x0032, "-324\\-324\\16", false);
  tags.SetValue(0x0020, 0x0037, "1\\0\\0\\0\\1\\0", false);
  tags.SetValue(0x0028, 0x0030, "3\\3", false);
  std::unique_ptr<Neuro::InputDicomInstance> dicom(new Neuro::InputDicomInstance(tags));
  dicom->GetCSAHeader().AddTag("NumberOfImagesInMosaic", "US").AddValue("5");
  dicom->GetCSAHeader().AddTag("PhaseEncodingDirectionPositive", "IS").AddValue("0");
  dicom->GetCSAHeader().AddTag("SliceNormalVector", "FD").AddValue("0.0").AddValue("0.0").AddValue("1.0");
  target.AddInstance(dicom.release(), "nope");
}

TEST(Autogenerated, Test_84)
{
  Neuro::DicomInstancesCollection instances;
  LoadInstance_84(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 72);
  ASSERT_NEAR(nifti.pixdim[1], 3.000000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 72);
  ASSERT_NEAR(nifti.pixdim[2], 3.000000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 5);
  ASSERT_NEAR(nifti.pixdim[3], 12.000000, 0.00001);
  ASSERT_EQ(nifti.ndim, 3);
  ASSERT_EQ(nifti.dim[0], 3);
  ASSERT_EQ(nifti.nvox, 25920u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(54));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(54));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(54));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 3);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -3.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 108.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 3.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], -105.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 12.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], 16.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, 1.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, 0.000000, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=50;Time=123002.285;phase=0");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadInstance_85(Neuro::DicomInstancesCollection& target)
{
  // ('dcm_qa_nih', 'In/20180918Si/mr_0006/epi_pe_lr-00002.dcm')
  Orthanc::DicomMap tags;
  tags.SetValue(0x0008, 0x0060, "MR", false);
  tags.SetValue(0x0008, 0x0070, "SIEMENS", false);
  tags.SetValue(0x0020, 0x0013, "2", false);
  tags.SetValue(0x0018, 0x0080, "2435.37", false);
  tags.SetValue(0x0018, 0x0081, "50", false);
  tags.SetValue(0x0018, 0x1312, "ROW", false);
  tags.SetValue(0x0019, 0x1029, "0.0\\1450.0\\482.5\\1932.5\\965.0", false);
  tags.SetValue(0x0008, 0x0016, "1.2.840.10008.5.1.4.1.1.4", false);
  tags.SetValue(0x0028, 0x0002, "1", false);
  tags.SetValue(0x0028, 0x0004, "MONOCHROME2", false);
  tags.SetValue(0x0028, 0x0010, "216", false);
  tags.SetValue(0x0028, 0x0011, "216", false);
  tags.SetValue(0x0028, 0x0100, "16", false);
  tags.SetValue(0x0028, 0x0101, "12", false);
  tags.SetValue(0x0028, 0x0102, "11", false);
  tags.SetValue(0x0028, 0x0103, "0", false);
  tags.SetValue(0x0008, 0x0032, "123004.720000", false);
  tags.SetValue(0x0018, 0x0050, "3", false);
  tags.SetValue(0x0018, 0x0088, "12", false);
  tags.SetValue(0x0020, 0x0032, "-324\\-324\\16", false);
  tags.SetValue(0x0020, 0x0037, "1\\0\\0\\0\\1\\0", false);
  tags.SetValue(0x0028, 0x0030, "3\\3", false);
  std::unique_ptr<Neuro::InputDicomInstance> dicom(new Neuro::InputDicomInstance(tags));
  dicom->GetCSAHeader().AddTag("NumberOfImagesInMosaic", "US").AddValue("5");
  dicom->GetCSAHeader().AddTag("PhaseEncodingDirectionPositive", "IS").AddValue("0");
  dicom->GetCSAHeader().AddTag("SliceNormalVector", "FD").AddValue("0.0").AddValue("0.0").AddValue("1.0");
  target.AddInstance(dicom.release(), "nope");
}

TEST(Autogenerated, Test_85)
{
  Neuro::DicomInstancesCollection instances;
  LoadInstance_85(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 72);
  ASSERT_NEAR(nifti.pixdim[1], 3.000000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 72);
  ASSERT_NEAR(nifti.pixdim[2], 3.000000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 5);
  ASSERT_NEAR(nifti.pixdim[3], 12.000000, 0.00001);
  ASSERT_EQ(nifti.ndim, 3);
  ASSERT_EQ(nifti.dim[0], 3);
  ASSERT_EQ(nifti.nvox, 25920u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(54));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(54));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(54));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 3);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -3.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 108.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 3.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], -105.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 12.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], 16.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, 1.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, 0.000000, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=50;Time=123004.720;phase=0");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadInstance_86(Neuro::DicomInstancesCollection& target)
{
  // ('dcm_qa_uih', 'In/DTI_132225/dti_tra_dir16_PA_rot_SaveBySlc__134057/00000001.dcm')
  Orthanc::DicomMap tags;
  tags.SetValue(0x0008, 0x0060, "MR", false);
  tags.SetValue(0x0008, 0x0070, "UIH", false);
  tags.SetValue(0x0020, 0x0013, "1", false);
  tags.SetValue(0x0018, 0x0080, "5865.0", false);
  tags.SetValue(0x0018, 0x0081, "70.7", false);
  tags.SetValue(0x0018, 0x1312, "COL", false);
  tags.SetValue(0x0008, 0x0016, "1.2.840.10008.5.1.4.1.1.4", false);
  tags.SetValue(0x0028, 0x0002, "1", false);
  tags.SetValue(0x0028, 0x0004, "MONOCHROME2", false);
  tags.SetValue(0x0028, 0x0010, "256", false);
  tags.SetValue(0x0028, 0x0011, "320", false);
  tags.SetValue(0x0028, 0x0100, "16", false);
  tags.SetValue(0x0028, 0x0101, "16", false);
  tags.SetValue(0x0028, 0x0102, "15", false);
  tags.SetValue(0x0028, 0x0103, "0", false);
  tags.SetValue(0x0018, 0x0050, "3.5", false);
  tags.SetValue(0x0018, 0x0088, "4.19999981", false);
  tags.SetValue(0x0020, 0x0037, "0.906671703\\-0.4210186\\0.026263684\\0.420842081\\0.90705198\\0.0121905897", false);
  tags.SetValue(0x0028, 0x0030, "3.5\\3.5", false);
  std::unique_ptr<Neuro::InputDicomInstance> dicom(new Neuro::InputDicomInstance(tags));
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925133748.283000", false);
    uih.SetValue(0x0008, 0x0032, "133748.283000", false);
    uih.SetValue(0x0018, 0x9073, "21.0", false);
    uih.SetValue(0x0020, 0x0032, "-154.647507\\-58.5786934\\-9.21860886", false);
    uih.SetValue(0x0020, 0x1041, "-4.73692799", false);
    uih.SetValue(0x0065, 0x100f, "F4.7", false);
    uih.SetValue(0x0065, 0x1015, "-5.96596575\\-4.14295006\\-4.91173077", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925133745.357000", false);
    uih.SetValue(0x0008, 0x0032, "133745.357000", false);
    uih.SetValue(0x0018, 0x9073, "21.0", false);
    uih.SetValue(0x0020, 0x0032, "-154.769119\\-58.5786934\\-5.02037001", false);
    uih.SetValue(0x0020, 0x1041, "-0.536926806", false);
    uih.SetValue(0x0065, 0x100f, "F0.5", false);
    uih.SetValue(0x0065, 0x1015, "-6.08757687\\-4.14295006\\-0.713491559", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925133747.990000", false);
    uih.SetValue(0x0008, 0x0032, "133747.990000", false);
    uih.SetValue(0x0018, 0x9073, "22.0", false);
    uih.SetValue(0x0020, 0x0032, "-154.890732\\-58.5786934\\-0.822131157", false);
    uih.SetValue(0x0020, 0x1041, "3.66307425", false);
    uih.SetValue(0x0065, 0x100f, "H3.7", false);
    uih.SetValue(0x0065, 0x1015, "-6.20918798\\-4.14295006\\3.48474741", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925133745.065000", false);
    uih.SetValue(0x0008, 0x0032, "133745.065000", false);
    uih.SetValue(0x0018, 0x9073, "21.0", false);
    uih.SetValue(0x0020, 0x0032, "-155.012344\\-58.5786934\\3.37610793", false);
    uih.SetValue(0x0020, 0x1041, "7.86307573", false);
    uih.SetValue(0x0065, 0x100f, "H7.9", false);
    uih.SetValue(0x0065, 0x1015, "-6.33079863\\-4.14295006\\7.68298626", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925133747.698000", false);
    uih.SetValue(0x0008, 0x0032, "133747.698000", false);
    uih.SetValue(0x0018, 0x9073, "21.0", false);
    uih.SetValue(0x0020, 0x0032, "-155.133957\\-58.5786934\\7.57434702", false);
    uih.SetValue(0x0020, 0x1041, "12.063077", false);
    uih.SetValue(0x0065, 0x100f, "H12.1", false);
    uih.SetValue(0x0065, 0x1015, "-6.45240974\\-4.14295006\\11.8812256", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925133744.772000", false);
    uih.SetValue(0x0008, 0x0032, "133744.772000", false);
    uih.SetValue(0x0018, 0x9073, "21.0", false);
    uih.SetValue(0x0020, 0x0032, "-155.255569\\-58.5786934\\11.7725859", false);
    uih.SetValue(0x0020, 0x1041, "16.2630787", false);
    uih.SetValue(0x0065, 0x100f, "H16.3", false);
    uih.SetValue(0x0065, 0x1015, "-6.57402086\\-4.14295006\\16.079464", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925133747.405000", false);
    uih.SetValue(0x0008, 0x0032, "133747.405000", false);
    uih.SetValue(0x0018, 0x9073, "21.0", false);
    uih.SetValue(0x0020, 0x0032, "-155.377167\\-58.5786934\\15.9708252", false);
    uih.SetValue(0x0020, 0x1041, "20.4630795", false);
    uih.SetValue(0x0065, 0x100f, "H20.5", false);
    uih.SetValue(0x0065, 0x1015, "-6.6956315\\-4.14295006\\20.2777042", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925133744.479000", false);
    uih.SetValue(0x0008, 0x0032, "133744.479000", false);
    uih.SetValue(0x0018, 0x9073, "22.0", false);
    uih.SetValue(0x0020, 0x0032, "-155.498779\\-58.5786934\\20.1690636", false);
    uih.SetValue(0x0020, 0x1041, "24.6630802", false);
    uih.SetValue(0x0065, 0x100f, "H24.7", false);
    uih.SetValue(0x0065, 0x1015, "-6.81724262\\-4.14295006\\24.4759426", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925133747.113000", false);
    uih.SetValue(0x0008, 0x0032, "133747.113000", false);
    uih.SetValue(0x0018, 0x9073, "21.0", false);
    uih.SetValue(0x0020, 0x0032, "-155.620392\\-58.5786934\\24.3673019", false);
    uih.SetValue(0x0020, 0x1041, "28.863081", false);
    uih.SetValue(0x0065, 0x100f, "H28.9", false);
    uih.SetValue(0x0065, 0x1015, "-6.93885374\\-4.14295006\\28.674181", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925133744.187000", false);
    uih.SetValue(0x0008, 0x0032, "133744.187000", false);
    uih.SetValue(0x0018, 0x9073, "21.0", false);
    uih.SetValue(0x0020, 0x0032, "-155.742004\\-58.5786934\\28.5655422", false);
    uih.SetValue(0x0020, 0x1041, "33.0630836", false);
    uih.SetValue(0x0065, 0x100f, "H33.1", false);
    uih.SetValue(0x0065, 0x1015, "-7.06046438\\-4.14295006\\32.8724213", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925133746.820000", false);
    uih.SetValue(0x0008, 0x0032, "133746.820000", false);
    uih.SetValue(0x0018, 0x9073, "21.0", false);
    uih.SetValue(0x0020, 0x0032, "-155.863617\\-58.5786934\\32.7637825", false);
    uih.SetValue(0x0020, 0x1041, "37.2630844", false);
    uih.SetValue(0x0065, 0x100f, "H37.3", false);
    uih.SetValue(0x0065, 0x1015, "-7.1820755\\-4.14295006\\37.0706596", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925133743.894000", false);
    uih.SetValue(0x0008, 0x0032, "133743.894000", false);
    uih.SetValue(0x0018, 0x9073, "22.0", false);
    uih.SetValue(0x0020, 0x0032, "-155.985229\\-58.5786934\\36.9620209", false);
    uih.SetValue(0x0020, 0x1041, "41.4630852", false);
    uih.SetValue(0x0065, 0x100f, "H41.5", false);
    uih.SetValue(0x0065, 0x1015, "-7.30368662\\-4.14295006\\41.268898", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925133746.527000", false);
    uih.SetValue(0x0008, 0x0032, "133746.527000", false);
    uih.SetValue(0x0018, 0x9073, "22.0", false);
    uih.SetValue(0x0020, 0x0032, "-156.106842\\-58.5786934\\41.1602592", false);
    uih.SetValue(0x0020, 0x1041, "45.6630859", false);
    uih.SetValue(0x0065, 0x100f, "H45.7", false);
    uih.SetValue(0x0065, 0x1015, "-7.42529726\\-4.14295006\\45.4671364", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925133743.602000", false);
    uih.SetValue(0x0008, 0x0032, "133743.602000", false);
    uih.SetValue(0x0018, 0x9073, "21.0", false);
    uih.SetValue(0x0020, 0x0032, "-156.228455\\-58.5786934\\45.3584976", false);
    uih.SetValue(0x0020, 0x1041, "49.8630867", false);
    uih.SetValue(0x0065, 0x100f, "H49.9", false);
    uih.SetValue(0x0065, 0x1015, "-7.54690838\\-4.14295006\\49.6653748", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925133746.235000", false);
    uih.SetValue(0x0008, 0x0032, "133746.235000", false);
    uih.SetValue(0x0018, 0x9073, "21.0", false);
    uih.SetValue(0x0020, 0x0032, "-156.350067\\-58.5786934\\49.556736", false);
    uih.SetValue(0x0020, 0x1041, "54.0630875", false);
    uih.SetValue(0x0065, 0x100f, "H54.1", false);
    uih.SetValue(0x0065, 0x1015, "-7.6685195\\-4.14295006\\53.8636169", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925133743.309000", false);
    uih.SetValue(0x0008, 0x0032, "133743.309000", false);
    uih.SetValue(0x0018, 0x9073, "21.0", false);
    uih.SetValue(0x0020, 0x0032, "-156.471664\\-58.5786934\\53.7549744", false);
    uih.SetValue(0x0020, 0x1041, "58.2630882", false);
    uih.SetValue(0x0065, 0x100f, "H58.3", false);
    uih.SetValue(0x0065, 0x1015, "-7.79013014\\-4.14295006\\58.0618553", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925133745.942000", false);
    uih.SetValue(0x0008, 0x0032, "133745.942000", false);
    uih.SetValue(0x0018, 0x9073, "22.0", false);
    uih.SetValue(0x0020, 0x0032, "-156.593277\\-58.5786934\\57.9532166", false);
    uih.SetValue(0x0020, 0x1041, "62.4630928", false);
    uih.SetValue(0x0065, 0x100f, "H62.5", false);
    uih.SetValue(0x0065, 0x1015, "-7.91174126\\-4.14295006\\62.2600937", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925133743.017000", false);
    uih.SetValue(0x0008, 0x0032, "133743.017000", false);
    uih.SetValue(0x0018, 0x9073, "21.0", false);
    uih.SetValue(0x0020, 0x0032, "-156.71489\\-58.5786934\\62.1514549", false);
    uih.SetValue(0x0020, 0x1041, "66.6630936", false);
    uih.SetValue(0x0065, 0x100f, "H66.7", false);
    uih.SetValue(0x0065, 0x1015, "-8.0333519\\-4.14295006\\66.4583359", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925133745.650000", false);
    uih.SetValue(0x0008, 0x0032, "133745.650000", false);
    uih.SetValue(0x0018, 0x9073, "21.0", false);
    uih.SetValue(0x0020, 0x0032, "-156.836502\\-58.5786934\\66.3496933", false);
    uih.SetValue(0x0020, 0x1041, "70.8630905", false);
    uih.SetValue(0x0065, 0x100f, "H70.9", false);
    uih.SetValue(0x0065, 0x1015, "-8.15496349\\-4.14295006\\70.6565704", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925133742.724000", false);
    uih.SetValue(0x0008, 0x0032, "133742.724000", false);
    uih.SetValue(0x0018, 0x9073, "21.0", false);
    uih.SetValue(0x0020, 0x0032, "-156.958115\\-58.5786934\\70.5479355", false);
    uih.SetValue(0x0020, 0x1041, "75.0630951", false);
    uih.SetValue(0x0065, 0x100f, "H75.1", false);
    uih.SetValue(0x0065, 0x1015, "-8.27657413\\-4.14295006\\74.8548126", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  target.AddInstance(dicom.release(), "nope");
}

TEST(Autogenerated, Test_86)
{
  Neuro::DicomInstancesCollection instances;
  LoadInstance_86(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 64);
  ASSERT_NEAR(nifti.pixdim[1], 3.500000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 64);
  ASSERT_NEAR(nifti.pixdim[2], 3.500000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 20);
  ASSERT_NEAR(nifti.pixdim[3], 4.200000, 0.00001);
  ASSERT_EQ(nifti.ndim, 3);
  ASSERT_EQ(nifti.dim[0], 3);
  ASSERT_EQ(nifti.nvox, 81920u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(57));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(57));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(57));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -3.173351, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 1.472947, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], 0.121611, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 61.851830, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], 1.473565, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 3.174682, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], -141.426270, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.091923, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], -0.042667, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 4.198239, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -6.530584, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, -0.215556, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, -0.976384, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, 0.003121, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=71;Time=133742.724");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadInstance_87(Neuro::DicomInstancesCollection& target)
{
  // ('dcm_qa_uih', 'In/DTI_132225/dti_tra_dir16_PA_rot_SaveBySlc__134057/00000002.dcm')
  Orthanc::DicomMap tags;
  tags.SetValue(0x0008, 0x0060, "MR", false);
  tags.SetValue(0x0008, 0x0070, "UIH", false);
  tags.SetValue(0x0020, 0x0013, "2", false);
  tags.SetValue(0x0018, 0x0080, "5865.0", false);
  tags.SetValue(0x0018, 0x0081, "70.7", false);
  tags.SetValue(0x0018, 0x1312, "COL", false);
  tags.SetValue(0x0008, 0x0016, "1.2.840.10008.5.1.4.1.1.4", false);
  tags.SetValue(0x0028, 0x0002, "1", false);
  tags.SetValue(0x0028, 0x0004, "MONOCHROME2", false);
  tags.SetValue(0x0028, 0x0010, "256", false);
  tags.SetValue(0x0028, 0x0011, "320", false);
  tags.SetValue(0x0028, 0x0100, "16", false);
  tags.SetValue(0x0028, 0x0101, "16", false);
  tags.SetValue(0x0028, 0x0102, "15", false);
  tags.SetValue(0x0028, 0x0103, "0", false);
  tags.SetValue(0x0018, 0x0050, "3.5", false);
  tags.SetValue(0x0018, 0x0088, "4.19999981", false);
  tags.SetValue(0x0020, 0x0037, "0.906671703\\-0.4210186\\0.026263684\\0.420842081\\0.90705198\\0.0121905897", false);
  tags.SetValue(0x0028, 0x0030, "3.5\\3.5", false);
  std::unique_ptr<Neuro::InputDicomInstance> dicom(new Neuro::InputDicomInstance(tags));
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925133759.986000", false);
    uih.SetValue(0x0008, 0x0032, "133759.986000", false);
    uih.SetValue(0x0018, 0x9073, "21.0", false);
    uih.SetValue(0x0020, 0x0032, "-154.647507\\-58.5786934\\-9.21860886", false);
    uih.SetValue(0x0020, 0x1041, "-4.73692799", false);
    uih.SetValue(0x0065, 0x100f, "F4.7", false);
    uih.SetValue(0x0065, 0x1015, "-5.96596575\\-4.14295006\\-4.91173077", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925133757.060000", false);
    uih.SetValue(0x0008, 0x0032, "133757.060000", false);
    uih.SetValue(0x0018, 0x9073, "21.0", false);
    uih.SetValue(0x0020, 0x0032, "-154.769119\\-58.5786934\\-5.02037001", false);
    uih.SetValue(0x0020, 0x1041, "-0.536926806", false);
    uih.SetValue(0x0065, 0x100f, "F0.5", false);
    uih.SetValue(0x0065, 0x1015, "-6.08757687\\-4.14295006\\-0.713491559", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925133759.693000", false);
    uih.SetValue(0x0008, 0x0032, "133759.693000", false);
    uih.SetValue(0x0018, 0x9073, "22.0", false);
    uih.SetValue(0x0020, 0x0032, "-154.890732\\-58.5786934\\-0.822131157", false);
    uih.SetValue(0x0020, 0x1041, "3.66307425", false);
    uih.SetValue(0x0065, 0x100f, "H3.7", false);
    uih.SetValue(0x0065, 0x1015, "-6.20918798\\-4.14295006\\3.48474741", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925133756.767000", false);
    uih.SetValue(0x0008, 0x0032, "133756.767000", false);
    uih.SetValue(0x0018, 0x9073, "22.0", false);
    uih.SetValue(0x0020, 0x0032, "-155.012344\\-58.5786934\\3.37610793", false);
    uih.SetValue(0x0020, 0x1041, "7.86307573", false);
    uih.SetValue(0x0065, 0x100f, "H7.9", false);
    uih.SetValue(0x0065, 0x1015, "-6.33079863\\-4.14295006\\7.68298626", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925133759.401000", false);
    uih.SetValue(0x0008, 0x0032, "133759.401000", false);
    uih.SetValue(0x0018, 0x9073, "21.0", false);
    uih.SetValue(0x0020, 0x0032, "-155.133957\\-58.5786934\\7.57434702", false);
    uih.SetValue(0x0020, 0x1041, "12.063077", false);
    uih.SetValue(0x0065, 0x100f, "H12.1", false);
    uih.SetValue(0x0065, 0x1015, "-6.45240974\\-4.14295006\\11.8812256", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925133756.475000", false);
    uih.SetValue(0x0008, 0x0032, "133756.475000", false);
    uih.SetValue(0x0018, 0x9073, "21.0", false);
    uih.SetValue(0x0020, 0x0032, "-155.255569\\-58.5786934\\11.7725859", false);
    uih.SetValue(0x0020, 0x1041, "16.2630787", false);
    uih.SetValue(0x0065, 0x100f, "H16.3", false);
    uih.SetValue(0x0065, 0x1015, "-6.57402086\\-4.14295006\\16.079464", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925133759.108000", false);
    uih.SetValue(0x0008, 0x0032, "133759.108000", false);
    uih.SetValue(0x0018, 0x9073, "21.0", false);
    uih.SetValue(0x0020, 0x0032, "-155.377167\\-58.5786934\\15.9708252", false);
    uih.SetValue(0x0020, 0x1041, "20.4630795", false);
    uih.SetValue(0x0065, 0x100f, "H20.5", false);
    uih.SetValue(0x0065, 0x1015, "-6.6956315\\-4.14295006\\20.2777042", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925133756.182000", false);
    uih.SetValue(0x0008, 0x0032, "133756.182000", false);
    uih.SetValue(0x0018, 0x9073, "22.0", false);
    uih.SetValue(0x0020, 0x0032, "-155.498779\\-58.5786934\\20.1690636", false);
    uih.SetValue(0x0020, 0x1041, "24.6630802", false);
    uih.SetValue(0x0065, 0x100f, "H24.7", false);
    uih.SetValue(0x0065, 0x1015, "-6.81724262\\-4.14295006\\24.4759426", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925133758.815000", false);
    uih.SetValue(0x0008, 0x0032, "133758.815000", false);
    uih.SetValue(0x0018, 0x9073, "22.0", false);
    uih.SetValue(0x0020, 0x0032, "-155.620392\\-58.5786934\\24.3673019", false);
    uih.SetValue(0x0020, 0x1041, "28.863081", false);
    uih.SetValue(0x0065, 0x100f, "H28.9", false);
    uih.SetValue(0x0065, 0x1015, "-6.93885374\\-4.14295006\\28.674181", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925133755.890000", false);
    uih.SetValue(0x0008, 0x0032, "133755.890000", false);
    uih.SetValue(0x0018, 0x9073, "21.0", false);
    uih.SetValue(0x0020, 0x0032, "-155.742004\\-58.5786934\\28.5655422", false);
    uih.SetValue(0x0020, 0x1041, "33.0630836", false);
    uih.SetValue(0x0065, 0x100f, "H33.1", false);
    uih.SetValue(0x0065, 0x1015, "-7.06046438\\-4.14295006\\32.8724213", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925133758.523000", false);
    uih.SetValue(0x0008, 0x0032, "133758.523000", false);
    uih.SetValue(0x0018, 0x9073, "21.0", false);
    uih.SetValue(0x0020, 0x0032, "-155.863617\\-58.5786934\\32.7637825", false);
    uih.SetValue(0x0020, 0x1041, "37.2630844", false);
    uih.SetValue(0x0065, 0x100f, "H37.3", false);
    uih.SetValue(0x0065, 0x1015, "-7.1820755\\-4.14295006\\37.0706596", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925133755.597000", false);
    uih.SetValue(0x0008, 0x0032, "133755.597000", false);
    uih.SetValue(0x0018, 0x9073, "22.0", false);
    uih.SetValue(0x0020, 0x0032, "-155.985229\\-58.5786934\\36.9620209", false);
    uih.SetValue(0x0020, 0x1041, "41.4630852", false);
    uih.SetValue(0x0065, 0x100f, "H41.5", false);
    uih.SetValue(0x0065, 0x1015, "-7.30368662\\-4.14295006\\41.268898", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925133758.230000", false);
    uih.SetValue(0x0008, 0x0032, "133758.230000", false);
    uih.SetValue(0x0018, 0x9073, "22.0", false);
    uih.SetValue(0x0020, 0x0032, "-156.106842\\-58.5786934\\41.1602592", false);
    uih.SetValue(0x0020, 0x1041, "45.6630859", false);
    uih.SetValue(0x0065, 0x100f, "H45.7", false);
    uih.SetValue(0x0065, 0x1015, "-7.42529726\\-4.14295006\\45.4671364", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925133755.305000", false);
    uih.SetValue(0x0008, 0x0032, "133755.305000", false);
    uih.SetValue(0x0018, 0x9073, "21.0", false);
    uih.SetValue(0x0020, 0x0032, "-156.228455\\-58.5786934\\45.3584976", false);
    uih.SetValue(0x0020, 0x1041, "49.8630867", false);
    uih.SetValue(0x0065, 0x100f, "H49.9", false);
    uih.SetValue(0x0065, 0x1015, "-7.54690838\\-4.14295006\\49.6653748", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925133757.938000", false);
    uih.SetValue(0x0008, 0x0032, "133757.938000", false);
    uih.SetValue(0x0018, 0x9073, "21.0", false);
    uih.SetValue(0x0020, 0x0032, "-156.350067\\-58.5786934\\49.556736", false);
    uih.SetValue(0x0020, 0x1041, "54.0630875", false);
    uih.SetValue(0x0065, 0x100f, "H54.1", false);
    uih.SetValue(0x0065, 0x1015, "-7.6685195\\-4.14295006\\53.8636169", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925133755.012000", false);
    uih.SetValue(0x0008, 0x0032, "133755.012000", false);
    uih.SetValue(0x0018, 0x9073, "21.0", false);
    uih.SetValue(0x0020, 0x0032, "-156.471664\\-58.5786934\\53.7549744", false);
    uih.SetValue(0x0020, 0x1041, "58.2630882", false);
    uih.SetValue(0x0065, 0x100f, "H58.3", false);
    uih.SetValue(0x0065, 0x1015, "-7.79013014\\-4.14295006\\58.0618553", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925133757.645000", false);
    uih.SetValue(0x0008, 0x0032, "133757.645000", false);
    uih.SetValue(0x0018, 0x9073, "22.0", false);
    uih.SetValue(0x0020, 0x0032, "-156.593277\\-58.5786934\\57.9532166", false);
    uih.SetValue(0x0020, 0x1041, "62.4630928", false);
    uih.SetValue(0x0065, 0x100f, "H62.5", false);
    uih.SetValue(0x0065, 0x1015, "-7.91174126\\-4.14295006\\62.2600937", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925133754.719000", false);
    uih.SetValue(0x0008, 0x0032, "133754.719000", false);
    uih.SetValue(0x0018, 0x9073, "22.0", false);
    uih.SetValue(0x0020, 0x0032, "-156.71489\\-58.5786934\\62.1514549", false);
    uih.SetValue(0x0020, 0x1041, "66.6630936", false);
    uih.SetValue(0x0065, 0x100f, "H66.7", false);
    uih.SetValue(0x0065, 0x1015, "-8.0333519\\-4.14295006\\66.4583359", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925133757.353000", false);
    uih.SetValue(0x0008, 0x0032, "133757.353000", false);
    uih.SetValue(0x0018, 0x9073, "21.0", false);
    uih.SetValue(0x0020, 0x0032, "-156.836502\\-58.5786934\\66.3496933", false);
    uih.SetValue(0x0020, 0x1041, "70.8630905", false);
    uih.SetValue(0x0065, 0x100f, "H70.9", false);
    uih.SetValue(0x0065, 0x1015, "-8.15496349\\-4.14295006\\70.6565704", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925133754.427000", false);
    uih.SetValue(0x0008, 0x0032, "133754.427000", false);
    uih.SetValue(0x0018, 0x9073, "21.0", false);
    uih.SetValue(0x0020, 0x0032, "-156.958115\\-58.5786934\\70.5479355", false);
    uih.SetValue(0x0020, 0x1041, "75.0630951", false);
    uih.SetValue(0x0065, 0x100f, "H75.1", false);
    uih.SetValue(0x0065, 0x1015, "-8.27657413\\-4.14295006\\74.8548126", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  target.AddInstance(dicom.release(), "nope");
}

TEST(Autogenerated, Test_87)
{
  Neuro::DicomInstancesCollection instances;
  LoadInstance_87(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 64);
  ASSERT_NEAR(nifti.pixdim[1], 3.500000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 64);
  ASSERT_NEAR(nifti.pixdim[2], 3.500000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 20);
  ASSERT_NEAR(nifti.pixdim[3], 4.200000, 0.00001);
  ASSERT_EQ(nifti.ndim, 3);
  ASSERT_EQ(nifti.dim[0], 3);
  ASSERT_EQ(nifti.nvox, 81920u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(57));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(57));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(57));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -3.173351, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 1.472947, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], 0.121611, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 61.851830, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], 1.473565, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 3.174682, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], -141.426270, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.091923, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], -0.042667, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 4.198239, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -6.530584, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, -0.215556, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, -0.976384, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, 0.003121, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=71;Time=133754.427");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadInstance_88(Neuro::DicomInstancesCollection& target)
{
  // ('dcm_qa_uih', 'In/DTI_132225/t1_gre_fsp_3d_sag__132750/00000001.dcm')
  Orthanc::DicomMap tags;
  tags.SetValue(0x0008, 0x0060, "MR", false);
  tags.SetValue(0x0008, 0x0070, "UIH", false);
  tags.SetValue(0x0020, 0x0013, "1", false);
  tags.SetValue(0x0018, 0x0080, "7.25", false);
  tags.SetValue(0x0018, 0x0081, "3.1", false);
  tags.SetValue(0x0018, 0x1312, "ROW", false);
  tags.SetValue(0x0008, 0x0016, "1.2.840.10008.5.1.4.1.1.4", false);
  tags.SetValue(0x0028, 0x0002, "1", false);
  tags.SetValue(0x0028, 0x0004, "MONOCHROME2", false);
  tags.SetValue(0x0028, 0x0010, "512", false);
  tags.SetValue(0x0028, 0x0011, "460", false);
  tags.SetValue(0x0028, 0x0100, "16", false);
  tags.SetValue(0x0028, 0x0101, "16", false);
  tags.SetValue(0x0028, 0x0102, "15", false);
  tags.SetValue(0x0028, 0x0103, "0", false);
  tags.SetValue(0x0008, 0x0032, "132417.803000", false);
  tags.SetValue(0x0018, 0x0050, "1", false);
  tags.SetValue(0x0018, 0x0088, "1", false);
  tags.SetValue(0x0020, 0x0032, "-93.2203293\\-110.039673\\126.103348", false);
  tags.SetValue(0x0020, 0x0037, "0.0679929852\\0.99768579\\-0\\-0\\-0\\-1", false);
  tags.SetValue(0x0028, 0x0030, "0.5\\0.5", false);
  std::unique_ptr<Neuro::InputDicomInstance> dicom(new Neuro::InputDicomInstance(tags));
  target.AddInstance(dicom.release(), "nope");
}

TEST(Autogenerated, Test_88)
{
  Neuro::DicomInstancesCollection instances;
  LoadInstance_88(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 460);
  ASSERT_NEAR(nifti.pixdim[1], 0.500000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 512);
  ASSERT_NEAR(nifti.pixdim[2], 0.500000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 1);
  ASSERT_NEAR(nifti.pixdim[3], 1.000000, 0.00001);
  ASSERT_EQ(nifti.ndim, 3);
  ASSERT_EQ(nifti.dim[0], 3);
  ASSERT_EQ(nifti.nvox, 235520u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(54));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(54));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(54));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -0.033996, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], 0.997686, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 93.220329, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], -0.498843, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], -0.067993, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], 110.039673, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 0.500000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -129.396652, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.482703, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, -0.516719, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, -0.516719, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=3.1;Time=132417.803");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadInstance_89(Neuro::DicomInstancesCollection& target)
{
  // ('dcm_qa_uih', 'In/DTI_132225/t1_gre_fsp_3d_sag__132750/00000002.dcm')
  Orthanc::DicomMap tags;
  tags.SetValue(0x0008, 0x0060, "MR", false);
  tags.SetValue(0x0008, 0x0070, "UIH", false);
  tags.SetValue(0x0020, 0x0013, "2", false);
  tags.SetValue(0x0018, 0x0080, "7.25", false);
  tags.SetValue(0x0018, 0x0081, "3.1", false);
  tags.SetValue(0x0018, 0x1312, "ROW", false);
  tags.SetValue(0x0008, 0x0016, "1.2.840.10008.5.1.4.1.1.4", false);
  tags.SetValue(0x0028, 0x0002, "1", false);
  tags.SetValue(0x0028, 0x0004, "MONOCHROME2", false);
  tags.SetValue(0x0028, 0x0010, "512", false);
  tags.SetValue(0x0028, 0x0011, "460", false);
  tags.SetValue(0x0028, 0x0100, "16", false);
  tags.SetValue(0x0028, 0x0101, "16", false);
  tags.SetValue(0x0028, 0x0102, "15", false);
  tags.SetValue(0x0028, 0x0103, "0", false);
  tags.SetValue(0x0008, 0x0032, "132417.803000", false);
  tags.SetValue(0x0018, 0x0050, "1", false);
  tags.SetValue(0x0018, 0x0088, "1", false);
  tags.SetValue(0x0020, 0x0032, "-92.222641\\-110.107666\\126.103348", false);
  tags.SetValue(0x0020, 0x0037, "0.0679929852\\0.99768579\\-0\\-0\\-0\\-1", false);
  tags.SetValue(0x0028, 0x0030, "0.5\\0.5", false);
  std::unique_ptr<Neuro::InputDicomInstance> dicom(new Neuro::InputDicomInstance(tags));
  target.AddInstance(dicom.release(), "nope");
}

TEST(Autogenerated, Test_89)
{
  Neuro::DicomInstancesCollection instances;
  LoadInstance_89(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 460);
  ASSERT_NEAR(nifti.pixdim[1], 0.500000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 512);
  ASSERT_NEAR(nifti.pixdim[2], 0.500000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 1);
  ASSERT_NEAR(nifti.pixdim[3], 1.000000, 0.00001);
  ASSERT_EQ(nifti.ndim, 3);
  ASSERT_EQ(nifti.dim[0], 3);
  ASSERT_EQ(nifti.nvox, 235520u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(54));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(54));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(54));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -0.033996, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], 0.997686, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 92.222641, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], -0.498843, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], -0.067993, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], 110.107666, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 0.500000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -129.396652, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.482703, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, -0.516719, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, -0.516719, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=3.1;Time=132417.803");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadInstance_90(Neuro::DicomInstancesCollection& target)
{
  // ('dcm_qa_uih', 'In/DTI_134434/dti_tra_dir16_AP_SaveBySlc__140028/00000001.dcm')
  Orthanc::DicomMap tags;
  tags.SetValue(0x0008, 0x0060, "MR", false);
  tags.SetValue(0x0008, 0x0070, "UIH", false);
  tags.SetValue(0x0020, 0x0013, "1", false);
  tags.SetValue(0x0018, 0x0080, "5865.0", false);
  tags.SetValue(0x0018, 0x0081, "71.5", false);
  tags.SetValue(0x0018, 0x1312, "COL", false);
  tags.SetValue(0x0008, 0x0016, "1.2.840.10008.5.1.4.1.1.4", false);
  tags.SetValue(0x0028, 0x0002, "1", false);
  tags.SetValue(0x0028, 0x0004, "MONOCHROME2", false);
  tags.SetValue(0x0028, 0x0010, "256", false);
  tags.SetValue(0x0028, 0x0011, "320", false);
  tags.SetValue(0x0028, 0x0100, "16", false);
  tags.SetValue(0x0028, 0x0101, "16", false);
  tags.SetValue(0x0028, 0x0102, "15", false);
  tags.SetValue(0x0028, 0x0103, "0", false);
  tags.SetValue(0x0018, 0x0050, "3.5", false);
  tags.SetValue(0x0018, 0x0088, "4.19999981", false);
  tags.SetValue(0x0020, 0x0037, "0.999113142\\0.0305822305\\0.0289414488\\-0.0305694081\\0.999532282\\-0.000885508256", false);
  tags.SetValue(0x0028, 0x0030, "3.5\\3.5", false);
  std::unique_ptr<Neuro::InputDicomInstance> dicom(new Neuro::InputDicomInstance(tags));
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925135718.843000", false);
    uih.SetValue(0x0008, 0x0032, "135718.843000", false);
    uih.SetValue(0x0018, 0x9073, "21.0", false);
    uih.SetValue(0x0020, 0x0032, "-114.442863\\-112.324295\\-6.39442587", false);
    uih.SetValue(0x0020, 0x1041, "-3.07805347", false);
    uih.SetValue(0x0065, 0x100f, "F3.1", false);
    uih.SetValue(0x0065, 0x1015, "-5.96596575\\3.0485301\\-3.25216055", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925135715.917000", false);
    uih.SetValue(0x0008, 0x0032, "135715.917000", false);
    uih.SetValue(0x0018, 0x9073, "21.0", false);
    uih.SetValue(0x0020, 0x0032, "-114.564476\\-112.324295\\-2.19618678", false);
    uih.SetValue(0x0020, 0x1041, "1.12194777", false);
    uih.SetValue(0x0065, 0x100f, "H1.1", false);
    uih.SetValue(0x0065, 0x1015, "-6.08757687\\3.0485301\\0.94607842", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925135718.550000", false);
    uih.SetValue(0x0008, 0x0032, "135718.550000", false);
    uih.SetValue(0x0018, 0x9073, "21.0", false);
    uih.SetValue(0x0020, 0x0032, "-114.686089\\-112.324295\\2.00205207", false);
    uih.SetValue(0x0020, 0x1041, "5.32194901", false);
    uih.SetValue(0x0065, 0x100f, "H5.3", false);
    uih.SetValue(0x0065, 0x1015, "-6.20918798\\3.0485301\\5.14431763", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925135715.624000", false);
    uih.SetValue(0x0008, 0x0032, "135715.624000", false);
    uih.SetValue(0x0018, 0x9073, "22.0", false);
    uih.SetValue(0x0020, 0x0032, "-114.807701\\-112.324295\\6.20029116", false);
    uih.SetValue(0x0020, 0x1041, "9.52194977", false);
    uih.SetValue(0x0065, 0x100f, "H9.5", false);
    uih.SetValue(0x0065, 0x1015, "-6.33079863\\3.0485301\\9.342556", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925135718.258000", false);
    uih.SetValue(0x0008, 0x0032, "135718.258000", false);
    uih.SetValue(0x0018, 0x9073, "21.0", false);
    uih.SetValue(0x0020, 0x0032, "-114.929314\\-112.324295\\10.39853", false);
    uih.SetValue(0x0020, 0x1041, "13.7219515", false);
    uih.SetValue(0x0065, 0x100f, "H13.7", false);
    uih.SetValue(0x0065, 0x1015, "-6.45240974\\3.0485301\\13.5407953", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925135715.332000", false);
    uih.SetValue(0x0008, 0x0032, "135715.332000", false);
    uih.SetValue(0x0018, 0x9073, "21.0", false);
    uih.SetValue(0x0020, 0x0032, "-115.050919\\-112.324295\\14.5967693", false);
    uih.SetValue(0x0020, 0x1041, "17.9219532", false);
    uih.SetValue(0x0065, 0x100f, "H17.9", false);
    uih.SetValue(0x0065, 0x1015, "-6.57402086\\3.0485301\\17.7390347", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925135717.965000", false);
    uih.SetValue(0x0008, 0x0032, "135717.965000", false);
    uih.SetValue(0x0018, 0x9073, "21.0", false);
    uih.SetValue(0x0020, 0x0032, "-115.172531\\-112.324295\\18.7950077", false);
    uih.SetValue(0x0020, 0x1041, "22.121954", false);
    uih.SetValue(0x0065, 0x100f, "H22.1", false);
    uih.SetValue(0x0065, 0x1015, "-6.6956315\\3.0485301\\21.937273", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925135715.039000", false);
    uih.SetValue(0x0008, 0x0032, "135715.039000", false);
    uih.SetValue(0x0018, 0x9073, "22.0", false);
    uih.SetValue(0x0020, 0x0032, "-115.294144\\-112.324295\\22.993248", false);
    uih.SetValue(0x0020, 0x1041, "26.3219547", false);
    uih.SetValue(0x0065, 0x100f, "H26.3", false);
    uih.SetValue(0x0065, 0x1015, "-6.81724262\\3.0485301\\26.1355133", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925135717.672000", false);
    uih.SetValue(0x0008, 0x0032, "135717.672000", false);
    uih.SetValue(0x0018, 0x9073, "22.0", false);
    uih.SetValue(0x0020, 0x0032, "-115.415756\\-112.324295\\27.1914864", false);
    uih.SetValue(0x0020, 0x1041, "30.5219555", false);
    uih.SetValue(0x0065, 0x100f, "H30.5", false);
    uih.SetValue(0x0065, 0x1015, "-6.93885374\\3.0485301\\30.3337517", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925135714.747000", false);
    uih.SetValue(0x0008, 0x0032, "135714.747000", false);
    uih.SetValue(0x0018, 0x9073, "21.0", false);
    uih.SetValue(0x0020, 0x0032, "-115.537369\\-112.324295\\31.3897247", false);
    uih.SetValue(0x0020, 0x1041, "34.7219582", false);
    uih.SetValue(0x0065, 0x100f, "H34.7", false);
    uih.SetValue(0x0065, 0x1015, "-7.06046438\\3.0485301\\34.5319901", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925135717.380000", false);
    uih.SetValue(0x0008, 0x0032, "135717.380000", false);
    uih.SetValue(0x0018, 0x9073, "21.0", false);
    uih.SetValue(0x0020, 0x0032, "-115.658974\\-112.324295\\35.5879631", false);
    uih.SetValue(0x0020, 0x1041, "38.9219589", false);
    uih.SetValue(0x0065, 0x100f, "H38.9", false);
    uih.SetValue(0x0065, 0x1015, "-7.1820755\\3.0485301\\38.7302284", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925135714.454000", false);
    uih.SetValue(0x0008, 0x0032, "135714.454000", false);
    uih.SetValue(0x0018, 0x9073, "21.0", false);
    uih.SetValue(0x0020, 0x0032, "-115.780586\\-112.324295\\39.7862015", false);
    uih.SetValue(0x0020, 0x1041, "43.1219597", false);
    uih.SetValue(0x0065, 0x100f, "H43.1", false);
    uih.SetValue(0x0065, 0x1015, "-7.30368662\\3.0485301\\42.9284668", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925135717.087000", false);
    uih.SetValue(0x0008, 0x0032, "135717.087000", false);
    uih.SetValue(0x0018, 0x9073, "22.0", false);
    uih.SetValue(0x0020, 0x0032, "-115.902199\\-112.324295\\43.9844437", false);
    uih.SetValue(0x0020, 0x1041, "47.3219604", false);
    uih.SetValue(0x0065, 0x100f, "H47.3", false);
    uih.SetValue(0x0065, 0x1015, "-7.42529726\\3.0485301\\47.126709", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925135714.161000", false);
    uih.SetValue(0x0008, 0x0032, "135714.161000", false);
    uih.SetValue(0x0018, 0x9073, "22.0", false);
    uih.SetValue(0x0020, 0x0032, "-116.023811\\-112.324295\\48.182682", false);
    uih.SetValue(0x0020, 0x1041, "51.5219612", false);
    uih.SetValue(0x0065, 0x100f, "H51.5", false);
    uih.SetValue(0x0065, 0x1015, "-7.54690838\\3.0485301\\51.3249474", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925135716.795000", false);
    uih.SetValue(0x0008, 0x0032, "135716.795000", false);
    uih.SetValue(0x0018, 0x9073, "21.0", false);
    uih.SetValue(0x0020, 0x0032, "-116.145424\\-112.324295\\52.3809204", false);
    uih.SetValue(0x0020, 0x1041, "55.721962", false);
    uih.SetValue(0x0065, 0x100f, "H55.7", false);
    uih.SetValue(0x0065, 0x1015, "-7.6685195\\3.0485301\\55.5231857", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925135713.869000", false);
    uih.SetValue(0x0008, 0x0032, "135713.869000", false);
    uih.SetValue(0x0018, 0x9073, "21.0", false);
    uih.SetValue(0x0020, 0x0032, "-116.267029\\-112.324295\\56.5791588", false);
    uih.SetValue(0x0020, 0x1041, "59.9219627", false);
    uih.SetValue(0x0065, 0x100f, "H59.9", false);
    uih.SetValue(0x0065, 0x1015, "-7.79013014\\3.0485301\\59.7214241", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925135716.502000", false);
    uih.SetValue(0x0008, 0x0032, "135716.502000", false);
    uih.SetValue(0x0018, 0x9073, "21.0", false);
    uih.SetValue(0x0020, 0x0032, "-116.388641\\-112.324295\\60.7773972", false);
    uih.SetValue(0x0020, 0x1041, "64.1219635", false);
    uih.SetValue(0x0065, 0x100f, "H64.1", false);
    uih.SetValue(0x0065, 0x1015, "-7.91174126\\3.0485301\\63.9196625", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925135713.576000", false);
    uih.SetValue(0x0008, 0x0032, "135713.576000", false);
    uih.SetValue(0x0018, 0x9073, "22.0", false);
    uih.SetValue(0x0020, 0x0032, "-116.510254\\-112.324295\\64.9756393", false);
    uih.SetValue(0x0020, 0x1041, "68.3219681", false);
    uih.SetValue(0x0065, 0x100f, "H68.3", false);
    uih.SetValue(0x0065, 0x1015, "-8.0333519\\3.0485301\\68.1179047", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925135716.209000", false);
    uih.SetValue(0x0008, 0x0032, "135716.209000", false);
    uih.SetValue(0x0018, 0x9073, "22.0", false);
    uih.SetValue(0x0020, 0x0032, "-116.631866\\-112.324295\\69.1738739", false);
    uih.SetValue(0x0020, 0x1041, "72.521965", false);
    uih.SetValue(0x0065, 0x100f, "H72.5", false);
    uih.SetValue(0x0065, 0x1015, "-8.15496349\\3.0485301\\72.3161392", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925135713.284000", false);
    uih.SetValue(0x0008, 0x0032, "135713.284000", false);
    uih.SetValue(0x0018, 0x9073, "21.0", false);
    uih.SetValue(0x0020, 0x0032, "-116.753471\\-112.324295\\73.3721161", false);
    uih.SetValue(0x0020, 0x1041, "76.7219696", false);
    uih.SetValue(0x0065, 0x100f, "H76.7", false);
    uih.SetValue(0x0065, 0x1015, "-8.27657413\\3.0485301\\76.5143814", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  target.AddInstance(dicom.release(), "nope");
}

TEST(Autogenerated, Test_90)
{
  Neuro::DicomInstancesCollection instances;
  LoadInstance_90(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 64);
  ASSERT_NEAR(nifti.pixdim[1], 3.500000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 64);
  ASSERT_NEAR(nifti.pixdim[2], 3.500000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 20);
  ASSERT_NEAR(nifti.pixdim[3], 4.200000, 0.00001);
  ASSERT_EQ(nifti.ndim, 3);
  ASSERT_EQ(nifti.dim[0], 3);
  ASSERT_EQ(nifti.nvox, 81920u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(57));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(57));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(57));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -3.496896, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], -0.106993, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], 0.121611, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 121.183418, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], -0.107038, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 3.498363, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], -108.072556, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.101295, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 0.003099, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 4.198239, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -6.589681, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.015291, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, -0.999778, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, -0.000221, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=72;Time=135713.284");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadInstance_91(Neuro::DicomInstancesCollection& target)
{
  // ('dcm_qa_uih', 'In/DTI_134434/dti_tra_dir16_AP_SaveBySlc__140028/00000002.dcm')
  Orthanc::DicomMap tags;
  tags.SetValue(0x0008, 0x0060, "MR", false);
  tags.SetValue(0x0008, 0x0070, "UIH", false);
  tags.SetValue(0x0020, 0x0013, "2", false);
  tags.SetValue(0x0018, 0x0080, "5865.0", false);
  tags.SetValue(0x0018, 0x0081, "71.5", false);
  tags.SetValue(0x0018, 0x1312, "COL", false);
  tags.SetValue(0x0008, 0x0016, "1.2.840.10008.5.1.4.1.1.4", false);
  tags.SetValue(0x0028, 0x0002, "1", false);
  tags.SetValue(0x0028, 0x0004, "MONOCHROME2", false);
  tags.SetValue(0x0028, 0x0010, "256", false);
  tags.SetValue(0x0028, 0x0011, "320", false);
  tags.SetValue(0x0028, 0x0100, "16", false);
  tags.SetValue(0x0028, 0x0101, "16", false);
  tags.SetValue(0x0028, 0x0102, "15", false);
  tags.SetValue(0x0028, 0x0103, "0", false);
  tags.SetValue(0x0018, 0x0050, "3.5", false);
  tags.SetValue(0x0018, 0x0088, "4.19999981", false);
  tags.SetValue(0x0020, 0x0037, "0.999113142\\0.0305822305\\0.0289414488\\-0.0305694081\\0.999532282\\-0.000885508256", false);
  tags.SetValue(0x0028, 0x0030, "3.5\\3.5", false);
  std::unique_ptr<Neuro::InputDicomInstance> dicom(new Neuro::InputDicomInstance(tags));
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925135730.546000", false);
    uih.SetValue(0x0008, 0x0032, "135730.546000", false);
    uih.SetValue(0x0018, 0x9073, "21.0", false);
    uih.SetValue(0x0020, 0x0032, "-114.442863\\-112.324295\\-6.39442587", false);
    uih.SetValue(0x0020, 0x1041, "-3.07805347", false);
    uih.SetValue(0x0065, 0x100f, "F3.1", false);
    uih.SetValue(0x0065, 0x1015, "-5.96596575\\3.0485301\\-3.25216055", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925135727.620000", false);
    uih.SetValue(0x0008, 0x0032, "135727.620000", false);
    uih.SetValue(0x0018, 0x9073, "21.0", false);
    uih.SetValue(0x0020, 0x0032, "-114.564476\\-112.324295\\-2.19618678", false);
    uih.SetValue(0x0020, 0x1041, "1.12194777", false);
    uih.SetValue(0x0065, 0x100f, "H1.1", false);
    uih.SetValue(0x0065, 0x1015, "-6.08757687\\3.0485301\\0.94607842", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925135730.253000", false);
    uih.SetValue(0x0008, 0x0032, "135730.253000", false);
    uih.SetValue(0x0018, 0x9073, "21.0", false);
    uih.SetValue(0x0020, 0x0032, "-114.686089\\-112.324295\\2.00205207", false);
    uih.SetValue(0x0020, 0x1041, "5.32194901", false);
    uih.SetValue(0x0065, 0x100f, "H5.3", false);
    uih.SetValue(0x0065, 0x1015, "-6.20918798\\3.0485301\\5.14431763", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925135727.327000", false);
    uih.SetValue(0x0008, 0x0032, "135727.327000", false);
    uih.SetValue(0x0018, 0x9073, "22.0", false);
    uih.SetValue(0x0020, 0x0032, "-114.807701\\-112.324295\\6.20029116", false);
    uih.SetValue(0x0020, 0x1041, "9.52194977", false);
    uih.SetValue(0x0065, 0x100f, "H9.5", false);
    uih.SetValue(0x0065, 0x1015, "-6.33079863\\3.0485301\\9.342556", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925135729.960000", false);
    uih.SetValue(0x0008, 0x0032, "135729.960000", false);
    uih.SetValue(0x0018, 0x9073, "22.0", false);
    uih.SetValue(0x0020, 0x0032, "-114.929314\\-112.324295\\10.39853", false);
    uih.SetValue(0x0020, 0x1041, "13.7219515", false);
    uih.SetValue(0x0065, 0x100f, "H13.7", false);
    uih.SetValue(0x0065, 0x1015, "-6.45240974\\3.0485301\\13.5407953", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925135727.035000", false);
    uih.SetValue(0x0008, 0x0032, "135727.035000", false);
    uih.SetValue(0x0018, 0x9073, "21.0", false);
    uih.SetValue(0x0020, 0x0032, "-115.050919\\-112.324295\\14.5967693", false);
    uih.SetValue(0x0020, 0x1041, "17.9219532", false);
    uih.SetValue(0x0065, 0x100f, "H17.9", false);
    uih.SetValue(0x0065, 0x1015, "-6.57402086\\3.0485301\\17.7390347", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925135729.668000", false);
    uih.SetValue(0x0008, 0x0032, "135729.668000", false);
    uih.SetValue(0x0018, 0x9073, "21.0", false);
    uih.SetValue(0x0020, 0x0032, "-115.172531\\-112.324295\\18.7950077", false);
    uih.SetValue(0x0020, 0x1041, "22.121954", false);
    uih.SetValue(0x0065, 0x100f, "H22.1", false);
    uih.SetValue(0x0065, 0x1015, "-6.6956315\\3.0485301\\21.937273", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925135726.742000", false);
    uih.SetValue(0x0008, 0x0032, "135726.742000", false);
    uih.SetValue(0x0018, 0x9073, "21.0", false);
    uih.SetValue(0x0020, 0x0032, "-115.294144\\-112.324295\\22.993248", false);
    uih.SetValue(0x0020, 0x1041, "26.3219547", false);
    uih.SetValue(0x0065, 0x100f, "H26.3", false);
    uih.SetValue(0x0065, 0x1015, "-6.81724262\\3.0485301\\26.1355133", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925135729.375000", false);
    uih.SetValue(0x0008, 0x0032, "135729.375000", false);
    uih.SetValue(0x0018, 0x9073, "22.0", false);
    uih.SetValue(0x0020, 0x0032, "-115.415756\\-112.324295\\27.1914864", false);
    uih.SetValue(0x0020, 0x1041, "30.5219555", false);
    uih.SetValue(0x0065, 0x100f, "H30.5", false);
    uih.SetValue(0x0065, 0x1015, "-6.93885374\\3.0485301\\30.3337517", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925135726.450000", false);
    uih.SetValue(0x0008, 0x0032, "135726.450000", false);
    uih.SetValue(0x0018, 0x9073, "21.0", false);
    uih.SetValue(0x0020, 0x0032, "-115.537369\\-112.324295\\31.3897247", false);
    uih.SetValue(0x0020, 0x1041, "34.7219582", false);
    uih.SetValue(0x0065, 0x100f, "H34.7", false);
    uih.SetValue(0x0065, 0x1015, "-7.06046438\\3.0485301\\34.5319901", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925135729.083000", false);
    uih.SetValue(0x0008, 0x0032, "135729.083000", false);
    uih.SetValue(0x0018, 0x9073, "21.0", false);
    uih.SetValue(0x0020, 0x0032, "-115.658974\\-112.324295\\35.5879631", false);
    uih.SetValue(0x0020, 0x1041, "38.9219589", false);
    uih.SetValue(0x0065, 0x100f, "H38.9", false);
    uih.SetValue(0x0065, 0x1015, "-7.1820755\\3.0485301\\38.7302284", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925135726.157000", false);
    uih.SetValue(0x0008, 0x0032, "135726.157000", false);
    uih.SetValue(0x0018, 0x9073, "21.0", false);
    uih.SetValue(0x0020, 0x0032, "-115.780586\\-112.324295\\39.7862015", false);
    uih.SetValue(0x0020, 0x1041, "43.1219597", false);
    uih.SetValue(0x0065, 0x100f, "H43.1", false);
    uih.SetValue(0x0065, 0x1015, "-7.30368662\\3.0485301\\42.9284668", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925135728.790000", false);
    uih.SetValue(0x0008, 0x0032, "135728.790000", false);
    uih.SetValue(0x0018, 0x9073, "22.0", false);
    uih.SetValue(0x0020, 0x0032, "-115.902199\\-112.324295\\43.9844437", false);
    uih.SetValue(0x0020, 0x1041, "47.3219604", false);
    uih.SetValue(0x0065, 0x100f, "H47.3", false);
    uih.SetValue(0x0065, 0x1015, "-7.42529726\\3.0485301\\47.126709", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925135725.864000", false);
    uih.SetValue(0x0008, 0x0032, "135725.864000", false);
    uih.SetValue(0x0018, 0x9073, "22.0", false);
    uih.SetValue(0x0020, 0x0032, "-116.023811\\-112.324295\\48.182682", false);
    uih.SetValue(0x0020, 0x1041, "51.5219612", false);
    uih.SetValue(0x0065, 0x100f, "H51.5", false);
    uih.SetValue(0x0065, 0x1015, "-7.54690838\\3.0485301\\51.3249474", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925135728.498000", false);
    uih.SetValue(0x0008, 0x0032, "135728.498000", false);
    uih.SetValue(0x0018, 0x9073, "21.0", false);
    uih.SetValue(0x0020, 0x0032, "-116.145424\\-112.324295\\52.3809204", false);
    uih.SetValue(0x0020, 0x1041, "55.721962", false);
    uih.SetValue(0x0065, 0x100f, "H55.7", false);
    uih.SetValue(0x0065, 0x1015, "-7.6685195\\3.0485301\\55.5231857", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925135725.572000", false);
    uih.SetValue(0x0008, 0x0032, "135725.572000", false);
    uih.SetValue(0x0018, 0x9073, "21.0", false);
    uih.SetValue(0x0020, 0x0032, "-116.267029\\-112.324295\\56.5791588", false);
    uih.SetValue(0x0020, 0x1041, "59.9219627", false);
    uih.SetValue(0x0065, 0x100f, "H59.9", false);
    uih.SetValue(0x0065, 0x1015, "-7.79013014\\3.0485301\\59.7214241", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925135728.205000", false);
    uih.SetValue(0x0008, 0x0032, "135728.205000", false);
    uih.SetValue(0x0018, 0x9073, "21.0", false);
    uih.SetValue(0x0020, 0x0032, "-116.388641\\-112.324295\\60.7773972", false);
    uih.SetValue(0x0020, 0x1041, "64.1219635", false);
    uih.SetValue(0x0065, 0x100f, "H64.1", false);
    uih.SetValue(0x0065, 0x1015, "-7.91174126\\3.0485301\\63.9196625", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925135725.279000", false);
    uih.SetValue(0x0008, 0x0032, "135725.279000", false);
    uih.SetValue(0x0018, 0x9073, "22.0", false);
    uih.SetValue(0x0020, 0x0032, "-116.510254\\-112.324295\\64.9756393", false);
    uih.SetValue(0x0020, 0x1041, "68.3219681", false);
    uih.SetValue(0x0065, 0x100f, "H68.3", false);
    uih.SetValue(0x0065, 0x1015, "-8.0333519\\3.0485301\\68.1179047", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925135727.912000", false);
    uih.SetValue(0x0008, 0x0032, "135727.912000", false);
    uih.SetValue(0x0018, 0x9073, "22.0", false);
    uih.SetValue(0x0020, 0x0032, "-116.631866\\-112.324295\\69.1738739", false);
    uih.SetValue(0x0020, 0x1041, "72.521965", false);
    uih.SetValue(0x0065, 0x100f, "H72.5", false);
    uih.SetValue(0x0065, 0x1015, "-8.15496349\\3.0485301\\72.3161392", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925135724.987000", false);
    uih.SetValue(0x0008, 0x0032, "135724.987000", false);
    uih.SetValue(0x0018, 0x9073, "21.0", false);
    uih.SetValue(0x0020, 0x0032, "-116.753471\\-112.324295\\73.3721161", false);
    uih.SetValue(0x0020, 0x1041, "76.7219696", false);
    uih.SetValue(0x0065, 0x100f, "H76.7", false);
    uih.SetValue(0x0065, 0x1015, "-8.27657413\\3.0485301\\76.5143814", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  target.AddInstance(dicom.release(), "nope");
}

TEST(Autogenerated, Test_91)
{
  Neuro::DicomInstancesCollection instances;
  LoadInstance_91(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 64);
  ASSERT_NEAR(nifti.pixdim[1], 3.500000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 64);
  ASSERT_NEAR(nifti.pixdim[2], 3.500000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 20);
  ASSERT_NEAR(nifti.pixdim[3], 4.200000, 0.00001);
  ASSERT_EQ(nifti.ndim, 3);
  ASSERT_EQ(nifti.dim[0], 3);
  ASSERT_EQ(nifti.nvox, 81920u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(57));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(57));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(57));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -3.496896, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], -0.106993, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], 0.121611, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 121.183418, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], -0.107038, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 3.498363, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], -108.072556, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.101295, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 0.003099, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 4.198239, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -6.589681, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.015291, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, -0.999778, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, -0.000221, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=72;Time=135724.987");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadInstance_92(Neuro::DicomInstancesCollection& target)
{
  // ('dcm_qa_uih', 'In/DTI_134434/dti_tra_dir16_PA_SaveBySlc__135612/00000001.dcm')
  Orthanc::DicomMap tags;
  tags.SetValue(0x0008, 0x0060, "MR", false);
  tags.SetValue(0x0008, 0x0070, "UIH", false);
  tags.SetValue(0x0020, 0x0013, "1", false);
  tags.SetValue(0x0018, 0x0080, "5865.0", false);
  tags.SetValue(0x0018, 0x0081, "70.7", false);
  tags.SetValue(0x0018, 0x1312, "COL", false);
  tags.SetValue(0x0008, 0x0016, "1.2.840.10008.5.1.4.1.1.4", false);
  tags.SetValue(0x0028, 0x0002, "1", false);
  tags.SetValue(0x0028, 0x0004, "MONOCHROME2", false);
  tags.SetValue(0x0028, 0x0010, "256", false);
  tags.SetValue(0x0028, 0x0011, "320", false);
  tags.SetValue(0x0028, 0x0100, "16", false);
  tags.SetValue(0x0028, 0x0101, "16", false);
  tags.SetValue(0x0028, 0x0102, "15", false);
  tags.SetValue(0x0028, 0x0103, "0", false);
  tags.SetValue(0x0018, 0x0050, "3.5", false);
  tags.SetValue(0x0018, 0x0088, "4.19999981", false);
  tags.SetValue(0x0020, 0x0037, "0.999113142\\0.0305822305\\0.0289414488\\-0.0305694081\\0.999532282\\-0.000885508256", false);
  tags.SetValue(0x0028, 0x0030, "3.5\\3.5", false);
  std::unique_ptr<Neuro::InputDicomInstance> dicom(new Neuro::InputDicomInstance(tags));
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925135303.242000", false);
    uih.SetValue(0x0008, 0x0032, "135303.242000", false);
    uih.SetValue(0x0018, 0x9073, "21.0", false);
    uih.SetValue(0x0020, 0x0032, "-114.442863\\-112.324295\\-6.39442587", false);
    uih.SetValue(0x0020, 0x1041, "-3.07805347", false);
    uih.SetValue(0x0065, 0x100f, "F3.1", false);
    uih.SetValue(0x0065, 0x1015, "-5.96596575\\3.0485301\\-3.25216055", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925135300.316000", false);
    uih.SetValue(0x0008, 0x0032, "135300.316000", false);
    uih.SetValue(0x0018, 0x9073, "22.0", false);
    uih.SetValue(0x0020, 0x0032, "-114.564476\\-112.324295\\-2.19618678", false);
    uih.SetValue(0x0020, 0x1041, "1.12194777", false);
    uih.SetValue(0x0065, 0x100f, "H1.1", false);
    uih.SetValue(0x0065, 0x1015, "-6.08757687\\3.0485301\\0.94607842", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925135302.950000", false);
    uih.SetValue(0x0008, 0x0032, "135302.950000", false);
    uih.SetValue(0x0018, 0x9073, "21.0", false);
    uih.SetValue(0x0020, 0x0032, "-114.686089\\-112.324295\\2.00205207", false);
    uih.SetValue(0x0020, 0x1041, "5.32194901", false);
    uih.SetValue(0x0065, 0x100f, "H5.3", false);
    uih.SetValue(0x0065, 0x1015, "-6.20918798\\3.0485301\\5.14431763", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925135300.024000", false);
    uih.SetValue(0x0008, 0x0032, "135300.024000", false);
    uih.SetValue(0x0018, 0x9073, "21.0", false);
    uih.SetValue(0x0020, 0x0032, "-114.807701\\-112.324295\\6.20029116", false);
    uih.SetValue(0x0020, 0x1041, "9.52194977", false);
    uih.SetValue(0x0065, 0x100f, "H9.5", false);
    uih.SetValue(0x0065, 0x1015, "-6.33079863\\3.0485301\\9.342556", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925135302.657000", false);
    uih.SetValue(0x0008, 0x0032, "135302.657000", false);
    uih.SetValue(0x0018, 0x9073, "21.0", false);
    uih.SetValue(0x0020, 0x0032, "-114.929314\\-112.324295\\10.39853", false);
    uih.SetValue(0x0020, 0x1041, "13.7219515", false);
    uih.SetValue(0x0065, 0x100f, "H13.7", false);
    uih.SetValue(0x0065, 0x1015, "-6.45240974\\3.0485301\\13.5407953", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925135259.731000", false);
    uih.SetValue(0x0008, 0x0032, "135259.731000", false);
    uih.SetValue(0x0018, 0x9073, "22.0", false);
    uih.SetValue(0x0020, 0x0032, "-115.050919\\-112.324295\\14.5967693", false);
    uih.SetValue(0x0020, 0x1041, "17.9219532", false);
    uih.SetValue(0x0065, 0x100f, "H17.9", false);
    uih.SetValue(0x0065, 0x1015, "-6.57402086\\3.0485301\\17.7390347", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925135302.364000", false);
    uih.SetValue(0x0008, 0x0032, "135302.364000", false);
    uih.SetValue(0x0018, 0x9073, "22.0", false);
    uih.SetValue(0x0020, 0x0032, "-115.172531\\-112.324295\\18.7950077", false);
    uih.SetValue(0x0020, 0x1041, "22.121954", false);
    uih.SetValue(0x0065, 0x100f, "H22.1", false);
    uih.SetValue(0x0065, 0x1015, "-6.6956315\\3.0485301\\21.937273", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925135259.439000", false);
    uih.SetValue(0x0008, 0x0032, "135259.439000", false);
    uih.SetValue(0x0018, 0x9073, "21.0", false);
    uih.SetValue(0x0020, 0x0032, "-115.294144\\-112.324295\\22.993248", false);
    uih.SetValue(0x0020, 0x1041, "26.3219547", false);
    uih.SetValue(0x0065, 0x100f, "H26.3", false);
    uih.SetValue(0x0065, 0x1015, "-6.81724262\\3.0485301\\26.1355133", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925135302.072000", false);
    uih.SetValue(0x0008, 0x0032, "135302.072000", false);
    uih.SetValue(0x0018, 0x9073, "21.0", false);
    uih.SetValue(0x0020, 0x0032, "-115.415756\\-112.324295\\27.1914864", false);
    uih.SetValue(0x0020, 0x1041, "30.5219555", false);
    uih.SetValue(0x0065, 0x100f, "H30.5", false);
    uih.SetValue(0x0065, 0x1015, "-6.93885374\\3.0485301\\30.3337517", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925135259.146000", false);
    uih.SetValue(0x0008, 0x0032, "135259.146000", false);
    uih.SetValue(0x0018, 0x9073, "21.0", false);
    uih.SetValue(0x0020, 0x0032, "-115.537369\\-112.324295\\31.3897247", false);
    uih.SetValue(0x0020, 0x1041, "34.7219582", false);
    uih.SetValue(0x0065, 0x100f, "H34.7", false);
    uih.SetValue(0x0065, 0x1015, "-7.06046438\\3.0485301\\34.5319901", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925135301.779000", false);
    uih.SetValue(0x0008, 0x0032, "135301.779000", false);
    uih.SetValue(0x0018, 0x9073, "22.0", false);
    uih.SetValue(0x0020, 0x0032, "-115.658974\\-112.324295\\35.5879631", false);
    uih.SetValue(0x0020, 0x1041, "38.9219589", false);
    uih.SetValue(0x0065, 0x100f, "H38.9", false);
    uih.SetValue(0x0065, 0x1015, "-7.1820755\\3.0485301\\38.7302284", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925135258.854000", false);
    uih.SetValue(0x0008, 0x0032, "135258.854000", false);
    uih.SetValue(0x0018, 0x9073, "21.0", false);
    uih.SetValue(0x0020, 0x0032, "-115.780586\\-112.324295\\39.7862015", false);
    uih.SetValue(0x0020, 0x1041, "43.1219597", false);
    uih.SetValue(0x0065, 0x100f, "H43.1", false);
    uih.SetValue(0x0065, 0x1015, "-7.30368662\\3.0485301\\42.9284668", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925135301.487000", false);
    uih.SetValue(0x0008, 0x0032, "135301.487000", false);
    uih.SetValue(0x0018, 0x9073, "21.0", false);
    uih.SetValue(0x0020, 0x0032, "-115.902199\\-112.324295\\43.9844437", false);
    uih.SetValue(0x0020, 0x1041, "47.3219604", false);
    uih.SetValue(0x0065, 0x100f, "H47.3", false);
    uih.SetValue(0x0065, 0x1015, "-7.42529726\\3.0485301\\47.126709", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925135258.561000", false);
    uih.SetValue(0x0008, 0x0032, "135258.561000", false);
    uih.SetValue(0x0018, 0x9073, "21.0", false);
    uih.SetValue(0x0020, 0x0032, "-116.023811\\-112.324295\\48.182682", false);
    uih.SetValue(0x0020, 0x1041, "51.5219612", false);
    uih.SetValue(0x0065, 0x100f, "H51.5", false);
    uih.SetValue(0x0065, 0x1015, "-7.54690838\\3.0485301\\51.3249474", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925135301.194000", false);
    uih.SetValue(0x0008, 0x0032, "135301.194000", false);
    uih.SetValue(0x0018, 0x9073, "21.0", false);
    uih.SetValue(0x0020, 0x0032, "-116.145424\\-112.324295\\52.3809204", false);
    uih.SetValue(0x0020, 0x1041, "55.721962", false);
    uih.SetValue(0x0065, 0x100f, "H55.7", false);
    uih.SetValue(0x0065, 0x1015, "-7.6685195\\3.0485301\\55.5231857", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925135258.268000", false);
    uih.SetValue(0x0008, 0x0032, "135258.268000", false);
    uih.SetValue(0x0018, 0x9073, "22.0", false);
    uih.SetValue(0x0020, 0x0032, "-116.267029\\-112.324295\\56.5791588", false);
    uih.SetValue(0x0020, 0x1041, "59.9219627", false);
    uih.SetValue(0x0065, 0x100f, "H59.9", false);
    uih.SetValue(0x0065, 0x1015, "-7.79013014\\3.0485301\\59.7214241", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925135300.902000", false);
    uih.SetValue(0x0008, 0x0032, "135300.902000", false);
    uih.SetValue(0x0018, 0x9073, "21.0", false);
    uih.SetValue(0x0020, 0x0032, "-116.388641\\-112.324295\\60.7773972", false);
    uih.SetValue(0x0020, 0x1041, "64.1219635", false);
    uih.SetValue(0x0065, 0x100f, "H64.1", false);
    uih.SetValue(0x0065, 0x1015, "-7.91174126\\3.0485301\\63.9196625", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925135257.976000", false);
    uih.SetValue(0x0008, 0x0032, "135257.976000", false);
    uih.SetValue(0x0018, 0x9073, "21.0", false);
    uih.SetValue(0x0020, 0x0032, "-116.510254\\-112.324295\\64.9756393", false);
    uih.SetValue(0x0020, 0x1041, "68.3219681", false);
    uih.SetValue(0x0065, 0x100f, "H68.3", false);
    uih.SetValue(0x0065, 0x1015, "-8.0333519\\3.0485301\\68.1179047", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925135300.609000", false);
    uih.SetValue(0x0008, 0x0032, "135300.609000", false);
    uih.SetValue(0x0018, 0x9073, "21.0", false);
    uih.SetValue(0x0020, 0x0032, "-116.631866\\-112.324295\\69.1738739", false);
    uih.SetValue(0x0020, 0x1041, "72.521965", false);
    uih.SetValue(0x0065, 0x100f, "H72.5", false);
    uih.SetValue(0x0065, 0x1015, "-8.15496349\\3.0485301\\72.3161392", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925135257.683000", false);
    uih.SetValue(0x0008, 0x0032, "135257.683000", false);
    uih.SetValue(0x0018, 0x9073, "22.0", false);
    uih.SetValue(0x0020, 0x0032, "-116.753471\\-112.324295\\73.3721161", false);
    uih.SetValue(0x0020, 0x1041, "76.7219696", false);
    uih.SetValue(0x0065, 0x100f, "H76.7", false);
    uih.SetValue(0x0065, 0x1015, "-8.27657413\\3.0485301\\76.5143814", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  target.AddInstance(dicom.release(), "nope");
}

TEST(Autogenerated, Test_92)
{
  Neuro::DicomInstancesCollection instances;
  LoadInstance_92(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 64);
  ASSERT_NEAR(nifti.pixdim[1], 3.500000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 64);
  ASSERT_NEAR(nifti.pixdim[2], 3.500000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 20);
  ASSERT_NEAR(nifti.pixdim[3], 4.200000, 0.00001);
  ASSERT_EQ(nifti.ndim, 3);
  ASSERT_EQ(nifti.dim[0], 3);
  ASSERT_EQ(nifti.nvox, 81920u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(57));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(57));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(57));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -3.496896, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], -0.106993, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], 0.121611, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 121.183418, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], -0.107038, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 3.498363, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], -108.072556, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.101295, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 0.003099, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 4.198239, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -6.589681, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.015291, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, -0.999778, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, -0.000221, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=71;Time=135257.683");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadInstance_93(Neuro::DicomInstancesCollection& target)
{
  // ('dcm_qa_uih', 'In/DTI_134434/dti_tra_dir16_PA_SaveBySlc__135612/00000002.dcm')
  Orthanc::DicomMap tags;
  tags.SetValue(0x0008, 0x0060, "MR", false);
  tags.SetValue(0x0008, 0x0070, "UIH", false);
  tags.SetValue(0x0020, 0x0013, "2", false);
  tags.SetValue(0x0018, 0x0080, "5865.0", false);
  tags.SetValue(0x0018, 0x0081, "70.7", false);
  tags.SetValue(0x0018, 0x1312, "COL", false);
  tags.SetValue(0x0008, 0x0016, "1.2.840.10008.5.1.4.1.1.4", false);
  tags.SetValue(0x0028, 0x0002, "1", false);
  tags.SetValue(0x0028, 0x0004, "MONOCHROME2", false);
  tags.SetValue(0x0028, 0x0010, "256", false);
  tags.SetValue(0x0028, 0x0011, "320", false);
  tags.SetValue(0x0028, 0x0100, "16", false);
  tags.SetValue(0x0028, 0x0101, "16", false);
  tags.SetValue(0x0028, 0x0102, "15", false);
  tags.SetValue(0x0028, 0x0103, "0", false);
  tags.SetValue(0x0018, 0x0050, "3.5", false);
  tags.SetValue(0x0018, 0x0088, "4.19999981", false);
  tags.SetValue(0x0020, 0x0037, "0.999113142\\0.0305822305\\0.0289414488\\-0.0305694081\\0.999532282\\-0.000885508256", false);
  tags.SetValue(0x0028, 0x0030, "3.5\\3.5", false);
  std::unique_ptr<Neuro::InputDicomInstance> dicom(new Neuro::InputDicomInstance(tags));
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925135314.945000", false);
    uih.SetValue(0x0008, 0x0032, "135314.945000", false);
    uih.SetValue(0x0018, 0x9073, "21.0", false);
    uih.SetValue(0x0020, 0x0032, "-114.442863\\-112.324295\\-6.39442587", false);
    uih.SetValue(0x0020, 0x1041, "-3.07805347", false);
    uih.SetValue(0x0065, 0x100f, "F3.1", false);
    uih.SetValue(0x0065, 0x1015, "-5.96596575\\3.0485301\\-3.25216055", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925135312.019000", false);
    uih.SetValue(0x0008, 0x0032, "135312.019000", false);
    uih.SetValue(0x0018, 0x9073, "22.0", false);
    uih.SetValue(0x0020, 0x0032, "-114.564476\\-112.324295\\-2.19618678", false);
    uih.SetValue(0x0020, 0x1041, "1.12194777", false);
    uih.SetValue(0x0065, 0x100f, "H1.1", false);
    uih.SetValue(0x0065, 0x1015, "-6.08757687\\3.0485301\\0.94607842", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925135314.653000", false);
    uih.SetValue(0x0008, 0x0032, "135314.653000", false);
    uih.SetValue(0x0018, 0x9073, "21.0", false);
    uih.SetValue(0x0020, 0x0032, "-114.686089\\-112.324295\\2.00205207", false);
    uih.SetValue(0x0020, 0x1041, "5.32194901", false);
    uih.SetValue(0x0065, 0x100f, "H5.3", false);
    uih.SetValue(0x0065, 0x1015, "-6.20918798\\3.0485301\\5.14431763", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925135311.727000", false);
    uih.SetValue(0x0008, 0x0032, "135311.727000", false);
    uih.SetValue(0x0018, 0x9073, "21.0", false);
    uih.SetValue(0x0020, 0x0032, "-114.807701\\-112.324295\\6.20029116", false);
    uih.SetValue(0x0020, 0x1041, "9.52194977", false);
    uih.SetValue(0x0065, 0x100f, "H9.5", false);
    uih.SetValue(0x0065, 0x1015, "-6.33079863\\3.0485301\\9.342556", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925135314.360000", false);
    uih.SetValue(0x0008, 0x0032, "135314.360000", false);
    uih.SetValue(0x0018, 0x9073, "21.0", false);
    uih.SetValue(0x0020, 0x0032, "-114.929314\\-112.324295\\10.39853", false);
    uih.SetValue(0x0020, 0x1041, "13.7219515", false);
    uih.SetValue(0x0065, 0x100f, "H13.7", false);
    uih.SetValue(0x0065, 0x1015, "-6.45240974\\3.0485301\\13.5407953", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925135311.434000", false);
    uih.SetValue(0x0008, 0x0032, "135311.434000", false);
    uih.SetValue(0x0018, 0x9073, "22.0", false);
    uih.SetValue(0x0020, 0x0032, "-115.050919\\-112.324295\\14.5967693", false);
    uih.SetValue(0x0020, 0x1041, "17.9219532", false);
    uih.SetValue(0x0065, 0x100f, "H17.9", false);
    uih.SetValue(0x0065, 0x1015, "-6.57402086\\3.0485301\\17.7390347", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925135314.067000", false);
    uih.SetValue(0x0008, 0x0032, "135314.067000", false);
    uih.SetValue(0x0018, 0x9073, "22.0", false);
    uih.SetValue(0x0020, 0x0032, "-115.172531\\-112.324295\\18.7950077", false);
    uih.SetValue(0x0020, 0x1041, "22.121954", false);
    uih.SetValue(0x0065, 0x100f, "H22.1", false);
    uih.SetValue(0x0065, 0x1015, "-6.6956315\\3.0485301\\21.937273", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925135311.142000", false);
    uih.SetValue(0x0008, 0x0032, "135311.142000", false);
    uih.SetValue(0x0018, 0x9073, "21.0", false);
    uih.SetValue(0x0020, 0x0032, "-115.294144\\-112.324295\\22.993248", false);
    uih.SetValue(0x0020, 0x1041, "26.3219547", false);
    uih.SetValue(0x0065, 0x100f, "H26.3", false);
    uih.SetValue(0x0065, 0x1015, "-6.81724262\\3.0485301\\26.1355133", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925135313.775000", false);
    uih.SetValue(0x0008, 0x0032, "135313.775000", false);
    uih.SetValue(0x0018, 0x9073, "21.0", false);
    uih.SetValue(0x0020, 0x0032, "-115.415756\\-112.324295\\27.1914864", false);
    uih.SetValue(0x0020, 0x1041, "30.5219555", false);
    uih.SetValue(0x0065, 0x100f, "H30.5", false);
    uih.SetValue(0x0065, 0x1015, "-6.93885374\\3.0485301\\30.3337517", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925135310.849000", false);
    uih.SetValue(0x0008, 0x0032, "135310.849000", false);
    uih.SetValue(0x0018, 0x9073, "21.0", false);
    uih.SetValue(0x0020, 0x0032, "-115.537369\\-112.324295\\31.3897247", false);
    uih.SetValue(0x0020, 0x1041, "34.7219582", false);
    uih.SetValue(0x0065, 0x100f, "H34.7", false);
    uih.SetValue(0x0065, 0x1015, "-7.06046438\\3.0485301\\34.5319901", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925135313.482000", false);
    uih.SetValue(0x0008, 0x0032, "135313.482000", false);
    uih.SetValue(0x0018, 0x9073, "22.0", false);
    uih.SetValue(0x0020, 0x0032, "-115.658974\\-112.324295\\35.5879631", false);
    uih.SetValue(0x0020, 0x1041, "38.9219589", false);
    uih.SetValue(0x0065, 0x100f, "H38.9", false);
    uih.SetValue(0x0065, 0x1015, "-7.1820755\\3.0485301\\38.7302284", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925135310.556000", false);
    uih.SetValue(0x0008, 0x0032, "135310.556000", false);
    uih.SetValue(0x0018, 0x9073, "22.0", false);
    uih.SetValue(0x0020, 0x0032, "-115.780586\\-112.324295\\39.7862015", false);
    uih.SetValue(0x0020, 0x1041, "43.1219597", false);
    uih.SetValue(0x0065, 0x100f, "H43.1", false);
    uih.SetValue(0x0065, 0x1015, "-7.30368662\\3.0485301\\42.9284668", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925135313.190000", false);
    uih.SetValue(0x0008, 0x0032, "135313.190000", false);
    uih.SetValue(0x0018, 0x9073, "21.0", false);
    uih.SetValue(0x0020, 0x0032, "-115.902199\\-112.324295\\43.9844437", false);
    uih.SetValue(0x0020, 0x1041, "47.3219604", false);
    uih.SetValue(0x0065, 0x100f, "H47.3", false);
    uih.SetValue(0x0065, 0x1015, "-7.42529726\\3.0485301\\47.126709", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925135310.264000", false);
    uih.SetValue(0x0008, 0x0032, "135310.264000", false);
    uih.SetValue(0x0018, 0x9073, "21.0", false);
    uih.SetValue(0x0020, 0x0032, "-116.023811\\-112.324295\\48.182682", false);
    uih.SetValue(0x0020, 0x1041, "51.5219612", false);
    uih.SetValue(0x0065, 0x100f, "H51.5", false);
    uih.SetValue(0x0065, 0x1015, "-7.54690838\\3.0485301\\51.3249474", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925135312.897000", false);
    uih.SetValue(0x0008, 0x0032, "135312.897000", false);
    uih.SetValue(0x0018, 0x9073, "21.0", false);
    uih.SetValue(0x0020, 0x0032, "-116.145424\\-112.324295\\52.3809204", false);
    uih.SetValue(0x0020, 0x1041, "55.721962", false);
    uih.SetValue(0x0065, 0x100f, "H55.7", false);
    uih.SetValue(0x0065, 0x1015, "-7.6685195\\3.0485301\\55.5231857", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925135309.971000", false);
    uih.SetValue(0x0008, 0x0032, "135309.971000", false);
    uih.SetValue(0x0018, 0x9073, "22.0", false);
    uih.SetValue(0x0020, 0x0032, "-116.267029\\-112.324295\\56.5791588", false);
    uih.SetValue(0x0020, 0x1041, "59.9219627", false);
    uih.SetValue(0x0065, 0x100f, "H59.9", false);
    uih.SetValue(0x0065, 0x1015, "-7.79013014\\3.0485301\\59.7214241", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925135312.604000", false);
    uih.SetValue(0x0008, 0x0032, "135312.604000", false);
    uih.SetValue(0x0018, 0x9073, "22.0", false);
    uih.SetValue(0x0020, 0x0032, "-116.388641\\-112.324295\\60.7773972", false);
    uih.SetValue(0x0020, 0x1041, "64.1219635", false);
    uih.SetValue(0x0065, 0x100f, "H64.1", false);
    uih.SetValue(0x0065, 0x1015, "-7.91174126\\3.0485301\\63.9196625", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925135309.679000", false);
    uih.SetValue(0x0008, 0x0032, "135309.679000", false);
    uih.SetValue(0x0018, 0x9073, "21.0", false);
    uih.SetValue(0x0020, 0x0032, "-116.510254\\-112.324295\\64.9756393", false);
    uih.SetValue(0x0020, 0x1041, "68.3219681", false);
    uih.SetValue(0x0065, 0x100f, "H68.3", false);
    uih.SetValue(0x0065, 0x1015, "-8.0333519\\3.0485301\\68.1179047", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925135312.312000", false);
    uih.SetValue(0x0008, 0x0032, "135312.312000", false);
    uih.SetValue(0x0018, 0x9073, "21.0", false);
    uih.SetValue(0x0020, 0x0032, "-116.631866\\-112.324295\\69.1738739", false);
    uih.SetValue(0x0020, 0x1041, "72.521965", false);
    uih.SetValue(0x0065, 0x100f, "H72.5", false);
    uih.SetValue(0x0065, 0x1015, "-8.15496349\\3.0485301\\72.3161392", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925135309.386000", false);
    uih.SetValue(0x0008, 0x0032, "135309.386000", false);
    uih.SetValue(0x0018, 0x9073, "22.0", false);
    uih.SetValue(0x0020, 0x0032, "-116.753471\\-112.324295\\73.3721161", false);
    uih.SetValue(0x0020, 0x1041, "76.7219696", false);
    uih.SetValue(0x0065, 0x100f, "H76.7", false);
    uih.SetValue(0x0065, 0x1015, "-8.27657413\\3.0485301\\76.5143814", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  target.AddInstance(dicom.release(), "nope");
}

TEST(Autogenerated, Test_93)
{
  Neuro::DicomInstancesCollection instances;
  LoadInstance_93(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 64);
  ASSERT_NEAR(nifti.pixdim[1], 3.500000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 64);
  ASSERT_NEAR(nifti.pixdim[2], 3.500000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 20);
  ASSERT_NEAR(nifti.pixdim[3], 4.200000, 0.00001);
  ASSERT_EQ(nifti.ndim, 3);
  ASSERT_EQ(nifti.dim[0], 3);
  ASSERT_EQ(nifti.nvox, 81920u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(57));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(57));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(57));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -3.496896, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], -0.106993, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], 0.121611, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 121.183418, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], -0.107038, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 3.498363, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], -108.072556, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.101295, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 0.003099, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 4.198239, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -6.589681, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.015291, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, -0.999778, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, -0.000221, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=71;Time=135309.386");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadInstance_94(Neuro::DicomInstancesCollection& target)
{
  // ('dcm_qa_uih', 'In/DTI_134434/dti_tra_dir16_PA_rot_SaveBySlc__140451/00000001.dcm')
  Orthanc::DicomMap tags;
  tags.SetValue(0x0008, 0x0060, "MR", false);
  tags.SetValue(0x0008, 0x0070, "UIH", false);
  tags.SetValue(0x0020, 0x0013, "1", false);
  tags.SetValue(0x0018, 0x0080, "5865.0", false);
  tags.SetValue(0x0018, 0x0081, "70.7", false);
  tags.SetValue(0x0018, 0x1312, "COL", false);
  tags.SetValue(0x0008, 0x0016, "1.2.840.10008.5.1.4.1.1.4", false);
  tags.SetValue(0x0028, 0x0002, "1", false);
  tags.SetValue(0x0028, 0x0004, "MONOCHROME2", false);
  tags.SetValue(0x0028, 0x0010, "256", false);
  tags.SetValue(0x0028, 0x0011, "320", false);
  tags.SetValue(0x0028, 0x0100, "16", false);
  tags.SetValue(0x0028, 0x0101, "16", false);
  tags.SetValue(0x0028, 0x0102, "15", false);
  tags.SetValue(0x0028, 0x0103, "0", false);
  tags.SetValue(0x0018, 0x0050, "3.5", false);
  tags.SetValue(0x0018, 0x0088, "4.19999981", false);
  tags.SetValue(0x0020, 0x0037, "0.961447895\\0.27357319\\0.0278503932\\-0.273458511\\0.96185118\\-0.00792131014", false);
  tags.SetValue(0x0028, 0x0030, "3.5\\3.5", false);
  std::unique_ptr<Neuro::InputDicomInstance> dicom(new Neuro::InputDicomInstance(tags));
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925140141.741000", false);
    uih.SetValue(0x0008, 0x0032, "140141.741000", false);
    uih.SetValue(0x0018, 0x9073, "22.0", false);
    uih.SetValue(0x0020, 0x0032, "-83.0207825\\-135.319\\-5.48421812", false);
    uih.SetValue(0x0020, 0x1041, "-3.07805347", false);
    uih.SetValue(0x0065, 0x100f, "F3.1", false);
    uih.SetValue(0x0065, 0x1015, "-5.96596575\\3.0485301\\-3.25216055", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925140138.816000", false);
    uih.SetValue(0x0008, 0x0032, "140138.816000", false);
    uih.SetValue(0x0018, 0x9073, "21.0", false);
    uih.SetValue(0x0020, 0x0032, "-83.1423874\\-135.319\\-1.28597891", false);
    uih.SetValue(0x0020, 0x1041, "1.12194777", false);
    uih.SetValue(0x0065, 0x100f, "H1.1", false);
    uih.SetValue(0x0065, 0x1015, "-6.08757687\\3.0485301\\0.94607842", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925140141.449000", false);
    uih.SetValue(0x0008, 0x0032, "140141.449000", false);
    uih.SetValue(0x0018, 0x9073, "21.0", false);
    uih.SetValue(0x0020, 0x0032, "-83.2639999\\-135.319\\2.91226006", false);
    uih.SetValue(0x0020, 0x1041, "5.32194901", false);
    uih.SetValue(0x0065, 0x100f, "H5.3", false);
    uih.SetValue(0x0065, 0x1015, "-6.20918798\\3.0485301\\5.14431763", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925140138.523000", false);
    uih.SetValue(0x0008, 0x0032, "140138.523000", false);
    uih.SetValue(0x0018, 0x9073, "21.0", false);
    uih.SetValue(0x0020, 0x0032, "-83.3856125\\-135.319\\7.11049891", false);
    uih.SetValue(0x0020, 0x1041, "9.52194977", false);
    uih.SetValue(0x0065, 0x100f, "H9.5", false);
    uih.SetValue(0x0065, 0x1015, "-6.33079863\\3.0485301\\9.342556", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925140141.156000", false);
    uih.SetValue(0x0008, 0x0032, "140141.156000", false);
    uih.SetValue(0x0018, 0x9073, "21.0", false);
    uih.SetValue(0x0020, 0x0032, "-83.507225\\-135.319\\11.3087378", false);
    uih.SetValue(0x0020, 0x1041, "13.7219515", false);
    uih.SetValue(0x0065, 0x100f, "H13.7", false);
    uih.SetValue(0x0065, 0x1015, "-6.45240974\\3.0485301\\13.5407953", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925140138.230000", false);
    uih.SetValue(0x0008, 0x0032, "140138.230000", false);
    uih.SetValue(0x0018, 0x9073, "22.0", false);
    uih.SetValue(0x0020, 0x0032, "-83.62883\\-135.319\\15.5069771", false);
    uih.SetValue(0x0020, 0x1041, "17.9219532", false);
    uih.SetValue(0x0065, 0x100f, "H17.9", false);
    uih.SetValue(0x0065, 0x1015, "-6.57402086\\3.0485301\\17.7390347", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925140140.864000", false);
    uih.SetValue(0x0008, 0x0032, "140140.864000", false);
    uih.SetValue(0x0018, 0x9073, "21.0", false);
    uih.SetValue(0x0020, 0x0032, "-83.7504425\\-135.319\\19.7052155", false);
    uih.SetValue(0x0020, 0x1041, "22.121954", false);
    uih.SetValue(0x0065, 0x100f, "H22.1", false);
    uih.SetValue(0x0065, 0x1015, "-6.6956315\\3.0485301\\21.937273", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925140137.938000", false);
    uih.SetValue(0x0008, 0x0032, "140137.938000", false);
    uih.SetValue(0x0018, 0x9073, "21.0", false);
    uih.SetValue(0x0020, 0x0032, "-83.8720551\\-135.319\\23.9034557", false);
    uih.SetValue(0x0020, 0x1041, "26.3219547", false);
    uih.SetValue(0x0065, 0x100f, "H26.3", false);
    uih.SetValue(0x0065, 0x1015, "-6.81724262\\3.0485301\\26.1355133", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925140140.571000", false);
    uih.SetValue(0x0008, 0x0032, "140140.571000", false);
    uih.SetValue(0x0018, 0x9073, "21.0", false);
    uih.SetValue(0x0020, 0x0032, "-83.9936676\\-135.319\\28.1016941", false);
    uih.SetValue(0x0020, 0x1041, "30.5219555", false);
    uih.SetValue(0x0065, 0x100f, "H30.5", false);
    uih.SetValue(0x0065, 0x1015, "-6.93885374\\3.0485301\\30.3337517", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925140137.645000", false);
    uih.SetValue(0x0008, 0x0032, "140137.645000", false);
    uih.SetValue(0x0018, 0x9073, "22.0", false);
    uih.SetValue(0x0020, 0x0032, "-84.1152802\\-135.319\\32.2999344", false);
    uih.SetValue(0x0020, 0x1041, "34.7219582", false);
    uih.SetValue(0x0065, 0x100f, "H34.7", false);
    uih.SetValue(0x0065, 0x1015, "-7.06046438\\3.0485301\\34.5319901", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925140140.278000", false);
    uih.SetValue(0x0008, 0x0032, "140140.278000", false);
    uih.SetValue(0x0018, 0x9073, "22.0", false);
    uih.SetValue(0x0020, 0x0032, "-84.2368851\\-135.319\\36.4981728", false);
    uih.SetValue(0x0020, 0x1041, "38.9219589", false);
    uih.SetValue(0x0065, 0x100f, "H38.9", false);
    uih.SetValue(0x0065, 0x1015, "-7.1820755\\3.0485301\\38.7302284", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925140137.353000", false);
    uih.SetValue(0x0008, 0x0032, "140137.353000", false);
    uih.SetValue(0x0018, 0x9073, "21.0", false);
    uih.SetValue(0x0020, 0x0032, "-84.3584976\\-135.319\\40.6964111", false);
    uih.SetValue(0x0020, 0x1041, "43.1219597", false);
    uih.SetValue(0x0065, 0x100f, "H43.1", false);
    uih.SetValue(0x0065, 0x1015, "-7.30368662\\3.0485301\\42.9284668", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925140139.986000", false);
    uih.SetValue(0x0008, 0x0032, "140139.986000", false);
    uih.SetValue(0x0018, 0x9073, "21.0", false);
    uih.SetValue(0x0020, 0x0032, "-84.4801102\\-135.319\\44.8946495", false);
    uih.SetValue(0x0020, 0x1041, "47.3219604", false);
    uih.SetValue(0x0065, 0x100f, "H47.3", false);
    uih.SetValue(0x0065, 0x1015, "-7.42529726\\3.0485301\\47.126709", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925140137.060000", false);
    uih.SetValue(0x0008, 0x0032, "140137.060000", false);
    uih.SetValue(0x0018, 0x9073, "21.0", false);
    uih.SetValue(0x0020, 0x0032, "-84.6017227\\-135.319\\49.0928879", false);
    uih.SetValue(0x0020, 0x1041, "51.5219612", false);
    uih.SetValue(0x0065, 0x100f, "H51.5", false);
    uih.SetValue(0x0065, 0x1015, "-7.54690838\\3.0485301\\51.3249474", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925140139.693000", false);
    uih.SetValue(0x0008, 0x0032, "140139.693000", false);
    uih.SetValue(0x0018, 0x9073, "22.0", false);
    uih.SetValue(0x0020, 0x0032, "-84.7233353\\-135.319\\53.2911301", false);
    uih.SetValue(0x0020, 0x1041, "55.721962", false);
    uih.SetValue(0x0065, 0x100f, "H55.7", false);
    uih.SetValue(0x0065, 0x1015, "-7.6685195\\3.0485301\\55.5231857", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925140136.768000", false);
    uih.SetValue(0x0008, 0x0032, "140136.768000", false);
    uih.SetValue(0x0018, 0x9073, "21.0", false);
    uih.SetValue(0x0020, 0x0032, "-84.8449402\\-135.319\\57.4893684", false);
    uih.SetValue(0x0020, 0x1041, "59.9219627", false);
    uih.SetValue(0x0065, 0x100f, "H59.9", false);
    uih.SetValue(0x0065, 0x1015, "-7.79013014\\3.0485301\\59.7214241", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925140139.401000", false);
    uih.SetValue(0x0008, 0x0032, "140139.401000", false);
    uih.SetValue(0x0018, 0x9073, "21.0", false);
    uih.SetValue(0x0020, 0x0032, "-84.9665527\\-135.319\\61.6876068", false);
    uih.SetValue(0x0020, 0x1041, "64.1219635", false);
    uih.SetValue(0x0065, 0x100f, "H64.1", false);
    uih.SetValue(0x0065, 0x1015, "-7.91174126\\3.0485301\\63.9196625", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925140136.475000", false);
    uih.SetValue(0x0008, 0x0032, "140136.475000", false);
    uih.SetValue(0x0018, 0x9073, "21.0", false);
    uih.SetValue(0x0020, 0x0032, "-85.0881653\\-135.319\\65.885849", false);
    uih.SetValue(0x0020, 0x1041, "68.3219681", false);
    uih.SetValue(0x0065, 0x100f, "H68.3", false);
    uih.SetValue(0x0065, 0x1015, "-8.0333519\\3.0485301\\68.1179047", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925140139.108000", false);
    uih.SetValue(0x0008, 0x0032, "140139.108000", false);
    uih.SetValue(0x0018, 0x9073, "21.0", false);
    uih.SetValue(0x0020, 0x0032, "-85.2097778\\-135.319\\70.0840836", false);
    uih.SetValue(0x0020, 0x1041, "72.521965", false);
    uih.SetValue(0x0065, 0x100f, "H72.5", false);
    uih.SetValue(0x0065, 0x1015, "-8.15496349\\3.0485301\\72.3161392", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925140136.182000", false);
    uih.SetValue(0x0008, 0x0032, "140136.182000", false);
    uih.SetValue(0x0018, 0x9073, "22.0", false);
    uih.SetValue(0x0020, 0x0032, "-85.3313904\\-135.319\\74.2823257", false);
    uih.SetValue(0x0020, 0x1041, "76.7219696", false);
    uih.SetValue(0x0065, 0x100f, "H76.7", false);
    uih.SetValue(0x0065, 0x1015, "-8.27657413\\3.0485301\\76.5143814", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  target.AddInstance(dicom.release(), "nope");
}

TEST(Autogenerated, Test_94)
{
  Neuro::DicomInstancesCollection instances;
  LoadInstance_94(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 64);
  ASSERT_NEAR(nifti.pixdim[1], 3.500000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 64);
  ASSERT_NEAR(nifti.pixdim[2], 3.500000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 20);
  ASSERT_NEAR(nifti.pixdim[3], 4.200000, 0.00001);
  ASSERT_EQ(nifti.ndim, 3);
  ASSERT_EQ(nifti.dim[0], 3);
  ASSERT_EQ(nifti.nvox, 81920u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(57));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(57));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(57));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -3.365068, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], -0.957105, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], 0.121611, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 143.318390, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], -0.957506, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 3.366479, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], -76.769180, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.097476, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 0.027725, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 4.198239, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -7.230867, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.138096, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, -0.990313, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, -0.002000, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=71;Time=140136.182");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadInstance_95(Neuro::DicomInstancesCollection& target)
{
  // ('dcm_qa_uih', 'In/DTI_134434/dti_tra_dir16_PA_rot_SaveBySlc__140451/00000002.dcm')
  Orthanc::DicomMap tags;
  tags.SetValue(0x0008, 0x0060, "MR", false);
  tags.SetValue(0x0008, 0x0070, "UIH", false);
  tags.SetValue(0x0020, 0x0013, "2", false);
  tags.SetValue(0x0018, 0x0080, "5865.0", false);
  tags.SetValue(0x0018, 0x0081, "70.7", false);
  tags.SetValue(0x0018, 0x1312, "COL", false);
  tags.SetValue(0x0008, 0x0016, "1.2.840.10008.5.1.4.1.1.4", false);
  tags.SetValue(0x0028, 0x0002, "1", false);
  tags.SetValue(0x0028, 0x0004, "MONOCHROME2", false);
  tags.SetValue(0x0028, 0x0010, "256", false);
  tags.SetValue(0x0028, 0x0011, "320", false);
  tags.SetValue(0x0028, 0x0100, "16", false);
  tags.SetValue(0x0028, 0x0101, "16", false);
  tags.SetValue(0x0028, 0x0102, "15", false);
  tags.SetValue(0x0028, 0x0103, "0", false);
  tags.SetValue(0x0018, 0x0050, "3.5", false);
  tags.SetValue(0x0018, 0x0088, "4.19999981", false);
  tags.SetValue(0x0020, 0x0037, "0.961447895\\0.27357319\\0.0278503932\\-0.273458511\\0.96185118\\-0.00792131014", false);
  tags.SetValue(0x0028, 0x0030, "3.5\\3.5", false);
  std::unique_ptr<Neuro::InputDicomInstance> dicom(new Neuro::InputDicomInstance(tags));
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925140153.444000", false);
    uih.SetValue(0x0008, 0x0032, "140153.444000", false);
    uih.SetValue(0x0018, 0x9073, "22.0", false);
    uih.SetValue(0x0020, 0x0032, "-83.0207825\\-135.319\\-5.48421812", false);
    uih.SetValue(0x0020, 0x1041, "-3.07805347", false);
    uih.SetValue(0x0065, 0x100f, "F3.1", false);
    uih.SetValue(0x0065, 0x1015, "-5.96596575\\3.0485301\\-3.25216055", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925140150.518000", false);
    uih.SetValue(0x0008, 0x0032, "140150.518000", false);
    uih.SetValue(0x0018, 0x9073, "22.0", false);
    uih.SetValue(0x0020, 0x0032, "-83.1423874\\-135.319\\-1.28597891", false);
    uih.SetValue(0x0020, 0x1041, "1.12194777", false);
    uih.SetValue(0x0065, 0x100f, "H1.1", false);
    uih.SetValue(0x0065, 0x1015, "-6.08757687\\3.0485301\\0.94607842", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925140153.152000", false);
    uih.SetValue(0x0008, 0x0032, "140153.152000", false);
    uih.SetValue(0x0018, 0x9073, "21.0", false);
    uih.SetValue(0x0020, 0x0032, "-83.2639999\\-135.319\\2.91226006", false);
    uih.SetValue(0x0020, 0x1041, "5.32194901", false);
    uih.SetValue(0x0065, 0x100f, "H5.3", false);
    uih.SetValue(0x0065, 0x1015, "-6.20918798\\3.0485301\\5.14431763", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925140150.226000", false);
    uih.SetValue(0x0008, 0x0032, "140150.226000", false);
    uih.SetValue(0x0018, 0x9073, "21.0", false);
    uih.SetValue(0x0020, 0x0032, "-83.3856125\\-135.319\\7.11049891", false);
    uih.SetValue(0x0020, 0x1041, "9.52194977", false);
    uih.SetValue(0x0065, 0x100f, "H9.5", false);
    uih.SetValue(0x0065, 0x1015, "-6.33079863\\3.0485301\\9.342556", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925140152.859000", false);
    uih.SetValue(0x0008, 0x0032, "140152.859000", false);
    uih.SetValue(0x0018, 0x9073, "21.0", false);
    uih.SetValue(0x0020, 0x0032, "-83.507225\\-135.319\\11.3087378", false);
    uih.SetValue(0x0020, 0x1041, "13.7219515", false);
    uih.SetValue(0x0065, 0x100f, "H13.7", false);
    uih.SetValue(0x0065, 0x1015, "-6.45240974\\3.0485301\\13.5407953", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925140149.933000", false);
    uih.SetValue(0x0008, 0x0032, "140149.933000", false);
    uih.SetValue(0x0018, 0x9073, "22.0", false);
    uih.SetValue(0x0020, 0x0032, "-83.62883\\-135.319\\15.5069771", false);
    uih.SetValue(0x0020, 0x1041, "17.9219532", false);
    uih.SetValue(0x0065, 0x100f, "H17.9", false);
    uih.SetValue(0x0065, 0x1015, "-6.57402086\\3.0485301\\17.7390347", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925140152.567000", false);
    uih.SetValue(0x0008, 0x0032, "140152.567000", false);
    uih.SetValue(0x0018, 0x9073, "21.0", false);
    uih.SetValue(0x0020, 0x0032, "-83.7504425\\-135.319\\19.7052155", false);
    uih.SetValue(0x0020, 0x1041, "22.121954", false);
    uih.SetValue(0x0065, 0x100f, "H22.1", false);
    uih.SetValue(0x0065, 0x1015, "-6.6956315\\3.0485301\\21.937273", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925140149.641000", false);
    uih.SetValue(0x0008, 0x0032, "140149.641000", false);
    uih.SetValue(0x0018, 0x9073, "21.0", false);
    uih.SetValue(0x0020, 0x0032, "-83.8720551\\-135.319\\23.9034557", false);
    uih.SetValue(0x0020, 0x1041, "26.3219547", false);
    uih.SetValue(0x0065, 0x100f, "H26.3", false);
    uih.SetValue(0x0065, 0x1015, "-6.81724262\\3.0485301\\26.1355133", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925140152.274000", false);
    uih.SetValue(0x0008, 0x0032, "140152.274000", false);
    uih.SetValue(0x0018, 0x9073, "21.0", false);
    uih.SetValue(0x0020, 0x0032, "-83.9936676\\-135.319\\28.1016941", false);
    uih.SetValue(0x0020, 0x1041, "30.5219555", false);
    uih.SetValue(0x0065, 0x100f, "H30.5", false);
    uih.SetValue(0x0065, 0x1015, "-6.93885374\\3.0485301\\30.3337517", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925140149.348000", false);
    uih.SetValue(0x0008, 0x0032, "140149.348000", false);
    uih.SetValue(0x0018, 0x9073, "22.0", false);
    uih.SetValue(0x0020, 0x0032, "-84.1152802\\-135.319\\32.2999344", false);
    uih.SetValue(0x0020, 0x1041, "34.7219582", false);
    uih.SetValue(0x0065, 0x100f, "H34.7", false);
    uih.SetValue(0x0065, 0x1015, "-7.06046438\\3.0485301\\34.5319901", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925140151.981000", false);
    uih.SetValue(0x0008, 0x0032, "140151.981000", false);
    uih.SetValue(0x0018, 0x9073, "22.0", false);
    uih.SetValue(0x0020, 0x0032, "-84.2368851\\-135.319\\36.4981728", false);
    uih.SetValue(0x0020, 0x1041, "38.9219589", false);
    uih.SetValue(0x0065, 0x100f, "H38.9", false);
    uih.SetValue(0x0065, 0x1015, "-7.1820755\\3.0485301\\38.7302284", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925140149.056000", false);
    uih.SetValue(0x0008, 0x0032, "140149.056000", false);
    uih.SetValue(0x0018, 0x9073, "21.0", false);
    uih.SetValue(0x0020, 0x0032, "-84.3584976\\-135.319\\40.6964111", false);
    uih.SetValue(0x0020, 0x1041, "43.1219597", false);
    uih.SetValue(0x0065, 0x100f, "H43.1", false);
    uih.SetValue(0x0065, 0x1015, "-7.30368662\\3.0485301\\42.9284668", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925140151.689000", false);
    uih.SetValue(0x0008, 0x0032, "140151.689000", false);
    uih.SetValue(0x0018, 0x9073, "21.0", false);
    uih.SetValue(0x0020, 0x0032, "-84.4801102\\-135.319\\44.8946495", false);
    uih.SetValue(0x0020, 0x1041, "47.3219604", false);
    uih.SetValue(0x0065, 0x100f, "H47.3", false);
    uih.SetValue(0x0065, 0x1015, "-7.42529726\\3.0485301\\47.126709", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925140148.763000", false);
    uih.SetValue(0x0008, 0x0032, "140148.763000", false);
    uih.SetValue(0x0018, 0x9073, "21.0", false);
    uih.SetValue(0x0020, 0x0032, "-84.6017227\\-135.319\\49.0928879", false);
    uih.SetValue(0x0020, 0x1041, "51.5219612", false);
    uih.SetValue(0x0065, 0x100f, "H51.5", false);
    uih.SetValue(0x0065, 0x1015, "-7.54690838\\3.0485301\\51.3249474", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925140151.396000", false);
    uih.SetValue(0x0008, 0x0032, "140151.396000", false);
    uih.SetValue(0x0018, 0x9073, "22.0", false);
    uih.SetValue(0x0020, 0x0032, "-84.7233353\\-135.319\\53.2911301", false);
    uih.SetValue(0x0020, 0x1041, "55.721962", false);
    uih.SetValue(0x0065, 0x100f, "H55.7", false);
    uih.SetValue(0x0065, 0x1015, "-7.6685195\\3.0485301\\55.5231857", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925140148.470000", false);
    uih.SetValue(0x0008, 0x0032, "140148.470000", false);
    uih.SetValue(0x0018, 0x9073, "22.0", false);
    uih.SetValue(0x0020, 0x0032, "-84.8449402\\-135.319\\57.4893684", false);
    uih.SetValue(0x0020, 0x1041, "59.9219627", false);
    uih.SetValue(0x0065, 0x100f, "H59.9", false);
    uih.SetValue(0x0065, 0x1015, "-7.79013014\\3.0485301\\59.7214241", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925140151.104000", false);
    uih.SetValue(0x0008, 0x0032, "140151.104000", false);
    uih.SetValue(0x0018, 0x9073, "21.0", false);
    uih.SetValue(0x0020, 0x0032, "-84.9665527\\-135.319\\61.6876068", false);
    uih.SetValue(0x0020, 0x1041, "64.1219635", false);
    uih.SetValue(0x0065, 0x100f, "H64.1", false);
    uih.SetValue(0x0065, 0x1015, "-7.91174126\\3.0485301\\63.9196625", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925140148.178000", false);
    uih.SetValue(0x0008, 0x0032, "140148.178000", false);
    uih.SetValue(0x0018, 0x9073, "21.0", false);
    uih.SetValue(0x0020, 0x0032, "-85.0881653\\-135.319\\65.885849", false);
    uih.SetValue(0x0020, 0x1041, "68.3219681", false);
    uih.SetValue(0x0065, 0x100f, "H68.3", false);
    uih.SetValue(0x0065, 0x1015, "-8.0333519\\3.0485301\\68.1179047", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925140150.811000", false);
    uih.SetValue(0x0008, 0x0032, "140150.811000", false);
    uih.SetValue(0x0018, 0x9073, "21.0", false);
    uih.SetValue(0x0020, 0x0032, "-85.2097778\\-135.319\\70.0840836", false);
    uih.SetValue(0x0020, 0x1041, "72.521965", false);
    uih.SetValue(0x0065, 0x100f, "H72.5", false);
    uih.SetValue(0x0065, 0x1015, "-8.15496349\\3.0485301\\72.3161392", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  {
    Orthanc::DicomMap uih;
    uih.SetValue(0x0008, 0x0022, "20180925", false);
    uih.SetValue(0x0008, 0x002a, "20180925140147.885000", false);
    uih.SetValue(0x0008, 0x0032, "140147.885000", false);
    uih.SetValue(0x0018, 0x9073, "22.0", false);
    uih.SetValue(0x0020, 0x0032, "-85.3313904\\-135.319\\74.2823257", false);
    uih.SetValue(0x0020, 0x1041, "76.7219696", false);
    uih.SetValue(0x0065, 0x100f, "H76.7", false);
    uih.SetValue(0x0065, 0x1015, "-8.27657413\\3.0485301\\76.5143814", false);
    uih.SetValue(0x0065, 0x1029, "0.02", false);
    dicom->AddUIHFrameSequenceItem(uih);
  }
  target.AddInstance(dicom.release(), "nope");
}

TEST(Autogenerated, Test_95)
{
  Neuro::DicomInstancesCollection instances;
  LoadInstance_95(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 64);
  ASSERT_NEAR(nifti.pixdim[1], 3.500000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 64);
  ASSERT_NEAR(nifti.pixdim[2], 3.500000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 20);
  ASSERT_NEAR(nifti.pixdim[3], 4.200000, 0.00001);
  ASSERT_EQ(nifti.ndim, 3);
  ASSERT_EQ(nifti.dim[0], 3);
  ASSERT_EQ(nifti.nvox, 81920u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(57));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(57));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(57));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -3.365068, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], -0.957105, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], 0.121611, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 143.318390, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], -0.957506, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 3.366479, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], -76.769180, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.097476, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 0.027725, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 4.198239, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -7.230867, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.138096, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, -0.990313, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, -0.002000, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=71;Time=140147.885");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadInstance_96(Neuro::DicomInstancesCollection& target)
{
  // ('dcm_qa_uih', 'In/DTI_134434/t1_gre_fsp_3d_sag__134917/00000001.dcm')
  Orthanc::DicomMap tags;
  tags.SetValue(0x0008, 0x0060, "MR", false);
  tags.SetValue(0x0008, 0x0070, "UIH", false);
  tags.SetValue(0x0020, 0x0013, "1", false);
  tags.SetValue(0x0018, 0x0080, "7.25", false);
  tags.SetValue(0x0018, 0x0081, "3.1", false);
  tags.SetValue(0x0018, 0x1312, "ROW", false);
  tags.SetValue(0x0008, 0x0016, "1.2.840.10008.5.1.4.1.1.4", false);
  tags.SetValue(0x0028, 0x0002, "1", false);
  tags.SetValue(0x0028, 0x0004, "MONOCHROME2", false);
  tags.SetValue(0x0028, 0x0010, "512", false);
  tags.SetValue(0x0028, 0x0011, "460", false);
  tags.SetValue(0x0028, 0x0100, "16", false);
  tags.SetValue(0x0028, 0x0101, "16", false);
  tags.SetValue(0x0028, 0x0102, "15", false);
  tags.SetValue(0x0028, 0x0103, "0", false);
  tags.SetValue(0x0008, 0x0032, "134548.132000", false);
  tags.SetValue(0x0018, 0x0050, "1", false);
  tags.SetValue(0x0018, 0x0088, "1", false);
  tags.SetValue(0x0020, 0x0032, "-90.5559769\\-112.983368\\120.478172", false);
  tags.SetValue(0x0020, 0x0037, "-0.00459422544\\0.9999879\\0.00176440715\\0.0258876719\\0.00188276928\\-0.999663115", false);
  tags.SetValue(0x0028, 0x0030, "0.5\\0.5", false);
  std::unique_ptr<Neuro::InputDicomInstance> dicom(new Neuro::InputDicomInstance(tags));
  target.AddInstance(dicom.release(), "nope");
}

TEST(Autogenerated, Test_96)
{
  Neuro::DicomInstancesCollection instances;
  LoadInstance_96(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 460);
  ASSERT_NEAR(nifti.pixdim[1], 0.500000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 512);
  ASSERT_NEAR(nifti.pixdim[2], 0.500000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 1);
  ASSERT_NEAR(nifti.pixdim[3], 1.000000, 0.00001);
  ASSERT_EQ(nifti.ndim, 3);
  ASSERT_EQ(nifti.dim[0], 3);
  ASSERT_EQ(nifti.nvox, 235520u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(54));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(54));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(54));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], 0.002297, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.012944, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], 0.999654, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 83.941681, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], -0.499994, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 0.000941, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], 0.004547, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], 112.502319, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000882, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 0.499832, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], -0.025896, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -134.935730, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.494170, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, -0.492796, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, -0.504831, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=3.1;Time=134548.132");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadInstance_97(Neuro::DicomInstancesCollection& target)
{
  // ('dcm_qa_uih', 'In/DTI_134434/t1_gre_fsp_3d_sag__134917/00000002.dcm')
  Orthanc::DicomMap tags;
  tags.SetValue(0x0008, 0x0060, "MR", false);
  tags.SetValue(0x0008, 0x0070, "UIH", false);
  tags.SetValue(0x0020, 0x0013, "2", false);
  tags.SetValue(0x0018, 0x0080, "7.25", false);
  tags.SetValue(0x0018, 0x0081, "3.1", false);
  tags.SetValue(0x0018, 0x1312, "ROW", false);
  tags.SetValue(0x0008, 0x0016, "1.2.840.10008.5.1.4.1.1.4", false);
  tags.SetValue(0x0028, 0x0002, "1", false);
  tags.SetValue(0x0028, 0x0004, "MONOCHROME2", false);
  tags.SetValue(0x0028, 0x0010, "512", false);
  tags.SetValue(0x0028, 0x0011, "460", false);
  tags.SetValue(0x0028, 0x0100, "16", false);
  tags.SetValue(0x0028, 0x0101, "16", false);
  tags.SetValue(0x0028, 0x0102, "15", false);
  tags.SetValue(0x0028, 0x0103, "0", false);
  tags.SetValue(0x0008, 0x0032, "134548.132000", false);
  tags.SetValue(0x0018, 0x0050, "1", false);
  tags.SetValue(0x0018, 0x0088, "1", false);
  tags.SetValue(0x0020, 0x0032, "-89.5563278\\-112.978821\\120.504074", false);
  tags.SetValue(0x0020, 0x0037, "-0.00459422544\\0.9999879\\0.00176440715\\0.0258876719\\0.00188276928\\-0.999663115", false);
  tags.SetValue(0x0028, 0x0030, "0.5\\0.5", false);
  std::unique_ptr<Neuro::InputDicomInstance> dicom(new Neuro::InputDicomInstance(tags));
  target.AddInstance(dicom.release(), "nope");
}

TEST(Autogenerated, Test_97)
{
  Neuro::DicomInstancesCollection instances;
  LoadInstance_97(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 460);
  ASSERT_NEAR(nifti.pixdim[1], 0.500000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 512);
  ASSERT_NEAR(nifti.pixdim[2], 0.500000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 1);
  ASSERT_NEAR(nifti.pixdim[3], 1.000000, 0.00001);
  ASSERT_EQ(nifti.ndim, 3);
  ASSERT_EQ(nifti.dim[0], 3);
  ASSERT_EQ(nifti.nvox, 235520u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(54));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(54));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(54));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], 0.002297, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.012944, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], 0.999654, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 82.942032, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], -0.499994, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 0.000941, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], 0.004547, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], 112.497772, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000882, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 0.499832, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], -0.025896, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -134.909836, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.494170, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, -0.492796, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, -0.504831, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=3.1;Time=134548.132");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadInstance_98(Neuro::DicomInstancesCollection& target)
{
  // ('heudiconv', 'inputs/rawdata/dicoms/MR.1.3.46.670589.11.38317.5.0.4476.2014042516093364426')
  Orthanc::DicomMap tags;
  tags.SetValue(0x0008, 0x0060, "MR", false);
  tags.SetValue(0x0008, 0x0070, "Philips Medical Systems", false);
  tags.SetValue(0x0020, 0x0013, "4835", false);
  tags.SetValue(0x2005, 0x1409, "0", false);
  tags.SetValue(0x2005, 0x140a, "4.00757020757020", false);
  tags.SetValue(0x0018, 0x0080, "2000.00061035156", false);
  tags.SetValue(0x0018, 0x0081, "30", false);
  tags.SetValue(0x0018, 0x1312, "COL", false);
  tags.SetValue(0x2005, 0x100e, "0.013238308019936085", false);
  tags.SetValue(0x0008, 0x0016, "1.2.840.10008.5.1.4.1.1.4", false);
  tags.SetValue(0x0028, 0x0002, "1", false);
  tags.SetValue(0x0028, 0x0004, "MONOCHROME2", false);
  tags.SetValue(0x0028, 0x0010, "80", false);
  tags.SetValue(0x0028, 0x0011, "80", false);
  tags.SetValue(0x0028, 0x0100, "16", false);
  tags.SetValue(0x0028, 0x0101, "12", false);
  tags.SetValue(0x0028, 0x0102, "11", false);
  tags.SetValue(0x0028, 0x0103, "0", false);
  tags.SetValue(0x0008, 0x0032, "160221.19", false);
  tags.SetValue(0x0018, 0x0050, "3", false);
  tags.SetValue(0x0018, 0x0088, "3.3", false);
  tags.SetValue(0x0020, 0x0032, "-120.9212288591\\-120.34524778928\\95.9285560837016", false);
  tags.SetValue(0x0020, 0x0037, "0.99980485439300\\-0.0114281568676\\0.01611445285379\\0.01479053776711\\0.97376281023025\\-0.2270841896533", false);
  tags.SetValue(0x0028, 0x0030, "3\\3", false);
  std::unique_ptr<Neuro::InputDicomInstance> dicom(new Neuro::InputDicomInstance(tags));
  target.AddInstance(dicom.release(), "nope");
}

TEST(Autogenerated, Test_98)
{
  Neuro::DicomInstancesCollection instances;
  LoadInstance_98(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 80);
  ASSERT_NEAR(nifti.pixdim[1], 3.000000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 80);
  ASSERT_NEAR(nifti.pixdim[2], 3.000000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 1);
  ASSERT_NEAR(nifti.pixdim[3], 3.300000, 0.00001);
  ASSERT_EQ(nifti.ndim, 3);
  ASSERT_EQ(nifti.dim[0], 3);
  ASSERT_EQ(nifti.nvox, 6400u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(57));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(57));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(57));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 4.007570);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -2.999414, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.044372, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], 0.043218, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 117.415871, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], 0.034284, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 2.921288, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], -0.750018, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], -110.436539, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.048343, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 0.681253, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 3.213348, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], 42.109604, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.006598, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, 0.993392, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, 0.114346, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=30;Time=160221.190");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadSeries_99(Neuro::DicomInstancesCollection& target)
{
  LoadInstance_2(target);
  LoadInstance_3(target);
}

TEST(Autogenerated, Test_99)
{
  Neuro::DicomInstancesCollection instances;
  LoadSeries_99(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 64);
  ASSERT_NEAR(nifti.pixdim[1], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 64);
  ASSERT_NEAR(nifti.pixdim[2], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 35);
  ASSERT_NEAR(nifti.pixdim[3], 3.600000, 0.00001);
  ASSERT_EQ(nifti.dim[4], 2);
  ASSERT_NEAR(nifti.pixdim[4], 3.000000, 0.00001);
  ASSERT_EQ(nifti.ndim, 4);
  ASSERT_EQ(nifti.dim[0], 4);
  ASSERT_EQ(nifti.nvox, 286720u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(57));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(57));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(57));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 1);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -3.250000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 104.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 3.230991, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], -0.388798, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], -58.684311, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 0.350998, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 3.578943, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -84.798035, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, 0.998537, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, 0.054079, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=30;Time=134935.305;phase=1");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadSeries_100(Neuro::DicomInstancesCollection& target)
{
  LoadInstance_4(target);
  LoadInstance_5(target);
}

TEST(Autogenerated, Test_100)
{
  Neuro::DicomInstancesCollection instances;
  LoadSeries_100(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 64);
  ASSERT_NEAR(nifti.pixdim[1], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 64);
  ASSERT_NEAR(nifti.pixdim[2], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 36);
  ASSERT_NEAR(nifti.pixdim[3], 3.600000, 0.00001);
  ASSERT_EQ(nifti.dim[4], 2);
  ASSERT_NEAR(nifti.pixdim[4], 3.000000, 0.00001);
  ASSERT_EQ(nifti.ndim, 4);
  ASSERT_EQ(nifti.dim[0], 4);
  ASSERT_EQ(nifti.nvox, 294912u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(57));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(57));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(57));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 1);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -3.250000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 104.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 3.230991, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], -0.388798, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], -58.489914, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 0.350998, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 3.578943, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -86.587509, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, 0.998537, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, 0.054079, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=30;Time=135252.445;phase=1");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadSeries_101(Neuro::DicomInstancesCollection& target)
{
  LoadInstance_6(target);
  LoadInstance_7(target);
}

TEST(Autogenerated, Test_101)
{
  Neuro::DicomInstancesCollection instances;
  LoadSeries_101(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 64);
  ASSERT_NEAR(nifti.pixdim[1], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 64);
  ASSERT_NEAR(nifti.pixdim[2], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 36);
  ASSERT_NEAR(nifti.pixdim[3], 3.600000, 0.00001);
  ASSERT_EQ(nifti.dim[4], 2);
  ASSERT_NEAR(nifti.pixdim[4], 3.000000, 0.00001);
  ASSERT_EQ(nifti.ndim, 4);
  ASSERT_EQ(nifti.dim[0], 4);
  ASSERT_EQ(nifti.nvox, 294912u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(57));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(57));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(57));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 1);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -3.250000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 104.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 3.230991, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], -0.388798, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], -58.489914, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 0.350998, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 3.578943, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -86.587509, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, 0.998537, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, 0.054079, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=30;Time=135416.225;phase=1");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadSeries_102(Neuro::DicomInstancesCollection& target)
{
  LoadInstance_8(target);
  LoadInstance_9(target);
}

TEST(Autogenerated, Test_102)
{
  Neuro::DicomInstancesCollection instances;
  LoadSeries_102(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 64);
  ASSERT_NEAR(nifti.pixdim[1], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 64);
  ASSERT_NEAR(nifti.pixdim[2], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 35);
  ASSERT_NEAR(nifti.pixdim[3], 3.600000, 0.00001);
  ASSERT_EQ(nifti.dim[4], 2);
  ASSERT_NEAR(nifti.pixdim[4], 3.000000, 0.00001);
  ASSERT_EQ(nifti.ndim, 4);
  ASSERT_EQ(nifti.dim[0], 4);
  ASSERT_EQ(nifti.nvox, 286720u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(57));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(57));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(57));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 2);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -3.250000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 104.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 3.230991, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], -0.388798, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], -58.684311, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 0.350998, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 3.578943, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -84.798035, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, 0.998537, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, 0.054079, 0.0001);
  ASSERT_TRUE(std::string(nifti.descrip) == "TE=30;Time=135041.527;phase=1" ||
              std::string(nifti.descrip) == "TE=30;Time=135041.528;phase=1");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadSeries_103(Neuro::DicomInstancesCollection& target)
{
  LoadInstance_10(target);
  LoadInstance_11(target);
}

TEST(Autogenerated, Test_103)
{
  Neuro::DicomInstancesCollection instances;
  LoadSeries_103(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 64);
  ASSERT_NEAR(nifti.pixdim[1], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 64);
  ASSERT_NEAR(nifti.pixdim[2], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 36);
  ASSERT_NEAR(nifti.pixdim[3], 3.600000, 0.00001);
  ASSERT_EQ(nifti.dim[4], 2);
  ASSERT_NEAR(nifti.pixdim[4], 3.000000, 0.00001);
  ASSERT_EQ(nifti.ndim, 4);
  ASSERT_EQ(nifti.dim[0], 4);
  ASSERT_EQ(nifti.nvox, 294912u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(57));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(57));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(57));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 2);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -3.250000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 104.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 3.230991, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], -0.388798, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], -58.489914, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 0.350998, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 3.578943, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -86.587509, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, 0.998537, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, 0.054079, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=30;Time=135332.235;phase=1");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadSeries_104(Neuro::DicomInstancesCollection& target)
{
  LoadInstance_12(target);
  LoadInstance_13(target);
}

TEST(Autogenerated, Test_104)
{
  Neuro::DicomInstancesCollection instances;
  LoadSeries_104(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 64);
  ASSERT_NEAR(nifti.pixdim[1], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 64);
  ASSERT_NEAR(nifti.pixdim[2], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 35);
  ASSERT_NEAR(nifti.pixdim[3], 3.600000, 0.00001);
  ASSERT_EQ(nifti.dim[4], 2);
  ASSERT_NEAR(nifti.pixdim[4], 3.000000, 0.00001);
  ASSERT_EQ(nifti.ndim, 4);
  ASSERT_EQ(nifti.dim[0], 4);
  ASSERT_EQ(nifti.nvox, 286720u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(57));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(57));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(57));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 3);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -3.250000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 104.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 3.230991, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], -0.388798, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], -58.684311, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 0.350998, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 3.578943, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -84.798035, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, 0.998537, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, 0.054079, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=30;Time=135110.870;phase=1");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadSeries_105(Neuro::DicomInstancesCollection& target)
{
  LoadInstance_14(target);
  LoadInstance_15(target);
}

TEST(Autogenerated, Test_105)
{
  Neuro::DicomInstancesCollection instances;
  LoadSeries_105(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 64);
  ASSERT_NEAR(nifti.pixdim[1], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 64);
  ASSERT_NEAR(nifti.pixdim[2], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 36);
  ASSERT_NEAR(nifti.pixdim[3], 3.600000, 0.00001);
  ASSERT_EQ(nifti.dim[4], 2);
  ASSERT_NEAR(nifti.pixdim[4], 3.000000, 0.00001);
  ASSERT_EQ(nifti.ndim, 4);
  ASSERT_EQ(nifti.dim[0], 4);
  ASSERT_EQ(nifti.nvox, 294912u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(57));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(57));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(57));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 5);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -3.250000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 104.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 3.230991, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], -0.388798, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], -58.489914, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 0.350998, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 3.578943, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -86.587509, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, 0.998537, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, 0.054079, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=30;Time=135444.723;phase=1");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadSeries_106(Neuro::DicomInstancesCollection& target)
{
  LoadInstance_16(target);
  LoadInstance_17(target);
}

TEST(Autogenerated, Test_106)
{
  Neuro::DicomInstancesCollection instances;
  LoadSeries_106(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 86);
  ASSERT_NEAR(nifti.pixdim[1], 2.697675, 0.00001);
  ASSERT_EQ(nifti.dim[2], 86);
  ASSERT_NEAR(nifti.pixdim[2], 2.697675, 0.00001);
  ASSERT_EQ(nifti.dim[3], 36);
  ASSERT_NEAR(nifti.pixdim[3], 3.600000, 0.00001);
  ASSERT_EQ(nifti.dim[4], 2);
  ASSERT_NEAR(nifti.pixdim[4], 3.000000, 0.00001);
  ASSERT_EQ(nifti.ndim, 4);
  ASSERT_EQ(nifti.dim[0], 4);
  ASSERT_EQ(nifti.nvox, 532512u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(57));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(57));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(57));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 1);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -2.697675, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 115.999977, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 2.654202, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], -0.643688, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], -58.807571, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 0.482350, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 3.541986, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -93.139343, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, -0.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, 0.995963, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, 0.089763, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=34;Time=140149.418;phase=1;mb=2");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadSeries_107(Neuro::DicomInstancesCollection& target)
{
  LoadInstance_18(target);
  LoadInstance_19(target);
}

TEST(Autogenerated, Test_107)
{
  Neuro::DicomInstancesCollection instances;
  LoadSeries_107(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 64);
  ASSERT_NEAR(nifti.pixdim[1], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 64);
  ASSERT_NEAR(nifti.pixdim[2], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 35);
  ASSERT_NEAR(nifti.pixdim[3], 3.600000, 0.00001);
  ASSERT_EQ(nifti.dim[4], 2);
  ASSERT_NEAR(nifti.pixdim[4], 3.000000, 0.00001);
  ASSERT_EQ(nifti.ndim, 4);
  ASSERT_EQ(nifti.dim[0], 4);
  ASSERT_EQ(nifti.nvox, 286720u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(54));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(54));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(54));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 1);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -3.250000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 104.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], -0.497204, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], -3.557622, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], 148.532135, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 3.211742, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], -0.550749, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -92.380424, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, 0.650774, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, 0.759271, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=30;Time=135701.908;phase=1");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadSeries_108(Neuro::DicomInstancesCollection& target)
{
  LoadInstance_20(target);
  LoadInstance_21(target);
}

TEST(Autogenerated, Test_108)
{
  Neuro::DicomInstancesCollection instances;
  LoadSeries_108(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 64);
  ASSERT_NEAR(nifti.pixdim[1], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 64);
  ASSERT_NEAR(nifti.pixdim[2], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 36);
  ASSERT_NEAR(nifti.pixdim[3], 3.600000, 0.00001);
  ASSERT_EQ(nifti.dim[4], 2);
  ASSERT_NEAR(nifti.pixdim[4], 3.000000, 0.00001);
  ASSERT_EQ(nifti.ndim, 4);
  ASSERT_EQ(nifti.dim[0], 4);
  ASSERT_EQ(nifti.nvox, 294912u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(54));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(54));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(54));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 1);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -3.250000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 104.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], -0.497204, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], -3.557622, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], 150.310944, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 3.211742, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], -0.550749, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -92.105034, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, 0.650774, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, 0.759271, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=30;Time=135534.370;phase=1");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadSeries_109(Neuro::DicomInstancesCollection& target)
{
  LoadInstance_22(target);
  LoadInstance_23(target);
}

TEST(Autogenerated, Test_109)
{
  Neuro::DicomInstancesCollection instances;
  LoadSeries_109(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 64);
  ASSERT_NEAR(nifti.pixdim[1], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 64);
  ASSERT_NEAR(nifti.pixdim[2], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 35);
  ASSERT_NEAR(nifti.pixdim[3], 3.600000, 0.00001);
  ASSERT_EQ(nifti.dim[4], 2);
  ASSERT_NEAR(nifti.pixdim[4], 3.000000, 0.00001);
  ASSERT_EQ(nifti.ndim, 4);
  ASSERT_EQ(nifti.dim[0], 4);
  ASSERT_EQ(nifti.nvox, 286720u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(54));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(54));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(54));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 2);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -3.250000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 104.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], -0.497204, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], -3.557622, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], 148.532135, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 3.211742, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], -0.550749, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -92.380424, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, 0.650774, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, 0.759271, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=30;Time=135734.797;phase=1");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadSeries_110(Neuro::DicomInstancesCollection& target)
{
  LoadInstance_24(target);
  LoadInstance_25(target);
}

TEST(Autogenerated, Test_110)
{
  Neuro::DicomInstancesCollection instances;
  LoadSeries_110(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 64);
  ASSERT_NEAR(nifti.pixdim[1], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 64);
  ASSERT_NEAR(nifti.pixdim[2], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 36);
  ASSERT_NEAR(nifti.pixdim[3], 3.600000, 0.00001);
  ASSERT_EQ(nifti.dim[4], 2);
  ASSERT_NEAR(nifti.pixdim[4], 3.000000, 0.00001);
  ASSERT_EQ(nifti.ndim, 4);
  ASSERT_EQ(nifti.dim[0], 4);
  ASSERT_EQ(nifti.nvox, 294912u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(54));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(54));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(54));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 2);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -3.250000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 104.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], -0.497204, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], -3.557622, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], 150.310944, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 3.211742, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], -0.550749, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -92.105034, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, 0.650774, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, 0.759271, 0.0001);
  ASSERT_TRUE(std::string(nifti.descrip) == "TE=30;Time=135604.842;phase=1" ||
              std::string(nifti.descrip) == "TE=30;Time=135604.843;phase=1");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadSeries_111(Neuro::DicomInstancesCollection& target)
{
  LoadInstance_26(target);
  LoadInstance_27(target);
}

TEST(Autogenerated, Test_111)
{
  Neuro::DicomInstancesCollection instances;
  LoadSeries_111(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 64);
  ASSERT_NEAR(nifti.pixdim[1], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 64);
  ASSERT_NEAR(nifti.pixdim[2], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 35);
  ASSERT_NEAR(nifti.pixdim[3], 3.600000, 0.00001);
  ASSERT_EQ(nifti.dim[4], 2);
  ASSERT_NEAR(nifti.pixdim[4], 3.000000, 0.00001);
  ASSERT_EQ(nifti.ndim, 4);
  ASSERT_EQ(nifti.dim[0], 4);
  ASSERT_EQ(nifti.nvox, 286720u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(54));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(54));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(54));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 3);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -3.250000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 104.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], -0.497204, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], -3.557622, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], 148.532135, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 3.211742, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], -0.550749, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -92.380424, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, 0.650774, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, 0.759271, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=30;Time=135801.485;phase=1");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadSeries_112(Neuro::DicomInstancesCollection& target)
{
  LoadInstance_28(target);
  LoadInstance_29(target);
}

TEST(Autogenerated, Test_112)
{
  Neuro::DicomInstancesCollection instances;
  LoadSeries_112(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 64);
  ASSERT_NEAR(nifti.pixdim[1], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 64);
  ASSERT_NEAR(nifti.pixdim[2], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 36);
  ASSERT_NEAR(nifti.pixdim[3], 3.600000, 0.00001);
  ASSERT_EQ(nifti.dim[4], 2);
  ASSERT_NEAR(nifti.pixdim[4], 3.000000, 0.00001);
  ASSERT_EQ(nifti.ndim, 4);
  ASSERT_EQ(nifti.dim[0], 4);
  ASSERT_EQ(nifti.nvox, 294912u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(54));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(54));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(54));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 5);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -3.250000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 104.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], -0.497204, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], -3.557622, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], 150.310944, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 3.211742, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], -0.550749, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -92.105034, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, 0.650774, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, 0.759271, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=30;Time=135633.508;phase=1");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadSeries_113(Neuro::DicomInstancesCollection& target)
{
  LoadInstance_30(target);
  LoadInstance_31(target);
}

TEST(Autogenerated, Test_113)
{
  Neuro::DicomInstancesCollection instances;
  LoadSeries_113(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 64);
  ASSERT_NEAR(nifti.pixdim[1], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 64);
  ASSERT_NEAR(nifti.pixdim[2], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 35);
  ASSERT_NEAR(nifti.pixdim[3], 3.600000, 0.00001);
  ASSERT_EQ(nifti.dim[4], 2);
  ASSERT_NEAR(nifti.pixdim[4], 3.000000, 0.00001);
  ASSERT_EQ(nifti.ndim, 4);
  ASSERT_EQ(nifti.dim[0], 4);
  ASSERT_EQ(nifti.nvox, 286720u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(54));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(54));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(54));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 1);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -3.600000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 61.200001, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], -3.250000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], 140.319641, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 3.250000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -126.173706, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.500000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, -0.500000, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, -0.500000, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=30;Time=140000.990;phase=1");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadSeries_114(Neuro::DicomInstancesCollection& target)
{
  LoadInstance_32(target);
  LoadInstance_33(target);
}

TEST(Autogenerated, Test_114)
{
  Neuro::DicomInstancesCollection instances;
  LoadSeries_114(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 64);
  ASSERT_NEAR(nifti.pixdim[1], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 64);
  ASSERT_NEAR(nifti.pixdim[2], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 36);
  ASSERT_NEAR(nifti.pixdim[3], 3.600000, 0.00001);
  ASSERT_EQ(nifti.dim[4], 2);
  ASSERT_NEAR(nifti.pixdim[4], 3.000000, 0.00001);
  ASSERT_EQ(nifti.ndim, 4);
  ASSERT_EQ(nifti.dim[0], 4);
  ASSERT_EQ(nifti.nvox, 294912u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(54));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(54));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(54));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 1);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -3.600000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 63.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], -3.250000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], 140.319641, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 3.250000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -126.173706, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.500000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, -0.500000, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, -0.500000, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=30;Time=135835.760;phase=1");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadSeries_115(Neuro::DicomInstancesCollection& target)
{
  LoadInstance_34(target);
  LoadInstance_35(target);
}

TEST(Autogenerated, Test_115)
{
  Neuro::DicomInstancesCollection instances;
  LoadSeries_115(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 64);
  ASSERT_NEAR(nifti.pixdim[1], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 64);
  ASSERT_NEAR(nifti.pixdim[2], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 35);
  ASSERT_NEAR(nifti.pixdim[3], 3.600000, 0.00001);
  ASSERT_EQ(nifti.dim[4], 2);
  ASSERT_NEAR(nifti.pixdim[4], 3.000000, 0.00001);
  ASSERT_EQ(nifti.ndim, 4);
  ASSERT_EQ(nifti.dim[0], 4);
  ASSERT_EQ(nifti.nvox, 286720u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(54));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(54));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(54));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 2);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -3.600000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 61.200001, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], -3.250000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], 140.319641, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 3.250000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -126.173706, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.500000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, -0.500000, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, -0.500000, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=30;Time=140034.397;phase=1");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadSeries_116(Neuro::DicomInstancesCollection& target)
{
  LoadInstance_36(target);
  LoadInstance_37(target);
}

TEST(Autogenerated, Test_116)
{
  Neuro::DicomInstancesCollection instances;
  LoadSeries_116(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 64);
  ASSERT_NEAR(nifti.pixdim[1], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 64);
  ASSERT_NEAR(nifti.pixdim[2], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 36);
  ASSERT_NEAR(nifti.pixdim[3], 3.600000, 0.00001);
  ASSERT_EQ(nifti.dim[4], 2);
  ASSERT_NEAR(nifti.pixdim[4], 3.000000, 0.00001);
  ASSERT_EQ(nifti.ndim, 4);
  ASSERT_EQ(nifti.dim[0], 4);
  ASSERT_EQ(nifti.nvox, 294912u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(54));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(54));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(54));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 2);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -3.600000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 63.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], -3.250000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], 140.319641, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 3.250000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -126.173706, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.500000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, -0.500000, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, -0.500000, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=30;Time=135905.272;phase=1");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadSeries_117(Neuro::DicomInstancesCollection& target)
{
  LoadInstance_38(target);
  LoadInstance_39(target);
}

TEST(Autogenerated, Test_117)
{
  Neuro::DicomInstancesCollection instances;
  LoadSeries_117(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 64);
  ASSERT_NEAR(nifti.pixdim[1], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 64);
  ASSERT_NEAR(nifti.pixdim[2], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 35);
  ASSERT_NEAR(nifti.pixdim[3], 3.600000, 0.00001);
  ASSERT_EQ(nifti.dim[4], 2);
  ASSERT_NEAR(nifti.pixdim[4], 3.000000, 0.00001);
  ASSERT_EQ(nifti.ndim, 4);
  ASSERT_EQ(nifti.dim[0], 4);
  ASSERT_EQ(nifti.nvox, 286720u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(54));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(54));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(54));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 3);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -3.600000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 61.200001, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], -3.250000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], 140.319641, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 3.250000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -126.173706, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.500000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, -0.500000, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, -0.500000, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=30;Time=140105.875;phase=1");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadSeries_118(Neuro::DicomInstancesCollection& target)
{
  LoadInstance_40(target);
  LoadInstance_41(target);
}

TEST(Autogenerated, Test_118)
{
  Neuro::DicomInstancesCollection instances;
  LoadSeries_118(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 64);
  ASSERT_NEAR(nifti.pixdim[1], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 64);
  ASSERT_NEAR(nifti.pixdim[2], 3.250000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 36);
  ASSERT_NEAR(nifti.pixdim[3], 3.600000, 0.00001);
  ASSERT_EQ(nifti.dim[4], 2);
  ASSERT_NEAR(nifti.pixdim[4], 3.000000, 0.00001);
  ASSERT_EQ(nifti.ndim, 4);
  ASSERT_EQ(nifti.dim[0], 4);
  ASSERT_EQ(nifti.nvox, 294912u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(54));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(54));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(54));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 5);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -3.600000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 63.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], -3.250000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], 140.319641, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 3.250000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -126.173706, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.500000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, -0.500000, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, -0.500000, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=30;Time=135931.837;phase=1");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadSeries_119(Neuro::DicomInstancesCollection& target)
{
  LoadInstance_70(target);
  LoadInstance_71(target);
}

TEST(Autogenerated, Test_119)
{
  Neuro::DicomInstancesCollection instances;
  LoadSeries_119(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 64);
  ASSERT_NEAR(nifti.pixdim[1], 3.750000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 64);
  ASSERT_NEAR(nifti.pixdim[2], 3.750000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 2);
  ASSERT_NEAR(nifti.pixdim[3], 5.000000, 0.00001);
  ASSERT_EQ(nifti.ndim, 3);
  ASSERT_EQ(nifti.dim[0], 3);
  ASSERT_EQ(nifti.nvox, 8192u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 4);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(57));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(57));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(57));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -3.750000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 118.125000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 3.750000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], -92.906006, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 5.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -17.687500, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, 1.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, 0.000000, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=30;Time=114815.000");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadSeries_120(Neuro::DicomInstancesCollection& target)
{
  LoadInstance_72(target);
  LoadInstance_73(target);
}

TEST(Autogenerated, Test_120)
{
  Neuro::DicomInstancesCollection instances;
  LoadSeries_120(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 64);
  ASSERT_NEAR(nifti.pixdim[1], 3.750000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 64);
  ASSERT_NEAR(nifti.pixdim[2], 3.750000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 2);
  ASSERT_NEAR(nifti.pixdim[3], 5.000000, 0.00001);
  ASSERT_EQ(nifti.ndim, 3);
  ASSERT_EQ(nifti.dim[0], 3);
  ASSERT_EQ(nifti.nvox, 8192u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 4);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(57));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(57));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(57));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -3.750000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 118.125000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 3.750000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], -92.906006, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 5.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -17.687500, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, 1.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, 0.000000, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=30;Time=115224.000");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadSeries_121(Neuro::DicomInstancesCollection& target)
{
  LoadInstance_74(target);
  LoadInstance_75(target);
}

TEST(Autogenerated, Test_121)
{
  Neuro::DicomInstancesCollection instances;
  LoadSeries_121(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 64);
  ASSERT_NEAR(nifti.pixdim[1], 3.750000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 64);
  ASSERT_NEAR(nifti.pixdim[2], 3.750000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 2);
  ASSERT_NEAR(nifti.pixdim[3], 5.000000, 0.00001);
  ASSERT_EQ(nifti.ndim, 3);
  ASSERT_EQ(nifti.dim[0], 3);
  ASSERT_EQ(nifti.nvox, 8192u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 4);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(57));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(57));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(57));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -3.750000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 118.125000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 3.750000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], -92.906006, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 5.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], 47.312500, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, 1.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, 0.000000, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=30;Time=115418.000");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadSeries_122(Neuro::DicomInstancesCollection& target)
{
  LoadInstance_76(target);
  LoadInstance_77(target);
}

TEST(Autogenerated, Test_122)
{
  Neuro::DicomInstancesCollection instances;
  LoadSeries_122(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 64);
  ASSERT_NEAR(nifti.pixdim[1], 3.750000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 64);
  ASSERT_NEAR(nifti.pixdim[2], 3.750000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 2);
  ASSERT_NEAR(nifti.pixdim[3], 5.000000, 0.00001);
  ASSERT_EQ(nifti.ndim, 3);
  ASSERT_EQ(nifti.dim[0], 3);
  ASSERT_EQ(nifti.nvox, 8192u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 4);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(57));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(57));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(57));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -3.750000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 118.125000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 3.750000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], -92.906006, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 5.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], 47.312500, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, 1.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, 0.000000, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=30;Time=115551.000");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadSeries_123(Neuro::DicomInstancesCollection& target)
{
  LoadInstance_78(target);
  LoadInstance_79(target);
}

TEST(Autogenerated, Test_123)
{
  Neuro::DicomInstancesCollection instances;
  LoadSeries_123(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 72);
  ASSERT_NEAR(nifti.pixdim[1], 3.000000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 72);
  ASSERT_NEAR(nifti.pixdim[2], 3.000000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 5);
  ASSERT_NEAR(nifti.pixdim[3], 12.000000, 0.00001);
  ASSERT_EQ(nifti.dim[4], 2);
  ASSERT_NEAR(nifti.pixdim[4], 2.435370, 0.00001);
  ASSERT_EQ(nifti.ndim, 4);
  ASSERT_EQ(nifti.dim[0], 4);
  ASSERT_EQ(nifti.nvox, 51840u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(57));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(57));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(57));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 3);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -3.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 108.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 3.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], -105.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 12.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], 16.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, 1.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, 0.000000, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=50;Time=122458.102;phase=1");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadSeries_124(Neuro::DicomInstancesCollection& target)
{
  LoadInstance_80(target);
  LoadInstance_81(target);
}

TEST(Autogenerated, Test_124)
{
  Neuro::DicomInstancesCollection instances;
  LoadSeries_124(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 72);
  ASSERT_NEAR(nifti.pixdim[1], 3.000000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 72);
  ASSERT_NEAR(nifti.pixdim[2], 3.000000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 5);
  ASSERT_NEAR(nifti.pixdim[3], 12.000000, 0.00001);
  ASSERT_EQ(nifti.dim[4], 2);
  ASSERT_NEAR(nifti.pixdim[4], 2.435370, 0.00001);
  ASSERT_EQ(nifti.ndim, 4);
  ASSERT_EQ(nifti.dim[0], 4);
  ASSERT_EQ(nifti.nvox, 51840u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(57));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(57));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(57));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 3);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -3.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 108.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 3.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], -105.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 12.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], 16.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, 1.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, 0.000000, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=50;Time=122654.518;phase=0");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadSeries_125(Neuro::DicomInstancesCollection& target)
{
  LoadInstance_82(target);
  LoadInstance_83(target);
}

TEST(Autogenerated, Test_125)
{
  Neuro::DicomInstancesCollection instances;
  LoadSeries_125(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 72);
  ASSERT_NEAR(nifti.pixdim[1], 3.000000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 72);
  ASSERT_NEAR(nifti.pixdim[2], 3.000000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 5);
  ASSERT_NEAR(nifti.pixdim[3], 12.000000, 0.00001);
  ASSERT_EQ(nifti.dim[4], 2);
  ASSERT_NEAR(nifti.pixdim[4], 2.435370, 0.00001);
  ASSERT_EQ(nifti.ndim, 4);
  ASSERT_EQ(nifti.dim[0], 4);
  ASSERT_EQ(nifti.nvox, 51840u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(54));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(54));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(54));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 3);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -3.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 108.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 3.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], -105.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 12.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], 16.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, 1.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, 0.000000, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=50;Time=122812.085;phase=1");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadSeries_126(Neuro::DicomInstancesCollection& target)
{
  LoadInstance_84(target);
  LoadInstance_85(target);
}

TEST(Autogenerated, Test_126)
{
  Neuro::DicomInstancesCollection instances;
  LoadSeries_126(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 72);
  ASSERT_NEAR(nifti.pixdim[1], 3.000000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 72);
  ASSERT_NEAR(nifti.pixdim[2], 3.000000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 5);
  ASSERT_NEAR(nifti.pixdim[3], 12.000000, 0.00001);
  ASSERT_EQ(nifti.dim[4], 2);
  ASSERT_NEAR(nifti.pixdim[4], 2.435370, 0.00001);
  ASSERT_EQ(nifti.ndim, 4);
  ASSERT_EQ(nifti.dim[0], 4);
  ASSERT_EQ(nifti.nvox, 51840u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(54));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(54));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(54));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 3);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -3.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 108.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 3.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], -105.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 12.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], 16.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, 1.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, 0.000000, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=50;Time=123002.285;phase=0");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadSeries_127(Neuro::DicomInstancesCollection& target)
{
  LoadInstance_86(target);
  LoadInstance_87(target);
}

TEST(Autogenerated, Test_127)
{
  Neuro::DicomInstancesCollection instances;
  LoadSeries_127(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 64);
  ASSERT_NEAR(nifti.pixdim[1], 3.500000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 64);
  ASSERT_NEAR(nifti.pixdim[2], 3.500000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 20);
  ASSERT_NEAR(nifti.pixdim[3], 4.200000, 0.00001);
  ASSERT_EQ(nifti.dim[4], 2);
  ASSERT_NEAR(nifti.pixdim[4], 5.865000, 0.00001);
  ASSERT_EQ(nifti.ndim, 4);
  ASSERT_EQ(nifti.dim[0], 4);
  ASSERT_EQ(nifti.nvox, 163840u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(57));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(57));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(57));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -3.173351, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 1.472947, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], 0.121611, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 61.851830, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], 1.473565, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 3.174682, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], -141.426270, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.091923, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], -0.042667, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 4.198239, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -6.530584, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, -0.215556, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, -0.976384, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, 0.003121, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=71;Time=133742.724");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadSeries_128(Neuro::DicomInstancesCollection& target)
{
  LoadInstance_88(target);
  LoadInstance_89(target);
}

TEST(Autogenerated, DISABLED_Test_128)
{
  Neuro::DicomInstancesCollection instances;
  LoadSeries_128(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 2);
  ASSERT_NEAR(nifti.pixdim[1], 1.000000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 460);
  ASSERT_NEAR(nifti.pixdim[2], 0.500000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 512);
  ASSERT_NEAR(nifti.pixdim[3], 0.500000, 0.00001);
  ASSERT_EQ(nifti.ndim, 3);
  ASSERT_EQ(nifti.dim[0], 3);
  ASSERT_EQ(nifti.nvox, 471040u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(54));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(54));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(54));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], 0.997688, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], 0.033996, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 76.618248, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], -0.067993, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 0.498843, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], -118.861221, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 0.500000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -129.396652, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, -0.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, -0.000000, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, 0.034016, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=3.1;Time=132417.803");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadSeries_129(Neuro::DicomInstancesCollection& target)
{
  LoadInstance_90(target);
  LoadInstance_91(target);
}

TEST(Autogenerated, Test_129)
{
  Neuro::DicomInstancesCollection instances;
  LoadSeries_129(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 64);
  ASSERT_NEAR(nifti.pixdim[1], 3.500000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 64);
  ASSERT_NEAR(nifti.pixdim[2], 3.500000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 20);
  ASSERT_NEAR(nifti.pixdim[3], 4.200000, 0.00001);
  ASSERT_EQ(nifti.dim[4], 2);
  ASSERT_NEAR(nifti.pixdim[4], 5.865000, 0.00001);
  ASSERT_EQ(nifti.ndim, 4);
  ASSERT_EQ(nifti.dim[0], 4);
  ASSERT_EQ(nifti.nvox, 163840u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(57));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(57));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(57));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -3.496896, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], -0.106993, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], 0.121611, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 121.183418, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], -0.107038, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 3.498363, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], -108.072556, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.101295, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 0.003099, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 4.198239, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -6.589681, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.015291, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, -0.999778, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, -0.000221, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=72;Time=135713.284");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadSeries_130(Neuro::DicomInstancesCollection& target)
{
  LoadInstance_92(target);
  LoadInstance_93(target);
}

TEST(Autogenerated, Test_130)
{
  Neuro::DicomInstancesCollection instances;
  LoadSeries_130(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 64);
  ASSERT_NEAR(nifti.pixdim[1], 3.500000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 64);
  ASSERT_NEAR(nifti.pixdim[2], 3.500000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 20);
  ASSERT_NEAR(nifti.pixdim[3], 4.200000, 0.00001);
  ASSERT_EQ(nifti.dim[4], 2);
  ASSERT_NEAR(nifti.pixdim[4], 5.865000, 0.00001);
  ASSERT_EQ(nifti.ndim, 4);
  ASSERT_EQ(nifti.dim[0], 4);
  ASSERT_EQ(nifti.nvox, 163840u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(57));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(57));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(57));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -3.496896, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], -0.106993, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], 0.121611, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 121.183418, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], -0.107038, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 3.498363, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], 0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], -108.072556, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.101295, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 0.003099, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 4.198239, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -6.589681, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.015291, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, -0.999778, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, -0.000221, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=71;Time=135257.683");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadSeries_131(Neuro::DicomInstancesCollection& target)
{
  LoadInstance_94(target);
  LoadInstance_95(target);
}

TEST(Autogenerated, Test_131)
{
  Neuro::DicomInstancesCollection instances;
  LoadSeries_131(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 64);
  ASSERT_NEAR(nifti.pixdim[1], 3.500000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 64);
  ASSERT_NEAR(nifti.pixdim[2], 3.500000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 20);
  ASSERT_NEAR(nifti.pixdim[3], 4.200000, 0.00001);
  ASSERT_EQ(nifti.dim[4], 2);
  ASSERT_NEAR(nifti.pixdim[4], 5.865000, 0.00001);
  ASSERT_EQ(nifti.ndim, 4);
  ASSERT_EQ(nifti.dim[0], 4);
  ASSERT_EQ(nifti.nvox, 163840u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(57));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(57));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(57));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], -1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], -3.365068, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], -0.957105, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], 0.121611, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 143.318390, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], -0.957506, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 3.366479, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], -0.000000, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], -76.769180, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], 0.097476, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], 0.027725, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 4.198239, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -7.230867, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, 0.138096, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, -0.990313, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, -0.002000, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=71;Time=140136.182");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}

static void LoadSeries_132(Neuro::DicomInstancesCollection& target)
{
  LoadInstance_96(target);
  LoadInstance_97(target);
}

TEST(Autogenerated, DISABLED_Test_132)
{
  Neuro::DicomInstancesCollection instances;
  LoadSeries_132(instances);
  nifti_image nifti;
  std::vector<Neuro::Slice> slices;
  instances.CreateNiftiHeader(nifti, slices);
  CheckConsistency(nifti);
  ASSERT_EQ(nifti.dim[1], 2);
  ASSERT_NEAR(nifti.pixdim[1], 1.000000, 0.00001);
  ASSERT_EQ(nifti.dim[2], 460);
  ASSERT_NEAR(nifti.pixdim[2], 0.500000, 0.00001);
  ASSERT_EQ(nifti.dim[3], 512);
  ASSERT_NEAR(nifti.pixdim[3], 0.500000, 0.00001);
  ASSERT_EQ(nifti.ndim, 3);
  ASSERT_EQ(nifti.dim[0], 3);
  ASSERT_EQ(nifti.nvox, 471040u);
  ASSERT_EQ(nifti.nbyper, 2);
  ASSERT_EQ(nifti.datatype, 512);
  ASSERT_EQ(nifti.freq_dim, DIM_INFO_TO_FREQ_DIM(54));
  ASSERT_EQ(nifti.phase_dim, DIM_INFO_TO_PHASE_DIM(54));
  ASSERT_EQ(nifti.slice_dim, DIM_INFO_TO_SLICE_DIM(54));
  ASSERT_EQ(nifti.intent_code, 0);
  ASSERT_EQ(nifti.slice_start, 0);
  ASSERT_EQ(nifti.slice_end, 0);
  ASSERT_EQ(nifti.slice_code, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p1, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p2, 0);
  ASSERT_FLOAT_EQ(nifti.intent_p3, 0);
  ASSERT_FLOAT_EQ(nifti.pixdim[0], 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_slope, 1.000000);
  ASSERT_FLOAT_EQ(nifti.scl_inter, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_min, 0.000000);
  ASSERT_FLOAT_EQ(nifti.cal_max, 0.000000);
  ASSERT_FLOAT_EQ(nifti.slice_duration, 0.000000);
  ASSERT_FLOAT_EQ(nifti.toffset, 0.000000);
  ASSERT_EQ(nifti.qform_code, 1);
  ASSERT_EQ(nifti.sform_code, 1);
  ASSERT_NEAR(nifti.sto_xyz.m[0][0], 0.999649, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][1], -0.002297, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][2], 0.012944, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[0][3], 83.996399, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][0], 0.004547, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][1], 0.499994, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][2], 0.000941, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[1][3], -116.999451, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][0], -0.025896, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][1], -0.000882, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][2], 0.499832, 0.0001);
  ASSERT_NEAR(nifti.sto_xyz.m[2][3], -134.504913, 0.0001);
  ASSERT_NEAR(nifti.quatern_b, -0.000912, 0.0001);
  ASSERT_NEAR(nifti.quatern_c, 0.012947, 0.0001);
  ASSERT_NEAR(nifti.quatern_d, 0.002286, 0.0001);
  ASSERT_STREQ(nifti.descrip, "TE=3.1;Time=134548.132");
  ASSERT_STREQ(nifti.intent_name, "");
  if (nifti.ndim == 3)
    ASSERT_EQ(nifti.nz, static_cast<int>(slices.size()));
  else
    ASSERT_EQ(nifti.nz * nifti.nt, static_cast<int>(slices.size()));
}
