/*
    Platypus - program for creating Mac OS X application wrappers around scripts
    Copyright (C) 2003-2010 Sveinbjorn Thordarson <sveinbjornt@simnet.is>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

/*
 
 A Swiss Army Knife class with a plethora of generic utility functions
 
 */
#import "STUtil.h"


@implementation STUtil

+ (void)alert: (NSString *)message subText: (NSString *)subtext
{
	NSAlert *alert = [[NSAlert alloc] init];
	[alert addButtonWithTitle:@"OK"];
	[alert setMessageText: message];
	[alert setInformativeText: subtext];
	[alert setAlertStyle:NSWarningAlertStyle];
	
	if ([alert runModal] == NSAlertFirstButtonReturn) 
	{
		[alert release];
		return;
	} 
	[alert release];
}

+ (void)fatalAlert: (NSString *)message subText: (NSString *)subtext
{
	NSAlert *alert = [[NSAlert alloc] init];
	[alert addButtonWithTitle:@"OK"];
	[alert setMessageText: message];
	[alert setInformativeText: subtext];
	[alert setAlertStyle: NSCriticalAlertStyle];
	
	if ([alert runModal] == NSAlertFirstButtonReturn) 
	{
		[alert release];
		[[NSApplication sharedApplication] terminate: self];
		return;
	}
	[alert release];
}

+ (void)sheetAlert: (NSString *)message subText: (NSString *)subtext forWindow: (NSWindow *)window
{
	NSAlert *alert = [[NSAlert alloc] init];
	[alert addButtonWithTitle:@"OK"];
	[alert setMessageText: message];
	[alert setInformativeText: subtext];
	[alert setAlertStyle:NSCriticalAlertStyle];
	
	[alert beginSheetModalForWindow:window modalDelegate:self didEndSelector: nil contextInfo:nil];
	[alert release];
}

+ (BOOL) proceedWarning: (NSString *)message subText: (NSString *)subtext withAction: (NSString *)action
{

	NSAlert *alert = [[NSAlert alloc] init];
	[alert addButtonWithTitle: action];
	[alert addButtonWithTitle:@"Cancel"];
	[alert setMessageText: message];
	[alert setInformativeText: subtext];
	[alert setAlertStyle: NSWarningAlertStyle];
	
	if ([alert runModal] == NSAlertFirstButtonReturn) 
	{
		[alert release];
		return YES;
	}
	
	[alert release];
	return NO;
}

+ (UInt64) fileOrFolderSize: (NSString *)path
{
	UInt64			size = 0;
	NSFileManager	*manager = [NSFileManager defaultManager];
	BOOL			isDir;
	
	if (![manager fileExistsAtPath: path isDirectory: &isDir])
		return 0;
	
	if (isDir)
	{
		NSDirectoryEnumerator	*dirEnumerator = [manager enumeratorAtPath: path];
		while ([dirEnumerator nextObject])
		{
			if ([NSFileTypeRegular isEqualToString:[[dirEnumerator fileAttributes] fileType]])
				size += [[dirEnumerator fileAttributes] fileSize];
		}
	}
	else
	{
		size = [[manager fileAttributesAtPath: path traverseLink:YES] fileSize];
	}
	return (UInt64)size;
}

+ (NSString *) fileOrFolderSizeAsHumanReadable: (NSString *)path
{
	return [self sizeAsHumanReadable: [self fileOrFolderSize: path]];
}

+ (NSString *) sizeAsHumanReadable: (UInt64)size
{
	NSString	*str;
	
	if( size < 1024ULL ) 
	{
		/* bytes */
		str = [NSString stringWithFormat:@"%u B", (unsigned int)size];
	} 
	else if( size < 1048576ULL) 
	{
		/* kbytes */
		str = [NSString stringWithFormat:@"%d KB", (long)size/1024];
	} 
	else if( size < 1073741824ULL ) 
	{
		/* megabytes */
		str = [NSString stringWithFormat:@"%.1f MB", size / 1048576.0];
	} 
	else 
	{
		/* gigabytes */
		str = [NSString stringWithFormat:@"%.1f GB", size / 1073741824.0];
	}
	return str;
}

+ (NSString *) cutSuffix: (NSString *)filename
{
	NSMutableArray	*components = (NSMutableArray *)[filename componentsSeparatedByString:@"."];

	//no suffix
	if ([components count] == 1)
		return filename;
	
	//*suffix* is too long
	if ((unsigned int)[(NSString *)[components objectAtIndex: [components count]-1] length] > 10)
		return filename;
	
	//remove suffix
	[components removeObjectAtIndex: [components count]-1];
	
	return ([components componentsJoinedByString: @"."]);
}


/*

 NSColor: Instantiate from Web-like Hex RRGGBB string
 Original Source: <http://cocoa.karelia.com/Foundation_Categories/NSColor__Instantiat.m>

*/

+ (NSColor *) colorFromHex:(NSString *) inColorString
{
	NSString *charStr		= [inColorString substringFromIndex: 1];
	NSColor *result			= NULL;
	unsigned int colorCode	= 0;
	unsigned char redByte	= 0;
	unsigned char greenByte = 0;
	unsigned char blueByte	= 0;
	
	if (charStr != NULL)
	{
		NSScanner *scanner = [NSScanner scannerWithString: charStr];
		(void) [scanner scanHexInt: &colorCode];	// ignore error
	}
	
	redByte		= (unsigned char) (colorCode >> 16);
	greenByte	= (unsigned char) (colorCode >> 8);
	blueByte	= (unsigned char) (colorCode);	// masks off high bits
	
	result = [NSColor
			  colorWithCalibratedRed:		(float)redByte	/ 0xff
			  green:						(float)greenByte/ 0xff
			  blue:							(float)blueByte	/ 0xff
			  alpha:						1.0];
	
	return result;
}


+ (NSString *)hexFromColor: (NSColor *)color
{
	float		redFloatValue, greenFloatValue, blueFloatValue;
	int			redIntValue, greenIntValue, blueIntValue;
	NSString	*redHexValue, *greenHexValue, *blueHexValue;
	
	//Convert the NSColor to the RGB color space before we can access its components
	NSColor *convertedColor = [color colorUsingColorSpaceName: NSCalibratedRGBColorSpace];
	
	if(convertedColor)
	{
		// Get the red, green, and blue components of the color
		[convertedColor getRed: &redFloatValue green: &greenFloatValue blue: &blueFloatValue alpha:NULL];
		
		// Convert the components to numbers (unsigned decimal integer) between 0 and 255
		redIntValue = redFloatValue * 255.99999f;
		greenIntValue = greenFloatValue* 255.99999f;
		blueIntValue = blueFloatValue * 255.99999f;
		
		// Convert the numbers to hex strings
		redHexValue = [NSString stringWithFormat:@"%02x", redIntValue];
		greenHexValue = [NSString stringWithFormat:@"%02x", greenIntValue];
		blueHexValue = [NSString stringWithFormat:@"%02x", blueIntValue];
		
		// Concatenate the red, green, and blue components' hex strings together with a "#"
		return [NSString stringWithFormat:@"#%@%@%@", redHexValue, greenHexValue, blueHexValue];
	}
	return NULL;
}

+ (NSArray *)imageFileSuffixes
{
	return [NSArray arrayWithObjects: 
			@"icns",
			@"pdf",
			@"jpg",
			@"png",
			@"jpeg",
			@"gif",
			@"tif",
			@"bmp",
			@"pcx",
			@"raw",
			@"pct",
			@"rsr",
			@"pxr",
			@"sct",
			@"tga",
			@"ICNS",
			@"PDF",
			@"JPG",
			@"PNG",
			@"JPEG",
			@"GIF",
			@"TIF",
			@"BMP",
			@"PCX",
			@"RAW",
			@"PCT",
			@"RSR",
			@"PXR",
			@"SCT",
			@"TGA", 
			NULL ];
}

@end
