"""Helper function and aspect to collect first-party packages.

These are used in node rules to link the node_modules before launching a program.
This supports path re-mapping, to support short module names.
See pathMapping doc: https://github.com/Microsoft/TypeScript/issues/5039

This reads the module_root and module_name attributes from rules in
the transitive closure, rolling these up to provide a mapping to the
linker, which uses the mappings to link a node_modules directory for
runtimes to locate all the first-party packages.
"""

load("//internal/providers:npm_package_info.bzl", "NpmPackageInfo")

def _debug(vars, *args):
    if "VERBOSE_LOGS" in vars.keys():
        print("[link_node_modules.bzl]", *args)

# Arbitrary name; must be chosen to globally avoid conflicts with any other aspect
_ASPECT_RESULT_NAME = "link_node_modules__aspect_result"

# Traverse 'srcs' in addition so that we can go across a genrule
_MODULE_MAPPINGS_DEPS_NAMES = ["data", "deps", "srcs"]

def add_arg(args, arg):
    """Add an argument

    Args:
        args: either a list or a ctx.actions.Args object
        arg: string arg to append on the end
    """
    if (type(args) == type([])):
        args.append(arg)
    else:
        args.add(arg)

def _link_mapping(label, mappings, k, v):
    if k in mappings and mappings[k] != v:
        if v[1] == mappings[k][1]:
            # Allow "src" and "bin" to win over "runfiles"
            # For example,
            #    mappings[k] = ["runfiles", "angular/packages/compiler"]
            #    v           = ["bin", "angular/packages/compiler"]
            if mappings[k][0] == "runfiles":
                return True
            elif v[0] == "runfiles":
                return False
        fail(("conflicting mapping at %s: %s maps to both %s and %s" % (label, k, mappings[k], v)), "deps")
    else:
        return True

def write_node_modules_manifest(ctx, extra_data = []):
    """Writes a manifest file read by the linker, containing info about resolving runtime dependencies

    Args:
        ctx: starlark rule execution context
        extra_data: labels to search for npm packages that need to be linked (ctx.attr.deps and ctx.attr.data will always be searched)
    """

    mappings = {
        # We always map the workspace to itself to support absolute require like
        # import from 'my_wksp/path/to/file'
        # and it's always a bin dependency, TODO: let user control via package.json
        ctx.workspace_name: ["bin", ctx.workspace_name],
    }
    node_modules_root = ""

    # Look through data/deps attributes to find...
    for dep in extra_data + getattr(ctx.attr, "data", []) + getattr(ctx.attr, "deps", []):
        # ...the root directory for the third-party node_modules; we'll symlink the local "node_modules" to it
        if NpmPackageInfo in dep:
            possible_root = "/".join([dep[NpmPackageInfo].workspace, "node_modules"])
            if not node_modules_root:
                node_modules_root = possible_root
            elif node_modules_root != possible_root:
                fail("All npm dependencies need to come from a single workspace. Found '%s' and '%s'." % (node_modules_root, possible_root))

        # ...first-party packages to be linked into the node_modules tree
        for k, v in getattr(dep, _ASPECT_RESULT_NAME, {}).items():
            if _link_mapping(dep.label, mappings, k, v):
                _debug(ctx.var, "Linking %s: %s" % (k, v))
                mappings[k] = v

    # Write the result to a file, and use the magic node option --bazel_node_modules_manifest
    # The launcher.sh will peel off this argument and pass it to the linker rather than the program.
    modules_manifest = ctx.actions.declare_file("_%s.module_mappings.json" % ctx.label.name)
    content = {
        "bin": ctx.bin_dir.path,
        "modules": mappings,
        "root": node_modules_root,
        "workspace": ctx.workspace_name,
    }
    ctx.actions.write(modules_manifest, str(content))
    return modules_manifest

def get_module_mappings(label, attrs, vars, rule_kind, srcs = [], workspace_name = None):
    """Returns the module_mappings from the given attrs.

    Collects a {module_name - module_root} hash from all transitive dependencies,
    checking for collisions. If a module has a non-empty `module_root` attribute,
    all sources underneath it are treated as if they were rooted at a folder
    `module_name`.

    Args:
      label: label
      attrs: attributes
      srcs: sources (defaults to [])
      workspace_name: workspace name (defaults to None)

    Returns:
      The module mappings
    """
    mappings = {}

    for name in _MODULE_MAPPINGS_DEPS_NAMES:
        for dep in getattr(attrs, name, []):
            for k, v in getattr(dep, _ASPECT_RESULT_NAME, {}).items():
                # A package which was reachable transitively via a *_binary or *_test
                # rule is assumed to be in the runfiles of that binary,
                # so we switch the linker target root.
                # In theory we ought to be able to tell that the files really are
                # propagated through the runfiles output of the target we are visiting
                # but it seems that this info is only available in Bazel Java internals.
                # TODO: revisit whether this is the correct condition after downstream testing
                if rule_kind.endswith("_binary") or rule_kind.endswith("_test"):
                    v = ["runfiles", v[1]]
                if _link_mapping(label, mappings, k, v):
                    mappings[k] = v
                    _debug(vars, "target %s propagating module mapping %s: %s" % (dep, k, v))

    if not getattr(attrs, "module_name", None) and not getattr(attrs, "module_root", None):
        # No mappings contributed here, short-circuit with the transitive ones we collected
        _debug(vars, "No module_name or module_root attr for", label)
        return mappings

    mn = getattr(attrs, "module_name", label.name)
    mr = label.package

    if workspace_name:
        mr = "%s/%s" % (workspace_name, mr)
    elif label.workspace_root:
        mr = "%s/%s" % (label.workspace_root, mr)

    # since our module mapping is currently based on attribute names,
    # allow a special one to instruct the linker that the package has no output
    # directory and is therefore meant to be used as sources.
    # TODO: This belongs in a different mechanism like a package.json field.
    if getattr(attrs, "module_from_src", False):
        mr = ["src", mr]
    else:
        mr = ["bin", mr]

    if _link_mapping(label, mappings, mn, mr):
        _debug(vars, "target %s adding module mapping %s: %s" % (label, mn, mr))
        mappings[mn] = mr

    return mappings

# When building a mapping for use at runtime, we need paths to be relative to
# the runfiles directory. This requires the workspace_name to be prefixed on
# each module root.
def _module_mappings_aspect_impl(target, ctx):
    if target.label.workspace_root:
        # We need the workspace_name for the target being visited.
        # Skylark doesn't have this - instead they have a workspace_root
        # which looks like "external/repo_name" - so grab the second path segment.
        # TODO(alexeagle): investigate a better way to get the workspace name
        workspace_name = target.label.workspace_root.split("/")[1]
    else:
        workspace_name = ctx.workspace_name

    # Use a dictionary to construct the result struct
    # so that we can reference the _ASPECT_RESULT_NAME variable
    return struct(**{
        _ASPECT_RESULT_NAME: get_module_mappings(
            target.label,
            ctx.rule.attr,
            ctx.var,
            ctx.rule.kind,
            workspace_name = workspace_name,
        ),
    })

module_mappings_aspect = aspect(
    _module_mappings_aspect_impl,
    attr_aspects = _MODULE_MAPPINGS_DEPS_NAMES,
)
