/***
*
* BNBT Beta 8.0 - A C++ BitTorrent Tracker
* Copyright (C) 2003-2004 Trevor Hogan
*
* This library is free software; you can redistribute it and/or
* modify it under the terms of the GNU Lesser General Public
* License as published by the Free Software Foundation; either
* version 2.1 of the License, or (at your option) any later version.
*
* This library is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this library; if not, write to the Free Software
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*
***/

#include "bnbt.h"
#include "bnbt_mysql.h"
#include "atom.h"
#include "bencode.h"
#include "tracker.h"
#include "util.h"

void CTracker :: serverResponseAnnounce( struct request_t *pRequest, struct response_t *pResponse )
{
	pResponse->strCode = "200 OK";

	pResponse->mapHeaders.insert( pair<string, string>( "Content-Type", "text/plain" ) );

	// retrieve info hash

	string strInfoHash = pRequest->mapParams["info_hash"];

	if( strInfoHash.empty( ) )
	{
		pResponse->strContent = UTIL_FailureReason( "info hash missing" );
		pResponse->bCompressOK = false;

		return;
	}

	if( m_pAllowed )
	{
		if( !m_pAllowed->getItem( strInfoHash ) )
		{
			pResponse->strContent = UTIL_FailureReason( "requested download is not authorized for use with this tracker" );
			pResponse->bCompressOK = false;

			return;
		}
	}

	// retrieve ip

	string strIP = inet_ntoa( pRequest->sin.sin_addr );
	string strTempIP = pRequest->mapParams["ip"];

	if( !strTempIP.empty( ) && strTempIP.find_first_not_of( "1234567890." ) == string :: npos )
		strIP = strTempIP;

	// retrieve a ton of parameters

	string strEvent = pRequest->mapParams["event"];
	string strPort = pRequest->mapParams["port"];
	string strUploaded = pRequest->mapParams["uploaded"];
	string strDownloaded = pRequest->mapParams["downloaded"];
	string strLeft = pRequest->mapParams["left"];
	string strPeerID = pRequest->mapParams["peer_id"];
	string strNumWant = pRequest->mapParams["num_want"];
	string strNoPeerID = pRequest->mapParams["no_peer_id"];
	string strCompact = pRequest->mapParams["compact"];

	if( !strEvent.empty( ) && strEvent != "started" && strEvent != "completed" && strEvent != "stopped" )
	{
		pResponse->strContent = UTIL_FailureReason( "invalid event" );
		pResponse->bCompressOK = false;

		return;
	}

	if( strPort.empty( ) )
	{
		pResponse->strContent = UTIL_FailureReason( "port missing" );
		pResponse->bCompressOK = false;

		return;
	}

	if( strUploaded.empty( ) )
	{
		pResponse->strContent = UTIL_FailureReason( "uploaded missing" );
		pResponse->bCompressOK = false;

		return;
	}

	if( strDownloaded.empty( ) )
	{
		pResponse->strContent = UTIL_FailureReason( "downloaded missing" );
		pResponse->bCompressOK = false;

		return;
	}

	if( strLeft.empty( ) )
	{
		pResponse->strContent = UTIL_FailureReason( "left missing" );
		pResponse->bCompressOK = false;

		return;
	}

	if( strPeerID.size( ) != 20 )
	{
		pResponse->strContent = UTIL_FailureReason( "peer id not of length 20" );
		pResponse->bCompressOK = false;

		return;
	}

	struct announce_t ann;

	ann.strInfoHash = strInfoHash;
	ann.strIP = strIP;
	ann.strEvent = strEvent;
	ann.iPort = (unsigned int)atoi( strPort.c_str( ) );
	ann.iUploaded = UTIL_StringTo64( strUploaded.c_str( ) );
	ann.iDownloaded = UTIL_StringTo64( strDownloaded.c_str( ) );
	ann.iLeft = UTIL_StringTo64( strLeft.c_str( ) );
	ann.strPeerID = strPeerID;

	Announce( ann );

	unsigned int iRSize = m_iResponseSize;

	if( !strNumWant.empty( ) )
		iRSize = atoi( strNumWant.c_str( ) );

	if( iRSize > (unsigned int)m_iMaxGive )
		iRSize = (unsigned int)m_iMaxGive;

	// populate cache

	if( !m_pCached->getItem( ann.strInfoHash ) )
		m_pCached->setItem( ann.strInfoHash, new CAtomList( ) );

	CAtom *pCache = m_pCached->getItem( ann.strInfoHash );

	if( pCache && dynamic_cast<CAtomList *>( pCache ) )
	{
		CAtomList *pCacheList = dynamic_cast<CAtomList *>( pCache );

		if( pCacheList->getValuePtr( )->size( ) < iRSize )
		{
#ifdef BNBT_MYSQL
			if( m_bMySQLOverrideDState )
			{
				CMySQLQuery *pQuery = new CMySQLQuery( "SELECT bid,bip,bport FROM dstate WHERE bhash=\'" + UTIL_StringToMySQL( ann.strInfoHash ) + "\'" );

				vector<string> vecQuery;

				while( ( vecQuery = pQuery->nextRow( ) ).size( ) == 3 )
				{
					CAtomDicti *pAdd = new CAtomDicti( );

					pAdd->setItem( "peer id", new CAtomString( vecQuery[0] ) );
					pAdd->setItem( "ip", new CAtomString( vecQuery[1] ) );
					pAdd->setItem( "port", new CAtomLong( atoi( vecQuery[2].c_str( ) ) ) );

					pCacheList->addItem( pAdd );
				}

				delete pQuery;
			}
			else
			{
#endif
				if( m_pDFile )
				{
					if( !m_pDFile->getItem( ann.strInfoHash ) )
						m_pDFile->setItem( ann.strInfoHash, new CAtomDicti( ) );

					CAtom *pPeers = m_pDFile->getItem( ann.strInfoHash );

					if( pPeers && pPeers->isDicti( ) )
					{
						map<string, CAtom *> *pmapPeersDicti = ( (CAtomDicti *)pPeers )->getValuePtr( );

						for( map<string, CAtom *> :: iterator i = pmapPeersDicti->begin( ); i != pmapPeersDicti->end( ); i++ )
						{
							if( (*i).second->isDicti( ) )
							{
								CAtomDicti *pAdd = new CAtomDicti( );

								CAtom *pIP = ( (CAtomDicti *)(*i).second )->getItem( "ip" );
								CAtom *pPort = ( (CAtomDicti *)(*i).second )->getItem( "port" );

								pAdd->setItem( "peer id", new CAtomString( (*i).first ) );

								if( pIP )
									pAdd->setItem( "ip", new CAtomString( pIP->toString( ) ) );

								if( pPort && dynamic_cast<CAtomLong *>( pPort ) )
									pAdd->setItem( "port", new CAtomLong( *dynamic_cast<CAtomLong *>( pPort ) ) );

								pCacheList->addItem( pAdd );
							}
						}
					}
				}

#ifdef BNBT_MYSQL
			}
#endif

			pCacheList->Randomize( );
		}
	}

	// clamp response

	if( pCache && dynamic_cast<CAtomList *>( pCache ) )
	{
		CAtomList *pCacheList = dynamic_cast<CAtomList *>( pCache );

		CAtom *pPeers = NULL;

		if( strCompact == "1" )
		{
			// compact announce

			string strPeers;

			vector<CAtom *> *pvecList = pCacheList->getValuePtr( );

			for( vector<CAtom *> :: iterator i = pvecList->begin( ); i != pvecList->end( ); )
			{
				if( (*i)->isDicti( ) )
				{
					if( strPeers.size( ) / 6 >= iRSize )
						break;

					CAtom *pIP = ( (CAtomDicti *)(*i) )->getItem( "ip" );
					CAtom *pPort = ( (CAtomDicti *)(*i) )->getItem( "port" );

					if( pIP && pPort && dynamic_cast<CAtomLong *>( pPort ) )
						strPeers += UTIL_Compact( pIP->toString( ), (unsigned int)dynamic_cast<CAtomLong *>( pPort )->getValue( ) );

					delete *i;

					i = pvecList->erase( i );
				}
				else
					i++;
			}

			pPeers = new CAtomString( strPeers );

			// don't compress

			pResponse->bCompressOK = false;
		}
		else
		{
			// regular announce

			CAtomList *pPeersList = new CAtomList( );

			vector<CAtom *> *pvecList = pCacheList->getValuePtr( );

			for( vector<CAtom *> :: iterator i = pvecList->begin( ); i != pvecList->end( ); )
			{
				if( (*i)->isDicti( ) )
				{
					if( pPeersList->getValuePtr( )->size( ) >= iRSize )
						break;

					if( strNoPeerID == "1" )
						( (CAtomDicti *)(*i) )->delItem( "peer id" );

					pPeersList->addItem( new CAtomDicti( *(CAtomDicti *)(*i) ) );

					delete *i;

					i = pvecList->erase( i );
				}
				else
					i++;
			}

			pPeers = pPeersList;
		}

		CAtomDicti *pData = new CAtomDicti( );

		pData->setItem( "interval", new CAtomLong( m_iAnnounceInterval ) );
		pData->setItem( "peers", pPeers );

		pResponse->strContent = Encode( pData );

		delete pData;
	}
}
