note
	description: "Structure describing a registry key value"
	legal: "See notice at end of class."
	status: "See notice at end of class.";
	date: "$Date: 2017-05-01 15:51:19 -0700 (Mon, 01 May 2017) $";
	revision: "$Revision: 100293 $"

class
	WEL_REGISTRY_KEY_VALUE

inherit
	WEL_REGISTRY_KEY_VALUE_TYPE

	PLATFORM
		export
			{NONE} all
		end

create
	make,
	make_with_data,
	make_with_value,
	make_with_dword_value

feature -- Initialization

	make (t: like type; v: READABLE_STRING_GENERAL)
			-- Create a new string key value.
		require
			v_not_void: v /= Void
			t_valid: t = reg_sz
		do
			set_string_value (v)
		ensure
			type_set: type = reg_sz
			string_value_set: string_value.same_string_general (v)
		end

	make_with_data (t: like type; v: MANAGED_POINTER)
			-- Create a new key value using data `v'.
		require
			v_not_void: v /= Void
		do
			set_type (t)
			data := v
		ensure
			type_set: type = t
			data_set: data = v
		end

	make_with_value (t: like type; v: like internal_value)
			-- Create a new key value of type `t' using `v'.
		obsolete
			"Use `make_with_data', `make' or `make_with_dword_value' instead [2017-05-31]."
		require
			v_not_void: v /= Void
		do
			set_type (t)
			create data.make (v.capacity)
			data.item.memory_copy (v.item, v.capacity)
		ensure
			type_set: type = t
			internal_value_set: internal_value ~ v
		end

	make_with_dword_value (v: like dword_value)
			-- Set `dword_value' with `v'.
			-- Set `type' with `reg_dword'.
		do
			set_dword_value (v)
		ensure
			type_set: type = reg_dword
			dword_value_set: dword_value = v
		end

feature -- Access

	type: INTEGER
			-- Type of value
			-- See class WEL_REGISTRY_KEY_VALUE_TYPE for possible
			-- values.

	data: MANAGED_POINTER
			-- Hold data for registry key value

	internal_value: WEL_STRING
			-- Storage for Current.
		obsolete
			"Use `string_value' or `dword_value' instead [2017-05-31]."
		do
			create Result.make_empty (data.count - 2)
			Result.item.memory_copy (data.item, data.count)
		end

	value: STRING_32
			-- String data.
		obsolete
			"Use `string_value' instead [2017-05-31]."
		require
			valid_type: type = Reg_sz
		do
			Result := string_value
		ensure
			string_value_not_void: Result /= Void
		end

	string_value: STRING_32
			-- String data.
		require
			valid_type: type = Reg_sz
		local
			l_count: INTEGER
			l_str: WEL_STRING
		do
			l_count := data.count
			if l_count > 0 and then (l_count \\ {WEL_STRING}.character_size = 0) then
				create l_str.share_from_pointer_and_count (data.item, l_count)
				Result := l_str.substring (1, l_count // l_str.character_size)
				check
					not_empty: not Result.is_empty
				end
				if Result.item (Result.count) = '%U' then
						-- Case where string was stored with its final null character,
						-- so we don't want it in our actual string.
					Result.remove (Result.count)
				end
			else
				create Result.make_empty
			end
		ensure
			string_value_not_void: Result /= Void
		end

	dword_value: INTEGER
			-- Data converted as integer.
		require
			valid_type: type = Reg_dword
		do
			($Result).memory_copy (data.item, Integer_32_bytes)
		end

feature -- Element Change

	set_type (t: INTEGER)
			-- Set type of value with `t'.
			-- See class WEL_REGISTRY_KEY_VALUE_TYPE for possible
			-- values for `t'.
		do
			type := t
		ensure
			type_set: type = t
		end

	set_value (v: like value)
			-- Set `value' with `v'.
		obsolete
			"Use `set_string_value' instead' [2017-05-31]."
		require
			v_not_void: v /= Void
		local
			l_str: WEL_STRING
		do
			create l_str.make (v)
			create data.make (l_str.capacity)
			data.item.memory_copy (l_str.item, l_str.capacity)
		ensure
			value_set: value ~ v
		end

	set_dword_value (v: like dword_value)
			-- Set `dword_value' with `v'.
			-- Set `type' with `reg_dword'.
		do
			type := reg_dword
			create data.make (integer_32_bytes)
			data.put_integer_32 (v, 0)
		ensure
			type_set: type = reg_dword
			dword_value_set: dword_value = v
		end

	set_string_value (v: READABLE_STRING_GENERAL)
			-- Set `string_value' with `v'.
			-- Set `type' with `reg_sz'.
		require
			v_not_void: v /= Void
		local
			l_str: WEL_STRING
		do
			type := reg_sz
			create l_str.make (v)
			create data.make (l_str.capacity)
			data.item.memory_copy (l_str.item, l_str.capacity)
		ensure
			type_set: type = reg_sz
			string_value_set: string_value.same_string_general (v)
		end

note
	copyright:	"Copyright (c) 1984-2017, Eiffel Software and others"
	license:	"Eiffel Forum License v2 (see http://www.eiffel.com/licensing/forum.txt)"
	source: "[
			Eiffel Software
			5949 Hollister Ave., Goleta, CA 93117 USA
			Telephone 805-685-1006, Fax 805-685-6869
			Website http://www.eiffel.com
			Customer support http://support.eiffel.com
		]"

end
