#define GE_USE_THREADS

/*
	description:

		"C declarations for the Gobo Eiffel runtime."

	system: "Gobo Eiffel Compiler"
	copyright: "Copyright (c) 2005-2018, Eric Bezault and others"
	license: "MIT License"
	date: "$Date: 2019-02-05 23:54:36 +0000 (Tue, 05 Feb 2019) $"
	revision: "$Revision: 102790 $"
*/

#ifndef GE_EIFFEL_H
#define GE_EIFFEL_H
#if defined(_MSC_VER) && (_MSC_VER >= 1020)
#pragma once
#endif

/* Class name mapping as defined in the FreeELKS library. */
#ifndef EIF_INTEGER
#define EIF_INTEGER EIF_INTEGER_32
#endif
#ifndef EIF_CHARACTER
#define EIF_CHARACTER EIF_CHARACTER_8
#endif
#ifndef EIF_REAL
#define EIF_REAL EIF_REAL_32
#endif
#ifndef EIF_DOUBLE
#define EIF_DOUBLE EIF_REAL_64
#endif
#ifndef GE_ms
#define GE_ms(s,c) GE_ms8((s),(c))
#endif

#if defined(__USE_POSIX) || defined(__unix__) || defined(_POSIX_C_SOURCE)
#include <unistd.h>
#endif
#if !defined(WIN32) && \
	(defined(WINVER) || defined(_WIN32_WINNT) || defined(_WIN32) || \
	defined(__WIN32__) || defined(__TOS_WIN__) || defined(_MSC_VER) || \
	defined(__MINGW32__))
#define WIN32 1
#endif
#ifdef WIN32
#define EIF_WINDOWS 1
#include <windows.h>
#endif

#include <stdlib.h>
#include <stdio.h>
#include <stdarg.h>
#include <string.h>
#include <stddef.h>

/* Platform definition */
/* Unix definition */
#define EIF_IS_UNIX EIF_TRUE
/* Windows definition */
#ifdef EIF_WINDOWS
#define EIF_IS_WINDOWS EIF_TRUE
#undef EIF_IS_UNIX
#define EIF_IS_UNIX EIF_FALSE
#else
#define EIF_IS_WINDOWS EIF_FALSE
#endif
/* VMS definition */
#ifdef EIF_VMS
#define EIF_IS_VMS EIF_TRUE
#undef EIF_IS_UNIX
#define EIF_IS_UNIX EIF_FALSE
#else
#define EIF_IS_VMS EIF_FALSE
#endif
/* MAC definition */
#ifdef EIF_MAC
#define EIF_IS_MAC EIF_TRUE
#undef EIF_IS_UNIX
#define EIF_IS_UNIX EIF_FALSE
#else
#define EIF_IS_MAC EIF_FALSE
#endif
/* VxWorks definition */
#ifdef EIF_VXWORKS
#define EIF_IS_VXWORKS EIF_TRUE
#undef EIF_IS_UNIX
#define EIF_IS_UNIX EIF_FALSE
#else
#define EIF_IS_VXWORKS EIF_FALSE
#endif

#ifdef __cplusplus
extern "C" {
#endif

#ifdef _MSC_VER /* MSVC */
typedef signed char int8_t;
typedef signed short int16_t;
typedef signed int int32_t;
typedef signed __int64 int64_t;
typedef unsigned char uint8_t;
typedef unsigned short uint16_t;
typedef unsigned int uint32_t;
typedef unsigned __int64 uint64_t;
#else
#if defined (__BORLANDC__) && (__BORLANDC__ < 0x600) /* Borland before 6.0 */
typedef signed char int8_t;
typedef signed short int16_t;
typedef signed long int int32_t;
typedef signed __int64 int64_t;
typedef unsigned char uint8_t;
typedef unsigned short uint16_t;
typedef unsigned long int uint32_t;
typedef unsigned __int64 uint64_t;
#else
#include <inttypes.h>
#endif
#endif

/* Portable integer pointers */
#ifdef EIF_WINDOWS
#ifndef _INTPTR_T_DEFINED
#define _INTPTR_T_DEFINED
#ifdef _WIN64
typedef __int64 intptr_t;
#else
typedef int intptr_t;
#endif
#endif
#ifndef _UINTPTR_T_DEFINED
#define _UINTPTR_T_DEFINED
#ifdef _WIN64
  typedef unsigned __int64 uintptr_t;
#else
  typedef unsigned int uintptr_t;
#endif
#endif
#endif

/* C type for underlying integer type identifying object's dynamic type. */
typedef uint16_t EIF_TYPE_INDEX;

/*
 * Abstraction representing an Eiffel type.
 * It is made of a compiler type-id,
 * and of some annotations (attached/detachable/separate/variant/frozen).
 */
typedef struct eif_type {
	EIF_TYPE_INDEX id;
	EIF_TYPE_INDEX annotations;
} EIF_TYPE;

/*
 * Since EIF_TYPE and EIF_ENCODED_TYPE have the same size, the encoded version
 * is basically a memcpy version of the EIF_TYPE representation.
 * It is used to provide backward compatibility to most Eiffel and
 * C APIs manipulating types as an INTEGER.
 */
typedef int32_t EIF_ENCODED_TYPE;
typedef EIF_ENCODED_TYPE EIF_TYPE_ID;
#define EIF_NO_TYPE (EIF_TYPE_ID)(-1)

/* Basic Eiffel types */
typedef char EIF_BOOLEAN;
typedef unsigned char EIF_CHARACTER_8;
typedef uint32_t EIF_CHARACTER_32;
typedef int8_t EIF_INTEGER_8;
typedef int16_t EIF_INTEGER_16;
typedef int32_t EIF_INTEGER_32;
typedef int64_t EIF_INTEGER_64;
typedef uint8_t EIF_NATURAL_8;
typedef uint16_t EIF_NATURAL_16;
typedef uint32_t EIF_NATURAL_32;
typedef uint64_t EIF_NATURAL_64;
typedef void* EIF_POINTER;
typedef float EIF_REAL_32;
typedef double EIF_REAL_64;
typedef struct {EIF_TYPE_INDEX id; uint16_t flags;} EIF_ANY;
typedef EIF_ANY* EIF_REFERENCE;
typedef struct {EIF_TYPE_INDEX id; uint16_t flags; EIF_REFERENCE area; EIF_INTEGER count;} EIF_STRING;
typedef struct {EIF_TYPE_INDEX id; uint16_t flags; uint32_t offset; EIF_INTEGER count; EIF_INTEGER capacity;} EIF_SPECIAL;

#ifdef EIF_WINDOWS
typedef wchar_t EIF_NATIVE_CHAR;
#else
typedef char EIF_NATIVE_CHAR;
#endif
typedef EIF_NATIVE_CHAR* EIF_FILENAME;

#define EIF_VOID ((EIF_REFERENCE)0)
#define EIF_FALSE ((EIF_BOOLEAN)'\0')
#define EIF_TRUE ((EIF_BOOLEAN)'\1')
#define EIF_TEST(x) ((x) ? EIF_TRUE : EIF_FALSE)

#define EIF_IS_WORKBENCH EIF_FALSE
#define EIF_POINTER_DISPLAY "lX"

/* For INTEGER and NATURAL manifest constants */
#define GE_int8(x) x
#define GE_nat8(x) x
#define GE_int16(x) x
#define GE_nat16(x) x
#define GE_int32(x) x##L
#define GE_nat32(x) x##U
#if defined (_MSC_VER) && (_MSC_VER < 1400) /* MSC older than v8 */
#define GE_int64(x) x##i64
#define GE_nat64(x) x##ui64
#elif defined(__BORLANDC__) && (__BORLANDC__ < 0x600) /* Borland before 6.0 */
#define GE_int64(x) x##i64
#define GE_nat64(x) x##ui64
#else /* ISO C 99 */
#define GE_int64(x) x##LL
#define GE_nat64(x) x##ULL
#endif
#ifdef __LCC__
/* lcc-win32 reports a constant overflow for -21474836478. */
#define GE_min_int32 (-GE_int32(2147483647)-GE_int32(1))
#else
#define GE_min_int32 GE_int32(-2147483648)
#endif
#if defined(__LCC__) || defined(__GNUC__) || defined(__MINGW32__)
/* lcc-win32 reports a constant overflow for -9223372036854775808. */
/* gcc and mingw-win64 warn that integer constant is so large that it is unsigned. */
#define GE_min_int64 (-GE_int64(9223372036854775807)-GE_int64(1))
#else
#define GE_min_int64 GE_int64(-9223372036854775808)
#endif
#if defined(__LCC__) && !defined(_WIN64)
/* lcc-win32 does not consider 64 bit constants as constants in case statement. */
#define GE_case_int64(x) ((int32_t)(x))
#define GE_case_nat64(x) ((uint32_t)(x))
#else
#define GE_case_int64(x) (x)
#define GE_case_nat64(x) (x)
#endif

#ifdef _WIN64
#define GE_IS_64_BITS EIF_TRUE
#else
#define GE_IS_64_BITS EIF_TEST(sizeof(void*)==64)
#endif

/* Posix threads */
#if !defined(EIF_WINDOWS)
#define GE_USE_POSIX_THREADS
#endif

#ifdef _MSC_VER /* MSVC */
/* MSVC does not support ISO C 99's 'snprintf' from stdio.h */
#define snprintf(a,b,c,d) sprintf(a,c,d)
#endif

/*
 * Gobo compiler version.
 * Starts with 6080 (looks like GOBO) followed by 5 digits.
 */
#define GE_compiler_version() 608000001

/*
	Interoperability with ISE.
*/
#define RTI64C(x) GE_int64(x)
#define EIF_OBJECT EIF_REFERENCE
#define EIF_OBJ EIF_OBJECT
/* Function pointer call to make sure all arguments are correctly pushed onto stack. */
/* FUNCTION_CAST is for standard C calls. */
/* FUNCTION_CAST_TYPE is for non-standard C calls. */
#define FUNCTION_CAST(r_type,arg_types) (r_type (*) arg_types)
#define FUNCTION_CAST_TYPE(r_type,call_type,arg_types) (r_type (call_type *) arg_types)
#define SIGBLOCK
#define SIGRESUME
#define rt_public				/* default C scope */
#define rt_private static		/* static outside a block means private */
#define rt_shared				/* data shared between modules, but not public */
typedef intptr_t rt_int_ptr;
typedef uintptr_t rt_uint_ptr;
#define RTMS(s) GE_str8(s)
#define RTMS_EX(s,c) GE_ms8((s),(c))

#ifdef __cplusplus
}
#endif

#endif

/*
	description:

		"C types used to implement class THREAD and related threading facilities"

	system: "Gobo Eiffel Compiler"
	copyright: "Copyright (c) 2017, Eric Bezault and others"
	license: "MIT License"
	date: "$Date: 2019-02-05 23:54:36 +0000 (Tue, 05 Feb 2019) $"
	revision: "$Revision: 102790 $"
*/

#ifndef GE_THREAD_TYPES_H
#define GE_THREAD_TYPES_H
#if defined(_MSC_VER) && (_MSC_VER >= 1020)
#pragma once
#endif

#ifdef GE_USE_THREADS

#define EIF_THREADS
#ifdef GE_USE_POSIX_THREADS
#define EIF_POSIX_THREADS
#endif

#ifndef GE_EIFFEL_H
#include "ge_eiffel.h"
#endif

#ifdef GE_USE_POSIX_THREADS
#include <pthread.h>
#include <semaphore.h>
#elif defined EIF_WINDOWS
#include <windows.h>
#include <process.h>
#endif

#ifdef __cplusplus
extern "C" {
#endif

#ifdef GE_USE_POSIX_THREADS

#define EIF_THR_TYPE            pthread_t
#define EIF_CS_TYPE				pthread_mutex_t
#define EIF_MUTEX_TYPE          pthread_mutex_t
#define EIF_COND_TYPE			pthread_cond_t
#define EIF_SEM_TYPE			sem_t
#define EIF_RWL_TYPE			pthread_rwlock_t

#elif defined EIF_WINDOWS

#define EIF_THR_TYPE            HANDLE
#define EIF_CS_TYPE				CRITICAL_SECTION
#define EIF_MUTEX_TYPE          CRITICAL_SECTION
#define EIF_SEM_TYPE            HANDLE

typedef struct {
		/* Semaphore used to queue up threads waiting for the condition to become signaled. */
	EIF_SEM_TYPE* semaphore;
		/* Serialize access to fields of Current. */
	EIF_CS_TYPE* csection;
		/* Number of waiters. */
	unsigned long num_waiting;
		/* Number of already awoken. */
	unsigned long num_wake;
		/* Number of time we signaled/broadcasted for improving fairness.
		 * This ensures one thread won't steal wakeups from other threads in queue. */
	unsigned long generation;
} EIF_COND_TYPE;

typedef struct {
	EIF_MUTEX_TYPE* m; /* Internal monitor lock. */
	int rwlock; /* >0 = # readers, <0 = writer, 0 = none */
	EIF_COND_TYPE* readers_ok; /* Start waiting readers. */
	unsigned int waiting_writers; /* Number of waiting writers. */
	EIF_COND_TYPE* writers_ok; /* Start a waiting writer. */
} EIF_RWL_TYPE;

#endif

typedef struct {
	unsigned int priority;
	unsigned int stack_size;
} EIF_THR_ATTR_TYPE;

/* Struct for thread context. */
typedef struct GE_thread_context_struct GE_thread_context;
struct GE_thread_context_struct {
	EIF_THR_TYPE thread_id; /* Thread identifier for associated thread. */
	EIF_REFERENCE current; /* Eiffel root object. */
	void (*routine)(EIF_REFERENCE); /* Eiffel routine. */
	void (*set_terminated)(EIF_REFERENCE,EIF_BOOLEAN); /* Eiffel routine to set {THREAD}.terminated. */
	unsigned int initial_priority; /* Initial priority. */
	EIF_THR_TYPE last_thread_id; /* Last thread created from current thread. */
	volatile int n_children; /* Number of direct thread children. */
	EIF_MUTEX_TYPE* children_mutex; /* Mutex to wait for thread children. */
	EIF_COND_TYPE* children_cond; /* Condition variable to wait for thread children. */
	GE_thread_context* parent_context;	/* Context of parent thread, NULL if main thread. */
	int thread_exiting; /* Has current thread already called GE_thread_exit? */
	volatile int is_alive; /* Is Current thread still alive? */
};

#ifdef __cplusplus
}
#endif

#endif
#endif

/*
	description:

		"C functions used to implement once features"

	system: "Gobo Eiffel Compiler"
	copyright: "Copyright (c) 2017, Eric Bezault and others"
	license: "MIT License"
	date: "$Date: 2019-02-05 23:54:36 +0000 (Tue, 05 Feb 2019) $"
	revision: "$Revision: 102790 $"
*/

#ifndef GE_ONCE_H
#define GE_ONCE_H
#if defined(_MSC_VER) && (_MSC_VER >= 1020)
#pragma once
#endif

#ifndef GE_EIFFEL_H
#include "ge_eiffel.h"
#endif

#ifdef __cplusplus
extern "C" {
#endif

/*
 * Struct to keep track of the call status
 * and results of once features.
 */
typedef struct {
	EIF_BOOLEAN* boolean_value;
	EIF_REFERENCE* boolean_exception;
	unsigned char* boolean_status;
	unsigned int boolean_count;
	EIF_CHARACTER_8* character_8_value;
	EIF_REFERENCE* character_8_exception;
	unsigned char* character_8_status;
	unsigned int character_8_count;
	EIF_CHARACTER_32* character_32_value;
	EIF_REFERENCE* character_32_exception;
	unsigned char* character_32_status;
	unsigned int character_32_count;
	EIF_INTEGER_8* integer_8_value;
	EIF_REFERENCE* integer_8_exception;
	unsigned char* integer_8_status;
	unsigned int integer_8_count;
	EIF_INTEGER_16* integer_16_value;
	EIF_REFERENCE* integer_16_exception;
	unsigned char* integer_16_status;
	unsigned int integer_16_count;
	EIF_INTEGER_32* integer_32_value;
	EIF_REFERENCE* integer_32_exception;
	unsigned char* integer_32_status;
	unsigned int integer_32_count;
	EIF_INTEGER_64* integer_64_value;
	EIF_REFERENCE* integer_64_exception;
	unsigned char* integer_64_status;
	unsigned int integer_64_count;
	EIF_NATURAL_8* natural_8_value;
	EIF_REFERENCE* natural_8_exception;
	unsigned char* natural_8_status;
	unsigned int natural_8_count;
	EIF_NATURAL_16* natural_16_value;
	EIF_REFERENCE* natural_16_exception;
	unsigned char* natural_16_status;
	unsigned int natural_16_count;
	EIF_NATURAL_32* natural_32_value;
	EIF_REFERENCE* natural_32_exception;
	unsigned char* natural_32_status;
	unsigned int natural_32_count;
	EIF_NATURAL_64* natural_64_value;
	EIF_REFERENCE* natural_64_exception;
	unsigned char* natural_64_status;
	unsigned int natural_64_count;
	EIF_POINTER* pointer_value;
	EIF_REFERENCE* pointer_exception;
	unsigned char* pointer_status;
	unsigned int pointer_count;
	EIF_REAL_32* real_32_value;
	EIF_REFERENCE* real_32_exception;
	unsigned char* real_32_status;
	unsigned int real_32_count;
	EIF_REAL_64* real_64_value;
	EIF_REFERENCE* real_64_exception;
	unsigned char* real_64_status;
	unsigned int real_64_count;
	EIF_REFERENCE* reference_value;
	EIF_REFERENCE* reference_exception;
	unsigned char* reference_status;
	unsigned int reference_count;
	EIF_REFERENCE* procedure_exception;
	unsigned char* procedure_status;
	unsigned int procedure_count;
} GE_onces;

/*
 * Variable to keep track of the call status
 * and results of once-per-process features.
 */
extern GE_onces* GE_process_onces;

/*
 * Initialize `GE_process_onces'.
 */
extern void GE_init_onces(
	unsigned int a_boolean_count,
	unsigned int a_character_8_count,
	unsigned int a_character_32_count,
	unsigned int a_integer_8_count,
	unsigned int a_integer_16_count,
	unsigned int a_integer_32_count,
	unsigned int a_integer_64_count,
	unsigned int a_natural_8_count,
	unsigned int a_natural_16_count,
	unsigned int a_natural_32_count,
	unsigned int a_natural_64_count,
	unsigned int a_pointer_count,
	unsigned int a_real_32_count,
	unsigned int a_real_64_count,
	unsigned int a_reference_count,
	unsigned int a_procedure_count);

/*
 * Create a new 'GE_onces' struct which can deal with the
 * numbers of once features passed as argument.
 */
extern GE_onces* GE_new_onces(
	unsigned int a_boolean_count,
	unsigned int a_character_8_count,
	unsigned int a_character_32_count,
	unsigned int a_integer_8_count,
	unsigned int a_integer_16_count,
	unsigned int a_integer_32_count,
	unsigned int a_integer_64_count,
	unsigned int a_natural_8_count,
	unsigned int a_natural_16_count,
	unsigned int a_natural_32_count,
	unsigned int a_natural_64_count,
	unsigned int a_pointer_count,
	unsigned int a_real_32_count,
	unsigned int a_real_64_count,
	unsigned int a_reference_count,
	unsigned int a_procedure_count);

/*
 * Free memory allocated by `a_onces'.
 */
extern void GE_free_onces(GE_onces* a_onces);

#ifdef __cplusplus
}
#endif

#endif

/*
	description:

		"C functions used to implement class EXCEPTION"

	system: "Gobo Eiffel Compiler"
	copyright: "Copyright (c) 2007-2018, Eric Bezault and others"
	license: "MIT License"
	date: "$Date: 2019-02-05 23:54:36 +0000 (Tue, 05 Feb 2019) $"
	revision: "$Revision: 102790 $"
*/

#ifndef GE_EXCEPTION_H
#define GE_EXCEPTION_H
#if defined(_MSC_VER) && (_MSC_VER >= 1020)
#pragma once
#endif

#ifndef GE_EIFFEL_H
#include "ge_eiffel.h"
#endif
#ifndef GE_ONCE_H
#include "ge_once.h"
#endif
#ifdef GE_USE_THREADS
#ifndef GE_THREAD_TYPES_H
#include "ge_thread_types.h"
#endif
#endif

#include <setjmp.h>

/*
 * On Linux glibc systems, we need to use sig* versions of jmp_buf,
 * setjmp and longjmp to preserve the signal handling context.
 * One way to detect this is if _SIGSET_H_types has
 * been defined in /usr/include/setjmp.h.
 * NOTE: ANSI only recognizes the non-sig versions.
 */
#if (defined(_SIGSET_H_types) && !defined(__STRICT_ANSI__))
#define GE_jmp_buf sigjmp_buf
#define GE_setjmp(x) sigsetjmp((x),1)
#define GE_longjmp(x,y) siglongjmp((x),(y))
#else
#define GE_jmp_buf jmp_buf
#define GE_setjmp(x) setjmp((x))
#define GE_longjmp(x,y) longjmp((x),(y))
#endif

#ifdef __cplusplus
extern "C" {
#endif

/*
 * Predefined exception codes.
 */
#define GE_EX_VOID		1			/* Feature applied to void reference */
#define GE_EX_MEM		2			/* No more memory */
#define GE_EX_PRE		3			/* Pre-condition violated */
#define GE_EX_POST		4			/* Post-condition violated */
#define GE_EX_FLOAT		5			/* Floating point exception (signal SIGFPE) */
#define GE_EX_CINV		6			/* Class invariant violated */
#define GE_EX_CHECK		7			/* Check instruction violated */
#define GE_EX_FAIL		8			/* Routine failure */
#define GE_EX_WHEN		9			/* Unmatched inspect value */
#define GE_EX_VAR		10			/* Non-decreasing loop variant */
#define GE_EX_LINV		11			/* Loop invariant violated */
#define GE_EX_SIG		12			/* Operating system signal */
#define GE_EX_BYE		13			/* Eiffel run-time panic */
#define GE_EX_RESC		14			/* Exception in rescue clause */
#define GE_EX_OMEM		15			/* Out of memory (cannot be ignored) */
#define GE_EX_RES		16			/* Resumption failed (retry did not succeed) */
#define GE_EX_CDEF		17			/* Create on deferred */
#define GE_EX_EXT		18			/* External event */
#define GE_EX_VEXP		19			/* Void assigned to expanded */
#define GE_EX_HDLR		20			/* Exception in signal handler */
#define GE_EX_IO		21			/* I/O error */
#define GE_EX_SYS		22			/* Operating system error */
#define GE_EX_RETR		23			/* Retrieval error */
#define GE_EX_PROG		24			/* Developer exception */
#define GE_EX_FATAL		25			/* Eiffel run-time fatal error */
#define GE_EX_DOL		26			/* $ applied to melted feature */
#define GE_EX_ISE_IO	27			/* I/O error raised by the ISE Eiffel runtime */
#define GE_EX_COM		28			/* COM error raised by EiffelCOM runtime */
#define GE_EX_RT_CHECK	29			/* Runtime check error such as out-of-bound array access */
#define GE_EX_OLD		30			/* Old violation */
#define GE_EX_SEL		31			/* Serialization failure */
#define GE_EX_DIRTY		32			/* SCOOP processor dirty exception. */
#define GE_EX_NEX		32			/* Number of internal exceptions */

/*
 * String buffer used to build the exception trace.
 */
typedef struct GE_exception_trace_buffer_struct GE_exception_trace_buffer;
struct GE_exception_trace_buffer_struct {
	char* area;
	uint32_t count;
	uint32_t capacity;
};

/*
 * Information about the feature being executed.
 */
typedef struct GE_call_struct GE_call;
struct GE_call_struct {
#ifdef GE_USE_CURRENT_IN_EXCEPTION_TRACE
	void* object; /* Current object */
#endif
	const char* class_name;
	const char* feature_name;
	GE_call* caller; /* previous feature in the call chain */
};

/*
 * Context of features containing a rescue clause.
 */
typedef struct GE_rescue_struct GE_rescue;
struct GE_rescue_struct {
	GE_jmp_buf jb;
	GE_rescue* previous; /* previous context in the call chain */
};

/*
 * Information about the execution context.
 * One such struct per thread.
 */
typedef struct GE_context_struct GE_context;
struct GE_context_struct {
	GE_call* call; /* Call stack */
	uint32_t in_assertion; /* Is an assertion evaluated? */
	GE_rescue* last_rescue; /* Context of last feature entered containing a rescue clause */
	uint32_t in_rescue; /* Number of rescue clauses currently being executed */
	EIF_REFERENCE exception_manager; /* Exception manager */
	char raising_exception; /* Is an exception currently being raised? */
	char exception_trace_enabled; /* Should exception trace be displayed? */
	long exception_code; /* Code of the exception currently being raised, 0 otherwise */
	const char* exception_tag; /* Tag of the exception currently being raised, NULL otherwise */
	GE_exception_trace_buffer exception_trace_buffer; /* String buffer used to build the exception trace */
	GE_exception_trace_buffer last_exception_trace; /* Last non-routine-failure exception trace */
	int pre_ecma_mapping_status; /* Do we map old names to new name? (i.e. STRING to STRING_8, INTEGER to INTEGER_32, ...). */
#ifdef GE_USE_THREADS
	GE_thread_context* thread; /* Thread context */
	GE_onces* process_onces; /* Cache for status and results of onces-per-process */
	GE_onces* thread_onces; /* Status and results of onces-per-thread */
	void* wel_per_thread_data; /* WEL private data */
#endif
};

/*
 * Default initialization for `GE_context'.
 */
extern GE_context GE_default_context;

/*
 * Execution context of main thread.
 */
extern GE_context* GE_main_context;

/*
 * Execution context of current thread.
 */
extern GE_context* GE_current_context(void);

/*
 * Initialization of exception handling.
 */
extern void GE_init_exception(GE_context* context);

/*
 * Free memory allocated in `a_context' for exception handling.
 */
extern void GE_free_exception(GE_context* a_context);

/*
 * Pointer to function to create a new exception manager object
 * (of type ISE_EXCEPTION_MANAGER).
 */
extern EIF_REFERENCE (*GE_new_exception_manager)(EIF_BOOLEAN);

/*
 * Pointer to Eiffel routine ISE_EXCEPTION_MANAGER.init_exception_manager.
 */
extern void (*GE_init_exception_manager)(GE_context*, EIF_REFERENCE);

/*
 * Pointer to Eiffel routine ISE_EXCEPTION_MANAGER.last_exception.
 */
extern EIF_REFERENCE (*GE_last_exception)(GE_context*, EIF_REFERENCE);

/*
 * Pointer to Eiffel routine ISE_EXCEPTION_MANAGER.once_raise.
 */
extern void (*GE_once_raise)(GE_context*, EIF_REFERENCE, EIF_REFERENCE);

/*
 * Pointer to Eiffel routine ISE_EXCEPTION_MANAGER.set_exception_data.
 */
extern void (*GE_set_exception_data)(GE_context*, EIF_REFERENCE, EIF_INTEGER_32, EIF_BOOLEAN, EIF_INTEGER_32, EIF_INTEGER_32, EIF_REFERENCE, EIF_REFERENCE, EIF_REFERENCE, EIF_REFERENCE, EIF_REFERENCE, EIF_REFERENCE, EIF_INTEGER_32, EIF_BOOLEAN);

/*
 * Exception tag associated with `a_code'.
 */
extern char* GE_exception_tag(long a_code);

/*
 * Raise an exception with code `a_code'.
 */
extern void GE_raise(long a_code);

/*
 * Raise an exception with code `a_code' and message `msg'.
 */
extern void GE_raise_with_message(long a_code, const char* msg);

/*
 * Raise an exception from EXCEPTION_MANAGER.
 */
extern void GE_developer_raise(long a_code, char* a_meaning, char* a_message);

/*
 * Raise exception which was raised the first time a once routine
 * was executed when executing it again.
 */
extern void GE_raise_once_exception(GE_context* a_context, EIF_REFERENCE a_exception);

/*
 * Exception, if any, which was last raised in `a_context'.
 */
extern EIF_REFERENCE GE_last_exception_raised(GE_context* a_context);

/*
 * Jump to execute the rescue of the last routine with a rescue
 * in the call stack.
 */
extern void GE_jump_to_last_rescue(GE_context* a_context);

/*
 * Set `in_assertion' to 'not b'.
 * Return the opposite of previous value.
 */
extern EIF_BOOLEAN GE_check_assert(EIF_BOOLEAN b);

/*
 * Check whether the type id of `obj' is not in `type_ids'.
 * If it is, then raise a CAT-call exception. Don't do anything if `obj' is Void.
 * `nb' is the number of ids in `type_ids' and is expected to be >0.
 * `type_ids' is sorted in increasing order.
 * Return `obj'.
 */
#define GE_catcall(obj,type_ids,nb) GE_check_catcall((obj),(type_ids),(nb))
extern EIF_REFERENCE GE_check_catcall(EIF_REFERENCE obj, EIF_TYPE_INDEX type_ids[], int nb);

/*
 * Check whether `obj' is Void.
 * If it is, then raise a call-on-void-target exception.
 * Return `obj'.
 */
#define GE_void(obj) (!(obj)?GE_check_void(obj):(obj))
extern EIF_REFERENCE GE_check_void(EIF_REFERENCE obj);

/*
 * Check whether `ptr' is a null pointer.
 * If it is, then raise a no-more-memory exception.
 * Return `ptr'.
 */
#define GE_null(ptr) GE_check_null(ptr)
extern void* GE_check_null(void* ptr);

#ifdef EIF_WINDOWS
/*
 * Set default exception handler.
 */
extern void GE_set_windows_exception_filter(void);
#endif

#ifdef __cplusplus
}
#endif

#endif

/*
	description:

		"C functions used to implement class THREAD and related threading facilities"

	system: "Gobo Eiffel Compiler"
	copyright: "Copyright (c) 2016-2017, Eric Bezault and others"
	license: "MIT License"
	date: "$Date: 2019-02-05 23:54:36 +0000 (Tue, 05 Feb 2019) $"
	revision: "$Revision: 102790 $"
*/

#ifndef GE_THREAD_H
#define GE_THREAD_H
#if defined(_MSC_VER) && (_MSC_VER >= 1020)
#pragma once
#endif

#ifdef GE_USE_THREADS

#ifndef GE_EIFFEL_H
#include "ge_eiffel.h"
#endif
#ifndef GE_THREAD_TYPES_H
#include "ge_thread_types.h"
#endif
#ifndef GE_EXCEPTION_H
#include "ge_exception.h"
#endif

#ifdef __cplusplus
extern "C" {
#endif

#ifdef GE_USE_POSIX_THREADS
#include <unistd.h>
#endif

#ifdef GE_USE_POSIX_THREADS
#	define EIF_TSD_VAL_TYPE        void*
#	define EIF_TSD_TYPE            pthread_key_t
#	define EIF_TSD_CREATE(key,msg) \
		if (pthread_key_create(&(key),NULL)) \
	        GE_raise_with_message(GE_EX_EXT, msg)
#	define EIF_TSD_SET(key,val,msg) \
		if (pthread_setspecific((key), (EIF_TSD_VAL_TYPE)(val))) \
			GE_raise_with_message(GE_EX_EXT, msg)
#	define EIF_TSD_GET0(val_type,key,val) (val = pthread_getspecific(key))
#	define EIF_TSD_GET(val_type,key,val,msg) \
		if (EIF_TSD_GET0(val_type,key,val) == (void*) 0) GE_raise_with_message(GE_EX_EXT, msg)
#	define EIF_TSD_DESTROY(key,msg) if (pthread_key_delete(key)) GE_raise_with_message(GE_EX_EXT, msg)
#elif defined EIF_WINDOWS
#	define EIF_TSD_VAL_TYPE        LPVOID
#	define EIF_TSD_TYPE            DWORD
#	define EIF_TSD_CREATE(key,msg) \
		if ((key=TlsAlloc())==TLS_OUT_OF_INDEXES) GE_raise_with_message(GE_EX_EXT, msg)
#	define EIF_TSD_SET(key,val,msg) \
		if (!TlsSetValue((key),(EIF_TSD_VAL_TYPE)(val))) GE_raise_with_message(GE_EX_EXT, msg)
#	define EIF_TSD_GET0(val_type,key,val) \
		val=(val_type) TlsGetValue(key)
#	define EIF_TSD_GET(val_type,key,val,msg) \
		EIF_TSD_GET0(val_type,key,val); \
		if (GetLastError() != NO_ERROR) GE_raise_with_message(GE_EX_EXT, msg)
#	define EIF_TSD_DESTROY(key,msg) \
		if (!TlsFree(key)) GE_raise_with_message(GE_EX_EXT, msg)
#endif

/* Thread priority levels. */
#define EIF_MIN_THR_PRIORITY			0L
#define EIF_BELOW_NORMAL_THR_PRIORITY	100L
#define EIF_DEFAULT_THR_PRIORITY		127L
#define EIF_ABOVE_NORMAL_THR_PRIORITY	154L
#define EIF_MAX_THR_PRIORITY			255L

/*
 * Mutexes used to protect the calls to once-per-process features.
 */
typedef struct {
	EIF_POINTER* boolean_mutex;
	EIF_POINTER* character_8_mutex;
	EIF_POINTER* character_32_mutex;
	EIF_POINTER* integer_8_mutex;
	EIF_POINTER* integer_16_mutex;
	EIF_POINTER* integer_32_mutex;
	EIF_POINTER* integer_64_mutex;
	EIF_POINTER* natural_8_mutex;
	EIF_POINTER* natural_16_mutex;
	EIF_POINTER* natural_32_mutex;
	EIF_POINTER* natural_64_mutex;
	EIF_POINTER* pointer_mutex;
	EIF_POINTER* real_32_mutex;
	EIF_POINTER* real_64_mutex;
	EIF_POINTER* reference_mutex;
	EIF_POINTER* procedure_mutex;
} GE_once_mutexes;

/*
 * Mutexes used to protect the calls to once-per-process features.
 */
extern GE_once_mutexes* GE_process_once_mutexes;

/*
 * Create a new 'GE_once_mutexes' struct which can deal with the
 * numbers of once features passed as argument.
 */
extern GE_once_mutexes* GE_new_once_mutexes(
	unsigned int a_boolean_count,
	unsigned int a_character_8_count,
	unsigned int a_character_32_count,
	unsigned int a_integer_8_count,
	unsigned int a_integer_16_count,
	unsigned int a_integer_32_count,
	unsigned int a_integer_64_count,
	unsigned int a_natural_8_count,
	unsigned int a_natural_16_count,
	unsigned int a_natural_32_count,
	unsigned int a_natural_64_count,
	unsigned int a_pointer_count,
	unsigned int a_real_32_count,
	unsigned int a_real_64_count,
	unsigned int a_reference_count,
	unsigned int a_procedure_count);

/*
 * Keep track of the numbers of once-per-thread features.
 */
extern void GE_thread_onces_set_counts(
	unsigned int a_boolean_count,
	unsigned int a_character_8_count,
	unsigned int a_character_32_count,
	unsigned int a_integer_8_count,
	unsigned int a_integer_16_count,
	unsigned int a_integer_32_count,
	unsigned int a_integer_64_count,
	unsigned int a_natural_8_count,
	unsigned int a_natural_16_count,
	unsigned int a_natural_32_count,
	unsigned int a_natural_64_count,
	unsigned int a_pointer_count,
	unsigned int a_real_32_count,
	unsigned int a_real_64_count,
	unsigned int a_reference_count,
	unsigned int a_procedure_count);

/*
 * Initialize data to handle threads.
 * To be called at the beginning of the main function
 * on the main thread.
 */
extern void GE_init_thread(GE_context* a_context);

/*
 * Create a new thread with attributes `attr' and execute
 * Eiffel routine `routine' on object `current'.
 */
extern void GE_thread_create_with_attr(EIF_REFERENCE current, void (*routine)(EIF_REFERENCE), void (*set_terminated)(EIF_REFERENCE,EIF_BOOLEAN), EIF_THR_ATTR_TYPE* attr);

/*
 * Execution context of current thread.
 */
extern GE_context* GE_thread_current_context(void);

/*
 * Thead ID of current thread.
 */
extern EIF_POINTER GE_thread_id(void);

/*
 * Thread ID of last thread created from current thread.
 */
extern EIF_POINTER GE_last_thread_created(void);

#ifdef EIF_WINDOWS
/*
 * Support for Windows GUI that requires that all GUI operations are performed in the same thread.
 * Allocate new structure of the given size `a_size', assign it to `wel_per_thread_data'.
 * Return newly allocated memory block. It will be freed automatically on thread termination.
 */
extern void* GE_thread_create_wel_per_thread_data(size_t a_size);
#endif

/*
 * Waits until a child thread sets `terminated' from `obj' to True,
 * which means it is terminated. The calling thread must be the
 * direct parent of the thread, or the function might loop indefinitely.
 */
extern void GE_thread_wait(EIF_REFERENCE obj, EIF_BOOLEAN (*get_terminated)(EIF_REFERENCE));

/*
 * Waits until a child thread sets `terminated' from `obj' to True,
 * which means it is terminated, or reaching `a_timeout_ms'.
 * The calling thread must be the direct parent of the thread,
 * or the function might loop indefinitely.
 */
extern EIF_BOOLEAN GE_thread_wait_with_timeout(EIF_REFERENCE obj, EIF_BOOLEAN (*get_terminated)(EIF_REFERENCE), EIF_NATURAL_64 a_timeout_ms);

/*
 * Yields execution to other threads.
 */
extern void GE_thread_yield(void);

/*
 * The calling thread waits for all other children threads to terminate.
 */
extern void GE_thread_join_all(void);

/*
 * Function called to terminate a thread launched by Eiffel with `GE_thread_create_with_attr'.
 * This function must be called from the thread itself (not the parent).
 */
extern void GE_thread_exit(void);

/*
 * Default thread priority level.
 */
extern EIF_INTEGER GE_thread_default_priority(void);

/*
 * Minimum thread priority level.
 */
extern EIF_INTEGER GE_thread_min_priority(void);

/*
 * Maximum thread priority level.
 */
extern EIF_INTEGER GE_thread_max_priority(void);

/*
 * Create a new mutex.
 */
extern EIF_POINTER GE_mutex_create(void);

/*
 * Lock mutex.
 */
extern void GE_mutex_lock(EIF_POINTER a_mutex);

/*
 * Try to lock mutex. Return True on success.
 */
extern EIF_BOOLEAN GE_mutex_try_lock(EIF_POINTER a_mutex);

/*
 * Unlock mutex.
 */
extern void GE_mutex_unlock(EIF_POINTER a_mutex);

/*
 * Destroy and free all resources used by mutex.
 */
extern void GE_mutex_destroy(EIF_POINTER a_mutex);

/*
 * Create a new semaphore allowing `a_count' threads
 * to go into a critical section.
 */
extern EIF_POINTER GE_semaphore_create(EIF_INTEGER a_count);

/*
 * Decrement semaphore count, waiting if necessary
 * until that becomes possible.
 */
extern void GE_semaphore_wait(EIF_POINTER a_semaphore);

/*
 * Has client been successful in decrementing semaphore
 * count without waiting?
 */
extern EIF_BOOLEAN GE_semaphore_try_wait(EIF_POINTER a_semaphore);

/*
 * Increment semaphore count.
 */
extern void GE_semaphore_post(EIF_POINTER a_semaphore);

/*
 * Destroy and free all resources used by semaphore.
 */
extern void GE_semaphore_destroy(EIF_POINTER a_semaphore);

/*
 * Create a new condition variable.
 */
extern EIF_POINTER GE_condition_variable_create(void);

/*
 * Unblock all threads blocked on condition variable.
 */
extern void GE_condition_variable_broadcast(EIF_POINTER a_condition_variable);

/*
 * Unblock one thread blocked on condition variable.
 */
extern void GE_condition_variable_signal(EIF_POINTER a_condition_variable);

/*
 * Block calling thread on condition variable.
 */
extern void GE_condition_variable_wait(EIF_POINTER a_condition_variable, EIF_POINTER a_mutex);

/*
 * Block calling thread on condition variable for at most `a_timeout' milliseconds.
 * Return 1 is we got the condition variable on time, otherwise return 0.
 */
extern EIF_INTEGER GE_condition_variable_wait_with_timeout(EIF_POINTER a_condition_variable, EIF_POINTER a_mutex, EIF_INTEGER a_timeout);

/*
 * Destroy and free all resources used by condition variable.
 */
extern void GE_condition_variable_destroy(EIF_POINTER a_condition_variable);

/*
 * Create a new read-write lock.
 */
extern EIF_POINTER GE_read_write_lock_create(void);

/*
 * Acquire a read lock. Multiple readers can go if there are no writer.
 */
extern void GE_read_write_lock_read_lock(EIF_POINTER a_read_write_lock);

/*
 * Acquire a write lock. Only a single write can proceed.
 */
extern void GE_read_write_lock_write_lock(EIF_POINTER a_read_write_lock);

/*
 * Unlock a read or write lock.
 */
extern void GE_read_write_lock_unlock(EIF_POINTER a_read_write_lock);

/*
 * Destroy and free all resources used by read-write lock.
 */
extern void GE_read_write_lock_destroy(EIF_POINTER a_read_write_lock);

#ifdef __cplusplus
}
#endif

#endif
#endif

/*
	description:

		"C functions used to manipulate native strings"

	system: "Gobo Eiffel Compiler"
	copyright: "Copyright (c) 2013-2018, Eric Bezault and others"
	license: "MIT License"
	date: "$Date: 2019-02-05 23:54:36 +0000 (Tue, 05 Feb 2019) $"
	revision: "$Revision: 102790 $"
*/

#ifndef GE_NATIVE_STRING_H
#define GE_NATIVE_STRING_H
#if defined(_MSC_VER) && (_MSC_VER >= 1020)
#pragma once
#endif

#ifndef GE_EIFFEL_H
#include "ge_eiffel.h"
#endif

#include <string.h>
#ifdef EIF_WINDOWS
#ifdef __LCC__
/* With lcc-win32, stat.h should be included before wchar.h. */
#include <sys/stat.h>
#endif
#include <wchar.h>
#else
#include <sys/types.h>
#endif

#ifdef __cplusplus
extern "C" {
#endif

#ifdef EIF_WINDOWS

/* Macro used to manipulate native strings, i.e: (wchar_t*) */
#define GE_nstrlen wcslen /* size of string */
#define GE_nstrncpy wcsncpy /* Copy n characters of one string to another */
#define GE_nstrcpy wcscpy /* Copy one string to another */
#define GE_nstrncat wcsncat /* Append characters of a string */
#define GE_nstrcat wcscat /* Append a string */
#define GE_nstrstr wcsstr /* Return a pointer to the first occurrence of a search string in a string. */
#define GE_nmakestr(quote) L##quote /* Manifest Native string declaration */
#define GE_nstr_fopen _wfopen /* Open file using native string name */
#define GE_nstrcmp wcscmp /* Compare two strings. */
#define GE_nstrdup _wcsdup /* Duplicate string. */
#define GE_nstr_cat_ascii(dest, src) {							\
		int i;													\
		size_t dest_len, src_len;								\
		dest_len = rt_nstrlen (dest);							\
		src_len = strlen (src);									\
		for (i = 0; i < src_len; i++) {							\
			dest[dest_len + i] = (EIF_NATIVE_CHAR)src[i];		\
		}														\
		dest[dest_len + src_len] = (EIF_NATIVE_CHAR)0;			\
	}

#else /* not EIF_WINDOWS */

/* Macro used to manipulate native strings, i.e: (char*) */
#define GE_nstrlen strlen /* size of string */
#define GE_nstrncpy strncpy /* Copy n characters of one string to another */
#define GE_nstrcpy strcpy /* Copy one string to another */
#define GE_nstrncat strncat /* Append characters of a string */
#define GE_nstrcat strcat /* Append a string */
#define GE_nstrstr strstr /* Return a pointer to the first occurrence of a search string in a string. */
#define GE_nmakestr(quote) quote /* Manifest Native string declaration */
#define GE_nstr_fopen fopen /* Open file using native string name */
#define GE_nstrcmp strcmp /* Compare two strings. */
#define GE_nstrdup strdup /* Duplicate string. */
#define GE_nstr_cat_ascii strcat

#endif


#ifdef __cplusplus
}
#endif

#endif

/*
	description:

		"C functions used to implement class ARGUMENTS"

	system: "Gobo Eiffel Compiler"
	copyright: "Copyright (c) 2007-2017, Eric Bezault and others"
	license: "MIT License"
	date: "$Date: 2019-02-05 23:54:36 +0000 (Tue, 05 Feb 2019) $"
	revision: "$Revision: 102790 $"
*/

#ifndef GE_ARGUMENTS_H
#define GE_ARGUMENTS_H
#if defined(_MSC_VER) && (_MSC_VER >= 1020)
#pragma once
#endif

#ifndef GE_EIFFEL_H
#include "ge_eiffel.h"
#endif

#ifdef __cplusplus
extern "C" {
#endif

extern int GE_argc;
extern EIF_NATIVE_CHAR** GE_argv;

#ifdef __cplusplus
}
#endif

#endif

/*
	description:

		"C functions used to implement type information"

	system: "Gobo Eiffel Compiler"
	copyright: "Copyright (c) 2016-2018, Eric Bezault and others"
	license: "MIT License"
	date: "$Date: 2019-02-05 23:54:36 +0000 (Tue, 05 Feb 2019) $"
	revision: "$Revision: 102790 $"
*/

#ifndef GE_TYPES_H
#define GE_TYPES_H
#if defined(_MSC_VER) && (_MSC_VER >= 1020)
#pragma once
#endif

#ifndef GE_EIFFEL_H
#include "ge_eiffel.h"
#endif
#ifndef GE_EXCEPTION_H
#include "ge_exception.h"
#endif

#ifdef __cplusplus
extern "C" {
#endif

/*
 * Type annotations.
 * When a type has no annotation, it means a detachable, non-separate, variant type.
 * In all other cases, there will be an annotation.
 */
#define ANNOTATION_MASK			0x007F	/* All possible annotations. */
#define ATTACHED_FLAG			0x0001
#define DETACHABLE_FLAG			0x0002	/* Only present when overriding an attached type. */
#define SEPARATE_FLAG			0x0004
#define VARIANT_FLAG			0x0008	/* Only present when overriding a frozen/poly type. */
#define UNUSABLE_FLAG			0x0010	/* Reserved for backward compatibility for storables. */
#define FROZEN_FLAG				0x0020
#define POLY_FLAG				0x0040

/*
 * Type flags.
 */
#define GE_TYPE_FLAG_SPECIAL		0x0010
#define GE_TYPE_FLAG_TUPLE			0x0020
#define GE_TYPE_FLAG_EXPANDED		0x0040
#define GE_TYPE_FLAG_DEFERRED		0x0080
#define GE_TYPE_FLAG_NONE			0x0100
#define GE_TYPE_FLAG_BASIC_MASK		0x000F /* One of "BOOLEAN", "CHARACTER_8", "CHARACTER_32", "INTEGER_8", "INTEGER_16", "INTEGER_32", "INTEGER_64", "NATURAL_8", "NATURAL_16", "NATURAL_32", "NATURAL_64", "POINTER", "REAL_32", "REAL_64" */
#define GE_TYPE_FLAG_BOOLEAN		0x0001
#define GE_TYPE_FLAG_CHARACTER_8	0x0002
#define GE_TYPE_FLAG_CHARACTER_32	0x0003
#define GE_TYPE_FLAG_INTEGER_8		0x0004
#define GE_TYPE_FLAG_INTEGER_16		0x0005
#define GE_TYPE_FLAG_INTEGER_32		0x0006
#define GE_TYPE_FLAG_INTEGER_64		0x0007
#define GE_TYPE_FLAG_NATURAL_8		0x0008
#define GE_TYPE_FLAG_NATURAL_16		0x0009
#define GE_TYPE_FLAG_NATURAL_32		0x000A
#define GE_TYPE_FLAG_NATURAL_64		0x000B
#define GE_TYPE_FLAG_POINTER		0x000C
#define GE_TYPE_FLAG_REAL_32		0x000D
#define GE_TYPE_FLAG_REAL_64		0x000E

/*
 * Convention for attribute types.
 * The values are in sync with REFLECTOR_CONSTANTS.
 */
#define GE_TYPE_KIND_INVALID		-1
#define GE_TYPE_KIND_POINTER		0
#define GE_TYPE_KIND_REFERENCE		1
#define GE_TYPE_KIND_CHARACTER_8	2
#define GE_TYPE_KIND_BOOLEAN		3
#define GE_TYPE_KIND_INTEGER_32		4
#define GE_TYPE_KIND_REAL_32		5
#define GE_TYPE_KIND_REAL_64		6
#define GE_TYPE_KIND_EXPANDED		7
#define GE_TYPE_KIND_INTEGER_8		9
#define GE_TYPE_KIND_INTEGER_16		10
#define GE_TYPE_KIND_INTEGER_64 	11
#define GE_TYPE_KIND_CHARACTER_32	12
#define GE_TYPE_KIND_NATURAL_8		13
#define GE_TYPE_KIND_NATURAL_16		14
#define GE_TYPE_KIND_NATURAL_32 	15
#define GE_TYPE_KIND_NATURAL_64 	16

/*
 * Object flags.
 */
#define GE_OBJECT_FLAG_MARKED		0x0001

/*
 * Ancestor relationship between two types X and Y.
 */
#ifdef GE_USE_ANCESTORS
typedef struct {
	EIF_TYPE_INDEX type_id; /* Type id of Y */
	EIF_BOOLEAN conforms; /* Does X conform to Y? */
	void (**qualified_calls)(); /* Function pointers, indexed by call id, when the static type of the target is Y and the dynamic type is X */
} GE_ancestor;
#endif

/*
 * Attribute.
 */
#ifdef GE_USE_ATTRIBUTES
typedef struct {
#ifdef GE_USE_ATTRIBUTE_NAME
	const char* name; /* Attribute name */
#endif
#ifdef GE_USE_ATTRIBUTE_TYPE_ID
	EIF_ENCODED_TYPE type_id; /* Static type id */
#endif
#ifdef GE_USE_ATTRIBUTE_OFFSET
	uint32_t offset; /* Address offset in object */
#endif
} GE_attribute;
#endif

/*
 * Type information.
 */
typedef struct {
	EIF_TYPE_INDEX type_id;
	uint16_t flags;
#ifdef GE_USE_TYPE_GENERATOR
	const char* generator; /* Generator class name */
#endif
#ifdef GE_USE_TYPE_NAME
	const char* name; /* Full type name */
#endif
#ifdef GE_USE_TYPE_GENERIC_PARAMETERS
	EIF_ENCODED_TYPE* generic_parameters;
	uint32_t generic_parameter_count;
#endif
#ifdef GE_USE_ANCESTORS
	GE_ancestor** ancestors;
	uint32_t ancestor_count;
#endif
#ifdef GE_USE_ATTRIBUTES
	GE_attribute** attributes;
	uint32_t attribute_count;
#endif
#ifdef GE_USE_TYPE_OBJECT_SIZE
	uint64_t object_size;
#endif
	EIF_REFERENCE (*new_instance)();
	void (*dispose)(GE_context*, EIF_REFERENCE);
} GE_type_info;

typedef struct {
	EIF_TYPE_INDEX id; /* Type id of the "TYPE [X]" object */
	EIF_INTEGER type_id; /* Type id of the type "X" */
	EIF_BOOLEAN is_special;
	void (*dispose)(GE_context*, EIF_REFERENCE);
	EIF_REFERENCE a1; /* internal_name */
	EIF_REFERENCE a2; /* internal_name_32 */
} EIF_TYPE_OBJ;

/*
 * Types indexed by type id.
 * Generated by the compiler.
 */
extern EIF_TYPE_OBJ GE_types[];
extern GE_type_info GE_type_infos[];

/*
 * Number of type infos in `GE_type_infos'.
 * Do not take into account the fake item at index 0.
 */
extern int GE_type_info_count;

/*
 * Encode a EIF_TYPE into a EIF_ENCODED_TYPE.
 * The lower part of EIF_ENCODED_TYPE contains the .id field,
 * and the upper part the .annotations.
 */
extern EIF_ENCODED_TYPE GE_encoded_type(EIF_TYPE a_type);

/*
 * Decode a EIF_ENCODED_TYPE into a EIF_TYPE.
 * The lower part of EIF_ENCODED_TYPE contains the .id field,
 * and the upper part the .annotations.
 */
extern EIF_TYPE GE_decoded_type(EIF_ENCODED_TYPE a_type);

/*
 * Type with `a_id' and `a_annotations'.
 */
extern EIF_TYPE GE_new_type(EIF_TYPE_INDEX a_id, EIF_TYPE_INDEX a_annotations);

/*
 * Type of object `obj'.
 */
#define GE_object_type(obj)	GE_new_type(((EIF_REFERENCE)(obj))->id, 0x0)
#define GE_object_encoded_type(obj) GE_encoded_type(GE_object_type(obj))

/*
 * Attachment status of `a_type'.
 */
#define GE_is_attached_type(a_type) EIF_TEST(((a_type).annotations & ATTACHED_FLAG) || GE_is_expanded_type_index((a_type).id))
#define GE_is_attached_encoded_type(a_type) GE_is_attached_type(GE_decoded_type(a_type))

/*
 * Associated detachable type of `a_type' if any,
 * otherwise `a_type'.
 */
extern EIF_TYPE GE_non_attached_type(EIF_TYPE a_type);
#define GE_non_attached_encoded_type(a_type) GE_encoded_type(GE_non_attached_type(GE_decoded_type(a_type)))

/*
 * Associated attached type of `a_type' if any,
 * otherwise `a_type'.
 */
extern EIF_TYPE GE_attached_type(EIF_TYPE a_type);
#define GE_attached_encoded_type(t) GE_encoded_type(GE_attached_type(GE_decoded_type(t)))

/*
 * Is `a_type' a SPECIAL type?
 */
#define GE_is_special_type_index(a_type) EIF_TEST(GE_type_infos[a_type].flags & GE_TYPE_FLAG_SPECIAL)
#define GE_is_special_encoded_type(a_type) GE_is_special_type_index(GE_decoded_type(a_type).id)
#define GE_is_special_object(obj) GE_is_special_type_index(((EIF_REFERENCE)(obj))->id)

/*
 * Is `a_type' a SPECIAL type of user-defined expanded type?
 */
extern EIF_BOOLEAN GE_is_special_of_expanded_type_index(EIF_TYPE_INDEX a_type);
#define GE_is_special_of_expanded_encoded_type(a_type) GE_is_special_of_expanded_type_index(GE_decoded_type(a_type).id)
#define GE_is_special_of_expanded_object(obj) GE_is_special_of_expanded_type_index(((EIF_REFERENCE)(obj))->id)

/*
 * Is `a_type' a SPECIAL type of reference type?
 */
extern EIF_BOOLEAN GE_is_special_of_reference_type_index(EIF_TYPE_INDEX a_type);
#define GE_is_special_of_reference_encoded_type(a_type) GE_is_special_of_reference_type_index(GE_decoded_type(a_type).id)
#define GE_is_special_of_reference_object(obj) GE_is_special_of_reference_type_index(((EIF_REFERENCE)(obj))->id)

/*
 * Is `a_type' a SPECIAL type of reference type or basic expanded type?
 * (Note that user-defined expanded types are excluded.)
 */
extern EIF_BOOLEAN GE_is_special_of_reference_or_basic_expanded_type_index(EIF_TYPE_INDEX a_type);
#define GE_is_special_of_reference_or_basic_expanded_encoded_type(a_type) GE_is_special_of_reference_or_basic_expanded_type_index(GE_decoded_type(a_type).id)
#define GE_is_special_of_reference_or_basic_expanded_object(obj) GE_is_special_of_reference_or_basic_expanded_type_index(((EIF_REFERENCE)(obj))->id)

/*
 * Is `a_type' a TUPLE type?
 */
#define GE_is_tuple_type_index(a_type) EIF_TEST(GE_type_infos[a_type].flags & GE_TYPE_FLAG_TUPLE)
#define GE_is_tuple_encoded_type(a_type) GE_is_tuple_type_index(GE_decoded_type(a_type).id)
#define GE_is_tuple_object(obj) GE_is_tuple_type_index(((EIF_REFERENCE)(obj))->id)

/*
 * Is `a_type' an expanded type?
 */
#define GE_is_expanded_type_index(a_type) EIF_TEST(GE_type_infos[a_type].flags & GE_TYPE_FLAG_EXPANDED)
#define GE_is_expanded_encoded_type(a_type) GE_is_expanded_type_index(GE_decoded_type(a_type).id)
#define GE_is_expanded_object(obj) GE_is_expanded_type_index(((EIF_REFERENCE)(obj))->id)

/*
 * Is `a_type' a type whose base class is deferred?
 */
#define GE_is_deferred_type_index(a_type) EIF_TEST(GE_type_infos[a_type].flags & GE_TYPE_FLAG_DEFERRED)
#define GE_is_deferred_encoded_type(a_type) GE_is_deferred_type_index(GE_decoded_type(a_type).id)

/*
 * Does `i'-th field of `a_object + a_physical_offset' (which is expected to be reference)
 * denote a reference with copy semantics?
 */
extern EIF_BOOLEAN GE_is_copy_semantics_field(EIF_INTEGER i, EIF_POINTER a_object, EIF_INTEGER a_physical_offset);

/*
 * Does `i'-th item of special `a_object' (which is expected to be reference)
 * denote a reference with copy semantics?
 */
extern EIF_BOOLEAN GE_is_special_copy_semantics_item(EIF_INTEGER i, EIF_POINTER a_object);

/*
 * Generator class name of `a_type'.
 */
extern EIF_REFERENCE GE_generator_of_type_index(EIF_TYPE_INDEX a_type);
#define GE_generator_of_encoded_type(a_type) GE_generator_of_type_index(GE_decoded_type(a_type).id)

/*
 * Full name of `a_type'.
 */
extern EIF_REFERENCE GE_generating_type_of_encoded_type(EIF_ENCODED_TYPE a_type);

/*
 * Encoded type whose name is `a_name'.
 * -1 if no such type.
 */
extern EIF_ENCODED_TYPE GE_encoded_type_from_name(EIF_POINTER a_name);

/*
 * Does `a_type_1' conform to `a_type_2'?
 */
extern EIF_BOOLEAN GE_encoded_type_conforms_to(EIF_ENCODED_TYPE a_type_1, EIF_ENCODED_TYPE a_type_2);

/*
 * Number of generic parameters.
 */
extern EIF_INTEGER GE_generic_parameter_count_of_type_index(EIF_TYPE_INDEX a_type);
#define GE_generic_parameter_count_of_encoded_type(a_type) GE_generic_parameter_count_of_type_index(GE_decoded_type(a_type).id)

/*
 * Type of `i'-th generic parameter of `a_type'.
 */
extern EIF_INTEGER GE_generic_parameter_of_type_index(EIF_TYPE_INDEX a_type, EIF_INTEGER i);
#define GE_generic_parameter_of_encoded_type(a_type,i) GE_generic_parameter_of_type_index(GE_decoded_type(a_type).id, (i))

/*
 * Number of fields of an object of dynamic type `a_type'.
 */
extern EIF_INTEGER GE_field_count_of_type_index(EIF_TYPE_INDEX a_type);
#define GE_field_count_of_encoded_type(a_type) GE_field_count_of_type_index(GE_decoded_type(a_type).id)

/*
 * Physical offset of the `i'-th field for an object of dynamic type `a_type'.
 */
extern EIF_INTEGER GE_field_offset_of_type_index(EIF_INTEGER i, EIF_TYPE_INDEX a_type);
#define GE_field_offset_of_encoded_type(i, a_type) GE_field_offset_of_type_index((i), GE_decoded_type(a_type).id)

/*
 * Name of the `i'-th field for an object of dynamic type `a_type'.
 */
extern EIF_POINTER GE_field_name_of_type_index(EIF_INTEGER i, EIF_TYPE_INDEX a_type);
#define GE_field_name_of_encoded_type(i, a_type) GE_field_name_of_type_index((i), GE_decoded_type(a_type).id)

/*
 * Static type of the `i'-th field for an object of dynamic type `a_type'.
 */
extern EIF_INTEGER GE_field_static_type_of_type_index(EIF_INTEGER i, EIF_TYPE_INDEX a_type);
#define GE_field_static_type_of_encoded_type(i, a_type) GE_field_static_type_of_type_index((i), GE_decoded_type(a_type).id)

/*
 * Kind of type of the `i'-th field for an object of dynamic type `a_type'.
 */
extern EIF_INTEGER GE_field_type_kind_of_type_index(EIF_INTEGER i, EIF_TYPE_INDEX a_type);
#define GE_field_type_kind_of_encoded_type(i, a_type) GE_field_type_kind_of_type_index((i), GE_decoded_type(a_type).id)

/*
 * Physical size of `a_object'.
 */
extern EIF_NATURAL_64 GE_object_size(EIF_POINTER a_object);

/*
 * Is `i'-th field of objects of type `a_type' a user-defined expanded attribute?
 */
extern EIF_BOOLEAN GE_is_field_expanded_of_type_index(EIF_INTEGER i, EIF_TYPE_INDEX a_type);
#define GE_is_field_expanded_of_encoded_type(i, a_type) GE_is_field_expanded_of_type_index((i), GE_decoded_type(a_type).id)

#define GE_field_address_at(a_field_offset, a_object, a_physical_offset) ((char*)(a_object) + (a_physical_offset) + (a_field_offset))
#define GE_object_at_offset(a_enclosing, a_physical_offset) (EIF_REFERENCE)(GE_field_address_at(0, (a_enclosing), (a_physical_offset)))
#define GE_raw_object_at_offset(a_enclosing, a_physical_offset) (EIF_POINTER)(GE_field_address_at(0, (a_enclosing), (a_physical_offset)))
#define GE_object_encoded_type_at_offset(a_enclosing, a_physical_offset) GE_object_encoded_type(GE_raw_object_at_offset((a_enclosing), (a_physical_offset)))
#define GE_boolean_field_at(a_field_offset, a_object, a_physical_offset) *(EIF_BOOLEAN*)(GE_field_address_at((a_field_offset), (a_object), (a_physical_offset)))
#define GE_character_8_field_at(a_field_offset, a_object, a_physical_offset) *(EIF_CHARACTER_8*)(GE_field_address_at((a_field_offset), (a_object), (a_physical_offset)))
#define GE_character_32_field_at(a_field_offset, a_object, a_physical_offset) *(EIF_CHARACTER_32*)(GE_field_address_at((a_field_offset), (a_object), (a_physical_offset)))
#define GE_integer_8_field_at(a_field_offset, a_object, a_physical_offset) *(EIF_INTEGER_8*)(GE_field_address_at((a_field_offset), (a_object), (a_physical_offset)))
#define GE_integer_16_field_at(a_field_offset, a_object, a_physical_offset) *(EIF_INTEGER_16*)(GE_field_address_at((a_field_offset), (a_object), (a_physical_offset)))
#define GE_integer_32_field_at(a_field_offset, a_object, a_physical_offset) *(EIF_INTEGER_32*)(GE_field_address_at((a_field_offset), (a_object), (a_physical_offset)))
#define GE_integer_64_field_at(a_field_offset, a_object, a_physical_offset) *(EIF_INTEGER_64*)(GE_field_address_at((a_field_offset), (a_object), (a_physical_offset)))
#define GE_natural_8_field_at(a_field_offset, a_object, a_physical_offset) *(EIF_NATURAL_8*)(GE_field_address_at((a_field_offset), (a_object), (a_physical_offset)))
#define GE_natural_16_field_at(a_field_offset, a_object, a_physical_offset) *(EIF_NATURAL_16*)(GE_field_address_at((a_field_offset), (a_object), (a_physical_offset)))
#define GE_natural_32_field_at(a_field_offset, a_object, a_physical_offset) *(EIF_NATURAL_32*)(GE_field_address_at((a_field_offset), (a_object), (a_physical_offset)))
#define GE_natural_64_field_at(a_field_offset, a_object, a_physical_offset) *(EIF_NATURAL_64*)(GE_field_address_at((a_field_offset), (a_object), (a_physical_offset)))
#define GE_pointer_field_at(a_field_offset, a_object, a_physical_offset) *(EIF_POINTER*)(GE_field_address_at((a_field_offset), (a_object), (a_physical_offset)))
#define GE_real_32_field_at(a_field_offset, a_object, a_physical_offset) *(EIF_REAL_32*)(GE_field_address_at((a_field_offset), (a_object), (a_physical_offset)))
#define GE_real_64_field_at(a_field_offset, a_object, a_physical_offset) *(EIF_REAL_64*)(GE_field_address_at((a_field_offset), (a_object), (a_physical_offset)))
#define GE_raw_reference_field_at(a_field_offset, a_object, a_physical_offset) (EIF_POINTER)*(EIF_REFERENCE*)(GE_field_address_at((a_field_offset), (a_object), (a_physical_offset)))
#define GE_reference_field_at(a_field_offset, a_object, a_physical_offset) *(EIF_REFERENCE*)(GE_field_address_at((a_field_offset), (a_object), (a_physical_offset)))
#define GE_set_boolean_field_at(a_field_offset, a_object, a_physical_offset, a_value) GE_boolean_field_at((a_field_offset), (a_object), (a_physical_offset)) = a_value
#define GE_set_character_8_field_at(a_field_offset, a_object, a_physical_offset, a_value) GE_character_8_field_at((a_field_offset), (a_object), (a_physical_offset)) = a_value
#define GE_set_character_32_field_at(a_field_offset, a_object, a_physical_offset, a_value) GE_character_32_field_at((a_field_offset), (a_object), (a_physical_offset)) = a_value
#define GE_set_integer_8_field_at(a_field_offset, a_object, a_physical_offset, a_value) GE_integer_8_field_at((a_field_offset), (a_object), (a_physical_offset)) = a_value
#define GE_set_integer_16_field_at(a_field_offset, a_object, a_physical_offset, a_value) GE_integer_16_field_at((a_field_offset), (a_object), (a_physical_offset)) = a_value
#define GE_set_integer_32_field_at(a_field_offset, a_object, a_physical_offset, a_value) GE_integer_32_field_at((a_field_offset), (a_object), (a_physical_offset)) = a_value
#define GE_set_integer_64_field_at(a_field_offset, a_object, a_physical_offset, a_value) GE_integer_64_field_at((a_field_offset), (a_object), (a_physical_offset)) = a_value
#define GE_set_natural_8_field_at(a_field_offset, a_object, a_physical_offset, a_value) GE_natural_8_field_at((a_field_offset), (a_object), (a_physical_offset)) = a_value
#define GE_set_natural_16_field_at(a_field_offset, a_object, a_physical_offset, a_value) GE_natural_16_field_at((a_field_offset), (a_object), (a_physical_offset)) = a_value
#define GE_set_natural_32_field_at(a_field_offset, a_object, a_physical_offset, a_value) GE_natural_32_field_at((a_field_offset), (a_object), (a_physical_offset)) = a_value
#define GE_set_natural_64_field_at(a_field_offset, a_object, a_physical_offset, a_value) GE_natural_64_field_at((a_field_offset), (a_object), (a_physical_offset)) = a_value
#define GE_set_pointer_field_at(a_field_offset, a_object, a_physical_offset, a_value) GE_pointer_field_at((a_field_offset), (a_object), (a_physical_offset)) = a_value
#define GE_set_real_32_field_at(a_field_offset, a_object, a_physical_offset, a_value) GE_real_32_field_at((a_field_offset), (a_object), (a_physical_offset)) = a_value
#define GE_set_real_64_field_at(a_field_offset, a_object, a_physical_offset, a_value) GE_real_64_field_at((a_field_offset), (a_object), (a_physical_offset)) = a_value
#define GE_set_reference_field_at(a_field_offset, a_object, a_physical_offset, a_value) GE_reference_field_at((a_field_offset), (a_object), (a_physical_offset)) = a_value

#if defined(GE_USE_ATTRIBUTES) && defined(GE_USE_ATTRIBUTE_OFFSET)
#define GE_field_address(i, a_object, a_physical_offset) GE_field_address_at(GE_type_infos[((EIF_REFERENCE)(a_object))->id].attributes[(i) - 1]->offset, (a_object), (a_physical_offset))
#define GE_boolean_field(i, a_object, a_physical_offset) *(EIF_BOOLEAN*)(GE_field_address((i), (a_object), (a_physical_offset)))
#define GE_character_8_field(i, a_object, a_physical_offset) *(EIF_CHARACTER_8*)(GE_field_address((i), (a_object), (a_physical_offset)))
#define GE_character_32_field(i, a_object, a_physical_offset) *(EIF_CHARACTER_32*)(GE_field_address((i), (a_object), (a_physical_offset)))
#define GE_integer_8_field(i, a_object, a_physical_offset) *(EIF_INTEGER_8*)(GE_field_address((i), (a_object), (a_physical_offset)))
#define GE_integer_16_field(i, a_object, a_physical_offset) *(EIF_INTEGER_16*)(GE_field_address((i), (a_object), (a_physical_offset)))
#define GE_integer_32_field(i, a_object, a_physical_offset) *(EIF_INTEGER_32*)(GE_field_address((i), (a_object), (a_physical_offset)))
#define GE_integer_64_field(i, a_object, a_physical_offset) *(EIF_INTEGER_64*)(GE_field_address((i), (a_object), (a_physical_offset)))
#define GE_natural_8_field(i, a_object, a_physical_offset) *(EIF_NATURAL_8*)(GE_field_address((i), (a_object), (a_physical_offset)))
#define GE_natural_16_field(i, a_object, a_physical_offset) *(EIF_NATURAL_16*)(GE_field_address((i), (a_object), (a_physical_offset)))
#define GE_natural_32_field(i, a_object, a_physical_offset) *(EIF_NATURAL_32*)(GE_field_address((i), (a_object), (a_physical_offset)))
#define GE_natural_64_field(i, a_object, a_physical_offset) *(EIF_NATURAL_64*)(GE_field_address((i), (a_object), (a_physical_offset)))
#define GE_pointer_field(i, a_object, a_physical_offset) *(EIF_POINTER*)(GE_field_address((i), (a_object), (a_physical_offset)))
#define GE_real_32_field(i, a_object, a_physical_offset) *(EIF_REAL_32*)(GE_field_address((i), (a_object), (a_physical_offset)))
#define GE_real_64_field(i, a_object, a_physical_offset) *(EIF_REAL_64*)(GE_field_address((i), (a_object), (a_physical_offset)))
#define GE_reference_field(i, a_object, a_physical_offset) *(EIF_REFERENCE*)(GE_field_address((i), (a_object), (a_physical_offset)))
#define GE_set_boolean_field(i, a_object, a_physical_offset, a_value) GE_boolean_field((i), (a_object), (a_physical_offset)) = (a_value)
#define GE_set_character_8_field(i, a_object, a_physical_offset, a_value) GE_character_8_field((i), (a_object), (a_physical_offset)) = (a_value)
#define GE_set_character_32_field(i, a_object, a_physical_offset, a_value) GE_character_32_field((i), (a_object), (a_physical_offset)) = (a_value)
#define GE_set_integer_8_field(i, a_object, a_physical_offset, a_value) GE_integer_8_field((i), (a_object), (a_physical_offset)) = (a_value)
#define GE_set_integer_16_field(i, a_object, a_physical_offset, a_value) GE_integer_16_field((i), (a_object), (a_physical_offset)) = (a_value)
#define GE_set_integer_32_field(i, a_object, a_physical_offset, a_value) GE_integer_32_field((i), (a_object), (a_physical_offset)) = (a_value)
#define GE_set_integer_64_field(i, a_object, a_physical_offset, a_value) GE_integer_64_field((i), (a_object), (a_physical_offset)) = (a_value)
#define GE_set_natural_8_field(i, a_object, a_physical_offset, a_value) GE_natural_8_field((i), (a_object), (a_physical_offset)) = (a_value)
#define GE_set_natural_16_field(i, a_object, a_physical_offset, a_value) GE_natural_16_field((i), (a_object), (a_physical_offset)) = (a_value)
#define GE_set_natural_32_field(i, a_object, a_physical_offset, a_value) GE_natural_32_field((i), (a_object), (a_physical_offset)) = (a_value)
#define GE_set_natural_64_field(i, a_object, a_physical_offset, a_value) GE_natural_64_field((i), (a_object), (a_physical_offset)) = (a_value)
#define GE_set_pointer_field(i, a_object, a_physical_offset, a_value) GE_pointer_field((i), (a_object), (a_physical_offset)) = (a_value)
#define GE_set_real_32_field(i, a_object, a_physical_offset, a_value) GE_real_32_field((i), (a_object), (a_physical_offset)) = (a_value)
#define GE_set_real_64_field(i, a_object, a_physical_offset, a_value) GE_real_64_field((i), (a_object), (a_physical_offset)) = (a_value)
#define GE_set_reference_field(i, a_object, a_physical_offset, a_value) GE_reference_field((i), (a_object), (a_physical_offset)) = (a_value)
#else
#define GE_boolean_field(i, a_object, a_physical_offset) (EIF_BOOLEAN)0
#define GE_character_8_field(i, a_object, a_physical_offset) (EIF_CHARACTER_8)0
#define GE_character_32_field(i, a_object, a_physical_offset) (EIF_CHARACTER_32)0
#define GE_integer_8_field(i, a_object, a_physical_offset) (EIF_INTEGER_8)0
#define GE_integer_16_field(i, a_object, a_physical_offset) (EIF_INTEGER_16)0
#define GE_integer_32_field(i, a_object, a_physical_offset) (EIF_INTEGER_32)0
#define GE_integer_64_field(i, a_object, a_physical_offset) (EIF_INTEGER_64)0
#define GE_natural_8_field(i, a_object, a_physical_offset) (EIF_NATURAL_8)0
#define GE_natural_16_field(i, a_object, a_physical_offset) (EIF_NATURAL_16)0
#define GE_natural_32_field(i, a_object, a_physical_offset) (EIF_NATURAL_32)0
#define GE_natural_64_field(i, a_object, a_physical_offset) (EIF_NATURAL_64)0
#define GE_pointer_field(i, a_object, a_physical_offset) (EIF_POINTER)0
#define GE_real_32_field(i, a_object, a_physical_offset) (EIF_REAL_32)0
#define GE_real_64_field(i, a_object, a_physical_offset) (EIF_REAL_64)0
#define GE_reference_field(i, a_object, a_physical_offset) (EIF_REFERENCE)0
#define GE_set_boolean_field(i, a_object, a_physical_offset, a_value)
#define GE_set_character_8_field(i, a_object, a_physical_offset, a_value)
#define GE_set_character_32_field(i, a_object, a_physical_offset, a_value)
#define GE_set_integer_8_field(i, a_object, a_physical_offset, a_value)
#define GE_set_integer_16_field(i, a_object, a_physical_offset, a_value)
#define GE_set_integer_32_field(i, a_object, a_physical_offset, a_value)
#define GE_set_integer_64_field(i, a_object, a_physical_offset, a_value)
#define GE_set_natural_8_field(i, a_object, a_physical_offset, a_value)
#define GE_set_natural_16_field(i, a_object, a_physical_offset, a_value)
#define GE_set_natural_32_field(i, a_object, a_physical_offset, a_value)
#define GE_set_natural_64_field(i, a_object, a_physical_offset, a_value)
#define GE_set_pointer_field(i, a_object, a_physical_offset, a_value)
#define GE_set_real_32_field(i, a_object, a_physical_offset, a_value)
#define GE_set_real_64_field(i, a_object, a_physical_offset, a_value)
#define GE_set_reference_field(i, a_object, a_physical_offset, a_value)
#endif

/*
 * Number of non-transient fields of an object of dynamic type `a_type'.
 * TODO: storable not implemented yet.
 */
#define GE_persistent_field_count_of_type_index(a_type) GE_field_count_of_type_index(a_type)
#define GE_persistent_field_count_of_encoded_type(a_type) GE_persistent_field_count_of_type_index(GE_decoded_type(a_type).id)

/*
 * Is `i'-th field of objects of type `a_type' a transient field?
 * TODO: storable not implemented yet.
 */
#define GE_is_field_transient_of_type_index(i, a_type) EIF_FALSE
#define GE_is_field_transient_of_encoded_type(i, a_type) GE_is_field_transient_of_type_index((i), GE_decoded_type(a_type).id)

/*
 * Storable version of objects of type `a_type'.
 * TODO: storable not implemented yet.
 */
#define GE_storable_version_of_type_index(a_type) EIF_VOID
#define GE_storable_version_of_encoded_type(a_type) GE_storable_version_of_type_index(GE_decoded_type(a_type).id)

/*
 * Get a lock on `GE_mark_object' and `GE_unmark_object' routines so that
 * 2 threads cannot `GE_mark_object' and `GE_unmark_object' at the same time.
 */
extern void GE_lock_marking(void);

/*
 * Release a lock on `GE_mark_object' and `GE_unmark_object', so that another
 * thread can use `GE_mark_object' and `GE_unmark_object'.
 */
extern void GE_unlock_marking(void);

/*
 * Is `obj' marked?
 */
extern EIF_BOOLEAN GE_is_object_marked(EIF_POINTER obj);

/*
 * Mark `obj'.
 */
extern void GE_mark_object(EIF_POINTER obj);

/*
 * Unmark `obj'.
 */
extern void GE_unmark_object(EIF_POINTER obj);

/*
 * New instance of dynamic `a_type'.
 * Note: returned object is not initialized and may
 * hence violate its invariant.
 * `a_type' cannot represent a SPECIAL type, use
 * `GE_new_special_instance_of_special_type_index' instead.
 */
extern EIF_REFERENCE GE_new_instance_of_type_index(EIF_TYPE_INDEX a_type);
#define GE_new_instance_of_encoded_type(a_type) GE_new_instance_of_type_index(GE_decoded_type(a_type).id)

/*
 * New instance of dynamic `a_type' that represents
 * a SPECIAL with can contain `a_capacity' element. To create a SPECIAL of
 * basic type, use `SPECIAL'.
 */
extern EIF_REFERENCE GE_new_special_any_instance_of_type_index(EIF_TYPE_INDEX a_type, EIF_INTEGER a_capacity);
#define GE_new_special_any_instance_of_encoded_type(a_type, a_capacity) GE_new_special_instance_of_type_index(GE_decoded_type(a_type).id, (a_capacity))

/*
 * New instance of tuple of type `a_type'.
 * Note: returned object is not initialized and may
 * hence violate its invariant.
 */
#define GE_new_tuple_instance_of_type_index(a_type) GE_new_instance_of_type_index(a_type)
#define GE_new_tuple_instance_of_encoded_type(a_type) GE_new_tuple_instance_of_type_index(GE_decoded_type(a_type).id)

/*
 * New instance of TYPE for object of type `a_type'.
 */
extern EIF_REFERENCE GE_new_type_instance_of_encoded_type(EIF_ENCODED_TYPE a_type);

#ifdef __cplusplus
}
#endif

#endif

/*
	description:

		"C functions used to manipulate strings"

	system: "Gobo Eiffel Compiler"
	copyright: "Copyright (c) 2016, Eric Bezault and others"
	license: "MIT License"
	date: "$Date: 2019-02-05 23:54:36 +0000 (Tue, 05 Feb 2019) $"
	revision: "$Revision: 102790 $"
*/

#ifndef GE_STRING_H
#define GE_STRING_H
#if defined(_MSC_VER) && (_MSC_VER >= 1020)
#pragma once
#endif

#ifndef GE_EIFFEL_H
#include "ge_eiffel.h"
#endif

#ifdef __cplusplus
extern "C" {
#endif

/*
 * New Eiffel empty string of type "STRING_8" with can
 * contain `c' characters.
 * Note: The implementation of this function is generated
 * by the Eiffel compiler.
 */
extern EIF_REFERENCE GE_new_str8(EIF_INTEGER c);

/*
 * New Eiffel empty string of type "STRING_32" with can
 * contain `c' characters.
 * Note: The implementation of this function is generated
 * by the Eiffel compiler.
 */
extern EIF_REFERENCE GE_new_str32(EIF_INTEGER c);

/*
 * New Eiffel empty string of type "IMMUTABLE_STRING_32" with can
 * contain `c' characters.
 * Note: The implementation of this function is generated
 * by the Eiffel compiler.
 */
extern EIF_REFERENCE GE_new_istr32(EIF_INTEGER c);

/*
 * New Eiffel string of type "STRING_8" containing the
 * first `c' characters found in ASCII string `s'.
 */
extern EIF_REFERENCE GE_ms8(const char* s, EIF_INTEGER c);

/*
 * New Eiffel string of type "STRING_8" containing all
 * characters found in the null-terminated ASCII string `s'.
 */
extern EIF_REFERENCE GE_str8(const char* s);

/*
 * New Eiffel string of type "STRING_32" containing the
 * first `c' characters found in ASCII string `s'.
 */
extern EIF_REFERENCE GE_ms32(const char* s, EIF_INTEGER c);

/*
 * New Eiffel string of type "STRING_32" containing all
 * characters found in the null-terminated ASCII string `s'.
 */
extern EIF_REFERENCE GE_str32(const char* s);

/*
 * New Eiffel string of type "IMMUTABLE_STRING_32" containing
 * the first `c' characters found in ASCII string `s'.
 */
extern EIF_REFERENCE GE_ims32(const char* s, EIF_INTEGER c);

/*
 * New Eiffel string of type "IMMUTABLE_STRING_32" containing all
 * characters found in the null-terminated ASCII string `s'.
 */
extern EIF_REFERENCE GE_istr32(const char* s);

/*
 * New Eiffel string of type "IMMUTABLE_STRING_32" containing the
 * first `n' native characters found in native string `s'.
 * Invalid native characters are escaped.
 */
extern EIF_REFERENCE GE_ims32_from_nstr(EIF_NATIVE_CHAR* s, EIF_INTEGER n);

/*
 * New Eiffel string of type "IMMUTABLE_STRING_32" containing all
 * characters found in the null-terminated native string `s'.
 * Invalid native characters are escaped.
 */
extern EIF_REFERENCE GE_istr32_from_nstr(EIF_NATIVE_CHAR* s);

/*
 * New Eiffel string of type "STRING" containing all
 * characters found in the null-terminated ASCII string `s'
 */
extern EIF_REFERENCE GE_str(const char* s);

#ifdef __cplusplus
}
#endif

#endif

/*
	description:

		"C functions used to implement class CONSOLE"

	system: "Gobo Eiffel Compiler"
	copyright: "Copyright (c) 2007-2017, Eric Bezault and others"
	license: "MIT License"
	date: "$Date: 2019-02-05 23:54:36 +0000 (Tue, 05 Feb 2019) $"
	revision: "$Revision: 102790 $"
*/

#ifndef GE_CONSOLE_H
#define GE_CONSOLE_H
#if defined(_MSC_VER) && (_MSC_VER >= 1020)
#pragma once
#endif

#ifdef __cplusplus
extern "C" {
#endif

/*
 * Initialize mutex to determine whether a new
 * console needs to be created.
 */
#ifdef EIF_WINDOWS
extern void GE_init_console(void);
#else
#define GE_init_console()
#endif

/*
 * Create a new DOS console if needed (i.e. in case of a Windows application).
 */
#ifdef EIF_WINDOWS
extern void GE_show_console(void);
#else
#define GE_show_console()
#endif

#ifdef __cplusplus
}
#endif

#endif

/*
	description:

		"C functions used to implement the program initialization"

	system: "Gobo Eiffel Compiler"
	copyright: "Copyright (c) 2007-2017, Eric Bezault and others"
	license: "MIT License"
	date: "$Date: 2019-02-05 23:54:36 +0000 (Tue, 05 Feb 2019) $"
	revision: "$Revision: 102790 $"
*/

#ifndef GE_MAIN_H
#define GE_MAIN_H
#if defined(_MSC_VER) && (_MSC_VER >= 1020)
#pragma once
#endif

#ifndef GE_EIFFEL_H
#include "ge_eiffel.h"
#endif

#ifdef __cplusplus
extern "C" {
#endif

extern int GE_main(int argc, EIF_NATIVE_CHAR** argv);

/*
 * System name.
 */
extern char* GE_system_name;

/*
 * Root class name.
 */
extern char* GE_root_class_name;

#ifdef EIF_WINDOWS

#include <windows.h>

/*
 * Used in WEL.
 */
extern HINSTANCE eif_hInstance;
extern HINSTANCE eif_hPrevInstance;
extern LPWSTR eif_lpCmdLine;
extern int eif_nCmdShow;

/*
 * Main entry point when compiling a Windows application.
 * See:
 *    http://en.wikipedia.org/wiki/WinMain
 *    http://msdn2.microsoft.com/en-us/library/ms633559.aspx
 */
extern int WINAPI WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance, LPSTR lpCmdLine, int nCmdShow);

#endif

#ifdef __cplusplus
}
#endif

#endif

/*
	description:

		"C functions used to access garbage collector facilities"

	system: "Gobo Eiffel Compiler"
	copyright: "Copyright (c) 2007-2017, Eric Bezault and others"
	license: "MIT License"
	date: "$Date: 2019-02-05 23:54:36 +0000 (Tue, 05 Feb 2019) $"
	revision: "$Revision: 102790 $"
*/

#ifndef GE_GC_H
#define GE_GC_H
#if defined(_MSC_VER) && (_MSC_VER >= 1020)
#pragma once
#endif

#ifndef GE_EXCEPTION_H
#include "ge_exception.h"
#endif

#ifdef GE_USE_BOEHM_GC
/*
 *	Use the Boehm garbage collector.
 *	See:
 *		http://en.wikipedia.org/wiki/Boehm_GC
 *		http://www.hpl.hp.com/personal/Hans_Boehm/gc/
 */
#define GC_IGNORE_WARN
#ifdef GE_USE_THREADS
#define GC_THREADS
#undef GC_NO_THREAD_REDIRECTS
#endif
#include "gc.h"
#endif

/*
 * GC initialization.
 */

#ifdef GE_USE_BOEHM_GC
#ifdef GE_USE_THREADS
#define GE_init_gc() \
	GC_INIT(); \
	GC_allow_register_threads(); \
	GC_enable_incremental()
#else
#define GE_init_gc() \
	GC_INIT(); \
	GC_enable_incremental()
#endif
#else /* No GC */
#define GE_init_gc() /* do nothing */
#endif

/*
 * Memory allocation.
 */

/*
 * Allocate memory that can contain pointers to collectable objects.
 * The allocated memory is not necessarily zeroed (unless `GE_malloc_cleared' is_defined).
 * The allocated object is itself collectable.
 * Raise an exception when no-more-memory.
 */
#ifdef GE_USE_BOEHM_GC
#define GE_malloc(size) GE_null(GC_MALLOC(size))
#else /* No GC */
#define GE_malloc(size) GE_null(malloc(size))
#endif

/*
 * Allocate memory that can contain pointers to collectable objects.
 * The allocated memory is not necessarily zeroed (unless `GE_malloc_cleared' is_defined).
 * The allocated object is itself collectable.
 * Do not raise an exception when no-more-memory.
 */
#ifdef GE_USE_BOEHM_GC
#define GE_unprotected_malloc(size) GC_MALLOC(size)
#else /* No GC */
#define GE_unprotected_malloc(size) malloc(size)
#endif

/*
 * Allocate memory that does not contain pointers to collectable objects.
 * The allocated memory is not necessarily zeroed (unless `GE_malloc_atomic_cleared' is_defined).
 * The allocated object is itself collectable.
 * Raise an exception when no-more-memory.
 */
#ifdef GE_USE_BOEHM_GC
#define GE_malloc_atomic(size) GE_null(GC_MALLOC_ATOMIC(size))
#else /* No GC */
#define GE_malloc_atomic(size) GE_null(malloc(size))
#endif

/*
 * Allocate memory that does not contain pointers to collectable objects.
 * The allocated memory is not necessarily zeroed (unless `GE_malloc_atomic_cleared' is_defined).
 * The allocated object is itself collectable.
 * Do not raise an exception when no-more-memory.
 */
#ifdef GE_USE_BOEHM_GC
#define GE_unprotected_malloc_atomic(size) GC_MALLOC_ATOMIC(size)
#else /* No GC */
#define GE_unprotected_malloc_atomic(size) malloc(size)
#endif

/*
 * Allocate memory that can contain pointers to collectable objects.
 * The allocated memory is zeroed.
 * The allocated object is itself collectable.
 * Raise an exception when no-more-memory.
 */
#ifdef GE_USE_BOEHM_GC
#define GE_calloc(nelem, elsize) GE_null(GC_MALLOC((nelem) * (elsize)))
#else /* No GC */
#define GE_calloc(nelem, elsize) GE_null(calloc((nelem), (elsize)))
#endif

/*
 * Allocate memory that can contain pointers to collectable objects.
 * The allocated memory is zeroed.
 * The allocated object is itself collectable.
 * Do not raise an exception when no-more-memory.
 */
#ifdef GE_USE_BOEHM_GC
#define GE_unprotected_calloc(nelem, elsize) GC_MALLOC((nelem) * (elsize))
#else /* No GC */
#define GE_unprotected_calloc(nelem, elsize) calloc((nelem), (elsize))
#endif

/*
 * Allocate memory that does not contain pointers to collectable objects.
 * The allocated memory is zeroed.
 * The allocated object is itself collectable.
 * Raise an exception when no-more-memory.
 */
#ifdef GE_USE_BOEHM_GC
#define GE_calloc_atomic(nelem, elsize) memset(GE_null(GC_MALLOC_ATOMIC((nelem) * (elsize))), 0, (nelem) * (elsize))
#else /* No GC */
#define GE_calloc(nelem, elsize) GE_null(calloc((nelem), (elsize)))
#endif

/*
 * Allocate memory that can contain pointers to collectable objects.
 * The allocated memory is not necessarily zeroed (unless `GE_malloc_cleared' is_defined).
 * The allocated object is itself not collectable.
 * Raise an exception when no-more-memory.
 */
#ifdef GE_USE_BOEHM_GC
#define GE_malloc_uncollectable(size) GE_null(GC_MALLOC_UNCOLLECTABLE(size))
#else /* No GC */
#define GE_malloc_uncollectable(size) GE_null(malloc(size))
#endif

/*
 * Allocate memory that can contain pointers to collectable objects.
 * The allocated memory is not necessarily zeroed (unless `GE_malloc_cleared' is_defined).
 * The allocated object is itself not collectable.
 * Do not raise an exception when no-more-memory.
 */
#ifdef GE_USE_BOEHM_GC
#define GE_unprotected_malloc_uncollectable(size) GC_MALLOC_UNCOLLECTABLE(size)
#else /* No GC */
#define GE_unprotected_malloc_uncollectable(size) malloc(size)
#endif

/*
 * Allocate memory that does not contain pointers to collectable objects.
 * The allocated memory is not necessarily zeroed (unless `GE_malloc_atomic_cleared' is_defined).
 * The allocated object is itself not collectable.
 * Raise an exception when no-more-memory.
 */
#ifdef GE_USE_BOEHM_GC
#define GE_malloc_atomic_uncollectable(size) GE_null(GC_malloc_atomic_uncollectable(size))
#else /* No GC */
#define GE_malloc_atomic_uncollectable(size) GE_null(malloc(size))
#endif

/*
 * Allocate memory that does not contain pointers to collectable objects.
 * The allocated memory is not necessarily zeroed (unless `GE_malloc_atomic_cleared' is_defined).
 * The allocated object is itself not collectable.
 * Do not raise an exception when no-more-memory.
 */
#ifdef GE_USE_BOEHM_GC
#define GE_unprotected_malloc_atomic_uncollectable(size) GC_malloc_atomic_uncollectable(size)
#else /* No GC */
#define GE_unprotected_malloc_atomic_uncollectable(size) malloc(size)
#endif

/*
 * Allocate memory that can contain pointers to collectable objects.
 * The allocated memory is zeroed.
 * The allocated object is itself not collectable.
 * Raise an exception when no-more-memory.
 */
#ifdef GE_USE_BOEHM_GC
#define GE_calloc_uncollectable(nelem, elsize) GE_null(GC_MALLOC_UNCOLLECTABLE((nelem) * (elsize)))
#else /* No GC */
#define GE_calloc_uncollectable(nelem, elsize) GE_null(calloc((nelem), (elsize)))
#endif

/*
 * Allocate memory that can contain pointers to collectable objects.
 * The allocated memory is zeroed.
 * The allocated object is itself not collectable.
 * Do not raise an exception when no-more-memory.
 */
#ifdef GE_USE_BOEHM_GC
#define GE_unprotected_calloc_uncollectable(nelem, elsize) GC_MALLOC_UNCOLLECTABLE((nelem) * (elsize))
#else /* No GC */
#define GE_unprotected_calloc_uncollectable(nelem, elsize) calloc((nelem), (elsize))
#endif

/*
 * Allocate memory that does not contain pointers to collectable objects.
 * The allocated memory is zeroed.
 * The allocated object is itself not collectable.
 * Raise an exception when no-more-memory.
 */
#ifdef GE_USE_BOEHM_GC
#define GE_calloc_atomic_uncollectable(nelem, elsize) memset(GE_null(GC_malloc_atomic_uncollectable((nelem) * (elsize))), 0, (nelem) * (elsize))
#else /* No GC */
#define GE_calloc_atomic_uncollectable(nelem, elsize) GE_null(calloc((nelem), (elsize)))
#endif

/*
 * Allocate memory that does not contain pointers to collectable objects.
 * The allocated memory is zeroed.
 * The allocated object is itself not collectable.
 * Do not raise an exception when no-more-memory.
 */
#ifdef GE_USE_BOEHM_GC
extern void* GE_unprotected_calloc_atomic_uncollectable(size_t nelem, size_t elsize);
#else /* No GC */
#define GE_unprotected_calloc_atomic_uncollectable(nelem, elsize) calloc((nelem), (elsize))
#endif

/*
 * Allocate more memory for the given pointer.
 * The reallocated pointer keeps the same properties (e.g. atomic or not, collectable or not).
 * The extra allocated memory is not necessarily zeroed.
 * Raise an exception when no-more-memory.
 */
#ifdef GE_USE_BOEHM_GC
#define GE_realloc(p, size) GE_null(GC_REALLOC((p), (size)))
#else /* No GC */
#define GE_realloc(p, size) GE_null(realloc((p), (size)))
#endif

/*
 * Allocate more memory for the given pointer.
 * The reallocated pointer keeps the same properties (e.g. atomic or not, collectable or not).
 * The extra allocated memory is not necessarily zeroed.
 * Do not raise an exception when no-more-memory.
 */
#ifdef GE_USE_BOEHM_GC
#define GE_unprotected_realloc(p, size) GC_REALLOC((p), (size))
#else /* No GC */
#define GE_unprotected_realloc(p, size) realloc((p), (size))
#endif

/*
 * Allocate more memory for the given pointer.
 * The reallocated pointer keeps the same properties (e.g. atomic or not, collectable or not).
 * The extra allocated memory is zeroed.
 * Raise an exception when no-more-memory.
 */
extern void* GE_recalloc(void* p, size_t old_nelem, size_t new_nelem, size_t elsize);

/*
 * Allocate more memory for the given pointer.
 * The reallocated pointer keeps the same properties (e.g. atomic or not, collectable or not).
 * The extra allocated memory is zeroed.
 * Do not raise an exception when no-more-memory.
 */
extern void* GE_unprotected_recalloc(void* p, size_t old_nelem, size_t new_nelem, size_t elsize);

/*
 * Explicitly deallocate an object.
 */
#ifdef GE_USE_BOEHM_GC
#define GE_free(p) GC_FREE(p)
#else /* No GC */
#define GE_free(p) free(p)
#endif

/*
 * When defined, `GE_malloc_cleared' means that `GE_malloc' and
 * `GE_malloc_uncollectable' make sure that the allocated memory
 * is zeroed.
 */
#ifdef GE_USE_BOEHM_GC
#define GE_malloc_cleared
#else /* No GC */
#endif

/*
 * When defined, `GE_malloc_atomic_cleared' means that `GE_malloc_atomic'
 * and `GE_malloc_atomic_uncollectable' make sure that the allocated memory
 * is zeroed.
 */
#ifdef GE_USE_BOEHM_GC
/* #define GE_malloc_atomic_cleared */
#else /* No GC */
#endif

/*
 * Dispose
 */

/*
 * Register dispose routine `disp' to be called on object `obj' when it will be collected.
 */
#ifdef GE_USE_BOEHM_GC
extern void GE_boehm_dispose(void* C, void* disp); /* Call dispose routine `disp' on object `C'. */
#define GE_register_dispose(obj, disp) GC_REGISTER_FINALIZER_NO_ORDER((void*)(obj), (void (*) (void*, void*)) &GE_boehm_dispose, NULL, NULL, NULL)
#else /* No GC */
#define GE_register_dispose(obj, disp) /* do nothing */
#endif

/*
 * Access to objects, useful with GCs which move objects in memory.
 * This is not the case here, since the Boehm GC is not a moving GC.
 */

/* Access object through hector. */
#define eif_access(obj) (obj)
/* Freeze memory address. */
#define eif_freeze(obj) (obj)
/* The C side adopts an object. */
#define eif_adopt(obj) (obj)
/* The C side protects an object. */
#define eif_protect(obj) (obj)
/* The C side weans adopted object. */
#define eif_wean(obj) (obj)
/* Forget a frozen memory address. */
#define eif_unfreeze(obj)
/* Always frozen since they do not move. */
#define eif_frozen(obj) 1
/* Always frozen since they do not move. */
#define spfrozen(obj) 1

#endif

/*
	description:

		"C functions used to implement class IDENTIFIED"

	system: "Gobo Eiffel Compiler"
	copyright: "Copyright (c) 2007-2017, Eric Bezault and others"
	license: "MIT License"
	date: "$Date: 2019-02-05 23:54:36 +0000 (Tue, 05 Feb 2019) $"
	revision: "$Revision: 102790 $"
*/

#ifndef GE_IDENTIFIED_H
#define GE_IDENTIFIED_H
#if defined(_MSC_VER) && (_MSC_VER >= 1020)
#pragma once
#endif

#ifndef GE_EIFFEL_H
#include "ge_eiffel.h"
#endif

#ifdef __cplusplus
extern "C" {
#endif

/*
 * Initialize data to keep track of object ids.
 */
extern void GE_init_identified(void);

/*
 * Get a new id for `object', assuming it is NOT in the stack.
 */
extern EIF_INTEGER_32 GE_object_id(EIF_OBJECT object);

/*
 * Return the object associated with `id'.
 */
extern EIF_REFERENCE GE_id_object(EIF_INTEGER_32 id);

/*
 * Remove the object associated with `id' from the stack.
 */
extern void GE_object_id_free(EIF_INTEGER_32 id);

#ifdef __cplusplus
}
#endif

#endif

#ifdef __cplusplus
extern "C" {
#endif

#define T0 EIF_ANY

/* CHARACTER */
#define EIF_CHARACTER EIF_CHARACTER_8

/* WIDE_CHARACTER */
#define EIF_WIDE_CHAR EIF_CHARACTER_32

/* INTEGER */
#define EIF_INTEGER EIF_INTEGER_32

/* NATURAL */
#define EIF_NATURAL EIF_NATURAL_32

/* REAL */
#define EIF_REAL EIF_REAL_32

/* DOUBLE */
#define EIF_DOUBLE EIF_REAL_64

/* BOOLEAN */
#define T1 EIF_BOOLEAN
extern T0* GE_boxed1(T1 a1);
typedef struct Sb1 Tb1;

/* CHARACTER_8 */
#define T2 EIF_CHARACTER_8
extern T0* GE_boxed2(T2 a1);
typedef struct Sb2 Tb2;

/* CHARACTER_32 */
#define T3 EIF_CHARACTER_32
extern T0* GE_boxed3(T3 a1);
typedef struct Sb3 Tb3;

/* INTEGER_8 */
#define T4 EIF_INTEGER_8
extern T0* GE_boxed4(T4 a1);
typedef struct Sb4 Tb4;

/* INTEGER_16 */
#define T5 EIF_INTEGER_16
extern T0* GE_boxed5(T5 a1);
typedef struct Sb5 Tb5;

/* INTEGER_32 */
#define T6 EIF_INTEGER_32
extern T0* GE_boxed6(T6 a1);
typedef struct Sb6 Tb6;

/* INTEGER_64 */
#define T7 EIF_INTEGER_64
extern T0* GE_boxed7(T7 a1);
typedef struct Sb7 Tb7;

/* NATURAL_8 */
#define T8 EIF_NATURAL_8
extern T0* GE_boxed8(T8 a1);
typedef struct Sb8 Tb8;

/* NATURAL_16 */
#define T9 EIF_NATURAL_16
extern T0* GE_boxed9(T9 a1);
typedef struct Sb9 Tb9;

/* NATURAL_32 */
#define T10 EIF_NATURAL_32
extern T0* GE_boxed10(T10 a1);
typedef struct Sb10 Tb10;

/* NATURAL_64 */
#define T11 EIF_NATURAL_64
extern T0* GE_boxed11(T11 a1);
typedef struct Sb11 Tb11;

/* REAL_32 */
#define T12 EIF_REAL_32
extern T0* GE_boxed12(T12 a1);
typedef struct Sb12 Tb12;

/* REAL_64 */
#define T13 EIF_REAL_64
extern T0* GE_boxed13(T13 a1);
typedef struct Sb13 Tb13;

/* POINTER */
#define T14 EIF_POINTER
extern T0* GE_boxed14(T14 a1);
typedef struct Sb14 Tb14;

/* SPECIAL [CHARACTER_8] */
typedef struct S15 T15;

/* SPECIAL [CHARACTER_32] */
typedef struct S16 T16;

/* STRING_8 */
typedef struct S17 T17;

/* STRING_32 */
typedef struct S18 T18;

/* IMMUTABLE_STRING_32 */
typedef struct S19 T19;

/* ISE_EXCEPTION_MANAGER */
typedef struct S20 T20;

/* GEC */
typedef struct S25 T25;

/* [attached] CELL [detachable EXCEPTION] */
typedef struct S26 T26;

/* [attached] HASH_TABLE [[attached] INTEGER_32, [attached] INTEGER_32] */
typedef struct S27 T27;

/* [attached] CELL [detachable TUPLE [[attached] INTEGER_32, [attached] INTEGER_32, [attached] INTEGER_32, [attached] STRING_8, [attached] STRING_8, [attached] STRING_8, [attached] STRING_8, [attached] STRING_8, [attached] STRING_8, [attached] INTEGER_32, [attached] BOOLEAN]] */
typedef struct S28 T28;

/* [attached] CELL [[attached] NO_MORE_MEMORY] */
typedef struct S29 T29;

/* C_STRING */
typedef struct S31 T31;

/* [attached] TUPLE [[attached] INTEGER_32, [attached] INTEGER_32, [attached] INTEGER_32, [attached] STRING_8, [attached] STRING_8, [attached] STRING_8, [attached] STRING_8, [attached] STRING_8, [attached] STRING_8, [attached] INTEGER_32, [attached] BOOLEAN] */
typedef struct S32 T32;

/* KL_TEXT_INPUT_FILE */
typedef struct S33 T33;

/* KL_ARGUMENTS */
typedef struct S34 T34;

/* ET_ISE_VARIABLES */
typedef struct S35 T35;

/* ET_ERROR_HANDLER */
typedef struct S36 T36;

/* KL_STRING_ROUTINES */
typedef struct S38 T38;

/* ET_SYSTEM */
typedef struct S40 T40;

/* KL_EXCEPTIONS */
typedef struct S41 T41;

/* AP_PARSER */
typedef struct S42 T42;

/* AP_ALTERNATIVE_OPTIONS_LIST */
typedef struct S43 T43;

/* AP_STRING_OPTION */
typedef struct S44 T44;

/* AP_FLAG */
typedef struct S46 T46;

/* AP_ENUMERATION_OPTION */
typedef struct S47 T47;

/* AP_BOOLEAN_OPTION */
typedef struct S48 T48;

/* AP_INTEGER_OPTION */
typedef struct S49 T49;

/* ET_NULL_ERROR_HANDLER */
typedef struct S53 T53;

/* ET_XACE_SYSTEM_PARSER */
typedef struct S55 T55;

/* ET_XACE_DEFAULT_ERROR_HANDLER */
typedef struct S56 T56;

/* [attached] DS_HASH_TABLE [[attached] STRING_8, [attached] STRING_8] */
typedef struct S57 T57;

/* UC_STRING_EQUALITY_TESTER */
typedef struct S58 T58;

/* ET_ECF_VARIABLES */
typedef struct S61 T61;

/* [attached] DS_HASH_TABLE_CURSOR [[attached] STRING_8, [attached] STRING_8] */
typedef struct S62 T62;

/* ET_ECF_SYSTEM_PARSER */
typedef struct S66 T66;

/* ET_ECF_ERROR_HANDLER */
typedef struct S67 T67;

/* ET_ECF_SYSTEM */
typedef struct S69 T69;

/* ET_ECF_TARGET */
typedef struct S70 T70;

/* ET_ECF_SETTINGS */
typedef struct S71 T71;

/* ET_LACE_PARSER */
typedef struct S77 T77;

/* ET_LACE_ERROR_HANDLER */
typedef struct S78 T78;

/* ET_DYNAMIC_SYSTEM */
typedef struct S80 T80;

/* ET_SYSTEM_PROCESSOR */
typedef struct S81 T81;

/* ET_C_GENERATOR */
typedef struct S84 T84;

/* KL_SHELL_COMMAND */
typedef struct S85 T85;

/* ET_SYSTEM_MULTIPROCESSOR */
typedef struct S87 T87;

/* UT_VERSION */
typedef struct S88 T88;

/* ET_DYNAMIC_PUSH_TYPE_SET_BUILDER */
typedef struct S91 T91;

/* ET_CLASS_TYPE */
typedef struct S93 T93;

/* ET_TOKEN_CONSTANTS */
typedef struct S94 T94;

/* ET_CLASS */
typedef struct S95 T95;

/* KL_OPERATING_SYSTEM */
typedef struct S97 T97;

/* UT_CANNOT_READ_FILE_ERROR */
typedef struct S100 T100;

/* UT_VERSION_NUMBER */
typedef struct S102 T102;

/* UT_MESSAGE */
typedef struct S103 T103;

/* ET_ECF_CAPABILITIES */
typedef struct S106 T106;

/* KL_EXECUTION_ENVIRONMENT */
typedef struct S107 T107;

/* AP_ERROR */
typedef struct S108 T108;

/* [attached] DS_HASH_SET [[attached] STRING_8] */
typedef struct S109 T109;

/* VOID_TARGET */
typedef struct S112 T112;

/* [attached] TYPE [[attached] VOID_TARGET] */
#define T113 EIF_TYPE_OBJ

/* ROUTINE_FAILURE */
typedef struct S114 T114;

/* [attached] TYPE [[attached] ROUTINE_FAILURE] */
#define T115 EIF_TYPE_OBJ

/* OLD_VIOLATION */
typedef struct S116 T116;

/* [attached] TYPE [[attached] OLD_VIOLATION] */
#define T117 EIF_TYPE_OBJ

/* NO_MORE_MEMORY */
typedef struct S119 T119;

/* INVARIANT_VIOLATION */
typedef struct S120 T120;

/* OPERATING_SYSTEM_SIGNAL_FAILURE */
typedef struct S121 T121;

/* IO_FAILURE */
typedef struct S122 T122;

/* OPERATING_SYSTEM_FAILURE */
typedef struct S123 T123;

/* COM_FAILURE */
typedef struct S124 T124;

/* EIFFEL_RUNTIME_PANIC */
typedef struct S125 T125;

/* PRECONDITION_VIOLATION */
typedef struct S128 T128;

/* POSTCONDITION_VIOLATION */
typedef struct S129 T129;

/* FLOATING_POINT_FAILURE */
typedef struct S130 T130;

/* CHECK_VIOLATION */
typedef struct S131 T131;

/* BAD_INSPECT_VALUE */
typedef struct S132 T132;

/* VARIANT_VIOLATION */
typedef struct S133 T133;

/* LOOP_INVARIANT_VIOLATION */
typedef struct S134 T134;

/* RESCUE_FAILURE */
typedef struct S135 T135;

/* RESUMPTION_FAILURE */
typedef struct S136 T136;

/* CREATE_ON_DEFERRED */
typedef struct S137 T137;

/* EXTERNAL_FAILURE */
typedef struct S138 T138;

/* VOID_ASSIGNED_TO_EXPANDED */
typedef struct S139 T139;

/* EXCEPTION_IN_SIGNAL_HANDLER_FAILURE */
typedef struct S140 T140;

/* MISMATCH_FAILURE */
typedef struct S141 T141;

/* DEVELOPER_EXCEPTION */
typedef struct S142 T142;

/* ADDRESS_APPLIED_TO_MELTED_FEATURE */
typedef struct S143 T143;

/* SERIALIZATION_FAILURE */
typedef struct S144 T144;

/* EXECUTION_ENVIRONMENT */
typedef struct S145 T145;

/* KL_WINDOWS_FILE_SYSTEM */
typedef struct S146 T146;

/* KL_UNIX_FILE_SYSTEM */
typedef struct S147 T147;

/* PRIMES */
typedef struct S148 T148;

/* [attached] SPECIAL [[attached] INTEGER_32] */
typedef struct S149 T149;

/* [attached] SPECIAL [[attached] BOOLEAN] */
typedef struct S150 T150;

/* MANAGED_POINTER */
typedef struct S154 T154;

/* [attached] KL_LINKABLE [[attached] CHARACTER_8] */
typedef struct S155 T155;

/* FILE_INFO */
typedef struct S157 T157;

/* MUTEX */
typedef struct S158 T158;

/* UT_ERROR_HANDLER */
typedef struct S159 T159;

/* KL_STANDARD_FILES */
typedef struct S160 T160;

/* KL_STDERR_FILE */
typedef struct S161 T161;

/* KL_STDOUT_FILE */
typedef struct S162 T162;

/* UC_STRING */
typedef struct S163 T163;

/* KL_ANY_ROUTINES */
typedef struct S164 T164;

/* ST_WORD_WRAPPER */
typedef struct S166 T166;

/* AP_DISPLAY_HELP_FLAG */
typedef struct S168 T168;

/* [attached] DS_ARRAYED_LIST [[attached] AP_OPTION] */
typedef struct S169 T169;

/* [attached] DS_ARRAYED_LIST [[attached] AP_ALTERNATIVE_OPTIONS_LIST] */
typedef struct S170 T170;

/* [attached] DS_ARRAYED_LIST [[attached] STRING_8] */
typedef struct S171 T171;

/* AP_ERROR_HANDLER */
typedef struct S172 T172;

/* [attached] DS_LINKED_LIST_CURSOR [[attached] AP_OPTION] */
typedef struct S175 T175;

/* [attached] DS_ARRAYED_LIST [detachable STRING_8] */
typedef struct S176 T176;

/* [attached] DS_ARRAYED_LIST_CURSOR [detachable STRING_8] */
typedef struct S177 T177;

/* [attached] DS_LINKED_LIST [[attached] STRING_8] */
typedef struct S178 T178;

/* KL_STRING_EQUALITY_TESTER */
typedef struct S179 T179;

/* [attached] DS_LINKED_LIST [[attached] BOOLEAN] */
typedef struct S182 T182;

/* [attached] DS_LINKED_LIST [[attached] INTEGER_32] */
typedef struct S185 T185;

/* KL_NULL_TEXT_OUTPUT_STREAM */
typedef struct S186 T186;

/* ET_XACE_AST_FACTORY */
typedef struct S187 T187;

/* XM_DOCUMENT */
typedef struct S188 T188;

/* XM_ELEMENT */
typedef struct S189 T189;

/* XM_TREE_CALLBACKS_PIPE */
typedef struct S192 T192;

/* XM_STOP_ON_ERROR_FILTER */
typedef struct S193 T193;

/* XM_CALLBACKS_TO_TREE_FILTER */
typedef struct S194 T194;

/* ET_XACE_VALIDATOR */
typedef struct S195 T195;

/* ET_XACE_PREPROCESSOR */
typedef struct S196 T196;

/* ET_XACE_SYSTEM */
typedef struct S197 T197;

/* [attached] DS_HASH_TABLE [[attached] ET_XACE_LIBRARY_CONFIG, [attached] STRING_8] */
typedef struct S198 T198;

/* XM_EXPAT_PARSER_FACTORY */
typedef struct S199 T199;

/* XM_EIFFEL_PARSER */
typedef struct S200 T200;

/* ET_XACE_LIBRARY_CONFIG_PARSER */
typedef struct S203 T203;

/* attached KL_EQUALITY_TESTER [[attached] STRING_8] */
typedef struct S206 T206;

/* [attached] KL_SPECIAL_ROUTINES [[attached] STRING_8] */
typedef struct S207 T207;

/* [attached] SPECIAL [[attached] STRING_8] */
typedef struct S208 T208;

/* [attached] KL_SPECIAL_ROUTINES [[attached] INTEGER_32] */
typedef struct S209 T209;

/* [attached] KL_AGENT_HASH_FUNCTION [[attached] STRING_8] */
typedef struct S210 T210;

/* KL_CASE_INSENSITIVE_STRING_EQUALITY_TESTER */
typedef struct S212 T212;

/* [attached] TUPLE [[attached] STRING_8] */
typedef struct S213 T213;

/* [attached] FUNCTION [[attached] TUPLE [[attached] STRING_8], [attached] INTEGER_32] */
typedef struct S214 T214;

/* [attached] TUPLE */
typedef struct S216 T216;

/* [attached] TUPLE [KL_STRING_ROUTINES] */
typedef struct S217 T217;

/* ET_ECF_AST_FACTORY */
typedef struct S219 T219;

/* [attached] DS_CELL [detachable ET_ECF_SYSTEM] */
typedef struct S220 T220;

/* [attached] TUPLE [[attached] ET_ECF_TARGET] */
typedef struct S221 T221;

/* [attached] PROCEDURE [[attached] TUPLE [[attached] ET_ECF_TARGET]] */
typedef struct S222 T222;

/* [attached] TUPLE [ET_ECF_SYSTEM_PARSER] */
typedef struct S223 T223;

/* [attached] TUPLE [[attached] XM_ELEMENT, detachable XM_POSITION_TABLE, [attached] STRING_8] */
typedef struct S225 T225;

/* [attached] PROCEDURE [[attached] TUPLE [[attached] XM_ELEMENT, detachable XM_POSITION_TABLE, [attached] STRING_8]] */
typedef struct S226 T226;

/* [attached] TUPLE [ET_ECF_SYSTEM_PARSER, detachable STRING_8, [attached] PROCEDURE [[attached] TUPLE [[attached] ET_ECF_TARGET]], [attached] DS_CELL [detachable ET_ECF_SYSTEM]] */
typedef struct S227 T227;

/* [attached] DS_HASH_TABLE [[attached] ET_ECF_LIBRARY, [attached] STRING_8] */
typedef struct S229 T229;

/* [attached] DS_HASH_TABLE [[attached] ET_ECF_DOTNET_ASSEMBLY, [attached] STRING_8] */
typedef struct S230 T230;

/* ET_ECF_STATE */
typedef struct S231 T231;

/* ET_IDENTIFIER */
typedef struct S232 T232;

/* ET_ECF_SYSTEM_CONFIG */
typedef struct S233 T233;

/* ET_ECF_LIBRARY */
typedef struct S236 T236;

/* ET_ADAPTED_LIBRARIES */
typedef struct S237 T237;

/* [attached] TUPLE [[attached] ET_ADAPTED_LIBRARY] */
typedef struct S239 T239;

/* [attached] PROCEDURE [[attached] TUPLE [[attached] ET_ADAPTED_LIBRARY]] */
typedef struct S240 T240;

/* ET_COMPRESSED_POSITION */
typedef struct S241 T241;

/* XM_ATTRIBUTE */
typedef struct S245 T245;

/* [attached] DS_HASH_TABLE [[attached] ET_ECF_SYSTEM_CONFIG, [attached] STRING_8] */
typedef struct S247 T247;

/* [attached] DS_HASH_TABLE [[attached] ET_ECF_TARGET, [attached] STRING_8] */
typedef struct S248 T248;

/* [attached] DS_CELL [detachable ET_ECF_SYSTEM_CONFIG] */
typedef struct S250 T250;

/* [attached] TUPLE [ET_ECF_SYSTEM_PARSER, ET_ECF_INTERNAL_UNIVERSE, [attached] DS_CELL [detachable ET_ECF_SYSTEM_CONFIG]] */
typedef struct S253 T253;

/* ET_ECF_TARGETS */
typedef struct S255 T255;

/* [attached] DS_HASH_TABLE [detachable RX_REGULAR_EXPRESSION, [attached] STRING_8] */
typedef struct S256 T256;

/* [attached] DS_HASH_TABLE_CURSOR [detachable RX_REGULAR_EXPRESSION, [attached] STRING_8] */
typedef struct S257 T257;

/* [attached] DS_CELL [detachable ET_ECF_LIBRARY] */
typedef struct S259 T259;

/* ET_ECF_ADAPTED_LIBRARY */
typedef struct S260 T260;

/* [attached] TUPLE [ET_ECF_SYSTEM_PARSER, ET_ECF_ADAPTED_LIBRARY, [attached] DS_CELL [detachable ET_ECF_LIBRARY]] */
typedef struct S261 T261;

/* ET_ADAPTED_DOTNET_ASSEMBLIES */
typedef struct S264 T264;

/* ET_ECF_DOTNET_ASSEMBLY */
typedef struct S265 T265;

/* ET_ECF_ADAPTED_DOTNET_ASSEMBLY */
typedef struct S267 T267;

/* ET_ECF_ERROR */
typedef struct S268 T268;

/* ET_LACE_AST_FACTORY */
typedef struct S269 T269;

/* [attached] DS_HASH_TABLE [[attached] ET_LACE_CLUSTER, [attached] ET_IDENTIFIER] */
typedef struct S270 T270;

/* YY_FILE_BUFFER */
typedef struct S271 T271;

/* YY_BUFFER */
typedef struct S272 T272;

/* [attached] DS_HASH_SET [[attached] ET_IDENTIFIER] */
typedef struct S273 T273;

/* [attached] SPECIAL [detachable ANY] */
typedef struct S279 T279;

/* [attached] KL_SPECIAL_ROUTINES [detachable ANY] */
typedef struct S280 T280;

/* [attached] SPECIAL [[attached] ET_IDENTIFIER] */
typedef struct S281 T281;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_IDENTIFIER] */
typedef struct S282 T282;

/* ET_LACE_SYSTEM */
typedef struct S283 T283;

/* ET_LACE_CLUSTERS */
typedef struct S285 T285;

/* ET_LACE_CLUSTER */
typedef struct S286 T286;

/* ET_LACE_EXCLUDE */
typedef struct S288 T288;

/* ET_LACE_DOTNET_ASSEMBLY */
typedef struct S290 T290;

/* [attached] SPECIAL [[attached] ET_LACE_SYSTEM] */
typedef struct S291 T291;

/* [attached] SPECIAL [detachable ET_LACE_CLUSTERS] */
typedef struct S292 T292;

/* [attached] SPECIAL [detachable ET_ADAPTED_DOTNET_ASSEMBLIES] */
typedef struct S293 T293;

/* [attached] SPECIAL [detachable ET_IDENTIFIER] */
typedef struct S294 T294;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_LACE_SYSTEM] */
typedef struct S295 T295;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_IDENTIFIER] */
typedef struct S296 T296;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_LACE_CLUSTERS] */
typedef struct S297 T297;

/* [attached] SPECIAL [[attached] ET_LACE_CLUSTERS] */
typedef struct S298 T298;

/* [attached] SPECIAL [[attached] ET_LACE_CLUSTER] */
typedef struct S299 T299;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_LACE_CLUSTERS] */
typedef struct S300 T300;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_LACE_CLUSTER] */
typedef struct S301 T301;

/* [attached] SPECIAL [detachable ET_LACE_EXCLUDE] */
typedef struct S302 T302;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_LACE_EXCLUDE] */
typedef struct S303 T303;

/* [attached] SPECIAL [[attached] ET_LACE_EXCLUDE] */
typedef struct S304 T304;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_LACE_EXCLUDE] */
typedef struct S305 T305;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_ADAPTED_DOTNET_ASSEMBLIES] */
typedef struct S306 T306;

/* [attached] SPECIAL [[attached] ET_ADAPTED_DOTNET_ASSEMBLIES] */
typedef struct S307 T307;

/* [attached] SPECIAL [[attached] ET_LACE_DOTNET_ASSEMBLY] */
typedef struct S308 T308;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_ADAPTED_DOTNET_ASSEMBLIES] */
typedef struct S309 T309;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_LACE_DOTNET_ASSEMBLY] */
typedef struct S310 T310;

/* ET_LACE_DOTNET_GAC_ASSEMBLY */
typedef struct S311 T311;

/* KL_PLATFORM */
typedef struct S312 T312;

/* KL_INTEGER_ROUTINES */
typedef struct S313 T313;

/* [attached] DS_HASH_TABLE [[attached] ET_MASTER_CLASS, [attached] ET_CLASS_NAME] */
typedef struct S314 T314;

/* [attached] DS_ARRAYED_LIST [[attached] ET_DYNAMIC_TYPE] */
typedef struct S315 T315;

/* [attached] TUPLE [[attached] ET_CLASS] */
typedef struct S316 T316;

/* [attached] PROCEDURE [[attached] TUPLE [[attached] ET_CLASS]] */
typedef struct S317 T317;

/* [attached] TUPLE [INTEGER_32] */
typedef struct S318 T318;

/* ET_DYNAMIC_NULL_TYPE_SET_BUILDER */
typedef struct S319 T319;

/* ET_DYNAMIC_TYPE */
typedef struct S320 T320;

/* [attached] TUPLE [ET_DYNAMIC_SYSTEM] */
typedef struct S321 T321;

/* ET_DYNAMIC_FEATURE */
typedef struct S332 T332;

/* ET_ACTUAL_PARAMETER_LIST */
typedef struct S333 T333;

/* ET_EXTERNAL_FUNCTION */
typedef struct S344 T344;

/* ET_TUPLE_TYPE */
typedef struct S345 T345;

/* ET_FORMAL_ARGUMENT_LIST */
typedef struct S347 T347;

/* [attached] ARRAY [[attached] ET_TYPE] */
typedef struct S348 T348;

/* [attached] SPECIAL [[attached] ET_TYPE] */
typedef struct S349 T349;

/* ET_FORMAL_ARGUMENT */
typedef struct S350 T350;

/* ET_QUERY_LIST */
typedef struct S351 T351;

/* ET_PROCEDURE_LIST */
typedef struct S352 T352;

/* DT_SHARED_SYSTEM_CLOCK */
typedef struct S353 T353;

/* DT_SYSTEM_CLOCK */
typedef struct S354 T354;

/* DT_DATE_TIME */
typedef struct S355 T355;

/* ET_EIFFEL_PREPARSER */
typedef struct S357 T357;

/* ET_MASTER_CLASS_CHECKER */
typedef struct S358 T358;

/* ET_EIFFEL_PARSER */
typedef struct S359 T359;

/* ET_PROVIDER_CHECKER */
typedef struct S360 T360;

/* ET_ANCESTOR_BUILDER */
typedef struct S361 T361;

/* ET_FEATURE_FLATTENER */
typedef struct S362 T362;

/* ET_INTERFACE_CHECKER */
typedef struct S363 T363;

/* ET_IMPLEMENTATION_CHECKER */
typedef struct S364 T364;

/* [attached] DS_ARRAYED_LIST [[attached] ET_CLASS] */
typedef struct S365 T365;

/* [attached] TUPLE [[attached] DS_ARRAYED_LIST [[attached] ET_CLASS]] */
typedef struct S366 T366;

/* ET_AST_NULL_PROCESSOR */
typedef struct S367 T367;

/* [attached] DS_ARRAYED_LIST [[attached] INTEGER_32] */
typedef struct S368 T368;

/* ET_AST_FACTORY */
typedef struct S369 T369;

/* ET_DOTNET_ASSEMBLY_CLASSIC_CONSUMER */
typedef struct S371 T371;

/* DT_DATE_TIME_DURATION */
typedef struct S372 T372;

/* ET_DYNAMIC_TYPE_SET_LIST */
typedef struct S373 T373;

/* KL_STRING_OUTPUT_STREAM */
typedef struct S374 T374;

/* [attached] DS_ARRAYED_LIST [[attached] ET_IDENTIFIER] */
typedef struct S375 T375;

/* [attached] DS_ARRAYED_LIST [detachable ET_DYNAMIC_TYPE] */
typedef struct S376 T376;

/* ET_DYNAMIC_TYPE_HASH_LIST */
typedef struct S377 T377;

/* ET_DYNAMIC_STANDALONE_TYPE_SET */
typedef struct S378 T378;

/* [attached] DS_ARRAYED_STACK [[attached] ET_EXPRESSION] */
typedef struct S379 T379;

/* [attached] DS_ARRAYED_LIST [[attached] ET_EXPRESSION] */
typedef struct S380 T380;

/* [attached] DS_HASH_TABLE [[attached] ET_DYNAMIC_TYPE, [attached] INTEGER_32] */
typedef struct S381 T381;

/* ET_DYNAMIC_STANDALONE_TYPE_SET_LIST */
typedef struct S382 T382;

/* [attached] DS_HASH_SET [[attached] ET_DYNAMIC_TYPE] */
typedef struct S383 T383;

/* [attached] DS_HASH_TABLE [[attached] ET_DYNAMIC_STANDALONE_TYPE_SET, [attached] ET_DYNAMIC_TYPE] */
typedef struct S384 T384;

/* [attached] DS_ARRAYED_LIST [[attached] ET_OBJECT_TEST] */
typedef struct S385 T385;

/* [attached] DS_ARRAYED_LIST [[attached] ET_OBJECT_EQUALITY_EXPRESSION] */
typedef struct S386 T386;

/* [attached] DS_ARRAYED_LIST [[attached] ET_EQUALITY_EXPRESSION] */
typedef struct S387 T387;

/* [attached] DS_ARRAYED_LIST [[attached] ET_AGENT] */
typedef struct S388 T388;

/* ET_EXPRESSION */
typedef struct S389 T389;

/* ET_CURRENT */
typedef struct S390 T390;

/* ET_ACTUAL_ARGUMENT_LIST */
typedef struct S391 T391;

/* ET_QUALIFIED_CALL_INSTRUCTION */
typedef struct S392 T392;

/* ET_QUALIFIED_CALL_EXPRESSION */
typedef struct S396 T396;

/* ET_UNQUALIFIED_CALL_EXPRESSION */
typedef struct S397 T397;

/* [attached] DS_ARRAYED_LIST [detachable ET_QUALIFIED_CALL_EXPRESSION] */
typedef struct S398 T398;

/* ET_MANIFEST_TUPLE */
typedef struct S399 T399;

/* [attached] DS_HASH_SET [[attached] ET_DYNAMIC_TUPLE_TYPE] */
typedef struct S400 T400;

/* [attached] DS_HASH_TABLE [[attached] INTEGER_32, [attached] ET_FEATURE] */
typedef struct S401 T401;

/* [attached] ARRAY [[attached] INTEGER_32] */
typedef struct S402 T402;

/* [attached] DS_HASH_TABLE [[attached] ET_CONSTANT, [attached] ET_FEATURE] */
typedef struct S403 T403;

/* [attached] DS_HASH_TABLE [[attached] ET_DYNAMIC_TYPE, [attached] ET_INLINE_CONSTANT] */
typedef struct S404 T404;

/* [attached] DS_HASH_TABLE [detachable ET_DYNAMIC_FEATURE, [attached] ET_DYNAMIC_TYPE] */
typedef struct S405 T405;

/* [attached] DS_ARRAYED_LIST [[attached] ET_DYNAMIC_FEATURE] */
typedef struct S406 T406;

/* [attached] DS_HASH_TABLE [[attached] BOOLEAN, [attached] STRING_8] */
typedef struct S407 T407;

/* ET_IDENTIFIER_TESTER */
typedef struct S408 T408;

/* RX_PCRE_REGULAR_EXPRESSION */
typedef struct S410 T410;

/* KL_TEXT_OUTPUT_FILE */
typedef struct S411 T411;

/* ET_DYNAMIC_TUPLE_TYPE */
typedef struct S412 T412;

/* UT_CANNOT_WRITE_TO_FILE_ERROR */
typedef struct S413 T413;

/* ET_DYNAMIC_FEATURE_LIST */
typedef struct S414 T414;

/* ET_DYNAMIC_PRECURSOR */
typedef struct S416 T416;

/* ET_DYNAMIC_PRECURSOR_LIST */
typedef struct S418 T418;

/* [attached] DS_HASH_TOPOLOGICAL_SORTER [[attached] ET_DYNAMIC_TYPE] */
typedef struct S419 T419;

/* ET_RESULT */
typedef struct S422 T422;

/* ET_DYNAMIC_QUALIFIED_QUERY_CALL */
typedef struct S424 T424;

/* ET_DYNAMIC_QUALIFIED_PROCEDURE_CALL */
typedef struct S429 T429;

/* ET_OBJECT_TEST */
typedef struct S431 T431;

/* ET_OBJECT_EQUALITY_EXPRESSION */
typedef struct S432 T432;

/* ET_EQUALITY_EXPRESSION */
typedef struct S433 T433;

/* ET_DYNAMIC_SPECIAL_TYPE */
typedef struct S434 T434;

/* [attached] DS_QUICK_SORTER [[attached] ET_DYNAMIC_TYPE] */
typedef struct S438 T438;

/* ET_DYNAMIC_TYPE_COMPARATOR_BY_ID */
typedef struct S439 T439;

/* ET_CREATE_EXPRESSION */
typedef struct S444 T444;

/* ET_QUALIFIED_CALL */
typedef struct S445 T445;

/* [attached] ARRAY [[attached] STRING_8] */
typedef struct S449 T449;

/* KL_DIRECTORY */
typedef struct S451 T451;

/* UT_TEMPLATE_EXPANDER */
typedef struct S452 T452;

/* ET_EXTENDED_ATTRIBUTE */
typedef struct S454 T454;

/* ET_COMPOUND */
typedef struct S456 T456;

/* [attached] DS_QUICK_SORTER [[attached] INTEGER_32] */
typedef struct S464 T464;

/* ET_IMPLICIT_TYPE_MARK */
typedef struct S472 T472;

/* ET_SYMBOL */
typedef struct S474 T474;

/* [attached] PROCEDURE [[attached] TUPLE] */
typedef struct S475 T475;

/* [attached] TUPLE [ET_C_GENERATOR, ET_IDENTIFIER, ET_CURRENT, ET_DYNAMIC_TYPE, BOOLEAN] */
typedef struct S476 T476;

/* [attached] TUPLE [ET_C_GENERATOR, ET_IDENTIFIER, ET_RESULT, ET_DYNAMIC_TYPE, BOOLEAN] */
typedef struct S477 T477;

/* [attached] TUPLE [ET_C_GENERATOR, ET_DYNAMIC_FEATURE, ET_RESULT, ET_DYNAMIC_TYPE, BOOLEAN] */
typedef struct S478 T478;

/* [attached] TUPLE [ET_C_GENERATOR, INTEGER_32, ET_RESULT, ET_DYNAMIC_TYPE, BOOLEAN] */
typedef struct S479 T479;

/* UT_INTEGER_FORMATTER */
typedef struct S480 T480;

/* STD_FILES */
typedef struct S485 T485;

/* [attached] TUPLE [ET_C_GENERATOR, INTEGER_32, ET_EXPRESSION, ET_DYNAMIC_TYPE, BOOLEAN] */
typedef struct S487 T487;

/* [attached] TUPLE [ET_C_GENERATOR, ET_EXPRESSION, ET_DYNAMIC_TYPE] */
typedef struct S490 T490;

/* [attached] TUPLE [ET_C_GENERATOR, ET_DYNAMIC_FEATURE, ET_DYNAMIC_TYPE, BOOLEAN] */
typedef struct S491 T491;

/* ET_CONSTANT_ATTRIBUTE */
typedef struct S492 T492;

/* ET_UNIQUE_ATTRIBUTE */
typedef struct S494 T494;

/* ET_REGULAR_INTEGER_CONSTANT */
typedef struct S495 T495;

/* ET_ATTRIBUTE */
typedef struct S496 T496;

/* ET_LIKE_CURRENT */
typedef struct S498 T498;

/* NATIVE_STRING */
typedef struct S500 T500;

/* [attached] DS_ARRAYED_LIST [[attached] ET_SYSTEM_PROCESSOR] */
typedef struct S501 T501;

/* WORKER_THREAD */
typedef struct S502 T502;

/* [attached] TUPLE [ET_SYSTEM_PROCESSOR, [attached] DS_ARRAYED_LIST [[attached] ET_CLASS]] */
typedef struct S503 T503;

/* ET_FEATURE_CHECKER */
typedef struct S505 T505;

/* [attached] DS_CELL [[attached] INTEGER_32] */
typedef struct S506 T506;

/* [attached] DS_HASH_TABLE [[attached] INTEGER_32, [attached] ET_DYNAMIC_TYPE] */
typedef struct S507 T507;

/* ET_TYPE_CHECKER */
typedef struct S510 T510;

/* [attached] DS_ARRAYED_LIST [[attached] ET_INLINE_AGENT] */
typedef struct S512 T512;

/* [attached] DS_ARRAYED_LIST [[attached] DS_ARRAYED_LIST [[attached] ET_PROCEDURE]] */
typedef struct S513 T513;

/* [attached] DS_ARRAYED_LIST [[attached] DS_ARRAYED_LIST [[attached] ET_QUERY]] */
typedef struct S514 T514;

/* [attached] DS_ARRAYED_LIST [[attached] DS_ARRAYED_LIST [[attached] ET_FEATURE]] */
typedef struct S515 T515;

/* [attached] DS_ARRAYED_LIST [[attached] ET_NESTED_TYPE_CONTEXT] */
typedef struct S516 T516;

/* ET_NESTED_TYPE_CONTEXT */
typedef struct S517 T517;

/* [attached] DS_HASH_TABLE [[attached] ET_NESTED_TYPE_CONTEXT, [attached] ET_NAMED_OBJECT_TEST] */
typedef struct S518 T518;

/* ET_OBJECT_TEST_SCOPE */
typedef struct S519 T519;

/* ET_OBJECT_TEST_SCOPE_BUILDER */
typedef struct S520 T520;

/* [attached] DS_HASH_TABLE [[attached] ET_NESTED_TYPE_CONTEXT, [attached] ET_ACROSS_COMPONENT] */
typedef struct S521 T521;

/* ET_ACROSS_CURSOR_SCOPE */
typedef struct S522 T522;

/* ET_ATTACHMENT_SCOPE */
typedef struct S523 T523;

/* ET_ATTACHMENT_SCOPE_BUILDER */
typedef struct S524 T524;

/* [attached] DS_ARRAYED_LIST [[attached] ET_ATTACHMENT_SCOPE] */
typedef struct S525 T525;

/* [attached] DS_ARRAYED_LIST [[attached] ET_INDEXING_TERM] */
typedef struct S526 T526;

/* attached DS_HASH_TABLE [[attached] ET_DYNAMIC_TYPE_SET, [attached] ET_DYNAMIC_TYPE] */
typedef struct S527 T527;

/* ET_PRECONDITIONS */
typedef struct S529 T529;

/* [attached] DS_HASH_TABLE [[attached] ET_ASSERTIONS, [attached] ET_FEATURE] */
typedef struct S530 T530;

/* ET_FEATURE_LIST */
typedef struct S533 T533;

/* ET_UNKNOWN_GROUP */
typedef struct S536 T536;

/* ET_BASE_TYPE_LIST */
typedef struct S538 T538;

/* ET_KEYWORD */
typedef struct S539 T539;

/* ET_CLASS_CODES */
typedef struct S540 T540;

/* PATH */
typedef struct S542 T542;

/* [attached] DS_HASH_TABLE [[attached] ET_IDENTIFIER, [attached] STRING_8] */
typedef struct S543 T543;

/* [attached] DS_HASH_SET_CURSOR [[attached] STRING_8] */
typedef struct S544 T544;

/* [attached] TYPE [[attached] STRING_8] */
#define T545 EIF_TYPE_OBJ

/* ARGUMENTS_32 */
typedef struct S547 T547;

/* ET_SYSTEM_ERROR */
typedef struct S548 T548;

/* ET_INTERNAL_ERROR */
typedef struct S549 T549;

/* ET_XACE_EXTERNALS */
typedef struct S551 T551;

/* [attached] DS_LINKED_LIST_CURSOR [[attached] STRING_8] */
typedef struct S552 T552;

/* ET_XACE_OPTIONS */
typedef struct S553 T553;

/* ET_XACE_TARGET */
typedef struct S554 T554;

/* ET_XACE_TARGETS */
typedef struct S557 T557;

/* ET_XACE_CLUSTERS */
typedef struct S559 T559;

/* ET_XACE_MOUNTED_LIBRARIES */
typedef struct S560 T560;

/* ET_XACE_CLUSTER */
typedef struct S564 T564;

/* [attached] DS_LINKED_LIST_CURSOR [[attached] XM_ELEMENT_NODE] */
typedef struct S569 T569;

/* ET_XACE_MOUNTED_LIBRARY */
typedef struct S571 T571;

/* ET_XACE_ELEMENT_OBSOLETED_BY_ELEMENT_WARNING */
typedef struct S575 T575;

/* ET_XACE_MULTIPLE_LIBRARY_PREFIX_ERROR */
typedef struct S576 T576;

/* ET_ECF_OPTIONS */
typedef struct S581 T581;

/* XM_NAMESPACE */
typedef struct S583 T583;

/* [attached] KL_ARRAY_ROUTINES [[attached] INTEGER_32] */
typedef struct S584 T584;

/* ET_DYNAMIC_PROCEDURE_TYPE */
typedef struct S586 T586;

/* ET_DYNAMIC_FUNCTION_TYPE */
typedef struct S587 T587;

/* UT_CONFIG_PARSER */
typedef struct S588 T588;

/* [attached] KL_COMPARABLE_COMPARATOR [[attached] INTEGER_32] */
typedef struct S589 T589;

/* ET_DO_PROCEDURE */
typedef struct S592 T592;

/* ET_DYNAMIC_PUSH_TYPE_SET */
typedef struct S595 T595;

/* ET_DEFERRED_PROCEDURE */
typedef struct S596 T596;

/* UTF_CONVERTER */
typedef struct S598 T598;
extern T0* GE_boxed598(T598 a1);
typedef struct Sb598 Tb598;

/* [attached] CELL [[attached] INTEGER_32] */
typedef struct S599 T599;

/* [attached] HASH_TABLE [[attached] NATIVE_STRING, [attached] IMMUTABLE_STRING_32] */
typedef struct S601 T601;

/* KL_PATHNAME */
typedef struct S603 T603;

/* EXCEPTIONS */
typedef struct S604 T604;

/* READABLE_STRING_32 */
typedef struct S605 T605;

/* [attached] SPECIAL [[attached] NATURAL_8] */
typedef struct S606 T606;

/* TYPED_POINTER [[attached] SPECIAL [[attached] NATURAL_8]] */
typedef struct S607 T607;
extern T0* GE_boxed607(T607 a1);
typedef struct Sb607 Tb607;

/* [attached] DS_ARRAYED_LIST_CURSOR [[attached] AP_OPTION] */
typedef struct S608 T608;

/* [attached] SPECIAL [[attached] AP_OPTION] */
typedef struct S611 T611;

/* [attached] KL_SPECIAL_ROUTINES [[attached] AP_OPTION] */
typedef struct S612 T612;

/* [attached] DS_ARRAYED_LIST_CURSOR [[attached] AP_ALTERNATIVE_OPTIONS_LIST] */
typedef struct S614 T614;

/* [attached] SPECIAL [[attached] AP_ALTERNATIVE_OPTIONS_LIST] */
typedef struct S615 T615;

/* [attached] KL_SPECIAL_ROUTINES [[attached] AP_ALTERNATIVE_OPTIONS_LIST] */
typedef struct S616 T616;

/* [attached] DS_ARRAYED_LIST_CURSOR [[attached] STRING_8] */
typedef struct S618 T618;

/* [attached] SPECIAL [detachable STRING_8] */
typedef struct S620 T620;

/* [attached] KL_SPECIAL_ROUTINES [detachable STRING_8] */
typedef struct S621 T621;

/* attached DS_LINKABLE [[attached] STRING_8] */
typedef struct S623 T623;

/* attached DS_LINKABLE [[attached] BOOLEAN] */
typedef struct S626 T626;

/* [attached] DS_LINKED_LIST_CURSOR [[attached] BOOLEAN] */
typedef struct S627 T627;

/* attached DS_LINKABLE [[attached] INTEGER_32] */
typedef struct S629 T629;

/* [attached] DS_LINKED_LIST_CURSOR [[attached] INTEGER_32] */
typedef struct S630 T630;

/* XM_NAMESPACE_RESOLVER */
typedef struct S632 T632;

/* [attached] ARRAY [[attached] XM_CALLBACKS_FILTER] */
typedef struct S633 T633;

/* [attached] SPECIAL [[attached] XM_CALLBACKS_FILTER] */
typedef struct S634 T634;

/* XM_CALLBACKS_NULL */
typedef struct S635 T635;

/* XM_POSITION_TABLE */
typedef struct S637 T637;

/* [attached] DS_HASH_SET [[attached] XM_NAMESPACE] */
typedef struct S638 T638;

/* UT_COUNTER */
typedef struct S642 T642;

/* [attached] KL_AGENT_ROUTINES [[attached] ET_CLASS] */
typedef struct S643 T643;

/* [attached] TUPLE [UT_COUNTER] */
typedef struct S644 T644;

/* [attached] TUPLE [[attached] KL_AGENT_ROUTINES [[attached] ET_CLASS], [attached] PROCEDURE [[attached] TUPLE]] */
typedef struct S645 T645;

/* [attached] DS_HASH_SET [[attached] ET_DOTNET_ASSEMBLY] */
typedef struct S647 T647;

/* ET_DOTNET_ASSEMBLIES */
typedef struct S648 T648;

/* ET_DOTNET_ASSEMBLY */
typedef struct S649 T649;

/* [attached] TUPLE [[attached] ET_DOTNET_ASSEMBLY] */
typedef struct S650 T650;

/* [attached] PREDICATE [[attached] TUPLE [[attached] ET_DOTNET_ASSEMBLY]] */
typedef struct S651 T651;

/* [attached] PROCEDURE [[attached] TUPLE [[attached] ET_DOTNET_ASSEMBLY]] */
typedef struct S652 T652;

/* [attached] TUPLE [[attached] DS_HASH_SET [[attached] ET_DOTNET_ASSEMBLY]] */
typedef struct S653 T653;

/* [attached] TUPLE [[attached] ET_INTERNAL_UNIVERSE] */
typedef struct S654 T654;

/* [attached] PROCEDURE [[attached] TUPLE [[attached] ET_INTERNAL_UNIVERSE]] */
typedef struct S655 T655;

/* [attached] TUPLE [[attached] PROCEDURE [[attached] TUPLE [[attached] ET_DOTNET_ASSEMBLY]], [attached] PREDICATE [[attached] TUPLE [[attached] ET_DOTNET_ASSEMBLY]]] */
typedef struct S656 T656;

/* [attached] TUPLE [ET_DOTNET_ASSEMBLIES] */
typedef struct S657 T657;

/* [attached] TUPLE [ET_SYSTEM_PROCESSOR] */
typedef struct S658 T658;

/* ET_LIBRARY */
typedef struct S659 T659;

/* [attached] TUPLE [[attached] ET_LIBRARY] */
typedef struct S660 T660;

/* [attached] PROCEDURE [[attached] TUPLE [[attached] ET_LIBRARY]] */
typedef struct S661 T661;

/* [attached] TUPLE [[attached] ET_UNIVERSE] */
typedef struct S662 T662;

/* [attached] PROCEDURE [[attached] TUPLE [[attached] ET_UNIVERSE]] */
typedef struct S663 T663;

/* [attached] TUPLE [[attached] PROCEDURE [[attached] TUPLE [[attached] ET_CLASS]]] */
typedef struct S664 T664;

/* [attached] TUPLE [[attached] PROCEDURE [[attached] TUPLE [[attached] ET_CLASS]], detachable FUNCTION [[attached] TUPLE, [attached] BOOLEAN]] */
typedef struct S665 T665;

/* [attached] DS_ARRAYED_LIST [[attached] ET_XACE_MOUNTED_LIBRARY] */
typedef struct S667 T667;

/* ET_XACE_LIBRARY_CONFIG */
typedef struct S668 T668;

/* [attached] TUPLE [[attached] ET_XACE_CLUSTER] */
typedef struct S669 T669;

/* [attached] PROCEDURE [[attached] TUPLE [[attached] ET_XACE_CLUSTER]] */
typedef struct S670 T670;

/* [attached] TUPLE [ET_XACE_CLUSTERS] */
typedef struct S671 T671;

/* ET_MASTER_CLASS */
typedef struct S672 T672;

/* [attached] TUPLE [[attached] ET_CLUSTER] */
typedef struct S674 T674;

/* [attached] PROCEDURE [[attached] TUPLE [[attached] ET_CLUSTER]] */
typedef struct S675 T675;

/* [attached] TUPLE [ET_XACE_SYSTEM] */
typedef struct S676 T676;

/* [attached] DS_HASH_SET [[attached] ET_INTERNAL_UNIVERSE] */
typedef struct S677 T677;

/* [attached] TUPLE [ET_AST_PROCESSOR] */
typedef struct S678 T678;

/* [attached] TUPLE [[attached] ET_MASTER_CLASS] */
typedef struct S679 T679;

/* [attached] PROCEDURE [[attached] TUPLE [[attached] ET_MASTER_CLASS]] */
typedef struct S680 T680;

/* [attached] TUPLE [[attached] ET_ADAPTED_DOTNET_ASSEMBLY] */
typedef struct S681 T681;

/* [attached] PROCEDURE [[attached] TUPLE [[attached] ET_ADAPTED_DOTNET_ASSEMBLY]] */
typedef struct S682 T682;

/* ET_ANCESTORS_STATUS_CHECKER */
typedef struct S683 T683;

/* ET_FLATTENING_STATUS_CHECKER */
typedef struct S684 T684;

/* ET_INTERFACE_STATUS_CHECKER */
typedef struct S685 T685;

/* ET_IMPLEMENTATION_STATUS_CHECKER */
typedef struct S686 T686;

/* [attached] PREDICATE [[attached] TUPLE [[attached] ET_CLASS]] */
typedef struct S687 T687;

/* [attached] TUPLE [ET_ANCESTORS_STATUS_CHECKER] */
typedef struct S689 T689;

/* [attached] TUPLE [ET_FLATTENING_STATUS_CHECKER] */
typedef struct S690 T690;

/* [attached] TUPLE [ET_INTERFACE_STATUS_CHECKER] */
typedef struct S691 T691;

/* [attached] TUPLE [ET_IMPLEMENTATION_STATUS_CHECKER] */
typedef struct S692 T692;

/* [attached] TUPLE [BOOLEAN] */
typedef struct S693 T693;

/* [attached] DS_HASH_SET [[attached] ET_UNIVERSE] */
typedef struct S694 T694;

/* ET_CLASS_NAME_TESTER */
typedef struct S696 T696;

/* ET_PARENT */
typedef struct S698 T698;

/* ET_PARENT_LIST */
typedef struct S699 T699;

/* ET_CLIENT_LIST */
typedef struct S700 T700;

/* ET_BUILTIN_CONVERT_FEATURE */
typedef struct S701 T701;

/* [attached] TUPLE [[attached] PROCEDURE [[attached] TUPLE [[attached] ET_CLUSTER]]] */
typedef struct S706 T706;

/* [attached] TUPLE [[attached] DS_HASH_SET [[attached] ET_INTERNAL_UNIVERSE]] */
typedef struct S712 T712;

/* [attached] TUPLE [[attached] PROCEDURE [[attached] TUPLE [[attached] ET_MASTER_CLASS]]] */
typedef struct S713 T713;

/* [attached] TUPLE [[attached] PROCEDURE [[attached] TUPLE [[attached] ET_CLASS]], [attached] FUNCTION [[attached] TUPLE [[attached] ET_CLASS], [attached] BOOLEAN]] */
typedef struct S714 T714;

/* [attached] TUPLE [[attached] DS_HASH_SET [[attached] ET_UNIVERSE]] */
typedef struct S715 T715;

/* ET_CLIENT */
typedef struct S716 T716;

/* ET_ATTACHMENT_SEPARATE_KEYWORDS */
typedef struct S717 T717;

/* ET_NONE_GROUP */
typedef struct S723 T723;

/* ET_UNFOLDED_EMPTY_TUPLE_ACTUAL_PARAMETERS */
typedef struct S724 T724;

/* [attached] DS_HASH_TABLE_CURSOR [[attached] ET_XACE_LIBRARY_CONFIG, [attached] STRING_8] */
typedef struct S729 T729;

/* [attached] SPECIAL [[attached] ET_XACE_LIBRARY_CONFIG] */
typedef struct S730 T730;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_XACE_LIBRARY_CONFIG] */
typedef struct S731 T731;

/* XM_EIFFEL_SCANNER */
typedef struct S732 T732;

/* XM_DEFAULT_POSITION */
typedef struct S733 T733;

/* [attached] DS_BILINKED_LIST [[attached] XM_POSITION] */
typedef struct S736 T736;

/* [attached] DS_LINKED_STACK [[attached] XM_EIFFEL_SCANNER] */
typedef struct S737 T737;

/* [attached] DS_HASH_TABLE [[attached] XM_EIFFEL_ENTITY_DEF, [attached] STRING_8] */
typedef struct S738 T738;

/* XM_NULL_EXTERNAL_RESOLVER */
typedef struct S740 T740;

/* XM_DTD_CALLBACKS_NULL */
typedef struct S742 T742;

/* [attached] KL_SPECIAL_ROUTINES [[attached] XM_EIFFEL_PARSER_NAME] */
typedef struct S743 T743;

/* XM_EIFFEL_PARSER_NAME */
typedef struct S744 T744;

/* [attached] SPECIAL [[attached] XM_EIFFEL_PARSER_NAME] */
typedef struct S745 T745;

/* [attached] KL_SPECIAL_ROUTINES [[attached] DS_HASH_SET [[attached] XM_EIFFEL_PARSER_NAME]] */
typedef struct S746 T746;

/* [attached] DS_HASH_SET [[attached] XM_EIFFEL_PARSER_NAME] */
typedef struct S747 T747;

/* [attached] SPECIAL [[attached] DS_HASH_SET [[attached] XM_EIFFEL_PARSER_NAME]] */
typedef struct S748 T748;

/* [attached] KL_SPECIAL_ROUTINES [[attached] XM_DTD_EXTERNAL_ID] */
typedef struct S749 T749;

/* XM_DTD_EXTERNAL_ID */
typedef struct S750 T750;

/* [attached] SPECIAL [[attached] XM_DTD_EXTERNAL_ID] */
typedef struct S751 T751;

/* [attached] KL_SPECIAL_ROUTINES [[attached] XM_DTD_ELEMENT_CONTENT] */
typedef struct S752 T752;

/* XM_DTD_ELEMENT_CONTENT */
typedef struct S753 T753;

/* [attached] SPECIAL [[attached] XM_DTD_ELEMENT_CONTENT] */
typedef struct S754 T754;

/* [attached] KL_SPECIAL_ROUTINES [[attached] DS_BILINKED_LIST [[attached] XM_DTD_ATTRIBUTE_CONTENT]] */
typedef struct S755 T755;

/* [attached] DS_BILINKED_LIST [[attached] XM_DTD_ATTRIBUTE_CONTENT] */
typedef struct S756 T756;

/* [attached] SPECIAL [[attached] DS_BILINKED_LIST [[attached] XM_DTD_ATTRIBUTE_CONTENT]] */
typedef struct S757 T757;

/* [attached] KL_SPECIAL_ROUTINES [[attached] XM_DTD_ATTRIBUTE_CONTENT] */
typedef struct S758 T758;

/* XM_DTD_ATTRIBUTE_CONTENT */
typedef struct S759 T759;

/* [attached] SPECIAL [[attached] XM_DTD_ATTRIBUTE_CONTENT] */
typedef struct S760 T760;

/* [attached] KL_SPECIAL_ROUTINES [[attached] DS_BILINKED_LIST [[attached] STRING_8]] */
typedef struct S761 T761;

/* [attached] DS_BILINKED_LIST [[attached] STRING_8] */
typedef struct S762 T762;

/* [attached] SPECIAL [[attached] DS_BILINKED_LIST [[attached] STRING_8]] */
typedef struct S763 T763;

/* [attached] KL_SPECIAL_ROUTINES [[attached] BOOLEAN] */
typedef struct S764 T764;

/* [attached] KL_SPECIAL_ROUTINES [[attached] XM_EIFFEL_DECLARATION] */
typedef struct S765 T765;

/* XM_EIFFEL_DECLARATION */
typedef struct S766 T766;

/* [attached] SPECIAL [[attached] XM_EIFFEL_DECLARATION] */
typedef struct S767 T767;

/* XM_EIFFEL_ENTITY_DEF */
typedef struct S772 T772;

/* XM_EIFFEL_SCANNER_DTD */
typedef struct S773 T773;

/* XM_EIFFEL_PE_ENTITY_DEF */
typedef struct S775 T775;

/* [attached] DS_HASH_TABLE_CURSOR [[attached] ET_ECF_LIBRARY, [attached] STRING_8] */
typedef struct S777 T777;

/* [attached] SPECIAL [[attached] ET_ECF_LIBRARY] */
typedef struct S778 T778;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_ECF_LIBRARY] */
typedef struct S783 T783;

/* [attached] SPECIAL [[attached] ET_ECF_DOTNET_ASSEMBLY] */
typedef struct S784 T784;

/* [attached] DS_HASH_TABLE_CURSOR [[attached] ET_ECF_DOTNET_ASSEMBLY, [attached] STRING_8] */
typedef struct S789 T789;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_ECF_DOTNET_ASSEMBLY] */
typedef struct S790 T790;

/* ET_CLUSTERS */
typedef struct S791 T791;

/* [attached] DS_ARRAYED_LIST [[attached] ET_ADAPTED_LIBRARY] */
typedef struct S792 T792;

/* [attached] DS_HASH_SET [[attached] ET_LIBRARY] */
typedef struct S793 T793;

/* [attached] TUPLE [[attached] DS_HASH_SET [[attached] ET_LIBRARY]] */
typedef struct S794 T794;

/* [attached] SPECIAL [[attached] ET_ECF_SYSTEM_CONFIG] */
typedef struct S796 T796;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_ECF_SYSTEM_CONFIG] */
typedef struct S798 T798;

/* [attached] DS_HASH_TABLE_CURSOR [[attached] ET_ECF_SYSTEM_CONFIG, [attached] STRING_8] */
typedef struct S799 T799;

/* [attached] DS_HASH_TABLE_CURSOR [[attached] ET_ECF_TARGET, [attached] STRING_8] */
typedef struct S800 T800;

/* [attached] SPECIAL [[attached] ET_ECF_TARGET] */
typedef struct S801 T801;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_ECF_TARGET] */
typedef struct S803 T803;

/* attached DS_SPARSE_TABLE_KEYS [detachable RX_REGULAR_EXPRESSION, [attached] STRING_8] */
typedef struct S807 T807;

/* [attached] KL_SPECIAL_ROUTINES [detachable RX_REGULAR_EXPRESSION] */
typedef struct S808 T808;

/* [attached] SPECIAL [detachable RX_REGULAR_EXPRESSION] */
typedef struct S809 T809;

/* [attached] DS_ARRAYED_LIST [[attached] ET_ADAPTED_DOTNET_ASSEMBLY] */
typedef struct S810 T810;

/* [attached] KL_EQUALITY_TESTER [[attached] ET_IDENTIFIER] */
typedef struct S812 T812;

/* [attached] DS_HASH_TABLE_CURSOR [[attached] ET_LACE_CLUSTER, [attached] ET_IDENTIFIER] */
typedef struct S815 T815;

/* [attached] DS_HASH_SET_CURSOR [[attached] ET_IDENTIFIER] */
typedef struct S816 T816;

/* [attached] TYPE [[attached] ET_IDENTIFIER] */
#define T818 EIF_TYPE_OBJ

/* [attached] TUPLE [ET_LACE_SYSTEM] */
typedef struct S819 T819;

/* [attached] DS_ARRAYED_LIST [[attached] ET_LACE_CLUSTER] */
typedef struct S820 T820;

/* [attached] TUPLE [[attached] ET_LACE_CLUSTER] */
typedef struct S821 T821;

/* [attached] DS_HASH_TABLE_CURSOR [[attached] ET_MASTER_CLASS, [attached] ET_CLASS_NAME] */
typedef struct S828 T828;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_MASTER_CLASS] */
typedef struct S829 T829;

/* [attached] SPECIAL [[attached] ET_MASTER_CLASS] */
typedef struct S830 T830;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_CLASS_NAME] */
typedef struct S831 T831;

/* [attached] SPECIAL [[attached] ET_CLASS_NAME] */
typedef struct S832 T832;

/* [attached] SPECIAL [[attached] ET_DYNAMIC_TYPE] */
typedef struct S833 T833;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_DYNAMIC_TYPE] */
typedef struct S834 T834;

/* [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_DYNAMIC_TYPE] */
typedef struct S835 T835;

/* [attached] SPECIAL [[attached] ET_ACTUAL_PARAMETER_ITEM] */
typedef struct S845 T845;

/* ET_BRACKET_SYMBOL */
typedef struct S846 T846;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_ACTUAL_PARAMETER_ITEM] */
typedef struct S848 T848;

/* C_DATE */
typedef struct S850 T850;

/* KL_STDIN_FILE */
typedef struct S851 T851;

/* [attached] DS_ARRAYED_STACK [[attached] INTEGER_32] */
typedef struct S853 T853;

/* [attached] DS_ARRAYED_STACK [detachable ET_FORMAL_ARGUMENT_LIST] */
typedef struct S854 T854;

/* [attached] DS_ARRAYED_STACK [detachable ET_LOCAL_VARIABLE_LIST] */
typedef struct S855 T855;

/* [attached] DS_ARRAYED_STACK [detachable ET_KEYWORD] */
typedef struct S856 T856;

/* [attached] DS_ARRAYED_STACK [detachable ET_SYMBOL] */
typedef struct S857 T857;

/* [attached] DS_ARRAYED_STACK [detachable ET_OBJECT_TEST_LIST] */
typedef struct S858 T858;

/* [attached] DS_ARRAYED_STACK [[attached] ET_OBJECT_TEST_LIST] */
typedef struct S859 T859;

/* [attached] DS_ARRAYED_STACK [detachable ET_ACROSS_COMPONENT_LIST] */
typedef struct S860 T860;

/* [attached] DS_ARRAYED_STACK [[attached] ET_ACROSS_COMPONENT_LIST] */
typedef struct S861 T861;

/* [attached] DS_ARRAYED_LIST [[attached] ET_ASSERTION_ITEM] */
typedef struct S862 T862;

/* [attached] DS_ARRAYED_LIST [[attached] ET_QUERY] */
typedef struct S863 T863;

/* [attached] DS_ARRAYED_LIST [[attached] ET_PROCEDURE] */
typedef struct S864 T864;

/* [attached] DS_ARRAYED_LIST [detachable ET_CONSTRAINT_TYPE] */
typedef struct S865 T865;

/* [attached] DS_HASH_SET [[attached] ET_NAMED_CLASS] */
typedef struct S866 T866;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_KEYWORD] */
typedef struct S867 T867;

/* [attached] SPECIAL [detachable ET_KEYWORD] */
typedef struct S869 T869;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_AGENT_KEYWORD] */
typedef struct S870 T870;

/* [attached] SPECIAL [detachable ET_AGENT_KEYWORD] */
typedef struct S872 T872;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_PRECURSOR_KEYWORD] */
typedef struct S873 T873;

/* [attached] SPECIAL [detachable ET_PRECURSOR_KEYWORD] */
typedef struct S875 T875;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_SYMBOL] */
typedef struct S876 T876;

/* [attached] SPECIAL [detachable ET_SYMBOL] */
typedef struct S878 T878;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_POSITION] */
typedef struct S879 T879;

/* [attached] SPECIAL [detachable ET_POSITION] */
typedef struct S881 T881;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_BOOLEAN_CONSTANT] */
typedef struct S882 T882;

/* [attached] SPECIAL [detachable ET_BOOLEAN_CONSTANT] */
typedef struct S884 T884;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_BREAK] */
typedef struct S885 T885;

/* [attached] SPECIAL [detachable ET_BREAK] */
typedef struct S887 T887;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_CHARACTER_CONSTANT] */
typedef struct S888 T888;

/* [attached] SPECIAL [detachable ET_CHARACTER_CONSTANT] */
typedef struct S890 T890;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_CURRENT] */
typedef struct S891 T891;

/* [attached] SPECIAL [detachable ET_CURRENT] */
typedef struct S893 T893;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_FREE_OPERATOR] */
typedef struct S894 T894;

/* [attached] SPECIAL [detachable ET_FREE_OPERATOR] */
typedef struct S896 T896;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_INTEGER_CONSTANT] */
typedef struct S897 T897;

/* [attached] SPECIAL [detachable ET_INTEGER_CONSTANT] */
typedef struct S899 T899;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_KEYWORD_OPERATOR] */
typedef struct S900 T900;

/* [attached] SPECIAL [detachable ET_KEYWORD_OPERATOR] */
typedef struct S902 T902;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_MANIFEST_STRING] */
typedef struct S903 T903;

/* [attached] SPECIAL [detachable ET_MANIFEST_STRING] */
typedef struct S905 T905;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_REAL_CONSTANT] */
typedef struct S906 T906;

/* [attached] SPECIAL [detachable ET_REAL_CONSTANT] */
typedef struct S908 T908;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_RESULT] */
typedef struct S909 T909;

/* [attached] SPECIAL [detachable ET_RESULT] */
typedef struct S910 T910;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_RETRY_INSTRUCTION] */
typedef struct S911 T911;

/* [attached] SPECIAL [detachable ET_RETRY_INSTRUCTION] */
typedef struct S913 T913;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_SYMBOL_OPERATOR] */
typedef struct S914 T914;

/* [attached] SPECIAL [detachable ET_SYMBOL_OPERATOR] */
typedef struct S916 T916;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_VOID] */
typedef struct S917 T917;

/* [attached] SPECIAL [detachable ET_VOID] */
typedef struct S919 T919;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_SEMICOLON_SYMBOL] */
typedef struct S920 T920;

/* [attached] SPECIAL [detachable ET_SEMICOLON_SYMBOL] */
typedef struct S922 T922;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_BRACKET_SYMBOL] */
typedef struct S923 T923;

/* [attached] SPECIAL [detachable ET_BRACKET_SYMBOL] */
typedef struct S925 T925;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_QUESTION_MARK_SYMBOL] */
typedef struct S926 T926;

/* [attached] SPECIAL [detachable ET_QUESTION_MARK_SYMBOL] */
typedef struct S928 T928;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_ACROSS_EXPRESSION] */
typedef struct S929 T929;

/* [attached] SPECIAL [detachable ET_ACROSS_EXPRESSION] */
typedef struct S931 T931;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_ACROSS_INSTRUCTION] */
typedef struct S932 T932;

/* [attached] SPECIAL [detachable ET_ACROSS_INSTRUCTION] */
typedef struct S934 T934;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_ACTUAL_ARGUMENT_LIST] */
typedef struct S935 T935;

/* [attached] SPECIAL [detachable ET_ACTUAL_ARGUMENT_LIST] */
typedef struct S936 T936;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_ACTUAL_PARAMETER_ITEM] */
typedef struct S937 T937;

/* [attached] SPECIAL [detachable ET_ACTUAL_PARAMETER_ITEM] */
typedef struct S939 T939;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_ACTUAL_PARAMETER_LIST] */
typedef struct S940 T940;

/* [attached] SPECIAL [detachable ET_ACTUAL_PARAMETER_LIST] */
typedef struct S941 T941;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_AGENT_ARGUMENT_OPERAND] */
typedef struct S942 T942;

/* [attached] SPECIAL [detachable ET_AGENT_ARGUMENT_OPERAND] */
typedef struct S944 T944;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_AGENT_ARGUMENT_OPERAND_ITEM] */
typedef struct S945 T945;

/* [attached] SPECIAL [detachable ET_AGENT_ARGUMENT_OPERAND_ITEM] */
typedef struct S947 T947;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_AGENT_ARGUMENT_OPERAND_LIST] */
typedef struct S948 T948;

/* [attached] SPECIAL [detachable ET_AGENT_ARGUMENT_OPERAND_LIST] */
typedef struct S950 T950;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_AGENT_TARGET] */
typedef struct S951 T951;

/* [attached] SPECIAL [detachable ET_AGENT_TARGET] */
typedef struct S953 T953;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_ALIAS_NAME] */
typedef struct S954 T954;

/* [attached] SPECIAL [detachable ET_ALIAS_NAME] */
typedef struct S956 T956;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_ASSIGNER] */
typedef struct S957 T957;

/* [attached] SPECIAL [detachable ET_ASSIGNER] */
typedef struct S959 T959;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_BRACKET_EXPRESSION] */
typedef struct S960 T960;

/* [attached] SPECIAL [detachable ET_BRACKET_EXPRESSION] */
typedef struct S962 T962;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_CALL_AGENT] */
typedef struct S963 T963;

/* [attached] SPECIAL [detachable ET_CALL_AGENT] */
typedef struct S965 T965;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_QUALIFIED_CALL_EXPRESSION] */
typedef struct S966 T966;

/* [attached] SPECIAL [detachable ET_QUALIFIED_CALL_EXPRESSION] */
typedef struct S967 T967;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_CHECK_INSTRUCTION] */
typedef struct S968 T968;

/* [attached] SPECIAL [detachable ET_CHECK_INSTRUCTION] */
typedef struct S970 T970;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_CHOICE] */
typedef struct S971 T971;

/* [attached] SPECIAL [detachable ET_CHOICE] */
typedef struct S973 T973;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_CHOICE_CONSTANT] */
typedef struct S974 T974;

/* [attached] SPECIAL [detachable ET_CHOICE_CONSTANT] */
typedef struct S976 T976;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_CHOICE_ITEM] */
typedef struct S977 T977;

/* [attached] SPECIAL [detachable ET_CHOICE_ITEM] */
typedef struct S979 T979;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_CHOICE_LIST] */
typedef struct S980 T980;

/* [attached] SPECIAL [detachable ET_CHOICE_LIST] */
typedef struct S982 T982;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_CLASS] */
typedef struct S983 T983;

/* [attached] SPECIAL [detachable ET_CLASS] */
typedef struct S985 T985;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_CLIENT_ITEM] */
typedef struct S986 T986;

/* [attached] SPECIAL [detachable ET_CLIENT_ITEM] */
typedef struct S988 T988;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_CLIENTS] */
typedef struct S989 T989;

/* [attached] SPECIAL [detachable ET_CLIENTS] */
typedef struct S991 T991;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_COMPOUND] */
typedef struct S992 T992;

/* [attached] SPECIAL [detachable ET_COMPOUND] */
typedef struct S993 T993;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_CONDITIONAL] */
typedef struct S994 T994;

/* [attached] SPECIAL [detachable ET_CONDITIONAL] */
typedef struct S996 T996;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_CONSTANT] */
typedef struct S997 T997;

/* [attached] SPECIAL [detachable ET_CONSTANT] */
typedef struct S999 T999;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_CONSTRAINT_ACTUAL_PARAMETER_ITEM] */
typedef struct S1000 T1000;

/* [attached] SPECIAL [detachable ET_CONSTRAINT_ACTUAL_PARAMETER_ITEM] */
typedef struct S1002 T1002;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_CONSTRAINT_ACTUAL_PARAMETER_LIST] */
typedef struct S1003 T1003;

/* [attached] SPECIAL [detachable ET_CONSTRAINT_ACTUAL_PARAMETER_LIST] */
typedef struct S1005 T1005;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_CONSTRAINT_CREATOR] */
typedef struct S1006 T1006;

/* [attached] SPECIAL [detachable ET_CONSTRAINT_CREATOR] */
typedef struct S1008 T1008;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_CONSTRAINT_TYPE] */
typedef struct S1009 T1009;

/* [attached] SPECIAL [detachable ET_CONSTRAINT_TYPE] */
typedef struct S1011 T1011;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_CONVERT_FEATURE] */
typedef struct S1012 T1012;

/* [attached] SPECIAL [detachable ET_CONVERT_FEATURE] */
typedef struct S1014 T1014;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_CONVERT_FEATURE_ITEM] */
typedef struct S1015 T1015;

/* [attached] SPECIAL [detachable ET_CONVERT_FEATURE_ITEM] */
typedef struct S1017 T1017;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_CONVERT_FEATURE_LIST] */
typedef struct S1018 T1018;

/* [attached] SPECIAL [detachable ET_CONVERT_FEATURE_LIST] */
typedef struct S1020 T1020;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_CREATE_EXPRESSION] */
typedef struct S1021 T1021;

/* [attached] SPECIAL [detachable ET_CREATE_EXPRESSION] */
typedef struct S1023 T1023;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_CREATION_REGION] */
typedef struct S1024 T1024;

/* [attached] SPECIAL [detachable ET_CREATION_REGION] */
typedef struct S1026 T1026;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_CREATOR] */
typedef struct S1027 T1027;

/* [attached] SPECIAL [detachable ET_CREATOR] */
typedef struct S1029 T1029;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_CREATOR_LIST] */
typedef struct S1030 T1030;

/* [attached] SPECIAL [detachable ET_CREATOR_LIST] */
typedef struct S1032 T1032;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_DEBUG_INSTRUCTION] */
typedef struct S1033 T1033;

/* [attached] SPECIAL [detachable ET_DEBUG_INSTRUCTION] */
typedef struct S1035 T1035;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_ELSEIF_EXPRESSION] */
typedef struct S1036 T1036;

/* [attached] SPECIAL [detachable ET_ELSEIF_EXPRESSION] */
typedef struct S1038 T1038;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_ELSEIF_EXPRESSION_LIST] */
typedef struct S1039 T1039;

/* [attached] SPECIAL [detachable ET_ELSEIF_EXPRESSION_LIST] */
typedef struct S1041 T1041;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_ELSEIF_PART] */
typedef struct S1042 T1042;

/* [attached] SPECIAL [detachable ET_ELSEIF_PART] */
typedef struct S1044 T1044;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_ELSEIF_PART_LIST] */
typedef struct S1045 T1045;

/* [attached] SPECIAL [detachable ET_ELSEIF_PART_LIST] */
typedef struct S1047 T1047;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_EXPORT] */
typedef struct S1048 T1048;

/* [attached] SPECIAL [detachable ET_EXPORT] */
typedef struct S1050 T1050;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_EXPORT_LIST] */
typedef struct S1051 T1051;

/* [attached] SPECIAL [detachable ET_EXPORT_LIST] */
typedef struct S1052 T1052;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_EXPRESSION] */
typedef struct S1053 T1053;

/* [attached] SPECIAL [detachable ET_EXPRESSION] */
typedef struct S1054 T1054;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_EXPRESSION_ITEM] */
typedef struct S1055 T1055;

/* [attached] SPECIAL [detachable ET_EXPRESSION_ITEM] */
typedef struct S1057 T1057;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_EXTENDED_FEATURE_NAME] */
typedef struct S1058 T1058;

/* [attached] SPECIAL [detachable ET_EXTENDED_FEATURE_NAME] */
typedef struct S1060 T1060;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_EXTERNAL_ALIAS] */
typedef struct S1061 T1061;

/* [attached] SPECIAL [detachable ET_EXTERNAL_ALIAS] */
typedef struct S1063 T1063;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_FEATURE_CLAUSE] */
typedef struct S1064 T1064;

/* [attached] SPECIAL [detachable ET_FEATURE_CLAUSE] */
typedef struct S1066 T1066;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_FEATURE_CLAUSE_LIST] */
typedef struct S1067 T1067;

/* [attached] SPECIAL [detachable ET_FEATURE_CLAUSE_LIST] */
typedef struct S1069 T1069;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_FEATURE_EXPORT] */
typedef struct S1070 T1070;

/* [attached] SPECIAL [detachable ET_FEATURE_EXPORT] */
typedef struct S1072 T1072;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_FEATURE_NAME] */
typedef struct S1073 T1073;

/* [attached] SPECIAL [detachable ET_FEATURE_NAME] */
typedef struct S1074 T1074;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_FEATURE_NAME_ITEM] */
typedef struct S1075 T1075;

/* [attached] SPECIAL [detachable ET_FEATURE_NAME_ITEM] */
typedef struct S1077 T1077;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_FORMAL_ARGUMENT] */
typedef struct S1078 T1078;

/* [attached] SPECIAL [detachable ET_FORMAL_ARGUMENT] */
typedef struct S1080 T1080;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_FORMAL_ARGUMENT_ITEM] */
typedef struct S1081 T1081;

/* [attached] SPECIAL [detachable ET_FORMAL_ARGUMENT_ITEM] */
typedef struct S1083 T1083;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_FORMAL_ARGUMENT_LIST] */
typedef struct S1084 T1084;

/* [attached] SPECIAL [detachable ET_FORMAL_ARGUMENT_LIST] */
typedef struct S1085 T1085;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_FORMAL_PARAMETER] */
typedef struct S1086 T1086;

/* [attached] SPECIAL [detachable ET_FORMAL_PARAMETER] */
typedef struct S1088 T1088;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_FORMAL_PARAMETER_ITEM] */
typedef struct S1089 T1089;

/* [attached] SPECIAL [detachable ET_FORMAL_PARAMETER_ITEM] */
typedef struct S1091 T1091;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_FORMAL_PARAMETER_LIST] */
typedef struct S1092 T1092;

/* [attached] SPECIAL [detachable ET_FORMAL_PARAMETER_LIST] */
typedef struct S1093 T1093;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_IF_EXPRESSION] */
typedef struct S1094 T1094;

/* [attached] SPECIAL [detachable ET_IF_EXPRESSION] */
typedef struct S1096 T1096;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_IF_INSTRUCTION] */
typedef struct S1097 T1097;

/* [attached] SPECIAL [detachable ET_IF_INSTRUCTION] */
typedef struct S1099 T1099;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_INDEXING_LIST] */
typedef struct S1100 T1100;

/* [attached] SPECIAL [detachable ET_INDEXING_LIST] */
typedef struct S1102 T1102;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_INDEXING_ITEM] */
typedef struct S1103 T1103;

/* [attached] SPECIAL [detachable ET_INDEXING_ITEM] */
typedef struct S1105 T1105;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_INDEXING_TERM] */
typedef struct S1106 T1106;

/* [attached] SPECIAL [detachable ET_INDEXING_TERM] */
typedef struct S1108 T1108;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_INDEXING_TERM_ITEM] */
typedef struct S1109 T1109;

/* [attached] SPECIAL [detachable ET_INDEXING_TERM_ITEM] */
typedef struct S1111 T1111;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_INDEXING_TERM_LIST] */
typedef struct S1112 T1112;

/* [attached] SPECIAL [detachable ET_INDEXING_TERM_LIST] */
typedef struct S1114 T1114;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_INLINE_AGENT] */
typedef struct S1115 T1115;

/* [attached] SPECIAL [detachable ET_INLINE_AGENT] */
typedef struct S1117 T1117;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_INSPECT_INSTRUCTION] */
typedef struct S1118 T1118;

/* [attached] SPECIAL [detachable ET_INSPECT_INSTRUCTION] */
typedef struct S1120 T1120;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_INSTRUCTION] */
typedef struct S1121 T1121;

/* [attached] SPECIAL [detachable ET_INSTRUCTION] */
typedef struct S1123 T1123;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_INVARIANTS] */
typedef struct S1124 T1124;

/* [attached] SPECIAL [detachable ET_INVARIANTS] */
typedef struct S1126 T1126;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_KEYWORD_FEATURE_NAME_LIST] */
typedef struct S1127 T1127;

/* [attached] SPECIAL [detachable ET_KEYWORD_FEATURE_NAME_LIST] */
typedef struct S1128 T1128;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_LIKE_TYPE] */
typedef struct S1129 T1129;

/* [attached] SPECIAL [detachable ET_LIKE_TYPE] */
typedef struct S1131 T1131;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_LOCAL_VARIABLE] */
typedef struct S1132 T1132;

/* [attached] SPECIAL [detachable ET_LOCAL_VARIABLE] */
typedef struct S1134 T1134;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_LOCAL_VARIABLE_ITEM] */
typedef struct S1135 T1135;

/* [attached] SPECIAL [detachable ET_LOCAL_VARIABLE_ITEM] */
typedef struct S1137 T1137;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_LOCAL_VARIABLE_LIST] */
typedef struct S1138 T1138;

/* [attached] SPECIAL [detachable ET_LOCAL_VARIABLE_LIST] */
typedef struct S1139 T1139;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_LOOP_INVARIANTS] */
typedef struct S1140 T1140;

/* [attached] SPECIAL [detachable ET_LOOP_INVARIANTS] */
typedef struct S1142 T1142;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_MANIFEST_ARRAY] */
typedef struct S1143 T1143;

/* [attached] SPECIAL [detachable ET_MANIFEST_ARRAY] */
typedef struct S1145 T1145;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_MANIFEST_STRING_ITEM] */
typedef struct S1146 T1146;

/* [attached] SPECIAL [detachable ET_MANIFEST_STRING_ITEM] */
typedef struct S1148 T1148;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_MANIFEST_STRING_LIST] */
typedef struct S1149 T1149;

/* [attached] SPECIAL [detachable ET_MANIFEST_STRING_LIST] */
typedef struct S1151 T1151;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_MANIFEST_TUPLE] */
typedef struct S1152 T1152;

/* [attached] SPECIAL [detachable ET_MANIFEST_TUPLE] */
typedef struct S1153 T1153;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_OBSOLETE] */
typedef struct S1154 T1154;

/* [attached] SPECIAL [detachable ET_OBSOLETE] */
typedef struct S1156 T1156;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_PARENTHESIZED_EXPRESSION] */
typedef struct S1157 T1157;

/* [attached] SPECIAL [detachable ET_PARENTHESIZED_EXPRESSION] */
typedef struct S1159 T1159;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_PARENT] */
typedef struct S1160 T1160;

/* [attached] SPECIAL [detachable ET_PARENT] */
typedef struct S1162 T1162;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_PARENT_CLAUSE_LIST] */
typedef struct S1163 T1163;

/* [attached] SPECIAL [detachable ET_PARENT_CLAUSE_LIST] */
typedef struct S1165 T1165;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_PARENT_ITEM] */
typedef struct S1166 T1166;

/* [attached] SPECIAL [detachable ET_PARENT_ITEM] */
typedef struct S1168 T1168;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_PARENT_LIST] */
typedef struct S1169 T1169;

/* [attached] SPECIAL [detachable ET_PARENT_LIST] */
typedef struct S1171 T1171;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_POSTCONDITIONS] */
typedef struct S1172 T1172;

/* [attached] SPECIAL [detachable ET_POSTCONDITIONS] */
typedef struct S1174 T1174;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_PRECONDITIONS] */
typedef struct S1175 T1175;

/* [attached] SPECIAL [detachable ET_PRECONDITIONS] */
typedef struct S1176 T1176;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_PROCEDURE] */
typedef struct S1177 T1177;

/* [attached] SPECIAL [detachable ET_PROCEDURE] */
typedef struct S1178 T1178;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_QUALIFIED_LIKE_IDENTIFIER] */
typedef struct S1179 T1179;

/* [attached] SPECIAL [detachable ET_QUALIFIED_LIKE_IDENTIFIER] */
typedef struct S1181 T1181;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_QUERY] */
typedef struct S1182 T1182;

/* [attached] SPECIAL [detachable ET_QUERY] */
typedef struct S1183 T1183;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_RENAME_ITEM] */
typedef struct S1184 T1184;

/* [attached] SPECIAL [detachable ET_RENAME_ITEM] */
typedef struct S1186 T1186;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_RENAME_LIST] */
typedef struct S1187 T1187;

/* [attached] SPECIAL [detachable ET_RENAME_LIST] */
typedef struct S1188 T1188;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_STATIC_CALL_EXPRESSION] */
typedef struct S1189 T1189;

/* [attached] SPECIAL [detachable ET_STATIC_CALL_EXPRESSION] */
typedef struct S1191 T1191;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_STRIP_EXPRESSION] */
typedef struct S1192 T1192;

/* [attached] SPECIAL [detachable ET_STRIP_EXPRESSION] */
typedef struct S1194 T1194;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_TYPE] */
typedef struct S1195 T1195;

/* [attached] SPECIAL [detachable ET_TYPE] */
typedef struct S1196 T1196;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_TYPE_ITEM] */
typedef struct S1197 T1197;

/* [attached] SPECIAL [detachable ET_TYPE_ITEM] */
typedef struct S1199 T1199;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_TYPE_LIST] */
typedef struct S1200 T1200;

/* [attached] SPECIAL [detachable ET_TYPE_LIST] */
typedef struct S1202 T1202;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_VARIANT] */
typedef struct S1203 T1203;

/* [attached] SPECIAL [detachable ET_VARIANT] */
typedef struct S1205 T1205;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_WHEN_PART] */
typedef struct S1206 T1206;

/* [attached] SPECIAL [detachable ET_WHEN_PART] */
typedef struct S1208 T1208;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_WHEN_PART_LIST] */
typedef struct S1209 T1209;

/* [attached] SPECIAL [detachable ET_WHEN_PART_LIST] */
typedef struct S1211 T1211;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_WRITABLE] */
typedef struct S1212 T1212;

/* [attached] SPECIAL [detachable ET_WRITABLE] */
typedef struct S1213 T1213;

/* [attached] DS_HASH_TOPOLOGICAL_SORTER [[attached] ET_CLASS] */
typedef struct S1214 T1214;

/* [attached] DS_HASH_TABLE [[attached] ET_BASE_TYPE, [attached] ET_CLASS] */
typedef struct S1215 T1215;

/* ET_PARENT_CHECKER1 */
typedef struct S1216 T1216;

/* ET_FORMAL_PARAMETER_CHECKER1 */
typedef struct S1217 T1217;

/* [attached] DS_HASH_TABLE [[attached] ET_FLATTENED_FEATURE, [attached] ET_FEATURE_NAME] */
typedef struct S1218 T1218;

/* ET_FEATURE_NAME_TESTER */
typedef struct S1219 T1219;

/* [attached] DS_HASH_TABLE [[attached] ET_FLATTENED_FEATURE, [attached] ET_ALIAS_NAME] */
typedef struct S1221 T1221;

/* ET_ALIAS_NAME_TESTER */
typedef struct S1222 T1222;

/* [attached] DS_ARRAYED_LIST [[attached] ET_CLIENT_LIST] */
typedef struct S1224 T1224;

/* [attached] DS_HASH_TABLE [[attached] ET_CLIENT, [attached] ET_CLASS] */
typedef struct S1225 T1225;

/* ET_FEATURE_ADAPTATION_RESOLVER */
typedef struct S1226 T1226;

/* ET_DOTNET_FEATURE_ADAPTATION_RESOLVER */
typedef struct S1227 T1227;

/* ET_IDENTIFIER_TYPE_RESOLVER */
typedef struct S1228 T1228;

/* ET_UNFOLDED_TUPLE_ACTUAL_PARAMETERS_RESOLVER1 */
typedef struct S1229 T1229;

/* ET_ANCHORED_TYPE_CHECKER */
typedef struct S1230 T1230;

/* ET_SIGNATURE_CHECKER */
typedef struct S1231 T1231;

/* ET_PARENT_CHECKER2 */
typedef struct S1232 T1232;

/* ET_FORMAL_PARAMETER_CHECKER2 */
typedef struct S1233 T1233;

/* ET_BUILTIN_FEATURE_CHECKER */
typedef struct S1234 T1234;

/* ET_PRECURSOR_CHECKER */
typedef struct S1235 T1235;

/* [attached] DS_HASH_TABLE [[attached] ET_FEATURE, [attached] INTEGER_32] */
typedef struct S1236 T1236;

/* [attached] DS_HASH_SET [[attached] ET_CLASS] */
typedef struct S1237 T1237;

/* ET_PARENT_CHECKER3 */
typedef struct S1238 T1238;

/* ET_QUALIFIED_ANCHORED_TYPE_CHECKER */
typedef struct S1239 T1239;

/* ET_UNFOLDED_TUPLE_ACTUAL_PARAMETERS_RESOLVER2 */
typedef struct S1240 T1240;

/* ET_FORMAL_PARAMETER_LIST */
typedef struct S1242 T1242;

/* ET_FORMAL_PARAMETER */
typedef struct S1243 T1243;

/* ET_CONSTRAINT_CREATOR */
typedef struct S1244 T1244;

/* ET_INHERITED_FEATURE */
typedef struct S1246 T1246;

/* ET_PARENT_FEATURE */
typedef struct S1247 T1247;

/* [attached] DS_HASH_SET [[attached] ET_PROCEDURE] */
typedef struct S1248 T1248;

/* [attached] DS_HASH_SET [[attached] ET_QUERY] */
typedef struct S1249 T1249;

/* ET_SUPPLIER_BUILDER */
typedef struct S1250 T1250;

/* [attached] SPECIAL [[attached] ET_CLASS] */
typedef struct S1251 T1251;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_CLASS] */
typedef struct S1252 T1252;

/* [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_CLASS] */
typedef struct S1253 T1253;

/* [attached] DS_ARRAYED_LIST_CURSOR [[attached] INTEGER_32] */
typedef struct S1255 T1255;

/* [attached] SPECIAL [[attached] ET_DYNAMIC_TYPE_SET] */
typedef struct S1258 T1258;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_DYNAMIC_TYPE_SET] */
typedef struct S1259 T1259;

/* [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_IDENTIFIER] */
typedef struct S1260 T1260;

/* [attached] SPECIAL [detachable ET_DYNAMIC_TYPE] */
typedef struct S1261 T1261;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_DYNAMIC_TYPE] */
typedef struct S1262 T1262;

/* [attached] DS_ARRAYED_LIST_CURSOR [detachable ET_DYNAMIC_TYPE] */
typedef struct S1263 T1263;

/* [attached] SPECIAL [[attached] ET_EXPRESSION] */
typedef struct S1265 T1265;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_EXPRESSION] */
typedef struct S1266 T1266;

/* [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_EXPRESSION] */
typedef struct S1267 T1267;

/* attached KL_EQUALITY_TESTER [[attached] INTEGER_32] */
typedef struct S1271 T1271;

/* [attached] DS_HASH_TABLE_CURSOR [[attached] ET_DYNAMIC_TYPE, [attached] INTEGER_32] */
typedef struct S1273 T1273;

/* [attached] SPECIAL [[attached] ET_DYNAMIC_STANDALONE_TYPE_SET] */
typedef struct S1274 T1274;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_DYNAMIC_STANDALONE_TYPE_SET] */
typedef struct S1275 T1275;

/* [attached] DS_HASH_SET_CURSOR [[attached] ET_DYNAMIC_TYPE] */
typedef struct S1276 T1276;

/* [attached] DS_HASH_TABLE_CURSOR [[attached] ET_DYNAMIC_STANDALONE_TYPE_SET, [attached] ET_DYNAMIC_TYPE] */
typedef struct S1279 T1279;

/* [attached] SPECIAL [[attached] ET_OBJECT_TEST] */
typedef struct S1282 T1282;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_OBJECT_TEST] */
typedef struct S1283 T1283;

/* [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_OBJECT_TEST] */
typedef struct S1284 T1284;

/* [attached] SPECIAL [[attached] ET_OBJECT_EQUALITY_EXPRESSION] */
typedef struct S1286 T1286;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_OBJECT_EQUALITY_EXPRESSION] */
typedef struct S1287 T1287;

/* [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_OBJECT_EQUALITY_EXPRESSION] */
typedef struct S1288 T1288;

/* [attached] SPECIAL [[attached] ET_EQUALITY_EXPRESSION] */
typedef struct S1290 T1290;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_EQUALITY_EXPRESSION] */
typedef struct S1291 T1291;

/* [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_EQUALITY_EXPRESSION] */
typedef struct S1292 T1292;

/* [attached] SPECIAL [[attached] ET_AGENT] */
typedef struct S1294 T1294;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_AGENT] */
typedef struct S1295 T1295;

/* [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_AGENT] */
typedef struct S1296 T1296;

/* [attached] SPECIAL [[attached] ET_EXPRESSION_ITEM] */
typedef struct S1298 T1298;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_EXPRESSION_ITEM] */
typedef struct S1300 T1300;

/* [attached] DS_ARRAYED_LIST_CURSOR [detachable ET_QUALIFIED_CALL_EXPRESSION] */
typedef struct S1301 T1301;

/* [attached] DS_HASH_SET_CURSOR [[attached] ET_DYNAMIC_TUPLE_TYPE] */
typedef struct S1302 T1302;

/* [attached] SPECIAL [[attached] ET_DYNAMIC_TUPLE_TYPE] */
typedef struct S1303 T1303;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_DYNAMIC_TUPLE_TYPE] */
typedef struct S1304 T1304;

/* [attached] DS_HASH_TABLE_CURSOR [[attached] INTEGER_32, [attached] ET_FEATURE] */
typedef struct S1309 T1309;

/* [attached] SPECIAL [[attached] ET_FEATURE] */
typedef struct S1310 T1310;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_FEATURE] */
typedef struct S1311 T1311;

/* [attached] DS_HASH_TABLE_CURSOR [[attached] ET_CONSTANT, [attached] ET_FEATURE] */
typedef struct S1312 T1312;

/* [attached] SPECIAL [[attached] ET_CONSTANT] */
typedef struct S1313 T1313;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_CONSTANT] */
typedef struct S1316 T1316;

/* [attached] DS_HASH_TABLE_CURSOR [[attached] ET_DYNAMIC_TYPE, [attached] ET_INLINE_CONSTANT] */
typedef struct S1317 T1317;

/* [attached] SPECIAL [[attached] ET_INLINE_CONSTANT] */
typedef struct S1318 T1318;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_INLINE_CONSTANT] */
typedef struct S1321 T1321;

/* [attached] SPECIAL [detachable ET_DYNAMIC_FEATURE] */
typedef struct S1322 T1322;

/* [attached] DS_HASH_TABLE_CURSOR [detachable ET_DYNAMIC_FEATURE, [attached] ET_DYNAMIC_TYPE] */
typedef struct S1325 T1325;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_DYNAMIC_FEATURE] */
typedef struct S1326 T1326;

/* [attached] SPECIAL [[attached] ET_DYNAMIC_FEATURE] */
typedef struct S1327 T1327;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_DYNAMIC_FEATURE] */
typedef struct S1328 T1328;

/* [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_DYNAMIC_FEATURE] */
typedef struct S1329 T1329;

/* [attached] DS_HASH_TABLE_CURSOR [[attached] BOOLEAN, [attached] STRING_8] */
typedef struct S1331 T1331;

/* RX_BYTE_CODE */
typedef struct S1337 T1337;

/* RX_CHARACTER_SET */
typedef struct S1338 T1338;

/* RX_CASE_MAPPING */
typedef struct S1339 T1339;

/* [attached] ARRAY [[attached] RX_CHARACTER_SET] */
typedef struct S1342 T1342;

/* [attached] SPECIAL [[attached] RX_CHARACTER_SET] */
typedef struct S1343 T1343;

/* TYPED_POINTER [[attached] ANY] */
typedef struct S1344 T1344;
extern T0* GE_boxed1344(T1344 a1);
typedef struct Sb1344 Tb1344;

/* [attached] DS_ARRAYED_LIST [detachable DS_LINKABLE [[attached] INTEGER_32]] */
typedef struct S1345 T1345;

/* [attached] ARRAY [[attached] BOOLEAN] */
typedef struct S1346 T1346;

/* [attached] SPECIAL [[attached] ET_SYSTEM_PROCESSOR] */
typedef struct S1349 T1349;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_SYSTEM_PROCESSOR] */
typedef struct S1350 T1350;

/* [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_SYSTEM_PROCESSOR] */
typedef struct S1351 T1351;

/* THREAD_ATTRIBUTES */
typedef struct S1352 T1352;

/* [attached] CELL [[attached] BOOLEAN] */
typedef struct S1353 T1353;

/* [attached] DS_HASH_TABLE_CURSOR [[attached] INTEGER_32, [attached] ET_DYNAMIC_TYPE] */
typedef struct S1359 T1359;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_INLINE_AGENT] */
typedef struct S1361 T1361;

/* [attached] SPECIAL [[attached] ET_INLINE_AGENT] */
typedef struct S1363 T1363;

/* [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_INLINE_AGENT] */
typedef struct S1364 T1364;

/* [attached] KL_SPECIAL_ROUTINES [[attached] DS_ARRAYED_LIST [[attached] ET_PROCEDURE]] */
typedef struct S1365 T1365;

/* [attached] SPECIAL [[attached] DS_ARRAYED_LIST [[attached] ET_PROCEDURE]] */
typedef struct S1366 T1366;

/* [attached] DS_ARRAYED_LIST_CURSOR [[attached] DS_ARRAYED_LIST [[attached] ET_PROCEDURE]] */
typedef struct S1367 T1367;

/* [attached] KL_SPECIAL_ROUTINES [[attached] DS_ARRAYED_LIST [[attached] ET_QUERY]] */
typedef struct S1368 T1368;

/* [attached] SPECIAL [[attached] DS_ARRAYED_LIST [[attached] ET_QUERY]] */
typedef struct S1369 T1369;

/* [attached] DS_ARRAYED_LIST_CURSOR [[attached] DS_ARRAYED_LIST [[attached] ET_QUERY]] */
typedef struct S1370 T1370;

/* [attached] KL_SPECIAL_ROUTINES [[attached] DS_ARRAYED_LIST [[attached] ET_FEATURE]] */
typedef struct S1371 T1371;

/* [attached] DS_ARRAYED_LIST [[attached] ET_FEATURE] */
typedef struct S1372 T1372;

/* [attached] SPECIAL [[attached] DS_ARRAYED_LIST [[attached] ET_FEATURE]] */
typedef struct S1373 T1373;

/* [attached] DS_ARRAYED_LIST_CURSOR [[attached] DS_ARRAYED_LIST [[attached] ET_FEATURE]] */
typedef struct S1374 T1374;

/* [attached] SPECIAL [[attached] ET_NESTED_TYPE_CONTEXT] */
typedef struct S1375 T1375;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_NESTED_TYPE_CONTEXT] */
typedef struct S1376 T1376;

/* [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_NESTED_TYPE_CONTEXT] */
typedef struct S1377 T1377;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_TYPE] */
typedef struct S1379 T1379;

/* [attached] DS_HASH_TABLE_CURSOR [[attached] ET_NESTED_TYPE_CONTEXT, [attached] ET_NAMED_OBJECT_TEST] */
typedef struct S1380 T1380;

/* ET_NAMED_OBJECT_TEST */
typedef struct S1384 T1384;

/* [attached] SPECIAL [[attached] ET_NAMED_OBJECT_TEST] */
typedef struct S1385 T1385;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_NAMED_OBJECT_TEST] */
typedef struct S1386 T1386;

/* [attached] DS_ARRAYED_LIST [[attached] ET_NAMED_OBJECT_TEST] */
typedef struct S1387 T1387;

/* [attached] DS_HASH_TABLE_CURSOR [[attached] ET_NESTED_TYPE_CONTEXT, [attached] ET_ACROSS_COMPONENT] */
typedef struct S1388 T1388;

/* [attached] SPECIAL [[attached] ET_ACROSS_COMPONENT] */
typedef struct S1392 T1392;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_ACROSS_COMPONENT] */
typedef struct S1393 T1393;

/* [attached] DS_ARRAYED_LIST [[attached] ET_ACROSS_COMPONENT] */
typedef struct S1394 T1394;

/* [attached] DS_HASH_SET [[attached] INTEGER_32] */
typedef struct S1395 T1395;

/* [attached] SPECIAL [[attached] ET_ATTACHMENT_SCOPE] */
typedef struct S1398 T1398;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_ATTACHMENT_SCOPE] */
typedef struct S1399 T1399;

/* [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_ATTACHMENT_SCOPE] */
typedef struct S1400 T1400;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_INDEXING_TERM] */
typedef struct S1402 T1402;

/* [attached] SPECIAL [[attached] ET_INDEXING_TERM] */
typedef struct S1404 T1404;

/* [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_INDEXING_TERM] */
typedef struct S1405 T1405;

/* [attached] DS_HASH_TABLE_CURSOR [[attached] ET_ASSERTIONS, [attached] ET_FEATURE] */
typedef struct S1406 T1406;

/* [attached] SPECIAL [[attached] ET_ASSERTIONS] */
typedef struct S1407 T1407;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_ASSERTIONS] */
typedef struct S1409 T1409;

/* [attached] DS_HASH_TABLE [[attached] NATURAL_8, [attached] ET_CLASS_NAME] */
typedef struct S1411 T1411;

/* [attached] DS_HASH_TABLE_CURSOR [[attached] ET_IDENTIFIER, [attached] STRING_8] */
typedef struct S1415 T1415;

/* STRING_TO_INTEGER_CONVERTOR */
typedef struct S1416 T1416;

/* ET_VALIDITY_ERROR */
typedef struct S1418 T1418;

/* UC_UNICODE_ROUTINES */
typedef struct S1419 T1419;

/* ET_XACE_CLASS_OPTIONS */
typedef struct S1422 T1422;

/* attached DS_LINKED_LIST [[attached] ET_XACE_FEATURE_OPTIONS] */
typedef struct S1424 T1424;

/* ET_XACE_FEATURE_OPTIONS */
typedef struct S1426 T1426;

/* ET_XACE_WRONG_ROOT_ELEMENT_ERROR */
typedef struct S1427 T1427;

/* ET_XACE_MISSING_ATTRIBUTE_ERROR */
typedef struct S1428 T1428;

/* ET_XACE_UNKNOWN_ELEMENT_ERROR */
typedef struct S1429 T1429;

/* ET_XACE_NON_EMPTY_ATTRIBUTE_EXPECTED_ERROR */
typedef struct S1430 T1430;

/* ET_XACE_MISSING_ELEMENT_ERROR */
typedef struct S1431 T1431;

/* ET_XACE_OBSOLETE_CLUSTER_ELEMENT_WARNING */
typedef struct S1432 T1432;

/* ET_XACE_OBSOLETE_EXCLUDE_ELEMENT_WARNING */
typedef struct S1433 T1433;

/* ET_XACE_ATTRIBUTE_OBSOLETED_BY_ELEMENT_WARNING */
typedef struct S1434 T1434;

/* ET_ECF_NOTE_ELEMENT */
typedef struct S1438 T1438;

/* [attached] DS_ARRAYED_LIST [[attached] ET_ECF_NOTE_ELEMENT] */
typedef struct S1439 T1439;

/* ET_ECF_TARGET_PARENT */
typedef struct S1453 T1453;

/* ET_ECF_ADAPTED_PRECOMPILED_LIBRARY */
typedef struct S1455 T1455;

/* ET_ECF_VERSION */
typedef struct S1459 T1459;

/* ET_ECF_ADAPTED_DOTNET_ASSEMBLIES */
typedef struct S1461 T1461;

/* ET_ECF_CLUSTER */
typedef struct S1463 T1463;

/* ET_ECF_CLUSTERS */
typedef struct S1464 T1464;

/* ET_ECF_EXTERNAL_CFLAG */
typedef struct S1466 T1466;

/* ET_ECF_EXTERNAL_CFLAGS */
typedef struct S1467 T1467;

/* ET_ECF_EXTERNAL_INCLUDE */
typedef struct S1469 T1469;

/* ET_ECF_EXTERNAL_INCLUDES */
typedef struct S1470 T1470;

/* ET_ECF_EXTERNAL_LIBRARY */
typedef struct S1472 T1472;

/* ET_ECF_EXTERNAL_LIBRARIES */
typedef struct S1473 T1473;

/* ET_ECF_EXTERNAL_LINKER_FLAG */
typedef struct S1475 T1475;

/* ET_ECF_EXTERNAL_LINKER_FLAGS */
typedef struct S1476 T1476;

/* ET_ECF_EXTERNAL_MAKE */
typedef struct S1478 T1478;

/* ET_ECF_EXTERNAL_MAKES */
typedef struct S1479 T1479;

/* ET_ECF_EXTERNAL_OBJECT */
typedef struct S1481 T1481;

/* ET_ECF_EXTERNAL_OBJECTS */
typedef struct S1482 T1482;

/* ET_ECF_EXTERNAL_RESOURCE */
typedef struct S1484 T1484;

/* ET_ECF_EXTERNAL_RESOURCES */
typedef struct S1485 T1485;

/* ET_ECF_FILE_RULE */
typedef struct S1487 T1487;

/* ET_ECF_FILE_RULES */
typedef struct S1488 T1488;

/* ET_ECF_ADAPTED_LIBRARIES */
typedef struct S1490 T1490;

/* ET_ECF_ACTION */
typedef struct S1492 T1492;

/* [attached] DS_ARRAYED_LIST [[attached] ET_ECF_ACTION] */
typedef struct S1493 T1493;

/* [attached] TUPLE [[attached] ET_ECF_OPTIONS] */
typedef struct S1497 T1497;

/* [attached] PROCEDURE [[attached] TUPLE [[attached] ET_ECF_OPTIONS]] */
typedef struct S1498 T1498;

/* ET_ECF_ROOT_ALL_CLASSES */
typedef struct S1500 T1500;

/* ET_ECF_ROOT_CLASS */
typedef struct S1501 T1501;

/* ET_ECF_ORED_CONDITIONS */
typedef struct S1503 T1503;

/* ET_ECF_ANDED_CONDITIONS */
typedef struct S1505 T1505;

/* [attached] DS_HASH_TABLE [[attached] ET_ECF_OPTIONS, [attached] STRING_8] */
typedef struct S1506 T1506;

/* ET_ECF_VISIBLE_CLASS */
typedef struct S1508 T1508;

/* [attached] DS_ARRAYED_LIST [[attached] ET_ECF_VISIBLE_CLASS] */
typedef struct S1509 T1509;

/* ET_ECF_BUILD_CONDITION */
typedef struct S1518 T1518;

/* ET_ECF_CONCURRENCY_CONDITION */
typedef struct S1519 T1519;

/* ET_ECF_CUSTOM_CONDITION */
typedef struct S1520 T1520;

/* ET_ECF_DOTNET_CONDITION */
typedef struct S1521 T1521;

/* ET_ECF_DYNAMIC_RUNTIME_CONDITION */
typedef struct S1522 T1522;

/* ET_ECF_PLATFORM_CONDITION */
typedef struct S1523 T1523;

/* ET_ECF_COMPILER_VERSION_CONDITION */
typedef struct S1524 T1524;

/* ET_ECF_MSIL_CLR_VERSION_CONDITION */
typedef struct S1525 T1525;

/* [attached] TUPLE [ET_ECF_SYSTEM] */
typedef struct S1526 T1526;

/* ET_PARENTHESIS_EXPRESSION */
typedef struct S1528 T1528;

/* ET_REGULAR_MANIFEST_STRING */
typedef struct S1532 T1532;

/* ET_INFIX_FREE_NAME */
typedef struct S1533 T1533;

/* ET_PARENT_CLAUSE_LIST */
typedef struct S1534 T1534;

/* ET_CREATOR_LIST */
typedef struct S1537 T1537;

/* TYPED_POINTER [[attached] NATURAL_8] */
typedef struct S1538 T1538;
extern T0* GE_boxed1538(T1538 a1);
typedef struct Sb1538 Tb1538;

/* [attached] XM_LINKED_LIST [[attached] XM_DOCUMENT_NODE] */
typedef struct S1540 T1540;

/* [attached] XM_LINKED_LIST [[attached] XM_ELEMENT_NODE] */
typedef struct S1542 T1542;

/* UT_TRISTATE */
typedef struct S1546 T1546;

/* [attached] KL_AGENT_ROUTINES [[attached] ANY] */
typedef struct S1547 T1547;

/* [attached] TUPLE [UT_TRISTATE] */
typedef struct S1548 T1548;

/* [attached] TUPLE [[attached] KL_AGENT_ROUTINES [[attached] ANY], [attached] PROCEDURE [[attached] TUPLE]] */
typedef struct S1549 T1549;

/* [attached] PREDICATE [[attached] TUPLE [[attached] ET_MASTER_CLASS]] */
typedef struct S1550 T1550;

/* [attached] PREDICATE [[attached] TUPLE] */
typedef struct S1551 T1551;

/* [attached] TUPLE [[attached] PROCEDURE [[attached] TUPLE [[attached] ET_MASTER_CLASS]], detachable FUNCTION [[attached] TUPLE, [attached] BOOLEAN]] */
typedef struct S1553 T1553;

/* [attached] TUPLE [[attached] PROCEDURE [[attached] TUPLE [[attached] ET_MASTER_CLASS]], [attached] FUNCTION [[attached] TUPLE [[attached] ET_MASTER_CLASS], [attached] BOOLEAN]] */
typedef struct S1554 T1554;

/* [attached] TUPLE [[attached] PROCEDURE [[attached] TUPLE [[attached] ET_MASTER_CLASS]], [attached] FUNCTION [[attached] TUPLE [[attached] ET_MASTER_CLASS], [attached] BOOLEAN], detachable FUNCTION [[attached] TUPLE, [attached] BOOLEAN]] */
typedef struct S1555 T1555;

/* UC_UTF8_ROUTINES */
typedef struct S1556 T1556;

/* UC_UTF8_STRING */
typedef struct S1557 T1557;

/* XM_EIFFEL_INPUT_STREAM */
typedef struct S1558 T1558;

/* [attached] DS_ARRAYED_LIST [[attached] ET_ECF_TARGET] */
typedef struct S1562 T1562;

/* [attached] DS_ARRAYED_LIST [[attached] ET_ADAPTED_UNIVERSE] */
typedef struct S1564 T1564;

/* [attached] DS_ARRAYED_LIST [[attached] ET_ECF_TARGET_PARENT] */
typedef struct S1566 T1566;

/* KL_CHARACTER_BUFFER */
typedef struct S1569 T1569;

/* [attached] SPECIAL [[attached] ET_QUERY] */
typedef struct S1572 T1572;

/* ET_ALIAS_NAME */
typedef struct S1573 T1573;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_QUERY] */
typedef struct S1574 T1574;

/* [attached] SPECIAL [[attached] ET_PROCEDURE] */
typedef struct S1575 T1575;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_PROCEDURE] */
typedef struct S1576 T1576;

/* KL_STRING_INPUT_STREAM */
typedef struct S1577 T1577;

/* ET_TEXT_GROUP */
typedef struct S1578 T1578;

/* ET_OBJECT_TEST_LIST */
typedef struct S1583 T1583;

/* ET_ACROSS_COMPONENT_LIST */
typedef struct S1585 T1585;

/* ET_CREATOR */
typedef struct S1625 T1625;

/* ET_FEATURE_CLAUSE */
typedef struct S1631 T1631;

/* ET_LOCAL_VARIABLE */
typedef struct S1655 T1655;

/* ET_ELSEIF_PART */
typedef struct S1675 T1675;

/* ET_ELSEIF_EXPRESSION */
typedef struct S1676 T1676;

/* ET_WHEN_PART */
typedef struct S1677 T1677;

/* ET_CONSTRAINED_FORMAL_PARAMETER */
typedef struct S1712 T1712;

/* ET_FILE_POSITION */
typedef struct S1716 T1716;

/* ET_TAGGED_ASSERTION */
typedef struct S1718 T1718;

/* [attached] DS_QUICK_SORTER [[attached] ET_QUERY] */
typedef struct S1721 T1721;

/* [attached] DS_QUICK_SORTER [[attached] ET_PROCEDURE] */
typedef struct S1722 T1722;

/* ET_REDECLARED_FEATURE */
typedef struct S1724 T1724;

/* ET_CONVERT_FEATURE_LIST */
typedef struct S1726 T1726;

/* ET_ALIASED_FEATURE_NAME */
typedef struct S1728 T1728;

/* ET_FEATURE_IDS */
typedef struct S1730 T1730;

/* ET_RENAME */
typedef struct S1731 T1731;

/* ET_POSTCONDITIONS */
typedef struct S1734 T1734;

/* ET_INVARIANTS */
typedef struct S1735 T1735;

/* [attached] TYPE [[attached] INTEGER_32] */
#define T1747 EIF_TYPE_OBJ

/* CONSOLE */
typedef struct S1748 T1748;

/* ET_LIKE_N */
typedef struct S1749 T1749;

/* [attached] DS_HASH_TABLE_CURSOR [[attached] ET_DYNAMIC_TYPE_SET, [attached] ET_DYNAMIC_TYPE] */
typedef struct S1750 T1750;

/* [attached] SPECIAL [[attached] ET_BASE_TYPE] */
typedef struct S1753 T1753;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_BASE_TYPE] */
typedef struct S1754 T1754;

/* [attached] ARRAY [[attached] IMMUTABLE_STRING_32] */
typedef struct S1755 T1755;

/* [attached] SPECIAL [[attached] IMMUTABLE_STRING_32] */
typedef struct S1756 T1756;

/* [attached] DS_HASH_TABLE [detachable DS_HASH_SET [[attached] STRING_8], [attached] STRING_8] */
typedef struct S1757 T1757;

/* [attached] DS_ARRAYED_LIST [[attached] ET_XACE_TARGET] */
typedef struct S1758 T1758;

/* [attached] DS_ARRAYED_LIST [[attached] ET_XACE_CLUSTER] */
typedef struct S1759 T1759;

/* [attached] DS_LINKED_LIST [[attached] ET_XACE_CLASS_OPTIONS] */
typedef struct S1762 T1762;

/* [attached] DS_ARRAYED_STACK [[attached] YY_BUFFER] */
typedef struct S1763 T1763;

/* UT_SYNTAX_ERROR */
typedef struct S1764 T1764;

/* UT_TOO_MANY_INCLUDES_ERROR */
typedef struct S1765 T1765;

/* ET_DYNAMIC_TARGET_LIST */
typedef struct S1768 T1768;

/* [attached] SPECIAL [[attached] NATIVE_STRING] */
typedef struct S1771 T1771;

/* [attached] ARRAY [detachable STRING_8] */
typedef struct S1772 T1772;

/* XM_NAMESPACE_RESOLVER_CONTEXT */
typedef struct S1773 T1773;

/* [attached] DS_LINKED_QUEUE [detachable STRING_8] */
typedef struct S1776 T1776;

/* [attached] DS_LINKED_QUEUE [[attached] STRING_8] */
typedef struct S1777 T1777;

/* [attached] DS_LINKED_LIST_CURSOR [[attached] DS_PAIR [[attached] XM_POSITION, [attached] XM_NODE]] */
typedef struct S1778 T1778;

/* [attached] DS_LINKED_LIST [[attached] DS_PAIR [[attached] XM_POSITION, [attached] XM_NODE]] */
typedef struct S1779 T1779;

/* [attached] DS_PAIR [[attached] XM_POSITION, [attached] XM_NODE] */
typedef struct S1780 T1780;

/* [attached] KL_EQUALITY_TESTER [[attached] XM_NAMESPACE] */
typedef struct S1783 T1783;

/* [attached] DS_HASH_SET_CURSOR [[attached] XM_NAMESPACE] */
typedef struct S1784 T1784;

/* [attached] KL_SPECIAL_ROUTINES [[attached] XM_NAMESPACE] */
typedef struct S1785 T1785;

/* [attached] SPECIAL [[attached] XM_NAMESPACE] */
typedef struct S1786 T1786;

/* [attached] DS_HASH_SET_CURSOR [[attached] ET_DOTNET_ASSEMBLY] */
typedef struct S1787 T1787;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_DOTNET_ASSEMBLY] */
typedef struct S1788 T1788;

/* [attached] SPECIAL [[attached] ET_DOTNET_ASSEMBLY] */
typedef struct S1789 T1789;

/* [attached] DS_ARRAYED_LIST [[attached] ET_DOTNET_ASSEMBLY] */
typedef struct S1792 T1792;

/* [attached] SPECIAL [[attached] ET_XACE_MOUNTED_LIBRARY] */
typedef struct S1793 T1793;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_XACE_MOUNTED_LIBRARY] */
typedef struct S1794 T1794;

/* [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_XACE_MOUNTED_LIBRARY] */
typedef struct S1795 T1795;

/* [attached] DS_ARRAYED_LIST [[attached] ET_MASTER_CLASS] */
typedef struct S1796 T1796;

/* [attached] DS_HASH_SET_CURSOR [[attached] ET_INTERNAL_UNIVERSE] */
typedef struct S1797 T1797;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_INTERNAL_UNIVERSE] */
typedef struct S1801 T1801;

/* [attached] SPECIAL [[attached] ET_INTERNAL_UNIVERSE] */
typedef struct S1802 T1802;

/* ET_CLASS_TYPE_STATUS_CHECKER1 */
typedef struct S1803 T1803;

/* ET_CLASS_TYPE_STATUS_CHECKER2 */
typedef struct S1804 T1804;

/* ET_QUALIFIED_ANCHORED_TYPE_STATUS_CHECKER */
typedef struct S1805 T1805;

/* ET_CLASS_TYPE_STATUS_CHECKER3 */
typedef struct S1806 T1806;

/* [attached] DS_HASH_SET_CURSOR [[attached] ET_NAMED_CLASS] */
typedef struct S1807 T1807;

/* [attached] DS_HASH_SET_CURSOR [[attached] ET_UNIVERSE] */
typedef struct S1808 T1808;

/* [attached] SPECIAL [[attached] ET_UNIVERSE] */
typedef struct S1809 T1809;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_UNIVERSE] */
typedef struct S1812 T1812;

/* [attached] SPECIAL [[attached] ET_PARENT_ITEM] */
typedef struct S1814 T1814;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_PARENT_ITEM] */
typedef struct S1816 T1816;

/* [attached] SPECIAL [[attached] ET_CLIENT_ITEM] */
typedef struct S1817 T1817;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_CLIENT_ITEM] */
typedef struct S1818 T1818;

/* ET_BRACED_TYPE_LIST */
typedef struct S1820 T1820;

/* XM_EIFFEL_CHARACTER_ENTITY */
typedef struct S1822 T1822;

/* attached DS_BILINKABLE [[attached] XM_POSITION] */
typedef struct S1825 T1825;

/* [attached] DS_BILINKED_LIST_CURSOR [[attached] XM_POSITION] */
typedef struct S1826 T1826;

/* attached DS_LINKABLE [[attached] XM_EIFFEL_SCANNER] */
typedef struct S1828 T1828;

/* [attached] SPECIAL [[attached] XM_EIFFEL_ENTITY_DEF] */
typedef struct S1829 T1829;

/* [attached] DS_HASH_TABLE_CURSOR [[attached] XM_EIFFEL_ENTITY_DEF, [attached] STRING_8] */
typedef struct S1834 T1834;

/* [attached] KL_SPECIAL_ROUTINES [[attached] XM_EIFFEL_ENTITY_DEF] */
typedef struct S1835 T1835;

/* [attached] KL_EQUALITY_TESTER [[attached] XM_EIFFEL_PARSER_NAME] */
typedef struct S1839 T1839;

/* [attached] DS_HASH_SET_CURSOR [[attached] XM_EIFFEL_PARSER_NAME] */
typedef struct S1840 T1840;

/* [attached] DS_BILINKED_LIST [[attached] XM_DTD_ELEMENT_CONTENT] */
typedef struct S1841 T1841;

/* [attached] DS_BILINKED_LIST_CURSOR [[attached] XM_DTD_ATTRIBUTE_CONTENT] */
typedef struct S1842 T1842;

/* attached DS_BILINKABLE [[attached] XM_DTD_ATTRIBUTE_CONTENT] */
typedef struct S1844 T1844;

/* [attached] DS_BILINKED_LIST_CURSOR [[attached] STRING_8] */
typedef struct S1846 T1846;

/* attached DS_BILINKABLE [[attached] STRING_8] */
typedef struct S1848 T1848;

/* [attached] DS_ARRAYED_LIST [[attached] ET_CLUSTER] */
typedef struct S1849 T1849;

/* [attached] SPECIAL [[attached] ET_ADAPTED_LIBRARY] */
typedef struct S1850 T1850;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_ADAPTED_LIBRARY] */
typedef struct S1851 T1851;

/* [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_ADAPTED_LIBRARY] */
typedef struct S1852 T1852;

/* [attached] DS_HASH_SET_CURSOR [[attached] ET_LIBRARY] */
typedef struct S1854 T1854;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_LIBRARY] */
typedef struct S1855 T1855;

/* [attached] SPECIAL [[attached] ET_LIBRARY] */
typedef struct S1856 T1856;

/* [attached] SPECIAL [[attached] ET_ADAPTED_DOTNET_ASSEMBLY] */
typedef struct S1857 T1857;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_ADAPTED_DOTNET_ASSEMBLY] */
typedef struct S1858 T1858;

/* [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_ADAPTED_DOTNET_ASSEMBLY] */
typedef struct S1859 T1859;

/* [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_LACE_CLUSTER] */
typedef struct S1861 T1861;

/* [attached] SPECIAL [detachable ET_OBJECT_TEST_LIST] */
typedef struct S1862 T1862;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_OBJECT_TEST_LIST] */
typedef struct S1863 T1863;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_OBJECT_TEST_LIST] */
typedef struct S1864 T1864;

/* [attached] SPECIAL [[attached] ET_OBJECT_TEST_LIST] */
typedef struct S1865 T1865;

/* [attached] SPECIAL [detachable ET_ACROSS_COMPONENT_LIST] */
typedef struct S1866 T1866;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_ACROSS_COMPONENT_LIST] */
typedef struct S1867 T1867;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_ACROSS_COMPONENT_LIST] */
typedef struct S1868 T1868;

/* [attached] SPECIAL [[attached] ET_ACROSS_COMPONENT_LIST] */
typedef struct S1869 T1869;

/* [attached] SPECIAL [[attached] ET_ASSERTION_ITEM] */
typedef struct S1870 T1870;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_ASSERTION_ITEM] */
typedef struct S1871 T1871;

/* [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_ASSERTION_ITEM] */
typedef struct S1872 T1872;

/* [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_QUERY] */
typedef struct S1874 T1874;

/* [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_PROCEDURE] */
typedef struct S1876 T1876;

/* [attached] DS_ARRAYED_LIST_CURSOR [detachable ET_CONSTRAINT_TYPE] */
typedef struct S1878 T1878;

/* [attached] SPECIAL [[attached] ET_NAMED_CLASS] */
typedef struct S1880 T1880;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_NAMED_CLASS] */
typedef struct S1882 T1882;

/* [attached] DS_HASH_TABLE [[attached] INTEGER_32, [attached] ET_CLASS] */
typedef struct S1886 T1886;

/* [attached] DS_HASH_TABLE_CURSOR [[attached] ET_BASE_TYPE, [attached] ET_CLASS] */
typedef struct S1887 T1887;

/* [attached] DS_HASH_TOPOLOGICAL_SORTER [[attached] ET_FORMAL_PARAMETER] */
typedef struct S1892 T1892;

/* attached DS_ARRAYED_LIST [[attached] ET_FORMAL_PARAMETER] */
typedef struct S1894 T1894;

/* ET_FORMAL_PARAMETER_TYPE */
typedef struct S1895 T1895;

/* [attached] DS_HASH_TABLE_CURSOR [[attached] ET_FLATTENED_FEATURE, [attached] ET_FEATURE_NAME] */
typedef struct S1896 T1896;

/* [attached] SPECIAL [[attached] ET_FLATTENED_FEATURE] */
typedef struct S1897 T1897;

/* [attached] SPECIAL [[attached] ET_FEATURE_NAME] */
typedef struct S1903 T1903;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_FLATTENED_FEATURE] */
typedef struct S1904 T1904;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_FEATURE_NAME] */
typedef struct S1905 T1905;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_ALIAS_NAME] */
typedef struct S1909 T1909;

/* [attached] SPECIAL [[attached] ET_ALIAS_NAME] */
typedef struct S1910 T1910;

/* [attached] DS_HASH_TABLE_CURSOR [[attached] ET_FLATTENED_FEATURE, [attached] ET_ALIAS_NAME] */
typedef struct S1912 T1912;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_CLIENT_LIST] */
typedef struct S1913 T1913;

/* [attached] SPECIAL [[attached] ET_CLIENT_LIST] */
typedef struct S1914 T1914;

/* [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_CLIENT_LIST] */
typedef struct S1915 T1915;

/* [attached] DS_HASH_TABLE_CURSOR [[attached] ET_CLIENT, [attached] ET_CLASS] */
typedef struct S1917 T1917;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_CLIENT] */
typedef struct S1918 T1918;

/* [attached] SPECIAL [[attached] ET_CLIENT] */
typedef struct S1919 T1919;

/* [attached] DS_HASH_TABLE [[attached] ET_RENAME, [attached] ET_CALL_NAME] */
typedef struct S1920 T1920;

/* ET_CALL_NAME_TESTER */
typedef struct S1921 T1921;

/* [attached] DS_HASH_SET [[attached] ET_CALL_NAME] */
typedef struct S1923 T1923;

/* [attached] DS_HASH_TABLE [[attached] BOOLEAN, [attached] ET_CALL_NAME] */
typedef struct S1924 T1924;

/* [attached] DS_HASH_TABLE [[attached] ET_FEATURE_NAME, [attached] ET_ALIAS_NAME] */
typedef struct S1925 T1925;

/* [attached] DS_HASH_TABLE [[attached] ET_REPLICABLE_FEATURE, [attached] INTEGER_32] */
typedef struct S1926 T1926;

/* ET_REPLICATED_FEATURE */
typedef struct S1930 T1930;

/* ET_RENAME_LIST */
typedef struct S1931 T1931;

/* ET_EXPORT_LIST */
typedef struct S1932 T1932;

/* ET_ALL_EXPORT */
typedef struct S1933 T1933;

/* ET_FEATURE_EXPORT */
typedef struct S1934 T1934;

/* ET_KEYWORD_FEATURE_NAME_LIST */
typedef struct S1935 T1935;

/* [attached] DS_LINKED_LIST [[attached] ET_ADAPTED_FEATURE] */
typedef struct S1936 T1936;

/* [attached] DS_ARRAYED_LIST [[attached] ET_PARENT_FEATURE] */
typedef struct S1937 T1937;

/* ET_DOTNET_SIGNATURE_TESTER */
typedef struct S1938 T1938;

/* [attached] DS_HASH_SET [[attached] ET_DOTNET_FEATURE] */
typedef struct S1939 T1939;

/* [attached] DS_HASH_TABLE [[attached] DS_LINKED_LIST [[attached] ET_DOTNET_FEATURE], [attached] ET_DOTNET_FEATURE] */
typedef struct S1941 T1941;

/* [attached] DS_LINKED_LIST [[attached] ET_DOTNET_FEATURE] */
typedef struct S1943 T1943;

/* [attached] DS_HASH_TOPOLOGICAL_SORTER [[attached] ET_LIKE_FEATURE] */
typedef struct S1944 T1944;

/* attached DS_ARRAYED_LIST [[attached] ET_LIKE_FEATURE] */
typedef struct S1946 T1946;

/* [attached] DS_LINKED_LIST_CURSOR [[attached] ET_PARENT_FEATURE] */
typedef struct S1947 T1947;

/* attached DS_LINKED_LIST [[attached] ET_PARENT_FEATURE] */
typedef struct S1949 T1949;

/* [attached] DS_HASH_TABLE [[attached] DS_HASH_TABLE [[attached] TUPLE [detachable ARRAY [[attached] ET_TYPE], detachable ET_TYPE, [attached] NATURAL_8], [attached] ET_FEATURE_NAME], [attached] NATURAL_8] */
typedef struct S1950 T1950;

/* [attached] DS_HASH_TABLE_CURSOR [[attached] ET_FEATURE, [attached] INTEGER_32] */
typedef struct S1951 T1951;

/* [attached] TYPE [[attached] ET_FEATURE] */
#define T1952 EIF_TYPE_OBJ

/* [attached] DS_HASH_SET_CURSOR [[attached] ET_CLASS] */
typedef struct S1956 T1956;

/* [attached] TYPE [[attached] ET_CLASS] */
#define T1957 EIF_TYPE_OBJ

/* [attached] DS_HASH_SET_CURSOR [[attached] ET_PROCEDURE] */
typedef struct S1959 T1959;

/* [attached] DS_HASH_SET_CURSOR [[attached] ET_QUERY] */
typedef struct S1961 T1961;

/* [attached] SPECIAL [detachable DS_LINKABLE [[attached] INTEGER_32]] */
typedef struct S1963 T1963;

/* [attached] KL_SPECIAL_ROUTINES [detachable DS_LINKABLE [[attached] INTEGER_32]] */
typedef struct S1964 T1964;

/* [attached] DS_ARRAYED_LIST_CURSOR [detachable DS_LINKABLE [[attached] INTEGER_32]] */
typedef struct S1965 T1965;

/* [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_NAMED_OBJECT_TEST] */
typedef struct S1968 T1968;

/* [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_ACROSS_COMPONENT] */
typedef struct S1970 T1970;

/* [attached] DS_HASH_SET_CURSOR [[attached] INTEGER_32] */
typedef struct S1972 T1972;

/* [attached] KL_SPECIAL_ROUTINES [[attached] NATURAL_8] */
typedef struct S1978 T1978;

/* [attached] DS_HASH_TABLE_CURSOR [[attached] NATURAL_8, [attached] ET_CLASS_NAME] */
typedef struct S1979 T1979;

/* STRING_8_SEARCHER */
typedef struct S1980 T1980;

/* attached ARRAYED_LIST [[attached] INTEGER_32] */
typedef struct S1982 T1982;

/* ET_XACE_INVALID_EXPRESSION_ERROR */
typedef struct S1986 T1986;

/* [attached] DS_HASH_TABLE [[attached] INTEGER_32, [attached] ET_UNIVERSE] */
typedef struct S1988 T1988;

/* [attached] DS_HASH_TABLE_CURSOR [[attached] INTEGER_32, [attached] ET_UNIVERSE] */
typedef struct S1989 T1989;

/* [attached] TUPLE [ET_ECF_SYSTEM, [attached] DS_HASH_TABLE [[attached] INTEGER_32, [attached] ET_UNIVERSE], INTEGER_32] */
typedef struct S1990 T1990;

/* [attached] KL_AGENT_ROUTINES [[attached] ET_UNIVERSE] */
typedef struct S1991 T1991;

/* [attached] PREDICATE [[attached] TUPLE [[attached] ET_UNIVERSE]] */
typedef struct S1992 T1992;

/* [attached] TUPLE [[attached] DS_HASH_TABLE [[attached] INTEGER_32, [attached] ET_UNIVERSE]] */
typedef struct S1993 T1993;

/* [attached] TUPLE [[attached] KL_AGENT_ROUTINES [[attached] ET_UNIVERSE], [attached] PREDICATE [[attached] TUPLE [[attached] ET_UNIVERSE]]] */
typedef struct S1994 T1994;

/* ET_DYNAMIC_AGENT_OPERAND_PUSH_TYPE_SET */
typedef struct S1998 T1998;

/* TYPED_POINTER [[attached] NATURAL_16] */
typedef struct S2000 T2000;
extern T0* GE_boxed2000(T2000 a1);
typedef struct Sb2000 Tb2000;

/* [attached] DS_LINKED_LIST_CURSOR [[attached] XM_DOCUMENT_NODE] */
typedef struct S2001 T2001;

/* XM_COMMENT */
typedef struct S2002 T2002;

/* XM_PROCESSING_INSTRUCTION */
typedef struct S2003 T2003;

/* XM_CHARACTER_DATA */
typedef struct S2004 T2004;

/* [attached] DS_CELL [detachable ET_CLASS] */
typedef struct S2009 T2009;

/* [attached] TUPLE [detachable DS_CELL [detachable ET_CLASS]] */
typedef struct S2010 T2010;

/* ST_SPLITTER */
typedef struct S2012 T2012;

/* ET_CLUSTER_DEPENDENCE_CONSTRAINT */
typedef struct S2014 T2014;

/* [attached] TUPLE [ET_EIFFEL_PREPARSER, [attached] DS_ARRAYED_LIST [[attached] STRING_8]] */
typedef struct S2015 T2015;

/* [attached] DS_HASH_TABLE [[attached] INTEGER_32, [attached] STRING_8] */
typedef struct S2022 T2022;

/* ET_LOCAL_VARIABLE_LIST */
typedef struct S2027 T2027;

/* ET_TRUE_CONSTANT */
typedef struct S2032 T2032;

/* ET_SEEDED_QUERY_COMPARATOR */
typedef struct S2034 T2034;

/* ET_SEEDED_PROCEDURE_COMPARATOR */
typedef struct S2037 T2037;

/* ET_CLIENTS */
typedef struct S2040 T2040;

/* [attached] TUPLE [detachable ET_CLIENT_LIST] */
typedef struct S2042 T2042;

/* ET_FREE_OPERATOR */
typedef struct S2044 T2044;

/* ET_C3_CHARACTER_CONSTANT */
typedef struct S2045 T2045;

/* ET_VERBATIM_STRING */
typedef struct S2046 T2046;

/* ET_SPECIAL_MANIFEST_STRING */
typedef struct S2047 T2047;

/* ET_UNDERSCORED_INTEGER_CONSTANT */
typedef struct S2048 T2048;

/* ET_HEXADECIMAL_INTEGER_CONSTANT */
typedef struct S2049 T2049;

/* ET_OCTAL_INTEGER_CONSTANT */
typedef struct S2050 T2050;

/* ET_BINARY_INTEGER_CONSTANT */
typedef struct S2051 T2051;

/* ET_REGULAR_REAL_CONSTANT */
typedef struct S2052 T2052;

/* ET_UNDERSCORED_REAL_CONSTANT */
typedef struct S2053 T2053;

/* ET_SYMBOL_OPERATOR */
typedef struct S2054 T2054;

/* ET_SEMICOLON_SYMBOL */
typedef struct S2055 T2055;

/* ET_QUESTION_MARK_SYMBOL */
typedef struct S2056 T2056;

/* ET_KEYWORD_OPERATOR */
typedef struct S2057 T2057;

/* ET_VOID */
typedef struct S2058 T2058;

/* ET_AGENT_KEYWORD */
typedef struct S2059 T2059;

/* ET_FALSE_CONSTANT */
typedef struct S2060 T2060;

/* ET_RETRY_INSTRUCTION */
typedef struct S2061 T2061;

/* ET_PRECURSOR_KEYWORD */
typedef struct S2062 T2062;

/* ET_C1_CHARACTER_CONSTANT */
typedef struct S2063 T2063;

/* ET_C2_CHARACTER_CONSTANT */
typedef struct S2064 T2064;

/* ET_INDEXING_LIST */
typedef struct S2065 T2065;

/* ET_TAGGED_INDEXING */
typedef struct S2066 T2066;

/* ET_INDEXING_TERM_LIST */
typedef struct S2068 T2068;

/* ET_INDEXING */
typedef struct S2069 T2069;

/* ET_CUSTOM_ATTRIBUTE */
typedef struct S2070 T2070;

/* ET_ATTACHMENT_SYMBOL_SEPARATE_KEYWORD */
typedef struct S2071 T2071;

/* ET_CONSTRAINT_ACTUAL_PARAMETER_LIST */
typedef struct S2072 T2072;

/* ET_CONSTRAINT_LABELED_ACTUAL_PARAMETER */
typedef struct S2073 T2073;

/* ET_CONSTRAINT_LABELED_COMMA_ACTUAL_PARAMETER */
typedef struct S2074 T2074;

/* ET_CONVERT_FUNCTION */
typedef struct S2075 T2075;

/* ET_CONVERT_PROCEDURE */
typedef struct S2076 T2076;

/* ET_FEATURE_CLAUSE_LIST */
typedef struct S2077 T2077;

/* ET_DO_FUNCTION */
typedef struct S2079 T2079;

/* ET_ONCE_FUNCTION */
typedef struct S2080 T2080;

/* ET_DEFERRED_FUNCTION */
typedef struct S2081 T2081;

/* ET_ONCE_PROCEDURE */
typedef struct S2082 T2082;

/* ET_PREFIX_NAME */
typedef struct S2083 T2083;

/* ET_PREFIX_FREE_NAME */
typedef struct S2084 T2084;

/* ET_INFIX_NAME */
typedef struct S2085 T2085;

/* ET_ALIAS_FREE_NAME */
typedef struct S2086 T2086;

/* ET_FORMAL_COMMA_ARGUMENT */
typedef struct S2087 T2087;

/* ET_LOCAL_COMMA_VARIABLE */
typedef struct S2089 T2089;

/* ET_CLASS_ASSERTION */
typedef struct S2091 T2091;

/* ET_VARIANT */
typedef struct S2092 T2092;

/* ET_LABELED_ACTUAL_PARAMETER */
typedef struct S2093 T2093;

/* ET_LABELED_COMMA_ACTUAL_PARAMETER */
typedef struct S2095 T2095;

/* ET_QUALIFIED_LIKE_BRACED_TYPE */
typedef struct S2097 T2097;

/* ET_QUALIFIED_LIKE_TYPE */
typedef struct S2098 T2098;

/* ET_ASSIGNER_INSTRUCTION */
typedef struct S2100 T2100;

/* ET_ASSIGNMENT */
typedef struct S2101 T2101;

/* ET_ASSIGNMENT_ATTEMPT */
typedef struct S2102 T2102;

/* ET_KEYWORD_EXPRESSION */
typedef struct S2103 T2103;

/* ET_LOOP_INSTRUCTION */
typedef struct S2104 T2104;

/* ET_BANG_INSTRUCTION */
typedef struct S2106 T2106;

/* ET_CREATE_INSTRUCTION */
typedef struct S2107 T2107;

/* ET_CREATION_REGION */
typedef struct S2108 T2108;

/* ET_IF_INSTRUCTION */
typedef struct S2109 T2109;

/* ET_ELSEIF_PART_LIST */
typedef struct S2110 T2110;

/* ET_IF_EXPRESSION */
typedef struct S2111 T2111;

/* ET_ELSEIF_EXPRESSION_LIST */
typedef struct S2112 T2112;

/* ET_INSPECT_INSTRUCTION */
typedef struct S2113 T2113;

/* ET_WHEN_PART_LIST */
typedef struct S2114 T2114;

/* ET_CHOICE_LIST */
typedef struct S2115 T2115;

/* ET_CHOICE_RANGE */
typedef struct S2116 T2116;

/* ET_DEBUG_INSTRUCTION */
typedef struct S2118 T2118;

/* ET_MANIFEST_STRING_LIST */
typedef struct S2119 T2119;

/* ET_PRECURSOR_INSTRUCTION */
typedef struct S2120 T2120;

/* ET_STATIC_CALL_INSTRUCTION */
typedef struct S2121 T2121;

/* ET_STATIC_CALL_EXPRESSION */
typedef struct S2122 T2122;

/* ET_PRECURSOR_EXPRESSION */
typedef struct S2123 T2123;

/* ET_CURRENT_ADDRESS */
typedef struct S2124 T2124;

/* ET_RESULT_ADDRESS */
typedef struct S2125 T2125;

/* ET_EXPRESSION_ADDRESS */
typedef struct S2126 T2126;

/* ET_PARENTHESIZED_EXPRESSION */
typedef struct S2127 T2127;

/* ET_INFIX_EXPRESSION */
typedef struct S2128 T2128;

/* ET_INFIX_AND_THEN_OPERATOR */
typedef struct S2130 T2130;

/* ET_INFIX_OR_ELSE_OPERATOR */
typedef struct S2132 T2132;

/* ET_MANIFEST_TYPE */
typedef struct S2133 T2133;

/* ET_PREFIX_EXPRESSION */
typedef struct S2134 T2134;

/* ET_OLD_EXPRESSION */
typedef struct S2135 T2135;

/* ET_BRACKET_EXPRESSION */
typedef struct S2136 T2136;

/* ET_MANIFEST_ARRAY */
typedef struct S2137 T2137;

/* ET_STRIP_EXPRESSION */
typedef struct S2138 T2138;

/* ET_CALL_AGENT */
typedef struct S2139 T2139;

/* ET_DO_FUNCTION_INLINE_AGENT */
typedef struct S2140 T2140;

/* ET_ONCE_FUNCTION_INLINE_AGENT */
typedef struct S2141 T2141;

/* ET_EXTERNAL_FUNCTION_INLINE_AGENT */
typedef struct S2142 T2142;

/* ET_DO_PROCEDURE_INLINE_AGENT */
typedef struct S2144 T2144;

/* ET_ONCE_PROCEDURE_INLINE_AGENT */
typedef struct S2145 T2145;

/* ET_EXTERNAL_PROCEDURE_INLINE_AGENT */
typedef struct S2146 T2146;

/* ET_AGENT_OPEN_TARGET */
typedef struct S2147 T2147;

/* ET_AGENT_ARGUMENT_OPERAND_LIST */
typedef struct S2148 T2148;

/* ET_AGENT_TYPED_OPEN_ARGUMENT */
typedef struct S2149 T2149;

/* ET_CONSTRAINT_GENERIC_NAMED_TYPE */
typedef struct S2150 T2150;

/* ET_CONSTRAINT_NAMED_TYPE */
typedef struct S2151 T2151;

/* ET_EXTERNAL_PROCEDURE */
typedef struct S2152 T2152;

/* ET_LOOP_INVARIANTS */
typedef struct S2153 T2153;

/* ET_LIKE_FEATURE */
typedef struct S2154 T2154;

/* ET_CHECK_INSTRUCTION */
typedef struct S2155 T2155;

/* ET_ACROSS_INSTRUCTION */
typedef struct S2156 T2156;

/* ET_UNQUALIFIED_CALL_INSTRUCTION */
typedef struct S2157 T2157;

/* ET_FEATURE_ADDRESS */
typedef struct S2158 T2158;

/* ET_OLD_OBJECT_TEST */
typedef struct S2159 T2159;

/* ET_ONCE_MANIFEST_STRING */
typedef struct S2160 T2160;

/* ET_ACROSS_EXPRESSION */
typedef struct S2161 T2161;

/* [attached] SPECIAL [[attached] ET_INSTRUCTION] */
typedef struct S2163 T2163;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_INSTRUCTION] */
typedef struct S2164 T2164;

/* HEXADECIMAL_STRING_TO_INTEGER_CONVERTER */
typedef struct S2166 T2166;

/* [attached] TUPLE [ET_MASTER_CLASS, ET_SYSTEM_PROCESSOR] */
typedef struct S2169 T2169;

/* attached DS_SPARSE_TABLE_KEYS [[attached] ET_CLIENT, [attached] ET_CLASS] */
typedef struct S2191 T2191;

/* [attached] SPECIAL [[attached] ET_FORMAL_PARAMETER_ITEM] */
typedef struct S2196 T2196;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_FORMAL_PARAMETER_ITEM] */
typedef struct S2197 T2197;

/* [attached] SPECIAL [[attached] ET_FEATURE_NAME_ITEM] */
typedef struct S2199 T2199;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_FEATURE_NAME_ITEM] */
typedef struct S2200 T2200;

/* INTEGER_OVERFLOW_CHECKER */
typedef struct S2202 T2202;

/* [attached] SPECIAL [[attached] ARRAY [[attached] INTEGER_32]] */
typedef struct S2203 T2203;

/* [attached] SPECIAL [[attached] SPECIAL [[attached] ARRAY [[attached] INTEGER_32]]] */
typedef struct S2204 T2204;

/* [attached] DS_LINKED_LIST_CURSOR [[attached] ET_XACE_FEATURE_OPTIONS] */
typedef struct S2205 T2205;

/* attached DS_LINKABLE [[attached] ET_XACE_FEATURE_OPTIONS] */
typedef struct S2207 T2207;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_ECF_NOTE_ELEMENT] */
typedef struct S2208 T2208;

/* [attached] SPECIAL [[attached] ET_ECF_NOTE_ELEMENT] */
typedef struct S2209 T2209;

/* [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_ECF_NOTE_ELEMENT] */
typedef struct S2210 T2210;

/* [attached] DS_ARRAYED_LIST [[attached] ET_ECF_ADAPTED_DOTNET_ASSEMBLY] */
typedef struct S2211 T2211;

/* [attached] DS_ARRAYED_LIST [[attached] ET_ECF_CLUSTER] */
typedef struct S2212 T2212;

/* [attached] DS_ARRAYED_LIST [[attached] ET_ECF_EXTERNAL_CFLAG] */
typedef struct S2213 T2213;

/* [attached] DS_ARRAYED_LIST [[attached] ET_ECF_EXTERNAL_INCLUDE] */
typedef struct S2214 T2214;

/* [attached] DS_ARRAYED_LIST [[attached] ET_ECF_EXTERNAL_LIBRARY] */
typedef struct S2215 T2215;

/* [attached] DS_ARRAYED_LIST [[attached] ET_ECF_EXTERNAL_LINKER_FLAG] */
typedef struct S2216 T2216;

/* [attached] DS_ARRAYED_LIST [[attached] ET_ECF_EXTERNAL_MAKE] */
typedef struct S2217 T2217;

/* [attached] DS_ARRAYED_LIST [[attached] ET_ECF_EXTERNAL_OBJECT] */
typedef struct S2218 T2218;

/* [attached] DS_ARRAYED_LIST [[attached] ET_ECF_EXTERNAL_RESOURCE] */
typedef struct S2219 T2219;

/* [attached] DS_ARRAYED_LIST [[attached] ET_ECF_FILE_RULE] */
typedef struct S2221 T2221;

/* [attached] DS_ARRAYED_LIST [[attached] ET_ECF_ADAPTED_LIBRARY] */
typedef struct S2222 T2222;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_ECF_ACTION] */
typedef struct S2223 T2223;

/* [attached] SPECIAL [[attached] ET_ECF_ACTION] */
typedef struct S2224 T2224;

/* [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_ECF_ACTION] */
typedef struct S2225 T2225;

/* [attached] DS_ARRAYED_LIST [[attached] ET_ECF_ANDED_CONDITIONS] */
typedef struct S2226 T2226;

/* [attached] DS_ARRAYED_LIST [[attached] ET_ECF_CONDITION_ITEM] */
typedef struct S2227 T2227;

/* [attached] SPECIAL [[attached] ET_ECF_OPTIONS] */
typedef struct S2228 T2228;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_ECF_OPTIONS] */
typedef struct S2233 T2233;

/* [attached] DS_HASH_TABLE_CURSOR [[attached] ET_ECF_OPTIONS, [attached] STRING_8] */
typedef struct S2234 T2234;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_ECF_VISIBLE_CLASS] */
typedef struct S2235 T2235;

/* [attached] SPECIAL [[attached] ET_ECF_VISIBLE_CLASS] */
typedef struct S2236 T2236;

/* [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_ECF_VISIBLE_CLASS] */
typedef struct S2237 T2237;

/* [attached] SPECIAL [[attached] ET_PARENT_LIST] */
typedef struct S2238 T2238;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_PARENT_LIST] */
typedef struct S2239 T2239;

/* [attached] SPECIAL [[attached] ET_CREATOR] */
typedef struct S2240 T2240;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_CREATOR] */
typedef struct S2241 T2241;

/* attached DS_LINKABLE [[attached] XM_DOCUMENT_NODE] */
typedef struct S2243 T2243;

/* [attached] DS_LINKED_QUEUE [[attached] CHARACTER_8] */
typedef struct S2249 T2249;

/* UC_UTF16_ROUTINES */
typedef struct S2250 T2250;

/* [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_ECF_TARGET] */
typedef struct S2251 T2251;

/* [attached] SPECIAL [[attached] ET_ADAPTED_UNIVERSE] */
typedef struct S2252 T2252;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_ADAPTED_UNIVERSE] */
typedef struct S2253 T2253;

/* [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_ADAPTED_UNIVERSE] */
typedef struct S2254 T2254;

/* [attached] SPECIAL [[attached] ET_ECF_TARGET_PARENT] */
typedef struct S2256 T2256;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_ECF_TARGET_PARENT] */
typedef struct S2257 T2257;

/* [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_ECF_TARGET_PARENT] */
typedef struct S2258 T2258;

/* [attached] SPECIAL [[attached] ET_CONVERT_FEATURE_ITEM] */
typedef struct S2260 T2260;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_CONVERT_FEATURE_ITEM] */
typedef struct S2261 T2261;

/* [attached] SPECIAL [detachable DS_HASH_SET [[attached] STRING_8]] */
typedef struct S2262 T2262;

/* [attached] KL_SPECIAL_ROUTINES [detachable DS_HASH_SET [[attached] STRING_8]] */
typedef struct S2266 T2266;

/* [attached] DS_HASH_TABLE_CURSOR [detachable DS_HASH_SET [[attached] STRING_8], [attached] STRING_8] */
typedef struct S2267 T2267;

/* [attached] SPECIAL [[attached] ET_XACE_TARGET] */
typedef struct S2268 T2268;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_XACE_TARGET] */
typedef struct S2269 T2269;

/* [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_XACE_TARGET] */
typedef struct S2270 T2270;

/* [attached] SPECIAL [[attached] ET_XACE_CLUSTER] */
typedef struct S2271 T2271;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_XACE_CLUSTER] */
typedef struct S2272 T2272;

/* [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_XACE_CLUSTER] */
typedef struct S2273 T2273;

/* [attached] DS_LINKED_LIST_CURSOR [[attached] ET_XACE_CLASS_OPTIONS] */
typedef struct S2274 T2274;

/* attached DS_LINKABLE [[attached] ET_XACE_CLASS_OPTIONS] */
typedef struct S2276 T2276;

/* [attached] SPECIAL [[attached] YY_BUFFER] */
typedef struct S2277 T2277;

/* [attached] KL_SPECIAL_ROUTINES [[attached] YY_BUFFER] */
typedef struct S2278 T2278;

/* [attached] SPECIAL [[attached] ET_DYNAMIC_TARGET] */
typedef struct S2279 T2279;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_DYNAMIC_TARGET] */
typedef struct S2280 T2280;

/* [attached] DS_BILINKED_LIST [[attached] DS_HASH_TABLE [[attached] STRING_8, [attached] STRING_8]] */
typedef struct S2282 T2282;

/* [attached] DS_BILINKED_LIST_CURSOR [[attached] DS_HASH_TABLE [[attached] STRING_8, [attached] STRING_8]] */
typedef struct S2283 T2283;

/* attached DS_LINKABLE [detachable STRING_8] */
typedef struct S2285 T2285;

/* attached DS_LINKABLE [[attached] DS_PAIR [[attached] XM_POSITION, [attached] XM_NODE]] */
typedef struct S2287 T2287;

/* [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_DOTNET_ASSEMBLY] */
typedef struct S2288 T2288;

/* [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_MASTER_CLASS] */
typedef struct S2289 T2289;

/* [attached] SPECIAL [[attached] ET_TYPE_ITEM] */
typedef struct S2293 T2293;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_TYPE_ITEM] */
typedef struct S2294 T2294;

/* [attached] DS_BILINKED_LIST_CURSOR [[attached] XM_DTD_ELEMENT_CONTENT] */
typedef struct S2295 T2295;

/* attached DS_BILINKABLE [[attached] XM_DTD_ELEMENT_CONTENT] */
typedef struct S2296 T2296;

/* [attached] SPECIAL [[attached] ET_CLUSTER] */
typedef struct S2298 T2298;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_CLUSTER] */
typedef struct S2299 T2299;

/* [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_CLUSTER] */
typedef struct S2300 T2300;

/* [attached] DS_HASH_TABLE_CURSOR [[attached] INTEGER_32, [attached] ET_CLASS] */
typedef struct S2302 T2302;

/* [attached] DS_HASH_TABLE [[attached] INTEGER_32, [attached] ET_FORMAL_PARAMETER] */
typedef struct S2303 T2303;

/* [attached] SPECIAL [[attached] ET_FORMAL_PARAMETER] */
typedef struct S2304 T2304;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_FORMAL_PARAMETER] */
typedef struct S2305 T2305;

/* [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_FORMAL_PARAMETER] */
typedef struct S2306 T2306;

/* [attached] DS_HASH_TABLE_CURSOR [[attached] ET_RENAME, [attached] ET_CALL_NAME] */
typedef struct S2308 T2308;

/* [attached] SPECIAL [[attached] ET_RENAME] */
typedef struct S2309 T2309;

/* [attached] TYPE [[attached] ET_RENAME] */
#define T2314 EIF_TYPE_OBJ

/* [attached] TYPE [[attached] ET_CALL_NAME] */
#define T2315 EIF_TYPE_OBJ

/* [attached] SPECIAL [[attached] ET_CALL_NAME] */
typedef struct S2317 T2317;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_RENAME] */
typedef struct S2318 T2318;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_CALL_NAME] */
typedef struct S2319 T2319;

/* [attached] DS_HASH_SET_CURSOR [[attached] ET_CALL_NAME] */
typedef struct S2320 T2320;

/* [attached] DS_HASH_TABLE_CURSOR [[attached] BOOLEAN, [attached] ET_CALL_NAME] */
typedef struct S2322 T2322;

/* [attached] DS_HASH_TABLE_CURSOR [[attached] ET_FEATURE_NAME, [attached] ET_ALIAS_NAME] */
typedef struct S2329 T2329;

/* [attached] DS_HASH_TABLE_CURSOR [[attached] ET_REPLICABLE_FEATURE, [attached] INTEGER_32] */
typedef struct S2330 T2330;

/* [attached] SPECIAL [[attached] ET_REPLICABLE_FEATURE] */
typedef struct S2331 T2331;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_REPLICABLE_FEATURE] */
typedef struct S2334 T2334;

/* [attached] SPECIAL [[attached] ET_RENAME_ITEM] */
typedef struct S2336 T2336;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_RENAME_ITEM] */
typedef struct S2337 T2337;

/* [attached] SPECIAL [[attached] ET_EXPORT] */
typedef struct S2339 T2339;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_EXPORT] */
typedef struct S2340 T2340;

/* [attached] DS_LINKED_LIST_CURSOR [[attached] ET_ADAPTED_FEATURE] */
typedef struct S2341 T2341;

/* attached DS_LINKABLE [[attached] ET_ADAPTED_FEATURE] */
typedef struct S2343 T2343;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_PARENT_FEATURE] */
typedef struct S2344 T2344;

/* [attached] SPECIAL [[attached] ET_PARENT_FEATURE] */
typedef struct S2345 T2345;

/* [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_PARENT_FEATURE] */
typedef struct S2346 T2346;

/* [attached] SPECIAL [[attached] ET_DOTNET_FEATURE] */
typedef struct S2347 T2347;

/* [attached] DS_HASH_SET_CURSOR [[attached] ET_DOTNET_FEATURE] */
typedef struct S2348 T2348;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_DOTNET_FEATURE] */
typedef struct S2349 T2349;

/* [attached] SPECIAL [[attached] DS_LINKED_LIST [[attached] ET_DOTNET_FEATURE]] */
typedef struct S2352 T2352;

/* [attached] DS_HASH_TABLE_CURSOR [[attached] DS_LINKED_LIST [[attached] ET_DOTNET_FEATURE], [attached] ET_DOTNET_FEATURE] */
typedef struct S2357 T2357;

/* [attached] KL_SPECIAL_ROUTINES [[attached] DS_LINKED_LIST [[attached] ET_DOTNET_FEATURE]] */
typedef struct S2358 T2358;

/* [attached] DS_LINKED_LIST_CURSOR [[attached] ET_DOTNET_FEATURE] */
typedef struct S2359 T2359;

/* attached DS_LINKABLE [[attached] ET_DOTNET_FEATURE] */
typedef struct S2361 T2361;

/* [attached] DS_HASH_TABLE [[attached] INTEGER_32, [attached] ET_LIKE_FEATURE] */
typedef struct S2362 T2362;

/* [attached] SPECIAL [[attached] ET_LIKE_FEATURE] */
typedef struct S2363 T2363;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_LIKE_FEATURE] */
typedef struct S2364 T2364;

/* [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_LIKE_FEATURE] */
typedef struct S2365 T2365;

/* [attached] DS_HASH_TABLE_CURSOR [[attached] DS_HASH_TABLE [[attached] TUPLE [detachable ARRAY [[attached] ET_TYPE], detachable ET_TYPE, [attached] NATURAL_8], [attached] ET_FEATURE_NAME], [attached] NATURAL_8] */
typedef struct S2368 T2368;

/* [attached] KL_SPECIAL_ROUTINES [[attached] DS_HASH_TABLE [[attached] TUPLE [detachable ARRAY [[attached] ET_TYPE], detachable ET_TYPE, [attached] NATURAL_8], [attached] ET_FEATURE_NAME]] */
typedef struct S2369 T2369;

/* [attached] DS_HASH_TABLE [[attached] TUPLE [detachable ARRAY [[attached] ET_TYPE], detachable ET_TYPE, [attached] NATURAL_8], [attached] ET_FEATURE_NAME] */
typedef struct S2370 T2370;

/* [attached] SPECIAL [[attached] DS_HASH_TABLE [[attached] TUPLE [detachable ARRAY [[attached] ET_TYPE], detachable ET_TYPE, [attached] NATURAL_8], [attached] ET_FEATURE_NAME]] */
typedef struct S2371 T2371;

/* ET_CLUSTER_ERROR */
typedef struct S2372 T2372;

/* ET_SYNTAX_ERROR */
typedef struct S2373 T2373;

/* ET_DOTNET_ASSEMBLY_ERROR */
typedef struct S2374 T2374;

/* [attached] TUPLE [ET_SYSTEM, [attached] DS_HASH_TABLE [[attached] INTEGER_32, [attached] ET_UNIVERSE], INTEGER_32] */
typedef struct S2377 T2377;

/* [attached] TUPLE [[attached] ET_ECF_CLUSTER] */
typedef struct S2378 T2378;

/* [attached] PROCEDURE [[attached] TUPLE [[attached] ET_ECF_CLUSTER]] */
typedef struct S2379 T2379;

/* [attached] TUPLE [ET_ECF_TARGET, STRING_8] */
typedef struct S2380 T2380;

/* [attached] TUPLE [[attached] ET_ECF_ADAPTED_LIBRARY] */
typedef struct S2382 T2382;

/* [attached] PROCEDURE [[attached] TUPLE [[attached] ET_ECF_ADAPTED_LIBRARY]] */
typedef struct S2383 T2383;

/* [attached] TUPLE [[attached] ET_ECF_ADAPTED_DOTNET_ASSEMBLY] */
typedef struct S2384 T2384;

/* [attached] PROCEDURE [[attached] TUPLE [[attached] ET_ECF_ADAPTED_DOTNET_ASSEMBLY]] */
typedef struct S2385 T2385;

/* [attached] PROCEDURE [[attached] TUPLE [[attached] STRING_8]] */
typedef struct S2388 T2388;

/* [attached] TUPLE [ET_ECF_OPTIONS, STRING_8] */
typedef struct S2389 T2389;

/* [attached] DS_HASH_TABLE [[attached] DS_HASH_TABLE [[attached] INTEGER_32, [attached] STRING_8], [attached] STRING_8] */
typedef struct S2395 T2395;

/* [attached] TUPLE [ET_XACE_SYSTEM, [attached] DS_HASH_TABLE [[attached] INTEGER_32, [attached] ET_UNIVERSE], INTEGER_32] */
typedef struct S2397 T2397;

/* [attached] TUPLE [ET_LACE_SYSTEM, [attached] DS_HASH_TABLE [[attached] INTEGER_32, [attached] ET_UNIVERSE], INTEGER_32] */
typedef struct S2398 T2398;

/* [attached] SPECIAL [[attached] ET_FORMAL_ARGUMENT_ITEM] */
typedef struct S2400 T2400;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_FORMAL_ARGUMENT_ITEM] */
typedef struct S2402 T2402;

/* attached DS_LINKABLE [[attached] XM_ELEMENT_NODE] */
typedef struct S2407 T2407;

/* [attached] TUPLE [detachable ARRAY [[attached] ET_TYPE], detachable ET_TYPE, [attached] NATURAL_8] */
typedef struct S2410 T2410;

/* [attached] TUPLE [[attached] ET_ECF_EXTERNAL_CFLAG] */
typedef struct S2418 T2418;

/* [attached] PROCEDURE [[attached] TUPLE [[attached] ET_ECF_EXTERNAL_CFLAG]] */
typedef struct S2419 T2419;

/* [attached] TUPLE [ET_ECF_INTERNAL_UNIVERSE, ET_ECF_STATE] */
typedef struct S2420 T2420;

/* [attached] TUPLE [[attached] ET_ECF_EXTERNAL_INCLUDE] */
typedef struct S2421 T2421;

/* [attached] PROCEDURE [[attached] TUPLE [[attached] ET_ECF_EXTERNAL_INCLUDE]] */
typedef struct S2422 T2422;

/* [attached] TUPLE [[attached] ET_ECF_EXTERNAL_LIBRARY] */
typedef struct S2423 T2423;

/* [attached] PROCEDURE [[attached] TUPLE [[attached] ET_ECF_EXTERNAL_LIBRARY]] */
typedef struct S2424 T2424;

/* [attached] TUPLE [[attached] ET_ECF_EXTERNAL_LINKER_FLAG] */
typedef struct S2425 T2425;

/* [attached] PROCEDURE [[attached] TUPLE [[attached] ET_ECF_EXTERNAL_LINKER_FLAG]] */
typedef struct S2426 T2426;

/* [attached] TUPLE [[attached] ET_ECF_EXTERNAL_MAKE] */
typedef struct S2427 T2427;

/* [attached] PROCEDURE [[attached] TUPLE [[attached] ET_ECF_EXTERNAL_MAKE]] */
typedef struct S2428 T2428;

/* [attached] TUPLE [[attached] ET_ECF_EXTERNAL_OBJECT] */
typedef struct S2429 T2429;

/* [attached] PROCEDURE [[attached] TUPLE [[attached] ET_ECF_EXTERNAL_OBJECT]] */
typedef struct S2430 T2430;

/* [attached] TUPLE [[attached] ET_ECF_EXTERNAL_RESOURCE] */
typedef struct S2431 T2431;

/* [attached] PROCEDURE [[attached] TUPLE [[attached] ET_ECF_EXTERNAL_RESOURCE]] */
typedef struct S2432 T2432;

/* [attached] DS_ARRAYED_LIST [[attached] RX_PCRE_REGULAR_EXPRESSION] */
typedef struct S2433 T2433;

/* [attached] DS_HASH_TABLE [[attached] LX_WILDCARD, [attached] STRING_8] */
typedef struct S2444 T2444;

/* [attached] TUPLE [[attached] LX_WILDCARD] */
typedef struct S2446 T2446;

/* [attached] PREDICATE [[attached] TUPLE [[attached] LX_WILDCARD]] */
typedef struct S2447 T2447;

/* [attached] TUPLE [ET_CLUSTER_DEPENDENCE_CONSTRAINT, ET_GROUP, STRING_8] */
typedef struct S2448 T2448;

/* [attached] PREDICATE [[attached] TUPLE [[attached] STRING_8]] */
typedef struct S2449 T2449;

/* [attached] TUPLE [KL_STRING_ROUTINES, STRING_8] */
typedef struct S2450 T2450;

/* LX_DFA_WILDCARD */
typedef struct S2452 T2452;

/* [attached] DS_HASH_TABLE_CURSOR [[attached] INTEGER_32, [attached] STRING_8] */
typedef struct S2457 T2457;

/* [attached] SPECIAL [[attached] ET_LOCAL_VARIABLE_ITEM] */
typedef struct S2459 T2459;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_LOCAL_VARIABLE_ITEM] */
typedef struct S2461 T2461;

/* [attached] SPECIAL [[attached] ET_INDEXING_ITEM] */
typedef struct S2463 T2463;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_INDEXING_ITEM] */
typedef struct S2464 T2464;

/* [attached] SPECIAL [[attached] ET_INDEXING_TERM_ITEM] */
typedef struct S2465 T2465;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_INDEXING_TERM_ITEM] */
typedef struct S2466 T2466;

/* [attached] SPECIAL [[attached] ET_CONSTRAINT_ACTUAL_PARAMETER_ITEM] */
typedef struct S2467 T2467;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_CONSTRAINT_ACTUAL_PARAMETER_ITEM] */
typedef struct S2469 T2469;

/* [attached] SPECIAL [[attached] ET_FEATURE_CLAUSE] */
typedef struct S2470 T2470;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_FEATURE_CLAUSE] */
typedef struct S2471 T2471;

/* ET_STANDARD_ONCE_KEYS */
typedef struct S2472 T2472;

/* [attached] SPECIAL [[attached] ET_ELSEIF_PART] */
typedef struct S2475 T2475;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_ELSEIF_PART] */
typedef struct S2476 T2476;

/* [attached] SPECIAL [[attached] ET_ELSEIF_EXPRESSION] */
typedef struct S2477 T2477;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_ELSEIF_EXPRESSION] */
typedef struct S2478 T2478;

/* [attached] SPECIAL [[attached] ET_WHEN_PART] */
typedef struct S2479 T2479;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_WHEN_PART] */
typedef struct S2480 T2480;

/* [attached] SPECIAL [[attached] ET_CHOICE_ITEM] */
typedef struct S2482 T2482;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_CHOICE_ITEM] */
typedef struct S2483 T2483;

/* [attached] SPECIAL [[attached] ET_MANIFEST_STRING_ITEM] */
typedef struct S2485 T2485;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_MANIFEST_STRING_ITEM] */
typedef struct S2486 T2486;

/* ET_UNFOLDED_TUPLE_ACTUAL_ARGUMENT_LIST */
typedef struct S2487 T2487;

/* ET_INFIX_CAST_EXPRESSION */
typedef struct S2489 T2489;

/* ET_AGENT_IMPLICIT_CURRENT_TARGET */
typedef struct S2493 T2493;

/* [attached] SPECIAL [[attached] ET_AGENT_ARGUMENT_OPERAND_ITEM] */
typedef struct S2495 T2495;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_AGENT_ARGUMENT_OPERAND_ITEM] */
typedef struct S2497 T2497;

/* [attached] DS_SPARSE_TABLE_KEYS_CURSOR [[attached] ET_CLIENT, [attached] ET_CLASS] */
typedef struct S2499 T2499;

/* [attached] SPECIAL [[attached] NATURAL_64] */
typedef struct S2501 T2501;

/* [attached] SPECIAL [[attached] ET_ECF_ADAPTED_DOTNET_ASSEMBLY] */
typedef struct S2502 T2502;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_ECF_ADAPTED_DOTNET_ASSEMBLY] */
typedef struct S2503 T2503;

/* [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_ECF_ADAPTED_DOTNET_ASSEMBLY] */
typedef struct S2504 T2504;

/* [attached] SPECIAL [[attached] ET_ECF_CLUSTER] */
typedef struct S2505 T2505;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_ECF_CLUSTER] */
typedef struct S2506 T2506;

/* [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_ECF_CLUSTER] */
typedef struct S2507 T2507;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_ECF_EXTERNAL_CFLAG] */
typedef struct S2508 T2508;

/* [attached] SPECIAL [[attached] ET_ECF_EXTERNAL_CFLAG] */
typedef struct S2509 T2509;

/* [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_ECF_EXTERNAL_CFLAG] */
typedef struct S2510 T2510;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_ECF_EXTERNAL_INCLUDE] */
typedef struct S2512 T2512;

/* [attached] SPECIAL [[attached] ET_ECF_EXTERNAL_INCLUDE] */
typedef struct S2513 T2513;

/* [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_ECF_EXTERNAL_INCLUDE] */
typedef struct S2514 T2514;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_ECF_EXTERNAL_LIBRARY] */
typedef struct S2516 T2516;

/* [attached] SPECIAL [[attached] ET_ECF_EXTERNAL_LIBRARY] */
typedef struct S2517 T2517;

/* [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_ECF_EXTERNAL_LIBRARY] */
typedef struct S2518 T2518;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_ECF_EXTERNAL_LINKER_FLAG] */
typedef struct S2520 T2520;

/* [attached] SPECIAL [[attached] ET_ECF_EXTERNAL_LINKER_FLAG] */
typedef struct S2521 T2521;

/* [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_ECF_EXTERNAL_LINKER_FLAG] */
typedef struct S2522 T2522;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_ECF_EXTERNAL_MAKE] */
typedef struct S2524 T2524;

/* [attached] SPECIAL [[attached] ET_ECF_EXTERNAL_MAKE] */
typedef struct S2525 T2525;

/* [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_ECF_EXTERNAL_MAKE] */
typedef struct S2526 T2526;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_ECF_EXTERNAL_OBJECT] */
typedef struct S2528 T2528;

/* [attached] SPECIAL [[attached] ET_ECF_EXTERNAL_OBJECT] */
typedef struct S2529 T2529;

/* [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_ECF_EXTERNAL_OBJECT] */
typedef struct S2530 T2530;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_ECF_EXTERNAL_RESOURCE] */
typedef struct S2532 T2532;

/* [attached] SPECIAL [[attached] ET_ECF_EXTERNAL_RESOURCE] */
typedef struct S2533 T2533;

/* [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_ECF_EXTERNAL_RESOURCE] */
typedef struct S2534 T2534;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_ECF_FILE_RULE] */
typedef struct S2536 T2536;

/* [attached] SPECIAL [[attached] ET_ECF_FILE_RULE] */
typedef struct S2537 T2537;

/* [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_ECF_FILE_RULE] */
typedef struct S2538 T2538;

/* [attached] SPECIAL [[attached] ET_ECF_ADAPTED_LIBRARY] */
typedef struct S2539 T2539;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_ECF_ADAPTED_LIBRARY] */
typedef struct S2540 T2540;

/* [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_ECF_ADAPTED_LIBRARY] */
typedef struct S2541 T2541;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_ECF_ANDED_CONDITIONS] */
typedef struct S2542 T2542;

/* [attached] SPECIAL [[attached] ET_ECF_ANDED_CONDITIONS] */
typedef struct S2543 T2543;

/* [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_ECF_ANDED_CONDITIONS] */
typedef struct S2544 T2544;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_ECF_CONDITION_ITEM] */
typedef struct S2545 T2545;

/* [attached] SPECIAL [[attached] ET_ECF_CONDITION_ITEM] */
typedef struct S2546 T2546;

/* [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_ECF_CONDITION_ITEM] */
typedef struct S2547 T2547;

/* attached DS_LINKABLE [[attached] CHARACTER_8] */
typedef struct S2549 T2549;

/* attached DS_BILINKABLE [[attached] DS_HASH_TABLE [[attached] STRING_8, [attached] STRING_8]] */
typedef struct S2551 T2551;

/* [attached] DS_HASH_TABLE_CURSOR [[attached] INTEGER_32, [attached] ET_FORMAL_PARAMETER] */
typedef struct S2555 T2555;

/* [attached] DS_HASH_TABLE_CURSOR [[attached] INTEGER_32, [attached] ET_LIKE_FEATURE] */
typedef struct S2558 T2558;

/* [attached] SPECIAL [[attached] TUPLE [detachable ARRAY [[attached] ET_TYPE], detachable ET_TYPE, [attached] NATURAL_8]] */
typedef struct S2559 T2559;

/* [attached] KL_SPECIAL_ROUTINES [[attached] TUPLE [detachable ARRAY [[attached] ET_TYPE], detachable ET_TYPE, [attached] NATURAL_8]] */
typedef struct S2563 T2563;

/* [attached] DS_HASH_TABLE_CURSOR [[attached] TUPLE [detachable ARRAY [[attached] ET_TYPE], detachable ET_TYPE, [attached] NATURAL_8], [attached] ET_FEATURE_NAME] */
typedef struct S2564 T2564;

/* [attached] TUPLE [ET_GROUP] */
typedef struct S2565 T2565;

/* [attached] TUPLE [[attached] PROCEDURE [[attached] TUPLE [[attached] ET_CLASS]], [attached] PREDICATE [[attached] TUPLE [[attached] ET_CLASS]]] */
typedef struct S2566 T2566;

/* AP_OPTION_COMPARATOR */
typedef struct S2567 T2567;

/* [attached] DS_QUICK_SORTER [[attached] AP_OPTION] */
typedef struct S2568 T2568;

/* KL_CHARACTER_ROUTINES */
typedef struct S2577 T2577;

/* [attached] TUPLE [ET_C_GENERATOR, INTEGER_32, ET_CURRENT, ET_DYNAMIC_TUPLE_TYPE, BOOLEAN] */
typedef struct S2578 T2578;

/* ET_UNFOLDED_TUPLE_ACTUAL_PARAMETERS */
typedef struct S2580 T2580;

/* ET_ACTUAL_PARAMETER_SUBLIST */
typedef struct S2581 T2581;

/* [attached] DS_ARRAYED_LIST [detachable ET_FORMAL_PARAMETER_TYPE] */
typedef struct S2583 T2583;

/* [attached] TYPE [detachable DEVELOPER_EXCEPTION] */
#define T2585 EIF_TYPE_OBJ

/* [attached] SPECIAL [[attached] ET_DYNAMIC_PRECURSOR] */
typedef struct S2594 T2594;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_DYNAMIC_PRECURSOR] */
typedef struct S2595 T2595;

/* ET_NULL_INSTRUCTION */
typedef struct S2596 T2596;

/* ET_PARENTHESIS_INSTRUCTION */
typedef struct S2597 T2597;

/* ET_PARENTHESIS_SYMBOL */
typedef struct S2615 T2615;

/* ET_AGENT_IMPLICIT_OPEN_ARGUMENT_LIST */
typedef struct S2621 T2621;

/* ET_AGENT_IMPLICIT_OPEN_ARGUMENT */
typedef struct S2622 T2622;

/* [attached] TYPE [[attached] ET_NESTED_TYPE_CONTEXT] */
#define T2627 EIF_TYPE_OBJ

/* [attached] TYPE [[attached] ET_ACROSS_COMPONENT] */
#define T2628 EIF_TYPE_OBJ

/* [attached] SPECIAL [[attached] DS_HASH_TABLE [[attached] INTEGER_32, [attached] STRING_8]] */
typedef struct S2646 T2646;

/* [attached] KL_SPECIAL_ROUTINES [[attached] DS_HASH_TABLE [[attached] INTEGER_32, [attached] STRING_8]] */
typedef struct S2650 T2650;

/* [attached] DS_HASH_TABLE_CURSOR [[attached] DS_HASH_TABLE [[attached] INTEGER_32, [attached] STRING_8], [attached] STRING_8] */
typedef struct S2651 T2651;

/* [attached] KL_SPECIAL_ROUTINES [[attached] RX_PCRE_REGULAR_EXPRESSION] */
typedef struct S2653 T2653;

/* [attached] SPECIAL [[attached] RX_PCRE_REGULAR_EXPRESSION] */
typedef struct S2654 T2654;

/* [attached] DS_ARRAYED_LIST_CURSOR [[attached] RX_PCRE_REGULAR_EXPRESSION] */
typedef struct S2655 T2655;

/* [attached] SPECIAL [[attached] LX_WILDCARD] */
typedef struct S2657 T2657;

/* [attached] KL_SPECIAL_ROUTINES [[attached] LX_WILDCARD] */
typedef struct S2661 T2661;

/* [attached] DS_HASH_TABLE_CURSOR [[attached] LX_WILDCARD, [attached] STRING_8] */
typedef struct S2662 T2662;

/* LX_WILDCARD_PARSER */
typedef struct S2664 T2664;

/* LX_DESCRIPTION */
typedef struct S2665 T2665;

/* LX_FULL_DFA */
typedef struct S2666 T2666;

/* ET_UNIVERSE_ERROR */
typedef struct S2670 T2670;

/* ET_DYNAMIC_EQUALITY_EXPRESSION */
typedef struct S2675 T2675;

/* ET_DYNAMIC_OBJECT_EQUALITY_EXPRESSION */
typedef struct S2676 T2676;

/* [attached] TYPE [[attached] RX_REGULAR_EXPRESSION] */
#define T2680 EIF_TYPE_OBJ

/* ET_CONVERT_BUILTIN_EXPRESSION */
typedef struct S2684 T2684;

/* ET_CONVERT_FROM_EXPRESSION */
typedef struct S2685 T2685;

/* ET_CONVERT_TO_EXPRESSION */
typedef struct S2686 T2686;

/* [attached] DS_SPARSE_TABLE_KEYS_CURSOR [detachable RX_REGULAR_EXPRESSION, [attached] STRING_8] */
typedef struct S2697 T2697;

/* [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_FEATURE] */
typedef struct S2702 T2702;

/* [attached] TUPLE [ET_ECF_TARGET, ET_ECF_STATE] */
typedef struct S2704 T2704;

/* [attached] TUPLE [[attached] ET_ECF_FILE_RULE] */
typedef struct S2705 T2705;

/* [attached] PROCEDURE [[attached] TUPLE [[attached] ET_ECF_FILE_RULE]] */
typedef struct S2706 T2706;

/* [attached] TUPLE [ET_ECF_FILE_RULES] */
typedef struct S2707 T2707;

/* [attached] PREDICATE [[attached] TUPLE [[attached] ET_ECF_FILE_RULE]] */
typedef struct S2708 T2708;

/* [attached] TUPLE [ET_ECF_STATE] */
typedef struct S2709 T2709;

/* attached DS_LINKABLE [[attached] ET_PARENT_FEATURE] */
typedef struct S2712 T2712;

/* [attached] SPECIAL [detachable ET_FORMAL_PARAMETER_TYPE] */
typedef struct S2718 T2718;

/* [attached] KL_SPECIAL_ROUTINES [detachable ET_FORMAL_PARAMETER_TYPE] */
typedef struct S2719 T2719;

/* [attached] DS_ARRAYED_LIST_CURSOR [detachable ET_FORMAL_PARAMETER_TYPE] */
typedef struct S2720 T2720;

/* [attached] SPECIAL [[attached] ET_AGENT_IMPLICIT_OPEN_ARGUMENT] */
typedef struct S2721 T2721;

/* [attached] KL_SPECIAL_ROUTINES [[attached] ET_AGENT_IMPLICIT_OPEN_ARGUMENT] */
typedef struct S2722 T2722;

/* [attached] DS_ARRAYED_LIST [[attached] LX_RULE] */
typedef struct S2723 T2723;

/* LX_START_CONDITIONS */
typedef struct S2724 T2724;

/* LX_ACTION_FACTORY */
typedef struct S2725 T2725;

/* LX_SYMBOL_CLASS */
typedef struct S2726 T2726;

/* [attached] DS_HASH_TABLE [[attached] LX_SYMBOL_CLASS, [attached] STRING_8] */
typedef struct S2727 T2727;

/* [attached] KL_SPECIAL_ROUTINES [[attached] LX_SYMBOL_CLASS] */
typedef struct S2728 T2728;

/* [attached] SPECIAL [[attached] LX_SYMBOL_CLASS] */
typedef struct S2729 T2729;

/* [attached] KL_SPECIAL_ROUTINES [[attached] LX_NFA] */
typedef struct S2730 T2730;

/* LX_NFA */
typedef struct S2731 T2731;

/* [attached] SPECIAL [[attached] LX_NFA] */
typedef struct S2732 T2732;

/* LX_RULE */
typedef struct S2735 T2735;

/* [attached] DS_HASH_TABLE_CURSOR [[attached] LX_SYMBOL_CLASS, [attached] STRING_8] */
typedef struct S2736 T2736;

/* LX_EQUIVALENCE_CLASSES */
typedef struct S2737 T2737;

/* LX_UNRECOGNIZED_RULE_ERROR */
typedef struct S2738 T2738;

/* LX_MISSING_QUOTE_ERROR */
typedef struct S2739 T2739;

/* LX_BAD_CHARACTER_CLASS_ERROR */
typedef struct S2740 T2740;

/* LX_BAD_CHARACTER_ERROR */
typedef struct S2741 T2741;

/* LX_FULL_AND_META_ERROR */
typedef struct S2742 T2742;

/* LX_FULL_AND_REJECT_ERROR */
typedef struct S2743 T2743;

/* LX_FULL_AND_VARIABLE_TRAILING_CONTEXT_ERROR */
typedef struct S2744 T2744;

/* LX_CHARACTER_OUT_OF_RANGE_ERROR */
typedef struct S2745 T2745;

/* [attached] ARRAY [[attached] LX_RULE] */
typedef struct S2746 T2746;

/* [attached] SPECIAL [[attached] LX_RULE] */
typedef struct S2747 T2747;

/* LX_DFA_STATE */
typedef struct S2748 T2748;

/* [attached] DS_ARRAYED_LIST [[attached] LX_NFA_STATE] */
typedef struct S2749 T2749;

/* [attached] DS_ARRAYED_LIST [[attached] LX_DFA_STATE] */
typedef struct S2750 T2750;

/* LX_SYMBOL_PARTITIONS */
typedef struct S2752 T2752;

/* [attached] KL_ARRAY_ROUTINES [[attached] LX_RULE] */
typedef struct S2753 T2753;

/* [attached] ARRAY [detachable LX_RULE] */
typedef struct S2754 T2754;

/* [attached] SPECIAL [detachable LX_RULE] */
typedef struct S2755 T2755;

/* LX_START_CONDITION */
typedef struct S2756 T2756;

/* [attached] LX_TRANSITION_TABLE [[attached] LX_DFA_STATE] */
typedef struct S2757 T2757;

/* [attached] DS_ARRAYED_LIST [[attached] LX_NFA] */
typedef struct S2759 T2759;

/* [attached] DS_HASH_TABLE [[attached] LX_NFA, [attached] INTEGER_32] */
typedef struct S2760 T2760;

/* LX_NFA_STATE */
typedef struct S2761 T2761;

/* [attached] TUPLE [ET_ECF_LIBRARY] */
typedef struct S2763 T2763;

/* [attached] TUPLE [[attached] DS_HASH_SET [[attached] STRING_8]] */
typedef struct S2771 T2771;

/* ET_AST_NULL_LEAF */
typedef struct S2776 T2776;

/* LX_NEGATIVE_RANGE_IN_CHARACTER_CLASS_ERROR */
typedef struct S2777 T2777;

/* [attached] KL_SPECIAL_ROUTINES [[attached] LX_RULE] */
typedef struct S2778 T2778;

/* [attached] DS_ARRAYED_LIST_CURSOR [[attached] LX_RULE] */
typedef struct S2779 T2779;

/* [attached] KL_ARRAY_ROUTINES [[attached] STRING_8] */
typedef struct S2780 T2780;

/* [attached] SPECIAL [[attached] LX_START_CONDITION] */
typedef struct S2781 T2781;

/* [attached] KL_SPECIAL_ROUTINES [[attached] LX_START_CONDITION] */
typedef struct S2782 T2782;

/* [attached] DS_ARRAYED_LIST_CURSOR [[attached] LX_START_CONDITION] */
typedef struct S2783 T2783;

/* [attached] LX_SYMBOL_CLASS_TRANSITION [[attached] LX_NFA_STATE] */
typedef struct S2788 T2788;

/* [attached] LX_EPSILON_TRANSITION [[attached] LX_NFA_STATE] */
typedef struct S2790 T2790;

/* [attached] LX_SYMBOL_TRANSITION [[attached] LX_NFA_STATE] */
typedef struct S2793 T2793;

/* LX_ACTION */
typedef struct S2795 T2795;

/* [attached] ARRAY [[attached] DS_BILINKABLE [[attached] INTEGER_32]] */
typedef struct S2796 T2796;

/* [attached] DS_BILINKABLE [[attached] INTEGER_32] */
typedef struct S2797 T2797;

/* [attached] SPECIAL [[attached] DS_BILINKABLE [[attached] INTEGER_32]] */
typedef struct S2798 T2798;

/* [attached] DS_BUBBLE_SORTER [[attached] LX_NFA_STATE] */
typedef struct S2800 T2800;

/* [attached] DS_BUBBLE_SORTER [[attached] LX_RULE] */
typedef struct S2802 T2802;

/* [attached] SPECIAL [[attached] LX_NFA_STATE] */
typedef struct S2804 T2804;

/* [attached] KL_SPECIAL_ROUTINES [[attached] LX_NFA_STATE] */
typedef struct S2807 T2807;

/* [attached] DS_ARRAYED_LIST_CURSOR [[attached] LX_NFA_STATE] */
typedef struct S2808 T2808;

/* [attached] SPECIAL [[attached] LX_DFA_STATE] */
typedef struct S2811 T2811;

/* [attached] KL_SPECIAL_ROUTINES [[attached] LX_DFA_STATE] */
typedef struct S2812 T2812;

/* [attached] DS_ARRAYED_LIST_CURSOR [[attached] LX_DFA_STATE] */
typedef struct S2813 T2813;

/* [attached] KL_ARRAY [[attached] LX_RULE] */
typedef struct S2814 T2814;

/* [attached] ARRAY [detachable LX_DFA_STATE] */
typedef struct S2815 T2815;

/* [attached] SPECIAL [detachable LX_DFA_STATE] */
typedef struct S2816 T2816;

/* [attached] KL_ARRAY_ROUTINES [detachable LX_DFA_STATE] */
typedef struct S2817 T2817;

/* [attached] DS_ARRAYED_LIST_CURSOR [[attached] LX_NFA] */
typedef struct S2818 T2818;

/* [attached] DS_HASH_TABLE_CURSOR [[attached] LX_NFA, [attached] INTEGER_32] */
typedef struct S2820 T2820;

/* CHARACTER_PROPERTY */
typedef struct S2821 T2821;

/* [attached] TUPLE [ET_ECF_LIBRARY, [attached] DS_HASH_TABLE [[attached] INTEGER_32, [attached] ET_UNIVERSE], INTEGER_32] */
typedef struct S2823 T2823;

/* [attached] TUPLE [ET_ECF_DOTNET_ASSEMBLY, [attached] DS_HASH_TABLE [[attached] INTEGER_32, [attached] ET_UNIVERSE], INTEGER_32] */
typedef struct S2824 T2824;

/* [attached] TUPLE [ET_LACE_DOTNET_ASSEMBLY, [attached] DS_HASH_TABLE [[attached] INTEGER_32, [attached] ET_UNIVERSE], INTEGER_32] */
typedef struct S2825 T2825;

/* [attached] TUPLE [ET_LACE_DOTNET_GAC_ASSEMBLY, [attached] DS_HASH_TABLE [[attached] INTEGER_32, [attached] ET_UNIVERSE], INTEGER_32] */
typedef struct S2826 T2826;

/* [attached] TUPLE [ET_DOTNET_ASSEMBLY, [attached] DS_HASH_TABLE [[attached] INTEGER_32, [attached] ET_UNIVERSE], INTEGER_32] */
typedef struct S2827 T2827;

/* [attached] TUPLE [ET_LIBRARY, [attached] DS_HASH_TABLE [[attached] INTEGER_32, [attached] ET_UNIVERSE], INTEGER_32] */
typedef struct S2828 T2828;

/* [attached] TYPE [[attached] BOOLEAN] */
#define T2829 EIF_TYPE_OBJ

/* [attached] TUPLE [[attached] RX_PCRE_REGULAR_EXPRESSION] */
typedef struct S2830 T2830;

/* [attached] PREDICATE [[attached] TUPLE [[attached] RX_PCRE_REGULAR_EXPRESSION]] */
typedef struct S2831 T2831;

/* [attached] DS_SHELL_SORTER [[attached] INTEGER_32] */
typedef struct S2836 T2836;

/* [attached] KL_COMPARABLE_COMPARATOR [[attached] LX_RULE] */
typedef struct S2838 T2838;

/* [attached] KL_COMPARABLE_COMPARATOR [[attached] LX_NFA_STATE] */
typedef struct S2841 T2841;

/* [attached] KL_ARRAY [[attached] STRING_8] */
typedef struct S2844 T2844;

/* [attached] SPECIAL [[attached] NATURAL_16] */
typedef struct S2845 T2845;

/* [attached] SPECIAL [[attached] NATURAL_32] */
typedef struct S2846 T2846;

/* [attached] ARRAY [[attached] NATURAL_16] */
typedef struct S2847 T2847;

/* [attached] ARRAY [[attached] NATURAL_32] */
typedef struct S2848 T2848;

/* Struct for boxed version of type BOOLEAN */
struct Sb1 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 z1; /* item */
};

/* Struct for boxed version of type CHARACTER_8 */
struct Sb2 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T2 z1; /* item */
};

/* Struct for boxed version of type CHARACTER_32 */
struct Sb3 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T3 z1; /* item */
};

/* Struct for boxed version of type INTEGER_8 */
struct Sb4 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T4 z1; /* item */
};

/* Struct for boxed version of type INTEGER_16 */
struct Sb5 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T5 z1; /* item */
};

/* Struct for boxed version of type INTEGER_32 */
struct Sb6 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 z1; /* item */
};

/* Struct for boxed version of type INTEGER_64 */
struct Sb7 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T7 z1; /* item */
};

/* Struct for boxed version of type NATURAL_8 */
struct Sb8 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T8 z1; /* item */
};

/* Struct for boxed version of type NATURAL_16 */
struct Sb9 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T9 z1; /* item */
};

/* Struct for boxed version of type NATURAL_32 */
struct Sb10 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T10 z1; /* item */
};

/* Struct for boxed version of type NATURAL_64 */
struct Sb11 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T11 z1; /* item */
};

/* Struct for boxed version of type REAL_32 */
struct Sb12 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T12 z1; /* item */
};

/* Struct for boxed version of type REAL_64 */
struct Sb13 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T13 z1; /* item */
};

/* Struct for boxed version of type POINTER */
struct Sb14 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T14 z1; /* item */
};

/* Struct for type UTF_CONVERTER */
struct S598 {
	char dummy;
};

/* Struct for boxed version of type UTF_CONVERTER */
struct Sb598 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T598 z1; /* item */
};

/* Struct for type TYPED_POINTER [[attached] NATURAL_16] */
struct S2000 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T14 a1; /* to_pointer */
};

/* Struct for boxed version of type TYPED_POINTER [[attached] NATURAL_16] */
struct Sb2000 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T2000 z1; /* item */
};

/* Struct for type TYPED_POINTER [[attached] NATURAL_8] */
struct S1538 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T14 a1; /* to_pointer */
};

/* Struct for boxed version of type TYPED_POINTER [[attached] NATURAL_8] */
struct Sb1538 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1538 z1; /* item */
};

/* Struct for type TYPED_POINTER [[attached] ANY] */
struct S1344 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T14 a1; /* to_pointer */
};

/* Struct for boxed version of type TYPED_POINTER [[attached] ANY] */
struct Sb1344 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1344 z1; /* item */
};

/* Struct for type TYPED_POINTER [[attached] SPECIAL [[attached] NATURAL_8]] */
struct S607 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T14 a1; /* to_pointer */
};

/* Struct for boxed version of type TYPED_POINTER [[attached] SPECIAL [[attached] NATURAL_8]] */
struct Sb607 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T607 z1; /* item */
};

/* Struct for type SPECIAL [CHARACTER_8] */
struct S15 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T2 z2[1]; /* item */
};

/* Struct for type SPECIAL [CHARACTER_32] */
struct S16 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T3 z2[1]; /* item */
};

/* Struct for type STRING_8 */
struct S17 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* area */
	T6 a2; /* count */
	T6 a3; /* internal_hash_code */
	T6 a4; /* internal_case_insensitive_hash_code */
};

/* Struct for type STRING_32 */
struct S18 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* area */
	T6 a2; /* count */
	T6 a3; /* internal_hash_code */
	T6 a4; /* internal_case_insensitive_hash_code */
};

/* Struct for type IMMUTABLE_STRING_32 */
struct S19 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* area */
	T6 a2; /* count */
	T6 a3; /* internal_hash_code */
	T6 a4; /* area_lower */
};

/* Struct for type ISE_EXCEPTION_MANAGER */
struct S20 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type GEC */
struct S25 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* error_handler */
	T0* a2; /* ecf_filename */
	T0* a3; /* last_system */
	T0* a4; /* target_option */
	T0* a5; /* setting_option */
	T0* a6; /* capability_option */
	T0* a7; /* variable_option */
	T0* a8; /* finalize_flag */
	T0* a9; /* gelint_flag */
	T0* a10; /* catcall_option */
	T0* a11; /* c_compile_option */
	T0* a12; /* split_option */
	T0* a13; /* split_size_option */
	T0* a14; /* new_instance_types_option */
	T0* a15; /* gc_option */
	T0* a16; /* thread_option */
	T0* a17; /* silent_flag */
	T0* a18; /* no_benchmark_flag */
	T0* a19; /* nested_benchmark_flag */
	T0* a20; /* metrics_flag */
	T0* a21; /* verbose_flag */
	T0* a22; /* version_flag */
	T0* a23; /* override_variables */
	T0* a24; /* override_settings */
	T0* a25; /* override_capabilities */
	T1 a26; /* ecf_gelint_option */
	T0* a27; /* new_instance_types */
	T6 a28; /* split_size */
};

/* Struct for type [attached] CELL [detachable EXCEPTION] */
struct S26 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* item */
};

/* Struct for type [attached] HASH_TABLE [[attached] INTEGER_32, [attached] INTEGER_32] */
struct S27 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* capacity */
	T0* a2; /* content */
	T0* a3; /* keys */
	T0* a4; /* deleted_marks */
	T0* a5; /* indexes_map */
	T6 a6; /* iteration_position */
	T6 a7; /* count */
	T6 a8; /* deleted_item_position */
	T6 a9; /* control */
	T6 a10; /* found_item */
	T1 a11; /* has_default */
	T6 a12; /* item_position */
	T6 a13; /* ht_lowest_deleted_position */
	T6 a14; /* ht_deleted_item */
	T6 a15; /* ht_deleted_key */
	T1 a16; /* object_comparison */
};

/* Struct for type [attached] CELL [detachable TUPLE [[attached] INTEGER_32, [attached] INTEGER_32, [attached] INTEGER_32, [attached] STRING_8, [attached] STRING_8, [attached] STRING_8, [attached] STRING_8, [attached] STRING_8, [attached] STRING_8, [attached] INTEGER_32, [attached] BOOLEAN]] */
struct S28 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* item */
};

/* Struct for type [attached] CELL [[attached] NO_MORE_MEMORY] */
struct S29 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* item */
};

/* Struct for type C_STRING */
struct S31 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* managed_data */
	T6 a2; /* count */
};

/* Struct for type [attached] TUPLE [[attached] INTEGER_32, [attached] INTEGER_32, [attached] INTEGER_32, [attached] STRING_8, [attached] STRING_8, [attached] STRING_8, [attached] STRING_8, [attached] STRING_8, [attached] STRING_8, [attached] INTEGER_32, [attached] BOOLEAN] */
struct S32 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 z1;
	T6 z2;
	T6 z3;
	T0* z4;
	T0* z5;
	T0* z6;
	T0* z7;
	T0* z8;
	T0* z9;
	T6 z10;
	T1 z11;
};

/* Struct for type KL_TEXT_INPUT_FILE */
struct S33 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* end_of_file */
	T0* a2; /* last_string */
	T6 a3; /* mode */
	T0* a4; /* name */
	T0* a5; /* character_buffer */
	T2 a6; /* last_character */
	T14 a7; /* file_pointer */
	T1 a8; /* descriptor_available */
	T0* a9; /* internal_name */
	T0* a10; /* internal_detachable_name_pointer */
};

/* Struct for type KL_ARGUMENTS */
struct S34 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* program_name */
};

/* Struct for type ET_ISE_VARIABLES */
struct S35 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type ET_ERROR_HANDLER */
struct S36 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* has_error */
	T0* a2; /* info_file */
	T0* a3; /* mutex */
	T0* a4; /* error_file */
	T0* a5; /* warning_file */
	T1 a6; /* is_ise */
	T1 a7; /* is_verbose */
	T1 a8; /* has_eiffel_error */
	T1 a9; /* has_internal_error */
	T1 a10; /* is_ge */
	T1 a11; /* is_pedantic */
};

/* Struct for type KL_STRING_ROUTINES */
struct S38 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type ET_SYSTEM */
struct S40 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* master_class_mutex */
	T0* a2; /* master_classes */
	T0* a3; /* register_class_mutex */
	T0* a4; /* external_include_pathnames */
	T0* a5; /* external_object_pathnames */
	T0* a6; /* external_library_pathnames */
	T0* a7; /* external_make_pathnames */
	T0* a8; /* external_resource_pathnames */
	T0* a9; /* external_cflags */
	T0* a10; /* external_linker_flags */
	T1 a11; /* console_application_mode */
	T1 a12; /* alias_transition_mode */
	T0* a13; /* current_system */
	T0* a14; /* universe */
	T0* a15; /* name */
	T0* a16; /* libraries */
	T0* a17; /* clusters */
	T0* a18; /* dotnet_assemblies */
	T0* a19; /* system_name */
	T0* a20; /* any_type */
	T0* a21; /* detachable_any_type */
	T0* a22; /* detachable_separate_any_type */
	T0* a23; /* separate_any_type */
	T0* a24; /* any_parent */
	T0* a25; /* any_parents */
	T0* a26; /* any_clients */
	T0* a27; /* tuple_type */
	T0* a28; /* detachable_tuple_type */
	T0* a29; /* tuple_like_current_type */
	T0* a30; /* unfolded_empty_tuple_actual_parameters */
	T0* a31; /* array_any_type */
	T0* a32; /* array_detachable_any_type */
	T0* a33; /* array_none_type */
	T0* a34; /* array_like_current_type */
	T0* a35; /* boolean_type */
	T0* a36; /* character_type */
	T0* a37; /* character_8_type */
	T0* a38; /* character_8_convert_feature */
	T0* a39; /* character_32_type */
	T0* a40; /* character_32_convert_feature */
	T0* a41; /* double_type */
	T0* a42; /* exception_type */
	T0* a43; /* detachable_exception_type */
	T0* a44; /* exception_manager_type */
	T0* a45; /* function_type */
	T0* a46; /* immutable_string_32_type */
	T0* a47; /* integer_type */
	T0* a48; /* integer_8_type */
	T0* a49; /* integer_8_convert_feature */
	T0* a50; /* integer_16_type */
	T0* a51; /* integer_16_convert_feature */
	T0* a52; /* integer_32_type */
	T0* a53; /* integer_32_convert_feature */
	T0* a54; /* integer_64_type */
	T0* a55; /* integer_64_convert_feature */
	T0* a56; /* ise_exception_manager_type */
	T0* a57; /* iterable_detachable_any_type */
	T0* a58; /* natural_type */
	T0* a59; /* natural_8_type */
	T0* a60; /* natural_8_convert_feature */
	T0* a61; /* natural_16_type */
	T0* a62; /* natural_16_convert_feature */
	T0* a63; /* natural_32_type */
	T0* a64; /* natural_32_convert_feature */
	T0* a65; /* natural_64_type */
	T0* a66; /* natural_64_convert_feature */
	T0* a67; /* none_type */
	T0* a68; /* detachable_none_type */
	T0* a69; /* pointer_type */
	T0* a70; /* predicate_like_current_type */
	T0* a71; /* procedure_like_current_type */
	T0* a72; /* real_type */
	T0* a73; /* real_32_type */
	T0* a74; /* real_32_convert_feature */
	T0* a75; /* real_64_type */
	T0* a76; /* real_64_convert_feature */
	T0* a77; /* routine_type */
	T0* a78; /* special_any_type */
	T0* a79; /* special_detachable_any_type */
	T0* a80; /* string_type */
	T0* a81; /* detachable_string_type */
	T0* a82; /* string_8_type */
	T0* a83; /* string_8_convert_feature */
	T0* a84; /* string_32_type */
	T0* a85; /* string_32_convert_feature */
	T0* a86; /* system_object_type */
	T0* a87; /* system_object_parents */
	T0* a88; /* system_string_type */
	T0* a89; /* type_any_type */
	T0* a90; /* type_type_any_type */
	T0* a91; /* type_detachable_any_type */
	T0* a92; /* type_detachable_like_current_type */
	T0* a93; /* detachable_type_exception_type */
	T0* a94; /* type_detachable_exception_type */
	T0* a95; /* detachable_type_detachable_any_type */
	T0* a96; /* type_like_current_type */
	T0* a97; /* typed_pointer_any_type */
	T0* a98; /* typed_pointer_like_current_type */
	T0* a99; /* wide_character_type */
	T1 a100; /* is_read_only */
	T6 a101; /* registered_class_count */
	T1 a102; /* attachment_type_conformance_mode */
	T1 a103; /* scoop_mode */
	T6 a104; /* default_create_seed */
	T0* a105; /* implicit_attachment_type_mark */
	T6 a106; /* copy_seed */
	T6 a107; /* is_equal_seed */
	T6 a108; /* routine_call_seed */
	T6 a109; /* function_item_seed */
	T6 a110; /* dispose_seed */
	T6 a111; /* iterable_new_cursor_seed */
	T6 a112; /* iteration_cursor_after_seed */
	T6 a113; /* iteration_cursor_forth_seed */
	T1 a114; /* is_dotnet */
	T1 a115; /* target_type_attachment_mode */
	T1 a116; /* obsolete_routine_type_mode */
};

/* Struct for type KL_EXCEPTIONS */
struct S41 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type AP_PARSER */
struct S42 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* options */
	T0* a2; /* alternative_options_lists */
	T0* a3; /* parameters */
	T0* a4; /* parameters_description */
	T0* a5; /* help_option */
	T0* a6; /* error_handler */
	T0* a7; /* application_description */
	T0* a8; /* current_options */
	T1 a9; /* is_first_option */
	T0* a10; /* last_option_parameter */
};

/* Struct for type AP_ALTERNATIVE_OPTIONS_LIST */
struct S43 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* introduction_option */
	T0* a2; /* parameters_description */
	T0* a3; /* internal_cursor */
	T0* a4; /* first_cell */
};

/* Struct for type AP_STRING_OPTION */
struct S44 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* parameters */
	T0* a2; /* default_parameter */
	T0* a3; /* description */
	T0* a4; /* parameter_description */
	T1 a5; /* needs_parameter */
	T0* a6; /* long_form */
	T1 a7; /* is_hidden */
	T1 a8; /* is_mandatory */
	T6 a9; /* maximum_occurrences */
	T1 a10; /* has_short_form */
	T2 a11; /* short_form */
};

/* Struct for type AP_FLAG */
struct S46 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* occurrences */
	T0* a2; /* description */
	T0* a3; /* long_form */
	T2 a4; /* short_form */
	T1 a5; /* has_short_form */
	T1 a6; /* is_hidden */
	T1 a7; /* is_mandatory */
	T6 a8; /* maximum_occurrences */
};

/* Struct for type AP_ENUMERATION_OPTION */
struct S47 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* parameters */
	T0* a2; /* default_parameter */
	T0* a3; /* description */
	T0* a4; /* possible_values */
	T0* a5; /* parameter_description */
	T1 a6; /* needs_parameter */
	T0* a7; /* long_form */
	T1 a8; /* is_hidden */
	T1 a9; /* is_mandatory */
	T6 a10; /* maximum_occurrences */
	T1 a11; /* has_short_form */
	T2 a12; /* short_form */
};

/* Struct for type AP_BOOLEAN_OPTION */
struct S48 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* parameters */
	T1 a2; /* default_parameter */
	T0* a3; /* description */
	T0* a4; /* parameter_description */
	T1 a5; /* needs_parameter */
	T0* a6; /* long_form */
	T1 a7; /* is_hidden */
	T1 a8; /* is_mandatory */
	T6 a9; /* maximum_occurrences */
	T1 a10; /* has_short_form */
	T2 a11; /* short_form */
};

/* Struct for type AP_INTEGER_OPTION */
struct S49 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* parameters */
	T6 a2; /* default_parameter */
	T0* a3; /* description */
	T0* a4; /* parameter_description */
	T1 a5; /* needs_parameter */
	T0* a6; /* long_form */
	T2 a7; /* short_form */
	T1 a8; /* is_hidden */
	T1 a9; /* is_mandatory */
	T6 a10; /* maximum_occurrences */
	T1 a11; /* has_short_form */
};

/* Struct for type ET_NULL_ERROR_HANDLER */
struct S53 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* has_error */
	T0* a2; /* info_file */
	T0* a3; /* mutex */
	T0* a4; /* error_file */
	T0* a5; /* warning_file */
	T1 a6; /* is_ise */
	T1 a7; /* is_verbose */
	T1 a8; /* has_eiffel_error */
	T1 a9; /* has_internal_error */
	T1 a10; /* is_ge */
	T1 a11; /* is_pedantic */
};

/* Struct for type ET_XACE_SYSTEM_PARSER */
struct S55 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* last_system */
	T0* a2; /* ast_factory */
	T0* a3; /* xml_parser */
	T0* a4; /* tree_pipe */
	T0* a5; /* xml_validator */
	T0* a6; /* xml_preprocessor */
	T0* a7; /* parsed_libraries */
	T0* a8; /* error_handler */
	T0* a9; /* library_parser */
	T6 a10; /* new_name_count */
	T1 a11; /* is_shallow */
};

/* Struct for type ET_XACE_DEFAULT_ERROR_HANDLER */
struct S56 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* error_count */
	T0* a2; /* error_file */
	T0* a3; /* warning_file */
	T0* a4; /* info_file */
};

/* Struct for type [attached] DS_HASH_TABLE [[attached] STRING_8, [attached] STRING_8] */
struct S57 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* key_equality_tester */
	T0* a2; /* internal_keys */
	T6 a3; /* position */
	T6 a4; /* last_position */
	T6 a5; /* capacity */
	T6 a6; /* slots_position */
	T6 a7; /* count */
	T0* a8; /* equality_tester */
	T6 a9; /* found_position */
	T6 a10; /* modulus */
	T6 a11; /* clashes_previous_position */
	T0* a12; /* special_item_routines */
	T0* a13; /* item_storage */
	T0* a14; /* clashes */
	T0* a15; /* slots */
	T0* a16; /* special_key_routines */
	T0* a17; /* key_storage */
	T6 a18; /* free_slot */
	T0* a19; /* internal_cursor */
	T0* a20; /* hash_function */
};

/* Struct for type UC_STRING_EQUALITY_TESTER */
struct S58 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type ET_ECF_VARIABLES */
struct S61 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* primary_variables */
	T0* a2; /* secondary_variables */
};

/* Struct for type [attached] DS_HASH_TABLE_CURSOR [[attached] STRING_8, [attached] STRING_8] */
struct S62 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* next_cursor */
	T0* a3; /* container */
};

/* Struct for type ET_ECF_SYSTEM_PARSER */
struct S66 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* last_system */
	T0* a2; /* ast_factory */
	T0* a3; /* override_settings */
	T0* a4; /* override_capabilities */
	T0* a5; /* override_variables */
	T0* a6; /* parsed_libraries */
	T0* a7; /* parsed_dotnet_assemblies */
	T0* a8; /* redirected_locations */
	T0* a9; /* xml_parser */
	T0* a10; /* tree_pipe */
	T0* a11; /* error_handler */
	T0* a12; /* ise_version */
};

/* Struct for type ET_ECF_ERROR_HANDLER */
struct S67 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* error_count */
	T0* a2; /* error_file */
	T0* a3; /* warning_file */
	T0* a4; /* info_file */
};

/* Struct for type ET_ECF_SYSTEM */
struct S69 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* filename */
	T0* a2; /* targets */
	T0* a3; /* library_target */
	T0* a4; /* libraries */
	T0* a5; /* selected_target */
	T0* a6; /* dotnet_assemblies */
	T0* a7; /* root_type */
	T0* a8; /* none_type */
	T0* a9; /* any_type */
	T0* a10; /* master_classes */
	T0* a11; /* root_creation */
	T6 a12; /* default_create_seed */
	T0* a13; /* boolean_type */
	T0* a14; /* character_8_type */
	T0* a15; /* character_32_type */
	T0* a16; /* integer_8_type */
	T0* a17; /* integer_16_type */
	T0* a18; /* integer_32_type */
	T0* a19; /* integer_64_type */
	T0* a20; /* natural_8_type */
	T0* a21; /* natural_16_type */
	T0* a22; /* natural_32_type */
	T0* a23; /* natural_64_type */
	T0* a24; /* real_32_type */
	T0* a25; /* real_64_type */
	T0* a26; /* pointer_type */
	T0* a27; /* special_any_type */
	T0* a28; /* string_8_type */
	T0* a29; /* string_32_type */
	T0* a30; /* immutable_string_32_type */
	T0* a31; /* array_any_type */
	T0* a32; /* typed_pointer_any_type */
	T0* a33; /* routine_type */
	T0* a34; /* detachable_tuple_type */
	T0* a35; /* universe */
	T0* a36; /* ise_exception_manager_type */
	T0* a37; /* detachable_none_type */
	T1 a38; /* attachment_type_conformance_mode */
	T1 a39; /* use_boehm_gc */
	T0* a40; /* external_include_pathnames */
	T0* a41; /* external_cflags */
	T0* a42; /* external_linker_flags */
	T0* a43; /* external_library_pathnames */
	T0* a44; /* external_object_pathnames */
	T0* a45; /* character_type */
	T0* a46; /* wide_character_type */
	T0* a47; /* integer_type */
	T0* a48; /* natural_type */
	T0* a49; /* real_type */
	T0* a50; /* double_type */
	T0* a51; /* external_make_pathnames */
	T0* a52; /* external_resource_pathnames */
	T6 a53; /* routine_call_seed */
	T6 a54; /* function_item_seed */
	T6 a55; /* is_equal_seed */
	T6 a56; /* dispose_seed */
	T1 a57; /* trace_mode */
	T1 a58; /* exception_trace_mode */
	T1 a59; /* multithreaded_mode */
	T1 a60; /* console_application_mode */
	T0* a61; /* system_name */
	T0* a62; /* name */
	T0* a63; /* ecf_namespace */
	T0* a64; /* ecf_version */
	T0* a65; /* description */
	T0* a66; /* notes */
	T1 a67; /* is_read_only */
	T0* a68; /* uuid */
	T0* a69; /* register_class_mutex */
	T1 a70; /* alias_transition_mode */
	T0* a71; /* current_system */
	T0* a72; /* tuple_type */
	T0* a73; /* detachable_exception_type */
	T0* a74; /* exception_type */
	T0* a75; /* string_type */
	T1 a76; /* is_preparsed */
	T0* a77; /* clusters */
	T0* a78; /* master_class_mutex */
	T0* a79; /* detachable_any_type */
	T0* a80; /* detachable_separate_any_type */
	T0* a81; /* separate_any_type */
	T0* a82; /* any_parent */
	T0* a83; /* any_parents */
	T0* a84; /* any_clients */
	T0* a85; /* tuple_like_current_type */
	T0* a86; /* unfolded_empty_tuple_actual_parameters */
	T0* a87; /* array_detachable_any_type */
	T0* a88; /* array_none_type */
	T0* a89; /* array_like_current_type */
	T0* a90; /* character_8_convert_feature */
	T0* a91; /* character_32_convert_feature */
	T0* a92; /* exception_manager_type */
	T0* a93; /* function_type */
	T0* a94; /* integer_8_convert_feature */
	T0* a95; /* integer_16_convert_feature */
	T0* a96; /* integer_32_convert_feature */
	T0* a97; /* integer_64_convert_feature */
	T0* a98; /* iterable_detachable_any_type */
	T0* a99; /* natural_8_convert_feature */
	T0* a100; /* natural_16_convert_feature */
	T0* a101; /* natural_32_convert_feature */
	T0* a102; /* natural_64_convert_feature */
	T0* a103; /* predicate_like_current_type */
	T0* a104; /* procedure_like_current_type */
	T0* a105; /* real_32_convert_feature */
	T0* a106; /* real_64_convert_feature */
	T0* a107; /* special_detachable_any_type */
	T0* a108; /* detachable_string_type */
	T0* a109; /* string_8_convert_feature */
	T0* a110; /* string_32_convert_feature */
	T0* a111; /* system_object_type */
	T0* a112; /* system_object_parents */
	T0* a113; /* system_string_type */
	T0* a114; /* type_any_type */
	T0* a115; /* type_type_any_type */
	T0* a116; /* type_detachable_any_type */
	T0* a117; /* type_detachable_like_current_type */
	T0* a118; /* detachable_type_exception_type */
	T0* a119; /* type_detachable_exception_type */
	T0* a120; /* detachable_type_detachable_any_type */
	T0* a121; /* type_like_current_type */
	T0* a122; /* typed_pointer_like_current_type */
	T0* a123; /* scm_read_mapping_builder */
	T0* a124; /* scm_write_mapping_builder */
	T6 a125; /* registered_class_count */
	T1 a126; /* scoop_mode */
	T0* a127; /* implicit_attachment_type_mark */
	T6 a128; /* copy_seed */
	T6 a129; /* iterable_new_cursor_seed */
	T6 a130; /* iteration_cursor_after_seed */
	T6 a131; /* iteration_cursor_forth_seed */
	T1 a132; /* is_dotnet */
	T1 a133; /* obsolete_routine_type_mode */
	T1 a134; /* target_type_attachment_mode */
};

/* Struct for type ET_ECF_TARGET */
struct S70 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* settings */
	T0* a2; /* capabilities */
	T0* a3; /* options */
	T0* a4; /* system_config */
	T0* a5; /* root */
	T0* a6; /* variables */
	T0* a7; /* name */
	T0* a8; /* parent */
	T0* a9; /* description */
	T0* a10; /* precompiled_library */
	T0* a11; /* version */
	T1 a12; /* is_abstract */
	T0* a13; /* clusters */
	T0* a14; /* libraries */
	T0* a15; /* dotnet_assemblies */
	T0* a16; /* file_rules */
	T0* a17; /* class_mappings */
	T0* a18; /* external_cflags */
	T0* a19; /* external_includes */
	T0* a20; /* external_libraries */
	T0* a21; /* external_linker_flags */
	T0* a22; /* external_makes */
	T0* a23; /* external_objects */
	T0* a24; /* external_resources */
	T0* a25; /* pre_compile_actions */
	T0* a26; /* post_compile_actions */
	T0* a27; /* notes */
};

/* Struct for type ET_ECF_SETTINGS */
struct S71 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* primary_settings */
	T0* a2; /* secondary_settings */
};

/* Struct for type ET_LACE_PARSER */
struct S77 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* last_system */
	T6 a2; /* yy_parsing_status */
	T0* a3; /* named_clusters */
	T0* a4; /* override_cluster_names */
	T0* a5; /* external_include_pathnames */
	T0* a6; /* external_object_pathnames */
	T0* a7; /* metadata_cache_path_value */
	T0* a8; /* msil_clr_version_value */
	T1 a9; /* console_application_value */
	T1 a10; /* exception_trace_value */
	T1 a11; /* trace_value */
	T0* a12; /* ast_factory */
	T6 a13; /* eif_lineno */
	T0* a14; /* eif_buffer */
	T0* a15; /* input_buffer */
	T6 a16; /* yy_end */
	T6 a17; /* yy_position */
	T6 a18; /* yy_line */
	T6 a19; /* yy_column */
	T6 a20; /* yy_suspended_yystacksize */
	T6 a21; /* yy_suspended_yystate */
	T6 a22; /* yy_suspended_yyn */
	T6 a23; /* yy_suspended_yychar1 */
	T6 a24; /* yy_suspended_index */
	T6 a25; /* yy_suspended_yyss_top */
	T6 a26; /* yy_suspended_yy_goto */
	T0* a27; /* yyr1 */
	T0* a28; /* yyss */
	T6 a29; /* yyssp */
	T0* a30; /* yypgoto */
	T0* a31; /* yycheck */
	T0* a32; /* yytable */
	T0* a33; /* yydefgoto */
	T6 a34; /* error_count */
	T1 a35; /* yy_lookahead_needed */
	T6 a36; /* yyerrstatus */
	T0* a37; /* yypact */
	T6 a38; /* last_token */
	T0* a39; /* yytranslate */
	T0* a40; /* yydefact */
	T0* a41; /* filename */
	T0* a42; /* error_handler */
	T0* a43; /* last_et_identifier_value */
	T1 a44; /* yy_rejected */
	T6 a45; /* yy_state_count */
	T6 a46; /* yy_full_match */
	T6 a47; /* yy_lp */
	T6 a48; /* yy_looking_for_trail_begin */
	T6 a49; /* yy_full_lp */
	T6 a50; /* yy_full_state */
	T6 a51; /* pushed_start_condition_count */
	T6 a52; /* yy_start_state */
	T6 a53; /* yy_start */
	T6 a54; /* line */
	T6 a55; /* column */
	T6 a56; /* position */
	T1 a57; /* yy_more_flag */
	T6 a58; /* yy_more_len */
	T6 a59; /* yy_last_accepting_state */
	T6 a60; /* yy_last_accepting_cpos */
	T6 a61; /* yyvsp1 */
	T6 a62; /* yyvsp2 */
	T6 a63; /* yyvsp3 */
	T6 a64; /* yyvsp4 */
	T6 a65; /* yyvsp5 */
	T6 a66; /* yyvsp6 */
	T6 a67; /* yyvsp7 */
	T6 a68; /* yyvsp8 */
	T6 a69; /* yyvsp9 */
	T6 a70; /* yyvsp10 */
	T6 a71; /* yyvsp11 */
	T6 a72; /* yyvsp12 */
	T0* a73; /* yy_state_stack */
	T0* a74; /* yy_content_area */
	T0* a75; /* yy_ec */
	T0* a76; /* yy_content */
	T0* a77; /* yy_accept */
	T0* a78; /* yy_chk */
	T0* a79; /* yy_base */
	T0* a80; /* yy_def */
	T0* a81; /* yy_meta */
	T0* a82; /* yy_nxt */
	T0* a83; /* yy_acclist */
	T0* a84; /* yytypes2 */
	T6 a85; /* yyvsc1 */
	T0* a86; /* yyvs1 */
	T0* a87; /* yyspecial_routines1 */
	T0* a88; /* last_detachable_any_value */
	T6 a89; /* yyvsc2 */
	T0* a90; /* yyvs2 */
	T0* a91; /* yyspecial_routines2 */
	T0* a92; /* yyvs10 */
	T0* a93; /* yyvs5 */
	T0* a94; /* yyvs12 */
	T0* a95; /* yyvs8 */
	T0* a96; /* yyspecial_routines10 */
	T6 a97; /* yyvsc10 */
	T6 a98; /* yyvsc8 */
	T0* a99; /* yyspecial_routines8 */
	T6 a100; /* yyvsc5 */
	T0* a101; /* yyspecial_routines5 */
	T0* a102; /* yyvs4 */
	T0* a103; /* yyvs3 */
	T6 a104; /* yyvsc4 */
	T0* a105; /* yyspecial_routines4 */
	T0* a106; /* yyspecial_routines3 */
	T0* a107; /* yyvs7 */
	T6 a108; /* yyvsc3 */
	T6 a109; /* yyvsc7 */
	T0* a110; /* yyspecial_routines7 */
	T0* a111; /* yyvs6 */
	T6 a112; /* yyvsc6 */
	T0* a113; /* yyspecial_routines6 */
	T6 a114; /* yyvsc12 */
	T0* a115; /* yyspecial_routines12 */
	T0* a116; /* yyvs11 */
	T0* a117; /* yyvs9 */
	T6 a118; /* yyvsc11 */
	T0* a119; /* yyspecial_routines11 */
	T6 a120; /* yyvsc9 */
	T0* a121; /* yyspecial_routines9 */
	T0* a122; /* yytypes1 */
	T0* a123; /* yy_pushed_start_conditions */
	T6 a124; /* code_ */
};

/* Struct for type ET_LACE_ERROR_HANDLER */
struct S78 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* error_file */
	T0* a2; /* warning_file */
	T0* a3; /* info_file */
};

/* Struct for type ET_DYNAMIC_SYSTEM */
struct S80 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* has_fatal_error */
	T1 a2; /* catcall_error_mode */
	T0* a3; /* current_system */
	T0* a4; /* system_processor */
	T0* a5; /* dynamic_types */
	T0* a6; /* null_dynamic_type_set_builder */
	T1 a7; /* full_class_checking */
	T1 a8; /* catcall_warning_mode */
	T0* a9; /* new_instance_types */
	T0* a10; /* dynamic_type_set_builder */
	T0* a11; /* unknown_type */
	T0* a12; /* boolean_type */
	T0* a13; /* character_8_type */
	T0* a14; /* character_32_type */
	T0* a15; /* immutable_string_32_type */
	T0* a16; /* integer_8_type */
	T0* a17; /* integer_16_type */
	T0* a18; /* integer_32_type */
	T0* a19; /* integer_64_type */
	T0* a20; /* natural_8_type */
	T0* a21; /* natural_16_type */
	T0* a22; /* natural_32_type */
	T0* a23; /* natural_64_type */
	T0* a24; /* real_32_type */
	T0* a25; /* real_64_type */
	T0* a26; /* pointer_type */
	T0* a27; /* string_8_type */
	T0* a28; /* string_32_type */
	T0* a29; /* special_character_8_type */
	T0* a30; /* special_character_32_type */
	T0* a31; /* ise_exception_manager_type */
	T0* a32; /* any_type */
	T0* a33; /* none_type */
	T0* a34; /* root_type */
	T0* a35; /* root_creation_procedure */
	T0* a36; /* special_count_feature */
	T0* a37; /* special_capacity_feature */
	T0* a38; /* array_area_feature */
	T0* a39; /* array_lower_feature */
	T0* a40; /* array_upper_feature */
	T0* a41; /* typed_pointer_to_pointer_feature */
	T0* a42; /* routine_closed_operands_feature */
	T0* a43; /* routine_rout_disp_feature */
	T0* a44; /* routine_set_rout_disp_final_feature */
	T0* a45; /* ise_exception_manager_init_exception_manager_feature */
	T0* a46; /* ise_exception_manager_last_exception_feature */
	T0* a47; /* ise_exception_manager_once_raise_feature */
	T0* a48; /* ise_exception_manager_set_exception_data_feature */
	T0* a49; /* ise_runtime_type_conforms_to_feature */
	T1 a50; /* in_dynamic_type */
	T0* a51; /* ise_runtime_new_type_instance_of_feature */
	T1 a52; /* in_create_meta_type */
	T1 a53; /* all_attributes_used */
	T0* a54; /* ise_runtime_reference_field_feature */
	T0* a55; /* ise_runtime_set_reference_field_feature */
};

/* Struct for type ET_SYSTEM_PROCESSOR */
struct S81 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* stop_request */
	T0* a2; /* eiffel_parser */
	T0* a3; /* interface_checker */
	T1 a4; /* benchmark_shown */
	T0* a5; /* ise_version */
	T0* a6; /* eiffel_preparser */
	T0* a7; /* master_class_checker */
	T0* a8; /* provider_checker */
	T0* a9; /* ancestor_builder */
	T0* a10; /* feature_flattener */
	T0* a11; /* implementation_checker */
	T0* a12; /* error_handler */
	T0* a13; /* processed_class_count_stack */
	T0* a14; /* ast_factory */
	T0* a15; /* dotnet_assembly_consumer */
	T0* a16; /* ecma_version */
	T1 a17; /* nested_benchmark_shown */
	T1 a18; /* metrics_shown */
	T1 a19; /* flat_mode */
	T1 a20; /* flat_dbc_mode */
	T1 a21; /* preparse_shallow_mode */
	T1 a22; /* preparse_single_mode */
	T1 a23; /* preparse_multiple_mode */
	T1 a24; /* unknown_builtin_reported */
	T1 a25; /* qualified_anchored_types_cycle_detection_enabled */
	T1 a26; /* use_attribute_keyword */
	T1 a27; /* use_note_keyword */
	T1 a28; /* use_reference_keyword */
	T1 a29; /* use_attached_keyword */
	T1 a30; /* use_detachable_keyword */
	T1 a31; /* cluster_dependence_enabled */
	T6 a32; /* postponed_class_count */
	T6 a33; /* processed_class_count */
	T1 a34; /* preparse_readonly_mode */
	T1 a35; /* preparse_override_mode */
	T1 a36; /* use_cluster_dependence_pathnames */
	T1 a37; /* providers_enabled */
	T1 a38; /* suppliers_enabled */
};

/* Struct for type ET_C_GENERATOR */
struct S84 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* never_void_target_count */
	T6 a2; /* can_be_void_target_count */
	T1 a3; /* has_fatal_error */
	T0* a4; /* system_processor */
	T0* a5; /* current_dynamic_system */
	T0* a6; /* dynamic_types */
	T1 a7; /* short_names */
	T1 a8; /* split_mode */
	T6 a9; /* split_threshold */
	T1 a10; /* use_boehm_gc */
	T0* a11; /* system_name */
	T0* a12; /* current_file */
	T0* a13; /* header_file */
	T0* a14; /* current_type */
	T0* a15; /* current_feature */
	T0* a16; /* extra_dynamic_type_sets */
	T0* a17; /* current_dynamic_type_sets */
	T0* a18; /* current_function_header_buffer */
	T0* a19; /* current_function_body_buffer */
	T0* a20; /* temp_variables */
	T0* a21; /* used_temp_variables */
	T0* a22; /* free_temp_variables */
	T0* a23; /* frozen_temp_variables */
	T0* a24; /* conforming_types */
	T0* a25; /* conforming_type_set */
	T0* a26; /* non_conforming_types */
	T0* a27; /* equality_type_set */
	T0* a28; /* equality_common_types */
	T0* a29; /* operand_stack */
	T0* a30; /* call_operands */
	T0* a31; /* attachment_dynamic_type_ids */
	T0* a32; /* target_dynamic_type_ids */
	T0* a33; /* target_dynamic_types */
	T0* a34; /* standalone_type_sets */
	T0* a35; /* deep_twin_types */
	T0* a36; /* deep_equal_types */
	T0* a37; /* deep_feature_target_type_sets */
	T0* a38; /* current_object_tests */
	T0* a39; /* current_object_equalities */
	T0* a40; /* current_equalities */
	T0* a41; /* current_agents */
	T0* a42; /* agent_open_operands */
	T0* a43; /* agent_closed_operands */
	T0* a44; /* agent_target */
	T0* a45; /* agent_arguments */
	T0* a46; /* agent_instruction */
	T0* a47; /* agent_expression */
	T0* a48; /* agent_closed_operands_type */
	T0* a49; /* wrapper_expression */
	T0* a50; /* agent_tuple_item_expressions */
	T0* a51; /* agent_manifest_tuple */
	T0* a52; /* formal_arguments */
	T0* a53; /* manifest_array_types */
	T0* a54; /* big_manifest_array_types */
	T0* a55; /* manifest_tuple_types */
	T0* a56; /* once_features */
	T0* a57; /* once_per_process_counts */
	T0* a58; /* once_per_thread_counts */
	T0* a59; /* constant_features */
	T0* a60; /* inline_constants */
	T0* a61; /* dispose_procedures */
	T0* a62; /* dynamic_type_id_set_names */
	T0* a63; /* called_features */
	T0* a64; /* called_static_features */
	T0* a65; /* included_header_filenames */
	T0* a66; /* included_runtime_header_files */
	T0* a67; /* included_runtime_c_files */
	T0* a68; /* c_filenames */
	T1 a69; /* finalize_mode */
	T0* a70; /* locals_written_in_body */
	T0* a71; /* locals_written_in_rescue */
	T0* a72; /* locals_written */
	T0* a73; /* locals_read_in_body */
	T0* a74; /* locals_read_in_rescue */
	T0* a75; /* locals_read */
	T0* a76; /* external_c_regexp */
	T0* a77; /* external_c_macro_regexp */
	T0* a78; /* external_c_struct_regexp */
	T0* a79; /* external_c_inline_regexp */
	T0* a80; /* old_external_c_regexp */
	T0* a81; /* old_external_c_macro_regexp */
	T0* a82; /* old_external_c_struct_regexp */
	T0* a83; /* external_cpp_regexp */
	T0* a84; /* external_cpp_inline_regexp */
	T0* a85; /* external_dllwin_regexp */
	T0* a86; /* old_external_dllwin32_regexp */
	T1 a87; /* in_static_feature */
	T1 a88; /* type_info_ancestors_used */
	T1 a89; /* type_info_attributes_used */
	T1 a90; /* type_info_attribute_name_used */
	T1 a91; /* type_info_attribute_type_id_used */
	T1 a92; /* type_info_attribute_offset_used */
	T1 a93; /* type_info_generator_used */
	T1 a94; /* type_info_name_used */
	T1 a95; /* type_info_generic_parameters_used */
	T1 a96; /* type_info_object_size_used */
	T0* a97; /* c_file */
	T6 a98; /* c_file_size */
	T0* a99; /* cpp_file */
	T6 a100; /* cpp_file_size */
	T6 a101; /* indentation */
	T0* a102; /* assignment_target */
	T1 a103; /* in_operand */
	T0* a104; /* call_target_type */
	T0* a105; /* current_agent */
	T1 a106; /* call_target_check_void */
	T1 a107; /* has_rescue */
	T1 a108; /* result_written */
	T1 a109; /* result_read */
	T1 a110; /* has_retry */
};

/* Struct for type KL_SHELL_COMMAND */
struct S85 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* exit_code */
	T0* a2; /* string_command */
	T0* a3; /* command */
	T6 a4; /* return_code */
	T1 a5; /* is_system_code */
};

/* Struct for type ET_SYSTEM_MULTIPROCESSOR */
struct S87 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* stop_request */
	T0* a2; /* eiffel_parser */
	T0* a3; /* interface_checker */
	T1 a4; /* benchmark_shown */
	T0* a5; /* ise_version */
	T0* a6; /* other_processors */
	T0* a7; /* error_handler */
	T0* a8; /* eiffel_preparser */
	T0* a9; /* master_class_checker */
	T0* a10; /* provider_checker */
	T0* a11; /* ancestor_builder */
	T0* a12; /* feature_flattener */
	T0* a13; /* implementation_checker */
	T0* a14; /* ecma_version */
	T1 a15; /* nested_benchmark_shown */
	T1 a16; /* metrics_shown */
	T1 a17; /* flat_mode */
	T1 a18; /* flat_dbc_mode */
	T0* a19; /* processed_class_count_stack */
	T0* a20; /* ast_factory */
	T0* a21; /* dotnet_assembly_consumer */
	T1 a22; /* use_attribute_keyword */
	T1 a23; /* use_note_keyword */
	T1 a24; /* use_reference_keyword */
	T1 a25; /* use_attached_keyword */
	T1 a26; /* use_detachable_keyword */
	T1 a27; /* preparse_shallow_mode */
	T1 a28; /* preparse_single_mode */
	T1 a29; /* preparse_multiple_mode */
	T1 a30; /* unknown_builtin_reported */
	T1 a31; /* qualified_anchored_types_cycle_detection_enabled */
	T1 a32; /* cluster_dependence_enabled */
	T6 a33; /* postponed_class_count */
	T6 a34; /* processed_class_count */
	T1 a35; /* preparse_readonly_mode */
	T1 a36; /* preparse_override_mode */
	T1 a37; /* use_cluster_dependence_pathnames */
	T1 a38; /* providers_enabled */
	T1 a39; /* suppliers_enabled */
};

/* Struct for type UT_VERSION */
struct S88 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* internal_major */
	T6 a2; /* internal_minor */
	T6 a3; /* internal_revision */
	T6 a4; /* internal_build */
};

/* Struct for type ET_DYNAMIC_PUSH_TYPE_SET_BUILDER */
struct S91 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* has_fatal_error */
	T0* a2; /* current_dynamic_system */
	T0* a3; /* feature_checker */
	T0* a4; /* current_dynamic_type */
	T0* a5; /* current_dynamic_feature */
	T0* a6; /* dynamic_type_sets */
	T0* a7; /* object_id_dynamic_type_set */
	T0* a8; /* current_index */
	T0* a9; /* result_index */
	T0* a10; /* constant_indexes */
	T0* a11; /* none_index */
	T1 a12; /* catcall_error_mode */
	T1 a13; /* catcall_warning_mode */
	T1 a14; /* no_debug */
	T1 a15; /* no_assertion */
	T1 a16; /* is_built */
	T0* a17; /* alive_conforming_descendants_per_type */
	T0* a18; /* type_checker */
	T0* a19; /* current_type */
	T0* a20; /* current_class */
	T0* a21; /* current_feature */
	T0* a22; /* current_feature_impl */
	T0* a23; /* current_class_impl */
	T0* a24; /* enclosing_inline_agents */
	T0* a25; /* unused_overloaded_procedures_list */
	T0* a26; /* unused_overloaded_queries_list */
	T0* a27; /* unused_overloaded_features_list */
	T0* a28; /* unused_contexts */
	T0* a29; /* current_context */
	T0* a30; /* current_target_type */
	T0* a31; /* current_object_test_types */
	T0* a32; /* current_object_test_scope */
	T0* a33; /* object_test_scope_builder */
	T0* a34; /* current_across_cursor_types */
	T0* a35; /* current_across_cursor_scope */
	T0* a36; /* current_initialization_scope */
	T0* a37; /* current_attachment_scope */
	T0* a38; /* attachment_scope_builder */
	T0* a39; /* unused_attachment_scopes */
	T0* a40; /* common_ancestor_type_list */
	T0* a41; /* indexing_term_list */
	T0* a42; /* default_creation_call_name */
	T0* a43; /* default_creation_call */
	T0* a44; /* system_processor */
	T1 a45; /* in_precursor */
	T1 a46; /* in_static_feature */
	T1 a47; /* in_rescue */
	T0* a48; /* current_inline_agent */
	T0* a49; /* supplier_handler */
	T1 a50; /* in_invariant */
	T1 a51; /* in_precondition */
	T1 a52; /* in_postcondition */
	T1 a53; /* in_check_instruction */
	T0* a54; /* precursor_procedures */
	T0* a55; /* precursor_queries */
};

/* Struct for type ET_CLASS_TYPE */
struct S93 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* type_mark */
	T0* a2; /* name */
	T0* a3; /* actual_parameters */
	T0* a4; /* named_base_class */
	T1 a5; /* tuple_actual_parameters_unfolded_2 */
	T1 a6; /* tuple_actual_parameters_unfolded_1 */
};

/* Struct for type ET_TOKEN_CONSTANTS */
struct S94 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type ET_CLASS */
struct S95 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* group */
	T0* a2; /* status_mutex */
	T1 a3; /* unprotected_is_parsed */
	T1 a4; /* unprotected_has_syntax_error */
	T1 a5; /* unprotected_ancestors_built */
	T1 a6; /* unprotected_has_ancestors_error */
	T1 a7; /* unprotected_features_flattened */
	T1 a8; /* unprotected_has_flattening_error */
	T1 a9; /* unprotected_interface_checked */
	T1 a10; /* unprotected_has_interface_error */
	T1 a11; /* unprotected_implementation_checked */
	T1 a12; /* unprotected_has_implementation_error */
	T0* a13; /* name */
	T6 a14; /* id */
	T0* a15; /* ancestors */
	T0* a16; /* conforming_ancestors */
	T0* a17; /* queries */
	T0* a18; /* procedures */
	T0* a19; /* class_keyword */
	T0* a20; /* end_keyword */
	T0* a21; /* processing_mutex */
	T0* a22; /* formal_parameter_types_mutex */
	T0* a23; /* named_base_class */
	T6 a24; /* time_stamp */
	T8 a25; /* class_code */
	T1 a26; /* tuple_actual_parameters_unfolded_1 */
	T1 a27; /* tuple_actual_parameters_unfolded_2 */
	T1 a28; /* is_checking_implementation */
	T0* a29; /* formal_parameters */
	T6 a30; /* index */
	T1 a31; /* redeclared_signatures_checked */
	T0* a32; /* parent_clauses */
	T0* a33; /* class_mark */
	T0* a34; /* creators */
	T1 a35; /* unprotected_is_marked */
	T0* a36; /* filename */
	T0* a37; /* convert_features */
	T0* a38; /* suppliers */
	T0* a39; /* invariants */
	T1 a40; /* is_ignored */
	T1 a41; /* is_interface */
	T1 a42; /* has_deferred_features */
	T6 a43; /* registered_feature_count */
	T0* a44; /* external_keyword */
	T0* a45; /* frozen_keyword */
	T0* a46; /* feature_clauses */
	T0* a47; /* first_indexing */
	T0* a48; /* second_indexing */
	T6 a49; /* tuple_constraint_position */
	T0* a50; /* obsolete_message */
	T6 a51; /* registered_inline_constant_count */
	T0* a52; /* leading_break */
	T0* a53; /* providers */
	T0* a54; /* formal_parameter_types */
};

/* Struct for type KL_OPERATING_SYSTEM */
struct S97 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type UT_CANNOT_READ_FILE_ERROR */
struct S100 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* parameters */
};

/* Struct for type UT_VERSION_NUMBER */
struct S102 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* parameters */
};

/* Struct for type UT_MESSAGE */
struct S103 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* parameters */
};

/* Struct for type ET_ECF_CAPABILITIES */
struct S106 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* primary_use_capabilities */
	T0* a2; /* primary_support_capabilities */
	T0* a3; /* secondary_capabilities */
};

/* Struct for type KL_EXECUTION_ENVIRONMENT */
struct S107 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type AP_ERROR */
struct S108 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* parameters */
	T0* a2; /* default_template */
	T0* a3; /* code */
};

/* Struct for type [attached] DS_HASH_SET [[attached] STRING_8] */
struct S109 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* internal_cursor */
	T0* a3; /* item_storage */
	T0* a4; /* equality_tester */
	T6 a5; /* count */
	T6 a6; /* capacity */
	T6 a7; /* slots_position */
	T6 a8; /* free_slot */
	T6 a9; /* last_position */
	T6 a10; /* modulus */
	T6 a11; /* clashes_previous_position */
	T6 a12; /* found_position */
	T0* a13; /* special_item_routines */
	T0* a14; /* clashes */
	T0* a15; /* slots */
	T0* a16; /* hash_function */
};

/* Struct for type VOID_TARGET */
struct S112 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* throwing_exception */
	T0* a2; /* internal_trace */
	T0* a3; /* c_description */
	T0* a4; /* recipient_name */
	T0* a5; /* type_name */
};

/* Struct for type ROUTINE_FAILURE */
struct S114 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* routine_name */
	T0* a2; /* class_name */
	T0* a3; /* throwing_exception */
	T0* a4; /* internal_trace */
	T0* a5; /* c_description */
	T0* a6; /* recipient_name */
	T0* a7; /* type_name */
};

/* Struct for type OLD_VIOLATION */
struct S116 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* throwing_exception */
	T0* a2; /* internal_trace */
	T0* a3; /* c_description */
	T0* a4; /* recipient_name */
	T0* a5; /* type_name */
};

/* Struct for type NO_MORE_MEMORY */
struct S119 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* internal_trace */
	T6 a2; /* internal_code */
	T0* a3; /* throwing_exception */
	T0* a4; /* c_description */
	T0* a5; /* recipient_name */
	T0* a6; /* type_name */
};

/* Struct for type INVARIANT_VIOLATION */
struct S120 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* throwing_exception */
	T1 a2; /* is_entry */
	T0* a3; /* internal_trace */
	T0* a4; /* c_description */
	T0* a5; /* recipient_name */
	T0* a6; /* type_name */
};

/* Struct for type OPERATING_SYSTEM_SIGNAL_FAILURE */
struct S121 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* throwing_exception */
	T6 a2; /* signal_code */
	T0* a3; /* internal_trace */
	T0* a4; /* c_description */
	T0* a5; /* recipient_name */
	T0* a6; /* type_name */
};

/* Struct for type IO_FAILURE */
struct S122 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* internal_code */
	T0* a2; /* throwing_exception */
	T6 a3; /* error_code */
	T0* a4; /* internal_trace */
	T0* a5; /* c_description */
	T0* a6; /* recipient_name */
	T0* a7; /* type_name */
};

/* Struct for type OPERATING_SYSTEM_FAILURE */
struct S123 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* throwing_exception */
	T6 a2; /* error_code */
	T0* a3; /* internal_trace */
	T0* a4; /* c_description */
	T0* a5; /* recipient_name */
	T0* a6; /* type_name */
};

/* Struct for type COM_FAILURE */
struct S124 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* throwing_exception */
	T6 a2; /* hresult_code */
	T0* a3; /* exception_information */
	T6 a4; /* hresult */
	T0* a5; /* internal_trace */
	T0* a6; /* c_description */
	T0* a7; /* recipient_name */
	T0* a8; /* type_name */
};

/* Struct for type EIFFEL_RUNTIME_PANIC */
struct S125 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* internal_code */
	T0* a2; /* throwing_exception */
	T0* a3; /* internal_trace */
	T0* a4; /* c_description */
	T0* a5; /* recipient_name */
	T0* a6; /* type_name */
};

/* Struct for type PRECONDITION_VIOLATION */
struct S128 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* throwing_exception */
	T0* a2; /* internal_trace */
	T0* a3; /* c_description */
	T0* a4; /* recipient_name */
	T0* a5; /* type_name */
};

/* Struct for type POSTCONDITION_VIOLATION */
struct S129 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* throwing_exception */
	T0* a2; /* internal_trace */
	T0* a3; /* c_description */
	T0* a4; /* recipient_name */
	T0* a5; /* type_name */
};

/* Struct for type FLOATING_POINT_FAILURE */
struct S130 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* throwing_exception */
	T0* a2; /* internal_trace */
	T0* a3; /* c_description */
	T0* a4; /* recipient_name */
	T0* a5; /* type_name */
};

/* Struct for type CHECK_VIOLATION */
struct S131 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* throwing_exception */
	T0* a2; /* internal_trace */
	T0* a3; /* c_description */
	T0* a4; /* recipient_name */
	T0* a5; /* type_name */
};

/* Struct for type BAD_INSPECT_VALUE */
struct S132 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* throwing_exception */
	T0* a2; /* internal_trace */
	T0* a3; /* c_description */
	T0* a4; /* recipient_name */
	T0* a5; /* type_name */
};

/* Struct for type VARIANT_VIOLATION */
struct S133 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* throwing_exception */
	T0* a2; /* internal_trace */
	T0* a3; /* c_description */
	T0* a4; /* recipient_name */
	T0* a5; /* type_name */
};

/* Struct for type LOOP_INVARIANT_VIOLATION */
struct S134 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* throwing_exception */
	T0* a2; /* internal_trace */
	T0* a3; /* c_description */
	T0* a4; /* recipient_name */
	T0* a5; /* type_name */
};

/* Struct for type RESCUE_FAILURE */
struct S135 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* throwing_exception */
	T0* a2; /* internal_trace */
	T0* a3; /* c_description */
	T0* a4; /* recipient_name */
	T0* a5; /* type_name */
};

/* Struct for type RESUMPTION_FAILURE */
struct S136 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* throwing_exception */
	T0* a2; /* internal_trace */
	T0* a3; /* c_description */
	T0* a4; /* recipient_name */
	T0* a5; /* type_name */
};

/* Struct for type CREATE_ON_DEFERRED */
struct S137 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* throwing_exception */
	T0* a2; /* internal_trace */
	T0* a3; /* c_description */
	T0* a4; /* recipient_name */
	T0* a5; /* type_name */
};

/* Struct for type EXTERNAL_FAILURE */
struct S138 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* throwing_exception */
	T0* a2; /* internal_trace */
	T0* a3; /* c_description */
	T0* a4; /* recipient_name */
	T0* a5; /* type_name */
};

/* Struct for type VOID_ASSIGNED_TO_EXPANDED */
struct S139 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* throwing_exception */
	T0* a2; /* internal_trace */
	T0* a3; /* c_description */
	T0* a4; /* recipient_name */
	T0* a5; /* type_name */
};

/* Struct for type EXCEPTION_IN_SIGNAL_HANDLER_FAILURE */
struct S140 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* throwing_exception */
	T0* a2; /* internal_trace */
	T0* a3; /* c_description */
	T0* a4; /* recipient_name */
	T0* a5; /* type_name */
};

/* Struct for type MISMATCH_FAILURE */
struct S141 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* throwing_exception */
	T0* a2; /* internal_trace */
	T0* a3; /* c_description */
	T0* a4; /* recipient_name */
	T0* a5; /* type_name */
};

/* Struct for type DEVELOPER_EXCEPTION */
struct S142 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* throwing_exception */
	T0* a2; /* internal_trace */
	T0* a3; /* c_description */
	T0* a4; /* recipient_name */
	T0* a5; /* type_name */
};

/* Struct for type ADDRESS_APPLIED_TO_MELTED_FEATURE */
struct S143 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* throwing_exception */
	T0* a2; /* internal_trace */
	T0* a3; /* c_description */
	T0* a4; /* recipient_name */
	T0* a5; /* type_name */
};

/* Struct for type SERIALIZATION_FAILURE */
struct S144 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* throwing_exception */
	T0* a2; /* internal_trace */
	T0* a3; /* c_description */
	T0* a4; /* recipient_name */
	T0* a5; /* type_name */
};

/* Struct for type EXECUTION_ENVIRONMENT */
struct S145 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* return_code */
};

/* Struct for type KL_WINDOWS_FILE_SYSTEM */
struct S146 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type KL_UNIX_FILE_SYSTEM */
struct S147 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type PRIMES */
struct S148 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [[attached] INTEGER_32] */
struct S149 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T6 z2[1]; /* item */
};

/* Struct for type [attached] SPECIAL [[attached] BOOLEAN] */
struct S150 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T1 z2[1]; /* item */
};

/* Struct for type MANAGED_POINTER */
struct S154 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T14 a1; /* item */
	T6 a2; /* count */
	T1 a3; /* is_shared */
};

/* Struct for type [attached] KL_LINKABLE [[attached] CHARACTER_8] */
struct S155 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T2 a1; /* item */
	T0* a2; /* right */
};

/* Struct for type FILE_INFO */
struct S157 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* buffered_file_info */
	T1 a2; /* is_following_symlinks */
	T1 a3; /* exists */
	T0* a4; /* internal_file_name */
	T0* a5; /* internal_name_pointer */
};

/* Struct for type MUTEX */
struct S158 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T14 a1; /* mutex_pointer */
	T14 a2; /* owner_thread_id */
};

/* Struct for type UT_ERROR_HANDLER */
struct S159 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* error_file */
	T0* a2; /* warning_file */
	T0* a3; /* info_file */
};

/* Struct for type KL_STANDARD_FILES */
struct S160 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type KL_STDERR_FILE */
struct S161 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type KL_STDOUT_FILE */
struct S162 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type UC_STRING */
struct S163 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T6 a2; /* byte_count */
	T6 a3; /* internal_hash_code */
	T6 a4; /* internal_case_insensitive_hash_code */
	T0* a5; /* area */
	T6 a6; /* last_byte_index_input */
	T6 a7; /* last_byte_index_result */
};

/* Struct for type KL_ANY_ROUTINES */
struct S164 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type ST_WORD_WRAPPER */
struct S166 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* new_line_indentation */
	T6 a2; /* broken_words */
	T6 a3; /* maximum_text_width */
};

/* Struct for type AP_DISPLAY_HELP_FLAG */
struct S168 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* description */
	T2 a2; /* short_form */
	T1 a3; /* has_short_form */
	T0* a4; /* long_form */
	T6 a5; /* occurrences */
	T1 a6; /* is_hidden */
	T1 a7; /* is_mandatory */
	T6 a8; /* maximum_occurrences */
};

/* Struct for type [attached] DS_ARRAYED_LIST [[attached] AP_OPTION] */
struct S169 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* internal_cursor */
	T6 a2; /* count */
	T0* a3; /* equality_tester */
	T0* a4; /* storage */
	T0* a5; /* special_routines */
	T6 a6; /* capacity */
};

/* Struct for type [attached] DS_ARRAYED_LIST [[attached] AP_ALTERNATIVE_OPTIONS_LIST] */
struct S170 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* internal_cursor */
	T0* a2; /* storage */
	T0* a3; /* special_routines */
	T6 a4; /* capacity */
	T6 a5; /* count */
};

/* Struct for type [attached] DS_ARRAYED_LIST [[attached] STRING_8] */
struct S171 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
	T0* a3; /* internal_cursor */
	T0* a4; /* special_routines */
	T6 a5; /* capacity */
	T0* a6; /* equality_tester */
};

/* Struct for type AP_ERROR_HANDLER */
struct S172 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* has_error */
	T0* a2; /* error_file */
	T0* a3; /* warning_file */
	T0* a4; /* info_file */
};

/* Struct for type [attached] DS_LINKED_LIST_CURSOR [[attached] AP_OPTION] */
struct S175 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T1 a2; /* before */
	T1 a3; /* after */
	T0* a4; /* current_cell */
	T0* a5; /* next_cursor */
};

/* Struct for type [attached] DS_ARRAYED_LIST [detachable STRING_8] */
struct S176 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
	T0* a3; /* special_routines */
	T6 a4; /* capacity */
	T0* a5; /* internal_cursor */
};

/* Struct for type [attached] DS_ARRAYED_LIST_CURSOR [detachable STRING_8] */
struct S177 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* container */
	T0* a3; /* next_cursor */
};

/* Struct for type [attached] DS_LINKED_LIST [[attached] STRING_8] */
struct S178 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* internal_cursor */
	T0* a2; /* last_cell */
	T6 a3; /* count */
	T0* a4; /* first_cell */
	T0* a5; /* equality_tester */
};

/* Struct for type KL_STRING_EQUALITY_TESTER */
struct S179 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] DS_LINKED_LIST [[attached] BOOLEAN] */
struct S182 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* last_cell */
	T0* a3; /* internal_cursor */
	T0* a4; /* first_cell */
};

/* Struct for type [attached] DS_LINKED_LIST [[attached] INTEGER_32] */
struct S185 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* last_cell */
	T0* a3; /* internal_cursor */
	T0* a4; /* first_cell */
};

/* Struct for type KL_NULL_TEXT_OUTPUT_STREAM */
struct S186 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* name */
};

/* Struct for type ET_XACE_AST_FACTORY */
struct S187 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type XM_DOCUMENT */
struct S188 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* root_element */
	T0* a2; /* children */
};

/* Struct for type XM_ELEMENT */
struct S189 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* name */
	T0* a2; /* namespace */
	T0* a3; /* parent */
	T0* a4; /* children */
};

/* Struct for type XM_TREE_CALLBACKS_PIPE */
struct S192 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* error */
	T0* a2; /* tree */
	T0* a3; /* start */
	T0* a4; /* last */
};

/* Struct for type XM_STOP_ON_ERROR_FILTER */
struct S193 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* has_error */
	T0* a2; /* last_error */
	T0* a3; /* next */
};

/* Struct for type XM_CALLBACKS_TO_TREE_FILTER */
struct S194 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* last_position_table */
	T0* a2; /* document */
	T0* a3; /* source_parser */
	T0* a4; /* current_element */
	T0* a5; /* namespace_cache */
	T0* a6; /* next */
};

/* Struct for type ET_XACE_VALIDATOR */
struct S195 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* has_error */
	T0* a2; /* error_handler */
};

/* Struct for type ET_XACE_PREPROCESSOR */
struct S196 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* variables */
	T0* a2; /* error_handler */
};

/* Struct for type ET_XACE_SYSTEM */
struct S197 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* root_type */
	T0* a2; /* none_type */
	T0* a3; /* any_type */
	T0* a4; /* master_classes */
	T0* a5; /* root_creation */
	T6 a6; /* default_create_seed */
	T0* a7; /* boolean_type */
	T0* a8; /* character_8_type */
	T0* a9; /* character_32_type */
	T0* a10; /* integer_8_type */
	T0* a11; /* integer_16_type */
	T0* a12; /* integer_32_type */
	T0* a13; /* integer_64_type */
	T0* a14; /* natural_8_type */
	T0* a15; /* natural_16_type */
	T0* a16; /* natural_32_type */
	T0* a17; /* natural_64_type */
	T0* a18; /* real_32_type */
	T0* a19; /* real_64_type */
	T0* a20; /* pointer_type */
	T0* a21; /* special_any_type */
	T0* a22; /* string_8_type */
	T0* a23; /* string_32_type */
	T0* a24; /* immutable_string_32_type */
	T0* a25; /* array_any_type */
	T0* a26; /* typed_pointer_any_type */
	T0* a27; /* routine_type */
	T0* a28; /* detachable_tuple_type */
	T0* a29; /* universe */
	T0* a30; /* ise_exception_manager_type */
	T0* a31; /* detachable_none_type */
	T0* a32; /* system_name */
	T0* a33; /* external_cflags */
	T0* a34; /* external_include_pathnames */
	T0* a35; /* external_library_pathnames */
	T0* a36; /* options */
	T0* a37; /* targets */
	T1 a38; /* attachment_type_conformance_mode */
	T1 a39; /* use_boehm_gc */
	T0* a40; /* external_linker_flags */
	T0* a41; /* external_object_pathnames */
	T0* a42; /* character_type */
	T0* a43; /* wide_character_type */
	T0* a44; /* integer_type */
	T0* a45; /* natural_type */
	T0* a46; /* real_type */
	T0* a47; /* double_type */
	T0* a48; /* external_make_pathnames */
	T0* a49; /* external_resource_pathnames */
	T6 a50; /* routine_call_seed */
	T6 a51; /* function_item_seed */
	T6 a52; /* is_equal_seed */
	T6 a53; /* dispose_seed */
	T1 a54; /* trace_mode */
	T1 a55; /* exception_trace_mode */
	T1 a56; /* multithreaded_mode */
	T1 a57; /* console_application_mode */
	T0* a58; /* name */
	T0* a59; /* selected_target */
	T0* a60; /* clusters */
	T0* a61; /* mounted_libraries */
	T0* a62; /* root_class_name */
	T0* a63; /* dotnet_assemblies */
	T0* a64; /* libraries */
	T0* a65; /* register_class_mutex */
	T1 a66; /* alias_transition_mode */
	T0* a67; /* current_system */
	T0* a68; /* tuple_type */
	T0* a69; /* detachable_exception_type */
	T0* a70; /* exception_type */
	T0* a71; /* string_type */
	T1 a72; /* is_preparsed */
	T1 a73; /* is_read_only */
	T0* a74; /* creation_procedure_name */
	T0* a75; /* master_class_mutex */
	T0* a76; /* detachable_any_type */
	T0* a77; /* detachable_separate_any_type */
	T0* a78; /* separate_any_type */
	T0* a79; /* any_parent */
	T0* a80; /* any_parents */
	T0* a81; /* any_clients */
	T0* a82; /* tuple_like_current_type */
	T0* a83; /* unfolded_empty_tuple_actual_parameters */
	T0* a84; /* array_detachable_any_type */
	T0* a85; /* array_none_type */
	T0* a86; /* array_like_current_type */
	T0* a87; /* character_8_convert_feature */
	T0* a88; /* character_32_convert_feature */
	T0* a89; /* exception_manager_type */
	T0* a90; /* function_type */
	T0* a91; /* integer_8_convert_feature */
	T0* a92; /* integer_16_convert_feature */
	T0* a93; /* integer_32_convert_feature */
	T0* a94; /* integer_64_convert_feature */
	T0* a95; /* iterable_detachable_any_type */
	T0* a96; /* natural_8_convert_feature */
	T0* a97; /* natural_16_convert_feature */
	T0* a98; /* natural_32_convert_feature */
	T0* a99; /* natural_64_convert_feature */
	T0* a100; /* predicate_like_current_type */
	T0* a101; /* procedure_like_current_type */
	T0* a102; /* real_32_convert_feature */
	T0* a103; /* real_64_convert_feature */
	T0* a104; /* special_detachable_any_type */
	T0* a105; /* detachable_string_type */
	T0* a106; /* string_8_convert_feature */
	T0* a107; /* string_32_convert_feature */
	T0* a108; /* system_object_type */
	T0* a109; /* system_object_parents */
	T0* a110; /* system_string_type */
	T0* a111; /* type_any_type */
	T0* a112; /* type_type_any_type */
	T0* a113; /* type_detachable_any_type */
	T0* a114; /* type_detachable_like_current_type */
	T0* a115; /* detachable_type_exception_type */
	T0* a116; /* type_detachable_exception_type */
	T0* a117; /* detachable_type_detachable_any_type */
	T0* a118; /* type_like_current_type */
	T0* a119; /* typed_pointer_like_current_type */
	T0* a120; /* scm_read_mapping_builder */
	T0* a121; /* scm_write_mapping_builder */
	T6 a122; /* registered_class_count */
	T1 a123; /* scoop_mode */
	T0* a124; /* implicit_attachment_type_mark */
	T6 a125; /* copy_seed */
	T6 a126; /* iterable_new_cursor_seed */
	T6 a127; /* iteration_cursor_after_seed */
	T6 a128; /* iteration_cursor_forth_seed */
	T1 a129; /* target_type_attachment_mode */
	T1 a130; /* obsolete_routine_type_mode */
	T1 a131; /* is_dotnet */
};

/* Struct for type [attached] DS_HASH_TABLE [[attached] ET_XACE_LIBRARY_CONFIG, [attached] STRING_8] */
struct S198 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T6 a2; /* last_position */
	T6 a3; /* free_slot */
	T6 a4; /* position */
	T0* a5; /* key_equality_tester */
	T0* a6; /* internal_keys */
	T0* a7; /* equality_tester */
	T0* a8; /* internal_cursor */
	T6 a9; /* found_position */
	T0* a10; /* item_storage */
	T0* a11; /* key_storage */
	T0* a12; /* clashes */
	T6 a13; /* modulus */
	T0* a14; /* slots */
	T6 a15; /* capacity */
	T0* a16; /* special_item_routines */
	T0* a17; /* special_key_routines */
	T6 a18; /* slots_position */
	T6 a19; /* clashes_previous_position */
	T0* a20; /* hash_function */
};

/* Struct for type XM_EXPAT_PARSER_FACTORY */
struct S199 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type XM_EIFFEL_PARSER */
struct S200 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* internal_last_error_description */
	T0* a2; /* scanner */
	T0* a3; /* error_positions */
	T0* a4; /* scanners */
	T1 a5; /* in_external_dtd */
	T0* a6; /* callbacks */
	T0* a7; /* entities */
	T0* a8; /* pe_entities */
	T0* a9; /* dtd_resolver */
	T0* a10; /* entity_resolver */
	T1 a11; /* use_namespaces */
	T6 a12; /* string_mode */
	T0* a13; /* dtd_callbacks */
	T0* a14; /* last_string_value */
	T0* a15; /* yyss */
	T0* a16; /* yyspecial_routines1 */
	T6 a17; /* yyvsc1 */
	T0* a18; /* yyvs1 */
	T0* a19; /* yyspecial_routines2 */
	T6 a20; /* yyvsc2 */
	T0* a21; /* yyvs2 */
	T0* a22; /* yyspecial_routines3 */
	T6 a23; /* yyvsc3 */
	T0* a24; /* yyvs3 */
	T0* a25; /* yyspecial_routines4 */
	T6 a26; /* yyvsc4 */
	T0* a27; /* yyvs4 */
	T0* a28; /* yyspecial_routines5 */
	T6 a29; /* yyvsc5 */
	T0* a30; /* yyvs5 */
	T0* a31; /* yyspecial_routines6 */
	T6 a32; /* yyvsc6 */
	T0* a33; /* yyvs6 */
	T0* a34; /* yyspecial_routines7 */
	T6 a35; /* yyvsc7 */
	T0* a36; /* yyvs7 */
	T0* a37; /* yyspecial_routines8 */
	T6 a38; /* yyvsc8 */
	T0* a39; /* yyvs8 */
	T0* a40; /* yyspecial_routines9 */
	T6 a41; /* yyvsc9 */
	T0* a42; /* yyvs9 */
	T0* a43; /* yyspecial_routines10 */
	T6 a44; /* yyvsc10 */
	T0* a45; /* yyvs10 */
	T0* a46; /* yyspecial_routines11 */
	T6 a47; /* yyvsc11 */
	T0* a48; /* yyvs11 */
	T0* a49; /* yytranslate */
	T0* a50; /* yyr1 */
	T0* a51; /* yytypes1 */
	T0* a52; /* yytypes2 */
	T0* a53; /* yydefact */
	T0* a54; /* yydefgoto */
	T0* a55; /* yypact */
	T0* a56; /* yypgoto */
	T0* a57; /* yytable */
	T0* a58; /* yycheck */
	T6 a59; /* yy_parsing_status */
	T6 a60; /* yy_suspended_yystacksize */
	T6 a61; /* yy_suspended_yystate */
	T6 a62; /* yy_suspended_yyn */
	T6 a63; /* yy_suspended_yychar1 */
	T6 a64; /* yy_suspended_index */
	T6 a65; /* yy_suspended_yyss_top */
	T6 a66; /* yy_suspended_yy_goto */
	T6 a67; /* yyssp */
	T6 a68; /* error_count */
	T1 a69; /* yy_lookahead_needed */
	T6 a70; /* yyerrstatus */
	T6 a71; /* last_token */
	T6 a72; /* yyvsp1 */
	T6 a73; /* yyvsp2 */
	T6 a74; /* yyvsp3 */
	T6 a75; /* yyvsp4 */
	T6 a76; /* yyvsp5 */
	T6 a77; /* yyvsp6 */
	T6 a78; /* yyvsp7 */
	T6 a79; /* yyvsp8 */
	T6 a80; /* yyvsp9 */
	T6 a81; /* yyvsp10 */
	T6 a82; /* yyvsp11 */
	T0* a83; /* last_detachable_any_value */
};

/* Struct for type ET_XACE_LIBRARY_CONFIG_PARSER */
struct S203 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* error_handler */
	T0* a2; /* ast_factory */
	T0* a3; /* parsed_libraries */
	T0* a4; /* xml_preprocessor */
	T0* a5; /* xml_validator */
	T0* a6; /* xml_parser */
	T0* a7; /* tree_pipe */
	T0* a8; /* library_parser */
	T0* a9; /* last_library */
	T1 a10; /* is_shallow */
	T6 a11; /* new_name_count */
};

/* Struct for type attached KL_EQUALITY_TESTER [[attached] STRING_8] */
struct S206 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] STRING_8] */
struct S207 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [[attached] STRING_8] */
struct S208 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] INTEGER_32] */
struct S209 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] KL_AGENT_HASH_FUNCTION [[attached] STRING_8] */
struct S210 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* hash_agent */
};

/* Struct for type KL_CASE_INSENSITIVE_STRING_EQUALITY_TESTER */
struct S212 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] TUPLE [[attached] STRING_8] */
struct S213 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type [attached] FUNCTION [[attached] TUPLE [[attached] STRING_8], [attached] INTEGER_32] */
struct S214 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T14 a1; /* rout_disp */
	T0* a2; /* closed_operands */
	T14 a3; /* encaps_rout_disp */
	T14 a4; /* calc_rout_addr */
	T1 a5; /* is_target_closed */
	T6 a6; /* open_count */
};

/* Struct for type [attached] TUPLE */
struct S216 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] TUPLE [KL_STRING_ROUTINES] */
struct S217 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type ET_ECF_AST_FACTORY */
struct S219 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] DS_CELL [detachable ET_ECF_SYSTEM] */
struct S220 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* item */
};

/* Struct for type [attached] TUPLE [[attached] ET_ECF_TARGET] */
struct S221 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type [attached] PROCEDURE [[attached] TUPLE [[attached] ET_ECF_TARGET]] */
struct S222 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T14 a1; /* rout_disp */
	T0* a2; /* closed_operands */
	T14 a3; /* encaps_rout_disp */
	T14 a4; /* calc_rout_addr */
	T1 a5; /* is_target_closed */
	T6 a6; /* open_count */
};

/* Struct for type [attached] TUPLE [ET_ECF_SYSTEM_PARSER] */
struct S223 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type [attached] TUPLE [[attached] XM_ELEMENT, detachable XM_POSITION_TABLE, [attached] STRING_8] */
struct S225 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
	T0* z2;
	T0* z3;
};

/* Struct for type [attached] PROCEDURE [[attached] TUPLE [[attached] XM_ELEMENT, detachable XM_POSITION_TABLE, [attached] STRING_8]] */
struct S226 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T14 a1; /* rout_disp */
	T0* a2; /* closed_operands */
	T14 a3; /* encaps_rout_disp */
	T14 a4; /* calc_rout_addr */
	T1 a5; /* is_target_closed */
	T6 a6; /* open_count */
};

/* Struct for type [attached] TUPLE [ET_ECF_SYSTEM_PARSER, detachable STRING_8, [attached] PROCEDURE [[attached] TUPLE [[attached] ET_ECF_TARGET]], [attached] DS_CELL [detachable ET_ECF_SYSTEM]] */
struct S227 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
	T0* z2;
	T0* z3;
	T0* z4;
};

/* Struct for type [attached] DS_HASH_TABLE [[attached] ET_ECF_LIBRARY, [attached] STRING_8] */
struct S229 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* internal_cursor */
	T6 a2; /* found_position */
	T0* a3; /* item_storage */
	T0* a4; /* key_equality_tester */
	T0* a5; /* internal_keys */
	T6 a6; /* count */
	T6 a7; /* last_position */
	T6 a8; /* free_slot */
	T6 a9; /* position */
	T6 a10; /* capacity */
	T0* a11; /* equality_tester */
	T0* a12; /* key_storage */
	T0* a13; /* clashes */
	T6 a14; /* modulus */
	T0* a15; /* slots */
	T6 a16; /* slots_position */
	T6 a17; /* clashes_previous_position */
	T0* a18; /* special_item_routines */
	T0* a19; /* special_key_routines */
	T0* a20; /* hash_function */
};

/* Struct for type [attached] DS_HASH_TABLE [[attached] ET_ECF_DOTNET_ASSEMBLY, [attached] STRING_8] */
struct S230 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* found_position */
	T0* a2; /* item_storage */
	T0* a3; /* key_equality_tester */
	T0* a4; /* internal_keys */
	T6 a5; /* count */
	T6 a6; /* last_position */
	T6 a7; /* free_slot */
	T6 a8; /* position */
	T6 a9; /* capacity */
	T0* a10; /* equality_tester */
	T0* a11; /* internal_cursor */
	T0* a12; /* key_storage */
	T0* a13; /* clashes */
	T6 a14; /* modulus */
	T0* a15; /* slots */
	T6 a16; /* slots_position */
	T6 a17; /* clashes_previous_position */
	T0* a18; /* special_item_routines */
	T0* a19; /* special_key_routines */
	T0* a20; /* hash_function */
};

/* Struct for type ET_ECF_STATE */
struct S231 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* target */
	T0* a2; /* ise_version */
	T1 a3; /* finalize_mode */
	T1 a4; /* is_dotnet */
	T1 a5; /* is_multithreaded */
	T1 a6; /* is_scoop */
};

/* Struct for type ET_IDENTIFIER */
struct S232 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* seed */
	T6 a2; /* index */
	T0* a3; /* name */
	T2 a4; /* status_code */
	T6 a5; /* compressed_position */
	T6 a6; /* hash_code */
};

/* Struct for type ET_ECF_SYSTEM_CONFIG */
struct S233 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* name */
	T0* a2; /* filename */
	T0* a3; /* universe */
	T0* a4; /* ecf_namespace */
	T0* a5; /* ecf_version */
	T0* a6; /* description */
	T0* a7; /* notes */
	T0* a8; /* targets */
	T1 a9; /* is_read_only */
	T0* a10; /* uuid */
	T0* a11; /* library_target */
};

/* Struct for type ET_ECF_LIBRARY */
struct S236 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* name */
	T0* a2; /* filename */
	T0* a3; /* ecf_namespace */
	T0* a4; /* ecf_version */
	T0* a5; /* description */
	T0* a6; /* notes */
	T0* a7; /* targets */
	T1 a8; /* is_read_only */
	T0* a9; /* uuid */
	T0* a10; /* library_target */
	T0* a11; /* current_system */
	T0* a12; /* library */
	T0* a13; /* libraries */
	T0* a14; /* clusters */
	T0* a15; /* dotnet_assemblies */
	T0* a16; /* master_classes */
	T0* a17; /* master_class_mutex */
	T0* a18; /* any_type */
	T0* a19; /* detachable_any_type */
	T0* a20; /* detachable_separate_any_type */
	T0* a21; /* separate_any_type */
	T0* a22; /* any_parent */
	T0* a23; /* any_parents */
	T0* a24; /* any_clients */
	T0* a25; /* tuple_type */
	T0* a26; /* detachable_tuple_type */
	T0* a27; /* tuple_like_current_type */
	T0* a28; /* unfolded_empty_tuple_actual_parameters */
	T0* a29; /* array_any_type */
	T0* a30; /* array_detachable_any_type */
	T0* a31; /* array_none_type */
	T0* a32; /* array_like_current_type */
	T0* a33; /* boolean_type */
	T0* a34; /* character_type */
	T0* a35; /* character_8_type */
	T0* a36; /* character_8_convert_feature */
	T0* a37; /* character_32_type */
	T0* a38; /* character_32_convert_feature */
	T0* a39; /* double_type */
	T0* a40; /* exception_type */
	T0* a41; /* detachable_exception_type */
	T0* a42; /* exception_manager_type */
	T0* a43; /* function_type */
	T0* a44; /* immutable_string_32_type */
	T0* a45; /* integer_type */
	T0* a46; /* integer_8_type */
	T0* a47; /* integer_8_convert_feature */
	T0* a48; /* integer_16_type */
	T0* a49; /* integer_16_convert_feature */
	T0* a50; /* integer_32_type */
	T0* a51; /* integer_32_convert_feature */
	T0* a52; /* integer_64_type */
	T0* a53; /* integer_64_convert_feature */
	T0* a54; /* ise_exception_manager_type */
	T0* a55; /* iterable_detachable_any_type */
	T0* a56; /* natural_type */
	T0* a57; /* natural_8_type */
	T0* a58; /* natural_8_convert_feature */
	T0* a59; /* natural_16_type */
	T0* a60; /* natural_16_convert_feature */
	T0* a61; /* natural_32_type */
	T0* a62; /* natural_32_convert_feature */
	T0* a63; /* natural_64_type */
	T0* a64; /* natural_64_convert_feature */
	T0* a65; /* none_type */
	T0* a66; /* detachable_none_type */
	T0* a67; /* pointer_type */
	T0* a68; /* predicate_like_current_type */
	T0* a69; /* procedure_like_current_type */
	T0* a70; /* real_type */
	T0* a71; /* real_32_type */
	T0* a72; /* real_32_convert_feature */
	T0* a73; /* real_64_type */
	T0* a74; /* real_64_convert_feature */
	T0* a75; /* routine_type */
	T0* a76; /* special_any_type */
	T0* a77; /* special_detachable_any_type */
	T0* a78; /* string_type */
	T0* a79; /* detachable_string_type */
	T0* a80; /* string_8_type */
	T0* a81; /* string_8_convert_feature */
	T0* a82; /* string_32_type */
	T0* a83; /* string_32_convert_feature */
	T0* a84; /* system_object_type */
	T0* a85; /* system_object_parents */
	T0* a86; /* system_string_type */
	T0* a87; /* type_any_type */
	T0* a88; /* type_type_any_type */
	T0* a89; /* type_detachable_any_type */
	T0* a90; /* type_detachable_like_current_type */
	T0* a91; /* detachable_type_exception_type */
	T0* a92; /* type_detachable_exception_type */
	T0* a93; /* detachable_type_detachable_any_type */
	T0* a94; /* type_like_current_type */
	T0* a95; /* typed_pointer_any_type */
	T0* a96; /* typed_pointer_like_current_type */
	T0* a97; /* wide_character_type */
	T0* a98; /* selected_target */
	T0* a99; /* implicit_attachment_type_mark */
	T1 a100; /* obsolete_routine_type_mode */
	T1 a101; /* is_preparsed */
};

/* Struct for type ET_ADAPTED_LIBRARIES */
struct S237 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* libraries */
};

/* Struct for type [attached] TUPLE [[attached] ET_ADAPTED_LIBRARY] */
struct S239 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type [attached] PROCEDURE [[attached] TUPLE [[attached] ET_ADAPTED_LIBRARY]] */
struct S240 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T14 a1; /* rout_disp */
	T0* a2; /* closed_operands */
	T14 a3; /* encaps_rout_disp */
	T14 a4; /* calc_rout_addr */
	T1 a5; /* is_target_closed */
	T6 a6; /* open_count */
};

/* Struct for type ET_COMPRESSED_POSITION */
struct S241 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* compressed_position */
};

/* Struct for type XM_ATTRIBUTE */
struct S245 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* name */
	T0* a2; /* namespace */
	T0* a3; /* value */
	T0* a4; /* parent */
};

/* Struct for type [attached] DS_HASH_TABLE [[attached] ET_ECF_SYSTEM_CONFIG, [attached] STRING_8] */
struct S247 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* found_position */
	T0* a2; /* item_storage */
	T6 a3; /* position */
	T6 a4; /* last_position */
	T6 a5; /* capacity */
	T6 a6; /* slots_position */
	T6 a7; /* count */
	T0* a8; /* key_equality_tester */
	T6 a9; /* modulus */
	T6 a10; /* clashes_previous_position */
	T0* a11; /* special_item_routines */
	T0* a12; /* clashes */
	T0* a13; /* slots */
	T0* a14; /* special_key_routines */
	T0* a15; /* key_storage */
	T0* a16; /* equality_tester */
	T6 a17; /* free_slot */
	T0* a18; /* internal_cursor */
	T0* a19; /* hash_function */
};

/* Struct for type [attached] DS_HASH_TABLE [[attached] ET_ECF_TARGET, [attached] STRING_8] */
struct S248 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T6 a2; /* position */
	T0* a3; /* internal_cursor */
	T0* a4; /* key_storage */
	T0* a5; /* item_storage */
	T6 a6; /* last_position */
	T6 a7; /* capacity */
	T6 a8; /* slots_position */
	T6 a9; /* modulus */
	T6 a10; /* clashes_previous_position */
	T0* a11; /* key_equality_tester */
	T6 a12; /* found_position */
	T0* a13; /* special_item_routines */
	T0* a14; /* clashes */
	T0* a15; /* slots */
	T0* a16; /* special_key_routines */
	T0* a17; /* equality_tester */
	T6 a18; /* free_slot */
	T0* a19; /* hash_function */
};

/* Struct for type [attached] DS_CELL [detachable ET_ECF_SYSTEM_CONFIG] */
struct S250 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* item */
};

/* Struct for type [attached] TUPLE [ET_ECF_SYSTEM_PARSER, ET_ECF_INTERNAL_UNIVERSE, [attached] DS_CELL [detachable ET_ECF_SYSTEM_CONFIG]] */
struct S253 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
	T0* z2;
	T0* z3;
};

/* Struct for type ET_ECF_TARGETS */
struct S255 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* targets */
};

/* Struct for type [attached] DS_HASH_TABLE [detachable RX_REGULAR_EXPRESSION, [attached] STRING_8] */
struct S256 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* key_equality_tester */
	T0* a2; /* internal_keys */
	T0* a3; /* hash_function */
	T6 a4; /* position */
	T6 a5; /* last_position */
	T6 a6; /* capacity */
	T6 a7; /* slots_position */
	T6 a8; /* count */
	T0* a9; /* equality_tester */
	T6 a10; /* found_position */
	T6 a11; /* modulus */
	T6 a12; /* clashes_previous_position */
	T0* a13; /* special_item_routines */
	T0* a14; /* item_storage */
	T0* a15; /* clashes */
	T0* a16; /* slots */
	T0* a17; /* special_key_routines */
	T0* a18; /* key_storage */
	T6 a19; /* free_slot */
	T0* a20; /* internal_cursor */
};

/* Struct for type [attached] DS_HASH_TABLE_CURSOR [detachable RX_REGULAR_EXPRESSION, [attached] STRING_8] */
struct S257 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* container */
	T0* a3; /* next_cursor */
};

/* Struct for type [attached] DS_CELL [detachable ET_ECF_LIBRARY] */
struct S259 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* item */
};

/* Struct for type ET_ECF_ADAPTED_LIBRARY */
struct S260 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* name_id */
	T0* a2; /* filename_id */
	T0* a3; /* universe */
	T0* a4; /* target */
	T0* a5; /* description */
	T0* a6; /* classname_prefix */
	T1 a7; /* is_read_only */
	T1 a8; /* use_application_options */
	T0* a9; /* options */
	T0* a10; /* class_options */
	T0* a11; /* class_renamings */
	T0* a12; /* visible_classes */
	T0* a13; /* conditions */
	T0* a14; /* notes */
	T0* a15; /* name */
	T0* a16; /* library */
};

/* Struct for type [attached] TUPLE [ET_ECF_SYSTEM_PARSER, ET_ECF_ADAPTED_LIBRARY, [attached] DS_CELL [detachable ET_ECF_LIBRARY]] */
struct S261 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
	T0* z2;
	T0* z3;
};

/* Struct for type ET_ADAPTED_DOTNET_ASSEMBLIES */
struct S264 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* dotnet_assemblies */
};

/* Struct for type ET_ECF_DOTNET_ASSEMBLY */
struct S265 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* pathname */
	T0* a2; /* referenced_assemblies */
	T0* a3; /* current_system */
	T0* a4; /* dotnet_assembly */
	T0* a5; /* name */
	T0* a6; /* master_classes */
	T0* a7; /* master_class_mutex */
	T0* a8; /* any_type */
	T0* a9; /* detachable_any_type */
	T0* a10; /* detachable_separate_any_type */
	T0* a11; /* separate_any_type */
	T0* a12; /* any_parent */
	T0* a13; /* any_parents */
	T0* a14; /* any_clients */
	T0* a15; /* tuple_type */
	T0* a16; /* detachable_tuple_type */
	T0* a17; /* tuple_like_current_type */
	T0* a18; /* unfolded_empty_tuple_actual_parameters */
	T0* a19; /* array_any_type */
	T0* a20; /* array_detachable_any_type */
	T0* a21; /* array_none_type */
	T0* a22; /* array_like_current_type */
	T0* a23; /* boolean_type */
	T0* a24; /* character_type */
	T0* a25; /* character_8_type */
	T0* a26; /* character_8_convert_feature */
	T0* a27; /* character_32_type */
	T0* a28; /* character_32_convert_feature */
	T0* a29; /* double_type */
	T0* a30; /* exception_type */
	T0* a31; /* detachable_exception_type */
	T0* a32; /* exception_manager_type */
	T0* a33; /* function_type */
	T0* a34; /* immutable_string_32_type */
	T0* a35; /* integer_type */
	T0* a36; /* integer_8_type */
	T0* a37; /* integer_8_convert_feature */
	T0* a38; /* integer_16_type */
	T0* a39; /* integer_16_convert_feature */
	T0* a40; /* integer_32_type */
	T0* a41; /* integer_32_convert_feature */
	T0* a42; /* integer_64_type */
	T0* a43; /* integer_64_convert_feature */
	T0* a44; /* ise_exception_manager_type */
	T0* a45; /* iterable_detachable_any_type */
	T0* a46; /* natural_type */
	T0* a47; /* natural_8_type */
	T0* a48; /* natural_8_convert_feature */
	T0* a49; /* natural_16_type */
	T0* a50; /* natural_16_convert_feature */
	T0* a51; /* natural_32_type */
	T0* a52; /* natural_32_convert_feature */
	T0* a53; /* natural_64_type */
	T0* a54; /* natural_64_convert_feature */
	T0* a55; /* none_type */
	T0* a56; /* detachable_none_type */
	T0* a57; /* pointer_type */
	T0* a58; /* predicate_like_current_type */
	T0* a59; /* procedure_like_current_type */
	T0* a60; /* real_type */
	T0* a61; /* real_32_type */
	T0* a62; /* real_32_convert_feature */
	T0* a63; /* real_64_type */
	T0* a64; /* real_64_convert_feature */
	T0* a65; /* routine_type */
	T0* a66; /* special_any_type */
	T0* a67; /* special_detachable_any_type */
	T0* a68; /* string_type */
	T0* a69; /* detachable_string_type */
	T0* a70; /* string_8_type */
	T0* a71; /* string_8_convert_feature */
	T0* a72; /* string_32_type */
	T0* a73; /* string_32_convert_feature */
	T0* a74; /* system_object_type */
	T0* a75; /* system_object_parents */
	T0* a76; /* system_string_type */
	T0* a77; /* type_any_type */
	T0* a78; /* type_type_any_type */
	T0* a79; /* type_detachable_any_type */
	T0* a80; /* type_detachable_like_current_type */
	T0* a81; /* detachable_type_exception_type */
	T0* a82; /* type_detachable_exception_type */
	T0* a83; /* detachable_type_detachable_any_type */
	T0* a84; /* type_like_current_type */
	T0* a85; /* typed_pointer_any_type */
	T0* a86; /* typed_pointer_like_current_type */
	T0* a87; /* wide_character_type */
	T1 a88; /* is_read_only */
	T1 a89; /* is_preparsed */
};

/* Struct for type ET_ECF_ADAPTED_DOTNET_ASSEMBLY */
struct S267 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* name_id */
	T0* a2; /* filename_id */
	T0* a3; /* universe */
	T0* a4; /* target */
	T0* a5; /* description */
	T0* a6; /* assembly_culture */
	T0* a7; /* assembly_key */
	T0* a8; /* assembly_name */
	T0* a9; /* assembly_version */
	T0* a10; /* classname_prefix */
	T1 a11; /* is_read_only */
	T0* a12; /* options */
	T0* a13; /* class_options */
	T0* a14; /* class_renamings */
	T0* a15; /* conditions */
	T0* a16; /* notes */
	T0* a17; /* name */
	T0* a18; /* dotnet_assembly */
};

/* Struct for type ET_ECF_ERROR */
struct S268 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* default_template */
	T0* a2; /* parameters */
	T0* a3; /* system_config */
	T0* a4; /* position */
	T0* a5; /* code */
};

/* Struct for type ET_LACE_AST_FACTORY */
struct S269 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] DS_HASH_TABLE [[attached] ET_LACE_CLUSTER, [attached] ET_IDENTIFIER] */
struct S270 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* found_position */
	T0* a2; /* item_storage */
	T0* a3; /* key_equality_tester */
	T6 a4; /* count */
	T6 a5; /* last_position */
	T6 a6; /* free_slot */
	T6 a7; /* position */
	T6 a8; /* capacity */
	T6 a9; /* slots_position */
	T0* a10; /* equality_tester */
	T0* a11; /* internal_cursor */
	T0* a12; /* key_storage */
	T0* a13; /* clashes */
	T6 a14; /* modulus */
	T0* a15; /* slots */
	T6 a16; /* clashes_previous_position */
	T0* a17; /* special_item_routines */
	T0* a18; /* special_key_routines */
	T0* a19; /* hash_function */
};

/* Struct for type YY_FILE_BUFFER */
struct S271 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* content */
	T6 a2; /* index */
	T6 a3; /* line */
	T6 a4; /* column */
	T6 a5; /* position */
	T1 a6; /* beginning_of_line */
	T6 a7; /* count */
	T1 a8; /* filled */
	T0* a9; /* file */
	T1 a10; /* end_of_file */
	T6 a11; /* capacity */
	T1 a12; /* interactive */
};

/* Struct for type YY_BUFFER */
struct S272 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* content */
	T6 a2; /* index */
	T6 a3; /* line */
	T6 a4; /* column */
	T6 a5; /* position */
	T1 a6; /* beginning_of_line */
	T6 a7; /* count */
	T1 a8; /* filled */
	T6 a9; /* capacity */
};

/* Struct for type [attached] DS_HASH_SET [[attached] ET_IDENTIFIER] */
struct S273 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* internal_cursor */
	T6 a2; /* position */
	T0* a3; /* item_storage */
	T6 a4; /* capacity */
	T6 a5; /* modulus */
	T6 a6; /* last_position */
	T6 a7; /* free_slot */
	T6 a8; /* count */
	T0* a9; /* equality_tester */
	T6 a10; /* slots_position */
	T6 a11; /* clashes_previous_position */
	T0* a12; /* special_item_routines */
	T0* a13; /* clashes */
	T0* a14; /* slots */
	T6 a15; /* found_position */
	T0* a16; /* hash_function */
};

/* Struct for type [attached] SPECIAL [detachable ANY] */
struct S279 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ANY] */
struct S280 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [[attached] ET_IDENTIFIER] */
struct S281 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_IDENTIFIER] */
struct S282 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type ET_LACE_SYSTEM */
struct S283 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* root_type */
	T0* a2; /* none_type */
	T0* a3; /* any_type */
	T0* a4; /* master_classes */
	T0* a5; /* root_creation */
	T6 a6; /* default_create_seed */
	T0* a7; /* boolean_type */
	T0* a8; /* character_8_type */
	T0* a9; /* character_32_type */
	T0* a10; /* integer_8_type */
	T0* a11; /* integer_16_type */
	T0* a12; /* integer_32_type */
	T0* a13; /* integer_64_type */
	T0* a14; /* natural_8_type */
	T0* a15; /* natural_16_type */
	T0* a16; /* natural_32_type */
	T0* a17; /* natural_64_type */
	T0* a18; /* real_32_type */
	T0* a19; /* real_64_type */
	T0* a20; /* pointer_type */
	T0* a21; /* special_any_type */
	T0* a22; /* string_8_type */
	T0* a23; /* string_32_type */
	T0* a24; /* immutable_string_32_type */
	T0* a25; /* array_any_type */
	T0* a26; /* typed_pointer_any_type */
	T0* a27; /* routine_type */
	T0* a28; /* detachable_tuple_type */
	T0* a29; /* universe */
	T0* a30; /* ise_exception_manager_type */
	T0* a31; /* detachable_none_type */
	T1 a32; /* attachment_type_conformance_mode */
	T1 a33; /* use_boehm_gc */
	T0* a34; /* external_include_pathnames */
	T0* a35; /* external_cflags */
	T0* a36; /* external_linker_flags */
	T0* a37; /* external_library_pathnames */
	T0* a38; /* external_object_pathnames */
	T0* a39; /* character_type */
	T0* a40; /* wide_character_type */
	T0* a41; /* integer_type */
	T0* a42; /* natural_type */
	T0* a43; /* real_type */
	T0* a44; /* double_type */
	T0* a45; /* external_make_pathnames */
	T0* a46; /* external_resource_pathnames */
	T6 a47; /* routine_call_seed */
	T6 a48; /* function_item_seed */
	T6 a49; /* is_equal_seed */
	T6 a50; /* dispose_seed */
	T1 a51; /* trace_mode */
	T1 a52; /* exception_trace_mode */
	T1 a53; /* multithreaded_mode */
	T1 a54; /* console_application_mode */
	T0* a55; /* system_name */
	T0* a56; /* register_class_mutex */
	T1 a57; /* alias_transition_mode */
	T0* a58; /* current_system */
	T0* a59; /* tuple_type */
	T0* a60; /* detachable_exception_type */
	T0* a61; /* exception_type */
	T0* a62; /* string_type */
	T0* a63; /* name */
	T0* a64; /* clusters */
	T0* a65; /* dotnet_assemblies */
	T0* a66; /* libraries */
	T0* a67; /* master_class_mutex */
	T0* a68; /* detachable_any_type */
	T0* a69; /* detachable_separate_any_type */
	T0* a70; /* separate_any_type */
	T0* a71; /* any_parent */
	T0* a72; /* any_parents */
	T0* a73; /* any_clients */
	T0* a74; /* tuple_like_current_type */
	T0* a75; /* unfolded_empty_tuple_actual_parameters */
	T0* a76; /* array_detachable_any_type */
	T0* a77; /* array_none_type */
	T0* a78; /* array_like_current_type */
	T0* a79; /* character_8_convert_feature */
	T0* a80; /* character_32_convert_feature */
	T0* a81; /* exception_manager_type */
	T0* a82; /* function_type */
	T0* a83; /* integer_8_convert_feature */
	T0* a84; /* integer_16_convert_feature */
	T0* a85; /* integer_32_convert_feature */
	T0* a86; /* integer_64_convert_feature */
	T0* a87; /* iterable_detachable_any_type */
	T0* a88; /* natural_8_convert_feature */
	T0* a89; /* natural_16_convert_feature */
	T0* a90; /* natural_32_convert_feature */
	T0* a91; /* natural_64_convert_feature */
	T0* a92; /* predicate_like_current_type */
	T0* a93; /* procedure_like_current_type */
	T0* a94; /* real_32_convert_feature */
	T0* a95; /* real_64_convert_feature */
	T0* a96; /* special_detachable_any_type */
	T0* a97; /* detachable_string_type */
	T0* a98; /* string_8_convert_feature */
	T0* a99; /* string_32_convert_feature */
	T0* a100; /* system_object_type */
	T0* a101; /* system_object_parents */
	T0* a102; /* system_string_type */
	T0* a103; /* type_any_type */
	T0* a104; /* type_type_any_type */
	T0* a105; /* type_detachable_any_type */
	T0* a106; /* type_detachable_like_current_type */
	T0* a107; /* detachable_type_exception_type */
	T0* a108; /* type_detachable_exception_type */
	T0* a109; /* detachable_type_detachable_any_type */
	T0* a110; /* type_like_current_type */
	T0* a111; /* typed_pointer_like_current_type */
	T1 a112; /* is_read_only */
	T1 a113; /* is_preparsed */
	T0* a114; /* scm_read_mapping_builder */
	T0* a115; /* scm_write_mapping_builder */
	T6 a116; /* registered_class_count */
	T1 a117; /* scoop_mode */
	T0* a118; /* implicit_attachment_type_mark */
	T6 a119; /* copy_seed */
	T6 a120; /* iterable_new_cursor_seed */
	T6 a121; /* iteration_cursor_after_seed */
	T6 a122; /* iteration_cursor_forth_seed */
	T1 a123; /* is_dotnet */
	T1 a124; /* target_type_attachment_mode */
	T1 a125; /* obsolete_routine_type_mode */
};

/* Struct for type ET_LACE_CLUSTERS */
struct S285 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* clusters */
};

/* Struct for type ET_LACE_CLUSTER */
struct S286 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* name_id */
	T0* a2; /* pathname_id */
	T1 a3; /* is_relative */
	T0* a4; /* universe */
	T1 a5; /* is_abstract */
	T1 a6; /* is_recursive */
	T1 a7; /* is_read_only */
	T0* a8; /* exclude */
	T0* a9; /* subclusters */
	T1 a10; /* overridden_constraint_enabled */
	T1 a11; /* scm_mapping_constraint_enabled */
	T0* a12; /* parent */
	T1 a13; /* is_override */
	T0* a14; /* provider_constraint */
	T0* a15; /* dependant_constraint */
	T0* a16; /* scm_read_mapping */
	T0* a17; /* scm_write_mapping */
	T1 a18; /* is_preparsed */
	T1 a19; /* is_implicit */
	T1 a20; /* use_obsolete_syntax */
	T0* a21; /* cached_absolute_pathname */
};

/* Struct for type ET_LACE_EXCLUDE */
struct S288 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* identifiers */
};

/* Struct for type ET_LACE_DOTNET_ASSEMBLY */
struct S290 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* name_id */
	T0* a2; /* pathname_id */
	T0* a3; /* classname_prefix_id */
	T0* a4; /* pathname */
	T0* a5; /* referenced_assemblies */
	T0* a6; /* classname_prefix */
	T0* a7; /* current_system */
	T0* a8; /* dotnet_assembly */
	T0* a9; /* name */
	T0* a10; /* master_classes */
	T0* a11; /* master_class_mutex */
	T0* a12; /* any_type */
	T0* a13; /* detachable_any_type */
	T0* a14; /* detachable_separate_any_type */
	T0* a15; /* separate_any_type */
	T0* a16; /* any_parent */
	T0* a17; /* any_parents */
	T0* a18; /* any_clients */
	T0* a19; /* tuple_type */
	T0* a20; /* detachable_tuple_type */
	T0* a21; /* tuple_like_current_type */
	T0* a22; /* unfolded_empty_tuple_actual_parameters */
	T0* a23; /* array_any_type */
	T0* a24; /* array_detachable_any_type */
	T0* a25; /* array_none_type */
	T0* a26; /* array_like_current_type */
	T0* a27; /* boolean_type */
	T0* a28; /* character_type */
	T0* a29; /* character_8_type */
	T0* a30; /* character_8_convert_feature */
	T0* a31; /* character_32_type */
	T0* a32; /* character_32_convert_feature */
	T0* a33; /* double_type */
	T0* a34; /* exception_type */
	T0* a35; /* detachable_exception_type */
	T0* a36; /* exception_manager_type */
	T0* a37; /* function_type */
	T0* a38; /* immutable_string_32_type */
	T0* a39; /* integer_type */
	T0* a40; /* integer_8_type */
	T0* a41; /* integer_8_convert_feature */
	T0* a42; /* integer_16_type */
	T0* a43; /* integer_16_convert_feature */
	T0* a44; /* integer_32_type */
	T0* a45; /* integer_32_convert_feature */
	T0* a46; /* integer_64_type */
	T0* a47; /* integer_64_convert_feature */
	T0* a48; /* ise_exception_manager_type */
	T0* a49; /* iterable_detachable_any_type */
	T0* a50; /* natural_type */
	T0* a51; /* natural_8_type */
	T0* a52; /* natural_8_convert_feature */
	T0* a53; /* natural_16_type */
	T0* a54; /* natural_16_convert_feature */
	T0* a55; /* natural_32_type */
	T0* a56; /* natural_32_convert_feature */
	T0* a57; /* natural_64_type */
	T0* a58; /* natural_64_convert_feature */
	T0* a59; /* none_type */
	T0* a60; /* detachable_none_type */
	T0* a61; /* pointer_type */
	T0* a62; /* predicate_like_current_type */
	T0* a63; /* procedure_like_current_type */
	T0* a64; /* real_type */
	T0* a65; /* real_32_type */
	T0* a66; /* real_32_convert_feature */
	T0* a67; /* real_64_type */
	T0* a68; /* real_64_convert_feature */
	T0* a69; /* routine_type */
	T0* a70; /* special_any_type */
	T0* a71; /* special_detachable_any_type */
	T0* a72; /* string_type */
	T0* a73; /* detachable_string_type */
	T0* a74; /* string_8_type */
	T0* a75; /* string_8_convert_feature */
	T0* a76; /* string_32_type */
	T0* a77; /* string_32_convert_feature */
	T0* a78; /* system_object_type */
	T0* a79; /* system_object_parents */
	T0* a80; /* system_string_type */
	T0* a81; /* type_any_type */
	T0* a82; /* type_type_any_type */
	T0* a83; /* type_detachable_any_type */
	T0* a84; /* type_detachable_like_current_type */
	T0* a85; /* detachable_type_exception_type */
	T0* a86; /* type_detachable_exception_type */
	T0* a87; /* detachable_type_detachable_any_type */
	T0* a88; /* type_like_current_type */
	T0* a89; /* typed_pointer_any_type */
	T0* a90; /* typed_pointer_like_current_type */
	T0* a91; /* wide_character_type */
	T1 a92; /* is_read_only */
	T1 a93; /* is_preparsed */
	T0* a94; /* class_renamings */
};

/* Struct for type [attached] SPECIAL [[attached] ET_LACE_SYSTEM] */
struct S291 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] SPECIAL [detachable ET_LACE_CLUSTERS] */
struct S292 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] SPECIAL [detachable ET_ADAPTED_DOTNET_ASSEMBLIES] */
struct S293 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] SPECIAL [detachable ET_IDENTIFIER] */
struct S294 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_LACE_SYSTEM] */
struct S295 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_IDENTIFIER] */
struct S296 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_LACE_CLUSTERS] */
struct S297 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [[attached] ET_LACE_CLUSTERS] */
struct S298 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] SPECIAL [[attached] ET_LACE_CLUSTER] */
struct S299 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_LACE_CLUSTERS] */
struct S300 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_LACE_CLUSTER] */
struct S301 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_LACE_EXCLUDE] */
struct S302 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_LACE_EXCLUDE] */
struct S303 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [[attached] ET_LACE_EXCLUDE] */
struct S304 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_LACE_EXCLUDE] */
struct S305 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_ADAPTED_DOTNET_ASSEMBLIES] */
struct S306 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [[attached] ET_ADAPTED_DOTNET_ASSEMBLIES] */
struct S307 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] SPECIAL [[attached] ET_LACE_DOTNET_ASSEMBLY] */
struct S308 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_ADAPTED_DOTNET_ASSEMBLIES] */
struct S309 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_LACE_DOTNET_ASSEMBLY] */
struct S310 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type ET_LACE_DOTNET_GAC_ASSEMBLY */
struct S311 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* name_id */
	T0* a2; /* assembly_name_id */
	T0* a3; /* classname_prefix_id */
	T0* a4; /* assembly_version_id */
	T0* a5; /* assembly_culture_id */
	T0* a6; /* assembly_public_key_token_id */
	T0* a7; /* assembly_name */
	T0* a8; /* classname_prefix */
	T0* a9; /* assembly_version */
	T0* a10; /* assembly_culture */
	T0* a11; /* assembly_public_key_token */
	T0* a12; /* pathname */
	T0* a13; /* referenced_assemblies */
	T0* a14; /* current_system */
	T0* a15; /* dotnet_assembly */
	T0* a16; /* name */
	T0* a17; /* master_classes */
	T0* a18; /* master_class_mutex */
	T0* a19; /* any_type */
	T0* a20; /* detachable_any_type */
	T0* a21; /* detachable_separate_any_type */
	T0* a22; /* separate_any_type */
	T0* a23; /* any_parent */
	T0* a24; /* any_parents */
	T0* a25; /* any_clients */
	T0* a26; /* tuple_type */
	T0* a27; /* detachable_tuple_type */
	T0* a28; /* tuple_like_current_type */
	T0* a29; /* unfolded_empty_tuple_actual_parameters */
	T0* a30; /* array_any_type */
	T0* a31; /* array_detachable_any_type */
	T0* a32; /* array_none_type */
	T0* a33; /* array_like_current_type */
	T0* a34; /* boolean_type */
	T0* a35; /* character_type */
	T0* a36; /* character_8_type */
	T0* a37; /* character_8_convert_feature */
	T0* a38; /* character_32_type */
	T0* a39; /* character_32_convert_feature */
	T0* a40; /* double_type */
	T0* a41; /* exception_type */
	T0* a42; /* detachable_exception_type */
	T0* a43; /* exception_manager_type */
	T0* a44; /* function_type */
	T0* a45; /* immutable_string_32_type */
	T0* a46; /* integer_type */
	T0* a47; /* integer_8_type */
	T0* a48; /* integer_8_convert_feature */
	T0* a49; /* integer_16_type */
	T0* a50; /* integer_16_convert_feature */
	T0* a51; /* integer_32_type */
	T0* a52; /* integer_32_convert_feature */
	T0* a53; /* integer_64_type */
	T0* a54; /* integer_64_convert_feature */
	T0* a55; /* ise_exception_manager_type */
	T0* a56; /* iterable_detachable_any_type */
	T0* a57; /* natural_type */
	T0* a58; /* natural_8_type */
	T0* a59; /* natural_8_convert_feature */
	T0* a60; /* natural_16_type */
	T0* a61; /* natural_16_convert_feature */
	T0* a62; /* natural_32_type */
	T0* a63; /* natural_32_convert_feature */
	T0* a64; /* natural_64_type */
	T0* a65; /* natural_64_convert_feature */
	T0* a66; /* none_type */
	T0* a67; /* detachable_none_type */
	T0* a68; /* pointer_type */
	T0* a69; /* predicate_like_current_type */
	T0* a70; /* procedure_like_current_type */
	T0* a71; /* real_type */
	T0* a72; /* real_32_type */
	T0* a73; /* real_32_convert_feature */
	T0* a74; /* real_64_type */
	T0* a75; /* real_64_convert_feature */
	T0* a76; /* routine_type */
	T0* a77; /* special_any_type */
	T0* a78; /* special_detachable_any_type */
	T0* a79; /* string_type */
	T0* a80; /* detachable_string_type */
	T0* a81; /* string_8_type */
	T0* a82; /* string_8_convert_feature */
	T0* a83; /* string_32_type */
	T0* a84; /* string_32_convert_feature */
	T0* a85; /* system_object_type */
	T0* a86; /* system_object_parents */
	T0* a87; /* system_string_type */
	T0* a88; /* type_any_type */
	T0* a89; /* type_type_any_type */
	T0* a90; /* type_detachable_any_type */
	T0* a91; /* type_detachable_like_current_type */
	T0* a92; /* detachable_type_exception_type */
	T0* a93; /* type_detachable_exception_type */
	T0* a94; /* detachable_type_detachable_any_type */
	T0* a95; /* type_like_current_type */
	T0* a96; /* typed_pointer_any_type */
	T0* a97; /* typed_pointer_like_current_type */
	T0* a98; /* wide_character_type */
	T1 a99; /* is_read_only */
	T1 a100; /* is_preparsed */
	T0* a101; /* class_renamings */
};

/* Struct for type KL_PLATFORM */
struct S312 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type KL_INTEGER_ROUTINES */
struct S313 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] DS_HASH_TABLE [[attached] ET_MASTER_CLASS, [attached] ET_CLASS_NAME] */
struct S314 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* capacity */
	T0* a2; /* key_equality_tester */
	T0* a3; /* internal_keys */
	T6 a4; /* last_position */
	T0* a5; /* equality_tester */
	T6 a6; /* modulus */
	T6 a7; /* free_slot */
	T6 a8; /* position */
	T0* a9; /* special_item_routines */
	T0* a10; /* item_storage */
	T0* a11; /* special_key_routines */
	T0* a12; /* key_storage */
	T0* a13; /* clashes */
	T0* a14; /* slots */
	T6 a15; /* found_position */
	T0* a16; /* internal_cursor */
	T6 a17; /* count */
	T6 a18; /* slots_position */
	T6 a19; /* clashes_previous_position */
	T0* a20; /* hash_function */
};

/* Struct for type [attached] DS_ARRAYED_LIST [[attached] ET_DYNAMIC_TYPE] */
struct S315 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
	T0* a3; /* special_routines */
	T6 a4; /* capacity */
	T0* a5; /* internal_cursor */
};

/* Struct for type [attached] TUPLE [[attached] ET_CLASS] */
struct S316 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type [attached] PROCEDURE [[attached] TUPLE [[attached] ET_CLASS]] */
struct S317 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T14 a1; /* rout_disp */
	T0* a2; /* closed_operands */
	T14 a3; /* encaps_rout_disp */
	T14 a4; /* calc_rout_addr */
	T1 a5; /* is_target_closed */
	T6 a6; /* open_count */
};

/* Struct for type [attached] TUPLE [INTEGER_32] */
struct S318 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 z1;
};

/* Struct for type ET_DYNAMIC_NULL_TYPE_SET_BUILDER */
struct S319 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* has_fatal_error */
	T0* a2; /* current_dynamic_system */
	T1 a3; /* catcall_error_mode */
	T1 a4; /* catcall_warning_mode */
	T1 a5; /* no_debug */
	T1 a6; /* no_assertion */
	T0* a7; /* alive_conforming_descendants_per_type */
};

/* Struct for type ET_DYNAMIC_TYPE */
struct S320 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* base_type */
	T6 a2; /* attribute_count */
	T0* a3; /* queries */
	T1 a4; /* is_alive */
	T6 a5; /* id */
	T0* a6; /* base_class */
	T0* a7; /* meta_type */
	T0* a8; /* procedures */
	T6 a9; /* hash_code */
	T0* a10; /* conforming_ancestors */
	T0* a11; /* conforming_descendants */
	T0* a12; /* alive_conforming_descendants */
	T0* a13; /* next_type */
	T1 a14; /* has_reference_attributes */
	T1 a15; /* has_generic_expanded_attributes */
	T0* a16; /* query_calls */
	T1 a17; /* has_static */
	T0* a18; /* procedure_calls */
	T0* a19; /* equality_expressions */
	T0* a20; /* object_equality_expressions */
};

/* Struct for type [attached] TUPLE [ET_DYNAMIC_SYSTEM] */
struct S321 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type ET_DYNAMIC_FEATURE */
struct S332 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* static_feature */
	T0* a2; /* dynamic_type_sets */
	T6 a3; /* id */
	T1 a4; /* is_generated */
	T0* a5; /* result_type_set */
	T0* a6; /* target_type */
	T8 a7; /* builtin_class_code */
	T8 a8; /* builtin_feature_code */
	T1 a9; /* is_regular */
	T0* a10; /* first_precursor */
	T1 a11; /* is_creation */
	T0* a12; /* other_precursors */
	T1 a13; /* is_built */
	T1 a14; /* is_current_type_needed */
	T1 a15; /* is_static_generated */
	T1 a16; /* is_address */
};

/* Struct for type ET_ACTUAL_PARAMETER_LIST */
struct S333 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
	T0* a3; /* left_bracket */
	T0* a4; /* right_bracket */
};

/* Struct for type ET_EXTERNAL_FUNCTION */
struct S344 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* implementation_feature */
	T0* a2; /* implementation_class */
	T0* a3; /* first_precursor */
	T0* a4; /* other_precursors */
	T0* a5; /* preconditions */
	T6 a6; /* hash_code */
	T6 a7; /* first_seed */
	T0* a8; /* arguments */
	T0* a9; /* assigner */
	T0* a10; /* postconditions */
	T8 a11; /* builtin_class_code */
	T8 a12; /* builtin_feature_code */
	T0* a13; /* other_seeds */
	T0* a14; /* frozen_keyword */
	T0* a15; /* extended_name */
	T0* a16; /* declared_type */
	T0* a17; /* language */
	T0* a18; /* obsolete_message */
	T0* a19; /* alias_clause */
	T0* a20; /* clients */
	T0* a21; /* is_keyword */
	T0* a22; /* end_keyword */
	T0* a23; /* semicolon */
	T0* a24; /* feature_clause */
	T0* a25; /* first_indexing */
	T0* a26; /* object_tests */
	T0* a27; /* across_components */
	T1 a28; /* validity_checked */
	T1 a29; /* has_validity_error */
	T6 a30; /* id */
	T6 a31; /* version */
	T0* a32; /* synonym */
};

/* Struct for type ET_TUPLE_TYPE */
struct S345 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* named_base_class */
	T0* a2; /* actual_parameters */
	T0* a3; /* type_mark */
	T0* a4; /* tuple_keyword */
};

/* Struct for type ET_FORMAL_ARGUMENT_LIST */
struct S347 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
	T0* a3; /* left_parenthesis */
	T0* a4; /* right_parenthesis */
};

/* Struct for type [attached] ARRAY [[attached] ET_TYPE] */
struct S348 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* area */
	T6 a2; /* lower */
	T6 a3; /* upper */
};

/* Struct for type [attached] SPECIAL [[attached] ET_TYPE] */
struct S349 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type ET_FORMAL_ARGUMENT */
struct S350 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* name_item */
	T0* a2; /* declared_type */
	T1 a3; /* is_used */
};

/* Struct for type ET_QUERY_LIST */
struct S351 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T6 a2; /* declared_count */
	T0* a3; /* storage */
};

/* Struct for type ET_PROCEDURE_LIST */
struct S352 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T6 a2; /* declared_count */
	T0* a3; /* storage */
};

/* Struct for type DT_SHARED_SYSTEM_CLOCK */
struct S353 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type DT_SYSTEM_CLOCK */
struct S354 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* second */
	T6 a2; /* year */
	T6 a3; /* month */
	T6 a4; /* day */
	T6 a5; /* hour */
	T6 a6; /* minute */
	T6 a7; /* millisecond */
	T0* a8; /* local_clock */
};

/* Struct for type DT_DATE_TIME */
struct S355 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* time_storage */
	T6 a2; /* date_storage */
};

/* Struct for type ET_EIFFEL_PREPARSER */
struct S357 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* eiffel_buffer */
	T0* a2; /* system_processor */
	T6 a3; /* last_text_count */
	T6 a4; /* last_literal_start */
	T0* a5; /* filename */
	T0* a6; /* group */
	T0* a7; /* verbatim_marker */
	T0* a8; /* verbatim_open_white_characters */
	T0* a9; /* verbatim_close_white_characters */
	T0* a10; /* yy_pushed_start_conditions */
	T0* a11; /* input_buffer */
	T0* a12; /* yy_state_stack */
	T6 a13; /* yy_start_state */
	T6 a14; /* yy_line */
	T6 a15; /* yy_column */
	T6 a16; /* yy_position */
	T6 a17; /* yy_start */
	T6 a18; /* yy_end */
	T6 a19; /* line */
	T6 a20; /* column */
	T6 a21; /* position */
	T0* a22; /* yy_nxt */
	T0* a23; /* yy_chk */
	T0* a24; /* yy_base */
	T0* a25; /* yy_def */
	T0* a26; /* yy_ec */
	T0* a27; /* yy_meta */
	T0* a28; /* yy_accept */
	T0* a29; /* yy_acclist */
	T0* a30; /* yy_content */
	T0* a31; /* yy_content_area */
	T0* a32; /* last_classname */
	T1 a33; /* class_keyword_found */
	T6 a34; /* last_token */
	T1 a35; /* yy_more_flag */
	T6 a36; /* yy_more_len */
	T6 a37; /* yy_state_count */
	T6 a38; /* yy_last_accepting_state */
	T6 a39; /* yy_last_accepting_cpos */
	T6 a40; /* yy_lp */
	T6 a41; /* yy_looking_for_trail_begin */
	T6 a42; /* yy_full_match */
	T6 a43; /* yy_full_state */
	T6 a44; /* yy_full_lp */
	T1 a45; /* yy_rejected */
	T6 a46; /* last_literal_end */
	T6 a47; /* pushed_start_condition_count */
	T6 a48; /* ms_column */
	T6 a49; /* ms_line */
	T6 a50; /* last_break_end */
	T6 a51; /* last_comment_end */
};

/* Struct for type ET_MASTER_CLASS_CHECKER */
struct S358 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* current_class */
	T0* a2; /* system_processor */
	T1 a3; /* has_fatal_error */
};

/* Struct for type ET_EIFFEL_PARSER */
struct S359 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* eiffel_buffer */
	T0* a2; /* counters */
	T0* a3; /* last_formal_arguments_stack */
	T0* a4; /* last_local_variables_stack */
	T0* a5; /* last_keywords */
	T0* a6; /* last_symbols */
	T0* a7; /* last_object_tests_stack */
	T0* a8; /* last_object_tests_pool */
	T0* a9; /* last_across_components_stack */
	T0* a10; /* last_across_components_pool */
	T0* a11; /* assertions */
	T0* a12; /* assertion_counters */
	T0* a13; /* assertion_kinds */
	T6 a14; /* assertion_kind */
	T0* a15; /* queries */
	T0* a16; /* procedures */
	T0* a17; /* constraints */
	T0* a18; /* providers */
	T0* a19; /* current_class */
	T0* a20; /* system_processor */
	T6 a21; /* last_text_count */
	T6 a22; /* last_literal_start */
	T0* a23; /* filename */
	T0* a24; /* group */
	T0* a25; /* verbatim_marker */
	T0* a26; /* verbatim_open_white_characters */
	T0* a27; /* verbatim_close_white_characters */
	T0* a28; /* yyss */
	T0* a29; /* yy_pushed_start_conditions */
	T0* a30; /* input_buffer */
	T0* a31; /* yyspecial_routines1 */
	T6 a32; /* yyvsc1 */
	T0* a33; /* yyvs1 */
	T0* a34; /* yyspecial_routines2 */
	T6 a35; /* yyvsc2 */
	T0* a36; /* yyvs2 */
	T0* a37; /* yyspecial_routines3 */
	T6 a38; /* yyvsc3 */
	T0* a39; /* yyvs3 */
	T0* a40; /* yyspecial_routines4 */
	T6 a41; /* yyvsc4 */
	T0* a42; /* yyvs4 */
	T0* a43; /* yyspecial_routines5 */
	T6 a44; /* yyvsc5 */
	T0* a45; /* yyvs5 */
	T0* a46; /* yyspecial_routines6 */
	T6 a47; /* yyvsc6 */
	T0* a48; /* yyvs6 */
	T0* a49; /* yyspecial_routines7 */
	T6 a50; /* yyvsc7 */
	T0* a51; /* yyvs7 */
	T0* a52; /* yyspecial_routines8 */
	T6 a53; /* yyvsc8 */
	T0* a54; /* yyvs8 */
	T0* a55; /* yyspecial_routines9 */
	T6 a56; /* yyvsc9 */
	T0* a57; /* yyvs9 */
	T0* a58; /* yyspecial_routines10 */
	T6 a59; /* yyvsc10 */
	T0* a60; /* yyvs10 */
	T0* a61; /* yyspecial_routines11 */
	T6 a62; /* yyvsc11 */
	T0* a63; /* yyvs11 */
	T0* a64; /* yyspecial_routines12 */
	T6 a65; /* yyvsc12 */
	T0* a66; /* yyvs12 */
	T0* a67; /* yyspecial_routines13 */
	T6 a68; /* yyvsc13 */
	T0* a69; /* yyvs13 */
	T0* a70; /* yyspecial_routines14 */
	T6 a71; /* yyvsc14 */
	T0* a72; /* yyvs14 */
	T0* a73; /* yyspecial_routines15 */
	T6 a74; /* yyvsc15 */
	T0* a75; /* yyvs15 */
	T0* a76; /* yyspecial_routines16 */
	T6 a77; /* yyvsc16 */
	T0* a78; /* yyvs16 */
	T0* a79; /* yyspecial_routines17 */
	T6 a80; /* yyvsc17 */
	T0* a81; /* yyvs17 */
	T0* a82; /* yyspecial_routines18 */
	T6 a83; /* yyvsc18 */
	T0* a84; /* yyvs18 */
	T0* a85; /* yyspecial_routines19 */
	T6 a86; /* yyvsc19 */
	T0* a87; /* yyvs19 */
	T0* a88; /* yyspecial_routines20 */
	T6 a89; /* yyvsc20 */
	T0* a90; /* yyvs20 */
	T0* a91; /* yyspecial_routines21 */
	T6 a92; /* yyvsc21 */
	T0* a93; /* yyvs21 */
	T0* a94; /* yyspecial_routines22 */
	T6 a95; /* yyvsc22 */
	T0* a96; /* yyvs22 */
	T0* a97; /* yyspecial_routines23 */
	T6 a98; /* yyvsc23 */
	T0* a99; /* yyvs23 */
	T0* a100; /* yyspecial_routines24 */
	T6 a101; /* yyvsc24 */
	T0* a102; /* yyvs24 */
	T0* a103; /* yyspecial_routines25 */
	T6 a104; /* yyvsc25 */
	T0* a105; /* yyvs25 */
	T0* a106; /* yyspecial_routines26 */
	T6 a107; /* yyvsc26 */
	T0* a108; /* yyvs26 */
	T0* a109; /* yyspecial_routines27 */
	T6 a110; /* yyvsc27 */
	T0* a111; /* yyvs27 */
	T0* a112; /* yyspecial_routines28 */
	T6 a113; /* yyvsc28 */
	T0* a114; /* yyvs28 */
	T0* a115; /* yyspecial_routines29 */
	T6 a116; /* yyvsc29 */
	T0* a117; /* yyvs29 */
	T0* a118; /* yyspecial_routines30 */
	T6 a119; /* yyvsc30 */
	T0* a120; /* yyvs30 */
	T0* a121; /* yyspecial_routines31 */
	T6 a122; /* yyvsc31 */
	T0* a123; /* yyvs31 */
	T0* a124; /* yyspecial_routines32 */
	T6 a125; /* yyvsc32 */
	T0* a126; /* yyvs32 */
	T0* a127; /* yyspecial_routines33 */
	T6 a128; /* yyvsc33 */
	T0* a129; /* yyvs33 */
	T0* a130; /* yyspecial_routines34 */
	T6 a131; /* yyvsc34 */
	T0* a132; /* yyvs34 */
	T0* a133; /* yyspecial_routines35 */
	T6 a134; /* yyvsc35 */
	T0* a135; /* yyvs35 */
	T0* a136; /* yyspecial_routines36 */
	T6 a137; /* yyvsc36 */
	T0* a138; /* yyvs36 */
	T0* a139; /* yyspecial_routines37 */
	T6 a140; /* yyvsc37 */
	T0* a141; /* yyvs37 */
	T0* a142; /* yyspecial_routines38 */
	T6 a143; /* yyvsc38 */
	T0* a144; /* yyvs38 */
	T0* a145; /* yyspecial_routines39 */
	T6 a146; /* yyvsc39 */
	T0* a147; /* yyvs39 */
	T0* a148; /* yyspecial_routines40 */
	T6 a149; /* yyvsc40 */
	T0* a150; /* yyvs40 */
	T0* a151; /* yyspecial_routines41 */
	T6 a152; /* yyvsc41 */
	T0* a153; /* yyvs41 */
	T0* a154; /* yyspecial_routines42 */
	T6 a155; /* yyvsc42 */
	T0* a156; /* yyvs42 */
	T0* a157; /* yyspecial_routines43 */
	T6 a158; /* yyvsc43 */
	T0* a159; /* yyvs43 */
	T0* a160; /* yyspecial_routines44 */
	T6 a161; /* yyvsc44 */
	T0* a162; /* yyvs44 */
	T0* a163; /* yyspecial_routines45 */
	T6 a164; /* yyvsc45 */
	T0* a165; /* yyvs45 */
	T0* a166; /* yyspecial_routines46 */
	T6 a167; /* yyvsc46 */
	T0* a168; /* yyvs46 */
	T0* a169; /* yyspecial_routines47 */
	T6 a170; /* yyvsc47 */
	T0* a171; /* yyvs47 */
	T0* a172; /* yyspecial_routines48 */
	T6 a173; /* yyvsc48 */
	T0* a174; /* yyvs48 */
	T0* a175; /* yyspecial_routines49 */
	T6 a176; /* yyvsc49 */
	T0* a177; /* yyvs49 */
	T0* a178; /* yyspecial_routines50 */
	T6 a179; /* yyvsc50 */
	T0* a180; /* yyvs50 */
	T0* a181; /* yyspecial_routines51 */
	T6 a182; /* yyvsc51 */
	T0* a183; /* yyvs51 */
	T0* a184; /* yyspecial_routines52 */
	T6 a185; /* yyvsc52 */
	T0* a186; /* yyvs52 */
	T0* a187; /* yyspecial_routines53 */
	T6 a188; /* yyvsc53 */
	T0* a189; /* yyvs53 */
	T0* a190; /* yyspecial_routines54 */
	T6 a191; /* yyvsc54 */
	T0* a192; /* yyvs54 */
	T0* a193; /* yyspecial_routines55 */
	T6 a194; /* yyvsc55 */
	T0* a195; /* yyvs55 */
	T0* a196; /* yyspecial_routines56 */
	T6 a197; /* yyvsc56 */
	T0* a198; /* yyvs56 */
	T0* a199; /* yyspecial_routines57 */
	T6 a200; /* yyvsc57 */
	T0* a201; /* yyvs57 */
	T0* a202; /* yyspecial_routines58 */
	T6 a203; /* yyvsc58 */
	T0* a204; /* yyvs58 */
	T0* a205; /* yyspecial_routines59 */
	T6 a206; /* yyvsc59 */
	T0* a207; /* yyvs59 */
	T0* a208; /* yyspecial_routines60 */
	T6 a209; /* yyvsc60 */
	T0* a210; /* yyvs60 */
	T0* a211; /* yyspecial_routines61 */
	T6 a212; /* yyvsc61 */
	T0* a213; /* yyvs61 */
	T0* a214; /* yyspecial_routines62 */
	T6 a215; /* yyvsc62 */
	T0* a216; /* yyvs62 */
	T0* a217; /* yyspecial_routines63 */
	T6 a218; /* yyvsc63 */
	T0* a219; /* yyvs63 */
	T0* a220; /* yyspecial_routines64 */
	T6 a221; /* yyvsc64 */
	T0* a222; /* yyvs64 */
	T0* a223; /* yyspecial_routines65 */
	T6 a224; /* yyvsc65 */
	T0* a225; /* yyvs65 */
	T0* a226; /* yyspecial_routines66 */
	T6 a227; /* yyvsc66 */
	T0* a228; /* yyvs66 */
	T0* a229; /* yyspecial_routines67 */
	T6 a230; /* yyvsc67 */
	T0* a231; /* yyvs67 */
	T0* a232; /* yyspecial_routines68 */
	T6 a233; /* yyvsc68 */
	T0* a234; /* yyvs68 */
	T0* a235; /* yyspecial_routines69 */
	T6 a236; /* yyvsc69 */
	T0* a237; /* yyvs69 */
	T0* a238; /* yyspecial_routines70 */
	T6 a239; /* yyvsc70 */
	T0* a240; /* yyvs70 */
	T0* a241; /* yyspecial_routines71 */
	T6 a242; /* yyvsc71 */
	T0* a243; /* yyvs71 */
	T0* a244; /* yyspecial_routines72 */
	T6 a245; /* yyvsc72 */
	T0* a246; /* yyvs72 */
	T0* a247; /* yyspecial_routines73 */
	T6 a248; /* yyvsc73 */
	T0* a249; /* yyvs73 */
	T0* a250; /* yyspecial_routines74 */
	T6 a251; /* yyvsc74 */
	T0* a252; /* yyvs74 */
	T0* a253; /* yyspecial_routines75 */
	T6 a254; /* yyvsc75 */
	T0* a255; /* yyvs75 */
	T0* a256; /* yyspecial_routines76 */
	T6 a257; /* yyvsc76 */
	T0* a258; /* yyvs76 */
	T0* a259; /* yyspecial_routines77 */
	T6 a260; /* yyvsc77 */
	T0* a261; /* yyvs77 */
	T0* a262; /* yyspecial_routines78 */
	T6 a263; /* yyvsc78 */
	T0* a264; /* yyvs78 */
	T0* a265; /* yyspecial_routines79 */
	T6 a266; /* yyvsc79 */
	T0* a267; /* yyvs79 */
	T0* a268; /* yyspecial_routines80 */
	T6 a269; /* yyvsc80 */
	T0* a270; /* yyvs80 */
	T0* a271; /* yyspecial_routines81 */
	T6 a272; /* yyvsc81 */
	T0* a273; /* yyvs81 */
	T0* a274; /* yyspecial_routines82 */
	T6 a275; /* yyvsc82 */
	T0* a276; /* yyvs82 */
	T0* a277; /* yyspecial_routines83 */
	T6 a278; /* yyvsc83 */
	T0* a279; /* yyvs83 */
	T0* a280; /* yyspecial_routines84 */
	T6 a281; /* yyvsc84 */
	T0* a282; /* yyvs84 */
	T0* a283; /* yyspecial_routines85 */
	T6 a284; /* yyvsc85 */
	T0* a285; /* yyvs85 */
	T0* a286; /* yyspecial_routines86 */
	T6 a287; /* yyvsc86 */
	T0* a288; /* yyvs86 */
	T0* a289; /* yyspecial_routines87 */
	T6 a290; /* yyvsc87 */
	T0* a291; /* yyvs87 */
	T0* a292; /* yyspecial_routines88 */
	T6 a293; /* yyvsc88 */
	T0* a294; /* yyvs88 */
	T0* a295; /* yyspecial_routines89 */
	T6 a296; /* yyvsc89 */
	T0* a297; /* yyvs89 */
	T0* a298; /* yyspecial_routines90 */
	T6 a299; /* yyvsc90 */
	T0* a300; /* yyvs90 */
	T0* a301; /* yyspecial_routines91 */
	T6 a302; /* yyvsc91 */
	T0* a303; /* yyvs91 */
	T0* a304; /* yyspecial_routines92 */
	T6 a305; /* yyvsc92 */
	T0* a306; /* yyvs92 */
	T0* a307; /* yyspecial_routines93 */
	T6 a308; /* yyvsc93 */
	T0* a309; /* yyvs93 */
	T0* a310; /* yyspecial_routines94 */
	T6 a311; /* yyvsc94 */
	T0* a312; /* yyvs94 */
	T0* a313; /* yyspecial_routines95 */
	T6 a314; /* yyvsc95 */
	T0* a315; /* yyvs95 */
	T0* a316; /* yyspecial_routines96 */
	T6 a317; /* yyvsc96 */
	T0* a318; /* yyvs96 */
	T0* a319; /* yyspecial_routines97 */
	T6 a320; /* yyvsc97 */
	T0* a321; /* yyvs97 */
	T0* a322; /* yyspecial_routines98 */
	T6 a323; /* yyvsc98 */
	T0* a324; /* yyvs98 */
	T0* a325; /* yyspecial_routines99 */
	T6 a326; /* yyvsc99 */
	T0* a327; /* yyvs99 */
	T0* a328; /* yyspecial_routines100 */
	T6 a329; /* yyvsc100 */
	T0* a330; /* yyvs100 */
	T0* a331; /* yyspecial_routines101 */
	T6 a332; /* yyvsc101 */
	T0* a333; /* yyvs101 */
	T0* a334; /* yyspecial_routines102 */
	T6 a335; /* yyvsc102 */
	T0* a336; /* yyvs102 */
	T0* a337; /* yyspecial_routines103 */
	T6 a338; /* yyvsc103 */
	T0* a339; /* yyvs103 */
	T0* a340; /* yyspecial_routines104 */
	T6 a341; /* yyvsc104 */
	T0* a342; /* yyvs104 */
	T0* a343; /* yyspecial_routines105 */
	T6 a344; /* yyvsc105 */
	T0* a345; /* yyvs105 */
	T0* a346; /* yyspecial_routines106 */
	T6 a347; /* yyvsc106 */
	T0* a348; /* yyvs106 */
	T0* a349; /* yyspecial_routines107 */
	T6 a350; /* yyvsc107 */
	T0* a351; /* yyvs107 */
	T0* a352; /* yyspecial_routines108 */
	T6 a353; /* yyvsc108 */
	T0* a354; /* yyvs108 */
	T0* a355; /* yyspecial_routines109 */
	T6 a356; /* yyvsc109 */
	T0* a357; /* yyvs109 */
	T0* a358; /* yyspecial_routines110 */
	T6 a359; /* yyvsc110 */
	T0* a360; /* yyvs110 */
	T0* a361; /* yyspecial_routines111 */
	T6 a362; /* yyvsc111 */
	T0* a363; /* yyvs111 */
	T0* a364; /* yyspecial_routines112 */
	T6 a365; /* yyvsc112 */
	T0* a366; /* yyvs112 */
	T0* a367; /* yyspecial_routines113 */
	T6 a368; /* yyvsc113 */
	T0* a369; /* yyvs113 */
	T0* a370; /* yyspecial_routines114 */
	T6 a371; /* yyvsc114 */
	T0* a372; /* yyvs114 */
	T0* a373; /* yyspecial_routines115 */
	T6 a374; /* yyvsc115 */
	T0* a375; /* yyvs115 */
	T0* a376; /* yyspecial_routines116 */
	T6 a377; /* yyvsc116 */
	T0* a378; /* yyvs116 */
	T0* a379; /* yyspecial_routines117 */
	T6 a380; /* yyvsc117 */
	T0* a381; /* yyvs117 */
	T0* a382; /* yyspecial_routines118 */
	T6 a383; /* yyvsc118 */
	T0* a384; /* yyvs118 */
	T0* a385; /* yyspecial_routines119 */
	T6 a386; /* yyvsc119 */
	T0* a387; /* yyvs119 */
	T0* a388; /* yyspecial_routines120 */
	T6 a389; /* yyvsc120 */
	T0* a390; /* yyvs120 */
	T0* a391; /* yyspecial_routines121 */
	T6 a392; /* yyvsc121 */
	T0* a393; /* yyvs121 */
	T0* a394; /* yyspecial_routines122 */
	T6 a395; /* yyvsc122 */
	T0* a396; /* yyvs122 */
	T0* a397; /* yyspecial_routines123 */
	T6 a398; /* yyvsc123 */
	T0* a399; /* yyvs123 */
	T0* a400; /* yyspecial_routines124 */
	T6 a401; /* yyvsc124 */
	T0* a402; /* yyvs124 */
	T0* a403; /* yytranslate */
	T0* a404; /* yyr1 */
	T0* a405; /* yytypes1 */
	T0* a406; /* yytypes2 */
	T0* a407; /* yydefact */
	T0* a408; /* yydefgoto */
	T0* a409; /* yypact */
	T0* a410; /* yypgoto */
	T0* a411; /* yytable */
	T0* a412; /* yycheck */
	T0* a413; /* yy_state_stack */
	T6 a414; /* yy_start_state */
	T6 a415; /* yy_line */
	T6 a416; /* yy_column */
	T6 a417; /* yy_position */
	T6 a418; /* yy_start */
	T6 a419; /* yy_end */
	T6 a420; /* line */
	T6 a421; /* column */
	T6 a422; /* position */
	T0* a423; /* yy_nxt */
	T0* a424; /* yy_chk */
	T0* a425; /* yy_base */
	T0* a426; /* yy_def */
	T0* a427; /* yy_ec */
	T0* a428; /* yy_meta */
	T0* a429; /* yy_accept */
	T0* a430; /* yy_acclist */
	T0* a431; /* yy_content */
	T0* a432; /* yy_content_area */
	T6 a433; /* time_stamp */
	T0* a434; /* last_class */
	T6 a435; /* yy_parsing_status */
	T6 a436; /* yy_suspended_yystacksize */
	T6 a437; /* yy_suspended_yystate */
	T6 a438; /* yy_suspended_yyn */
	T6 a439; /* yy_suspended_yychar1 */
	T6 a440; /* yy_suspended_index */
	T6 a441; /* yy_suspended_yyss_top */
	T6 a442; /* yy_suspended_yy_goto */
	T6 a443; /* yyssp */
	T6 a444; /* error_count */
	T1 a445; /* yy_lookahead_needed */
	T6 a446; /* yyerrstatus */
	T6 a447; /* last_token */
	T0* a448; /* last_clients */
	T0* a449; /* last_export_clients */
	T0* a450; /* last_feature_clause */
	T6 a451; /* last_literal_end */
	T1 a452; /* yy_rejected */
	T6 a453; /* yy_state_count */
	T6 a454; /* yy_full_match */
	T6 a455; /* yy_lp */
	T6 a456; /* yy_looking_for_trail_begin */
	T6 a457; /* yy_full_lp */
	T6 a458; /* yy_full_state */
	T6 a459; /* pushed_start_condition_count */
	T1 a460; /* yy_more_flag */
	T6 a461; /* yy_more_len */
	T6 a462; /* yy_last_accepting_state */
	T6 a463; /* yy_last_accepting_cpos */
	T6 a464; /* yyvsp1 */
	T6 a465; /* yyvsp2 */
	T6 a466; /* yyvsp3 */
	T6 a467; /* yyvsp4 */
	T6 a468; /* yyvsp5 */
	T6 a469; /* yyvsp6 */
	T6 a470; /* yyvsp7 */
	T6 a471; /* yyvsp8 */
	T6 a472; /* yyvsp9 */
	T6 a473; /* yyvsp10 */
	T6 a474; /* yyvsp11 */
	T6 a475; /* yyvsp12 */
	T6 a476; /* yyvsp13 */
	T6 a477; /* yyvsp14 */
	T6 a478; /* yyvsp15 */
	T6 a479; /* yyvsp16 */
	T6 a480; /* yyvsp17 */
	T6 a481; /* yyvsp18 */
	T6 a482; /* yyvsp19 */
	T6 a483; /* yyvsp20 */
	T6 a484; /* yyvsp21 */
	T6 a485; /* yyvsp22 */
	T6 a486; /* yyvsp23 */
	T6 a487; /* yyvsp24 */
	T6 a488; /* yyvsp25 */
	T6 a489; /* yyvsp26 */
	T6 a490; /* yyvsp27 */
	T6 a491; /* yyvsp28 */
	T6 a492; /* yyvsp29 */
	T6 a493; /* yyvsp30 */
	T6 a494; /* yyvsp31 */
	T6 a495; /* yyvsp32 */
	T6 a496; /* yyvsp33 */
	T6 a497; /* yyvsp34 */
	T6 a498; /* yyvsp35 */
	T6 a499; /* yyvsp36 */
	T6 a500; /* yyvsp37 */
	T6 a501; /* yyvsp38 */
	T6 a502; /* yyvsp39 */
	T6 a503; /* yyvsp40 */
	T6 a504; /* yyvsp41 */
	T6 a505; /* yyvsp42 */
	T6 a506; /* yyvsp43 */
	T6 a507; /* yyvsp44 */
	T6 a508; /* yyvsp45 */
	T6 a509; /* yyvsp46 */
	T6 a510; /* yyvsp47 */
	T6 a511; /* yyvsp48 */
	T6 a512; /* yyvsp49 */
	T6 a513; /* yyvsp50 */
	T6 a514; /* yyvsp51 */
	T6 a515; /* yyvsp52 */
	T6 a516; /* yyvsp53 */
	T6 a517; /* yyvsp54 */
	T6 a518; /* yyvsp55 */
	T6 a519; /* yyvsp56 */
	T6 a520; /* yyvsp57 */
	T6 a521; /* yyvsp58 */
	T6 a522; /* yyvsp59 */
	T6 a523; /* yyvsp60 */
	T6 a524; /* yyvsp61 */
	T6 a525; /* yyvsp62 */
	T6 a526; /* yyvsp63 */
	T6 a527; /* yyvsp64 */
	T6 a528; /* yyvsp65 */
	T6 a529; /* yyvsp66 */
	T6 a530; /* yyvsp67 */
	T6 a531; /* yyvsp68 */
	T6 a532; /* yyvsp69 */
	T6 a533; /* yyvsp70 */
	T6 a534; /* yyvsp71 */
	T6 a535; /* yyvsp72 */
	T6 a536; /* yyvsp73 */
	T6 a537; /* yyvsp74 */
	T6 a538; /* yyvsp75 */
	T6 a539; /* yyvsp76 */
	T6 a540; /* yyvsp77 */
	T6 a541; /* yyvsp78 */
	T6 a542; /* yyvsp79 */
	T6 a543; /* yyvsp80 */
	T6 a544; /* yyvsp81 */
	T6 a545; /* yyvsp82 */
	T6 a546; /* yyvsp83 */
	T6 a547; /* yyvsp84 */
	T6 a548; /* yyvsp85 */
	T6 a549; /* yyvsp86 */
	T6 a550; /* yyvsp87 */
	T6 a551; /* yyvsp88 */
	T6 a552; /* yyvsp89 */
	T6 a553; /* yyvsp90 */
	T6 a554; /* yyvsp91 */
	T6 a555; /* yyvsp92 */
	T6 a556; /* yyvsp93 */
	T6 a557; /* yyvsp94 */
	T6 a558; /* yyvsp95 */
	T6 a559; /* yyvsp96 */
	T6 a560; /* yyvsp97 */
	T6 a561; /* yyvsp98 */
	T6 a562; /* yyvsp99 */
	T6 a563; /* yyvsp100 */
	T6 a564; /* yyvsp101 */
	T6 a565; /* yyvsp102 */
	T6 a566; /* yyvsp103 */
	T6 a567; /* yyvsp104 */
	T6 a568; /* yyvsp105 */
	T6 a569; /* yyvsp106 */
	T6 a570; /* yyvsp107 */
	T6 a571; /* yyvsp108 */
	T6 a572; /* yyvsp109 */
	T6 a573; /* yyvsp110 */
	T6 a574; /* yyvsp111 */
	T6 a575; /* yyvsp112 */
	T6 a576; /* yyvsp113 */
	T6 a577; /* yyvsp114 */
	T6 a578; /* yyvsp115 */
	T6 a579; /* yyvsp116 */
	T6 a580; /* yyvsp117 */
	T6 a581; /* yyvsp118 */
	T6 a582; /* yyvsp119 */
	T6 a583; /* yyvsp120 */
	T6 a584; /* yyvsp121 */
	T6 a585; /* yyvsp122 */
	T6 a586; /* yyvsp123 */
	T6 a587; /* yyvsp124 */
	T0* a588; /* last_detachable_any_value */
	T0* a589; /* last_detachable_et_keyword_value */
	T0* a590; /* last_detachable_et_agent_keyword_value */
	T0* a591; /* last_detachable_et_precursor_keyword_value */
	T0* a592; /* last_detachable_et_symbol_value */
	T0* a593; /* last_detachable_et_position_value */
	T0* a594; /* last_detachable_et_boolean_constant_value */
	T0* a595; /* last_detachable_et_break_value */
	T0* a596; /* last_detachable_et_character_constant_value */
	T0* a597; /* last_detachable_et_current_value */
	T0* a598; /* last_detachable_et_free_operator_value */
	T0* a599; /* last_detachable_et_identifier_value */
	T0* a600; /* last_detachable_et_integer_constant_value */
	T0* a601; /* last_detachable_et_keyword_operator_value */
	T0* a602; /* last_detachable_et_manifest_string_value */
	T0* a603; /* last_detachable_et_real_constant_value */
	T0* a604; /* last_detachable_et_result_value */
	T0* a605; /* last_detachable_et_retry_instruction_value */
	T0* a606; /* last_detachable_et_symbol_operator_value */
	T0* a607; /* last_detachable_et_void_value */
	T0* a608; /* last_detachable_et_semicolon_symbol_value */
	T0* a609; /* last_detachable_et_bracket_symbol_value */
	T0* a610; /* last_detachable_et_question_mark_symbol_value */
	T0* a611; /* last_formal_arguments */
	T0* a612; /* last_local_variables */
	T0* a613; /* last_object_tests */
	T0* a614; /* last_across_components */
	T6 a615; /* last_break_end */
	T6 a616; /* last_comment_end */
	T6 a617; /* ms_column */
	T6 a618; /* ms_line */
	T6 a619; /* break_kind */
};

/* Struct for type ET_PROVIDER_CHECKER */
struct S360 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* current_class */
	T0* a2; /* system_processor */
};

/* Struct for type ET_ANCESTOR_BUILDER */
struct S361 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* class_sorter */
	T0* a2; /* ancestors */
	T0* a3; /* conforming_ancestors */
	T0* a4; /* parent_checker */
	T0* a5; /* formal_parameter_checker */
	T0* a6; /* parent_context */
	T0* a7; /* current_class */
	T0* a8; /* system_processor */
};

/* Struct for type ET_FEATURE_FLATTENER */
struct S362 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* named_features */
	T0* a2; /* queries */
	T0* a3; /* procedures */
	T0* a4; /* aliased_features */
	T0* a5; /* clients_list */
	T0* a6; /* client_classes */
	T0* a7; /* feature_adaptation_resolver */
	T0* a8; /* dotnet_feature_adaptation_resolver */
	T0* a9; /* identifier_type_resolver */
	T0* a10; /* unfolded_tuple_actual_parameters_resolver */
	T0* a11; /* anchored_type_checker */
	T0* a12; /* signature_checker */
	T0* a13; /* parent_checker */
	T0* a14; /* formal_parameter_checker */
	T0* a15; /* builtin_feature_checker */
	T0* a16; /* precursor_checker */
	T0* a17; /* precursors */
	T0* a18; /* current_class */
	T0* a19; /* system_processor */
	T1 a20; /* has_signature_error */
};

/* Struct for type ET_INTERFACE_CHECKER */
struct S363 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* classes_to_be_processed */
	T0* a2; /* parent_checker3 */
	T0* a3; /* qualified_anchored_type_checker */
	T0* a4; /* named_features */
	T0* a5; /* feature_adaptation_resolver */
	T0* a6; /* dotnet_feature_adaptation_resolver */
	T0* a7; /* signature_checker */
	T0* a8; /* unfolded_tuple_actual_parameters_resolver */
	T0* a9; /* current_class */
	T0* a10; /* system_processor */
};

/* Struct for type ET_IMPLEMENTATION_CHECKER */
struct S364 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* feature_checker */
	T0* a2; /* precursor_procedures */
	T0* a3; /* precursor_queries */
	T0* a4; /* supplier_builder */
	T0* a5; /* current_class */
	T0* a6; /* no_suppliers */
	T0* a7; /* system_processor */
};

/* Struct for type [attached] DS_ARRAYED_LIST [[attached] ET_CLASS] */
struct S365 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
	T0* a3; /* special_routines */
	T6 a4; /* capacity */
	T0* a5; /* internal_cursor */
	T0* a6; /* equality_tester */
};

/* Struct for type [attached] TUPLE [[attached] DS_ARRAYED_LIST [[attached] ET_CLASS]] */
struct S366 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type ET_AST_NULL_PROCESSOR */
struct S367 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] DS_ARRAYED_LIST [[attached] INTEGER_32] */
struct S368 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* a3; /* storage */
	T0* a4; /* special_routines */
	T0* a5; /* internal_cursor */
};

/* Struct for type ET_AST_FACTORY */
struct S369 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type ET_DOTNET_ASSEMBLY_CLASSIC_CONSUMER */
struct S371 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* system_processor */
};

/* Struct for type DT_DATE_TIME_DURATION */
struct S372 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* year */
	T6 a2; /* month */
	T6 a3; /* day */
	T6 a4; /* hour */
	T6 a5; /* minute */
	T6 a6; /* second */
	T6 a7; /* millisecond */
};

/* Struct for type ET_DYNAMIC_TYPE_SET_LIST */
struct S373 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
};

/* Struct for type KL_STRING_OUTPUT_STREAM */
struct S374 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* string */
};

/* Struct for type [attached] DS_ARRAYED_LIST [[attached] ET_IDENTIFIER] */
struct S375 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* a3; /* storage */
	T0* a4; /* special_routines */
	T0* a5; /* internal_cursor */
};

/* Struct for type [attached] DS_ARRAYED_LIST [detachable ET_DYNAMIC_TYPE] */
struct S376 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
	T0* a3; /* special_routines */
	T6 a4; /* capacity */
	T0* a5; /* internal_cursor */
};

/* Struct for type ET_DYNAMIC_TYPE_HASH_LIST */
struct S377 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
	T0* a3; /* clashes */
	T0* a4; /* slots */
};

/* Struct for type ET_DYNAMIC_STANDALONE_TYPE_SET */
struct S378 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* static_type */
	T6 a2; /* count */
	T1 a3; /* is_never_void */
	T0* a4; /* dynamic_types */
};

/* Struct for type [attached] DS_ARRAYED_STACK [[attached] ET_EXPRESSION] */
struct S379 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
	T0* a3; /* special_routines */
	T6 a4; /* capacity */
};

/* Struct for type [attached] DS_ARRAYED_LIST [[attached] ET_EXPRESSION] */
struct S380 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* capacity */
	T6 a2; /* count */
	T0* a3; /* storage */
	T0* a4; /* special_routines */
	T0* a5; /* internal_cursor */
};

/* Struct for type [attached] DS_HASH_TABLE [[attached] ET_DYNAMIC_TYPE, [attached] INTEGER_32] */
struct S381 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* item_storage */
	T6 a3; /* last_position */
	T6 a4; /* capacity */
	T6 a5; /* count */
	T6 a6; /* free_slot */
	T6 a7; /* slots_position */
	T6 a8; /* modulus */
	T6 a9; /* clashes_previous_position */
	T0* a10; /* key_equality_tester */
	T0* a11; /* equality_tester */
	T6 a12; /* found_position */
	T0* a13; /* clashes */
	T0* a14; /* slots */
	T0* a15; /* special_item_routines */
	T0* a16; /* special_key_routines */
	T0* a17; /* key_storage */
	T0* a18; /* internal_cursor */
	T0* a19; /* hash_function */
};

/* Struct for type ET_DYNAMIC_STANDALONE_TYPE_SET_LIST */
struct S382 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
};

/* Struct for type [attached] DS_HASH_SET [[attached] ET_DYNAMIC_TYPE] */
struct S383 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* internal_cursor */
	T6 a2; /* count */
	T0* a3; /* item_storage */
	T6 a4; /* capacity */
	T6 a5; /* modulus */
	T6 a6; /* last_position */
	T6 a7; /* free_slot */
	T6 a8; /* position */
	T6 a9; /* slots_position */
	T0* a10; /* special_item_routines */
	T0* a11; /* clashes */
	T0* a12; /* slots */
	T6 a13; /* found_position */
	T6 a14; /* clashes_previous_position */
	T0* a15; /* hash_function */
	T0* a16; /* equality_tester */
};

/* Struct for type [attached] DS_HASH_TABLE [[attached] ET_DYNAMIC_STANDALONE_TYPE_SET, [attached] ET_DYNAMIC_TYPE] */
struct S384 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* internal_cursor */
	T6 a3; /* found_position */
	T0* a4; /* item_storage */
	T6 a5; /* last_position */
	T6 a6; /* free_slot */
	T6 a7; /* position */
	T6 a8; /* capacity */
	T0* a9; /* equality_tester */
	T0* a10; /* key_equality_tester */
	T0* a11; /* key_storage */
	T0* a12; /* clashes */
	T6 a13; /* modulus */
	T0* a14; /* slots */
	T6 a15; /* slots_position */
	T6 a16; /* clashes_previous_position */
	T0* a17; /* special_item_routines */
	T0* a18; /* special_key_routines */
	T0* a19; /* hash_function */
};

/* Struct for type [attached] DS_ARRAYED_LIST [[attached] ET_OBJECT_TEST] */
struct S385 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
	T0* a3; /* special_routines */
	T6 a4; /* capacity */
	T0* a5; /* internal_cursor */
};

/* Struct for type [attached] DS_ARRAYED_LIST [[attached] ET_OBJECT_EQUALITY_EXPRESSION] */
struct S386 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
	T0* a3; /* special_routines */
	T6 a4; /* capacity */
	T0* a5; /* internal_cursor */
};

/* Struct for type [attached] DS_ARRAYED_LIST [[attached] ET_EQUALITY_EXPRESSION] */
struct S387 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
	T0* a3; /* special_routines */
	T6 a4; /* capacity */
	T0* a5; /* internal_cursor */
};

/* Struct for type [attached] DS_ARRAYED_LIST [[attached] ET_AGENT] */
struct S388 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
	T0* a3; /* special_routines */
	T6 a4; /* capacity */
	T0* a5; /* internal_cursor */
};

/* Struct for type ET_EXPRESSION */
struct S389 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type ET_CURRENT */
struct S390 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* index */
	T2 a2; /* code */
	T0* a3; /* text */
	T6 a4; /* compressed_position */
};

/* Struct for type ET_ACTUAL_ARGUMENT_LIST */
struct S391 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
	T0* a3; /* left_symbol */
	T0* a4; /* right_symbol */
};

/* Struct for type ET_QUALIFIED_CALL_INSTRUCTION */
struct S392 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* target */
	T0* a2; /* qualified_name */
	T0* a3; /* arguments */
	T0* a4; /* parenthesis_call */
};

/* Struct for type ET_QUALIFIED_CALL_EXPRESSION */
struct S396 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* target */
	T6 a2; /* index */
	T0* a3; /* qualified_name */
	T0* a4; /* arguments */
	T0* a5; /* parenthesis_call */
};

/* Struct for type ET_UNQUALIFIED_CALL_EXPRESSION */
struct S397 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* name */
	T0* a2; /* arguments */
	T6 a3; /* index */
	T0* a4; /* parenthesis_call */
};

/* Struct for type [attached] DS_ARRAYED_LIST [detachable ET_QUALIFIED_CALL_EXPRESSION] */
struct S398 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* a3; /* storage */
	T0* a4; /* special_routines */
	T0* a5; /* internal_cursor */
};

/* Struct for type ET_MANIFEST_TUPLE */
struct S399 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* left_symbol */
	T0* a2; /* right_symbol */
	T6 a3; /* count */
	T0* a4; /* storage */
	T6 a5; /* index */
};

/* Struct for type [attached] DS_HASH_SET [[attached] ET_DYNAMIC_TUPLE_TYPE] */
struct S400 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* internal_cursor */
	T0* a2; /* item_storage */
	T6 a3; /* capacity */
	T6 a4; /* modulus */
	T6 a5; /* last_position */
	T6 a6; /* free_slot */
	T6 a7; /* position */
	T6 a8; /* count */
	T0* a9; /* special_item_routines */
	T0* a10; /* clashes */
	T0* a11; /* slots */
	T6 a12; /* found_position */
	T6 a13; /* slots_position */
	T6 a14; /* clashes_previous_position */
	T0* a15; /* hash_function */
	T0* a16; /* equality_tester */
};

/* Struct for type [attached] DS_HASH_TABLE [[attached] INTEGER_32, [attached] ET_FEATURE] */
struct S401 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T6 a2; /* count */
	T6 a3; /* last_position */
	T6 a4; /* free_slot */
	T6 a5; /* modulus */
	T6 a6; /* slots_position */
	T6 a7; /* clashes_previous_position */
	T0* a8; /* key_equality_tester */
	T0* a9; /* equality_tester */
	T0* a10; /* internal_cursor */
	T6 a11; /* found_position */
	T0* a12; /* item_storage */
	T0* a13; /* key_storage */
	T0* a14; /* clashes */
	T0* a15; /* slots */
	T6 a16; /* capacity */
	T0* a17; /* special_item_routines */
	T0* a18; /* special_key_routines */
	T0* a19; /* hash_function */
};

/* Struct for type [attached] ARRAY [[attached] INTEGER_32] */
struct S402 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* area */
	T6 a2; /* lower */
	T6 a3; /* upper */
};

/* Struct for type [attached] DS_HASH_TABLE [[attached] ET_CONSTANT, [attached] ET_FEATURE] */
struct S403 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* internal_cursor */
	T0* a2; /* key_storage */
	T0* a3; /* item_storage */
	T6 a4; /* count */
	T6 a5; /* last_position */
	T6 a6; /* free_slot */
	T6 a7; /* position */
	T6 a8; /* capacity */
	T6 a9; /* slots_position */
	T0* a10; /* equality_tester */
	T0* a11; /* key_equality_tester */
	T6 a12; /* found_position */
	T0* a13; /* clashes */
	T6 a14; /* modulus */
	T0* a15; /* slots */
	T6 a16; /* clashes_previous_position */
	T0* a17; /* special_item_routines */
	T0* a18; /* special_key_routines */
	T0* a19; /* hash_function */
};

/* Struct for type [attached] DS_HASH_TABLE [[attached] ET_DYNAMIC_TYPE, [attached] ET_INLINE_CONSTANT] */
struct S404 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* internal_cursor */
	T0* a2; /* key_storage */
	T0* a3; /* item_storage */
	T6 a4; /* count */
	T6 a5; /* last_position */
	T6 a6; /* free_slot */
	T6 a7; /* position */
	T0* a8; /* equality_tester */
	T0* a9; /* key_equality_tester */
	T6 a10; /* found_position */
	T0* a11; /* clashes */
	T6 a12; /* modulus */
	T0* a13; /* slots */
	T6 a14; /* capacity */
	T0* a15; /* special_item_routines */
	T0* a16; /* special_key_routines */
	T6 a17; /* slots_position */
	T6 a18; /* clashes_previous_position */
	T0* a19; /* hash_function */
};

/* Struct for type [attached] DS_HASH_TABLE [detachable ET_DYNAMIC_FEATURE, [attached] ET_DYNAMIC_TYPE] */
struct S405 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* found_position */
	T0* a2; /* item_storage */
	T6 a3; /* count */
	T6 a4; /* last_position */
	T6 a5; /* free_slot */
	T6 a6; /* position */
	T6 a7; /* capacity */
	T0* a8; /* equality_tester */
	T0* a9; /* key_equality_tester */
	T0* a10; /* internal_cursor */
	T0* a11; /* key_storage */
	T0* a12; /* clashes */
	T6 a13; /* modulus */
	T0* a14; /* slots */
	T6 a15; /* slots_position */
	T6 a16; /* clashes_previous_position */
	T0* a17; /* special_item_routines */
	T0* a18; /* special_key_routines */
	T0* a19; /* hash_function */
};

/* Struct for type [attached] DS_ARRAYED_LIST [[attached] ET_DYNAMIC_FEATURE] */
struct S406 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
	T0* a3; /* special_routines */
	T6 a4; /* capacity */
	T0* a5; /* internal_cursor */
};

/* Struct for type [attached] DS_HASH_TABLE [[attached] BOOLEAN, [attached] STRING_8] */
struct S407 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* internal_cursor */
	T6 a2; /* position */
	T0* a3; /* item_storage */
	T0* a4; /* key_storage */
	T0* a5; /* key_equality_tester */
	T0* a6; /* internal_keys */
	T6 a7; /* count */
	T6 a8; /* last_position */
	T6 a9; /* free_slot */
	T6 a10; /* capacity */
	T6 a11; /* slots_position */
	T6 a12; /* modulus */
	T6 a13; /* clashes_previous_position */
	T0* a14; /* equality_tester */
	T6 a15; /* found_position */
	T0* a16; /* clashes */
	T0* a17; /* slots */
	T0* a18; /* special_item_routines */
	T0* a19; /* special_key_routines */
	T0* a20; /* hash_function */
};

/* Struct for type ET_IDENTIFIER_TESTER */
struct S408 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type RX_PCRE_REGULAR_EXPRESSION */
struct S410 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* subject */
	T6 a2; /* subject_end */
	T6 a3; /* subject_start */
	T6 a4; /* match_count */
	T0* a5; /* offset_vector */
	T6 a6; /* offset_vector_count */
	T6 a7; /* brastart_capacity */
	T0* a8; /* brastart_vector */
	T6 a9; /* brastart_lower */
	T6 a10; /* brastart_count */
	T6 a11; /* eptr_capacity */
	T0* a12; /* eptr_vector */
	T6 a13; /* eptr_lower */
	T6 a14; /* eptr_upper */
	T0* a15; /* byte_code */
	T0* a16; /* internal_start_bits */
	T0* a17; /* pattern_buffer */
	T6 a18; /* subexpression_count */
	T0* a19; /* pattern */
	T6 a20; /* pattern_count */
	T6 a21; /* pattern_position */
	T6 a22; /* code_index */
	T6 a23; /* maxbackrefs */
	T6 a24; /* optchanged */
	T6 a25; /* first_character */
	T6 a26; /* required_character */
	T6 a27; /* regexp_countlits */
	T0* a28; /* start_bits */
	T0* a29; /* error_message */
	T1 a30; /* is_anchored */
	T0* a31; /* character_case_mapping */
	T0* a32; /* word_set */
	T6 a33; /* subject_next_start */
	T6 a34; /* error_code */
	T6 a35; /* error_position */
	T1 a36; /* is_startline */
	T1 a37; /* is_caseless */
	T1 a38; /* is_bol */
	T1 a39; /* is_eol */
	T1 a40; /* is_multiline */
	T1 a41; /* is_dotall */
	T1 a42; /* is_extended */
	T1 a43; /* is_empty_allowed */
	T1 a44; /* is_dollar_endonly */
	T1 a45; /* is_greedy */
	T1 a46; /* is_strict */
	T1 a47; /* is_ichanged */
	T6 a48; /* first_matched_index */
	T6 a49; /* eptr */
	T6 a50; /* offset_top */
	T1 a51; /* is_matching_caseless */
	T1 a52; /* is_matching_multiline */
	T1 a53; /* is_matching_dotall */
};

/* Struct for type KL_TEXT_OUTPUT_FILE */
struct S411 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* name */
	T6 a2; /* mode */
	T14 a3; /* file_pointer */
	T1 a4; /* descriptor_available */
	T0* a5; /* last_string */
	T0* a6; /* internal_name */
	T0* a7; /* internal_detachable_name_pointer */
};

/* Struct for type ET_DYNAMIC_TUPLE_TYPE */
struct S412 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* id */
	T0* a2; /* base_class */
	T0* a3; /* base_type */
	T1 a4; /* is_alive */
	T6 a5; /* attribute_count */
	T0* a6; /* queries */
	T0* a7; /* meta_type */
	T0* a8; /* procedures */
	T0* a9; /* item_type_sets */
	T1 a10; /* has_reference_attributes */
	T1 a11; /* has_generic_expanded_attributes */
	T0* a12; /* alive_conforming_descendants */
	T6 a13; /* hash_code */
	T0* a14; /* next_type */
	T0* a15; /* query_calls */
	T0* a16; /* conforming_ancestors */
	T0* a17; /* conforming_descendants */
	T1 a18; /* has_static */
	T0* a19; /* procedure_calls */
	T0* a20; /* equality_expressions */
	T0* a21; /* object_equality_expressions */
};

/* Struct for type UT_CANNOT_WRITE_TO_FILE_ERROR */
struct S413 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* parameters */
};

/* Struct for type ET_DYNAMIC_FEATURE_LIST */
struct S414 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
};

/* Struct for type ET_DYNAMIC_PRECURSOR */
struct S416 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* is_static_generated */
	T0* a2; /* target_type */
	T0* a3; /* static_feature */
	T0* a4; /* dynamic_type_sets */
	T0* a5; /* current_feature */
	T6 a6; /* id */
	T0* a7; /* result_type_set */
	T1 a8; /* is_regular */
	T1 a9; /* is_creation */
	T1 a10; /* is_address */
	T8 a11; /* builtin_class_code */
	T1 a12; /* is_generated */
	T8 a13; /* builtin_feature_code */
	T0* a14; /* first_precursor */
	T0* a15; /* other_precursors */
	T1 a16; /* is_built */
	T0* a17; /* parent_type */
	T1 a18; /* is_current_type_needed */
};

/* Struct for type ET_DYNAMIC_PRECURSOR_LIST */
struct S418 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
};

/* Struct for type [attached] DS_HASH_TOPOLOGICAL_SORTER [[attached] ET_DYNAMIC_TYPE] */
struct S419 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* sorted_items */
	T0* a2; /* cycle */
	T0* a3; /* indexes */
	T0* a4; /* items */
	T0* a5; /* counts */
	T0* a6; /* successors */
};

/* Struct for type ET_RESULT */
struct S422 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* index */
	T2 a2; /* code */
	T0* a3; /* text */
	T6 a4; /* compressed_position */
};

/* Struct for type ET_DYNAMIC_QUALIFIED_QUERY_CALL */
struct S424 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* static_call */
	T0* a2; /* target_type_set */
	T0* a3; /* result_type_set */
	T0* a4; /* current_feature */
	T0* a5; /* current_type */
	T0* a6; /* next */
};

/* Struct for type ET_DYNAMIC_QUALIFIED_PROCEDURE_CALL */
struct S429 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* target_type_set */
	T0* a2; /* next */
	T0* a3; /* static_call */
	T0* a4; /* current_feature */
	T0* a5; /* current_type */
};

/* Struct for type ET_OBJECT_TEST */
struct S431 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* index */
	T0* a2; /* attached_keyword */
	T0* a3; /* declared_type */
	T0* a4; /* expression */
};

/* Struct for type ET_OBJECT_EQUALITY_EXPRESSION */
struct S432 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* index */
	T0* a2; /* left */
	T0* a3; /* operator */
	T0* a4; /* right */
};

/* Struct for type ET_EQUALITY_EXPRESSION */
struct S433 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* index */
	T0* a2; /* left */
	T0* a3; /* operator */
	T0* a4; /* right */
};

/* Struct for type ET_DYNAMIC_SPECIAL_TYPE */
struct S434 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* id */
	T0* a2; /* base_class */
	T0* a3; /* base_type */
	T1 a4; /* is_alive */
	T6 a5; /* attribute_count */
	T0* a6; /* queries */
	T0* a7; /* meta_type */
	T0* a8; /* procedures */
	T0* a9; /* item_type_set */
	T1 a10; /* has_reference_attributes */
	T1 a11; /* has_generic_expanded_attributes */
	T0* a12; /* alive_conforming_descendants */
	T6 a13; /* hash_code */
	T0* a14; /* next_type */
	T0* a15; /* query_calls */
	T0* a16; /* conforming_ancestors */
	T0* a17; /* conforming_descendants */
	T1 a18; /* has_static */
	T0* a19; /* procedure_calls */
	T0* a20; /* equality_expressions */
	T0* a21; /* object_equality_expressions */
};

/* Struct for type [attached] DS_QUICK_SORTER [[attached] ET_DYNAMIC_TYPE] */
struct S438 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* comparator */
};

/* Struct for type ET_DYNAMIC_TYPE_COMPARATOR_BY_ID */
struct S439 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type ET_CREATE_EXPRESSION */
struct S444 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* index */
	T0* a2; /* creation_call */
	T0* a3; /* creation_type */
	T0* a4; /* create_keyword */
	T0* a5; /* creation_region */
};

/* Struct for type ET_QUALIFIED_CALL */
struct S445 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* arguments */
	T0* a2; /* qualified_name */
};

/* Struct for type [attached] ARRAY [[attached] STRING_8] */
struct S449 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* area */
	T6 a2; /* lower */
	T6 a3; /* upper */
};

/* Struct for type KL_DIRECTORY */
struct S451 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* end_of_input */
	T0* a2; /* last_entry */
	T6 a3; /* mode */
	T0* a4; /* name */
	T0* a5; /* entry_buffer */
	T1 a6; /* old_end_of_input */
	T14 a7; /* directory_pointer */
	T14 a8; /* last_entry_pointer */
	T0* a9; /* lastentry */
	T0* a10; /* internal_name */
	T0* a11; /* internal_detachable_name_pointer */
};

/* Struct for type UT_TEMPLATE_EXPANDER */
struct S452 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type ET_EXTENDED_ATTRIBUTE */
struct S454 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* declared_type */
	T0* a2; /* implementation_class */
	T0* a3; /* assigner */
	T0* a4; /* obsolete_message */
	T0* a5; /* preconditions */
	T0* a6; /* locals */
	T0* a7; /* compound */
	T0* a8; /* postconditions */
	T0* a9; /* rescue_clause */
	T0* a10; /* clients */
	T0* a11; /* end_keyword */
	T0* a12; /* semicolon */
	T0* a13; /* feature_clause */
	T0* a14; /* first_indexing */
	T0* a15; /* object_tests */
	T0* a16; /* across_components */
	T0* a17; /* extended_name */
	T6 a18; /* hash_code */
	T0* a19; /* implementation_feature */
	T0* a20; /* frozen_keyword */
	T0* a21; /* synonym */
	T1 a22; /* validity_checked */
	T1 a23; /* has_validity_error */
	T0* a24; /* first_precursor */
	T0* a25; /* other_precursors */
	T6 a26; /* first_seed */
	T0* a27; /* other_seeds */
	T6 a28; /* id */
	T6 a29; /* version */
};

/* Struct for type ET_COMPOUND */
struct S456 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* keyword */
	T6 a2; /* count */
	T0* a3; /* storage */
};

/* Struct for type [attached] DS_QUICK_SORTER [[attached] INTEGER_32] */
struct S464 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* comparator */
};

/* Struct for type ET_IMPLICIT_TYPE_MARK */
struct S472 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* is_expanded_mark */
	T1 a2; /* is_reference_mark */
	T1 a3; /* is_attached_mark */
	T1 a4; /* is_separate_mark */
	T1 a5; /* is_detachable_mark */
};

/* Struct for type ET_SYMBOL */
struct S474 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T2 a1; /* code */
	T6 a2; /* compressed_position */
};

/* Struct for type [attached] PROCEDURE [[attached] TUPLE] */
struct S475 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T14 a1; /* rout_disp */
	T0* a2; /* closed_operands */
	T14 a3; /* encaps_rout_disp */
	T14 a4; /* calc_rout_addr */
	T1 a5; /* is_target_closed */
	T6 a6; /* open_count */
};

/* Struct for type [attached] TUPLE [ET_C_GENERATOR, ET_IDENTIFIER, ET_CURRENT, ET_DYNAMIC_TYPE, BOOLEAN] */
struct S476 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
	T0* z2;
	T0* z3;
	T0* z4;
	T1 z5;
};

/* Struct for type [attached] TUPLE [ET_C_GENERATOR, ET_IDENTIFIER, ET_RESULT, ET_DYNAMIC_TYPE, BOOLEAN] */
struct S477 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
	T0* z2;
	T0* z3;
	T0* z4;
	T1 z5;
};

/* Struct for type [attached] TUPLE [ET_C_GENERATOR, ET_DYNAMIC_FEATURE, ET_RESULT, ET_DYNAMIC_TYPE, BOOLEAN] */
struct S478 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
	T0* z2;
	T0* z3;
	T0* z4;
	T1 z5;
};

/* Struct for type [attached] TUPLE [ET_C_GENERATOR, INTEGER_32, ET_RESULT, ET_DYNAMIC_TYPE, BOOLEAN] */
struct S479 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
	T6 z2;
	T0* z3;
	T0* z4;
	T1 z5;
};

/* Struct for type UT_INTEGER_FORMATTER */
struct S480 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type STD_FILES */
struct S485 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* default_output */
};

/* Struct for type [attached] TUPLE [ET_C_GENERATOR, INTEGER_32, ET_EXPRESSION, ET_DYNAMIC_TYPE, BOOLEAN] */
struct S487 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
	T6 z2;
	T0* z3;
	T0* z4;
	T1 z5;
};

/* Struct for type [attached] TUPLE [ET_C_GENERATOR, ET_EXPRESSION, ET_DYNAMIC_TYPE] */
struct S490 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
	T0* z2;
	T0* z3;
};

/* Struct for type [attached] TUPLE [ET_C_GENERATOR, ET_DYNAMIC_FEATURE, ET_DYNAMIC_TYPE, BOOLEAN] */
struct S491 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
	T0* z2;
	T0* z3;
	T1 z4;
};

/* Struct for type ET_CONSTANT_ATTRIBUTE */
struct S492 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* declared_type */
	T0* a2; /* constant */
	T0* a3; /* implementation_class */
	T0* a4; /* assigner */
	T0* a5; /* clients */
	T0* a6; /* is_keyword */
	T0* a7; /* semicolon */
	T0* a8; /* feature_clause */
	T0* a9; /* first_indexing */
	T0* a10; /* object_tests */
	T0* a11; /* across_components */
	T0* a12; /* extended_name */
	T6 a13; /* hash_code */
	T0* a14; /* implementation_feature */
	T0* a15; /* frozen_keyword */
	T0* a16; /* synonym */
	T1 a17; /* validity_checked */
	T1 a18; /* has_validity_error */
	T0* a19; /* first_precursor */
	T0* a20; /* other_precursors */
	T6 a21; /* first_seed */
	T0* a22; /* other_seeds */
	T6 a23; /* id */
	T6 a24; /* version */
};

/* Struct for type ET_UNIQUE_ATTRIBUTE */
struct S494 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* declared_type */
	T0* a2; /* implementation_class */
	T0* a3; /* assigner */
	T0* a4; /* clients */
	T0* a5; /* is_keyword */
	T0* a6; /* unique_keyword */
	T0* a7; /* semicolon */
	T0* a8; /* feature_clause */
	T0* a9; /* first_indexing */
	T0* a10; /* object_tests */
	T0* a11; /* across_components */
	T0* a12; /* constant */
	T0* a13; /* extended_name */
	T6 a14; /* hash_code */
	T0* a15; /* implementation_feature */
	T0* a16; /* frozen_keyword */
	T0* a17; /* synonym */
	T6 a18; /* id */
	T1 a19; /* validity_checked */
	T1 a20; /* has_validity_error */
	T0* a21; /* first_precursor */
	T0* a22; /* other_precursors */
	T6 a23; /* first_seed */
	T0* a24; /* other_seeds */
	T6 a25; /* version */
};

/* Struct for type ET_REGULAR_INTEGER_CONSTANT */
struct S495 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* sign */
	T6 a2; /* index */
	T0* a3; /* literal */
	T0* a4; /* cast_type */
	T6 a5; /* compressed_position */
	T1 a6; /* has_overflow */
	T11 a7; /* value */
	T0* a8; /* type */
};

/* Struct for type ET_ATTRIBUTE */
struct S496 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* declared_type */
	T0* a2; /* implementation_class */
	T0* a3; /* assigner */
	T0* a4; /* clients */
	T0* a5; /* semicolon */
	T0* a6; /* feature_clause */
	T0* a7; /* first_indexing */
	T0* a8; /* object_tests */
	T0* a9; /* across_components */
	T0* a10; /* extended_name */
	T6 a11; /* hash_code */
	T0* a12; /* implementation_feature */
	T0* a13; /* frozen_keyword */
	T0* a14; /* synonym */
	T1 a15; /* validity_checked */
	T1 a16; /* has_validity_error */
	T0* a17; /* first_precursor */
	T0* a18; /* other_precursors */
	T6 a19; /* first_seed */
	T0* a20; /* other_seeds */
	T6 a21; /* id */
	T6 a22; /* version */
};

/* Struct for type ET_LIKE_CURRENT */
struct S498 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* type_mark */
	T0* a2; /* like_keyword */
	T0* a3; /* current_keyword */
};

/* Struct for type NATIVE_STRING */
struct S500 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* managed_data */
	T6 a2; /* unit_count */
};

/* Struct for type [attached] DS_ARRAYED_LIST [[attached] ET_SYSTEM_PROCESSOR] */
struct S501 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
	T0* a3; /* special_routines */
	T6 a4; /* capacity */
	T0* a5; /* internal_cursor */
};

/* Struct for type WORKER_THREAD */
struct S502 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* thread_procedure */
	T0* a2; /* launch_mutex */
	T14 a3; /* thread_id */
	T1 a4; /* terminated */
};

/* Struct for type [attached] TUPLE [ET_SYSTEM_PROCESSOR, [attached] DS_ARRAYED_LIST [[attached] ET_CLASS]] */
struct S503 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
	T0* z2;
};

/* Struct for type ET_FEATURE_CHECKER */
struct S505 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* system_processor */
	T0* a2; /* type_checker */
	T0* a3; /* current_type */
	T0* a4; /* current_class */
	T0* a5; /* current_feature */
	T0* a6; /* current_feature_impl */
	T0* a7; /* current_class_impl */
	T0* a8; /* enclosing_inline_agents */
	T0* a9; /* unused_overloaded_procedures_list */
	T0* a10; /* unused_overloaded_queries_list */
	T0* a11; /* unused_overloaded_features_list */
	T0* a12; /* unused_contexts */
	T0* a13; /* current_context */
	T0* a14; /* current_target_type */
	T0* a15; /* current_object_test_types */
	T0* a16; /* current_object_test_scope */
	T0* a17; /* object_test_scope_builder */
	T0* a18; /* current_across_cursor_types */
	T0* a19; /* current_across_cursor_scope */
	T0* a20; /* current_initialization_scope */
	T0* a21; /* current_attachment_scope */
	T0* a22; /* attachment_scope_builder */
	T0* a23; /* unused_attachment_scopes */
	T0* a24; /* common_ancestor_type_list */
	T0* a25; /* indexing_term_list */
	T0* a26; /* default_creation_call_name */
	T0* a27; /* default_creation_call */
	T1 a28; /* has_fatal_error */
	T1 a29; /* in_static_feature */
	T1 a30; /* in_precondition */
	T0* a31; /* supplier_handler */
	T1 a32; /* in_invariant */
	T0* a33; /* precursor_queries */
	T0* a34; /* precursor_procedures */
	T1 a35; /* in_postcondition */
	T1 a36; /* in_precursor */
	T1 a37; /* in_rescue */
	T0* a38; /* current_inline_agent */
	T1 a39; /* in_check_instruction */
};

/* Struct for type [attached] DS_CELL [[attached] INTEGER_32] */
struct S506 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* item */
};

/* Struct for type [attached] DS_HASH_TABLE [[attached] INTEGER_32, [attached] ET_DYNAMIC_TYPE] */
struct S507 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T6 a2; /* count */
	T6 a3; /* last_position */
	T6 a4; /* free_slot */
	T6 a5; /* modulus */
	T6 a6; /* capacity */
	T6 a7; /* slots_position */
	T6 a8; /* clashes_previous_position */
	T0* a9; /* key_equality_tester */
	T0* a10; /* equality_tester */
	T0* a11; /* internal_cursor */
	T6 a12; /* found_position */
	T0* a13; /* item_storage */
	T0* a14; /* key_storage */
	T0* a15; /* clashes */
	T0* a16; /* slots */
	T0* a17; /* special_item_routines */
	T0* a18; /* special_key_routines */
	T0* a19; /* hash_function */
};

/* Struct for type ET_TYPE_CHECKER */
struct S510 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* current_context */
	T0* a2; /* current_class_impl */
	T0* a3; /* current_feature_impl */
	T0* a4; /* constraint_context */
	T0* a5; /* current_class */
	T0* a6; /* system_processor */
	T0* a7; /* supplier_handler */
	T1 a8; /* has_fatal_error */
	T1 a9; /* class_interface_error_only */
	T1 a10; /* in_qualified_anchored_type */
};

/* Struct for type [attached] DS_ARRAYED_LIST [[attached] ET_INLINE_AGENT] */
struct S512 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* special_routines */
	T0* a2; /* storage */
	T6 a3; /* capacity */
	T0* a4; /* internal_cursor */
	T6 a5; /* count */
};

/* Struct for type [attached] DS_ARRAYED_LIST [[attached] DS_ARRAYED_LIST [[attached] ET_PROCEDURE]] */
struct S513 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* special_routines */
	T0* a2; /* storage */
	T6 a3; /* capacity */
	T0* a4; /* internal_cursor */
	T6 a5; /* count */
};

/* Struct for type [attached] DS_ARRAYED_LIST [[attached] DS_ARRAYED_LIST [[attached] ET_QUERY]] */
struct S514 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* special_routines */
	T0* a2; /* storage */
	T6 a3; /* capacity */
	T0* a4; /* internal_cursor */
	T6 a5; /* count */
};

/* Struct for type [attached] DS_ARRAYED_LIST [[attached] DS_ARRAYED_LIST [[attached] ET_FEATURE]] */
struct S515 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* special_routines */
	T0* a2; /* storage */
	T6 a3; /* capacity */
	T0* a4; /* internal_cursor */
	T6 a5; /* count */
};

/* Struct for type [attached] DS_ARRAYED_LIST [[attached] ET_NESTED_TYPE_CONTEXT] */
struct S516 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
	T0* a3; /* special_routines */
	T6 a4; /* capacity */
	T0* a5; /* internal_cursor */
};

/* Struct for type ET_NESTED_TYPE_CONTEXT */
struct S517 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* root_context */
	T6 a2; /* count */
	T0* a3; /* storage */
};

/* Struct for type [attached] DS_HASH_TABLE [[attached] ET_NESTED_TYPE_CONTEXT, [attached] ET_NAMED_OBJECT_TEST] */
struct S518 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* internal_cursor */
	T0* a2; /* item_storage */
	T6 a3; /* count */
	T6 a4; /* last_position */
	T6 a5; /* free_slot */
	T6 a6; /* position */
	T0* a7; /* equality_tester */
	T0* a8; /* key_equality_tester */
	T6 a9; /* found_position */
	T0* a10; /* key_storage */
	T0* a11; /* clashes */
	T6 a12; /* modulus */
	T0* a13; /* slots */
	T6 a14; /* capacity */
	T0* a15; /* special_item_routines */
	T0* a16; /* special_key_routines */
	T6 a17; /* slots_position */
	T6 a18; /* clashes_previous_position */
	T0* a19; /* hash_function */
};

/* Struct for type ET_OBJECT_TEST_SCOPE */
struct S519 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* object_tests */
	T6 a2; /* hidden_count */
};

/* Struct for type ET_OBJECT_TEST_SCOPE_BUILDER */
struct S520 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* has_fatal_error */
	T0* a2; /* current_class */
	T0* a3; /* system_processor */
	T0* a4; /* scope */
	T1 a5; /* is_negated */
};

/* Struct for type [attached] DS_HASH_TABLE [[attached] ET_NESTED_TYPE_CONTEXT, [attached] ET_ACROSS_COMPONENT] */
struct S521 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* internal_cursor */
	T0* a2; /* item_storage */
	T6 a3; /* count */
	T6 a4; /* last_position */
	T6 a5; /* free_slot */
	T6 a6; /* position */
	T0* a7; /* equality_tester */
	T0* a8; /* key_equality_tester */
	T6 a9; /* found_position */
	T0* a10; /* key_storage */
	T0* a11; /* clashes */
	T6 a12; /* modulus */
	T0* a13; /* slots */
	T6 a14; /* capacity */
	T0* a15; /* special_item_routines */
	T0* a16; /* special_key_routines */
	T6 a17; /* slots_position */
	T6 a18; /* clashes_previous_position */
	T0* a19; /* internal_keys */
	T0* a20; /* hash_function */
};

/* Struct for type ET_ACROSS_CURSOR_SCOPE */
struct S522 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* hidden_count */
	T0* a2; /* across_components */
};

/* Struct for type ET_ATTACHMENT_SCOPE */
struct S523 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* locals_attached */
	T0* a2; /* arguments_attached */
	T0* a3; /* attributes_attached */
	T1 a4; /* result_attached */
	T1 a5; /* is_code_unreachable */
};

/* Struct for type ET_ATTACHMENT_SCOPE_BUILDER */
struct S524 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* scope */
	T1 a2; /* is_negated */
};

/* Struct for type [attached] DS_ARRAYED_LIST [[attached] ET_ATTACHMENT_SCOPE] */
struct S525 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
	T0* a3; /* special_routines */
	T6 a4; /* capacity */
	T0* a5; /* internal_cursor */
};

/* Struct for type [attached] DS_ARRAYED_LIST [[attached] ET_INDEXING_TERM] */
struct S526 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* special_routines */
	T0* a2; /* storage */
	T6 a3; /* capacity */
	T0* a4; /* internal_cursor */
	T6 a5; /* count */
};

/* Struct for type attached DS_HASH_TABLE [[attached] ET_DYNAMIC_TYPE_SET, [attached] ET_DYNAMIC_TYPE] */
struct S527 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* internal_cursor */
	T6 a2; /* found_position */
	T0* a3; /* item_storage */
	T0* a4; /* key_storage */
	T6 a5; /* position */
	T6 a6; /* last_position */
	T6 a7; /* modulus */
	T6 a8; /* slots_position */
	T6 a9; /* clashes_previous_position */
	T0* a10; /* key_equality_tester */
	T0* a11; /* equality_tester */
	T6 a12; /* capacity */
	T6 a13; /* free_slot */
	T0* a14; /* special_item_routines */
	T0* a15; /* special_key_routines */
	T0* a16; /* clashes */
	T0* a17; /* slots */
	T6 a18; /* count */
	T0* a19; /* hash_function */
};

/* Struct for type ET_PRECONDITIONS */
struct S529 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* require_keyword */
	T6 a2; /* count */
	T0* a3; /* storage */
	T0* a4; /* else_keyword */
	T1 a5; /* validity_checked */
	T1 a6; /* has_validity_error */
};

/* Struct for type [attached] DS_HASH_TABLE [[attached] ET_ASSERTIONS, [attached] ET_FEATURE] */
struct S530 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* internal_cursor */
	T0* a3; /* clashes */
	T0* a4; /* item_storage */
	T6 a5; /* last_position */
	T6 a6; /* position */
	T6 a7; /* capacity */
	T6 a8; /* slots_position */
	T0* a9; /* equality_tester */
	T0* a10; /* key_equality_tester */
	T6 a11; /* found_position */
	T6 a12; /* modulus */
	T6 a13; /* clashes_previous_position */
	T0* a14; /* special_item_routines */
	T0* a15; /* slots */
	T0* a16; /* special_key_routines */
	T0* a17; /* key_storage */
	T6 a18; /* free_slot */
	T0* a19; /* hash_function */
};

/* Struct for type ET_FEATURE_LIST */
struct S533 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* storage */
	T6 a2; /* count */
};

/* Struct for type ET_UNKNOWN_GROUP */
struct S536 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* name */
	T1 a2; /* use_obsolete_syntax */
	T0* a3; /* cached_absolute_pathname */
};

/* Struct for type ET_BASE_TYPE_LIST */
struct S538 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
};

/* Struct for type ET_KEYWORD */
struct S539 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T2 a1; /* code */
	T0* a2; /* text */
	T6 a3; /* compressed_position */
};

/* Struct for type ET_CLASS_CODES */
struct S540 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type PATH */
struct S542 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* internal_name */
	T0* a2; /* storage */
	T1 a3; /* is_normalized */
};

/* Struct for type [attached] DS_HASH_TABLE [[attached] ET_IDENTIFIER, [attached] STRING_8] */
struct S543 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* key_equality_tester */
	T0* a2; /* internal_keys */
	T0* a3; /* hash_function */
	T6 a4; /* position */
	T6 a5; /* last_position */
	T6 a6; /* capacity */
	T6 a7; /* slots_position */
	T6 a8; /* count */
	T0* a9; /* equality_tester */
	T6 a10; /* found_position */
	T6 a11; /* modulus */
	T6 a12; /* clashes_previous_position */
	T0* a13; /* special_item_routines */
	T0* a14; /* item_storage */
	T0* a15; /* clashes */
	T0* a16; /* slots */
	T0* a17; /* special_key_routines */
	T0* a18; /* key_storage */
	T6 a19; /* free_slot */
	T0* a20; /* internal_cursor */
};

/* Struct for type [attached] DS_HASH_SET_CURSOR [[attached] STRING_8] */
struct S544 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* next_cursor */
	T0* a3; /* container */
};

/* Struct for type ARGUMENTS_32 */
struct S547 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type ET_SYSTEM_ERROR */
struct S548 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* default_template */
	T0* a2; /* parameters */
	T0* a3; /* code */
	T0* a4; /* etl_code */
};

/* Struct for type ET_INTERNAL_ERROR */
struct S549 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* default_template */
	T0* a2; /* parameters */
	T0* a3; /* code */
	T0* a4; /* etl_code */
};

/* Struct for type ET_XACE_EXTERNALS */
struct S551 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* c_compiler_options */
	T0* a2; /* include_directories */
	T0* a3; /* link_libraries */
};

/* Struct for type [attached] DS_LINKED_LIST_CURSOR [[attached] STRING_8] */
struct S552 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* after */
	T0* a2; /* current_cell */
	T0* a3; /* next_cursor */
	T0* a4; /* container */
	T1 a5; /* before */
};

/* Struct for type ET_XACE_OPTIONS */
struct S553 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* primary_options */
	T0* a2; /* secondary_options */
	T0* a3; /* primary_multivalue_options */
};

/* Struct for type ET_XACE_TARGET */
struct S554 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* options */
	T0* a2; /* name */
	T0* a3; /* root_class_name */
	T0* a4; /* creation_procedure_name */
	T0* a5; /* clusters */
	T0* a6; /* libraries */
};

/* Struct for type ET_XACE_TARGETS */
struct S557 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* targets */
};

/* Struct for type ET_XACE_CLUSTERS */
struct S559 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* clusters */
};

/* Struct for type ET_XACE_MOUNTED_LIBRARIES */
struct S560 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* libraries */
	T6 a2; /* root_count */
};

/* Struct for type ET_XACE_CLUSTER */
struct S564 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* name */
	T0* a2; /* pathname */
	T1 a3; /* is_abstract */
	T0* a4; /* options */
	T0* a5; /* subclusters */
	T0* a6; /* libraries */
	T1 a7; /* is_relative */
	T0* a8; /* universe */
	T0* a9; /* library_prefix */
	T0* a10; /* cluster_prefix */
	T0* a11; /* class_options */
	T1 a12; /* is_recursive */
	T1 a13; /* is_read_only */
	T1 a14; /* overridden_constraint_enabled */
	T1 a15; /* scm_mapping_constraint_enabled */
	T0* a16; /* scm_read_mapping */
	T0* a17; /* scm_write_mapping */
	T1 a18; /* is_implicit */
	T1 a19; /* use_obsolete_syntax */
	T1 a20; /* is_override */
	T0* a21; /* parent */
	T1 a22; /* is_preparsed */
	T0* a23; /* provider_constraint */
	T0* a24; /* dependant_constraint */
	T1 a25; /* is_mounted */
	T0* a26; /* cached_absolute_pathname */
};

/* Struct for type [attached] DS_LINKED_LIST_CURSOR [[attached] XM_ELEMENT_NODE] */
struct S569 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* after */
	T0* a2; /* current_cell */
	T0* a3; /* container */
	T1 a4; /* before */
	T0* a5; /* next_cursor */
};

/* Struct for type ET_XACE_MOUNTED_LIBRARY */
struct S571 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* pathname */
	T0* a2; /* library_prefix */
	T0* a3; /* library */
	T0* a4; /* position */
};

/* Struct for type ET_XACE_ELEMENT_OBSOLETED_BY_ELEMENT_WARNING */
struct S575 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* parameters */
};

/* Struct for type ET_XACE_MULTIPLE_LIBRARY_PREFIX_ERROR */
struct S576 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* parameters */
};

/* Struct for type ET_ECF_OPTIONS */
struct S581 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* primary_assertions */
	T0* a2; /* primary_debugs */
	T0* a3; /* primary_warnings */
	T0* a4; /* primary_options */
	T0* a5; /* description */
	T0* a6; /* secondary_options */
};

/* Struct for type XM_NAMESPACE */
struct S583 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* uri */
	T0* a2; /* ns_prefix */
};

/* Struct for type [attached] KL_ARRAY_ROUTINES [[attached] INTEGER_32] */
struct S584 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type ET_DYNAMIC_PROCEDURE_TYPE */
struct S586 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* id */
	T0* a2; /* base_class */
	T0* a3; /* base_type */
	T1 a4; /* is_alive */
	T6 a5; /* attribute_count */
	T0* a6; /* queries */
	T0* a7; /* meta_type */
	T6 a8; /* hash_code */
	T0* a9; /* procedures */
	T0* a10; /* set_rout_disp_final_feature */
	T0* a11; /* open_operand_type_sets */
	T0* a12; /* alive_conforming_descendants */
	T0* a13; /* next_type */
	T0* a14; /* query_calls */
	T1 a15; /* has_reference_attributes */
	T1 a16; /* has_generic_expanded_attributes */
	T0* a17; /* conforming_ancestors */
	T0* a18; /* conforming_descendants */
	T1 a19; /* has_static */
	T0* a20; /* procedure_calls */
	T0* a21; /* equality_expressions */
	T0* a22; /* object_equality_expressions */
};

/* Struct for type ET_DYNAMIC_FUNCTION_TYPE */
struct S587 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* id */
	T0* a2; /* base_class */
	T0* a3; /* base_type */
	T1 a4; /* is_alive */
	T6 a5; /* attribute_count */
	T0* a6; /* queries */
	T0* a7; /* meta_type */
	T6 a8; /* hash_code */
	T0* a9; /* procedures */
	T0* a10; /* set_rout_disp_final_feature */
	T0* a11; /* result_type_set */
	T0* a12; /* open_operand_type_sets */
	T0* a13; /* alive_conforming_descendants */
	T0* a14; /* next_type */
	T0* a15; /* query_calls */
	T1 a16; /* has_reference_attributes */
	T1 a17; /* has_generic_expanded_attributes */
	T0* a18; /* conforming_ancestors */
	T0* a19; /* conforming_descendants */
	T1 a20; /* has_static */
	T0* a21; /* procedure_calls */
	T0* a22; /* equality_expressions */
	T0* a23; /* object_equality_expressions */
};

/* Struct for type UT_CONFIG_PARSER */
struct S588 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* has_error */
	T0* a2; /* config_values */
	T0* a3; /* error_handler */
	T0* a4; /* defined_values */
	T0* a5; /* include_stack */
	T0* a6; /* line_nb_stack */
	T0* a7; /* last_string_value */
	T6 a8; /* line_nb */
	T0* a9; /* yyss */
	T0* a10; /* input_buffer */
	T6 a11; /* yy_end */
	T6 a12; /* yy_position */
	T6 a13; /* yy_line */
	T6 a14; /* yy_column */
	T6 a15; /* yy_parsing_status */
	T6 a16; /* yy_suspended_yystacksize */
	T6 a17; /* yy_suspended_yystate */
	T6 a18; /* yy_suspended_yyn */
	T6 a19; /* yy_suspended_yychar1 */
	T6 a20; /* yy_suspended_index */
	T6 a21; /* yy_suspended_yyss_top */
	T6 a22; /* yy_suspended_yy_goto */
	T0* a23; /* yyr1 */
	T6 a24; /* yyssp */
	T0* a25; /* yypgoto */
	T0* a26; /* yycheck */
	T0* a27; /* yytable */
	T0* a28; /* yydefgoto */
	T6 a29; /* error_count */
	T1 a30; /* yy_lookahead_needed */
	T6 a31; /* yyerrstatus */
	T0* a32; /* yypact */
	T6 a33; /* last_token */
	T0* a34; /* yytranslate */
	T0* a35; /* yydefact */
	T0* a36; /* yy_pushed_start_conditions */
	T0* a37; /* yyspecial_routines1 */
	T6 a38; /* yyvsc1 */
	T0* a39; /* yyvs1 */
	T0* a40; /* yyspecial_routines2 */
	T6 a41; /* yyvsc2 */
	T0* a42; /* yyvs2 */
	T0* a43; /* yyspecial_routines3 */
	T6 a44; /* yyvsc3 */
	T0* a45; /* yyvs3 */
	T0* a46; /* yytypes1 */
	T0* a47; /* yytypes2 */
	T6 a48; /* yy_start */
	T6 a49; /* yyvsp1 */
	T6 a50; /* yyvsp2 */
	T6 a51; /* yyvsp3 */
	T1 a52; /* yy_more_flag */
	T6 a53; /* yy_more_len */
	T6 a54; /* line */
	T6 a55; /* column */
	T6 a56; /* position */
	T6 a57; /* yy_start_state */
	T0* a58; /* yy_state_stack */
	T6 a59; /* yy_state_count */
	T0* a60; /* yy_content_area */
	T0* a61; /* yy_ec */
	T0* a62; /* yy_content */
	T0* a63; /* yy_accept */
	T6 a64; /* yy_last_accepting_state */
	T6 a65; /* yy_last_accepting_cpos */
	T0* a66; /* yy_chk */
	T0* a67; /* yy_base */
	T0* a68; /* yy_def */
	T0* a69; /* yy_meta */
	T0* a70; /* yy_nxt */
	T6 a71; /* yy_lp */
	T0* a72; /* yy_acclist */
	T6 a73; /* yy_looking_for_trail_begin */
	T6 a74; /* yy_full_match */
	T6 a75; /* yy_full_state */
	T6 a76; /* yy_full_lp */
	T1 a77; /* yy_rejected */
	T0* a78; /* last_detachable_any_value */
	T6 a79; /* if_level */
	T6 a80; /* ignored_level */
};

/* Struct for type [attached] KL_COMPARABLE_COMPARATOR [[attached] INTEGER_32] */
struct S589 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type ET_DO_PROCEDURE */
struct S592 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* implementation_class */
	T6 a2; /* first_seed */
	T0* a3; /* arguments */
	T0* a4; /* implementation_feature */
	T0* a5; /* extended_name */
	T6 a6; /* hash_code */
	T0* a7; /* end_keyword */
	T0* a8; /* clients */
	T0* a9; /* compound */
	T1 a10; /* validity_checked */
	T1 a11; /* has_validity_error */
	T0* a12; /* preconditions */
	T0* a13; /* first_precursor */
	T0* a14; /* other_precursors */
	T0* a15; /* other_seeds */
	T0* a16; /* obsolete_message */
	T0* a17; /* locals */
	T0* a18; /* postconditions */
	T0* a19; /* rescue_clause */
	T0* a20; /* is_keyword */
	T0* a21; /* semicolon */
	T0* a22; /* feature_clause */
	T0* a23; /* first_indexing */
	T0* a24; /* object_tests */
	T0* a25; /* across_components */
	T0* a26; /* frozen_keyword */
	T0* a27; /* synonym */
	T6 a28; /* id */
	T6 a29; /* version */
};

/* Struct for type ET_DYNAMIC_PUSH_TYPE_SET */
struct S595 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* static_type */
	T1 a2; /* is_never_void */
	T6 a3; /* count */
	T0* a4; /* targets */
	T0* a5; /* dynamic_types */
	T1 a6; /* is_dynamic_types_readonly */
};

/* Struct for type ET_DEFERRED_PROCEDURE */
struct S596 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* implementation_feature */
	T0* a2; /* implementation_class */
	T6 a3; /* first_seed */
	T0* a4; /* arguments */
	T0* a5; /* extended_name */
	T0* a6; /* deferred_keyword */
	T6 a7; /* hash_code */
	T0* a8; /* end_keyword */
	T0* a9; /* clients */
	T1 a10; /* validity_checked */
	T1 a11; /* has_validity_error */
	T0* a12; /* preconditions */
	T0* a13; /* first_precursor */
	T0* a14; /* other_precursors */
	T0* a15; /* other_seeds */
	T0* a16; /* obsolete_message */
	T0* a17; /* postconditions */
	T0* a18; /* is_keyword */
	T0* a19; /* semicolon */
	T0* a20; /* feature_clause */
	T0* a21; /* first_indexing */
	T0* a22; /* object_tests */
	T0* a23; /* across_components */
	T0* a24; /* frozen_keyword */
	T0* a25; /* synonym */
	T6 a26; /* id */
	T6 a27; /* version */
};

/* Struct for type [attached] CELL [[attached] INTEGER_32] */
struct S599 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* item */
};

/* Struct for type [attached] HASH_TABLE [[attached] NATIVE_STRING, [attached] IMMUTABLE_STRING_32] */
struct S601 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* capacity */
	T0* a2; /* content */
	T0* a3; /* keys */
	T0* a4; /* deleted_marks */
	T0* a5; /* indexes_map */
	T6 a6; /* iteration_position */
	T6 a7; /* count */
	T6 a8; /* deleted_item_position */
	T6 a9; /* control */
	T0* a10; /* found_item */
	T1 a11; /* has_default */
	T6 a12; /* item_position */
	T6 a13; /* ht_lowest_deleted_position */
	T0* a14; /* ht_deleted_item */
	T0* a15; /* ht_deleted_key */
	T1 a16; /* object_comparison */
};

/* Struct for type KL_PATHNAME */
struct S603 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* drive */
	T0* a3; /* hostname */
	T0* a4; /* sharename */
	T1 a5; /* is_relative */
	T0* a6; /* components */
};

/* Struct for type EXCEPTIONS */
struct S604 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type READABLE_STRING_32 */
struct S605 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [[attached] NATURAL_8] */
struct S606 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T8 z2[1]; /* item */
};

/* Struct for type [attached] DS_ARRAYED_LIST_CURSOR [[attached] AP_OPTION] */
struct S608 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* next_cursor */
	T0* a3; /* container */
};

/* Struct for type [attached] SPECIAL [[attached] AP_OPTION] */
struct S611 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] AP_OPTION] */
struct S612 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] DS_ARRAYED_LIST_CURSOR [[attached] AP_ALTERNATIVE_OPTIONS_LIST] */
struct S614 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* next_cursor */
	T0* a3; /* container */
};

/* Struct for type [attached] SPECIAL [[attached] AP_ALTERNATIVE_OPTIONS_LIST] */
struct S615 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] AP_ALTERNATIVE_OPTIONS_LIST] */
struct S616 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] DS_ARRAYED_LIST_CURSOR [[attached] STRING_8] */
struct S618 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* next_cursor */
	T0* a3; /* container */
};

/* Struct for type [attached] SPECIAL [detachable STRING_8] */
struct S620 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable STRING_8] */
struct S621 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type attached DS_LINKABLE [[attached] STRING_8] */
struct S623 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* item */
	T0* a2; /* right */
};

/* Struct for type attached DS_LINKABLE [[attached] BOOLEAN] */
struct S626 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* item */
	T0* a2; /* right */
};

/* Struct for type [attached] DS_LINKED_LIST_CURSOR [[attached] BOOLEAN] */
struct S627 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T1 a2; /* before */
};

/* Struct for type attached DS_LINKABLE [[attached] INTEGER_32] */
struct S629 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* item */
	T0* a2; /* right */
};

/* Struct for type [attached] DS_LINKED_LIST_CURSOR [[attached] INTEGER_32] */
struct S630 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T1 a2; /* before */
};

/* Struct for type XM_NAMESPACE_RESOLVER */
struct S632 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* next */
	T0* a2; /* element_local_part */
	T0* a3; /* element_prefix */
	T0* a4; /* context */
	T1 a5; /* forward_xmlns */
	T0* a6; /* attributes_prefix */
	T0* a7; /* attributes_local_part */
	T0* a8; /* attributes_value */
};

/* Struct for type [attached] ARRAY [[attached] XM_CALLBACKS_FILTER] */
struct S633 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* area */
	T6 a2; /* lower */
	T6 a3; /* upper */
};

/* Struct for type [attached] SPECIAL [[attached] XM_CALLBACKS_FILTER] */
struct S634 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type XM_CALLBACKS_NULL */
struct S635 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type XM_POSITION_TABLE */
struct S637 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* table */
};

/* Struct for type [attached] DS_HASH_SET [[attached] XM_NAMESPACE] */
struct S638 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* equality_tester */
	T6 a2; /* capacity */
	T6 a3; /* modulus */
	T6 a4; /* last_position */
	T6 a5; /* free_slot */
	T6 a6; /* position */
	T0* a7; /* special_item_routines */
	T0* a8; /* item_storage */
	T0* a9; /* clashes */
	T0* a10; /* slots */
	T6 a11; /* found_position */
	T0* a12; /* internal_cursor */
	T6 a13; /* slots_position */
	T6 a14; /* count */
	T6 a15; /* clashes_previous_position */
	T0* a16; /* hash_function */
};

/* Struct for type UT_COUNTER */
struct S642 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* item */
};

/* Struct for type [attached] KL_AGENT_ROUTINES [[attached] ET_CLASS] */
struct S643 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] TUPLE [UT_COUNTER] */
struct S644 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type [attached] TUPLE [[attached] KL_AGENT_ROUTINES [[attached] ET_CLASS], [attached] PROCEDURE [[attached] TUPLE]] */
struct S645 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
	T0* z2;
};

/* Struct for type [attached] DS_HASH_SET [[attached] ET_DOTNET_ASSEMBLY] */
struct S647 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* capacity */
	T6 a2; /* modulus */
	T6 a3; /* last_position */
	T6 a4; /* free_slot */
	T6 a5; /* position */
	T6 a6; /* slots_position */
	T6 a7; /* count */
	T0* a8; /* special_item_routines */
	T0* a9; /* item_storage */
	T0* a10; /* clashes */
	T0* a11; /* slots */
	T6 a12; /* found_position */
	T0* a13; /* internal_cursor */
	T6 a14; /* clashes_previous_position */
	T0* a15; /* hash_function */
	T0* a16; /* equality_tester */
};

/* Struct for type ET_DOTNET_ASSEMBLIES */
struct S648 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* dotnet_assemblies */
};

/* Struct for type ET_DOTNET_ASSEMBLY */
struct S649 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* pathname */
	T0* a2; /* referenced_assemblies */
	T0* a3; /* current_system */
	T0* a4; /* dotnet_assembly */
	T0* a5; /* name */
	T0* a6; /* master_classes */
	T0* a7; /* master_class_mutex */
	T0* a8; /* any_type */
	T0* a9; /* detachable_any_type */
	T0* a10; /* detachable_separate_any_type */
	T0* a11; /* separate_any_type */
	T0* a12; /* any_parent */
	T0* a13; /* any_parents */
	T0* a14; /* any_clients */
	T0* a15; /* tuple_type */
	T0* a16; /* detachable_tuple_type */
	T0* a17; /* tuple_like_current_type */
	T0* a18; /* unfolded_empty_tuple_actual_parameters */
	T0* a19; /* array_any_type */
	T0* a20; /* array_detachable_any_type */
	T0* a21; /* array_none_type */
	T0* a22; /* array_like_current_type */
	T0* a23; /* boolean_type */
	T0* a24; /* character_type */
	T0* a25; /* character_8_type */
	T0* a26; /* character_8_convert_feature */
	T0* a27; /* character_32_type */
	T0* a28; /* character_32_convert_feature */
	T0* a29; /* double_type */
	T0* a30; /* exception_type */
	T0* a31; /* detachable_exception_type */
	T0* a32; /* exception_manager_type */
	T0* a33; /* function_type */
	T0* a34; /* immutable_string_32_type */
	T0* a35; /* integer_type */
	T0* a36; /* integer_8_type */
	T0* a37; /* integer_8_convert_feature */
	T0* a38; /* integer_16_type */
	T0* a39; /* integer_16_convert_feature */
	T0* a40; /* integer_32_type */
	T0* a41; /* integer_32_convert_feature */
	T0* a42; /* integer_64_type */
	T0* a43; /* integer_64_convert_feature */
	T0* a44; /* ise_exception_manager_type */
	T0* a45; /* iterable_detachable_any_type */
	T0* a46; /* natural_type */
	T0* a47; /* natural_8_type */
	T0* a48; /* natural_8_convert_feature */
	T0* a49; /* natural_16_type */
	T0* a50; /* natural_16_convert_feature */
	T0* a51; /* natural_32_type */
	T0* a52; /* natural_32_convert_feature */
	T0* a53; /* natural_64_type */
	T0* a54; /* natural_64_convert_feature */
	T0* a55; /* none_type */
	T0* a56; /* detachable_none_type */
	T0* a57; /* pointer_type */
	T0* a58; /* predicate_like_current_type */
	T0* a59; /* procedure_like_current_type */
	T0* a60; /* real_type */
	T0* a61; /* real_32_type */
	T0* a62; /* real_32_convert_feature */
	T0* a63; /* real_64_type */
	T0* a64; /* real_64_convert_feature */
	T0* a65; /* routine_type */
	T0* a66; /* special_any_type */
	T0* a67; /* special_detachable_any_type */
	T0* a68; /* string_type */
	T0* a69; /* detachable_string_type */
	T0* a70; /* string_8_type */
	T0* a71; /* string_8_convert_feature */
	T0* a72; /* string_32_type */
	T0* a73; /* string_32_convert_feature */
	T0* a74; /* system_object_type */
	T0* a75; /* system_object_parents */
	T0* a76; /* system_string_type */
	T0* a77; /* type_any_type */
	T0* a78; /* type_type_any_type */
	T0* a79; /* type_detachable_any_type */
	T0* a80; /* type_detachable_like_current_type */
	T0* a81; /* detachable_type_exception_type */
	T0* a82; /* type_detachable_exception_type */
	T0* a83; /* detachable_type_detachable_any_type */
	T0* a84; /* type_like_current_type */
	T0* a85; /* typed_pointer_any_type */
	T0* a86; /* typed_pointer_like_current_type */
	T0* a87; /* wide_character_type */
	T1 a88; /* is_read_only */
	T1 a89; /* is_preparsed */
};

/* Struct for type [attached] TUPLE [[attached] ET_DOTNET_ASSEMBLY] */
struct S650 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type [attached] PREDICATE [[attached] TUPLE [[attached] ET_DOTNET_ASSEMBLY]] */
struct S651 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T14 a1; /* rout_disp */
	T0* a2; /* closed_operands */
	T14 a3; /* encaps_rout_disp */
	T14 a4; /* calc_rout_addr */
	T1 a5; /* is_target_closed */
	T6 a6; /* open_count */
};

/* Struct for type [attached] PROCEDURE [[attached] TUPLE [[attached] ET_DOTNET_ASSEMBLY]] */
struct S652 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T14 a1; /* rout_disp */
	T0* a2; /* closed_operands */
	T14 a3; /* encaps_rout_disp */
	T14 a4; /* calc_rout_addr */
	T1 a5; /* is_target_closed */
	T6 a6; /* open_count */
};

/* Struct for type [attached] TUPLE [[attached] DS_HASH_SET [[attached] ET_DOTNET_ASSEMBLY]] */
struct S653 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type [attached] TUPLE [[attached] ET_INTERNAL_UNIVERSE] */
struct S654 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type [attached] PROCEDURE [[attached] TUPLE [[attached] ET_INTERNAL_UNIVERSE]] */
struct S655 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T14 a1; /* rout_disp */
	T0* a2; /* closed_operands */
	T14 a3; /* encaps_rout_disp */
	T14 a4; /* calc_rout_addr */
	T1 a5; /* is_target_closed */
	T6 a6; /* open_count */
};

/* Struct for type [attached] TUPLE [[attached] PROCEDURE [[attached] TUPLE [[attached] ET_DOTNET_ASSEMBLY]], [attached] PREDICATE [[attached] TUPLE [[attached] ET_DOTNET_ASSEMBLY]]] */
struct S656 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
	T0* z2;
};

/* Struct for type [attached] TUPLE [ET_DOTNET_ASSEMBLIES] */
struct S657 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type [attached] TUPLE [ET_SYSTEM_PROCESSOR] */
struct S658 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type ET_LIBRARY */
struct S659 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* current_system */
	T0* a2; /* library */
	T0* a3; /* name */
	T0* a4; /* libraries */
	T0* a5; /* clusters */
	T0* a6; /* dotnet_assemblies */
	T0* a7; /* master_classes */
	T0* a8; /* master_class_mutex */
	T0* a9; /* any_type */
	T0* a10; /* detachable_any_type */
	T0* a11; /* detachable_separate_any_type */
	T0* a12; /* separate_any_type */
	T0* a13; /* any_parent */
	T0* a14; /* any_parents */
	T0* a15; /* any_clients */
	T0* a16; /* tuple_type */
	T0* a17; /* detachable_tuple_type */
	T0* a18; /* tuple_like_current_type */
	T0* a19; /* unfolded_empty_tuple_actual_parameters */
	T0* a20; /* array_any_type */
	T0* a21; /* array_detachable_any_type */
	T0* a22; /* array_none_type */
	T0* a23; /* array_like_current_type */
	T0* a24; /* boolean_type */
	T0* a25; /* character_type */
	T0* a26; /* character_8_type */
	T0* a27; /* character_8_convert_feature */
	T0* a28; /* character_32_type */
	T0* a29; /* character_32_convert_feature */
	T0* a30; /* double_type */
	T0* a31; /* exception_type */
	T0* a32; /* detachable_exception_type */
	T0* a33; /* exception_manager_type */
	T0* a34; /* function_type */
	T0* a35; /* immutable_string_32_type */
	T0* a36; /* integer_type */
	T0* a37; /* integer_8_type */
	T0* a38; /* integer_8_convert_feature */
	T0* a39; /* integer_16_type */
	T0* a40; /* integer_16_convert_feature */
	T0* a41; /* integer_32_type */
	T0* a42; /* integer_32_convert_feature */
	T0* a43; /* integer_64_type */
	T0* a44; /* integer_64_convert_feature */
	T0* a45; /* ise_exception_manager_type */
	T0* a46; /* iterable_detachable_any_type */
	T0* a47; /* natural_type */
	T0* a48; /* natural_8_type */
	T0* a49; /* natural_8_convert_feature */
	T0* a50; /* natural_16_type */
	T0* a51; /* natural_16_convert_feature */
	T0* a52; /* natural_32_type */
	T0* a53; /* natural_32_convert_feature */
	T0* a54; /* natural_64_type */
	T0* a55; /* natural_64_convert_feature */
	T0* a56; /* none_type */
	T0* a57; /* detachable_none_type */
	T0* a58; /* pointer_type */
	T0* a59; /* predicate_like_current_type */
	T0* a60; /* procedure_like_current_type */
	T0* a61; /* real_type */
	T0* a62; /* real_32_type */
	T0* a63; /* real_32_convert_feature */
	T0* a64; /* real_64_type */
	T0* a65; /* real_64_convert_feature */
	T0* a66; /* routine_type */
	T0* a67; /* special_any_type */
	T0* a68; /* special_detachable_any_type */
	T0* a69; /* string_type */
	T0* a70; /* detachable_string_type */
	T0* a71; /* string_8_type */
	T0* a72; /* string_8_convert_feature */
	T0* a73; /* string_32_type */
	T0* a74; /* string_32_convert_feature */
	T0* a75; /* system_object_type */
	T0* a76; /* system_object_parents */
	T0* a77; /* system_string_type */
	T0* a78; /* type_any_type */
	T0* a79; /* type_type_any_type */
	T0* a80; /* type_detachable_any_type */
	T0* a81; /* type_detachable_like_current_type */
	T0* a82; /* detachable_type_exception_type */
	T0* a83; /* type_detachable_exception_type */
	T0* a84; /* detachable_type_detachable_any_type */
	T0* a85; /* type_like_current_type */
	T0* a86; /* typed_pointer_any_type */
	T0* a87; /* typed_pointer_like_current_type */
	T0* a88; /* wide_character_type */
	T1 a89; /* is_read_only */
	T1 a90; /* is_preparsed */
};

/* Struct for type [attached] TUPLE [[attached] ET_LIBRARY] */
struct S660 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type [attached] PROCEDURE [[attached] TUPLE [[attached] ET_LIBRARY]] */
struct S661 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T14 a1; /* rout_disp */
	T0* a2; /* closed_operands */
	T14 a3; /* encaps_rout_disp */
	T14 a4; /* calc_rout_addr */
	T1 a5; /* is_target_closed */
	T6 a6; /* open_count */
};

/* Struct for type [attached] TUPLE [[attached] ET_UNIVERSE] */
struct S662 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type [attached] PROCEDURE [[attached] TUPLE [[attached] ET_UNIVERSE]] */
struct S663 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T14 a1; /* rout_disp */
	T0* a2; /* closed_operands */
	T14 a3; /* encaps_rout_disp */
	T14 a4; /* calc_rout_addr */
	T1 a5; /* is_target_closed */
	T6 a6; /* open_count */
};

/* Struct for type [attached] TUPLE [[attached] PROCEDURE [[attached] TUPLE [[attached] ET_CLASS]]] */
struct S664 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type [attached] TUPLE [[attached] PROCEDURE [[attached] TUPLE [[attached] ET_CLASS]], detachable FUNCTION [[attached] TUPLE, [attached] BOOLEAN]] */
struct S665 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
	T0* z2;
};

/* Struct for type [attached] DS_ARRAYED_LIST [[attached] ET_XACE_MOUNTED_LIBRARY] */
struct S667 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
	T0* a3; /* special_routines */
	T6 a4; /* capacity */
	T0* a5; /* internal_cursor */
};

/* Struct for type ET_XACE_LIBRARY_CONFIG */
struct S668 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* targets */
	T0* a2; /* library_prefix */
	T0* a3; /* name */
	T0* a4; /* selected_target */
	T0* a5; /* options */
	T0* a6; /* clusters */
	T0* a7; /* libraries */
};

/* Struct for type [attached] TUPLE [[attached] ET_XACE_CLUSTER] */
struct S669 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type [attached] PROCEDURE [[attached] TUPLE [[attached] ET_XACE_CLUSTER]] */
struct S670 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T14 a1; /* rout_disp */
	T0* a2; /* closed_operands */
	T14 a3; /* encaps_rout_disp */
	T14 a4; /* calc_rout_addr */
	T1 a5; /* is_target_closed */
	T6 a6; /* open_count */
};

/* Struct for type [attached] TUPLE [ET_XACE_CLUSTERS] */
struct S671 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type ET_MASTER_CLASS */
struct S672 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* name */
	T0* a2; /* mapped_class */
	T0* a3; /* first_overriding_class */
	T0* a4; /* intrinsic_class */
	T0* a5; /* universe */
	T0* a6; /* other_local_override_classes */
	T0* a7; /* other_local_non_override_classes */
	T0* a8; /* other_local_ignored_classes */
	T0* a9; /* other_local_hidden_classes */
	T0* a10; /* other_imported_classes */
	T0* a11; /* other_overriding_classes */
	T0* a12; /* status_mutex */
	T0* a13; /* processing_mutex */
	T1 a14; /* unprotected_is_marked */
	T1 a15; /* is_modified */
	T0* a16; /* first_imported_class */
	T0* a17; /* first_local_ignored_class */
	T0* a18; /* first_local_override_class */
	T0* a19; /* first_local_non_override_class */
	T0* a20; /* first_local_hidden_class */
};

/* Struct for type [attached] TUPLE [[attached] ET_CLUSTER] */
struct S674 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type [attached] PROCEDURE [[attached] TUPLE [[attached] ET_CLUSTER]] */
struct S675 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T14 a1; /* rout_disp */
	T0* a2; /* closed_operands */
	T14 a3; /* encaps_rout_disp */
	T14 a4; /* calc_rout_addr */
	T1 a5; /* is_target_closed */
	T6 a6; /* open_count */
};

/* Struct for type [attached] TUPLE [ET_XACE_SYSTEM] */
struct S676 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type [attached] DS_HASH_SET [[attached] ET_INTERNAL_UNIVERSE] */
struct S677 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T6 a2; /* capacity */
	T6 a3; /* modulus */
	T6 a4; /* last_position */
	T6 a5; /* free_slot */
	T6 a6; /* slots_position */
	T6 a7; /* count */
	T6 a8; /* clashes_previous_position */
	T0* a9; /* special_item_routines */
	T0* a10; /* item_storage */
	T0* a11; /* clashes */
	T0* a12; /* slots */
	T6 a13; /* found_position */
	T0* a14; /* internal_cursor */
	T0* a15; /* equality_tester */
	T0* a16; /* hash_function */
};

/* Struct for type [attached] TUPLE [ET_AST_PROCESSOR] */
struct S678 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type [attached] TUPLE [[attached] ET_MASTER_CLASS] */
struct S679 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type [attached] PROCEDURE [[attached] TUPLE [[attached] ET_MASTER_CLASS]] */
struct S680 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T14 a1; /* rout_disp */
	T0* a2; /* closed_operands */
	T14 a3; /* encaps_rout_disp */
	T14 a4; /* calc_rout_addr */
	T1 a5; /* is_target_closed */
	T6 a6; /* open_count */
};

/* Struct for type [attached] TUPLE [[attached] ET_ADAPTED_DOTNET_ASSEMBLY] */
struct S681 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type [attached] PROCEDURE [[attached] TUPLE [[attached] ET_ADAPTED_DOTNET_ASSEMBLY]] */
struct S682 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T14 a1; /* rout_disp */
	T0* a2; /* closed_operands */
	T14 a3; /* encaps_rout_disp */
	T14 a4; /* calc_rout_addr */
	T1 a5; /* is_target_closed */
	T6 a6; /* open_count */
};

/* Struct for type ET_ANCESTORS_STATUS_CHECKER */
struct S683 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* class_type_checker */
	T0* a2; /* current_class */
	T0* a3; /* system_processor */
};

/* Struct for type ET_FLATTENING_STATUS_CHECKER */
struct S684 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* class_type_checker */
	T0* a2; /* current_class */
	T0* a3; /* system_processor */
};

/* Struct for type ET_INTERFACE_STATUS_CHECKER */
struct S685 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* qualified_anchored_type_checker */
	T0* a2; /* class_type_checker */
	T0* a3; /* current_class */
	T0* a4; /* system_processor */
};

/* Struct for type ET_IMPLEMENTATION_STATUS_CHECKER */
struct S686 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* current_class */
	T0* a2; /* system_processor */
};

/* Struct for type [attached] PREDICATE [[attached] TUPLE [[attached] ET_CLASS]] */
struct S687 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T14 a1; /* rout_disp */
	T0* a2; /* closed_operands */
	T14 a3; /* encaps_rout_disp */
	T14 a4; /* calc_rout_addr */
	T1 a5; /* is_target_closed */
	T6 a6; /* open_count */
};

/* Struct for type [attached] TUPLE [ET_ANCESTORS_STATUS_CHECKER] */
struct S689 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type [attached] TUPLE [ET_FLATTENING_STATUS_CHECKER] */
struct S690 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type [attached] TUPLE [ET_INTERFACE_STATUS_CHECKER] */
struct S691 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type [attached] TUPLE [ET_IMPLEMENTATION_STATUS_CHECKER] */
struct S692 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type [attached] TUPLE [BOOLEAN] */
struct S693 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 z1;
};

/* Struct for type [attached] DS_HASH_SET [[attached] ET_UNIVERSE] */
struct S694 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* internal_cursor */
	T6 a2; /* position */
	T0* a3; /* item_storage */
	T6 a4; /* capacity */
	T6 a5; /* modulus */
	T6 a6; /* last_position */
	T6 a7; /* free_slot */
	T6 a8; /* slots_position */
	T6 a9; /* count */
	T6 a10; /* clashes_previous_position */
	T0* a11; /* special_item_routines */
	T0* a12; /* clashes */
	T0* a13; /* slots */
	T6 a14; /* found_position */
	T0* a15; /* hash_function */
	T0* a16; /* equality_tester */
};

/* Struct for type ET_CLASS_NAME_TESTER */
struct S696 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type ET_PARENT */
struct S698 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* type */
	T0* a2; /* renames */
	T0* a3; /* exports */
	T0* a4; /* undefines */
	T0* a5; /* redefines */
	T0* a6; /* selects */
	T0* a7; /* end_keyword */
};

/* Struct for type ET_PARENT_LIST */
struct S699 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* clients_clause */
	T0* a3; /* storage */
	T0* a4; /* inherit_keyword */
};

/* Struct for type ET_CLIENT_LIST */
struct S700 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
};

/* Struct for type ET_BUILTIN_CONVERT_FEATURE */
struct S701 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* name */
	T0* a2; /* types */
	T0* a3; /* type */
};

/* Struct for type [attached] TUPLE [[attached] PROCEDURE [[attached] TUPLE [[attached] ET_CLUSTER]]] */
struct S706 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type [attached] TUPLE [[attached] DS_HASH_SET [[attached] ET_INTERNAL_UNIVERSE]] */
struct S712 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type [attached] TUPLE [[attached] PROCEDURE [[attached] TUPLE [[attached] ET_MASTER_CLASS]]] */
struct S713 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type [attached] TUPLE [[attached] PROCEDURE [[attached] TUPLE [[attached] ET_CLASS]], [attached] FUNCTION [[attached] TUPLE [[attached] ET_CLASS], [attached] BOOLEAN]] */
struct S714 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
	T0* z2;
};

/* Struct for type [attached] TUPLE [[attached] DS_HASH_SET [[attached] ET_UNIVERSE]] */
struct S715 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type ET_CLIENT */
struct S716 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* name */
	T0* a2; /* named_base_class */
};

/* Struct for type ET_ATTACHMENT_SEPARATE_KEYWORDS */
struct S717 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* attachment_keyword */
	T0* a2; /* separateness_keyword */
};

/* Struct for type ET_NONE_GROUP */
struct S723 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* universe */
	T0* a2; /* name */
	T1 a3; /* use_obsolete_syntax */
	T0* a4; /* cached_absolute_pathname */
};

/* Struct for type ET_UNFOLDED_EMPTY_TUPLE_ACTUAL_PARAMETERS */
struct S724 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* tuple_type */
};

/* Struct for type [attached] DS_HASH_TABLE_CURSOR [[attached] ET_XACE_LIBRARY_CONFIG, [attached] STRING_8] */
struct S729 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* next_cursor */
	T0* a2; /* container */
	T6 a3; /* position */
};

/* Struct for type [attached] SPECIAL [[attached] ET_XACE_LIBRARY_CONFIG] */
struct S730 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_XACE_LIBRARY_CONFIG] */
struct S731 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type XM_EIFFEL_SCANNER */
struct S732 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* last_token */
	T0* a2; /* last_value */
	T0* a3; /* input_name */
	T6 a4; /* position */
	T6 a5; /* column */
	T6 a6; /* line */
	T0* a7; /* input_filter */
	T6 a8; /* yy_start_state */
	T0* a9; /* character_entity */
	T0* a10; /* input_stream */
	T0* a11; /* input_resolver */
	T1 a12; /* yy_more_flag */
	T6 a13; /* yy_more_len */
	T6 a14; /* yy_end */
	T6 a15; /* yy_start */
	T6 a16; /* yy_line */
	T6 a17; /* yy_column */
	T6 a18; /* yy_position */
	T0* a19; /* input_buffer */
	T0* a20; /* yy_state_stack */
	T6 a21; /* yy_state_count */
	T0* a22; /* yy_content_area */
	T0* a23; /* yy_ec */
	T0* a24; /* yy_content */
	T0* a25; /* yy_accept */
	T6 a26; /* yy_last_accepting_state */
	T6 a27; /* yy_last_accepting_cpos */
	T0* a28; /* yy_chk */
	T0* a29; /* yy_base */
	T0* a30; /* yy_def */
	T0* a31; /* yy_meta */
	T0* a32; /* yy_nxt */
	T6 a33; /* yy_lp */
	T0* a34; /* yy_acclist */
	T6 a35; /* yy_looking_for_trail_begin */
	T6 a36; /* yy_full_match */
	T6 a37; /* yy_full_state */
	T6 a38; /* yy_full_lp */
	T1 a39; /* yy_rejected */
	T0* a40; /* yy_pushed_start_conditions */
	T0* a41; /* last_string_value */
	T0* a42; /* last_error */
	T6 a43; /* pushed_start_condition_count */
};

/* Struct for type XM_DEFAULT_POSITION */
struct S733 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* row */
	T6 a2; /* column */
	T0* a3; /* source_name */
	T6 a4; /* byte_index */
};

/* Struct for type [attached] DS_BILINKED_LIST [[attached] XM_POSITION] */
struct S736 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* first_cell */
	T0* a2; /* last_cell */
	T6 a3; /* count */
	T0* a4; /* internal_cursor */
};

/* Struct for type [attached] DS_LINKED_STACK [[attached] XM_EIFFEL_SCANNER] */
struct S737 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* first_cell */
};

/* Struct for type [attached] DS_HASH_TABLE [[attached] XM_EIFFEL_ENTITY_DEF, [attached] STRING_8] */
struct S738 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* item_storage */
	T6 a3; /* count */
	T6 a4; /* last_position */
	T6 a5; /* free_slot */
	T6 a6; /* capacity */
	T0* a7; /* key_equality_tester */
	T0* a8; /* internal_keys */
	T6 a9; /* modulus */
	T6 a10; /* slots_position */
	T6 a11; /* clashes_previous_position */
	T0* a12; /* internal_cursor */
	T6 a13; /* found_position */
	T0* a14; /* key_storage */
	T0* a15; /* clashes */
	T0* a16; /* slots */
	T0* a17; /* special_item_routines */
	T0* a18; /* special_key_routines */
	T0* a19; /* equality_tester */
	T0* a20; /* hash_function */
};

/* Struct for type XM_NULL_EXTERNAL_RESOLVER */
struct S740 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type XM_DTD_CALLBACKS_NULL */
struct S742 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] XM_EIFFEL_PARSER_NAME] */
struct S743 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type XM_EIFFEL_PARSER_NAME */
struct S744 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* use_namespaces */
	T6 a2; /* count */
	T0* a3; /* first */
	T0* a4; /* second */
	T0* a5; /* tail */
};

/* Struct for type [attached] SPECIAL [[attached] XM_EIFFEL_PARSER_NAME] */
struct S745 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] DS_HASH_SET [[attached] XM_EIFFEL_PARSER_NAME]] */
struct S746 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] DS_HASH_SET [[attached] XM_EIFFEL_PARSER_NAME] */
struct S747 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* equality_tester */
	T6 a3; /* count */
	T6 a4; /* capacity */
	T6 a5; /* free_slot */
	T6 a6; /* last_position */
	T6 a7; /* modulus */
	T6 a8; /* slots_position */
	T6 a9; /* clashes_previous_position */
	T6 a10; /* found_position */
	T0* a11; /* clashes */
	T0* a12; /* slots */
	T0* a13; /* special_item_routines */
	T0* a14; /* item_storage */
	T0* a15; /* internal_cursor */
	T0* a16; /* hash_function */
};

/* Struct for type [attached] SPECIAL [[attached] DS_HASH_SET [[attached] XM_EIFFEL_PARSER_NAME]] */
struct S748 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] XM_DTD_EXTERNAL_ID] */
struct S749 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type XM_DTD_EXTERNAL_ID */
struct S750 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* system_id */
	T0* a2; /* public_id */
};

/* Struct for type [attached] SPECIAL [[attached] XM_DTD_EXTERNAL_ID] */
struct S751 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] XM_DTD_ELEMENT_CONTENT] */
struct S752 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type XM_DTD_ELEMENT_CONTENT */
struct S753 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* items */
	T2 a2; /* repetition */
	T0* a3; /* name */
	T2 a4; /* type */
	T1 a5; /* is_character_data_allowed */
};

/* Struct for type [attached] SPECIAL [[attached] XM_DTD_ELEMENT_CONTENT] */
struct S754 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] DS_BILINKED_LIST [[attached] XM_DTD_ATTRIBUTE_CONTENT]] */
struct S755 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] DS_BILINKED_LIST [[attached] XM_DTD_ATTRIBUTE_CONTENT] */
struct S756 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* last_cell */
	T6 a2; /* count */
	T0* a3; /* first_cell */
	T0* a4; /* internal_cursor */
};

/* Struct for type [attached] SPECIAL [[attached] DS_BILINKED_LIST [[attached] XM_DTD_ATTRIBUTE_CONTENT]] */
struct S757 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] XM_DTD_ATTRIBUTE_CONTENT] */
struct S758 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type XM_DTD_ATTRIBUTE_CONTENT */
struct S759 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* name */
	T0* a2; /* default_value */
	T2 a3; /* type */
	T1 a4; /* is_list_type */
	T0* a5; /* enumeration_list */
	T2 a6; /* value */
};

/* Struct for type [attached] SPECIAL [[attached] XM_DTD_ATTRIBUTE_CONTENT] */
struct S760 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] DS_BILINKED_LIST [[attached] STRING_8]] */
struct S761 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] DS_BILINKED_LIST [[attached] STRING_8] */
struct S762 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* last_cell */
	T6 a2; /* count */
	T0* a3; /* first_cell */
	T0* a4; /* equality_tester */
	T0* a5; /* internal_cursor */
};

/* Struct for type [attached] SPECIAL [[attached] DS_BILINKED_LIST [[attached] STRING_8]] */
struct S763 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] BOOLEAN] */
struct S764 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] XM_EIFFEL_DECLARATION] */
struct S765 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type XM_EIFFEL_DECLARATION */
struct S766 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* encoding */
	T0* a2; /* version */
	T1 a3; /* stand_alone */
};

/* Struct for type [attached] SPECIAL [[attached] XM_EIFFEL_DECLARATION] */
struct S767 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type XM_EIFFEL_ENTITY_DEF */
struct S772 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* literal_name */
	T0* a2; /* value */
	T0* a3; /* resolver */
	T0* a4; /* external_id */
	T0* a5; /* character_entity */
	T0* a6; /* yy_pushed_start_conditions */
	T0* a7; /* input_buffer */
	T1 a8; /* in_use */
	T0* a9; /* input_name */
	T0* a10; /* last_value */
	T0* a11; /* last_string_value */
	T0* a12; /* last_error */
	T1 a13; /* yy_rejected */
	T6 a14; /* yy_state_count */
	T6 a15; /* yy_full_match */
	T6 a16; /* yy_lp */
	T6 a17; /* yy_looking_for_trail_begin */
	T6 a18; /* yy_full_lp */
	T6 a19; /* yy_full_state */
	T6 a20; /* pushed_start_condition_count */
	T6 a21; /* yy_start_state */
	T6 a22; /* yy_line */
	T6 a23; /* yy_column */
	T6 a24; /* yy_position */
	T6 a25; /* yy_start */
	T6 a26; /* yy_end */
	T6 a27; /* line */
	T6 a28; /* column */
	T6 a29; /* position */
	T1 a30; /* yy_more_flag */
	T6 a31; /* yy_more_len */
	T6 a32; /* yy_last_accepting_state */
	T6 a33; /* yy_last_accepting_cpos */
	T0* a34; /* yy_state_stack */
	T0* a35; /* yy_nxt */
	T0* a36; /* yy_chk */
	T0* a37; /* yy_base */
	T0* a38; /* yy_def */
	T0* a39; /* yy_ec */
	T0* a40; /* yy_meta */
	T0* a41; /* yy_accept */
	T0* a42; /* yy_content */
	T0* a43; /* yy_content_area */
	T6 a44; /* last_token */
	T0* a45; /* input_filter */
	T0* a46; /* input_stream */
	T0* a47; /* input_resolver */
	T0* a48; /* yy_acclist */
};

/* Struct for type XM_EIFFEL_SCANNER_DTD */
struct S773 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* last_token */
	T0* a2; /* last_value */
	T0* a3; /* input_name */
	T6 a4; /* position */
	T6 a5; /* column */
	T6 a6; /* line */
	T0* a7; /* input_filter */
	T6 a8; /* yy_start_state */
	T0* a9; /* character_entity */
	T0* a10; /* input_stream */
	T0* a11; /* input_resolver */
	T1 a12; /* decl_start_sent */
	T1 a13; /* decl_end_sent */
	T1 a14; /* yy_more_flag */
	T6 a15; /* yy_more_len */
	T6 a16; /* yy_end */
	T6 a17; /* yy_start */
	T6 a18; /* yy_line */
	T6 a19; /* yy_column */
	T6 a20; /* yy_position */
	T0* a21; /* input_buffer */
	T0* a22; /* yy_state_stack */
	T6 a23; /* yy_state_count */
	T0* a24; /* yy_content_area */
	T0* a25; /* yy_ec */
	T0* a26; /* yy_content */
	T0* a27; /* yy_accept */
	T6 a28; /* yy_last_accepting_state */
	T6 a29; /* yy_last_accepting_cpos */
	T0* a30; /* yy_chk */
	T0* a31; /* yy_base */
	T0* a32; /* yy_def */
	T0* a33; /* yy_meta */
	T0* a34; /* yy_nxt */
	T6 a35; /* yy_lp */
	T0* a36; /* yy_acclist */
	T6 a37; /* yy_looking_for_trail_begin */
	T6 a38; /* yy_full_match */
	T6 a39; /* yy_full_state */
	T6 a40; /* yy_full_lp */
	T1 a41; /* yy_rejected */
	T0* a42; /* yy_pushed_start_conditions */
	T0* a43; /* last_string_value */
	T0* a44; /* last_error */
	T6 a45; /* pushed_start_condition_count */
};

/* Struct for type XM_EIFFEL_PE_ENTITY_DEF */
struct S775 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* resolver */
	T0* a2; /* external_id */
	T0* a3; /* literal_name */
	T0* a4; /* value */
	T0* a5; /* character_entity */
	T0* a6; /* yy_pushed_start_conditions */
	T0* a7; /* input_buffer */
	T1 a8; /* in_use */
	T0* a9; /* input_name */
	T0* a10; /* last_value */
	T0* a11; /* last_string_value */
	T0* a12; /* last_error */
	T1 a13; /* yy_rejected */
	T6 a14; /* yy_state_count */
	T6 a15; /* yy_full_match */
	T6 a16; /* yy_lp */
	T6 a17; /* yy_looking_for_trail_begin */
	T6 a18; /* yy_full_lp */
	T6 a19; /* yy_full_state */
	T6 a20; /* pushed_start_condition_count */
	T6 a21; /* yy_start_state */
	T6 a22; /* yy_line */
	T6 a23; /* yy_column */
	T6 a24; /* yy_position */
	T6 a25; /* yy_start */
	T6 a26; /* yy_end */
	T6 a27; /* line */
	T6 a28; /* column */
	T6 a29; /* position */
	T1 a30; /* yy_more_flag */
	T6 a31; /* yy_more_len */
	T6 a32; /* yy_last_accepting_state */
	T6 a33; /* yy_last_accepting_cpos */
	T0* a34; /* yy_state_stack */
	T1 a35; /* pre_sent */
	T1 a36; /* post_sent */
	T0* a37; /* yy_nxt */
	T0* a38; /* yy_chk */
	T0* a39; /* yy_base */
	T0* a40; /* yy_def */
	T0* a41; /* yy_ec */
	T0* a42; /* yy_meta */
	T0* a43; /* yy_accept */
	T0* a44; /* yy_content */
	T0* a45; /* yy_content_area */
	T6 a46; /* last_token */
	T0* a47; /* input_filter */
	T0* a48; /* input_stream */
	T0* a49; /* input_resolver */
	T0* a50; /* yy_acclist */
};

/* Struct for type [attached] DS_HASH_TABLE_CURSOR [[attached] ET_ECF_LIBRARY, [attached] STRING_8] */
struct S777 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* next_cursor */
	T0* a3; /* container */
};

/* Struct for type [attached] SPECIAL [[attached] ET_ECF_LIBRARY] */
struct S778 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_ECF_LIBRARY] */
struct S783 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [[attached] ET_ECF_DOTNET_ASSEMBLY] */
struct S784 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] DS_HASH_TABLE_CURSOR [[attached] ET_ECF_DOTNET_ASSEMBLY, [attached] STRING_8] */
struct S789 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* next_cursor */
	T0* a2; /* container */
	T6 a3; /* position */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_ECF_DOTNET_ASSEMBLY] */
struct S790 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type ET_CLUSTERS */
struct S791 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* clusters */
};

/* Struct for type [attached] DS_ARRAYED_LIST [[attached] ET_ADAPTED_LIBRARY] */
struct S792 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
	T0* a3; /* special_routines */
	T6 a4; /* capacity */
	T0* a5; /* internal_cursor */
};

/* Struct for type [attached] DS_HASH_SET [[attached] ET_LIBRARY] */
struct S793 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* capacity */
	T6 a2; /* modulus */
	T6 a3; /* last_position */
	T6 a4; /* free_slot */
	T6 a5; /* position */
	T0* a6; /* special_item_routines */
	T0* a7; /* item_storage */
	T0* a8; /* clashes */
	T0* a9; /* slots */
	T6 a10; /* found_position */
	T0* a11; /* internal_cursor */
	T6 a12; /* slots_position */
	T6 a13; /* count */
	T6 a14; /* clashes_previous_position */
	T0* a15; /* hash_function */
	T0* a16; /* equality_tester */
};

/* Struct for type [attached] TUPLE [[attached] DS_HASH_SET [[attached] ET_LIBRARY]] */
struct S794 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type [attached] SPECIAL [[attached] ET_ECF_SYSTEM_CONFIG] */
struct S796 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_ECF_SYSTEM_CONFIG] */
struct S798 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] DS_HASH_TABLE_CURSOR [[attached] ET_ECF_SYSTEM_CONFIG, [attached] STRING_8] */
struct S799 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
};

/* Struct for type [attached] DS_HASH_TABLE_CURSOR [[attached] ET_ECF_TARGET, [attached] STRING_8] */
struct S800 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* next_cursor */
	T0* a3; /* container */
};

/* Struct for type [attached] SPECIAL [[attached] ET_ECF_TARGET] */
struct S801 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_ECF_TARGET] */
struct S803 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type attached DS_SPARSE_TABLE_KEYS [detachable RX_REGULAR_EXPRESSION, [attached] STRING_8] */
struct S807 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* table */
	T0* a2; /* equality_tester */
	T0* a3; /* internal_cursor */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable RX_REGULAR_EXPRESSION] */
struct S808 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable RX_REGULAR_EXPRESSION] */
struct S809 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] DS_ARRAYED_LIST [[attached] ET_ADAPTED_DOTNET_ASSEMBLY] */
struct S810 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
	T0* a3; /* special_routines */
	T6 a4; /* capacity */
	T0* a5; /* internal_cursor */
};

/* Struct for type [attached] KL_EQUALITY_TESTER [[attached] ET_IDENTIFIER] */
struct S812 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] DS_HASH_TABLE_CURSOR [[attached] ET_LACE_CLUSTER, [attached] ET_IDENTIFIER] */
struct S815 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* next_cursor */
	T0* a2; /* container */
	T6 a3; /* position */
};

/* Struct for type [attached] DS_HASH_SET_CURSOR [[attached] ET_IDENTIFIER] */
struct S816 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* next_cursor */
	T0* a3; /* container */
};

/* Struct for type [attached] TUPLE [ET_LACE_SYSTEM] */
struct S819 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type [attached] DS_ARRAYED_LIST [[attached] ET_LACE_CLUSTER] */
struct S820 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
	T0* a3; /* special_routines */
	T6 a4; /* capacity */
	T0* a5; /* internal_cursor */
};

/* Struct for type [attached] TUPLE [[attached] ET_LACE_CLUSTER] */
struct S821 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type [attached] DS_HASH_TABLE_CURSOR [[attached] ET_MASTER_CLASS, [attached] ET_CLASS_NAME] */
struct S828 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
	T0* a3; /* next_cursor */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_MASTER_CLASS] */
struct S829 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [[attached] ET_MASTER_CLASS] */
struct S830 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_CLASS_NAME] */
struct S831 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [[attached] ET_CLASS_NAME] */
struct S832 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] SPECIAL [[attached] ET_DYNAMIC_TYPE] */
struct S833 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_DYNAMIC_TYPE] */
struct S834 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_DYNAMIC_TYPE] */
struct S835 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* next_cursor */
	T0* a2; /* container */
	T6 a3; /* position */
};

/* Struct for type [attached] SPECIAL [[attached] ET_ACTUAL_PARAMETER_ITEM] */
struct S845 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type ET_BRACKET_SYMBOL */
struct S846 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T2 a1; /* code */
	T6 a2; /* compressed_position */
	T6 a3; /* seed */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_ACTUAL_PARAMETER_ITEM] */
struct S848 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type C_DATE */
struct S850 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* millisecond_now */
	T0* a2; /* internal_item */
	T1 a3; /* is_utc */
};

/* Struct for type KL_STDIN_FILE */
struct S851 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* end_of_file */
	T2 a2; /* last_character */
	T0* a3; /* last_string */
	T0* a4; /* character_buffer */
};

/* Struct for type [attached] DS_ARRAYED_STACK [[attached] INTEGER_32] */
struct S853 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* storage */
	T6 a2; /* count */
	T0* a3; /* special_routines */
	T6 a4; /* capacity */
};

/* Struct for type [attached] DS_ARRAYED_STACK [detachable ET_FORMAL_ARGUMENT_LIST] */
struct S854 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
	T0* a3; /* special_routines */
	T6 a4; /* capacity */
};

/* Struct for type [attached] DS_ARRAYED_STACK [detachable ET_LOCAL_VARIABLE_LIST] */
struct S855 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
	T0* a3; /* special_routines */
	T6 a4; /* capacity */
};

/* Struct for type [attached] DS_ARRAYED_STACK [detachable ET_KEYWORD] */
struct S856 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* special_routines */
	T0* a2; /* storage */
	T6 a3; /* capacity */
	T6 a4; /* count */
};

/* Struct for type [attached] DS_ARRAYED_STACK [detachable ET_SYMBOL] */
struct S857 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* special_routines */
	T0* a2; /* storage */
	T6 a3; /* capacity */
	T6 a4; /* count */
};

/* Struct for type [attached] DS_ARRAYED_STACK [detachable ET_OBJECT_TEST_LIST] */
struct S858 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
	T0* a3; /* special_routines */
	T6 a4; /* capacity */
};

/* Struct for type [attached] DS_ARRAYED_STACK [[attached] ET_OBJECT_TEST_LIST] */
struct S859 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* special_routines */
	T0* a2; /* storage */
	T6 a3; /* capacity */
	T6 a4; /* count */
};

/* Struct for type [attached] DS_ARRAYED_STACK [detachable ET_ACROSS_COMPONENT_LIST] */
struct S860 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
	T0* a3; /* special_routines */
	T6 a4; /* capacity */
};

/* Struct for type [attached] DS_ARRAYED_STACK [[attached] ET_ACROSS_COMPONENT_LIST] */
struct S861 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* special_routines */
	T0* a2; /* storage */
	T6 a3; /* capacity */
	T6 a4; /* count */
};

/* Struct for type [attached] DS_ARRAYED_LIST [[attached] ET_ASSERTION_ITEM] */
struct S862 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
	T0* a3; /* special_routines */
	T6 a4; /* capacity */
	T0* a5; /* internal_cursor */
};

/* Struct for type [attached] DS_ARRAYED_LIST [[attached] ET_QUERY] */
struct S863 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
	T0* a3; /* internal_cursor */
	T0* a4; /* special_routines */
	T6 a5; /* capacity */
};

/* Struct for type [attached] DS_ARRAYED_LIST [[attached] ET_PROCEDURE] */
struct S864 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
	T0* a3; /* internal_cursor */
	T0* a4; /* special_routines */
	T6 a5; /* capacity */
};

/* Struct for type [attached] DS_ARRAYED_LIST [detachable ET_CONSTRAINT_TYPE] */
struct S865 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* storage */
	T0* a2; /* special_routines */
	T6 a3; /* capacity */
	T6 a4; /* count */
	T0* a5; /* internal_cursor */
};

/* Struct for type [attached] DS_HASH_SET [[attached] ET_NAMED_CLASS] */
struct S866 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* internal_cursor */
	T0* a3; /* item_storage */
	T6 a4; /* capacity */
	T6 a5; /* modulus */
	T6 a6; /* last_position */
	T6 a7; /* free_slot */
	T6 a8; /* position */
	T6 a9; /* slots_position */
	T0* a10; /* special_item_routines */
	T0* a11; /* clashes */
	T0* a12; /* slots */
	T6 a13; /* found_position */
	T6 a14; /* clashes_previous_position */
	T0* a15; /* equality_tester */
	T0* a16; /* hash_function */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_KEYWORD] */
struct S867 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_KEYWORD] */
struct S869 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_AGENT_KEYWORD] */
struct S870 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_AGENT_KEYWORD] */
struct S872 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_PRECURSOR_KEYWORD] */
struct S873 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_PRECURSOR_KEYWORD] */
struct S875 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_SYMBOL] */
struct S876 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_SYMBOL] */
struct S878 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_POSITION] */
struct S879 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_POSITION] */
struct S881 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_BOOLEAN_CONSTANT] */
struct S882 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_BOOLEAN_CONSTANT] */
struct S884 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_BREAK] */
struct S885 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_BREAK] */
struct S887 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_CHARACTER_CONSTANT] */
struct S888 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_CHARACTER_CONSTANT] */
struct S890 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_CURRENT] */
struct S891 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_CURRENT] */
struct S893 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_FREE_OPERATOR] */
struct S894 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_FREE_OPERATOR] */
struct S896 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_INTEGER_CONSTANT] */
struct S897 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_INTEGER_CONSTANT] */
struct S899 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_KEYWORD_OPERATOR] */
struct S900 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_KEYWORD_OPERATOR] */
struct S902 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_MANIFEST_STRING] */
struct S903 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_MANIFEST_STRING] */
struct S905 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_REAL_CONSTANT] */
struct S906 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_REAL_CONSTANT] */
struct S908 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_RESULT] */
struct S909 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_RESULT] */
struct S910 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_RETRY_INSTRUCTION] */
struct S911 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_RETRY_INSTRUCTION] */
struct S913 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_SYMBOL_OPERATOR] */
struct S914 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_SYMBOL_OPERATOR] */
struct S916 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_VOID] */
struct S917 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_VOID] */
struct S919 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_SEMICOLON_SYMBOL] */
struct S920 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_SEMICOLON_SYMBOL] */
struct S922 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_BRACKET_SYMBOL] */
struct S923 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_BRACKET_SYMBOL] */
struct S925 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_QUESTION_MARK_SYMBOL] */
struct S926 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_QUESTION_MARK_SYMBOL] */
struct S928 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_ACROSS_EXPRESSION] */
struct S929 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_ACROSS_EXPRESSION] */
struct S931 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_ACROSS_INSTRUCTION] */
struct S932 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_ACROSS_INSTRUCTION] */
struct S934 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_ACTUAL_ARGUMENT_LIST] */
struct S935 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_ACTUAL_ARGUMENT_LIST] */
struct S936 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_ACTUAL_PARAMETER_ITEM] */
struct S937 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_ACTUAL_PARAMETER_ITEM] */
struct S939 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_ACTUAL_PARAMETER_LIST] */
struct S940 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_ACTUAL_PARAMETER_LIST] */
struct S941 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_AGENT_ARGUMENT_OPERAND] */
struct S942 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_AGENT_ARGUMENT_OPERAND] */
struct S944 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_AGENT_ARGUMENT_OPERAND_ITEM] */
struct S945 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_AGENT_ARGUMENT_OPERAND_ITEM] */
struct S947 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_AGENT_ARGUMENT_OPERAND_LIST] */
struct S948 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_AGENT_ARGUMENT_OPERAND_LIST] */
struct S950 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_AGENT_TARGET] */
struct S951 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_AGENT_TARGET] */
struct S953 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_ALIAS_NAME] */
struct S954 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_ALIAS_NAME] */
struct S956 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_ASSIGNER] */
struct S957 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_ASSIGNER] */
struct S959 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_BRACKET_EXPRESSION] */
struct S960 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_BRACKET_EXPRESSION] */
struct S962 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_CALL_AGENT] */
struct S963 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_CALL_AGENT] */
struct S965 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_QUALIFIED_CALL_EXPRESSION] */
struct S966 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_QUALIFIED_CALL_EXPRESSION] */
struct S967 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_CHECK_INSTRUCTION] */
struct S968 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_CHECK_INSTRUCTION] */
struct S970 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_CHOICE] */
struct S971 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_CHOICE] */
struct S973 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_CHOICE_CONSTANT] */
struct S974 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_CHOICE_CONSTANT] */
struct S976 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_CHOICE_ITEM] */
struct S977 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_CHOICE_ITEM] */
struct S979 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_CHOICE_LIST] */
struct S980 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_CHOICE_LIST] */
struct S982 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_CLASS] */
struct S983 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_CLASS] */
struct S985 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_CLIENT_ITEM] */
struct S986 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_CLIENT_ITEM] */
struct S988 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_CLIENTS] */
struct S989 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_CLIENTS] */
struct S991 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_COMPOUND] */
struct S992 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_COMPOUND] */
struct S993 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_CONDITIONAL] */
struct S994 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_CONDITIONAL] */
struct S996 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_CONSTANT] */
struct S997 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_CONSTANT] */
struct S999 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_CONSTRAINT_ACTUAL_PARAMETER_ITEM] */
struct S1000 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_CONSTRAINT_ACTUAL_PARAMETER_ITEM] */
struct S1002 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_CONSTRAINT_ACTUAL_PARAMETER_LIST] */
struct S1003 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_CONSTRAINT_ACTUAL_PARAMETER_LIST] */
struct S1005 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_CONSTRAINT_CREATOR] */
struct S1006 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_CONSTRAINT_CREATOR] */
struct S1008 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_CONSTRAINT_TYPE] */
struct S1009 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_CONSTRAINT_TYPE] */
struct S1011 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_CONVERT_FEATURE] */
struct S1012 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_CONVERT_FEATURE] */
struct S1014 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_CONVERT_FEATURE_ITEM] */
struct S1015 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_CONVERT_FEATURE_ITEM] */
struct S1017 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_CONVERT_FEATURE_LIST] */
struct S1018 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_CONVERT_FEATURE_LIST] */
struct S1020 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_CREATE_EXPRESSION] */
struct S1021 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_CREATE_EXPRESSION] */
struct S1023 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_CREATION_REGION] */
struct S1024 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_CREATION_REGION] */
struct S1026 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_CREATOR] */
struct S1027 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_CREATOR] */
struct S1029 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_CREATOR_LIST] */
struct S1030 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_CREATOR_LIST] */
struct S1032 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_DEBUG_INSTRUCTION] */
struct S1033 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_DEBUG_INSTRUCTION] */
struct S1035 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_ELSEIF_EXPRESSION] */
struct S1036 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_ELSEIF_EXPRESSION] */
struct S1038 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_ELSEIF_EXPRESSION_LIST] */
struct S1039 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_ELSEIF_EXPRESSION_LIST] */
struct S1041 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_ELSEIF_PART] */
struct S1042 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_ELSEIF_PART] */
struct S1044 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_ELSEIF_PART_LIST] */
struct S1045 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_ELSEIF_PART_LIST] */
struct S1047 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_EXPORT] */
struct S1048 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_EXPORT] */
struct S1050 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_EXPORT_LIST] */
struct S1051 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_EXPORT_LIST] */
struct S1052 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_EXPRESSION] */
struct S1053 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_EXPRESSION] */
struct S1054 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_EXPRESSION_ITEM] */
struct S1055 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_EXPRESSION_ITEM] */
struct S1057 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_EXTENDED_FEATURE_NAME] */
struct S1058 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_EXTENDED_FEATURE_NAME] */
struct S1060 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_EXTERNAL_ALIAS] */
struct S1061 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_EXTERNAL_ALIAS] */
struct S1063 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_FEATURE_CLAUSE] */
struct S1064 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_FEATURE_CLAUSE] */
struct S1066 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_FEATURE_CLAUSE_LIST] */
struct S1067 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_FEATURE_CLAUSE_LIST] */
struct S1069 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_FEATURE_EXPORT] */
struct S1070 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_FEATURE_EXPORT] */
struct S1072 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_FEATURE_NAME] */
struct S1073 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_FEATURE_NAME] */
struct S1074 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_FEATURE_NAME_ITEM] */
struct S1075 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_FEATURE_NAME_ITEM] */
struct S1077 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_FORMAL_ARGUMENT] */
struct S1078 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_FORMAL_ARGUMENT] */
struct S1080 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_FORMAL_ARGUMENT_ITEM] */
struct S1081 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_FORMAL_ARGUMENT_ITEM] */
struct S1083 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_FORMAL_ARGUMENT_LIST] */
struct S1084 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_FORMAL_ARGUMENT_LIST] */
struct S1085 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_FORMAL_PARAMETER] */
struct S1086 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_FORMAL_PARAMETER] */
struct S1088 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_FORMAL_PARAMETER_ITEM] */
struct S1089 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_FORMAL_PARAMETER_ITEM] */
struct S1091 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_FORMAL_PARAMETER_LIST] */
struct S1092 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_FORMAL_PARAMETER_LIST] */
struct S1093 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_IF_EXPRESSION] */
struct S1094 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_IF_EXPRESSION] */
struct S1096 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_IF_INSTRUCTION] */
struct S1097 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_IF_INSTRUCTION] */
struct S1099 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_INDEXING_LIST] */
struct S1100 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_INDEXING_LIST] */
struct S1102 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_INDEXING_ITEM] */
struct S1103 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_INDEXING_ITEM] */
struct S1105 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_INDEXING_TERM] */
struct S1106 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_INDEXING_TERM] */
struct S1108 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_INDEXING_TERM_ITEM] */
struct S1109 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_INDEXING_TERM_ITEM] */
struct S1111 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_INDEXING_TERM_LIST] */
struct S1112 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_INDEXING_TERM_LIST] */
struct S1114 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_INLINE_AGENT] */
struct S1115 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_INLINE_AGENT] */
struct S1117 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_INSPECT_INSTRUCTION] */
struct S1118 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_INSPECT_INSTRUCTION] */
struct S1120 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_INSTRUCTION] */
struct S1121 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_INSTRUCTION] */
struct S1123 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_INVARIANTS] */
struct S1124 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_INVARIANTS] */
struct S1126 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_KEYWORD_FEATURE_NAME_LIST] */
struct S1127 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_KEYWORD_FEATURE_NAME_LIST] */
struct S1128 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_LIKE_TYPE] */
struct S1129 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_LIKE_TYPE] */
struct S1131 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_LOCAL_VARIABLE] */
struct S1132 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_LOCAL_VARIABLE] */
struct S1134 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_LOCAL_VARIABLE_ITEM] */
struct S1135 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_LOCAL_VARIABLE_ITEM] */
struct S1137 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_LOCAL_VARIABLE_LIST] */
struct S1138 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_LOCAL_VARIABLE_LIST] */
struct S1139 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_LOOP_INVARIANTS] */
struct S1140 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_LOOP_INVARIANTS] */
struct S1142 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_MANIFEST_ARRAY] */
struct S1143 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_MANIFEST_ARRAY] */
struct S1145 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_MANIFEST_STRING_ITEM] */
struct S1146 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_MANIFEST_STRING_ITEM] */
struct S1148 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_MANIFEST_STRING_LIST] */
struct S1149 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_MANIFEST_STRING_LIST] */
struct S1151 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_MANIFEST_TUPLE] */
struct S1152 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_MANIFEST_TUPLE] */
struct S1153 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_OBSOLETE] */
struct S1154 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_OBSOLETE] */
struct S1156 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_PARENTHESIZED_EXPRESSION] */
struct S1157 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_PARENTHESIZED_EXPRESSION] */
struct S1159 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_PARENT] */
struct S1160 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_PARENT] */
struct S1162 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_PARENT_CLAUSE_LIST] */
struct S1163 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_PARENT_CLAUSE_LIST] */
struct S1165 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_PARENT_ITEM] */
struct S1166 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_PARENT_ITEM] */
struct S1168 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_PARENT_LIST] */
struct S1169 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_PARENT_LIST] */
struct S1171 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_POSTCONDITIONS] */
struct S1172 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_POSTCONDITIONS] */
struct S1174 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_PRECONDITIONS] */
struct S1175 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_PRECONDITIONS] */
struct S1176 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_PROCEDURE] */
struct S1177 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_PROCEDURE] */
struct S1178 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_QUALIFIED_LIKE_IDENTIFIER] */
struct S1179 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_QUALIFIED_LIKE_IDENTIFIER] */
struct S1181 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_QUERY] */
struct S1182 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_QUERY] */
struct S1183 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_RENAME_ITEM] */
struct S1184 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_RENAME_ITEM] */
struct S1186 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_RENAME_LIST] */
struct S1187 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_RENAME_LIST] */
struct S1188 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_STATIC_CALL_EXPRESSION] */
struct S1189 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_STATIC_CALL_EXPRESSION] */
struct S1191 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_STRIP_EXPRESSION] */
struct S1192 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_STRIP_EXPRESSION] */
struct S1194 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_TYPE] */
struct S1195 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_TYPE] */
struct S1196 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_TYPE_ITEM] */
struct S1197 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_TYPE_ITEM] */
struct S1199 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_TYPE_LIST] */
struct S1200 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_TYPE_LIST] */
struct S1202 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_VARIANT] */
struct S1203 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_VARIANT] */
struct S1205 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_WHEN_PART] */
struct S1206 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_WHEN_PART] */
struct S1208 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_WHEN_PART_LIST] */
struct S1209 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_WHEN_PART_LIST] */
struct S1211 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_WRITABLE] */
struct S1212 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_WRITABLE] */
struct S1213 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] DS_HASH_TOPOLOGICAL_SORTER [[attached] ET_CLASS] */
struct S1214 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* sorted_items */
	T0* a2; /* cycle */
	T0* a3; /* indexes */
	T0* a4; /* counts */
	T0* a5; /* successors */
	T0* a6; /* items */
};

/* Struct for type [attached] DS_HASH_TABLE [[attached] ET_BASE_TYPE, [attached] ET_CLASS] */
struct S1215 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* internal_cursor */
	T6 a3; /* found_position */
	T0* a4; /* item_storage */
	T6 a5; /* last_position */
	T6 a6; /* free_slot */
	T6 a7; /* position */
	T6 a8; /* capacity */
	T0* a9; /* equality_tester */
	T0* a10; /* key_equality_tester */
	T0* a11; /* key_storage */
	T0* a12; /* clashes */
	T6 a13; /* modulus */
	T0* a14; /* slots */
	T6 a15; /* slots_position */
	T6 a16; /* clashes_previous_position */
	T0* a17; /* special_item_routines */
	T0* a18; /* special_key_routines */
	T0* a19; /* hash_function */
};

/* Struct for type ET_PARENT_CHECKER1 */
struct S1216 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* has_fatal_error */
	T0* a2; /* current_class */
	T0* a3; /* system_processor */
	T0* a4; /* current_parent */
};

/* Struct for type ET_FORMAL_PARAMETER_CHECKER1 */
struct S1217 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* has_fatal_error */
	T0* a2; /* formal_parameter_sorter */
	T0* a3; /* direct_formal_parameter_sorter */
	T0* a4; /* current_class */
	T0* a5; /* system_processor */
	T0* a6; /* current_formal */
};

/* Struct for type [attached] DS_HASH_TABLE [[attached] ET_FLATTENED_FEATURE, [attached] ET_FEATURE_NAME] */
struct S1218 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* internal_cursor */
	T6 a3; /* found_position */
	T6 a4; /* position */
	T0* a5; /* item_storage */
	T0* a6; /* key_equality_tester */
	T0* a7; /* internal_keys */
	T6 a8; /* last_position */
	T6 a9; /* free_slot */
	T6 a10; /* modulus */
	T6 a11; /* slots_position */
	T6 a12; /* clashes_previous_position */
	T0* a13; /* equality_tester */
	T0* a14; /* key_storage */
	T0* a15; /* clashes */
	T0* a16; /* slots */
	T6 a17; /* capacity */
	T0* a18; /* special_item_routines */
	T0* a19; /* special_key_routines */
	T0* a20; /* hash_function */
};

/* Struct for type ET_FEATURE_NAME_TESTER */
struct S1219 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] DS_HASH_TABLE [[attached] ET_FLATTENED_FEATURE, [attached] ET_ALIAS_NAME] */
struct S1221 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* found_position */
	T0* a2; /* item_storage */
	T0* a3; /* key_equality_tester */
	T0* a4; /* internal_keys */
	T6 a5; /* position */
	T6 a6; /* last_position */
	T6 a7; /* capacity */
	T6 a8; /* count */
	T6 a9; /* free_slot */
	T0* a10; /* equality_tester */
	T6 a11; /* modulus */
	T6 a12; /* slots_position */
	T6 a13; /* clashes_previous_position */
	T0* a14; /* clashes */
	T0* a15; /* slots */
	T0* a16; /* special_item_routines */
	T0* a17; /* special_key_routines */
	T0* a18; /* key_storage */
	T0* a19; /* internal_cursor */
	T0* a20; /* hash_function */
};

/* Struct for type ET_ALIAS_NAME_TESTER */
struct S1222 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] DS_ARRAYED_LIST [[attached] ET_CLIENT_LIST] */
struct S1224 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* special_routines */
	T0* a2; /* storage */
	T6 a3; /* capacity */
	T0* a4; /* internal_cursor */
	T6 a5; /* count */
};

/* Struct for type [attached] DS_HASH_TABLE [[attached] ET_CLIENT, [attached] ET_CLASS] */
struct S1225 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* equality_tester */
	T0* a2; /* key_equality_tester */
	T6 a3; /* capacity */
	T6 a4; /* modulus */
	T6 a5; /* last_position */
	T6 a6; /* free_slot */
	T6 a7; /* position */
	T0* a8; /* special_item_routines */
	T0* a9; /* item_storage */
	T0* a10; /* special_key_routines */
	T0* a11; /* key_storage */
	T0* a12; /* clashes */
	T0* a13; /* slots */
	T6 a14; /* found_position */
	T0* a15; /* internal_cursor */
	T6 a16; /* count */
	T0* a17; /* internal_keys */
	T6 a18; /* slots_position */
	T6 a19; /* clashes_previous_position */
	T0* a20; /* hash_function */
};

/* Struct for type ET_FEATURE_ADAPTATION_RESOLVER */
struct S1226 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* has_fatal_error */
	T0* a2; /* rename_table */
	T0* a3; /* export_table */
	T0* a4; /* undefine_table */
	T0* a5; /* redefine_table */
	T0* a6; /* select_table */
	T0* a7; /* alias_mapping */
	T0* a8; /* replicable_features */
	T0* a9; /* current_class */
	T0* a10; /* system_processor */
	T0* a11; /* free_parent_feature */
	T0* a12; /* parent_feature_list */
	T0* a13; /* free_inherited_feature */
	T0* a14; /* inherited_feature_list */
	T0* a15; /* free_redeclared_feature */
	T0* a16; /* redeclared_feature_list */
};

/* Struct for type ET_DOTNET_FEATURE_ADAPTATION_RESOLVER */
struct S1227 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* has_fatal_error */
	T0* a2; /* dotnet_features */
	T0* a3; /* other_dotnet_features */
	T0* a4; /* current_class */
	T0* a5; /* system_processor */
	T0* a6; /* free_parent_feature */
	T0* a7; /* parent_feature_list */
	T0* a8; /* free_inherited_feature */
	T0* a9; /* inherited_feature_list */
	T0* a10; /* free_redeclared_feature */
	T0* a11; /* redeclared_feature_list */
};

/* Struct for type ET_IDENTIFIER_TYPE_RESOLVER */
struct S1228 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* has_fatal_error */
	T0* a2; /* current_class */
	T0* a3; /* system_processor */
	T0* a4; /* current_feature */
};

/* Struct for type ET_UNFOLDED_TUPLE_ACTUAL_PARAMETERS_RESOLVER1 */
struct S1229 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* constraint_context */
	T0* a2; /* current_class */
	T0* a3; /* system_processor */
};

/* Struct for type ET_ANCHORED_TYPE_CHECKER */
struct S1230 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* has_fatal_error */
	T0* a2; /* anchored_type_sorter */
	T0* a3; /* current_class */
	T0* a4; /* system_processor */
	T0* a5; /* current_anchored_type */
};

/* Struct for type ET_SIGNATURE_CHECKER */
struct S1231 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* has_fatal_error */
	T0* a2; /* parent_context */
	T0* a3; /* current_class */
	T0* a4; /* system_processor */
	T6 a5; /* processing_mode */
};

/* Struct for type ET_PARENT_CHECKER2 */
struct S1232 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* has_fatal_error */
	T0* a2; /* constraint_context */
	T0* a3; /* current_class */
	T0* a4; /* system_processor */
};

/* Struct for type ET_FORMAL_PARAMETER_CHECKER2 */
struct S1233 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* has_fatal_error */
	T0* a2; /* constraint_context */
	T0* a3; /* current_class */
	T0* a4; /* system_processor */
};

/* Struct for type ET_BUILTIN_FEATURE_CHECKER */
struct S1234 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* has_fatal_error */
	T0* a2; /* builtin_features */
	T0* a3; /* current_class */
	T0* a4; /* system_processor */
};

/* Struct for type ET_PRECURSOR_CHECKER */
struct S1235 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* has_fatal_error */
	T0* a2; /* current_feature */
	T0* a3; /* current_class */
	T0* a4; /* system_processor */
};

/* Struct for type [attached] DS_HASH_TABLE [[attached] ET_FEATURE, [attached] INTEGER_32] */
struct S1236 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* internal_cursor */
	T0* a3; /* item_storage */
	T6 a4; /* position */
	T6 a5; /* last_position */
	T6 a6; /* capacity */
	T6 a7; /* slots_position */
	T6 a8; /* free_slot */
	T0* a9; /* equality_tester */
	T0* a10; /* key_equality_tester */
	T6 a11; /* found_position */
	T6 a12; /* modulus */
	T6 a13; /* clashes_previous_position */
	T0* a14; /* special_item_routines */
	T0* a15; /* clashes */
	T0* a16; /* slots */
	T0* a17; /* special_key_routines */
	T0* a18; /* key_storage */
	T0* a19; /* internal_keys */
	T0* a20; /* hash_function */
};

/* Struct for type [attached] DS_HASH_SET [[attached] ET_CLASS] */
struct S1237 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T6 a2; /* last_position */
	T0* a3; /* clashes */
	T0* a4; /* item_storage */
	T6 a5; /* capacity */
	T6 a6; /* modulus */
	T6 a7; /* free_slot */
	T6 a8; /* position */
	T0* a9; /* special_item_routines */
	T0* a10; /* slots */
	T6 a11; /* found_position */
	T0* a12; /* internal_cursor */
	T6 a13; /* slots_position */
	T6 a14; /* clashes_previous_position */
	T0* a15; /* equality_tester */
	T0* a16; /* hash_function */
};

/* Struct for type ET_PARENT_CHECKER3 */
struct S1238 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* has_fatal_error */
	T0* a2; /* classes_to_be_processed */
	T0* a3; /* current_class */
	T0* a4; /* system_processor */
};

/* Struct for type ET_QUALIFIED_ANCHORED_TYPE_CHECKER */
struct S1239 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* has_fatal_error */
	T0* a2; /* current_class_impl */
	T0* a3; /* current_class */
	T0* a4; /* system_processor */
	T1 a5; /* in_qualified_anchored_type */
};

/* Struct for type ET_UNFOLDED_TUPLE_ACTUAL_PARAMETERS_RESOLVER2 */
struct S1240 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* constraint_context */
	T0* a2; /* current_class */
	T0* a3; /* system_processor */
};

/* Struct for type ET_FORMAL_PARAMETER_LIST */
struct S1242 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
	T0* a3; /* left_bracket */
	T0* a4; /* right_bracket */
};

/* Struct for type ET_FORMAL_PARAMETER */
struct S1243 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* name */
	T6 a2; /* index */
	T0* a3; /* implementation_class */
	T0* a4; /* type_mark */
};

/* Struct for type ET_CONSTRAINT_CREATOR */
struct S1244 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* create_keyword */
	T0* a2; /* end_keyword */
	T6 a3; /* count */
	T0* a4; /* storage */
};

/* Struct for type ET_INHERITED_FEATURE */
struct S1246 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* next */
	T0* a2; /* parent_feature */
	T0* a3; /* flattened_feature */
	T0* a4; /* flattened_parent */
	T6 a5; /* first_seed */
	T0* a6; /* other_seeds */
	T1 a7; /* is_selected */
	T0* a8; /* replicated_features */
	T0* a9; /* replicated_seeds */
};

/* Struct for type ET_PARENT_FEATURE */
struct S1247 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* next */
	T0* a2; /* precursor_feature */
	T0* a3; /* parent */
	T0* a4; /* new_name */
	T0* a5; /* undefine_name */
	T0* a6; /* redefine_name */
	T0* a7; /* select_name */
	T0* a8; /* merged_feature */
};

/* Struct for type [attached] DS_HASH_SET [[attached] ET_PROCEDURE] */
struct S1248 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* internal_cursor */
	T0* a2; /* item_storage */
	T6 a3; /* capacity */
	T6 a4; /* modulus */
	T6 a5; /* last_position */
	T6 a6; /* free_slot */
	T6 a7; /* position */
	T6 a8; /* count */
	T0* a9; /* special_item_routines */
	T0* a10; /* clashes */
	T0* a11; /* slots */
	T6 a12; /* found_position */
	T6 a13; /* slots_position */
	T6 a14; /* clashes_previous_position */
	T0* a15; /* hash_function */
	T0* a16; /* equality_tester */
};

/* Struct for type [attached] DS_HASH_SET [[attached] ET_QUERY] */
struct S1249 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* internal_cursor */
	T0* a2; /* item_storage */
	T6 a3; /* capacity */
	T6 a4; /* modulus */
	T6 a5; /* last_position */
	T6 a6; /* free_slot */
	T6 a7; /* position */
	T6 a8; /* count */
	T0* a9; /* special_item_routines */
	T0* a10; /* clashes */
	T0* a11; /* slots */
	T6 a12; /* found_position */
	T6 a13; /* slots_position */
	T6 a14; /* clashes_previous_position */
	T0* a15; /* hash_function */
	T0* a16; /* equality_tester */
};

/* Struct for type ET_SUPPLIER_BUILDER */
struct S1250 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* supplier_classes */
	T0* a2; /* current_class */
};

/* Struct for type [attached] SPECIAL [[attached] ET_CLASS] */
struct S1251 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_CLASS] */
struct S1252 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_CLASS] */
struct S1253 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* next_cursor */
	T0* a2; /* container */
	T6 a3; /* position */
};

/* Struct for type [attached] DS_ARRAYED_LIST_CURSOR [[attached] INTEGER_32] */
struct S1255 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* next_cursor */
	T0* a2; /* container */
	T6 a3; /* position */
};

/* Struct for type [attached] SPECIAL [[attached] ET_DYNAMIC_TYPE_SET] */
struct S1258 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_DYNAMIC_TYPE_SET] */
struct S1259 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_IDENTIFIER] */
struct S1260 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
};

/* Struct for type [attached] SPECIAL [detachable ET_DYNAMIC_TYPE] */
struct S1261 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_DYNAMIC_TYPE] */
struct S1262 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] DS_ARRAYED_LIST_CURSOR [detachable ET_DYNAMIC_TYPE] */
struct S1263 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* next_cursor */
	T0* a2; /* container */
	T6 a3; /* position */
};

/* Struct for type [attached] SPECIAL [[attached] ET_EXPRESSION] */
struct S1265 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_EXPRESSION] */
struct S1266 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_EXPRESSION] */
struct S1267 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* next_cursor */
	T0* a2; /* container */
	T6 a3; /* position */
};

/* Struct for type attached KL_EQUALITY_TESTER [[attached] INTEGER_32] */
struct S1271 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] DS_HASH_TABLE_CURSOR [[attached] ET_DYNAMIC_TYPE, [attached] INTEGER_32] */
struct S1273 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* next_cursor */
	T0* a2; /* container */
	T6 a3; /* position */
};

/* Struct for type [attached] SPECIAL [[attached] ET_DYNAMIC_STANDALONE_TYPE_SET] */
struct S1274 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_DYNAMIC_STANDALONE_TYPE_SET] */
struct S1275 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] DS_HASH_SET_CURSOR [[attached] ET_DYNAMIC_TYPE] */
struct S1276 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* next_cursor */
	T0* a3; /* container */
};

/* Struct for type [attached] DS_HASH_TABLE_CURSOR [[attached] ET_DYNAMIC_STANDALONE_TYPE_SET, [attached] ET_DYNAMIC_TYPE] */
struct S1279 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* next_cursor */
	T0* a3; /* container */
};

/* Struct for type [attached] SPECIAL [[attached] ET_OBJECT_TEST] */
struct S1282 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_OBJECT_TEST] */
struct S1283 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_OBJECT_TEST] */
struct S1284 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* next_cursor */
	T0* a2; /* container */
	T6 a3; /* position */
};

/* Struct for type [attached] SPECIAL [[attached] ET_OBJECT_EQUALITY_EXPRESSION] */
struct S1286 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_OBJECT_EQUALITY_EXPRESSION] */
struct S1287 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_OBJECT_EQUALITY_EXPRESSION] */
struct S1288 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* next_cursor */
	T0* a2; /* container */
	T6 a3; /* position */
};

/* Struct for type [attached] SPECIAL [[attached] ET_EQUALITY_EXPRESSION] */
struct S1290 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_EQUALITY_EXPRESSION] */
struct S1291 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_EQUALITY_EXPRESSION] */
struct S1292 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* next_cursor */
	T0* a2; /* container */
	T6 a3; /* position */
};

/* Struct for type [attached] SPECIAL [[attached] ET_AGENT] */
struct S1294 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_AGENT] */
struct S1295 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_AGENT] */
struct S1296 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* next_cursor */
	T0* a2; /* container */
	T6 a3; /* position */
};

/* Struct for type [attached] SPECIAL [[attached] ET_EXPRESSION_ITEM] */
struct S1298 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_EXPRESSION_ITEM] */
struct S1300 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] DS_ARRAYED_LIST_CURSOR [detachable ET_QUALIFIED_CALL_EXPRESSION] */
struct S1301 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
};

/* Struct for type [attached] DS_HASH_SET_CURSOR [[attached] ET_DYNAMIC_TUPLE_TYPE] */
struct S1302 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* next_cursor */
	T0* a3; /* container */
};

/* Struct for type [attached] SPECIAL [[attached] ET_DYNAMIC_TUPLE_TYPE] */
struct S1303 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_DYNAMIC_TUPLE_TYPE] */
struct S1304 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] DS_HASH_TABLE_CURSOR [[attached] INTEGER_32, [attached] ET_FEATURE] */
struct S1309 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* next_cursor */
	T0* a2; /* container */
	T6 a3; /* position */
};

/* Struct for type [attached] SPECIAL [[attached] ET_FEATURE] */
struct S1310 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_FEATURE] */
struct S1311 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] DS_HASH_TABLE_CURSOR [[attached] ET_CONSTANT, [attached] ET_FEATURE] */
struct S1312 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* next_cursor */
	T0* a3; /* container */
};

/* Struct for type [attached] SPECIAL [[attached] ET_CONSTANT] */
struct S1313 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_CONSTANT] */
struct S1316 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] DS_HASH_TABLE_CURSOR [[attached] ET_DYNAMIC_TYPE, [attached] ET_INLINE_CONSTANT] */
struct S1317 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* next_cursor */
	T0* a3; /* container */
};

/* Struct for type [attached] SPECIAL [[attached] ET_INLINE_CONSTANT] */
struct S1318 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_INLINE_CONSTANT] */
struct S1321 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable ET_DYNAMIC_FEATURE] */
struct S1322 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] DS_HASH_TABLE_CURSOR [detachable ET_DYNAMIC_FEATURE, [attached] ET_DYNAMIC_TYPE] */
struct S1325 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* next_cursor */
	T0* a2; /* container */
	T6 a3; /* position */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_DYNAMIC_FEATURE] */
struct S1326 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [[attached] ET_DYNAMIC_FEATURE] */
struct S1327 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_DYNAMIC_FEATURE] */
struct S1328 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_DYNAMIC_FEATURE] */
struct S1329 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* next_cursor */
	T0* a3; /* container */
};

/* Struct for type [attached] DS_HASH_TABLE_CURSOR [[attached] BOOLEAN, [attached] STRING_8] */
struct S1331 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* next_cursor */
	T0* a3; /* container */
};

/* Struct for type RX_BYTE_CODE */
struct S1337 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* byte_code */
	T0* a3; /* character_sets */
	T6 a4; /* capacity */
	T6 a5; /* character_sets_count */
	T6 a6; /* character_sets_capacity */
};

/* Struct for type RX_CHARACTER_SET */
struct S1338 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* set */
};

/* Struct for type RX_CASE_MAPPING */
struct S1339 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* lower_table */
	T0* a2; /* flip_table */
};

/* Struct for type [attached] ARRAY [[attached] RX_CHARACTER_SET] */
struct S1342 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* area */
	T6 a2; /* lower */
	T6 a3; /* upper */
};

/* Struct for type [attached] SPECIAL [[attached] RX_CHARACTER_SET] */
struct S1343 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] DS_ARRAYED_LIST [detachable DS_LINKABLE [[attached] INTEGER_32]] */
struct S1345 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* storage */
	T0* a2; /* special_routines */
	T6 a3; /* capacity */
	T6 a4; /* count */
	T0* a5; /* internal_cursor */
};

/* Struct for type [attached] ARRAY [[attached] BOOLEAN] */
struct S1346 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* area */
	T6 a2; /* lower */
	T6 a3; /* upper */
};

/* Struct for type [attached] SPECIAL [[attached] ET_SYSTEM_PROCESSOR] */
struct S1349 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_SYSTEM_PROCESSOR] */
struct S1350 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_SYSTEM_PROCESSOR] */
struct S1351 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
};

/* Struct for type THREAD_ATTRIBUTES */
struct S1352 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* shared */
	T14 a2; /* internal_item */
	T0* a3; /* managed_pointer */
};

/* Struct for type [attached] CELL [[attached] BOOLEAN] */
struct S1353 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* item */
};

/* Struct for type [attached] DS_HASH_TABLE_CURSOR [[attached] INTEGER_32, [attached] ET_DYNAMIC_TYPE] */
struct S1359 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* next_cursor */
	T0* a2; /* container */
	T6 a3; /* position */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_INLINE_AGENT] */
struct S1361 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [[attached] ET_INLINE_AGENT] */
struct S1363 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_INLINE_AGENT] */
struct S1364 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
	T0* a3; /* next_cursor */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] DS_ARRAYED_LIST [[attached] ET_PROCEDURE]] */
struct S1365 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [[attached] DS_ARRAYED_LIST [[attached] ET_PROCEDURE]] */
struct S1366 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] DS_ARRAYED_LIST_CURSOR [[attached] DS_ARRAYED_LIST [[attached] ET_PROCEDURE]] */
struct S1367 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
	T0* a3; /* next_cursor */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] DS_ARRAYED_LIST [[attached] ET_QUERY]] */
struct S1368 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [[attached] DS_ARRAYED_LIST [[attached] ET_QUERY]] */
struct S1369 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] DS_ARRAYED_LIST_CURSOR [[attached] DS_ARRAYED_LIST [[attached] ET_QUERY]] */
struct S1370 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
	T0* a3; /* next_cursor */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] DS_ARRAYED_LIST [[attached] ET_FEATURE]] */
struct S1371 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] DS_ARRAYED_LIST [[attached] ET_FEATURE] */
struct S1372 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
	T0* a3; /* equality_tester */
	T0* a4; /* special_routines */
	T6 a5; /* capacity */
	T0* a6; /* internal_cursor */
};

/* Struct for type [attached] SPECIAL [[attached] DS_ARRAYED_LIST [[attached] ET_FEATURE]] */
struct S1373 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] DS_ARRAYED_LIST_CURSOR [[attached] DS_ARRAYED_LIST [[attached] ET_FEATURE]] */
struct S1374 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
	T0* a3; /* next_cursor */
};

/* Struct for type [attached] SPECIAL [[attached] ET_NESTED_TYPE_CONTEXT] */
struct S1375 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_NESTED_TYPE_CONTEXT] */
struct S1376 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_NESTED_TYPE_CONTEXT] */
struct S1377 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* next_cursor */
	T0* a3; /* container */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_TYPE] */
struct S1379 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] DS_HASH_TABLE_CURSOR [[attached] ET_NESTED_TYPE_CONTEXT, [attached] ET_NAMED_OBJECT_TEST] */
struct S1380 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* next_cursor */
	T0* a3; /* container */
};

/* Struct for type ET_NAMED_OBJECT_TEST */
struct S1384 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* index */
	T0* a2; /* name */
	T0* a3; /* as_keyword */
	T0* a4; /* expression */
	T0* a5; /* attached_keyword */
	T0* a6; /* declared_type */
};

/* Struct for type [attached] SPECIAL [[attached] ET_NAMED_OBJECT_TEST] */
struct S1385 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_NAMED_OBJECT_TEST] */
struct S1386 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] DS_ARRAYED_LIST [[attached] ET_NAMED_OBJECT_TEST] */
struct S1387 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* special_routines */
	T0* a3; /* storage */
	T6 a4; /* capacity */
	T0* a5; /* internal_cursor */
};

/* Struct for type [attached] DS_HASH_TABLE_CURSOR [[attached] ET_NESTED_TYPE_CONTEXT, [attached] ET_ACROSS_COMPONENT] */
struct S1388 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* next_cursor */
	T0* a3; /* container */
};

/* Struct for type [attached] SPECIAL [[attached] ET_ACROSS_COMPONENT] */
struct S1392 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_ACROSS_COMPONENT] */
struct S1393 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] DS_ARRAYED_LIST [[attached] ET_ACROSS_COMPONENT] */
struct S1394 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* special_routines */
	T0* a2; /* storage */
	T6 a3; /* capacity */
	T6 a4; /* count */
	T0* a5; /* internal_cursor */
};

/* Struct for type [attached] DS_HASH_SET [[attached] INTEGER_32] */
struct S1395 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* internal_cursor */
	T6 a2; /* position */
	T0* a3; /* item_storage */
	T6 a4; /* capacity */
	T6 a5; /* modulus */
	T6 a6; /* last_position */
	T6 a7; /* free_slot */
	T6 a8; /* count */
	T6 a9; /* slots_position */
	T6 a10; /* clashes_previous_position */
	T0* a11; /* special_item_routines */
	T0* a12; /* clashes */
	T0* a13; /* slots */
	T6 a14; /* found_position */
	T0* a15; /* equality_tester */
	T0* a16; /* hash_function */
};

/* Struct for type [attached] SPECIAL [[attached] ET_ATTACHMENT_SCOPE] */
struct S1398 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_ATTACHMENT_SCOPE] */
struct S1399 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_ATTACHMENT_SCOPE] */
struct S1400 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* next_cursor */
	T0* a3; /* container */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_INDEXING_TERM] */
struct S1402 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [[attached] ET_INDEXING_TERM] */
struct S1404 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_INDEXING_TERM] */
struct S1405 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
	T0* a3; /* next_cursor */
};

/* Struct for type [attached] DS_HASH_TABLE_CURSOR [[attached] ET_ASSERTIONS, [attached] ET_FEATURE] */
struct S1406 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* next_cursor */
	T0* a3; /* container */
};

/* Struct for type [attached] SPECIAL [[attached] ET_ASSERTIONS] */
struct S1407 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_ASSERTIONS] */
struct S1409 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] DS_HASH_TABLE [[attached] NATURAL_8, [attached] ET_CLASS_NAME] */
struct S1411 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* found_position */
	T0* a2; /* item_storage */
	T6 a3; /* position */
	T0* a4; /* key_equality_tester */
	T0* a5; /* internal_keys */
	T6 a6; /* last_position */
	T6 a7; /* capacity */
	T6 a8; /* slots_position */
	T6 a9; /* count */
	T0* a10; /* equality_tester */
	T6 a11; /* modulus */
	T6 a12; /* clashes_previous_position */
	T0* a13; /* special_item_routines */
	T0* a14; /* clashes */
	T0* a15; /* slots */
	T0* a16; /* special_key_routines */
	T0* a17; /* key_storage */
	T6 a18; /* free_slot */
	T0* a19; /* internal_cursor */
	T0* a20; /* hash_function */
};

/* Struct for type [attached] DS_HASH_TABLE_CURSOR [[attached] ET_IDENTIFIER, [attached] STRING_8] */
struct S1415 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
	T0* a3; /* next_cursor */
};

/* Struct for type STRING_TO_INTEGER_CONVERTOR */
struct S1416 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* sign */
	T11 a2; /* part1 */
	T11 a3; /* part2 */
	T6 a4; /* last_state */
	T1 a5; /* internal_overflowed */
	T0* a6; /* leading_separators */
	T0* a7; /* trailing_separators */
	T1 a8; /* leading_separators_acceptable */
	T1 a9; /* trailing_separators_acceptable */
	T6 a10; /* conversion_type */
};

/* Struct for type ET_VALIDITY_ERROR */
struct S1418 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* ise_reported */
	T1 a2; /* ge_reported */
	T0* a3; /* default_template */
	T0* a4; /* parameters */
	T0* a5; /* current_class */
	T0* a6; /* class_impl */
	T0* a7; /* position */
	T0* a8; /* code */
	T0* a9; /* etl_code */
	T1 a10; /* ise_fatal */
	T1 a11; /* ge_fatal */
};

/* Struct for type UC_UNICODE_ROUTINES */
struct S1419 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type ET_XACE_CLASS_OPTIONS */
struct S1422 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* feature_options */
	T0* a2; /* class_name */
	T0* a3; /* options */
};

/* Struct for type attached DS_LINKED_LIST [[attached] ET_XACE_FEATURE_OPTIONS] */
struct S1424 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* last_cell */
	T0* a3; /* first_cell */
	T0* a4; /* internal_cursor */
};

/* Struct for type ET_XACE_FEATURE_OPTIONS */
struct S1426 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* feature_name */
	T0* a2; /* options */
};

/* Struct for type ET_XACE_WRONG_ROOT_ELEMENT_ERROR */
struct S1427 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* parameters */
};

/* Struct for type ET_XACE_MISSING_ATTRIBUTE_ERROR */
struct S1428 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* parameters */
};

/* Struct for type ET_XACE_UNKNOWN_ELEMENT_ERROR */
struct S1429 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* parameters */
};

/* Struct for type ET_XACE_NON_EMPTY_ATTRIBUTE_EXPECTED_ERROR */
struct S1430 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* parameters */
};

/* Struct for type ET_XACE_MISSING_ELEMENT_ERROR */
struct S1431 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* parameters */
};

/* Struct for type ET_XACE_OBSOLETE_CLUSTER_ELEMENT_WARNING */
struct S1432 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* parameters */
};

/* Struct for type ET_XACE_OBSOLETE_EXCLUDE_ELEMENT_WARNING */
struct S1433 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* parameters */
};

/* Struct for type ET_XACE_ATTRIBUTE_OBSOLETED_BY_ELEMENT_WARNING */
struct S1434 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* parameters */
};

/* Struct for type ET_ECF_NOTE_ELEMENT */
struct S1438 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* name */
	T0* a2; /* attributes */
	T0* a3; /* elements */
	T0* a4; /* content */
};

/* Struct for type [attached] DS_ARRAYED_LIST [[attached] ET_ECF_NOTE_ELEMENT] */
struct S1439 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* special_routines */
	T0* a2; /* storage */
	T6 a3; /* capacity */
	T6 a4; /* count */
	T0* a5; /* internal_cursor */
};

/* Struct for type ET_ECF_TARGET_PARENT */
struct S1453 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* target */
	T0* a2; /* name_id */
	T0* a3; /* location_id */
};

/* Struct for type ET_ECF_ADAPTED_PRECOMPILED_LIBRARY */
struct S1455 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* name_id */
	T0* a2; /* filename_id */
	T0* a3; /* universe */
	T0* a4; /* target */
	T0* a5; /* description */
	T0* a6; /* eifgens_location */
	T0* a7; /* classname_prefix */
	T1 a8; /* is_read_only */
	T1 a9; /* use_application_options */
	T0* a10; /* options */
	T0* a11; /* class_options */
	T0* a12; /* class_renamings */
	T0* a13; /* visible_classes */
	T0* a14; /* conditions */
	T0* a15; /* name */
	T0* a16; /* library */
};

/* Struct for type ET_ECF_VERSION */
struct S1459 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* internal_major */
	T6 a2; /* internal_minor */
	T6 a3; /* internal_release */
	T6 a4; /* internal_build */
	T0* a5; /* product */
	T0* a6; /* company */
	T0* a7; /* copyright */
};

/* Struct for type ET_ECF_ADAPTED_DOTNET_ASSEMBLIES */
struct S1461 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* dotnet_assemblies */
};

/* Struct for type ET_ECF_CLUSTER */
struct S1463 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* name */
	T0* a2; /* pathname */
	T1 a3; /* is_relative */
	T0* a4; /* universe */
	T0* a5; /* target */
	T0* a6; /* description */
	T1 a7; /* is_hidden */
	T0* a8; /* classname_prefix */
	T1 a9; /* is_read_only */
	T1 a10; /* is_recursive */
	T0* a11; /* conditioned_subclusters */
	T0* a12; /* conditioned_file_rules */
	T0* a13; /* options */
	T0* a14; /* class_options */
	T0* a15; /* class_renamings */
	T0* a16; /* class_mappings */
	T0* a17; /* visible_classes */
	T0* a18; /* provider_groups */
	T0* a19; /* conditions */
	T0* a20; /* notes */
	T1 a21; /* is_override */
	T0* a22; /* overridden_group */
	T1 a23; /* overridden_constraint_enabled */
	T1 a24; /* scm_mapping_constraint_enabled */
	T0* a25; /* subclusters */
	T1 a26; /* use_obsolete_syntax */
	T0* a27; /* scm_read_mapping */
	T0* a28; /* scm_write_mapping */
	T1 a29; /* is_implicit */
	T0* a30; /* parent */
	T0* a31; /* provider_constraint */
	T0* a32; /* dependant_constraint */
	T1 a33; /* is_preparsed */
	T1 a34; /* is_abstract */
	T0* a35; /* file_rules */
	T0* a36; /* cached_absolute_pathname */
};

/* Struct for type ET_ECF_CLUSTERS */
struct S1464 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* clusters */
};

/* Struct for type ET_ECF_EXTERNAL_CFLAG */
struct S1466 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* flag */
	T0* a2; /* description */
	T0* a3; /* conditions */
};

/* Struct for type ET_ECF_EXTERNAL_CFLAGS */
struct S1467 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* external_values */
};

/* Struct for type ET_ECF_EXTERNAL_INCLUDE */
struct S1469 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* pathname */
	T0* a2; /* description */
	T0* a3; /* conditions */
};

/* Struct for type ET_ECF_EXTERNAL_INCLUDES */
struct S1470 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* external_values */
};

/* Struct for type ET_ECF_EXTERNAL_LIBRARY */
struct S1472 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* pathname */
	T0* a2; /* description */
	T0* a3; /* conditions */
};

/* Struct for type ET_ECF_EXTERNAL_LIBRARIES */
struct S1473 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* external_values */
};

/* Struct for type ET_ECF_EXTERNAL_LINKER_FLAG */
struct S1475 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* flag */
	T0* a2; /* description */
	T0* a3; /* conditions */
};

/* Struct for type ET_ECF_EXTERNAL_LINKER_FLAGS */
struct S1476 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* external_values */
};

/* Struct for type ET_ECF_EXTERNAL_MAKE */
struct S1478 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* pathname */
	T0* a2; /* description */
	T0* a3; /* conditions */
};

/* Struct for type ET_ECF_EXTERNAL_MAKES */
struct S1479 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* external_values */
};

/* Struct for type ET_ECF_EXTERNAL_OBJECT */
struct S1481 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* pathname */
	T0* a2; /* description */
	T0* a3; /* conditions */
};

/* Struct for type ET_ECF_EXTERNAL_OBJECTS */
struct S1482 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* external_values */
};

/* Struct for type ET_ECF_EXTERNAL_RESOURCE */
struct S1484 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* pathname */
	T0* a2; /* description */
	T0* a3; /* conditions */
};

/* Struct for type ET_ECF_EXTERNAL_RESOURCES */
struct S1485 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* external_values */
};

/* Struct for type ET_ECF_FILE_RULE */
struct S1487 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* exclude */
	T0* a2; /* include */
	T0* a3; /* exclude_regexp */
	T0* a4; /* include_regexp */
	T0* a5; /* conditions */
	T0* a6; /* description */
};

/* Struct for type ET_ECF_FILE_RULES */
struct S1488 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* file_rules */
};

/* Struct for type ET_ECF_ADAPTED_LIBRARIES */
struct S1490 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* libraries */
};

/* Struct for type ET_ECF_ACTION */
struct S1492 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* command_name */
	T0* a2; /* description */
	T1 a3; /* must_succeed */
	T0* a4; /* working_directory */
	T0* a5; /* conditions */
};

/* Struct for type [attached] DS_ARRAYED_LIST [[attached] ET_ECF_ACTION] */
struct S1493 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* special_routines */
	T0* a2; /* storage */
	T6 a3; /* capacity */
	T6 a4; /* count */
	T0* a5; /* internal_cursor */
};

/* Struct for type [attached] TUPLE [[attached] ET_ECF_OPTIONS] */
struct S1497 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type [attached] PROCEDURE [[attached] TUPLE [[attached] ET_ECF_OPTIONS]] */
struct S1498 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T14 a1; /* rout_disp */
	T0* a2; /* closed_operands */
	T14 a3; /* encaps_rout_disp */
	T14 a4; /* calc_rout_addr */
	T1 a5; /* is_target_closed */
	T6 a6; /* open_count */
};

/* Struct for type ET_ECF_ROOT_ALL_CLASSES */
struct S1500 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type ET_ECF_ROOT_CLASS */
struct S1501 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* class_name */
	T0* a2; /* creation_procedure_name */
	T0* a3; /* cluster_name */
};

/* Struct for type ET_ECF_ORED_CONDITIONS */
struct S1503 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* conditions */
};

/* Struct for type ET_ECF_ANDED_CONDITIONS */
struct S1505 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* conditions */
};

/* Struct for type [attached] DS_HASH_TABLE [[attached] ET_ECF_OPTIONS, [attached] STRING_8] */
struct S1506 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* found_position */
	T0* a2; /* item_storage */
	T6 a3; /* last_position */
	T6 a4; /* position */
	T6 a5; /* capacity */
	T6 a6; /* slots_position */
	T6 a7; /* count */
	T0* a8; /* key_equality_tester */
	T0* a9; /* internal_keys */
	T0* a10; /* equality_tester */
	T6 a11; /* modulus */
	T6 a12; /* clashes_previous_position */
	T0* a13; /* special_item_routines */
	T0* a14; /* clashes */
	T0* a15; /* slots */
	T0* a16; /* special_key_routines */
	T0* a17; /* key_storage */
	T6 a18; /* free_slot */
	T0* a19; /* internal_cursor */
	T0* a20; /* hash_function */
};

/* Struct for type ET_ECF_VISIBLE_CLASS */
struct S1508 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* class_name */
	T0* a2; /* feature_name */
	T0* a3; /* new_class_name */
	T0* a4; /* new_feature_name */
};

/* Struct for type [attached] DS_ARRAYED_LIST [[attached] ET_ECF_VISIBLE_CLASS] */
struct S1509 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* special_routines */
	T0* a2; /* storage */
	T6 a3; /* capacity */
	T6 a4; /* count */
	T0* a5; /* internal_cursor */
};

/* Struct for type ET_ECF_BUILD_CONDITION */
struct S1518 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* value */
	T1 a2; /* is_excluded */
};

/* Struct for type ET_ECF_CONCURRENCY_CONDITION */
struct S1519 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* value */
	T1 a2; /* is_excluded */
};

/* Struct for type ET_ECF_CUSTOM_CONDITION */
struct S1520 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* name */
	T0* a2; /* value */
	T0* a3; /* match */
	T1 a4; /* is_excluded */
};

/* Struct for type ET_ECF_DOTNET_CONDITION */
struct S1521 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* value */
};

/* Struct for type ET_ECF_DYNAMIC_RUNTIME_CONDITION */
struct S1522 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* value */
};

/* Struct for type ET_ECF_PLATFORM_CONDITION */
struct S1523 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* value */
	T1 a2; /* is_excluded */
};

/* Struct for type ET_ECF_COMPILER_VERSION_CONDITION */
struct S1524 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* min_value */
	T0* a2; /* max_value */
};

/* Struct for type ET_ECF_MSIL_CLR_VERSION_CONDITION */
struct S1525 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* min_value */
	T0* a2; /* max_value */
};

/* Struct for type [attached] TUPLE [ET_ECF_SYSTEM] */
struct S1526 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type ET_PARENTHESIS_EXPRESSION */
struct S1528 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* target */
	T0* a2; /* name */
	T0* a3; /* arguments */
	T6 a4; /* index */
};

/* Struct for type ET_REGULAR_MANIFEST_STRING */
struct S1532 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* value */
	T6 a2; /* index */
	T0* a3; /* cast_type */
	T6 a4; /* compressed_position */
	T0* a5; /* type */
};

/* Struct for type ET_INFIX_FREE_NAME */
struct S1533 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* hash_code */
	T6 a2; /* seed */
	T0* a3; /* operator_string */
	T0* a4; /* infix_keyword */
	T2 a5; /* code */
};

/* Struct for type ET_PARENT_CLAUSE_LIST */
struct S1534 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
};

/* Struct for type ET_CREATOR_LIST */
struct S1537 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
};

/* Struct for type [attached] XM_LINKED_LIST [[attached] XM_DOCUMENT_NODE] */
struct S1540 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* last_cell */
	T0* a3; /* first_cell */
	T0* a4; /* internal_cursor */
};

/* Struct for type [attached] XM_LINKED_LIST [[attached] XM_ELEMENT_NODE] */
struct S1542 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* internal_cursor */
	T0* a2; /* last_cell */
	T6 a3; /* count */
	T0* a4; /* first_cell */
};

/* Struct for type UT_TRISTATE */
struct S1546 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T2 a1; /* value */
};

/* Struct for type [attached] KL_AGENT_ROUTINES [[attached] ANY] */
struct S1547 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] TUPLE [UT_TRISTATE] */
struct S1548 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type [attached] TUPLE [[attached] KL_AGENT_ROUTINES [[attached] ANY], [attached] PROCEDURE [[attached] TUPLE]] */
struct S1549 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
	T0* z2;
};

/* Struct for type [attached] PREDICATE [[attached] TUPLE [[attached] ET_MASTER_CLASS]] */
struct S1550 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T14 a1; /* rout_disp */
	T0* a2; /* closed_operands */
	T14 a3; /* encaps_rout_disp */
	T14 a4; /* calc_rout_addr */
	T1 a5; /* is_target_closed */
	T6 a6; /* open_count */
};

/* Struct for type [attached] PREDICATE [[attached] TUPLE] */
struct S1551 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T14 a1; /* rout_disp */
	T0* a2; /* closed_operands */
	T14 a3; /* encaps_rout_disp */
	T14 a4; /* calc_rout_addr */
	T1 a5; /* is_target_closed */
	T6 a6; /* open_count */
};

/* Struct for type [attached] TUPLE [[attached] PROCEDURE [[attached] TUPLE [[attached] ET_MASTER_CLASS]], detachable FUNCTION [[attached] TUPLE, [attached] BOOLEAN]] */
struct S1553 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
	T0* z2;
};

/* Struct for type [attached] TUPLE [[attached] PROCEDURE [[attached] TUPLE [[attached] ET_MASTER_CLASS]], [attached] FUNCTION [[attached] TUPLE [[attached] ET_MASTER_CLASS], [attached] BOOLEAN]] */
struct S1554 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
	T0* z2;
};

/* Struct for type [attached] TUPLE [[attached] PROCEDURE [[attached] TUPLE [[attached] ET_MASTER_CLASS]], [attached] FUNCTION [[attached] TUPLE [[attached] ET_MASTER_CLASS], [attached] BOOLEAN], detachable FUNCTION [[attached] TUPLE, [attached] BOOLEAN]] */
struct S1555 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
	T0* z2;
	T0* z3;
};

/* Struct for type UC_UTF8_ROUTINES */
struct S1556 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type UC_UTF8_STRING */
struct S1557 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* area */
	T6 a3; /* byte_count */
	T6 a4; /* internal_hash_code */
	T6 a5; /* last_byte_index_input */
	T6 a6; /* last_byte_index_result */
	T6 a7; /* internal_case_insensitive_hash_code */
};

/* Struct for type XM_EIFFEL_INPUT_STREAM */
struct S1558 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* last_string */
	T6 a2; /* encoding */
	T0* a3; /* utf_queue */
	T0* a4; /* impl */
};

/* Struct for type [attached] DS_ARRAYED_LIST [[attached] ET_ECF_TARGET] */
struct S1562 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
	T0* a3; /* special_routines */
	T6 a4; /* capacity */
	T0* a5; /* internal_cursor */
};

/* Struct for type [attached] DS_ARRAYED_LIST [[attached] ET_ADAPTED_UNIVERSE] */
struct S1564 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
	T0* a3; /* special_routines */
	T6 a4; /* capacity */
	T0* a5; /* internal_cursor */
};

/* Struct for type [attached] DS_ARRAYED_LIST [[attached] ET_ECF_TARGET_PARENT] */
struct S1566 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
	T0* a3; /* special_routines */
	T6 a4; /* capacity */
	T0* a5; /* internal_cursor */
};

/* Struct for type KL_CHARACTER_BUFFER */
struct S1569 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* as_special */
	T0* a2; /* area */
};

/* Struct for type [attached] SPECIAL [[attached] ET_QUERY] */
struct S1572 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type ET_ALIAS_NAME */
struct S1573 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* alias_keyword */
	T0* a2; /* alias_string */
	T2 a3; /* code */
	T6 a4; /* seed */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_QUERY] */
struct S1574 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [[attached] ET_PROCEDURE] */
struct S1575 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_PROCEDURE] */
struct S1576 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type KL_STRING_INPUT_STREAM */
struct S1577 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* end_of_input */
	T2 a2; /* last_character */
	T0* a3; /* last_string */
	T0* a4; /* string */
	T6 a5; /* location */
};

/* Struct for type ET_TEXT_GROUP */
struct S1578 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type ET_OBJECT_TEST_LIST */
struct S1583 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
};

/* Struct for type ET_ACROSS_COMPONENT_LIST */
struct S1585 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
};

/* Struct for type ET_CREATOR */
struct S1625 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* creation_keyword */
	T0* a2; /* clients */
	T0* a3; /* storage */
	T6 a4; /* count */
};

/* Struct for type ET_FEATURE_CLAUSE */
struct S1631 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* feature_keyword */
	T0* a2; /* clients */
};

/* Struct for type ET_LOCAL_VARIABLE */
struct S1655 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* name_item */
	T0* a2; /* declared_type */
	T1 a3; /* is_used */
};

/* Struct for type ET_ELSEIF_PART */
struct S1675 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* conditional */
	T0* a2; /* then_compound */
};

/* Struct for type ET_ELSEIF_EXPRESSION */
struct S1676 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* conditional */
	T0* a2; /* then_keyword */
	T0* a3; /* then_expression */
};

/* Struct for type ET_WHEN_PART */
struct S1677 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* choices */
	T0* a2; /* then_compound */
};

/* Struct for type ET_CONSTRAINED_FORMAL_PARAMETER */
struct S1712 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* arrow_symbol */
	T0* a2; /* constraint */
	T0* a3; /* creation_procedures */
	T0* a4; /* type_mark */
	T6 a5; /* index */
	T0* a6; /* name */
	T0* a7; /* implementation_class */
	T0* a8; /* constraint_base_type */
};

/* Struct for type ET_FILE_POSITION */
struct S1716 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* compressed_position */
	T0* a2; /* filename */
};

/* Struct for type ET_TAGGED_ASSERTION */
struct S1718 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* tag */
	T0* a2; /* untagged_assertion */
};

/* Struct for type [attached] DS_QUICK_SORTER [[attached] ET_QUERY] */
struct S1721 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* comparator */
};

/* Struct for type [attached] DS_QUICK_SORTER [[attached] ET_PROCEDURE] */
struct S1722 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* comparator */
};

/* Struct for type ET_REDECLARED_FEATURE */
struct S1724 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* next */
	T0* a2; /* flattened_feature */
	T0* a3; /* parent_feature */
	T6 a4; /* first_seed */
	T0* a5; /* other_seeds */
	T1 a6; /* is_selected */
	T0* a7; /* replicated_features */
	T0* a8; /* replicated_seeds */
};

/* Struct for type ET_CONVERT_FEATURE_LIST */
struct S1726 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* convert_keyword */
	T6 a2; /* count */
	T0* a3; /* storage */
};

/* Struct for type ET_ALIASED_FEATURE_NAME */
struct S1728 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* feature_name */
	T0* a2; /* alias_name */
};

/* Struct for type ET_FEATURE_IDS */
struct S1730 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* feature_ids */
};

/* Struct for type ET_RENAME */
struct S1731 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* old_name */
	T0* a2; /* as_keyword */
	T0* a3; /* new_name */
};

/* Struct for type ET_POSTCONDITIONS */
struct S1734 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* validity_checked */
	T1 a2; /* has_validity_error */
	T6 a3; /* count */
	T0* a4; /* storage */
	T0* a5; /* ensure_keyword */
	T0* a6; /* then_keyword */
};

/* Struct for type ET_INVARIANTS */
struct S1735 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* implementation_class */
	T0* a2; /* invariant_keyword */
	T0* a3; /* storage */
	T6 a4; /* count */
	T0* a5; /* object_tests */
	T0* a6; /* across_components */
	T1 a7; /* validity_checked */
	T1 a8; /* has_validity_error */
};

/* Struct for type CONSOLE */
struct S1748 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T14 a1; /* file_pointer */
	T6 a2; /* mode */
	T0* a3; /* last_string */
	T0* a4; /* internal_name */
	T0* a5; /* internal_detachable_name_pointer */
	T2 a6; /* last_character */
};

/* Struct for type ET_LIKE_N */
struct S1749 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* index */
	T0* a2; /* previous */
	T0* a3; /* type_mark */
	T0* a4; /* like_keyword */
};

/* Struct for type [attached] DS_HASH_TABLE_CURSOR [[attached] ET_DYNAMIC_TYPE_SET, [attached] ET_DYNAMIC_TYPE] */
struct S1750 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* next_cursor */
	T0* a3; /* container */
};

/* Struct for type [attached] SPECIAL [[attached] ET_BASE_TYPE] */
struct S1753 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_BASE_TYPE] */
struct S1754 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] ARRAY [[attached] IMMUTABLE_STRING_32] */
struct S1755 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* area */
	T6 a2; /* lower */
	T6 a3; /* upper */
	T1 a4; /* object_comparison */
};

/* Struct for type [attached] SPECIAL [[attached] IMMUTABLE_STRING_32] */
struct S1756 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] DS_HASH_TABLE [detachable DS_HASH_SET [[attached] STRING_8], [attached] STRING_8] */
struct S1757 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* found_position */
	T0* a2; /* item_storage */
	T6 a3; /* position */
	T0* a4; /* key_equality_tester */
	T0* a5; /* internal_keys */
	T0* a6; /* hash_function */
	T6 a7; /* last_position */
	T6 a8; /* capacity */
	T6 a9; /* slots_position */
	T6 a10; /* count */
	T0* a11; /* equality_tester */
	T6 a12; /* modulus */
	T6 a13; /* clashes_previous_position */
	T0* a14; /* special_item_routines */
	T0* a15; /* clashes */
	T0* a16; /* slots */
	T0* a17; /* special_key_routines */
	T0* a18; /* key_storage */
	T6 a19; /* free_slot */
	T0* a20; /* internal_cursor */
};

/* Struct for type [attached] DS_ARRAYED_LIST [[attached] ET_XACE_TARGET] */
struct S1758 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
	T0* a3; /* special_routines */
	T6 a4; /* capacity */
	T0* a5; /* internal_cursor */
};

/* Struct for type [attached] DS_ARRAYED_LIST [[attached] ET_XACE_CLUSTER] */
struct S1759 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
	T0* a3; /* special_routines */
	T6 a4; /* capacity */
	T0* a5; /* internal_cursor */
};

/* Struct for type [attached] DS_LINKED_LIST [[attached] ET_XACE_CLASS_OPTIONS] */
struct S1762 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* last_cell */
	T6 a2; /* count */
	T0* a3; /* first_cell */
	T0* a4; /* internal_cursor */
};

/* Struct for type [attached] DS_ARRAYED_STACK [[attached] YY_BUFFER] */
struct S1763 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* a3; /* storage */
	T0* a4; /* special_routines */
};

/* Struct for type UT_SYNTAX_ERROR */
struct S1764 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* parameters */
};

/* Struct for type UT_TOO_MANY_INCLUDES_ERROR */
struct S1765 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* parameters */
};

/* Struct for type ET_DYNAMIC_TARGET_LIST */
struct S1768 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
};

/* Struct for type [attached] SPECIAL [[attached] NATIVE_STRING] */
struct S1771 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] ARRAY [detachable STRING_8] */
struct S1772 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* area */
	T6 a2; /* lower */
	T6 a3; /* upper */
};

/* Struct for type XM_NAMESPACE_RESOLVER_CONTEXT */
struct S1773 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* context */
};

/* Struct for type [attached] DS_LINKED_QUEUE [detachable STRING_8] */
struct S1776 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* first_cell */
	T6 a2; /* count */
	T0* a3; /* last_cell */
};

/* Struct for type [attached] DS_LINKED_QUEUE [[attached] STRING_8] */
struct S1777 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* first_cell */
	T0* a2; /* last_cell */
	T6 a3; /* count */
};

/* Struct for type [attached] DS_LINKED_LIST_CURSOR [[attached] DS_PAIR [[attached] XM_POSITION, [attached] XM_NODE]] */
struct S1778 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* after */
	T0* a2; /* current_cell */
	T0* a3; /* container */
	T1 a4; /* before */
	T0* a5; /* next_cursor */
};

/* Struct for type [attached] DS_LINKED_LIST [[attached] DS_PAIR [[attached] XM_POSITION, [attached] XM_NODE]] */
struct S1779 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* last_cell */
	T6 a2; /* count */
	T0* a3; /* first_cell */
	T0* a4; /* internal_cursor */
};

/* Struct for type [attached] DS_PAIR [[attached] XM_POSITION, [attached] XM_NODE] */
struct S1780 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* first */
	T0* a2; /* second */
};

/* Struct for type [attached] KL_EQUALITY_TESTER [[attached] XM_NAMESPACE] */
struct S1783 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] DS_HASH_SET_CURSOR [[attached] XM_NAMESPACE] */
struct S1784 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] XM_NAMESPACE] */
struct S1785 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [[attached] XM_NAMESPACE] */
struct S1786 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] DS_HASH_SET_CURSOR [[attached] ET_DOTNET_ASSEMBLY] */
struct S1787 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_DOTNET_ASSEMBLY] */
struct S1788 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [[attached] ET_DOTNET_ASSEMBLY] */
struct S1789 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] DS_ARRAYED_LIST [[attached] ET_DOTNET_ASSEMBLY] */
struct S1792 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
	T0* a3; /* special_routines */
	T6 a4; /* capacity */
	T0* a5; /* internal_cursor */
	T0* a6; /* equality_tester */
};

/* Struct for type [attached] SPECIAL [[attached] ET_XACE_MOUNTED_LIBRARY] */
struct S1793 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_XACE_MOUNTED_LIBRARY] */
struct S1794 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_XACE_MOUNTED_LIBRARY] */
struct S1795 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
};

/* Struct for type [attached] DS_ARRAYED_LIST [[attached] ET_MASTER_CLASS] */
struct S1796 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
	T0* a3; /* special_routines */
	T6 a4; /* capacity */
	T0* a5; /* equality_tester */
	T0* a6; /* internal_cursor */
};

/* Struct for type [attached] DS_HASH_SET_CURSOR [[attached] ET_INTERNAL_UNIVERSE] */
struct S1797 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_INTERNAL_UNIVERSE] */
struct S1801 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [[attached] ET_INTERNAL_UNIVERSE] */
struct S1802 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type ET_CLASS_TYPE_STATUS_CHECKER1 */
struct S1803 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* has_fatal_error */
};

/* Struct for type ET_CLASS_TYPE_STATUS_CHECKER2 */
struct S1804 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* has_fatal_error */
};

/* Struct for type ET_QUALIFIED_ANCHORED_TYPE_STATUS_CHECKER */
struct S1805 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* has_fatal_error */
	T0* a2; /* current_class */
	T0* a3; /* system_processor */
};

/* Struct for type ET_CLASS_TYPE_STATUS_CHECKER3 */
struct S1806 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* has_fatal_error */
};

/* Struct for type [attached] DS_HASH_SET_CURSOR [[attached] ET_NAMED_CLASS] */
struct S1807 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* next_cursor */
	T0* a3; /* container */
};

/* Struct for type [attached] DS_HASH_SET_CURSOR [[attached] ET_UNIVERSE] */
struct S1808 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* next_cursor */
	T0* a3; /* container */
};

/* Struct for type [attached] SPECIAL [[attached] ET_UNIVERSE] */
struct S1809 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_UNIVERSE] */
struct S1812 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [[attached] ET_PARENT_ITEM] */
struct S1814 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_PARENT_ITEM] */
struct S1816 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [[attached] ET_CLIENT_ITEM] */
struct S1817 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_CLIENT_ITEM] */
struct S1818 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type ET_BRACED_TYPE_LIST */
struct S1820 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* left_brace */
	T0* a2; /* right_brace */
	T6 a3; /* count */
	T0* a4; /* storage */
};

/* Struct for type XM_EIFFEL_CHARACTER_ENTITY */
struct S1822 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* code */
};

/* Struct for type attached DS_BILINKABLE [[attached] XM_POSITION] */
struct S1825 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* item */
	T0* a2; /* right */
	T0* a3; /* left */
};

/* Struct for type [attached] DS_BILINKED_LIST_CURSOR [[attached] XM_POSITION] */
struct S1826 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T1 a2; /* before */
};

/* Struct for type attached DS_LINKABLE [[attached] XM_EIFFEL_SCANNER] */
struct S1828 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* item */
	T0* a2; /* right */
};

/* Struct for type [attached] SPECIAL [[attached] XM_EIFFEL_ENTITY_DEF] */
struct S1829 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] DS_HASH_TABLE_CURSOR [[attached] XM_EIFFEL_ENTITY_DEF, [attached] STRING_8] */
struct S1834 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* next_cursor */
	T0* a2; /* container */
	T6 a3; /* position */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] XM_EIFFEL_ENTITY_DEF] */
struct S1835 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] KL_EQUALITY_TESTER [[attached] XM_EIFFEL_PARSER_NAME] */
struct S1839 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] DS_HASH_SET_CURSOR [[attached] XM_EIFFEL_PARSER_NAME] */
struct S1840 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
};

/* Struct for type [attached] DS_BILINKED_LIST [[attached] XM_DTD_ELEMENT_CONTENT] */
struct S1841 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* first_cell */
	T6 a2; /* count */
	T0* a3; /* last_cell */
	T0* a4; /* internal_cursor */
};

/* Struct for type [attached] DS_BILINKED_LIST_CURSOR [[attached] XM_DTD_ATTRIBUTE_CONTENT] */
struct S1842 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* after */
	T0* a2; /* current_cell */
	T0* a3; /* container */
	T1 a4; /* before */
	T0* a5; /* next_cursor */
};

/* Struct for type attached DS_BILINKABLE [[attached] XM_DTD_ATTRIBUTE_CONTENT] */
struct S1844 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* item */
	T0* a2; /* right */
	T0* a3; /* left */
};

/* Struct for type [attached] DS_BILINKED_LIST_CURSOR [[attached] STRING_8] */
struct S1846 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* after */
	T0* a2; /* current_cell */
	T0* a3; /* container */
	T1 a4; /* before */
	T0* a5; /* next_cursor */
};

/* Struct for type attached DS_BILINKABLE [[attached] STRING_8] */
struct S1848 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* item */
	T0* a2; /* right */
	T0* a3; /* left */
};

/* Struct for type [attached] DS_ARRAYED_LIST [[attached] ET_CLUSTER] */
struct S1849 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
	T0* a3; /* special_routines */
	T6 a4; /* capacity */
	T0* a5; /* internal_cursor */
};

/* Struct for type [attached] SPECIAL [[attached] ET_ADAPTED_LIBRARY] */
struct S1850 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_ADAPTED_LIBRARY] */
struct S1851 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_ADAPTED_LIBRARY] */
struct S1852 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
};

/* Struct for type [attached] DS_HASH_SET_CURSOR [[attached] ET_LIBRARY] */
struct S1854 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_LIBRARY] */
struct S1855 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [[attached] ET_LIBRARY] */
struct S1856 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] SPECIAL [[attached] ET_ADAPTED_DOTNET_ASSEMBLY] */
struct S1857 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_ADAPTED_DOTNET_ASSEMBLY] */
struct S1858 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_ADAPTED_DOTNET_ASSEMBLY] */
struct S1859 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
};

/* Struct for type [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_LACE_CLUSTER] */
struct S1861 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* next_cursor */
	T0* a3; /* container */
};

/* Struct for type [attached] SPECIAL [detachable ET_OBJECT_TEST_LIST] */
struct S1862 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_OBJECT_TEST_LIST] */
struct S1863 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_OBJECT_TEST_LIST] */
struct S1864 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [[attached] ET_OBJECT_TEST_LIST] */
struct S1865 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] SPECIAL [detachable ET_ACROSS_COMPONENT_LIST] */
struct S1866 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_ACROSS_COMPONENT_LIST] */
struct S1867 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_ACROSS_COMPONENT_LIST] */
struct S1868 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [[attached] ET_ACROSS_COMPONENT_LIST] */
struct S1869 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] SPECIAL [[attached] ET_ASSERTION_ITEM] */
struct S1870 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_ASSERTION_ITEM] */
struct S1871 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_ASSERTION_ITEM] */
struct S1872 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* next_cursor */
	T6 a2; /* position */
	T0* a3; /* container */
};

/* Struct for type [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_QUERY] */
struct S1874 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* next_cursor */
	T0* a3; /* container */
};

/* Struct for type [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_PROCEDURE] */
struct S1876 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* next_cursor */
	T0* a3; /* container */
};

/* Struct for type [attached] DS_ARRAYED_LIST_CURSOR [detachable ET_CONSTRAINT_TYPE] */
struct S1878 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* next_cursor */
	T0* a2; /* container */
	T6 a3; /* position */
};

/* Struct for type [attached] SPECIAL [[attached] ET_NAMED_CLASS] */
struct S1880 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_NAMED_CLASS] */
struct S1882 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] DS_HASH_TABLE [[attached] INTEGER_32, [attached] ET_CLASS] */
struct S1886 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T6 a2; /* found_position */
	T0* a3; /* item_storage */
	T6 a4; /* count */
	T6 a5; /* last_position */
	T6 a6; /* free_slot */
	T6 a7; /* modulus */
	T6 a8; /* capacity */
	T6 a9; /* slots_position */
	T6 a10; /* clashes_previous_position */
	T0* a11; /* key_equality_tester */
	T0* a12; /* equality_tester */
	T0* a13; /* internal_cursor */
	T0* a14; /* key_storage */
	T0* a15; /* clashes */
	T0* a16; /* slots */
	T0* a17; /* special_item_routines */
	T0* a18; /* special_key_routines */
	T0* a19; /* hash_function */
};

/* Struct for type [attached] DS_HASH_TABLE_CURSOR [[attached] ET_BASE_TYPE, [attached] ET_CLASS] */
struct S1887 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* next_cursor */
	T0* a3; /* container */
};

/* Struct for type [attached] DS_HASH_TOPOLOGICAL_SORTER [[attached] ET_FORMAL_PARAMETER] */
struct S1892 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* cycle */
	T0* a2; /* sorted_items */
	T0* a3; /* items */
	T0* a4; /* counts */
	T0* a5; /* successors */
	T0* a6; /* indexes */
};

/* Struct for type attached DS_ARRAYED_LIST [[attached] ET_FORMAL_PARAMETER] */
struct S1894 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
	T0* a3; /* special_routines */
	T6 a4; /* capacity */
	T0* a5; /* internal_cursor */
};

/* Struct for type ET_FORMAL_PARAMETER_TYPE */
struct S1895 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* implementation_class */
	T6 a2; /* index */
	T0* a3; /* type_mark */
	T0* a4; /* name */
};

/* Struct for type [attached] DS_HASH_TABLE_CURSOR [[attached] ET_FLATTENED_FEATURE, [attached] ET_FEATURE_NAME] */
struct S1896 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* next_cursor */
	T0* a3; /* container */
};

/* Struct for type [attached] SPECIAL [[attached] ET_FLATTENED_FEATURE] */
struct S1897 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] SPECIAL [[attached] ET_FEATURE_NAME] */
struct S1903 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_FLATTENED_FEATURE] */
struct S1904 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_FEATURE_NAME] */
struct S1905 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_ALIAS_NAME] */
struct S1909 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [[attached] ET_ALIAS_NAME] */
struct S1910 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] DS_HASH_TABLE_CURSOR [[attached] ET_FLATTENED_FEATURE, [attached] ET_ALIAS_NAME] */
struct S1912 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* next_cursor */
	T0* a2; /* container */
	T6 a3; /* position */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_CLIENT_LIST] */
struct S1913 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [[attached] ET_CLIENT_LIST] */
struct S1914 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_CLIENT_LIST] */
struct S1915 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* next_cursor */
	T0* a3; /* container */
};

/* Struct for type [attached] DS_HASH_TABLE_CURSOR [[attached] ET_CLIENT, [attached] ET_CLASS] */
struct S1917 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* next_cursor */
	T0* a3; /* container */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_CLIENT] */
struct S1918 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [[attached] ET_CLIENT] */
struct S1919 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] DS_HASH_TABLE [[attached] ET_RENAME, [attached] ET_CALL_NAME] */
struct S1920 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* capacity */
	T6 a2; /* found_position */
	T0* a3; /* internal_cursor */
	T6 a4; /* count */
	T0* a5; /* item_storage */
	T0* a6; /* key_equality_tester */
	T0* a7; /* internal_keys */
	T6 a8; /* position */
	T6 a9; /* last_position */
	T6 a10; /* free_slot */
	T6 a11; /* modulus */
	T0* a12; /* equality_tester */
	T6 a13; /* slots_position */
	T6 a14; /* clashes_previous_position */
	T0* a15; /* key_storage */
	T0* a16; /* clashes */
	T0* a17; /* slots */
	T0* a18; /* special_item_routines */
	T0* a19; /* special_key_routines */
	T0* a20; /* hash_function */
};

/* Struct for type ET_CALL_NAME_TESTER */
struct S1921 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] DS_HASH_SET [[attached] ET_CALL_NAME] */
struct S1923 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* capacity */
	T6 a2; /* found_position */
	T0* a3; /* internal_cursor */
	T6 a4; /* count */
	T0* a5; /* item_storage */
	T6 a6; /* modulus */
	T6 a7; /* last_position */
	T6 a8; /* free_slot */
	T6 a9; /* position */
	T0* a10; /* equality_tester */
	T0* a11; /* special_item_routines */
	T0* a12; /* clashes */
	T0* a13; /* slots */
	T6 a14; /* slots_position */
	T6 a15; /* clashes_previous_position */
	T0* a16; /* hash_function */
};

/* Struct for type [attached] DS_HASH_TABLE [[attached] BOOLEAN, [attached] ET_CALL_NAME] */
struct S1924 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T6 a3; /* found_position */
	T0* a4; /* internal_cursor */
	T0* a5; /* key_storage */
	T0* a6; /* item_storage */
	T0* a7; /* key_equality_tester */
	T0* a8; /* internal_keys */
	T6 a9; /* position */
	T6 a10; /* last_position */
	T6 a11; /* free_slot */
	T6 a12; /* modulus */
	T0* a13; /* equality_tester */
	T6 a14; /* slots_position */
	T6 a15; /* clashes_previous_position */
	T0* a16; /* special_item_routines */
	T0* a17; /* clashes */
	T0* a18; /* slots */
	T0* a19; /* special_key_routines */
	T0* a20; /* hash_function */
};

/* Struct for type [attached] DS_HASH_TABLE [[attached] ET_FEATURE_NAME, [attached] ET_ALIAS_NAME] */
struct S1925 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T6 a2; /* found_position */
	T0* a3; /* item_storage */
	T0* a4; /* key_equality_tester */
	T0* a5; /* internal_keys */
	T6 a6; /* count */
	T6 a7; /* last_position */
	T6 a8; /* free_slot */
	T6 a9; /* capacity */
	T6 a10; /* modulus */
	T6 a11; /* slots_position */
	T6 a12; /* clashes_previous_position */
	T0* a13; /* equality_tester */
	T0* a14; /* internal_cursor */
	T0* a15; /* key_storage */
	T0* a16; /* clashes */
	T0* a17; /* slots */
	T0* a18; /* special_item_routines */
	T0* a19; /* special_key_routines */
	T0* a20; /* hash_function */
};

/* Struct for type [attached] DS_HASH_TABLE [[attached] ET_REPLICABLE_FEATURE, [attached] INTEGER_32] */
struct S1926 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* internal_cursor */
	T6 a2; /* found_position */
	T0* a3; /* item_storage */
	T0* a4; /* key_storage */
	T6 a5; /* count */
	T6 a6; /* last_position */
	T6 a7; /* free_slot */
	T6 a8; /* position */
	T6 a9; /* capacity */
	T0* a10; /* equality_tester */
	T0* a11; /* key_equality_tester */
	T0* a12; /* clashes */
	T6 a13; /* modulus */
	T0* a14; /* slots */
	T6 a15; /* slots_position */
	T6 a16; /* clashes_previous_position */
	T0* a17; /* special_item_routines */
	T0* a18; /* special_key_routines */
	T0* a19; /* hash_function */
};

/* Struct for type ET_REPLICATED_FEATURE */
struct S1930 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* features */
	T6 a2; /* selected_count */
};

/* Struct for type ET_RENAME_LIST */
struct S1931 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* rename_keyword */
	T6 a2; /* count */
	T0* a3; /* storage */
};

/* Struct for type ET_EXPORT_LIST */
struct S1932 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* export_keyword */
	T6 a2; /* count */
	T0* a3; /* storage */
};

/* Struct for type ET_ALL_EXPORT */
struct S1933 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* clients_clause */
	T0* a2; /* all_keyword */
};

/* Struct for type ET_FEATURE_EXPORT */
struct S1934 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* clients_clause */
	T0* a2; /* storage */
	T6 a3; /* count */
};

/* Struct for type ET_KEYWORD_FEATURE_NAME_LIST */
struct S1935 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* keyword */
	T6 a2; /* count */
	T0* a3; /* storage */
};

/* Struct for type [attached] DS_LINKED_LIST [[attached] ET_ADAPTED_FEATURE] */
struct S1936 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* last_cell */
	T6 a2; /* count */
	T0* a3; /* first_cell */
	T0* a4; /* internal_cursor */
};

/* Struct for type [attached] DS_ARRAYED_LIST [[attached] ET_PARENT_FEATURE] */
struct S1937 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* special_routines */
	T0* a2; /* storage */
	T6 a3; /* capacity */
	T6 a4; /* count */
	T0* a5; /* internal_cursor */
};

/* Struct for type ET_DOTNET_SIGNATURE_TESTER */
struct S1938 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] DS_HASH_SET [[attached] ET_DOTNET_FEATURE] */
struct S1939 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* capacity */
	T6 a2; /* found_position */
	T0* a3; /* item_storage */
	T6 a4; /* modulus */
	T6 a5; /* last_position */
	T6 a6; /* free_slot */
	T6 a7; /* position */
	T0* a8; /* equality_tester */
	T6 a9; /* count */
	T6 a10; /* slots_position */
	T0* a11; /* special_item_routines */
	T0* a12; /* clashes */
	T0* a13; /* slots */
	T0* a14; /* internal_cursor */
	T6 a15; /* clashes_previous_position */
	T0* a16; /* hash_function */
};

/* Struct for type [attached] DS_HASH_TABLE [[attached] DS_LINKED_LIST [[attached] ET_DOTNET_FEATURE], [attached] ET_DOTNET_FEATURE] */
struct S1941 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* found_position */
	T0* a2; /* item_storage */
	T0* a3; /* key_equality_tester */
	T0* a4; /* internal_keys */
	T6 a5; /* count */
	T6 a6; /* last_position */
	T6 a7; /* free_slot */
	T6 a8; /* position */
	T6 a9; /* capacity */
	T0* a10; /* equality_tester */
	T0* a11; /* internal_cursor */
	T0* a12; /* key_storage */
	T0* a13; /* clashes */
	T6 a14; /* modulus */
	T0* a15; /* slots */
	T6 a16; /* slots_position */
	T6 a17; /* clashes_previous_position */
	T0* a18; /* special_item_routines */
	T0* a19; /* special_key_routines */
	T0* a20; /* hash_function */
};

/* Struct for type [attached] DS_LINKED_LIST [[attached] ET_DOTNET_FEATURE] */
struct S1943 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* last_cell */
	T6 a2; /* count */
	T0* a3; /* first_cell */
	T0* a4; /* internal_cursor */
};

/* Struct for type [attached] DS_HASH_TOPOLOGICAL_SORTER [[attached] ET_LIKE_FEATURE] */
struct S1944 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* cycle */
	T0* a2; /* counts */
	T0* a3; /* successors */
	T0* a4; /* items */
	T0* a5; /* sorted_items */
	T0* a6; /* indexes */
};

/* Struct for type attached DS_ARRAYED_LIST [[attached] ET_LIKE_FEATURE] */
struct S1946 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
	T0* a3; /* special_routines */
	T6 a4; /* capacity */
	T0* a5; /* internal_cursor */
};

/* Struct for type [attached] DS_LINKED_LIST_CURSOR [[attached] ET_PARENT_FEATURE] */
struct S1947 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* after */
	T0* a2; /* current_cell */
	T0* a3; /* container */
	T1 a4; /* before */
	T0* a5; /* next_cursor */
};

/* Struct for type attached DS_LINKED_LIST [[attached] ET_PARENT_FEATURE] */
struct S1949 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* last_cell */
	T6 a2; /* count */
	T0* a3; /* first_cell */
	T0* a4; /* internal_cursor */
};

/* Struct for type [attached] DS_HASH_TABLE [[attached] DS_HASH_TABLE [[attached] TUPLE [detachable ARRAY [[attached] ET_TYPE], detachable ET_TYPE, [attached] NATURAL_8], [attached] ET_FEATURE_NAME], [attached] NATURAL_8] */
struct S1950 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* equality_tester */
	T0* a2; /* key_equality_tester */
	T6 a3; /* capacity */
	T6 a4; /* modulus */
	T6 a5; /* last_position */
	T6 a6; /* free_slot */
	T6 a7; /* position */
	T0* a8; /* special_item_routines */
	T0* a9; /* item_storage */
	T0* a10; /* special_key_routines */
	T0* a11; /* key_storage */
	T0* a12; /* clashes */
	T0* a13; /* slots */
	T6 a14; /* found_position */
	T0* a15; /* internal_cursor */
	T6 a16; /* slots_position */
	T6 a17; /* count */
	T6 a18; /* clashes_previous_position */
	T0* a19; /* hash_function */
};

/* Struct for type [attached] DS_HASH_TABLE_CURSOR [[attached] ET_FEATURE, [attached] INTEGER_32] */
struct S1951 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* next_cursor */
	T0* a3; /* container */
};

/* Struct for type [attached] DS_HASH_SET_CURSOR [[attached] ET_CLASS] */
struct S1956 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* next_cursor */
	T0* a3; /* container */
};

/* Struct for type [attached] DS_HASH_SET_CURSOR [[attached] ET_PROCEDURE] */
struct S1959 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* next_cursor */
	T0* a3; /* container */
};

/* Struct for type [attached] DS_HASH_SET_CURSOR [[attached] ET_QUERY] */
struct S1961 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* next_cursor */
	T0* a3; /* container */
};

/* Struct for type [attached] SPECIAL [detachable DS_LINKABLE [[attached] INTEGER_32]] */
struct S1963 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable DS_LINKABLE [[attached] INTEGER_32]] */
struct S1964 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] DS_ARRAYED_LIST_CURSOR [detachable DS_LINKABLE [[attached] INTEGER_32]] */
struct S1965 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* next_cursor */
	T0* a2; /* container */
	T6 a3; /* position */
};

/* Struct for type [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_NAMED_OBJECT_TEST] */
struct S1968 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* next_cursor */
	T0* a2; /* container */
	T6 a3; /* position */
};

/* Struct for type [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_ACROSS_COMPONENT] */
struct S1970 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* next_cursor */
	T0* a2; /* container */
	T6 a3; /* position */
};

/* Struct for type [attached] DS_HASH_SET_CURSOR [[attached] INTEGER_32] */
struct S1972 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* next_cursor */
	T0* a3; /* container */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] NATURAL_8] */
struct S1978 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] DS_HASH_TABLE_CURSOR [[attached] NATURAL_8, [attached] ET_CLASS_NAME] */
struct S1979 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
};

/* Struct for type STRING_8_SEARCHER */
struct S1980 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* deltas */
};

/* Struct for type attached ARRAYED_LIST [[attached] INTEGER_32] */
struct S1982 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* area_v2 */
	T6 a2; /* index */
};

/* Struct for type ET_XACE_INVALID_EXPRESSION_ERROR */
struct S1986 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* parameters */
};

/* Struct for type [attached] DS_HASH_TABLE [[attached] INTEGER_32, [attached] ET_UNIVERSE] */
struct S1988 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T6 a2; /* last_position */
	T6 a3; /* capacity */
	T6 a4; /* slots_position */
	T6 a5; /* count */
	T6 a6; /* modulus */
	T6 a7; /* clashes_previous_position */
	T0* a8; /* key_equality_tester */
	T0* a9; /* equality_tester */
	T6 a10; /* found_position */
	T0* a11; /* special_item_routines */
	T0* a12; /* item_storage */
	T0* a13; /* clashes */
	T0* a14; /* slots */
	T0* a15; /* special_key_routines */
	T0* a16; /* key_storage */
	T6 a17; /* free_slot */
	T0* a18; /* internal_cursor */
	T0* a19; /* hash_function */
};

/* Struct for type [attached] DS_HASH_TABLE_CURSOR [[attached] INTEGER_32, [attached] ET_UNIVERSE] */
struct S1989 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* container */
	T0* a3; /* next_cursor */
};

/* Struct for type [attached] TUPLE [ET_ECF_SYSTEM, [attached] DS_HASH_TABLE [[attached] INTEGER_32, [attached] ET_UNIVERSE], INTEGER_32] */
struct S1990 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
	T0* z2;
	T6 z3;
};

/* Struct for type [attached] KL_AGENT_ROUTINES [[attached] ET_UNIVERSE] */
struct S1991 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] PREDICATE [[attached] TUPLE [[attached] ET_UNIVERSE]] */
struct S1992 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T14 a1; /* rout_disp */
	T0* a2; /* closed_operands */
	T14 a3; /* encaps_rout_disp */
	T14 a4; /* calc_rout_addr */
	T1 a5; /* is_target_closed */
	T6 a6; /* open_count */
};

/* Struct for type [attached] TUPLE [[attached] DS_HASH_TABLE [[attached] INTEGER_32, [attached] ET_UNIVERSE]] */
struct S1993 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type [attached] TUPLE [[attached] KL_AGENT_ROUTINES [[attached] ET_UNIVERSE], [attached] PREDICATE [[attached] TUPLE [[attached] ET_UNIVERSE]]] */
struct S1994 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
	T0* z2;
};

/* Struct for type ET_DYNAMIC_AGENT_OPERAND_PUSH_TYPE_SET */
struct S1998 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* static_type */
	T6 a2; /* count */
	T1 a3; /* is_never_void */
	T0* a4; /* dynamic_types */
	T0* a5; /* agent_type */
	T1 a6; /* is_dynamic_types_readonly */
};

/* Struct for type [attached] DS_LINKED_LIST_CURSOR [[attached] XM_DOCUMENT_NODE] */
struct S2001 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* after */
	T0* a2; /* current_cell */
	T0* a3; /* container */
	T1 a4; /* before */
	T0* a5; /* next_cursor */
};

/* Struct for type XM_COMMENT */
struct S2002 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* data */
	T0* a2; /* parent */
};

/* Struct for type XM_PROCESSING_INSTRUCTION */
struct S2003 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* target */
	T0* a2; /* data */
	T0* a3; /* parent */
};

/* Struct for type XM_CHARACTER_DATA */
struct S2004 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* content */
	T0* a2; /* parent */
};

/* Struct for type [attached] DS_CELL [detachable ET_CLASS] */
struct S2009 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* item */
};

/* Struct for type [attached] TUPLE [detachable DS_CELL [detachable ET_CLASS]] */
struct S2010 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type ST_SPLITTER */
struct S2012 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* has_escape_character */
	T2 a2; /* escape_character */
	T0* a3; /* separator_codes */
	T0* a4; /* separators */
};

/* Struct for type ET_CLUSTER_DEPENDENCE_CONSTRAINT */
struct S2014 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* current_cluster */
	T0* a2; /* group_pathnames */
	T0* a3; /* group_names */
	T0* a4; /* pathname_buffer */
};

/* Struct for type [attached] TUPLE [ET_EIFFEL_PREPARSER, [attached] DS_ARRAYED_LIST [[attached] STRING_8]] */
struct S2015 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
	T0* z2;
};

/* Struct for type [attached] DS_HASH_TABLE [[attached] INTEGER_32, [attached] STRING_8] */
struct S2022 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* found_position */
	T0* a2; /* key_storage */
	T0* a3; /* item_storage */
	T6 a4; /* position */
	T6 a5; /* count */
	T6 a6; /* capacity */
	T6 a7; /* free_slot */
	T6 a8; /* last_position */
	T0* a9; /* key_equality_tester */
	T0* a10; /* internal_keys */
	T0* a11; /* hash_function */
	T6 a12; /* slots_position */
	T0* a13; /* equality_tester */
	T6 a14; /* modulus */
	T6 a15; /* clashes_previous_position */
	T0* a16; /* clashes */
	T0* a17; /* slots */
	T0* a18; /* special_item_routines */
	T0* a19; /* special_key_routines */
	T0* a20; /* internal_cursor */
};

/* Struct for type ET_LOCAL_VARIABLE_LIST */
struct S2027 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
	T0* a3; /* local_keyword */
};

/* Struct for type ET_TRUE_CONSTANT */
struct S2032 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* index */
	T6 a2; /* compressed_position */
	T2 a3; /* code */
	T0* a4; /* text */
};

/* Struct for type ET_SEEDED_QUERY_COMPARATOR */
struct S2034 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type ET_SEEDED_PROCEDURE_COMPARATOR */
struct S2037 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type ET_CLIENTS */
struct S2040 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
	T0* a3; /* left_brace */
	T0* a4; /* right_brace */
};

/* Struct for type [attached] TUPLE [detachable ET_CLIENT_LIST] */
struct S2042 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type ET_FREE_OPERATOR */
struct S2044 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T2 a1; /* code */
	T6 a2; /* hash_code */
	T6 a3; /* compressed_position */
	T0* a4; /* operator_name */
	T6 a5; /* seed */
};

/* Struct for type ET_C3_CHARACTER_CONSTANT */
struct S2045 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* index */
	T0* a2; /* cast_type */
	T6 a3; /* compressed_position */
	T0* a4; /* literal */
	T3 a5; /* value */
	T0* a6; /* type */
};

/* Struct for type ET_VERBATIM_STRING */
struct S2046 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* value */
	T6 a2; /* index */
	T0* a3; /* cast_type */
	T6 a4; /* compressed_position */
	T0* a5; /* literal */
	T0* a6; /* marker */
	T0* a7; /* open_white_characters */
	T0* a8; /* close_white_characters */
	T1 a9; /* is_left_aligned */
	T0* a10; /* type */
};

/* Struct for type ET_SPECIAL_MANIFEST_STRING */
struct S2047 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* value */
	T6 a2; /* index */
	T0* a3; /* cast_type */
	T6 a4; /* compressed_position */
	T0* a5; /* literal */
	T0* a6; /* type */
};

/* Struct for type ET_UNDERSCORED_INTEGER_CONSTANT */
struct S2048 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* sign */
	T6 a2; /* index */
	T0* a3; /* cast_type */
	T11 a4; /* value */
	T1 a5; /* has_overflow */
	T6 a6; /* compressed_position */
	T0* a7; /* literal */
	T0* a8; /* type */
};

/* Struct for type ET_HEXADECIMAL_INTEGER_CONSTANT */
struct S2049 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* sign */
	T6 a2; /* index */
	T0* a3; /* cast_type */
	T11 a4; /* value */
	T1 a5; /* has_overflow */
	T6 a6; /* compressed_position */
	T0* a7; /* literal */
	T0* a8; /* type */
};

/* Struct for type ET_OCTAL_INTEGER_CONSTANT */
struct S2050 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* sign */
	T6 a2; /* index */
	T0* a3; /* cast_type */
	T11 a4; /* value */
	T1 a5; /* has_overflow */
	T6 a6; /* compressed_position */
	T0* a7; /* literal */
	T0* a8; /* type */
};

/* Struct for type ET_BINARY_INTEGER_CONSTANT */
struct S2051 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* sign */
	T6 a2; /* index */
	T0* a3; /* cast_type */
	T11 a4; /* value */
	T1 a5; /* has_overflow */
	T6 a6; /* compressed_position */
	T0* a7; /* literal */
	T0* a8; /* type */
};

/* Struct for type ET_REGULAR_REAL_CONSTANT */
struct S2052 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* sign */
	T6 a2; /* index */
	T0* a3; /* cast_type */
	T6 a4; /* compressed_position */
	T0* a5; /* literal */
	T0* a6; /* type */
};

/* Struct for type ET_UNDERSCORED_REAL_CONSTANT */
struct S2053 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* sign */
	T6 a2; /* index */
	T0* a3; /* cast_type */
	T6 a4; /* compressed_position */
	T0* a5; /* literal */
	T0* a6; /* type */
};

/* Struct for type ET_SYMBOL_OPERATOR */
struct S2054 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T2 a1; /* code */
	T6 a2; /* compressed_position */
	T6 a3; /* seed */
};

/* Struct for type ET_SEMICOLON_SYMBOL */
struct S2055 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T2 a1; /* code */
	T6 a2; /* compressed_position */
};

/* Struct for type ET_QUESTION_MARK_SYMBOL */
struct S2056 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T2 a1; /* code */
	T6 a2; /* compressed_position */
	T6 a3; /* index */
};

/* Struct for type ET_KEYWORD_OPERATOR */
struct S2057 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* compressed_position */
	T2 a2; /* code */
	T0* a3; /* text */
	T6 a4; /* seed */
};

/* Struct for type ET_VOID */
struct S2058 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* index */
	T6 a2; /* compressed_position */
	T2 a3; /* code */
	T0* a4; /* text */
};

/* Struct for type ET_AGENT_KEYWORD */
struct S2059 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T2 a1; /* code */
	T6 a2; /* compressed_position */
	T0* a3; /* name */
};

/* Struct for type ET_FALSE_CONSTANT */
struct S2060 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* index */
	T6 a2; /* compressed_position */
	T2 a3; /* code */
	T0* a4; /* text */
};

/* Struct for type ET_RETRY_INSTRUCTION */
struct S2061 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T2 a1; /* code */
	T6 a2; /* compressed_position */
	T0* a3; /* text */
};

/* Struct for type ET_PRECURSOR_KEYWORD */
struct S2062 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T2 a1; /* code */
	T6 a2; /* compressed_position */
	T0* a3; /* name */
	T6 a4; /* seed */
};

/* Struct for type ET_C1_CHARACTER_CONSTANT */
struct S2063 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* index */
	T0* a2; /* cast_type */
	T6 a3; /* compressed_position */
	T3 a4; /* value */
	T0* a5; /* type */
};

/* Struct for type ET_C2_CHARACTER_CONSTANT */
struct S2064 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* index */
	T0* a2; /* cast_type */
	T6 a3; /* compressed_position */
	T3 a4; /* value */
	T0* a5; /* type */
};

/* Struct for type ET_INDEXING_LIST */
struct S2065 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* indexing_keyword */
	T6 a2; /* count */
	T0* a3; /* storage */
};

/* Struct for type ET_TAGGED_INDEXING */
struct S2066 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* tag */
	T0* a2; /* terms */
};

/* Struct for type ET_INDEXING_TERM_LIST */
struct S2068 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
};

/* Struct for type ET_INDEXING */
struct S2069 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* terms */
};

/* Struct for type ET_CUSTOM_ATTRIBUTE */
struct S2070 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* creation_expression */
	T0* a2; /* settings */
	T0* a3; /* end_keyword */
};

/* Struct for type ET_ATTACHMENT_SYMBOL_SEPARATE_KEYWORD */
struct S2071 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* attachment_symbol */
	T0* a2; /* separateness_keyword */
};

/* Struct for type ET_CONSTRAINT_ACTUAL_PARAMETER_LIST */
struct S2072 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
	T0* a3; /* left_bracket */
	T0* a4; /* right_bracket */
};

/* Struct for type ET_CONSTRAINT_LABELED_ACTUAL_PARAMETER */
struct S2073 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* label */
	T0* a2; /* colon */
	T0* a3; /* type */
};

/* Struct for type ET_CONSTRAINT_LABELED_COMMA_ACTUAL_PARAMETER */
struct S2074 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* label */
	T0* a2; /* comma */
	T0* a3; /* type */
};

/* Struct for type ET_CONVERT_FUNCTION */
struct S2075 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* name */
	T0* a2; /* types */
	T0* a3; /* colon */
};

/* Struct for type ET_CONVERT_PROCEDURE */
struct S2076 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* name */
	T0* a2; /* types */
	T0* a3; /* left_parenthesis */
	T0* a4; /* right_parenthesis */
};

/* Struct for type ET_FEATURE_CLAUSE_LIST */
struct S2077 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
};

/* Struct for type ET_DO_FUNCTION */
struct S2079 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* implementation_feature */
	T0* a2; /* implementation_class */
	T0* a3; /* first_precursor */
	T0* a4; /* other_precursors */
	T0* a5; /* preconditions */
	T6 a6; /* hash_code */
	T6 a7; /* first_seed */
	T0* a8; /* arguments */
	T0* a9; /* assigner */
	T0* a10; /* postconditions */
	T0* a11; /* other_seeds */
	T0* a12; /* clients */
	T6 a13; /* version */
	T6 a14; /* id */
	T0* a15; /* extended_name */
	T0* a16; /* declared_type */
	T0* a17; /* obsolete_message */
	T0* a18; /* locals */
	T0* a19; /* compound */
	T0* a20; /* rescue_clause */
	T0* a21; /* is_keyword */
	T0* a22; /* end_keyword */
	T0* a23; /* semicolon */
	T0* a24; /* feature_clause */
	T0* a25; /* first_indexing */
	T0* a26; /* object_tests */
	T0* a27; /* across_components */
	T0* a28; /* frozen_keyword */
	T1 a29; /* validity_checked */
	T1 a30; /* has_validity_error */
	T0* a31; /* synonym */
};

/* Struct for type ET_ONCE_FUNCTION */
struct S2080 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* implementation_feature */
	T0* a2; /* implementation_class */
	T0* a3; /* first_precursor */
	T0* a4; /* other_precursors */
	T0* a5; /* preconditions */
	T6 a6; /* hash_code */
	T6 a7; /* first_seed */
	T0* a8; /* arguments */
	T0* a9; /* assigner */
	T0* a10; /* postconditions */
	T0* a11; /* other_seeds */
	T0* a12; /* clients */
	T6 a13; /* version */
	T6 a14; /* id */
	T0* a15; /* extended_name */
	T1 a16; /* validity_checked */
	T1 a17; /* has_validity_error */
	T0* a18; /* keys */
	T0* a19; /* declared_type */
	T0* a20; /* obsolete_message */
	T0* a21; /* locals */
	T0* a22; /* compound */
	T0* a23; /* rescue_clause */
	T0* a24; /* is_keyword */
	T0* a25; /* end_keyword */
	T0* a26; /* semicolon */
	T0* a27; /* feature_clause */
	T0* a28; /* first_indexing */
	T0* a29; /* object_tests */
	T0* a30; /* across_components */
	T0* a31; /* frozen_keyword */
	T0* a32; /* synonym */
};

/* Struct for type ET_DEFERRED_FUNCTION */
struct S2081 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* implementation_feature */
	T0* a2; /* implementation_class */
	T0* a3; /* first_precursor */
	T0* a4; /* other_precursors */
	T0* a5; /* preconditions */
	T6 a6; /* hash_code */
	T6 a7; /* first_seed */
	T0* a8; /* arguments */
	T0* a9; /* assigner */
	T0* a10; /* postconditions */
	T0* a11; /* other_seeds */
	T0* a12; /* clients */
	T6 a13; /* version */
	T6 a14; /* id */
	T0* a15; /* extended_name */
	T1 a16; /* validity_checked */
	T1 a17; /* has_validity_error */
	T0* a18; /* declared_type */
	T0* a19; /* obsolete_message */
	T0* a20; /* is_keyword */
	T0* a21; /* deferred_keyword */
	T0* a22; /* end_keyword */
	T0* a23; /* semicolon */
	T0* a24; /* feature_clause */
	T0* a25; /* first_indexing */
	T0* a26; /* object_tests */
	T0* a27; /* across_components */
	T0* a28; /* frozen_keyword */
	T0* a29; /* synonym */
};

/* Struct for type ET_ONCE_PROCEDURE */
struct S2082 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* implementation_feature */
	T0* a2; /* implementation_class */
	T0* a3; /* first_precursor */
	T0* a4; /* other_precursors */
	T0* a5; /* preconditions */
	T6 a6; /* hash_code */
	T6 a7; /* first_seed */
	T0* a8; /* arguments */
	T0* a9; /* postconditions */
	T0* a10; /* compound */
	T0* a11; /* other_seeds */
	T0* a12; /* clients */
	T6 a13; /* version */
	T6 a14; /* id */
	T0* a15; /* extended_name */
	T1 a16; /* validity_checked */
	T1 a17; /* has_validity_error */
	T0* a18; /* keys */
	T0* a19; /* first_indexing */
	T0* a20; /* obsolete_message */
	T0* a21; /* is_keyword */
	T0* a22; /* end_keyword */
	T0* a23; /* semicolon */
	T0* a24; /* feature_clause */
	T0* a25; /* object_tests */
	T0* a26; /* across_components */
	T0* a27; /* locals */
	T0* a28; /* rescue_clause */
	T0* a29; /* frozen_keyword */
	T0* a30; /* synonym */
};

/* Struct for type ET_PREFIX_NAME */
struct S2083 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* seed */
	T0* a2; /* prefix_keyword */
	T0* a3; /* operator_string */
	T2 a4; /* code */
};

/* Struct for type ET_PREFIX_FREE_NAME */
struct S2084 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* seed */
	T6 a2; /* hash_code */
	T0* a3; /* prefix_keyword */
	T0* a4; /* operator_string */
	T2 a5; /* code */
};

/* Struct for type ET_INFIX_NAME */
struct S2085 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* seed */
	T0* a2; /* infix_keyword */
	T0* a3; /* operator_string */
	T2 a4; /* code */
};

/* Struct for type ET_ALIAS_FREE_NAME */
struct S2086 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* hash_code */
	T2 a2; /* code */
	T0* a3; /* alias_string */
	T0* a4; /* alias_keyword */
	T6 a5; /* seed */
};

/* Struct for type ET_FORMAL_COMMA_ARGUMENT */
struct S2087 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* name_item */
	T0* a2; /* declared_type */
	T1 a3; /* is_used */
};

/* Struct for type ET_LOCAL_COMMA_VARIABLE */
struct S2089 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* name_item */
	T0* a2; /* declared_type */
	T1 a3; /* is_used */
};

/* Struct for type ET_CLASS_ASSERTION */
struct S2091 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* class_keyword */
};

/* Struct for type ET_VARIANT */
struct S2092 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* variant_keyword */
	T0* a2; /* tag */
	T0* a3; /* expression */
};

/* Struct for type ET_LABELED_ACTUAL_PARAMETER */
struct S2093 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* declared_type */
	T0* a2; /* label_item */
};

/* Struct for type ET_LABELED_COMMA_ACTUAL_PARAMETER */
struct S2095 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* declared_type */
	T0* a2; /* label_item */
};

/* Struct for type ET_QUALIFIED_LIKE_BRACED_TYPE */
struct S2097 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* type_mark */
	T0* a2; /* like_keyword */
	T0* a3; /* target_type */
	T0* a4; /* qualified_name */
	T0* a5; /* left_brace */
	T0* a6; /* right_brace */
};

/* Struct for type ET_QUALIFIED_LIKE_TYPE */
struct S2098 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* type_mark */
	T0* a2; /* target_type */
	T0* a3; /* qualified_name */
};

/* Struct for type ET_ASSIGNER_INSTRUCTION */
struct S2100 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* call */
	T0* a2; /* source */
	T0* a3; /* assign_symbol */
	T0* a4; /* name */
};

/* Struct for type ET_ASSIGNMENT */
struct S2101 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* target */
	T0* a2; /* source */
	T0* a3; /* assign_symbol */
};

/* Struct for type ET_ASSIGNMENT_ATTEMPT */
struct S2102 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* target */
	T0* a2; /* source */
	T0* a3; /* assign_attempt_symbol */
};

/* Struct for type ET_KEYWORD_EXPRESSION */
struct S2103 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* keyword */
	T0* a2; /* expression */
};

/* Struct for type ET_LOOP_INSTRUCTION */
struct S2104 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* from_compound */
	T0* a2; /* until_conditional */
	T0* a3; /* loop_compound */
	T0* a4; /* end_keyword */
	T0* a5; /* invariant_part */
	T0* a6; /* variant_part */
	T1 a7; /* has_old_variant_syntax */
};

/* Struct for type ET_BANG_INSTRUCTION */
struct S2106 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* type */
	T0* a2; /* target */
	T0* a3; /* creation_call */
	T0* a4; /* left_bang */
	T0* a5; /* right_bang */
};

/* Struct for type ET_CREATE_INSTRUCTION */
struct S2107 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* creation_type */
	T0* a2; /* target */
	T0* a3; /* creation_call */
	T0* a4; /* create_keyword */
	T0* a5; /* creation_region */
};

/* Struct for type ET_CREATION_REGION */
struct S2108 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* class_name */
	T0* a2; /* left_symbol */
	T0* a3; /* right_symbol */
};

/* Struct for type ET_IF_INSTRUCTION */
struct S2109 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* conditional */
	T0* a2; /* then_compound */
	T0* a3; /* end_keyword */
	T0* a4; /* elseif_parts */
	T0* a5; /* else_compound */
};

/* Struct for type ET_ELSEIF_PART_LIST */
struct S2110 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
};

/* Struct for type ET_IF_EXPRESSION */
struct S2111 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* index */
	T0* a2; /* conditional */
	T0* a3; /* then_expression */
	T0* a4; /* elseif_parts */
	T0* a5; /* else_expression */
	T0* a6; /* then_keyword */
	T0* a7; /* else_keyword */
	T0* a8; /* end_keyword */
};

/* Struct for type ET_ELSEIF_EXPRESSION_LIST */
struct S2112 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
};

/* Struct for type ET_INSPECT_INSTRUCTION */
struct S2113 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* conditional */
	T0* a2; /* when_parts */
	T0* a3; /* end_keyword */
	T0* a4; /* else_compound */
};

/* Struct for type ET_WHEN_PART_LIST */
struct S2114 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
};

/* Struct for type ET_CHOICE_LIST */
struct S2115 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* when_keyword */
	T6 a2; /* count */
	T0* a3; /* storage */
};

/* Struct for type ET_CHOICE_RANGE */
struct S2116 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* lower */
	T0* a2; /* upper */
	T0* a3; /* dotdot */
};

/* Struct for type ET_DEBUG_INSTRUCTION */
struct S2118 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* keys */
	T0* a2; /* compound */
	T0* a3; /* end_keyword */
};

/* Struct for type ET_MANIFEST_STRING_LIST */
struct S2119 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* left_parenthesis */
	T0* a2; /* right_parenthesis */
	T6 a3; /* count */
	T0* a4; /* storage */
};

/* Struct for type ET_PRECURSOR_INSTRUCTION */
struct S2120 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* parent_name */
	T0* a2; /* arguments */
	T0* a3; /* precursor_keyword */
	T1 a4; /* is_parent_prefixed */
	T0* a5; /* parenthesis_call */
	T0* a6; /* parent_type */
};

/* Struct for type ET_STATIC_CALL_INSTRUCTION */
struct S2121 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* static_type */
	T0* a2; /* feature_keyword */
	T0* a3; /* qualified_name */
	T0* a4; /* arguments */
	T0* a5; /* parenthesis_call */
};

/* Struct for type ET_STATIC_CALL_EXPRESSION */
struct S2122 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* index */
	T0* a2; /* feature_keyword */
	T0* a3; /* static_type */
	T0* a4; /* parenthesis_call */
	T0* a5; /* qualified_name */
	T0* a6; /* arguments */
};

/* Struct for type ET_PRECURSOR_EXPRESSION */
struct S2123 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* index */
	T1 a2; /* is_parent_prefixed */
	T0* a3; /* parent_name */
	T0* a4; /* precursor_keyword */
	T0* a5; /* arguments */
	T0* a6; /* parenthesis_call */
	T0* a7; /* parent_type */
};

/* Struct for type ET_CURRENT_ADDRESS */
struct S2124 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* index */
	T0* a2; /* dollar */
	T0* a3; /* current_keyword */
};

/* Struct for type ET_RESULT_ADDRESS */
struct S2125 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* index */
	T0* a2; /* dollar */
	T0* a3; /* result_keyword */
};

/* Struct for type ET_EXPRESSION_ADDRESS */
struct S2126 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* index */
	T0* a2; /* dollar */
	T0* a3; /* expression */
};

/* Struct for type ET_PARENTHESIZED_EXPRESSION */
struct S2127 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* index */
	T0* a2; /* left_parenthesis */
	T0* a3; /* expression */
	T0* a4; /* right_parenthesis */
};

/* Struct for type ET_INFIX_EXPRESSION */
struct S2128 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* index */
	T0* a2; /* left */
	T0* a3; /* right */
	T0* a4; /* name */
	T1 a5; /* is_boolean_operator */
};

/* Struct for type ET_INFIX_AND_THEN_OPERATOR */
struct S2130 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* and_keyword */
	T0* a2; /* then_keyword */
	T6 a3; /* seed */
};

/* Struct for type ET_INFIX_OR_ELSE_OPERATOR */
struct S2132 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* or_keyword */
	T0* a2; /* else_keyword */
	T6 a3; /* seed */
};

/* Struct for type ET_MANIFEST_TYPE */
struct S2133 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* index */
	T0* a2; /* left_brace */
	T0* a3; /* type */
	T0* a4; /* right_brace */
};

/* Struct for type ET_PREFIX_EXPRESSION */
struct S2134 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* index */
	T0* a2; /* name */
	T0* a3; /* expression */
	T1 a4; /* is_boolean_operator */
};

/* Struct for type ET_OLD_EXPRESSION */
struct S2135 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* index */
	T0* a2; /* old_keyword */
	T0* a3; /* expression */
};

/* Struct for type ET_BRACKET_EXPRESSION */
struct S2136 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* index */
	T0* a2; /* target */
	T0* a3; /* arguments */
	T0* a4; /* name */
};

/* Struct for type ET_MANIFEST_ARRAY */
struct S2137 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* index */
	T0* a2; /* cast_type */
	T0* a3; /* left_symbol */
	T6 a4; /* count */
	T0* a5; /* storage */
	T0* a6; /* right_symbol */
};

/* Struct for type ET_STRIP_EXPRESSION */
struct S2138 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* index */
	T0* a2; /* strip_keyword */
	T0* a3; /* left_parenthesis */
	T6 a4; /* count */
	T0* a5; /* storage */
	T0* a6; /* right_parenthesis */
};

/* Struct for type ET_CALL_AGENT */
struct S2139 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* index */
	T0* a2; /* agent_keyword */
	T0* a3; /* target */
	T0* a4; /* arguments */
	T0* a5; /* qualified_name */
	T0* a6; /* implicit_result */
};

/* Struct for type ET_DO_FUNCTION_INLINE_AGENT */
struct S2140 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* index */
	T0* a2; /* agent_keyword */
	T0* a3; /* formal_arguments */
	T0* a4; /* declared_type */
	T0* a5; /* end_keyword */
	T0* a6; /* implicit_result */
	T0* a7; /* object_tests */
	T0* a8; /* across_components */
	T0* a9; /* actual_arguments */
	T0* a10; /* preconditions */
	T0* a11; /* locals */
	T0* a12; /* compound */
	T0* a13; /* postconditions */
	T0* a14; /* rescue_clause */
	T0* a15; /* target */
};

/* Struct for type ET_ONCE_FUNCTION_INLINE_AGENT */
struct S2141 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* index */
	T0* a2; /* agent_keyword */
	T0* a3; /* formal_arguments */
	T0* a4; /* declared_type */
	T0* a5; /* end_keyword */
	T0* a6; /* implicit_result */
	T0* a7; /* object_tests */
	T0* a8; /* across_components */
	T0* a9; /* actual_arguments */
	T0* a10; /* preconditions */
	T0* a11; /* locals */
	T0* a12; /* keys */
	T0* a13; /* compound */
	T0* a14; /* postconditions */
	T0* a15; /* rescue_clause */
	T0* a16; /* target */
};

/* Struct for type ET_EXTERNAL_FUNCTION_INLINE_AGENT */
struct S2142 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* index */
	T0* a2; /* agent_keyword */
	T0* a3; /* language */
	T0* a4; /* object_tests */
	T0* a5; /* across_components */
	T0* a6; /* actual_arguments */
	T0* a7; /* preconditions */
	T0* a8; /* alias_clause */
	T0* a9; /* postconditions */
	T0* a10; /* target */
	T0* a11; /* formal_arguments */
	T0* a12; /* declared_type */
	T0* a13; /* end_keyword */
	T0* a14; /* implicit_result */
};

/* Struct for type ET_DO_PROCEDURE_INLINE_AGENT */
struct S2144 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* index */
	T0* a2; /* agent_keyword */
	T0* a3; /* formal_arguments */
	T0* a4; /* end_keyword */
	T0* a5; /* object_tests */
	T0* a6; /* across_components */
	T0* a7; /* actual_arguments */
	T0* a8; /* preconditions */
	T0* a9; /* locals */
	T0* a10; /* compound */
	T0* a11; /* postconditions */
	T0* a12; /* rescue_clause */
	T0* a13; /* target */
};

/* Struct for type ET_ONCE_PROCEDURE_INLINE_AGENT */
struct S2145 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* index */
	T0* a2; /* agent_keyword */
	T0* a3; /* formal_arguments */
	T0* a4; /* end_keyword */
	T0* a5; /* object_tests */
	T0* a6; /* across_components */
	T0* a7; /* actual_arguments */
	T0* a8; /* preconditions */
	T0* a9; /* locals */
	T0* a10; /* keys */
	T0* a11; /* compound */
	T0* a12; /* postconditions */
	T0* a13; /* rescue_clause */
	T0* a14; /* target */
};

/* Struct for type ET_EXTERNAL_PROCEDURE_INLINE_AGENT */
struct S2146 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* index */
	T0* a2; /* agent_keyword */
	T0* a3; /* language */
	T0* a4; /* object_tests */
	T0* a5; /* across_components */
	T0* a6; /* actual_arguments */
	T0* a7; /* preconditions */
	T0* a8; /* alias_clause */
	T0* a9; /* postconditions */
	T0* a10; /* target */
	T0* a11; /* formal_arguments */
	T0* a12; /* end_keyword */
};

/* Struct for type ET_AGENT_OPEN_TARGET */
struct S2147 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* left_brace */
	T0* a2; /* type */
	T0* a3; /* right_brace */
	T6 a4; /* index */
};

/* Struct for type ET_AGENT_ARGUMENT_OPERAND_LIST */
struct S2148 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
	T0* a3; /* left_parenthesis */
	T0* a4; /* right_parenthesis */
};

/* Struct for type ET_AGENT_TYPED_OPEN_ARGUMENT */
struct S2149 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* type */
	T0* a2; /* left_brace */
	T0* a3; /* right_brace */
	T0* a4; /* question_mark */
	T6 a5; /* index */
};

/* Struct for type ET_CONSTRAINT_GENERIC_NAMED_TYPE */
struct S2150 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* type_mark */
	T0* a2; /* name */
	T0* a3; /* actual_parameters */
};

/* Struct for type ET_CONSTRAINT_NAMED_TYPE */
struct S2151 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* type_mark */
	T0* a2; /* name */
};

/* Struct for type ET_EXTERNAL_PROCEDURE */
struct S2152 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* implementation_feature */
	T0* a2; /* implementation_class */
	T0* a3; /* first_precursor */
	T0* a4; /* other_precursors */
	T0* a5; /* preconditions */
	T6 a6; /* hash_code */
	T6 a7; /* first_seed */
	T0* a8; /* arguments */
	T0* a9; /* postconditions */
	T8 a10; /* builtin_class_code */
	T8 a11; /* builtin_feature_code */
	T0* a12; /* other_seeds */
	T0* a13; /* clients */
	T6 a14; /* version */
	T6 a15; /* id */
	T0* a16; /* extended_name */
	T1 a17; /* validity_checked */
	T1 a18; /* has_validity_error */
	T0* a19; /* frozen_keyword */
	T0* a20; /* obsolete_message */
	T0* a21; /* is_keyword */
	T0* a22; /* end_keyword */
	T0* a23; /* semicolon */
	T0* a24; /* feature_clause */
	T0* a25; /* first_indexing */
	T0* a26; /* object_tests */
	T0* a27; /* across_components */
	T0* a28; /* language */
	T0* a29; /* alias_clause */
	T0* a30; /* synonym */
};

/* Struct for type ET_LOOP_INVARIANTS */
struct S2153 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* invariant_keyword */
	T6 a2; /* count */
	T0* a3; /* storage */
};

/* Struct for type ET_LIKE_FEATURE */
struct S2154 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* like_keyword */
	T0* a2; /* type_mark */
	T0* a3; /* name */
	T6 a4; /* seed */
	T1 a5; /* is_procedure */
};

/* Struct for type ET_CHECK_INSTRUCTION */
struct S2155 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* check_keyword */
	T0* a2; /* end_keyword */
	T6 a3; /* count */
	T0* a4; /* storage */
	T0* a5; /* then_compound */
};

/* Struct for type ET_ACROSS_INSTRUCTION */
struct S2156 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* cursor_name */
	T0* a2; /* across_keyword */
	T0* a3; /* iterable_expression */
	T0* a4; /* as_keyword */
	T0* a5; /* from_compound */
	T0* a6; /* until_conditional */
	T0* a7; /* loop_compound */
	T0* a8; /* end_keyword */
	T0* a9; /* invariant_part */
	T0* a10; /* variant_part */
	T0* a11; /* new_cursor_expression */
	T0* a12; /* cursor_after_expression */
	T0* a13; /* cursor_forth_instruction */
};

/* Struct for type ET_UNQUALIFIED_CALL_INSTRUCTION */
struct S2157 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* name */
	T0* a2; /* arguments */
	T0* a3; /* parenthesis_call */
};

/* Struct for type ET_FEATURE_ADDRESS */
struct S2158 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* index */
	T0* a2; /* dollar */
	T0* a3; /* name */
};

/* Struct for type ET_OLD_OBJECT_TEST */
struct S2159 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* index */
	T0* a2; /* name */
	T0* a3; /* left_brace */
	T0* a4; /* expression */
	T0* a5; /* attached_keyword */
	T0* a6; /* as_keyword */
	T0* a7; /* colon */
	T0* a8; /* declared_type */
	T0* a9; /* right_brace */
};

/* Struct for type ET_ONCE_MANIFEST_STRING */
struct S2160 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* index */
	T0* a2; /* once_keyword */
	T0* a3; /* manifest_string */
	T6 a4; /* id */
};

/* Struct for type ET_ACROSS_EXPRESSION */
struct S2161 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* index */
	T0* a2; /* cursor_name */
	T0* a3; /* across_keyword */
	T0* a4; /* iterable_expression */
	T0* a5; /* iteration_conditional */
	T0* a6; /* until_conditional */
	T0* a7; /* invariant_part */
	T0* a8; /* variant_part */
	T0* a9; /* as_keyword */
	T0* a10; /* end_keyword */
	T1 a11; /* is_all */
	T0* a12; /* new_cursor_expression */
	T0* a13; /* cursor_after_expression */
	T0* a14; /* cursor_forth_instruction */
};

/* Struct for type [attached] SPECIAL [[attached] ET_INSTRUCTION] */
struct S2163 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_INSTRUCTION] */
struct S2164 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type HEXADECIMAL_STRING_TO_INTEGER_CONVERTER */
struct S2166 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* last_state */
	T1 a2; /* internal_overflowed */
	T11 a3; /* part1 */
	T11 a4; /* part2 */
	T0* a5; /* leading_separators */
	T0* a6; /* trailing_separators */
	T6 a7; /* sign */
	T6 a8; /* conversion_type */
	T2 a9; /* internal_lookahead */
	T1 a10; /* leading_separators_acceptable */
	T1 a11; /* trailing_separators_acceptable */
};

/* Struct for type [attached] TUPLE [ET_MASTER_CLASS, ET_SYSTEM_PROCESSOR] */
struct S2169 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
	T0* z2;
};

/* Struct for type attached DS_SPARSE_TABLE_KEYS [[attached] ET_CLIENT, [attached] ET_CLASS] */
struct S2191 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* table */
	T0* a2; /* equality_tester */
	T0* a3; /* internal_cursor */
};

/* Struct for type [attached] SPECIAL [[attached] ET_FORMAL_PARAMETER_ITEM] */
struct S2196 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_FORMAL_PARAMETER_ITEM] */
struct S2197 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [[attached] ET_FEATURE_NAME_ITEM] */
struct S2199 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_FEATURE_NAME_ITEM] */
struct S2200 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type INTEGER_OVERFLOW_CHECKER */
struct S2202 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* integer_overflow_state1 */
	T0* a2; /* integer_overflow_state2 */
	T0* a3; /* natural_overflow_state1 */
	T0* a4; /* natural_overflow_state2 */
};

/* Struct for type [attached] SPECIAL [[attached] ARRAY [[attached] INTEGER_32]] */
struct S2203 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] SPECIAL [[attached] SPECIAL [[attached] ARRAY [[attached] INTEGER_32]]] */
struct S2204 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] DS_LINKED_LIST_CURSOR [[attached] ET_XACE_FEATURE_OPTIONS] */
struct S2205 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T1 a2; /* before */
};

/* Struct for type attached DS_LINKABLE [[attached] ET_XACE_FEATURE_OPTIONS] */
struct S2207 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* item */
	T0* a2; /* right */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_ECF_NOTE_ELEMENT] */
struct S2208 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [[attached] ET_ECF_NOTE_ELEMENT] */
struct S2209 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_ECF_NOTE_ELEMENT] */
struct S2210 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
};

/* Struct for type [attached] DS_ARRAYED_LIST [[attached] ET_ECF_ADAPTED_DOTNET_ASSEMBLY] */
struct S2211 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
	T0* a3; /* special_routines */
	T6 a4; /* capacity */
	T0* a5; /* internal_cursor */
};

/* Struct for type [attached] DS_ARRAYED_LIST [[attached] ET_ECF_CLUSTER] */
struct S2212 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
	T0* a3; /* special_routines */
	T6 a4; /* capacity */
	T0* a5; /* internal_cursor */
};

/* Struct for type [attached] DS_ARRAYED_LIST [[attached] ET_ECF_EXTERNAL_CFLAG] */
struct S2213 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* special_routines */
	T0* a2; /* storage */
	T6 a3; /* capacity */
	T6 a4; /* count */
	T0* a5; /* internal_cursor */
};

/* Struct for type [attached] DS_ARRAYED_LIST [[attached] ET_ECF_EXTERNAL_INCLUDE] */
struct S2214 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* special_routines */
	T0* a2; /* storage */
	T6 a3; /* capacity */
	T6 a4; /* count */
	T0* a5; /* internal_cursor */
};

/* Struct for type [attached] DS_ARRAYED_LIST [[attached] ET_ECF_EXTERNAL_LIBRARY] */
struct S2215 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* special_routines */
	T0* a2; /* storage */
	T6 a3; /* capacity */
	T6 a4; /* count */
	T0* a5; /* internal_cursor */
};

/* Struct for type [attached] DS_ARRAYED_LIST [[attached] ET_ECF_EXTERNAL_LINKER_FLAG] */
struct S2216 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* special_routines */
	T0* a2; /* storage */
	T6 a3; /* capacity */
	T6 a4; /* count */
	T0* a5; /* internal_cursor */
};

/* Struct for type [attached] DS_ARRAYED_LIST [[attached] ET_ECF_EXTERNAL_MAKE] */
struct S2217 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* special_routines */
	T0* a2; /* storage */
	T6 a3; /* capacity */
	T6 a4; /* count */
	T0* a5; /* internal_cursor */
};

/* Struct for type [attached] DS_ARRAYED_LIST [[attached] ET_ECF_EXTERNAL_OBJECT] */
struct S2218 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* special_routines */
	T0* a2; /* storage */
	T6 a3; /* capacity */
	T6 a4; /* count */
	T0* a5; /* internal_cursor */
};

/* Struct for type [attached] DS_ARRAYED_LIST [[attached] ET_ECF_EXTERNAL_RESOURCE] */
struct S2219 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* special_routines */
	T0* a2; /* storage */
	T6 a3; /* capacity */
	T6 a4; /* count */
	T0* a5; /* internal_cursor */
};

/* Struct for type [attached] DS_ARRAYED_LIST [[attached] ET_ECF_FILE_RULE] */
struct S2221 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* special_routines */
	T0* a2; /* storage */
	T6 a3; /* capacity */
	T6 a4; /* count */
	T0* a5; /* internal_cursor */
};

/* Struct for type [attached] DS_ARRAYED_LIST [[attached] ET_ECF_ADAPTED_LIBRARY] */
struct S2222 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
	T0* a3; /* special_routines */
	T6 a4; /* capacity */
	T0* a5; /* internal_cursor */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_ECF_ACTION] */
struct S2223 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [[attached] ET_ECF_ACTION] */
struct S2224 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_ECF_ACTION] */
struct S2225 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
};

/* Struct for type [attached] DS_ARRAYED_LIST [[attached] ET_ECF_ANDED_CONDITIONS] */
struct S2226 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* special_routines */
	T0* a2; /* storage */
	T6 a3; /* capacity */
	T6 a4; /* count */
	T0* a5; /* internal_cursor */
};

/* Struct for type [attached] DS_ARRAYED_LIST [[attached] ET_ECF_CONDITION_ITEM] */
struct S2227 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* special_routines */
	T0* a2; /* storage */
	T6 a3; /* capacity */
	T6 a4; /* count */
	T0* a5; /* internal_cursor */
};

/* Struct for type [attached] SPECIAL [[attached] ET_ECF_OPTIONS] */
struct S2228 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_ECF_OPTIONS] */
struct S2233 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] DS_HASH_TABLE_CURSOR [[attached] ET_ECF_OPTIONS, [attached] STRING_8] */
struct S2234 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_ECF_VISIBLE_CLASS] */
struct S2235 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [[attached] ET_ECF_VISIBLE_CLASS] */
struct S2236 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_ECF_VISIBLE_CLASS] */
struct S2237 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
};

/* Struct for type [attached] SPECIAL [[attached] ET_PARENT_LIST] */
struct S2238 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_PARENT_LIST] */
struct S2239 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [[attached] ET_CREATOR] */
struct S2240 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_CREATOR] */
struct S2241 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type attached DS_LINKABLE [[attached] XM_DOCUMENT_NODE] */
struct S2243 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* item */
	T0* a2; /* right */
};

/* Struct for type [attached] DS_LINKED_QUEUE [[attached] CHARACTER_8] */
struct S2249 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* first_cell */
	T0* a3; /* last_cell */
};

/* Struct for type UC_UTF16_ROUTINES */
struct S2250 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_ECF_TARGET] */
struct S2251 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
};

/* Struct for type [attached] SPECIAL [[attached] ET_ADAPTED_UNIVERSE] */
struct S2252 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_ADAPTED_UNIVERSE] */
struct S2253 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_ADAPTED_UNIVERSE] */
struct S2254 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* next_cursor */
	T0* a3; /* container */
};

/* Struct for type [attached] SPECIAL [[attached] ET_ECF_TARGET_PARENT] */
struct S2256 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_ECF_TARGET_PARENT] */
struct S2257 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_ECF_TARGET_PARENT] */
struct S2258 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
};

/* Struct for type [attached] SPECIAL [[attached] ET_CONVERT_FEATURE_ITEM] */
struct S2260 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_CONVERT_FEATURE_ITEM] */
struct S2261 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [detachable DS_HASH_SET [[attached] STRING_8]] */
struct S2262 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable DS_HASH_SET [[attached] STRING_8]] */
struct S2266 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] DS_HASH_TABLE_CURSOR [detachable DS_HASH_SET [[attached] STRING_8], [attached] STRING_8] */
struct S2267 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
};

/* Struct for type [attached] SPECIAL [[attached] ET_XACE_TARGET] */
struct S2268 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_XACE_TARGET] */
struct S2269 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_XACE_TARGET] */
struct S2270 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
};

/* Struct for type [attached] SPECIAL [[attached] ET_XACE_CLUSTER] */
struct S2271 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_XACE_CLUSTER] */
struct S2272 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_XACE_CLUSTER] */
struct S2273 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
	T0* a3; /* next_cursor */
};

/* Struct for type [attached] DS_LINKED_LIST_CURSOR [[attached] ET_XACE_CLASS_OPTIONS] */
struct S2274 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T1 a2; /* before */
};

/* Struct for type attached DS_LINKABLE [[attached] ET_XACE_CLASS_OPTIONS] */
struct S2276 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* item */
	T0* a2; /* right */
};

/* Struct for type [attached] SPECIAL [[attached] YY_BUFFER] */
struct S2277 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] YY_BUFFER] */
struct S2278 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [[attached] ET_DYNAMIC_TARGET] */
struct S2279 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_DYNAMIC_TARGET] */
struct S2280 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] DS_BILINKED_LIST [[attached] DS_HASH_TABLE [[attached] STRING_8, [attached] STRING_8]] */
struct S2282 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* last_cell */
	T0* a3; /* first_cell */
	T0* a4; /* internal_cursor */
};

/* Struct for type [attached] DS_BILINKED_LIST_CURSOR [[attached] DS_HASH_TABLE [[attached] STRING_8, [attached] STRING_8]] */
struct S2283 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* before */
	T0* a2; /* current_cell */
	T0* a3; /* next_cursor */
	T0* a4; /* container */
	T1 a5; /* after */
};

/* Struct for type attached DS_LINKABLE [detachable STRING_8] */
struct S2285 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* item */
	T0* a2; /* right */
};

/* Struct for type attached DS_LINKABLE [[attached] DS_PAIR [[attached] XM_POSITION, [attached] XM_NODE]] */
struct S2287 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* item */
	T0* a2; /* right */
};

/* Struct for type [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_DOTNET_ASSEMBLY] */
struct S2288 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
};

/* Struct for type [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_MASTER_CLASS] */
struct S2289 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* next_cursor */
	T6 a2; /* position */
	T0* a3; /* container */
};

/* Struct for type [attached] SPECIAL [[attached] ET_TYPE_ITEM] */
struct S2293 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_TYPE_ITEM] */
struct S2294 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] DS_BILINKED_LIST_CURSOR [[attached] XM_DTD_ELEMENT_CONTENT] */
struct S2295 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T1 a2; /* before */
};

/* Struct for type attached DS_BILINKABLE [[attached] XM_DTD_ELEMENT_CONTENT] */
struct S2296 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* item */
	T0* a2; /* right */
	T0* a3; /* left */
};

/* Struct for type [attached] SPECIAL [[attached] ET_CLUSTER] */
struct S2298 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_CLUSTER] */
struct S2299 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_CLUSTER] */
struct S2300 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
};

/* Struct for type [attached] DS_HASH_TABLE_CURSOR [[attached] INTEGER_32, [attached] ET_CLASS] */
struct S2302 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* next_cursor */
	T0* a2; /* container */
	T6 a3; /* position */
};

/* Struct for type [attached] DS_HASH_TABLE [[attached] INTEGER_32, [attached] ET_FORMAL_PARAMETER] */
struct S2303 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T6 a2; /* last_position */
	T6 a3; /* free_slot */
	T6 a4; /* position */
	T0* a5; /* equality_tester */
	T0* a6; /* key_equality_tester */
	T0* a7; /* internal_cursor */
	T6 a8; /* found_position */
	T0* a9; /* item_storage */
	T0* a10; /* key_storage */
	T0* a11; /* clashes */
	T6 a12; /* modulus */
	T0* a13; /* slots */
	T6 a14; /* capacity */
	T0* a15; /* special_item_routines */
	T0* a16; /* special_key_routines */
	T0* a17; /* hash_function */
	T6 a18; /* slots_position */
	T6 a19; /* clashes_previous_position */
};

/* Struct for type [attached] SPECIAL [[attached] ET_FORMAL_PARAMETER] */
struct S2304 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_FORMAL_PARAMETER] */
struct S2305 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_FORMAL_PARAMETER] */
struct S2306 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* next_cursor */
	T0* a2; /* container */
	T6 a3; /* position */
};

/* Struct for type [attached] DS_HASH_TABLE_CURSOR [[attached] ET_RENAME, [attached] ET_CALL_NAME] */
struct S2308 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* next_cursor */
	T0* a3; /* container */
};

/* Struct for type [attached] SPECIAL [[attached] ET_RENAME] */
struct S2309 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] SPECIAL [[attached] ET_CALL_NAME] */
struct S2317 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_RENAME] */
struct S2318 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_CALL_NAME] */
struct S2319 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] DS_HASH_SET_CURSOR [[attached] ET_CALL_NAME] */
struct S2320 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* next_cursor */
	T0* a3; /* container */
};

/* Struct for type [attached] DS_HASH_TABLE_CURSOR [[attached] BOOLEAN, [attached] ET_CALL_NAME] */
struct S2322 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* next_cursor */
	T0* a3; /* container */
};

/* Struct for type [attached] DS_HASH_TABLE_CURSOR [[attached] ET_FEATURE_NAME, [attached] ET_ALIAS_NAME] */
struct S2329 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* next_cursor */
	T0* a2; /* container */
	T6 a3; /* position */
};

/* Struct for type [attached] DS_HASH_TABLE_CURSOR [[attached] ET_REPLICABLE_FEATURE, [attached] INTEGER_32] */
struct S2330 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* next_cursor */
	T0* a3; /* container */
};

/* Struct for type [attached] SPECIAL [[attached] ET_REPLICABLE_FEATURE] */
struct S2331 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_REPLICABLE_FEATURE] */
struct S2334 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [[attached] ET_RENAME_ITEM] */
struct S2336 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_RENAME_ITEM] */
struct S2337 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [[attached] ET_EXPORT] */
struct S2339 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_EXPORT] */
struct S2340 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] DS_LINKED_LIST_CURSOR [[attached] ET_ADAPTED_FEATURE] */
struct S2341 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T1 a2; /* before */
	T1 a3; /* after */
	T0* a4; /* current_cell */
	T0* a5; /* next_cursor */
};

/* Struct for type attached DS_LINKABLE [[attached] ET_ADAPTED_FEATURE] */
struct S2343 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* item */
	T0* a2; /* right */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_PARENT_FEATURE] */
struct S2344 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [[attached] ET_PARENT_FEATURE] */
struct S2345 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_PARENT_FEATURE] */
struct S2346 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
	T0* a3; /* next_cursor */
};

/* Struct for type [attached] SPECIAL [[attached] ET_DOTNET_FEATURE] */
struct S2347 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] DS_HASH_SET_CURSOR [[attached] ET_DOTNET_FEATURE] */
struct S2348 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* next_cursor */
	T0* a2; /* container */
	T6 a3; /* position */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_DOTNET_FEATURE] */
struct S2349 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [[attached] DS_LINKED_LIST [[attached] ET_DOTNET_FEATURE]] */
struct S2352 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] DS_HASH_TABLE_CURSOR [[attached] DS_LINKED_LIST [[attached] ET_DOTNET_FEATURE], [attached] ET_DOTNET_FEATURE] */
struct S2357 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* next_cursor */
	T0* a2; /* container */
	T6 a3; /* position */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] DS_LINKED_LIST [[attached] ET_DOTNET_FEATURE]] */
struct S2358 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] DS_LINKED_LIST_CURSOR [[attached] ET_DOTNET_FEATURE] */
struct S2359 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T1 a2; /* before */
	T1 a3; /* after */
	T0* a4; /* current_cell */
	T0* a5; /* next_cursor */
};

/* Struct for type attached DS_LINKABLE [[attached] ET_DOTNET_FEATURE] */
struct S2361 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* item */
	T0* a2; /* right */
};

/* Struct for type [attached] DS_HASH_TABLE [[attached] INTEGER_32, [attached] ET_LIKE_FEATURE] */
struct S2362 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T6 a2; /* last_position */
	T6 a3; /* free_slot */
	T6 a4; /* position */
	T0* a5; /* equality_tester */
	T0* a6; /* key_equality_tester */
	T0* a7; /* internal_cursor */
	T6 a8; /* found_position */
	T0* a9; /* item_storage */
	T0* a10; /* key_storage */
	T0* a11; /* clashes */
	T6 a12; /* modulus */
	T0* a13; /* slots */
	T6 a14; /* capacity */
	T0* a15; /* special_item_routines */
	T0* a16; /* special_key_routines */
	T0* a17; /* hash_function */
	T6 a18; /* slots_position */
	T6 a19; /* clashes_previous_position */
};

/* Struct for type [attached] SPECIAL [[attached] ET_LIKE_FEATURE] */
struct S2363 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_LIKE_FEATURE] */
struct S2364 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_LIKE_FEATURE] */
struct S2365 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* next_cursor */
	T6 a2; /* position */
	T0* a3; /* container */
};

/* Struct for type [attached] DS_HASH_TABLE_CURSOR [[attached] DS_HASH_TABLE [[attached] TUPLE [detachable ARRAY [[attached] ET_TYPE], detachable ET_TYPE, [attached] NATURAL_8], [attached] ET_FEATURE_NAME], [attached] NATURAL_8] */
struct S2368 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] DS_HASH_TABLE [[attached] TUPLE [detachable ARRAY [[attached] ET_TYPE], detachable ET_TYPE, [attached] NATURAL_8], [attached] ET_FEATURE_NAME]] */
struct S2369 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] DS_HASH_TABLE [[attached] TUPLE [detachable ARRAY [[attached] ET_TYPE], detachable ET_TYPE, [attached] NATURAL_8], [attached] ET_FEATURE_NAME] */
struct S2370 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* found_position */
	T0* a2; /* item_storage */
	T0* a3; /* key_equality_tester */
	T0* a4; /* internal_keys */
	T6 a5; /* position */
	T6 a6; /* last_position */
	T6 a7; /* capacity */
	T6 a8; /* slots_position */
	T6 a9; /* count */
	T0* a10; /* equality_tester */
	T6 a11; /* modulus */
	T6 a12; /* clashes_previous_position */
	T0* a13; /* special_item_routines */
	T0* a14; /* clashes */
	T0* a15; /* slots */
	T0* a16; /* special_key_routines */
	T0* a17; /* key_storage */
	T6 a18; /* free_slot */
	T0* a19; /* internal_cursor */
	T0* a20; /* hash_function */
};

/* Struct for type [attached] SPECIAL [[attached] DS_HASH_TABLE [[attached] TUPLE [detachable ARRAY [[attached] ET_TYPE], detachable ET_TYPE, [attached] NATURAL_8], [attached] ET_FEATURE_NAME]] */
struct S2371 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type ET_CLUSTER_ERROR */
struct S2372 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* default_template */
	T0* a2; /* parameters */
	T0* a3; /* code */
	T0* a4; /* etl_code */
	T0* a5; /* cluster */
};

/* Struct for type ET_SYNTAX_ERROR */
struct S2373 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* default_template */
	T0* a2; /* parameters */
	T0* a3; /* code */
	T0* a4; /* etl_code */
	T0* a5; /* filename */
	T0* a6; /* position */
	T1 a7; /* ise_reported */
	T1 a8; /* ise_fatal */
	T1 a9; /* ge_reported */
	T1 a10; /* ge_fatal */
};

/* Struct for type ET_DOTNET_ASSEMBLY_ERROR */
struct S2374 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* default_template */
	T0* a2; /* parameters */
	T0* a3; /* code */
	T0* a4; /* etl_code */
	T0* a5; /* assembly */
};

/* Struct for type [attached] TUPLE [ET_SYSTEM, [attached] DS_HASH_TABLE [[attached] INTEGER_32, [attached] ET_UNIVERSE], INTEGER_32] */
struct S2377 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
	T0* z2;
	T6 z3;
};

/* Struct for type [attached] TUPLE [[attached] ET_ECF_CLUSTER] */
struct S2378 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type [attached] PROCEDURE [[attached] TUPLE [[attached] ET_ECF_CLUSTER]] */
struct S2379 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T14 a1; /* rout_disp */
	T0* a2; /* closed_operands */
	T14 a3; /* encaps_rout_disp */
	T14 a4; /* calc_rout_addr */
	T1 a5; /* is_target_closed */
	T6 a6; /* open_count */
};

/* Struct for type [attached] TUPLE [ET_ECF_TARGET, STRING_8] */
struct S2380 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
	T0* z2;
};

/* Struct for type [attached] TUPLE [[attached] ET_ECF_ADAPTED_LIBRARY] */
struct S2382 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type [attached] PROCEDURE [[attached] TUPLE [[attached] ET_ECF_ADAPTED_LIBRARY]] */
struct S2383 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T14 a1; /* rout_disp */
	T0* a2; /* closed_operands */
	T14 a3; /* encaps_rout_disp */
	T14 a4; /* calc_rout_addr */
	T1 a5; /* is_target_closed */
	T6 a6; /* open_count */
};

/* Struct for type [attached] TUPLE [[attached] ET_ECF_ADAPTED_DOTNET_ASSEMBLY] */
struct S2384 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type [attached] PROCEDURE [[attached] TUPLE [[attached] ET_ECF_ADAPTED_DOTNET_ASSEMBLY]] */
struct S2385 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T14 a1; /* rout_disp */
	T0* a2; /* closed_operands */
	T14 a3; /* encaps_rout_disp */
	T14 a4; /* calc_rout_addr */
	T1 a5; /* is_target_closed */
	T6 a6; /* open_count */
};

/* Struct for type [attached] PROCEDURE [[attached] TUPLE [[attached] STRING_8]] */
struct S2388 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T14 a1; /* rout_disp */
	T0* a2; /* closed_operands */
	T14 a3; /* encaps_rout_disp */
	T14 a4; /* calc_rout_addr */
	T1 a5; /* is_target_closed */
	T6 a6; /* open_count */
};

/* Struct for type [attached] TUPLE [ET_ECF_OPTIONS, STRING_8] */
struct S2389 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
	T0* z2;
};

/* Struct for type [attached] DS_HASH_TABLE [[attached] DS_HASH_TABLE [[attached] INTEGER_32, [attached] STRING_8], [attached] STRING_8] */
struct S2395 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* found_position */
	T0* a2; /* item_storage */
	T6 a3; /* position */
	T0* a4; /* key_equality_tester */
	T0* a5; /* internal_keys */
	T0* a6; /* hash_function */
	T6 a7; /* last_position */
	T6 a8; /* capacity */
	T6 a9; /* slots_position */
	T6 a10; /* count */
	T0* a11; /* equality_tester */
	T6 a12; /* modulus */
	T6 a13; /* clashes_previous_position */
	T0* a14; /* special_item_routines */
	T0* a15; /* clashes */
	T0* a16; /* slots */
	T0* a17; /* special_key_routines */
	T0* a18; /* key_storage */
	T6 a19; /* free_slot */
	T0* a20; /* internal_cursor */
};

/* Struct for type [attached] TUPLE [ET_XACE_SYSTEM, [attached] DS_HASH_TABLE [[attached] INTEGER_32, [attached] ET_UNIVERSE], INTEGER_32] */
struct S2397 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
	T0* z2;
	T6 z3;
};

/* Struct for type [attached] TUPLE [ET_LACE_SYSTEM, [attached] DS_HASH_TABLE [[attached] INTEGER_32, [attached] ET_UNIVERSE], INTEGER_32] */
struct S2398 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
	T0* z2;
	T6 z3;
};

/* Struct for type [attached] SPECIAL [[attached] ET_FORMAL_ARGUMENT_ITEM] */
struct S2400 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_FORMAL_ARGUMENT_ITEM] */
struct S2402 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type attached DS_LINKABLE [[attached] XM_ELEMENT_NODE] */
struct S2407 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* right */
	T0* a2; /* item */
};

/* Struct for type [attached] TUPLE [detachable ARRAY [[attached] ET_TYPE], detachable ET_TYPE, [attached] NATURAL_8] */
struct S2410 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
	T0* z2;
	T8 z3;
};

/* Struct for type [attached] TUPLE [[attached] ET_ECF_EXTERNAL_CFLAG] */
struct S2418 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type [attached] PROCEDURE [[attached] TUPLE [[attached] ET_ECF_EXTERNAL_CFLAG]] */
struct S2419 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T14 a1; /* rout_disp */
	T0* a2; /* closed_operands */
	T14 a3; /* encaps_rout_disp */
	T14 a4; /* calc_rout_addr */
	T1 a5; /* is_target_closed */
	T6 a6; /* open_count */
};

/* Struct for type [attached] TUPLE [ET_ECF_INTERNAL_UNIVERSE, ET_ECF_STATE] */
struct S2420 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
	T0* z2;
};

/* Struct for type [attached] TUPLE [[attached] ET_ECF_EXTERNAL_INCLUDE] */
struct S2421 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type [attached] PROCEDURE [[attached] TUPLE [[attached] ET_ECF_EXTERNAL_INCLUDE]] */
struct S2422 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T14 a1; /* rout_disp */
	T0* a2; /* closed_operands */
	T14 a3; /* encaps_rout_disp */
	T14 a4; /* calc_rout_addr */
	T1 a5; /* is_target_closed */
	T6 a6; /* open_count */
};

/* Struct for type [attached] TUPLE [[attached] ET_ECF_EXTERNAL_LIBRARY] */
struct S2423 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type [attached] PROCEDURE [[attached] TUPLE [[attached] ET_ECF_EXTERNAL_LIBRARY]] */
struct S2424 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T14 a1; /* rout_disp */
	T0* a2; /* closed_operands */
	T14 a3; /* encaps_rout_disp */
	T14 a4; /* calc_rout_addr */
	T1 a5; /* is_target_closed */
	T6 a6; /* open_count */
};

/* Struct for type [attached] TUPLE [[attached] ET_ECF_EXTERNAL_LINKER_FLAG] */
struct S2425 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type [attached] PROCEDURE [[attached] TUPLE [[attached] ET_ECF_EXTERNAL_LINKER_FLAG]] */
struct S2426 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T14 a1; /* rout_disp */
	T0* a2; /* closed_operands */
	T14 a3; /* encaps_rout_disp */
	T14 a4; /* calc_rout_addr */
	T1 a5; /* is_target_closed */
	T6 a6; /* open_count */
};

/* Struct for type [attached] TUPLE [[attached] ET_ECF_EXTERNAL_MAKE] */
struct S2427 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type [attached] PROCEDURE [[attached] TUPLE [[attached] ET_ECF_EXTERNAL_MAKE]] */
struct S2428 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T14 a1; /* rout_disp */
	T0* a2; /* closed_operands */
	T14 a3; /* encaps_rout_disp */
	T14 a4; /* calc_rout_addr */
	T1 a5; /* is_target_closed */
	T6 a6; /* open_count */
};

/* Struct for type [attached] TUPLE [[attached] ET_ECF_EXTERNAL_OBJECT] */
struct S2429 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type [attached] PROCEDURE [[attached] TUPLE [[attached] ET_ECF_EXTERNAL_OBJECT]] */
struct S2430 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T14 a1; /* rout_disp */
	T0* a2; /* closed_operands */
	T14 a3; /* encaps_rout_disp */
	T14 a4; /* calc_rout_addr */
	T1 a5; /* is_target_closed */
	T6 a6; /* open_count */
};

/* Struct for type [attached] TUPLE [[attached] ET_ECF_EXTERNAL_RESOURCE] */
struct S2431 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type [attached] PROCEDURE [[attached] TUPLE [[attached] ET_ECF_EXTERNAL_RESOURCE]] */
struct S2432 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T14 a1; /* rout_disp */
	T0* a2; /* closed_operands */
	T14 a3; /* encaps_rout_disp */
	T14 a4; /* calc_rout_addr */
	T1 a5; /* is_target_closed */
	T6 a6; /* open_count */
};

/* Struct for type [attached] DS_ARRAYED_LIST [[attached] RX_PCRE_REGULAR_EXPRESSION] */
struct S2433 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* special_routines */
	T0* a2; /* storage */
	T6 a3; /* capacity */
	T6 a4; /* count */
	T0* a5; /* internal_cursor */
};

/* Struct for type [attached] DS_HASH_TABLE [[attached] LX_WILDCARD, [attached] STRING_8] */
struct S2444 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T6 a2; /* last_position */
	T0* a3; /* clashes */
	T0* a4; /* item_storage */
	T0* a5; /* key_equality_tester */
	T0* a6; /* internal_keys */
	T6 a7; /* position */
	T6 a8; /* capacity */
	T6 a9; /* slots_position */
	T0* a10; /* equality_tester */
	T6 a11; /* found_position */
	T6 a12; /* modulus */
	T6 a13; /* clashes_previous_position */
	T0* a14; /* special_item_routines */
	T0* a15; /* slots */
	T0* a16; /* special_key_routines */
	T0* a17; /* key_storage */
	T6 a18; /* free_slot */
	T0* a19; /* internal_cursor */
	T0* a20; /* hash_function */
};

/* Struct for type [attached] TUPLE [[attached] LX_WILDCARD] */
struct S2446 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type [attached] PREDICATE [[attached] TUPLE [[attached] LX_WILDCARD]] */
struct S2447 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T14 a1; /* rout_disp */
	T0* a2; /* closed_operands */
	T14 a3; /* encaps_rout_disp */
	T14 a4; /* calc_rout_addr */
	T1 a5; /* is_target_closed */
	T6 a6; /* open_count */
};

/* Struct for type [attached] TUPLE [ET_CLUSTER_DEPENDENCE_CONSTRAINT, ET_GROUP, STRING_8] */
struct S2448 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
	T0* z2;
	T0* z3;
};

/* Struct for type [attached] PREDICATE [[attached] TUPLE [[attached] STRING_8]] */
struct S2449 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T14 a1; /* rout_disp */
	T0* a2; /* closed_operands */
	T14 a3; /* encaps_rout_disp */
	T14 a4; /* calc_rout_addr */
	T1 a5; /* is_target_closed */
	T6 a6; /* open_count */
};

/* Struct for type [attached] TUPLE [KL_STRING_ROUTINES, STRING_8] */
struct S2450 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
	T0* z2;
};

/* Struct for type LX_DFA_WILDCARD */
struct S2452 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* pattern */
	T1 a2; /* is_case_insensitive */
	T0* a3; /* yy_nxt */
	T0* a4; /* yy_accept */
	T6 a5; /* yyNb_rows */
	T6 a6; /* match_count */
	T6 a7; /* subject_start */
	T6 a8; /* subject_end */
	T0* a9; /* subject */
	T6 a10; /* matched_start */
	T6 a11; /* matched_end */
};

/* Struct for type [attached] DS_HASH_TABLE_CURSOR [[attached] INTEGER_32, [attached] STRING_8] */
struct S2457 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* next_cursor */
	T0* a3; /* container */
};

/* Struct for type [attached] SPECIAL [[attached] ET_LOCAL_VARIABLE_ITEM] */
struct S2459 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_LOCAL_VARIABLE_ITEM] */
struct S2461 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [[attached] ET_INDEXING_ITEM] */
struct S2463 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_INDEXING_ITEM] */
struct S2464 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [[attached] ET_INDEXING_TERM_ITEM] */
struct S2465 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_INDEXING_TERM_ITEM] */
struct S2466 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [[attached] ET_CONSTRAINT_ACTUAL_PARAMETER_ITEM] */
struct S2467 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_CONSTRAINT_ACTUAL_PARAMETER_ITEM] */
struct S2469 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [[attached] ET_FEATURE_CLAUSE] */
struct S2470 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_FEATURE_CLAUSE] */
struct S2471 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type ET_STANDARD_ONCE_KEYS */
struct S2472 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [[attached] ET_ELSEIF_PART] */
struct S2475 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_ELSEIF_PART] */
struct S2476 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [[attached] ET_ELSEIF_EXPRESSION] */
struct S2477 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_ELSEIF_EXPRESSION] */
struct S2478 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [[attached] ET_WHEN_PART] */
struct S2479 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_WHEN_PART] */
struct S2480 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [[attached] ET_CHOICE_ITEM] */
struct S2482 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_CHOICE_ITEM] */
struct S2483 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [[attached] ET_MANIFEST_STRING_ITEM] */
struct S2485 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_MANIFEST_STRING_ITEM] */
struct S2486 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type ET_UNFOLDED_TUPLE_ACTUAL_ARGUMENT_LIST */
struct S2487 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* actual_arguments */
	T0* a3; /* left_symbol */
	T0* a4; /* storage */
	T0* a5; /* right_symbol */
};

/* Struct for type ET_INFIX_CAST_EXPRESSION */
struct S2489 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* index */
	T0* a2; /* expression */
	T0* a3; /* type */
};

/* Struct for type ET_AGENT_IMPLICIT_CURRENT_TARGET */
struct S2493 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* index */
	T0* a2; /* agent_expression */
};

/* Struct for type [attached] SPECIAL [[attached] ET_AGENT_ARGUMENT_OPERAND_ITEM] */
struct S2495 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_AGENT_ARGUMENT_OPERAND_ITEM] */
struct S2497 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] DS_SPARSE_TABLE_KEYS_CURSOR [[attached] ET_CLIENT, [attached] ET_CLASS] */
struct S2499 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* table_cursor */
	T0* a2; /* container */
};

/* Struct for type [attached] SPECIAL [[attached] NATURAL_64] */
struct S2501 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T11 z2[1]; /* item */
};

/* Struct for type [attached] SPECIAL [[attached] ET_ECF_ADAPTED_DOTNET_ASSEMBLY] */
struct S2502 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_ECF_ADAPTED_DOTNET_ASSEMBLY] */
struct S2503 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_ECF_ADAPTED_DOTNET_ASSEMBLY] */
struct S2504 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
};

/* Struct for type [attached] SPECIAL [[attached] ET_ECF_CLUSTER] */
struct S2505 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_ECF_CLUSTER] */
struct S2506 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_ECF_CLUSTER] */
struct S2507 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
	T0* a3; /* next_cursor */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_ECF_EXTERNAL_CFLAG] */
struct S2508 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [[attached] ET_ECF_EXTERNAL_CFLAG] */
struct S2509 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_ECF_EXTERNAL_CFLAG] */
struct S2510 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_ECF_EXTERNAL_INCLUDE] */
struct S2512 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [[attached] ET_ECF_EXTERNAL_INCLUDE] */
struct S2513 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_ECF_EXTERNAL_INCLUDE] */
struct S2514 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_ECF_EXTERNAL_LIBRARY] */
struct S2516 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [[attached] ET_ECF_EXTERNAL_LIBRARY] */
struct S2517 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_ECF_EXTERNAL_LIBRARY] */
struct S2518 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_ECF_EXTERNAL_LINKER_FLAG] */
struct S2520 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [[attached] ET_ECF_EXTERNAL_LINKER_FLAG] */
struct S2521 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_ECF_EXTERNAL_LINKER_FLAG] */
struct S2522 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_ECF_EXTERNAL_MAKE] */
struct S2524 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [[attached] ET_ECF_EXTERNAL_MAKE] */
struct S2525 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_ECF_EXTERNAL_MAKE] */
struct S2526 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_ECF_EXTERNAL_OBJECT] */
struct S2528 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [[attached] ET_ECF_EXTERNAL_OBJECT] */
struct S2529 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_ECF_EXTERNAL_OBJECT] */
struct S2530 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_ECF_EXTERNAL_RESOURCE] */
struct S2532 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [[attached] ET_ECF_EXTERNAL_RESOURCE] */
struct S2533 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_ECF_EXTERNAL_RESOURCE] */
struct S2534 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_ECF_FILE_RULE] */
struct S2536 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [[attached] ET_ECF_FILE_RULE] */
struct S2537 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_ECF_FILE_RULE] */
struct S2538 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
};

/* Struct for type [attached] SPECIAL [[attached] ET_ECF_ADAPTED_LIBRARY] */
struct S2539 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_ECF_ADAPTED_LIBRARY] */
struct S2540 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_ECF_ADAPTED_LIBRARY] */
struct S2541 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_ECF_ANDED_CONDITIONS] */
struct S2542 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [[attached] ET_ECF_ANDED_CONDITIONS] */
struct S2543 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_ECF_ANDED_CONDITIONS] */
struct S2544 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_ECF_CONDITION_ITEM] */
struct S2545 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [[attached] ET_ECF_CONDITION_ITEM] */
struct S2546 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_ECF_CONDITION_ITEM] */
struct S2547 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
};

/* Struct for type attached DS_LINKABLE [[attached] CHARACTER_8] */
struct S2549 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T2 a1; /* item */
	T0* a2; /* right */
};

/* Struct for type attached DS_BILINKABLE [[attached] DS_HASH_TABLE [[attached] STRING_8, [attached] STRING_8]] */
struct S2551 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* item */
	T0* a2; /* left */
	T0* a3; /* right */
};

/* Struct for type [attached] DS_HASH_TABLE_CURSOR [[attached] INTEGER_32, [attached] ET_FORMAL_PARAMETER] */
struct S2555 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* next_cursor */
	T0* a2; /* container */
	T6 a3; /* position */
};

/* Struct for type [attached] DS_HASH_TABLE_CURSOR [[attached] INTEGER_32, [attached] ET_LIKE_FEATURE] */
struct S2558 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* next_cursor */
	T0* a2; /* container */
	T6 a3; /* position */
};

/* Struct for type [attached] SPECIAL [[attached] TUPLE [detachable ARRAY [[attached] ET_TYPE], detachable ET_TYPE, [attached] NATURAL_8]] */
struct S2559 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] TUPLE [detachable ARRAY [[attached] ET_TYPE], detachable ET_TYPE, [attached] NATURAL_8]] */
struct S2563 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] DS_HASH_TABLE_CURSOR [[attached] TUPLE [detachable ARRAY [[attached] ET_TYPE], detachable ET_TYPE, [attached] NATURAL_8], [attached] ET_FEATURE_NAME] */
struct S2564 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
};

/* Struct for type [attached] TUPLE [ET_GROUP] */
struct S2565 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type [attached] TUPLE [[attached] PROCEDURE [[attached] TUPLE [[attached] ET_CLASS]], [attached] PREDICATE [[attached] TUPLE [[attached] ET_CLASS]]] */
struct S2566 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
	T0* z2;
};

/* Struct for type AP_OPTION_COMPARATOR */
struct S2567 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] DS_QUICK_SORTER [[attached] AP_OPTION] */
struct S2568 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* comparator */
};

/* Struct for type KL_CHARACTER_ROUTINES */
struct S2577 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] TUPLE [ET_C_GENERATOR, INTEGER_32, ET_CURRENT, ET_DYNAMIC_TUPLE_TYPE, BOOLEAN] */
struct S2578 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
	T6 z2;
	T0* z3;
	T0* z4;
	T1 z5;
};

/* Struct for type ET_UNFOLDED_TUPLE_ACTUAL_PARAMETERS */
struct S2580 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* actual_parameters */
	T6 a3; /* tuple_position */
	T0* a4; /* tuple_type */
};

/* Struct for type ET_ACTUAL_PARAMETER_SUBLIST */
struct S2581 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* actual_parameters */
	T6 a2; /* upper */
	T6 a3; /* lower */
};

/* Struct for type [attached] DS_ARRAYED_LIST [detachable ET_FORMAL_PARAMETER_TYPE] */
struct S2583 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* a3; /* storage */
	T0* a4; /* special_routines */
	T0* a5; /* internal_cursor */
};

/* Struct for type [attached] SPECIAL [[attached] ET_DYNAMIC_PRECURSOR] */
struct S2594 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_DYNAMIC_PRECURSOR] */
struct S2595 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type ET_NULL_INSTRUCTION */
struct S2596 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type ET_PARENTHESIS_INSTRUCTION */
struct S2597 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* target */
	T0* a2; /* name */
	T0* a3; /* arguments */
};

/* Struct for type ET_PARENTHESIS_SYMBOL */
struct S2615 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* seed */
	T6 a2; /* compressed_position */
	T2 a3; /* code */
};

/* Struct for type ET_AGENT_IMPLICIT_OPEN_ARGUMENT_LIST */
struct S2621 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
};

/* Struct for type ET_AGENT_IMPLICIT_OPEN_ARGUMENT */
struct S2622 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* agent_expression */
	T6 a2; /* argument_index */
	T6 a3; /* index */
};

/* Struct for type [attached] SPECIAL [[attached] DS_HASH_TABLE [[attached] INTEGER_32, [attached] STRING_8]] */
struct S2646 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] DS_HASH_TABLE [[attached] INTEGER_32, [attached] STRING_8]] */
struct S2650 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] DS_HASH_TABLE_CURSOR [[attached] DS_HASH_TABLE [[attached] INTEGER_32, [attached] STRING_8], [attached] STRING_8] */
struct S2651 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* next_cursor */
	T0* a3; /* container */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] RX_PCRE_REGULAR_EXPRESSION] */
struct S2653 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [[attached] RX_PCRE_REGULAR_EXPRESSION] */
struct S2654 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] DS_ARRAYED_LIST_CURSOR [[attached] RX_PCRE_REGULAR_EXPRESSION] */
struct S2655 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
};

/* Struct for type [attached] SPECIAL [[attached] LX_WILDCARD] */
struct S2657 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] LX_WILDCARD] */
struct S2661 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] DS_HASH_TABLE_CURSOR [[attached] LX_WILDCARD, [attached] STRING_8] */
struct S2662 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* next_cursor */
	T0* a3; /* container */
};

/* Struct for type LX_WILDCARD_PARSER */
struct S2664 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* successful */
	T0* a2; /* pending_rules */
	T0* a3; /* start_condition_stack */
	T0* a4; /* action_factory */
	T0* a5; /* old_singleton_lines */
	T0* a6; /* old_singleton_columns */
	T0* a7; /* old_singleton_counts */
	T0* a8; /* old_regexp_lines */
	T0* a9; /* old_regexp_columns */
	T0* a10; /* old_regexp_counts */
	T0* a11; /* description */
	T0* a12; /* last_string_value */
	T0* a13; /* last_lx_symbol_class_value */
	T0* a14; /* error_handler */
	T0* a15; /* name_definitions */
	T0* a16; /* character_classes */
	T6 a17; /* line_nb */
	T0* a18; /* yyss */
	T0* a19; /* input_buffer */
	T6 a20; /* yy_end */
	T6 a21; /* yy_position */
	T6 a22; /* yy_line */
	T6 a23; /* yy_column */
	T6 a24; /* yy_parsing_status */
	T6 a25; /* yy_suspended_yystacksize */
	T6 a26; /* yy_suspended_yystate */
	T6 a27; /* yy_suspended_yyn */
	T6 a28; /* yy_suspended_yychar1 */
	T6 a29; /* yy_suspended_index */
	T6 a30; /* yy_suspended_yyss_top */
	T6 a31; /* yy_suspended_yy_goto */
	T0* a32; /* yyr1 */
	T6 a33; /* yyssp */
	T0* a34; /* yypgoto */
	T0* a35; /* yycheck */
	T0* a36; /* yytable */
	T0* a37; /* yydefgoto */
	T6 a38; /* error_count */
	T1 a39; /* yy_lookahead_needed */
	T6 a40; /* yyerrstatus */
	T0* a41; /* yypact */
	T6 a42; /* last_token */
	T0* a43; /* yytranslate */
	T0* a44; /* yydefact */
	T0* a45; /* yy_pushed_start_conditions */
	T0* a46; /* yyspecial_routines1 */
	T6 a47; /* yyvsc1 */
	T0* a48; /* yyvs1 */
	T0* a49; /* yyspecial_routines2 */
	T6 a50; /* yyvsc2 */
	T0* a51; /* yyvs2 */
	T0* a52; /* yyspecial_routines3 */
	T6 a53; /* yyvsc3 */
	T0* a54; /* yyvs3 */
	T0* a55; /* yyspecial_routines4 */
	T6 a56; /* yyvsc4 */
	T0* a57; /* yyvs4 */
	T0* a58; /* yyspecial_routines5 */
	T6 a59; /* yyvsc5 */
	T0* a60; /* yyvs5 */
	T0* a61; /* yytypes1 */
	T0* a62; /* yytypes2 */
	T6 a63; /* yy_start */
	T6 a64; /* yyvsp1 */
	T6 a65; /* yyvsp2 */
	T6 a66; /* yyvsp3 */
	T6 a67; /* yyvsp4 */
	T6 a68; /* yyvsp5 */
	T1 a69; /* yy_more_flag */
	T6 a70; /* yy_more_len */
	T6 a71; /* line */
	T6 a72; /* column */
	T6 a73; /* position */
	T6 a74; /* yy_start_state */
	T0* a75; /* yy_state_stack */
	T6 a76; /* yy_state_count */
	T0* a77; /* yy_content_area */
	T0* a78; /* yy_ec */
	T0* a79; /* yy_content */
	T0* a80; /* yy_accept */
	T6 a81; /* yy_last_accepting_state */
	T6 a82; /* yy_last_accepting_cpos */
	T0* a83; /* yy_chk */
	T0* a84; /* yy_base */
	T0* a85; /* yy_def */
	T0* a86; /* yy_meta */
	T0* a87; /* yy_nxt */
	T6 a88; /* yy_lp */
	T0* a89; /* yy_acclist */
	T6 a90; /* yy_looking_for_trail_begin */
	T6 a91; /* yy_full_match */
	T6 a92; /* yy_full_state */
	T6 a93; /* yy_full_lp */
	T1 a94; /* yy_rejected */
	T0* a95; /* last_detachable_any_value */
	T6 a96; /* last_integer_value */
	T1 a97; /* in_trail_context */
	T0* a98; /* rule */
	T1 a99; /* has_trail_context */
	T6 a100; /* head_count */
	T6 a101; /* head_line */
	T6 a102; /* head_column */
	T6 a103; /* trail_count */
	T6 a104; /* rule_line_nb */
};

/* Struct for type LX_DESCRIPTION */
struct S2665 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* equiv_classes */
	T1 a2; /* equiv_classes_used */
	T1 a3; /* full_table */
	T1 a4; /* meta_equiv_classes_used */
	T1 a5; /* reject_used */
	T1 a6; /* variable_trail_context */
	T0* a7; /* rules */
	T0* a8; /* start_conditions */
	T6 a9; /* characters_count */
	T6 a10; /* array_size */
	T1 a11; /* line_pragma */
	T1 a12; /* inspect_used */
	T0* a13; /* eof_rules */
	T0* a14; /* eiffel_header */
	T1 a15; /* case_insensitive */
	T0* a16; /* input_filename */
	T1 a17; /* actions_separated */
	T0* a18; /* eiffel_code */
	T1 a19; /* bol_needed */
	T1 a20; /* pre_action_used */
	T1 a21; /* post_action_used */
	T1 a22; /* pre_eof_action_used */
	T1 a23; /* post_eof_action_used */
	T1 a24; /* line_used */
	T1 a25; /* position_used */
};

/* Struct for type LX_FULL_DFA */
struct S2666 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* yy_nxt */
	T0* a2; /* yy_accept */
	T6 a3; /* yyNb_rows */
	T0* a4; /* input_filename */
	T6 a5; /* characters_count */
	T6 a6; /* array_size */
	T1 a7; /* inspect_used */
	T1 a8; /* actions_separated */
	T0* a9; /* eiffel_code */
	T0* a10; /* eiffel_header */
	T1 a11; /* bol_needed */
	T1 a12; /* pre_action_used */
	T1 a13; /* post_action_used */
	T1 a14; /* pre_eof_action_used */
	T1 a15; /* post_eof_action_used */
	T1 a16; /* line_pragma */
	T0* a17; /* yy_start_conditions */
	T0* a18; /* yy_ec */
	T6 a19; /* yyNull_equiv_class */
	T6 a20; /* yyNb_rules */
	T0* a21; /* yy_rules */
	T6 a22; /* yyEnd_of_buffer */
	T1 a23; /* yyLine_used */
	T1 a24; /* yyPosition_used */
	T6 a25; /* minimum_symbol */
	T6 a26; /* maximum_symbol */
	T0* a27; /* states */
	T6 a28; /* backing_up_count */
	T0* a29; /* partitions */
	T6 a30; /* start_states_count */
	T1 a31; /* yyBacking_up */
	T0* a32; /* yy_eof_rules */
};

/* Struct for type ET_UNIVERSE_ERROR */
struct S2670 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* default_template */
	T0* a2; /* parameters */
	T0* a3; /* code */
	T0* a4; /* etl_code */
	T0* a5; /* universe */
};

/* Struct for type ET_DYNAMIC_EQUALITY_EXPRESSION */
struct S2675 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* target_type_set */
	T0* a2; /* static_equality */
	T0* a3; /* current_feature */
	T0* a4; /* current_type */
	T0* a5; /* next */
};

/* Struct for type ET_DYNAMIC_OBJECT_EQUALITY_EXPRESSION */
struct S2676 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* target_type_set */
	T0* a2; /* static_equality */
	T0* a3; /* current_feature */
	T0* a4; /* current_type */
	T0* a5; /* next */
};

/* Struct for type ET_CONVERT_BUILTIN_EXPRESSION */
struct S2684 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* index */
	T0* a2; /* expression */
	T0* a3; /* convert_feature */
	T0* a4; /* type */
};

/* Struct for type ET_CONVERT_FROM_EXPRESSION */
struct S2685 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* index */
	T0* a2; /* expression */
	T0* a3; /* convert_feature */
	T0* a4; /* type */
};

/* Struct for type ET_CONVERT_TO_EXPRESSION */
struct S2686 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* index */
	T0* a2; /* expression */
	T0* a3; /* convert_feature */
};

/* Struct for type [attached] DS_SPARSE_TABLE_KEYS_CURSOR [detachable RX_REGULAR_EXPRESSION, [attached] STRING_8] */
struct S2697 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* table_cursor */
	T0* a2; /* container */
};

/* Struct for type [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_FEATURE] */
struct S2702 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* next_cursor */
	T6 a2; /* position */
	T0* a3; /* container */
};

/* Struct for type [attached] TUPLE [ET_ECF_TARGET, ET_ECF_STATE] */
struct S2704 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
	T0* z2;
};

/* Struct for type [attached] TUPLE [[attached] ET_ECF_FILE_RULE] */
struct S2705 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type [attached] PROCEDURE [[attached] TUPLE [[attached] ET_ECF_FILE_RULE]] */
struct S2706 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T14 a1; /* rout_disp */
	T0* a2; /* closed_operands */
	T14 a3; /* encaps_rout_disp */
	T14 a4; /* calc_rout_addr */
	T1 a5; /* is_target_closed */
	T6 a6; /* open_count */
};

/* Struct for type [attached] TUPLE [ET_ECF_FILE_RULES] */
struct S2707 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type [attached] PREDICATE [[attached] TUPLE [[attached] ET_ECF_FILE_RULE]] */
struct S2708 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T14 a1; /* rout_disp */
	T0* a2; /* closed_operands */
	T14 a3; /* encaps_rout_disp */
	T14 a4; /* calc_rout_addr */
	T1 a5; /* is_target_closed */
	T6 a6; /* open_count */
};

/* Struct for type [attached] TUPLE [ET_ECF_STATE] */
struct S2709 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type attached DS_LINKABLE [[attached] ET_PARENT_FEATURE] */
struct S2712 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* item */
	T0* a2; /* right */
};

/* Struct for type [attached] SPECIAL [detachable ET_FORMAL_PARAMETER_TYPE] */
struct S2718 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [detachable ET_FORMAL_PARAMETER_TYPE] */
struct S2719 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] DS_ARRAYED_LIST_CURSOR [detachable ET_FORMAL_PARAMETER_TYPE] */
struct S2720 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
};

/* Struct for type [attached] SPECIAL [[attached] ET_AGENT_IMPLICIT_OPEN_ARGUMENT] */
struct S2721 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] ET_AGENT_IMPLICIT_OPEN_ARGUMENT] */
struct S2722 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] DS_ARRAYED_LIST [[attached] LX_RULE] */
struct S2723 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
	T0* a3; /* special_routines */
	T6 a4; /* capacity */
	T0* a5; /* internal_cursor */
};

/* Struct for type LX_START_CONDITIONS */
struct S2724 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
	T0* a3; /* special_routines */
	T6 a4; /* capacity */
	T0* a5; /* internal_cursor */
};

/* Struct for type LX_ACTION_FACTORY */
struct S2725 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type LX_SYMBOL_CLASS */
struct S2726 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* special_routines */
	T0* a2; /* storage */
	T6 a3; /* capacity */
	T1 a4; /* sort_needed */
	T1 a5; /* negated */
	T0* a6; /* internal_cursor */
	T6 a7; /* count */
	T0* a8; /* equality_tester */
};

/* Struct for type [attached] DS_HASH_TABLE [[attached] LX_SYMBOL_CLASS, [attached] STRING_8] */
struct S2727 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T6 a2; /* found_position */
	T0* a3; /* item_storage */
	T0* a4; /* key_equality_tester */
	T0* a5; /* internal_keys */
	T6 a6; /* count */
	T6 a7; /* capacity */
	T6 a8; /* slots_position */
	T6 a9; /* free_slot */
	T6 a10; /* last_position */
	T6 a11; /* modulus */
	T6 a12; /* clashes_previous_position */
	T0* a13; /* equality_tester */
	T0* a14; /* special_item_routines */
	T0* a15; /* clashes */
	T0* a16; /* slots */
	T0* a17; /* special_key_routines */
	T0* a18; /* key_storage */
	T0* a19; /* internal_cursor */
	T0* a20; /* hash_function */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] LX_SYMBOL_CLASS] */
struct S2728 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [[attached] LX_SYMBOL_CLASS] */
struct S2729 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] LX_NFA] */
struct S2730 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type LX_NFA */
struct S2731 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* in_trail_context */
	T0* a2; /* states */
};

/* Struct for type [attached] SPECIAL [[attached] LX_NFA] */
struct S2732 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type LX_RULE */
struct S2735 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* id */
	T0* a2; /* pattern */
	T0* a3; /* action */
	T6 a4; /* head_count */
	T6 a5; /* trail_count */
	T6 a6; /* line_count */
	T6 a7; /* column_count */
	T6 a8; /* line_nb */
	T1 a9; /* has_trail_context */
	T1 a10; /* is_useful */
};

/* Struct for type [attached] DS_HASH_TABLE_CURSOR [[attached] LX_SYMBOL_CLASS, [attached] STRING_8] */
struct S2736 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* container */
	T0* a3; /* next_cursor */
};

/* Struct for type LX_EQUIVALENCE_CLASSES */
struct S2737 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
};

/* Struct for type LX_UNRECOGNIZED_RULE_ERROR */
struct S2738 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* parameters */
};

/* Struct for type LX_MISSING_QUOTE_ERROR */
struct S2739 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* parameters */
};

/* Struct for type LX_BAD_CHARACTER_CLASS_ERROR */
struct S2740 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* parameters */
};

/* Struct for type LX_BAD_CHARACTER_ERROR */
struct S2741 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* parameters */
};

/* Struct for type LX_FULL_AND_META_ERROR */
struct S2742 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* parameters */
};

/* Struct for type LX_FULL_AND_REJECT_ERROR */
struct S2743 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* parameters */
};

/* Struct for type LX_FULL_AND_VARIABLE_TRAILING_CONTEXT_ERROR */
struct S2744 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* parameters */
};

/* Struct for type LX_CHARACTER_OUT_OF_RANGE_ERROR */
struct S2745 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* parameters */
};

/* Struct for type [attached] ARRAY [[attached] LX_RULE] */
struct S2746 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* area */
	T6 a2; /* lower */
	T6 a3; /* upper */
};

/* Struct for type [attached] SPECIAL [[attached] LX_RULE] */
struct S2747 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type LX_DFA_STATE */
struct S2748 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* accepted_rules */
	T0* a2; /* states */
	T0* a3; /* transitions */
	T0* a4; /* accepted_head_rules */
	T6 a5; /* code */
	T6 a6; /* id */
};

/* Struct for type [attached] DS_ARRAYED_LIST [[attached] LX_NFA_STATE] */
struct S2749 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
	T0* a3; /* equality_tester */
	T0* a4; /* special_routines */
	T6 a5; /* capacity */
	T0* a6; /* internal_cursor */
};

/* Struct for type [attached] DS_ARRAYED_LIST [[attached] LX_DFA_STATE] */
struct S2750 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* a3; /* storage */
	T0* a4; /* special_routines */
	T0* a5; /* internal_cursor */
};

/* Struct for type LX_SYMBOL_PARTITIONS */
struct S2752 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* symbols */
	T0* a2; /* storage */
	T6 a3; /* count */
};

/* Struct for type [attached] KL_ARRAY_ROUTINES [[attached] LX_RULE] */
struct S2753 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] ARRAY [detachable LX_RULE] */
struct S2754 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* area */
	T6 a2; /* lower */
	T6 a3; /* upper */
};

/* Struct for type [attached] SPECIAL [detachable LX_RULE] */
struct S2755 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type LX_START_CONDITION */
struct S2756 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* name */
	T6 a2; /* id */
	T1 a3; /* is_exclusive */
	T0* a4; /* patterns */
	T0* a5; /* bol_patterns */
};

/* Struct for type [attached] LX_TRANSITION_TABLE [[attached] LX_DFA_STATE] */
struct S2757 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* storage */
	T0* a2; /* array_routines */
	T6 a3; /* count */
};

/* Struct for type [attached] DS_ARRAYED_LIST [[attached] LX_NFA] */
struct S2759 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* special_routines */
	T0* a2; /* storage */
	T6 a3; /* capacity */
	T0* a4; /* internal_cursor */
	T6 a5; /* count */
};

/* Struct for type [attached] DS_HASH_TABLE [[attached] LX_NFA, [attached] INTEGER_32] */
struct S2760 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* item_storage */
	T0* a3; /* key_equality_tester */
	T6 a4; /* count */
	T6 a5; /* capacity */
	T6 a6; /* slots_position */
	T6 a7; /* free_slot */
	T6 a8; /* last_position */
	T6 a9; /* modulus */
	T6 a10; /* clashes_previous_position */
	T0* a11; /* equality_tester */
	T6 a12; /* found_position */
	T0* a13; /* special_item_routines */
	T0* a14; /* clashes */
	T0* a15; /* slots */
	T0* a16; /* special_key_routines */
	T0* a17; /* key_storage */
	T0* a18; /* internal_cursor */
	T0* a19; /* hash_function */
};

/* Struct for type LX_NFA_STATE */
struct S2761 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* in_trail_context */
	T0* a2; /* transition */
	T0* a3; /* epsilon_transition */
	T6 a4; /* id */
	T0* a5; /* accepted_rule */
};

/* Struct for type [attached] TUPLE [ET_ECF_LIBRARY] */
struct S2763 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type [attached] TUPLE [[attached] DS_HASH_SET [[attached] STRING_8]] */
struct S2771 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type ET_AST_NULL_LEAF */
struct S2776 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* compressed_position */
};

/* Struct for type LX_NEGATIVE_RANGE_IN_CHARACTER_CLASS_ERROR */
struct S2777 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* parameters */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] LX_RULE] */
struct S2778 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] DS_ARRAYED_LIST_CURSOR [[attached] LX_RULE] */
struct S2779 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* next_cursor */
	T0* a3; /* container */
};

/* Struct for type [attached] KL_ARRAY_ROUTINES [[attached] STRING_8] */
struct S2780 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] SPECIAL [[attached] LX_START_CONDITION] */
struct S2781 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] LX_START_CONDITION] */
struct S2782 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] DS_ARRAYED_LIST_CURSOR [[attached] LX_START_CONDITION] */
struct S2783 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* next_cursor */
	T0* a3; /* container */
};

/* Struct for type [attached] LX_SYMBOL_CLASS_TRANSITION [[attached] LX_NFA_STATE] */
struct S2788 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* label */
	T0* a2; /* target */
};

/* Struct for type [attached] LX_EPSILON_TRANSITION [[attached] LX_NFA_STATE] */
struct S2790 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* target */
};

/* Struct for type [attached] LX_SYMBOL_TRANSITION [[attached] LX_NFA_STATE] */
struct S2793 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* label */
	T0* a2; /* target */
};

/* Struct for type LX_ACTION */
struct S2795 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* text */
};

/* Struct for type [attached] ARRAY [[attached] DS_BILINKABLE [[attached] INTEGER_32]] */
struct S2796 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* area */
	T6 a2; /* lower */
	T6 a3; /* upper */
};

/* Struct for type [attached] DS_BILINKABLE [[attached] INTEGER_32] */
struct S2797 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* item */
	T0* a2; /* left */
	T0* a3; /* right */
};

/* Struct for type [attached] SPECIAL [[attached] DS_BILINKABLE [[attached] INTEGER_32]] */
struct S2798 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] DS_BUBBLE_SORTER [[attached] LX_NFA_STATE] */
struct S2800 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* comparator */
};

/* Struct for type [attached] DS_BUBBLE_SORTER [[attached] LX_RULE] */
struct S2802 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* comparator */
};

/* Struct for type [attached] SPECIAL [[attached] LX_NFA_STATE] */
struct S2804 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] LX_NFA_STATE] */
struct S2807 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] DS_ARRAYED_LIST_CURSOR [[attached] LX_NFA_STATE] */
struct S2808 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* next_cursor */
	T0* a3; /* container */
};

/* Struct for type [attached] SPECIAL [[attached] LX_DFA_STATE] */
struct S2811 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_SPECIAL_ROUTINES [[attached] LX_DFA_STATE] */
struct S2812 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] DS_ARRAYED_LIST_CURSOR [[attached] LX_DFA_STATE] */
struct S2813 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
};

/* Struct for type [attached] KL_ARRAY [[attached] LX_RULE] */
struct S2814 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* area */
	T6 a2; /* lower */
	T6 a3; /* upper */
};

/* Struct for type [attached] ARRAY [detachable LX_DFA_STATE] */
struct S2815 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* area */
	T6 a2; /* lower */
	T6 a3; /* upper */
};

/* Struct for type [attached] SPECIAL [detachable LX_DFA_STATE] */
struct S2816 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [attached] KL_ARRAY_ROUTINES [detachable LX_DFA_STATE] */
struct S2817 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] DS_ARRAYED_LIST_CURSOR [[attached] LX_NFA] */
struct S2818 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
};

/* Struct for type [attached] DS_HASH_TABLE_CURSOR [[attached] LX_NFA, [attached] INTEGER_32] */
struct S2820 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
};

/* Struct for type CHARACTER_PROPERTY */
struct S2821 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] TUPLE [ET_ECF_LIBRARY, [attached] DS_HASH_TABLE [[attached] INTEGER_32, [attached] ET_UNIVERSE], INTEGER_32] */
struct S2823 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
	T0* z2;
	T6 z3;
};

/* Struct for type [attached] TUPLE [ET_ECF_DOTNET_ASSEMBLY, [attached] DS_HASH_TABLE [[attached] INTEGER_32, [attached] ET_UNIVERSE], INTEGER_32] */
struct S2824 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
	T0* z2;
	T6 z3;
};

/* Struct for type [attached] TUPLE [ET_LACE_DOTNET_ASSEMBLY, [attached] DS_HASH_TABLE [[attached] INTEGER_32, [attached] ET_UNIVERSE], INTEGER_32] */
struct S2825 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
	T0* z2;
	T6 z3;
};

/* Struct for type [attached] TUPLE [ET_LACE_DOTNET_GAC_ASSEMBLY, [attached] DS_HASH_TABLE [[attached] INTEGER_32, [attached] ET_UNIVERSE], INTEGER_32] */
struct S2826 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
	T0* z2;
	T6 z3;
};

/* Struct for type [attached] TUPLE [ET_DOTNET_ASSEMBLY, [attached] DS_HASH_TABLE [[attached] INTEGER_32, [attached] ET_UNIVERSE], INTEGER_32] */
struct S2827 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
	T0* z2;
	T6 z3;
};

/* Struct for type [attached] TUPLE [ET_LIBRARY, [attached] DS_HASH_TABLE [[attached] INTEGER_32, [attached] ET_UNIVERSE], INTEGER_32] */
struct S2828 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
	T0* z2;
	T6 z3;
};

/* Struct for type [attached] TUPLE [[attached] RX_PCRE_REGULAR_EXPRESSION] */
struct S2830 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* z1;
};

/* Struct for type [attached] PREDICATE [[attached] TUPLE [[attached] RX_PCRE_REGULAR_EXPRESSION]] */
struct S2831 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T14 a1; /* rout_disp */
	T0* a2; /* closed_operands */
	T14 a3; /* encaps_rout_disp */
	T14 a4; /* calc_rout_addr */
	T1 a5; /* is_target_closed */
	T6 a6; /* open_count */
};

/* Struct for type [attached] DS_SHELL_SORTER [[attached] INTEGER_32] */
struct S2836 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* comparator */
};

/* Struct for type [attached] KL_COMPARABLE_COMPARATOR [[attached] LX_RULE] */
struct S2838 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] KL_COMPARABLE_COMPARATOR [[attached] LX_NFA_STATE] */
struct S2841 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [attached] KL_ARRAY [[attached] STRING_8] */
struct S2844 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* area */
	T6 a2; /* lower */
	T6 a3; /* upper */
};

/* Struct for type [attached] SPECIAL [[attached] NATURAL_16] */
struct S2845 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T9 z2[1]; /* item */
};

/* Struct for type [attached] SPECIAL [[attached] NATURAL_32] */
struct S2846 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T10 z2[1]; /* item */
};

/* Struct for type [attached] ARRAY [[attached] NATURAL_16] */
struct S2847 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* area */
	T6 a2; /* lower */
	T6 a3; /* upper */
};

/* Struct for type [attached] ARRAY [[attached] NATURAL_32] */
struct S2848 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* area */
	T6 a2; /* lower */
	T6 a3; /* upper */
};


extern T1 GE_default1;
extern T2 GE_default2;
extern T3 GE_default3;
extern T4 GE_default4;
extern T5 GE_default5;
extern T6 GE_default6;
extern T7 GE_default7;
extern T8 GE_default8;
extern T9 GE_default9;
extern T10 GE_default10;
extern T11 GE_default11;
extern T12 GE_default12;
extern T13 GE_default13;
extern T14 GE_default14;
extern T15 GE_default15;
extern T16 GE_default16;
extern T17 GE_default17;
extern T18 GE_default18;
extern T19 GE_default19;
extern T20 GE_default20;
extern T25 GE_default25;
extern T26 GE_default26;
extern T27 GE_default27;
extern T28 GE_default28;
extern T29 GE_default29;
extern T31 GE_default31;
extern T32 GE_default32;
extern T33 GE_default33;
extern T34 GE_default34;
extern T35 GE_default35;
extern T36 GE_default36;
extern T38 GE_default38;
extern T40 GE_default40;
extern T41 GE_default41;
extern T42 GE_default42;
extern T43 GE_default43;
extern T44 GE_default44;
extern T46 GE_default46;
extern T47 GE_default47;
extern T48 GE_default48;
extern T49 GE_default49;
extern T53 GE_default53;
extern T55 GE_default55;
extern T56 GE_default56;
extern T57 GE_default57;
extern T58 GE_default58;
extern T61 GE_default61;
extern T62 GE_default62;
extern T66 GE_default66;
extern T67 GE_default67;
extern T69 GE_default69;
extern T70 GE_default70;
extern T71 GE_default71;
extern T77 GE_default77;
extern T78 GE_default78;
extern T80 GE_default80;
extern T81 GE_default81;
extern T84 GE_default84;
extern T85 GE_default85;
extern T87 GE_default87;
extern T88 GE_default88;
extern T91 GE_default91;
extern T93 GE_default93;
extern T94 GE_default94;
extern T95 GE_default95;
extern T97 GE_default97;
extern T100 GE_default100;
extern T102 GE_default102;
extern T103 GE_default103;
extern T106 GE_default106;
extern T107 GE_default107;
extern T108 GE_default108;
extern T109 GE_default109;
extern T112 GE_default112;
extern T114 GE_default114;
extern T116 GE_default116;
extern T119 GE_default119;
extern T120 GE_default120;
extern T121 GE_default121;
extern T122 GE_default122;
extern T123 GE_default123;
extern T124 GE_default124;
extern T125 GE_default125;
extern T128 GE_default128;
extern T129 GE_default129;
extern T130 GE_default130;
extern T131 GE_default131;
extern T132 GE_default132;
extern T133 GE_default133;
extern T134 GE_default134;
extern T135 GE_default135;
extern T136 GE_default136;
extern T137 GE_default137;
extern T138 GE_default138;
extern T139 GE_default139;
extern T140 GE_default140;
extern T141 GE_default141;
extern T142 GE_default142;
extern T143 GE_default143;
extern T144 GE_default144;
extern T145 GE_default145;
extern T146 GE_default146;
extern T147 GE_default147;
extern T148 GE_default148;
extern T149 GE_default149;
extern T150 GE_default150;
extern T154 GE_default154;
extern T155 GE_default155;
extern T157 GE_default157;
extern T158 GE_default158;
extern T159 GE_default159;
extern T160 GE_default160;
extern T161 GE_default161;
extern T162 GE_default162;
extern T163 GE_default163;
extern T164 GE_default164;
extern T166 GE_default166;
extern T168 GE_default168;
extern T169 GE_default169;
extern T170 GE_default170;
extern T171 GE_default171;
extern T172 GE_default172;
extern T175 GE_default175;
extern T176 GE_default176;
extern T177 GE_default177;
extern T178 GE_default178;
extern T179 GE_default179;
extern T182 GE_default182;
extern T185 GE_default185;
extern T186 GE_default186;
extern T187 GE_default187;
extern T188 GE_default188;
extern T189 GE_default189;
extern T192 GE_default192;
extern T193 GE_default193;
extern T194 GE_default194;
extern T195 GE_default195;
extern T196 GE_default196;
extern T197 GE_default197;
extern T198 GE_default198;
extern T199 GE_default199;
extern T200 GE_default200;
extern T203 GE_default203;
extern T206 GE_default206;
extern T207 GE_default207;
extern T208 GE_default208;
extern T209 GE_default209;
extern T210 GE_default210;
extern T212 GE_default212;
extern T213 GE_default213;
extern T214 GE_default214;
extern T216 GE_default216;
extern T217 GE_default217;
extern T219 GE_default219;
extern T220 GE_default220;
extern T221 GE_default221;
extern T222 GE_default222;
extern T223 GE_default223;
extern T225 GE_default225;
extern T226 GE_default226;
extern T227 GE_default227;
extern T229 GE_default229;
extern T230 GE_default230;
extern T231 GE_default231;
extern T232 GE_default232;
extern T233 GE_default233;
extern T236 GE_default236;
extern T237 GE_default237;
extern T239 GE_default239;
extern T240 GE_default240;
extern T241 GE_default241;
extern T245 GE_default245;
extern T247 GE_default247;
extern T248 GE_default248;
extern T250 GE_default250;
extern T253 GE_default253;
extern T255 GE_default255;
extern T256 GE_default256;
extern T257 GE_default257;
extern T259 GE_default259;
extern T260 GE_default260;
extern T261 GE_default261;
extern T264 GE_default264;
extern T265 GE_default265;
extern T267 GE_default267;
extern T268 GE_default268;
extern T269 GE_default269;
extern T270 GE_default270;
extern T271 GE_default271;
extern T272 GE_default272;
extern T273 GE_default273;
extern T279 GE_default279;
extern T280 GE_default280;
extern T281 GE_default281;
extern T282 GE_default282;
extern T283 GE_default283;
extern T285 GE_default285;
extern T286 GE_default286;
extern T288 GE_default288;
extern T290 GE_default290;
extern T291 GE_default291;
extern T292 GE_default292;
extern T293 GE_default293;
extern T294 GE_default294;
extern T295 GE_default295;
extern T296 GE_default296;
extern T297 GE_default297;
extern T298 GE_default298;
extern T299 GE_default299;
extern T300 GE_default300;
extern T301 GE_default301;
extern T302 GE_default302;
extern T303 GE_default303;
extern T304 GE_default304;
extern T305 GE_default305;
extern T306 GE_default306;
extern T307 GE_default307;
extern T308 GE_default308;
extern T309 GE_default309;
extern T310 GE_default310;
extern T311 GE_default311;
extern T312 GE_default312;
extern T313 GE_default313;
extern T314 GE_default314;
extern T315 GE_default315;
extern T316 GE_default316;
extern T317 GE_default317;
extern T318 GE_default318;
extern T319 GE_default319;
extern T320 GE_default320;
extern T321 GE_default321;
extern T332 GE_default332;
extern T333 GE_default333;
extern T344 GE_default344;
extern T345 GE_default345;
extern T347 GE_default347;
extern T348 GE_default348;
extern T349 GE_default349;
extern T350 GE_default350;
extern T351 GE_default351;
extern T352 GE_default352;
extern T353 GE_default353;
extern T354 GE_default354;
extern T355 GE_default355;
extern T357 GE_default357;
extern T358 GE_default358;
extern T359 GE_default359;
extern T360 GE_default360;
extern T361 GE_default361;
extern T362 GE_default362;
extern T363 GE_default363;
extern T364 GE_default364;
extern T365 GE_default365;
extern T366 GE_default366;
extern T367 GE_default367;
extern T368 GE_default368;
extern T369 GE_default369;
extern T371 GE_default371;
extern T372 GE_default372;
extern T373 GE_default373;
extern T374 GE_default374;
extern T375 GE_default375;
extern T376 GE_default376;
extern T377 GE_default377;
extern T378 GE_default378;
extern T379 GE_default379;
extern T380 GE_default380;
extern T381 GE_default381;
extern T382 GE_default382;
extern T383 GE_default383;
extern T384 GE_default384;
extern T385 GE_default385;
extern T386 GE_default386;
extern T387 GE_default387;
extern T388 GE_default388;
extern T389 GE_default389;
extern T390 GE_default390;
extern T391 GE_default391;
extern T392 GE_default392;
extern T396 GE_default396;
extern T397 GE_default397;
extern T398 GE_default398;
extern T399 GE_default399;
extern T400 GE_default400;
extern T401 GE_default401;
extern T402 GE_default402;
extern T403 GE_default403;
extern T404 GE_default404;
extern T405 GE_default405;
extern T406 GE_default406;
extern T407 GE_default407;
extern T408 GE_default408;
extern T410 GE_default410;
extern T411 GE_default411;
extern T412 GE_default412;
extern T413 GE_default413;
extern T414 GE_default414;
extern T416 GE_default416;
extern T418 GE_default418;
extern T419 GE_default419;
extern T422 GE_default422;
extern T424 GE_default424;
extern T429 GE_default429;
extern T431 GE_default431;
extern T432 GE_default432;
extern T433 GE_default433;
extern T434 GE_default434;
extern T438 GE_default438;
extern T439 GE_default439;
extern T444 GE_default444;
extern T445 GE_default445;
extern T449 GE_default449;
extern T451 GE_default451;
extern T452 GE_default452;
extern T454 GE_default454;
extern T456 GE_default456;
extern T464 GE_default464;
extern T472 GE_default472;
extern T474 GE_default474;
extern T475 GE_default475;
extern T476 GE_default476;
extern T477 GE_default477;
extern T478 GE_default478;
extern T479 GE_default479;
extern T480 GE_default480;
extern T485 GE_default485;
extern T487 GE_default487;
extern T490 GE_default490;
extern T491 GE_default491;
extern T492 GE_default492;
extern T494 GE_default494;
extern T495 GE_default495;
extern T496 GE_default496;
extern T498 GE_default498;
extern T500 GE_default500;
extern T501 GE_default501;
extern T502 GE_default502;
extern T503 GE_default503;
extern T505 GE_default505;
extern T506 GE_default506;
extern T507 GE_default507;
extern T510 GE_default510;
extern T512 GE_default512;
extern T513 GE_default513;
extern T514 GE_default514;
extern T515 GE_default515;
extern T516 GE_default516;
extern T517 GE_default517;
extern T518 GE_default518;
extern T519 GE_default519;
extern T520 GE_default520;
extern T521 GE_default521;
extern T522 GE_default522;
extern T523 GE_default523;
extern T524 GE_default524;
extern T525 GE_default525;
extern T526 GE_default526;
extern T527 GE_default527;
extern T529 GE_default529;
extern T530 GE_default530;
extern T533 GE_default533;
extern T536 GE_default536;
extern T538 GE_default538;
extern T539 GE_default539;
extern T540 GE_default540;
extern T542 GE_default542;
extern T543 GE_default543;
extern T544 GE_default544;
extern T547 GE_default547;
extern T548 GE_default548;
extern T549 GE_default549;
extern T551 GE_default551;
extern T552 GE_default552;
extern T553 GE_default553;
extern T554 GE_default554;
extern T557 GE_default557;
extern T559 GE_default559;
extern T560 GE_default560;
extern T564 GE_default564;
extern T569 GE_default569;
extern T571 GE_default571;
extern T575 GE_default575;
extern T576 GE_default576;
extern T581 GE_default581;
extern T583 GE_default583;
extern T584 GE_default584;
extern T586 GE_default586;
extern T587 GE_default587;
extern T588 GE_default588;
extern T589 GE_default589;
extern T592 GE_default592;
extern T595 GE_default595;
extern T596 GE_default596;
extern T598 GE_default598;
extern T599 GE_default599;
extern T601 GE_default601;
extern T603 GE_default603;
extern T604 GE_default604;
extern T605 GE_default605;
extern T606 GE_default606;
extern T607 GE_default607;
extern T608 GE_default608;
extern T611 GE_default611;
extern T612 GE_default612;
extern T614 GE_default614;
extern T615 GE_default615;
extern T616 GE_default616;
extern T618 GE_default618;
extern T620 GE_default620;
extern T621 GE_default621;
extern T623 GE_default623;
extern T626 GE_default626;
extern T627 GE_default627;
extern T629 GE_default629;
extern T630 GE_default630;
extern T632 GE_default632;
extern T633 GE_default633;
extern T634 GE_default634;
extern T635 GE_default635;
extern T637 GE_default637;
extern T638 GE_default638;
extern T642 GE_default642;
extern T643 GE_default643;
extern T644 GE_default644;
extern T645 GE_default645;
extern T647 GE_default647;
extern T648 GE_default648;
extern T649 GE_default649;
extern T650 GE_default650;
extern T651 GE_default651;
extern T652 GE_default652;
extern T653 GE_default653;
extern T654 GE_default654;
extern T655 GE_default655;
extern T656 GE_default656;
extern T657 GE_default657;
extern T658 GE_default658;
extern T659 GE_default659;
extern T660 GE_default660;
extern T661 GE_default661;
extern T662 GE_default662;
extern T663 GE_default663;
extern T664 GE_default664;
extern T665 GE_default665;
extern T667 GE_default667;
extern T668 GE_default668;
extern T669 GE_default669;
extern T670 GE_default670;
extern T671 GE_default671;
extern T672 GE_default672;
extern T674 GE_default674;
extern T675 GE_default675;
extern T676 GE_default676;
extern T677 GE_default677;
extern T678 GE_default678;
extern T679 GE_default679;
extern T680 GE_default680;
extern T681 GE_default681;
extern T682 GE_default682;
extern T683 GE_default683;
extern T684 GE_default684;
extern T685 GE_default685;
extern T686 GE_default686;
extern T687 GE_default687;
extern T689 GE_default689;
extern T690 GE_default690;
extern T691 GE_default691;
extern T692 GE_default692;
extern T693 GE_default693;
extern T694 GE_default694;
extern T696 GE_default696;
extern T698 GE_default698;
extern T699 GE_default699;
extern T700 GE_default700;
extern T701 GE_default701;
extern T706 GE_default706;
extern T712 GE_default712;
extern T713 GE_default713;
extern T714 GE_default714;
extern T715 GE_default715;
extern T716 GE_default716;
extern T717 GE_default717;
extern T723 GE_default723;
extern T724 GE_default724;
extern T729 GE_default729;
extern T730 GE_default730;
extern T731 GE_default731;
extern T732 GE_default732;
extern T733 GE_default733;
extern T736 GE_default736;
extern T737 GE_default737;
extern T738 GE_default738;
extern T740 GE_default740;
extern T742 GE_default742;
extern T743 GE_default743;
extern T744 GE_default744;
extern T745 GE_default745;
extern T746 GE_default746;
extern T747 GE_default747;
extern T748 GE_default748;
extern T749 GE_default749;
extern T750 GE_default750;
extern T751 GE_default751;
extern T752 GE_default752;
extern T753 GE_default753;
extern T754 GE_default754;
extern T755 GE_default755;
extern T756 GE_default756;
extern T757 GE_default757;
extern T758 GE_default758;
extern T759 GE_default759;
extern T760 GE_default760;
extern T761 GE_default761;
extern T762 GE_default762;
extern T763 GE_default763;
extern T764 GE_default764;
extern T765 GE_default765;
extern T766 GE_default766;
extern T767 GE_default767;
extern T772 GE_default772;
extern T773 GE_default773;
extern T775 GE_default775;
extern T777 GE_default777;
extern T778 GE_default778;
extern T783 GE_default783;
extern T784 GE_default784;
extern T789 GE_default789;
extern T790 GE_default790;
extern T791 GE_default791;
extern T792 GE_default792;
extern T793 GE_default793;
extern T794 GE_default794;
extern T796 GE_default796;
extern T798 GE_default798;
extern T799 GE_default799;
extern T800 GE_default800;
extern T801 GE_default801;
extern T803 GE_default803;
extern T807 GE_default807;
extern T808 GE_default808;
extern T809 GE_default809;
extern T810 GE_default810;
extern T812 GE_default812;
extern T815 GE_default815;
extern T816 GE_default816;
extern T819 GE_default819;
extern T820 GE_default820;
extern T821 GE_default821;
extern T828 GE_default828;
extern T829 GE_default829;
extern T830 GE_default830;
extern T831 GE_default831;
extern T832 GE_default832;
extern T833 GE_default833;
extern T834 GE_default834;
extern T835 GE_default835;
extern T845 GE_default845;
extern T846 GE_default846;
extern T848 GE_default848;
extern T850 GE_default850;
extern T851 GE_default851;
extern T853 GE_default853;
extern T854 GE_default854;
extern T855 GE_default855;
extern T856 GE_default856;
extern T857 GE_default857;
extern T858 GE_default858;
extern T859 GE_default859;
extern T860 GE_default860;
extern T861 GE_default861;
extern T862 GE_default862;
extern T863 GE_default863;
extern T864 GE_default864;
extern T865 GE_default865;
extern T866 GE_default866;
extern T867 GE_default867;
extern T869 GE_default869;
extern T870 GE_default870;
extern T872 GE_default872;
extern T873 GE_default873;
extern T875 GE_default875;
extern T876 GE_default876;
extern T878 GE_default878;
extern T879 GE_default879;
extern T881 GE_default881;
extern T882 GE_default882;
extern T884 GE_default884;
extern T885 GE_default885;
extern T887 GE_default887;
extern T888 GE_default888;
extern T890 GE_default890;
extern T891 GE_default891;
extern T893 GE_default893;
extern T894 GE_default894;
extern T896 GE_default896;
extern T897 GE_default897;
extern T899 GE_default899;
extern T900 GE_default900;
extern T902 GE_default902;
extern T903 GE_default903;
extern T905 GE_default905;
extern T906 GE_default906;
extern T908 GE_default908;
extern T909 GE_default909;
extern T910 GE_default910;
extern T911 GE_default911;
extern T913 GE_default913;
extern T914 GE_default914;
extern T916 GE_default916;
extern T917 GE_default917;
extern T919 GE_default919;
extern T920 GE_default920;
extern T922 GE_default922;
extern T923 GE_default923;
extern T925 GE_default925;
extern T926 GE_default926;
extern T928 GE_default928;
extern T929 GE_default929;
extern T931 GE_default931;
extern T932 GE_default932;
extern T934 GE_default934;
extern T935 GE_default935;
extern T936 GE_default936;
extern T937 GE_default937;
extern T939 GE_default939;
extern T940 GE_default940;
extern T941 GE_default941;
extern T942 GE_default942;
extern T944 GE_default944;
extern T945 GE_default945;
extern T947 GE_default947;
extern T948 GE_default948;
extern T950 GE_default950;
extern T951 GE_default951;
extern T953 GE_default953;
extern T954 GE_default954;
extern T956 GE_default956;
extern T957 GE_default957;
extern T959 GE_default959;
extern T960 GE_default960;
extern T962 GE_default962;
extern T963 GE_default963;
extern T965 GE_default965;
extern T966 GE_default966;
extern T967 GE_default967;
extern T968 GE_default968;
extern T970 GE_default970;
extern T971 GE_default971;
extern T973 GE_default973;
extern T974 GE_default974;
extern T976 GE_default976;
extern T977 GE_default977;
extern T979 GE_default979;
extern T980 GE_default980;
extern T982 GE_default982;
extern T983 GE_default983;
extern T985 GE_default985;
extern T986 GE_default986;
extern T988 GE_default988;
extern T989 GE_default989;
extern T991 GE_default991;
extern T992 GE_default992;
extern T993 GE_default993;
extern T994 GE_default994;
extern T996 GE_default996;
extern T997 GE_default997;
extern T999 GE_default999;
extern T1000 GE_default1000;
extern T1002 GE_default1002;
extern T1003 GE_default1003;
extern T1005 GE_default1005;
extern T1006 GE_default1006;
extern T1008 GE_default1008;
extern T1009 GE_default1009;
extern T1011 GE_default1011;
extern T1012 GE_default1012;
extern T1014 GE_default1014;
extern T1015 GE_default1015;
extern T1017 GE_default1017;
extern T1018 GE_default1018;
extern T1020 GE_default1020;
extern T1021 GE_default1021;
extern T1023 GE_default1023;
extern T1024 GE_default1024;
extern T1026 GE_default1026;
extern T1027 GE_default1027;
extern T1029 GE_default1029;
extern T1030 GE_default1030;
extern T1032 GE_default1032;
extern T1033 GE_default1033;
extern T1035 GE_default1035;
extern T1036 GE_default1036;
extern T1038 GE_default1038;
extern T1039 GE_default1039;
extern T1041 GE_default1041;
extern T1042 GE_default1042;
extern T1044 GE_default1044;
extern T1045 GE_default1045;
extern T1047 GE_default1047;
extern T1048 GE_default1048;
extern T1050 GE_default1050;
extern T1051 GE_default1051;
extern T1052 GE_default1052;
extern T1053 GE_default1053;
extern T1054 GE_default1054;
extern T1055 GE_default1055;
extern T1057 GE_default1057;
extern T1058 GE_default1058;
extern T1060 GE_default1060;
extern T1061 GE_default1061;
extern T1063 GE_default1063;
extern T1064 GE_default1064;
extern T1066 GE_default1066;
extern T1067 GE_default1067;
extern T1069 GE_default1069;
extern T1070 GE_default1070;
extern T1072 GE_default1072;
extern T1073 GE_default1073;
extern T1074 GE_default1074;
extern T1075 GE_default1075;
extern T1077 GE_default1077;
extern T1078 GE_default1078;
extern T1080 GE_default1080;
extern T1081 GE_default1081;
extern T1083 GE_default1083;
extern T1084 GE_default1084;
extern T1085 GE_default1085;
extern T1086 GE_default1086;
extern T1088 GE_default1088;
extern T1089 GE_default1089;
extern T1091 GE_default1091;
extern T1092 GE_default1092;
extern T1093 GE_default1093;
extern T1094 GE_default1094;
extern T1096 GE_default1096;
extern T1097 GE_default1097;
extern T1099 GE_default1099;
extern T1100 GE_default1100;
extern T1102 GE_default1102;
extern T1103 GE_default1103;
extern T1105 GE_default1105;
extern T1106 GE_default1106;
extern T1108 GE_default1108;
extern T1109 GE_default1109;
extern T1111 GE_default1111;
extern T1112 GE_default1112;
extern T1114 GE_default1114;
extern T1115 GE_default1115;
extern T1117 GE_default1117;
extern T1118 GE_default1118;
extern T1120 GE_default1120;
extern T1121 GE_default1121;
extern T1123 GE_default1123;
extern T1124 GE_default1124;
extern T1126 GE_default1126;
extern T1127 GE_default1127;
extern T1128 GE_default1128;
extern T1129 GE_default1129;
extern T1131 GE_default1131;
extern T1132 GE_default1132;
extern T1134 GE_default1134;
extern T1135 GE_default1135;
extern T1137 GE_default1137;
extern T1138 GE_default1138;
extern T1139 GE_default1139;
extern T1140 GE_default1140;
extern T1142 GE_default1142;
extern T1143 GE_default1143;
extern T1145 GE_default1145;
extern T1146 GE_default1146;
extern T1148 GE_default1148;
extern T1149 GE_default1149;
extern T1151 GE_default1151;
extern T1152 GE_default1152;
extern T1153 GE_default1153;
extern T1154 GE_default1154;
extern T1156 GE_default1156;
extern T1157 GE_default1157;
extern T1159 GE_default1159;
extern T1160 GE_default1160;
extern T1162 GE_default1162;
extern T1163 GE_default1163;
extern T1165 GE_default1165;
extern T1166 GE_default1166;
extern T1168 GE_default1168;
extern T1169 GE_default1169;
extern T1171 GE_default1171;
extern T1172 GE_default1172;
extern T1174 GE_default1174;
extern T1175 GE_default1175;
extern T1176 GE_default1176;
extern T1177 GE_default1177;
extern T1178 GE_default1178;
extern T1179 GE_default1179;
extern T1181 GE_default1181;
extern T1182 GE_default1182;
extern T1183 GE_default1183;
extern T1184 GE_default1184;
extern T1186 GE_default1186;
extern T1187 GE_default1187;
extern T1188 GE_default1188;
extern T1189 GE_default1189;
extern T1191 GE_default1191;
extern T1192 GE_default1192;
extern T1194 GE_default1194;
extern T1195 GE_default1195;
extern T1196 GE_default1196;
extern T1197 GE_default1197;
extern T1199 GE_default1199;
extern T1200 GE_default1200;
extern T1202 GE_default1202;
extern T1203 GE_default1203;
extern T1205 GE_default1205;
extern T1206 GE_default1206;
extern T1208 GE_default1208;
extern T1209 GE_default1209;
extern T1211 GE_default1211;
extern T1212 GE_default1212;
extern T1213 GE_default1213;
extern T1214 GE_default1214;
extern T1215 GE_default1215;
extern T1216 GE_default1216;
extern T1217 GE_default1217;
extern T1218 GE_default1218;
extern T1219 GE_default1219;
extern T1221 GE_default1221;
extern T1222 GE_default1222;
extern T1224 GE_default1224;
extern T1225 GE_default1225;
extern T1226 GE_default1226;
extern T1227 GE_default1227;
extern T1228 GE_default1228;
extern T1229 GE_default1229;
extern T1230 GE_default1230;
extern T1231 GE_default1231;
extern T1232 GE_default1232;
extern T1233 GE_default1233;
extern T1234 GE_default1234;
extern T1235 GE_default1235;
extern T1236 GE_default1236;
extern T1237 GE_default1237;
extern T1238 GE_default1238;
extern T1239 GE_default1239;
extern T1240 GE_default1240;
extern T1242 GE_default1242;
extern T1243 GE_default1243;
extern T1244 GE_default1244;
extern T1246 GE_default1246;
extern T1247 GE_default1247;
extern T1248 GE_default1248;
extern T1249 GE_default1249;
extern T1250 GE_default1250;
extern T1251 GE_default1251;
extern T1252 GE_default1252;
extern T1253 GE_default1253;
extern T1255 GE_default1255;
extern T1258 GE_default1258;
extern T1259 GE_default1259;
extern T1260 GE_default1260;
extern T1261 GE_default1261;
extern T1262 GE_default1262;
extern T1263 GE_default1263;
extern T1265 GE_default1265;
extern T1266 GE_default1266;
extern T1267 GE_default1267;
extern T1271 GE_default1271;
extern T1273 GE_default1273;
extern T1274 GE_default1274;
extern T1275 GE_default1275;
extern T1276 GE_default1276;
extern T1279 GE_default1279;
extern T1282 GE_default1282;
extern T1283 GE_default1283;
extern T1284 GE_default1284;
extern T1286 GE_default1286;
extern T1287 GE_default1287;
extern T1288 GE_default1288;
extern T1290 GE_default1290;
extern T1291 GE_default1291;
extern T1292 GE_default1292;
extern T1294 GE_default1294;
extern T1295 GE_default1295;
extern T1296 GE_default1296;
extern T1298 GE_default1298;
extern T1300 GE_default1300;
extern T1301 GE_default1301;
extern T1302 GE_default1302;
extern T1303 GE_default1303;
extern T1304 GE_default1304;
extern T1309 GE_default1309;
extern T1310 GE_default1310;
extern T1311 GE_default1311;
extern T1312 GE_default1312;
extern T1313 GE_default1313;
extern T1316 GE_default1316;
extern T1317 GE_default1317;
extern T1318 GE_default1318;
extern T1321 GE_default1321;
extern T1322 GE_default1322;
extern T1325 GE_default1325;
extern T1326 GE_default1326;
extern T1327 GE_default1327;
extern T1328 GE_default1328;
extern T1329 GE_default1329;
extern T1331 GE_default1331;
extern T1337 GE_default1337;
extern T1338 GE_default1338;
extern T1339 GE_default1339;
extern T1342 GE_default1342;
extern T1343 GE_default1343;
extern T1344 GE_default1344;
extern T1345 GE_default1345;
extern T1346 GE_default1346;
extern T1349 GE_default1349;
extern T1350 GE_default1350;
extern T1351 GE_default1351;
extern T1352 GE_default1352;
extern T1353 GE_default1353;
extern T1359 GE_default1359;
extern T1361 GE_default1361;
extern T1363 GE_default1363;
extern T1364 GE_default1364;
extern T1365 GE_default1365;
extern T1366 GE_default1366;
extern T1367 GE_default1367;
extern T1368 GE_default1368;
extern T1369 GE_default1369;
extern T1370 GE_default1370;
extern T1371 GE_default1371;
extern T1372 GE_default1372;
extern T1373 GE_default1373;
extern T1374 GE_default1374;
extern T1375 GE_default1375;
extern T1376 GE_default1376;
extern T1377 GE_default1377;
extern T1379 GE_default1379;
extern T1380 GE_default1380;
extern T1384 GE_default1384;
extern T1385 GE_default1385;
extern T1386 GE_default1386;
extern T1387 GE_default1387;
extern T1388 GE_default1388;
extern T1392 GE_default1392;
extern T1393 GE_default1393;
extern T1394 GE_default1394;
extern T1395 GE_default1395;
extern T1398 GE_default1398;
extern T1399 GE_default1399;
extern T1400 GE_default1400;
extern T1402 GE_default1402;
extern T1404 GE_default1404;
extern T1405 GE_default1405;
extern T1406 GE_default1406;
extern T1407 GE_default1407;
extern T1409 GE_default1409;
extern T1411 GE_default1411;
extern T1415 GE_default1415;
extern T1416 GE_default1416;
extern T1418 GE_default1418;
extern T1419 GE_default1419;
extern T1422 GE_default1422;
extern T1424 GE_default1424;
extern T1426 GE_default1426;
extern T1427 GE_default1427;
extern T1428 GE_default1428;
extern T1429 GE_default1429;
extern T1430 GE_default1430;
extern T1431 GE_default1431;
extern T1432 GE_default1432;
extern T1433 GE_default1433;
extern T1434 GE_default1434;
extern T1438 GE_default1438;
extern T1439 GE_default1439;
extern T1453 GE_default1453;
extern T1455 GE_default1455;
extern T1459 GE_default1459;
extern T1461 GE_default1461;
extern T1463 GE_default1463;
extern T1464 GE_default1464;
extern T1466 GE_default1466;
extern T1467 GE_default1467;
extern T1469 GE_default1469;
extern T1470 GE_default1470;
extern T1472 GE_default1472;
extern T1473 GE_default1473;
extern T1475 GE_default1475;
extern T1476 GE_default1476;
extern T1478 GE_default1478;
extern T1479 GE_default1479;
extern T1481 GE_default1481;
extern T1482 GE_default1482;
extern T1484 GE_default1484;
extern T1485 GE_default1485;
extern T1487 GE_default1487;
extern T1488 GE_default1488;
extern T1490 GE_default1490;
extern T1492 GE_default1492;
extern T1493 GE_default1493;
extern T1497 GE_default1497;
extern T1498 GE_default1498;
extern T1500 GE_default1500;
extern T1501 GE_default1501;
extern T1503 GE_default1503;
extern T1505 GE_default1505;
extern T1506 GE_default1506;
extern T1508 GE_default1508;
extern T1509 GE_default1509;
extern T1518 GE_default1518;
extern T1519 GE_default1519;
extern T1520 GE_default1520;
extern T1521 GE_default1521;
extern T1522 GE_default1522;
extern T1523 GE_default1523;
extern T1524 GE_default1524;
extern T1525 GE_default1525;
extern T1526 GE_default1526;
extern T1528 GE_default1528;
extern T1532 GE_default1532;
extern T1533 GE_default1533;
extern T1534 GE_default1534;
extern T1537 GE_default1537;
extern T1538 GE_default1538;
extern T1540 GE_default1540;
extern T1542 GE_default1542;
extern T1546 GE_default1546;
extern T1547 GE_default1547;
extern T1548 GE_default1548;
extern T1549 GE_default1549;
extern T1550 GE_default1550;
extern T1551 GE_default1551;
extern T1553 GE_default1553;
extern T1554 GE_default1554;
extern T1555 GE_default1555;
extern T1556 GE_default1556;
extern T1557 GE_default1557;
extern T1558 GE_default1558;
extern T1562 GE_default1562;
extern T1564 GE_default1564;
extern T1566 GE_default1566;
extern T1569 GE_default1569;
extern T1572 GE_default1572;
extern T1573 GE_default1573;
extern T1574 GE_default1574;
extern T1575 GE_default1575;
extern T1576 GE_default1576;
extern T1577 GE_default1577;
extern T1578 GE_default1578;
extern T1583 GE_default1583;
extern T1585 GE_default1585;
extern T1625 GE_default1625;
extern T1631 GE_default1631;
extern T1655 GE_default1655;
extern T1675 GE_default1675;
extern T1676 GE_default1676;
extern T1677 GE_default1677;
extern T1712 GE_default1712;
extern T1716 GE_default1716;
extern T1718 GE_default1718;
extern T1721 GE_default1721;
extern T1722 GE_default1722;
extern T1724 GE_default1724;
extern T1726 GE_default1726;
extern T1728 GE_default1728;
extern T1730 GE_default1730;
extern T1731 GE_default1731;
extern T1734 GE_default1734;
extern T1735 GE_default1735;
extern T1748 GE_default1748;
extern T1749 GE_default1749;
extern T1750 GE_default1750;
extern T1753 GE_default1753;
extern T1754 GE_default1754;
extern T1755 GE_default1755;
extern T1756 GE_default1756;
extern T1757 GE_default1757;
extern T1758 GE_default1758;
extern T1759 GE_default1759;
extern T1762 GE_default1762;
extern T1763 GE_default1763;
extern T1764 GE_default1764;
extern T1765 GE_default1765;
extern T1768 GE_default1768;
extern T1771 GE_default1771;
extern T1772 GE_default1772;
extern T1773 GE_default1773;
extern T1776 GE_default1776;
extern T1777 GE_default1777;
extern T1778 GE_default1778;
extern T1779 GE_default1779;
extern T1780 GE_default1780;
extern T1783 GE_default1783;
extern T1784 GE_default1784;
extern T1785 GE_default1785;
extern T1786 GE_default1786;
extern T1787 GE_default1787;
extern T1788 GE_default1788;
extern T1789 GE_default1789;
extern T1792 GE_default1792;
extern T1793 GE_default1793;
extern T1794 GE_default1794;
extern T1795 GE_default1795;
extern T1796 GE_default1796;
extern T1797 GE_default1797;
extern T1801 GE_default1801;
extern T1802 GE_default1802;
extern T1803 GE_default1803;
extern T1804 GE_default1804;
extern T1805 GE_default1805;
extern T1806 GE_default1806;
extern T1807 GE_default1807;
extern T1808 GE_default1808;
extern T1809 GE_default1809;
extern T1812 GE_default1812;
extern T1814 GE_default1814;
extern T1816 GE_default1816;
extern T1817 GE_default1817;
extern T1818 GE_default1818;
extern T1820 GE_default1820;
extern T1822 GE_default1822;
extern T1825 GE_default1825;
extern T1826 GE_default1826;
extern T1828 GE_default1828;
extern T1829 GE_default1829;
extern T1834 GE_default1834;
extern T1835 GE_default1835;
extern T1839 GE_default1839;
extern T1840 GE_default1840;
extern T1841 GE_default1841;
extern T1842 GE_default1842;
extern T1844 GE_default1844;
extern T1846 GE_default1846;
extern T1848 GE_default1848;
extern T1849 GE_default1849;
extern T1850 GE_default1850;
extern T1851 GE_default1851;
extern T1852 GE_default1852;
extern T1854 GE_default1854;
extern T1855 GE_default1855;
extern T1856 GE_default1856;
extern T1857 GE_default1857;
extern T1858 GE_default1858;
extern T1859 GE_default1859;
extern T1861 GE_default1861;
extern T1862 GE_default1862;
extern T1863 GE_default1863;
extern T1864 GE_default1864;
extern T1865 GE_default1865;
extern T1866 GE_default1866;
extern T1867 GE_default1867;
extern T1868 GE_default1868;
extern T1869 GE_default1869;
extern T1870 GE_default1870;
extern T1871 GE_default1871;
extern T1872 GE_default1872;
extern T1874 GE_default1874;
extern T1876 GE_default1876;
extern T1878 GE_default1878;
extern T1880 GE_default1880;
extern T1882 GE_default1882;
extern T1886 GE_default1886;
extern T1887 GE_default1887;
extern T1892 GE_default1892;
extern T1894 GE_default1894;
extern T1895 GE_default1895;
extern T1896 GE_default1896;
extern T1897 GE_default1897;
extern T1903 GE_default1903;
extern T1904 GE_default1904;
extern T1905 GE_default1905;
extern T1909 GE_default1909;
extern T1910 GE_default1910;
extern T1912 GE_default1912;
extern T1913 GE_default1913;
extern T1914 GE_default1914;
extern T1915 GE_default1915;
extern T1917 GE_default1917;
extern T1918 GE_default1918;
extern T1919 GE_default1919;
extern T1920 GE_default1920;
extern T1921 GE_default1921;
extern T1923 GE_default1923;
extern T1924 GE_default1924;
extern T1925 GE_default1925;
extern T1926 GE_default1926;
extern T1930 GE_default1930;
extern T1931 GE_default1931;
extern T1932 GE_default1932;
extern T1933 GE_default1933;
extern T1934 GE_default1934;
extern T1935 GE_default1935;
extern T1936 GE_default1936;
extern T1937 GE_default1937;
extern T1938 GE_default1938;
extern T1939 GE_default1939;
extern T1941 GE_default1941;
extern T1943 GE_default1943;
extern T1944 GE_default1944;
extern T1946 GE_default1946;
extern T1947 GE_default1947;
extern T1949 GE_default1949;
extern T1950 GE_default1950;
extern T1951 GE_default1951;
extern T1956 GE_default1956;
extern T1959 GE_default1959;
extern T1961 GE_default1961;
extern T1963 GE_default1963;
extern T1964 GE_default1964;
extern T1965 GE_default1965;
extern T1968 GE_default1968;
extern T1970 GE_default1970;
extern T1972 GE_default1972;
extern T1978 GE_default1978;
extern T1979 GE_default1979;
extern T1980 GE_default1980;
extern T1982 GE_default1982;
extern T1986 GE_default1986;
extern T1988 GE_default1988;
extern T1989 GE_default1989;
extern T1990 GE_default1990;
extern T1991 GE_default1991;
extern T1992 GE_default1992;
extern T1993 GE_default1993;
extern T1994 GE_default1994;
extern T1998 GE_default1998;
extern T2000 GE_default2000;
extern T2001 GE_default2001;
extern T2002 GE_default2002;
extern T2003 GE_default2003;
extern T2004 GE_default2004;
extern T2009 GE_default2009;
extern T2010 GE_default2010;
extern T2012 GE_default2012;
extern T2014 GE_default2014;
extern T2015 GE_default2015;
extern T2022 GE_default2022;
extern T2027 GE_default2027;
extern T2032 GE_default2032;
extern T2034 GE_default2034;
extern T2037 GE_default2037;
extern T2040 GE_default2040;
extern T2042 GE_default2042;
extern T2044 GE_default2044;
extern T2045 GE_default2045;
extern T2046 GE_default2046;
extern T2047 GE_default2047;
extern T2048 GE_default2048;
extern T2049 GE_default2049;
extern T2050 GE_default2050;
extern T2051 GE_default2051;
extern T2052 GE_default2052;
extern T2053 GE_default2053;
extern T2054 GE_default2054;
extern T2055 GE_default2055;
extern T2056 GE_default2056;
extern T2057 GE_default2057;
extern T2058 GE_default2058;
extern T2059 GE_default2059;
extern T2060 GE_default2060;
extern T2061 GE_default2061;
extern T2062 GE_default2062;
extern T2063 GE_default2063;
extern T2064 GE_default2064;
extern T2065 GE_default2065;
extern T2066 GE_default2066;
extern T2068 GE_default2068;
extern T2069 GE_default2069;
extern T2070 GE_default2070;
extern T2071 GE_default2071;
extern T2072 GE_default2072;
extern T2073 GE_default2073;
extern T2074 GE_default2074;
extern T2075 GE_default2075;
extern T2076 GE_default2076;
extern T2077 GE_default2077;
extern T2079 GE_default2079;
extern T2080 GE_default2080;
extern T2081 GE_default2081;
extern T2082 GE_default2082;
extern T2083 GE_default2083;
extern T2084 GE_default2084;
extern T2085 GE_default2085;
extern T2086 GE_default2086;
extern T2087 GE_default2087;
extern T2089 GE_default2089;
extern T2091 GE_default2091;
extern T2092 GE_default2092;
extern T2093 GE_default2093;
extern T2095 GE_default2095;
extern T2097 GE_default2097;
extern T2098 GE_default2098;
extern T2100 GE_default2100;
extern T2101 GE_default2101;
extern T2102 GE_default2102;
extern T2103 GE_default2103;
extern T2104 GE_default2104;
extern T2106 GE_default2106;
extern T2107 GE_default2107;
extern T2108 GE_default2108;
extern T2109 GE_default2109;
extern T2110 GE_default2110;
extern T2111 GE_default2111;
extern T2112 GE_default2112;
extern T2113 GE_default2113;
extern T2114 GE_default2114;
extern T2115 GE_default2115;
extern T2116 GE_default2116;
extern T2118 GE_default2118;
extern T2119 GE_default2119;
extern T2120 GE_default2120;
extern T2121 GE_default2121;
extern T2122 GE_default2122;
extern T2123 GE_default2123;
extern T2124 GE_default2124;
extern T2125 GE_default2125;
extern T2126 GE_default2126;
extern T2127 GE_default2127;
extern T2128 GE_default2128;
extern T2130 GE_default2130;
extern T2132 GE_default2132;
extern T2133 GE_default2133;
extern T2134 GE_default2134;
extern T2135 GE_default2135;
extern T2136 GE_default2136;
extern T2137 GE_default2137;
extern T2138 GE_default2138;
extern T2139 GE_default2139;
extern T2140 GE_default2140;
extern T2141 GE_default2141;
extern T2142 GE_default2142;
extern T2144 GE_default2144;
extern T2145 GE_default2145;
extern T2146 GE_default2146;
extern T2147 GE_default2147;
extern T2148 GE_default2148;
extern T2149 GE_default2149;
extern T2150 GE_default2150;
extern T2151 GE_default2151;
extern T2152 GE_default2152;
extern T2153 GE_default2153;
extern T2154 GE_default2154;
extern T2155 GE_default2155;
extern T2156 GE_default2156;
extern T2157 GE_default2157;
extern T2158 GE_default2158;
extern T2159 GE_default2159;
extern T2160 GE_default2160;
extern T2161 GE_default2161;
extern T2163 GE_default2163;
extern T2164 GE_default2164;
extern T2166 GE_default2166;
extern T2169 GE_default2169;
extern T2191 GE_default2191;
extern T2196 GE_default2196;
extern T2197 GE_default2197;
extern T2199 GE_default2199;
extern T2200 GE_default2200;
extern T2202 GE_default2202;
extern T2203 GE_default2203;
extern T2204 GE_default2204;
extern T2205 GE_default2205;
extern T2207 GE_default2207;
extern T2208 GE_default2208;
extern T2209 GE_default2209;
extern T2210 GE_default2210;
extern T2211 GE_default2211;
extern T2212 GE_default2212;
extern T2213 GE_default2213;
extern T2214 GE_default2214;
extern T2215 GE_default2215;
extern T2216 GE_default2216;
extern T2217 GE_default2217;
extern T2218 GE_default2218;
extern T2219 GE_default2219;
extern T2221 GE_default2221;
extern T2222 GE_default2222;
extern T2223 GE_default2223;
extern T2224 GE_default2224;
extern T2225 GE_default2225;
extern T2226 GE_default2226;
extern T2227 GE_default2227;
extern T2228 GE_default2228;
extern T2233 GE_default2233;
extern T2234 GE_default2234;
extern T2235 GE_default2235;
extern T2236 GE_default2236;
extern T2237 GE_default2237;
extern T2238 GE_default2238;
extern T2239 GE_default2239;
extern T2240 GE_default2240;
extern T2241 GE_default2241;
extern T2243 GE_default2243;
extern T2249 GE_default2249;
extern T2250 GE_default2250;
extern T2251 GE_default2251;
extern T2252 GE_default2252;
extern T2253 GE_default2253;
extern T2254 GE_default2254;
extern T2256 GE_default2256;
extern T2257 GE_default2257;
extern T2258 GE_default2258;
extern T2260 GE_default2260;
extern T2261 GE_default2261;
extern T2262 GE_default2262;
extern T2266 GE_default2266;
extern T2267 GE_default2267;
extern T2268 GE_default2268;
extern T2269 GE_default2269;
extern T2270 GE_default2270;
extern T2271 GE_default2271;
extern T2272 GE_default2272;
extern T2273 GE_default2273;
extern T2274 GE_default2274;
extern T2276 GE_default2276;
extern T2277 GE_default2277;
extern T2278 GE_default2278;
extern T2279 GE_default2279;
extern T2280 GE_default2280;
extern T2282 GE_default2282;
extern T2283 GE_default2283;
extern T2285 GE_default2285;
extern T2287 GE_default2287;
extern T2288 GE_default2288;
extern T2289 GE_default2289;
extern T2293 GE_default2293;
extern T2294 GE_default2294;
extern T2295 GE_default2295;
extern T2296 GE_default2296;
extern T2298 GE_default2298;
extern T2299 GE_default2299;
extern T2300 GE_default2300;
extern T2302 GE_default2302;
extern T2303 GE_default2303;
extern T2304 GE_default2304;
extern T2305 GE_default2305;
extern T2306 GE_default2306;
extern T2308 GE_default2308;
extern T2309 GE_default2309;
extern T2317 GE_default2317;
extern T2318 GE_default2318;
extern T2319 GE_default2319;
extern T2320 GE_default2320;
extern T2322 GE_default2322;
extern T2329 GE_default2329;
extern T2330 GE_default2330;
extern T2331 GE_default2331;
extern T2334 GE_default2334;
extern T2336 GE_default2336;
extern T2337 GE_default2337;
extern T2339 GE_default2339;
extern T2340 GE_default2340;
extern T2341 GE_default2341;
extern T2343 GE_default2343;
extern T2344 GE_default2344;
extern T2345 GE_default2345;
extern T2346 GE_default2346;
extern T2347 GE_default2347;
extern T2348 GE_default2348;
extern T2349 GE_default2349;
extern T2352 GE_default2352;
extern T2357 GE_default2357;
extern T2358 GE_default2358;
extern T2359 GE_default2359;
extern T2361 GE_default2361;
extern T2362 GE_default2362;
extern T2363 GE_default2363;
extern T2364 GE_default2364;
extern T2365 GE_default2365;
extern T2368 GE_default2368;
extern T2369 GE_default2369;
extern T2370 GE_default2370;
extern T2371 GE_default2371;
extern T2372 GE_default2372;
extern T2373 GE_default2373;
extern T2374 GE_default2374;
extern T2377 GE_default2377;
extern T2378 GE_default2378;
extern T2379 GE_default2379;
extern T2380 GE_default2380;
extern T2382 GE_default2382;
extern T2383 GE_default2383;
extern T2384 GE_default2384;
extern T2385 GE_default2385;
extern T2388 GE_default2388;
extern T2389 GE_default2389;
extern T2395 GE_default2395;
extern T2397 GE_default2397;
extern T2398 GE_default2398;
extern T2400 GE_default2400;
extern T2402 GE_default2402;
extern T2407 GE_default2407;
extern T2410 GE_default2410;
extern T2418 GE_default2418;
extern T2419 GE_default2419;
extern T2420 GE_default2420;
extern T2421 GE_default2421;
extern T2422 GE_default2422;
extern T2423 GE_default2423;
extern T2424 GE_default2424;
extern T2425 GE_default2425;
extern T2426 GE_default2426;
extern T2427 GE_default2427;
extern T2428 GE_default2428;
extern T2429 GE_default2429;
extern T2430 GE_default2430;
extern T2431 GE_default2431;
extern T2432 GE_default2432;
extern T2433 GE_default2433;
extern T2444 GE_default2444;
extern T2446 GE_default2446;
extern T2447 GE_default2447;
extern T2448 GE_default2448;
extern T2449 GE_default2449;
extern T2450 GE_default2450;
extern T2452 GE_default2452;
extern T2457 GE_default2457;
extern T2459 GE_default2459;
extern T2461 GE_default2461;
extern T2463 GE_default2463;
extern T2464 GE_default2464;
extern T2465 GE_default2465;
extern T2466 GE_default2466;
extern T2467 GE_default2467;
extern T2469 GE_default2469;
extern T2470 GE_default2470;
extern T2471 GE_default2471;
extern T2472 GE_default2472;
extern T2475 GE_default2475;
extern T2476 GE_default2476;
extern T2477 GE_default2477;
extern T2478 GE_default2478;
extern T2479 GE_default2479;
extern T2480 GE_default2480;
extern T2482 GE_default2482;
extern T2483 GE_default2483;
extern T2485 GE_default2485;
extern T2486 GE_default2486;
extern T2487 GE_default2487;
extern T2489 GE_default2489;
extern T2493 GE_default2493;
extern T2495 GE_default2495;
extern T2497 GE_default2497;
extern T2499 GE_default2499;
extern T2501 GE_default2501;
extern T2502 GE_default2502;
extern T2503 GE_default2503;
extern T2504 GE_default2504;
extern T2505 GE_default2505;
extern T2506 GE_default2506;
extern T2507 GE_default2507;
extern T2508 GE_default2508;
extern T2509 GE_default2509;
extern T2510 GE_default2510;
extern T2512 GE_default2512;
extern T2513 GE_default2513;
extern T2514 GE_default2514;
extern T2516 GE_default2516;
extern T2517 GE_default2517;
extern T2518 GE_default2518;
extern T2520 GE_default2520;
extern T2521 GE_default2521;
extern T2522 GE_default2522;
extern T2524 GE_default2524;
extern T2525 GE_default2525;
extern T2526 GE_default2526;
extern T2528 GE_default2528;
extern T2529 GE_default2529;
extern T2530 GE_default2530;
extern T2532 GE_default2532;
extern T2533 GE_default2533;
extern T2534 GE_default2534;
extern T2536 GE_default2536;
extern T2537 GE_default2537;
extern T2538 GE_default2538;
extern T2539 GE_default2539;
extern T2540 GE_default2540;
extern T2541 GE_default2541;
extern T2542 GE_default2542;
extern T2543 GE_default2543;
extern T2544 GE_default2544;
extern T2545 GE_default2545;
extern T2546 GE_default2546;
extern T2547 GE_default2547;
extern T2549 GE_default2549;
extern T2551 GE_default2551;
extern T2555 GE_default2555;
extern T2558 GE_default2558;
extern T2559 GE_default2559;
extern T2563 GE_default2563;
extern T2564 GE_default2564;
extern T2565 GE_default2565;
extern T2566 GE_default2566;
extern T2567 GE_default2567;
extern T2568 GE_default2568;
extern T2577 GE_default2577;
extern T2578 GE_default2578;
extern T2580 GE_default2580;
extern T2581 GE_default2581;
extern T2583 GE_default2583;
extern T2594 GE_default2594;
extern T2595 GE_default2595;
extern T2596 GE_default2596;
extern T2597 GE_default2597;
extern T2615 GE_default2615;
extern T2621 GE_default2621;
extern T2622 GE_default2622;
extern T2646 GE_default2646;
extern T2650 GE_default2650;
extern T2651 GE_default2651;
extern T2653 GE_default2653;
extern T2654 GE_default2654;
extern T2655 GE_default2655;
extern T2657 GE_default2657;
extern T2661 GE_default2661;
extern T2662 GE_default2662;
extern T2664 GE_default2664;
extern T2665 GE_default2665;
extern T2666 GE_default2666;
extern T2670 GE_default2670;
extern T2675 GE_default2675;
extern T2676 GE_default2676;
extern T2684 GE_default2684;
extern T2685 GE_default2685;
extern T2686 GE_default2686;
extern T2697 GE_default2697;
extern T2702 GE_default2702;
extern T2704 GE_default2704;
extern T2705 GE_default2705;
extern T2706 GE_default2706;
extern T2707 GE_default2707;
extern T2708 GE_default2708;
extern T2709 GE_default2709;
extern T2712 GE_default2712;
extern T2718 GE_default2718;
extern T2719 GE_default2719;
extern T2720 GE_default2720;
extern T2721 GE_default2721;
extern T2722 GE_default2722;
extern T2723 GE_default2723;
extern T2724 GE_default2724;
extern T2725 GE_default2725;
extern T2726 GE_default2726;
extern T2727 GE_default2727;
extern T2728 GE_default2728;
extern T2729 GE_default2729;
extern T2730 GE_default2730;
extern T2731 GE_default2731;
extern T2732 GE_default2732;
extern T2735 GE_default2735;
extern T2736 GE_default2736;
extern T2737 GE_default2737;
extern T2738 GE_default2738;
extern T2739 GE_default2739;
extern T2740 GE_default2740;
extern T2741 GE_default2741;
extern T2742 GE_default2742;
extern T2743 GE_default2743;
extern T2744 GE_default2744;
extern T2745 GE_default2745;
extern T2746 GE_default2746;
extern T2747 GE_default2747;
extern T2748 GE_default2748;
extern T2749 GE_default2749;
extern T2750 GE_default2750;
extern T2752 GE_default2752;
extern T2753 GE_default2753;
extern T2754 GE_default2754;
extern T2755 GE_default2755;
extern T2756 GE_default2756;
extern T2757 GE_default2757;
extern T2759 GE_default2759;
extern T2760 GE_default2760;
extern T2761 GE_default2761;
extern T2763 GE_default2763;
extern T2771 GE_default2771;
extern T2776 GE_default2776;
extern T2777 GE_default2777;
extern T2778 GE_default2778;
extern T2779 GE_default2779;
extern T2780 GE_default2780;
extern T2781 GE_default2781;
extern T2782 GE_default2782;
extern T2783 GE_default2783;
extern T2788 GE_default2788;
extern T2790 GE_default2790;
extern T2793 GE_default2793;
extern T2795 GE_default2795;
extern T2796 GE_default2796;
extern T2797 GE_default2797;
extern T2798 GE_default2798;
extern T2800 GE_default2800;
extern T2802 GE_default2802;
extern T2804 GE_default2804;
extern T2807 GE_default2807;
extern T2808 GE_default2808;
extern T2811 GE_default2811;
extern T2812 GE_default2812;
extern T2813 GE_default2813;
extern T2814 GE_default2814;
extern T2815 GE_default2815;
extern T2816 GE_default2816;
extern T2817 GE_default2817;
extern T2818 GE_default2818;
extern T2820 GE_default2820;
extern T2821 GE_default2821;
extern T2823 GE_default2823;
extern T2824 GE_default2824;
extern T2825 GE_default2825;
extern T2826 GE_default2826;
extern T2827 GE_default2827;
extern T2828 GE_default2828;
extern T2830 GE_default2830;
extern T2831 GE_default2831;
extern T2836 GE_default2836;
extern T2838 GE_default2838;
extern T2841 GE_default2841;
extern T2844 GE_default2844;
extern T2845 GE_default2845;
extern T2846 GE_default2846;
extern T2847 GE_default2847;
extern T2848 GE_default2848;

extern T0* GE_new_str8(T6 c);
extern T0* GE_new_str32(T6 c);
extern T0* GE_new_istr32(T6 c);
/* Call to ANY.same_type */
extern T1 T23x49156T0(GE_context* ac, T0* C, T0* a1);
/* Call to KI_CHARACTER_INPUT_STREAM.name */
extern T0* T39x5849095(GE_context* ac, T0* C);
/* Call to KI_CHARACTER_INPUT_STREAM.read_to_string */
extern T6 T39x5734401T0T6T6(GE_context* ac, T0* C, T0* a1, T6 a2, T6 a3);
/* Call to KI_CHARACTER_INPUT_STREAM.is_closable */
extern T1 T39x5849090(GE_context* ac, T0* C);
/* Call to KI_CHARACTER_INPUT_STREAM.end_of_input */
extern T1 T39x5849093(GE_context* ac, T0* C);
/* Call to KI_CHARACTER_INPUT_STREAM.last_character */
extern T2 T39x5849096(GE_context* ac, T0* C);
/* Call to ET_SYSTEM.is_equal_seed */
extern T6 T40x20037639(GE_context* ac, T0* C);
/* Call to ET_SYSTEM.attachment_type_conformance_mode */
extern T1 T40x20037651(GE_context* ac, T0* C);
/* Call to ET_SYSTEM.default_create_seed */
extern T6 T40x20037637(GE_context* ac, T0* C);
/* Call to ET_SYSTEM.detachable_any_type */
extern T0* T40x20070432(GE_context* ac, T0* C);
/* Call to ET_SYSTEM.is_dotnet */
extern T1 T40x20037633(GE_context* ac, T0* C);
/* Call to ET_SYSTEM.target_type_attachment_mode */
extern T1 T40x20037652(GE_context* ac, T0* C);
/* Call to ET_SYSTEM.any_type */
extern T0* T40x20070431(GE_context* ac, T0* C);
/* Call to ET_SYSTEM.array_like_current_type */
extern T0* T40x20070441(GE_context* ac, T0* C);
/* Call to ET_SYSTEM.none_type */
extern T0* T40x20070464(GE_context* ac, T0* C);
/* Call to ET_SYSTEM.array_any_type */
extern T0* T40x20070438(GE_context* ac, T0* C);
/* Call to ET_SYSTEM.array_none_type */
extern T0* T40x20070440(GE_context* ac, T0* C);
/* Call to ET_SYSTEM.array_detachable_any_type */
extern T0* T40x20070439(GE_context* ac, T0* C);
/* Call to ET_SYSTEM.detachable_none_type */
extern T0* T40x20070465(GE_context* ac, T0* C);
/* Call to ET_SYSTEM.iteration_cursor_forth_seed */
extern T6 T40x20037645(GE_context* ac, T0* C);
/* Call to ET_SYSTEM.iteration_cursor_after_seed */
extern T6 T40x20037644(GE_context* ac, T0* C);
/* Call to ET_SYSTEM.iterable_new_cursor_seed */
extern T6 T40x20037643(GE_context* ac, T0* C);
/* Call to ET_SYSTEM.routine_call_seed */
extern T6 T40x20037641(GE_context* ac, T0* C);
/* Call to ET_SYSTEM.character_32_type */
extern T0* T40x20070445(GE_context* ac, T0* C);
/* Call to ET_SYSTEM.any_clients */
extern T0* T40x20070437(GE_context* ac, T0* C);
/* Call to ET_SYSTEM.copy_seed */
extern T6 T40x20037638(GE_context* ac, T0* C);
/* Call to ET_SYSTEM.real_32_type */
extern T0* T40x20070470(GE_context* ac, T0* C);
/* Call to ET_SYSTEM.string_32_type */
extern T0* T40x20070478(GE_context* ac, T0* C);
/* Call to ET_SYSTEM.string_8_type */
extern T0* T40x20070477(GE_context* ac, T0* C);
/* Call to ET_SYSTEM.function_item_seed */
extern T6 T40x20037642(GE_context* ac, T0* C);
/* Call to ET_SYSTEM.any_parent */
extern T0* T40x20070435(GE_context* ac, T0* C);
/* Call to ET_SYSTEM.alias_transition_mode */
extern T1 T40x20037655(GE_context* ac, T0* C);
/* Call to ET_SYSTEM.scoop_mode */
extern T1 T40x20037650(GE_context* ac, T0* C);
/* Call to ET_SYSTEM.system_object_parents */
extern T0* T40x20070480(GE_context* ac, T0* C);
/* Call to ET_SYSTEM.any_parents */
extern T0* T40x20070436(GE_context* ac, T0* C);
/* Call to ET_SYSTEM.type_any_type */
extern T0* T40x20070486(GE_context* ac, T0* C);
/* Call to ET_SYSTEM.dispose_seed */
extern T6 T40x20037640(GE_context* ac, T0* C);
/* Call to ET_SYSTEM.console_application_mode */
extern T1 T40x20037648(GE_context* ac, T0* C);
/* Call to ET_SYSTEM.multithreaded_mode */
extern T1 T40x20037649(GE_context* ac, T0* C);
/* Call to ET_SYSTEM.exception_trace_mode */
extern T1 T40x20037653(GE_context* ac, T0* C);
/* Call to ET_SYSTEM.trace_mode */
extern T1 T40x20037654(GE_context* ac, T0* C);
/* Call to ET_SYSTEM.class_count_recursive */
extern T6 T40x20070428(GE_context* ac, T0* C);
/* Call to ET_SYSTEM.root_type */
extern T0* T40x20037634(GE_context* ac, T0* C);
/* Call to ET_SYSTEM.external_resource_pathnames */
extern T0* T40x20037662(GE_context* ac, T0* C);
/* Call to ET_SYSTEM.external_make_pathnames */
extern T0* T40x20037661(GE_context* ac, T0* C);
/* Call to ET_SYSTEM.external_object_pathnames */
extern T0* T40x20037659(GE_context* ac, T0* C);
/* Call to ET_SYSTEM.external_library_pathnames */
extern T0* T40x20037660(GE_context* ac, T0* C);
/* Call to ET_SYSTEM.external_linker_flags */
extern T0* T40x20037664(GE_context* ac, T0* C);
/* Call to ET_SYSTEM.external_cflags */
extern T0* T40x20037663(GE_context* ac, T0* C);
/* Call to ET_SYSTEM.external_include_pathnames */
extern T0* T40x20037658(GE_context* ac, T0* C);
/* Call to ET_SYSTEM.use_boehm_gc */
extern T1 T40x20037656(GE_context* ac, T0* C);
/* Call to ET_SYSTEM.universe */
extern T0* T40x19709958(GE_context* ac, T0* C);
/* Call to ET_SYSTEM.ise_exception_manager_type */
extern T0* T40x20070457(GE_context* ac, T0* C);
/* Call to ET_SYSTEM.pointer_type */
extern T0* T40x20070466(GE_context* ac, T0* C);
/* Call to ET_SYSTEM.detachable_tuple_type */
extern T0* T40x20070483(GE_context* ac, T0* C);
/* Call to ET_SYSTEM.routine_type */
extern T0* T40x20070472(GE_context* ac, T0* C);
/* Call to ET_SYSTEM.typed_pointer_any_type */
extern T0* T40x20070494(GE_context* ac, T0* C);
/* Call to ET_SYSTEM.special_any_type */
extern T0* T40x20070473(GE_context* ac, T0* C);
/* Call to ET_SYSTEM.immutable_string_32_type */
extern T0* T40x20070451(GE_context* ac, T0* C);
/* Call to ET_SYSTEM.character_8_type */
extern T0* T40x20070444(GE_context* ac, T0* C);
/* Call to ET_SYSTEM.real_64_type */
extern T0* T40x20070471(GE_context* ac, T0* C);
/* Call to ET_SYSTEM.natural_64_type */
extern T0* T40x20070463(GE_context* ac, T0* C);
/* Call to ET_SYSTEM.natural_32_type */
extern T0* T40x20070462(GE_context* ac, T0* C);
/* Call to ET_SYSTEM.natural_16_type */
extern T0* T40x20070461(GE_context* ac, T0* C);
/* Call to ET_SYSTEM.natural_8_type */
extern T0* T40x20070460(GE_context* ac, T0* C);
/* Call to ET_SYSTEM.integer_64_type */
extern T0* T40x20070456(GE_context* ac, T0* C);
/* Call to ET_SYSTEM.integer_32_type */
extern T0* T40x20070455(GE_context* ac, T0* C);
/* Call to ET_SYSTEM.integer_16_type */
extern T0* T40x20070454(GE_context* ac, T0* C);
/* Call to ET_SYSTEM.integer_8_type */
extern T0* T40x20070453(GE_context* ac, T0* C);
/* Call to ET_SYSTEM.boolean_type */
extern T0* T40x20070442(GE_context* ac, T0* C);
/* Call to ET_SYSTEM.root_creation */
extern T0* T40x20037635(GE_context* ac, T0* C);
/* Call to ET_SYSTEM.master_classes */
extern T0* T40x20070406(GE_context* ac, T0* C);
/* Call to ET_SYSTEM.system_name */
extern T0* T40x20037657(GE_context* ac, T0* C);
/* Call to AP_OPTION.short_form */
extern T2 T51x5275656(GE_context* ac, T0* C);
/* Call to AP_OPTION.long_form */
extern T0* T51x5275651(GE_context* ac, T0* C);
/* Call to AP_OPTION.description */
extern T0* T51x5275649(GE_context* ac, T0* C);
/* Call to AP_OPTION.names */
extern T0* T51x5275653(GE_context* ac, T0* C);
/* Call to AP_OPTION.is_hidden */
extern T1 T51x5275660(GE_context* ac, T0* C);
/* Call to AP_OPTION.name */
extern T0* T51x5275652(GE_context* ac, T0* C);
/* Call to AP_OPTION.has_short_form */
extern T1 T51x5275659(GE_context* ac, T0* C);
/* Call to AP_OPTION.maximum_occurrences */
extern T6 T51x5275655(GE_context* ac, T0* C);
/* Call to AP_OPTION.occurrences */
extern T6 T51x5275654(GE_context* ac, T0* C);
/* Call to AP_OPTION.was_found */
extern T1 T51x5275664(GE_context* ac, T0* C);
/* Call to AP_OPTION.is_mandatory */
extern T1 T51x5275661(GE_context* ac, T0* C);
/* Call to AP_OPTION.example */
extern T0* T51x5275650(GE_context* ac, T0* C);
/* Call to ET_BASE_TYPE.conforms_to_type */
extern T1 T83x23003194T0T0T0T0(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3, T0* a4);
/* Call to ET_BASE_TYPE.shallow_named_type_with_type_mark */
extern T0* T83x23003151T0T0(GE_context* ac, T0* C, T0* a1, T0* a2);
/* Call to ET_BASE_TYPE.same_base_type_with_type_marks */
extern T1 T83x23003181T0T0T0T0T0(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3, T0* a4, T0* a5);
/* Call to ET_BASE_TYPE.base_class */
extern T0* T83x23035907(GE_context* ac, T0* C);
/* Call to ET_BASE_TYPE.type_with_type_mark */
extern T0* T83x23003154T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_BASE_TYPE.actual_parameters */
extern T0* T83x22462468(GE_context* ac, T0* C);
/* Call to ET_BASE_TYPE.base_type_index_of_label */
extern T6 T83x23003147T0T0(GE_context* ac, T0* C, T0* a1, T0* a2);
/* Call to ET_BASE_TYPE.base_type_actual */
extern T0* T83x23003145T6T0(GE_context* ac, T0* C, T6 a1, T0* a2);
/* Call to ET_BASE_TYPE.is_type_attached_with_type_mark */
extern T1 T83x23003165T0T0(GE_context* ac, T0* C, T0* a1, T0* a2);
/* Call to ET_BASE_TYPE.is_type_reference_with_type_mark */
extern T1 T83x23003163T0T0(GE_context* ac, T0* C, T0* a1, T0* a2);
/* Call to ET_BASE_TYPE.base_type_with_type_mark */
extern T0* T83x23003142T0T0(GE_context* ac, T0* C, T0* a1, T0* a2);
/* Call to ET_BASE_TYPE.is_type_detachable_with_type_mark */
extern T1 T83x23003167T0T0(GE_context* ac, T0* C, T0* a1, T0* a2);
/* Call to ET_BASE_TYPE.same_base_tuple_type_with_type_marks */
extern T1 T83x23003193T0T0T0T0T0(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3, T0* a4, T0* a5);
/* Call to ET_BASE_TYPE.same_base_class_type_with_type_marks */
extern T1 T83x23003191T0T0T0T0T0(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3, T0* a4, T0* a5);
/* Call to ET_BASE_TYPE.actual_parameter_count */
extern T6 T83x22462476(GE_context* ac, T0* C);
/* Call to ET_BASE_TYPE.named_base_class */
extern T0* T83x23035908(GE_context* ac, T0* C);
/* Call to ET_BASE_TYPE.index_of_label */
extern T6 T83x22462473T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_BASE_TYPE.shallow_base_type_with_type_mark */
extern T0* T83x23003144T0T0(GE_context* ac, T0* C, T0* a1, T0* a2);
/* Call to ET_BASE_TYPE.to_text */
extern T0* T83x23003206(GE_context* ac, T0* C);
/* Call to ET_BASE_TYPE.unaliased_to_text */
extern T0* T83x23003207(GE_context* ac, T0* C);
/* Call to ET_BASE_TYPE.is_type_expanded_with_type_mark */
extern T1 T83x23003161T0T0(GE_context* ac, T0* C, T0* a1, T0* a2);
/* Call to ET_BASE_TYPE.same_named_formal_parameter_type_with_type_marks */
extern T1 T83x23003189T0T0T0T0T0(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3, T0* a4, T0* a5);
/* Call to ET_BASE_TYPE.conforms_from_formal_parameter_type_with_type_marks */
extern T1 T83x23003197T0T0T0T0T0T0(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3, T0* a4, T0* a5, T0* a6);
/* Call to ET_BASE_TYPE.conforms_from_tuple_type_with_type_marks */
extern T1 T83x23003198T0T0T0T0T0T0(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3, T0* a4, T0* a5, T0* a6);
/* Call to ET_BASE_TYPE.conforms_from_class_type_with_type_marks */
extern T1 T83x23003196T0T0T0T0T0T0(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3, T0* a4, T0* a5, T0* a6);
/* Call to ET_BASE_TYPE.base_type_actual_count */
extern T6 T83x23003156T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_BASE_TYPE.conforms_to_type_with_type_marks */
extern T1 T83x23003195T0T0T0T0T0T0(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3, T0* a4, T0* a5, T0* a6);
/* Call to ET_BASE_TYPE.same_named_tuple_type_with_type_marks */
extern T1 T83x23003190T0T0T0T0T0(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3, T0* a4, T0* a5);
/* Call to ET_BASE_TYPE.same_named_class_type_with_type_marks */
extern T1 T83x23003188T0T0T0T0T0(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3, T0* a4, T0* a5);
/* Call to ET_BASE_TYPE.named_type_with_type_mark */
extern T0* T83x23003149T0T0(GE_context* ac, T0* C, T0* a1, T0* a2);
/* Call to ET_BASE_TYPE.same_named_type_with_type_marks */
extern T1 T83x23003179T0T0T0T0T0(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3, T0* a4, T0* a5);
/* Call to ET_BASE_TYPE.is_expanded */
extern T1 T83x22462480(GE_context* ac, T0* C);
/* Call to ET_BASE_TYPE.same_as_base_class */
extern T1 T83x22462485(GE_context* ac, T0* C);
/* Call to ET_BASE_TYPE.same_named_type */
extern T1 T83x23003178T0T0T0(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* Call to READABLE_STRING_GENERAL.code */
extern T10 T99x2031617T6(GE_context* ac, T0* C, T6 a1);
/* Call to READABLE_STRING_GENERAL.count */
extern T6 T99x2031655(GE_context* ac, T0* C);
/* Call to READABLE_STRING_GENERAL.substring */
extern T0* T99x2031696T6T6(GE_context* ac, T0* C, T6 a1, T6 a2);
/* Call to READABLE_STRING_GENERAL.item */
extern T3 T99x2031618T6(GE_context* ac, T0* C, T6 a1);
/* Call to UT_ERROR.default_message */
extern T0* T101x11436035(GE_context* ac, T0* C);
/* Call to detachable AP_OPTION.needs_parameter */
extern T1 T174x5275662(GE_context* ac, T0* C);
/* Call to detachable AP_OPTION.allows_parameter */
extern T1 T174x5275657(GE_context* ac, T0* C);
/* Call to attached KL_EQUALITY_TESTER [[attached] STRING_8].test */
extern T1 T206x7258113T0T0(GE_context* ac, T0* C, T0* a1, T0* a2);
/* Call to ET_ECF_SYSTEM_CONFIG.filename */
extern T0* T233x15106051(GE_context* ac, T0* C);
/* Call to ET_ECF_SYSTEM_CONFIG.ecf_version */
extern T0* T233x15106054(GE_context* ac, T0* C);
/* Call to ET_ECF_SYSTEM_CONFIG.universe */
extern T0* T233x15106060(GE_context* ac, T0* C);
/* Call to ET_ECF_SYSTEM_CONFIG.library_target */
extern T0* T233x15106056(GE_context* ac, T0* C);
/* Call to ET_ECF_SYSTEM_CONFIG.targets */
extern T0* T233x15106055(GE_context* ac, T0* C);
/* Call to ET_POSITION.column */
extern T6 T246x21184514(GE_context* ac, T0* C);
/* Call to ET_POSITION.line */
extern T6 T246x21184513(GE_context* ac, T0* C);
/* Call to ET_POSITION.to_text */
extern T0* T246x21184519(GE_context* ac, T0* C);
/* Call to ET_POSITION.is_null */
extern T1 T246x21184515(GE_context* ac, T0* C);
/* Call to ET_ADAPTED_DOTNET_ASSEMBLY.dotnet_assembly */
extern T0* T266x19709958(GE_context* ac, T0* C);
/* Call to ET_DYNAMIC_TYPE.conforms_to_type */
extern T1 T320x24641549T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_DYNAMIC_TYPE.seeded_dynamic_query */
extern T0* T320x24641570T6T0(GE_context* ac, T0* C, T6 a1, T0* a2);
/* Call to ET_DYNAMIC_TYPE.is_basic */
extern T1 T320x24641545(GE_context* ac, T0* C);
/* Call to ET_DYNAMIC_TYPE.is_expanded */
extern T1 T320x24739841(GE_context* ac, T0* C);
/* Call to ET_DYNAMIC_TYPE.base_type */
extern T0* T320x24641550(GE_context* ac, T0* C);
/* Call to ET_DYNAMIC_TYPE.base_class */
extern T0* T320x24641551(GE_context* ac, T0* C);
/* Call to ET_DYNAMIC_TYPE.is_agent_type */
extern T1 T320x24641542(GE_context* ac, T0* C);
/* Call to ET_DYNAMIC_TYPE.seeded_dynamic_procedure */
extern T0* T320x24641571T6T0(GE_context* ac, T0* C, T6 a1, T0* a2);
/* Call to ET_DYNAMIC_TYPE.dynamic_query */
extern T0* T320x24641568T0T0(GE_context* ac, T0* C, T0* a1, T0* a2);
/* Call to ET_DYNAMIC_TYPE.dynamic_procedure */
extern T0* T320x24641569T0T0(GE_context* ac, T0* C, T0* a1, T0* a2);
/* Call to ET_DYNAMIC_TYPE.queries */
extern T0* T320x24641566(GE_context* ac, T0* C);
/* Call to ET_DYNAMIC_TYPE.attribute_count */
extern T6 T320x24641562(GE_context* ac, T0* C);
/* Call to ET_DYNAMIC_TYPE.id */
extern T6 T320x24641560(GE_context* ac, T0* C);
/* Call to ET_DYNAMIC_TYPE.is_special */
extern T1 T320x24641543(GE_context* ac, T0* C);
/* Call to ET_DYNAMIC_TYPE.is_builtin_attribute */
extern T1 T320x24641572T0T8T8(GE_context* ac, T0* C, T0* a1, T8 a2, T8 a3);
/* Call to ET_DYNAMIC_TYPE.meta_type */
extern T0* T320x24641552(GE_context* ac, T0* C);
/* Call to ET_DYNAMIC_TYPE.has_nested_reference_attributes */
extern T1 T320x24641564(GE_context* ac, T0* C);
/* Call to ET_DYNAMIC_TYPE.hash_code */
extern T6 T320x458753(GE_context* ac, T0* C);
/* Call to ET_DYNAMIC_TYPE.is_generic */
extern T1 T320x24641541(GE_context* ac, T0* C);
/* Call to ET_DYNAMIC_TYPE.is_alive */
extern T1 T320x24641537(GE_context* ac, T0* C);
/* Call to ET_DYNAMIC_TYPE.next_type */
extern T0* T320x24641578(GE_context* ac, T0* C);
/* Call to ET_DYNAMIC_TYPE.procedure_calls */
extern T0* T320x24641574(GE_context* ac, T0* C);
/* Call to ET_DYNAMIC_TYPE.query_calls */
extern T0* T320x24641573(GE_context* ac, T0* C);
/* Call to ET_DYNAMIC_TYPE.procedures */
extern T0* T320x24641567(GE_context* ac, T0* C);
/* Call to ET_DYNAMIC_TYPE.static_type */
extern T0* T320x24739845(GE_context* ac, T0* C);
/* Call to ET_DYNAMIC_TYPE.conforming_ancestors */
extern T0* T320x24641556(GE_context* ac, T0* C);
/* Call to ET_DYNAMIC_TYPE.is_tuple */
extern T1 T320x24641544(GE_context* ac, T0* C);
/* Call to ET_DYNAMIC_TYPE.has_static */
extern T1 T320x24641546(GE_context* ac, T0* C);
/* Call to detachable ET_DYNAMIC_TYPE.is_basic */
extern T1 T323x24641545(GE_context* ac, T0* C);
/* Call to detachable ET_DYNAMIC_TYPE.is_expanded */
extern T1 T323x24739841(GE_context* ac, T0* C);
/* Call to detachable ET_DYNAMIC_TYPE.has_nested_reference_attributes */
extern T1 T323x24641564(GE_context* ac, T0* C);
/* Call to detachable ET_DYNAMIC_TYPE.base_type */
extern T0* T323x24641550(GE_context* ac, T0* C);
/* Call to detachable ET_DYNAMIC_TYPE.base_class */
extern T0* T323x24641551(GE_context* ac, T0* C);
/* Call to detachable ET_DYNAMIC_TYPE.id */
extern T6 T323x24641560(GE_context* ac, T0* C);
/* Call to detachable ET_DYNAMIC_TYPE.dynamic_procedure */
extern T0* T323x24641569T0T0(GE_context* ac, T0* C, T0* a1, T0* a2);
/* Call to detachable ET_FEATURE_NAME.lower_name */
extern T0* T324x21544962(GE_context* ac, T0* C);
/* Call to detachable ET_PROCEDURE.is_creation_exported_to */
extern T1 T325x18841656T0T0T0(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* Call to detachable ET_PROCEDURE.first_seed */
extern T6 T325x18923531(GE_context* ac, T0* C);
/* Call to detachable ET_PROCEDURE.arguments_count */
extern T6 T325x18841653(GE_context* ac, T0* C);
/* Call to detachable ET_PROCEDURE.arguments */
extern T0* T325x18923530(GE_context* ac, T0* C);
/* Call to detachable ET_PROCEDURE.has_seed */
extern T1 T325x18923526T6(GE_context* ac, T0* C, T6 a1);
/* Call to detachable ET_PROCEDURE.new_synonym */
extern T0* T325x18841660T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable ET_PROCEDURE.name */
extern T0* T325x18923528(GE_context* ac, T0* C);
/* Call to detachable ET_QUERY.type */
extern T0* T326x18923529(GE_context* ac, T0* C);
/* Call to detachable ET_QUERY.assigner */
extern T0* T326x19349507(GE_context* ac, T0* C);
/* Call to detachable ET_QUERY.is_exported_to */
extern T1 T326x18841654T0T0(GE_context* ac, T0* C, T0* a1, T0* a2);
/* Call to detachable ET_QUERY.first_seed */
extern T6 T326x18923531(GE_context* ac, T0* C);
/* Call to detachable ET_QUERY.is_attribute */
extern T1 T326x18841631(GE_context* ac, T0* C);
/* Call to detachable ET_QUERY.arguments */
extern T0* T326x18923530(GE_context* ac, T0* C);
/* Call to detachable ET_QUERY.has_seed */
extern T1 T326x18923526T6(GE_context* ac, T0* C, T6 a1);
/* Call to detachable ET_QUERY.is_unique_attribute */
extern T1 T326x18841633(GE_context* ac, T0* C);
/* Call to detachable ET_QUERY.is_constant_attribute */
extern T1 T326x18841632(GE_context* ac, T0* C);
/* Call to detachable ET_QUERY.new_synonym */
extern T0* T326x18841660T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_FEATURE_NAME.last_leaf */
extern T0* T328x20955141(GE_context* ac, T0* C);
/* Call to ET_FEATURE_NAME.is_tuple_label */
extern T1 T328x21544969(GE_context* ac, T0* C);
/* Call to ET_FEATURE_NAME.name */
extern T0* T328x21544961(GE_context* ac, T0* C);
/* Call to ET_FEATURE_NAME.lower_name */
extern T0* T328x21544962(GE_context* ac, T0* C);
/* Call to ET_FEATURE_NAME.position */
extern T0* T328x20955137(GE_context* ac, T0* C);
/* Call to ET_FEATURE_NAME.seed */
extern T6 T328x21544963(GE_context* ac, T0* C);
/* Call to ET_FEATURE_NAME.first_leaf */
extern T0* T328x20955140(GE_context* ac, T0* C);
/* Call to ET_FEATURE_NAME.across_cursor_name */
extern T0* T328x21545005(GE_context* ac, T0* C);
/* Call to ET_FEATURE_NAME.is_across_cursor */
extern T1 T328x21544967(GE_context* ac, T0* C);
/* Call to ET_FEATURE_NAME.object_test_local_name */
extern T0* T328x21545004(GE_context* ac, T0* C);
/* Call to ET_FEATURE_NAME.is_object_test_local */
extern T1 T328x21544966(GE_context* ac, T0* C);
/* Call to ET_FEATURE_NAME.local_name */
extern T0* T328x21545002(GE_context* ac, T0* C);
/* Call to ET_FEATURE_NAME.is_local */
extern T1 T328x21544965(GE_context* ac, T0* C);
/* Call to ET_FEATURE_NAME.argument_name */
extern T0* T328x21545003(GE_context* ac, T0* C);
/* Call to ET_FEATURE_NAME.is_argument */
extern T1 T328x21544968(GE_context* ac, T0* C);
/* Call to ET_FEATURE_NAME.same_feature_name */
extern T1 T328x21676035T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_FEATURE_NAME.hash_code */
extern T6 T328x458753(GE_context* ac, T0* C);
/* Call to ET_FEATURE_NAME.is_prefix */
extern T1 T328x21544991(GE_context* ac, T0* C);
/* Call to ET_FEATURE_NAME.is_infix */
extern T1 T328x21544971(GE_context* ac, T0* C);
/* Call to ET_TYPE.last_leaf */
extern T0* T330x20955141(GE_context* ac, T0* C);
/* Call to ET_TYPE.position */
extern T0* T330x20955137(GE_context* ac, T0* C);
/* Call to ET_TYPE.base_class */
extern T0* T330x23003139T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_TYPE.has_identifier_anchored_type */
extern T1 T330x23003169(GE_context* ac, T0* C);
/* Call to ET_TYPE.depends_on_qualified_anchored_type */
extern T1 T330x23003170T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_TYPE.conforms_to_type */
extern T1 T330x23003194T0T0T0T0(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3, T0* a4);
/* Call to ET_TYPE.named_type */
extern T0* T330x23003148T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_TYPE.same_syntactical_type */
extern T1 T330x23003176T0T0T0(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* Call to ET_TYPE.resolved_formal_parameters */
extern T0* T330x23003199T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_TYPE.has_formal_types */
extern T1 T330x23003171T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_TYPE.has_anchored_type */
extern T1 T330x23003168(GE_context* ac, T0* C);
/* Call to ET_TYPE.same_named_type */
extern T1 T330x23003178T0T0T0(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* Call to ET_TYPE.same_named_type_with_type_marks */
extern T1 T330x23003179T0T0T0T0T0(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3, T0* a4, T0* a5);
/* Call to ET_TYPE.shallow_named_type_with_type_mark */
extern T0* T330x23003151T0T0(GE_context* ac, T0* C, T0* a1, T0* a2);
/* Call to ET_TYPE.same_base_type_with_type_marks */
extern T1 T330x23003181T0T0T0T0T0(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3, T0* a4, T0* a5);
/* Call to ET_TYPE.to_text */
extern T0* T330x23003206(GE_context* ac, T0* C);
/* Call to ET_TYPE.is_type_detachable */
extern T1 T330x23003166T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_TYPE.is_base_type */
extern T1 T330x23003159(GE_context* ac, T0* C);
/* Call to ET_TYPE.is_type_attached */
extern T1 T330x23003164T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_TYPE.is_like_current */
extern T1 T330x23003157(GE_context* ac, T0* C);
/* Call to ET_TYPE.shallow_named_type */
extern T0* T330x23003150T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_TYPE.is_type_expanded */
extern T1 T330x23003160T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_TYPE.shallow_base_type_with_type_mark */
extern T0* T330x23003144T0T0(GE_context* ac, T0* C, T0* a1, T0* a2);
/* Call to ET_TYPE.base_type_index_of_label */
extern T6 T330x23003147T0T0(GE_context* ac, T0* C, T0* a1, T0* a2);
/* Call to ET_TYPE.base_type_with_type_mark */
extern T0* T330x23003142T0T0(GE_context* ac, T0* C, T0* a1, T0* a2);
/* Call to ET_TYPE.base_type_actual */
extern T0* T330x23003145T6T0(GE_context* ac, T0* C, T6 a1, T0* a2);
/* Call to ET_TYPE.type_with_type_mark */
extern T0* T330x23003154T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_TYPE.same_syntactical_like_current_with_type_marks */
extern T1 T330x23003184T0T0T0T0T0(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3, T0* a4, T0* a5);
/* Call to ET_TYPE.same_syntactical_tuple_type_with_type_marks */
extern T1 T330x23003187T0T0T0T0T0(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3, T0* a4, T0* a5);
/* Call to ET_TYPE.same_syntactical_class_type_with_type_marks */
extern T1 T330x23003182T0T0T0T0T0(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3, T0* a4, T0* a5);
/* Call to ET_TYPE.same_syntactical_like_feature_with_type_marks */
extern T1 T330x23003185T0T0T0T0T0(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3, T0* a4, T0* a5);
/* Call to ET_TYPE.same_syntactical_qualified_like_identifier_with_type_marks */
extern T1 T330x23003186T0T0T0T0T0(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3, T0* a4, T0* a5);
/* Call to ET_TYPE.named_base_class */
extern T0* T330x23003140T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_TYPE.is_type_attached_with_type_mark */
extern T1 T330x23003165T0T0(GE_context* ac, T0* C, T0* a1, T0* a2);
/* Call to ET_TYPE.is_type_reference_with_type_mark */
extern T1 T330x23003163T0T0(GE_context* ac, T0* C, T0* a1, T0* a2);
/* Call to ET_TYPE.is_type_detachable_with_type_mark */
extern T1 T330x23003167T0T0(GE_context* ac, T0* C, T0* a1, T0* a2);
/* Call to ET_TYPE.same_base_tuple_type_with_type_marks */
extern T1 T330x23003193T0T0T0T0T0(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3, T0* a4, T0* a5);
/* Call to ET_TYPE.same_base_class_type_with_type_marks */
extern T1 T330x23003191T0T0T0T0T0(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3, T0* a4, T0* a5);
/* Call to ET_TYPE.same_syntactical_formal_parameter_type_with_type_marks */
extern T1 T330x23003183T0T0T0T0T0(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3, T0* a4, T0* a5);
/* Call to ET_TYPE.same_syntactical_type_with_type_marks */
extern T1 T330x23003177T0T0T0T0T0(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3, T0* a4, T0* a5);
/* Call to ET_TYPE.is_type_expanded_with_type_mark */
extern T1 T330x23003161T0T0(GE_context* ac, T0* C, T0* a1, T0* a2);
/* Call to ET_TYPE.same_named_formal_parameter_type_with_type_marks */
extern T1 T330x23003189T0T0T0T0T0(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3, T0* a4, T0* a5);
/* Call to ET_TYPE.conforms_to_type_with_type_marks */
extern T1 T330x23003195T0T0T0T0T0T0(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3, T0* a4, T0* a5, T0* a6);
/* Call to ET_TYPE.conforms_from_formal_parameter_type_with_type_marks */
extern T1 T330x23003197T0T0T0T0T0T0(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3, T0* a4, T0* a5, T0* a6);
/* Call to ET_TYPE.conforms_from_tuple_type_with_type_marks */
extern T1 T330x23003198T0T0T0T0T0T0(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3, T0* a4, T0* a5, T0* a6);
/* Call to ET_TYPE.conforms_from_class_type_with_type_marks */
extern T1 T330x23003196T0T0T0T0T0T0(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3, T0* a4, T0* a5, T0* a6);
/* Call to ET_TYPE.base_type_actual_count */
extern T6 T330x23003156T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_TYPE.same_named_tuple_type_with_type_marks */
extern T1 T330x23003190T0T0T0T0T0(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3, T0* a4, T0* a5);
/* Call to ET_TYPE.same_named_class_type_with_type_marks */
extern T1 T330x23003188T0T0T0T0T0(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3, T0* a4, T0* a5);
/* Call to ET_TYPE.named_type_with_type_mark */
extern T0* T330x23003149T0T0(GE_context* ac, T0* C, T0* a1, T0* a2);
/* Call to ET_TYPE.is_type_reference */
extern T1 T330x23003162T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_TYPE.is_named_type */
extern T1 T330x23003158(GE_context* ac, T0* C);
/* Call to ET_TYPE.shallow_base_type */
extern T0* T330x23003143T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_TYPE.base_type */
extern T0* T330x23003141T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_TYPE.type */
extern T0* T330x23052289(GE_context* ac, T0* C);
/* Call to ET_TYPE_CONTEXT.to_nested_type_context */
extern T0* T331x23035952(GE_context* ac, T0* C);
/* Call to ET_TYPE_CONTEXT.as_nested_type_context */
extern T0* T331x23035951(GE_context* ac, T0* C);
/* Call to ET_TYPE_CONTEXT.root_context */
extern T0* T331x23035905(GE_context* ac, T0* C);
/* Call to ET_TYPE_CONTEXT.is_root_context */
extern T1 T331x23035918(GE_context* ac, T0* C);
/* Call to ET_TYPE_CONTEXT.base_type_actual */
extern T0* T331x23035911T6(GE_context* ac, T0* C, T6 a1);
/* Call to ET_TYPE_CONTEXT.base_type_actual_count */
extern T6 T331x23035916(GE_context* ac, T0* C);
/* Call to ET_TYPE_CONTEXT.base_class */
extern T0* T331x23035907(GE_context* ac, T0* C);
/* Call to ET_TYPE_CONTEXT.scoop_mode */
extern T1 T331x23035931(GE_context* ac, T0* C);
/* Call to ET_TYPE_CONTEXT.attachment_type_conformance_mode */
extern T1 T331x23035930(GE_context* ac, T0* C);
/* Call to ET_TYPE_CONTEXT.named_type */
extern T0* T331x23035914(GE_context* ac, T0* C);
/* Call to ET_TYPE_CONTEXT.same_base_type_with_type_marks */
extern T1 T331x23035937T0T0T0T0(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3, T0* a4);
/* Call to ET_TYPE_CONTEXT.is_type_attached_with_type_mark */
extern T1 T331x23035925T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_TYPE_CONTEXT.is_type_reference_with_type_mark */
extern T1 T331x23035923T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_TYPE_CONTEXT.same_named_formal_parameter_type_with_type_marks */
extern T1 T331x23035939T0T0T0T0(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3, T0* a4);
/* Call to ET_TYPE_CONTEXT.conforms_from_formal_parameter_type_with_type_marks */
extern T1 T331x23035949T0T0T0T0T0(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3, T0* a4, T0* a5);
/* Call to ET_TYPE_CONTEXT.named_type_with_type_mark */
extern T0* T331x23035915T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_TYPE_CONTEXT.same_named_type_with_type_marks */
extern T1 T331x23035933T0T0T0T0(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3, T0* a4);
/* Call to ET_TYPE_CONTEXT.conforms_to_type_with_type_marks */
extern T1 T331x23035945T0T0T0T0T0(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3, T0* a4, T0* a5);
/* Call to ET_TYPE_CONTEXT.is_type_detachable_with_type_mark */
extern T1 T331x23035927T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_TYPE_CONTEXT.same_named_tuple_type_with_type_marks */
extern T1 T331x23035940T0T0T0T0(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3, T0* a4);
/* Call to ET_TYPE_CONTEXT.same_named_class_type_with_type_marks */
extern T1 T331x23035938T0T0T0T0(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3, T0* a4);
/* Call to ET_TYPE_CONTEXT.conforms_from_tuple_type_with_type_marks */
extern T1 T331x23035950T0T0T0T0T0(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3, T0* a4, T0* a5);
/* Call to ET_TYPE_CONTEXT.conforms_from_class_type_with_type_marks */
extern T1 T331x23035948T0T0T0T0T0(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3, T0* a4, T0* a5);
/* Call to ET_TYPE_CONTEXT.same_base_tuple_type_with_type_marks */
extern T1 T331x23035943T0T0T0T0(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3, T0* a4);
/* Call to ET_TYPE_CONTEXT.same_base_class_type_with_type_marks */
extern T1 T331x23035941T0T0T0T0(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3, T0* a4);
/* Call to ET_TYPE_CONTEXT.base_type_with_type_mark */
extern T0* T331x23035910T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_TYPE_CONTEXT.is_type_expanded_with_type_mark */
extern T1 T331x23035921T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_TYPE_CONTEXT.named_base_class */
extern T0* T331x23035908(GE_context* ac, T0* C);
/* Call to ET_QUERY.lower_name */
extern T0* T334x19546115(GE_context* ac, T0* C);
/* Call to ET_QUERY.is_attribute */
extern T1 T334x18841631(GE_context* ac, T0* C);
/* Call to ET_QUERY.hash_code */
extern T6 T334x458753(GE_context* ac, T0* C);
/* Call to ET_QUERY.type */
extern T0* T334x18923529(GE_context* ac, T0* C);
/* Call to ET_QUERY.first_seed */
extern T6 T334x18923531(GE_context* ac, T0* C);
/* Call to ET_QUERY.overloaded_alias_name */
extern T0* T334x18841607(GE_context* ac, T0* C);
/* Call to ET_QUERY.alias_name */
extern T0* T334x18841603(GE_context* ac, T0* C);
/* Call to ET_QUERY.overloaded_name */
extern T0* T334x18841606(GE_context* ac, T0* C);
/* Call to ET_QUERY.name */
extern T0* T334x18923528(GE_context* ac, T0* C);
/* Call to ET_QUERY.arguments */
extern T0* T334x18923530(GE_context* ac, T0* C);
/* Call to ET_QUERY.is_exported_to */
extern T1 T334x18841654T0T0(GE_context* ac, T0* C, T0* a1, T0* a2);
/* Call to ET_QUERY.is_static */
extern T1 T334x18841637(GE_context* ac, T0* C);
/* Call to ET_QUERY.arguments_count */
extern T6 T334x18841653(GE_context* ac, T0* C);
/* Call to ET_QUERY.implementation_feature */
extern T0* T334x18432013(GE_context* ac, T0* C);
/* Call to ET_QUERY.implementation_class */
extern T0* T334x19546117(GE_context* ac, T0* C);
/* Call to ET_QUERY.is_prefixable */
extern T1 T334x18841642(GE_context* ac, T0* C);
/* Call to ET_QUERY.is_infixable */
extern T1 T334x18841641(GE_context* ac, T0* C);
/* Call to ET_QUERY.postconditions */
extern T0* T334x18432010(GE_context* ac, T0* C);
/* Call to ET_QUERY.preconditions */
extern T0* T334x18432009(GE_context* ac, T0* C);
/* Call to ET_QUERY.assigner */
extern T0* T334x19349507(GE_context* ac, T0* C);
/* Call to ET_QUERY.is_once_per_object */
extern T1 T334x18432004(GE_context* ac, T0* C);
/* Call to ET_QUERY.is_once */
extern T1 T334x18432001(GE_context* ac, T0* C);
/* Call to ET_QUERY.is_parenthesisable */
extern T1 T334x18841644(GE_context* ac, T0* C);
/* Call to ET_QUERY.is_bracketable */
extern T1 T334x18841643(GE_context* ac, T0* C);
/* Call to ET_QUERY.is_frozen */
extern T1 T334x18841627(GE_context* ac, T0* C);
/* Call to ET_QUERY.is_deferred */
extern T1 T334x18841628(GE_context* ac, T0* C);
/* Call to ET_QUERY.has_seed */
extern T1 T334x18923526T6(GE_context* ac, T0* C, T6 a1);
/* Call to ET_PROCEDURE.hash_code */
extern T6 T335x458753(GE_context* ac, T0* C);
/* Call to ET_PROCEDURE.is_static */
extern T1 T335x18841637(GE_context* ac, T0* C);
/* Call to ET_PROCEDURE.is_exported_to */
extern T1 T335x18841654T0T0(GE_context* ac, T0* C, T0* a1, T0* a2);
/* Call to ET_PROCEDURE.first_seed */
extern T6 T335x18923531(GE_context* ac, T0* C);
/* Call to ET_PROCEDURE.overloaded_alias_name */
extern T0* T335x18841607(GE_context* ac, T0* C);
/* Call to ET_PROCEDURE.alias_name */
extern T0* T335x18841603(GE_context* ac, T0* C);
/* Call to ET_PROCEDURE.overloaded_name */
extern T0* T335x18841606(GE_context* ac, T0* C);
/* Call to ET_PROCEDURE.name */
extern T0* T335x18923528(GE_context* ac, T0* C);
/* Call to ET_PROCEDURE.arguments */
extern T0* T335x18923530(GE_context* ac, T0* C);
/* Call to ET_PROCEDURE.implementation_class */
extern T0* T335x19546117(GE_context* ac, T0* C);
/* Call to ET_PROCEDURE.postconditions */
extern T0* T335x18432010(GE_context* ac, T0* C);
/* Call to ET_PROCEDURE.preconditions */
extern T0* T335x18432009(GE_context* ac, T0* C);
/* Call to ET_PROCEDURE.is_parenthesisable */
extern T1 T335x18841644(GE_context* ac, T0* C);
/* Call to ET_PROCEDURE.is_frozen */
extern T1 T335x18841627(GE_context* ac, T0* C);
/* Call to ET_PROCEDURE.is_deferred */
extern T1 T335x18841628(GE_context* ac, T0* C);
/* Call to ET_PROCEDURE.has_seed */
extern T1 T335x18923526T6(GE_context* ac, T0* C, T6 a1);
/* Call to ET_UNIVERSE.adapted_universe */
extern T0* T336x20070418T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_UNIVERSE.shortest_path */
extern T0* T336x20070419T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_UNIVERSE.full_lower_name */
extern T0* T336x20070423T2(GE_context* ac, T0* C, T2 a1);
/* Call to ET_UNIVERSE.kind_name */
extern T0* T336x20070425(GE_context* ac, T0* C);
/* Call to ET_UNIVERSE.real_64_convert_feature */
extern T0* T336x20070510(GE_context* ac, T0* C);
/* Call to ET_UNIVERSE.real_64_type */
extern T0* T336x20070471(GE_context* ac, T0* C);
/* Call to ET_UNIVERSE.real_32_convert_feature */
extern T0* T336x20070509(GE_context* ac, T0* C);
/* Call to ET_UNIVERSE.real_32_type */
extern T0* T336x20070470(GE_context* ac, T0* C);
/* Call to ET_UNIVERSE.natural_64_convert_feature */
extern T0* T336x20070508(GE_context* ac, T0* C);
/* Call to ET_UNIVERSE.natural_64_type */
extern T0* T336x20070463(GE_context* ac, T0* C);
/* Call to ET_UNIVERSE.natural_32_convert_feature */
extern T0* T336x20070507(GE_context* ac, T0* C);
/* Call to ET_UNIVERSE.natural_32_type */
extern T0* T336x20070462(GE_context* ac, T0* C);
/* Call to ET_UNIVERSE.natural_16_convert_feature */
extern T0* T336x20070506(GE_context* ac, T0* C);
/* Call to ET_UNIVERSE.natural_16_type */
extern T0* T336x20070461(GE_context* ac, T0* C);
/* Call to ET_UNIVERSE.natural_8_convert_feature */
extern T0* T336x20070505(GE_context* ac, T0* C);
/* Call to ET_UNIVERSE.natural_8_type */
extern T0* T336x20070460(GE_context* ac, T0* C);
/* Call to ET_UNIVERSE.integer_64_convert_feature */
extern T0* T336x20070504(GE_context* ac, T0* C);
/* Call to ET_UNIVERSE.integer_64_type */
extern T0* T336x20070456(GE_context* ac, T0* C);
/* Call to ET_UNIVERSE.integer_32_convert_feature */
extern T0* T336x20070503(GE_context* ac, T0* C);
/* Call to ET_UNIVERSE.integer_32_type */
extern T0* T336x20070455(GE_context* ac, T0* C);
/* Call to ET_UNIVERSE.integer_16_convert_feature */
extern T0* T336x20070502(GE_context* ac, T0* C);
/* Call to ET_UNIVERSE.integer_16_type */
extern T0* T336x20070454(GE_context* ac, T0* C);
/* Call to ET_UNIVERSE.integer_8_convert_feature */
extern T0* T336x20070501(GE_context* ac, T0* C);
/* Call to ET_UNIVERSE.integer_8_type */
extern T0* T336x20070453(GE_context* ac, T0* C);
/* Call to ET_UNIVERSE.character_32_convert_feature */
extern T0* T336x20070500(GE_context* ac, T0* C);
/* Call to ET_UNIVERSE.character_32_type */
extern T0* T336x20070445(GE_context* ac, T0* C);
/* Call to ET_UNIVERSE.character_8_convert_feature */
extern T0* T336x20070499(GE_context* ac, T0* C);
/* Call to ET_UNIVERSE.character_8_type */
extern T0* T336x20070444(GE_context* ac, T0* C);
/* Call to ET_UNIVERSE.string_32_convert_feature */
extern T0* T336x20070512(GE_context* ac, T0* C);
/* Call to ET_UNIVERSE.string_32_type */
extern T0* T336x20070478(GE_context* ac, T0* C);
/* Call to ET_UNIVERSE.string_8_convert_feature */
extern T0* T336x20070511(GE_context* ac, T0* C);
/* Call to ET_UNIVERSE.string_8_type */
extern T0* T336x20070477(GE_context* ac, T0* C);
/* Call to ET_UNIVERSE.boolean_type */
extern T0* T336x20070442(GE_context* ac, T0* C);
/* Call to ET_UNIVERSE.master_class */
extern T0* T336x20070407T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_UNIVERSE.tuple_type */
extern T0* T336x20070482(GE_context* ac, T0* C);
/* Call to ET_UNIVERSE.function_type */
extern T0* T336x20070450(GE_context* ac, T0* C);
/* Call to ET_UNIVERSE.predicate_like_current_type */
extern T0* T336x20070467(GE_context* ac, T0* C);
/* Call to ET_UNIVERSE.procedure_like_current_type */
extern T0* T336x20070468(GE_context* ac, T0* C);
/* Call to ET_UNIVERSE.pointer_type */
extern T0* T336x20070466(GE_context* ac, T0* C);
/* Call to ET_UNIVERSE.typed_pointer_like_current_type */
extern T0* T336x20070495(GE_context* ac, T0* C);
/* Call to ET_UNIVERSE.tuple_like_current_type */
extern T0* T336x20070484(GE_context* ac, T0* C);
/* Call to ET_UNIVERSE.type_like_current_type */
extern T0* T336x20070493(GE_context* ac, T0* C);
/* Call to ET_UNIVERSE.any_type */
extern T0* T336x20070431(GE_context* ac, T0* C);
/* Call to ET_UNIVERSE.integer_type */
extern T0* T336x20070452(GE_context* ac, T0* C);
/* Call to ET_UNIVERSE.iterable_detachable_any_type */
extern T0* T336x20070458(GE_context* ac, T0* C);
/* Call to ET_UNIVERSE.detachable_any_type */
extern T0* T336x20070432(GE_context* ac, T0* C);
/* Call to ET_UNIVERSE.hash_code */
extern T6 T336x458753(GE_context* ac, T0* C);
/* Call to ET_UNIVERSE.has_master_class */
extern T1 T336x20070401T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_UNIVERSE.relative_name */
extern T0* T336x20070420T0T2(GE_context* ac, T0* C, T0* a1, T2 a2);
/* Call to ET_UNIVERSE.relative_lower_name */
extern T0* T336x20070421T0T2(GE_context* ac, T0* C, T0* a1, T2 a2);
/* Call to ET_UNIVERSE.implicit_attachment_type_mark */
extern T0* T336x20070497(GE_context* ac, T0* C);
/* Call to ET_UNIVERSE.unfolded_empty_tuple_actual_parameters */
extern T0* T336x20070485(GE_context* ac, T0* C);
/* Call to ET_UNIVERSE.obsolete_routine_type_mode */
extern T1 T336x20070498(GE_context* ac, T0* C);
/* Call to ET_UNIVERSE.immutable_string_32_type */
extern T0* T336x20070451(GE_context* ac, T0* C);
/* Call to ET_UNIVERSE.system_string_type */
extern T0* T336x20070481(GE_context* ac, T0* C);
/* Call to ET_UNIVERSE.string_type */
extern T0* T336x20070475(GE_context* ac, T0* C);
/* Call to ET_UNIVERSE.detachable_separate_any_type */
extern T0* T336x20070433(GE_context* ac, T0* C);
/* Call to ET_UNIVERSE.double_type */
extern T0* T336x20070446(GE_context* ac, T0* C);
/* Call to ET_UNIVERSE.real_type */
extern T0* T336x20070469(GE_context* ac, T0* C);
/* Call to ET_UNIVERSE.detachable_string_type */
extern T0* T336x20070476(GE_context* ac, T0* C);
/* Call to ET_UNIVERSE.special_any_type */
extern T0* T336x20070473(GE_context* ac, T0* C);
/* Call to ET_UNIVERSE.type_detachable_any_type */
extern T0* T336x20070488(GE_context* ac, T0* C);
/* Call to ET_UNIVERSE.special_detachable_any_type */
extern T0* T336x20070474(GE_context* ac, T0* C);
/* Call to ET_UNIVERSE.exception_manager_type */
extern T0* T336x20070449(GE_context* ac, T0* C);
/* Call to ET_UNIVERSE.type_detachable_exception_type */
extern T0* T336x20070492(GE_context* ac, T0* C);
/* Call to ET_UNIVERSE.exception_type */
extern T0* T336x20070447(GE_context* ac, T0* C);
/* Call to ET_UNIVERSE.detachable_type_exception_type */
extern T0* T336x20070491(GE_context* ac, T0* C);
/* Call to ET_UNIVERSE.detachable_exception_type */
extern T0* T336x20070448(GE_context* ac, T0* C);
/* Call to ET_UNIVERSE.type_detachable_like_current_type */
extern T0* T336x20070490(GE_context* ac, T0* C);
/* Call to ET_UNIVERSE.current_system */
extern T0* T336x20070417(GE_context* ac, T0* C);
/* Call to ET_UNIVERSE.full_name */
extern T0* T336x20070422T2(GE_context* ac, T0* C, T2 a1);
/* Call to ET_UNIVERSE.classes_in_group */
extern T0* T336x20070412T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_UNIVERSE.system_object_parents */
extern T0* T336x20070480(GE_context* ac, T0* C);
/* Call to ET_UNIVERSE.any_parents */
extern T0* T336x20070436(GE_context* ac, T0* C);
/* Call to ET_UNIVERSE.system_object_type */
extern T0* T336x20070479(GE_context* ac, T0* C);
/* Call to ET_UNIVERSE.natural_type */
extern T0* T336x20070459(GE_context* ac, T0* C);
/* Call to ET_UNIVERSE.wide_character_type */
extern T0* T336x20070496(GE_context* ac, T0* C);
/* Call to ET_UNIVERSE.character_type */
extern T0* T336x20070443(GE_context* ac, T0* C);
/* Call to ET_ACTUAL_PARAMETER_ITEM.resolved_formal_parameters_with_type */
extern T0* T337x22413317T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_ACTUAL_PARAMETER_ITEM.type */
extern T0* T337x22413314(GE_context* ac, T0* C);
/* Call to ET_ACTUAL_PARAMETER_ITEM.actual_parameter */
extern T0* T337x22413313(GE_context* ac, T0* C);
/* Call to detachable ET_TYPE_MARK.is_attached_mark */
extern T1 T339x21397511(GE_context* ac, T0* C);
/* Call to detachable ET_TYPE_MARK.is_detachable_mark */
extern T1 T339x21397512(GE_context* ac, T0* C);
/* Call to detachable ET_TYPE_MARK.is_attachment_mark */
extern T1 T339x21397510(GE_context* ac, T0* C);
/* Call to detachable ET_TYPE_MARK.position */
extern T0* T339x20955137(GE_context* ac, T0* C);
/* Call to detachable ET_TYPE_MARK.is_separateness_mark */
extern T1 T339x21397508(GE_context* ac, T0* C);
/* Call to detachable ET_TYPE_MARK.is_expanded_mark */
extern T1 T339x21397506(GE_context* ac, T0* C);
/* Call to detachable ET_TYPE_MARK.is_expandedness_mark */
extern T1 T339x21397505(GE_context* ac, T0* C);
/* Call to detachable ET_TYPE_MARK.is_separate_mark */
extern T1 T339x21397509(GE_context* ac, T0* C);
/* Call to detachable ET_TYPE_MARK.is_reference_mark */
extern T1 T339x21397507(GE_context* ac, T0* C);
/* Call to ET_ACTUAL_PARAMETERS.last_leaf */
extern T0* T340x20955141(GE_context* ac, T0* C);
/* Call to ET_ACTUAL_PARAMETERS.type */
extern T0* T340x22380547T6(GE_context* ac, T0* C, T6 a1);
/* Call to ET_ACTUAL_PARAMETERS.actual_parameter */
extern T0* T340x22380546T6(GE_context* ac, T0* C, T6 a1);
/* Call to ET_ACTUAL_PARAMETERS.count */
extern T6 T340x22380545(GE_context* ac, T0* C);
/* Call to ET_ACTUAL_PARAMETERS.is_empty */
extern T1 T340x22380551(GE_context* ac, T0* C);
/* Call to ET_ACTUAL_PARAMETERS.has_identifier_anchored_type */
extern T1 T340x22380553(GE_context* ac, T0* C);
/* Call to ET_ACTUAL_PARAMETERS.index_of_label */
extern T6 T340x22380549T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_ACTUAL_PARAMETERS.depends_on_qualified_anchored_type */
extern T1 T340x22380554T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_ACTUAL_PARAMETERS.same_syntactical_types */
extern T1 T340x22380557T0T0T0(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* Call to ET_ACTUAL_PARAMETERS.has_formal_types */
extern T1 T340x22380555T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_ACTUAL_PARAMETERS.has_anchored_type */
extern T1 T340x22380552(GE_context* ac, T0* C);
/* Call to ET_ACTUAL_PARAMETERS.same_named_types */
extern T1 T340x22380558T0T0T0(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* Call to ET_ACTUAL_PARAMETERS.conforms_to_types */
extern T1 T340x22380559T0T0T0T0(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3, T0* a4);
/* Call to detachable ET_DYNAMIC_TYPE_SET.static_type */
extern T0* T342x24739845(GE_context* ac, T0* C);
/* Call to detachable ET_DYNAMIC_TYPE_SET.is_expanded */
extern T1 T342x24739841(GE_context* ac, T0* C);
/* Call to detachable ET_DYNAMIC_TYPE_SET.is_empty */
extern T1 T342x24657925(GE_context* ac, T0* C);
/* Call to detachable ET_DYNAMIC_TYPE_SET.special_type */
extern T0* T342x24657922(GE_context* ac, T0* C);
/* Call to detachable ET_DYNAMIC_TYPE_SET.dynamic_type */
extern T0* T342x24657921T6(GE_context* ac, T0* C, T6 a1);
/* Call to detachable ET_DYNAMIC_TYPE_SET.count */
extern T6 T342x24657924(GE_context* ac, T0* C);
/* Call to ET_DYNAMIC_TYPE_SET.static_type */
extern T0* T343x24739845(GE_context* ac, T0* C);
/* Call to ET_DYNAMIC_TYPE_SET.is_never_void */
extern T1 T343x24739842(GE_context* ac, T0* C);
/* Call to ET_DYNAMIC_TYPE_SET.can_be_void */
extern T1 T343x24739843(GE_context* ac, T0* C);
/* Call to ET_DYNAMIC_TYPE_SET.dynamic_type */
extern T0* T343x24657921T6(GE_context* ac, T0* C, T6 a1);
/* Call to ET_DYNAMIC_TYPE_SET.count */
extern T6 T343x24657924(GE_context* ac, T0* C);
/* Call to ET_DYNAMIC_TYPE_SET.has_expanded */
extern T1 T343x24657928(GE_context* ac, T0* C);
/* Call to ET_DYNAMIC_TYPE_SET.is_expanded */
extern T1 T343x24739841(GE_context* ac, T0* C);
/* Call to ET_DYNAMIC_TYPE_SET.dynamic_types */
extern T0* T343x24739847(GE_context* ac, T0* C);
/* Call to ET_DYNAMIC_TYPE_SET.is_empty */
extern T1 T343x24657925(GE_context* ac, T0* C);
/* Call to ET_DYNAMIC_TYPE_SET.has_type */
extern T1 T343x24657926T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_EXPRESSION.last_leaf */
extern T0* T389x20955141(GE_context* ac, T0* C);
/* Call to ET_EXPRESSION.position */
extern T0* T389x20955137(GE_context* ac, T0* C);
/* Call to ET_EXPRESSION.is_instance_free */
extern T1 T389x17416196(GE_context* ac, T0* C);
/* Call to ET_EXPRESSION.is_false */
extern T1 T389x17694727(GE_context* ac, T0* C);
/* Call to ET_EXPRESSION.manifest_constant_convert_feature */
extern T0* T389x17694731T0T0T0(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* Call to ET_EXPRESSION.unparenthesized_expression */
extern T0* T389x17694724(GE_context* ac, T0* C);
/* Call to ET_EXPRESSION.index */
extern T6 T389x18071555(GE_context* ac, T0* C);
/* Call to ET_EXPRESSION.is_current */
extern T1 T389x17694726(GE_context* ac, T0* C);
/* Call to ET_EXPRESSION.is_instance_free */
extern T1 T389x18071554(GE_context* ac, T0* C);
/* Call to ET_EXPRESSION.is_never_void */
extern T1 T389x17694725(GE_context* ac, T0* C);
/* Call to ET_FEATURE.lower_name */
extern T0* T393x19546115(GE_context* ac, T0* C);
/* Call to ET_FEATURE.name */
extern T0* T393x18923528(GE_context* ac, T0* C);
/* Call to ET_FEATURE.has_seed */
extern T1 T393x18923526T6(GE_context* ac, T0* C, T6 a1);
/* Call to ET_FEATURE.first_seed */
extern T6 T393x18923531(GE_context* ac, T0* C);
/* Call to ET_FEATURE.is_exported_to */
extern T1 T393x18841654T0T0(GE_context* ac, T0* C, T0* a1, T0* a2);
/* Call to ET_FEATURE.clients */
extern T0* T393x19546119(GE_context* ac, T0* C);
/* Call to ET_FEATURE.is_static */
extern T1 T393x18841637(GE_context* ac, T0* C);
/* Call to ET_FEATURE.arguments */
extern T0* T393x18923530(GE_context* ac, T0* C);
/* Call to ET_FEATURE.extended_name */
extern T0* T393x18841604(GE_context* ac, T0* C);
/* Call to ET_FEATURE.arguments_count */
extern T6 T393x18841653(GE_context* ac, T0* C);
/* Call to ET_FEATURE.implementation_feature */
extern T0* T393x18432013(GE_context* ac, T0* C);
/* Call to ET_FEATURE.implementation_class */
extern T0* T393x19546117(GE_context* ac, T0* C);
/* Call to ET_FEATURE.is_procedure */
extern T1 T393x18841635(GE_context* ac, T0* C);
/* Call to ET_FEATURE.are_postconditions_instance_free_recursive */
extern T1 T393x18841646(GE_context* ac, T0* C);
/* Call to ET_FEATURE.are_preconditions_instance_free_recursive */
extern T1 T393x18841645(GE_context* ac, T0* C);
/* Call to ET_FEATURE.is_deferred */
extern T1 T393x18841628(GE_context* ac, T0* C);
/* Call to ET_FEATURE.version */
extern T6 T393x18841612(GE_context* ac, T0* C);
/* Call to ET_FEATURE.is_unique_attribute */
extern T1 T393x18841633(GE_context* ac, T0* C);
/* Call to ET_FEATURE.is_constant_attribute */
extern T1 T393x18841632(GE_context* ac, T0* C);
/* Call to ET_FEATURE.type */
extern T0* T393x18923529(GE_context* ac, T0* C);
/* Call to ET_FEATURE.is_once_per_process */
extern T1 T393x18432002(GE_context* ac, T0* C);
/* Call to ET_FEATURE.is_attribute */
extern T1 T393x18841631(GE_context* ac, T0* C);
/* Call to ET_FEATURE.has_class_postcondition_recursive */
extern T1 T393x18841647(GE_context* ac, T0* C);
/* Call to ET_FEATURE.alias_name */
extern T0* T393x18841603(GE_context* ac, T0* C);
/* Call to ET_FEATURE.other_seeds */
extern T0* T393x18923532(GE_context* ac, T0* C);
/* Call to ET_FEATURE.hash_code */
extern T6 T393x458753(GE_context* ac, T0* C);
/* Call to ET_FEATURE.preconditions */
extern T0* T393x18432009(GE_context* ac, T0* C);
/* Call to ET_FEATURE.other_precursors */
extern T0* T393x18841614(GE_context* ac, T0* C);
/* Call to ET_FEATURE.first_precursor */
extern T0* T393x19546116(GE_context* ac, T0* C);
/* Call to ET_FEATURE.has_validity_error */
extern T1 T393x19546121(GE_context* ac, T0* C);
/* Call to ET_FEATURE.validity_checked */
extern T1 T393x19546120(GE_context* ac, T0* C);
/* Call to ET_FEATURE.postconditions */
extern T0* T393x18432010(GE_context* ac, T0* C);
/* Call to ET_FEATURE.is_frozen */
extern T1 T393x18841627(GE_context* ac, T0* C);
/* Call to ET_FEATURE.is_infixable */
extern T1 T393x18841641(GE_context* ac, T0* C);
/* Call to ET_FEATURE.is_prefixable */
extern T1 T393x18841642(GE_context* ac, T0* C);
/* Call to ET_FEATURE.is_parenthesisable */
extern T1 T393x18841644(GE_context* ac, T0* C);
/* Call to ET_FEATURE.is_bracketable */
extern T1 T393x18841643(GE_context* ac, T0* C);
/* Call to ET_FEATURE.id */
extern T6 T393x18841611(GE_context* ac, T0* C);
/* Call to ET_FEATURE.renamed_feature */
extern T0* T393x18841661T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_FEATURE.undefined_feature */
extern T0* T393x18841662T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_FEATURE.same_version */
extern T1 T393x18841659T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_FEATURE.is_dotnet */
extern T1 T393x18923525(GE_context* ac, T0* C);
/* Call to ET_FEATURE.is_query */
extern T1 T393x18841634(GE_context* ac, T0* C);
/* Call to ET_FEATURE.is_once */
extern T1 T393x18432001(GE_context* ac, T0* C);
/* Call to ET_QUALIFIED_FEATURE_NAME.last_leaf */
extern T0* T394x20955141(GE_context* ac, T0* C);
/* Call to ET_QUALIFIED_FEATURE_NAME.feature_name */
extern T0* T394x22151169(GE_context* ac, T0* C);
/* Call to ET_CALL_COMPONENT.name */
extern T0* T425x17465346(GE_context* ac, T0* C);
/* Call to ET_CALL_COMPONENT.arguments */
extern T0* T425x17465347(GE_context* ac, T0* C);
/* Call to ET_CALL_COMPONENT.position */
extern T0* T425x20955137(GE_context* ac, T0* C);
/* Call to ET_CALL_COMPONENT.is_tuple_label */
extern T1 T425x17465349(GE_context* ac, T0* C);
/* Call to ET_CALL_COMPONENT.is_call_agent */
extern T1 T425x17465350(GE_context* ac, T0* C);
/* Call to ET_CALL_NAME.position */
extern T0* T426x20955137(GE_context* ac, T0* C);
/* Call to ET_CALL_NAME.seed */
extern T6 T426x21544963(GE_context* ac, T0* C);
/* Call to ET_CALL_NAME.is_tuple_label */
extern T1 T426x21544969(GE_context* ac, T0* C);
/* Call to ET_CALL_NAME.lower_name */
extern T0* T426x21544962(GE_context* ac, T0* C);
/* Call to ET_CALL_NAME.name */
extern T0* T426x21544961(GE_context* ac, T0* C);
/* Call to ET_CALL_NAME.is_infix_or_else */
extern T1 T426x21544985(GE_context* ac, T0* C);
/* Call to ET_CALL_NAME.is_infix_implies */
extern T1 T426x21544979(GE_context* ac, T0* C);
/* Call to ET_CALL_NAME.is_infix_and_then */
extern T1 T426x21544973(GE_context* ac, T0* C);
/* Call to ET_CALL_NAME.precursor_keyword */
extern T0* T426x21545001(GE_context* ac, T0* C);
/* Call to ET_CALL_NAME.is_precursor */
extern T1 T426x21544999(GE_context* ac, T0* C);
/* Call to ET_CALL_NAME.is_infix_xor */
extern T1 T426x21544989(GE_context* ac, T0* C);
/* Call to ET_CALL_NAME.is_infix_times */
extern T1 T426x21544988(GE_context* ac, T0* C);
/* Call to ET_CALL_NAME.is_infix_power */
extern T1 T426x21544987(GE_context* ac, T0* C);
/* Call to ET_CALL_NAME.is_infix_plus */
extern T1 T426x21544986(GE_context* ac, T0* C);
/* Call to ET_CALL_NAME.is_infix_or */
extern T1 T426x21544984(GE_context* ac, T0* C);
/* Call to ET_CALL_NAME.is_infix_mod */
extern T1 T426x21544983(GE_context* ac, T0* C);
/* Call to ET_CALL_NAME.is_infix_minus */
extern T1 T426x21544982(GE_context* ac, T0* C);
/* Call to ET_CALL_NAME.is_infix_lt */
extern T1 T426x21544981(GE_context* ac, T0* C);
/* Call to ET_CALL_NAME.is_infix_le */
extern T1 T426x21544980(GE_context* ac, T0* C);
/* Call to ET_CALL_NAME.is_infix_gt */
extern T1 T426x21544978(GE_context* ac, T0* C);
/* Call to ET_CALL_NAME.is_infix_ge */
extern T1 T426x21544977(GE_context* ac, T0* C);
/* Call to ET_CALL_NAME.is_infix_divide */
extern T1 T426x21544975(GE_context* ac, T0* C);
/* Call to ET_CALL_NAME.is_infix_div */
extern T1 T426x21544974(GE_context* ac, T0* C);
/* Call to ET_CALL_NAME.is_infix_and */
extern T1 T426x21544972(GE_context* ac, T0* C);
/* Call to ET_CALL_NAME.is_prefix_not */
extern T1 T426x21544995(GE_context* ac, T0* C);
/* Call to ET_CALL_NAME.is_prefix_plus */
extern T1 T426x21544993(GE_context* ac, T0* C);
/* Call to ET_CALL_NAME.is_prefix_minus */
extern T1 T426x21544992(GE_context* ac, T0* C);
/* Call to ET_CALL_NAME.hash_code */
extern T6 T426x458753(GE_context* ac, T0* C);
/* Call to ET_CALL_NAME.is_infix_dotdot */
extern T1 T426x21544990(GE_context* ac, T0* C);
/* Call to ET_CALL_NAME.is_parenthesis */
extern T1 T426x21544997(GE_context* ac, T0* C);
/* Call to ET_CALL_NAME.is_bracket */
extern T1 T426x21544996(GE_context* ac, T0* C);
/* Call to ET_CALL_NAME.same_call_name */
extern T1 T426x21545000T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_CALL_NAME.feature_name */
extern T0* T426x21545006(GE_context* ac, T0* C);
/* Call to ET_AGENT.target */
extern T0* T430x15400962(GE_context* ac, T0* C);
/* Call to ET_AGENT.arguments */
extern T0* T430x15400963(GE_context* ac, T0* C);
/* Call to ET_AGENT.implicit_target_position */
extern T0* T430x15400970(GE_context* ac, T0* C);
/* Call to ET_AGENT.position */
extern T0* T430x20955137(GE_context* ac, T0* C);
/* Call to ET_AGENT.is_inline_agent */
extern T1 T430x15400968(GE_context* ac, T0* C);
/* Call to ET_AGENT.index */
extern T6 T430x18071555(GE_context* ac, T0* C);
/* Call to ET_AGENT.is_qualified_call */
extern T1 T430x15400965(GE_context* ac, T0* C);
/* Call to ET_AGENT.implicit_result */
extern T0* T430x15400964(GE_context* ac, T0* C);
/* Call to ET_OBJECT_TEST.expression */
extern T0* T431x18022404(GE_context* ac, T0* C);
/* Call to ET_OBJECT_TEST.index */
extern T6 T431x18071555(GE_context* ac, T0* C);
/* Call to ET_OBJECT_TEST.type */
extern T0* T431x18022403(GE_context* ac, T0* C);
/* Call to ET_OBJECT_TEST.name */
extern T0* T431x18022405(GE_context* ac, T0* C);
/* Call to detachable ET_TYPE.to_text */
extern T0* T436x23003206(GE_context* ac, T0* C);
/* Call to detachable ET_TYPE.shallow_named_type_with_type_mark */
extern T0* T436x23003151T0T0(GE_context* ac, T0* C, T0* a1, T0* a2);
/* Call to detachable ET_TYPE.is_like_current */
extern T1 T436x23003157(GE_context* ac, T0* C);
/* Call to detachable ET_TYPE.shallow_named_type */
extern T0* T436x23003150T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable ET_TYPE.named_type */
extern T0* T436x23003148T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable ET_TYPE.base_class */
extern T0* T436x23003139T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable ET_TYPE.is_base_type */
extern T1 T436x23003159(GE_context* ac, T0* C);
/* Call to detachable ET_TYPE.position */
extern T0* T436x20955137(GE_context* ac, T0* C);
/* Call to detachable ET_TYPE.is_named_type */
extern T1 T436x23003158(GE_context* ac, T0* C);
/* Call to detachable ET_TYPE.same_syntactical_type */
extern T1 T436x23003176T0T0T0(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* Call to detachable ET_TYPE.conforms_to_type */
extern T1 T436x23003194T0T0T0T0(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3, T0* a4);
/* Call to detachable ET_TYPE.same_syntactical_type_with_type_marks */
extern T1 T436x23003177T0T0T0T0T0(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3, T0* a4, T0* a5);
/* Call to detachable ET_TYPE.same_named_type */
extern T1 T436x23003178T0T0T0(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* Call to detachable ET_TYPE.is_type_reference */
extern T1 T436x23003162T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable ET_TYPE.is_type_expanded */
extern T1 T436x23003160T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_CONSTANT.is_string_constant */
extern T1 T437x17039365(GE_context* ac, T0* C);
/* Call to ET_CONSTANT.is_character_constant */
extern T1 T437x17039362(GE_context* ac, T0* C);
/* Call to ET_CONSTANT.is_boolean_constant */
extern T1 T437x17039361(GE_context* ac, T0* C);
/* Call to ET_CONSTANT.index */
extern T6 T437x18071555(GE_context* ac, T0* C);
/* Call to detachable ET_ACTUAL_PARAMETERS.type */
extern T0* T443x22380547T6(GE_context* ac, T0* C, T6 a1);
/* Call to detachable ET_ACTUAL_PARAMETERS.count */
extern T6 T443x22380545(GE_context* ac, T0* C);
/* Call to detachable ET_ACTUAL_PARAMETERS.resolved_formal_parameters */
extern T0* T443x22380562T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable ET_ACTUAL_PARAMETERS.is_empty */
extern T1 T443x22380551(GE_context* ac, T0* C);
/* Call to detachable ET_ACTUAL_PARAMETERS.named_types */
extern T0* T443x22380548T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_TARGET_TYPE.position */
extern T0* T446x20955137(GE_context* ac, T0* C);
/* Call to ET_TARGET_TYPE.type */
extern T0* T446x22970369(GE_context* ac, T0* C);
/* Call to ET_TARGET_TYPE.last_leaf */
extern T0* T446x20955141(GE_context* ac, T0* C);
/* Call to detachable ET_ARGUMENT_OPERANDS.actual_argument */
extern T0* T458x17416197T6(GE_context* ac, T0* C, T6 a1);
/* Call to detachable ET_ARGUMENT_OPERANDS.count */
extern T6 T458x17416198(GE_context* ac, T0* C);
/* Call to detachable ET_ARGUMENT_OPERANDS.is_empty */
extern T1 T458x17416194(GE_context* ac, T0* C);
/* Call to detachable ET_FEATURE.has_seed */
extern T1 T460x18923526T6(GE_context* ac, T0* C, T6 a1);
/* Call to detachable ET_FEATURE.arguments_count */
extern T6 T460x18841653(GE_context* ac, T0* C);
/* Call to detachable ET_FEATURE.arguments */
extern T0* T460x18923530(GE_context* ac, T0* C);
/* Call to detachable ET_FEATURE.type */
extern T0* T460x18923529(GE_context* ac, T0* C);
/* Call to ET_EXPRESSION_ITEM.expression */
extern T0* T463x17760257(GE_context* ac, T0* C);
/* Call to ET_EXPRESSION_ITEM.position */
extern T0* T463x20955137(GE_context* ac, T0* C);
/* Call to ET_OPERAND.index */
extern T6 T466x18071555(GE_context* ac, T0* C);
/* Call to ET_OPERAND.is_open_operand */
extern T1 T466x18071553(GE_context* ac, T0* C);
/* Call to ET_AGENT_TARGET.index */
extern T6 T467x18071555(GE_context* ac, T0* C);
/* Call to ET_AGENT_TARGET.is_instance_free */
extern T1 T467x18071554(GE_context* ac, T0* C);
/* Call to ET_AGENT_TARGET.position */
extern T0* T467x20955137(GE_context* ac, T0* C);
/* Call to ET_AGENT_TARGET.is_open_operand */
extern T1 T467x18071553(GE_context* ac, T0* C);
/* Call to ET_AGENT_ARGUMENT_OPERAND.is_instance_free */
extern T1 T470x18071554(GE_context* ac, T0* C);
/* Call to ET_AGENT_ARGUMENT_OPERAND.is_open_operand */
extern T1 T470x18071553(GE_context* ac, T0* C);
/* Call to ET_AGENT_ARGUMENT_OPERAND.index */
extern T6 T470x18071555(GE_context* ac, T0* C);
/* Call to ET_SYMBOL.position */
extern T0* T474x20955137(GE_context* ac, T0* C);
/* Call to ET_SYMBOL.is_question_mark */
extern T1 T474x21397519(GE_context* ac, T0* C);
/* Call to ET_SYMBOL.is_bang */
extern T1 T474x21397520(GE_context* ac, T0* C);
/* Call to detachable ET_ACTUAL_ARGUMENTS.actual_argument */
extern T0* T482x17416197T6(GE_context* ac, T0* C, T6 a1);
/* Call to detachable ET_ACTUAL_ARGUMENTS.count */
extern T6 T482x17416198(GE_context* ac, T0* C);
/* Call to detachable ET_ACTUAL_ARGUMENTS.is_empty */
extern T1 T482x17416194(GE_context* ac, T0* C);
/* Call to ET_ACTUAL_ARGUMENTS.actual_argument */
extern T0* T483x17416197T6(GE_context* ac, T0* C, T6 a1);
/* Call to ET_ACTUAL_ARGUMENTS.count */
extern T6 T483x17416198(GE_context* ac, T0* C);
/* Call to ET_DYNAMIC_TYPES.dynamic_type */
extern T0* T486x24657921T6(GE_context* ac, T0* C, T6 a1);
/* Call to ET_DYNAMIC_TYPES.count */
extern T6 T486x24657924(GE_context* ac, T0* C);
/* Call to ET_MANIFEST_STRING.cast_type */
extern T0* T493x17121285(GE_context* ac, T0* C);
/* Call to ET_MANIFEST_STRING.value */
extern T0* T493x17121283(GE_context* ac, T0* C);
/* Call to ET_MANIFEST_STRING.position */
extern T0* T493x20955137(GE_context* ac, T0* C);
/* Call to ET_MANIFEST_STRING.index */
extern T6 T493x18071555(GE_context* ac, T0* C);
/* Call to ET_STANDALONE_CLOSURE.as_invariants */
extern T0* T511x19546123(GE_context* ac, T0* C);
/* Call to ET_STANDALONE_CLOSURE.is_invariants */
extern T1 T511x19546114(GE_context* ac, T0* C);
/* Call to ET_STANDALONE_CLOSURE.as_feature */
extern T0* T511x19546122(GE_context* ac, T0* C);
/* Call to ET_STANDALONE_CLOSURE.is_feature */
extern T1 T511x19546113(GE_context* ac, T0* C);
/* Call to ET_STANDALONE_CLOSURE.first_precursor */
extern T0* T511x19546116(GE_context* ac, T0* C);
/* Call to ET_STANDALONE_CLOSURE.has_validity_error */
extern T1 T511x19546121(GE_context* ac, T0* C);
/* Call to ET_STANDALONE_CLOSURE.implementation_feature */
extern T0* T511x18432013(GE_context* ac, T0* C);
/* Call to ET_STANDALONE_CLOSURE.postconditions */
extern T0* T511x18432010(GE_context* ac, T0* C);
/* Call to ET_STANDALONE_CLOSURE.preconditions */
extern T0* T511x18432009(GE_context* ac, T0* C);
/* Call to ET_STANDALONE_CLOSURE.locals */
extern T0* T511x18432008(GE_context* ac, T0* C);
/* Call to ET_STANDALONE_CLOSURE.arguments */
extern T0* T511x18432006(GE_context* ac, T0* C);
/* Call to ET_STANDALONE_CLOSURE.type */
extern T0* T511x18432005(GE_context* ac, T0* C);
/* Call to ET_STANDALONE_CLOSURE.clients */
extern T0* T511x19546119(GE_context* ac, T0* C);
/* Call to ET_STANDALONE_CLOSURE.implementation_class */
extern T0* T511x19546117(GE_context* ac, T0* C);
/* Call to ET_ASSERTION.expression */
extern T0* T534x15876098(GE_context* ac, T0* C);
/* Call to ET_ASSERTION.is_class_assertion */
extern T1 T534x15876097(GE_context* ac, T0* C);
/* Call to ET_PRIMARY_GROUP.relative_lower_name */
extern T0* T537x19922964T0T2(GE_context* ac, T0* C, T0* a1, T2 a2);
/* Call to ET_PRIMARY_GROUP.kind_name */
extern T0* T537x19922970(GE_context* ac, T0* C);
/* Call to ET_PRIMARY_GROUP.full_lower_name */
extern T0* T537x19922966T2(GE_context* ac, T0* C, T2 a1);
/* Call to ET_PRIMARY_GROUP.is_cluster */
extern T1 T537x19922947(GE_context* ac, T0* C);
/* Call to ET_PRIMARY_GROUP.use_obsolete_syntax */
extern T1 T537x19922954(GE_context* ac, T0* C);
/* Call to ET_PRIMARY_GROUP.current_system */
extern T0* T537x19922960(GE_context* ac, T0* C);
/* Call to ET_PRIMARY_GROUP.universe */
extern T0* T537x19922959(GE_context* ac, T0* C);
/* Call to ET_PRIMARY_GROUP.cluster */
extern T0* T537x19922983(GE_context* ac, T0* C);
/* Call to ET_PRIMARY_GROUP.is_override */
extern T1 T537x19922952(GE_context* ac, T0* C);
/* Call to ET_PRIMARY_GROUP.dotnet_assembly */
extern T0* T537x19922985(GE_context* ac, T0* C);
/* Call to ET_PRIMARY_GROUP.is_dotnet_assembly */
extern T1 T537x19922949(GE_context* ac, T0* C);
/* Call to ET_PRIMARY_GROUP.is_unknown */
extern T1 T537x19922951(GE_context* ac, T0* C);
/* Call to ET_PRIMARY_GROUP.is_none */
extern T1 T537x19922950(GE_context* ac, T0* C);
/* Call to ET_EXTENDED_FEATURE_NAME.alias_name */
extern T0* T593x21643266(GE_context* ac, T0* C);
/* Call to ET_EXTENDED_FEATURE_NAME.feature_name */
extern T0* T593x21643265(GE_context* ac, T0* C);
/* Call to ET_EXTENDED_FEATURE_NAME.same_call_name */
extern T1 T593x21643268T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_EXTENDED_FEATURE_NAME.position */
extern T0* T593x20955137(GE_context* ac, T0* C);
/* Call to ET_EXTENDED_FEATURE_NAME.same_extended_feature_name */
extern T1 T593x21643267T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_TYPE_MARK.position */
extern T0* T597x20955137(GE_context* ac, T0* C);
/* Call to ET_TYPE_MARK.is_implicit_mark */
extern T1 T597x21397513(GE_context* ac, T0* C);
/* Call to ET_TYPE_MARK.text */
extern T0* T597x21397522(GE_context* ac, T0* C);
/* Call to ET_TYPE_MARK.overridden_type_mark */
extern T0* T597x21397521T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_TYPE_MARK.is_attached_mark */
extern T1 T597x21397511(GE_context* ac, T0* C);
/* Call to ET_TYPE_MARK.is_attachment_mark */
extern T1 T597x21397510(GE_context* ac, T0* C);
/* Call to ET_TYPE_MARK.is_separate_mark */
extern T1 T597x21397509(GE_context* ac, T0* C);
/* Call to ET_TYPE_MARK.is_separateness_mark */
extern T1 T597x21397508(GE_context* ac, T0* C);
/* Call to ET_TYPE_MARK.is_expanded_mark */
extern T1 T597x21397506(GE_context* ac, T0* C);
/* Call to ET_TYPE_MARK.is_expandedness_mark */
extern T1 T597x21397505(GE_context* ac, T0* C);
/* Call to ET_TYPE_MARK.is_reference_mark */
extern T1 T597x21397507(GE_context* ac, T0* C);
/* Call to ET_INTERNAL_UNIVERSE.hash_code */
extern T6 T646x458753(GE_context* ac, T0* C);
/* Call to ET_DOTNET_ASSEMBLY.dotnet_assembly */
extern T0* T649x19709958(GE_context* ac, T0* C);
/* Call to ET_DOTNET_ASSEMBLY.full_lower_name */
extern T0* T649x19922966T2(GE_context* ac, T0* C, T2 a1);
/* Call to ET_DOTNET_ASSEMBLY.hash_code */
extern T6 T649x458753(GE_context* ac, T0* C);
/* Call to ET_DOTNET_ASSEMBLY.is_consumable */
extern T1 T649x19890195(GE_context* ac, T0* C);
/* Call to ET_CLUSTER.full_lower_name */
extern T0* T673x19922966T2(GE_context* ac, T0* C, T2 a1);
/* Call to ET_CLUSTER.scm_write_mapping_recursive */
extern T0* T673x19742760(GE_context* ac, T0* C);
/* Call to ET_CLUSTER.absolute_pathname */
extern T0* T673x19922969(GE_context* ac, T0* C);
/* Call to ET_CLUSTER.dependant_constraint */
extern T0* T673x19742754(GE_context* ac, T0* C);
/* Call to ET_CLUSTER.overridden_constraint_enabled */
extern T1 T673x19742755(GE_context* ac, T0* C);
/* Call to ET_CLUSTER.scm_mapping_constraint_enabled */
extern T1 T673x19742756(GE_context* ac, T0* C);
/* Call to ET_CLUSTER.provider_constraint */
extern T0* T673x19742753(GE_context* ac, T0* C);
/* Call to ET_CLUSTER.is_abstract */
extern T1 T673x19742722(GE_context* ac, T0* C);
/* Call to ET_CLUSTER.is_read_only */
extern T1 T673x19922953(GE_context* ac, T0* C);
/* Call to ET_CLUSTER.is_override */
extern T1 T673x19922952(GE_context* ac, T0* C);
/* Call to ET_CLUSTER.full_pathname */
extern T0* T673x19922967(GE_context* ac, T0* C);
/* Call to ET_CLUSTER.is_implicit */
extern T1 T673x19742730(GE_context* ac, T0* C);
/* Call to ET_CLUSTER.is_preparsed */
extern T1 T673x19922955(GE_context* ac, T0* C);
/* Call to ET_CLUSTER.parent */
extern T0* T673x19922986(GE_context* ac, T0* C);
/* Call to ET_CLUSTER.universe */
extern T0* T673x19922959(GE_context* ac, T0* C);
/* Call to ET_CLUSTER.subclusters */
extern T0* T673x19742752(GE_context* ac, T0* C);
/* Call to ET_CLUSTER.is_valid_directory_name */
extern T1 T673x19742736T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_CLUSTER.is_recursive */
extern T1 T673x19742724(GE_context* ac, T0* C);
/* Call to ET_CLUSTER.is_valid_eiffel_filename */
extern T1 T673x19742735T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_CLUSTER.scm_write_mapping */
extern T0* T673x19742758(GE_context* ac, T0* C);
/* Call to ET_CLUSTER.scm_read_mapping */
extern T0* T673x19742757(GE_context* ac, T0* C);
/* Call to XM_EIFFEL_SCANNER.start_condition */
extern T6 T732x12156933(GE_context* ac, T0* C);
/* Call to XM_EIFFEL_SCANNER.is_applicable_encoding */
extern T1 T732x27754500T0(GE_context* ac, T0* C, T0* a1);
/* Call to XM_EIFFEL_SCANNER.end_of_file */
extern T1 T732x12156939(GE_context* ac, T0* C);
/* Call to XM_EIFFEL_SCANNER.last_value */
extern T0* T732x27754509(GE_context* ac, T0* C);
/* Call to XM_EIFFEL_SCANNER.last_token */
extern T6 T732x12156929(GE_context* ac, T0* C);
/* Call to XM_EIFFEL_SCANNER.error_position */
extern T0* T732x27754507(GE_context* ac, T0* C);
/* Call to ET_CLUSTERS.clusters */
extern T0* T791x19759114(GE_context* ac, T0* C);
/* Call to ET_ACTUAL_PARAMETER.last_leaf */
extern T0* T841x20955141(GE_context* ac, T0* C);
/* Call to ET_ACTUAL_PARAMETER.label */
extern T0* T841x22413315(GE_context* ac, T0* C);
/* Call to ET_ACTUAL_PARAMETER.position */
extern T0* T841x20955137(GE_context* ac, T0* C);
/* Call to ET_ACTUAL_PARAMETER.named_parameter_with_type */
extern T0* T841x22364164T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_ACTUAL_PARAMETER.type */
extern T0* T841x22413314(GE_context* ac, T0* C);
/* Call to ET_NAMED_TYPE.to_text */
extern T0* T844x23003206(GE_context* ac, T0* C);
/* Call to ET_NAMED_TYPE.same_base_type_with_type_marks */
extern T1 T844x23003181T0T0T0T0T0(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3, T0* a4, T0* a5);
/* Call to ET_NAMED_TYPE.base_type_index_of_label */
extern T6 T844x23003147T0T0(GE_context* ac, T0* C, T0* a1, T0* a2);
/* Call to ET_NAMED_TYPE.same_syntactical_like_current_with_type_marks */
extern T1 T844x23003184T0T0T0T0T0(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3, T0* a4, T0* a5);
/* Call to ET_NAMED_TYPE.same_syntactical_tuple_type_with_type_marks */
extern T1 T844x23003187T0T0T0T0T0(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3, T0* a4, T0* a5);
/* Call to ET_NAMED_TYPE.same_syntactical_class_type_with_type_marks */
extern T1 T844x23003182T0T0T0T0T0(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3, T0* a4, T0* a5);
/* Call to ET_NAMED_TYPE.same_syntactical_like_feature_with_type_marks */
extern T1 T844x23003185T0T0T0T0T0(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3, T0* a4, T0* a5);
/* Call to ET_NAMED_TYPE.base_type_actual */
extern T0* T844x23003145T6T0(GE_context* ac, T0* C, T6 a1, T0* a2);
/* Call to ET_NAMED_TYPE.same_syntactical_qualified_like_identifier_with_type_marks */
extern T1 T844x23003186T0T0T0T0T0(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3, T0* a4, T0* a5);
/* Call to ET_NAMED_TYPE.type_with_type_mark */
extern T0* T844x23003154T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_NAMED_TYPE.has_formal_types */
extern T1 T844x23003171T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_NAMED_TYPE.same_syntactical_type_with_type_marks */
extern T1 T844x23003177T0T0T0T0T0(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3, T0* a4, T0* a5);
/* Call to ET_NAMED_TYPE.same_syntactical_formal_parameter_type_with_type_marks */
extern T1 T844x23003183T0T0T0T0T0(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3, T0* a4, T0* a5);
/* Call to ET_NAMED_TYPE.is_type_attached_with_type_mark */
extern T1 T844x23003165T0T0(GE_context* ac, T0* C, T0* a1, T0* a2);
/* Call to ET_NAMED_TYPE.is_type_reference_with_type_mark */
extern T1 T844x23003163T0T0(GE_context* ac, T0* C, T0* a1, T0* a2);
/* Call to ET_NAMED_TYPE.is_type_detachable_with_type_mark */
extern T1 T844x23003167T0T0(GE_context* ac, T0* C, T0* a1, T0* a2);
/* Call to ET_NAMED_TYPE.same_base_tuple_type_with_type_marks */
extern T1 T844x23003193T0T0T0T0T0(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3, T0* a4, T0* a5);
/* Call to ET_NAMED_TYPE.same_base_class_type_with_type_marks */
extern T1 T844x23003191T0T0T0T0T0(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3, T0* a4, T0* a5);
/* Call to ET_NAMED_TYPE.is_type_expanded_with_type_mark */
extern T1 T844x23003161T0T0(GE_context* ac, T0* C, T0* a1, T0* a2);
/* Call to ET_NAMED_TYPE.same_named_formal_parameter_type_with_type_marks */
extern T1 T844x23003189T0T0T0T0T0(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3, T0* a4, T0* a5);
/* Call to ET_NAMED_TYPE.conforms_from_formal_parameter_type_with_type_marks */
extern T1 T844x23003197T0T0T0T0T0T0(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3, T0* a4, T0* a5, T0* a6);
/* Call to ET_NAMED_TYPE.conforms_from_tuple_type_with_type_marks */
extern T1 T844x23003198T0T0T0T0T0T0(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3, T0* a4, T0* a5, T0* a6);
/* Call to ET_NAMED_TYPE.conforms_from_class_type_with_type_marks */
extern T1 T844x23003196T0T0T0T0T0T0(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3, T0* a4, T0* a5, T0* a6);
/* Call to ET_NAMED_TYPE.conforms_to_type_with_type_marks */
extern T1 T844x23003195T0T0T0T0T0T0(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3, T0* a4, T0* a5, T0* a6);
/* Call to ET_NAMED_TYPE.same_named_tuple_type_with_type_marks */
extern T1 T844x23003190T0T0T0T0T0(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3, T0* a4, T0* a5);
/* Call to ET_NAMED_TYPE.same_named_class_type_with_type_marks */
extern T1 T844x23003188T0T0T0T0T0(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3, T0* a4, T0* a5);
/* Call to ET_NAMED_TYPE.named_base_class */
extern T0* T844x23003140T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_NAMED_TYPE.same_named_type_with_type_marks */
extern T1 T844x23003179T0T0T0T0T0(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3, T0* a4, T0* a5);
/* Call to ET_NAMED_TYPE.base_class */
extern T0* T844x23003139T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable ET_CHARACTER_CONSTANT.value */
extern T3 T889x17022979(GE_context* ac, T0* C);
/* Call to detachable ET_INTEGER_CONSTANT.to_integer_32 */
extern T6 T898x17104901(GE_context* ac, T0* C);
/* Call to detachable ET_MANIFEST_STRING.value */
extern T0* T904x17121283(GE_context* ac, T0* C);
/* Call to detachable ET_ALIAS_NAME.alias_lower_name */
extern T0* T955x21463073(GE_context* ac, T0* C);
/* Call to detachable ET_ALIAS_NAME.is_prefixable */
extern T1 T955x21463068(GE_context* ac, T0* C);
/* Call to detachable ET_ALIAS_NAME.is_infixable */
extern T1 T955x21463069(GE_context* ac, T0* C);
/* Call to detachable ET_ALIAS_NAME.is_infix */
extern T1 T955x21544971(GE_context* ac, T0* C);
/* Call to detachable ET_ALIAS_NAME.is_parenthesis */
extern T1 T955x21544997(GE_context* ac, T0* C);
/* Call to detachable ET_ALIAS_NAME.is_bracket */
extern T1 T955x21544996(GE_context* ac, T0* C);
/* Call to detachable ET_ALIAS_NAME.same_alias_name */
extern T1 T955x21463083T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable ET_ASSIGNER.feature_name */
extern T0* T958x18382849(GE_context* ac, T0* C);
/* Call to detachable ET_CONSTANT.is_integer_constant */
extern T1 T998x17039363(GE_context* ac, T0* C);
/* Call to detachable ET_CONSTANT.is_character_constant */
extern T1 T998x17039362(GE_context* ac, T0* C);
/* Call to detachable ET_CONSTRAINT_TYPE.resolved_syntactical_constraint */
extern T0* T1010x22708226T0T0T0(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* Call to detachable ET_CONVERT_FEATURE.name */
extern T0* T1013x16252931(GE_context* ac, T0* C);
/* Call to detachable ET_CONVERT_FEATURE.is_convert_to */
extern T1 T1013x16252930(GE_context* ac, T0* C);
/* Call to detachable ET_CONVERT_FEATURE.is_convert_from */
extern T1 T1013x16252929(GE_context* ac, T0* C);
/* Call to detachable ET_EXTERNAL_ALIAS.manifest_string */
extern T0* T1062x18743297(GE_context* ac, T0* C);
/* Call to ET_FLATTENED_FEATURE.adapted_feature */
extern T0* T1241x18923537(GE_context* ac, T0* C);
/* Call to ET_FLATTENED_FEATURE.is_adapted */
extern T1 T1241x18923524(GE_context* ac, T0* C);
/* Call to ET_FLATTENED_FEATURE.inherited_feature */
extern T0* T1241x18923535(GE_context* ac, T0* C);
/* Call to ET_FLATTENED_FEATURE.is_inherited */
extern T1 T1241x18923522(GE_context* ac, T0* C);
/* Call to ET_FLATTENED_FEATURE.redeclared_feature */
extern T0* T1241x18923536(GE_context* ac, T0* C);
/* Call to ET_FLATTENED_FEATURE.is_redeclared */
extern T1 T1241x18923523(GE_context* ac, T0* C);
/* Call to ET_FLATTENED_FEATURE.is_immediate */
extern T1 T1241x18923521(GE_context* ac, T0* C);
/* Call to ET_FLATTENED_FEATURE.flattened_feature */
extern T0* T1241x18923533(GE_context* ac, T0* C);
/* Call to ET_FLATTENED_FEATURE.immediate_feature */
extern T0* T1241x18923534(GE_context* ac, T0* C);
/* Call to ET_CLOSURE.type */
extern T0* T1360x18432005(GE_context* ac, T0* C);
/* Call to ET_CLOSURE.locals */
extern T0* T1360x18432008(GE_context* ac, T0* C);
/* Call to ET_CLOSURE.arguments */
extern T0* T1360x18432006(GE_context* ac, T0* C);
/* Call to ET_CLOSURE.across_components */
extern T0* T1360x18432012(GE_context* ac, T0* C);
/* Call to ET_CLOSURE.object_tests */
extern T0* T1360x18432011(GE_context* ac, T0* C);
/* Call to ET_INLINE_AGENT.position */
extern T0* T1362x20955137(GE_context* ac, T0* C);
/* Call to ET_INLINE_AGENT.implicit_result */
extern T0* T1362x15400964(GE_context* ac, T0* C);
/* Call to ET_INLINE_AGENT.formal_arguments */
extern T0* T1362x18432006(GE_context* ac, T0* C);
/* Call to ET_INLINE_AGENT.actual_arguments */
extern T0* T1362x15400963(GE_context* ac, T0* C);
/* Call to ET_INLINE_AGENT.target */
extern T0* T1362x15400962(GE_context* ac, T0* C);
/* Call to ET_INLINE_AGENT.locals */
extern T0* T1362x18432008(GE_context* ac, T0* C);
/* Call to ET_INLINE_AGENT.type */
extern T0* T1362x18432005(GE_context* ac, T0* C);
/* Call to ET_INDEXING_TERM.indexing_term_value */
extern T0* T1403x16646146(GE_context* ac, T0* C);
/* Call to ET_INDEXING_TERM.position */
extern T0* T1403x20955137(GE_context* ac, T0* C);
/* Call to ET_INDEXING_TERM.has_indexing_term_value */
extern T1 T1403x16646147T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_QUALIFIED_FEATURE_CALL_EXPRESSION.name */
extern T0* T1529x17465346(GE_context* ac, T0* C);
/* Call to ET_QUALIFIED_FEATURE_CALL_EXPRESSION.target */
extern T0* T1529x17465345(GE_context* ac, T0* C);
/* Call to ET_QUALIFIED_FEATURE_CALL_EXPRESSION.arguments_count */
extern T6 T1529x20217859(GE_context* ac, T0* C);
/* Call to ET_QUALIFIED_FEATURE_CALL_EXPRESSION.parenthesis_call */
extern T0* T1529x20496387(GE_context* ac, T0* C);
/* Call to ET_QUALIFIED_FEATURE_CALL_EXPRESSION.arguments */
extern T0* T1529x17465347(GE_context* ac, T0* C);
/* Call to ET_QUALIFIED_FEATURE_CALL_EXPRESSION.index */
extern T6 T1529x18071555(GE_context* ac, T0* C);
/* Call to ET_QUALIFIED_FEATURE_CALL_EXPRESSION.is_call_agent */
extern T1 T1529x17465350(GE_context* ac, T0* C);
/* Call to ET_QUALIFIED_FEATURE_CALL_EXPRESSION.is_tuple_label */
extern T1 T1529x17465349(GE_context* ac, T0* C);
/* Call to XM_DOCUMENT_NODE.parent */
extern T0* T1541x28672001(GE_context* ac, T0* C);
/* Call to ET_ECF_CONDITION_ITEM.is_enabled */
extern T1 T1561x14303233T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_ECF_CONDITION_ITEM.is_capability_supported */
extern T1 T1561x14303234T0T0T0(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* Call to ET_ECF_CONDITION_ITEM.is_capability_aware */
extern T1 T1561x14352385T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_ADAPTED_UNIVERSE.lower_name */
extern T0* T1565x19709955(GE_context* ac, T0* C);
/* Call to ET_ADAPTED_UNIVERSE.name */
extern T0* T1565x19709954(GE_context* ac, T0* C);
/* Call to ET_ALIAS_NAME.same_call_name */
extern T1 T1573x21545000T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_ALIAS_NAME.same_alias_name */
extern T1 T1573x21463083T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_ALIAS_NAME.is_infix_minus */
extern T1 T1573x21544982(GE_context* ac, T0* C);
/* Call to ET_ALIAS_NAME.is_infix_plus */
extern T1 T1573x21544986(GE_context* ac, T0* C);
/* Call to ET_ALIAS_NAME.is_prefix_minus */
extern T1 T1573x21544992(GE_context* ac, T0* C);
/* Call to ET_ALIAS_NAME.is_prefix_plus */
extern T1 T1573x21544993(GE_context* ac, T0* C);
/* Call to ET_ALIAS_NAME.hash_code */
extern T6 T1573x458753(GE_context* ac, T0* C);
/* Call to ET_ALIAS_NAME.is_prefix */
extern T1 T1573x21544991(GE_context* ac, T0* C);
/* Call to ET_ALIAS_NAME.is_parenthesis */
extern T1 T1573x21544997(GE_context* ac, T0* C);
/* Call to ET_ALIAS_NAME.is_bracket */
extern T1 T1573x21544996(GE_context* ac, T0* C);
/* Call to ET_ALIAS_NAME.is_infix */
extern T1 T1573x21544971(GE_context* ac, T0* C);
/* Call to ET_ALIAS_NAME.is_infixable */
extern T1 T1573x21463069(GE_context* ac, T0* C);
/* Call to ET_ALIAS_NAME.is_prefixable */
extern T1 T1573x21463068(GE_context* ac, T0* C);
/* Call to ET_INDEXING_TERM_ITEM.indexing_term */
extern T0* T1606x16646145(GE_context* ac, T0* C);
/* Call to ET_INDEXING_TERM_ITEM.position */
extern T0* T1606x20955137(GE_context* ac, T0* C);
/* Call to ET_INDEXING_TERM_ITEM.has_indexing_term_value */
extern T1 T1606x16646147T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_FEATURE_NAME_ITEM.feature_name */
extern T0* T1610x21643265(GE_context* ac, T0* C);
/* Call to ET_FEATURE_NAME_ITEM.position */
extern T0* T1610x20955137(GE_context* ac, T0* C);
/* Call to ET_CONSTRAINT_TYPE.resolved_syntactical_constraint */
extern T0* T1614x22708226T0T0T0(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* Call to ET_CONSTRAINT_ACTUAL_PARAMETER_ITEM.resolved_syntactical_constraint_with_type */
extern T0* T1616x22593540T0T0(GE_context* ac, T0* C, T0* a1, T0* a2);
/* Call to ET_CONSTRAINT_ACTUAL_PARAMETER_ITEM.type */
extern T0* T1616x22593538(GE_context* ac, T0* C);
/* Call to ET_CONSTRAINT_ACTUAL_PARAMETER_ITEM.position */
extern T0* T1616x22593539(GE_context* ac, T0* C);
/* Call to ET_CONVERT_FEATURE.name */
extern T0* T1626x16252931(GE_context* ac, T0* C);
/* Call to ET_TYPE_ITEM.type */
extern T0* T1630x23052289(GE_context* ac, T0* C);
/* Call to detachable ET_TARGET_TYPE.type */
extern T0* T1673x22970369(GE_context* ac, T0* C);
/* Call to ET_CHOICE.upper */
extern T0* T1678x20267011(GE_context* ac, T0* C);
/* Call to ET_CHOICE.is_range */
extern T1 T1678x20267009(GE_context* ac, T0* C);
/* Call to ET_CHOICE.lower */
extern T0* T1678x20267010(GE_context* ac, T0* C);
/* Call to ET_CHOICE_ITEM.choice */
extern T0* T1679x20316161(GE_context* ac, T0* C);
/* Call to ET_MANIFEST_STRING_ITEM.manifest_string */
extern T0* T1681x17940481(GE_context* ac, T0* C);
/* Call to ET_AGENT_ARGUMENT_OPERAND_ITEM.agent_actual_argument */
extern T0* T1710x15466497(GE_context* ac, T0* C);
/* Call to ET_ASSERTION_ITEM.assertion */
extern T0* T1717x15908865(GE_context* ac, T0* C);
/* Call to ET_ASSERTION_ITEM.position */
extern T0* T1717x20955137(GE_context* ac, T0* C);
/* Call to detachable ET_FLATTENED_FEATURE.flattened_feature */
extern T0* T1720x18923533(GE_context* ac, T0* C);
/* Call to detachable ET_FLATTENED_FEATURE.inherited_feature */
extern T0* T1720x18923535(GE_context* ac, T0* C);
/* Call to detachable ET_FLATTENED_FEATURE.is_inherited */
extern T1 T1720x18923522(GE_context* ac, T0* C);
/* Call to ET_ASSIGNER.feature_name */
extern T0* T1725x18382849(GE_context* ac, T0* C);
/* Call to detachable ET_DYNAMIC_TYPES.dynamic_type */
extern T0* T1769x24657921T6(GE_context* ac, T0* C, T6 a1);
/* Call to detachable ET_DYNAMIC_TYPES.count */
extern T6 T1769x24657924(GE_context* ac, T0* C);
/* Call to detachable ET_DYNAMIC_TYPES.index_of */
extern T6 T1769x24657923T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable ET_DYNAMIC_TYPES.has_type */
extern T1 T1769x24657926T0(GE_context* ac, T0* C, T0* a1);
/* Call to [attached] DS_ARRAYED_LIST [[attached] ET_CLUSTER].item */
extern T0* T1849x9125890T6(GE_context* ac, T0* C, T6 a1);
/* Call to [attached] DS_ARRAYED_LIST [[attached] ET_CLUSTER].count */
extern T6 T1849x9060353(GE_context* ac, T0* C);
/* Call to ET_FORMAL_PARAMETER_TYPE.index */
extern T6 T1895x22740995(GE_context* ac, T0* C);
/* Call to ET_FORMAL_PARAMETER_TYPE.implementation_class */
extern T0* T1895x22740996(GE_context* ac, T0* C);
/* Call to ET_FORMAL_PARAMETER_TYPE.is_type_detachable_with_type_mark */
extern T1 T1895x23003167T0T0(GE_context* ac, T0* C, T0* a1, T0* a2);
/* Call to ET_FORMAL_PARAMETER_TYPE.is_type_attached_with_type_mark */
extern T1 T1895x23003165T0T0(GE_context* ac, T0* C, T0* a1, T0* a2);
/* Call to ET_FORMAL_PARAMETER_TYPE.overridden_type_mark */
extern T0* T1895x23003138T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_FORMAL_PARAMETER_TYPE.name */
extern T0* T1895x22888449(GE_context* ac, T0* C);
/* Call to ET_FORMAL_PARAMETER_TYPE.type */
extern T0* T1895x23052289(GE_context* ac, T0* C);
/* Call to ET_REPLICABLE_FEATURE.first_feature */
extern T0* T1929x19398659(GE_context* ac, T0* C);
/* Call to ET_REPLICABLE_FEATURE.replicated_feature */
extern T0* T1929x19398658(GE_context* ac, T0* C);
/* Call to ET_REPLICABLE_FEATURE.has_replication */
extern T1 T1929x19398657(GE_context* ac, T0* C);
/* Call to ET_REPLICABLE_FEATURE.selected_count */
extern T6 T1929x19398660(GE_context* ac, T0* C);
/* Call to XM_ELEMENT_NODE.parent */
extern T0* T2005x28672001(GE_context* ac, T0* C);
/* Call to detachable ET_FORMAL_PARAMETER_TYPE.index */
extern T6 T2026x22740995(GE_context* ac, T0* C);
/* Call to ET_INTEGER_CONSTANT.cast_type */
extern T0* T2030x17104908(GE_context* ac, T0* C);
/* Call to ET_INTEGER_CONSTANT.index */
extern T6 T2030x18071555(GE_context* ac, T0* C);
/* Call to ET_INTEGER_CONSTANT.is_negative */
extern T1 T2030x17104915(GE_context* ac, T0* C);
/* Call to ET_INTEGER_CONSTANT.value */
extern T11 T2030x17104898(GE_context* ac, T0* C);
/* Call to ET_INTEGER_CONSTANT.literal */
extern T0* T2030x17104897(GE_context* ac, T0* C);
/* Call to ET_INTEGER_CONSTANT.sign */
extern T0* T2030x17104907(GE_context* ac, T0* C);
/* Call to ET_INTEGER_CONSTANT.value_position */
extern T0* T2030x17104912(GE_context* ac, T0* C);
/* Call to ET_INTEGER_CONSTANT.is_natural_64 */
extern T1 T2030x17104927(GE_context* ac, T0* C);
/* Call to ET_INTEGER_CONSTANT.is_binary */
extern T1 T2030x17104918(GE_context* ac, T0* C);
/* Call to ET_INTEGER_CONSTANT.is_hexadecimal */
extern T1 T2030x17104917(GE_context* ac, T0* C);
/* Call to ET_INTEGER_CONSTANT.is_integer_64 */
extern T1 T2030x17104923(GE_context* ac, T0* C);
/* Call to ET_INTEGER_CONSTANT.is_integer_32 */
extern T1 T2030x17104922(GE_context* ac, T0* C);
/* Call to ET_INTEGER_CONSTANT.is_natural_32 */
extern T1 T2030x17104926(GE_context* ac, T0* C);
/* Call to ET_INTEGER_CONSTANT.is_natural_16 */
extern T1 T2030x17104925(GE_context* ac, T0* C);
/* Call to ET_INTEGER_CONSTANT.is_natural_8 */
extern T1 T2030x17104924(GE_context* ac, T0* C);
/* Call to ET_INTEGER_CONSTANT.is_integer_16 */
extern T1 T2030x17104921(GE_context* ac, T0* C);
/* Call to ET_INTEGER_CONSTANT.is_integer_8 */
extern T1 T2030x17104920(GE_context* ac, T0* C);
/* Call to ET_DECLARED_TYPE.last_leaf */
extern T0* T2078x20955141(GE_context* ac, T0* C);
/* Call to ET_DECLARED_TYPE.type */
extern T0* T2078x22724609(GE_context* ac, T0* C);
/* Call to ET_LIKE_TYPE.to_text */
extern T0* T2099x23003206(GE_context* ac, T0* C);
/* Call to ET_LIKE_TYPE.position */
extern T0* T2099x20955137(GE_context* ac, T0* C);
/* Call to ET_LIKE_TYPE.resolved_formal_parameters */
extern T0* T2099x23003199T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_LIKE_TYPE.like_keyword */
extern T0* T2099x22872066(GE_context* ac, T0* C);
/* Call to ET_LIKE_TYPE.has_formal_types */
extern T1 T2099x23003171T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_CONDITIONAL.expression */
extern T0* T2105x17498113(GE_context* ac, T0* C);
/* Call to ET_CONDITIONAL.position */
extern T0* T2105x20955137(GE_context* ac, T0* C);
/* Call to ET_CHOICE_CONSTANT.position */
extern T0* T2117x20955137(GE_context* ac, T0* C);
/* Call to ET_OPERATOR.is_tuple_label */
extern T1 T2129x21544969(GE_context* ac, T0* C);
/* Call to ET_OPERATOR.is_prefix_not */
extern T1 T2129x21544995(GE_context* ac, T0* C);
/* Call to ET_OPERATOR.is_infix_and_then */
extern T1 T2129x21544973(GE_context* ac, T0* C);
/* Call to ET_OPERATOR.is_infix_and */
extern T1 T2129x21544972(GE_context* ac, T0* C);
/* Call to ET_OPERATOR.is_infix_implies */
extern T1 T2129x21544979(GE_context* ac, T0* C);
/* Call to ET_OPERATOR.is_infix_or_else */
extern T1 T2129x21544985(GE_context* ac, T0* C);
/* Call to ET_OPERATOR.is_infix_or */
extern T1 T2129x21544984(GE_context* ac, T0* C);
/* Call to ET_OPERATOR.position */
extern T0* T2129x20955137(GE_context* ac, T0* C);
/* Call to ET_EXTERNAL_LANGUAGE.manifest_string */
extern T0* T2143x18776065(GE_context* ac, T0* C);
/* Call to ET_AST_LEAF.last_position */
extern T0* T2165x20955139(GE_context* ac, T0* C);
/* Call to ET_GROUP.full_lower_name */
extern T0* T2171x19922966T2(GE_context* ac, T0* C, T2 a1);
/* Call to ET_GROUP.kind_lower_name */
extern T0* T2171x19922971(GE_context* ac, T0* C);
/* Call to ET_GROUP.is_cluster */
extern T1 T2171x19922947(GE_context* ac, T0* C);
/* Call to ET_GROUP.absolute_pathname */
extern T0* T2171x19922969(GE_context* ac, T0* C);
/* Call to ET_GROUP.universe */
extern T0* T2171x19922959(GE_context* ac, T0* C);
/* Call to ET_GROUP.parent */
extern T0* T2171x19922986(GE_context* ac, T0* C);
/* Call to ET_GROUP.full_name */
extern T0* T2171x19922965T2(GE_context* ac, T0* C, T2 a1);
/* Call to ET_GROUP.cluster */
extern T0* T2171x19922983(GE_context* ac, T0* C);
/* Call to ET_ECF_GROUP.class_options */
extern T0* T2381x14811142(GE_context* ac, T0* C);
/* Call to ET_ECF_GROUP.options */
extern T0* T2381x14811141(GE_context* ac, T0* C);
/* Call to ET_INTERNAL_ROUTINE.implementation_feature */
extern T0* T2391x18432013(GE_context* ac, T0* C);
/* Call to ET_INTERNAL_ROUTINE.is_once_per_process */
extern T1 T2391x18432002(GE_context* ac, T0* C);
/* Call to ET_INTERNAL_ROUTINE.is_once */
extern T1 T2391x18432001(GE_context* ac, T0* C);
/* Call to ET_INTERNAL_ROUTINE.lower_name */
extern T0* T2391x19546115(GE_context* ac, T0* C);
/* Call to ET_INTERNAL_ROUTINE.rescue_clause */
extern T0* T2391x19103747(GE_context* ac, T0* C);
/* Call to ET_INTERNAL_ROUTINE.compound */
extern T0* T2391x19103745(GE_context* ac, T0* C);
/* Call to ET_INTERNAL_ROUTINE.arguments */
extern T0* T2391x18923530(GE_context* ac, T0* C);
/* Call to ET_ROUTINE.implementation_feature */
extern T0* T2392x18432013(GE_context* ac, T0* C);
/* Call to ET_ROUTINE.arguments */
extern T0* T2392x18923530(GE_context* ac, T0* C);
/* Call to ET_INTERNAL_ROUTINE_CLOSURE.locals */
extern T0* T2393x18432008(GE_context* ac, T0* C);
/* Call to ET_INTERNAL_ROUTINE_CLOSURE.compound */
extern T0* T2393x19103745(GE_context* ac, T0* C);
/* Call to ET_INTERNAL_ROUTINE_CLOSURE.is_once_per_process */
extern T1 T2393x18432002(GE_context* ac, T0* C);
/* Call to ET_INTERNAL_ROUTINE_CLOSURE.rescue_clause */
extern T0* T2393x19103747(GE_context* ac, T0* C);
/* Call to detachable ET_AST_NODE.position */
extern T0* T2412x20955137(GE_context* ac, T0* C);
/* Call to ET_FREE_NAME.operator_name */
extern T0* T2435x21790722(GE_context* ac, T0* C);
/* Call to ET_FREE_NAME.hash_code */
extern T6 T2435x458753(GE_context* ac, T0* C);
/* Call to ET_FREE_NAME.is_infix_freeop */
extern T1 T2435x21544976(GE_context* ac, T0* C);
/* Call to ET_UNTAGGED_ASSERTION.is_class_assertion */
extern T1 T2437x15876097(GE_context* ac, T0* C);
/* Call to ET_CONVERT_EXPRESSION.expression */
extern T0* T2490x17530881(GE_context* ac, T0* C);
/* Call to ET_CONVERT_EXPRESSION.convert_feature */
extern T0* T2490x17530882(GE_context* ac, T0* C);
/* Call to ET_CHARACTER_CONSTANT.cast_type */
extern T0* T2571x17022980(GE_context* ac, T0* C);
/* Call to ET_CHARACTER_CONSTANT.index */
extern T6 T2571x18071555(GE_context* ac, T0* C);
/* Call to ET_CHARACTER_CONSTANT.value */
extern T3 T2571x17022979(GE_context* ac, T0* C);
/* Call to ET_QUALIFIED_FEATURE_CALL_INSTRUCTION.target */
extern T0* T2573x17465345(GE_context* ac, T0* C);
/* Call to ET_QUALIFIED_FEATURE_CALL_INSTRUCTION.name */
extern T0* T2573x17465346(GE_context* ac, T0* C);
/* Call to ET_QUALIFIED_FEATURE_CALL_INSTRUCTION.is_call_agent */
extern T1 T2573x17465350(GE_context* ac, T0* C);
/* Call to ET_QUALIFIED_FEATURE_CALL_INSTRUCTION.is_tuple_label */
extern T1 T2573x17465349(GE_context* ac, T0* C);
/* Call to ET_QUALIFIED_FEATURE_CALL_INSTRUCTION.arguments */
extern T0* T2573x17465347(GE_context* ac, T0* C);
/* Call to ET_INTERNAL_ROUTINE_INLINE_AGENT.implicit_result */
extern T0* T2574x15400964(GE_context* ac, T0* C);
/* Call to ET_EXTERNAL_ALIAS.manifest_string */
extern T0* T2576x18743297(GE_context* ac, T0* C);
/* Call to detachable ET_CONVERT_EXPRESSION.index */
extern T6 T2599x18071555(GE_context* ac, T0* C);
/* Call to ET_QUERY_INLINE_AGENT.type */
extern T0* T2608x18432005(GE_context* ac, T0* C);
/* Call to ET_QUERY_INLINE_AGENT.formal_arguments */
extern T0* T2608x18432006(GE_context* ac, T0* C);
/* Call to ET_PROCEDURE_INLINE_AGENT.formal_arguments */
extern T0* T2611x18432006(GE_context* ac, T0* C);
/* Call to ET_CALL_WITH_ACTUAL_ARGUMENTS.arguments */
extern T0* T2614x20217858(GE_context* ac, T0* C);
/* Call to ET_CALL_WITH_ACTUAL_ARGUMENTS.name */
extern T0* T2614x20217857(GE_context* ac, T0* C);
/* Call to ET_REGULAR_FEATURE_CALL.parenthesis_call */
extern T0* T2616x20496387(GE_context* ac, T0* C);
/* Call to ET_REGULAR_FEATURE_CALL.name */
extern T0* T2616x17465346(GE_context* ac, T0* C);
/* Call to ET_REGULAR_FEATURE_CALL.target */
extern T0* T2616x17465345(GE_context* ac, T0* C);
/* Call to ET_REGULAR_FEATURE_CALL.arguments */
extern T0* T2616x17465347(GE_context* ac, T0* C);
/* Call to ET_QUALIFIED_FEATURE_CALL.target */
extern T0* T2617x17465345(GE_context* ac, T0* C);
/* Call to ET_QUALIFIED_FEATURE_CALL.name */
extern T0* T2617x17465346(GE_context* ac, T0* C);
/* Call to ET_FEATURE_CALL_EXPRESSION.arguments */
extern T0* T2618x17465347(GE_context* ac, T0* C);
/* Call to ET_CALL_WITH_ACTUAL_ARGUMENT_LIST.arguments */
extern T0* T2620x20217858(GE_context* ac, T0* C);
/* Call to ET_LIKE_IDENTIFIER.name */
extern T0* T2623x22839297(GE_context* ac, T0* C);
/* Call to ET_LIKE_IDENTIFIER.to_text */
extern T0* T2623x23003206(GE_context* ac, T0* C);
/* Call to ET_AST_NODE.position */
extern T0* T2669x20955137(GE_context* ac, T0* C);
/* Call to ET_ARGUMENT_OPERANDS.actual_argument */
extern T0* T2683x17416197T6(GE_context* ac, T0* C, T6 a1);
/* Call to ET_ARGUMENT_OPERANDS.count */
extern T6 T2683x17416198(GE_context* ac, T0* C);
/* Call to detachable LX_TRANSITION [[attached] LX_NFA_STATE].target */
extern T0* T2789x12435457(GE_context* ac, T0* C);
/* Call to detachable LX_TRANSITION [[attached] LX_NFA_STATE].labeled */
extern T1 T2789x12435458T6(GE_context* ac, T0* C, T6 a1);
/* Call to EXCEPTION.set_type_name */
extern void T22x1556504T0(GE_context* ac, T0* C, T0* a1);
/* Call to EXCEPTION.set_recipient_name */
extern void T22x1556507T0(GE_context* ac, T0* C, T0* a1);
/* Call to EXCEPTION.set_description */
extern void T22x1556509T0(GE_context* ac, T0* C, T0* a1);
/* Call to EXCEPTION.set_exception_trace */
extern void T22x1556503T0(GE_context* ac, T0* C, T0* a1);
/* Call to EXCEPTION.set_throwing_exception */
extern void T22x1556508T0(GE_context* ac, T0* C, T0* a1);
/* Call to KI_CHARACTER_INPUT_STREAM.close */
extern void T39x5849098(GE_context* ac, T0* C);
/* Call to KI_CHARACTER_INPUT_STREAM.read_character */
extern void T39x5849100(GE_context* ac, T0* C);
/* Call to ET_SYSTEM.register_class */
extern void T40x20037697T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_SYSTEM.set_iteration_cursor_forth_seed */
extern void T40x20037698T6(GE_context* ac, T0* C, T6 a1);
/* Call to ET_SYSTEM.set_iteration_cursor_after_seed */
extern void T40x20037699T6(GE_context* ac, T0* C, T6 a1);
/* Call to ET_SYSTEM.set_iterable_new_cursor_seed */
extern void T40x20037700T6(GE_context* ac, T0* C, T6 a1);
/* Call to ET_SYSTEM.set_dispose_seed */
extern void T40x20037703T6(GE_context* ac, T0* C, T6 a1);
/* Call to ET_SYSTEM.set_function_item_seed */
extern void T40x20037701T6(GE_context* ac, T0* C, T6 a1);
/* Call to ET_SYSTEM.set_routine_call_seed */
extern void T40x20037702T6(GE_context* ac, T0* C, T6 a1);
/* Call to ET_SYSTEM.set_is_equal_seed */
extern void T40x20037704T6(GE_context* ac, T0* C, T6 a1);
/* Call to ET_SYSTEM.set_copy_seed */
extern void T40x20037705T6(GE_context* ac, T0* C, T6 a1);
/* Call to ET_SYSTEM.set_default_create_seed */
extern void T40x20037706T6(GE_context* ac, T0* C, T6 a1);
/* Call to ET_SYSTEM.master_classes_do_recursive_until */
extern void T40x20070538T0T0(GE_context* ac, T0* C, T0* a1, T0* a2);
/* Call to ET_SYSTEM.preparse_recursive */
extern void T40x20070524T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_SYSTEM.classes_do_recursive */
extern void T40x20070548T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_SYSTEM.classes_do_recursive_until */
extern void T40x20070547T0T0(GE_context* ac, T0* C, T0* a1, T0* a2);
/* Call to AP_OPTION.reset */
extern void T51x5275665(GE_context* ac, T0* C);
/* Call to KI_TEXT_OUTPUT_STREAM.put_integer */
extern void T96x5750798T6(GE_context* ac, T0* C, T6 a1);
/* Call to KI_TEXT_OUTPUT_STREAM.put_character */
extern void T96x5881862T2(GE_context* ac, T0* C, T2 a1);
/* Call to KI_TEXT_OUTPUT_STREAM.put_new_line */
extern void T96x5963778(GE_context* ac, T0* C);
/* Call to KI_TEXT_OUTPUT_STREAM.put_string */
extern void T96x5750800T0(GE_context* ac, T0* C, T0* a1);
/* Call to KI_TEXT_OUTPUT_STREAM.append */
extern void T96x5881861T0(GE_context* ac, T0* C, T0* a1);
/* Call to KI_TEXT_OUTPUT_STREAM.put_line */
extern void T96x5963779T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable AP_OPTION.record_occurrence */
extern void T174x5275666T0(GE_context* ac, T0* C, T0* a1);
/* Call to XM_CALLBACKS_FILTER.set_next */
extern void T201x26804226T0(GE_context* ac, T0* C, T0* a1);
/* Call to XM_CALLBACKS.on_xml_declaration */
extern void T202x26738701T0T0T1(GE_context* ac, T0* C, T0* a1, T0* a2, T1 a3);
/* Call to XM_CALLBACKS.on_error */
extern void T202x26738700T0(GE_context* ac, T0* C, T0* a1);
/* Call to XM_CALLBACKS.on_start_tag_finish */
extern void T202x26738695(GE_context* ac, T0* C);
/* Call to XM_CALLBACKS.on_finish */
extern void T202x26738702(GE_context* ac, T0* C);
/* Call to XM_CALLBACKS.on_attribute */
extern void T202x26738696T0T0T0T0(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3, T0* a4);
/* Call to XM_CALLBACKS.on_end_tag */
extern void T202x26738694T0T0T0(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* Call to XM_CALLBACKS.on_start_tag */
extern void T202x26738697T0T0T0(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* Call to XM_CALLBACKS.on_content */
extern void T202x26738693T0(GE_context* ac, T0* C, T0* a1);
/* Call to XM_CALLBACKS.on_processing_instruction */
extern void T202x26738699T0T0(GE_context* ac, T0* C, T0* a1, T0* a2);
/* Call to XM_CALLBACKS.on_comment */
extern void T202x26738698T0(GE_context* ac, T0* C, T0* a1);
/* Call to XM_CALLBACKS.on_start */
extern void T202x26738703(GE_context* ac, T0* C);
/* Call to ET_ECF_SYSTEM_CONFIG.set_library_target */
extern void T233x15106066T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_ECF_SYSTEM_CONFIG.set_uuid */
extern void T233x15106065T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_ECF_SYSTEM_CONFIG.set_read_only */
extern void T233x15106069T1(GE_context* ac, T0* C, T1 a1);
/* Call to ET_ECF_SYSTEM_CONFIG.set_targets */
extern void T233x15106067T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_ECF_SYSTEM_CONFIG.set_notes */
extern void T233x15106061T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_ECF_SYSTEM_CONFIG.set_description */
extern void T233x15106062T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_ECF_SYSTEM_CONFIG.set_ecf_version */
extern void T233x15106063T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_ECF_SYSTEM_CONFIG.set_ecf_namespace */
extern void T233x15106064T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_ADAPTED_DOTNET_ASSEMBLY.export_classes */
extern void T266x19709960T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_DYNAMIC_TYPE.put_query_call */
extern void T320x24641589T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_DYNAMIC_TYPE.put_procedure_call */
extern void T320x24641588T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_DYNAMIC_TYPE.put_target */
extern void T320x24739850T0T0(GE_context* ac, T0* C, T0* a1, T0* a2);
/* Call to ET_DYNAMIC_TYPE.put_object_equality_expression */
extern void T320x24641586T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_DYNAMIC_TYPE.put_equality_expression */
extern void T320x24641587T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_DYNAMIC_TYPE.set_static */
extern void T320x24641596T1(GE_context* ac, T0* C, T1 a1);
/* Call to ET_DYNAMIC_TYPE.use_all_attributes */
extern void T320x24641591T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_DYNAMIC_TYPE.set_meta_type */
extern void T320x24641595T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_DYNAMIC_TYPE.set_alive */
extern void T320x24641598(GE_context* ac, T0* C);
/* Call to ET_DYNAMIC_TYPE.set_attribute_position */
extern void T320x24641592T0T6(GE_context* ac, T0* C, T0* a1, T6 a2);
/* Call to ET_DYNAMIC_TYPE.set_next_type */
extern void T320x24641583T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_DYNAMIC_TYPE.set_hash_code */
extern void T320x24641593T6(GE_context* ac, T0* C, T6 a1);
/* Call to ET_DYNAMIC_TYPE.set_id */
extern void T320x24641594T6(GE_context* ac, T0* C, T6 a1);
/* Call to detachable ET_PROCEDURE.set_across_components */
extern void T325x18432014T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable ET_PROCEDURE.set_object_tests */
extern void T325x18432015T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable ET_PROCEDURE.set_frozen_keyword */
extern void T325x18841672T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable ET_QUERY.set_across_components */
extern void T326x18432014T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable ET_QUERY.set_object_tests */
extern void T326x18432015T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable ET_QUERY.set_frozen_keyword */
extern void T326x18841672T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_FEATURE_NAME.reset */
extern void T328x21545008(GE_context* ac, T0* C);
/* Call to ET_FEATURE_NAME.set_seed */
extern void T328x21545007T6(GE_context* ac, T0* C, T6 a1);
/* Call to ET_AST_PROCESSOR.process_convert_to_expression */
extern void T329x25297057T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_AST_PROCESSOR.process_convert_from_expression */
extern void T329x25297059T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_AST_PROCESSOR.process_convert_builtin_expression */
extern void T329x25297063T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_AST_PROCESSOR.process_infix_cast_expression */
extern void T329x25296995T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_AST_PROCESSOR.process_unqualified_call_instruction */
extern void T329x25296904T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_AST_PROCESSOR.process_across_instruction */
extern void T329x25297114T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_AST_PROCESSOR.process_check_instruction */
extern void T329x25297078T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_AST_PROCESSOR.process_static_call_instruction */
extern void T329x25296920T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_AST_PROCESSOR.process_precursor_instruction */
extern void T329x25296943T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_AST_PROCESSOR.process_debug_instruction */
extern void T329x25297048T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_AST_PROCESSOR.process_inspect_instruction */
extern void T329x25296989T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_AST_PROCESSOR.process_if_instruction */
extern void T329x25297002T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_AST_PROCESSOR.process_create_instruction */
extern void T329x25297055T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_AST_PROCESSOR.process_bang_instruction */
extern void T329x25297089T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_AST_PROCESSOR.process_loop_instruction */
extern void T329x25296972T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_AST_PROCESSOR.process_assignment_attempt */
extern void T329x25297094T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_AST_PROCESSOR.process_assignment */
extern void T329x25297095T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_AST_PROCESSOR.process_assigner_instruction */
extern void T329x25297096T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_AST_PROCESSOR.process_retry_instruction */
extern void T329x25296924T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_AST_PROCESSOR.process_constrained_formal_parameter */
extern void T329x25297065T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_AST_PROCESSOR.process_formal_parameter */
extern void T329x25297012T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_AST_PROCESSOR.process_qualified_call_instruction */
extern void T329x25296936T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_AST_PROCESSOR.process_class_type */
extern void T329x25297072T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_AST_PROCESSOR.process_across_expression */
extern void T329x25297115T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_AST_PROCESSOR.process_once_manifest_string */
extern void T329x25296957T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_AST_PROCESSOR.process_old_object_test */
extern void T329x25296960T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_AST_PROCESSOR.process_feature_address */
extern void T329x25297021T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_AST_PROCESSOR.process_like_feature */
extern void T329x25296978T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_AST_PROCESSOR.process_external_procedure */
extern void T329x25297024T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_AST_PROCESSOR.process_external_procedure_inline_agent */
extern void T329x25297023T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_AST_PROCESSOR.process_once_procedure_inline_agent */
extern void T329x25296955T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_AST_PROCESSOR.process_do_procedure_inline_agent */
extern void T329x25297042T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_AST_PROCESSOR.process_external_function_inline_agent */
extern void T329x25297025T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_AST_PROCESSOR.process_once_function_inline_agent */
extern void T329x25296958T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_AST_PROCESSOR.process_do_function_inline_agent */
extern void T329x25297044T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_AST_PROCESSOR.process_call_agent */
extern void T329x25297079T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_AST_PROCESSOR.process_strip_expression */
extern void T329x25296919T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_AST_PROCESSOR.process_manifest_array */
extern void T329x25296970T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_AST_PROCESSOR.process_bracket_expression */
extern void T329x25297084T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_AST_PROCESSOR.process_old_expression */
extern void T329x25296961T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_AST_PROCESSOR.process_prefix_expression */
extern void T329x25296941T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_AST_PROCESSOR.process_manifest_type */
extern void T329x25296966T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_AST_PROCESSOR.process_infix_expression */
extern void T329x25296993T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_AST_PROCESSOR.process_parenthesized_expression */
extern void T329x25296947T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_AST_PROCESSOR.process_expression_address */
extern void T329x25297030T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_AST_PROCESSOR.process_result_address */
extern void T329x25296925T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_AST_PROCESSOR.process_current_address */
extern void T329x25297050T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_AST_PROCESSOR.process_precursor_expression */
extern void T329x25296944T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_AST_PROCESSOR.process_static_call_expression */
extern void T329x25296921T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_AST_PROCESSOR.process_if_expression */
extern void T329x25297003T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_AST_PROCESSOR.process_qualified_like_type */
extern void T329x25296934T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_AST_PROCESSOR.process_qualified_like_braced_type */
extern void T329x25296935T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_AST_PROCESSOR.process_once_procedure */
extern void T329x25296956T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_AST_PROCESSOR.process_deferred_function */
extern void T329x25297047T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_AST_PROCESSOR.process_once_function */
extern void T329x25296959T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_AST_PROCESSOR.process_do_function */
extern void T329x25297045T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_AST_PROCESSOR.process_c2_character_constant */
extern void T329x25297081T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_AST_PROCESSOR.process_c1_character_constant */
extern void T329x25297082T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_AST_PROCESSOR.process_false_constant */
extern void T329x25297022T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_AST_PROCESSOR.process_void */
extern void T329x25296901T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_AST_PROCESSOR.process_underscored_real_constant */
extern void T329x25296910T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_AST_PROCESSOR.process_regular_real_constant */
extern void T329x25296930T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_AST_PROCESSOR.process_binary_integer_constant */
extern void T329x25297088T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_AST_PROCESSOR.process_octal_integer_constant */
extern void T329x25296962T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_AST_PROCESSOR.process_hexadecimal_integer_constant */
extern void T329x25297007T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_AST_PROCESSOR.process_underscored_integer_constant */
extern void T329x25296911T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_AST_PROCESSOR.process_special_manifest_string */
extern void T329x25296922T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_AST_PROCESSOR.process_verbatim_string */
extern void T329x25296902T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_AST_PROCESSOR.process_c3_character_constant */
extern void T329x25297080T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_AST_PROCESSOR.process_true_constant */
extern void T329x25296914T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_AST_PROCESSOR.process_formal_parameter_type */
extern void T329x25297009T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_AST_PROCESSOR.process_like_current */
extern void T329x25296979T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_AST_PROCESSOR.process_attribute */
extern void T329x25297090T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_AST_PROCESSOR.process_unique_attribute */
extern void T329x25296906T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_AST_PROCESSOR.process_constant_attribute */
extern void T329x25297066T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_AST_PROCESSOR.process_extended_attribute */
extern void T329x25297028T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_AST_PROCESSOR.process_tuple_type */
extern void T329x25296913T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_AST_PROCESSOR.process_external_function */
extern void T329x25297026T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_AST_PROCESSOR.process_regular_manifest_string */
extern void T329x25296931T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_AST_PROCESSOR.process_named_object_test */
extern void T329x25296965T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_AST_PROCESSOR.process_deferred_procedure */
extern void T329x25297046T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_AST_PROCESSOR.process_do_procedure */
extern void T329x25297043T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_AST_PROCESSOR.process_regular_integer_constant */
extern void T329x25296932T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_AST_PROCESSOR.process_create_expression */
extern void T329x25297056T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_AST_PROCESSOR.process_equality_expression */
extern void T329x25297032T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_AST_PROCESSOR.process_object_equality_expression */
extern void T329x25296964T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_AST_PROCESSOR.process_object_test */
extern void T329x25296963T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_AST_PROCESSOR.process_manifest_tuple */
extern void T329x25296967T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_AST_PROCESSOR.process_unqualified_call_expression */
extern void T329x25296905T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_AST_PROCESSOR.process_result */
extern void T329x25296926T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_AST_PROCESSOR.process_qualified_call_expression */
extern void T329x25296937T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_AST_PROCESSOR.process_current */
extern void T329x25297051T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_AST_PROCESSOR.process_identifier */
extern void T329x25297006T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_AST_PROCESSOR.process_class */
extern void T329x25297074T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_TYPE.reset_qualified_anchored_types */
extern void T330x23003211(GE_context* ac, T0* C);
/* Call to ET_TYPE.reset */
extern void T330x23003212(GE_context* ac, T0* C);
/* Call to ET_TYPE.append_to_string */
extern void T330x23003210T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_TYPE.append_unaliased_to_string */
extern void T330x23003209T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_TYPE.process */
extern void T330x20955146T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_QUERY.reset_after_interface_checked */
extern void T334x18841685(GE_context* ac, T0* C);
/* Call to ET_QUERY.reset_after_features_flattened */
extern void T334x18841686(GE_context* ac, T0* C);
/* Call to ET_QUERY.reset */
extern void T334x18841687(GE_context* ac, T0* C);
/* Call to ET_QUERY.reset_validity_checked */
extern void T334x19546125(GE_context* ac, T0* C);
/* Call to ET_PROCEDURE.reset_after_interface_checked */
extern void T335x18841685(GE_context* ac, T0* C);
/* Call to ET_PROCEDURE.reset_after_features_flattened */
extern void T335x18841686(GE_context* ac, T0* C);
/* Call to ET_PROCEDURE.reset */
extern void T335x18841687(GE_context* ac, T0* C);
/* Call to ET_PROCEDURE.reset_validity_checked */
extern void T335x19546125(GE_context* ac, T0* C);
/* Call to ET_UNIVERSE.universes_do_if */
extern void T336x20070534T0T0(GE_context* ac, T0* C, T0* a1, T0* a2);
/* Call to ET_UNIVERSE.universes_do_all */
extern void T336x20070535T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_UNIVERSE.master_classes_do_if_until */
extern void T336x20070540T0T0T0(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* Call to ET_UNIVERSE.master_classes_do_if */
extern void T336x20070541T0T0(GE_context* ac, T0* C, T0* a1, T0* a2);
/* Call to ET_UNIVERSE.master_classes_do_until */
extern void T336x20070542T0T0(GE_context* ac, T0* C, T0* a1, T0* a2);
/* Call to ET_UNIVERSE.master_classes_do_all */
extern void T336x20070543T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_UNIVERSE.classes_do_if */
extern void T336x20070551T0T0(GE_context* ac, T0* C, T0* a1, T0* a2);
/* Call to ET_UNIVERSE.classes_do_until */
extern void T336x20070552T0T0(GE_context* ac, T0* C, T0* a1, T0* a2);
/* Call to ET_UNIVERSE.classes_do_all */
extern void T336x20070553T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_ACTUAL_PARAMETERS.reset_qualified_anchored_types */
extern void T340x22380565(GE_context* ac, T0* C);
/* Call to ET_ACTUAL_PARAMETERS.reset */
extern void T340x22380566(GE_context* ac, T0* C);
/* Call to ET_ACTUAL_PARAMETERS.append_to_string */
extern void T340x22380564T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_ACTUAL_PARAMETERS.append_unaliased_to_string */
extern void T340x22380563T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable ET_DYNAMIC_TYPE_SET.put_target */
extern void T342x24739850T0T0(GE_context* ac, T0* C, T0* a1, T0* a2);
/* Call to detachable ET_DYNAMIC_TYPE_SET.propagate_can_be_void */
extern void T342x24559617T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_DYNAMIC_TYPE_SET.put_target */
extern void T343x24739850T0T0(GE_context* ac, T0* C, T0* a1, T0* a2);
/* Call to ET_DYNAMIC_TYPE_SET.set_never_void */
extern void T343x24739851(GE_context* ac, T0* C);
/* Call to ET_DYNAMIC_TYPE_SET.propagate_can_be_void */
extern void T343x24559617T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_DYNAMIC_TYPE_SET.put_type_from_type_set */
extern void T343x24559619T0T0T0(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* Call to ET_EXPRESSION.process */
extern void T389x20955146T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_EXPRESSION.reset */
extern void T389x15876100(GE_context* ac, T0* C);
/* Call to ET_EXPRESSION.set_index */
extern void T389x18071556T6(GE_context* ac, T0* C, T6 a1);
/* Call to ET_FEATURE.set_id */
extern void T393x18841682T6(GE_context* ac, T0* C, T6 a1);
/* Call to ET_FEATURE.process */
extern void T393x20955146T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_FEATURE.set_validity_error */
extern void T393x19546126(GE_context* ac, T0* C);
/* Call to ET_FEATURE.set_validity_checked */
extern void T393x19546127(GE_context* ac, T0* C);
/* Call to ET_FEATURE.set_other_seeds */
extern void T393x18841675T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_FEATURE.set_first_seed */
extern void T393x18841676T6(GE_context* ac, T0* C, T6 a1);
/* Call to ET_FEATURE.set_clients */
extern void T393x18841683T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_FEATURE.set_version */
extern void T393x18841679T6(GE_context* ac, T0* C, T6 a1);
/* Call to ET_FEATURE.set_other_precursors */
extern void T393x18841673T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_FEATURE.set_first_precursor */
extern void T393x18841674T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_FEATURE.reset_postconditions */
extern void T393x18841668(GE_context* ac, T0* C);
/* Call to ET_FEATURE.reset_preconditions */
extern void T393x18841669(GE_context* ac, T0* C);
/* Call to ET_FEATURE.set_implementation_class */
extern void T393x18841678T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_FEATURE.set_implementation_feature */
extern void T393x18841677T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_CALL_NAME.set_seed */
extern void T426x21545007T6(GE_context* ac, T0* C, T6 a1);
/* Call to ET_AGENT.set_arguments */
extern void T430x15400972T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_AGENT.process */
extern void T430x20955146T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable ET_TYPE.process */
extern void T436x20955146T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_CONSTANT.set_index */
extern void T437x18071556T6(GE_context* ac, T0* C, T6 a1);
/* Call to ET_CONSTANT.process */
extern void T437x20955146T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_OPERAND.set_index */
extern void T466x18071556T6(GE_context* ac, T0* C, T6 a1);
/* Call to ET_AGENT_TARGET.set_index */
extern void T467x18071556T6(GE_context* ac, T0* C, T6 a1);
/* Call to ET_AGENT_TARGET.process */
extern void T467x20955146T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_AGENT_TARGET.reset */
extern void T467x18071557(GE_context* ac, T0* C);
/* Call to ET_AGENT_ARGUMENT_OPERAND.process */
extern void T470x20955146T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_AGENT_ARGUMENT_OPERAND.reset */
extern void T470x18071557(GE_context* ac, T0* C);
/* Call to ET_MANIFEST_STRING.set_index */
extern void T493x18071556T6(GE_context* ac, T0* C, T6 a1);
/* Call to ET_MANIFEST_STRING.set_type */
extern void T493x17121292T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_ASSERTION.process */
extern void T534x20955146T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_ASSERTION.reset */
extern void T534x15876100(GE_context* ac, T0* C);
/* Call to ET_EXTENDED_FEATURE_NAME.reset */
extern void T593x21643269(GE_context* ac, T0* C);
/* Call to ET_INTERNAL_UNIVERSE.clusters_do_explicit */
extern void T646x19955754T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_INTERNAL_UNIVERSE.dotnet_assemblies_do_if */
extern void T646x19955747T0T0(GE_context* ac, T0* C, T0* a1, T0* a2);
/* Call to ET_DOTNET_ASSEMBLY.add_dotnet_assembly_recursive */
extern void T649x19890201T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_DOTNET_ASSEMBLY.add_universe_recursive */
extern void T649x20070529T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_DOTNET_ASSEMBLY.import_classes */
extern void T649x20070521(GE_context* ac, T0* C);
/* Call to ET_DOTNET_ASSEMBLY.preparse */
extern void T649x20070525T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_DOTNET_ASSEMBLY.consume */
extern void T649x19890196T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_CLUSTER.process */
extern void T673x19742771T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_CLUSTER.set_dependant_constraint */
extern void T673x19742782T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_CLUSTER.set_provider_constraint */
extern void T673x19742783T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_CLUSTER.add_recursive_cluster */
extern void T673x19742778T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_CLUSTER.set_preparsed */
extern void T673x19742788T1(GE_context* ac, T0* C, T1 a1);
/* Call to ET_CLUSTER.set_scm_write_mapping */
extern void T673x19742780T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_CLUSTER.set_scm_read_mapping */
extern void T673x19742781T0(GE_context* ac, T0* C, T0* a1);
/* Call to XM_EIFFEL_SCANNER.close_input */
extern void T732x27754519(GE_context* ac, T0* C);
/* Call to XM_EIFFEL_SCANNER.set_input_from_resolver */
extern void T732x27754520T0(GE_context* ac, T0* C, T0* a1);
/* Call to XM_EIFFEL_SCANNER.set_encoding */
extern void T732x27754518T0(GE_context* ac, T0* C, T0* a1);
/* Call to XM_EIFFEL_SCANNER.push_start_condition_dtd_ignore */
extern void T732x27754523(GE_context* ac, T0* C);
/* Call to XM_EIFFEL_SCANNER.read_token */
extern void T732x12156969(GE_context* ac, T0* C);
/* Call to XM_EIFFEL_SCANNER.set_input_stream */
extern void T732x27754521T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable ET_CHARACTER_CONSTANT.set_cast_type */
extern void T889x17022987T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable ET_INTEGER_CONSTANT.set_cast_type */
extern void T898x17104935T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable ET_INTEGER_CONSTANT.set_sign */
extern void T898x17104936T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable ET_MANIFEST_STRING.set_cast_type */
extern void T904x17121293T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable ET_ALIAS_NAME.set_prefix */
extern void T955x21463088(GE_context* ac, T0* C);
/* Call to detachable ET_ALIAS_NAME.set_infix */
extern void T955x21463087(GE_context* ac, T0* C);
/* Call to detachable ET_INLINE_AGENT.set_across_components */
extern void T1116x18432014T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable ET_INLINE_AGENT.set_object_tests */
extern void T1116x18432015T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable ET_INLINE_AGENT.set_actual_arguments */
extern void T1116x15400972T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_QUALIFIED_FEATURE_CALL_EXPRESSION.set_index */
extern void T1529x18071556T6(GE_context* ac, T0* C, T6 a1);
/* Call to ET_DYNAMIC_TARGET.propagate_can_be_void */
extern void T1531x24559617T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_DYNAMIC_TARGET.put_type_from_type_set */
extern void T1531x24559619T0T0T0(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* Call to XM_DOCUMENT_NODE.node_set_parent */
extern void T1541x28524546T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_ALIAS_NAME.reset */
extern void T1573x21545008(GE_context* ac, T0* C);
/* Call to ET_ALIAS_NAME.set_infix */
extern void T1573x21463087(GE_context* ac, T0* C);
/* Call to ET_ALIAS_NAME.set_prefix */
extern void T1573x21463088(GE_context* ac, T0* C);
/* Call to ET_INSTRUCTION.process */
extern void T1666x20955146T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_INSTRUCTION.reset */
extern void T1666x20561922(GE_context* ac, T0* C);
/* Call to ET_CHOICE.reset */
extern void T1678x20267013(GE_context* ac, T0* C);
/* Call to ET_ASSIGNER.reset */
extern void T1725x18382850(GE_context* ac, T0* C);
/* Call to [attached] DS_ARRAYED_LIST [[attached] ET_CLUSTER].remove */
extern void T1849x9125906T6(GE_context* ac, T0* C, T6 a1);
/* Call to XM_ELEMENT_NODE.node_set_parent */
extern void T2005x28524546T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_INTEGER_CONSTANT.set_index */
extern void T2030x18071556T6(GE_context* ac, T0* C, T6 a1);
/* Call to ET_INTEGER_CONSTANT.set_cast_type */
extern void T2030x17104935T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_INTEGER_CONSTANT.set_type */
extern void T2030x17104934T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_INTEGER_CONSTANT.set_sign */
extern void T2030x17104936T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_LIKE_TYPE.reset_qualified_anchored_types */
extern void T2099x23003211(GE_context* ac, T0* C);
/* Call to ET_LIKE_TYPE.reset */
extern void T2099x23003212(GE_context* ac, T0* C);
/* Call to ET_LIKE_TYPE.append_to_string */
extern void T2099x23003210T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_CHOICE_CONSTANT.reset */
extern void T2117x20267013(GE_context* ac, T0* C);
/* Call to ET_OPERATOR.reset */
extern void T2129x21545008(GE_context* ac, T0* C);
/* Call to ET_ECF_GROUP.set_options */
extern void T2381x14811152T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_UNTAGGED_ASSERTION.reset */
extern void T2437x15876100(GE_context* ac, T0* C);
/* Call to ET_CHARACTER_CONSTANT.set_index */
extern void T2571x18071556T6(GE_context* ac, T0* C, T6 a1);
/* Call to ET_CHARACTER_CONSTANT.set_cast_type */
extern void T2571x17022987T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_CHARACTER_CONSTANT.set_type */
extern void T2571x17022986T0(GE_context* ac, T0* C, T0* a1);
/* Call to ET_REGULAR_FEATURE_CALL.set_parenthesis_call */
extern void T2616x20774914T0T0T0(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* Call to ET_CALL_WITH_ACTUAL_ARGUMENT_LIST.set_arguments */
extern void T2620x20234242T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable LX_TRANSITION [[attached] LX_NFA_STATE].record */
extern void T2789x12435460T0(GE_context* ac, T0* C, T0* a1);
/* New instance of type SPECIAL [CHARACTER_8] */
extern T0* GE_new15(T6 a1, T1 initialize);
/* New instance of type SPECIAL [CHARACTER_32] */
extern T0* GE_new16(T6 a1, T1 initialize);
/* New instance of type STRING_8 */
extern T0* GE_new17(T1 initialize);
/* New instance of type STRING_32 */
extern T0* GE_new18(T1 initialize);
/* New instance of type IMMUTABLE_STRING_32 */
extern T0* GE_new19(T1 initialize);
/* New instance of type ISE_EXCEPTION_MANAGER */
extern T0* GE_new20(T1 initialize);
/* New instance of type GEC */
extern T0* GE_new25(T1 initialize);
/* New instance of type [attached] CELL [detachable EXCEPTION] */
extern T0* GE_new26(T1 initialize);
/* New instance of type [attached] HASH_TABLE [[attached] INTEGER_32, [attached] INTEGER_32] */
extern T0* GE_new27(T1 initialize);
/* New instance of type [attached] CELL [detachable TUPLE [[attached] INTEGER_32, [attached] INTEGER_32, [attached] INTEGER_32, [attached] STRING_8, [attached] STRING_8, [attached] STRING_8, [attached] STRING_8, [attached] STRING_8, [attached] STRING_8, [attached] INTEGER_32, [attached] BOOLEAN]] */
extern T0* GE_new28(T1 initialize);
/* New instance of type [attached] CELL [[attached] NO_MORE_MEMORY] */
extern T0* GE_new29(T1 initialize);
/* New instance of type C_STRING */
extern T0* GE_new31(T1 initialize);
/* New instance of type [attached] TUPLE [[attached] INTEGER_32, [attached] INTEGER_32, [attached] INTEGER_32, [attached] STRING_8, [attached] STRING_8, [attached] STRING_8, [attached] STRING_8, [attached] STRING_8, [attached] STRING_8, [attached] INTEGER_32, [attached] BOOLEAN] */
extern T0* GE_new32(T1 initialize);
/* New instance of type KL_TEXT_INPUT_FILE */
extern T0* GE_new33(T1 initialize);
/* New instance of type KL_ARGUMENTS */
extern T0* GE_new34(T1 initialize);
/* New instance of type ET_ISE_VARIABLES */
extern T0* GE_new35(T1 initialize);
/* New instance of type ET_ERROR_HANDLER */
extern T0* GE_new36(T1 initialize);
/* New instance of type KL_STRING_ROUTINES */
extern T0* GE_new38(T1 initialize);
/* New instance of type ET_SYSTEM */
extern T0* GE_new40(T1 initialize);
/* New instance of type KL_EXCEPTIONS */
extern T0* GE_new41(T1 initialize);
/* New instance of type AP_PARSER */
extern T0* GE_new42(T1 initialize);
/* New instance of type AP_ALTERNATIVE_OPTIONS_LIST */
extern T0* GE_new43(T1 initialize);
/* New instance of type AP_STRING_OPTION */
extern T0* GE_new44(T1 initialize);
/* New instance of type AP_FLAG */
extern T0* GE_new46(T1 initialize);
/* New instance of type AP_ENUMERATION_OPTION */
extern T0* GE_new47(T1 initialize);
/* New instance of type AP_BOOLEAN_OPTION */
extern T0* GE_new48(T1 initialize);
/* New instance of type AP_INTEGER_OPTION */
extern T0* GE_new49(T1 initialize);
/* New instance of type ET_NULL_ERROR_HANDLER */
extern T0* GE_new53(T1 initialize);
/* New instance of type ET_XACE_SYSTEM_PARSER */
extern T0* GE_new55(T1 initialize);
/* New instance of type ET_XACE_DEFAULT_ERROR_HANDLER */
extern T0* GE_new56(T1 initialize);
/* New instance of type [attached] DS_HASH_TABLE [[attached] STRING_8, [attached] STRING_8] */
extern T0* GE_new57(T1 initialize);
/* New instance of type UC_STRING_EQUALITY_TESTER */
extern T0* GE_new58(T1 initialize);
/* New instance of type ET_ECF_VARIABLES */
extern T0* GE_new61(T1 initialize);
/* New instance of type [attached] DS_HASH_TABLE_CURSOR [[attached] STRING_8, [attached] STRING_8] */
extern T0* GE_new62(T1 initialize);
/* New instance of type ET_ECF_SYSTEM_PARSER */
extern T0* GE_new66(T1 initialize);
/* New instance of type ET_ECF_ERROR_HANDLER */
extern T0* GE_new67(T1 initialize);
/* New instance of type ET_ECF_SYSTEM */
extern T0* GE_new69(T1 initialize);
/* New instance of type ET_ECF_TARGET */
extern T0* GE_new70(T1 initialize);
/* New instance of type ET_ECF_SETTINGS */
extern T0* GE_new71(T1 initialize);
/* New instance of type ET_LACE_PARSER */
extern T0* GE_new77(T1 initialize);
/* New instance of type ET_LACE_ERROR_HANDLER */
extern T0* GE_new78(T1 initialize);
/* New instance of type ET_DYNAMIC_SYSTEM */
extern T0* GE_new80(T1 initialize);
/* New instance of type ET_SYSTEM_PROCESSOR */
extern T0* GE_new81(T1 initialize);
/* New instance of type ET_C_GENERATOR */
extern T0* GE_new84(T1 initialize);
/* New instance of type KL_SHELL_COMMAND */
extern T0* GE_new85(T1 initialize);
/* New instance of type ET_SYSTEM_MULTIPROCESSOR */
extern T0* GE_new87(T1 initialize);
/* New instance of type UT_VERSION */
extern T0* GE_new88(T1 initialize);
/* New instance of type ET_DYNAMIC_PUSH_TYPE_SET_BUILDER */
extern T0* GE_new91(T1 initialize);
/* New instance of type ET_CLASS_TYPE */
extern T0* GE_new93(T1 initialize);
/* New instance of type ET_TOKEN_CONSTANTS */
extern T0* GE_new94(T1 initialize);
/* New instance of type ET_CLASS */
extern T0* GE_new95(T1 initialize);
/* New instance of type KL_OPERATING_SYSTEM */
extern T0* GE_new97(T1 initialize);
/* New instance of type UT_CANNOT_READ_FILE_ERROR */
extern T0* GE_new100(T1 initialize);
/* New instance of type UT_VERSION_NUMBER */
extern T0* GE_new102(T1 initialize);
/* New instance of type UT_MESSAGE */
extern T0* GE_new103(T1 initialize);
/* New instance of type ET_ECF_CAPABILITIES */
extern T0* GE_new106(T1 initialize);
/* New instance of type KL_EXECUTION_ENVIRONMENT */
extern T0* GE_new107(T1 initialize);
/* New instance of type AP_ERROR */
extern T0* GE_new108(T1 initialize);
/* New instance of type [attached] DS_HASH_SET [[attached] STRING_8] */
extern T0* GE_new109(T1 initialize);
/* New instance of type VOID_TARGET */
extern T0* GE_new112(T1 initialize);
/* New instance of type ROUTINE_FAILURE */
extern T0* GE_new114(T1 initialize);
/* New instance of type OLD_VIOLATION */
extern T0* GE_new116(T1 initialize);
/* New instance of type NO_MORE_MEMORY */
extern T0* GE_new119(T1 initialize);
/* New instance of type INVARIANT_VIOLATION */
extern T0* GE_new120(T1 initialize);
/* New instance of type OPERATING_SYSTEM_SIGNAL_FAILURE */
extern T0* GE_new121(T1 initialize);
/* New instance of type IO_FAILURE */
extern T0* GE_new122(T1 initialize);
/* New instance of type OPERATING_SYSTEM_FAILURE */
extern T0* GE_new123(T1 initialize);
/* New instance of type COM_FAILURE */
extern T0* GE_new124(T1 initialize);
/* New instance of type EIFFEL_RUNTIME_PANIC */
extern T0* GE_new125(T1 initialize);
/* New instance of type PRECONDITION_VIOLATION */
extern T0* GE_new128(T1 initialize);
/* New instance of type POSTCONDITION_VIOLATION */
extern T0* GE_new129(T1 initialize);
/* New instance of type FLOATING_POINT_FAILURE */
extern T0* GE_new130(T1 initialize);
/* New instance of type CHECK_VIOLATION */
extern T0* GE_new131(T1 initialize);
/* New instance of type BAD_INSPECT_VALUE */
extern T0* GE_new132(T1 initialize);
/* New instance of type VARIANT_VIOLATION */
extern T0* GE_new133(T1 initialize);
/* New instance of type LOOP_INVARIANT_VIOLATION */
extern T0* GE_new134(T1 initialize);
/* New instance of type RESCUE_FAILURE */
extern T0* GE_new135(T1 initialize);
/* New instance of type RESUMPTION_FAILURE */
extern T0* GE_new136(T1 initialize);
/* New instance of type CREATE_ON_DEFERRED */
extern T0* GE_new137(T1 initialize);
/* New instance of type EXTERNAL_FAILURE */
extern T0* GE_new138(T1 initialize);
/* New instance of type VOID_ASSIGNED_TO_EXPANDED */
extern T0* GE_new139(T1 initialize);
/* New instance of type EXCEPTION_IN_SIGNAL_HANDLER_FAILURE */
extern T0* GE_new140(T1 initialize);
/* New instance of type MISMATCH_FAILURE */
extern T0* GE_new141(T1 initialize);
/* New instance of type DEVELOPER_EXCEPTION */
extern T0* GE_new142(T1 initialize);
/* New instance of type ADDRESS_APPLIED_TO_MELTED_FEATURE */
extern T0* GE_new143(T1 initialize);
/* New instance of type SERIALIZATION_FAILURE */
extern T0* GE_new144(T1 initialize);
/* New instance of type EXECUTION_ENVIRONMENT */
extern T0* GE_new145(T1 initialize);
/* New instance of type KL_WINDOWS_FILE_SYSTEM */
extern T0* GE_new146(T1 initialize);
/* New instance of type KL_UNIX_FILE_SYSTEM */
extern T0* GE_new147(T1 initialize);
/* New instance of type PRIMES */
extern T0* GE_new148(T1 initialize);
/* New instance of type [attached] SPECIAL [[attached] INTEGER_32] */
extern T0* GE_new149(T6 a1, T1 initialize);
/* New instance of type [attached] SPECIAL [[attached] BOOLEAN] */
extern T0* GE_new150(T6 a1, T1 initialize);
/* New instance of type MANAGED_POINTER */
extern T0* GE_new154(T1 initialize);
/* New instance of type [attached] KL_LINKABLE [[attached] CHARACTER_8] */
extern T0* GE_new155(T1 initialize);
/* New instance of type FILE_INFO */
extern T0* GE_new157(T1 initialize);
/* New instance of type MUTEX */
extern T0* GE_new158(T1 initialize);
/* New instance of type UT_ERROR_HANDLER */
extern T0* GE_new159(T1 initialize);
/* New instance of type KL_STANDARD_FILES */
extern T0* GE_new160(T1 initialize);
/* New instance of type KL_STDERR_FILE */
extern T0* GE_new161(T1 initialize);
/* New instance of type KL_STDOUT_FILE */
extern T0* GE_new162(T1 initialize);
/* New instance of type UC_STRING */
extern T0* GE_new163(T1 initialize);
/* New instance of type KL_ANY_ROUTINES */
extern T0* GE_new164(T1 initialize);
/* New instance of type ST_WORD_WRAPPER */
extern T0* GE_new166(T1 initialize);
/* New instance of type AP_DISPLAY_HELP_FLAG */
extern T0* GE_new168(T1 initialize);
/* New instance of type [attached] DS_ARRAYED_LIST [[attached] AP_OPTION] */
extern T0* GE_new169(T1 initialize);
/* New instance of type [attached] DS_ARRAYED_LIST [[attached] AP_ALTERNATIVE_OPTIONS_LIST] */
extern T0* GE_new170(T1 initialize);
/* New instance of type [attached] DS_ARRAYED_LIST [[attached] STRING_8] */
extern T0* GE_new171(T1 initialize);
/* New instance of type AP_ERROR_HANDLER */
extern T0* GE_new172(T1 initialize);
/* New instance of type [attached] DS_LINKED_LIST_CURSOR [[attached] AP_OPTION] */
extern T0* GE_new175(T1 initialize);
/* New instance of type [attached] DS_ARRAYED_LIST [detachable STRING_8] */
extern T0* GE_new176(T1 initialize);
/* New instance of type [attached] DS_ARRAYED_LIST_CURSOR [detachable STRING_8] */
extern T0* GE_new177(T1 initialize);
/* New instance of type [attached] DS_LINKED_LIST [[attached] STRING_8] */
extern T0* GE_new178(T1 initialize);
/* New instance of type KL_STRING_EQUALITY_TESTER */
extern T0* GE_new179(T1 initialize);
/* New instance of type [attached] DS_LINKED_LIST [[attached] BOOLEAN] */
extern T0* GE_new182(T1 initialize);
/* New instance of type [attached] DS_LINKED_LIST [[attached] INTEGER_32] */
extern T0* GE_new185(T1 initialize);
/* New instance of type KL_NULL_TEXT_OUTPUT_STREAM */
extern T0* GE_new186(T1 initialize);
/* New instance of type ET_XACE_AST_FACTORY */
extern T0* GE_new187(T1 initialize);
/* New instance of type XM_DOCUMENT */
extern T0* GE_new188(T1 initialize);
/* New instance of type XM_ELEMENT */
extern T0* GE_new189(T1 initialize);
/* New instance of type XM_TREE_CALLBACKS_PIPE */
extern T0* GE_new192(T1 initialize);
/* New instance of type XM_STOP_ON_ERROR_FILTER */
extern T0* GE_new193(T1 initialize);
/* New instance of type XM_CALLBACKS_TO_TREE_FILTER */
extern T0* GE_new194(T1 initialize);
/* New instance of type ET_XACE_VALIDATOR */
extern T0* GE_new195(T1 initialize);
/* New instance of type ET_XACE_PREPROCESSOR */
extern T0* GE_new196(T1 initialize);
/* New instance of type ET_XACE_SYSTEM */
extern T0* GE_new197(T1 initialize);
/* New instance of type [attached] DS_HASH_TABLE [[attached] ET_XACE_LIBRARY_CONFIG, [attached] STRING_8] */
extern T0* GE_new198(T1 initialize);
/* New instance of type XM_EXPAT_PARSER_FACTORY */
extern T0* GE_new199(T1 initialize);
/* New instance of type XM_EIFFEL_PARSER */
extern T0* GE_new200(T1 initialize);
/* New instance of type ET_XACE_LIBRARY_CONFIG_PARSER */
extern T0* GE_new203(T1 initialize);
/* New instance of type attached KL_EQUALITY_TESTER [[attached] STRING_8] */
extern T0* GE_new206(T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [[attached] STRING_8] */
extern T0* GE_new207(T1 initialize);
/* New instance of type [attached] SPECIAL [[attached] STRING_8] */
extern T0* GE_new208(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [[attached] INTEGER_32] */
extern T0* GE_new209(T1 initialize);
/* New instance of type [attached] KL_AGENT_HASH_FUNCTION [[attached] STRING_8] */
extern T0* GE_new210(T1 initialize);
/* New instance of type KL_CASE_INSENSITIVE_STRING_EQUALITY_TESTER */
extern T0* GE_new212(T1 initialize);
/* New instance of type [attached] TUPLE [[attached] STRING_8] */
extern T0* GE_new213(T1 initialize);
/* New instance of type [attached] FUNCTION [[attached] TUPLE [[attached] STRING_8], [attached] INTEGER_32] */
extern T0* GE_new214(T1 initialize);
/* New instance of type [attached] TUPLE */
extern T0* GE_new216(T1 initialize);
/* New instance of type [attached] TUPLE [KL_STRING_ROUTINES] */
extern T0* GE_new217(T1 initialize);
/* New instance of type ET_ECF_AST_FACTORY */
extern T0* GE_new219(T1 initialize);
/* New instance of type [attached] DS_CELL [detachable ET_ECF_SYSTEM] */
extern T0* GE_new220(T1 initialize);
/* New instance of type [attached] TUPLE [[attached] ET_ECF_TARGET] */
extern T0* GE_new221(T1 initialize);
/* New instance of type [attached] PROCEDURE [[attached] TUPLE [[attached] ET_ECF_TARGET]] */
extern T0* GE_new222(T1 initialize);
/* New instance of type [attached] TUPLE [ET_ECF_SYSTEM_PARSER] */
extern T0* GE_new223(T1 initialize);
/* New instance of type [attached] TUPLE [[attached] XM_ELEMENT, detachable XM_POSITION_TABLE, [attached] STRING_8] */
extern T0* GE_new225(T1 initialize);
/* New instance of type [attached] PROCEDURE [[attached] TUPLE [[attached] XM_ELEMENT, detachable XM_POSITION_TABLE, [attached] STRING_8]] */
extern T0* GE_new226(T1 initialize);
/* New instance of type [attached] TUPLE [ET_ECF_SYSTEM_PARSER, detachable STRING_8, [attached] PROCEDURE [[attached] TUPLE [[attached] ET_ECF_TARGET]], [attached] DS_CELL [detachable ET_ECF_SYSTEM]] */
extern T0* GE_new227(T1 initialize);
/* New instance of type [attached] DS_HASH_TABLE [[attached] ET_ECF_LIBRARY, [attached] STRING_8] */
extern T0* GE_new229(T1 initialize);
/* New instance of type [attached] DS_HASH_TABLE [[attached] ET_ECF_DOTNET_ASSEMBLY, [attached] STRING_8] */
extern T0* GE_new230(T1 initialize);
/* New instance of type ET_ECF_STATE */
extern T0* GE_new231(T1 initialize);
/* New instance of type ET_IDENTIFIER */
extern T0* GE_new232(T1 initialize);
/* New instance of type ET_ECF_SYSTEM_CONFIG */
extern T0* GE_new233(T1 initialize);
/* New instance of type ET_ECF_LIBRARY */
extern T0* GE_new236(T1 initialize);
/* New instance of type ET_ADAPTED_LIBRARIES */
extern T0* GE_new237(T1 initialize);
/* New instance of type [attached] TUPLE [[attached] ET_ADAPTED_LIBRARY] */
extern T0* GE_new239(T1 initialize);
/* New instance of type [attached] PROCEDURE [[attached] TUPLE [[attached] ET_ADAPTED_LIBRARY]] */
extern T0* GE_new240(T1 initialize);
/* New instance of type ET_COMPRESSED_POSITION */
extern T0* GE_new241(T1 initialize);
/* New instance of type XM_ATTRIBUTE */
extern T0* GE_new245(T1 initialize);
/* New instance of type [attached] DS_HASH_TABLE [[attached] ET_ECF_SYSTEM_CONFIG, [attached] STRING_8] */
extern T0* GE_new247(T1 initialize);
/* New instance of type [attached] DS_HASH_TABLE [[attached] ET_ECF_TARGET, [attached] STRING_8] */
extern T0* GE_new248(T1 initialize);
/* New instance of type [attached] DS_CELL [detachable ET_ECF_SYSTEM_CONFIG] */
extern T0* GE_new250(T1 initialize);
/* New instance of type [attached] TUPLE [ET_ECF_SYSTEM_PARSER, ET_ECF_INTERNAL_UNIVERSE, [attached] DS_CELL [detachable ET_ECF_SYSTEM_CONFIG]] */
extern T0* GE_new253(T1 initialize);
/* New instance of type ET_ECF_TARGETS */
extern T0* GE_new255(T1 initialize);
/* New instance of type [attached] DS_HASH_TABLE [detachable RX_REGULAR_EXPRESSION, [attached] STRING_8] */
extern T0* GE_new256(T1 initialize);
/* New instance of type [attached] DS_HASH_TABLE_CURSOR [detachable RX_REGULAR_EXPRESSION, [attached] STRING_8] */
extern T0* GE_new257(T1 initialize);
/* New instance of type [attached] DS_CELL [detachable ET_ECF_LIBRARY] */
extern T0* GE_new259(T1 initialize);
/* New instance of type ET_ECF_ADAPTED_LIBRARY */
extern T0* GE_new260(T1 initialize);
/* New instance of type [attached] TUPLE [ET_ECF_SYSTEM_PARSER, ET_ECF_ADAPTED_LIBRARY, [attached] DS_CELL [detachable ET_ECF_LIBRARY]] */
extern T0* GE_new261(T1 initialize);
/* New instance of type ET_ADAPTED_DOTNET_ASSEMBLIES */
extern T0* GE_new264(T1 initialize);
/* New instance of type ET_ECF_DOTNET_ASSEMBLY */
extern T0* GE_new265(T1 initialize);
/* New instance of type ET_ECF_ADAPTED_DOTNET_ASSEMBLY */
extern T0* GE_new267(T1 initialize);
/* New instance of type ET_ECF_ERROR */
extern T0* GE_new268(T1 initialize);
/* New instance of type ET_LACE_AST_FACTORY */
extern T0* GE_new269(T1 initialize);
/* New instance of type [attached] DS_HASH_TABLE [[attached] ET_LACE_CLUSTER, [attached] ET_IDENTIFIER] */
extern T0* GE_new270(T1 initialize);
/* New instance of type YY_FILE_BUFFER */
extern T0* GE_new271(T1 initialize);
/* New instance of type YY_BUFFER */
extern T0* GE_new272(T1 initialize);
/* New instance of type [attached] DS_HASH_SET [[attached] ET_IDENTIFIER] */
extern T0* GE_new273(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ANY] */
extern T0* GE_new279(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ANY] */
extern T0* GE_new280(T1 initialize);
/* New instance of type [attached] SPECIAL [[attached] ET_IDENTIFIER] */
extern T0* GE_new281(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [[attached] ET_IDENTIFIER] */
extern T0* GE_new282(T1 initialize);
/* New instance of type ET_LACE_SYSTEM */
extern T0* GE_new283(T1 initialize);
/* New instance of type ET_LACE_CLUSTERS */
extern T0* GE_new285(T1 initialize);
/* New instance of type ET_LACE_CLUSTER */
extern T0* GE_new286(T1 initialize);
/* New instance of type ET_LACE_EXCLUDE */
extern T0* GE_new288(T1 initialize);
/* New instance of type ET_LACE_DOTNET_ASSEMBLY */
extern T0* GE_new290(T1 initialize);
/* New instance of type [attached] SPECIAL [[attached] ET_LACE_SYSTEM] */
extern T0* GE_new291(T6 a1, T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_LACE_CLUSTERS] */
extern T0* GE_new292(T6 a1, T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_ADAPTED_DOTNET_ASSEMBLIES] */
extern T0* GE_new293(T6 a1, T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_IDENTIFIER] */
extern T0* GE_new294(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [[attached] ET_LACE_SYSTEM] */
extern T0* GE_new295(T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_IDENTIFIER] */
extern T0* GE_new296(T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_LACE_CLUSTERS] */
extern T0* GE_new297(T1 initialize);
/* New instance of type [attached] SPECIAL [[attached] ET_LACE_CLUSTERS] */
extern T0* GE_new298(T6 a1, T1 initialize);
/* New instance of type [attached] SPECIAL [[attached] ET_LACE_CLUSTER] */
extern T0* GE_new299(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [[attached] ET_LACE_CLUSTERS] */
extern T0* GE_new300(T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [[attached] ET_LACE_CLUSTER] */
extern T0* GE_new301(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_LACE_EXCLUDE] */
extern T0* GE_new302(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_LACE_EXCLUDE] */
extern T0* GE_new303(T1 initialize);
/* New instance of type [attached] SPECIAL [[attached] ET_LACE_EXCLUDE] */
extern T0* GE_new304(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [[attached] ET_LACE_EXCLUDE] */
extern T0* GE_new305(T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_ADAPTED_DOTNET_ASSEMBLIES] */
extern T0* GE_new306(T1 initialize);
/* New instance of type [attached] SPECIAL [[attached] ET_ADAPTED_DOTNET_ASSEMBLIES] */
extern T0* GE_new307(T6 a1, T1 initialize);
/* New instance of type [attached] SPECIAL [[attached] ET_LACE_DOTNET_ASSEMBLY] */
extern T0* GE_new308(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [[attached] ET_ADAPTED_DOTNET_ASSEMBLIES] */
extern T0* GE_new309(T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [[attached] ET_LACE_DOTNET_ASSEMBLY] */
extern T0* GE_new310(T1 initialize);
/* New instance of type ET_LACE_DOTNET_GAC_ASSEMBLY */
extern T0* GE_new311(T1 initialize);
/* New instance of type KL_PLATFORM */
extern T0* GE_new312(T1 initialize);
/* New instance of type KL_INTEGER_ROUTINES */
extern T0* GE_new313(T1 initialize);
/* New instance of type [attached] DS_HASH_TABLE [[attached] ET_MASTER_CLASS, [attached] ET_CLASS_NAME] */
extern T0* GE_new314(T1 initialize);
/* New instance of type [attached] DS_ARRAYED_LIST [[attached] ET_DYNAMIC_TYPE] */
extern T0* GE_new315(T1 initialize);
/* New instance of type [attached] TUPLE [[attached] ET_CLASS] */
extern T0* GE_new316(T1 initialize);
/* New instance of type [attached] PROCEDURE [[attached] TUPLE [[attached] ET_CLASS]] */
extern T0* GE_new317(T1 initialize);
/* New instance of type [attached] TUPLE [INTEGER_32] */
extern T0* GE_new318(T1 initialize);
/* New instance of type ET_DYNAMIC_NULL_TYPE_SET_BUILDER */
extern T0* GE_new319(T1 initialize);
/* New instance of type ET_DYNAMIC_TYPE */
extern T0* GE_new320(T1 initialize);
/* New instance of type [attached] TUPLE [ET_DYNAMIC_SYSTEM] */
extern T0* GE_new321(T1 initialize);
/* New instance of type ET_DYNAMIC_FEATURE */
extern T0* GE_new332(T1 initialize);
/* New instance of type ET_ACTUAL_PARAMETER_LIST */
extern T0* GE_new333(T1 initialize);
/* New instance of type ET_EXTERNAL_FUNCTION */
extern T0* GE_new344(T1 initialize);
/* New instance of type ET_TUPLE_TYPE */
extern T0* GE_new345(T1 initialize);
/* New instance of type ET_FORMAL_ARGUMENT_LIST */
extern T0* GE_new347(T1 initialize);
/* New instance of type [attached] ARRAY [[attached] ET_TYPE] */
extern T0* GE_new348(T1 initialize);
/* New instance of type [attached] SPECIAL [[attached] ET_TYPE] */
extern T0* GE_new349(T6 a1, T1 initialize);
/* New instance of type ET_FORMAL_ARGUMENT */
extern T0* GE_new350(T1 initialize);
/* New instance of type ET_QUERY_LIST */
extern T0* GE_new351(T1 initialize);
/* New instance of type ET_PROCEDURE_LIST */
extern T0* GE_new352(T1 initialize);
/* New instance of type DT_SHARED_SYSTEM_CLOCK */
extern T0* GE_new353(T1 initialize);
/* New instance of type DT_SYSTEM_CLOCK */
extern T0* GE_new354(T1 initialize);
/* New instance of type DT_DATE_TIME */
extern T0* GE_new355(T1 initialize);
/* New instance of type ET_EIFFEL_PREPARSER */
extern T0* GE_new357(T1 initialize);
/* New instance of type ET_MASTER_CLASS_CHECKER */
extern T0* GE_new358(T1 initialize);
/* New instance of type ET_EIFFEL_PARSER */
extern T0* GE_new359(T1 initialize);
/* New instance of type ET_PROVIDER_CHECKER */
extern T0* GE_new360(T1 initialize);
/* New instance of type ET_ANCESTOR_BUILDER */
extern T0* GE_new361(T1 initialize);
/* New instance of type ET_FEATURE_FLATTENER */
extern T0* GE_new362(T1 initialize);
/* New instance of type ET_INTERFACE_CHECKER */
extern T0* GE_new363(T1 initialize);
/* New instance of type ET_IMPLEMENTATION_CHECKER */
extern T0* GE_new364(T1 initialize);
/* New instance of type [attached] DS_ARRAYED_LIST [[attached] ET_CLASS] */
extern T0* GE_new365(T1 initialize);
/* New instance of type [attached] TUPLE [[attached] DS_ARRAYED_LIST [[attached] ET_CLASS]] */
extern T0* GE_new366(T1 initialize);
/* New instance of type ET_AST_NULL_PROCESSOR */
extern T0* GE_new367(T1 initialize);
/* New instance of type [attached] DS_ARRAYED_LIST [[attached] INTEGER_32] */
extern T0* GE_new368(T1 initialize);
/* New instance of type ET_AST_FACTORY */
extern T0* GE_new369(T1 initialize);
/* New instance of type ET_DOTNET_ASSEMBLY_CLASSIC_CONSUMER */
extern T0* GE_new371(T1 initialize);
/* New instance of type DT_DATE_TIME_DURATION */
extern T0* GE_new372(T1 initialize);
/* New instance of type ET_DYNAMIC_TYPE_SET_LIST */
extern T0* GE_new373(T1 initialize);
/* New instance of type KL_STRING_OUTPUT_STREAM */
extern T0* GE_new374(T1 initialize);
/* New instance of type [attached] DS_ARRAYED_LIST [[attached] ET_IDENTIFIER] */
extern T0* GE_new375(T1 initialize);
/* New instance of type [attached] DS_ARRAYED_LIST [detachable ET_DYNAMIC_TYPE] */
extern T0* GE_new376(T1 initialize);
/* New instance of type ET_DYNAMIC_TYPE_HASH_LIST */
extern T0* GE_new377(T1 initialize);
/* New instance of type ET_DYNAMIC_STANDALONE_TYPE_SET */
extern T0* GE_new378(T1 initialize);
/* New instance of type [attached] DS_ARRAYED_STACK [[attached] ET_EXPRESSION] */
extern T0* GE_new379(T1 initialize);
/* New instance of type [attached] DS_ARRAYED_LIST [[attached] ET_EXPRESSION] */
extern T0* GE_new380(T1 initialize);
/* New instance of type [attached] DS_HASH_TABLE [[attached] ET_DYNAMIC_TYPE, [attached] INTEGER_32] */
extern T0* GE_new381(T1 initialize);
/* New instance of type ET_DYNAMIC_STANDALONE_TYPE_SET_LIST */
extern T0* GE_new382(T1 initialize);
/* New instance of type [attached] DS_HASH_SET [[attached] ET_DYNAMIC_TYPE] */
extern T0* GE_new383(T1 initialize);
/* New instance of type [attached] DS_HASH_TABLE [[attached] ET_DYNAMIC_STANDALONE_TYPE_SET, [attached] ET_DYNAMIC_TYPE] */
extern T0* GE_new384(T1 initialize);
/* New instance of type [attached] DS_ARRAYED_LIST [[attached] ET_OBJECT_TEST] */
extern T0* GE_new385(T1 initialize);
/* New instance of type [attached] DS_ARRAYED_LIST [[attached] ET_OBJECT_EQUALITY_EXPRESSION] */
extern T0* GE_new386(T1 initialize);
/* New instance of type [attached] DS_ARRAYED_LIST [[attached] ET_EQUALITY_EXPRESSION] */
extern T0* GE_new387(T1 initialize);
/* New instance of type [attached] DS_ARRAYED_LIST [[attached] ET_AGENT] */
extern T0* GE_new388(T1 initialize);
/* New instance of type ET_EXPRESSION */
extern T0* GE_new389(T1 initialize);
/* New instance of type ET_CURRENT */
extern T0* GE_new390(T1 initialize);
/* New instance of type ET_ACTUAL_ARGUMENT_LIST */
extern T0* GE_new391(T1 initialize);
/* New instance of type ET_QUALIFIED_CALL_INSTRUCTION */
extern T0* GE_new392(T1 initialize);
/* New instance of type ET_QUALIFIED_CALL_EXPRESSION */
extern T0* GE_new396(T1 initialize);
/* New instance of type ET_UNQUALIFIED_CALL_EXPRESSION */
extern T0* GE_new397(T1 initialize);
/* New instance of type [attached] DS_ARRAYED_LIST [detachable ET_QUALIFIED_CALL_EXPRESSION] */
extern T0* GE_new398(T1 initialize);
/* New instance of type ET_MANIFEST_TUPLE */
extern T0* GE_new399(T1 initialize);
/* New instance of type [attached] DS_HASH_SET [[attached] ET_DYNAMIC_TUPLE_TYPE] */
extern T0* GE_new400(T1 initialize);
/* New instance of type [attached] DS_HASH_TABLE [[attached] INTEGER_32, [attached] ET_FEATURE] */
extern T0* GE_new401(T1 initialize);
/* New instance of type [attached] ARRAY [[attached] INTEGER_32] */
extern T0* GE_new402(T1 initialize);
/* New instance of type [attached] DS_HASH_TABLE [[attached] ET_CONSTANT, [attached] ET_FEATURE] */
extern T0* GE_new403(T1 initialize);
/* New instance of type [attached] DS_HASH_TABLE [[attached] ET_DYNAMIC_TYPE, [attached] ET_INLINE_CONSTANT] */
extern T0* GE_new404(T1 initialize);
/* New instance of type [attached] DS_HASH_TABLE [detachable ET_DYNAMIC_FEATURE, [attached] ET_DYNAMIC_TYPE] */
extern T0* GE_new405(T1 initialize);
/* New instance of type [attached] DS_ARRAYED_LIST [[attached] ET_DYNAMIC_FEATURE] */
extern T0* GE_new406(T1 initialize);
/* New instance of type [attached] DS_HASH_TABLE [[attached] BOOLEAN, [attached] STRING_8] */
extern T0* GE_new407(T1 initialize);
/* New instance of type ET_IDENTIFIER_TESTER */
extern T0* GE_new408(T1 initialize);
/* New instance of type RX_PCRE_REGULAR_EXPRESSION */
extern T0* GE_new410(T1 initialize);
/* New instance of type KL_TEXT_OUTPUT_FILE */
extern T0* GE_new411(T1 initialize);
/* New instance of type ET_DYNAMIC_TUPLE_TYPE */
extern T0* GE_new412(T1 initialize);
/* New instance of type UT_CANNOT_WRITE_TO_FILE_ERROR */
extern T0* GE_new413(T1 initialize);
/* New instance of type ET_DYNAMIC_FEATURE_LIST */
extern T0* GE_new414(T1 initialize);
/* New instance of type ET_DYNAMIC_PRECURSOR */
extern T0* GE_new416(T1 initialize);
/* New instance of type ET_DYNAMIC_PRECURSOR_LIST */
extern T0* GE_new418(T1 initialize);
/* New instance of type [attached] DS_HASH_TOPOLOGICAL_SORTER [[attached] ET_DYNAMIC_TYPE] */
extern T0* GE_new419(T1 initialize);
/* New instance of type ET_RESULT */
extern T0* GE_new422(T1 initialize);
/* New instance of type ET_DYNAMIC_QUALIFIED_QUERY_CALL */
extern T0* GE_new424(T1 initialize);
/* New instance of type ET_DYNAMIC_QUALIFIED_PROCEDURE_CALL */
extern T0* GE_new429(T1 initialize);
/* New instance of type ET_OBJECT_TEST */
extern T0* GE_new431(T1 initialize);
/* New instance of type ET_OBJECT_EQUALITY_EXPRESSION */
extern T0* GE_new432(T1 initialize);
/* New instance of type ET_EQUALITY_EXPRESSION */
extern T0* GE_new433(T1 initialize);
/* New instance of type ET_DYNAMIC_SPECIAL_TYPE */
extern T0* GE_new434(T1 initialize);
/* New instance of type [attached] DS_QUICK_SORTER [[attached] ET_DYNAMIC_TYPE] */
extern T0* GE_new438(T1 initialize);
/* New instance of type ET_DYNAMIC_TYPE_COMPARATOR_BY_ID */
extern T0* GE_new439(T1 initialize);
/* New instance of type ET_CREATE_EXPRESSION */
extern T0* GE_new444(T1 initialize);
/* New instance of type ET_QUALIFIED_CALL */
extern T0* GE_new445(T1 initialize);
/* New instance of type [attached] ARRAY [[attached] STRING_8] */
extern T0* GE_new449(T1 initialize);
/* New instance of type KL_DIRECTORY */
extern T0* GE_new451(T1 initialize);
/* New instance of type UT_TEMPLATE_EXPANDER */
extern T0* GE_new452(T1 initialize);
/* New instance of type ET_EXTENDED_ATTRIBUTE */
extern T0* GE_new454(T1 initialize);
/* New instance of type ET_COMPOUND */
extern T0* GE_new456(T1 initialize);
/* New instance of type [attached] DS_QUICK_SORTER [[attached] INTEGER_32] */
extern T0* GE_new464(T1 initialize);
/* New instance of type ET_IMPLICIT_TYPE_MARK */
extern T0* GE_new472(T1 initialize);
/* New instance of type ET_SYMBOL */
extern T0* GE_new474(T1 initialize);
/* New instance of type [attached] PROCEDURE [[attached] TUPLE] */
extern T0* GE_new475(T1 initialize);
/* New instance of type [attached] TUPLE [ET_C_GENERATOR, ET_IDENTIFIER, ET_CURRENT, ET_DYNAMIC_TYPE, BOOLEAN] */
extern T0* GE_new476(T1 initialize);
/* New instance of type [attached] TUPLE [ET_C_GENERATOR, ET_IDENTIFIER, ET_RESULT, ET_DYNAMIC_TYPE, BOOLEAN] */
extern T0* GE_new477(T1 initialize);
/* New instance of type [attached] TUPLE [ET_C_GENERATOR, ET_DYNAMIC_FEATURE, ET_RESULT, ET_DYNAMIC_TYPE, BOOLEAN] */
extern T0* GE_new478(T1 initialize);
/* New instance of type [attached] TUPLE [ET_C_GENERATOR, INTEGER_32, ET_RESULT, ET_DYNAMIC_TYPE, BOOLEAN] */
extern T0* GE_new479(T1 initialize);
/* New instance of type UT_INTEGER_FORMATTER */
extern T0* GE_new480(T1 initialize);
/* New instance of type STD_FILES */
extern T0* GE_new485(T1 initialize);
/* New instance of type [attached] TUPLE [ET_C_GENERATOR, INTEGER_32, ET_EXPRESSION, ET_DYNAMIC_TYPE, BOOLEAN] */
extern T0* GE_new487(T1 initialize);
/* New instance of type [attached] TUPLE [ET_C_GENERATOR, ET_EXPRESSION, ET_DYNAMIC_TYPE] */
extern T0* GE_new490(T1 initialize);
/* New instance of type [attached] TUPLE [ET_C_GENERATOR, ET_DYNAMIC_FEATURE, ET_DYNAMIC_TYPE, BOOLEAN] */
extern T0* GE_new491(T1 initialize);
/* New instance of type ET_CONSTANT_ATTRIBUTE */
extern T0* GE_new492(T1 initialize);
/* New instance of type ET_UNIQUE_ATTRIBUTE */
extern T0* GE_new494(T1 initialize);
/* New instance of type ET_REGULAR_INTEGER_CONSTANT */
extern T0* GE_new495(T1 initialize);
/* New instance of type ET_ATTRIBUTE */
extern T0* GE_new496(T1 initialize);
/* New instance of type ET_LIKE_CURRENT */
extern T0* GE_new498(T1 initialize);
/* New instance of type NATIVE_STRING */
extern T0* GE_new500(T1 initialize);
/* New instance of type [attached] DS_ARRAYED_LIST [[attached] ET_SYSTEM_PROCESSOR] */
extern T0* GE_new501(T1 initialize);
/* New instance of type WORKER_THREAD */
extern T0* GE_new502(T1 initialize);
/* New instance of type [attached] TUPLE [ET_SYSTEM_PROCESSOR, [attached] DS_ARRAYED_LIST [[attached] ET_CLASS]] */
extern T0* GE_new503(T1 initialize);
/* New instance of type ET_FEATURE_CHECKER */
extern T0* GE_new505(T1 initialize);
/* New instance of type [attached] DS_CELL [[attached] INTEGER_32] */
extern T0* GE_new506(T1 initialize);
/* New instance of type [attached] DS_HASH_TABLE [[attached] INTEGER_32, [attached] ET_DYNAMIC_TYPE] */
extern T0* GE_new507(T1 initialize);
/* New instance of type ET_TYPE_CHECKER */
extern T0* GE_new510(T1 initialize);
/* New instance of type [attached] DS_ARRAYED_LIST [[attached] ET_INLINE_AGENT] */
extern T0* GE_new512(T1 initialize);
/* New instance of type [attached] DS_ARRAYED_LIST [[attached] DS_ARRAYED_LIST [[attached] ET_PROCEDURE]] */
extern T0* GE_new513(T1 initialize);
/* New instance of type [attached] DS_ARRAYED_LIST [[attached] DS_ARRAYED_LIST [[attached] ET_QUERY]] */
extern T0* GE_new514(T1 initialize);
/* New instance of type [attached] DS_ARRAYED_LIST [[attached] DS_ARRAYED_LIST [[attached] ET_FEATURE]] */
extern T0* GE_new515(T1 initialize);
/* New instance of type [attached] DS_ARRAYED_LIST [[attached] ET_NESTED_TYPE_CONTEXT] */
extern T0* GE_new516(T1 initialize);
/* New instance of type ET_NESTED_TYPE_CONTEXT */
extern T0* GE_new517(T1 initialize);
/* New instance of type [attached] DS_HASH_TABLE [[attached] ET_NESTED_TYPE_CONTEXT, [attached] ET_NAMED_OBJECT_TEST] */
extern T0* GE_new518(T1 initialize);
/* New instance of type ET_OBJECT_TEST_SCOPE */
extern T0* GE_new519(T1 initialize);
/* New instance of type ET_OBJECT_TEST_SCOPE_BUILDER */
extern T0* GE_new520(T1 initialize);
/* New instance of type [attached] DS_HASH_TABLE [[attached] ET_NESTED_TYPE_CONTEXT, [attached] ET_ACROSS_COMPONENT] */
extern T0* GE_new521(T1 initialize);
/* New instance of type ET_ACROSS_CURSOR_SCOPE */
extern T0* GE_new522(T1 initialize);
/* New instance of type ET_ATTACHMENT_SCOPE */
extern T0* GE_new523(T1 initialize);
/* New instance of type ET_ATTACHMENT_SCOPE_BUILDER */
extern T0* GE_new524(T1 initialize);
/* New instance of type [attached] DS_ARRAYED_LIST [[attached] ET_ATTACHMENT_SCOPE] */
extern T0* GE_new525(T1 initialize);
/* New instance of type [attached] DS_ARRAYED_LIST [[attached] ET_INDEXING_TERM] */
extern T0* GE_new526(T1 initialize);
/* New instance of type attached DS_HASH_TABLE [[attached] ET_DYNAMIC_TYPE_SET, [attached] ET_DYNAMIC_TYPE] */
extern T0* GE_new527(T1 initialize);
/* New instance of type ET_PRECONDITIONS */
extern T0* GE_new529(T1 initialize);
/* New instance of type [attached] DS_HASH_TABLE [[attached] ET_ASSERTIONS, [attached] ET_FEATURE] */
extern T0* GE_new530(T1 initialize);
/* New instance of type ET_FEATURE_LIST */
extern T0* GE_new533(T1 initialize);
/* New instance of type ET_UNKNOWN_GROUP */
extern T0* GE_new536(T1 initialize);
/* New instance of type ET_BASE_TYPE_LIST */
extern T0* GE_new538(T1 initialize);
/* New instance of type ET_KEYWORD */
extern T0* GE_new539(T1 initialize);
/* New instance of type ET_CLASS_CODES */
extern T0* GE_new540(T1 initialize);
/* New instance of type PATH */
extern T0* GE_new542(T1 initialize);
/* New instance of type [attached] DS_HASH_TABLE [[attached] ET_IDENTIFIER, [attached] STRING_8] */
extern T0* GE_new543(T1 initialize);
/* New instance of type [attached] DS_HASH_SET_CURSOR [[attached] STRING_8] */
extern T0* GE_new544(T1 initialize);
/* New instance of type ARGUMENTS_32 */
extern T0* GE_new547(T1 initialize);
/* New instance of type ET_SYSTEM_ERROR */
extern T0* GE_new548(T1 initialize);
/* New instance of type ET_INTERNAL_ERROR */
extern T0* GE_new549(T1 initialize);
/* New instance of type ET_XACE_EXTERNALS */
extern T0* GE_new551(T1 initialize);
/* New instance of type [attached] DS_LINKED_LIST_CURSOR [[attached] STRING_8] */
extern T0* GE_new552(T1 initialize);
/* New instance of type ET_XACE_OPTIONS */
extern T0* GE_new553(T1 initialize);
/* New instance of type ET_XACE_TARGET */
extern T0* GE_new554(T1 initialize);
/* New instance of type ET_XACE_TARGETS */
extern T0* GE_new557(T1 initialize);
/* New instance of type ET_XACE_CLUSTERS */
extern T0* GE_new559(T1 initialize);
/* New instance of type ET_XACE_MOUNTED_LIBRARIES */
extern T0* GE_new560(T1 initialize);
/* New instance of type ET_XACE_CLUSTER */
extern T0* GE_new564(T1 initialize);
/* New instance of type [attached] DS_LINKED_LIST_CURSOR [[attached] XM_ELEMENT_NODE] */
extern T0* GE_new569(T1 initialize);
/* New instance of type ET_XACE_MOUNTED_LIBRARY */
extern T0* GE_new571(T1 initialize);
/* New instance of type ET_XACE_ELEMENT_OBSOLETED_BY_ELEMENT_WARNING */
extern T0* GE_new575(T1 initialize);
/* New instance of type ET_XACE_MULTIPLE_LIBRARY_PREFIX_ERROR */
extern T0* GE_new576(T1 initialize);
/* New instance of type ET_ECF_OPTIONS */
extern T0* GE_new581(T1 initialize);
/* New instance of type XM_NAMESPACE */
extern T0* GE_new583(T1 initialize);
/* New instance of type [attached] KL_ARRAY_ROUTINES [[attached] INTEGER_32] */
extern T0* GE_new584(T1 initialize);
/* New instance of type ET_DYNAMIC_PROCEDURE_TYPE */
extern T0* GE_new586(T1 initialize);
/* New instance of type ET_DYNAMIC_FUNCTION_TYPE */
extern T0* GE_new587(T1 initialize);
/* New instance of type UT_CONFIG_PARSER */
extern T0* GE_new588(T1 initialize);
/* New instance of type [attached] KL_COMPARABLE_COMPARATOR [[attached] INTEGER_32] */
extern T0* GE_new589(T1 initialize);
/* New instance of type ET_DO_PROCEDURE */
extern T0* GE_new592(T1 initialize);
/* New instance of type ET_DYNAMIC_PUSH_TYPE_SET */
extern T0* GE_new595(T1 initialize);
/* New instance of type ET_DEFERRED_PROCEDURE */
extern T0* GE_new596(T1 initialize);
/* New instance of type [attached] CELL [[attached] INTEGER_32] */
extern T0* GE_new599(T1 initialize);
/* New instance of type [attached] HASH_TABLE [[attached] NATIVE_STRING, [attached] IMMUTABLE_STRING_32] */
extern T0* GE_new601(T1 initialize);
/* New instance of type KL_PATHNAME */
extern T0* GE_new603(T1 initialize);
/* New instance of type EXCEPTIONS */
extern T0* GE_new604(T1 initialize);
/* New instance of type READABLE_STRING_32 */
extern T0* GE_new605(T1 initialize);
/* New instance of type [attached] SPECIAL [[attached] NATURAL_8] */
extern T0* GE_new606(T6 a1, T1 initialize);
/* New instance of type [attached] DS_ARRAYED_LIST_CURSOR [[attached] AP_OPTION] */
extern T0* GE_new608(T1 initialize);
/* New instance of type [attached] SPECIAL [[attached] AP_OPTION] */
extern T0* GE_new611(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [[attached] AP_OPTION] */
extern T0* GE_new612(T1 initialize);
/* New instance of type [attached] DS_ARRAYED_LIST_CURSOR [[attached] AP_ALTERNATIVE_OPTIONS_LIST] */
extern T0* GE_new614(T1 initialize);
/* New instance of type [attached] SPECIAL [[attached] AP_ALTERNATIVE_OPTIONS_LIST] */
extern T0* GE_new615(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [[attached] AP_ALTERNATIVE_OPTIONS_LIST] */
extern T0* GE_new616(T1 initialize);
/* New instance of type [attached] DS_ARRAYED_LIST_CURSOR [[attached] STRING_8] */
extern T0* GE_new618(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable STRING_8] */
extern T0* GE_new620(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable STRING_8] */
extern T0* GE_new621(T1 initialize);
/* New instance of type attached DS_LINKABLE [[attached] STRING_8] */
extern T0* GE_new623(T1 initialize);
/* New instance of type attached DS_LINKABLE [[attached] BOOLEAN] */
extern T0* GE_new626(T1 initialize);
/* New instance of type [attached] DS_LINKED_LIST_CURSOR [[attached] BOOLEAN] */
extern T0* GE_new627(T1 initialize);
/* New instance of type attached DS_LINKABLE [[attached] INTEGER_32] */
extern T0* GE_new629(T1 initialize);
/* New instance of type [attached] DS_LINKED_LIST_CURSOR [[attached] INTEGER_32] */
extern T0* GE_new630(T1 initialize);
/* New instance of type XM_NAMESPACE_RESOLVER */
extern T0* GE_new632(T1 initialize);
/* New instance of type [attached] ARRAY [[attached] XM_CALLBACKS_FILTER] */
extern T0* GE_new633(T1 initialize);
/* New instance of type [attached] SPECIAL [[attached] XM_CALLBACKS_FILTER] */
extern T0* GE_new634(T6 a1, T1 initialize);
/* New instance of type XM_CALLBACKS_NULL */
extern T0* GE_new635(T1 initialize);
/* New instance of type XM_POSITION_TABLE */
extern T0* GE_new637(T1 initialize);
/* New instance of type [attached] DS_HASH_SET [[attached] XM_NAMESPACE] */
extern T0* GE_new638(T1 initialize);
/* New instance of type UT_COUNTER */
extern T0* GE_new642(T1 initialize);
/* New instance of type [attached] KL_AGENT_ROUTINES [[attached] ET_CLASS] */
extern T0* GE_new643(T1 initialize);
/* New instance of type [attached] TUPLE [UT_COUNTER] */
extern T0* GE_new644(T1 initialize);
/* New instance of type [attached] TUPLE [[attached] KL_AGENT_ROUTINES [[attached] ET_CLASS], [attached] PROCEDURE [[attached] TUPLE]] */
extern T0* GE_new645(T1 initialize);
/* New instance of type [attached] DS_HASH_SET [[attached] ET_DOTNET_ASSEMBLY] */
extern T0* GE_new647(T1 initialize);
/* New instance of type ET_DOTNET_ASSEMBLIES */
extern T0* GE_new648(T1 initialize);
/* New instance of type ET_DOTNET_ASSEMBLY */
extern T0* GE_new649(T1 initialize);
/* New instance of type [attached] TUPLE [[attached] ET_DOTNET_ASSEMBLY] */
extern T0* GE_new650(T1 initialize);
/* New instance of type [attached] PREDICATE [[attached] TUPLE [[attached] ET_DOTNET_ASSEMBLY]] */
extern T0* GE_new651(T1 initialize);
/* New instance of type [attached] PROCEDURE [[attached] TUPLE [[attached] ET_DOTNET_ASSEMBLY]] */
extern T0* GE_new652(T1 initialize);
/* New instance of type [attached] TUPLE [[attached] DS_HASH_SET [[attached] ET_DOTNET_ASSEMBLY]] */
extern T0* GE_new653(T1 initialize);
/* New instance of type [attached] TUPLE [[attached] ET_INTERNAL_UNIVERSE] */
extern T0* GE_new654(T1 initialize);
/* New instance of type [attached] PROCEDURE [[attached] TUPLE [[attached] ET_INTERNAL_UNIVERSE]] */
extern T0* GE_new655(T1 initialize);
/* New instance of type [attached] TUPLE [[attached] PROCEDURE [[attached] TUPLE [[attached] ET_DOTNET_ASSEMBLY]], [attached] PREDICATE [[attached] TUPLE [[attached] ET_DOTNET_ASSEMBLY]]] */
extern T0* GE_new656(T1 initialize);
/* New instance of type [attached] TUPLE [ET_DOTNET_ASSEMBLIES] */
extern T0* GE_new657(T1 initialize);
/* New instance of type [attached] TUPLE [ET_SYSTEM_PROCESSOR] */
extern T0* GE_new658(T1 initialize);
/* New instance of type ET_LIBRARY */
extern T0* GE_new659(T1 initialize);
/* New instance of type [attached] TUPLE [[attached] ET_LIBRARY] */
extern T0* GE_new660(T1 initialize);
/* New instance of type [attached] PROCEDURE [[attached] TUPLE [[attached] ET_LIBRARY]] */
extern T0* GE_new661(T1 initialize);
/* New instance of type [attached] TUPLE [[attached] ET_UNIVERSE] */
extern T0* GE_new662(T1 initialize);
/* New instance of type [attached] PROCEDURE [[attached] TUPLE [[attached] ET_UNIVERSE]] */
extern T0* GE_new663(T1 initialize);
/* New instance of type [attached] TUPLE [[attached] PROCEDURE [[attached] TUPLE [[attached] ET_CLASS]]] */
extern T0* GE_new664(T1 initialize);
/* New instance of type [attached] TUPLE [[attached] PROCEDURE [[attached] TUPLE [[attached] ET_CLASS]], detachable FUNCTION [[attached] TUPLE, [attached] BOOLEAN]] */
extern T0* GE_new665(T1 initialize);
/* New instance of type [attached] DS_ARRAYED_LIST [[attached] ET_XACE_MOUNTED_LIBRARY] */
extern T0* GE_new667(T1 initialize);
/* New instance of type ET_XACE_LIBRARY_CONFIG */
extern T0* GE_new668(T1 initialize);
/* New instance of type [attached] TUPLE [[attached] ET_XACE_CLUSTER] */
extern T0* GE_new669(T1 initialize);
/* New instance of type [attached] PROCEDURE [[attached] TUPLE [[attached] ET_XACE_CLUSTER]] */
extern T0* GE_new670(T1 initialize);
/* New instance of type [attached] TUPLE [ET_XACE_CLUSTERS] */
extern T0* GE_new671(T1 initialize);
/* New instance of type ET_MASTER_CLASS */
extern T0* GE_new672(T1 initialize);
/* New instance of type [attached] TUPLE [[attached] ET_CLUSTER] */
extern T0* GE_new674(T1 initialize);
/* New instance of type [attached] PROCEDURE [[attached] TUPLE [[attached] ET_CLUSTER]] */
extern T0* GE_new675(T1 initialize);
/* New instance of type [attached] TUPLE [ET_XACE_SYSTEM] */
extern T0* GE_new676(T1 initialize);
/* New instance of type [attached] DS_HASH_SET [[attached] ET_INTERNAL_UNIVERSE] */
extern T0* GE_new677(T1 initialize);
/* New instance of type [attached] TUPLE [ET_AST_PROCESSOR] */
extern T0* GE_new678(T1 initialize);
/* New instance of type [attached] TUPLE [[attached] ET_MASTER_CLASS] */
extern T0* GE_new679(T1 initialize);
/* New instance of type [attached] PROCEDURE [[attached] TUPLE [[attached] ET_MASTER_CLASS]] */
extern T0* GE_new680(T1 initialize);
/* New instance of type [attached] TUPLE [[attached] ET_ADAPTED_DOTNET_ASSEMBLY] */
extern T0* GE_new681(T1 initialize);
/* New instance of type [attached] PROCEDURE [[attached] TUPLE [[attached] ET_ADAPTED_DOTNET_ASSEMBLY]] */
extern T0* GE_new682(T1 initialize);
/* New instance of type ET_ANCESTORS_STATUS_CHECKER */
extern T0* GE_new683(T1 initialize);
/* New instance of type ET_FLATTENING_STATUS_CHECKER */
extern T0* GE_new684(T1 initialize);
/* New instance of type ET_INTERFACE_STATUS_CHECKER */
extern T0* GE_new685(T1 initialize);
/* New instance of type ET_IMPLEMENTATION_STATUS_CHECKER */
extern T0* GE_new686(T1 initialize);
/* New instance of type [attached] PREDICATE [[attached] TUPLE [[attached] ET_CLASS]] */
extern T0* GE_new687(T1 initialize);
/* New instance of type [attached] TUPLE [ET_ANCESTORS_STATUS_CHECKER] */
extern T0* GE_new689(T1 initialize);
/* New instance of type [attached] TUPLE [ET_FLATTENING_STATUS_CHECKER] */
extern T0* GE_new690(T1 initialize);
/* New instance of type [attached] TUPLE [ET_INTERFACE_STATUS_CHECKER] */
extern T0* GE_new691(T1 initialize);
/* New instance of type [attached] TUPLE [ET_IMPLEMENTATION_STATUS_CHECKER] */
extern T0* GE_new692(T1 initialize);
/* New instance of type [attached] TUPLE [BOOLEAN] */
extern T0* GE_new693(T1 initialize);
/* New instance of type [attached] DS_HASH_SET [[attached] ET_UNIVERSE] */
extern T0* GE_new694(T1 initialize);
/* New instance of type ET_CLASS_NAME_TESTER */
extern T0* GE_new696(T1 initialize);
/* New instance of type ET_PARENT */
extern T0* GE_new698(T1 initialize);
/* New instance of type ET_PARENT_LIST */
extern T0* GE_new699(T1 initialize);
/* New instance of type ET_CLIENT_LIST */
extern T0* GE_new700(T1 initialize);
/* New instance of type ET_BUILTIN_CONVERT_FEATURE */
extern T0* GE_new701(T1 initialize);
/* New instance of type [attached] TUPLE [[attached] PROCEDURE [[attached] TUPLE [[attached] ET_CLUSTER]]] */
extern T0* GE_new706(T1 initialize);
/* New instance of type [attached] TUPLE [[attached] DS_HASH_SET [[attached] ET_INTERNAL_UNIVERSE]] */
extern T0* GE_new712(T1 initialize);
/* New instance of type [attached] TUPLE [[attached] PROCEDURE [[attached] TUPLE [[attached] ET_MASTER_CLASS]]] */
extern T0* GE_new713(T1 initialize);
/* New instance of type [attached] TUPLE [[attached] PROCEDURE [[attached] TUPLE [[attached] ET_CLASS]], [attached] FUNCTION [[attached] TUPLE [[attached] ET_CLASS], [attached] BOOLEAN]] */
extern T0* GE_new714(T1 initialize);
/* New instance of type [attached] TUPLE [[attached] DS_HASH_SET [[attached] ET_UNIVERSE]] */
extern T0* GE_new715(T1 initialize);
/* New instance of type ET_CLIENT */
extern T0* GE_new716(T1 initialize);
/* New instance of type ET_ATTACHMENT_SEPARATE_KEYWORDS */
extern T0* GE_new717(T1 initialize);
/* New instance of type ET_NONE_GROUP */
extern T0* GE_new723(T1 initialize);
/* New instance of type ET_UNFOLDED_EMPTY_TUPLE_ACTUAL_PARAMETERS */
extern T0* GE_new724(T1 initialize);
/* New instance of type [attached] DS_HASH_TABLE_CURSOR [[attached] ET_XACE_LIBRARY_CONFIG, [attached] STRING_8] */
extern T0* GE_new729(T1 initialize);
/* New instance of type [attached] SPECIAL [[attached] ET_XACE_LIBRARY_CONFIG] */
extern T0* GE_new730(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [[attached] ET_XACE_LIBRARY_CONFIG] */
extern T0* GE_new731(T1 initialize);
/* New instance of type XM_EIFFEL_SCANNER */
extern T0* GE_new732(T1 initialize);
/* New instance of type XM_DEFAULT_POSITION */
extern T0* GE_new733(T1 initialize);
/* New instance of type [attached] DS_BILINKED_LIST [[attached] XM_POSITION] */
extern T0* GE_new736(T1 initialize);
/* New instance of type [attached] DS_LINKED_STACK [[attached] XM_EIFFEL_SCANNER] */
extern T0* GE_new737(T1 initialize);
/* New instance of type [attached] DS_HASH_TABLE [[attached] XM_EIFFEL_ENTITY_DEF, [attached] STRING_8] */
extern T0* GE_new738(T1 initialize);
/* New instance of type XM_NULL_EXTERNAL_RESOLVER */
extern T0* GE_new740(T1 initialize);
/* New instance of type XM_DTD_CALLBACKS_NULL */
extern T0* GE_new742(T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [[attached] XM_EIFFEL_PARSER_NAME] */
extern T0* GE_new743(T1 initialize);
/* New instance of type XM_EIFFEL_PARSER_NAME */
extern T0* GE_new744(T1 initialize);
/* New instance of type [attached] SPECIAL [[attached] XM_EIFFEL_PARSER_NAME] */
extern T0* GE_new745(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [[attached] DS_HASH_SET [[attached] XM_EIFFEL_PARSER_NAME]] */
extern T0* GE_new746(T1 initialize);
/* New instance of type [attached] DS_HASH_SET [[attached] XM_EIFFEL_PARSER_NAME] */
extern T0* GE_new747(T1 initialize);
/* New instance of type [attached] SPECIAL [[attached] DS_HASH_SET [[attached] XM_EIFFEL_PARSER_NAME]] */
extern T0* GE_new748(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [[attached] XM_DTD_EXTERNAL_ID] */
extern T0* GE_new749(T1 initialize);
/* New instance of type XM_DTD_EXTERNAL_ID */
extern T0* GE_new750(T1 initialize);
/* New instance of type [attached] SPECIAL [[attached] XM_DTD_EXTERNAL_ID] */
extern T0* GE_new751(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [[attached] XM_DTD_ELEMENT_CONTENT] */
extern T0* GE_new752(T1 initialize);
/* New instance of type XM_DTD_ELEMENT_CONTENT */
extern T0* GE_new753(T1 initialize);
/* New instance of type [attached] SPECIAL [[attached] XM_DTD_ELEMENT_CONTENT] */
extern T0* GE_new754(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [[attached] DS_BILINKED_LIST [[attached] XM_DTD_ATTRIBUTE_CONTENT]] */
extern T0* GE_new755(T1 initialize);
/* New instance of type [attached] DS_BILINKED_LIST [[attached] XM_DTD_ATTRIBUTE_CONTENT] */
extern T0* GE_new756(T1 initialize);
/* New instance of type [attached] SPECIAL [[attached] DS_BILINKED_LIST [[attached] XM_DTD_ATTRIBUTE_CONTENT]] */
extern T0* GE_new757(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [[attached] XM_DTD_ATTRIBUTE_CONTENT] */
extern T0* GE_new758(T1 initialize);
/* New instance of type XM_DTD_ATTRIBUTE_CONTENT */
extern T0* GE_new759(T1 initialize);
/* New instance of type [attached] SPECIAL [[attached] XM_DTD_ATTRIBUTE_CONTENT] */
extern T0* GE_new760(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [[attached] DS_BILINKED_LIST [[attached] STRING_8]] */
extern T0* GE_new761(T1 initialize);
/* New instance of type [attached] DS_BILINKED_LIST [[attached] STRING_8] */
extern T0* GE_new762(T1 initialize);
/* New instance of type [attached] SPECIAL [[attached] DS_BILINKED_LIST [[attached] STRING_8]] */
extern T0* GE_new763(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [[attached] BOOLEAN] */
extern T0* GE_new764(T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [[attached] XM_EIFFEL_DECLARATION] */
extern T0* GE_new765(T1 initialize);
/* New instance of type XM_EIFFEL_DECLARATION */
extern T0* GE_new766(T1 initialize);
/* New instance of type [attached] SPECIAL [[attached] XM_EIFFEL_DECLARATION] */
extern T0* GE_new767(T6 a1, T1 initialize);
/* New instance of type XM_EIFFEL_ENTITY_DEF */
extern T0* GE_new772(T1 initialize);
/* New instance of type XM_EIFFEL_SCANNER_DTD */
extern T0* GE_new773(T1 initialize);
/* New instance of type XM_EIFFEL_PE_ENTITY_DEF */
extern T0* GE_new775(T1 initialize);
/* New instance of type [attached] DS_HASH_TABLE_CURSOR [[attached] ET_ECF_LIBRARY, [attached] STRING_8] */
extern T0* GE_new777(T1 initialize);
/* New instance of type [attached] SPECIAL [[attached] ET_ECF_LIBRARY] */
extern T0* GE_new778(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [[attached] ET_ECF_LIBRARY] */
extern T0* GE_new783(T1 initialize);
/* New instance of type [attached] SPECIAL [[attached] ET_ECF_DOTNET_ASSEMBLY] */
extern T0* GE_new784(T6 a1, T1 initialize);
/* New instance of type [attached] DS_HASH_TABLE_CURSOR [[attached] ET_ECF_DOTNET_ASSEMBLY, [attached] STRING_8] */
extern T0* GE_new789(T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [[attached] ET_ECF_DOTNET_ASSEMBLY] */
extern T0* GE_new790(T1 initialize);
/* New instance of type ET_CLUSTERS */
extern T0* GE_new791(T1 initialize);
/* New instance of type [attached] DS_ARRAYED_LIST [[attached] ET_ADAPTED_LIBRARY] */
extern T0* GE_new792(T1 initialize);
/* New instance of type [attached] DS_HASH_SET [[attached] ET_LIBRARY] */
extern T0* GE_new793(T1 initialize);
/* New instance of type [attached] TUPLE [[attached] DS_HASH_SET [[attached] ET_LIBRARY]] */
extern T0* GE_new794(T1 initialize);
/* New instance of type [attached] SPECIAL [[attached] ET_ECF_SYSTEM_CONFIG] */
extern T0* GE_new796(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [[attached] ET_ECF_SYSTEM_CONFIG] */
extern T0* GE_new798(T1 initialize);
/* New instance of type [attached] DS_HASH_TABLE_CURSOR [[attached] ET_ECF_SYSTEM_CONFIG, [attached] STRING_8] */
extern T0* GE_new799(T1 initialize);
/* New instance of type [attached] DS_HASH_TABLE_CURSOR [[attached] ET_ECF_TARGET, [attached] STRING_8] */
extern T0* GE_new800(T1 initialize);
/* New instance of type [attached] SPECIAL [[attached] ET_ECF_TARGET] */
extern T0* GE_new801(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [[attached] ET_ECF_TARGET] */
extern T0* GE_new803(T1 initialize);
/* New instance of type attached DS_SPARSE_TABLE_KEYS [detachable RX_REGULAR_EXPRESSION, [attached] STRING_8] */
extern T0* GE_new807(T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable RX_REGULAR_EXPRESSION] */
extern T0* GE_new808(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable RX_REGULAR_EXPRESSION] */
extern T0* GE_new809(T6 a1, T1 initialize);
/* New instance of type [attached] DS_ARRAYED_LIST [[attached] ET_ADAPTED_DOTNET_ASSEMBLY] */
extern T0* GE_new810(T1 initialize);
/* New instance of type [attached] KL_EQUALITY_TESTER [[attached] ET_IDENTIFIER] */
extern T0* GE_new812(T1 initialize);
/* New instance of type [attached] DS_HASH_TABLE_CURSOR [[attached] ET_LACE_CLUSTER, [attached] ET_IDENTIFIER] */
extern T0* GE_new815(T1 initialize);
/* New instance of type [attached] DS_HASH_SET_CURSOR [[attached] ET_IDENTIFIER] */
extern T0* GE_new816(T1 initialize);
/* New instance of type [attached] TUPLE [ET_LACE_SYSTEM] */
extern T0* GE_new819(T1 initialize);
/* New instance of type [attached] DS_ARRAYED_LIST [[attached] ET_LACE_CLUSTER] */
extern T0* GE_new820(T1 initialize);
/* New instance of type [attached] TUPLE [[attached] ET_LACE_CLUSTER] */
extern T0* GE_new821(T1 initialize);
/* New instance of type [attached] DS_HASH_TABLE_CURSOR [[attached] ET_MASTER_CLASS, [attached] ET_CLASS_NAME] */
extern T0* GE_new828(T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [[attached] ET_MASTER_CLASS] */
extern T0* GE_new829(T1 initialize);
/* New instance of type [attached] SPECIAL [[attached] ET_MASTER_CLASS] */
extern T0* GE_new830(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [[attached] ET_CLASS_NAME] */
extern T0* GE_new831(T1 initialize);
/* New instance of type [attached] SPECIAL [[attached] ET_CLASS_NAME] */
extern T0* GE_new832(T6 a1, T1 initialize);
/* New instance of type [attached] SPECIAL [[attached] ET_DYNAMIC_TYPE] */
extern T0* GE_new833(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [[attached] ET_DYNAMIC_TYPE] */
extern T0* GE_new834(T1 initialize);
/* New instance of type [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_DYNAMIC_TYPE] */
extern T0* GE_new835(T1 initialize);
/* New instance of type [attached] SPECIAL [[attached] ET_ACTUAL_PARAMETER_ITEM] */
extern T0* GE_new845(T6 a1, T1 initialize);
/* New instance of type ET_BRACKET_SYMBOL */
extern T0* GE_new846(T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [[attached] ET_ACTUAL_PARAMETER_ITEM] */
extern T0* GE_new848(T1 initialize);
/* New instance of type C_DATE */
extern T0* GE_new850(T1 initialize);
/* New instance of type KL_STDIN_FILE */
extern T0* GE_new851(T1 initialize);
/* New instance of type [attached] DS_ARRAYED_STACK [[attached] INTEGER_32] */
extern T0* GE_new853(T1 initialize);
/* New instance of type [attached] DS_ARRAYED_STACK [detachable ET_FORMAL_ARGUMENT_LIST] */
extern T0* GE_new854(T1 initialize);
/* New instance of type [attached] DS_ARRAYED_STACK [detachable ET_LOCAL_VARIABLE_LIST] */
extern T0* GE_new855(T1 initialize);
/* New instance of type [attached] DS_ARRAYED_STACK [detachable ET_KEYWORD] */
extern T0* GE_new856(T1 initialize);
/* New instance of type [attached] DS_ARRAYED_STACK [detachable ET_SYMBOL] */
extern T0* GE_new857(T1 initialize);
/* New instance of type [attached] DS_ARRAYED_STACK [detachable ET_OBJECT_TEST_LIST] */
extern T0* GE_new858(T1 initialize);
/* New instance of type [attached] DS_ARRAYED_STACK [[attached] ET_OBJECT_TEST_LIST] */
extern T0* GE_new859(T1 initialize);
/* New instance of type [attached] DS_ARRAYED_STACK [detachable ET_ACROSS_COMPONENT_LIST] */
extern T0* GE_new860(T1 initialize);
/* New instance of type [attached] DS_ARRAYED_STACK [[attached] ET_ACROSS_COMPONENT_LIST] */
extern T0* GE_new861(T1 initialize);
/* New instance of type [attached] DS_ARRAYED_LIST [[attached] ET_ASSERTION_ITEM] */
extern T0* GE_new862(T1 initialize);
/* New instance of type [attached] DS_ARRAYED_LIST [[attached] ET_QUERY] */
extern T0* GE_new863(T1 initialize);
/* New instance of type [attached] DS_ARRAYED_LIST [[attached] ET_PROCEDURE] */
extern T0* GE_new864(T1 initialize);
/* New instance of type [attached] DS_ARRAYED_LIST [detachable ET_CONSTRAINT_TYPE] */
extern T0* GE_new865(T1 initialize);
/* New instance of type [attached] DS_HASH_SET [[attached] ET_NAMED_CLASS] */
extern T0* GE_new866(T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_KEYWORD] */
extern T0* GE_new867(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_KEYWORD] */
extern T0* GE_new869(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_AGENT_KEYWORD] */
extern T0* GE_new870(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_AGENT_KEYWORD] */
extern T0* GE_new872(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_PRECURSOR_KEYWORD] */
extern T0* GE_new873(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_PRECURSOR_KEYWORD] */
extern T0* GE_new875(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_SYMBOL] */
extern T0* GE_new876(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_SYMBOL] */
extern T0* GE_new878(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_POSITION] */
extern T0* GE_new879(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_POSITION] */
extern T0* GE_new881(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_BOOLEAN_CONSTANT] */
extern T0* GE_new882(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_BOOLEAN_CONSTANT] */
extern T0* GE_new884(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_BREAK] */
extern T0* GE_new885(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_BREAK] */
extern T0* GE_new887(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_CHARACTER_CONSTANT] */
extern T0* GE_new888(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_CHARACTER_CONSTANT] */
extern T0* GE_new890(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_CURRENT] */
extern T0* GE_new891(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_CURRENT] */
extern T0* GE_new893(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_FREE_OPERATOR] */
extern T0* GE_new894(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_FREE_OPERATOR] */
extern T0* GE_new896(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_INTEGER_CONSTANT] */
extern T0* GE_new897(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_INTEGER_CONSTANT] */
extern T0* GE_new899(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_KEYWORD_OPERATOR] */
extern T0* GE_new900(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_KEYWORD_OPERATOR] */
extern T0* GE_new902(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_MANIFEST_STRING] */
extern T0* GE_new903(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_MANIFEST_STRING] */
extern T0* GE_new905(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_REAL_CONSTANT] */
extern T0* GE_new906(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_REAL_CONSTANT] */
extern T0* GE_new908(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_RESULT] */
extern T0* GE_new909(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_RESULT] */
extern T0* GE_new910(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_RETRY_INSTRUCTION] */
extern T0* GE_new911(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_RETRY_INSTRUCTION] */
extern T0* GE_new913(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_SYMBOL_OPERATOR] */
extern T0* GE_new914(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_SYMBOL_OPERATOR] */
extern T0* GE_new916(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_VOID] */
extern T0* GE_new917(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_VOID] */
extern T0* GE_new919(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_SEMICOLON_SYMBOL] */
extern T0* GE_new920(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_SEMICOLON_SYMBOL] */
extern T0* GE_new922(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_BRACKET_SYMBOL] */
extern T0* GE_new923(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_BRACKET_SYMBOL] */
extern T0* GE_new925(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_QUESTION_MARK_SYMBOL] */
extern T0* GE_new926(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_QUESTION_MARK_SYMBOL] */
extern T0* GE_new928(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_ACROSS_EXPRESSION] */
extern T0* GE_new929(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_ACROSS_EXPRESSION] */
extern T0* GE_new931(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_ACROSS_INSTRUCTION] */
extern T0* GE_new932(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_ACROSS_INSTRUCTION] */
extern T0* GE_new934(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_ACTUAL_ARGUMENT_LIST] */
extern T0* GE_new935(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_ACTUAL_ARGUMENT_LIST] */
extern T0* GE_new936(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_ACTUAL_PARAMETER_ITEM] */
extern T0* GE_new937(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_ACTUAL_PARAMETER_ITEM] */
extern T0* GE_new939(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_ACTUAL_PARAMETER_LIST] */
extern T0* GE_new940(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_ACTUAL_PARAMETER_LIST] */
extern T0* GE_new941(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_AGENT_ARGUMENT_OPERAND] */
extern T0* GE_new942(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_AGENT_ARGUMENT_OPERAND] */
extern T0* GE_new944(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_AGENT_ARGUMENT_OPERAND_ITEM] */
extern T0* GE_new945(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_AGENT_ARGUMENT_OPERAND_ITEM] */
extern T0* GE_new947(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_AGENT_ARGUMENT_OPERAND_LIST] */
extern T0* GE_new948(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_AGENT_ARGUMENT_OPERAND_LIST] */
extern T0* GE_new950(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_AGENT_TARGET] */
extern T0* GE_new951(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_AGENT_TARGET] */
extern T0* GE_new953(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_ALIAS_NAME] */
extern T0* GE_new954(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_ALIAS_NAME] */
extern T0* GE_new956(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_ASSIGNER] */
extern T0* GE_new957(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_ASSIGNER] */
extern T0* GE_new959(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_BRACKET_EXPRESSION] */
extern T0* GE_new960(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_BRACKET_EXPRESSION] */
extern T0* GE_new962(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_CALL_AGENT] */
extern T0* GE_new963(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_CALL_AGENT] */
extern T0* GE_new965(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_QUALIFIED_CALL_EXPRESSION] */
extern T0* GE_new966(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_QUALIFIED_CALL_EXPRESSION] */
extern T0* GE_new967(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_CHECK_INSTRUCTION] */
extern T0* GE_new968(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_CHECK_INSTRUCTION] */
extern T0* GE_new970(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_CHOICE] */
extern T0* GE_new971(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_CHOICE] */
extern T0* GE_new973(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_CHOICE_CONSTANT] */
extern T0* GE_new974(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_CHOICE_CONSTANT] */
extern T0* GE_new976(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_CHOICE_ITEM] */
extern T0* GE_new977(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_CHOICE_ITEM] */
extern T0* GE_new979(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_CHOICE_LIST] */
extern T0* GE_new980(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_CHOICE_LIST] */
extern T0* GE_new982(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_CLASS] */
extern T0* GE_new983(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_CLASS] */
extern T0* GE_new985(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_CLIENT_ITEM] */
extern T0* GE_new986(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_CLIENT_ITEM] */
extern T0* GE_new988(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_CLIENTS] */
extern T0* GE_new989(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_CLIENTS] */
extern T0* GE_new991(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_COMPOUND] */
extern T0* GE_new992(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_COMPOUND] */
extern T0* GE_new993(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_CONDITIONAL] */
extern T0* GE_new994(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_CONDITIONAL] */
extern T0* GE_new996(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_CONSTANT] */
extern T0* GE_new997(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_CONSTANT] */
extern T0* GE_new999(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_CONSTRAINT_ACTUAL_PARAMETER_ITEM] */
extern T0* GE_new1000(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_CONSTRAINT_ACTUAL_PARAMETER_ITEM] */
extern T0* GE_new1002(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_CONSTRAINT_ACTUAL_PARAMETER_LIST] */
extern T0* GE_new1003(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_CONSTRAINT_ACTUAL_PARAMETER_LIST] */
extern T0* GE_new1005(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_CONSTRAINT_CREATOR] */
extern T0* GE_new1006(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_CONSTRAINT_CREATOR] */
extern T0* GE_new1008(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_CONSTRAINT_TYPE] */
extern T0* GE_new1009(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_CONSTRAINT_TYPE] */
extern T0* GE_new1011(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_CONVERT_FEATURE] */
extern T0* GE_new1012(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_CONVERT_FEATURE] */
extern T0* GE_new1014(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_CONVERT_FEATURE_ITEM] */
extern T0* GE_new1015(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_CONVERT_FEATURE_ITEM] */
extern T0* GE_new1017(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_CONVERT_FEATURE_LIST] */
extern T0* GE_new1018(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_CONVERT_FEATURE_LIST] */
extern T0* GE_new1020(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_CREATE_EXPRESSION] */
extern T0* GE_new1021(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_CREATE_EXPRESSION] */
extern T0* GE_new1023(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_CREATION_REGION] */
extern T0* GE_new1024(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_CREATION_REGION] */
extern T0* GE_new1026(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_CREATOR] */
extern T0* GE_new1027(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_CREATOR] */
extern T0* GE_new1029(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_CREATOR_LIST] */
extern T0* GE_new1030(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_CREATOR_LIST] */
extern T0* GE_new1032(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_DEBUG_INSTRUCTION] */
extern T0* GE_new1033(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_DEBUG_INSTRUCTION] */
extern T0* GE_new1035(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_ELSEIF_EXPRESSION] */
extern T0* GE_new1036(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_ELSEIF_EXPRESSION] */
extern T0* GE_new1038(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_ELSEIF_EXPRESSION_LIST] */
extern T0* GE_new1039(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_ELSEIF_EXPRESSION_LIST] */
extern T0* GE_new1041(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_ELSEIF_PART] */
extern T0* GE_new1042(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_ELSEIF_PART] */
extern T0* GE_new1044(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_ELSEIF_PART_LIST] */
extern T0* GE_new1045(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_ELSEIF_PART_LIST] */
extern T0* GE_new1047(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_EXPORT] */
extern T0* GE_new1048(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_EXPORT] */
extern T0* GE_new1050(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_EXPORT_LIST] */
extern T0* GE_new1051(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_EXPORT_LIST] */
extern T0* GE_new1052(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_EXPRESSION] */
extern T0* GE_new1053(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_EXPRESSION] */
extern T0* GE_new1054(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_EXPRESSION_ITEM] */
extern T0* GE_new1055(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_EXPRESSION_ITEM] */
extern T0* GE_new1057(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_EXTENDED_FEATURE_NAME] */
extern T0* GE_new1058(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_EXTENDED_FEATURE_NAME] */
extern T0* GE_new1060(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_EXTERNAL_ALIAS] */
extern T0* GE_new1061(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_EXTERNAL_ALIAS] */
extern T0* GE_new1063(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_FEATURE_CLAUSE] */
extern T0* GE_new1064(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_FEATURE_CLAUSE] */
extern T0* GE_new1066(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_FEATURE_CLAUSE_LIST] */
extern T0* GE_new1067(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_FEATURE_CLAUSE_LIST] */
extern T0* GE_new1069(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_FEATURE_EXPORT] */
extern T0* GE_new1070(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_FEATURE_EXPORT] */
extern T0* GE_new1072(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_FEATURE_NAME] */
extern T0* GE_new1073(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_FEATURE_NAME] */
extern T0* GE_new1074(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_FEATURE_NAME_ITEM] */
extern T0* GE_new1075(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_FEATURE_NAME_ITEM] */
extern T0* GE_new1077(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_FORMAL_ARGUMENT] */
extern T0* GE_new1078(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_FORMAL_ARGUMENT] */
extern T0* GE_new1080(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_FORMAL_ARGUMENT_ITEM] */
extern T0* GE_new1081(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_FORMAL_ARGUMENT_ITEM] */
extern T0* GE_new1083(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_FORMAL_ARGUMENT_LIST] */
extern T0* GE_new1084(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_FORMAL_ARGUMENT_LIST] */
extern T0* GE_new1085(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_FORMAL_PARAMETER] */
extern T0* GE_new1086(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_FORMAL_PARAMETER] */
extern T0* GE_new1088(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_FORMAL_PARAMETER_ITEM] */
extern T0* GE_new1089(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_FORMAL_PARAMETER_ITEM] */
extern T0* GE_new1091(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_FORMAL_PARAMETER_LIST] */
extern T0* GE_new1092(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_FORMAL_PARAMETER_LIST] */
extern T0* GE_new1093(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_IF_EXPRESSION] */
extern T0* GE_new1094(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_IF_EXPRESSION] */
extern T0* GE_new1096(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_IF_INSTRUCTION] */
extern T0* GE_new1097(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_IF_INSTRUCTION] */
extern T0* GE_new1099(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_INDEXING_LIST] */
extern T0* GE_new1100(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_INDEXING_LIST] */
extern T0* GE_new1102(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_INDEXING_ITEM] */
extern T0* GE_new1103(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_INDEXING_ITEM] */
extern T0* GE_new1105(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_INDEXING_TERM] */
extern T0* GE_new1106(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_INDEXING_TERM] */
extern T0* GE_new1108(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_INDEXING_TERM_ITEM] */
extern T0* GE_new1109(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_INDEXING_TERM_ITEM] */
extern T0* GE_new1111(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_INDEXING_TERM_LIST] */
extern T0* GE_new1112(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_INDEXING_TERM_LIST] */
extern T0* GE_new1114(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_INLINE_AGENT] */
extern T0* GE_new1115(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_INLINE_AGENT] */
extern T0* GE_new1117(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_INSPECT_INSTRUCTION] */
extern T0* GE_new1118(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_INSPECT_INSTRUCTION] */
extern T0* GE_new1120(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_INSTRUCTION] */
extern T0* GE_new1121(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_INSTRUCTION] */
extern T0* GE_new1123(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_INVARIANTS] */
extern T0* GE_new1124(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_INVARIANTS] */
extern T0* GE_new1126(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_KEYWORD_FEATURE_NAME_LIST] */
extern T0* GE_new1127(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_KEYWORD_FEATURE_NAME_LIST] */
extern T0* GE_new1128(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_LIKE_TYPE] */
extern T0* GE_new1129(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_LIKE_TYPE] */
extern T0* GE_new1131(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_LOCAL_VARIABLE] */
extern T0* GE_new1132(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_LOCAL_VARIABLE] */
extern T0* GE_new1134(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_LOCAL_VARIABLE_ITEM] */
extern T0* GE_new1135(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_LOCAL_VARIABLE_ITEM] */
extern T0* GE_new1137(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_LOCAL_VARIABLE_LIST] */
extern T0* GE_new1138(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_LOCAL_VARIABLE_LIST] */
extern T0* GE_new1139(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_LOOP_INVARIANTS] */
extern T0* GE_new1140(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_LOOP_INVARIANTS] */
extern T0* GE_new1142(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_MANIFEST_ARRAY] */
extern T0* GE_new1143(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_MANIFEST_ARRAY] */
extern T0* GE_new1145(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_MANIFEST_STRING_ITEM] */
extern T0* GE_new1146(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_MANIFEST_STRING_ITEM] */
extern T0* GE_new1148(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_MANIFEST_STRING_LIST] */
extern T0* GE_new1149(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_MANIFEST_STRING_LIST] */
extern T0* GE_new1151(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_MANIFEST_TUPLE] */
extern T0* GE_new1152(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_MANIFEST_TUPLE] */
extern T0* GE_new1153(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_OBSOLETE] */
extern T0* GE_new1154(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_OBSOLETE] */
extern T0* GE_new1156(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_PARENTHESIZED_EXPRESSION] */
extern T0* GE_new1157(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_PARENTHESIZED_EXPRESSION] */
extern T0* GE_new1159(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_PARENT] */
extern T0* GE_new1160(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_PARENT] */
extern T0* GE_new1162(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_PARENT_CLAUSE_LIST] */
extern T0* GE_new1163(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_PARENT_CLAUSE_LIST] */
extern T0* GE_new1165(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_PARENT_ITEM] */
extern T0* GE_new1166(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_PARENT_ITEM] */
extern T0* GE_new1168(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_PARENT_LIST] */
extern T0* GE_new1169(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_PARENT_LIST] */
extern T0* GE_new1171(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_POSTCONDITIONS] */
extern T0* GE_new1172(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_POSTCONDITIONS] */
extern T0* GE_new1174(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_PRECONDITIONS] */
extern T0* GE_new1175(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_PRECONDITIONS] */
extern T0* GE_new1176(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_PROCEDURE] */
extern T0* GE_new1177(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_PROCEDURE] */
extern T0* GE_new1178(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_QUALIFIED_LIKE_IDENTIFIER] */
extern T0* GE_new1179(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_QUALIFIED_LIKE_IDENTIFIER] */
extern T0* GE_new1181(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_QUERY] */
extern T0* GE_new1182(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_QUERY] */
extern T0* GE_new1183(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_RENAME_ITEM] */
extern T0* GE_new1184(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_RENAME_ITEM] */
extern T0* GE_new1186(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_RENAME_LIST] */
extern T0* GE_new1187(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_RENAME_LIST] */
extern T0* GE_new1188(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_STATIC_CALL_EXPRESSION] */
extern T0* GE_new1189(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_STATIC_CALL_EXPRESSION] */
extern T0* GE_new1191(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_STRIP_EXPRESSION] */
extern T0* GE_new1192(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_STRIP_EXPRESSION] */
extern T0* GE_new1194(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_TYPE] */
extern T0* GE_new1195(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_TYPE] */
extern T0* GE_new1196(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_TYPE_ITEM] */
extern T0* GE_new1197(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_TYPE_ITEM] */
extern T0* GE_new1199(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_TYPE_LIST] */
extern T0* GE_new1200(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_TYPE_LIST] */
extern T0* GE_new1202(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_VARIANT] */
extern T0* GE_new1203(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_VARIANT] */
extern T0* GE_new1205(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_WHEN_PART] */
extern T0* GE_new1206(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_WHEN_PART] */
extern T0* GE_new1208(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_WHEN_PART_LIST] */
extern T0* GE_new1209(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_WHEN_PART_LIST] */
extern T0* GE_new1211(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_WRITABLE] */
extern T0* GE_new1212(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_WRITABLE] */
extern T0* GE_new1213(T6 a1, T1 initialize);
/* New instance of type [attached] DS_HASH_TOPOLOGICAL_SORTER [[attached] ET_CLASS] */
extern T0* GE_new1214(T1 initialize);
/* New instance of type [attached] DS_HASH_TABLE [[attached] ET_BASE_TYPE, [attached] ET_CLASS] */
extern T0* GE_new1215(T1 initialize);
/* New instance of type ET_PARENT_CHECKER1 */
extern T0* GE_new1216(T1 initialize);
/* New instance of type ET_FORMAL_PARAMETER_CHECKER1 */
extern T0* GE_new1217(T1 initialize);
/* New instance of type [attached] DS_HASH_TABLE [[attached] ET_FLATTENED_FEATURE, [attached] ET_FEATURE_NAME] */
extern T0* GE_new1218(T1 initialize);
/* New instance of type ET_FEATURE_NAME_TESTER */
extern T0* GE_new1219(T1 initialize);
/* New instance of type [attached] DS_HASH_TABLE [[attached] ET_FLATTENED_FEATURE, [attached] ET_ALIAS_NAME] */
extern T0* GE_new1221(T1 initialize);
/* New instance of type ET_ALIAS_NAME_TESTER */
extern T0* GE_new1222(T1 initialize);
/* New instance of type [attached] DS_ARRAYED_LIST [[attached] ET_CLIENT_LIST] */
extern T0* GE_new1224(T1 initialize);
/* New instance of type [attached] DS_HASH_TABLE [[attached] ET_CLIENT, [attached] ET_CLASS] */
extern T0* GE_new1225(T1 initialize);
/* New instance of type ET_FEATURE_ADAPTATION_RESOLVER */
extern T0* GE_new1226(T1 initialize);
/* New instance of type ET_DOTNET_FEATURE_ADAPTATION_RESOLVER */
extern T0* GE_new1227(T1 initialize);
/* New instance of type ET_IDENTIFIER_TYPE_RESOLVER */
extern T0* GE_new1228(T1 initialize);
/* New instance of type ET_UNFOLDED_TUPLE_ACTUAL_PARAMETERS_RESOLVER1 */
extern T0* GE_new1229(T1 initialize);
/* New instance of type ET_ANCHORED_TYPE_CHECKER */
extern T0* GE_new1230(T1 initialize);
/* New instance of type ET_SIGNATURE_CHECKER */
extern T0* GE_new1231(T1 initialize);
/* New instance of type ET_PARENT_CHECKER2 */
extern T0* GE_new1232(T1 initialize);
/* New instance of type ET_FORMAL_PARAMETER_CHECKER2 */
extern T0* GE_new1233(T1 initialize);
/* New instance of type ET_BUILTIN_FEATURE_CHECKER */
extern T0* GE_new1234(T1 initialize);
/* New instance of type ET_PRECURSOR_CHECKER */
extern T0* GE_new1235(T1 initialize);
/* New instance of type [attached] DS_HASH_TABLE [[attached] ET_FEATURE, [attached] INTEGER_32] */
extern T0* GE_new1236(T1 initialize);
/* New instance of type [attached] DS_HASH_SET [[attached] ET_CLASS] */
extern T0* GE_new1237(T1 initialize);
/* New instance of type ET_PARENT_CHECKER3 */
extern T0* GE_new1238(T1 initialize);
/* New instance of type ET_QUALIFIED_ANCHORED_TYPE_CHECKER */
extern T0* GE_new1239(T1 initialize);
/* New instance of type ET_UNFOLDED_TUPLE_ACTUAL_PARAMETERS_RESOLVER2 */
extern T0* GE_new1240(T1 initialize);
/* New instance of type ET_FORMAL_PARAMETER_LIST */
extern T0* GE_new1242(T1 initialize);
/* New instance of type ET_FORMAL_PARAMETER */
extern T0* GE_new1243(T1 initialize);
/* New instance of type ET_CONSTRAINT_CREATOR */
extern T0* GE_new1244(T1 initialize);
/* New instance of type ET_INHERITED_FEATURE */
extern T0* GE_new1246(T1 initialize);
/* New instance of type ET_PARENT_FEATURE */
extern T0* GE_new1247(T1 initialize);
/* New instance of type [attached] DS_HASH_SET [[attached] ET_PROCEDURE] */
extern T0* GE_new1248(T1 initialize);
/* New instance of type [attached] DS_HASH_SET [[attached] ET_QUERY] */
extern T0* GE_new1249(T1 initialize);
/* New instance of type ET_SUPPLIER_BUILDER */
extern T0* GE_new1250(T1 initialize);
/* New instance of type [attached] SPECIAL [[attached] ET_CLASS] */
extern T0* GE_new1251(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [[attached] ET_CLASS] */
extern T0* GE_new1252(T1 initialize);
/* New instance of type [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_CLASS] */
extern T0* GE_new1253(T1 initialize);
/* New instance of type [attached] DS_ARRAYED_LIST_CURSOR [[attached] INTEGER_32] */
extern T0* GE_new1255(T1 initialize);
/* New instance of type [attached] SPECIAL [[attached] ET_DYNAMIC_TYPE_SET] */
extern T0* GE_new1258(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [[attached] ET_DYNAMIC_TYPE_SET] */
extern T0* GE_new1259(T1 initialize);
/* New instance of type [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_IDENTIFIER] */
extern T0* GE_new1260(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_DYNAMIC_TYPE] */
extern T0* GE_new1261(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_DYNAMIC_TYPE] */
extern T0* GE_new1262(T1 initialize);
/* New instance of type [attached] DS_ARRAYED_LIST_CURSOR [detachable ET_DYNAMIC_TYPE] */
extern T0* GE_new1263(T1 initialize);
/* New instance of type [attached] SPECIAL [[attached] ET_EXPRESSION] */
extern T0* GE_new1265(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [[attached] ET_EXPRESSION] */
extern T0* GE_new1266(T1 initialize);
/* New instance of type [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_EXPRESSION] */
extern T0* GE_new1267(T1 initialize);
/* New instance of type attached KL_EQUALITY_TESTER [[attached] INTEGER_32] */
extern T0* GE_new1271(T1 initialize);
/* New instance of type [attached] DS_HASH_TABLE_CURSOR [[attached] ET_DYNAMIC_TYPE, [attached] INTEGER_32] */
extern T0* GE_new1273(T1 initialize);
/* New instance of type [attached] SPECIAL [[attached] ET_DYNAMIC_STANDALONE_TYPE_SET] */
extern T0* GE_new1274(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [[attached] ET_DYNAMIC_STANDALONE_TYPE_SET] */
extern T0* GE_new1275(T1 initialize);
/* New instance of type [attached] DS_HASH_SET_CURSOR [[attached] ET_DYNAMIC_TYPE] */
extern T0* GE_new1276(T1 initialize);
/* New instance of type [attached] DS_HASH_TABLE_CURSOR [[attached] ET_DYNAMIC_STANDALONE_TYPE_SET, [attached] ET_DYNAMIC_TYPE] */
extern T0* GE_new1279(T1 initialize);
/* New instance of type [attached] SPECIAL [[attached] ET_OBJECT_TEST] */
extern T0* GE_new1282(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [[attached] ET_OBJECT_TEST] */
extern T0* GE_new1283(T1 initialize);
/* New instance of type [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_OBJECT_TEST] */
extern T0* GE_new1284(T1 initialize);
/* New instance of type [attached] SPECIAL [[attached] ET_OBJECT_EQUALITY_EXPRESSION] */
extern T0* GE_new1286(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [[attached] ET_OBJECT_EQUALITY_EXPRESSION] */
extern T0* GE_new1287(T1 initialize);
/* New instance of type [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_OBJECT_EQUALITY_EXPRESSION] */
extern T0* GE_new1288(T1 initialize);
/* New instance of type [attached] SPECIAL [[attached] ET_EQUALITY_EXPRESSION] */
extern T0* GE_new1290(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [[attached] ET_EQUALITY_EXPRESSION] */
extern T0* GE_new1291(T1 initialize);
/* New instance of type [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_EQUALITY_EXPRESSION] */
extern T0* GE_new1292(T1 initialize);
/* New instance of type [attached] SPECIAL [[attached] ET_AGENT] */
extern T0* GE_new1294(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [[attached] ET_AGENT] */
extern T0* GE_new1295(T1 initialize);
/* New instance of type [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_AGENT] */
extern T0* GE_new1296(T1 initialize);
/* New instance of type [attached] SPECIAL [[attached] ET_EXPRESSION_ITEM] */
extern T0* GE_new1298(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [[attached] ET_EXPRESSION_ITEM] */
extern T0* GE_new1300(T1 initialize);
/* New instance of type [attached] DS_ARRAYED_LIST_CURSOR [detachable ET_QUALIFIED_CALL_EXPRESSION] */
extern T0* GE_new1301(T1 initialize);
/* New instance of type [attached] DS_HASH_SET_CURSOR [[attached] ET_DYNAMIC_TUPLE_TYPE] */
extern T0* GE_new1302(T1 initialize);
/* New instance of type [attached] SPECIAL [[attached] ET_DYNAMIC_TUPLE_TYPE] */
extern T0* GE_new1303(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [[attached] ET_DYNAMIC_TUPLE_TYPE] */
extern T0* GE_new1304(T1 initialize);
/* New instance of type [attached] DS_HASH_TABLE_CURSOR [[attached] INTEGER_32, [attached] ET_FEATURE] */
extern T0* GE_new1309(T1 initialize);
/* New instance of type [attached] SPECIAL [[attached] ET_FEATURE] */
extern T0* GE_new1310(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [[attached] ET_FEATURE] */
extern T0* GE_new1311(T1 initialize);
/* New instance of type [attached] DS_HASH_TABLE_CURSOR [[attached] ET_CONSTANT, [attached] ET_FEATURE] */
extern T0* GE_new1312(T1 initialize);
/* New instance of type [attached] SPECIAL [[attached] ET_CONSTANT] */
extern T0* GE_new1313(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [[attached] ET_CONSTANT] */
extern T0* GE_new1316(T1 initialize);
/* New instance of type [attached] DS_HASH_TABLE_CURSOR [[attached] ET_DYNAMIC_TYPE, [attached] ET_INLINE_CONSTANT] */
extern T0* GE_new1317(T1 initialize);
/* New instance of type [attached] SPECIAL [[attached] ET_INLINE_CONSTANT] */
extern T0* GE_new1318(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [[attached] ET_INLINE_CONSTANT] */
extern T0* GE_new1321(T1 initialize);
/* New instance of type [attached] SPECIAL [detachable ET_DYNAMIC_FEATURE] */
extern T0* GE_new1322(T6 a1, T1 initialize);
/* New instance of type [attached] DS_HASH_TABLE_CURSOR [detachable ET_DYNAMIC_FEATURE, [attached] ET_DYNAMIC_TYPE] */
extern T0* GE_new1325(T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [detachable ET_DYNAMIC_FEATURE] */
extern T0* GE_new1326(T1 initialize);
/* New instance of type [attached] SPECIAL [[attached] ET_DYNAMIC_FEATURE] */
extern T0* GE_new1327(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [[attached] ET_DYNAMIC_FEATURE] */
extern T0* GE_new1328(T1 initialize);
/* New instance of type [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_DYNAMIC_FEATURE] */
extern T0* GE_new1329(T1 initialize);
/* New instance of type [attached] DS_HASH_TABLE_CURSOR [[attached] BOOLEAN, [attached] STRING_8] */
extern T0* GE_new1331(T1 initialize);
/* New instance of type RX_BYTE_CODE */
extern T0* GE_new1337(T1 initialize);
/* New instance of type RX_CHARACTER_SET */
extern T0* GE_new1338(T1 initialize);
/* New instance of type RX_CASE_MAPPING */
extern T0* GE_new1339(T1 initialize);
/* New instance of type [attached] ARRAY [[attached] RX_CHARACTER_SET] */
extern T0* GE_new1342(T1 initialize);
/* New instance of type [attached] SPECIAL [[attached] RX_CHARACTER_SET] */
extern T0* GE_new1343(T6 a1, T1 initialize);
/* New instance of type [attached] DS_ARRAYED_LIST [detachable DS_LINKABLE [[attached] INTEGER_32]] */
extern T0* GE_new1345(T1 initialize);
/* New instance of type [attached] ARRAY [[attached] BOOLEAN] */
extern T0* GE_new1346(T1 initialize);
/* New instance of type [attached] SPECIAL [[attached] ET_SYSTEM_PROCESSOR] */
extern T0* GE_new1349(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [[attached] ET_SYSTEM_PROCESSOR] */
extern T0* GE_new1350(T1 initialize);
/* New instance of type [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_SYSTEM_PROCESSOR] */
extern T0* GE_new1351(T1 initialize);
/* New instance of type THREAD_ATTRIBUTES */
extern T0* GE_new1352(T1 initialize);
/* New instance of type [attached] CELL [[attached] BOOLEAN] */
extern T0* GE_new1353(T1 initialize);
/* New instance of type [attached] DS_HASH_TABLE_CURSOR [[attached] INTEGER_32, [attached] ET_DYNAMIC_TYPE] */
extern T0* GE_new1359(T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [[attached] ET_INLINE_AGENT] */
extern T0* GE_new1361(T1 initialize);
/* New instance of type [attached] SPECIAL [[attached] ET_INLINE_AGENT] */
extern T0* GE_new1363(T6 a1, T1 initialize);
/* New instance of type [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_INLINE_AGENT] */
extern T0* GE_new1364(T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [[attached] DS_ARRAYED_LIST [[attached] ET_PROCEDURE]] */
extern T0* GE_new1365(T1 initialize);
/* New instance of type [attached] SPECIAL [[attached] DS_ARRAYED_LIST [[attached] ET_PROCEDURE]] */
extern T0* GE_new1366(T6 a1, T1 initialize);
/* New instance of type [attached] DS_ARRAYED_LIST_CURSOR [[attached] DS_ARRAYED_LIST [[attached] ET_PROCEDURE]] */
extern T0* GE_new1367(T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [[attached] DS_ARRAYED_LIST [[attached] ET_QUERY]] */
extern T0* GE_new1368(T1 initialize);
/* New instance of type [attached] SPECIAL [[attached] DS_ARRAYED_LIST [[attached] ET_QUERY]] */
extern T0* GE_new1369(T6 a1, T1 initialize);
/* New instance of type [attached] DS_ARRAYED_LIST_CURSOR [[attached] DS_ARRAYED_LIST [[attached] ET_QUERY]] */
extern T0* GE_new1370(T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [[attached] DS_ARRAYED_LIST [[attached] ET_FEATURE]] */
extern T0* GE_new1371(T1 initialize);
/* New instance of type [attached] DS_ARRAYED_LIST [[attached] ET_FEATURE] */
extern T0* GE_new1372(T1 initialize);
/* New instance of type [attached] SPECIAL [[attached] DS_ARRAYED_LIST [[attached] ET_FEATURE]] */
extern T0* GE_new1373(T6 a1, T1 initialize);
/* New instance of type [attached] DS_ARRAYED_LIST_CURSOR [[attached] DS_ARRAYED_LIST [[attached] ET_FEATURE]] */
extern T0* GE_new1374(T1 initialize);
/* New instance of type [attached] SPECIAL [[attached] ET_NESTED_TYPE_CONTEXT] */
extern T0* GE_new1375(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [[attached] ET_NESTED_TYPE_CONTEXT] */
extern T0* GE_new1376(T1 initialize);
/* New instance of type [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_NESTED_TYPE_CONTEXT] */
extern T0* GE_new1377(T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [[attached] ET_TYPE] */
extern T0* GE_new1379(T1 initialize);
/* New instance of type [attached] DS_HASH_TABLE_CURSOR [[attached] ET_NESTED_TYPE_CONTEXT, [attached] ET_NAMED_OBJECT_TEST] */
extern T0* GE_new1380(T1 initialize);
/* New instance of type ET_NAMED_OBJECT_TEST */
extern T0* GE_new1384(T1 initialize);
/* New instance of type [attached] SPECIAL [[attached] ET_NAMED_OBJECT_TEST] */
extern T0* GE_new1385(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [[attached] ET_NAMED_OBJECT_TEST] */
extern T0* GE_new1386(T1 initialize);
/* New instance of type [attached] DS_ARRAYED_LIST [[attached] ET_NAMED_OBJECT_TEST] */
extern T0* GE_new1387(T1 initialize);
/* New instance of type [attached] DS_HASH_TABLE_CURSOR [[attached] ET_NESTED_TYPE_CONTEXT, [attached] ET_ACROSS_COMPONENT] */
extern T0* GE_new1388(T1 initialize);
/* New instance of type [attached] SPECIAL [[attached] ET_ACROSS_COMPONENT] */
extern T0* GE_new1392(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [[attached] ET_ACROSS_COMPONENT] */
extern T0* GE_new1393(T1 initialize);
/* New instance of type [attached] DS_ARRAYED_LIST [[attached] ET_ACROSS_COMPONENT] */
extern T0* GE_new1394(T1 initialize);
/* New instance of type [attached] DS_HASH_SET [[attached] INTEGER_32] */
extern T0* GE_new1395(T1 initialize);
/* New instance of type [attached] SPECIAL [[attached] ET_ATTACHMENT_SCOPE] */
extern T0* GE_new1398(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [[attached] ET_ATTACHMENT_SCOPE] */
extern T0* GE_new1399(T1 initialize);
/* New instance of type [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_ATTACHMENT_SCOPE] */
extern T0* GE_new1400(T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [[attached] ET_INDEXING_TERM] */
extern T0* GE_new1402(T1 initialize);
/* New instance of type [attached] SPECIAL [[attached] ET_INDEXING_TERM] */
extern T0* GE_new1404(T6 a1, T1 initialize);
/* New instance of type [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_INDEXING_TERM] */
extern T0* GE_new1405(T1 initialize);
/* New instance of type [attached] DS_HASH_TABLE_CURSOR [[attached] ET_ASSERTIONS, [attached] ET_FEATURE] */
extern T0* GE_new1406(T1 initialize);
/* New instance of type [attached] SPECIAL [[attached] ET_ASSERTIONS] */
extern T0* GE_new1407(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [[attached] ET_ASSERTIONS] */
extern T0* GE_new1409(T1 initialize);
/* New instance of type [attached] DS_HASH_TABLE [[attached] NATURAL_8, [attached] ET_CLASS_NAME] */
extern T0* GE_new1411(T1 initialize);
/* New instance of type [attached] DS_HASH_TABLE_CURSOR [[attached] ET_IDENTIFIER, [attached] STRING_8] */
extern T0* GE_new1415(T1 initialize);
/* New instance of type STRING_TO_INTEGER_CONVERTOR */
extern T0* GE_new1416(T1 initialize);
/* New instance of type ET_VALIDITY_ERROR */
extern T0* GE_new1418(T1 initialize);
/* New instance of type UC_UNICODE_ROUTINES */
extern T0* GE_new1419(T1 initialize);
/* New instance of type ET_XACE_CLASS_OPTIONS */
extern T0* GE_new1422(T1 initialize);
/* New instance of type attached DS_LINKED_LIST [[attached] ET_XACE_FEATURE_OPTIONS] */
extern T0* GE_new1424(T1 initialize);
/* New instance of type ET_XACE_FEATURE_OPTIONS */
extern T0* GE_new1426(T1 initialize);
/* New instance of type ET_XACE_WRONG_ROOT_ELEMENT_ERROR */
extern T0* GE_new1427(T1 initialize);
/* New instance of type ET_XACE_MISSING_ATTRIBUTE_ERROR */
extern T0* GE_new1428(T1 initialize);
/* New instance of type ET_XACE_UNKNOWN_ELEMENT_ERROR */
extern T0* GE_new1429(T1 initialize);
/* New instance of type ET_XACE_NON_EMPTY_ATTRIBUTE_EXPECTED_ERROR */
extern T0* GE_new1430(T1 initialize);
/* New instance of type ET_XACE_MISSING_ELEMENT_ERROR */
extern T0* GE_new1431(T1 initialize);
/* New instance of type ET_XACE_OBSOLETE_CLUSTER_ELEMENT_WARNING */
extern T0* GE_new1432(T1 initialize);
/* New instance of type ET_XACE_OBSOLETE_EXCLUDE_ELEMENT_WARNING */
extern T0* GE_new1433(T1 initialize);
/* New instance of type ET_XACE_ATTRIBUTE_OBSOLETED_BY_ELEMENT_WARNING */
extern T0* GE_new1434(T1 initialize);
/* New instance of type ET_ECF_NOTE_ELEMENT */
extern T0* GE_new1438(T1 initialize);
/* New instance of type [attached] DS_ARRAYED_LIST [[attached] ET_ECF_NOTE_ELEMENT] */
extern T0* GE_new1439(T1 initialize);
/* New instance of type ET_ECF_TARGET_PARENT */
extern T0* GE_new1453(T1 initialize);
/* New instance of type ET_ECF_ADAPTED_PRECOMPILED_LIBRARY */
extern T0* GE_new1455(T1 initialize);
/* New instance of type ET_ECF_VERSION */
extern T0* GE_new1459(T1 initialize);
/* New instance of type ET_ECF_ADAPTED_DOTNET_ASSEMBLIES */
extern T0* GE_new1461(T1 initialize);
/* New instance of type ET_ECF_CLUSTER */
extern T0* GE_new1463(T1 initialize);
/* New instance of type ET_ECF_CLUSTERS */
extern T0* GE_new1464(T1 initialize);
/* New instance of type ET_ECF_EXTERNAL_CFLAG */
extern T0* GE_new1466(T1 initialize);
/* New instance of type ET_ECF_EXTERNAL_CFLAGS */
extern T0* GE_new1467(T1 initialize);
/* New instance of type ET_ECF_EXTERNAL_INCLUDE */
extern T0* GE_new1469(T1 initialize);
/* New instance of type ET_ECF_EXTERNAL_INCLUDES */
extern T0* GE_new1470(T1 initialize);
/* New instance of type ET_ECF_EXTERNAL_LIBRARY */
extern T0* GE_new1472(T1 initialize);
/* New instance of type ET_ECF_EXTERNAL_LIBRARIES */
extern T0* GE_new1473(T1 initialize);
/* New instance of type ET_ECF_EXTERNAL_LINKER_FLAG */
extern T0* GE_new1475(T1 initialize);
/* New instance of type ET_ECF_EXTERNAL_LINKER_FLAGS */
extern T0* GE_new1476(T1 initialize);
/* New instance of type ET_ECF_EXTERNAL_MAKE */
extern T0* GE_new1478(T1 initialize);
/* New instance of type ET_ECF_EXTERNAL_MAKES */
extern T0* GE_new1479(T1 initialize);
/* New instance of type ET_ECF_EXTERNAL_OBJECT */
extern T0* GE_new1481(T1 initialize);
/* New instance of type ET_ECF_EXTERNAL_OBJECTS */
extern T0* GE_new1482(T1 initialize);
/* New instance of type ET_ECF_EXTERNAL_RESOURCE */
extern T0* GE_new1484(T1 initialize);
/* New instance of type ET_ECF_EXTERNAL_RESOURCES */
extern T0* GE_new1485(T1 initialize);
/* New instance of type ET_ECF_FILE_RULE */
extern T0* GE_new1487(T1 initialize);
/* New instance of type ET_ECF_FILE_RULES */
extern T0* GE_new1488(T1 initialize);
/* New instance of type ET_ECF_ADAPTED_LIBRARIES */
extern T0* GE_new1490(T1 initialize);
/* New instance of type ET_ECF_ACTION */
extern T0* GE_new1492(T1 initialize);
/* New instance of type [attached] DS_ARRAYED_LIST [[attached] ET_ECF_ACTION] */
extern T0* GE_new1493(T1 initialize);
/* New instance of type [attached] TUPLE [[attached] ET_ECF_OPTIONS] */
extern T0* GE_new1497(T1 initialize);
/* New instance of type [attached] PROCEDURE [[attached] TUPLE [[attached] ET_ECF_OPTIONS]] */
extern T0* GE_new1498(T1 initialize);
/* New instance of type ET_ECF_ROOT_ALL_CLASSES */
extern T0* GE_new1500(T1 initialize);
/* New instance of type ET_ECF_ROOT_CLASS */
extern T0* GE_new1501(T1 initialize);
/* New instance of type ET_ECF_ORED_CONDITIONS */
extern T0* GE_new1503(T1 initialize);
/* New instance of type ET_ECF_ANDED_CONDITIONS */
extern T0* GE_new1505(T1 initialize);
/* New instance of type [attached] DS_HASH_TABLE [[attached] ET_ECF_OPTIONS, [attached] STRING_8] */
extern T0* GE_new1506(T1 initialize);
/* New instance of type ET_ECF_VISIBLE_CLASS */
extern T0* GE_new1508(T1 initialize);
/* New instance of type [attached] DS_ARRAYED_LIST [[attached] ET_ECF_VISIBLE_CLASS] */
extern T0* GE_new1509(T1 initialize);
/* New instance of type ET_ECF_BUILD_CONDITION */
extern T0* GE_new1518(T1 initialize);
/* New instance of type ET_ECF_CONCURRENCY_CONDITION */
extern T0* GE_new1519(T1 initialize);
/* New instance of type ET_ECF_CUSTOM_CONDITION */
extern T0* GE_new1520(T1 initialize);
/* New instance of type ET_ECF_DOTNET_CONDITION */
extern T0* GE_new1521(T1 initialize);
/* New instance of type ET_ECF_DYNAMIC_RUNTIME_CONDITION */
extern T0* GE_new1522(T1 initialize);
/* New instance of type ET_ECF_PLATFORM_CONDITION */
extern T0* GE_new1523(T1 initialize);
/* New instance of type ET_ECF_COMPILER_VERSION_CONDITION */
extern T0* GE_new1524(T1 initialize);
/* New instance of type ET_ECF_MSIL_CLR_VERSION_CONDITION */
extern T0* GE_new1525(T1 initialize);
/* New instance of type [attached] TUPLE [ET_ECF_SYSTEM] */
extern T0* GE_new1526(T1 initialize);
/* New instance of type ET_PARENTHESIS_EXPRESSION */
extern T0* GE_new1528(T1 initialize);
/* New instance of type ET_REGULAR_MANIFEST_STRING */
extern T0* GE_new1532(T1 initialize);
/* New instance of type ET_INFIX_FREE_NAME */
extern T0* GE_new1533(T1 initialize);
/* New instance of type ET_PARENT_CLAUSE_LIST */
extern T0* GE_new1534(T1 initialize);
/* New instance of type ET_CREATOR_LIST */
extern T0* GE_new1537(T1 initialize);
/* New instance of type [attached] XM_LINKED_LIST [[attached] XM_DOCUMENT_NODE] */
extern T0* GE_new1540(T1 initialize);
/* New instance of type [attached] XM_LINKED_LIST [[attached] XM_ELEMENT_NODE] */
extern T0* GE_new1542(T1 initialize);
/* New instance of type UT_TRISTATE */
extern T0* GE_new1546(T1 initialize);
/* New instance of type [attached] KL_AGENT_ROUTINES [[attached] ANY] */
extern T0* GE_new1547(T1 initialize);
/* New instance of type [attached] TUPLE [UT_TRISTATE] */
extern T0* GE_new1548(T1 initialize);
/* New instance of type [attached] TUPLE [[attached] KL_AGENT_ROUTINES [[attached] ANY], [attached] PROCEDURE [[attached] TUPLE]] */
extern T0* GE_new1549(T1 initialize);
/* New instance of type [attached] PREDICATE [[attached] TUPLE [[attached] ET_MASTER_CLASS]] */
extern T0* GE_new1550(T1 initialize);
/* New instance of type [attached] PREDICATE [[attached] TUPLE] */
extern T0* GE_new1551(T1 initialize);
/* New instance of type [attached] TUPLE [[attached] PROCEDURE [[attached] TUPLE [[attached] ET_MASTER_CLASS]], detachable FUNCTION [[attached] TUPLE, [attached] BOOLEAN]] */
extern T0* GE_new1553(T1 initialize);
/* New instance of type [attached] TUPLE [[attached] PROCEDURE [[attached] TUPLE [[attached] ET_MASTER_CLASS]], [attached] FUNCTION [[attached] TUPLE [[attached] ET_MASTER_CLASS], [attached] BOOLEAN]] */
extern T0* GE_new1554(T1 initialize);
/* New instance of type [attached] TUPLE [[attached] PROCEDURE [[attached] TUPLE [[attached] ET_MASTER_CLASS]], [attached] FUNCTION [[attached] TUPLE [[attached] ET_MASTER_CLASS], [attached] BOOLEAN], detachable FUNCTION [[attached] TUPLE, [attached] BOOLEAN]] */
extern T0* GE_new1555(T1 initialize);
/* New instance of type UC_UTF8_ROUTINES */
extern T0* GE_new1556(T1 initialize);
/* New instance of type UC_UTF8_STRING */
extern T0* GE_new1557(T1 initialize);
/* New instance of type XM_EIFFEL_INPUT_STREAM */
extern T0* GE_new1558(T1 initialize);
/* New instance of type [attached] DS_ARRAYED_LIST [[attached] ET_ECF_TARGET] */
extern T0* GE_new1562(T1 initialize);
/* New instance of type [attached] DS_ARRAYED_LIST [[attached] ET_ADAPTED_UNIVERSE] */
extern T0* GE_new1564(T1 initialize);
/* New instance of type [attached] DS_ARRAYED_LIST [[attached] ET_ECF_TARGET_PARENT] */
extern T0* GE_new1566(T1 initialize);
/* New instance of type KL_CHARACTER_BUFFER */
extern T0* GE_new1569(T1 initialize);
/* New instance of type [attached] SPECIAL [[attached] ET_QUERY] */
extern T0* GE_new1572(T6 a1, T1 initialize);
/* New instance of type ET_ALIAS_NAME */
extern T0* GE_new1573(T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [[attached] ET_QUERY] */
extern T0* GE_new1574(T1 initialize);
/* New instance of type [attached] SPECIAL [[attached] ET_PROCEDURE] */
extern T0* GE_new1575(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [[attached] ET_PROCEDURE] */
extern T0* GE_new1576(T1 initialize);
/* New instance of type KL_STRING_INPUT_STREAM */
extern T0* GE_new1577(T1 initialize);
/* New instance of type ET_TEXT_GROUP */
extern T0* GE_new1578(T1 initialize);
/* New instance of type ET_OBJECT_TEST_LIST */
extern T0* GE_new1583(T1 initialize);
/* New instance of type ET_ACROSS_COMPONENT_LIST */
extern T0* GE_new1585(T1 initialize);
/* New instance of type ET_CREATOR */
extern T0* GE_new1625(T1 initialize);
/* New instance of type ET_FEATURE_CLAUSE */
extern T0* GE_new1631(T1 initialize);
/* New instance of type ET_LOCAL_VARIABLE */
extern T0* GE_new1655(T1 initialize);
/* New instance of type ET_ELSEIF_PART */
extern T0* GE_new1675(T1 initialize);
/* New instance of type ET_ELSEIF_EXPRESSION */
extern T0* GE_new1676(T1 initialize);
/* New instance of type ET_WHEN_PART */
extern T0* GE_new1677(T1 initialize);
/* New instance of type ET_CONSTRAINED_FORMAL_PARAMETER */
extern T0* GE_new1712(T1 initialize);
/* New instance of type ET_FILE_POSITION */
extern T0* GE_new1716(T1 initialize);
/* New instance of type ET_TAGGED_ASSERTION */
extern T0* GE_new1718(T1 initialize);
/* New instance of type [attached] DS_QUICK_SORTER [[attached] ET_QUERY] */
extern T0* GE_new1721(T1 initialize);
/* New instance of type [attached] DS_QUICK_SORTER [[attached] ET_PROCEDURE] */
extern T0* GE_new1722(T1 initialize);
/* New instance of type ET_REDECLARED_FEATURE */
extern T0* GE_new1724(T1 initialize);
/* New instance of type ET_CONVERT_FEATURE_LIST */
extern T0* GE_new1726(T1 initialize);
/* New instance of type ET_ALIASED_FEATURE_NAME */
extern T0* GE_new1728(T1 initialize);
/* New instance of type ET_FEATURE_IDS */
extern T0* GE_new1730(T1 initialize);
/* New instance of type ET_RENAME */
extern T0* GE_new1731(T1 initialize);
/* New instance of type ET_POSTCONDITIONS */
extern T0* GE_new1734(T1 initialize);
/* New instance of type ET_INVARIANTS */
extern T0* GE_new1735(T1 initialize);
/* New instance of type CONSOLE */
extern T0* GE_new1748(T1 initialize);
/* New instance of type ET_LIKE_N */
extern T0* GE_new1749(T1 initialize);
/* New instance of type [attached] DS_HASH_TABLE_CURSOR [[attached] ET_DYNAMIC_TYPE_SET, [attached] ET_DYNAMIC_TYPE] */
extern T0* GE_new1750(T1 initialize);
/* New instance of type [attached] SPECIAL [[attached] ET_BASE_TYPE] */
extern T0* GE_new1753(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [[attached] ET_BASE_TYPE] */
extern T0* GE_new1754(T1 initialize);
/* New instance of type [attached] ARRAY [[attached] IMMUTABLE_STRING_32] */
extern T0* GE_new1755(T1 initialize);
/* New instance of type [attached] SPECIAL [[attached] IMMUTABLE_STRING_32] */
extern T0* GE_new1756(T6 a1, T1 initialize);
/* New instance of type [attached] DS_HASH_TABLE [detachable DS_HASH_SET [[attached] STRING_8], [attached] STRING_8] */
extern T0* GE_new1757(T1 initialize);
/* New instance of type [attached] DS_ARRAYED_LIST [[attached] ET_XACE_TARGET] */
extern T0* GE_new1758(T1 initialize);
/* New instance of type [attached] DS_ARRAYED_LIST [[attached] ET_XACE_CLUSTER] */
extern T0* GE_new1759(T1 initialize);
/* New instance of type [attached] DS_LINKED_LIST [[attached] ET_XACE_CLASS_OPTIONS] */
extern T0* GE_new1762(T1 initialize);
/* New instance of type [attached] DS_ARRAYED_STACK [[attached] YY_BUFFER] */
extern T0* GE_new1763(T1 initialize);
/* New instance of type UT_SYNTAX_ERROR */
extern T0* GE_new1764(T1 initialize);
/* New instance of type UT_TOO_MANY_INCLUDES_ERROR */
extern T0* GE_new1765(T1 initialize);
/* New instance of type ET_DYNAMIC_TARGET_LIST */
extern T0* GE_new1768(T1 initialize);
/* New instance of type [attached] SPECIAL [[attached] NATIVE_STRING] */
extern T0* GE_new1771(T6 a1, T1 initialize);
/* New instance of type [attached] ARRAY [detachable STRING_8] */
extern T0* GE_new1772(T1 initialize);
/* New instance of type XM_NAMESPACE_RESOLVER_CONTEXT */
extern T0* GE_new1773(T1 initialize);
/* New instance of type [attached] DS_LINKED_QUEUE [detachable STRING_8] */
extern T0* GE_new1776(T1 initialize);
/* New instance of type [attached] DS_LINKED_QUEUE [[attached] STRING_8] */
extern T0* GE_new1777(T1 initialize);
/* New instance of type [attached] DS_LINKED_LIST_CURSOR [[attached] DS_PAIR [[attached] XM_POSITION, [attached] XM_NODE]] */
extern T0* GE_new1778(T1 initialize);
/* New instance of type [attached] DS_LINKED_LIST [[attached] DS_PAIR [[attached] XM_POSITION, [attached] XM_NODE]] */
extern T0* GE_new1779(T1 initialize);
/* New instance of type [attached] DS_PAIR [[attached] XM_POSITION, [attached] XM_NODE] */
extern T0* GE_new1780(T1 initialize);
/* New instance of type [attached] KL_EQUALITY_TESTER [[attached] XM_NAMESPACE] */
extern T0* GE_new1783(T1 initialize);
/* New instance of type [attached] DS_HASH_SET_CURSOR [[attached] XM_NAMESPACE] */
extern T0* GE_new1784(T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [[attached] XM_NAMESPACE] */
extern T0* GE_new1785(T1 initialize);
/* New instance of type [attached] SPECIAL [[attached] XM_NAMESPACE] */
extern T0* GE_new1786(T6 a1, T1 initialize);
/* New instance of type [attached] DS_HASH_SET_CURSOR [[attached] ET_DOTNET_ASSEMBLY] */
extern T0* GE_new1787(T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [[attached] ET_DOTNET_ASSEMBLY] */
extern T0* GE_new1788(T1 initialize);
/* New instance of type [attached] SPECIAL [[attached] ET_DOTNET_ASSEMBLY] */
extern T0* GE_new1789(T6 a1, T1 initialize);
/* New instance of type [attached] DS_ARRAYED_LIST [[attached] ET_DOTNET_ASSEMBLY] */
extern T0* GE_new1792(T1 initialize);
/* New instance of type [attached] SPECIAL [[attached] ET_XACE_MOUNTED_LIBRARY] */
extern T0* GE_new1793(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [[attached] ET_XACE_MOUNTED_LIBRARY] */
extern T0* GE_new1794(T1 initialize);
/* New instance of type [attached] DS_ARRAYED_LIST_CURSOR [[attached] ET_XACE_MOUNTED_LIBRARY] */
extern T0* GE_new1795(T1 initialize);
/* New instance of type [attached] DS_ARRAYED_LIST [[attached] ET_MASTER_CLASS] */
extern T0* GE_new1796(T1 initialize);
/* New instance of type [attached] DS_HASH_SET_CURSOR [[attached] ET_INTERNAL_UNIVERSE] */
extern T0* GE_new1797(T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [[attached] ET_INTERNAL_UNIVERSE] */
extern T0* GE_new1801(T1 initialize);
/* New instance of type [attached] SPECIAL [[attached] ET_INTERNAL_UNIVERSE] */
extern T0* GE_new1802(T6 a1, T1 initialize);
/* New instance of type ET_CLASS_TYPE_STATUS_CHECKER1 */
extern T0* GE_new1803(T1 initialize);
/* New instance of type ET_CLASS_TYPE_STATUS_CHECKER2 */
extern T0* GE_new1804(T1 initialize);
/* New instance of type ET_QUALIFIED_ANCHORED_TYPE_STATUS_CHECKER */
extern T0* GE_new1805(T1 initialize);
/* New instance of type ET_CLASS_TYPE_STATUS_CHECKER3 */
extern T0* GE_new1806(T1 initialize);
/* New instance of type [attached] DS_HASH_SET_CURSOR [[attached] ET_NAMED_CLASS] */
extern T0* GE_new1807(T1 initialize);
/* New instance of type [attached] DS_HASH_SET_CURSOR [[attached] ET_UNIVERSE] */
extern T0* GE_new1808(T1 initialize);
/* New instance of type [attached] SPECIAL [[attached] ET_UNIVERSE] */
extern T0* GE_new1809(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [[attached] ET_UNIVERSE] */
extern T0* GE_new1812(T1 initialize);
/* New instance of type [attached] SPECIAL [[attached] ET_PARENT_ITEM] */
extern T0* GE_new1814(T6 a1, T1 initialize);
/* New instance of type [attached] KL_SPECIAL_ROUTINES [[attached] ET_PARENT_ITEM] */
extern T0* GE_new1816(T1 ini