note
	description: "EiffelVision screen, implementation interface."
	legal: "See notice at end of class."
	status: "See notice at end of class."
	date: "$Date: 2008-12-29 12:27:11 -0800 (Mon, 29 Dec 2008) $"
	revision: "$Revision: 76420 $"

class
	EV_SCREEN_IMP

inherit
	EV_SCREEN_I
		redefine
			interface, virtual_width, virtual_height, virtual_x, virtual_y
		end

	EV_DRAWABLE_IMP
		redefine
			interface, destroy
		end

	WEL_INPUT_EVENT
		export
			{NONE} all
		end

	WEL_UNIT_CONVERSION
		rename
			horizontal_resolution as wel_horizontal_resolution,
			vertical_resolution as wel_vertical_resolution
		export
			{NONE} all
		end

create
	make

feature {NONE} -- Initialization

	make (an_interface: like interface)
			-- Create `Current', a screen object.
		do
			base_make (an_interface)
			create dc
			dc.get
		end

feature -- Access

	dc: WEL_SCREEN_DC
			-- DC for drawing.

feature -- Status report

	destroyed: BOOLEAN
			-- Is `Current' destroyed?
		do
			Result := not dc.exists
		end

	pointer_position: EV_COORDINATE
			-- Position of the screen pointer.
		local
			wel_point: WEL_POINT
		do
			create wel_point.make (0, 0)
			wel_point.set_cursor_position
			create Result.set (wel_point.x, wel_point.y)
		end

	widget_at_position (x, y: INTEGER): EV_WIDGET
			-- Widget at position (`x', `y') if any.
		local
			l_window: WEL_WINDOW
			wel_point: WEL_POINT
			widget_imp: EV_WIDGET_IMP
			internal_combo_box: EV_INTERNAL_COMBO_BOX_IMP
			internal_combo_field: EV_INTERNAL_COMBO_FIELD_IMP
		do
				-- Assign the cursor position to `wel_point'.
			create wel_point.make (x, y)
				-- Retrieve WEL_WINDOW at `wel_point'.
			l_window := wel_point.window_at

				-- If there is a window at `wel_point'.
			widget_imp ?= l_window
			if widget_imp /= Void then
					-- Result is interface of `widget_imp'.
				Result := widget_imp.interface
			else
					-- Combo boxes must be handled as a special case, as
					-- they are comprised of two widgets.
				internal_combo_box ?= l_window
				if internal_combo_box /= Void then
					Result := internal_combo_box.parent.interface
				else
					internal_combo_field ?= l_window
					if internal_combo_field /= Void then
						Result := internal_combo_field.parent.interface
					end
				end
			end
		end

feature -- Basic operation

	set_pointer_position (x, y: INTEGER)
			-- Set `pointer_position' to (`x',`y`).
		do
			cwin_set_cursor_position (x, y)
		end

	set_default_colors
			-- Set foreground and background color to their default values.
		local
			a_default_colors: EV_STOCK_COLORS
		do
			create a_default_colors
			set_background_color (a_default_colors.default_background_color)
			set_foreground_color (a_default_colors.default_foreground_color)
		end

	fake_pointer_button_press (a_button: INTEGER)
			-- Simulate the user pressing a `a_button'
			-- on the pointing device.
		do
			inspect a_button
			when 1 then
				send_mouse_left_button_down
			when 2 then
				send_mouse_right_button_down
			when 3 then
				send_mouse_middle_button_down
			end
		end

	fake_pointer_button_release (a_button: INTEGER)
			-- Simulate the user releasing a `a_button'
			-- on the pointing device.
		do
			inspect a_button
			when 1 then
				send_mouse_left_button_up
			when 2 then
				send_mouse_right_button_up
			when 3 then
				send_mouse_middle_button_up
			end
		end

	fake_pointer_wheel_up
			-- Simulate the user rotating the mouse wheel up.
		do
			send_mouse_wheel_up
		end

	fake_pointer_wheel_down
			-- Simulate the user rotating the mouse wheel down.
		do
			send_mouse_wheel_down
		end

	fake_key_press (a_key: EV_KEY)
			-- Simulate the user pressing `a_key'.
		local
			vk_code: INTEGER
		do
			vk_code := Key_conversion.key_code_to_wel(a_key.code)
			send_keyboard_key_down (vk_code)
		end

	fake_key_release (a_key: EV_KEY)
			-- Simulate the user releasing `a_key'.
		local
			vk_code: INTEGER
		do
			vk_code := Key_conversion.key_code_to_wel(a_key.code)
			send_keyboard_key_up (vk_code)
		end

feature -- Measurement

	width: INTEGER
			-- Width of `Current'.
		do
			Result := dc.width
		end

	height: INTEGER
			-- Height of `Current'.
		do
			Result := dc.height
		end

	virtual_width: INTEGER
			-- Virtual width of `Current'
		local
			l_metrics: WEL_SYSTEM_METRICS
		do
			create l_metrics
			Result := l_metrics.virtual_screen_width
		end

	virtual_height: INTEGER
			-- Virtual height of `Current'
		local
			l_metrics: WEL_SYSTEM_METRICS
		do
			create l_metrics
			Result := l_metrics.virtual_screen_height
		end

	virtual_x: INTEGER
			-- X position of virtual screen in main display coordinates
		local
			l_metrics: WEL_SYSTEM_METRICS
		do
			create l_metrics
			Result := l_metrics.virtual_screen_x
		end

	virtual_y: INTEGER
			-- Y position of virtual screen in main display coordinates
		local
			l_metrics: WEL_SYSTEM_METRICS
		do
			create l_metrics
			Result := l_metrics.virtual_screen_y
		end

	vertical_resolution: INTEGER
			-- Number of pixels per inch along screen height.
		do
			Result := get_device_caps (dc.item, logical_pixels_y)
		end

	horizontal_resolution: INTEGER
			-- Number of pixels per inch along screen width.
		do
			Result := get_device_caps (dc.item, logical_pixels_x)
		end

	redraw
			-- Force screen to redraw itself.
		external
			"C inline use %"wel.h%""
		alias
			"InvalidateRect(NULL, NULL, FALSE)"
		end

feature -- Status setting

	destroy
			-- Destroy actual object.
		do
			dc.release
			Precursor {EV_DRAWABLE_IMP}
		end

feature -- Implementation

	interface: EV_SCREEN


feature {NONE} -- Constants

	System_metrics_constants: WEL_SYSTEM_METRICS
			-- System metrics constants.
		once
			create Result
		end

	Key_conversion: EV_WEL_KEY_CONVERSION
			-- Key conversion routines.
		once
			create Result
		end

invariant
	dc_not_void: dc /= Void

note
	copyright:	"Copyright (c) 1984-2006, Eiffel Software and others"
	license:	"Eiffel Forum License v2 (see http://www.eiffel.com/licensing/forum.txt)"
	source: "[
			 Eiffel Software
			 356 Storke Road, Goleta, CA 93117 USA
			 Telephone 805-685-1006, Fax 805-685-6869
			 Website http://www.eiffel.com
			 Customer support http://support.eiffel.com
		]"




end -- class EV_SCREEN_IMP

