!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!
module cubefield_observe
  use cube_types
  use cubetools_parameters
  use cubetools_structure
  use cubeadm_cubeid_types
  use cubefield_messaging
  ! 
  public :: observe
  public :: cubefield_observe_command
  private
  !
  type :: observe_comm_t
     type(option_t), pointer :: comm
   contains
     procedure, public  :: register => cubefield_observe_register
     procedure, private :: parse    => cubefield_observe_parse
     procedure, private :: main     => cubefield_observe_main
  end type observe_comm_t
  type(observe_comm_t) :: observe
  !
  integer(kind=4), parameter :: ivoldens = 1
  integer(kind=4), parameter :: ivx = 2
  integer(kind=4), parameter :: ivy = 3
  integer(kind=4), parameter :: ivz = 4
  type observe_user_t
     type(cubeid_user_t)   :: cubeids
     real(kind=8)          :: distance    = 400 ! [pc]
     real(kind=8)          :: inclination =  30 ! [deg]
   contains
     procedure, private :: toprog => cubefield_observe_user_toprog
  end type observe_user_t
  !
  type observe_prog_t
     type(cube_t), pointer :: voldens
     type(cube_t), pointer :: vx
     type(cube_t), pointer :: vy
     type(cube_t), pointer :: vz
     type(cube_t), pointer :: coldens
     type(cube_t), pointer :: vcentroid
     integer(kind=chan_k)  :: nz
     real(kind=8)          :: factor
     real(kind=8)          :: distance
     real(kind=8)          :: cosi
     real(kind=8)          :: sini
   contains
     procedure, private :: header => cubefield_observe_prog_header
     procedure, private :: data   => cubefield_observe_prog_data
     procedure, private :: loop   => cubefield_observe_prog_loop
     procedure, private :: act    => cubefield_observe_prog_act
  end type observe_prog_t
  !
contains
  !
  subroutine cubefield_observe_command(line,error)
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    character(len=*), intent(in)    :: line
    logical,          intent(inout) :: error
    !
    type(observe_user_t) :: user
    character(len=*), parameter :: rname='OBSERVE>COMMAND'
    !
    call cubefield_message(fieldseve%trace,rname,'Welcome')
    !
    call observe%parse(line,user,error)
    if (error) return
    call observe%main(user,error)
    if (error) continue
  end subroutine cubefield_observe_command
  !
  subroutine cubefield_observe_register(observe,error)
    use cubedag_allflags
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(observe_comm_t), intent(inout) :: observe
    logical,               intent(inout) :: error
    !
    type(cubeid_arg_t) :: cubearg
    character(len=*), parameter :: comm_abstract='Compute column density and centroid velocity'
    character(len=*), parameter :: comm_help=&
         'It uses the 3D volume density and velocity fields&
         &and it assumes the optically thin limit'
    character(len=*), parameter :: rname='OBSERVE>REGISTER'
    !
    call cubefield_message(fieldseve%trace,rname,'Welcome')
    !
    call cubetools_register_command(&
         'OBSERVE','voldens vx vy vz',&
         comm_abstract,&
         comm_help,&
         cubefield_observe_command,&
         observe%comm,error)
    if (error) return
    call cubearg%register(&
         'VOLDENS',&
         'Volume density',&
         strg_id,&
         code_arg_mandatory,&
         [flag_any],&
         error)
    call cubearg%register(&
         'VX',&
         'x-axis velocity component',&
         strg_id,&
         code_arg_mandatory,&
         [flag_any],&
         error)
    call cubearg%register(&
         'VY',&
         'y-axis velocity component',&
         strg_id,&
         code_arg_mandatory,&
         [flag_any],&
         error)
    call cubearg%register(&
         'VZ',&
         'z-axis velocity component',&
         strg_id,&
         code_arg_mandatory,&
         [flag_any],&
         error)
    if (error) return
  end subroutine cubefield_observe_register
  !
  subroutine cubefield_observe_parse(comm,line,user,error)
    !----------------------------------------------------------------------
    ! OBSERVE voldensid vxid vyid vzid
    !----------------------------------------------------------------------
    class(observe_comm_t), intent(in)    :: comm
    character(len=*),      intent(in)    :: line
    type(observe_user_t),  intent(out)   :: user
    logical,               intent(inout) :: error
    !
    character(len=*), parameter :: rname='OBSERVE>PARSE'
    !
    call cubefield_message(fieldseve%trace,rname,'Welcome')
    !
    call cubeadm_cubeid_parse(line,observe%comm,user%cubeids,error)
    if (error) return
  end subroutine cubefield_observe_parse
  !
  subroutine cubefield_observe_main(comm,user,error) 
    use cubeadm_timing
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(observe_comm_t), intent(in)    :: comm
    type(observe_user_t),  intent(inout) :: user
    logical,               intent(inout) :: error
    !
    type(observe_prog_t) :: prog
    character(len=*), parameter :: rname='OBSERVE>MAIN'
    !
    call cubefield_message(fieldseve%trace,rname,'Welcome')
    !
    call user%toprog(comm,prog,error)
    if (error) return
    call prog%header(error)
    if (error) return
    call cubeadm_timing_prepro2process()
    call prog%data(error)
    if (error) return
    call cubeadm_timing_process2postpro()
  end subroutine cubefield_observe_main
  !
  !------------------------------------------------------------------------
  !
  subroutine cubefield_observe_user_toprog(user,comm,prog,error)
    use cubeadm_get
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(observe_user_t), intent(in)    :: user
    type(observe_comm_t),  intent(in)    :: comm
    type(observe_prog_t),  intent(out)   :: prog
    logical,               intent(inout) :: error
    !
    character(len=*), parameter :: rname='OBSERVE>USER>TOPROG'
    !
    call cubefield_message(fieldseve%trace,rname,'Welcome')
    !
    call cubeadm_cubeid_get_header(comm%comm,ivoldens,user%cubeids,&
         code_access_speset,code_read,prog%voldens,error)
    if (error) return
    call cubeadm_cubeid_get_header(comm%comm,ivx,user%cubeids,&
         code_access_speset,code_read,prog%vx,error)
    if (error) return
    call cubeadm_cubeid_get_header(comm%comm,ivy,user%cubeids,&
         code_access_speset,code_read,prog%vy,error)
    if (error) return
    call cubeadm_cubeid_get_header(comm%comm,ivz,user%cubeids,&
         code_access_speset,code_read,prog%vz,error)
    if (error) return
    prog%distance = user%distance
  end subroutine cubefield_observe_user_toprog
  !
  !------------------------------------------------------------------------
  !
  subroutine cubefield_observe_prog_header(prog,error)
    use phys_const
    use cubetools_axis_types
    use cubetools_header_methods
    use cubedag_allflags
    use cubemain_header_tools
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(observe_prog_t), intent(inout) :: prog
    logical,               intent(inout) :: error
    !
    type(axis_t) :: axis
    real(kind=8), parameter :: Av_per_pscm = 1d0/0.9d21
    real(kind=8), parameter :: cm_per_au = 14959787070000
    integer(kind=chan_k), parameter :: onechan=1
    character(len=*), parameter :: rname='OBSERVE>PROG>HEADER'
    !
    call cubefield_message(fieldseve%trace,rname,'Welcome')
    !
    ! *** JP: there should be a check that the input cubes are consistent
    call cubemain_header_tools_compress_c(&
         prog%coldens,prog%voldens,&
         [flag_observed,flag_column,flag_density],&
         'magn',onechan,error)
    call cubemain_header_tools_compress_c(&
         prog%vcentroid,prog%vz,&
         [flag_observed,flag_centroid,flag_velocity],&
         'km/s',onechan,error)
    !
    call cubetools_header_get_axis_head_c(prog%voldens%head,axis,error)
    if (error) return
    prog%nz = axis%n
    prog%factor = Av_per_pscm*cm_per_au*prog%distance*sec_per_rad*abs(axis%inc)
  end subroutine cubefield_observe_prog_header
  !
  subroutine cubefield_observe_prog_data(prog,error)
    use cubeadm_opened
    !----------------------------------------------------------------------
    ! 
    !----------------------------------------------------------------------
    class(observe_prog_t), intent(inout) :: prog
    logical,               intent(inout) :: error
    !
    type(cubeadm_iterator_t) :: itertask
    character(len=*), parameter :: rname='OBSERVE>PROG>DATA'
    !
    call cubefield_message(fieldseve%trace,rname,'Welcome')
    !
    call cubeadm_datainit_all(itertask,error)
    if (error) return
    !$OMP PARALLEL DEFAULT(none) SHARED(prog,error) FIRSTPRIVATE(itertask)
    !$OMP SINGLE
    do while (cubeadm_dataiterate_all(itertask,error))
       if (error) exit
       !$OMP TASK SHARED(prog,error) FIRSTPRIVATE(itertask)
       if (.not.error) &
         call prog%loop(itertask%first,itertask%last,error)
       !$OMP END TASK
    enddo ! itertask
    !$OMP END SINGLE
    !$OMP END PARALLEL
  end subroutine cubefield_observe_prog_data
  !   
  subroutine cubefield_observe_prog_loop(prog,first,last,error)
    use cubeadm_entryloop
    use cubeadm_spectrum_types
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(observe_prog_t), intent(inout) :: prog
    integer(kind=entr_k),  intent(in)    :: first
    integer(kind=entr_k),  intent(in)    :: last
    logical,               intent(inout) :: error
    !
    integer(kind=entr_k) :: ie
    type(spectrum_t) :: voldens,vx,vy,vz
    type(spectrum_t) :: coldens,vcentroid
    character(len=*), parameter :: rname='OBSERVE>PROG>LOOP'
    !
    call voldens%associate('voldens',prog%voldens,error)
    if (error) return
    call vx%associate('vx',prog%vx,error)
    if (error) return
    call vy%associate('vy',prog%vy,error)
    if (error) return
    call vz%associate('vz',prog%vz,error)
    if (error) return
    call coldens%allocate('coldens',prog%coldens,error)
    if (error) return
    call vcentroid%allocate('vcentroid',prog%vcentroid,error)
    if (error) return
    !
    do ie=first,last
      call cubeadm_entryloop_iterate(ie,error)
      if (error) return
      call prog%act(ie,voldens,vx,vy,vz,coldens,vcentroid,error)
      if (error) return
    enddo ! ie
  end subroutine cubefield_observe_prog_loop
  !   
  subroutine cubefield_observe_prog_act(prog,ie,voldens,vx,vy,vz,coldens,vcentroid,error)
    use cubeadm_spectrum_types
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(observe_prog_t), intent(inout) :: prog
    integer(kind=entr_k),  intent(in)    :: ie
    type(spectrum_t),      intent(inout) :: voldens
    type(spectrum_t),      intent(inout) :: vx
    type(spectrum_t),      intent(inout) :: vy
    type(spectrum_t),      intent(inout) :: vz
    type(spectrum_t),      intent(inout) :: coldens
    type(spectrum_t),      intent(inout) :: vcentroid
    logical,               intent(inout) :: error
    !
    real(kind=sign_k) :: dens,cent
    integer(kind=chan_k) :: iz
    character(len=*), parameter :: rname='OBSERVE>PROG>ACT'
    !
    call voldens%get(ie,error)
    if (error) return
!!$ Not needed in this first implementation
!!$    call vx%get(prog%vx,ie,error)
!!$    if (error) return
!!$    call vy%get(prog%vy,ie,error)
!!$    if (error) return
    call vz%get(ie,error)
    if (error) return
    dens = 0
    cent = 0
    do iz=1,prog%nz
       dens = dens+voldens%y%val(iz)
       cent = cent+voldens%y%val(iz)*vz%y%val(iz)
    enddo ! iz
    cent = cent/dens
    coldens%y%val(1) = dens*prog%factor
    vcentroid%y%val(1) = cent
    call coldens%put(ie,error)
    if (error) return
    call vcentroid%put(ie,error)
    if (error) return
  end subroutine cubefield_observe_prog_act
end module cubefield_observe
!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
