!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!
module cubemain_feather
  use cubetools_structure
  use cubetools_beam_types
  use cube_types
  use cubeadm_cubeid_types
  use cubemain_messaging
  !
  public :: feather
  public :: cubemain_feather_command
  private
  !
  type :: feather_comm_t
     type(option_t), pointer :: comm
     type(option_t), pointer :: factor
   contains
     procedure, public  :: register     => cubemain_feather_register
     procedure, private :: parse        => cubemain_feather_parse
     procedure, private :: parse_factor => cubemain_feather_parse_factor
     procedure, private :: main         => cubemain_feather_main
  end type feather_comm_t
  type(feather_comm_t) :: feather
  !
  integer(kind=4), parameter :: ilores = 1
  integer(kind=4), parameter :: ihires = 2
  type feather_user_t
     type(cubeid_user_t)   :: cubeids
     real(kind=sign_k)     :: lofac = 1.0  ! [---] Low resolution cube scalling factor
   contains
     procedure, private :: toprog => cubemain_feather_user_toprog
  end type feather_user_t
  type feather_prog_t
     type(cube_t), pointer :: lores                    ! Low resolution input cube
     type(cube_t), pointer :: hires                    ! High resolution input cube
     type(cube_t), pointer :: feather                  ! Output cube 
     complex(kind=sign_k), allocatable :: cweight(:,:) ! [---] Complex weight for the high resolution cube 
     real(kind=sign_k)     :: lofac                    ! [---] Low resolution cube scalling factor
     ! VVV Dim has to have kind=4 because of previous interfaces
     integer(kind=4)       :: dim(2)                   ! [pix,pix] Dimensions for the FFT
     integer(kind=pixe_k)  :: fftdim(2)                ! FFT dimensions
     type(beam_t)          :: lobeam,hibeam            ! Low and high resolution beams
   contains
     procedure, private :: header       => cubemain_feather_prog_header
     procedure, private :: hires_weight => cubemain_feather_prog_hires_weight
     procedure, private :: data         => cubemain_feather_prog_data
     procedure, private :: loop         => cubemain_feather_prog_loop
     procedure, private :: image        => cubemain_feather_prog_image
  end type feather_prog_t
  !
contains
  !
  subroutine cubemain_feather_command(line,error)
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    character(len=*), intent(in)    :: line
    logical,          intent(inout) :: error
    !
    type(feather_user_t) :: user
    character(len=*), parameter :: rname='FEATHER>COMMAND'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call feather%parse(line,user,error)
    if (error) return
    call feather%main(user,error)
    if (error) continue
  end subroutine cubemain_feather_command
  !
  !----------------------------------------------------------------------
  !
  subroutine cubemain_feather_register(feather,error)
    use cubedag_allflags
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(feather_comm_t), intent(inout) :: feather
    logical,               intent(inout) :: error
    !
    type(cubeid_arg_t) :: cubearg
    type(standard_arg_t) :: stdarg
    character(len=*), parameter :: comm_abstract = &
         'Combine two cubes of different spatial resolutions'
    character(len=*), parameter :: comm_help = &
         'The combination is done on the UV plane via the fourier&
         & transform of the input images. lores must have a coarser&
         & spatial resolution than hires. lores and hires must have&
         & the same spectral and spatial gridding, this can be&
         & achieved using CUBE\RESAMPLE and CUBE\REPROJECT'
    character(len=*), parameter :: rname='FEATHER>REGISTER'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call cubetools_register_command(&
         'FEATHER','lores hires',&
         comm_abstract,&
         comm_help,&
         cubemain_feather_command,&
         feather%comm,error)
    if (error) return
    call cubearg%register( &
         'LORES', &
         'Lower resolution cube',  &
         strg_id,&
         code_arg_mandatory,  &
         [flag_cube], &
         error)
    if (error) return
    call cubearg%register( &
         'HIRES', &
         'Higher resolution cube',  &
         strg_id,&
         code_arg_mandatory,  &
         [flag_cube], &
         error)
    if (error) return
    !    
    call cubetools_register_option(&
         'FACTOR','lofac',&
         'Define a flux density scaling factor for LORES',&
         strg_id,&
         feather%factor,error)
    if (error) return
    call stdarg%register( &
         'lofac',  &
         'flux density scaling factor for LORES', &
         strg_id,&
         code_arg_mandatory, &
         error)
    if (error) return
  end subroutine cubemain_feather_register
  !
  subroutine cubemain_feather_parse(feather,line,user,error)
    use cubetools_parse
    !----------------------------------------------------------------------
    ! FEATHER lores hires
    ! [/FACTOR lofac]
    !----------------------------------------------------------------------
    class(feather_comm_t), intent(in)    :: feather
    character(len=*),      intent(in)    :: line
    type(feather_user_t),  intent(out)   :: user
    logical,               intent(inout) :: error
    !
    character(len=*), parameter :: rname='FEATHER>PARSE'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call cubeadm_cubeid_parse(line,feather%comm,user%cubeids,error)
    if (error) return
    call feather%parse_factor(line,user,error)
    if(error) return
  end subroutine cubemain_feather_parse
  !
  subroutine cubemain_feather_parse_factor(feather,line,user,error)
    !----------------------------------------------------------------------
    ! FEATHER 
    ! /FACTOR lofac
    !----------------------------------------------------------------------
    class(feather_comm_t), intent(in)    :: feather
    character(len=*),      intent(in)    :: line
    type(feather_user_t),  intent(inout) :: user
    logical,               intent(inout) :: error
    !
    logical :: present
    character(len=*), parameter :: rname='FEATHER>PARSE>FACTOR'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call feather%factor%present(line,present,error)
    if (error) return
    if (present) then
       call cubetools_getarg(line,feather%factor,1,user%lofac,mandatory,error)
       if(error) return
    else
       user%lofac = 1.0
    endif
  end subroutine cubemain_feather_parse_factor
  !
  subroutine cubemain_feather_main(feather,user,error)
    use cubeadm_timing
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(feather_comm_t), intent(in)    :: feather
    type(feather_user_t),  intent(in)    :: user
    logical,               intent(inout) :: error
    !
    type(feather_prog_t) :: prog
    character(len=*), parameter :: rname='FEATHER>MAIN'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call user%toprog(prog,error)
    if (error) return
    call prog%header(error)
    if (error) return
    !
    call cubeadm_timing_prepro2process()
    call prog%data(error)
    if (error) return
    call cubeadm_timing_process2postpro()
  end subroutine cubemain_feather_main
  !
  !----------------------------------------------------------------------
  !
  subroutine cubemain_feather_user_toprog(user,prog,error)
    use cubetools_header_methods
    use cubetools_consistency_methods
    use cubetools_brightness
    use cubeadm_get
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(feather_user_t), intent(in)    :: user
    type(feather_prog_t),  intent(out)   :: prog
    logical,               intent(inout) :: error
    !
    logical :: prob
    character(len=unit_l) :: lounit,hiunit
    real(kind=coor_k) :: loarea, hiarea

    character(len=*), parameter :: rname='FEATHER>USER>TOPROG'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call cubeadm_cubeid_get_header(feather%comm,ilores,user%cubeids,code_access_imaset,code_read,&
         prog%lores,error)
    if (error) return
    call cubeadm_cubeid_get_header(feather%comm,ihires,user%cubeids,code_access_imaset,code_read,&
         prog%hires,error)
    if (error) return
    !
    ! Unit checks
    call cubetools_header_get_array_unit(prog%hires%head,hiunit,error)
    if (error) return
    call cubetools_header_get_array_unit(prog%lores%head,lounit,error)
    if (error) return
    call cubetools_header_get_spabeam(prog%lores%head,prog%lobeam,error)
    if (error) return
    call cubetools_header_get_spabeam(prog%hires%head,prog%hibeam,error)
    if (error) return
    if (trim(hiunit).ne.brightness_unit(code_unit_jyperbeam)) then
       call cubemain_message(seve%e,rname,'High resolution image unit&
            & must be '//brightness_unit(code_unit_jyperbeam))
       error = .true.
    endif
    if (trim(lounit).ne.brightness_unit(code_unit_jyperbeam)) then
       call cubemain_message(seve%e,rname,'Low resolution image unit&
            & must be '//brightness_unit(code_unit_jyperbeam))
       error = .true.
    endif
    loarea = prog%lobeam%major*prog%lobeam%minor
    hiarea = prog%hibeam%major*prog%hibeam%minor
    if (loarea.le.hiarea) then
       call cubemain_message(seve%e,rname,'Low resolution cube must have the largest beam')
       error = .true.
    endif
    !
    if (error) return
    prob =.false.
    call cubetools_consistency_grid('Low resolution',prog%lores%head,&
         'High resolution',prog%hires%head,prob,error)
    if (error) return
    if (cubetools_consistency_failed(rname,prob,error)) return
    !
    prog%lofac = user%lofac*hiarea/loarea
    prog%dim(1) = int(prog%hires%head%arr%n%l,4)
    prog%dim(2) = int(prog%hires%head%arr%n%m,4)
    !
  end subroutine cubemain_feather_user_toprog
  !
  !----------------------------------------------------------------------
  !
  subroutine cubemain_feather_prog_header(prog,error)
    use cubetools_header_methods
    use cubedag_allflags
    use cubeadm_clone
    use cubemain_fft_utils
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(feather_prog_t), intent(inout) :: prog
    logical,               intent(inout) :: error
    !
    character(len=*), parameter :: rname='FEATHER>PROG>HEADER'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call cubeadm_clone_header(prog%hires,[flag_feather,flag_cube],prog%feather,error)
    if (error) return
    !  
    call cubetools_header_add_observatories(prog%lores%head,prog%feather%head,error)
    if (error) return
    !
    call prog%hires_weight(error)
    if (error) return
  end subroutine cubemain_feather_prog_header
  !
  subroutine cubemain_feather_prog_hires_weight(prog,error)
    use gkernel_interfaces
    use cubemain_fft_utils
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(feather_prog_t), intent(inout) :: prog
    logical,               intent(inout) :: error
    !
    character(len=*), parameter :: rname='FEATHER>PROG>HIRES>WEIGHT'
    integer(kind=4) :: ier
    integer(kind=pixe_k) :: nl,nm,il,im,icl,icm
    real(kind=coor_k) :: cl,cm,pl,pm,sigx,sigy,acoeff,bcoeff,ccoeff,pang
    
    real(kind=sign_k), allocatable :: beam(:,:),work(:)
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    nl = prog%lores%head%arr%n%l
    nm = prog%lores%head%arr%n%m
    allocate(prog%cweight(nl,nm),beam(nl,nm),work(2*max(nl,nm)),stat=ier)
    if (failed_allocate(rname,'work arrays',ier,error)) return
    !
    icl = nl/2
    icm = nm/2
    if (2*icl.ne.nl) icl = icl+1
    if (2*icm.ne.nm) icm = icm+1
    cl = prog%lores%head%spa%l%coord(icl)
    cm = prog%lores%head%spa%m%coord(icm)
    if (prog%lobeam%minor.gt.0) then
       sigx = prog%lobeam%major/(sqrt(8.*log(2.)))
       sigy = prog%lobeam%minor/(sqrt(8.*log(2.)))
       ! VVV is this correct, or is there a correction to be applied here?
       pang = prog%lobeam%pang
       !
       acoeff = (cos(pang)**2)/(2*sigx**2)+(sin(pang)**2)/(2*sigy**2)
       bcoeff = (sin(2*pang))/(4*sigx**2)+(sin(2*pang))/(4*sigy**2)
       ccoeff = (sin(pang)**2)/(2*sigx**2)+(cos(pang)**2)/(2*sigy**2)
       !
       do im=1,nm
          pm = prog%lores%head%spa%m%coord(im)-cm
          do il=1,nl
             pl = prog%lores%head%spa%l%coord(il)-cl
             beam(il,im) = exp(-(acoeff*pl**2+2*bcoeff*pl*pm+ccoeff*pm**2))
          enddo
       enddo
    else
       sigx = prog%lobeam%major/(sqrt(8.*log(2.)))
       do im=1,nm
          pm = prog%lores%head%spa%m%coord(im)-cm
          do il=1,nl
             pl = prog%lores%head%spa%l%coord(il)-cl
             beam(il,im) = exp(-(pl**2+pm**2)/(2*sigx**2))
          enddo
       enddo
    endif
    !
    prog%cweight(:,:) = cmplx(beam(:,:),0.0)
    call fourt(prog%cweight,prog%dim,2,code_dire,code_rdata,work)
    prog%cweight(:,:) = prog%cweight(:,:)/(nl*nm)
    prog%cweight(:,:) = 1.0-prog%cweight(:,:)
    !
    deallocate(beam,work)
  end subroutine cubemain_feather_prog_hires_weight
  !
  subroutine cubemain_feather_prog_data(prog,error)
    use cubeadm_opened
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(feather_prog_t), intent(inout) :: prog
    logical,               intent(inout) :: error
    !
    character(len=*), parameter :: rname='FEATHER>PROG>DATA'
    type(cubeadm_iterator_t) :: iter
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call cubeadm_datainit_all(iter,error)
    if (error) return
    !$OMP PARALLEL DEFAULT(none) SHARED(prog,error) FIRSTPRIVATE(iter)
    !$OMP SINGLE
    do while (cubeadm_dataiterate_all(iter,error))
       if (error)  exit
       !$OMP TASK SHARED(prog) FIRSTPRIVATE(iter,error)
       if (.not.error)  &
         call prog%loop(iter%first,iter%last,error)
       !$OMP END TASK
    enddo ! ie
    !$OMP END SINGLE
    !$OMP END PARALLEL
  end subroutine cubemain_feather_prog_data
  !
  subroutine cubemain_feather_prog_loop(prog,first,last,error)
    use gkernel_interfaces
    use cubeadm_entryloop
    use cubemain_image_real
    use cubemain_image_cplx
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(feather_prog_t), intent(inout) :: prog
    integer(kind=entr_k),  intent(in)    :: first
    integer(kind=entr_k),  intent(in)    :: last
    logical,               intent(inout) :: error
    !
    character(len=*), parameter :: rname='FEATHER>PROG>LOOP'
    integer(kind=entr_k) :: ie
    integer(kind=pixe_k) :: nl,nm
    integer(kind=4) :: ier
    real(kind=sign_k),allocatable :: work(:)
    type(image_cplx_t) :: lofft,hifft
    type(image_t) :: loima,hiima,ouima
    !
    nl = prog%feather%head%arr%n%l
    nm = prog%feather%head%arr%n%m
    call hiima%init(prog%hires,error)
    if (error) return
    call loima%init(prog%lores,error)
    if (error) return
    call ouima%reallocate('feathered',nl,nm,error)
    if (error) return
    call lofft%reallocate('Low res FFT',nl,nm,error)
    if (error) return
    call hifft%reallocate('High res FFT',nl,nm,error)
    if (error) return
    
    allocate(work(2*max(nl,nm)),stat=ier)
    if (failed_allocate(rname,'work array',ier,error)) return
    !
    do ie=first,last
      call cubeadm_entryloop_iterate(ie,error)
      if (error)  return
      call prog%image(ie,loima,hiima,ouima,work,lofft,hifft,error)
      if (error)  return
    enddo
  end subroutine cubemain_feather_prog_loop
  !
  subroutine cubemain_feather_prog_image(prog,ie,loima,hiima,ouima,work,lofft,hifft,error)
    use cubemain_image_real
    use cubemain_image_cplx
    use cubemain_fft_utils
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(feather_prog_t), intent(inout) :: prog
    integer(kind=entr_k),  intent(in)    :: ie
    type(image_t),         intent(inout) :: loima  ! Working buffer
    type(image_t),         intent(inout) :: hiima  ! Working buffer
    type(image_t),         intent(inout) :: ouima  ! Working buffer
    real(kind=sign_k),     intent(inout) :: work(:)! Working buffer
    type(image_cplx_t),    intent(inout) :: lofft  ! Working buffer
    type(image_cplx_t),    intent(inout) :: hifft  ! Working buffer
    logical,               intent(inout) :: error
    !
    character(len=*), parameter :: rname='FEATHER>PROG>IMAGE'
    integer(kind=pixe_k) :: il,im,nl,nm
    !
    nl = prog%feather%head%arr%n%l
    nm = prog%feather%head%arr%n%m
    call hiima%get(prog%hires,ie,error)
    if (error)  return
    call loima%get(prog%lores,ie,error)
    if (error)  return
    !
    call cubemain_fft_plunge(nl,nm,hiima,nl,nm,hifft,error)
    if (error) return
    call cubemain_fft_plunge(nl,nm,loima,nl,nm,lofft,error)
    if (error) return
    !
    call fourt(lofft%z,prog%dim,2,code_inve,code_rdata,work)
    call fourt(hifft%z,prog%dim,2,code_inve,code_rdata,work)
    !
    do im=1,nm
       do il=1,nl
          hifft%z(il,im) = hifft%z(il,im)*prog%cweight(il,im)+lofft%z(il,im)*prog%lofac
       end do
    enddo
    !
    call fourt(hifft%z,prog%dim,2,code_dire,code_cdata,work)
    !
    call cubemain_fft_deplunge(nl,nm,hifft,nl,nm,ouima,error)
    if (error) return
    !
    call cubemain_image_reblank(hiima,ouima,error)
    if (error) return
    call ouima%put(prog%feather,ie,error)
    if (error)  return
  end subroutine cubemain_feather_prog_image
end module cubemain_feather
!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
