!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!
module cubemain_export
  use cubetools_structure
  use cubetools_keyword_arg
  use cube_types
  use cubedag_parameters
  use cubedag_allflags
  use cubemain_messaging
  use cubeadm_directory_type
  use cubeadm_cubeid_types
  use cubeadm_timing
  use cubemain_identifier
  !
  public :: export
  public :: cubemain_export_command
  private
  !
  type :: export_comm_t
     type(option_t),      pointer :: comm
     type(option_t),      pointer :: format
     type(keyword_arg_t), pointer :: format_arg
     type(option_t),      pointer :: blanking
     type(option_t),      pointer :: name
     type(identifier_opt_t)       :: as
     type(option_t),      pointer :: dir
     type(option_t),      pointer :: access
     type(keyword_arg_t), pointer :: access_arg
   contains
     procedure, public  :: register => cubemain_export_register
     procedure, private :: parse    => cubemain_export_parse
     procedure, private :: main     => cubemain_export_main
  end type export_comm_t
  type(export_comm_t) :: export
  !
  integer(kind=4), parameter :: icube = 1
  type export_user_t
     type(cubeid_user_t)     :: cubeids
     type(identifier_user_t) :: identifier
     logical                 :: tofits
     logical                 :: doblank
     logical                 :: dodir
     logical                 :: dofilename
     real(kind=4)            :: bval=-1000.
     real(kind=4)            :: eval=0.
     character(len=file_l)   :: filename
     character(len=file_l)   :: dir
     integer(kind=code_k)    :: access
   contains
     procedure, private :: toprog => cubemain_export_user_toprog
  end type export_user_t
  type export_prog_t
     type(cubeid_prog_t)     :: incube
     character(len=file_l)   :: name
     logical                 :: tofits
     logical                 :: doblank
     type(identifier_prog_t) :: identifier
     real(kind=4)            :: bval,eval
     type(cube_t), pointer   :: cube
     type(cube_t), pointer   :: exported
   contains
     procedure, private :: header => cubemain_export_prog_header
     procedure, private :: data   => cubemain_export_prog_data
  end type export_prog_t
  !
contains
  !
  subroutine cubemain_export_command(line,error)
    !---------------------------------------------------------------------
    ! Support routine for command:
    !   EXPORT [CubeID] [/FORMAT FITS|GDF]
    !---------------------------------------------------------------------
    character(len=*), intent(in)    :: line
    logical,          intent(inout) :: error
    !
    type(export_user_t) :: user
    character(len=*), parameter :: rname='EXPORT>COMMAND'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call export%parse(line,user,error)
    if (error)  return
    call export%main(user,error)
    if (error)  return
  end subroutine cubemain_export_command
  !
  !----------------------------------------------------------------------
  !
  subroutine cubemain_export_register(export,error)
    !---------------------------------------------------------------------
    ! 
    !---------------------------------------------------------------------
    class(export_comm_t), intent(inout) :: export
    logical,              intent(inout) :: error
    !
    type(cubeid_arg_t) :: cubearg
    type(keyword_arg_t) :: keyarg
    type(standard_arg_t) :: stdarg
    integer(kind=4), parameter :: nformats=2
    character(len=*), parameter :: formats(nformats) =  &
         (/ 'FITS','GDF ' /)
    integer(kind=4), parameter :: naccesses=2
    character(len=*), parameter :: accesses(naccesses) =  &
         (/ 'IMAGE   ','SPECTRUM' /)
    character(len=*), parameter :: comm_abstract = &
         'Export reduced data'
    character(len=*), parameter :: comm_help = &
         'Export a cube to the RED (reduction) directory, by default&
         & the file name is based on its family name and flag. The&
         & option /NAME allows to customize the output file base name&
         &, and optionally the directory and the file extension.'
    character(len=*), parameter :: rname='EXPORT>REGISTER'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call cubetools_register_command(&
         'EXPORT','[cube]',&
         comm_abstract,&
         comm_help,&
         cubemain_export_command,&
         export%comm,error)
    if (error) return
    call cubearg%register( &
         'CUBE', &
         'Cube to be exported',  &
         strg_id,&
         code_arg_optional,  &
         [flag_any], &
         error)
    if (error) return
    !
    call cubetools_register_option(&
         'FORMAT','type',&
         'Set the data format to be used',&
         'Default is FITS if the option is omitted',&
         export%format,error)
    if (error) return
    call keyarg%register( &
         'type',  &
         'Format type', &
         strg_id,&
         code_arg_mandatory, &
         formats,  &
         .not.flexible,  &
         export%format_arg,  &
         error)
    if (error) return
    !
    call cubetools_register_option(&
         'BLANKING','[Bval [Eval]]',&
         'Export with blank values instead of NaN (GDF only)',&
         strg_id,&
         export%blanking,error)
    if (error) return
    call stdarg%register( &
         'Bval',  &
         'Blanking value (default -1000)', &
         strg_id,&
         code_arg_optional, &
         error)
    if (error) return
    call stdarg%register( &
         'Eval',  &
         'Tolerance on blanking value (default 0)', &
         strg_id,&
         code_arg_optional, &
         error)
    if (error) return
    !
    call cubetools_register_option(&
         'FILENAME','filename',&
         'Customize the name of exported file on disk',&
         'This option regards customizing only the name of the file&
         & that is to be created on disk. The family nams is not&
         & changed by this option. To change the family name please&
         & use option /AS. If this option is not used the default&
         & file name on disk is: family-flag1-...-flagn',&
         export%name,error)
    if (error) return
    call stdarg%register( &
         'filename',  &
         'Customized file name', &
         strg_id,&
         code_arg_optional, &
         error)
    if (error) return
    !
    call export%as%register(&
         'Customize the identifier of the exported cube',&
         changeflags, error)
    if (error) return
    !
    call cubetools_register_option(&
         'DIRECTORY','dirname',&
         'Customize the export directory',&
         'If option is not given defaults to the default RED&
         & directory (set with command ADM\DIRECTORY)',&
         export%dir,error)
    if (error) return
    call stdarg%register(&
         'directory', &
         'Export directory', &
         strg_id,&
         code_arg_optional, error)
    if (error) return
    !
    call cubetools_register_option(&
         'ACCESS','type',&
         'Set the access order of the output file',&
         'Defaults to last used access',&
         export%access,error)
    if (error) return
    call keyarg%register( &
         'type',  &
         'Access type', &
         strg_id,&
         code_arg_mandatory, &
         accesses,  &
         .not.flexible,  &
         export%access_arg,  &
         error)
    if (error) return
  end subroutine cubemain_export_register
  !
  subroutine cubemain_export_parse(export,line,user,error)
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(export_comm_t), intent(in)    :: export
    character(len=*),     intent(in)    :: line
    type(export_user_t),  intent(out)   :: user
    logical,              intent(inout) :: error
    !
    character(len=argu_l) :: argum
    character(len=12) :: key
    integer(kind=4) :: iformat,iaccess
    logical :: doaccess
    character(len=*), parameter :: rname='EXPORT>PARSE'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call cubeadm_cubeid_parse(line,export%comm,user%cubeids,error)
    if (error) return
    !
    ! /FORMAT
    argum = 'FITS'  ! Default
    call cubetools_getarg(line,export%format,1,argum,.not.mandatory,error)
    if (error)  return
    call cubetools_keyword_user2prog(export%format_arg,argum,iformat,key,error)
    if (error)  return
    user%tofits = key.eq.'FITS'
    !
    ! /BLANKING
    call export%blanking%present(line,user%doblank,error)
    if (error) return
    if (user%tofits .and. user%doblank) then
      call cubemain_message(seve%e,rname,  &
        '/'//trim(export%blanking%name)//' is only relevant for GDF format')
      error = .true.
      return
    endif
    call cubetools_getarg(line,export%blanking,1,user%bval,.not.mandatory,error)
    if (error)  return
    call cubetools_getarg(line,export%blanking,2,user%eval,.not.mandatory,error)
    if (error)  return
    !
    ! /AS family[:flags]
    call export%as%parse(line,user%identifier,error)
    if (error) return
    !
    ! /DIRECTORY
    call export%dir%present(line,user%dodir,error)
    if (error) return
    if (user%dodir) then
       call cubetools_getarg(line,export%dir,1,user%dir,mandatory,error)
       if (error) return
    endif
    !
    ! /FILENAME
    call export%name%present(line,user%dofilename,error)
    if (error) return
    if (user%dofilename) then
       call cubetools_getarg(line,export%name,1,user%filename,mandatory,error)
       if (error) return
    endif    
    !
    ! /ACCESS IMAGE|SPECTRUM
    call export%access%present(line,doaccess,error)
    if (error) return
    if (doaccess) then
      call cubetools_getarg(line,export%access,1,argum,mandatory,error)
      if (error)  return
      call cubetools_keyword_user2prog(export%access_arg,argum,iaccess,key,error)
      if (error)  return
      select case (key)
      case ('IMAGE')
        user%access = code_access_imaset
      case ('SPECTRUM')
         user%access = code_access_speset
      case default
         call cubemain_message(seve%e,rname,'Unknown access')
         error = .true.
         return
      end select
    else
      user%access = code_access_imaset_or_speset
    endif
  end subroutine cubemain_export_parse
  !
  subroutine cubemain_export_main(export,user,error)
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(export_comm_t), intent(in)    :: export
    type(export_user_t),  intent(inout) :: user
    logical,              intent(inout) :: error
    !
    type(export_prog_t) :: prog
    character(len=*), parameter :: rname='EXPORT>MAIN'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call user%toprog(prog,error)
    if (error) return
    call prog%header(error)
    if (error)  return
    call cubeadm_timing_prepro2process()
    call prog%data(error)
    if (error)  return
    call cubeadm_timing_process2postpro()
  end subroutine cubemain_export_main
  !
  !----------------------------------------------------------------------
  !
  subroutine cubemain_export_user_toprog(user,prog,error)
    use cubeadm_get
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(export_user_t), intent(in)    :: user
    type(export_prog_t),  intent(inout) :: prog
    logical,              intent(inout) :: error
    !
    character(file_l):: dirname
    integer(kind=4) :: ldir,iflag
    type(flag_t), pointer :: flag
    character(len=*), parameter :: rname='EXPORT>USER>TOPROG'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call cubeadm_cubeid_get_header(export%comm,icube,user%cubeids,user%access,  &
      code_read,prog%cube,error)
    if (error) return
    !
    prog%tofits  = user%tofits
    prog%doblank = user%doblank
    prog%bval    = user%bval
    prog%eval    = user%eval
    !
    call user%identifier%toprog(prog%cube,prog%identifier,error)
    if (error) return
    !
    if (user%dodir) then
       dirname = user%dir
       ldir = len_trim(dirname)
       if (dirname(ldir:ldir).ne.'/')  dirname(ldir+1:ldir+1) = '/'
    else
       dirname = dir%red
    endif
    call cubeadm_directory_create(dirname,error)
    if (error)  return
    !
    if (user%dofilename) then
       prog%name = trim(dirname)//user%filename
    else
       prog%name = trim(dirname)//prog%identifier%family
       if (allocated(prog%identifier%flags)) then
          do iflag=1,size(prog%identifier%flags)
             prog%name = trim(prog%name)//prog%identifier%flags(iflag)%get_suffix()
          enddo
       else
          do iflag=1,prog%cube%node%flag%n
             flag => cubedag_flag_ptr(prog%cube%node%flag%list(iflag)%p,error)
             if (error)  return
             prog%name = trim(prog%name)//flag%get_suffix()
          enddo
       endif
    endif
  end subroutine cubemain_export_user_toprog
  !
  !----------------------------------------------------------------------
  !
  subroutine cubemain_export_prog_header(prog,error)
    use cubedag_allflags
    use cubeadm_clone
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(export_prog_t), intent(inout) :: prog
    logical,              intent(inout) :: error
    !
    type(flag_t) :: noflag(0)
    character(len=*), parameter :: rname='EXPORT>PROG>HEADER'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call cubeadm_clone_header(prog%cube,noflag,prog%exported,error,keepflags=.true.)
    if (error)  return
    !
    call prog%identifier%apply(prog%exported,error)
    if (error) return
  end subroutine cubemain_export_prog_header
  !
  subroutine cubemain_export_prog_data(prog,error)
    use cubetools_dataformat
    use cubeio_cube_define
    use cubeio_file
    use cubemain_copy
    !---------------------------------------------------------------------
    !
    !---------------------------------------------------------------------
    class(export_prog_t), intent(inout) :: prog
    logical,              intent(inout) :: error
    !
    character(len=*), parameter :: rname='EXPORT>PROG>CUBE'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    ! Sanity
    if (prog%cube%node%origin.eq.code_origin_imported) then
      call cubemain_message(seve%w,rname,'Exporting an imported cube')
    elseif (prog%cube%node%origin.eq.code_origin_exported) then
      call cubemain_message(seve%w,rname,'Re-exporting an exported cube')
    endif
    !
    call cubeio_cube_define_buffering(prog%exported%prog,code_buffer_disk,error)
    if (error)  return
    call cubeio_cube_define_filename(prog%exported%prog,prog%name,error)
    if (error)  return
    if (prog%tofits) then
      call cubeio_cube_define_filekind(prog%exported%prog,code_dataformat_fits,error)
    else
      call cubeio_cube_define_filekind(prog%exported%prog,code_dataformat_gdf,error)
    endif
    if (error)  return
    if (prog%doblank) then
      call cubeio_cube_define_reblank(prog%exported%prog,prog%bval,prog%eval,error)
      if (error)  return
    endif
    !
    call cubemain_copy_data(prog%cube,prog%exported,error)
    if (error)  return
    !
    ! Update cube in memory
    prog%exported%node%origin = code_origin_exported
  end subroutine cubemain_export_prog_data
end module cubemain_export
!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
