!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!
module cubemain_noise
  use cube_types
  use cubetools_structure
  use cubeadm_cubeid_types
  use cubemain_messaging
  use cubemain_range
  use cubemain_windowing
  !
  public :: noise
  public :: cubemain_noise_command
  private
  !
  type :: noise_comm_t
     type(option_t), pointer :: comm
     type(option_t), pointer :: mad     
     type(option_t), pointer :: nchan
     type(range_opt_t)       :: range
   contains
     procedure, public  :: register => cubemain_noise_register
     procedure, private :: parse    => cubemain_noise_parse
     procedure, private :: main     => cubemain_noise_main
  end type noise_comm_t
  type(noise_comm_t) :: noise
  !
  integer(kind=4), parameter :: icube = 1
  integer(kind=4), parameter :: iwind = 2
  type noise_user_t
     type(cubeid_user_t)   :: cubeids           ! Signal and window cubeids
     character(len=argu_l) :: nchan             ! Number of channels to be used  
     type(range_array_t)   :: range             ! Range(s) to be ignored
     logical               :: domad = .false.   ! Do MAD noise
     logical               :: dorange = .false. ! Do range 
   contains
     procedure, private :: toprog => cubemain_noise_user_toprog
  end type noise_user_t
  !
  type noise_prog_t
     logical               :: domad = .false.
     logical               :: dowin = .false.
     logical               :: doloc = .false.
     integer(kind=chan_k)  :: nchan = 0  ! number of channels to be used
     integer(kind=chan_k)  :: nnois = 0  ! Number of noise estimates
     integer(kind=chan_k)  :: nhalf = 0  ! Half number of used channels
     type(window_array_t)  :: global     ! One set of windows for the full index
     type(cube_t), pointer :: local      ! Number and positions of windows depend on the spectrum
     type(cube_t), pointer :: cube
     type(cube_t), pointer :: noise
   contains
     procedure, private :: header        => cubemain_noise_prog_header
     procedure, private :: data          => cubemain_noise_prog_data
     procedure, private :: loop          => cubemain_noise_prog_data_loop
     procedure, private :: mad_loop      => cubemain_noise_prog_data_mad_loop
     procedure, private :: mad_act       => cubemain_noise_prog_data_mad_act
     procedure, private :: locwind_loop  => cubemain_noise_prog_data_locwind_loop
     procedure, private :: locwind_act   => cubemain_noise_prog_data_locwind_act
     procedure, private :: glowind_loop  => cubemain_noise_prog_data_glowind_loop
     procedure, private :: glowind_act   => cubemain_noise_prog_data_glowind_act
     procedure, private :: compute_noise => cubemain_noise_prog_data_compute_noise
  end type noise_prog_t
  !
contains
  !
  subroutine cubemain_noise_command(line,error)
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    character(len=*), intent(in)  :: line
    logical,          intent(out) :: error
    !
    type(noise_user_t) :: user
    type(noise_prog_t) :: prog
    character(len=*), parameter :: rname='NOISE>COMMAND'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call noise%parse(line,user,error)
    if (error) return
    call noise%main(user,prog,error)
    if (error) return
  end subroutine cubemain_noise_command
  !
  !------------------------------------------------------------------------
  !
  subroutine cubemain_noise_register(noise,error)
    use cubedag_allflags
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(noise_comm_t), intent(inout) :: noise
    logical,             intent(inout) :: error
    !
    type(cubeid_arg_t) :: cubearg
    type(standard_arg_t) :: stdarg
    character(len=*), parameter :: comm_abstract = &
         'Compute the noise RMS of each spectrum of a cube'
    character(len=*), parameter :: comm_help = &
         'NOISE uses either a global window to compute the moments &
         &(/RANGE), or individual windows set for each spectrum&
         & (created by CUBE\WINDOW).'
    character(len=*), parameter :: rname='NOISE>REGISTER'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call cubetools_register_command(&
         'NOISE','[cube [window]]',&
         comm_abstract,&
         comm_help,&
         cubemain_noise_command,&
         noise%comm,error)
    if (error) return
    call cubearg%register(&
         'SIGNAL',&
         'Signal cube',&
         strg_id,&
         code_arg_optional,&
         [flag_cube],&
         error)
    if (error) return
    call cubearg%register(&
         'WINDOW',&
         'Local spectral range(s) containing signal',&
         strg_id,&
         code_arg_optional,&
         [flag_window],&
         error)
    if (error) return
    !
    call noise%range%register(&
         'RANGE',&
         'Define global spectral range(s) containing signal',&
         range_is_multiple,error)
    if (error) return
    !
    call cubetools_register_option(&
         'MAD','',&
         'Compute Median Absolute Deviation noise instead of RMS',&
         strg_id,&
         noise%mad,error)
    if (error) return
    !
    call cubetools_register_option(&
         'NCHAN','nchan',&
         'Number of channels in each block to be used to compute noise',&
         strg_id,&
         noise%nchan,error)
    if (error) return
    call stdarg%register(&
         'nchan',&
         'number of channels',&
         strg_id,&
         code_arg_mandatory,&
         error)
    if (error) return
  end subroutine cubemain_noise_register
  !
  subroutine cubemain_noise_parse(noise,line,user,error)
    !----------------------------------------------------------------------
    ! NOISE cubeid [windowid]
    ! /NCHAN nchan
    ! /RANGE vfirst vlast
    ! /MAD
    !----------------------------------------------------------------------
    class(noise_comm_t), intent(in)    :: noise
    character(len=*),    intent(in)    :: line
    type(noise_user_t),  intent(out)   :: user
    logical,             intent(inout) :: error
    !
    character(len=*), parameter :: rname='NOISE>PARSE'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call cubeadm_cubeid_parse(line,noise%comm,user%cubeids,error)
    if (error) return
    call noise%mad%present(line,user%domad,error)
    if (error) return
    call cubemain_noise_parse_nchan(line,noise%nchan,1,user%nchan,error)
    if (error) return
    call noise%range%parse(line,user%dorange,user%range,error)
    if (error) return
  end subroutine cubemain_noise_parse
  !
  subroutine cubemain_noise_parse_nchan(line,opt,iarg,nchan,error)
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    character(len=*),      intent(in)    :: line
    type(option_t),        intent(in)    :: opt
    integer(kind=4),       intent(in)    :: iarg
    character(len=argu_l), intent(out)   :: nchan
    logical,               intent(inout) :: error
    !
    logical :: present
    character(len=*), parameter :: rname='NOISE>PARSE>NCHAN'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call opt%present(line,present,error)
    if (error) return
    if (present) then
       call cubetools_getarg(line,opt,iarg,nchan,mandatory,error)
       if (error) return
    else
       nchan = strg_star
    endif
  end subroutine cubemain_noise_parse_nchan
  !
  subroutine cubemain_noise_main(noise,user,prog,error)
    use cubeadm_timing
    use cubeadm_get
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(noise_comm_t), intent(in)    :: noise
    type(noise_user_t),  intent(inout) :: user
    type(noise_prog_t),  intent(inout) :: prog
    logical,             intent(inout) :: error
    !
    character(len=*), parameter :: rname='NOISE>MAIN'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call user%toprog(noise,prog,error)
    if (error) return
    call prog%header(error)
    if (error) return
    call cubeadm_timing_prepro2process()
    call prog%data(error)
    if (error) return
    call cubeadm_timing_process2postpro()
  end subroutine cubemain_noise_main
  !
  !------------------------------------------------------------------------
  !
  subroutine cubemain_noise_user_toprog(user,noise,prog,error)
    use cubetools_user2prog
    use cubetools_consistency_methods
    use cubedag_allflags
    use cubeadm_get
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(noise_user_t), intent(inout) :: user
    class(noise_comm_t), intent(in)    :: noise
    type(noise_prog_t),  intent(inout) :: prog
    logical,             intent(inout) :: error
    !
    logical :: prob
    integer(kind=chan_k) :: default
    character(len=*), parameter :: rname='NOISE>USER>TOPROG'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    default = 0
    call cubetools_user2prog_resolve_star(user%nchan,default,prog%nchan,error)
    if (error) return    
    call cubeadm_cubeid_get_header(noise%comm,icube,user%cubeids,code_access_speset,code_read,prog%cube,error)
    if (error) return
    !
    prog%domad = user%domad
    if (user%domad) then
       ! No window searched
       prog%dowin = .false.
    else
       prog%dowin = .true.
       if (user%dorange) then
          ! Define a global window
          prog%doloc = .false.
          ! Ranges are truncated to avoid segfault
          user%range%val(:)%truncate = .true.
          call noise%range%user2prog(prog%cube,user%range,prog%global,error)
          if (error) return
       else
          prog%doloc = .true.
          prob = .false.
          ! Search for a data cube with the definitions of the local windows
          call cubeadm_cubeid_get_header(noise%comm,iwind,user%cubeids,code_access_speset,code_read,prog%local,error)
          if (error) return
          call cubetools_consistency_spatial('Input cube',prog%cube%head,'Window',prog%local%head,prob,error)
          if (error) return
          if (cubetools_consistency_failed(rname,prob,error)) return
       endif
    endif
  end subroutine cubemain_noise_user_toprog
  !
  !----------------------------------------------------------------------
  !
  subroutine cubemain_noise_prog_header(prog,error)
    use cubetools_header_methods
    use cubedag_allflags
    use cubeadm_clone
    !---------------------------------------------------------------------- 
    ! To deal with edge effects, we cheat on the last frequency segment
    ! that we attribute beyond the original frequency range while we 
    ! always compute the noise inside this range. This should have no 
    ! consequence as the noise should be interpolated in frequency
    ! in later use.
    !----------------------------------------------------------------------
    class(noise_prog_t), intent(inout) :: prog
    logical,             intent(inout) :: error
    !
    integer(kind=chan_k) :: in_nchan,nhalf
    character(len=mess_l) :: mess
    character(len=*), parameter :: rname='NOISE>PROG>HEADER'
    !
    if ((prog%domad).and.(prog%dowin)) then
       call cubemain_message(seve%e,rname,'Options /RANGE and /MAD are exclusives')
       error = .true.
       return
    else if (.not.((prog%domad).or.(prog%dowin))) then
       call cubemain_message(seve%e,rname,'Do not understand what you want')
       error = .true.
       return
    endif
    !
    call cubetools_header_get_nchan(prog%cube%head,in_nchan,error)
    if (error) return
    call cubeadm_clone_header(prog%cube,flag_noise,prog%noise,error)
    if (error) return
    if ((prog%nchan.lt.0).or.(prog%nchan.gt.in_nchan)) then
       call cubemain_message(seve%e,rname,'Number of asked channels outside cube range')
       error = .true.
       return
    else if (prog%nchan.eq.0) then
       ! Default value
       prog%nchan = in_nchan
       nhalf = prog%nchan
       prog%nnois = 1
    else
       ! User value
       ! Ensure that prog%nchan will be odd in output because the median definition 
       ! is simpler
       if (mod(prog%nchan,2).eq.0) then
          prog%nchan = prog%nchan-1
       endif
       nhalf = floor(0.5*prog%nchan)
       prog%nnois  = floor(dble(in_nchan)/dble(nhalf))
       if (nhalf*prog%nnois.lt.in_nchan) then
          prog%nnois = prog%nnois+1
       endif
    endif
    if ((prog%nchan.lt.5)) then
       call cubemain_message(seve%e,rname,'Minimum number of channels must be 5')
       error = .true.
       return
    else if (prog%nchan.gt.in_nchan) then
       call cubemain_message(seve%e,rname,'You asked to compute the noise rms on more samples than channels in the spectrum!')
       error = .true.
       return
    endif
    prog%nhalf = floor(0.5*prog%nchan)
    ! Update noise header accordingly
    call cubetools_header_put_nchan(prog%nnois,prog%noise%head,error)
    if (error) return
    call cubetools_header_multiply_spectral_spacing(nhalf,prog%noise%head,error)
    if (error) return
    call cubetools_header_rederive_spectral_axes(prog%noise%head,error)
    if (error) return
    ! User feedback
    write(mess,'(a,i0,a,i0,a)') 'Computing ',prog%nnois,' noise values, each using ',prog%nchan,' channels'
    call cubemain_message(seve%i,rname,mess)
  end subroutine cubemain_noise_prog_header
  !
  subroutine cubemain_noise_prog_data(prog,error)
    use cubeadm_opened
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(noise_prog_t), intent(inout) :: prog
    logical,             intent(inout) :: error
    !
    type(cubeadm_iterator_t) :: iter
    character(len=*), parameter :: rname='NOISE>PROG>DATA'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call cubeadm_datainit_all(iter,error)
    if (error) return
    !$OMP PARALLEL DEFAULT(none) SHARED(prog,error) FIRSTPRIVATE(iter)
    !$OMP SINGLE
    do while (cubeadm_dataiterate_all(iter,error))
       if (error)  exit
       !$OMP TASK SHARED(prog) FIRSTPRIVATE(iter,error)
       if (.not.error) call prog%loop(iter%first,iter%last,error)
       !$OMP END TASK
    enddo ! ie
    !$OMP END SINGLE
    !$OMP END PARALLEL
  end subroutine cubemain_noise_prog_data
  !
  subroutine cubemain_noise_prog_data_loop(prog,first,last,error)
    !----------------------------------------------------------------------
    ! 
    !----------------------------------------------------------------------
    class(noise_prog_t),  intent(inout) :: prog
    integer(kind=entr_k), intent(in)    :: first
    integer(kind=entr_k), intent(in)    :: last
    logical,              intent(inout) :: error
    !
    character(len=*), parameter :: rname='NOISE>PROG>DATA>LOOP'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    if (prog%domad) then
       call prog%mad_loop(first,last,error)
    else if (prog%dowin) then
       if (prog%doloc) then
          call prog%locwind_loop(first,last,error)
       else
          call prog%glowind_loop(first,last,error)
       endif
    endif
  end subroutine cubemain_noise_prog_data_loop  
  !
  !------------------------------------------------------------------------
  !
  subroutine cubemain_noise_prog_data_mad_loop(prog,first,last,error)
    use cubeadm_entryloop
    use cubemain_spectrum_real
    !----------------------------------------------------------------------
    ! 
    !----------------------------------------------------------------------
    class(noise_prog_t),  intent(inout) :: prog
    integer(kind=entr_k), intent(in)    :: first
    integer(kind=entr_k), intent(in)    :: last
    logical,              intent(inout) :: error
    !
    type(spectrum_t) :: spec,line,good,diff,noise
    integer(kind=entr_k) :: ie
    character(len=*), parameter :: rname='NOISE>PROG>DATA>MAD>LOOP'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    call noise%reallocate('noise',prog%nnois,error)
    if (error) return
    call good%reallocate('good',prog%nchan,error)
    if (error) return
    call diff%reallocate('diff',prog%nchan,error)
    if (error) return
    call spec%reallocate('spec',prog%cube%head%arr%n%c,error)
    if (error) return
    !
    do ie=first,last
       call cubeadm_entryloop_iterate(ie,error)
       if (error)  return
       call prog%mad_act(ie,spec,line,good,diff,noise,error)
       if (error)  return
    enddo
  end subroutine cubemain_noise_prog_data_mad_loop
  !
  subroutine cubemain_noise_prog_data_mad_act(prog,ie,spec,line,good,diff,noise,error)
    use cubetools_nan
    use cubemain_spectrum_real
    use cubemain_spectrum_blanking
    use cubemain_statistics
    !----------------------------------------------------------------------
    ! Compute a running median absolute deviation filter
    !----------------------------------------------------------------------
    class(noise_prog_t),  intent(inout) :: prog
    integer(kind=entr_k), intent(in)    :: ie
    type(spectrum_t),     intent(inout) :: spec
    type(spectrum_t),     intent(inout) :: line
    type(spectrum_t),     intent(inout) :: good
    type(spectrum_t),     intent(inout) :: diff
    type(spectrum_t),     intent(inout) :: noise
    logical,              intent(inout) :: error
    !  
    integer(kind=chan_k) :: nn,in
    integer(kind=chan_k) :: nc,first,last,nhalf
    integer(kind=chan_k) :: nchan
    real(kind=sign_k) :: median
    character(len=*), parameter :: rname='NOISE>PROG>DATA>MAD>ACT'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    nchan = prog%nchan
    !
    nhalf = floor(0.5*nchan)
    nn = prog%noise%head%arr%n%c
    nc = prog%cube%head%arr%n%c
    !
    spec%noi = gr4nan
    !
    call spec%get(prog%cube,ie,error)
    if (error) return
    first = 1
    do in=1,nn
       ! The following ensures that
       !    1) we never get past the number of channels
       !    2) we always compute the mad on nchan contiguous samples
       last  = min(first+nchan-1,nc)
       first = last-nchan+1
       call line%point_to(spec,first,last,1.0,error)
       if (error) return
       call cubemain_spectrum_unblank(line,good,error)
       if (error) return
       if (good%n.ge.1) then
          median = cubemain_median(good%t(1:good%n))
          noise%t(in) = cubemain_mad(good%t(1:good%n),median)
       else
          noise%t(in) = gr4nan
       endif
       first = first+nhalf
    enddo ! in
    call noise%put(prog%noise,ie,error)
    if (error) return   
  end subroutine cubemain_noise_prog_data_mad_act
  !
  !------------------------------------------------------------------------
  !
  subroutine cubemain_noise_prog_data_locwind_loop(prog,first,last,error)
    use cubeadm_entryloop
    use cubemain_spectrum_real
    !----------------------------------------------------------------------
    ! 
    !----------------------------------------------------------------------
    class(noise_prog_t),  intent(inout) :: prog
    integer(kind=entr_k), intent(in)    :: first
    integer(kind=entr_k), intent(in)    :: last
    logical,              intent(inout) :: error
    !
    type(spectrum_t) :: spec,base,window,noise
    integer(kind=entr_k) :: ie
    character(len=*), parameter :: rname='NOISE>DATA>LOCWIND>LOOP'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    if (prog%local%head%arr%n%c/2.le.0) then
       call cubemain_message(seve%e,rname,'Negative or zero number of windows')
       error = .true.
       return
    endif
    !
    call base%reallocate('base',prog%nchan,error)
    if (error) return
    call spec%reallocate('spec',prog%cube%head%arr%n%c,error)
    if (error) return
    call window%reallocate('window',prog%local%head%arr%n%c,error)
    if (error) return
    call noise%reallocate('noise',prog%noise%head%arr%n%c,error)
    if (error) return
    !
    do ie=first,last
       call cubeadm_entryloop_iterate(ie,error)
       if (error)  return
       call prog%locwind_act(ie,window,spec,base,noise,error)
       if (error)  return
    enddo
  end subroutine cubemain_noise_prog_data_locwind_loop
  !
  subroutine cubemain_noise_prog_data_locwind_act(prog,ie,window,spec,base,noise,error)
    use cubetools_nan
    use cubemain_topology
    use cubemain_spectrum_real
    !----------------------------------------------------------------------
    ! 
    !----------------------------------------------------------------------
    class(noise_prog_t), target, intent(inout) :: prog
    integer(kind=entr_k),        intent(in)    :: ie
    type(spectrum_t),            intent(inout) :: window
    type(spectrum_t),            intent(inout) :: spec
    type(spectrum_t),            intent(inout) :: base
    type(spectrum_t),            intent(inout) :: noise
    logical,                     intent(inout) :: error
    !
    integer(kind=wind_k) :: nw,iw
    integer(kind=chan_k) :: ichan(2)
    real(kind=coor_k) :: vrange(2)
    type(cube_t), pointer :: wind
    character(len=*), parameter :: rname='NOISE>DATA>LOCAL'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    wind => prog%local
    nw = wind%head%arr%n%c/2
    !
    call spec%get(prog%cube,ie,error)
    if (error) return
    call window%get(wind,ie,error)
    if (error) return
    ! Blank signal
    do iw=1,nw
       if (cubemain_window_blank(window,iw)) cycle
       vrange(:) = window%t(2*iw-1:2*iw)
       call cubemain_topo_vrange2crange(prog%cube,vrange,ichan,error)
       if (error) return
       spec%t(ichan(1):ichan(2)) = gr4nan
    enddo
    call prog%compute_noise(spec,base,noise,error)
    if (error) return
    call noise%put(prog%noise,ie,error)
    if (error) return
  end subroutine cubemain_noise_prog_data_locwind_act
  !
  !------------------------------------------------------------------------
  !
  subroutine cubemain_noise_prog_data_glowind_loop(prog,first,last,error)
    use cubeadm_entryloop
    use cubemain_spectrum_real
    !----------------------------------------------------------------------
    ! 
    !----------------------------------------------------------------------
    class(noise_prog_t), target, intent(inout) :: prog
    integer(kind=entr_k),        intent(in)    :: first
    integer(kind=entr_k),        intent(in)    :: last
    logical,                     intent(inout) :: error
    !
    type(spectrum_t) :: spec,base,noise
    integer(kind=entr_k) :: ie
    character(len=*), parameter :: rname='NOISE>DATA>WIN>GLOBAL>LOOP'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    if (prog%global%n.le.0) then
       call cubemain_message(seve%e,rname,'Negative or zero number of windows')
       error = .true.
       return
    endif
    !
    call base%reallocate('base',prog%cube%head%arr%n%c,error)
    if (error) return
    call spec%reallocate('spec',prog%cube%head%arr%n%c,error)
    if (error) return
    call noise%reallocate('noise',prog%noise%head%arr%n%c,error)
    if (error) return
    !
    do ie=first,last
       call cubeadm_entryloop_iterate(ie,error)
       if (error) return
       call prog%glowind_act(ie,spec,base,noise,error)
       if (error) return
    enddo
  end subroutine cubemain_noise_prog_data_glowind_loop
  !
  subroutine cubemain_noise_prog_data_glowind_act(prog,ie,spec,base,noise,error)
    use cubetools_nan
    use cubemain_spectrum_real
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(noise_prog_t), target, intent(inout) :: prog
    integer(kind=entr_k),        intent(in)    :: ie
    type(spectrum_t),            intent(inout) :: spec
    type(spectrum_t),            intent(inout) :: base
    type(spectrum_t),            intent(inout) :: noise
    logical,                     intent(inout) :: error
    !
    integer(kind=chan_k) :: iw
    type(window_array_t), pointer :: wind
    character(len=*), parameter :: rname='NOISE>PROG>DATA>GLOWIND>ACT'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    wind => prog%global
    !
    call spec%get(prog%cube,ie,error)
    if (error) return
    ! Blank signal
    do iw=1,wind%n
       spec%t(wind%val(iw)%o(1):wind%val(iw)%o(2)) = gr4nan
    enddo
    call prog%compute_noise(spec,base,noise,error)
    if (error) return
    call noise%put(prog%noise,ie,error)
    if (error) return
  end subroutine cubemain_noise_prog_data_glowind_act
  !
  !------------------------------------------------------------------------
  !
  subroutine cubemain_noise_prog_data_compute_noise(prog,spec,base,noise,error)
    use cubemain_spectrum_real
    use cubemain_spectrum_blanking
    use cubemain_spectrum_operations
    !----------------------------------------------------------------------
    !
    !----------------------------------------------------------------------
    class(noise_prog_t), intent(inout) :: prog
    type(spectrum_t),    intent(inout) :: spec
    type(spectrum_t),    intent(inout) :: base
    type(spectrum_t),    intent(inout) :: noise
    logical,             intent(inout) :: error
    !
    type(spectrum_t) :: line
    integer(kind=chan_k) :: first,last,in
    character(len=*), parameter :: rname='NOISE>PROG>DATA>COMPUTE>NOISE'
    !
    call cubemain_message(mainseve%trace,rname,'Welcome')
    !
    first = 1
    do in=1,prog%nnois
       last  = min(first+prog%nchan-1,prog%cube%head%arr%n%c)
       first = last-prog%nchan+1
       call line%point_to(spec,first,last,1.0,error)
       if (error) return
       call cubemain_spectrum_unblank(line,base,error)
       if (error) return
       call cubemain_spectrum_base_noise(base,error)
       if (error) return
       noise%t(in) = base%noi
       first = first+prog%nhalf
    enddo
  end subroutine cubemain_noise_prog_data_compute_noise
end module cubemain_noise
!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
