------------------------------------------------------------------------------
--                                                                          --
--                      CHARLES CONTAINER LIBRARY                           --
--                                                                          --
--              Copyright (C) 2001-2003 Matthew J Heaney                    --
--                                                                          --
-- The Charles Container Library ("Charles") is free software; you can      --
-- redistribute it and/or modify it under terms of the GNU General Public   --
-- License as published by the Free Software Foundation; either version 2,  --
-- or (at your option) any later version.  Charles is distributed in the    --
-- hope that it will be useful, but WITHOUT ANY WARRANTY; without even the  --
-- implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. --
-- See the GNU General Public License for more details.  You should have    --
-- received a copy of the GNU General Public License distributed with       --
-- Charles;  see file COPYING.TXT.  If not, write to the Free Software      --
-- Foundation,  59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.    --
--                                                                          --
-- As a special exception, if other files instantiate generics from this    --
-- unit, or you link this unit with other files to produce an executable,   --
-- this unit does not by itself cause the resulting executable to be        --
-- covered by the GNU General Public License.  This exception does not      --
-- however invalidate any other reasons why the executable file might be    --
-- covered by the GNU Public License.                                       --
--                                                                          --
-- Charles is maintained by Matthew J Heaney.                               --
--                                                                          --
-- http://home.earthlink.net/~matthewjheaney/index.html                     --
-- mailto:matthewjheaney@earthlink.net                                      --
--                                                                          --
------------------------------------------------------------------------------

with Ada.Unchecked_Deallocation;
--with System;  use type System.Address;

package body Charles.Sets.Hashed.Unbounded is

   type Node_Type is
      record
         Element : aliased Element_Type;
         Next    : Node_Access;
      end record;

   function Next (Node : Node_Access) return Node_Access is
   begin
      return Node.Next;
   end;

   procedure Set_Next (Node : Node_Access; Next : Node_Access) is
   begin
      Node.Next := Next;
   end;


   function Hash (Node : Node_Access) return Hash_Type is
   begin
      return Hash (Node.Element);
   end;


   procedure Free (X : in out Node_Access) is
      procedure Deallocate_Node is
         new Ada.Unchecked_Deallocation (Node_Type, Node_Access);
   begin
      Deallocate_Node (X);
   end;


   procedure Adjust (Container : in out Container_Type) is

      function New_Node (Src : Node_Access) return Node_Access is
      begin
         return new Node_Type'(Src.Element, null);
      end;

      procedure Adjust is
         new Hash_Table_Types.Generic_Adjust (New_Node);
   begin
      Adjust (Container.Hash_Table);
   end;


   procedure Finalize (Container : in out Container_Type) is
   begin
      Finalize (Container.Hash_Table);
   end;


   procedure Assign
     (Target : in out Container_Type;
      Source : in     Container_Type) is

      function New_Node (Src : Node_Access) return Node_Access is
      begin
         return new Node_Type'(Src.Element, null);
      end;

      procedure Assign is
        new Hash_Table_Types.Generic_Assign (New_Node);
   begin
      Assign (Target.Hash_Table, Source.Hash_Table);
   end;



   function "=" (Left, Right : Container_Type) return Boolean is

      function Is_Equal_Element (L, R : Node_Access) return Boolean is
         pragma Inline (Is_Equal_Element);
      begin
         return L.Element = R.Element;
      end;

      function Is_Equal is
        new Hash_Table_Types.Generic_Equal (Is_Equal_Element);
   begin
      return Is_Equal (Left.Hash_Table, Right.Hash_Table);
   end;


   function Length (Container : Container_Type) return Natural is
   begin
      return Container.Hash_Table.Length;
   end;


   function Is_Empty (Container : Container_Type) return Boolean is
   begin
      return Length (Container) = 0;
   end;


   procedure Clear (Container : in out Container_Type) is
   begin
      Clear (Container.Hash_Table);
   end;


   procedure Swap (Left, Right : in out Container_Type) is
   begin
      Swap (Left.Hash_Table, Right.Hash_Table);
   end;


   function Is_Equal_Key
     (Node : Node_Access;
      Key  : Element_Type) return Boolean is

      pragma Inline (Is_Equal_Key);
   begin
      return Is_Equal_Key (Node.Element, Key);
   end;


   package Element_Keys is
     new Hash_Table_Types.Generic_Keys
     (Element_Type,
      Hash,
      Is_Equal_Key);

   use Element_Keys;


   procedure Insert
     (Container : in out Container_Type;
      New_Item  : in     Element_Type;
      Iterator  :    out Iterator_Type;
      Success   :    out Boolean) is

      function New_Node (Next : Node_Access) return Node_Access is
         Node : constant Node_Access := new Node_Type'(New_Item, Next);
      begin
         return Node;
      end;

      procedure Insert is
        new Element_Keys.Generic_Conditional_Insert (New_Node);

      HT : Hash_Table_Type renames Container.Hash_Table;
   begin
      Resize (HT, HT.Length + 1);
      Insert (HT, New_Item, Iterator.Node, Success);
   end;


   procedure Insert
     (Container : in out Container_Type;
      New_Item  : in     Element_Type) is

      Iterator : Iterator_Type;
      Success  : Boolean;
   begin
      Insert (Container, New_Item, Iterator, Success);
   end;



   procedure Insert_Sans_Resize
     (Container : in out Container_Type;
      New_Item  : in     Element_Type;
      Iterator  :    out Iterator_Type;
      Success   :    out Boolean) is

      function New_Node (Next : Node_Access) return Node_Access is
         Node : constant Node_Access := new Node_Type'(New_Item, Next);
      begin
         return Node;
      end;

      procedure Insert is
        new Element_Keys.Generic_Conditional_Insert (New_Node);

      HT : Hash_Table_Type renames Container.Hash_Table;
   begin
      Resize (HT, Length => 1);
      Insert (HT, New_Item, Iterator.Node, Success);
   end;


   procedure Insert_Sans_Resize
     (Container : in out Container_Type;
      New_Item  : in     Element_Type) is

      Iterator : Iterator_Type;
      Success  : Boolean;
   begin
      Insert_Sans_Resize (Container, New_Item, Iterator, Success);
   end;


   procedure Delete
     (Container : in out Container_Type;
      Item      : in     Element_Type) is
   begin
      Delete (Container.Hash_Table, Item);
   end;


   procedure Delete
     (Container : in out Container_Type;
      Iterator  : in out Iterator_Type) is
   begin
      Delete (Container.Hash_Table, Iterator.Node);
   end;


--     procedure Delete
--       (Container : in out Container_Type;
--        First     : in out Iterator_Type;
--        Back      : in     Iterator_Type) is
--     begin
--        Delete (Container.Hash_Table, First.Node, Back.Node);
--     end;


   function Find
     (Container : Container_Type;
      Item      : Element_Type) return Iterator_Type is
   begin
      return (Node => Find (Container.Hash_Table, Item));
   end;


   function Is_In
     (Item      : Element_Type;
      Container : Container_Type) return Boolean is
   begin
      return Find (Container, Item) /= Null_Iterator;
   end;


--     procedure Generic_Equal_Range
--       (Container   : in     Container_Type;
--        Item        : in     Element_Type) is

--        procedure Process (Node : Node_Access) is
--           pragma Inline (Process);
--        begin
--           Process (Iterator_Type'(Node => Node));
--        end;

--        procedure Iterate is
--          new Element_Keys.Generic_Equal_Range (Process);
--     begin
--        Iterate (Container.Hash_Table, Item);
--     end;


   function Is_Equal_Key
     (Left, Right : Iterator_Type) return Boolean is
   begin
      return Is_Equal_Key (Left.Node.Element, Right.Node.Element);
   end;


   function Is_Equal_Key
     (Left  : Iterator_Type;
      Right : Element_Type) return Boolean is
   begin
      return Is_Equal_Key (Left.Node.Element, Right);
   end;


   function Is_Equal_Key
     (Left  : Element_Type;
      Right : Iterator_Type) return Boolean is
   begin
      return Is_Equal_Key (Right, Left);
   end;


   function Is_Equal (Left, Right : Iterator_Type) return Boolean is
   begin
      return Left.Node.Element = Right.Node.Element;
   end;


   function Is_Equal
     (Left  : Iterator_Type;
      Right : Element_Type) return Boolean is
   begin
      return Left.Node.Element = Right;
   end;


   function Is_Equal
     (Left  : Element_Type;
      Right : Iterator_Type) return Boolean is
   begin
      return Is_Equal (Right, Left);
   end;


   function Generic_Compare_Elements
     (Left, Right : Iterator_Type) return Boolean is
   begin
      return Compare (Left.Node.Element, Right.Node.Element);
   end;


--     function Count
--       (Container : Container_Type;
--        Item      : Element_Type) return Natural is

--        Result : constant Natural := Count (Container.Hash_Table, Item);
--        pragma Assert (Result <= 1);
--     begin
--        return Result;
--     end;


   function First (Container : Container_Type)
      return Iterator_Type is
   begin
      return (Node => First (Container.Hash_Table));
   end;

   function First_Element (Container : Container_Type)
     return Element_Type is

      Node : constant Node_Access :=
        First (Container.Hash_Table);
   begin
      return Node.Element;
   end;


   function Back (Container : Container_Type)
     return Iterator_Type is

      pragma Warnings (Off, Container);
   begin
      return Null_Iterator;
   end;


   function Succ
     (Container : Container_Type;
      Iterator  : Iterator_Type) return Iterator_Type is
   begin
      return (Node => Succ (Container.Hash_Table, Iterator.Node));
   end;


   procedure Increment
     (Container : in     Container_Type;
      Iterator  : in out Iterator_Type) is
   begin
      Iterator := Succ (Container, Iterator);
   end;


   function Element (Iterator : Iterator_Type) return Element_Type is
   begin
      return Iterator.Node.Element;
   end;


   function Generic_Element
     (Iterator : Iterator_Type) return Element_Access is
   begin
      return Iterator.Node.Element'Access;
   end;


   function Generic_Modify_Element
     (Iterator : Iterator_Type) return Element_Access is
   begin
      return Iterator.Node.Element'Access;
   end;


   procedure Copy_Element
     (Iterator : in     Iterator_Type;
      Item     :    out Element_Type) is
   begin
      Item := Iterator.Node.Element;
   end;


   procedure Generic_Select_Element
     (Iterator : in Iterator_Type) is
   begin
      Process (Iterator.Node.Element);
   end;


   procedure Generic_Iteration
     (Container : in Container_Type) is

      procedure Process (Node : Node_Access) is
      begin
         Process (Iterator_Type'(Node => Node));
      end;

      procedure Iterate is
         new Hash_Table_Types.Generic_Iteration (Process);
   begin
      Iterate (Container.Hash_Table);
   end;


   procedure Generic_Select_Elements
     (Container : in Container_Type) is

      procedure Process (Node : Node_Access) is
      begin
         Process (Node.Element);
      end;

      procedure Iterate is
         new Hash_Table_Types.Generic_Iteration (Process);
   begin
      Iterate (Container.Hash_Table);
   end;



   function Size (Container : Container_Type) return Natural is
   begin
      return Size (Container.Hash_Table);
   end;


   procedure Resize
     (Container : in out Container_Type;
      Length    : in     Natural) is
   begin
      Resize (Container.Hash_Table, Length);
   end;


--TODO:
--     function Index
--       (Container : Container_Type;
--        Item      : Element_Type) return Positive is
--     begin
--        return Index (Container.Hash_Table, Item);
--     end;


--     function Index
--       (Container : Container_Type;
--        Iterator  : Iterator_Type) return Positive is
--     begin
--        return Index (Container.Hash_Table, Iterator.Node);
--     end;


--     function Iterator
--       (Container : Container_Type;
--        Index     : Positive) return Iterator_Type is
--     begin
--        return (Node => Container.Hash_Table.Buckets (Index));
--     end;


   procedure Swap_Iterator (Left, Right : in out Iterator_Type) is
      LI : constant Iterator_Type := Left;
   begin
      Left := Right;
      Right := LI;
   end;


   package body Generic_Keys is

      function Is_Equal_Key
        (Node : Node_Access;
         Key  : Key_Type) return Boolean is

         pragma Inline (Is_Equal_Key);
      begin
         return Is_Equal_Key (Node.Element, Key);
      end;

      package Keys is
        new Hash_Table_Types.Generic_Keys
        (Key_Type,
         Hash,
         Is_Equal_Key);

      use Keys;


      function Find
        (Container : Container_Type;
         Key       : Key_Type) return Iterator_Type is
      begin
         return (Node => Find (Container.Hash_Table, Key));
      end;


      function Is_In
        (Key       : Key_Type;
         Container : Container_Type) return Boolean is
      begin
         return Find (Container, Key) /= Null_Iterator;
      end;


--        function Count
--          (Container : Container_Type;
--           Key       : Key_Type) return Natural is

--           Result : constant Natural := Count (Container.Hash_Table, Key);
--           pragma Assert (Result <= 1);
--        begin
--           return Result;
--        end;


      function Element
        (Container : Container_Type;
         Key       : Key_Type) return Element_Type is

         Node : constant Node_Access := Find (Container.Hash_Table, Key);
      begin
         return Node.Element;
      end;


      procedure Generic_Equal_Range
        (Container   : in     Container_Type;
         Key         : in     Key_Type) is

         procedure Process (Node : Node_Access) is
            pragma Inline (Process);
         begin
            Process (Iterator_Type'(Node => Node));
         end;

         procedure Iterate is
           new Keys.Generic_Equal_Range (Process);
      begin
         Iterate (Container.Hash_Table, Key);
      end;


      function Is_Equal_Key
        (Left  : Iterator_Type;
         Right : Key_Type) return Boolean is
      begin
         return Is_Equal_Key (Left.Node.Element, Right);
      end;


      function Is_Equal_Key
        (Left  : Key_Type;
         Right : Iterator_Type) return Boolean is
      begin
         return Is_Equal_Key (Right, Left);
      end;


--TODO:
--        function Index
--          (Container : Container_Type;
--           Key       : Key_Type) return Positive is
--        begin
--           return Index (Container.Hash_Table, Key);
--        end;


      procedure Delete
        (Container : in out Container_Type;
         Key       : in     Key_Type) is
      begin
         Delete (Container.Hash_Table, Key);
      end;


      package body Generic_Insertion is

         procedure Insert
           (Container : in out Container_Type;
            Key       : in     Key_Type;
            Iterator  :    out Iterator_Type;
            Success   :    out Boolean) is

            function New_Node (Next : Node_Access) return Node_Access is
               Node : Node_Access := new Node_Type;
            begin
               Set_Key (Node.Element, Key);
               Node.Next := Next;

               return Node;
            exception
               when others =>
                  Free (Node);
                  raise;
            end;

            procedure Insert is
              new Keys.Generic_Conditional_Insert (New_Node);

            HT : Hash_Table_Type renames Container.Hash_Table;
         begin
            Resize (HT, HT.Length + 1);
            Insert (HT, Key, Iterator.Node, Success);
         end;


         procedure Insert_Sans_Resize
           (Container : in out Container_Type;
            Key       : in     Key_Type;
            Iterator  :    out Iterator_Type;
            Success   :    out Boolean) is

            function New_Node (Next : Node_Access) return Node_Access is
               Node : Node_Access := new Node_Type;
            begin
               Set_Key (Node.Element, Key);
               Node.Next := Next;

               return Node;
            exception
               when others =>
                  Free (Node);
                  raise;
            end;

            procedure Insert is
              new Keys.Generic_Conditional_Insert (New_Node);

            HT : Hash_Table_Type renames Container.Hash_Table;
         begin
            Resize (HT, Length => 1);
            Insert (HT, Key, Iterator.Node, Success);
         end;

      end Generic_Insertion;

   end Generic_Keys;


end Charles.Sets.Hashed.Unbounded;


