#!/usr/bin/ruby
#
# Copyright 2009-2011 the original author or authors.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#
require 'socket'
require 'base64'

#-------------------------------------------
# Constants
#-------------------------------------------

DESTHOST = "localhost" # for security
DESTPORT = ENV.fetch("GROOVYSERVER_PORT", 1961)
IS_WINDOWS = RUBY_PLATFORM.downcase =~ /mswin(?!ce)|mingw|cygwin|bccwin/
HOME_DIR = IS_WINDOWS ? ENV['USERPROFILE'] : ENV['HOME']
COOKIE_FILE_BASE = HOME_DIR + "/.groovy/groovyserv/cookie"
GROOVYSERVER_CMD = File.expand_path(ENV.fetch("GROOVYSERV_HOME", File.dirname($0)+"/..") + "/bin/groovyserver")

#-------------------------------------------
# Classes
#-------------------------------------------

class Options
  attr_reader :client, :server
  def initialize
    @client = {
      :without_invoking_server => false,
      :port => DESTPORT,
      :quiet => false,
      :env_all => false,
      :env_include_mask => [],
      :env_exclude_mask => [],
      :help => false,
      :groovyserver_opt => [],
    }
    @server = {
      :help => false,
      :args => [],
    }
  end
end

#-------------------------------------------
# Global variables
#-------------------------------------------

$options = nil    # FIXME shouldn't use global variables

#-------------------------------------------
# Functions
#-------------------------------------------

def usage()
  puts "\
usage: groovyclient.rb -C[option for groovyclient] [args/options for groovy]
options:
  -Ch,-Chelp                       show this usage
  -Cp,-Cport <port>                specify the port to connect to groovyserver
  -Ck,-Ckill-server                kill the running groovyserver
  -Cr,-Crestart-server             restart the running groovyserver
  -Cq,-Cquiet                      suppress statring messages
  -Cenv <substr>                   pass environment variables of which a name
                                   includes specified substr
  -Cenv-all                        pass all environment variables
  -Cenv-exclude <substr>           don't pass environment variables of which a
                                   name includes specified substr"
end

def start_server(args)
  unless FileTest.executable? GROOVYSERVER_CMD
    STDERR.puts "ERROR: Command not found: #{GROOVYSERVER_CMD}"
    exit 1
  end
  if $options.client[:quiet]
    args << "-q"
  else
    command_str = "'#{GROOVYSERVER_CMD}' -p #{$options.client[:port]} #{args.join(' ')}"
    STDERR.printf "Invoking server: %s\n", command_str
  end
  system(GROOVYSERVER_CMD, "-p", $options.client[:port].to_s, *args)
end

def session(socket, args)
  send_command(socket, args)
  while true do
    IO.select([socket, $stdin]).each do |ins|
      if ins[0] == socket
        handle_socket(socket)
      elsif ins[0] == $stdin
        handle_stdin(socket)
      end
    end
  end
end

def send_envvars(socket)
  ENV.each{|key,value|
    if $options.client[:env_all] || $options.client[:env_include_mask].any?{|item| key.include?(item) }
      if !$options.client[:env_exclude_mask].any?{|item| key.include?(item) }
        socket.puts "Env: #{key}=#{value}"
      end
    end
  }
end

def send_command(socket, args)
  socket.puts "Cwd: #{current_dir}"
  args.each do |arg|
    # why using gsub? because Base64.encode64 happens to break lines automatically.
    # TODO using default encoding.
    socket.puts "Arg: #{Base64.encode64(arg).gsub(/\s/, '')}"
  end
  File.open(COOKIE_FILE_BASE + "-" + $options.client[:port].to_s) { |f|
    socket.puts "Cookie: #{f.read}"
  }
  send_envvars(socket)
  if ENV['CLASSPATH']
    socket.puts "Cp: #{ENV['CLASSPATH']}"
  end
  socket.puts ""
end

def current_dir()
  if IS_WINDOWS
    # native path expression including a drive letter is needed.
    `cmd.exe /c cd`.strip # FIXME it's ugly...
  else
    Dir::pwd
  end
end

def handle_socket(socket)
  headers = read_headers(socket)
  if headers.empty?
    exit 1
  end
  if headers['Status']
    if $options.server[:help]
      puts "\n"
      usage()
    end
    exit headers['Status'].to_i
  end
  data = socket.read(headers['Size'].to_i)
  return unless data

  if headers['Channel'] == 'out'
    $stdout.print data
  elsif headers['Channel'] == 'err'
    $stderr.print data
  end
end

$closedStdin = false # FIXME
def handle_stdin(socket)
  return if $closedStdin
  begin
    data = $stdin.read_nonblock(512)
  rescue EOFError
    socket.write "Size: 0\n\n"
    $closedStdin = true
  else
    socket.write "Size: #{data.length}\n\n"
    socket.write data
  end
end

def send_interrupt(socket)
    socket.write "Size: -1\n\n"
    socket.close()
end

def read_headers(socket)
  headers = {}
  while (line = socket.gets) != nil && line != "\n" do
    line.chomp!
    /([a-zA-Z]+): (.+)/ =~ line
    headers[$1] = $2
  end
  headers
end

def parse_option(args)
  options = Options.new
  args.each_with_index do |arg, i|
    case arg
    when "-Cwithout-invoking-server"
      options.client[:without_invoking_server] = true
    when "-Cp", "-Cport"
      port = args.delete_at(i + 1)
      unless port =~ /^[0-9]+$/
        raise "Invalid port number #{port} for #{arg}"
      end
      options.client[:port] = port
    when "-Ck" , "-Ckill-server"
      options.client[:groovyserver_opt] << "-k"
    when "-Cr", "-Crestart-server"
      options.client[:groovyserver_opt] << "-r"
    when "-Cq", "-Cquiet"
      options.client[:groovyserver_opt] << "-q"
      options.client[:quiet] = true
    when "-Cenv-all"
      options.client[:env_all] = true
    when "-Cenv"
      val = args.delete_at(i + 1)
      unless val
        raise "Invalid mask string #{val} for #{arg}"
      end
      options.client[:env_include_mask] << val
    when "-Cenv-exclude"
      val = args.delete_at(i + 1)
      unless val
        raise "Invalid mask string #{val} for #{arg}"
      end
      options.client[:env_exclude_mask] << val
    when "-Ch", "-Chelp"
      options.client[:help] = true
    when "--help", "-help", "-h"
      options.server[:help] = true
      options.server[:args] << arg
    when /-C.*/
      raise "Unknown option #{arg}"
    else
      options.server[:args] << arg
    end
  end
  if options.server[:args].empty?
    # display additionally client's usage at the end of session
    options.server[:help] = true
  end
  options
end

#-------------------------------------------
# Main
#-------------------------------------------

# Parsing options
begin
  $options = parse_option(ARGV)
rescue => e
  STDERR.puts "ERROR: #{e.message}"
  usage()
  exit 1
end
#puts "Original ARGV: #{ARGV.inspect}"
#puts "Parsed options: #{$options.inspect}"

# Only show usage (highest priority)
if $options.client[:help]
  usage()
  exit 0
end

# Start or stop server when specified
unless $options.client[:groovyserver_opt].empty?
  start_server($options.client[:groovyserver_opt].uniq)
  if $options.client[:groovyserver_opt].include?("-k")
    exit 0
  end
end

# Invoke script (before, start server if down)
failCount = 0
begin
  TCPSocket.open(DESTHOST, $options.client[:port]) { |socket|
    Signal.trap(:INT) {
      send_interrupt(socket)
      exit 8
    }
    session(socket, $options.server[:args])
  }
rescue Errno::ECONNREFUSED
  if $options.client[:without_invoking_server]
    STDERR.puts "ERROR: groovyserver isn't running"
    exit 9
  end
  if failCount >= 3
    STDERR.puts "ERROR: Failed to start up groovyserver: #{GROOVYSERVER_CMD}"
    exit 1
  end
  start_server([])
  sleep 3
  failCount += 1
  retry
rescue Errno::ECONNRESET, Errno::EPIPE
  # normally exit if reset by peer or broken pipe
  exit 0
rescue => e
  STDERR.puts "ERROR: #{e.message}"
  STDERR.puts e.backtrace
  exit 1
end

