'use strict';

const {
  ObjectDefineProperty,
  globalThis,
} = primordials;

const {
  defineOperation,
  exposeInterface,
  lazyDOMExceptionClass,
  defineLazyProperties,
  defineReplaceableLazyAttribute,
  exposeLazyInterfaces,
} = require('internal/util');
const config = internalBinding('config');

// https://console.spec.whatwg.org/#console-namespace
exposeNamespace(globalThis, 'console',
                createGlobalConsole());

const { URL, URLSearchParams } = require('internal/url');
// https://url.spec.whatwg.org/#url
exposeInterface(globalThis, 'URL', URL);
// https://url.spec.whatwg.org/#urlsearchparams
exposeInterface(globalThis, 'URLSearchParams', URLSearchParams);
exposeGetterAndSetter(globalThis,
                      'DOMException',
                      lazyDOMExceptionClass,
                      (value) => {
                        exposeInterface(globalThis, 'DOMException', value);
                      });

// https://html.spec.whatwg.org/multipage/webappapis.html#windoworworkerglobalscope
const timers = require('timers');
defineOperation(globalThis, 'clearInterval', timers.clearInterval);
defineOperation(globalThis, 'clearTimeout', timers.clearTimeout);
defineOperation(globalThis, 'setInterval', timers.setInterval);
defineOperation(globalThis, 'setTimeout', timers.setTimeout);

// Lazy ones.
exposeLazyInterfaces(globalThis, 'internal/abort_controller', [
  'AbortController', 'AbortSignal',
]);
exposeLazyInterfaces(globalThis, 'internal/event_target', [
  'EventTarget', 'Event',
]);
exposeLazyInterfaces(globalThis, 'internal/worker/io', [
  'MessageChannel', 'MessagePort', 'MessageEvent',
]);
defineLazyProperties(globalThis, 'buffer', ['atob', 'btoa']);
// https://www.w3.org/TR/FileAPI/#dfn-Blob
exposeLazyInterfaces(globalThis, 'internal/blob', ['Blob']);
// https://www.w3.org/TR/hr-time-2/#the-performance-attribute
exposeLazyInterfaces(globalThis, 'perf_hooks', [
  'Performance', 'PerformanceEntry', 'PerformanceMark', 'PerformanceMeasure',
  'PerformanceObserver', 'PerformanceObserverEntryList', 'PerformanceResourceTiming',
]);

defineReplaceableLazyAttribute(globalThis, 'perf_hooks', ['performance']);

// https://encoding.spec.whatwg.org/#textencoder
// https://encoding.spec.whatwg.org/#textdecoder
exposeLazyInterfaces(globalThis,
                     'internal/encoding',
                     ['TextEncoder', 'TextDecoder']);

function createGlobalConsole() {
  const consoleFromNode =
    require('internal/console/global');
  if (config.hasInspector) {
    const inspector = require('internal/util/inspector');
    // TODO(joyeecheung): postpone this until the first time inspector
    // is activated.
    inspector.wrapConsole(consoleFromNode);
    const { setConsoleExtensionInstaller } = internalBinding('inspector');
    // Setup inspector command line API.
    setConsoleExtensionInstaller(inspector.installConsoleExtensions);
  }
  return consoleFromNode;
}

// https://heycam.github.io/webidl/#es-namespaces
function exposeNamespace(target, name, namespaceObject) {
  ObjectDefineProperty(target, name, {
    __proto__: null,
    writable: true,
    enumerable: false,
    configurable: true,
    value: namespaceObject
  });
}

function exposeGetterAndSetter(target, name, getter, setter = undefined) {
  ObjectDefineProperty(target, name, {
    __proto__: null,
    enumerable: false,
    configurable: true,
    get: getter,
    set: setter,
  });
}

// Web Streams API
exposeLazyInterfaces(
  globalThis,
  'internal/webstreams/transformstream',
  ['TransformStream', 'TransformStreamDefaultController']);

exposeLazyInterfaces(
  globalThis,
  'internal/webstreams/writablestream',
  ['WritableStream', 'WritableStreamDefaultController', 'WritableStreamDefaultWriter']);

exposeLazyInterfaces(
  globalThis,
  'internal/webstreams/readablestream',
  [
    'ReadableStream', 'ReadableStreamDefaultReader',
    'ReadableStreamBYOBReader', 'ReadableStreamBYOBRequest',
    'ReadableByteStreamController', 'ReadableStreamDefaultController',
  ]);

exposeLazyInterfaces(
  globalThis,
  'internal/webstreams/queuingstrategies',
  [
    'ByteLengthQueuingStrategy', 'CountQueuingStrategy',
  ]);

exposeLazyInterfaces(
  globalThis,
  'internal/webstreams/encoding',
  [
    'TextEncoderStream', 'TextDecoderStream',
  ]);

exposeLazyInterfaces(
  globalThis,
  'internal/webstreams/compression',
  [
    'CompressionStream', 'DecompressionStream',
  ]);
