#!/usr/bin/env python

# Capstone Python bindings, by Sebastian Macke <Sebastian Macke>
from __future__ import print_function
from capstone import *
from capstone.mos65xx import *
from xprint import to_hex, to_x

M6502_CODE = b"\xa1\x12\xa5\x12\xa9\x12\xad\x34\x12\xb1\x12\xb5\x12\xb9\x34\x12\xbd\x34\x12\x0d\x34\x12\x00\x81\x87\x6c\x01\x00\x85\xFF\x10\x00\x19\x42\x42\x00\x49\x42"
M65C02_CODE = b"\x1a\x3a\x02\x12\x03\x5c\x34\x12"
MW65C02_CODE = b"\x07\x12\x27\x12\x47\x12\x67\x12\x87\x12\xa7\x12\xc7\x12\xe7\x12\x10\xfe\x0f\x12\xfd\x4f\x12\xfd\x8f\x12\xfd\xcf\x12\xfd"
M65816_CODE = b"\xa9\x34\x12\xad\x34\x12\xbd\x34\x12\xb9\x34\x12\xaf\x56\x34\x12\xbf\x56\x34\x12\xa5\x12\xb5\x12\xb2\x12\xa1\x12\xb1\x12\xa7\x12\xb7\x12\xa3\x12\xb3\x12\xc2\x00\xe2\x00\x54\x34\x12\x44\x34\x12\x02\x12"

all_tests = (
        (CS_ARCH_MOS65XX, CS_MODE_MOS65XX_6502, M6502_CODE, "MOS65XX_6502"),
        (CS_ARCH_MOS65XX, CS_MODE_MOS65XX_65C02, M65C02_CODE, "MOS65XX_65C02"),
        (CS_ARCH_MOS65XX, CS_MODE_MOS65XX_W65C02, MW65C02_CODE, "MOS65XX_W65C02"),
        (CS_ARCH_MOS65XX, CS_MODE_MOS65XX_65816_LONG_MX, M65816_CODE, "MOS65XX_65816 (long m/x)"),
)

address_modes=[
    "No address mode",
    "implied",
    "accumulator",
    "immediate value",
    "relative",
    "interrupt signature",
    "block move",
    "zero page",
    "zero page indexed with x",
    "zero page indexed with y",
    "relative bit branch",
    "zero page indirect",
    "zero page indexed with x indirect",
    "zero page indirect indexed with y",
    "zero page indirect long",
    "zero page indirect long indexed with y",
    "absolute",
    "absolute indexed with x",
    "absolute indexed with y",
    "absolute indirect",
    "absolute indexed with x indirect",
    "absolute indirect long",
    "absolute long",
    "absolute long indexed with x",
    "stack relative",
    "stack relative indirect indexed with y",
]


def print_insn_detail(insn):
    # print address, mnemonic and operands
    print("0x%x:\t%s\t%s" % (insn.address, insn.mnemonic, insn.op_str))

    # "data" instruction generated by SKIPDATA option has no detail
    if insn.id == 0:
        return
    print("\taddress mode: %s" % (address_modes[insn.am]))
    print("\tmodifies flags: %s" % ('true' if insn.modifies_flags != 0 else 'false'))
    if len(insn.operands) > 0:
        print("\top_count: %u" % len(insn.operands))
        c = -1
        for i in insn.operands:
            c += 1
            if i.type == MOS65XX_OP_REG:
                print("\t\toperands[%u].type: REG = %s" % (c, insn.reg_name(i.reg)))
            if i.type == MOS65XX_OP_IMM:
                print("\t\toperands[%u].type: IMM = 0x%s" % (c, to_x(i.imm)))
            if i.type == MOS65XX_OP_MEM:
                print("\t\toperands[%u].type: MEM = 0x%s" % (c, to_x(i.mem)))


# ## Test class Cs
def test_class():
    for (arch, mode, code, comment) in all_tests:
        print("*" * 16)
        print("Platform: %s" % comment)
        print("Code: %s" % to_hex(code))
        print("Disasm:")

        try:
            md = Cs(arch, mode)
            md.detail = True
            md.syntax = CS_OPT_SYNTAX_MOTOROLA
            for insn in md.disasm(code, 0x1000):
                print_insn_detail(insn)
                print()

            print("0x%x:\n" % (insn.address + insn.size))
        except CsError as e:
            print("ERROR: %s" % e)


if __name__ == '__main__':
    test_class()
