/*
 *  This file is part of the WebKit open source project.
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this library; see the file COPYING.LIB.  If not, write to
 *  the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 */

#include "config.h"
#include "WebKitDOMElement.h"

#include <WebCore/CSSImportRule.h>
#include "DOMObjectCache.h"
#include <WebCore/Document.h>
#include <WebCore/ExceptionCode.h>
#include <WebCore/ExceptionCodeDescription.h>
#include "GObjectEventListener.h"
#include <WebCore/HTMLNames.h>
#include <WebCore/JSMainThreadExecState.h>
#include "WebKitDOMAttrPrivate.h"
#include "WebKitDOMCSSStyleDeclarationPrivate.h"
#include "WebKitDOMDOMTokenListPrivate.h"
#include "WebKitDOMElementPrivate.h"
#include "WebKitDOMEventPrivate.h"
#include "WebKitDOMEventTarget.h"
#include "WebKitDOMHTMLCollectionPrivate.h"
#include "WebKitDOMNamedNodeMapPrivate.h"
#include "WebKitDOMNodeListPrivate.h"
#include "WebKitDOMNodePrivate.h"
#include "WebKitDOMPrivate.h"
#include "ConvertToUTF8String.h"
#include "WebKitDOMElementUnstable.h"
#include <wtf/GetPtr.h>
#include <wtf/RefPtr.h>

namespace WebKit {

WebKitDOMElement* kit(WebCore::Element* obj)
{
    return WEBKIT_DOM_ELEMENT(kit(static_cast<WebCore::Node*>(obj)));
}

WebCore::Element* core(WebKitDOMElement* request)
{
    return request ? static_cast<WebCore::Element*>(WEBKIT_DOM_OBJECT(request)->coreObject) : 0;
}

WebKitDOMElement* wrapElement(WebCore::Element* coreObject)
{
    ASSERT(coreObject);
    return WEBKIT_DOM_ELEMENT(g_object_new(WEBKIT_DOM_TYPE_ELEMENT, "core-object", coreObject, nullptr));
}

} // namespace WebKit

static gboolean webkit_dom_element_dispatch_event(WebKitDOMEventTarget* target, WebKitDOMEvent* event, GError** error)
{
    WebCore::Event* coreEvent = WebKit::core(event);
    if (!coreEvent)
        return false;
    WebCore::Element* coreTarget = static_cast<WebCore::Element*>(WEBKIT_DOM_OBJECT(target)->coreObject);

    WebCore::ExceptionCode ec = 0;
    gboolean result = coreTarget->dispatchEventForBindings(*coreEvent, ec);
    if (ec) {
        WebCore::ExceptionCodeDescription description(ec);
        g_set_error_literal(error, g_quark_from_string("WEBKIT_DOM"), description.code, description.name);
    }
    return result;
}

static gboolean webkit_dom_element_add_event_listener(WebKitDOMEventTarget* target, const char* eventName, GClosure* handler, gboolean useCapture)
{
    WebCore::Element* coreTarget = static_cast<WebCore::Element*>(WEBKIT_DOM_OBJECT(target)->coreObject);
    return WebKit::GObjectEventListener::addEventListener(G_OBJECT(target), coreTarget, eventName, handler, useCapture);
}

static gboolean webkit_dom_element_remove_event_listener(WebKitDOMEventTarget* target, const char* eventName, GClosure* handler, gboolean useCapture)
{
    WebCore::Element* coreTarget = static_cast<WebCore::Element*>(WEBKIT_DOM_OBJECT(target)->coreObject);
    return WebKit::GObjectEventListener::removeEventListener(G_OBJECT(target), coreTarget, eventName, handler, useCapture);
}

static void webkit_dom_event_target_init(WebKitDOMEventTargetIface* iface)
{
    iface->dispatch_event = webkit_dom_element_dispatch_event;
    iface->add_event_listener = webkit_dom_element_add_event_listener;
    iface->remove_event_listener = webkit_dom_element_remove_event_listener;
}

G_DEFINE_TYPE_WITH_CODE(WebKitDOMElement, webkit_dom_element, WEBKIT_DOM_TYPE_NODE, G_IMPLEMENT_INTERFACE(WEBKIT_DOM_TYPE_EVENT_TARGET, webkit_dom_event_target_init))

enum {
    PROP_0,
    PROP_TAG_NAME,
    PROP_ATTRIBUTES,
    PROP_STYLE,
    PROP_ID,
    PROP_NAMESPACE_URI,
    PROP_PREFIX,
    PROP_LOCAL_NAME,
    PROP_OFFSET_LEFT,
    PROP_OFFSET_TOP,
    PROP_OFFSET_WIDTH,
    PROP_OFFSET_HEIGHT,
    PROP_CLIENT_LEFT,
    PROP_CLIENT_TOP,
    PROP_CLIENT_WIDTH,
    PROP_CLIENT_HEIGHT,
    PROP_SCROLL_LEFT,
    PROP_SCROLL_TOP,
    PROP_SCROLL_WIDTH,
    PROP_SCROLL_HEIGHT,
    PROP_OFFSET_PARENT,
    PROP_INNER_HTML,
    PROP_OUTER_HTML,
    PROP_CLASS_NAME,
    PROP_CLASS_LIST,
    PROP_WEBKIT_REGION_OVERSET,
    PROP_PREVIOUS_ELEMENT_SIBLING,
    PROP_NEXT_ELEMENT_SIBLING,
    PROP_CHILDREN,
    PROP_FIRST_ELEMENT_CHILD,
    PROP_LAST_ELEMENT_CHILD,
    PROP_CHILD_ELEMENT_COUNT,
};

static void webkit_dom_element_set_property(GObject* object, guint propertyId, const GValue* value, GParamSpec* pspec)
{
    WebKitDOMElement* self = WEBKIT_DOM_ELEMENT(object);

    switch (propertyId) {
    case PROP_ID:
        webkit_dom_element_set_id(self, g_value_get_string(value));
        break;
    case PROP_SCROLL_LEFT:
        webkit_dom_element_set_scroll_left(self, g_value_get_long(value));
        break;
    case PROP_SCROLL_TOP:
        webkit_dom_element_set_scroll_top(self, g_value_get_long(value));
        break;
    case PROP_INNER_HTML:
        webkit_dom_element_set_inner_html(self, g_value_get_string(value), nullptr);
        break;
    case PROP_OUTER_HTML:
        webkit_dom_element_set_outer_html(self, g_value_get_string(value), nullptr);
        break;
    case PROP_CLASS_NAME:
        webkit_dom_element_set_class_name(self, g_value_get_string(value));
        break;
    default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, propertyId, pspec);
        break;
    }
}

static void webkit_dom_element_get_property(GObject* object, guint propertyId, GValue* value, GParamSpec* pspec)
{
    WebKitDOMElement* self = WEBKIT_DOM_ELEMENT(object);

    switch (propertyId) {
    case PROP_TAG_NAME:
        g_value_take_string(value, webkit_dom_element_get_tag_name(self));
        break;
    case PROP_ATTRIBUTES:
        g_value_set_object(value, webkit_dom_element_get_attributes(self));
        break;
    case PROP_STYLE:
        g_value_set_object(value, webkit_dom_element_get_style(self));
        break;
    case PROP_ID:
        g_value_take_string(value, webkit_dom_element_get_id(self));
        break;
    case PROP_NAMESPACE_URI:
        g_value_take_string(value, webkit_dom_element_get_namespace_uri(self));
        break;
    case PROP_PREFIX:
        g_value_take_string(value, webkit_dom_element_get_prefix(self));
        break;
    case PROP_LOCAL_NAME:
        g_value_take_string(value, webkit_dom_element_get_local_name(self));
        break;
    case PROP_OFFSET_LEFT:
        g_value_set_double(value, webkit_dom_element_get_offset_left(self));
        break;
    case PROP_OFFSET_TOP:
        g_value_set_double(value, webkit_dom_element_get_offset_top(self));
        break;
    case PROP_OFFSET_WIDTH:
        g_value_set_double(value, webkit_dom_element_get_offset_width(self));
        break;
    case PROP_OFFSET_HEIGHT:
        g_value_set_double(value, webkit_dom_element_get_offset_height(self));
        break;
    case PROP_CLIENT_LEFT:
        g_value_set_double(value, webkit_dom_element_get_client_left(self));
        break;
    case PROP_CLIENT_TOP:
        g_value_set_double(value, webkit_dom_element_get_client_top(self));
        break;
    case PROP_CLIENT_WIDTH:
        g_value_set_double(value, webkit_dom_element_get_client_width(self));
        break;
    case PROP_CLIENT_HEIGHT:
        g_value_set_double(value, webkit_dom_element_get_client_height(self));
        break;
    case PROP_SCROLL_LEFT:
        g_value_set_long(value, webkit_dom_element_get_scroll_left(self));
        break;
    case PROP_SCROLL_TOP:
        g_value_set_long(value, webkit_dom_element_get_scroll_top(self));
        break;
    case PROP_SCROLL_WIDTH:
        g_value_set_long(value, webkit_dom_element_get_scroll_width(self));
        break;
    case PROP_SCROLL_HEIGHT:
        g_value_set_long(value, webkit_dom_element_get_scroll_height(self));
        break;
    case PROP_OFFSET_PARENT:
        g_value_set_object(value, webkit_dom_element_get_offset_parent(self));
        break;
    case PROP_INNER_HTML:
        g_value_take_string(value, webkit_dom_element_get_inner_html(self));
        break;
    case PROP_OUTER_HTML:
        g_value_take_string(value, webkit_dom_element_get_outer_html(self));
        break;
    case PROP_CLASS_NAME:
        g_value_take_string(value, webkit_dom_element_get_class_name(self));
        break;
    case PROP_CLASS_LIST:
        g_value_set_object(value, webkit_dom_element_get_class_list(self));
        break;
    case PROP_WEBKIT_REGION_OVERSET:
        g_value_take_string(value, webkit_dom_element_get_webkit_region_overset(self));
        break;
    case PROP_PREVIOUS_ELEMENT_SIBLING:
        g_value_set_object(value, webkit_dom_element_get_previous_element_sibling(self));
        break;
    case PROP_NEXT_ELEMENT_SIBLING:
        g_value_set_object(value, webkit_dom_element_get_next_element_sibling(self));
        break;
    case PROP_CHILDREN:
        g_value_set_object(value, webkit_dom_element_get_children(self));
        break;
    case PROP_FIRST_ELEMENT_CHILD:
        g_value_set_object(value, webkit_dom_element_get_first_element_child(self));
        break;
    case PROP_LAST_ELEMENT_CHILD:
        g_value_set_object(value, webkit_dom_element_get_last_element_child(self));
        break;
    case PROP_CHILD_ELEMENT_COUNT:
        g_value_set_ulong(value, webkit_dom_element_get_child_element_count(self));
        break;
    default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, propertyId, pspec);
        break;
    }
}

static void webkit_dom_element_class_init(WebKitDOMElementClass* requestClass)
{
    GObjectClass* gobjectClass = G_OBJECT_CLASS(requestClass);
    gobjectClass->set_property = webkit_dom_element_set_property;
    gobjectClass->get_property = webkit_dom_element_get_property;

    g_object_class_install_property(
        gobjectClass,
        PROP_TAG_NAME,
        g_param_spec_string(
            "tag-name",
            "Element:tag-name",
            "read-only gchar* Element:tag-name",
            "",
            WEBKIT_PARAM_READABLE));

    g_object_class_install_property(
        gobjectClass,
        PROP_ATTRIBUTES,
        g_param_spec_object(
            "attributes",
            "Element:attributes",
            "read-only WebKitDOMNamedNodeMap* Element:attributes",
            WEBKIT_DOM_TYPE_NAMED_NODE_MAP,
            WEBKIT_PARAM_READABLE));

    g_object_class_install_property(
        gobjectClass,
        PROP_STYLE,
        g_param_spec_object(
            "style",
            "Element:style",
            "read-only WebKitDOMCSSStyleDeclaration* Element:style",
            WEBKIT_DOM_TYPE_CSS_STYLE_DECLARATION,
            WEBKIT_PARAM_READABLE));

    g_object_class_install_property(
        gobjectClass,
        PROP_ID,
        g_param_spec_string(
            "id",
            "Element:id",
            "read-write gchar* Element:id",
            "",
            WEBKIT_PARAM_READWRITE));

    g_object_class_install_property(
        gobjectClass,
        PROP_NAMESPACE_URI,
        g_param_spec_string(
            "namespace-uri",
            "Element:namespace-uri",
            "read-only gchar* Element:namespace-uri",
            "",
            WEBKIT_PARAM_READABLE));

    g_object_class_install_property(
        gobjectClass,
        PROP_PREFIX,
        g_param_spec_string(
            "prefix",
            "Element:prefix",
            "read-only gchar* Element:prefix",
            "",
            WEBKIT_PARAM_READABLE));

    g_object_class_install_property(
        gobjectClass,
        PROP_LOCAL_NAME,
        g_param_spec_string(
            "local-name",
            "Element:local-name",
            "read-only gchar* Element:local-name",
            "",
            WEBKIT_PARAM_READABLE));

    g_object_class_install_property(
        gobjectClass,
        PROP_OFFSET_LEFT,
        g_param_spec_double(
            "offset-left",
            "Element:offset-left",
            "read-only gdouble Element:offset-left",
            -G_MAXDOUBLE, G_MAXDOUBLE, 0,
            WEBKIT_PARAM_READABLE));

    g_object_class_install_property(
        gobjectClass,
        PROP_OFFSET_TOP,
        g_param_spec_double(
            "offset-top",
            "Element:offset-top",
            "read-only gdouble Element:offset-top",
            -G_MAXDOUBLE, G_MAXDOUBLE, 0,
            WEBKIT_PARAM_READABLE));

    g_object_class_install_property(
        gobjectClass,
        PROP_OFFSET_WIDTH,
        g_param_spec_double(
            "offset-width",
            "Element:offset-width",
            "read-only gdouble Element:offset-width",
            -G_MAXDOUBLE, G_MAXDOUBLE, 0,
            WEBKIT_PARAM_READABLE));

    g_object_class_install_property(
        gobjectClass,
        PROP_OFFSET_HEIGHT,
        g_param_spec_double(
            "offset-height",
            "Element:offset-height",
            "read-only gdouble Element:offset-height",
            -G_MAXDOUBLE, G_MAXDOUBLE, 0,
            WEBKIT_PARAM_READABLE));

    g_object_class_install_property(
        gobjectClass,
        PROP_CLIENT_LEFT,
        g_param_spec_double(
            "client-left",
            "Element:client-left",
            "read-only gdouble Element:client-left",
            -G_MAXDOUBLE, G_MAXDOUBLE, 0,
            WEBKIT_PARAM_READABLE));

    g_object_class_install_property(
        gobjectClass,
        PROP_CLIENT_TOP,
        g_param_spec_double(
            "client-top",
            "Element:client-top",
            "read-only gdouble Element:client-top",
            -G_MAXDOUBLE, G_MAXDOUBLE, 0,
            WEBKIT_PARAM_READABLE));

    g_object_class_install_property(
        gobjectClass,
        PROP_CLIENT_WIDTH,
        g_param_spec_double(
            "client-width",
            "Element:client-width",
            "read-only gdouble Element:client-width",
            -G_MAXDOUBLE, G_MAXDOUBLE, 0,
            WEBKIT_PARAM_READABLE));

    g_object_class_install_property(
        gobjectClass,
        PROP_CLIENT_HEIGHT,
        g_param_spec_double(
            "client-height",
            "Element:client-height",
            "read-only gdouble Element:client-height",
            -G_MAXDOUBLE, G_MAXDOUBLE, 0,
            WEBKIT_PARAM_READABLE));

    g_object_class_install_property(
        gobjectClass,
        PROP_SCROLL_LEFT,
        g_param_spec_long(
            "scroll-left",
            "Element:scroll-left",
            "read-write glong Element:scroll-left",
            G_MINLONG, G_MAXLONG, 0,
            WEBKIT_PARAM_READWRITE));

    g_object_class_install_property(
        gobjectClass,
        PROP_SCROLL_TOP,
        g_param_spec_long(
            "scroll-top",
            "Element:scroll-top",
            "read-write glong Element:scroll-top",
            G_MINLONG, G_MAXLONG, 0,
            WEBKIT_PARAM_READWRITE));

    g_object_class_install_property(
        gobjectClass,
        PROP_SCROLL_WIDTH,
        g_param_spec_long(
            "scroll-width",
            "Element:scroll-width",
            "read-only glong Element:scroll-width",
            G_MINLONG, G_MAXLONG, 0,
            WEBKIT_PARAM_READABLE));

    g_object_class_install_property(
        gobjectClass,
        PROP_SCROLL_HEIGHT,
        g_param_spec_long(
            "scroll-height",
            "Element:scroll-height",
            "read-only glong Element:scroll-height",
            G_MINLONG, G_MAXLONG, 0,
            WEBKIT_PARAM_READABLE));

    g_object_class_install_property(
        gobjectClass,
        PROP_OFFSET_PARENT,
        g_param_spec_object(
            "offset-parent",
            "Element:offset-parent",
            "read-only WebKitDOMElement* Element:offset-parent",
            WEBKIT_DOM_TYPE_ELEMENT,
            WEBKIT_PARAM_READABLE));

    g_object_class_install_property(
        gobjectClass,
        PROP_INNER_HTML,
        g_param_spec_string(
            "inner-html",
            "Element:inner-html",
            "read-write gchar* Element:inner-html",
            "",
            WEBKIT_PARAM_READWRITE));

    g_object_class_install_property(
        gobjectClass,
        PROP_OUTER_HTML,
        g_param_spec_string(
            "outer-html",
            "Element:outer-html",
            "read-write gchar* Element:outer-html",
            "",
            WEBKIT_PARAM_READWRITE));

    g_object_class_install_property(
        gobjectClass,
        PROP_CLASS_NAME,
        g_param_spec_string(
            "class-name",
            "Element:class-name",
            "read-write gchar* Element:class-name",
            "",
            WEBKIT_PARAM_READWRITE));

    g_object_class_install_property(
        gobjectClass,
        PROP_CLASS_LIST,
        g_param_spec_object(
            "class-list",
            "Element:class-list",
            "read-only WebKitDOMDOMTokenList* Element:class-list",
            WEBKIT_DOM_TYPE_DOM_TOKEN_LIST,
            WEBKIT_PARAM_READABLE));

    g_object_class_install_property(
        gobjectClass,
        PROP_WEBKIT_REGION_OVERSET,
        g_param_spec_string(
            "webkit-region-overset",
            "Element:webkit-region-overset",
            "read-only gchar* Element:webkit-region-overset",
            "",
            WEBKIT_PARAM_READABLE));

    g_object_class_install_property(
        gobjectClass,
        PROP_PREVIOUS_ELEMENT_SIBLING,
        g_param_spec_object(
            "previous-element-sibling",
            "Element:previous-element-sibling",
            "read-only WebKitDOMElement* Element:previous-element-sibling",
            WEBKIT_DOM_TYPE_ELEMENT,
            WEBKIT_PARAM_READABLE));

    g_object_class_install_property(
        gobjectClass,
        PROP_NEXT_ELEMENT_SIBLING,
        g_param_spec_object(
            "next-element-sibling",
            "Element:next-element-sibling",
            "read-only WebKitDOMElement* Element:next-element-sibling",
            WEBKIT_DOM_TYPE_ELEMENT,
            WEBKIT_PARAM_READABLE));

    g_object_class_install_property(
        gobjectClass,
        PROP_CHILDREN,
        g_param_spec_object(
            "children",
            "Element:children",
            "read-only WebKitDOMHTMLCollection* Element:children",
            WEBKIT_DOM_TYPE_HTML_COLLECTION,
            WEBKIT_PARAM_READABLE));

    g_object_class_install_property(
        gobjectClass,
        PROP_FIRST_ELEMENT_CHILD,
        g_param_spec_object(
            "first-element-child",
            "Element:first-element-child",
            "read-only WebKitDOMElement* Element:first-element-child",
            WEBKIT_DOM_TYPE_ELEMENT,
            WEBKIT_PARAM_READABLE));

    g_object_class_install_property(
        gobjectClass,
        PROP_LAST_ELEMENT_CHILD,
        g_param_spec_object(
            "last-element-child",
            "Element:last-element-child",
            "read-only WebKitDOMElement* Element:last-element-child",
            WEBKIT_DOM_TYPE_ELEMENT,
            WEBKIT_PARAM_READABLE));

    g_object_class_install_property(
        gobjectClass,
        PROP_CHILD_ELEMENT_COUNT,
        g_param_spec_ulong(
            "child-element-count",
            "Element:child-element-count",
            "read-only gulong Element:child-element-count",
            0, G_MAXULONG, 0,
            WEBKIT_PARAM_READABLE));

}

static void webkit_dom_element_init(WebKitDOMElement* request)
{
    UNUSED_PARAM(request);
}

gchar* webkit_dom_element_get_attribute(WebKitDOMElement* self, const gchar* name)
{
    WebCore::JSMainThreadNullState state;
    g_return_val_if_fail(WEBKIT_DOM_IS_ELEMENT(self), 0);
    g_return_val_if_fail(name, 0);
    WebCore::Element* item = WebKit::core(self);
    WTF::String convertedName = WTF::String::fromUTF8(name);
    gchar* result = convertToUTF8String(item->getAttribute(convertedName));
    return result;
}

void webkit_dom_element_set_attribute(WebKitDOMElement* self, const gchar* name, const gchar* value, GError** error)
{
    WebCore::JSMainThreadNullState state;
    g_return_if_fail(WEBKIT_DOM_IS_ELEMENT(self));
    g_return_if_fail(name);
    g_return_if_fail(value);
    g_return_if_fail(!error || !*error);
    WebCore::Element* item = WebKit::core(self);
    WTF::String convertedName = WTF::String::fromUTF8(name);
    WTF::String convertedValue = WTF::String::fromUTF8(value);
    WebCore::ExceptionCode ec = 0;
    item->setAttribute(convertedName, convertedValue, ec);
    if (ec) {
        WebCore::ExceptionCodeDescription ecdesc(ec);
        g_set_error_literal(error, g_quark_from_string("WEBKIT_DOM"), ecdesc.code, ecdesc.name);
    }
}

void webkit_dom_element_remove_attribute(WebKitDOMElement* self, const gchar* name)
{
    WebCore::JSMainThreadNullState state;
    g_return_if_fail(WEBKIT_DOM_IS_ELEMENT(self));
    g_return_if_fail(name);
    WebCore::Element* item = WebKit::core(self);
    WTF::String convertedName = WTF::String::fromUTF8(name);
    item->removeAttribute(convertedName);
}

WebKitDOMAttr* webkit_dom_element_get_attribute_node(WebKitDOMElement* self, const gchar* name)
{
    WebCore::JSMainThreadNullState state;
    g_return_val_if_fail(WEBKIT_DOM_IS_ELEMENT(self), 0);
    g_return_val_if_fail(name, 0);
    WebCore::Element* item = WebKit::core(self);
    WTF::String convertedName = WTF::String::fromUTF8(name);
    RefPtr<WebCore::Attr> gobjectResult = WTF::getPtr(item->getAttributeNode(convertedName));
    return WebKit::kit(gobjectResult.get());
}

WebKitDOMAttr* webkit_dom_element_set_attribute_node(WebKitDOMElement* self, WebKitDOMAttr* newAttr, GError** error)
{
    WebCore::JSMainThreadNullState state;
    g_return_val_if_fail(WEBKIT_DOM_IS_ELEMENT(self), 0);
    g_return_val_if_fail(WEBKIT_DOM_IS_ATTR(newAttr), 0);
    g_return_val_if_fail(!error || !*error, 0);
    WebCore::Element* item = WebKit::core(self);
    WebCore::Attr* convertedNewAttr = WebKit::core(newAttr);
    WebCore::ExceptionCode ec = 0;
    RefPtr<WebCore::Attr> gobjectResult = WTF::getPtr(item->setAttributeNode(*convertedNewAttr, ec));
    if (ec) {
        WebCore::ExceptionCodeDescription ecdesc(ec);
        g_set_error_literal(error, g_quark_from_string("WEBKIT_DOM"), ecdesc.code, ecdesc.name);
    }
    return WebKit::kit(gobjectResult.get());
}

WebKitDOMAttr* webkit_dom_element_remove_attribute_node(WebKitDOMElement* self, WebKitDOMAttr* oldAttr, GError** error)
{
    WebCore::JSMainThreadNullState state;
    g_return_val_if_fail(WEBKIT_DOM_IS_ELEMENT(self), 0);
    g_return_val_if_fail(WEBKIT_DOM_IS_ATTR(oldAttr), 0);
    g_return_val_if_fail(!error || !*error, 0);
    WebCore::Element* item = WebKit::core(self);
    WebCore::Attr* convertedOldAttr = WebKit::core(oldAttr);
    WebCore::ExceptionCode ec = 0;
    RefPtr<WebCore::Attr> gobjectResult = WTF::getPtr(item->removeAttributeNode(*convertedOldAttr, ec));
    if (ec) {
        WebCore::ExceptionCodeDescription ecdesc(ec);
        g_set_error_literal(error, g_quark_from_string("WEBKIT_DOM"), ecdesc.code, ecdesc.name);
    }
    return WebKit::kit(gobjectResult.get());
}

WebKitDOMHTMLCollection* webkit_dom_element_get_elements_by_tag_name_as_html_collection(WebKitDOMElement* self, const gchar* name)
{
    WebCore::JSMainThreadNullState state;
    g_return_val_if_fail(WEBKIT_DOM_IS_ELEMENT(self), 0);
    g_return_val_if_fail(name, 0);
    WebCore::Element* item = WebKit::core(self);
    WTF::String convertedName = WTF::String::fromUTF8(name);
    RefPtr<WebCore::HTMLCollection> gobjectResult = WTF::getPtr(item->getElementsByTagName(convertedName));
    return WebKit::kit(gobjectResult.get());
}

gboolean webkit_dom_element_has_attributes(WebKitDOMElement* self)
{
    WebCore::JSMainThreadNullState state;
    g_return_val_if_fail(WEBKIT_DOM_IS_ELEMENT(self), FALSE);
    WebCore::Element* item = WebKit::core(self);
    gboolean result = item->hasAttributes();
    return result;
}

gchar* webkit_dom_element_get_attribute_ns(WebKitDOMElement* self, const gchar* namespaceURI, const gchar* localName)
{
    WebCore::JSMainThreadNullState state;
    g_return_val_if_fail(WEBKIT_DOM_IS_ELEMENT(self), 0);
    g_return_val_if_fail(namespaceURI, 0);
    g_return_val_if_fail(localName, 0);
    WebCore::Element* item = WebKit::core(self);
    WTF::String convertedNamespaceURI = WTF::String::fromUTF8(namespaceURI);
    WTF::String convertedLocalName = WTF::String::fromUTF8(localName);
    gchar* result = convertToUTF8String(item->getAttributeNS(convertedNamespaceURI, convertedLocalName));
    return result;
}

void webkit_dom_element_set_attribute_ns(WebKitDOMElement* self, const gchar* namespaceURI, const gchar* qualifiedName, const gchar* value, GError** error)
{
    WebCore::JSMainThreadNullState state;
    g_return_if_fail(WEBKIT_DOM_IS_ELEMENT(self));
    g_return_if_fail(qualifiedName);
    g_return_if_fail(value);
    g_return_if_fail(!error || !*error);
    WebCore::Element* item = WebKit::core(self);
    WTF::String convertedNamespaceURI = WTF::String::fromUTF8(namespaceURI);
    WTF::String convertedQualifiedName = WTF::String::fromUTF8(qualifiedName);
    WTF::String convertedValue = WTF::String::fromUTF8(value);
    WebCore::ExceptionCode ec = 0;
    item->setAttributeNS(convertedNamespaceURI, convertedQualifiedName, convertedValue, ec);
    if (ec) {
        WebCore::ExceptionCodeDescription ecdesc(ec);
        g_set_error_literal(error, g_quark_from_string("WEBKIT_DOM"), ecdesc.code, ecdesc.name);
    }
}

void webkit_dom_element_remove_attribute_ns(WebKitDOMElement* self, const gchar* namespaceURI, const gchar* localName)
{
    WebCore::JSMainThreadNullState state;
    g_return_if_fail(WEBKIT_DOM_IS_ELEMENT(self));
    g_return_if_fail(namespaceURI);
    g_return_if_fail(localName);
    WebCore::Element* item = WebKit::core(self);
    WTF::String convertedNamespaceURI = WTF::String::fromUTF8(namespaceURI);
    WTF::String convertedLocalName = WTF::String::fromUTF8(localName);
    item->removeAttributeNS(convertedNamespaceURI, convertedLocalName);
}

WebKitDOMHTMLCollection* webkit_dom_element_get_elements_by_tag_name_ns_as_html_collection(WebKitDOMElement* self, const gchar* namespaceURI, const gchar* localName)
{
    WebCore::JSMainThreadNullState state;
    g_return_val_if_fail(WEBKIT_DOM_IS_ELEMENT(self), 0);
    g_return_val_if_fail(namespaceURI, 0);
    g_return_val_if_fail(localName, 0);
    WebCore::Element* item = WebKit::core(self);
    WTF::String convertedNamespaceURI = WTF::String::fromUTF8(namespaceURI);
    WTF::String convertedLocalName = WTF::String::fromUTF8(localName);
    RefPtr<WebCore::HTMLCollection> gobjectResult = WTF::getPtr(item->getElementsByTagNameNS(convertedNamespaceURI, convertedLocalName));
    return WebKit::kit(gobjectResult.get());
}

WebKitDOMAttr* webkit_dom_element_get_attribute_node_ns(WebKitDOMElement* self, const gchar* namespaceURI, const gchar* localName)
{
    WebCore::JSMainThreadNullState state;
    g_return_val_if_fail(WEBKIT_DOM_IS_ELEMENT(self), 0);
    g_return_val_if_fail(namespaceURI, 0);
    g_return_val_if_fail(localName, 0);
    WebCore::Element* item = WebKit::core(self);
    WTF::String convertedNamespaceURI = WTF::String::fromUTF8(namespaceURI);
    WTF::String convertedLocalName = WTF::String::fromUTF8(localName);
    RefPtr<WebCore::Attr> gobjectResult = WTF::getPtr(item->getAttributeNodeNS(convertedNamespaceURI, convertedLocalName));
    return WebKit::kit(gobjectResult.get());
}

WebKitDOMAttr* webkit_dom_element_set_attribute_node_ns(WebKitDOMElement* self, WebKitDOMAttr* newAttr, GError** error)
{
    WebCore::JSMainThreadNullState state;
    g_return_val_if_fail(WEBKIT_DOM_IS_ELEMENT(self), 0);
    g_return_val_if_fail(WEBKIT_DOM_IS_ATTR(newAttr), 0);
    g_return_val_if_fail(!error || !*error, 0);
    WebCore::Element* item = WebKit::core(self);
    WebCore::Attr* convertedNewAttr = WebKit::core(newAttr);
    WebCore::ExceptionCode ec = 0;
    RefPtr<WebCore::Attr> gobjectResult = WTF::getPtr(item->setAttributeNodeNS(*convertedNewAttr, ec));
    if (ec) {
        WebCore::ExceptionCodeDescription ecdesc(ec);
        g_set_error_literal(error, g_quark_from_string("WEBKIT_DOM"), ecdesc.code, ecdesc.name);
    }
    return WebKit::kit(gobjectResult.get());
}

gboolean webkit_dom_element_has_attribute(WebKitDOMElement* self, const gchar* name)
{
    WebCore::JSMainThreadNullState state;
    g_return_val_if_fail(WEBKIT_DOM_IS_ELEMENT(self), FALSE);
    g_return_val_if_fail(name, FALSE);
    WebCore::Element* item = WebKit::core(self);
    WTF::String convertedName = WTF::String::fromUTF8(name);
    gboolean result = item->hasAttribute(convertedName);
    return result;
}

gboolean webkit_dom_element_has_attribute_ns(WebKitDOMElement* self, const gchar* namespaceURI, const gchar* localName)
{
    WebCore::JSMainThreadNullState state;
    g_return_val_if_fail(WEBKIT_DOM_IS_ELEMENT(self), FALSE);
    g_return_val_if_fail(namespaceURI, FALSE);
    g_return_val_if_fail(localName, FALSE);
    WebCore::Element* item = WebKit::core(self);
    WTF::String convertedNamespaceURI = WTF::String::fromUTF8(namespaceURI);
    WTF::String convertedLocalName = WTF::String::fromUTF8(localName);
    gboolean result = item->hasAttributeNS(convertedNamespaceURI, convertedLocalName);
    return result;
}

void webkit_dom_element_focus(WebKitDOMElement* self)
{
    WebCore::JSMainThreadNullState state;
    g_return_if_fail(WEBKIT_DOM_IS_ELEMENT(self));
    WebCore::Element* item = WebKit::core(self);
    item->focus();
}

void webkit_dom_element_blur(WebKitDOMElement* self)
{
    WebCore::JSMainThreadNullState state;
    g_return_if_fail(WEBKIT_DOM_IS_ELEMENT(self));
    WebCore::Element* item = WebKit::core(self);
    item->blur();
}

void webkit_dom_element_scroll_into_view(WebKitDOMElement* self, gboolean alignWithTop)
{
    WebCore::JSMainThreadNullState state;
    g_return_if_fail(WEBKIT_DOM_IS_ELEMENT(self));
    WebCore::Element* item = WebKit::core(self);
    item->scrollIntoView(alignWithTop);
}

void webkit_dom_element_scroll_into_view_if_needed(WebKitDOMElement* self, gboolean centerIfNeeded)
{
    WebCore::JSMainThreadNullState state;
    g_return_if_fail(WEBKIT_DOM_IS_ELEMENT(self));
    WebCore::Element* item = WebKit::core(self);
    item->scrollIntoViewIfNeeded(centerIfNeeded);
}

void webkit_dom_element_scroll_by_lines(WebKitDOMElement* self, glong lines)
{
    WebCore::JSMainThreadNullState state;
    g_return_if_fail(WEBKIT_DOM_IS_ELEMENT(self));
    WebCore::Element* item = WebKit::core(self);
    item->scrollByLines(lines);
}

void webkit_dom_element_scroll_by_pages(WebKitDOMElement* self, glong pages)
{
    WebCore::JSMainThreadNullState state;
    g_return_if_fail(WEBKIT_DOM_IS_ELEMENT(self));
    WebCore::Element* item = WebKit::core(self);
    item->scrollByPages(pages);
}

WebKitDOMHTMLCollection* webkit_dom_element_get_elements_by_class_name_as_html_collection(WebKitDOMElement* self, const gchar* name)
{
    WebCore::JSMainThreadNullState state;
    g_return_val_if_fail(WEBKIT_DOM_IS_ELEMENT(self), 0);
    g_return_val_if_fail(name, 0);
    WebCore::Element* item = WebKit::core(self);
    WTF::String convertedName = WTF::String::fromUTF8(name);
    RefPtr<WebCore::HTMLCollection> gobjectResult = WTF::getPtr(item->getElementsByClassName(convertedName));
    return WebKit::kit(gobjectResult.get());
}

gboolean webkit_dom_element_matches(WebKitDOMElement* self, const gchar* selectors, GError** error)
{
    WebCore::JSMainThreadNullState state;
    g_return_val_if_fail(WEBKIT_DOM_IS_ELEMENT(self), FALSE);
    g_return_val_if_fail(selectors, FALSE);
    g_return_val_if_fail(!error || !*error, FALSE);
    WebCore::Element* item = WebKit::core(self);
    WTF::String convertedSelectors = WTF::String::fromUTF8(selectors);
    WebCore::ExceptionCode ec = 0;
    gboolean result = item->matches(convertedSelectors, ec);
    if (ec) {
        WebCore::ExceptionCodeDescription ecdesc(ec);
        g_set_error_literal(error, g_quark_from_string("WEBKIT_DOM"), ecdesc.code, ecdesc.name);
    }
    return result;
}

WebKitDOMElement* webkit_dom_element_closest(WebKitDOMElement* self, const gchar* selectors, GError** error)
{
    WebCore::JSMainThreadNullState state;
    g_return_val_if_fail(WEBKIT_DOM_IS_ELEMENT(self), 0);
    g_return_val_if_fail(selectors, 0);
    g_return_val_if_fail(!error || !*error, 0);
    WebCore::Element* item = WebKit::core(self);
    WTF::String convertedSelectors = WTF::String::fromUTF8(selectors);
    WebCore::ExceptionCode ec = 0;
    RefPtr<WebCore::Element> gobjectResult = WTF::getPtr(item->closest(convertedSelectors, ec));
    if (ec) {
        WebCore::ExceptionCodeDescription ecdesc(ec);
        g_set_error_literal(error, g_quark_from_string("WEBKIT_DOM"), ecdesc.code, ecdesc.name);
    }
    return WebKit::kit(gobjectResult.get());
}

gboolean webkit_dom_element_webkit_matches_selector(WebKitDOMElement* self, const gchar* selectors, GError** error)
{
    WebCore::JSMainThreadNullState state;
    g_return_val_if_fail(WEBKIT_DOM_IS_ELEMENT(self), FALSE);
    g_return_val_if_fail(selectors, FALSE);
    g_return_val_if_fail(!error || !*error, FALSE);
    WebCore::Element* item = WebKit::core(self);
    WTF::String convertedSelectors = WTF::String::fromUTF8(selectors);
    WebCore::ExceptionCode ec = 0;
    gboolean result = item->matches(convertedSelectors, ec);
    if (ec) {
        WebCore::ExceptionCodeDescription ecdesc(ec);
        g_set_error_literal(error, g_quark_from_string("WEBKIT_DOM"), ecdesc.code, ecdesc.name);
    }
    return result;
}

void webkit_dom_element_webkit_request_fullscreen(WebKitDOMElement* self)
{
    WebCore::JSMainThreadNullState state;
    g_return_if_fail(WEBKIT_DOM_IS_ELEMENT(self));
    WebCore::Element* item = WebKit::core(self);
    item->webkitRequestFullscreen();
}

WebKitDOMElement* webkit_dom_element_insert_adjacent_element(WebKitDOMElement* self, const gchar* where, WebKitDOMElement* element, GError** error)
{
    WebCore::JSMainThreadNullState state;
    g_return_val_if_fail(WEBKIT_DOM_IS_ELEMENT(self), 0);
    g_return_val_if_fail(where, 0);
    g_return_val_if_fail(WEBKIT_DOM_IS_ELEMENT(element), 0);
    g_return_val_if_fail(!error || !*error, 0);
    WebCore::Element* item = WebKit::core(self);
    WTF::String convertedWhere = WTF::String::fromUTF8(where);
    WebCore::Element* convertedElement = WebKit::core(element);
    WebCore::ExceptionCode ec = 0;
    RefPtr<WebCore::Element> gobjectResult = WTF::getPtr(item->insertAdjacentElement(convertedWhere, *convertedElement, ec));
    if (ec) {
        WebCore::ExceptionCodeDescription ecdesc(ec);
        g_set_error_literal(error, g_quark_from_string("WEBKIT_DOM"), ecdesc.code, ecdesc.name);
    }
    return WebKit::kit(gobjectResult.get());
}

void webkit_dom_element_insert_adjacent_html(WebKitDOMElement* self, const gchar* where, const gchar* html, GError** error)
{
    WebCore::JSMainThreadNullState state;
    g_return_if_fail(WEBKIT_DOM_IS_ELEMENT(self));
    g_return_if_fail(where);
    g_return_if_fail(html);
    g_return_if_fail(!error || !*error);
    WebCore::Element* item = WebKit::core(self);
    WTF::String convertedWhere = WTF::String::fromUTF8(where);
    WTF::String convertedHtml = WTF::String::fromUTF8(html);
    WebCore::ExceptionCode ec = 0;
    item->insertAdjacentHTML(convertedWhere, convertedHtml, ec);
    if (ec) {
        WebCore::ExceptionCodeDescription ecdesc(ec);
        g_set_error_literal(error, g_quark_from_string("WEBKIT_DOM"), ecdesc.code, ecdesc.name);
    }
}

void webkit_dom_element_insert_adjacent_text(WebKitDOMElement* self, const gchar* where, const gchar* text, GError** error)
{
    WebCore::JSMainThreadNullState state;
    g_return_if_fail(WEBKIT_DOM_IS_ELEMENT(self));
    g_return_if_fail(where);
    g_return_if_fail(text);
    g_return_if_fail(!error || !*error);
    WebCore::Element* item = WebKit::core(self);
    WTF::String convertedWhere = WTF::String::fromUTF8(where);
    WTF::String convertedText = WTF::String::fromUTF8(text);
    WebCore::ExceptionCode ec = 0;
    item->insertAdjacentText(convertedWhere, convertedText, ec);
    if (ec) {
        WebCore::ExceptionCodeDescription ecdesc(ec);
        g_set_error_literal(error, g_quark_from_string("WEBKIT_DOM"), ecdesc.code, ecdesc.name);
    }
}

void webkit_dom_element_request_pointer_lock(WebKitDOMElement* self)
{
#if ENABLE(POINTER_LOCK)
    WebCore::JSMainThreadNullState state;
    g_return_if_fail(WEBKIT_DOM_IS_ELEMENT(self));
    WebCore::Element* item = WebKit::core(self);
    item->requestPointerLock();
#else
    UNUSED_PARAM(self);
    WEBKIT_WARN_FEATURE_NOT_PRESENT("Pointer Lock")
#endif /* ENABLE(POINTER_LOCK) */
}

void webkit_dom_element_remove(WebKitDOMElement* self, GError** error)
{
    WebCore::JSMainThreadNullState state;
    g_return_if_fail(WEBKIT_DOM_IS_ELEMENT(self));
    g_return_if_fail(!error || !*error);
    WebCore::Element* item = WebKit::core(self);
    WebCore::ExceptionCode ec = 0;
    item->remove(ec);
    if (ec) {
        WebCore::ExceptionCodeDescription ecdesc(ec);
        g_set_error_literal(error, g_quark_from_string("WEBKIT_DOM"), ecdesc.code, ecdesc.name);
    }
}

WebKitDOMElement* webkit_dom_element_query_selector(WebKitDOMElement* self, const gchar* selectors, GError** error)
{
    WebCore::JSMainThreadNullState state;
    g_return_val_if_fail(WEBKIT_DOM_IS_ELEMENT(self), 0);
    g_return_val_if_fail(selectors, 0);
    g_return_val_if_fail(!error || !*error, 0);
    WebCore::Element* item = WebKit::core(self);
    WTF::String convertedSelectors = WTF::String::fromUTF8(selectors);
    WebCore::ExceptionCode ec = 0;
    RefPtr<WebCore::Element> gobjectResult = WTF::getPtr(item->querySelector(convertedSelectors, ec));
    if (ec) {
        WebCore::ExceptionCodeDescription ecdesc(ec);
        g_set_error_literal(error, g_quark_from_string("WEBKIT_DOM"), ecdesc.code, ecdesc.name);
    }
    return WebKit::kit(gobjectResult.get());
}

WebKitDOMNodeList* webkit_dom_element_query_selector_all(WebKitDOMElement* self, const gchar* selectors, GError** error)
{
    WebCore::JSMainThreadNullState state;
    g_return_val_if_fail(WEBKIT_DOM_IS_ELEMENT(self), 0);
    g_return_val_if_fail(selectors, 0);
    g_return_val_if_fail(!error || !*error, 0);
    WebCore::Element* item = WebKit::core(self);
    WTF::String convertedSelectors = WTF::String::fromUTF8(selectors);
    WebCore::ExceptionCode ec = 0;
    RefPtr<WebCore::NodeList> gobjectResult = WTF::getPtr(item->querySelectorAll(convertedSelectors, ec));
    if (ec) {
        WebCore::ExceptionCodeDescription ecdesc(ec);
        g_set_error_literal(error, g_quark_from_string("WEBKIT_DOM"), ecdesc.code, ecdesc.name);
    }
    return WebKit::kit(gobjectResult.get());
}

gchar* webkit_dom_element_get_tag_name(WebKitDOMElement* self)
{
    WebCore::JSMainThreadNullState state;
    g_return_val_if_fail(WEBKIT_DOM_IS_ELEMENT(self), 0);
    WebCore::Element* item = WebKit::core(self);
    gchar* result = convertToUTF8String(item->tagName());
    return result;
}

WebKitDOMNamedNodeMap* webkit_dom_element_get_attributes(WebKitDOMElement* self)
{
    WebCore::JSMainThreadNullState state;
    g_return_val_if_fail(WEBKIT_DOM_IS_ELEMENT(self), 0);
    WebCore::Element* item = WebKit::core(self);
    RefPtr<WebCore::NamedNodeMap> gobjectResult = WTF::getPtr(item->attributes());
    return WebKit::kit(gobjectResult.get());
}

WebKitDOMCSSStyleDeclaration* webkit_dom_element_get_style(WebKitDOMElement* self)
{
    WebCore::JSMainThreadNullState state;
    g_return_val_if_fail(WEBKIT_DOM_IS_ELEMENT(self), 0);
    WebCore::Element* item = WebKit::core(self);
    RefPtr<WebCore::CSSStyleDeclaration> gobjectResult = WTF::getPtr(item->cssomStyle());
    return WebKit::kit(gobjectResult.get());
}

gchar* webkit_dom_element_get_id(WebKitDOMElement* self)
{
    WebCore::JSMainThreadNullState state;
    g_return_val_if_fail(WEBKIT_DOM_IS_ELEMENT(self), 0);
    WebCore::Element* item = WebKit::core(self);
    gchar* result = convertToUTF8String(item->getIdAttribute());
    return result;
}

void webkit_dom_element_set_id(WebKitDOMElement* self, const gchar* value)
{
    WebCore::JSMainThreadNullState state;
    g_return_if_fail(WEBKIT_DOM_IS_ELEMENT(self));
    g_return_if_fail(value);
    WebCore::Element* item = WebKit::core(self);
    WTF::String convertedValue = WTF::String::fromUTF8(value);
    item->setAttributeWithoutSynchronization(WebCore::HTMLNames::idAttr, convertedValue);
}

gchar* webkit_dom_element_get_namespace_uri(WebKitDOMElement* self)
{
    WebCore::JSMainThreadNullState state;
    g_return_val_if_fail(WEBKIT_DOM_IS_ELEMENT(self), 0);
    WebCore::Element* item = WebKit::core(self);
    gchar* result = convertToUTF8String(item->namespaceURI());
    return result;
}

gchar* webkit_dom_element_get_prefix(WebKitDOMElement* self)
{
    WebCore::JSMainThreadNullState state;
    g_return_val_if_fail(WEBKIT_DOM_IS_ELEMENT(self), 0);
    WebCore::Element* item = WebKit::core(self);
    gchar* result = convertToUTF8String(item->prefix());
    return result;
}

gchar* webkit_dom_element_get_local_name(WebKitDOMElement* self)
{
    WebCore::JSMainThreadNullState state;
    g_return_val_if_fail(WEBKIT_DOM_IS_ELEMENT(self), 0);
    WebCore::Element* item = WebKit::core(self);
    gchar* result = convertToUTF8String(item->localName());
    return result;
}

gdouble webkit_dom_element_get_offset_left(WebKitDOMElement* self)
{
    WebCore::JSMainThreadNullState state;
    g_return_val_if_fail(WEBKIT_DOM_IS_ELEMENT(self), 0);
    WebCore::Element* item = WebKit::core(self);
    gdouble result = item->offsetLeft();
    return result;
}

gdouble webkit_dom_element_get_offset_top(WebKitDOMElement* self)
{
    WebCore::JSMainThreadNullState state;
    g_return_val_if_fail(WEBKIT_DOM_IS_ELEMENT(self), 0);
    WebCore::Element* item = WebKit::core(self);
    gdouble result = item->offsetTop();
    return result;
}

gdouble webkit_dom_element_get_offset_width(WebKitDOMElement* self)
{
    WebCore::JSMainThreadNullState state;
    g_return_val_if_fail(WEBKIT_DOM_IS_ELEMENT(self), 0);
    WebCore::Element* item = WebKit::core(self);
    gdouble result = item->offsetWidth();
    return result;
}

gdouble webkit_dom_element_get_offset_height(WebKitDOMElement* self)
{
    WebCore::JSMainThreadNullState state;
    g_return_val_if_fail(WEBKIT_DOM_IS_ELEMENT(self), 0);
    WebCore::Element* item = WebKit::core(self);
    gdouble result = item->offsetHeight();
    return result;
}

gdouble webkit_dom_element_get_client_left(WebKitDOMElement* self)
{
    WebCore::JSMainThreadNullState state;
    g_return_val_if_fail(WEBKIT_DOM_IS_ELEMENT(self), 0);
    WebCore::Element* item = WebKit::core(self);
    gdouble result = item->clientLeft();
    return result;
}

gdouble webkit_dom_element_get_client_top(WebKitDOMElement* self)
{
    WebCore::JSMainThreadNullState state;
    g_return_val_if_fail(WEBKIT_DOM_IS_ELEMENT(self), 0);
    WebCore::Element* item = WebKit::core(self);
    gdouble result = item->clientTop();
    return result;
}

gdouble webkit_dom_element_get_client_width(WebKitDOMElement* self)
{
    WebCore::JSMainThreadNullState state;
    g_return_val_if_fail(WEBKIT_DOM_IS_ELEMENT(self), 0);
    WebCore::Element* item = WebKit::core(self);
    gdouble result = item->clientWidth();
    return result;
}

gdouble webkit_dom_element_get_client_height(WebKitDOMElement* self)
{
    WebCore::JSMainThreadNullState state;
    g_return_val_if_fail(WEBKIT_DOM_IS_ELEMENT(self), 0);
    WebCore::Element* item = WebKit::core(self);
    gdouble result = item->clientHeight();
    return result;
}

glong webkit_dom_element_get_scroll_left(WebKitDOMElement* self)
{
    WebCore::JSMainThreadNullState state;
    g_return_val_if_fail(WEBKIT_DOM_IS_ELEMENT(self), 0);
    WebCore::Element* item = WebKit::core(self);
    glong result = item->scrollLeft();
    return result;
}

void webkit_dom_element_set_scroll_left(WebKitDOMElement* self, glong value)
{
    WebCore::JSMainThreadNullState state;
    g_return_if_fail(WEBKIT_DOM_IS_ELEMENT(self));
    WebCore::Element* item = WebKit::core(self);
    item->setScrollLeft(value);
}

glong webkit_dom_element_get_scroll_top(WebKitDOMElement* self)
{
    WebCore::JSMainThreadNullState state;
    g_return_val_if_fail(WEBKIT_DOM_IS_ELEMENT(self), 0);
    WebCore::Element* item = WebKit::core(self);
    glong result = item->scrollTop();
    return result;
}

void webkit_dom_element_set_scroll_top(WebKitDOMElement* self, glong value)
{
    WebCore::JSMainThreadNullState state;
    g_return_if_fail(WEBKIT_DOM_IS_ELEMENT(self));
    WebCore::Element* item = WebKit::core(self);
    item->setScrollTop(value);
}

glong webkit_dom_element_get_scroll_width(WebKitDOMElement* self)
{
    WebCore::JSMainThreadNullState state;
    g_return_val_if_fail(WEBKIT_DOM_IS_ELEMENT(self), 0);
    WebCore::Element* item = WebKit::core(self);
    glong result = item->scrollWidth();
    return result;
}

glong webkit_dom_element_get_scroll_height(WebKitDOMElement* self)
{
    WebCore::JSMainThreadNullState state;
    g_return_val_if_fail(WEBKIT_DOM_IS_ELEMENT(self), 0);
    WebCore::Element* item = WebKit::core(self);
    glong result = item->scrollHeight();
    return result;
}

WebKitDOMElement* webkit_dom_element_get_offset_parent(WebKitDOMElement* self)
{
    WebCore::JSMainThreadNullState state;
    g_return_val_if_fail(WEBKIT_DOM_IS_ELEMENT(self), 0);
    WebCore::Element* item = WebKit::core(self);
    RefPtr<WebCore::Element> gobjectResult = WTF::getPtr(item->bindingsOffsetParent());
    return WebKit::kit(gobjectResult.get());
}

gchar* webkit_dom_element_get_inner_html(WebKitDOMElement* self)
{
    WebCore::JSMainThreadNullState state;
    g_return_val_if_fail(WEBKIT_DOM_IS_ELEMENT(self), 0);
    WebCore::Element* item = WebKit::core(self);
    gchar* result = convertToUTF8String(item->innerHTML());
    return result;
}

void webkit_dom_element_set_inner_html(WebKitDOMElement* self, const gchar* value, GError** error)
{
    WebCore::JSMainThreadNullState state;
    g_return_if_fail(WEBKIT_DOM_IS_ELEMENT(self));
    g_return_if_fail(value);
    g_return_if_fail(!error || !*error);
    WebCore::Element* item = WebKit::core(self);
    WTF::String convertedValue = WTF::String::fromUTF8(value);
    WebCore::ExceptionCode ec = 0;
    item->setInnerHTML(convertedValue, ec);
    if (ec) {
        WebCore::ExceptionCodeDescription ecdesc(ec);
        g_set_error_literal(error, g_quark_from_string("WEBKIT_DOM"), ecdesc.code, ecdesc.name);
    }
}

gchar* webkit_dom_element_get_outer_html(WebKitDOMElement* self)
{
    WebCore::JSMainThreadNullState state;
    g_return_val_if_fail(WEBKIT_DOM_IS_ELEMENT(self), 0);
    WebCore::Element* item = WebKit::core(self);
    gchar* result = convertToUTF8String(item->outerHTML());
    return result;
}

void webkit_dom_element_set_outer_html(WebKitDOMElement* self, const gchar* value, GError** error)
{
    WebCore::JSMainThreadNullState state;
    g_return_if_fail(WEBKIT_DOM_IS_ELEMENT(self));
    g_return_if_fail(value);
    g_return_if_fail(!error || !*error);
    WebCore::Element* item = WebKit::core(self);
    WTF::String convertedValue = WTF::String::fromUTF8(value);
    WebCore::ExceptionCode ec = 0;
    item->setOuterHTML(convertedValue, ec);
    if (ec) {
        WebCore::ExceptionCodeDescription ecdesc(ec);
        g_set_error_literal(error, g_quark_from_string("WEBKIT_DOM"), ecdesc.code, ecdesc.name);
    }
}

gchar* webkit_dom_element_get_class_name(WebKitDOMElement* self)
{
    WebCore::JSMainThreadNullState state;
    g_return_val_if_fail(WEBKIT_DOM_IS_ELEMENT(self), 0);
    WebCore::Element* item = WebKit::core(self);
    gchar* result = convertToUTF8String(item->attributeWithoutSynchronization(WebCore::HTMLNames::classAttr));
    return result;
}

void webkit_dom_element_set_class_name(WebKitDOMElement* self, const gchar* value)
{
    WebCore::JSMainThreadNullState state;
    g_return_if_fail(WEBKIT_DOM_IS_ELEMENT(self));
    g_return_if_fail(value);
    WebCore::Element* item = WebKit::core(self);
    WTF::String convertedValue = WTF::String::fromUTF8(value);
    item->setAttributeWithoutSynchronization(WebCore::HTMLNames::classAttr, convertedValue);
}

WebKitDOMDOMTokenList* webkit_dom_element_get_class_list(WebKitDOMElement* self)
{
    WebCore::JSMainThreadNullState state;
    g_return_val_if_fail(WEBKIT_DOM_IS_ELEMENT(self), 0);
    WebCore::Element* item = WebKit::core(self);
    RefPtr<WebCore::DOMTokenList> gobjectResult = WTF::getPtr(item->classList());
    return WebKit::kit(gobjectResult.get());
}

gchar* webkit_dom_element_get_webkit_region_overset(WebKitDOMElement* self)
{
#if ENABLE(CSS_REGIONS)
    WebCore::JSMainThreadNullState state;
    g_return_val_if_fail(WEBKIT_DOM_IS_ELEMENT(self), 0);
    WebCore::Element* item = WebKit::core(self);
    gchar* result = convertToUTF8String(item->webkitRegionOverset());
    return result;
#else
    UNUSED_PARAM(self);
    WEBKIT_WARN_FEATURE_NOT_PRESENT("Css Regions")
    return 0;
#endif /* ENABLE(CSS_REGIONS) */
}

WebKitDOMElement* webkit_dom_element_get_previous_element_sibling(WebKitDOMElement* self)
{
    WebCore::JSMainThreadNullState state;
    g_return_val_if_fail(WEBKIT_DOM_IS_ELEMENT(self), 0);
    WebCore::Element* item = WebKit::core(self);
    RefPtr<WebCore::Element> gobjectResult = WTF::getPtr(item->previousElementSibling());
    return WebKit::kit(gobjectResult.get());
}

WebKitDOMElement* webkit_dom_element_get_next_element_sibling(WebKitDOMElement* self)
{
    WebCore::JSMainThreadNullState state;
    g_return_val_if_fail(WEBKIT_DOM_IS_ELEMENT(self), 0);
    WebCore::Element* item = WebKit::core(self);
    RefPtr<WebCore::Element> gobjectResult = WTF::getPtr(item->nextElementSibling());
    return WebKit::kit(gobjectResult.get());
}

WebKitDOMHTMLCollection* webkit_dom_element_get_children(WebKitDOMElement* self)
{
    WebCore::JSMainThreadNullState state;
    g_return_val_if_fail(WEBKIT_DOM_IS_ELEMENT(self), 0);
    WebCore::Element* item = WebKit::core(self);
    RefPtr<WebCore::HTMLCollection> gobjectResult = WTF::getPtr(item->children());
    return WebKit::kit(gobjectResult.get());
}

WebKitDOMElement* webkit_dom_element_get_first_element_child(WebKitDOMElement* self)
{
    WebCore::JSMainThreadNullState state;
    g_return_val_if_fail(WEBKIT_DOM_IS_ELEMENT(self), 0);
    WebCore::Element* item = WebKit::core(self);
    RefPtr<WebCore::Element> gobjectResult = WTF::getPtr(item->firstElementChild());
    return WebKit::kit(gobjectResult.get());
}

WebKitDOMElement* webkit_dom_element_get_last_element_child(WebKitDOMElement* self)
{
    WebCore::JSMainThreadNullState state;
    g_return_val_if_fail(WEBKIT_DOM_IS_ELEMENT(self), 0);
    WebCore::Element* item = WebKit::core(self);
    RefPtr<WebCore::Element> gobjectResult = WTF::getPtr(item->lastElementChild());
    return WebKit::kit(gobjectResult.get());
}

gulong webkit_dom_element_get_child_element_count(WebKitDOMElement* self)
{
    WebCore::JSMainThreadNullState state;
    g_return_val_if_fail(WEBKIT_DOM_IS_ELEMENT(self), 0);
    WebCore::Element* item = WebKit::core(self);
    gulong result = item->childElementCount();
    return result;
}

