/*
 * Copyright (c) 2020 Jason Liu
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#ifndef _MACPORTS_APPKIT_H_
#define _MACPORTS_APPKIT_H_

/*
 * Compatibility mapping for old versions of AppKit. By using the
 * mapping in this file, we can still compile newer versions of software
 * using older versions of Mac OS X.
 * 
 * Many AppKit enum constants were deprecated/renamed in macOS 10.12
 * Sierra in order to make naming more consistent. The directives in
 * this wrapper file replace the old enum names in the system AppKit
 * header files with the new ones, so that code that has been written
 * for newer versions of AppKit will still be able to be compiled on a
 * system with a pre-10.12 AppKit SDK.
 *
 * References:
 * * https://stackoverflow.com/a/44322273
 * * https://lists.gnu.org/archive/html/emacs-devel/2016-08/msg00358.html
 * * https://github.com/qemu/qemu/commit/4ba967ad7454c08d7e01b047d34d0c3d98f2a10d#diff-1fdc5419bda303ee3f720fb8ccdab2c3
 * * https://patchwork.kernel.org/patch/9697563/
 *
 * Note that in the references cited above, the #define directives map
 * the new enum names to the old names. This is because their directives
 * are located in their own applications' source code files. However,
 * our wrapper file performs the mapping in the opposite direction, and
 * replaces the old enum names with the new names. This is because our
 * file is patching the system AppKit header, instead of changing the
 * application's source code to conform to an old system AppKit header.
 * 
 * Also, note that our file only fixes the Objective-C header file. It
 * does not affect the bridge metadata file (a.k.a. bridge support
 * file), so any software that uses RubyCocoa or PyObjC to access AppKit
 * will likely still be broken. This is despite the fact that gathering
 * the list of enum names in our wrapper file was done by comparing the
 * bridge metadata files taken from a macOS 10.11 and 10.12 machine,
 * located at
 * /System/Library/Frameworks/AppKit.framework/Resources/BridgeSupport/AppKit.bridgesupport
 *
 * Theoretically, you could use the 'gen_bridge_metadata' command to try
 * to generate a new bridge support file, but this would only work if
 * you went through and actually replaced the old enum names in ALL of
 * AppKit's header files (not just AppKit.h, which is simply a top-level
 * header that #includes a bunch of other headers), and our wrapper
 * would not help in this regard. In addition, it's probably not a good
 * idea to be using the 'gen_bridge_metadata' tool to replace the bridge
 * support file in a system framework; it is really intended to be used
 * by software authors who want to provide bridge support for RubyCocoa
 * and PyObjC in frameworks generated by their own projects. In the
 * future, if someone tries to create a MacPorts package for some piece
 * of software that does use RubyCocoa or PyObjC to access AppKit, then
 * a different sort of solution than this wrapper file may be needed for
 * that scenario.
 *
 * References:
 * * https://developer.apple.com/library/archive/documentation/Cocoa/Conceptual/RubyPythonCocoa/Articles/GenerateFrameworkMetadata.html
 * * https://bugs.eclipse.org/bugs/show_bug.cgi?id=502090
 * * Execute 'man gen_bridge_metadata' in a terminal window.
 */

/********************************************************************
 *                          IMPORTANT NOTE                          *
 *                                                                  *
 * This wrapper file is currently being used by a few ports:        *
 * Blender, MaterialX, and Godot. This file will eventually be      *
 * added to the MacPorts legacy support package. After it gets      *
 * added, we will be able to remove this file from the ports and    *
 * simply add the MacPorts legacy support package as a dependency   *
 * in the portfiles for Blender, MaterialX, and Godot.              *
 ********************************************************************/

#if __ENVIRONMENT_MAC_OS_X_VERSION_MIN_REQUIRED__ < 101200

/*
 * Note: A small number of the #defines below are commented out because
 * they cause compile errors when building Blender. MaterialX doesn't
 * use them in its source, so they can be safely commented out without
 * causing any ill effects.
 */

#define NS16BitBigEndianBitmapFormat         NSBitmapFormatSixteenBitBigEndian
#define NS16BitLittleEndianBitmapFormat      NSBitmapFormatSixteenBitLittleEndian
#define NS32BitBigEndianBitmapFormat         NSBitmapFormatThirtyTwoBitBigEndian
#define NS32BitLittleEndianBitmapFormat      NSBitmapFormatThirtyTwoBitLittleEndian
#define NSAlphaFirstBitmapFormat             NSBitmapFormatAlphaFirst
#define NSAlphaNonpremultipliedBitmapFormat  NSBitmapFormatAlphaNonpremultiplied
#define NSAlternateKeyMask                   NSEventModifierFlagOption
#define NSAnyEventMask                       NSEventMaskAny
#define NSAppKitDefined                      NSEventTypeAppKitDefined
#define NSAppKitDefinedMask                  NSEventMaskAppKitDefined
#define NSApplicationActivatedEventType      NSEventSubtypeApplicationActivated
#define NSApplicationDeactivatedEventType    NSEventSubtypeApplicationDeactivated
#define NSApplicationDefined                 NSEventTypeApplicationDefined
#define NSApplicationDefinedMask             NSEventMaskApplicationDefined
#define NSBMPFileType                        NSBitmapImageFileTypeBMP
#define NSBorderlessWindowMask               NSWindowStyleMaskBorderless
#define NSCMYKColorSpaceModel                NSColorSpaceModelCMYK
#define NSCMYKModeColorPanel                 NSColorPanelModeCMYK
/* #define NSCenterTextAlignment                NSTextAlignmentCenter */
#define NSCircularBezelStyle                 NSBezelStyleCircular
#define NSCircularSlider                     NSSliderTypeCircular
#define NSClosableWindowMask                 NSWindowStyleMaskClosable
#define NSColorListModeColorPanel            NSColorPanelModeColorList
#define NSCommandKeyMask                     NSEventModifierFlagCommand
#define NSCompositeClear                     NSCompositingOperationClear
#define NSCompositeCopy                      NSCompositingOperationCopy
#define NSCompositeDestinationAtop           NSCompositingOperationDestinationAtop
#define NSCompositeDestinationIn             NSCompositingOperationDestinationIn
#define NSCompositeDestinationOut            NSCompositingOperationDestinationOut
#define NSCompositeDestinationOver           NSCompositingOperationDestinationOver
#define NSCompositeHighlight                 NSCompositingOperationHighlight
#define NSCompositePlusDarker                NSCompositingOperationPlusDarker
#define NSCompositeSourceAtop                NSCompositingOperationSourceAtop
#define NSCompositeSourceIn                  NSCompositingOperationSourceIn
#define NSCompositeSourceOut                 NSCompositingOperationSourceOut
#define NSCompositeSourceOver                NSCompositingOperationSourceOver
#define NSCompositeXOR                       NSCompositingOperationXOR
#define NSControlKeyMask                     NSEventModifierFlagControl
#define NSCrayonModeColorPanel               NSColorPanelModeCrayon
#define NSCriticalAlertStyle                 NSAlertStyleCritical
#define NSCursorPointingDevice               NSPointingDeviceTypeCursor
#define NSCursorUpdate                       NSEventTypeCursorUpdate
#define NSCursorUpdateMask                   NSEventMaskCursorUpdate
#define NSCustomPaletteModeColorPanel        NSColorPanelModeCustomPalette
/* #define NSDefaultTokenStyle                  NSTokenStyleDefault */
#define NSDeviceIndependentModifierFlagsMask NSEventModifierFlagDeviceIndependentFlagsMask
#define NSAlphaShiftKeyMask                  NSEventModifierFlagCapsLock
#define NSDeviceNColorSpaceModel             NSColorSpaceModelDeviceN
#define NSDisclosureBezelStyle               NSBezelStyleDisclosure
#define NSDocModalWindowMask                 NSWindowStyleMaskDocModalWindow
#define NSEraserPointingDevice               NSPointingDeviceTypeEraser
#define NSFlagsChanged                       NSEventTypeFlagsChanged
#define NSFlagsChangedMask                   NSEventMaskFlagsChanged
#define NSFloatingPointSamplesBitmapFormat   NSBitmapFormatFloatingPointSamples
#define NSFullScreenWindowMask               NSWindowStyleMaskFullScreen
#define NSFullSizeContentViewWindowMask      NSWindowStyleMaskFullSizeContentView
#define NSFunctionKeyMask                    NSEventModifierFlagFunction
#define NSGIFFileType                        NSBitmapImageFileTypeGIF
#define NSGrayColorSpaceModel                NSColorSpaceModelGray
#define NSGrayModeColorPanel                 NSColorPanelModeGray
#define NSHSBModeColorPanel                  NSColorPanelModeHSB
#define NSHUDWindowMask                      NSWindowStyleMaskHUDWindow
#define NSHelpButtonBezelStyle               NSBezelStyleHelpButton
#define NSHelpKeyMask                        NSEventModifierFlagHelp
#define NSIndexedColorSpaceModel             NSColorSpaceModelIndexed
#define NSInformationalAlertStyle            NSAlertStyleInformational
#define NSInlineBezelStyle                   NSBezelStyleInline
#define NSJPEG2000FileType                   NSBitmapImageFileTypeJPEG2000
#define NSJPEGFileType                       NSBitmapImageFileTypeJPEG
/* #define NSJustifiedTextAlignment             NSTextAlignmentJustified */
#define NSKeyDown                            NSEventTypeKeyDown
#define NSKeyDownMask                        NSEventMaskKeyDown
#define NSKeyUp                              NSEventTypeKeyUp
#define NSKeyUpMask                          NSEventMaskKeyUp
#define NSLABColorSpaceModel                 NSColorSpaceModelLAB
#define NSLeftMouseDown                      NSEventTypeLeftMouseDown
#define NSLeftMouseDownMask                  NSEventMaskLeftMouseDown
#define NSLeftMouseDragged                   NSEventTypeLeftMouseDragged
#define NSLeftMouseDraggedMask               NSEventMaskLeftMouseDragged
#define NSLeftMouseUp                        NSEventTypeLeftMouseUp
#define NSLeftMouseUpMask                    NSEventMaskLeftMouseUp
/* #define NSLeftTextAlignment                  NSTextAlignmentLeft */
#define NSLinearSlider                       NSSliderTypeLinear
#define NSMiniControlSize                    NSControlSizeMini
#define NSMiniaturizableWindowMask           NSWindowStyleMaskMiniaturizable
/* #define NSMomentaryPushButton                NSMomentaryLightButton */
#define NSMouseEntered                       NSEventTypeMouseEntered
#define NSMouseEnteredMask                   NSEventMaskMouseEntered
#define NSMouseEventSubtype                  NSEventSubtypeMouseEvent
#define NSMouseExited                        NSEventTypeMouseExited
#define NSMouseExitedMask                    NSEventMaskMouseExited
#define NSMouseMoved                         NSEventTypeMouseMoved
#define NSMouseMovedMask                     NSEventMaskMouseMoved
/* #define NSNaturalTextAlignment               NSTextAlignmentNatural */
#define NSNoModeColorPanel                   NSColorPanelModeNone
#define NSNonactivatingPanelMask             NSWindowStyleMaskNonactivatingPanel
#define NSNumericPadKeyMask                  NSEventModifierFlagNumericPad
#define NSOtherMouseDown                     NSEventTypeOtherMouseDown
#define NSOtherMouseDownMask                 NSEventMaskOtherMouseDown
#define NSOtherMouseDragged                  NSEventTypeOtherMouseDragged
#define NSOtherMouseDraggedMask              NSEventMaskOtherMouseDragged
#define NSOtherMouseUp                       NSEventTypeOtherMouseUp
#define NSOtherMouseUpMask                   NSEventMaskOtherMouseUp
#define NSPNGFileType                        NSBitmapImageFileTypePNG
#define NSPatternColorSpaceModel             NSColorSpaceModelPatterned
#define NSPenLowerSideMask                   NSEventButtonMaskPenLowerSide
#define NSPenPointingDevice                  NSPointingDeviceTypePen
#define NSPenTipMask                         NSEventButtonMaskPenTip
#define NSPenUpperSideMask                   NSEventButtonMaskPenUpperSide
#define NSPeriodic                           NSEventTypePeriodic
#define NSPowerOffEventType                  NSEventSubtypePowerOff
#define NSRGBColorSpaceModel                 NSColorSpaceModelRGB
#define NSRGBModeColorPanel                  NSColorPanelModeRGB
#define NSRecessedBezelStyle                 NSBezelStyleRecessed
#define NSRegularControlSize                 NSControlSizeRegular
#define NSRegularSquareBezelStyle            NSBezelStyleRegularSquare
#define NSResizableWindowMask                NSWindowStyleMaskResizable
#define NSRightMouseDown                     NSEventTypeRightMouseDown
#define NSRightMouseDownMask                 NSEventMaskRightMouseDown
#define NSRightMouseDragged                  NSEventTypeRightMouseDragged
#define NSRightMouseDraggedMask              NSEventMaskRightMouseDragged
#define NSRightMouseUp                       NSEventTypeRightMouseUp
#define NSRightMouseUpMask                   NSEventMaskRightMouseUp
/* #define NSRightTextAlignment                 NSTextAlignmentRight */
#define NSRoundRectBezelStyle                NSBezelStyleRoundRect
#define NSRoundedBezelStyle                  NSBezelStyleRounded
#define NSRoundedDisclosureBezelStyle        NSBezelStyleRoundedDisclosure
/* #define NSRoundedTokenStyle                  NSTokenStyleRounded */
#define NSScreenChangedEventType             NSEventSubtypeScreenChanged
#define NSScrollWheel                        NSEventTypeScrollWheel
#define NSShadowlessSquareBezelStyle         NSBezelStyleShadowlessSquare
#define NSShiftKeyMask                       NSEventModifierFlagShift
#define NSSmallControlSize                   NSControlSizeSmall
#define NSSmallSquareBezelStyle              NSBezelStyleSmallSquare
#define NSSystemDefined                      NSEventTypeSystemDefined
#define NSSystemDefinedMask                  NSEventMaskSystemDefined
#define NSTIFFFileType                       NSBitmapImageFileTypeTIFF
#define NSTabletPoint                        NSEventTypeTabletPoint
#define NSTabletPointEventSubtype            NSEventSubtypeTabletPoint
#define NSTabletPointMask                    NSEventMaskTabletPoint
#define NSTabletProximity                    NSEventTypeTabletProximity
#define NSTabletProximityEventSubtype        NSEventSubtypeTabletProximity
#define NSTabletProximityMask                NSEventMaskTabletProximity
#define NSTexturedBackgroundWindowMask       NSWindowStyleMaskTexturedBackground
#define NSTexturedRoundedBezelStyle          NSBezelStyleTexturedRounded
#define NSTexturedSquareBezelStyle           NSBezelStyleTexturedSquare
#define NSTickMarkAbove                      NSTickMarkPositionAbove
#define NSTickMarkBelow                      NSTickMarkPositionBelow
#define NSTickMarkLeft                       NSTickMarkPositionLeading
#define NSTickMarkRight                      NSTickMarkPositionTrailing
#define NSTitledWindowMask                   NSWindowStyleMaskTitled
#define NSTouchEventSubtype                  NSEventSubtypeTouch
#define NSUnifiedTitleAndToolbarWindowMask   NSWindowStyleMaskUnifiedTitleAndToolbar
#define NSUnknownColorSpaceModel             NSColorSpaceModelUnknown
#define NSUnknownPointingDevice              NSPointingDeviceTypeUnknown
#define NSUtilityWindowMask                  NSWindowStyleMaskUtilityWindow
#define NSWarningAlertStyle                  NSAlertStyleWarning
#define NSWheelModeColorPanel                NSColorPanelModeWheel
#define NSWindowExposedEventType             NSEventSubtypeWindowExposed
#define NSWindowMovedEventType               NSEventSubtypeWindowMoved

/*
 * New enum: NSWindowStyleMask is an enumerated type that was introduced
 * starting in macOS 10.12; there is no pre-existing equivalent enum
 * type in macOS 10.11 and earlier. However, since this enum's typedef
 * is essentially equivalent to that of an unsigned integer, we can
 * substitute NSUInteger in older versions of macOS and things should
 * still work.
 *
 * Reference: https://developer.apple.com/documentation/appkit/nswindowstylemask
 */
#define NSWindowStyleMask                    NSUInteger

#endif /* __ENVIRONMENT_MAC_OS_X_VERSION_MIN_REQUIRED__ < 101200 */

/* Include the primary system AppKit.h */
#include_next <AppKit/AppKit.h>

#endif /* _MACPORTS_APPKIT_H_ */
