﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/storagegateway/StorageGatewayRequest.h>
#include <aws/storagegateway/StorageGateway_EXPORTS.h>
#include <aws/storagegateway/model/Tag.h>

#include <utility>

namespace Aws {
namespace StorageGateway {
namespace Model {

/**
 * <p>A JSON object containing one or more of the following fields:</p> <ul> <li>
 * <p> <a>CreateSnapshotInput$SnapshotDescription</a> </p> </li> <li> <p>
 * <a>CreateSnapshotInput$VolumeARN</a> </p> </li> </ul><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/storagegateway-2013-06-30/CreateSnapshotInput">AWS
 * API Reference</a></p>
 */
class CreateSnapshotRequest : public StorageGatewayRequest {
 public:
  AWS_STORAGEGATEWAY_API CreateSnapshotRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateSnapshot"; }

  AWS_STORAGEGATEWAY_API Aws::String SerializePayload() const override;

  AWS_STORAGEGATEWAY_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the volume. Use the <a>ListVolumes</a>
   * operation to return a list of gateway volumes.</p>
   */
  inline const Aws::String& GetVolumeARN() const { return m_volumeARN; }
  inline bool VolumeARNHasBeenSet() const { return m_volumeARNHasBeenSet; }
  template <typename VolumeARNT = Aws::String>
  void SetVolumeARN(VolumeARNT&& value) {
    m_volumeARNHasBeenSet = true;
    m_volumeARN = std::forward<VolumeARNT>(value);
  }
  template <typename VolumeARNT = Aws::String>
  CreateSnapshotRequest& WithVolumeARN(VolumeARNT&& value) {
    SetVolumeARN(std::forward<VolumeARNT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Textual description of the snapshot that appears in the Amazon EC2 console,
   * Elastic Block Store snapshots panel in the <b>Description</b> field, and in the
   * Storage Gateway snapshot <b>Details</b> pane, <b>Description</b> field.</p>
   */
  inline const Aws::String& GetSnapshotDescription() const { return m_snapshotDescription; }
  inline bool SnapshotDescriptionHasBeenSet() const { return m_snapshotDescriptionHasBeenSet; }
  template <typename SnapshotDescriptionT = Aws::String>
  void SetSnapshotDescription(SnapshotDescriptionT&& value) {
    m_snapshotDescriptionHasBeenSet = true;
    m_snapshotDescription = std::forward<SnapshotDescriptionT>(value);
  }
  template <typename SnapshotDescriptionT = Aws::String>
  CreateSnapshotRequest& WithSnapshotDescription(SnapshotDescriptionT&& value) {
    SetSnapshotDescription(std::forward<SnapshotDescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A list of up to 50 tags that can be assigned to a snapshot. Each tag is a
   * key-value pair.</p>  <p>Valid characters for key and value are letters,
   * spaces, and numbers representable in UTF-8 format, and the following special
   * characters: + - = . _ : / @. The maximum length of a tag's key is 128
   * characters, and the maximum length for a tag's value is 256.</p>
   */
  inline const Aws::Vector<Tag>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Vector<Tag>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Vector<Tag>>
  CreateSnapshotRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsT = Tag>
  CreateSnapshotRequest& AddTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace_back(std::forward<TagsT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_volumeARN;
  bool m_volumeARNHasBeenSet = false;

  Aws::String m_snapshotDescription;
  bool m_snapshotDescriptionHasBeenSet = false;

  Aws::Vector<Tag> m_tags;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace StorageGateway
}  // namespace Aws
