%% -*- erlang-indent-level: 2 -*-
%%--------------------------------------------------------------------
%% Tests that basic cases of binary construction work
%%--------------------------------------------------------------------
-module(bs_construct).

-export([test/0]).

test() ->
  <<42>> = sz(8),
  <<42:8/little>> = sz_little(8),
  <<55>> = take_five(1, 3, 1, 7, 4),
  ok = bs5(),
  16#10000008 = bit_size(large_bin(1, 2, 3, 4)),
  ok = bad_ones(),
  ok.

%%--------------------------------------------------------------------
%% Taken from a bug report submitted by Dan Wallin (24 Oct 2003), the
%% following cases test construction of binaries whose segments have
%% sizes that are statically unknown.

sz(S) ->
  <<42:S>>.

sz_little(S) ->
  <<42:S/little>>.

take_five(A, Head, FB, C, Tail) ->
  <<A:Head, FB:1, C:Tail>>.

%%--------------------------------------------------------------------

bs5() ->
  Const = mk_constant(),
  Pairs = mk_pairs(),
  true = are_same(Const, Pairs),
  true = lists:all(fun ({B, L}) -> binary_to_list(B) =:= L end, Pairs),
  ok.

are_same(C, L) ->
  C =:= L.

mk_constant() ->
  [{<<213>>,[213]},
   {<<56>>,[56]},
   {<<1,2>>,[1,2]},
   {<<71>>,[71]},
   {<<8,1>>,[8,1]},
   {<<3,9>>,[3,9]},
   {<<9,3>>,[9,3]},
   {<<0,0,0,0>>,[0,0,0,0]},
   {<<62,0,0,0>>,[62,0,0,0]},
   {<<0,0,0,62>>,[0,0,0,62]},
   {<<138,99,0,147>>,[138,99,0,147]},
   {<<138,99,0,148>>,[138,99,0,148]},
   {<<147,0,99,138>>,[147,0,99,138]},
   {<<255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255>>,
    [255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255]},
   {<<13>>,[13]},
   {<<0,4,0,5>>,[0,4,0,5]},
   {<<129>>,[129]},
   {<<129>>,[129]},
   {<<1,2>>,[1,2]},
   {<<1>>,[1]},
   {<<4,3,1>>,[4,3,1]},
   {<<47>>,[47]},
   {<<>>,[]},
   {<<97,112,97>>,[97,112,97]},
   {<<46,110,142,77,45,204,233>>,[46,110,142,77,45,204,233]},
   {<<>>,[]}].

mk_pairs() ->
  L4 = [138,99,0,147],
  [{<<-43>>,[256-43]},
   {<<56>>,[56]},
   {<<1,2>>,[1,2]},
   {<<4:4,7:4>>,[4*16+7]},
   {<<1:5,1:11>>,[1*8,1]},
   {<<777:16/big>>,[3,9]},
   {<<777:16/little>>,[9,3]},
   {<<0.0:32/float>>,[0,0,0,0]},
   {<<0.125:32/float>>,[62,0,0,0]},
   {<<0.125:32/little-float>>,[0,0,0,62]},
   {<<57285702734876389752897683:32>>,L4},
   {<<57285702734876389752897684:32>>,[138,99,0,148]},
   {<<57285702734876389752897683:32/little>>,lists:reverse(L4)},
   {<<-1:17/unit:8>>,lists:duplicate(17,255)},
   {<<13>>,[13]},
   {<<4:8/unit:2,5:2/unit:8>>,[0,4,0,5]},
   {<<1:1,0:6,1:1>>,[129]},
   {<<1:1/little,0:6/little,1:1/little>>,[129]},
   {<<<<1,2>>/binary>>,[1,2]},
   {<<<<1,2>>:1/binary>>,[1]},
   {<<4,3,<<1,2>>:1/binary>>,[4,3,1]},
   {<<(256*45+47)>>,[47]},
   {<<57:0>>,[]},
   {<<"apa">>,"apa"},
   {<<1:3,"string",9:5>>,[46,110,142,77,45,204,233]},
   {<<>>,[]}].

%%--------------------------------------------------------------------
%% Constructs a big enough binary to have a bit size that needs a
%% bignum on 32-bit architectures

large_bin(X1, X2, X3, X4) ->
  Sz = 16#4000000,
  <<1, <<X1:Sz, X2:Sz, X3:Sz, X4:Sz>>/bits>>.

%%--------------------------------------------------------------------
%% Test construction of "bad" binaries

-define(FAIL(Expr), {'EXIT', {badarg, _}} = (catch Expr)).

bad_ones() ->
  PI = math:pi(),
  ?FAIL(<<PI>>),
  Bin12 = <<1,2>>,
  ?FAIL(<<Bin12>>),
  E = 2.71,
  ?FAIL(<<E/binary>>),
  Int = 24334,
  ?FAIL(<<Int/binary>>),
  BigInt = 24334344294788947129487129487219847,
  ?FAIL(<<BigInt/binary>>),
  Bin123 = <<1,2,3>>,
  ?FAIL(<<Bin123/float>>),
  ok.
