/* -*- mode: C; c-file-style: "gnu"; indent-tabs-mode: nil; -*- */
/*
 * Copyright (C) 2011 Red Hat, Inc.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General
 * Public License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: David Zeuthen <davidz@redhat.com>
 */

#include "config.h"
#include <glib/gi18n.h>
#include <libnotify/notify.h>
#include <gio/gdesktopappinfo.h>
#include <rest/rest-proxy.h>
#include <libsoup/soup.h>

#include "goadaemon.h"
#include "goabackend/goabackend.h"

struct _GoaDaemon
{
  GObject parent_instance;

  GDBusConnection *connection;

  GFileMonitor *system_conf_dir_monitor;
  GFileMonitor *home_conf_file_monitor;
  GFileMonitor *home_conf_dir_monitor;

  GDBusObjectManagerServer *object_manager;

  GoaManager *manager;

  NotifyNotification *notification;

  guint config_timeout_id;
};

typedef struct
{
  GObjectClass parent_class;
} GoaDaemonClass;

static void on_file_monitor_changed (GFileMonitor     *monitor,
                                     GFile            *file,
                                     GFile            *other_file,
                                     GFileMonitorEvent event_type,
                                     gpointer          user_data);

static gboolean on_manager_handle_add_account (GoaManager            *object,
                                               GDBusMethodInvocation *invocation,
                                               const gchar           *provider,
                                               const gchar           *identity,
                                               const gchar           *presentation_identity,
                                               GVariant              *details,
                                               gpointer               user_data);

static gboolean on_account_handle_remove (GoaAccount            *account,
                                          GDBusMethodInvocation *invocation,
                                          gpointer               user_data);

static gboolean on_account_handle_ensure_credentials (GoaAccount            *account,
                                                      GDBusMethodInvocation *invocation,
                                                      gpointer               user_data);

static void goa_daemon_reload_configuration (GoaDaemon *daemon);

static void on_notification_closed (NotifyNotification *notification,
                                    gpointer            user_data);

static void account_on_attention_needed_notify (GObject     *object,
                                                GParamSpec  *pspec,
                                                gpointer     user_data);

G_DEFINE_TYPE (GoaDaemon, goa_daemon, G_TYPE_OBJECT);

static void
goa_daemon_finalize (GObject *object)
{
  GoaDaemon *daemon = GOA_DAEMON (object);

  if (daemon->config_timeout_id != 0)
    {
      g_source_remove (daemon->config_timeout_id);
    }

  if (daemon->notification != NULL)
    {
      g_signal_handlers_disconnect_by_func (daemon->notification,
                                            G_CALLBACK (on_notification_closed),
                                            daemon);
      g_object_unref (daemon->notification);
    }
  if (daemon->system_conf_dir_monitor != NULL)
    {
      g_signal_handlers_disconnect_by_func (daemon->system_conf_dir_monitor, on_file_monitor_changed, daemon);
      g_object_unref (daemon->system_conf_dir_monitor);
    }
  if (daemon->home_conf_dir_monitor != NULL)
    {
      g_signal_handlers_disconnect_by_func (daemon->home_conf_dir_monitor, on_file_monitor_changed, daemon);
      g_object_unref (daemon->home_conf_dir_monitor);
    }
  if (daemon->home_conf_file_monitor != NULL)
    {
      g_signal_handlers_disconnect_by_func (daemon->home_conf_file_monitor, on_file_monitor_changed, daemon);
      g_object_unref (daemon->home_conf_file_monitor);
    }

  g_object_unref (daemon->manager);
  g_object_unref (daemon->object_manager);
  g_object_unref (daemon->connection);

  G_OBJECT_CLASS (goa_daemon_parent_class)->finalize (object);
}

static GFileMonitor *
create_monitor (const gchar *path, gboolean is_dir)
{
  GFile *file;
  GFileMonitor *monitor;
  GError *error;

  error = NULL;
  file = g_file_new_for_path (path);
  if (is_dir)
    monitor = g_file_monitor_directory (file, G_FILE_MONITOR_NONE, NULL, &error);
  else
    monitor = g_file_monitor_file (file, G_FILE_MONITOR_NONE, NULL, &error);

  if (monitor == NULL)
    {
      goa_warning ("Error monitoring %s at %s: %s (%s, %d)",
                   is_dir ? "directory" : "file",
                   path,
                   error->message, g_quark_to_string (error->domain), error->code);
      g_error_free (error);
    }
  g_object_unref (file);

  return monitor;
}

static gboolean
on_config_file_monitor_timeout (gpointer user_data)
{
  GoaDaemon *daemon = GOA_DAEMON (user_data);

  daemon->config_timeout_id = 0;
  goa_info ("Reloading configuration files\n");
  goa_daemon_reload_configuration (daemon);

  return FALSE;
}

static void
on_file_monitor_changed (GFileMonitor      *monitor,
                         GFile             *file,
                         GFile             *other_file,
                         GFileMonitorEvent  event_type,
                         gpointer           user_data)
{
  GoaDaemon *daemon = GOA_DAEMON (user_data);

  if (daemon->config_timeout_id == 0)
    {
      daemon->config_timeout_id = g_timeout_add (200, on_config_file_monitor_timeout, daemon);
    }
}

static void
goa_daemon_init (GoaDaemon *daemon)
{
  static volatile GQuark goa_error_domain = 0;
  GoaObjectSkeleton *object;
  gchar *path;

  /* this will force associating errors in the GOA_ERROR error domain
   * with org.freedesktop.Goa.Error.* errors via g_dbus_error_register_error_domain().
   */
  goa_error_domain = GOA_ERROR;
  goa_error_domain; /* shut up -Wunused-but-set-variable */

  notify_init ("goa-daemon");

  /* TODO: maybe nicer to pass in a GDBusConnection* construct property */
  daemon->connection = g_bus_get_sync (G_BUS_TYPE_SESSION, NULL, NULL);

  /* Create object manager */
  daemon->object_manager = g_dbus_object_manager_server_new ("/org/gnome/OnlineAccounts");

  /* Create and export Manager */
  daemon->manager = goa_manager_skeleton_new ();
  g_signal_connect (daemon->manager,
                    "handle-add-account",
                    G_CALLBACK (on_manager_handle_add_account),
                    daemon);
  object = goa_object_skeleton_new ("/org/gnome/OnlineAccounts/Manager");
  goa_object_skeleton_set_manager (object, daemon->manager);
  g_dbus_object_manager_server_export (daemon->object_manager, G_DBUS_OBJECT_SKELETON (object));
  g_object_unref (object);

  /* create ~/.config/goa-1.0 directory */
  path = g_strdup_printf ("%s/goa-1.0", g_get_user_config_dir ());
  if (g_mkdir_with_parents (path, 0755) != 0)
    {
      goa_warning ("Error creating directory %s: %m", path);
    }
  g_free (path);

  /* set up file monitoring */
  path = g_strdup_printf ("%s/goa-1.0/accounts.conf", g_get_user_config_dir ());
  daemon->home_conf_file_monitor = create_monitor (path, FALSE);
  if (daemon->home_conf_file_monitor != NULL)
    g_signal_connect (daemon->home_conf_file_monitor, "changed", G_CALLBACK (on_file_monitor_changed), daemon);
  g_free (path);

  /* prime the list of accounts */
  goa_daemon_reload_configuration (daemon);

  /* Export objects */
  g_dbus_object_manager_server_set_connection (daemon->object_manager, daemon->connection);
}

static void
goa_daemon_class_init (GoaDaemonClass *klass)
{
  GObjectClass *gobject_class;

  gobject_class = G_OBJECT_CLASS (klass);
  gobject_class->finalize     = goa_daemon_finalize;
}

GoaDaemon *
goa_daemon_new (void)
{
  return GOA_DAEMON (g_object_new (GOA_TYPE_DAEMON, NULL));
}


/* ---------------------------------------------------------------------------------------------------- */

static void
diff_sorted_lists (GList *list1,
                   GList *list2,
                   GCompareFunc compare,
                   GList **added,
                   GList **removed,
                   GList **unchanged)
{
  gint order;

  *added = *removed = *unchanged = NULL;

  while (list1 != NULL && list2 != NULL)
    {
      order = (*compare) (list1->data, list2->data);
      if (order < 0)
        {
          *removed = g_list_prepend (*removed, list1->data);
          list1 = list1->next;
        }
      else if (order > 0)
        {
          *added = g_list_prepend (*added, list2->data);
          list2 = list2->next;
        }
      else
        { /* same item */
          *unchanged = g_list_prepend (*unchanged, list1->data);
          list1 = list1->next;
          list2 = list2->next;
        }
    }

  while (list1 != NULL)
    {
      *removed = g_list_prepend (*removed, list1->data);
      list1 = list1->next;
    }
  while (list2 != NULL)
    {
      *added = g_list_prepend (*added, list2->data);
      list2 = list2->next;
    }
}

/* ---------------------------------------------------------------------------------------------------- */

typedef struct
{
  GKeyFile *key_file;
  gchar *path;
} KeyFileData;

static void
key_file_data_free (KeyFileData *data)
{
  /* the key_file member is freed elsewhere */
  g_free (data->path);
  g_slice_free (KeyFileData, data);
}

static KeyFileData *
key_file_data_new (GKeyFile    *key_file,
                   const gchar *path)
{
  KeyFileData *data;
  data = g_slice_new (KeyFileData);
  data->key_file = key_file;
  data->path = g_strdup (path);
  return data;
}

/* ---------------------------------------------------------------------------------------------------- */

static void
add_config_file (const gchar   *path,
                 GHashTable    *group_name_to_key_file_data,
                 GList        **key_files_to_free)
{
  GKeyFile *key_file;
  GError *error;

  key_file = g_key_file_new ();

  error = NULL;
  if (!g_key_file_load_from_file (key_file,
                                  path,
                                  G_KEY_FILE_NONE,
                                  &error))
    {
      if (!(error->domain == G_FILE_ERROR && error->code == G_FILE_ERROR_NOENT))
        {
          goa_warning ("Error loading %s: %s (%s, %d)",
                       path,
                       error->message, g_quark_to_string (error->domain), error->code);
        }
      g_error_free (error);
      g_key_file_free (key_file);
    }
  else
    {
      gchar **groups;
      gsize num_groups;
      guint n;

      groups = g_key_file_get_groups (key_file, &num_groups);
      for (n = 0; n < num_groups; n++)
        {
          if (g_str_has_prefix (groups[n], "Account "))
            {
              g_hash_table_insert (group_name_to_key_file_data,
                                   groups[n], /* steals string */
                                   key_file_data_new (key_file, path));
            }
          else
            {
              goa_warning ("Unexpected group \"%s\" in file %s", groups[n], path);
              g_free (groups[n]);
            }
        }
      g_free (groups);

      *key_files_to_free = g_list_prepend (*key_files_to_free, key_file);
    }
}

/* ---------------------------------------------------------------------------------------------------- */

/* returns FALSE if object is not (or no longer) valid */
static gboolean
update_account_object (GoaDaemon           *daemon,
                       GoaObjectSkeleton   *object,
                       const gchar         *path,
                       const gchar         *group,
                       GKeyFile            *key_file,
                       gboolean             just_added)
{
  GoaAccount *account;
  GoaProvider *provider;
  gboolean ret;
  gchar *identity;
  gchar *presentation_identity;
  gchar *type;
  gchar *name;
  GIcon *icon;
  gchar *serialized_icon;
  GError *error;

  g_return_val_if_fail (GOA_IS_DAEMON (daemon), FALSE);
  g_return_val_if_fail (G_IS_DBUS_OBJECT_SKELETON (object), FALSE);
  g_return_val_if_fail (group != NULL, FALSE);
  g_return_val_if_fail (key_file != NULL, FALSE);

  ret = FALSE;
  identity = NULL;
  type = NULL;
  account = NULL;
  name = NULL;
  icon = NULL;
  serialized_icon = NULL;

  goa_debug ("updating %s %d", g_dbus_object_get_object_path (G_DBUS_OBJECT (object)), just_added);

  type = g_key_file_get_string (key_file, group, "Provider", NULL);
  identity = g_key_file_get_string (key_file, group, "Identity", NULL);
  presentation_identity = g_key_file_get_string (key_file, group, "PresentationIdentity", NULL);
  if (just_added)
    {
      account = goa_account_skeleton_new ();
      goa_object_skeleton_set_account (object, account);
    }
  else
    {
      account = goa_object_get_account (GOA_OBJECT (object));
    }

  provider = goa_provider_get_for_provider_type (type);
  if (provider == NULL)
    {
      goa_warning ("Unsupported account type %s for identity %s (no provider)", type, identity);
      goto out;
    }

  name = goa_provider_get_provider_name (provider, GOA_OBJECT (object));
  icon = goa_provider_get_provider_icon (provider, GOA_OBJECT (object));
  serialized_icon = g_icon_to_string (icon);

  goa_account_set_id (account, g_strrstr (g_dbus_object_get_object_path (G_DBUS_OBJECT (object)), "/") + 1);
  goa_account_set_provider_type (account, type);
  goa_account_set_provider_name (account, name);
  goa_account_set_provider_icon (account, serialized_icon);
  goa_account_set_identity (account, identity);
  goa_account_set_presentation_identity (account, presentation_identity);

  error = NULL;
  if (!goa_provider_build_object (provider, object, key_file, group, &error))
    {
      goa_warning ("Error parsing account: %s (%s, %d)",
                   error->message, g_quark_to_string (error->domain), error->code);
      g_error_free (error);
      goto out;
    }

  ret = TRUE;

 out:
  g_free (serialized_icon);
  if (icon != NULL)
    g_object_unref (icon);
  g_free (name);
  if (provider != NULL)
    g_object_unref (provider);
  g_object_unref (account);
  g_free (type);
  g_free (identity);
  return ret;
}

static gchar *
object_path_to_group (const gchar *object_path)
{
  g_return_val_if_fail (g_str_has_prefix (object_path, "/org/gnome/OnlineAccounts/Accounts/"), NULL);
  return g_strdup_printf ("Account %s", object_path + sizeof "/org/gnome/OnlineAccounts/Accounts/" - 1);
}

static void
process_config_entries (GoaDaemon  *daemon,
                        GHashTable *group_name_to_key_file_data)
{
  GHashTableIter iter;
  const gchar *id;
  KeyFileData *key_file_data;
  GList *existing_object_paths;
  GList *config_object_paths;
  GList *added;
  GList *removed;
  GList *unchanged;
  GList *l;

  existing_object_paths = NULL;
  {
    GList *existing_objects;
    existing_objects = g_dbus_object_manager_get_objects (G_DBUS_OBJECT_MANAGER (daemon->object_manager));
    for (l = existing_objects; l != NULL; l = l->next)
      {
        GoaObject *object = GOA_OBJECT (l->data);
        const gchar *object_path;
        object_path = g_dbus_object_get_object_path (G_DBUS_OBJECT (object));
        if (g_str_has_prefix (object_path, "/org/gnome/OnlineAccounts/Accounts/"))
          existing_object_paths = g_list_prepend (existing_object_paths, g_strdup (object_path));
      }
    g_list_foreach (existing_objects, (GFunc) g_object_unref, NULL);
    g_list_free (existing_objects);
  }

  config_object_paths = NULL;
  g_hash_table_iter_init (&iter, group_name_to_key_file_data);
  while (g_hash_table_iter_next (&iter, (gpointer*) &id, (gpointer*) &key_file_data))
    {
      gchar *object_path;

      /* create and validate object path */
      object_path = g_strdup_printf ("/org/gnome/OnlineAccounts/Accounts/%s", id + sizeof "Account " - 1);
      if (strstr (id + sizeof "Account " - 1, "/") != NULL || !g_variant_is_object_path (object_path))
        {
          goa_warning ("`%s' is not a valid account identifier", id);
          g_free (object_path);
          continue;
        }
      /* steals object_path variable */
      config_object_paths = g_list_prepend (config_object_paths, object_path);
    }

  existing_object_paths = g_list_sort (existing_object_paths, (GCompareFunc) g_strcmp0);
  config_object_paths = g_list_sort (config_object_paths, (GCompareFunc) g_strcmp0);
  diff_sorted_lists (existing_object_paths,
                     config_object_paths,
                     (GCompareFunc) g_strcmp0,
                     &added,
                     &removed,
                     &unchanged);

  for (l = removed; l != NULL; l = l->next)
    {
      const gchar *object_path = l->data;
      GoaObject *object;
      object = GOA_OBJECT (g_dbus_object_manager_get_object (G_DBUS_OBJECT_MANAGER (daemon->object_manager), object_path));
      g_warn_if_fail (object != NULL);
      g_signal_handlers_disconnect_by_func (goa_object_peek_account (object),
                                            G_CALLBACK (account_on_attention_needed_notify),
                                            daemon);
      g_signal_handlers_disconnect_by_func (goa_object_peek_account (object),
                                            G_CALLBACK (on_account_handle_remove),
                                            daemon);
      goa_debug ("removing %s", object_path);
      g_warn_if_fail (g_dbus_object_manager_server_unexport (daemon->object_manager, object_path));
    }
  for (l = added; l != NULL; l = l->next)
    {
      const gchar *object_path = l->data;
      GoaObjectSkeleton *object;
      gchar *group;

      goa_debug ("adding %s", object_path);

      group = object_path_to_group (object_path);
      key_file_data = g_hash_table_lookup (group_name_to_key_file_data, group);
      g_warn_if_fail (key_file_data != NULL);

      object = goa_object_skeleton_new (object_path);
      if (update_account_object (daemon,
                                 object,
                                 key_file_data->path,
                                 group,
                                 key_file_data->key_file,
                                 TRUE))
        {
          g_dbus_object_manager_server_export (daemon->object_manager, G_DBUS_OBJECT_SKELETON (object));
          g_signal_connect (goa_object_peek_account (GOA_OBJECT (object)),
                            "notify::attention-needed",
                            G_CALLBACK (account_on_attention_needed_notify),
                            daemon);
          g_signal_connect (goa_object_peek_account (GOA_OBJECT (object)),
                            "handle-remove",
                            G_CALLBACK (on_account_handle_remove),
                            daemon);
          g_signal_connect (goa_object_peek_account (GOA_OBJECT (object)),
                            "handle-ensure-credentials",
                            G_CALLBACK (on_account_handle_ensure_credentials),
                            daemon);
        }
      g_object_unref (object);
      g_free (group);
    }
  for (l = unchanged; l != NULL; l = l->next)
    {
      const gchar *object_path = l->data;
      GoaObject *object;
      gchar *group;

      goa_debug ("unchanged %s", object_path);

      group = object_path_to_group (object_path);
      key_file_data = g_hash_table_lookup (group_name_to_key_file_data, group);
      g_warn_if_fail (key_file_data != NULL);

      object = GOA_OBJECT (g_dbus_object_manager_get_object (G_DBUS_OBJECT_MANAGER (daemon->object_manager), object_path));
      g_warn_if_fail (object != NULL);
      if (!update_account_object (daemon,
                                  GOA_OBJECT_SKELETON (object),
                                  key_file_data->path,
                                  group,
                                  key_file_data->key_file,
                                  FALSE))
        {
          g_signal_handlers_disconnect_by_func (goa_object_peek_account (object),
                                                G_CALLBACK (account_on_attention_needed_notify),
                                                daemon);
          g_signal_handlers_disconnect_by_func (goa_object_peek_account (object),
                                                G_CALLBACK (on_account_handle_remove),
                                                daemon);
          g_signal_handlers_disconnect_by_func (goa_object_peek_account (object),
                                                G_CALLBACK (on_account_handle_ensure_credentials),
                                                daemon);
          g_warn_if_fail (g_dbus_object_manager_server_unexport (daemon->object_manager, object_path));
        }
      g_object_unref (object);
      g_free (group);
    }

  g_list_free (removed);
  g_list_free (added);
  g_list_free (unchanged);
  g_list_foreach (existing_object_paths, (GFunc) g_free, NULL);
  g_list_free (existing_object_paths);
  g_list_foreach (config_object_paths, (GFunc) g_free, NULL);
  g_list_free (config_object_paths);
}

/* <internal>
 * goa_daemon_reload_configuration:
 * @daemon: A #GoaDaemon
 *
 * Updates the accounts_objects member from stored configuration -
 * typically called at startup or when a change on the configuration
 * files has been detected.
 */
static void
goa_daemon_reload_configuration (GoaDaemon *daemon)
{
  GList *key_files_to_free;
  GHashTable *group_name_to_key_file_data;
  gchar *path;

  key_files_to_free = NULL;
  group_name_to_key_file_data = g_hash_table_new_full (g_str_hash,
                                                       g_str_equal,
                                                       g_free,
                                                       (GDestroyNotify) key_file_data_free);

  /* Read the main user config file at $HOME/.config/goa-1.0/accounts.conf */
  path = g_strdup_printf ("%s/goa-1.0/accounts.conf", g_get_user_config_dir ());
  add_config_file (path, group_name_to_key_file_data, &key_files_to_free);
  g_free (path);

  /* now process the group_name_to_key_file_data hash table */
  process_config_entries (daemon, group_name_to_key_file_data);

  g_hash_table_unref (group_name_to_key_file_data);
  g_list_foreach (key_files_to_free, (GFunc) g_key_file_free, NULL);
  g_list_free (key_files_to_free);
}

static gchar *
generate_new_id (GoaDaemon *daemon)
{
  GDateTime *dt;
  gchar *ret;

  dt = g_date_time_new_now_local ();
  ret = g_date_time_format (dt, "account_%s"); /* seconds since Epoch */
  /* TODO: handle collisions */
  g_date_time_unref (dt);

  return ret;
}

static gboolean
on_manager_handle_add_account (GoaManager             *manager,
                               GDBusMethodInvocation  *invocation,
                               const gchar            *provider,
                               const gchar            *identity,
                               const gchar            *presentation_identity,
                               GVariant               *details,
                               gpointer                user_data)
{
  GoaDaemon *daemon = GOA_DAEMON (user_data);
  GKeyFile *key_file;
  GError *error;
  gchar *path;
  gchar *id;
  gchar *group;
  gchar *data;
  gsize length;
  gchar *object_path;
  GVariantIter iter;
  const gchar *key;
  const gchar *value;

  /* TODO: could check for @type */

  key_file = NULL;
  path = NULL;
  id = NULL;
  group = NULL;
  data = NULL;
  object_path = NULL;

  key_file = g_key_file_new ();
  path = g_strdup_printf ("%s/goa-1.0/accounts.conf", g_get_user_config_dir ());
  error = NULL;
  if (!g_file_get_contents (path,
                            &data,
                            &length,
                            &error))
    {
      if (error->domain == G_FILE_ERROR && error->code == G_FILE_ERROR_NOENT)
        {
          g_error_free (error);
        }
      else
        {
          g_prefix_error (&error, "Error loading file %s: ", path);
          g_dbus_method_invocation_return_gerror (invocation, error);
          goto out;
        }
    }
  else
    {
      if (length > 0)
        {
          error = NULL;
          if (!g_key_file_load_from_data (key_file, data, length, G_KEY_FILE_KEEP_COMMENTS, &error))
            {
              g_prefix_error (&error, "Error parsing key-value-file %s: ", path);
              g_dbus_method_invocation_return_gerror (invocation, error);
              goto out;
            }
        }
    }

  id = generate_new_id (daemon);
  group = g_strdup_printf ("Account %s", id);
  g_key_file_set_string (key_file, group, "Provider", provider);
  g_key_file_set_string (key_file, group, "Identity", identity);
  g_key_file_set_string (key_file, group, "PresentationIdentity", presentation_identity);

  g_variant_iter_init (&iter, details);
  while (g_variant_iter_next (&iter, "{&s&s}", &key, &value))
    {
      g_key_file_set_string (key_file, group, key, value);
    }

  g_free (data);
  error = NULL;
  data = g_key_file_to_data (key_file,
                             &length,
                             &error);
  if (data == NULL)
    {
      g_prefix_error (&error, "Error generating key-value-file: ");
      g_dbus_method_invocation_return_gerror (invocation, error);
      goto out;
    }

  error = NULL;
  if (!g_file_set_contents (path,
                            data,
                            length,
                            &error))
    {
      g_prefix_error (&error, "Error writing key-value-file %s: ", path);
      g_dbus_method_invocation_return_gerror (invocation, error);
      goto out;
    }

  goa_daemon_reload_configuration (daemon);

  object_path = g_strdup_printf ("/org/gnome/OnlineAccounts/Accounts/%s", id);
  goa_manager_complete_add_account (manager, invocation, object_path);

 out:
  g_free (object_path);
  g_free (data);
  g_free (group);
  g_free (id);
  g_free (path);
  if (key_file != NULL)
    g_key_file_free (key_file);

  return TRUE; /* invocation was handled */
}

/* ---------------------------------------------------------------------------------------------------- */

static void
notification_cb (NotifyNotification *notification,
                 gchar              *action,
                 gpointer            user_data)
{
  GAppLaunchContext *ctx;
  GDesktopAppInfo *app;
  GError *error;

  /* TODO: Hmm, would be nice to set the screen, timestamp etc etc */
  ctx = g_app_launch_context_new ();

  app = g_desktop_app_info_new ("gnome-online-accounts-panel.desktop");

  error = NULL;
  if (!g_app_info_launch (G_APP_INFO (app),
                          NULL, /* files */
                          ctx,
                          &error))
    {
      goa_warning ("Error launching: %s (%s, %d)",
                   error->message, g_quark_to_string (error->domain), error->code);
      g_error_free (error);
    }
  g_object_unref (app);
  g_object_unref (ctx);
}

static void
on_notification_closed (NotifyNotification *notification,
                        gpointer            user_data)
{
  GoaDaemon *daemon = GOA_DAEMON (user_data);
  g_signal_handlers_disconnect_by_func (daemon->notification,
                                        G_CALLBACK (on_notification_closed),
                                        daemon);
  g_object_unref (daemon->notification);
  daemon->notification = NULL;
}

static void
goa_daemon_update_notifications (GoaDaemon *daemon)
{
  gboolean show_notification;
  GList *objects;
  GList *l;

  show_notification = FALSE;
  objects = g_dbus_object_manager_get_objects (G_DBUS_OBJECT_MANAGER (daemon->object_manager));
  for (l = objects; l != NULL; l = l->next)
    {
      GoaObject *object = GOA_OBJECT (l->data);
      GoaAccount *account;
      account = goa_object_peek_account (object);
      if (account != NULL)
        {
          if (goa_account_get_attention_needed (account))
            {
              show_notification = TRUE;
              break;
            }
        }
    }

  if (show_notification)
    {
      if (daemon->notification == NULL)
        {
          GError *error;

          daemon->notification = notify_notification_new (_("An online account needs attention"),
                                                          NULL,
                                                          NULL);
          notify_notification_set_timeout (daemon->notification, NOTIFY_EXPIRES_NEVER);
          g_object_set (daemon->notification, "icon-name", "gtk-dialog-error", NULL);
          notify_notification_add_action (daemon->notification,
                                          "open-online-accounts",
                                          _("Open Online Accounts..."),
                                          notification_cb,
                                          daemon,
                                          NULL); /* GFreeFunc */

          error = NULL;
          if (!notify_notification_show (daemon->notification, &error))
            {
              goa_warning ("Error showing notification: %s (%s, %d)",
                           error->message, g_quark_to_string (error->domain), error->code);
              g_error_free (error);
              g_object_unref (daemon->notification);
              daemon->notification = NULL;
            }
          else
            {
              g_signal_connect (daemon->notification,
                                "closed",
                                G_CALLBACK (on_notification_closed),
                                daemon);
            }
        }
    }
  else
    {
      if (daemon->notification != NULL)
        {
          GError *error;
          error = NULL;
          if (!notify_notification_close (daemon->notification, &error))
            {
              goa_warning ("Error closing notification: %s (%s, %d)",
                           error->message, g_quark_to_string (error->domain), error->code);
              g_error_free (error);
            }
          g_signal_handlers_disconnect_by_func (daemon->notification,
                                                G_CALLBACK (on_notification_closed),
                                                daemon);
          g_object_unref (daemon->notification);
          daemon->notification = NULL;
        }
    }

  g_list_foreach (objects, (GFunc) g_object_unref, NULL);
  g_list_free (objects);
}

static void
account_on_attention_needed_notify (GObject     *object,
                                    GParamSpec  *pspec,
                                    gpointer     user_data)
{
  GoaDaemon *daemon = GOA_DAEMON (user_data);
  goa_daemon_update_notifications (daemon);
}

/* ---------------------------------------------------------------------------------------------------- */

static gboolean
on_account_handle_remove (GoaAccount            *account,
                          GDBusMethodInvocation *invocation,
                          gpointer               user_data)
{
  GoaDaemon *daemon = GOA_DAEMON (user_data);
  GKeyFile *key_file;
  gchar *path;
  gchar *group;
  gchar *data;
  gsize length;
  GError *error;

  path = NULL;
  group = NULL;
  key_file = NULL;
  data = NULL;

  /* update key-file - right now we only support removing the account
   * if the entry is in ~/.config/goa-1.0/accounts.conf
   */

  key_file = g_key_file_new ();
  path = g_strdup_printf ("%s/goa-1.0/accounts.conf", g_get_user_config_dir ());

  error = NULL;
  if (!g_key_file_load_from_file (key_file,
                                  path,
                                  G_KEY_FILE_KEEP_COMMENTS,
                                  &error))
    {
      g_dbus_method_invocation_return_gerror (invocation, error);
      g_error_free (error);
      goto out;
    }

  group = g_strdup_printf ("Account %s", goa_account_get_id (account));

  error = NULL;
  if (!g_key_file_remove_group (key_file, group, &error))
    {
      g_dbus_method_invocation_return_gerror (invocation, error);
      g_error_free (error);
      goto out;
    }

  error = NULL;
  data = g_key_file_to_data (key_file,
                             &length,
                             &error);
  if (data == NULL)
    {
      g_prefix_error (&error, "Error generating key-value-file: ");
      g_dbus_method_invocation_return_gerror (invocation, error);
      goto out;
    }

  error = NULL;
  if (!g_file_set_contents (path,
                            data,
                            length,
                            &error))
    {
      g_prefix_error (&error, "Error writing key-value-file %s: ", path);
      g_dbus_method_invocation_return_gerror (invocation, error);
      goto out;
    }

  goa_daemon_reload_configuration (daemon);

  goa_account_complete_remove (account, invocation);

 out:
  g_free (data);
  if (key_file != NULL)
    g_key_file_free (key_file);
  g_free (group);
  g_free (path);
  return TRUE; /* invocation was handled */
}

/* ---------------------------------------------------------------------------------------------------- */

typedef struct
{
  GoaDaemon *daemon;
  GoaObject *object;
  GDBusMethodInvocation *invocation;
} EnsureCredentialsData;

static EnsureCredentialsData *
ensure_credentials_data_new (GoaDaemon             *daemon,
                             GoaObject             *object,
                             GDBusMethodInvocation *invocation)
{
  EnsureCredentialsData *data;
  data = g_slice_new0 (EnsureCredentialsData);
  data->daemon = g_object_ref (daemon);
  data->object = g_object_ref (object);
  data->invocation = invocation;
  return data;
}

static void
ensure_credentials_data_unref (EnsureCredentialsData *data)
{
  g_object_unref (data->daemon);
  g_object_unref (data->object);
  g_slice_free (EnsureCredentialsData, data);
}

static gboolean
is_authorization_error (GError *error)
{
  gboolean ret;

  g_return_val_if_fail (error != NULL, FALSE);

  ret = FALSE;
  if (error->domain == REST_PROXY_ERROR || error->domain == SOUP_HTTP_ERROR)
    {
      if (SOUP_STATUS_IS_CLIENT_ERROR (error->code))
        ret = TRUE;
    }
  else if (error->domain == GOA_ERROR)
    {
      if (error->code == GOA_ERROR_NOT_AUTHORIZED)
        ret = TRUE;
    }
  return ret;
}

static void
ensure_credentials_cb (GoaProvider   *provider,
                       GAsyncResult  *res,
                       gpointer       user_data)
{
  EnsureCredentialsData *data = user_data;
  gint expires_in;
  GError *error;

  error= NULL;
  if (!goa_provider_ensure_credentials_finish (provider, &expires_in, res, &error))
    {
      /* Set AttentionNeeded only if the error is an authorization error */
      if (is_authorization_error (error))
        {
          GoaAccount *account;
          account = goa_object_peek_account (data->object);
          if (!goa_account_get_attention_needed (account))
            {
              goa_account_set_attention_needed (account, TRUE);
              g_dbus_interface_skeleton_flush (G_DBUS_INTERFACE_SKELETON (account));
              goa_notice ("%s: Setting AttentionNeeded to TRUE because EnsureCredentials() failed with: %s (%s, %d)",
                          g_dbus_object_get_object_path (G_DBUS_OBJECT (data->object)),
                          error->message, g_quark_to_string (error->domain), error->code);
            }
        }
      g_dbus_method_invocation_return_gerror (data->invocation, error);
      g_error_free (error);
    }
  else
    {
      GoaAccount *account;
      account = goa_object_peek_account (data->object);

      /* Clear AttentionNeeded flag if set */
      if (goa_account_get_attention_needed (account))
        {
          goa_account_set_attention_needed (account, FALSE);
          g_dbus_interface_skeleton_flush (G_DBUS_INTERFACE_SKELETON (account));
          goa_notice ("%s: Setting AttentionNeeded to FALSE because EnsureCredentials() succeded\n",
                      g_dbus_object_get_object_path (G_DBUS_OBJECT (data->object)));
        }
      goa_account_complete_ensure_credentials (goa_object_peek_account (data->object),
                                               data->invocation,
                                               expires_in);
    }
  ensure_credentials_data_unref (data);
}

static gboolean
on_account_handle_ensure_credentials (GoaAccount            *account,
                                      GDBusMethodInvocation *invocation,
                                      gpointer               user_data)
{
  GoaDaemon *daemon = GOA_DAEMON (user_data);
  GoaProvider *provider;
  GoaObject *object;

  object = GOA_OBJECT (g_dbus_interface_get_object (G_DBUS_INTERFACE (account)));
  provider = goa_provider_get_for_provider_type (goa_account_get_provider_type (account));
  if (provider == NULL)
    {
      g_dbus_method_invocation_return_error (invocation,
                                             GOA_ERROR,
                                             GOA_ERROR_FAILED,
                                             "Unsupported account type %s for id %s (no provider)",
                                             goa_account_get_provider_type (account),
                                             goa_account_get_id (account));
      goto out;
    }

  goa_provider_ensure_credentials (provider,
                                   object,
                                   NULL, /* GCancellable */
                                   (GAsyncReadyCallback) ensure_credentials_cb,
                                   ensure_credentials_data_new (daemon, object, invocation));

 out:
  return TRUE; /* invocation was handled */
}

