/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */
/*
 * This file is part of the LibreOffice project.
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 * This file incorporates work covered by the following license notice:
 *
 *   Licensed to the Apache Software Foundation (ASF) under one or more
 *   contributor license agreements. See the NOTICE file distributed
 *   with this work for additional information regarding copyright
 *   ownership. The ASF licenses this file to you under the Apache
 *   License, Version 2.0 (the "License"); you may not use this file
 *   except in compliance with the License. You may obtain a copy of
 *   the License at http://www.apache.org/licenses/LICENSE-2.0 .
 */

#ifndef INCLUDED_SW_INC_VIEWOPT_HXX
#define INCLUDED_SW_INC_VIEWOPT_HXX

#include <config_features.h>

#include <tools/gen.hxx>
#include <tools/color.hxx>

#include <sfx2/zoomitem.hxx>
#include <svx/svxids.hrc>
#include "swdllapi.h"
#include "authratr.hxx"

class SwRect;
namespace vcl { class Window; }
class OutputDevice;
class SwViewShell;
class SwDocShell;
namespace svtools{ class ColorConfig;}

enum class ViewOptFlags1 {
    Tab           = 0x00000002,
    Blank         = 0x00000004,
    HardBlank     = 0x00000008,
    Paragraph     = 0x00000010,
    Linebreak     = 0x00000020,
    Pagebreak     = 0x00000040,
    Columnbreak   = 0x00000080,
    SoftHyph      = 0x00000100,
    Ref           = 0x00000400,
    FieldName     = 0x00000800,
    Postits       = 0x00004000,
    FieldHidden   = 0x00008000,
    CharHidden    = 0x00010000,
    Graphic       = 0x00020000,
    Table         = 0x00040000,
    Draw          = 0x00080000,
    Control       = 0x00100000,
    Crosshair     = 0x00400000,
    Snap          = 0x00800000,
    Synchronize   = 0x01000000,
    GridVisible   = 0x02000000,
    OnlineSpell   = 0x04000000,
    ViewMetachars = 0x20000000,
    Pageback      = 0x40000000
};
namespace o3tl {
    template<> struct typed_flags<ViewOptFlags1> : is_typed_flags<ViewOptFlags1, 0x67dfcdfe> {};
}

enum class ViewOptCoreFlags2 {
    BlackFont         = 0x0001,
    HiddenPara        = 0x0002,
    SmoothScroll      = 0x0004,
    CursorInProt      = 0x0008,
    PdfExport         = 0x0010,
    Printing          = 0x0020,
};
namespace o3tl {
    template<> struct typed_flags<ViewOptCoreFlags2> : is_typed_flags<ViewOptCoreFlags2, 0x007f> {};
};

enum class ViewOptFlags2 {
    HRuler          = 0x00000400,
    VScrollbar      = 0x00000800,
    HScrollbar      = 0x00001000,
    VRuler          = 0x00004000,
    AnyRuler        = 0x00008000,
    Modified        = 0x00010000,
    KeepAspectRatio = 0x00020000,
    GrfKeepZoom     = 0x00040000,
    ContentTips     = 0x00100000,
    ScrollbarTips   = 0x00400000,
    PrintFormat     = 0x00800000,
    ShadowCursor    = 0x01000000,
    VRulerRight     = 0x02000000
};
namespace o3tl {
    template<> struct typed_flags<ViewOptFlags2> : is_typed_flags<ViewOptFlags2, 0x03d7dc00> {};
};

// Table background.
#define TBL_DEST_CELL   0
#define TBL_DEST_ROW    1
#define TBL_DEST_TBL    2

// Appearance flags.
enum class ViewOptFlags {
    NONE               = 0x0000,
    DocBoundaries      = 0x0001,
    ObjectBoundaries   = 0x0002,
    TableBoundaries    = 0x0004,
    IndexShadings      = 0x0008,
    Links              = 0x0010,
    VisitedLinks       = 0x0020,
    FieldShadings      = 0x0040,
    SectionBoundaries  = 0x0080,
    Shadow             = 0x0100,
};
namespace o3tl {
    template<> struct typed_flags<ViewOptFlags> : is_typed_flags<ViewOptFlags, 0x01ff> {};
}

class SW_DLLPUBLIC SwViewOption
{
    static Color    m_aDocColor;  // color of document boundaries
    static Color    m_aDocBoundColor;  // color of document boundaries
    static Color    m_aObjectBoundColor; // color of object boundaries
    static Color    m_aAppBackgroundColor; // application background
    static Color    m_aTableBoundColor; // color of table boundaries
    static Color    m_aFontColor;
    static Color    m_aIndexShadingsColor; // background color of indexes
    static Color    m_aLinksColor;
    static Color    m_aVisitedLinksColor;
    static Color    m_aDirectCursorColor;
    static Color    m_aTextGridColor;
    static Color    m_aSpellColor;     // mark color of online spell checking
    static Color    m_aSmarttagColor;
    static Color    m_aFieldShadingsColor;
    static Color    m_aSectionBoundColor;
    static Color    m_aPageBreakColor;
    static Color    m_aScriptIndicatorColor;
    static Color    m_aShadowColor;
    static Color    m_aHeaderFooterMarkColor;

    static ViewOptFlags m_nAppearanceFlags;
protected:
    static sal_uInt16   m_nPixelTwips;// 1 Pixel == ? Twips

    OUString        m_sSymbolFont;        // Symbolfont.
    ViewOptFlags1   m_nCoreOptions;       // Bits for SwViewShell.
    ViewOptCoreFlags2 m_nCore2Options;    // Bits for SwViewShell.
    ViewOptFlags2   m_nUIOptions;         // UI-Bits
    Color           m_aRetouchColor;      // DefaultBackground for BrowseView
    Size            m_aSnapSize;          // Describes horizontal and vertical snap.
    sal_uInt16      mnViewLayoutColumns;  // # columns for edit view
    short           m_nDivisionX;         // Grid division.
    short           m_nDivisionY;
    sal_uInt8       m_nPagePreviewRow;       // Page Preview Row/Columns.
    sal_uInt8       m_nPagePreviewCol;       // Page Preview Row/Columns.
    sal_uInt8       m_nShadowCursorFillMode;  // FillMode for ShadowCursor.
    bool            m_bReadonly : 1;      // Readonly-Doc.
    bool            m_bStarOneSetting : 1;// Prevent from UI automatics (no scrollbars in readonly documents).
    bool            m_bIsPagePreview : 1; // The preview mustn't print field/footnote/... shadings.
    bool            m_bSelectionInReadonly : 1; // Determines whether selection is switched on in readonly documents.
    bool            mbFormView : 1;
    bool            mbBrowseMode : 1;
    bool            mbBookView : 1;      // View mode for page preview.
    bool            mbViewLayoutBookMode : 1; // Book view mode for edit view.
    bool            mbHideWhitespaceMode : 1; // Hide header, footer, and pagebreak.
    bool            m_bShowPlaceHolderFields : 1; // Only used in printing!
    mutable bool    m_bIdle;

    // Scale
    sal_uInt16          m_nZoom;          // In percent.
    SvxZoomType     m_eZoom;              // 'enum' for zoom.

    sal_uInt8            m_nTableDestination;      // Destination for table background.

#ifdef DBG_UTIL
    // Corresponds to statements in ui/config/cfgvw.src.
    bool  m_bTest1        :1;     // Test-flag  "Layout not loading"
    bool  m_bTest2        :1;     // Test-flag  "WYSIWYG++"
    bool  m_bTest3        :1;     // Test-flag  ""
    bool  m_bTest4        :1;     // Test-flag  "WYSIWYG debug"
    bool  m_bTest5        :1;     // Test-flag  "No idle format"
    bool  m_bTest6        :1;     // Test-flag  "No screen adj"
    bool  m_bTest7        :1;     // Test-flag  "win format"
    bool  m_bTest8        :1;     // Test-flag  ""
    static bool  m_bTest9;    // Test-Flag  "DrawingLayerNotLoading"
    bool  m_bTest10       :1;     // Test-Flag  "Format by Input"
#endif

public:
            SwViewOption();                     // CTOR
            SwViewOption(const SwViewOption&);
            ~SwViewOption();

    static void Init( vcl::Window *pWin );        // Initializing of static data.

    inline ViewOptFlags1   GetCoreOptions() const {return m_nCoreOptions;}
    inline void     SetUIOptions( const SwViewOption& );

    // Options from nCoreOptions
    inline bool IsIdle() const
        { return m_bIdle; }

    // Logically this is a const function since it does not modify the viewoptions
    // but only effects idle formatting. Of course that member is already implement
    // in the wrong place here... Also currently there are many const modifying casts in the code
    // just to call this function on otherwise const objects. Thus declaring it as const now.
    inline void SetIdle( bool b ) const
        { m_bIdle = b; }

    inline bool IsTab(bool bHard = false) const
                    {   return !m_bReadonly && (m_nCoreOptions & ViewOptFlags1::Tab) &&
                            ((m_nCoreOptions & ViewOptFlags1::ViewMetachars)||bHard); }
    inline void SetTab( bool b )        {
        b ? (m_nCoreOptions |= ViewOptFlags1::Tab ) : ( m_nCoreOptions &= ~ViewOptFlags1::Tab); }

    inline bool IsBlank(bool bHard = false) const
                    { return !m_bReadonly && (m_nCoreOptions & ViewOptFlags1::Blank) &&
                            ((m_nCoreOptions & ViewOptFlags1::ViewMetachars)||bHard); }
    inline void SetBlank( bool b )
        { b ? (m_nCoreOptions |= ViewOptFlags1::Blank ) : ( m_nCoreOptions &= ~ViewOptFlags1::Blank); }

    inline bool IsHardBlank() const
                    { return !m_bReadonly && (m_nCoreOptions & ViewOptFlags1::HardBlank); }
    inline void SetHardBlank( bool b )
        { b ? (m_nCoreOptions |= ViewOptFlags1::HardBlank ) : ( m_nCoreOptions &= ~ViewOptFlags1::HardBlank); }

    inline bool IsParagraph(bool bHard = false) const
                    {   return !m_bReadonly && (m_nCoreOptions & ViewOptFlags1::Paragraph) &&
                            ((m_nCoreOptions & ViewOptFlags1::ViewMetachars)||bHard); }
    inline void SetParagraph( bool b )
        { b ? (m_nCoreOptions |= ViewOptFlags1::Paragraph ) : ( m_nCoreOptions &= ~ViewOptFlags1::Paragraph); }

    inline bool IsLineBreak(bool bHard = false) const
                    {   return !m_bReadonly && (m_nCoreOptions & ViewOptFlags1::Linebreak) &&
                            ((m_nCoreOptions & ViewOptFlags1::ViewMetachars)||bHard); }
    inline void SetLineBreak( bool b )
        { b ? (m_nCoreOptions |= ViewOptFlags1::Linebreak ) : ( m_nCoreOptions &= ~ViewOptFlags1::Linebreak); }

    inline void SetPageBreak( bool b )
        { b ? (m_nCoreOptions |= ViewOptFlags1::Pagebreak ) : ( m_nCoreOptions &= ~ViewOptFlags1::Pagebreak); }

    inline void SetColumnBreak( bool b)
        { b ? (m_nCoreOptions |= ViewOptFlags1::Columnbreak ) : ( m_nCoreOptions &= ~ViewOptFlags1::Columnbreak); }

    inline bool IsSoftHyph() const
                    { return !m_bReadonly && (m_nCoreOptions & ViewOptFlags1::SoftHyph); }
    inline void SetSoftHyph( bool b )
        { b ? (m_nCoreOptions |= ViewOptFlags1::SoftHyph ) : ( m_nCoreOptions &= ~ViewOptFlags1::SoftHyph); }

    inline bool IsFieldName() const       { return !m_bReadonly && (m_nCoreOptions & ViewOptFlags1::FieldName); }
    inline void SetFieldName( bool b )
        { b ? (m_nCoreOptions |= ViewOptFlags1::FieldName ) : ( m_nCoreOptions &= ~ViewOptFlags1::FieldName); }

    inline bool IsPostIts() const
        { return bool(m_nCoreOptions & ViewOptFlags1::Postits); }
    inline void SetPostIts( bool b )
        { b ? (m_nCoreOptions |= ViewOptFlags1::Postits ) : ( m_nCoreOptions &= ~ViewOptFlags1::Postits); }
    static void PaintPostIts( OutputDevice *pOut, const SwRect &rRect,
                              bool bIsScript );
    static sal_uInt16 GetPostItsWidth( const OutputDevice *pOut );

    inline bool IsShowHiddenChar(bool bHard = false) const
        { return !m_bReadonly && (m_nCoreOptions & ViewOptFlags1::CharHidden) &&
                            ((m_nCoreOptions & ViewOptFlags1::ViewMetachars)||bHard); }

    inline void SetShowHiddenChar( bool b )
        { b ? (m_nCoreOptions |= ViewOptFlags1::CharHidden ) : ( m_nCoreOptions &= ~ViewOptFlags1::CharHidden); }

    inline bool IsShowHiddenField() const
        { return !m_bReadonly && (m_nCoreOptions & ViewOptFlags1::FieldHidden); }
    inline void SetShowHiddenField( bool b )
        { b ? (m_nCoreOptions |= ViewOptFlags1::FieldHidden ) : ( m_nCoreOptions &= ~ViewOptFlags1::FieldHidden); }

    inline bool IsGraphic() const
        { return bool(m_nCoreOptions & ViewOptFlags1::Graphic); }
    inline void SetGraphic( bool b )
        { b ? (m_nCoreOptions |= ViewOptFlags1::Graphic ) : ( m_nCoreOptions &= ~ViewOptFlags1::Graphic); }

    inline bool IsPageBack() const
        { return bool(m_nCoreOptions & ViewOptFlags1::Pageback); }
    inline void SetPageBack( bool b )
        { b ? (m_nCoreOptions |= ViewOptFlags1::Pageback) : ( m_nCoreOptions &= ~ViewOptFlags1::Pageback); }

    inline bool IsTable() const
        { return bool(m_nCoreOptions & ViewOptFlags1::Table); }
    inline void SetTable( bool b )
        { b ? (m_nCoreOptions |= ViewOptFlags1::Table ) : ( m_nCoreOptions &= ~ViewOptFlags1::Table); }

    inline bool IsDraw() const
        { return bool(m_nCoreOptions & ViewOptFlags1::Draw); }
    inline void SetDraw( bool b )
        { b ? (m_nCoreOptions |= ViewOptFlags1::Draw ) : ( m_nCoreOptions &= ~ViewOptFlags1::Draw); }

    inline bool IsControl() const
        { return bool(m_nCoreOptions & ViewOptFlags1::Control); }
    inline void SetControl( bool b )
        { b ? (m_nCoreOptions |= ViewOptFlags1::Control ) : ( m_nCoreOptions &= ~ViewOptFlags1::Control); }

    inline bool IsSnap() const
        { return bool(m_nCoreOptions & ViewOptFlags1::Snap); }
    inline void SetSnap( bool b )
        { b ? (m_nCoreOptions |= ViewOptFlags1::Snap ) : ( m_nCoreOptions &= ~ViewOptFlags1::Snap); }

    inline void SetSnapSize( Size &rSz ){ m_aSnapSize = rSz; }
    inline const Size &GetSnapSize() const { return m_aSnapSize; }

    inline bool IsGridVisible() const
        { return !m_bReadonly && (m_nCoreOptions & ViewOptFlags1::GridVisible); }
    inline void SetGridVisible( bool b )
        { b ? (m_nCoreOptions |= ViewOptFlags1::GridVisible ) : ( m_nCoreOptions &= ~ViewOptFlags1::GridVisible); }

    inline bool IsOnlineSpell() const
        { return !m_bReadonly && (m_nCoreOptions & ViewOptFlags1::OnlineSpell); }
    void SetOnlineSpell( bool b );

    inline bool IsViewMetaChars() const
        { return !m_bReadonly && (m_nCoreOptions & ViewOptFlags1::ViewMetachars); }
    inline void SetViewMetaChars( bool b)
        { b ? (m_nCoreOptions |= ViewOptFlags1::ViewMetachars ) : ( m_nCoreOptions &= ~ViewOptFlags1::ViewMetachars); }

    inline bool IsSynchronize() const
        {  return bool(m_nCoreOptions & ViewOptFlags1::Synchronize); }
    inline void SetSynchronize( bool b )
        { b ? (m_nCoreOptions |= ViewOptFlags1::Synchronize ) : ( m_nCoreOptions &= ~ViewOptFlags1::Synchronize); }

    inline bool IsCrossHair() const
        { return bool(m_nCoreOptions & ViewOptFlags1::Crosshair); }
    inline void SetCrossHair( bool b )
        { b ? (m_nCoreOptions |= ViewOptFlags1::Crosshair ) : ( m_nCoreOptions &= ~ViewOptFlags1::Crosshair); }

    // Options from nCore2Options
    inline bool IsBlackFont() const
        {return bool(m_nCore2Options & ViewOptCoreFlags2::BlackFont); }

    inline void SetBlackFont(bool b)
        { b ? (m_nCore2Options |= ViewOptCoreFlags2::BlackFont) : (m_nCore2Options &= ~ViewOptCoreFlags2::BlackFont);}

    inline bool IsShowHiddenPara() const
        {return bool(m_nCore2Options & ViewOptCoreFlags2::HiddenPara); }

    inline void SetShowHiddenPara(bool b)
        { b ? (m_nCore2Options |= ViewOptCoreFlags2::HiddenPara) : (m_nCore2Options &= ~ViewOptCoreFlags2::HiddenPara);}

    inline bool IsSmoothScroll() const
        {return bool(m_nCore2Options & ViewOptCoreFlags2::SmoothScroll); }

    inline void SetSmoothScroll(bool b)
        { b ? (m_nCore2Options |= ViewOptCoreFlags2::SmoothScroll) : (m_nCore2Options &= ~ViewOptCoreFlags2::SmoothScroll);}

    inline bool IsCursorInProtectedArea() const
        {return bool(m_nCore2Options & ViewOptCoreFlags2::CursorInProt); }

    inline void SetCursorInProtectedArea(bool b)
        { b ? (m_nCore2Options |= ViewOptCoreFlags2::CursorInProt) : (m_nCore2Options &= ~ViewOptCoreFlags2::CursorInProt);}

    static bool IsIgnoreProtectedArea();

    inline bool IsPDFExport() const
        {return bool(m_nCore2Options & ViewOptCoreFlags2::PdfExport); }

    inline void SetPDFExport(bool b)
        { b ? (m_nCore2Options |= ViewOptCoreFlags2::PdfExport) : (m_nCore2Options &= ~ViewOptCoreFlags2::PdfExport);}

    inline bool IsPrinting() const
        {return bool(m_nCore2Options & ViewOptCoreFlags2::Printing); }

    inline void SetPrinting(bool b)
        { b ? (m_nCore2Options |= ViewOptCoreFlags2::Printing) : (m_nCore2Options &= ~ViewOptCoreFlags2::Printing);}

    inline short GetDivisionX() const   { return m_nDivisionX; }
    inline void  SetDivisionX( short n ){ m_nDivisionX = n; }
    inline short GetDivisionY() const   { return m_nDivisionY; }
    inline void  SetDivisionY( short n ){ m_nDivisionY = n; }

    // Default margin left and above document: 284 twips == 5.0 mm.
    static SAL_CONSTEXPR sal_uInt16 GetDefDocumentBorder() { return 284; }
    // Default gap between pages: 284 twips == 5.0 mm.
    static SAL_CONSTEXPR sal_uInt16 GetDefGapBetweenPages() { return 284; }
    // Minimum edge-to-text distance: 22 twips == 0.4 mm.
    static SAL_CONSTEXPR sal_uInt16 GetMinGapBetweenPages() { return 22; }

    inline sal_uInt16 GetDocumentBorder() const { return IsWhitespaceHidden() ? GetMinGapBetweenPages() : GetDefDocumentBorder(); }
    inline sal_uInt16 GetGapBetweenPages() const { return IsWhitespaceHidden() ? GetMinGapBetweenPages() : GetDefGapBetweenPages(); }

    inline sal_uInt8  GetPagePrevRow() const      { return m_nPagePreviewRow; }
    inline void  SetPagePrevRow( sal_uInt8 n ) { m_nPagePreviewRow = n; }
    inline sal_uInt8  GetPagePrevCol() const      { return m_nPagePreviewCol; }
    inline void  SetPagePrevCol( sal_uInt8 n ) { m_nPagePreviewCol = n; }
    bool         IsReadonly() const { return m_bReadonly; }
    void         SetReadonly(bool bSet) { m_bReadonly = bSet; }

    bool         IsSelectionInReadonly() const {return m_bSelectionInReadonly;}
    void         SetSelectionInReadonly(bool bSet) {m_bSelectionInReadonly = bSet;}

    bool         IsFormView() const { return mbFormView; }
    void         SetFormView( bool bSet ) { mbFormView = bSet; }

    inline bool  getBrowseMode() const { return mbBrowseMode; }
    inline void  setBrowseMode(bool bSet) { mbBrowseMode = bSet; }
    inline bool  IsPagePrevBookview() const { return mbBookView; }
    inline void  SetPagePrevBookview(bool bSet) { mbBookView = bSet; }

    static bool IsAutoCompleteWords();

    bool   IsViewLayoutBookMode() const { return mbViewLayoutBookMode; }
    void   SetViewLayoutBookMode( bool bNew ) { mbViewLayoutBookMode = bNew; }
    sal_uInt16 GetViewLayoutColumns() const { return mnViewLayoutColumns; }
    void   SetViewLayoutColumns( sal_uInt16 nNew ) { mnViewLayoutColumns = nNew; }
    bool   IsHideWhitespaceMode() const { return mbHideWhitespaceMode; }
    void   SetHideWhitespaceMode( bool bMode ) { mbHideWhitespaceMode = bMode; }

    /// The rules that allow for hiding whitespace.
    bool   CanHideWhitespace() const { return !IsMultipageView(); }
    bool   IsWhitespaceHidden() const { return IsHideWhitespaceMode() && !IsMultipageView(); }
    bool   IsMultipageView() const { return IsViewLayoutBookMode() ||
                                            GetViewLayoutColumns() == 0; }

#ifdef DBG_UTIL
    // Correspond to statements in ui/config/cfgvw.src.
    inline bool IsTest1() const     { return m_bTest1; }
    inline bool IsTest2() const     { return m_bTest2; }
    inline bool IsTest3() const     { return m_bTest3; }
    inline bool IsTest4() const     { return m_bTest4; }
    inline bool IsTest5() const     { return m_bTest5; }
    inline bool IsTest6() const     { return m_bTest6; }
    inline bool IsTest7() const     { return m_bTest7; }
    inline bool IsTest8() const     { return m_bTest8; }
    inline bool IsTest10() const    { return m_bTest10; }
#endif

    inline sal_uInt16 GetZoom() const    { return m_nZoom; }
    inline void   SetZoom( sal_uInt16 n ){ m_nZoom = n; }

    static void DrawRect( OutputDevice* pOut, const SwRect &rRect, long nCol );
    static void DrawRectPrinter( OutputDevice* pOut, const SwRect& rRect );

    SwViewOption& operator=( const SwViewOption &rOpt );

    // Compare methods.
    bool IsEqualFlags ( const SwViewOption &rOpt ) const;
    inline bool operator == ( const SwViewOption &rOpt ) const;
    inline bool operator != ( const SwViewOption &rOpt ) const  { return !(*this == rOpt); }

    // Options from nUIOptions
    bool    IsViewVScrollBar() const
        {
#if HAVE_FEATURE_DESKTOP
            return bool(m_nUIOptions & ViewOptFlags2::VScrollbar);
#else
            return false;
#endif
        }
    bool    IsViewHScrollBar() const
        {
#if HAVE_FEATURE_DESKTOP
            return bool(m_nUIOptions & ViewOptFlags2::HScrollbar);
#else
            return false;
#endif
        }
    bool    IsKeepRatio()      const
        { return bool(m_nUIOptions & ViewOptFlags2::KeepAspectRatio); }
    bool    IsGrfKeepZoom()    const
        { return bool(m_nUIOptions & ViewOptFlags2::GrfKeepZoom); }
    bool    IsShowContentTips() const
        { return bool(m_nUIOptions & ViewOptFlags2::ContentTips); }
    bool    IsPrtFormat() const
        { return bool(m_nUIOptions & ViewOptFlags2::PrintFormat); }
    bool    IsShowScrollBarTips() const
        { return bool(m_nUIOptions & ViewOptFlags2::ScrollbarTips); }

    SvxZoomType    GetZoomType()      const { return m_eZoom; }

    sal_uInt8   GetTableDest() const    { return m_nTableDestination; }

    void   SetViewVScrollBar(bool b)
        { b ? (m_nUIOptions |= ViewOptFlags2::VScrollbar ) : ( m_nUIOptions &= ~ViewOptFlags2::VScrollbar); }
    void   SetViewHScrollBar(bool b)
        { b ? (m_nUIOptions |= ViewOptFlags2::HScrollbar ) : ( m_nUIOptions &= ~ViewOptFlags2::HScrollbar); }
    void   SetKeepRatio     (bool b)
        { b ? (m_nUIOptions |= ViewOptFlags2::KeepAspectRatio ) : ( m_nUIOptions &= ~ViewOptFlags2::KeepAspectRatio); }
    void   SetGrfKeepZoom   (bool b)
        { b ? (m_nUIOptions |= ViewOptFlags2::GrfKeepZoom ) : ( m_nUIOptions &= ~ViewOptFlags2::GrfKeepZoom); }
    void SetShowContentTips( bool b)
        { b ? (m_nUIOptions |= ViewOptFlags2::ContentTips) : (m_nUIOptions &= ~ViewOptFlags2::ContentTips); }
    void SetPrtFormat( bool b)
        { b ? (m_nUIOptions |= ViewOptFlags2::PrintFormat) : (m_nUIOptions &= ~ViewOptFlags2::PrintFormat); }
    void SetShowScrollBarTips( bool b)
        { b ? (m_nUIOptions |= ViewOptFlags2::ScrollbarTips) : (m_nUIOptions &= ~ViewOptFlags2::ScrollbarTips); }

    void            SetZoomType     (SvxZoomType eZoom_){ m_eZoom = eZoom_;  }
    void            SetTableDest( sal_uInt8 nNew )    { m_nTableDestination = nNew;  }

    const OUString& GetSymbolFont() const {return m_sSymbolFont;}
    void            SetSymbolFont(const OUString& sSet) {m_sSymbolFont = sSet;}

    const Color&    GetRetoucheColor() const        { return m_aRetouchColor;}
    void            SetRetoucheColor(const Color&r) { m_aRetouchColor = r;   }

    bool        IsViewAnyRuler() const
        {
#if HAVE_FEATURE_DESKTOP
            return bool(m_nUIOptions & ViewOptFlags2::AnyRuler);
#else
            return false;
#endif
        }
    void            SetViewAnyRuler(bool bSet)
                        { bSet ? (m_nUIOptions |= ViewOptFlags2::AnyRuler) : (m_nUIOptions &= ~ViewOptFlags2::AnyRuler);}

    bool        IsViewHRuler(bool bDirect = false)     const
                        {
#if HAVE_FEATURE_DESKTOP
                            return bDirect
                                   ? bool(m_nUIOptions & ViewOptFlags2::HRuler)
                                   : !m_bReadonly && (m_nUIOptions & (ViewOptFlags2::AnyRuler|ViewOptFlags2::HRuler)) == (ViewOptFlags2::AnyRuler|ViewOptFlags2::HRuler);
#else
                            (void) bDirect;
                            return false;
#endif
                        }
    void            SetViewHRuler   (bool b)
                        {    b ? (m_nUIOptions |= ViewOptFlags2::HRuler ) : ( m_nUIOptions &= ~ViewOptFlags2::HRuler);}

    bool            IsViewVRuler(bool bDirect = false) const
                        {
#if HAVE_FEATURE_DESKTOP
                            return bDirect
                                   ? bool(m_nUIOptions & ViewOptFlags2::VRuler)
                                   : !m_bReadonly && (m_nUIOptions & (ViewOptFlags2::AnyRuler|ViewOptFlags2::VRuler)) == (ViewOptFlags2::AnyRuler|ViewOptFlags2::VRuler);
#else
                            (void) bDirect;
                            return false;
#endif
                        }
    void            SetViewVRuler     (bool b)
                        { b ? (m_nUIOptions |= ViewOptFlags2::VRuler ) : ( m_nUIOptions &= ~ViewOptFlags2::VRuler);}

    // ShadowCursor, switch on/off, get/set color/mode.
    bool    IsShadowCursor()    const
        { return bool(m_nUIOptions & ViewOptFlags2::ShadowCursor); }
    void   SetShadowCursor(bool b)
        { b ? (m_nUIOptions |= ViewOptFlags2::ShadowCursor ) : ( m_nUIOptions &= ~ViewOptFlags2::ShadowCursor); }

    //move vertical ruler to the right
    bool    IsVRulerRight()    const
        { return bool(m_nUIOptions & ViewOptFlags2::VRulerRight); }
    void   SetVRulerRight(bool b)
        { b ? (m_nUIOptions |= ViewOptFlags2::VRulerRight ) : ( m_nUIOptions &= ~ViewOptFlags2::VRulerRight); }

    bool            IsStarOneSetting() const {return m_bStarOneSetting; }
    void            SetStarOneSetting(bool bSet) {m_bStarOneSetting = bSet; }

    bool            IsPagePreview() const {return m_bIsPagePreview; }
    void            SetPagePreview(bool bSet) { m_bIsPagePreview= bSet; }

    sal_uInt8           GetShdwCursorFillMode() const { return m_nShadowCursorFillMode; }
    void            SetShdwCursorFillMode( sal_uInt8 nMode ) { m_nShadowCursorFillMode = nMode; };

    bool        IsShowPlaceHolderFields() const { return m_bShowPlaceHolderFields; }
    void            SetShowPlaceHolderFields(bool bSet) { m_bShowPlaceHolderFields = bSet; }

    static Color&   GetDocColor();
    static Color&   GetDocBoundariesColor();
    static Color&   GetAppBackgroundColor();
    static Color&   GetObjectBoundariesColor();
    static Color&   GetTableBoundariesColor();
    static Color&   GetIndexShadingsColor();
    static Color&   GetLinksColor();
    static Color&   GetVisitedLinksColor();
    static Color&   GetDirectCursorColor();
    static Color&   GetTextGridColor();
    static Color&   GetSpellColor();
    static Color&   GetSmarttagColor();
    static Color&   GetShadowColor();
    static Color&   GetFontColor();
    static Color&   GetFieldShadingsColor();
    static Color&   GetSectionBoundColor();
    static Color&   GetPageBreakColor();
    static Color&   GetHeaderFooterMarkColor();

    static bool     IsAppearanceFlag(ViewOptFlags nFlag);

    static bool     IsDocBoundaries()     {return IsAppearanceFlag(ViewOptFlags::DocBoundaries);}
    static bool     IsObjectBoundaries()  {return IsAppearanceFlag(ViewOptFlags::ObjectBoundaries);}
    static bool     IsTableBoundaries()   {return IsAppearanceFlag(ViewOptFlags::TableBoundaries );}
    static bool     IsIndexShadings()     {return IsAppearanceFlag(ViewOptFlags::IndexShadings   );}
    static bool     IsLinks()             {return IsAppearanceFlag(ViewOptFlags::Links            );}
    static bool     IsVisitedLinks()      {return IsAppearanceFlag(ViewOptFlags::VisitedLinks    );}
    static bool     IsFieldShadings()     {return IsAppearanceFlag(ViewOptFlags::FieldShadings);}
    static bool     IsSectionBoundaries() {return IsAppearanceFlag(ViewOptFlags::SectionBoundaries);}
    static bool     IsShadow()            {return IsAppearanceFlag(ViewOptFlags::Shadow           );}

    static void     SetAppearanceFlag(ViewOptFlags nFlag, bool bSet, bool bSaveInConfig = false);

    static void     SetDocBoundaries(bool bSet)   {SetAppearanceFlag(ViewOptFlags::DocBoundaries, bSet);}

    static void     ApplyColorConfigValues(const svtools::ColorConfig& rConfig);
};

inline bool SwViewOption::operator==( const SwViewOption &rOpt ) const
{
    return IsEqualFlags( rOpt ) && m_nZoom == rOpt.GetZoom();
}

inline void SwViewOption::SetUIOptions( const SwViewOption& rVOpt )
{
    m_nUIOptions = rVOpt.m_nUIOptions;
    m_nTableDestination = rVOpt.m_nTableDestination;
    m_nShadowCursorFillMode = rVOpt.m_nShadowCursorFillMode;
}

// Helper function for checking HTML-capabilities.
SW_DLLPUBLIC sal_uInt16     GetHtmlMode(const SwDocShell*);

#endif

/* vim:set shiftwidth=4 softtabstop=4 expandtab: */
