/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */
/*
 * This file is part of the libgltf project.
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 */

#ifndef COMMON_H
#define COMMON_H
#include <GL/glew.h>
#include <glm/glm.hpp>
#include <string>
#include <vector>
#include <map>
#include <algorithm>
#include <functional>
#include "types.h"
#include "Texture.h"
#include "Shaders.h"

namespace libgltf
{

int gltf_get_file_index_by_name(const std::vector<glTFFile>& inputFiles, const char* name);

struct TechniqueState
{
    unsigned int blendEnable;
    unsigned int blendEquation;
    unsigned int blendFuncSfactor;
    unsigned int blendFuncDfactor;
    unsigned int cullFaceEnable;
    unsigned int depthMask;
    unsigned int depthTestEnable;
};

enum DataType
{
    DataType_UNKNOW                         = 0x0,
    DataType_BYTE                           = 0x1400,
    DataType_UNSIGNED_BYTE                  = 0x1401,
    DataType_SHORT                          = 0x1402,
    DataType_UNSIGNED_SHORT                 = 0x1403,
    DataType_INT                            = 0x1404,
    DataType_UNSIGNED_INT                   = 0x1405,
    DataType_FLOAT                          = 0x1406,

    DataType_ARRAY_BUFFER                   = 0x8892,
    DataType_ELEMENT_ARRAY_BUFFER           = 0x8893,
    DataType_ARRAY_BUFFER_BINDING           = 0x8894,
    DataType_ELEMENT_ARRAY_BUFFER_BINDING   = 0x8895,

    DataType_FLOAT_VEC2                     = 0x8B50,
    DataType_FLOAT_VEC3                     = 0x8B51,
    DataType_FLOAT_VEC4                     = 0x8B52,
    DataType_INT_VEC2                       = 0x8B53,
    DataType_INT_VEC3                       = 0x8B54,
    DataType_INT_VEC4                       = 0x8B55,
    DataType_BOOL                           = 0x8B56,
    DataType_BOOL_VEC2                      = 0x8B57,
    DataType_BOOL_VEC3                      = 0x8B58,
    DataType_BOOL_VEC4                      = 0x8B59,
    DataType_FLOAT_MAT2                     = 0x8B5A,
    DataType_FLOAT_MAT3                     = 0x8B5B,
    DataType_FLOAT_MAT4                     = 0x8B5C,
    DataType_SAMPLER_2D                     = 0x8B5E,
    DataType_SAMPLER_CUBE                   = 0x8B60
};

class Attribute
{
public:
    void setDataCount(unsigned int count);
    unsigned int getDataCount() const;

    void setDataType(DataType type);
    DataType getDataType() const;

    void setByteStride(unsigned int byteStride);
    unsigned int getByteStride() const;

    void setAttributeData(const char* pValue, unsigned int length);
    const char* getAttributeData() const;

    Attribute();
    ~Attribute();
private:
    Attribute(const Attribute&);
    Attribute& operator=(const Attribute&);

    unsigned int mByteStride;
    DataType mDataType;
    unsigned int mCount;
    char* pData;
};

enum LightSourceType
{
    LightSource_UNDEFINED     = 0x0,
    LightSource_DIRECTIONAL   = 0x1,
    LightSource_POINT         = 0x2,
    LightSource_SPOT          = 0x3,
    LightSource_AMBIET        = 0x4
};

class Light
{
public:
    Light();
    ~Light();
    Light(const Light& light);

    void setLightName(std::string);
    std::string getLightName() const;

    void setType(LightSourceType Type);

    void setColor(glm::vec3 color);
    glm::vec3 getColor() const;

    void setAttenuationConstant(float);
    float getAttenuationConstant() const;

    void setAttenuationLinear(float);
    float getmAttenuationLinear() const;

    void setAttenuationQuadratic(float);
    float getAttenuationQuadratic() const;

private:
    std::string mName;
    LightSourceType mType;
    float mAttenuationConstant;
    float mAttenuationLinear;
    float mAttenuationQuadratic;
    glm::vec3 mColor;
};

class TechAttribute
{
public:
    void setAttributeName(const std::string& name);
    const std::string getAttributeName();

    void setAttributeIndex(const std::string& index);
    const std::string getAttributeIndex();

    TechAttribute();
    ~TechAttribute();
private:
    std::string mAttributeName;
    std::string mAttributeIndex;
};

class TechUniform
{
public:
    void setUniformName(const std::string& name);

    void setUniformIndex(const std::string& index);
    TechUniform();
    ~TechUniform();
private:
    std::string mUniformName;
    std::string mUniformIndex;
};

struct techLight
{
    std::string mName;
    std::string mSource;
    float floatValue;
    glm::vec3 vecValue;
    DataType type;

    techLight(): mName(), mSource(), floatValue(0.0), vecValue(1.0f), type(DataType_UNKNOW) {}
};

class Technique
{
public:
    void setProgramState(bool flag);
    bool getProgramState() const;

    void setTechId(std::string techId);
    const std::string& getTechId();

    void setVertexShader(const std::string& shaderName);
    void setFragmentShader(const std::string& shaderName);
    unsigned int getProgramId() const;

    void insertTechAttribute(const std::string& key, TechAttribute* pTechAttr);
    TechAttribute* getTechAttribute(const std::string& key);

    void pushTLight(techLight* ptLight);
    const std::vector<techLight*> poptLight();

    void pushTechUniform(TechUniform* pUniform);

    TechniqueState* getTechState();

    int initTechnique(const std::vector<glTFFile>& inputFiles);
    bool useTechnique();
    void freeTechnique();

    Technique();
    ~Technique();
private:
    Technique(const Technique&);
    Technique& operator=(const Technique&);

    ShaderProgram mShaderProg;

    std::map<std::string, TechAttribute*> mTechAttrMap;
    std::vector<TechUniform*> mTechUnifVec;

    std::vector<techLight*>mTLight;

    unsigned int mProgramId;
    std::string mTechniqueId;
    std::string mVShaderName;
    std::string mFShaderName;
    bool mTechniqueState;
    TechniqueState* pState;
};

enum TextureType
{
    TextureType_NONE = 0x0,
    TextureType_DIFFUSE = 0x1,
    TextureType_SPECULAR = 0x2,
    TextureType_AMBIENT = 0x3,
    TextureType_EMISSIVE = 0x4,
    TextureType_HEIGHT = 0x5,
    TextureType_NORMALS = 0x6,
    TextureType_SHININESS = 0x7,
    TextureType_OPACITY = 0x8,
    TextureType_DISPLACEMENT = 0x9,
    TextureType_LIGHTMAP = 0xA,
    TextureType_REFLECTION = 0xB,
    TextureType_UNKNOWN = 0xC
};

class MaterialProperty
{
public:
    void setPropertyName(const std::string& key);
    const std::string& getPropertyName() const;

    void setImagePath(const std::string& index);
    const std::string& getImagePath() const;

    void setDataLength(unsigned int length);

    void setDataType(DataType type);
    DataType getDataType() const;

    void setPropertyData(char* pValue, unsigned int size);
    const char* getPropertyData() const;

    MaterialProperty();
    ~MaterialProperty();
private:
    MaterialProperty(const MaterialProperty&);
    MaterialProperty& operator=(const MaterialProperty&);

    /** Specifies the name of the property (key) */
    std::string mPropertyName;

    /** Specifies their texture type.
    * If non-texture properties, this member is TextureType_NONE(0)
    */
    TextureType mTextureType;

    /** Specifies image if it's exist. */
    std::string mImagePath;

    /** Specifies their data type.
    * If non-texture properties, this member is TextureType_NONE(0)
    */
    DataType mDataType;

    /** Size of the buffer pData is pointing to, in bytes.
    *  This value may not be 0.
    */
    unsigned int mDataLength;

    /** Binary buffer to hold the property's value.
    * The size of the buffer is always mDataLength.
    */
    char* pData;
};

class Material
{
public:
    void setTechniqueId(const std::string& techniqueId);
    const std::string& getTechniqueId() const;

    void pushMaterialProper(MaterialProperty* pProperty);
    const MaterialProperty* getMaterialProper(unsigned int index);
    unsigned int getMaterialProperSize();

    Material();
    ~Material();
private:
    std::string mTechniqueId;
    std::vector<MaterialProperty*> mPropertyVec;
};

class Primitives
{
public:
    void setMaterialIndex(std::string materialIndex);
    const std::string getMaterialIndex() const;

    void setIndicesIndex(std::string indicesIndex);
    const std::string getIndicesIndex() const;

    void insertAttribute(const std::string& key, const std::string& value);
    const std::string getAttributeIndex(const std::string& key) const;

    Primitives();
    ~Primitives();
private:
    /** Index of attribute map. VerticeIndex NormalIndex TexCoordIndex*/
    std::map<std::string, std::string> mAttributeMap;

    /** Index of attribute IndexIndex*/
    std::string mIndicesIndex;

    /** Index of material map*/
    std::string mMaterialIndex;
};

class Mesh
{
public:
    void setMeshName(const std::string& name);

    void setPrimitiveVec(Primitives* pPrim);
    const Primitives* getPrimitiveVec(unsigned int index) const;
    unsigned int getPrimitiveVecSize() const;

    Mesh();
    ~Mesh();
private:
    std::string mName;
    std::vector<Primitives*> mPrimitiveVec;
};



class Skin
{
public:
    void setBindMatrix(glm::mat4* matrix);
    glm::mat4* getBindMatrix();

    void setBindMatrixCount(unsigned int count);
    unsigned int getBindMatrixCount();

    void pushBoneId(std::string boneId);
    const std::string getBoneId(unsigned int index);
    unsigned int getBoneIdSize();

    void setSkinName(const std::string& name);
    const std::string& getSkinName();

    Skin();
    ~Skin();
private:
    Skin(const Skin&);
    Skin& operator=(const Skin&);

    std::string mName;
    glm::mat4* pBindMatrix;
    unsigned int mBindMatrixCount;
    std::vector<std::string> mBoneIdVec;
};

#define TRANS_CHANNEL  1
#define ROTATE_CHANNEL 2
#define SCALE_CHANNEL  4
#define ALL_CHANNEL    7

class Animation
{
    struct QuatKey
    {
        double mTime;
        glm::mat4 mValue;

        QuatKey() : mTime(0.0), mValue(1.0) {}
    };
public:
    void setDuration(double duration);
    double getDuration();

    void setBoneId(std::string boneId);
    const std::string& getBoneId();

    const glm::mat4& findTimeValue(double time);
    void pushTimeValue(double time, const glm::mat4& matrix);

    void setTimeValue(const glm::mat4& matrix, unsigned int index);
    const glm::mat4& getTimeValue(unsigned int index);

    static bool compareQuatKey(const QuatKey& qk, double time)
    {
        if (time > qk.mTime)
        {
            return true;
        }
        return false;
    }

    void setChannelBits(unsigned int channel)
    {
        mChannelBits |= channel;
    }

    unsigned int getChannelBits()
    {
        return mChannelBits;
    }

    Animation();
    ~Animation();
private:
    /** Bone id of every animation.  */
    std::string mBoneId;

    /** Duration of the animation in ticks.  */
    double mDuration;

    unsigned char mChannelBits;

    std::vector<QuatKey> mTimeValueVec;
};

enum NodeType
{
    NodeType_Init        = 0x0,
    NodeType_Node        = 0x1,
    NodeType_Mesh        = 0x2,
    NodeType_Camera      = 0x4,
    NodeType_Light       = 0x8
};

class Node
{
public:
    void setGlobalMatrix(const glm::mat4& matrix);
    glm::mat4& getGlobalMatrix();

    void setLocalMatrix(const glm::mat4& matrix);
    glm::mat4& getLocalMatrix();

    void setRotate(const glm::mat4& rotate);
    void setRotate(const float* pbuf);
    const glm::mat4& getRotate();

    void setScale(const glm::mat4& scale);
    void setScale(const float* pbuf);
    const glm::mat4& getScale();

    void setTranslate(const glm::mat4& trans);
    void setTranslate(const float* pbuf);
    const glm::mat4& getTranslate();

    void setNodeType(NodeType type);
    NodeType getNodeType() const;

    void setNodeName(const std::string& name);
    const std::string& getNodeName() const;

    void setJointId(const std::string& jointId);
    const std::string& getJointId() const;

    void setSkinIndex(const std::string& index);
    const std::string& getSkinIndex() const;

    void setSkinPoint(Skin* pSkin);
    Skin* getSkinPoint();

    void pushBoneNode(Node* pBone);
    Node* getBoneNode(unsigned int index);
    unsigned int getBoneNodeSize();

    void setAnimPoint(Animation* pAnima);
    Animation* getAnimPoint();

    void setSkeleIndex(const std::string& index);
    const std::string& getSkeleIndex() const;

    void setJointFlag(bool flag);
    bool getJointFlag();

    void setMatrixFlag(bool flag);
    bool getMatrixFlag();

    void setUpdateFlag(bool flag);
    bool getUpdateFlag();

    void setLightIndex(const std::string& index);
    const std::string getLightIndex() const;

    void setCameraIndex(std::string index);
    const std::string getCameraIndex();

    void pushMeshIndex(const std::string& index);
    const std::string getMeshIndex(unsigned int index) const;
    unsigned int getMeshIndexSize() const;

    void pushChildNode(Node* pNode);
    Node* getChildNode(unsigned int index);
    unsigned int getChildNodeSize() const;

    void setParentNode(Node* pParent);
    Node* getParentNode();

    inline bool hasChildren() const
    {
        return mChildrenVec.size() > 0 ? true : false;
    }

    inline bool hasMesh() const
    {
        return mMesheIndexVec.size() > 0 ? true : false;
    }

    Node();
    ~Node();
private:
    Node(const Node&);
    Node& operator=(const Node&);

    /** There are four types of node, Maybe it's a mesh array, or a camera,
    *    light, also it could be just a node, children node are others type.
    */
    NodeType mNodeType;

    /** The name of this node. */
    std::string mName;

    /** The transformation relative to the node's local. */
    glm::mat4 mLocalMatrix;

    /** The transformation relative to the node's global. */
    glm::mat4 mGlobalMatrix;

    glm::mat4 mScaleMatrix;
    glm::mat4 mTransMatrix;
    glm::mat4 mRotateMatrix;

    /** Parent node. NULL if this node is the root node. */
    Node* pParent;

    /** The child nodes of this node. */
    std::vector<Node*> mChildrenVec;

    /** The meshes of this node. Each entry is an index into the mesh */
    std::vector<std::string> mMesheIndexVec;

    /** The camera of this node. Each entry is an index into the camera */
    std::string mCameraIndex;

    /** The light of this node. Each entry is an index into the light */
    std::string mLightIndex;

    /** Index into the joint, if it is a animation. */
    std::string mJointIndex;

    std::string mSkinIndex;
    Skin* mpSkin;

    std::vector<Node*> mBoneVec;

    Animation* pAnimation;

    std::string mSkeleIndex;

    bool bJointFlag;
    bool bMatrixFlag;
    bool bUpdateFlag;
};

class ParseCamera
{
public:
    void setXFov(float x);
    float getXFov();

    void setYFov(float y);
    float getYFov();

    void setFar(float f);
    float getFar();

    void setNear(float n);
    float getNear();

    void setAspectRatio(float a);
    float getAspectRatio();
    void setCameraNode(Node* pNode);
    Node* getCameraNode();
    ParseCamera();
    ~ParseCamera();
private:
    ParseCamera(const ParseCamera&);
    ParseCamera& operator=(const ParseCamera&);

    std::string mName;
    float mXFov;
    float mYFov;
    float mNear;
    float mFar;
    float mAspectRatio;
    Node* cameraNode;
};



class Scene
{
public:
    void insertMeshMap(const std::string& key, Mesh* pMesh);
    Mesh* findMesh(const std::string& key);

    void insertMaterialMap(const std::string& key, Material* pMaterial);
    Material* findMaterial(const std::string& key);

    void insertAttributeMap(std::string key, Attribute* pAttribute);
    const Attribute* findAttribute(const std::string& key);
    void clearAttributeMap();

    void setVertexMax(float x, float y, float z);
    glm::vec3& getVertexMax();
    void setVertexMin(float x, float y, float z);
    glm::vec3& getVertexMin();

    void insertLightMap(const std::string& key, Light* pLight);
    Light* findLight(const std::string& key);

    void setRootNode(Node* pRootNode);
    Node* getRootNode();

    void pushNode(Node* pNode);
    Node* getNode(unsigned int index);
    unsigned int getNodeSize();

    void pushSkin(Skin* pSkin);
    Skin* getSkin(unsigned int index);
    unsigned int getSkinSize();

    void insertCameraMap(const std::string& key, ParseCamera* pCamera);
    ParseCamera* findCamera(const std::string& key);
    std::string getCameraIndex();
    const glm::mat4 findCameraTransformation(const std::string& key);

    void pushTechnique(Technique* pTechnique);
    Technique* getTechnique(unsigned int index);
    unsigned int getTechSize();

    char* getBuffer();
    int  setBuffer(const std::string& binName, const unsigned int length, const std::vector<glTFFile>& inputFiles);
    void removeBuffer();

    int loadTexture(const std::string& imagePath, const std::vector<glTFFile>& inputFiles);
    Texture* findTexture(const std::string& key);
    void insertTextureMap(const std::string& key, Texture* pTexture);

    void setGltfHandle(glTFHandle* handle);
    glTFHandle* getGltfHandle();
    const glTFFile* getGltfFileByFileName(const std::string& fileName, const std::vector<glTFFile>& inputFiles);

    Animation* findAnimation(const std::string& nodeId);
    void insertAnimMap(const std::string& key, Animation* pAnimation);
    unsigned int getAnimationCount();

    void setUseCameraInJson(bool bECamera)
    {
        bUseCameraInJson = bECamera;
    }

    bool getUseCameraInJson() const
    {
        return bUseCameraInJson;
    }

    double getDuration();
    void setDuration(double duration);

    void insertLightNodeMap(const std::string& LightIndex, Node* pNode);
    Node* findLightNodeMap(std::string key);

    Scene();
    ~Scene();
private:
    Scene(const Scene&);
    Scene& operator=(const Scene&);

    template<typename M> void freeMap(M& aMap);

    /** The map of Animation. */
    std::map<std::string, Animation*> mAnimaMap;
    double mDuration;

    /** The vector of Node */
    std::map<std::string, Node*> mLightNodeMap;

    /** The vector of Skins. */
    std::vector<Skin*> mSkinVec;

    /** The root node of the hierarchy.
    * There will always be at least the root node if the parser
    * was successful done(and no special flags have been set).
    */
    Node* pRootNode;

    std::vector<Node*> mNodeVec;

    /** The map of meshes.
    * Use the indices given in the Node structure to access
    * this array.
    */
    std::map<std::string, Mesh*> mMeshMap;

    /** The map of material. */
    std::map<std::string, Material*> mMaterialMap;

    /** The map of light sources. */
    std::map<std::string, Light*> mLightMap;

    /** The map of cameras.
    * The first camera in the array (if existing)
    * is the default camera view into the scene.
    */
    std::map<std::string, ParseCamera*> mCameraMap;

    glm::vec3 mVertexMax;
    glm::vec3 mVertexMin;

    /** The map of attribute. */
    std::map<std::string, Attribute*> mAttributeMap;

    /** The map of technique. */
    std::vector<Technique*> mTechniqueVec;

    /** Store the data of binary */
    char* pBuffer;

    /** Whether analysis the camera node*/
    bool bUseCameraInJson;

    glTFHandle* pGltfHandle;
    std::map<std::string, Texture*> mTexturesMap;
};

} // namespace libgltf

#endif

/* vim:set shiftwidth=4 softtabstop=4 expandtab: */
