/*
 * Copyright (c) 2007, 2018, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <jvmti.h>
#include "agent_common.h"
#include <jvmti_tools.h>
#include "JVMTITools.h"

/* ============================================================================= */

static jvmtiEnv *jvmti = NULL;

/* ============================================================================= */

/* Used to explicitly initiate class retransformation process from Java code
 *
 * Parameters:
 *    jclass klass                      - class, which possess this native method
 *                                        (nsk.jvmti.RetransformClasses.retransform004)
 *
 *    jclass class_for_retransformation - class, which should be retransformed
 */
JNIEXPORT jboolean JNICALL
Java_nsk_jvmti_RetransformClasses_retransform004_forceLoadedClassesRetransformation(
        JNIEnv *jni
        , jclass klass
        , jclass class_for_retransformation
        )
{
    if (!NSK_JVMTI_VERIFY(
                NSK_CPP_STUB3(
                    RetransformClasses
                    , jvmti
                    , 1
                    , &class_for_retransformation
                    )
                )
       )
        return JNI_FALSE;

    return JNI_TRUE;
}


/* ============================================================================= */

/* Class retransformation hook */
static void JNICALL
ClassFileLoadHook (
        jvmtiEnv *jvmti
        , JNIEnv *jni
        , jclass class_being_redefined
        , jobject loader
        , const char* name
        , jobject protection_domain
        , jint class_data_len
        , const unsigned char* class_data
        , jint *new_class_data_len
        , unsigned char** new_class_data
        )
{
    // Allocate space for "retransformed" class version
    if (!NSK_JVMTI_VERIFY(
                NSK_CPP_STUB3(
                    Allocate
                    , jvmti
                    , class_data_len
                    , new_class_data
                    )
                )
       )
        return;

    // Copy old code
    memcpy(*new_class_data, class_data, class_data_len);
    *new_class_data_len = class_data_len;
}


/* ============================================================================= */

/* Agent initialization procedure */
#ifdef STATIC_BUILD
JNIEXPORT jint JNICALL Agent_OnLoad_retransform004(JavaVM *jvm, char *options, void *reserved) {
    return Agent_Initialize(jvm, options, reserved);
}
JNIEXPORT jint JNICALL Agent_OnAttach_retransform004(JavaVM *jvm, char *options, void *reserved) {
    return Agent_Initialize(jvm, options, reserved);
}
JNIEXPORT jint JNI_OnLoad_retransform004(JavaVM *jvm, char *options, void *reserved) {
    return JNI_VERSION_1_8;
}
#endif
jint Agent_Initialize(JavaVM *vm, char *options, void *reserved)
{
    jvmtiEventCallbacks callbacks;
    jvmtiCapabilities caps;

    if (!NSK_VERIFY(
                (jvmti = nsk_jvmti_createJVMTIEnv(vm, reserved)) != NULL
                )
       )
        return JNI_ERR;

    if (!NSK_JVMTI_VERIFY(
                NSK_CPP_STUB2(
                    GetCapabilities
                    , jvmti
                    , &caps)
                )
       )
        return JNI_ERR;

    caps.can_retransform_classes = 1;

    // Register all necessary JVM capabilities
    if (!NSK_JVMTI_VERIFY(
                NSK_CPP_STUB2(
                    AddCapabilities
                    , jvmti
                    , &caps)
                )
       )
        return JNI_ERR;

    // Register all necessary event callbacks
    memset(&callbacks, 0, sizeof(callbacks));
    callbacks.ClassFileLoadHook = &ClassFileLoadHook;

    if (!NSK_JVMTI_VERIFY(
                NSK_CPP_STUB3(
                    SetEventCallbacks
                    , jvmti
                    , &callbacks
                    , sizeof(callbacks)
                    )
                )
       )
        return JNI_ERR;

    // Enable class retransformation
    if (!NSK_JVMTI_VERIFY(
                NSK_CPP_STUB4(
                    SetEventNotificationMode
                    , jvmti
                    , JVMTI_ENABLE
                    , JVMTI_EVENT_CLASS_FILE_LOAD_HOOK
                    , NULL
                    )
                )
       )
        return JNI_ERR;

    return JNI_OK;
}

/* ============================================================================= */
