/**
 * @file xcb/geis_xcb_backend.c
 * @brief Implements the GEIS XCB back end.
 *
 * Copyright 2011 Canonical Ltd.
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option) any
 * later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.	 See the GNU Lesser General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
#include "geis_config.h"
#include "geis_xcb_backend.h"

#include <dlfcn.h>
#include "geis_attr.h"
#include "geis_backend_protected.h"
#include "geis_class.h"
#include "geis_device.h"
#include "geis_event.h"
#include "geis_frame.h"
#include "geis_filter.h"
#include "geis_group.h"
#include "geis_logging.h"
#include "geis_private.h"
#include "geis_region.h"
#include "geis_touch.h"
#include "geis_xcb_backend_sub_table.h"
#include "geis_xcb_backend_token.h"
#include "grail_gestures.h"
#include <grail.h>
#include <grail-types.h>
#include <stdio.h>
#include <string.h>
#include "xcb_gesture.h"
#include <X11/extensions/XI2proto.h>
#include <X11/extensions/XInput2.h>
#include <X11/X.h>
#include <X11/Xlib-xcb.h>
#include <xcb/xcb.h>
#ifdef HAVE_XI_2_1
# include <xorg/xserver-properties.h>
#endif

#define MIN_TOUCHES 1
#define MAX_TOUCHES 5
#define MAX_NUM_DEVICES 10
#define MAX_NUM_WINDOWS 10

#define GRAIL_XCB_BITMASK_LEN 2


static inline GeisSize
_min(GeisSize a, GeisSize b)
{
  return (a < b) ? a : b;
}


static inline GeisSize
_max(GeisSize a, GeisSize b)
{
  return (a > b) ? a : b;
}


struct GeisXcbBackend
{
  Geis                    geis;
  Display                *x11_display;
  xcb_connection_t       *xcb_connection;
  int                     xcb_fd;
  int                     grail_is_old_version;
  GeisXcbBackendSubTable  sub_table;
  int                     xi_opcode;
};


/*
 * Information required for each xcb-gesture subscription.
 *
 * @todo: unhardcode sizes and lengths (and malloc them instead).
 */
typedef struct _XcbGestureSub
{
  int           num_device_filters;
  uint16_t      devices[MAX_NUM_DEVICES]; 
  uint16_t      device_count;
  int           num_class_filters;
  uint32_t      mask[GRAIL_XCB_BITMASK_LEN];
  uint32_t      mask_len;
  int           num_region_filters;
  xcb_window_t  windows[MAX_NUM_WINDOWS];
  int           window_count;
} *XcbGestureSub;

static void _construct(void *mem, Geis geis);
static void _finalize(GeisBackend g);
static GeisBackendToken _create_token(GeisBackend be, GeisBackendTokenInitState);
static GeisStatus _gxcb_accept_gesture(GeisBackend, GeisGroup, GeisGestureId);
static GeisStatus _gxcb_reject_gesture(GeisBackend, GeisGroup, GeisGestureId);
static void _fd_callback(int fd, GeisBackendMultiplexorActivity ev, void *ctx);


static struct GeisBackendVtable be_vtbl = {
  _construct,
  _finalize,
  _create_token,
  _gxcb_accept_gesture,
  _gxcb_reject_gesture
};


/* A handy little table to map grail types to geis gesture classes. */
static struct GrailTypeToGeisClass
{
  GeisInteger      grail_type;
  GeisGestureClass geis_class;
} s_grail_type_map[] = {
  { GRAIL_TYPE_DRAG1, NULL },
  { GRAIL_TYPE_PINCH1, NULL },
  { GRAIL_TYPE_ROTATE1, NULL },
  { GRAIL_TYPE_DRAG2, NULL },
  { GRAIL_TYPE_PINCH2, NULL },
  { GRAIL_TYPE_ROTATE2, NULL },
  { GRAIL_TYPE_DRAG3, NULL },
  { GRAIL_TYPE_PINCH3, NULL },
  { GRAIL_TYPE_ROTATE3, NULL },
  { GRAIL_TYPE_DRAG4, NULL },
  { GRAIL_TYPE_PINCH4, NULL },
  { GRAIL_TYPE_ROTATE4, NULL },
  { GRAIL_TYPE_DRAG5, NULL },
  { GRAIL_TYPE_PINCH5, NULL },
  { GRAIL_TYPE_ROTATE5, NULL },
  { GRAIL_TYPE_TAP1, NULL },
  { GRAIL_TYPE_TAP2, NULL },
  { GRAIL_TYPE_TAP3, NULL },
  { GRAIL_TYPE_TAP4, NULL },
  { GRAIL_TYPE_TAP5, NULL },
  { GRAIL_TYPE_EDRAG, NULL },
  { GRAIL_TYPE_EPINCH, NULL },
  { GRAIL_TYPE_EROTATE, NULL },
  { GRAIL_TYPE_MDRAG, NULL },
  { GRAIL_TYPE_MPINCH, NULL },
  { GRAIL_TYPE_MROTATE, NULL },
  { GRAIL_TYPE_TOUCH1, NULL },
  { GRAIL_TYPE_TOUCH2, NULL },
  { GRAIL_TYPE_TOUCH3, NULL },
  { GRAIL_TYPE_TOUCH4, NULL },
  { GRAIL_TYPE_TOUCH5, NULL },
  { GRAIL_TYPE_ETOUCH, NULL },
  { GRAIL_TYPE_MTOUCH, NULL }
};

static GeisSize s_grail_type_map_size = sizeof(s_grail_type_map)
                                      / sizeof(struct GrailTypeToGeisClass);

static void
_set_grail_type_class(GeisInteger grail_type, GeisGestureClass geis_class)
{
  GeisSize i;
  for (i = 0; i < s_grail_type_map_size; ++i)
  {
    if (s_grail_type_map[i].grail_type == grail_type)
    { 
      s_grail_type_map[i].geis_class = geis_class;
      break;
    }
  }
}


static GeisGestureClass
_get_geis_class_from_grail_type(GeisInteger grail_type)
{
  GeisSize i;
  for (i = 0; i < s_grail_type_map_size; ++i)
  {
    if (s_grail_type_map[i].grail_type == grail_type)
    { 
      return s_grail_type_map[i].geis_class;
    }
  }
  return NULL;
}


/*
 * Makes an attempt to determine the local version of grail.  Pretty useless if
 * X is remote.
 */
static int
_grail_is_old_version(void)
{
  int (*fn)(void);
  int is_old_version = 1;
  void* lib = dlopen("libutouch-grail.so.1", RTLD_LAZY | RTLD_LOCAL);
  if (!lib)
  {
    geis_error("can not open libutouch-grail.so.1: %s", dlerror());
    goto final_exit;
  }

  *(void **)(&fn) = dlsym(lib, "grail_get_version");
  if (fn)
  {
    /* Check against magic in a particular version of grail.h */
    is_old_version = fn() < 0x00011000;
  }

  dlclose(lib);
final_exit:
  return is_old_version;
}


static void
_report_init_complete(GeisXcbBackend be)
{
  geis_post_event(be->geis, geis_event_new(GEIS_EVENT_INIT_COMPLETE));
}


#ifdef HAVE_XI_2_1
static void
_map_xi2_mode_to_geis_device_attrs(int xi2_mode, GeisDevice geis_device)
{
  GeisAttr device_attr;
  GeisBoolean is_direct = GEIS_FALSE;
  GeisBoolean is_independent = GEIS_FALSE;

  if (xi2_mode == XIDirectTouch)
    is_direct = GEIS_TRUE;

  device_attr = geis_attr_new(GEIS_DEVICE_ATTRIBUTE_DIRECT_TOUCH,
                              GEIS_ATTR_TYPE_BOOLEAN,
                              &is_direct);
  if (!device_attr)
  {
    geis_error("failed to create device attr");
  }
  else
  {
    geis_device_add_attr(geis_device, device_attr);
  }

  if (xi2_mode == XIIndependentPointer)
    is_independent = GEIS_TRUE;

  device_attr = geis_attr_new(GEIS_DEVICE_ATTRIBUTE_INDEPENDENT_TOUCH,
                              GEIS_ATTR_TYPE_BOOLEAN,
                              &is_independent);
  if (!device_attr)
  {
    geis_error("failed to create device attr");
  }
  else
  {
    geis_device_add_attr(geis_device, device_attr);
  }
}
#endif


static GeisBoolean
_verify_xcb_version(xcb_connection_t *xcb_connection)
{
  GeisBoolean                         is_valid_version = GEIS_FALSE;
  xcb_gesture_query_version_cookie_t  version_cookie;
  xcb_gesture_query_version_reply_t  *version_reply = NULL;
  xcb_generic_error_t                *error = NULL;

  version_cookie = xcb_gesture_query_version(xcb_connection,
                                             XCB_GESTURE_MAJOR_VERSION,
                                             XCB_GESTURE_MINOR_VERSION);
  version_reply = xcb_gesture_query_version_reply(xcb_connection,
                                                  version_cookie,
                                                  &error);
  if (!version_reply)
  {
    geis_error("failed to receive XCB gesture version reply.");
    goto final_exit;
  }

  if (version_reply->major_version != XCB_GESTURE_MAJOR_VERSION
   && version_reply->minor_version != XCB_GESTURE_MINOR_VERSION)
  {
    geis_error("server supports unrecognized version: %d.%d",
               version_reply->major_version, version_reply->minor_version);
  }
  else
  {
    is_valid_version = GEIS_TRUE;
  }

 free(version_reply);
final_exit:
 return is_valid_version;
}


#ifdef HAVE_XI_2_1
static void
_report_xi2_valuator_attrs(GeisDevice geis_device,
                           XITouchValuatorClassInfo *v,
                           GeisString min_label,
                           GeisString max_label,
                           GeisString res_label)
{
  GeisFloat f;
  GeisAttr device_attr;

  f = v->min;
  device_attr = geis_attr_new(min_label, GEIS_ATTR_TYPE_FLOAT, &f);
  if (!device_attr)
  {
    geis_error("failed to create device attr");
  }
  else
  {
    geis_device_add_attr(geis_device, device_attr);
  }

  f = v->max;
  device_attr = geis_attr_new(max_label, GEIS_ATTR_TYPE_FLOAT, &f);
  if (!device_attr)
  {
    geis_error("failed to create device attr");
  }
  else
  {
    geis_device_add_attr(geis_device, device_attr);
  }

  f = (GeisFloat)v->resolution;
  device_attr = geis_attr_new(res_label, GEIS_ATTR_TYPE_FLOAT, &f);
  if (!device_attr)
  {
    geis_error("failed to create device attr");
  }
  else
  {
    geis_device_add_attr(geis_device, device_attr);
  }
}


/*
 * Reports a new gesture-capable device.
 */
static void
_report_an_xcb_device(GeisXcbBackend be,  XIDeviceInfo *xcb_device)
{
  int        filter_attr_index = 0;
  GeisSize   filter_attr_count = 2;
  GeisFilterableAttribute filter_attrs = NULL;
  GeisDevice geis_device;
  GeisEvent  device_event;
  int        class_index;

  geis_device = geis_device_new(xcb_device->name, xcb_device->deviceid);
  if (!geis_device)
  {
    geis_error("failed to create device");
    goto final_exit;
  }

  /* Add attributes to the device. */
  for (class_index = 0; class_index < xcb_device->num_classes; ++class_index)
  {
    XIAnyClassInfo *any = xcb_device->classes[class_index];
    if (any->type == XITouchClass)
    {
      GeisAttr   device_attr;
      XITouchClassInfo *v = (XITouchClassInfo *)any;
      filter_attr_count += 3;
      _map_xi2_mode_to_geis_device_attrs(v->mode, geis_device);
      geis_debug("touch class for device %d \"%s\": mode %d num_touches %d",
                 xcb_device->deviceid, xcb_device->name,
                 v->mode, v->num_touches);

      device_attr = geis_attr_new(GEIS_DEVICE_ATTRIBUTE_TOUCHES,
                                  GEIS_ATTR_TYPE_INTEGER,
                                  &v->num_touches);
      if (!device_attr)
      {
	geis_error("failed to create device attr");
      }
      else
      {
	geis_device_add_attr(geis_device, device_attr);
      }
    }
    else if (any->type == XITouchValuatorClass)
    {
      XITouchValuatorClassInfo *v = (XITouchValuatorClassInfo *)any;
      char *label = v->label ? XGetAtomName(be->x11_display, v->label) : NULL;
      if (label && 0 == strcmp(label, AXIS_LABEL_PROP_ABS_MT_POSITION_X))
      {
	_report_xi2_valuator_attrs(geis_device,
	                           v,
	                           GEIS_DEVICE_ATTRIBUTE_MIN_X,
	                           GEIS_DEVICE_ATTRIBUTE_MAX_X,
	                           GEIS_DEVICE_ATTRIBUTE_RES_X);
      }
      else if (label && 0 == strcmp(label, AXIS_LABEL_PROP_ABS_MT_POSITION_Y))
      {
	_report_xi2_valuator_attrs(geis_device,
	                           v,
	                           GEIS_DEVICE_ATTRIBUTE_MIN_Y,
	                           GEIS_DEVICE_ATTRIBUTE_MAX_Y,
	                           GEIS_DEVICE_ATTRIBUTE_RES_Y);
      }
      else
      {
	char min_name[64];
	char max_name[64];
	char res_name[64];
	char *axis_name = label ? label : "";
	sprintf(min_name, "device %.48s %d minimum", axis_name, v->number);
	sprintf(max_name, "device %.48s %d maximum", axis_name, v->number);
	sprintf(res_name, "device %.48s %d resolution", axis_name, v->number);
	_report_xi2_valuator_attrs(geis_device, v, min_name, max_name, res_name);
      }
      if (label)
        XFree(label);
    }
  }

  /* Register the device for subscriptions. */
  filter_attrs = calloc(filter_attr_count, sizeof(struct GeisFilterableAttribute));
  if (!filter_attrs)
  {
    geis_debug("error allocating device attr table");
    goto unroll_device;
  }

  geis_filterable_attribute_init(&filter_attrs[filter_attr_index++],
                     GEIS_DEVICE_ATTRIBUTE_NAME, GEIS_ATTR_TYPE_STRING,
                     geis_xcb_token_add_device_term, geis_device);
  geis_filterable_attribute_init(&filter_attrs[filter_attr_index++],
                     GEIS_DEVICE_ATTRIBUTE_ID, GEIS_ATTR_TYPE_INTEGER,
                     geis_xcb_token_add_device_term, geis_device);
  geis_filterable_attribute_init(&filter_attrs[filter_attr_index++],
                     GEIS_DEVICE_ATTRIBUTE_TOUCHES, GEIS_ATTR_TYPE_INTEGER,
                     geis_xcb_token_add_device_term, geis_device);
  geis_filterable_attribute_init(&filter_attrs[filter_attr_index++],
                     GEIS_DEVICE_ATTRIBUTE_DIRECT_TOUCH, GEIS_ATTR_TYPE_BOOLEAN,
                     geis_xcb_token_add_device_term, geis_device);
  geis_filterable_attribute_init(&filter_attrs[filter_attr_index++],
                     GEIS_DEVICE_ATTRIBUTE_INDEPENDENT_TOUCH, GEIS_ATTR_TYPE_BOOLEAN,
                     geis_xcb_token_add_device_term, geis_device);
  geis_register_device(be->geis, geis_device, filter_attr_index, filter_attrs);
  free(filter_attrs);
  goto final_exit;

  geis_event_delete(device_event);
unroll_device:
  geis_device_unref(geis_device);
final_exit:
  return;
}


/*
 * Enumerates all input devices known to the X server and reports those that may
 * give gestural input.
 */
static void
_report_xcb_devices(GeisXcbBackend be, int deviceid)
{
  int num_devices;
  XIDeviceInfo *devices = XIQueryDevice(be->x11_display, deviceid, &num_devices);
  if (devices == NULL)
  {
    geis_warning("error retrieving device from XIQueryDevice()");
    return;
  }

  for (int device_index = 0; device_index < num_devices; ++device_index)
  {
    int class_index;
    for (class_index = 0;
         class_index < devices[device_index].num_classes;
         ++class_index)
    {
      XIAnyClassInfo *any = devices[device_index].classes[class_index];
      if (any->type == XITouchClass)
      {
	_report_an_xcb_device(be, &devices[device_index]);
	break;
      }
    }
  }

  XIFreeDeviceInfo(devices);
}
#endif


/*
 * Generates the events for gesture classes.
 */
static void
_report_grail_classes(GeisXcbBackend be)
{
  static struct GeisFilterableAttribute attrs[] = {
    { GEIS_CLASS_ATTRIBUTE_NAME,      GEIS_ATTR_TYPE_STRING,  geis_xcb_token_add_class_term, NULL },
    { GEIS_CLASS_ATTRIBUTE_ID,        GEIS_ATTR_TYPE_INTEGER, geis_xcb_token_add_class_term, NULL },
    { GEIS_GESTURE_ATTRIBUTE_TOUCHES, GEIS_ATTR_TYPE_INTEGER, geis_xcb_token_add_class_term, NULL }
  };
  GeisSize attr_count = sizeof(attrs) / sizeof(struct GeisFilterableAttribute);

  GeisGestureClass drag_class = geis_gesture_class_new(GEIS_GESTURE_DRAG,
                                                       GRAIL_TYPE_DRAG1);
  geis_xcb_backend_add_drag_attrs(drag_class);
  attrs[0].add_term_context = attrs[1].add_term_context = attrs[2].add_term_context = drag_class;
  geis_register_gesture_class(be->geis, drag_class, attr_count, attrs);
  _set_grail_type_class(GRAIL_TYPE_DRAG1, drag_class);
  _set_grail_type_class(GRAIL_TYPE_DRAG2, drag_class);
  _set_grail_type_class(GRAIL_TYPE_DRAG3, drag_class);
  _set_grail_type_class(GRAIL_TYPE_DRAG4, drag_class);
  _set_grail_type_class(GRAIL_TYPE_DRAG5, drag_class);
  _set_grail_type_class(GRAIL_TYPE_EDRAG, drag_class);
  _set_grail_type_class(GRAIL_TYPE_MDRAG, drag_class);

  GeisGestureClass pinch_class = geis_gesture_class_new(GEIS_GESTURE_PINCH,
                                                        GRAIL_TYPE_PINCH1);
  geis_xcb_backend_add_pinch_attrs(pinch_class);
  attrs[0].add_term_context = attrs[1].add_term_context = attrs[2].add_term_context = pinch_class;
  geis_register_gesture_class(be->geis, pinch_class, attr_count, attrs);
  _set_grail_type_class(GRAIL_TYPE_PINCH1, pinch_class);
  _set_grail_type_class(GRAIL_TYPE_PINCH2, pinch_class);
  _set_grail_type_class(GRAIL_TYPE_PINCH3, pinch_class);
  _set_grail_type_class(GRAIL_TYPE_PINCH4, pinch_class);
  _set_grail_type_class(GRAIL_TYPE_PINCH5, pinch_class);
  _set_grail_type_class(GRAIL_TYPE_EPINCH, pinch_class);
  _set_grail_type_class(GRAIL_TYPE_MPINCH, pinch_class);

  GeisGestureClass rotate_class = geis_gesture_class_new(GEIS_GESTURE_ROTATE,
                                                         GRAIL_TYPE_ROTATE1);
  geis_xcb_backend_add_rotate_attrs(rotate_class);
  attrs[0].add_term_context = attrs[1].add_term_context = attrs[2].add_term_context = rotate_class;
  geis_register_gesture_class(be->geis, rotate_class, attr_count, attrs);
  _set_grail_type_class(GRAIL_TYPE_ROTATE1, rotate_class);
  _set_grail_type_class(GRAIL_TYPE_ROTATE2, rotate_class);
  _set_grail_type_class(GRAIL_TYPE_ROTATE3, rotate_class);
  _set_grail_type_class(GRAIL_TYPE_ROTATE4, rotate_class);
  _set_grail_type_class(GRAIL_TYPE_ROTATE5, rotate_class);
  _set_grail_type_class(GRAIL_TYPE_EROTATE, rotate_class);
  _set_grail_type_class(GRAIL_TYPE_MROTATE, rotate_class);

  GeisGestureClass tap_class = geis_gesture_class_new(GEIS_GESTURE_TAP,
                                                      GRAIL_TYPE_TAP1);
  geis_xcb_backend_add_tap_attrs(tap_class);
  attrs[0].add_term_context = attrs[1].add_term_context = attrs[2].add_term_context = tap_class;
  geis_register_gesture_class(be->geis, tap_class, attr_count, attrs);
  _set_grail_type_class(GRAIL_TYPE_TAP1, tap_class);
  _set_grail_type_class(GRAIL_TYPE_TAP2, tap_class);
  _set_grail_type_class(GRAIL_TYPE_TAP3, tap_class);
  _set_grail_type_class(GRAIL_TYPE_TAP4, tap_class);
  _set_grail_type_class(GRAIL_TYPE_TAP5, tap_class);

  GeisGestureClass touch_class = geis_gesture_class_new(GEIS_GESTURE_TOUCH,
                                                        GRAIL_TYPE_TOUCH1);
  attrs[0].add_term_context = attrs[1].add_term_context = attrs[2].add_term_context = touch_class;
  geis_register_gesture_class(be->geis, touch_class, attr_count, attrs);
  _set_grail_type_class(GRAIL_TYPE_TOUCH1, touch_class);
  _set_grail_type_class(GRAIL_TYPE_TOUCH2, touch_class);
  _set_grail_type_class(GRAIL_TYPE_TOUCH3, touch_class);
  _set_grail_type_class(GRAIL_TYPE_TOUCH4, touch_class);
  _set_grail_type_class(GRAIL_TYPE_TOUCH5, touch_class);
  _set_grail_type_class(GRAIL_TYPE_ETOUCH, touch_class);
  _set_grail_type_class(GRAIL_TYPE_MTOUCH, touch_class);
}


static void
_report_xcb_regions(GeisXcbBackend be)
{
  static struct GeisFilterableAttribute attrs[] = {
    { GEIS_REGION_ATTRIBUTE_WINDOWID, GEIS_ATTR_TYPE_INTEGER,  geis_xcb_token_add_region_term, NULL },
  };
  GeisSize attr_count = sizeof(attrs) / sizeof(struct GeisFilterableAttribute);

  geis_register_region(be->geis, NULL, attr_count, attrs);
}


static void
_report_xcb_specials(GeisXcbBackend be)
{
  static struct GeisFilterableAttribute attrs[] = {
    { GEIS_GESTURE_TYPE_SYSTEM, GEIS_ATTR_TYPE_BOOLEAN, geis_xcb_token_add_feature_term, NULL },
    { "GRAB", GEIS_ATTR_TYPE_BOOLEAN, geis_xcb_token_add_feature_term, NULL }
  };
  GeisSize attr_count = sizeof(attrs) / sizeof(struct GeisFilterableAttribute);

  geis_register_special(be->geis, attr_count, attrs);
}

#ifdef HAVE_XI_2_1
static void
_select_device_events(GeisXcbBackend be)
{
  XIEventMask xi_event_mask;
  int event;
  int error;

  XQueryExtension(be->x11_display, "XInputExtension", &be->xi_opcode, &event,
                  &error);

  xi_event_mask.deviceid = XIAllDevices;
  xi_event_mask.mask_len = XIMaskLen(XI_HierarchyChanged);
  xi_event_mask.mask = calloc(xi_event_mask.mask_len, sizeof(char));
  XISetMask(xi_event_mask.mask, XI_HierarchyChanged);
  if (XISelectEvents(be->x11_display, XDefaultRootWindow(be->x11_display),
                     &xi_event_mask, 1) != Success)
    geis_warning("failed to select for XI hierarchy change events");
}
#endif

void 
_construct(void *mem, Geis geis)
{
  GeisXcbBackend be = (GeisXcbBackend)mem;
  be->geis = geis;
  be->grail_is_old_version = _grail_is_old_version();

  be->x11_display = XOpenDisplay(NULL);
  if (!be->x11_display)
  {
    geis_error("error opening X server.");
    geis_error_push(geis, GEIS_STATUS_UNKNOWN_ERROR);
    goto final_exit;
  }

  be->xcb_connection = XGetXCBConnection(be->x11_display);
  if (!be->xcb_connection)
  {
    geis_error("error connecting to X server.");
    geis_error_push(geis, GEIS_STATUS_UNKNOWN_ERROR);
    goto unwind_x11;
  }
  if (!_verify_xcb_version(be->xcb_connection))
  {
    geis_error_push(geis, GEIS_STATUS_UNKNOWN_ERROR);
    goto unwind_x11;
  }

  be->xcb_fd = xcb_get_file_descriptor(be->xcb_connection);
  geis_multiplex_fd(be->geis, be->xcb_fd, GEIS_BE_MX_READ_AVAILABLE,
                    _fd_callback, be);

  be->sub_table = geis_xcb_backend_sub_table_new();

#ifdef HAVE_XI_2_1
  _report_xcb_devices(be, XIAllDevices);
  _select_device_events(be);
#endif
  _report_grail_classes(be);
  _report_xcb_regions(be);
  _report_xcb_specials(be);
  _report_init_complete(be);

  goto final_exit;

unwind_x11:
  /* XCB has a bug that causes XCloseDisplay to fire a fatal IO error if the xcb
   * connection has an error. The best we can do right now is to leave the
   * connection dangling. */
  if (!be->xcb_connection || !xcb_connection_has_error(be->xcb_connection))
    XCloseDisplay(be->x11_display);
final_exit:
  return;
}


void 
_finalize(GeisBackend g)
{
  GeisXcbBackend be = (GeisXcbBackend)g;

  geis_xcb_backend_sub_table_delete(be->sub_table);

  /* XCB has a bug that causes XCloseDisplay to fire a fatal IO error if the xcb
   * connection has an error. The best we can do right now is to leave the
   * connection dangling. */
  if (!be->xcb_connection || !xcb_connection_has_error(be->xcb_connection))
    XCloseDisplay(be->x11_display);

  geis_debug("XCB back end finalized");
}


GeisBackendToken
_create_token(GeisBackend gbe, GeisBackendTokenInitState init_state)
{
  GeisXcbBackend xbe = (GeisXcbBackend)gbe;
  GeisBackendToken token = geis_xcb_token_new(gbe, init_state);
  geis_xcb_token_set_xcb(token, xbe->xcb_connection);
  return token;
}


static void
_dispatch_gesture(GeisXcbBackend be, xcb_gesture_notify_event_t *grail_event)
{
  GeisEvent     geis_event = NULL;
  GeisGroupSet  groupset = geis_groupset_new();
  GeisAttr      group_attr = geis_attr_new(GEIS_EVENT_ATTRIBUTE_GROUPSET,
                                           GEIS_ATTR_TYPE_POINTER,
                                           groupset);
  GeisTouchSet  touchset = geis_touchset_new();
  GeisAttr      touch_attr = geis_attr_new(GEIS_EVENT_ATTRIBUTE_TOUCHSET,
                                           GEIS_ATTR_TYPE_POINTER,
                                           touchset);

  /* Play games with the grail event to get the grail gesture properties. */
  float        *properties = (float *)(grail_event + 1);
  GeisSize      num_properties = grail_event->num_props;
  GeisSize      num_unmapped_properties = num_properties;
  GeisSize      touch_count;

  geis_attr_set_destructor(group_attr, (GeisAttrDestructor)geis_groupset_delete);
  geis_attr_set_destructor(touch_attr, (GeisAttrDestructor)geis_touchset_delete);

  switch (grail_event->status)
  {
    case GRAIL_STATUS_BEGIN:
      geis_event = geis_event_new(GEIS_EVENT_GESTURE_BEGIN);
      break;
    case GRAIL_STATUS_UPDATE:
      geis_event = geis_event_new(GEIS_EVENT_GESTURE_UPDATE);
      break;
    case GRAIL_STATUS_END:
      geis_event = geis_event_new(GEIS_EVENT_GESTURE_END);
      break;
  }

  GeisGroup group = geis_group_new(1);
  geis_groupset_insert(groupset, group);

  GeisFrame frame = geis_frame_new(grail_event->gesture_id);
  geis_group_insert_frame(group, frame);
  geis_frame_set_is_class(frame, _get_geis_class_from_grail_type(grail_event->gesture_type));

  GeisAttr attr = NULL;

  GeisInteger ival = grail_event->device_id;
  attr = geis_attr_new(GEIS_GESTURE_ATTRIBUTE_DEVICE_ID,
                       GEIS_ATTR_TYPE_INTEGER,
                       &ival);
  geis_frame_add_attr(frame, attr);

  ival = grail_event->time;
  attr = geis_attr_new(GEIS_GESTURE_ATTRIBUTE_TIMESTAMP,
                       GEIS_ATTR_TYPE_INTEGER,
                       &ival);
  geis_frame_add_attr(frame, attr);

  ival = grail_event->root;
  attr = geis_attr_new(GEIS_GESTURE_ATTRIBUTE_ROOT_WINDOW_ID,
                       GEIS_ATTR_TYPE_INTEGER,
                       &ival);
  geis_frame_add_attr(frame, attr);

  ival = grail_event->event;
  attr = geis_attr_new(GEIS_GESTURE_ATTRIBUTE_EVENT_WINDOW_ID,
                       GEIS_ATTR_TYPE_INTEGER,
                       &ival);
  geis_frame_add_attr(frame, attr);

  ival = grail_event->child;
  attr = geis_attr_new(GEIS_GESTURE_ATTRIBUTE_CHILD_WINDOW_ID,
                       GEIS_ATTR_TYPE_INTEGER,
                       &ival);
  geis_frame_add_attr(frame, attr);

  attr = geis_attr_new(GEIS_GESTURE_ATTRIBUTE_FOCUS_X,
                       GEIS_ATTR_TYPE_FLOAT,
                       &grail_event->focus_x);
  geis_frame_add_attr(frame, attr);

  attr = geis_attr_new(GEIS_GESTURE_ATTRIBUTE_FOCUS_Y,
                       GEIS_ATTR_TYPE_FLOAT,
                       &grail_event->focus_y);
  geis_frame_add_attr(frame, attr);

  num_unmapped_properties = geis_xcb_backend_map_grail_attrs(
                                   grail_event->gesture_type,
                                   be->grail_is_old_version ? 1 : 2,
                                   num_properties, properties, frame);

  /* any remaining properties are touch properties */
  touch_count = num_unmapped_properties / 3;
  properties += num_properties - num_unmapped_properties;
  attr = geis_attr_new(GEIS_GESTURE_ATTRIBUTE_TOUCHES,
                       GEIS_ATTR_TYPE_INTEGER,
                       &touch_count);
  geis_frame_add_attr(frame, attr);

  GeisSize i;
  for (i = 0; i < touch_count; ++i)
  {
    GeisTouch  touch = geis_touch_new(i);

    if (num_unmapped_properties > 0)
    {
      GeisInteger touch_slot = *properties;
      attr = geis_attr_new(GEIS_TOUCH_ATTRIBUTE_ID,
                           GEIS_ATTR_TYPE_INTEGER,
                           &touch_slot);
      geis_touch_add_attr(touch, attr);
      ++properties;
      --num_unmapped_properties;
    }
    if (num_unmapped_properties > 0)
    {
      attr = geis_attr_new(GEIS_TOUCH_ATTRIBUTE_X,
                           GEIS_ATTR_TYPE_FLOAT,
                           properties);
      geis_touch_add_attr(touch, attr);
      ++properties;
      --num_unmapped_properties;
    }
    if (num_unmapped_properties > 0)
    {
      attr = geis_attr_new(GEIS_TOUCH_ATTRIBUTE_Y,
                           GEIS_ATTR_TYPE_FLOAT,
                           properties);
      geis_touch_add_attr(touch, attr);
      ++properties;
      --num_unmapped_properties;
    }
    geis_touchset_insert(touchset, touch);
    geis_frame_add_touchid(frame, geis_touch_id(touch));
  }

  geis_event_add_attr(geis_event, group_attr);
  geis_event_add_attr(geis_event, touch_attr);
  geis_post_event(be->geis, geis_event);
}

#ifdef HAVE_XI_2_1
static void
_report_hierarchy_change(GeisXcbBackend be, xcb_ge_event_t *event)
{
  xXIHierarchyEvent *he = (xXIHierarchyEvent *)event;
  xXIHierarchyInfo *info = (xXIHierarchyInfo *)(event + 1);
  int i;

  for (i = 0; i < he->num_info; ++i, ++info)
  {
    if (info->flags == XISlaveAdded) {
      _report_xcb_devices(be, info->deviceid);
    }
    else if (info->flags == XISlaveRemoved)
    {
      GeisDeviceBag bag = geis_devices(be->geis);
      GeisSize i;

      for (i = 0; i < geis_device_bag_count(bag); ++i)
      {
        GeisDevice device = geis_device_bag_device(bag, i);

        if (geis_device_id(device) == info->deviceid)
        {
          geis_unregister_device(be->geis, device);
          break;
        }
      }
    }
  }
}
#endif

/**
 * Dispatches events coming from XCB.
 */
static void
_xcb_dispatch(GeisXcbBackend be)
{
  if (be->xcb_connection)
  {
    const xcb_query_extension_reply_t *extension_info;
    extension_info = xcb_get_extension_data(be->xcb_connection, &xcb_gesture_id);

    xcb_generic_event_t *event = NULL;
    while ((event = xcb_poll_for_event(be->xcb_connection)))
    {
      xcb_ge_event_t *ge;

      if (event->response_type != GenericEvent) {
	geis_warning("received non-generic event type: %d",
	             event->response_type);
	goto next_event;
      }

      ge = (xcb_ge_event_t *)event;

      /* an error in the xcb protocol, pad0 is actually the extension opcode */
      if (ge->pad0 == extension_info->major_opcode)
      {
        xcb_gesture_notify_event_t *gesture_event =
          (xcb_gesture_notify_event_t*)event;

        if (gesture_event->event_type != XCB_GESTURE_NOTIFY)
        {
          geis_warning("received unrecognized gesture event type: %d",
                       gesture_event->event_type);
          goto next_event;
        }

        _dispatch_gesture(be, gesture_event);
      }
#ifdef HAVE_XI_2_1
      else if (ge->pad0 == be->xi_opcode)
      {
        if (ge->event_type == XI_HierarchyChanged)
          _report_hierarchy_change(be, ge);
        else
          geis_warning("received unrecognized XI event type: %d",
                       ge->event_type);
      }
#endif
      else
        geis_warning("received unrecognized generic event for extension: %d",
                     ge->pad0);

next_event:
      free(event);
    }
  }
}

/** @todo implement this */
void
_fd_callback(int                             fd GEIS_UNUSED,
             GeisBackendMultiplexorActivity  ev GEIS_UNUSED,
             void                           *ctx)
{
  GeisXcbBackend be = (GeisXcbBackend)ctx;
  _xcb_dispatch(be);
}


Geis
geis_xcb_backend_geis(GeisXcbBackend be)
{
  return be->geis;
}


void
geis_xcb_backend_add_token(GeisXcbBackend be, XcbBackendToken token)
{
  uint16_t dev;
  int win;

  for (win = 0; win < geis_xcb_token_window_count(token); ++win)
  {
    xcb_window_t window_id = geis_xcb_token_window(token, win);
    for (dev = 0; dev < geis_xcb_token_device_count(token); ++dev)
    {
      uint16_t device_id = geis_xcb_token_device(token, dev);
      geis_xcb_backend_sub_table_insert(be->sub_table,
                                        window_id,
                                        device_id,
                                        token);
    }
  }
}


void
geis_xcb_backend_remove_token(GeisXcbBackend be, XcbBackendToken token)
{
  geis_xcb_backend_sub_table_remove_token(be->sub_table, token);
}


static void
_compose_grail_masks(XcbBackendToken  token,
                     void            *context)
{
  uint16_t grail_mask_len = geis_xcb_token_grail_mask_len(token);
  uint32_t *grail_mask = geis_xcb_token_grail_mask(token);
  uint32_t *mask = (uint32_t *)context;

  _grail_mask_or(grail_mask_len, mask, grail_mask);
}


void
geis_xcb_backend_select_events(GeisXcbBackend be,
                               uint16_t       device_id,
                               xcb_window_t   window_id)
{
  uint16_t grail_mask_len = 2;
  uint32_t grail_mask[2];
  xcb_generic_error_t  *error;
  xcb_void_cookie_t select_cookie;

  _grail_mask_clear(grail_mask_len, grail_mask);
  geis_xcb_backend_sub_table_foreach(be->sub_table, window_id, device_id,
                                    _compose_grail_masks, grail_mask);

  geis_debug("window_id=0x%08x device_id=%d mask_len=%d mask=0x%08x %08x",
             window_id, device_id, grail_mask_len, grail_mask[1], grail_mask[0]);

  select_cookie = xcb_gesture_select_events_checked(be->xcb_connection,
                                                    window_id,
                                                    device_id,
                                                    grail_mask_len,
                                                    grail_mask);
  error = xcb_request_check(be->xcb_connection, select_cookie);
  if (error)
  {
    geis_error("failed to select events for window 0x%08x", window_id);
  }
}


__attribute__((constructor))
static void _register_xcb_backend()
{
  geis_register_backend(GEIS_INIT_UTOUCH_XCB_BACKEND,
                        sizeof(struct GeisXcbBackend),
                        &be_vtbl);
}

/* A dummy routine to force linkage of this module without dlopening it */
void
geis_include_backend_xcb()
{
}


GeisStatus
_gxcb_accept_gesture(GeisBackend   be GEIS_UNUSED,
                     GeisGroup     group GEIS_UNUSED,
                     GeisGestureId gesture_id GEIS_UNUSED)
{
  return GEIS_STATUS_UNKNOWN_ERROR;
}


GeisStatus
_gxcb_reject_gesture(GeisBackend   be GEIS_UNUSED,
                     GeisGroup     group GEIS_UNUSED,
                     GeisGestureId gesture_id GEIS_UNUSED)
{
  return GEIS_STATUS_UNKNOWN_ERROR;
}

