/* $Id: UIColorThemeEditor.cpp $ */
/** @file
 * VBox Qt GUI - UIColorThemeEditor class implementation.
 */

/*
 * Copyright (C) 2019-2022 Oracle and/or its affiliates.
 *
 * This file is part of VirtualBox base platform packages, as
 * available from https://www.virtualbox.org.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation, in version 3 of the
 * License.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <https://www.gnu.org/licenses>.
 *
 * SPDX-License-Identifier: GPL-3.0-only
 */

/* Qt includes: */
#include <QComboBox>
#include <QGridLayout>
#include <QHBoxLayout>
#include <QLabel>

/* GUI includes: */
#include "UIColorThemeEditor.h"
#include "UIConverter.h"


UIColorThemeEditor::UIColorThemeEditor(QWidget *pParent /* = 0 */)
    : QIWithRetranslateUI<QWidget>(pParent)
    , m_enmValue(UIColorThemeType_Auto)
    , m_pLabel(0)
    , m_pCombo(0)
{
    prepare();
}

void UIColorThemeEditor::setValue(UIColorThemeType enmValue)
{
    /* Update cached value and
     * combo if value has changed: */
    if (m_enmValue != enmValue)
    {
        m_enmValue = enmValue;
        populateCombo();
    }
}

UIColorThemeType UIColorThemeEditor::value() const
{
    return m_pCombo ? m_pCombo->currentData().value<UIColorThemeType>() : m_enmValue;
}

void UIColorThemeEditor::retranslateUi()
{
    if (m_pLabel)
        m_pLabel->setText(tr("Color &Theme:"));
    if (m_pCombo)
    {
        for (int i = 0; i < m_pCombo->count(); ++i)
        {
            const UIColorThemeType enmType = m_pCombo->itemData(i).value<UIColorThemeType>();
            m_pCombo->setItemText(i, gpConverter->toString(enmType));
        }
        m_pCombo->setToolTip(tr("Selects the color theme. It can be Light, Dark or automatically detected (default)."));
    }
}

void UIColorThemeEditor::prepare()
{
    /* Create main layout: */
    QGridLayout *pLayout = new QGridLayout(this);
    if (pLayout)
    {
        pLayout->setContentsMargins(0, 0, 0, 0);

        /* Create label: */
        m_pLabel = new QLabel(this);
        if (m_pLabel)
            pLayout->addWidget(m_pLabel, 0, 0);

        /* Create combo layout: */
        QHBoxLayout *pComboLayout = new QHBoxLayout;
        if (pComboLayout)
        {
            /* Create combo: */
            m_pCombo = new QComboBox(this);
            if (m_pCombo)
            {
                /* This is necessary since contents is dynamical now: */
                m_pCombo->setSizeAdjustPolicy(QComboBox::AdjustToContents);
                if (m_pLabel)
                    m_pLabel->setBuddy(m_pCombo);
                pComboLayout->addWidget(m_pCombo);
            }

            /* Add stretch: */
            pComboLayout->addStretch();

            /* Add combo-layout into main-layout: */
            pLayout->addLayout(pComboLayout, 0, 1);
        }
    }

    /* Populate combo: */
    populateCombo();

    /* Apply language settings: */
    retranslateUi();
}

void UIColorThemeEditor::populateCombo()
{
    if (m_pCombo)
    {
        /* Clear combo first of all: */
        m_pCombo->clear();

        /* Get possible values: */
        QList<UIColorThemeType> possibleValues;
        possibleValues << UIColorThemeType_Auto
                       << UIColorThemeType_Light
                       << UIColorThemeType_Dark;

        /* Update combo with all the possible values: */
        foreach (const UIColorThemeType &enmType, possibleValues)
            m_pCombo->addItem(QString(), QVariant::fromValue(enmType));

        /* Look for proper index to choose: */
        const int iIndex = m_pCombo->findData(QVariant::fromValue(m_enmValue));
        if (iIndex != -1)
            m_pCombo->setCurrentIndex(iIndex);

        /* Retranslate finally: */
        retranslateUi();
    }
}
