/*
 * Copyright (c) 2018-2020, ARM Limited and Contributors. All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

/*
 * ZynqMP system level PM-API functions for pin control.
 */

#include <string.h>

#include <arch_helpers.h>
#include <plat/common/platform.h>

#include "pm_api_pinctrl.h"
#include "pm_api_sys.h"
#include "pm_client.h"
#include "pm_common.h"
#include "pm_ipi.h"

struct pinctrl_function {
	char name[FUNCTION_NAME_LEN];
	uint16_t group_base;
	uint8_t group_size;
	uint8_t regval;
};

/* Max groups for one pin */
#define MAX_PIN_GROUPS	(13U)

struct zynqmp_pin_group {
	uint16_t (*groups)[];
};

static struct pinctrl_function pinctrl_functions[MAX_FUNCTION] =  {
	[PINCTRL_FUNC_CAN0] = {
		.name = "can0",
		.regval = 0x20,
		.group_base = PINCTRL_GRP_CAN0_0,
		.group_size = PINCTRL_GRP_CAN0_18 - PINCTRL_GRP_CAN0_0 + 1U,
	},
	[PINCTRL_FUNC_CAN1] = {
		.name = "can1",
		.regval = 0x20,
		.group_base = PINCTRL_GRP_CAN1_0,
		.group_size = PINCTRL_GRP_CAN1_19 - PINCTRL_GRP_CAN1_0 + 1U,
	},
	[PINCTRL_FUNC_ETHERNET0] = {
		.name = "ethernet0",
		.regval = 0x02,
		.group_base = PINCTRL_GRP_ETHERNET0_0,
		.group_size = PINCTRL_GRP_ETHERNET0_0 - PINCTRL_GRP_ETHERNET0_0 + 1U,
	},
	[PINCTRL_FUNC_ETHERNET1] = {
		.name = "ethernet1",
		.regval = 0x02,
		.group_base = PINCTRL_GRP_ETHERNET1_0,
		.group_size = PINCTRL_GRP_ETHERNET1_0 - PINCTRL_GRP_ETHERNET1_0 + 1U,
	},
	[PINCTRL_FUNC_ETHERNET2] = {
		.name = "ethernet2",
		.regval = 0x02,
		.group_base = PINCTRL_GRP_ETHERNET2_0,
		.group_size = PINCTRL_GRP_ETHERNET2_0 - PINCTRL_GRP_ETHERNET2_0 + 1U,
	},
	[PINCTRL_FUNC_ETHERNET3] = {
		.name = "ethernet3",
		.regval = 0x02,
		.group_base = PINCTRL_GRP_ETHERNET3_0,
		.group_size = PINCTRL_GRP_ETHERNET3_0 - PINCTRL_GRP_ETHERNET3_0 + 1U,
	},
	[PINCTRL_FUNC_GEMTSU0] = {
		.name = "gemtsu0",
		.regval = 0x02,
		.group_base = PINCTRL_GRP_GEMTSU0_0,
		.group_size = PINCTRL_GRP_GEMTSU0_2 - PINCTRL_GRP_GEMTSU0_0 + 1U,
	},
	[PINCTRL_FUNC_GPIO0] = {
		.name = "gpio0",
		.regval = 0x00,
		.group_base = PINCTRL_GRP_GPIO0_0,
		.group_size = PINCTRL_GRP_GPIO0_77 - PINCTRL_GRP_GPIO0_0 + 1U,
	},
	[PINCTRL_FUNC_I2C0] = {
		.name = "i2c0",
		.regval = 0x40,
		.group_base = PINCTRL_GRP_I2C0_0,
		.group_size = PINCTRL_GRP_I2C0_18 - PINCTRL_GRP_I2C0_0 + 1U,
	},
	[PINCTRL_FUNC_I2C1] = {
		.name = "i2c1",
		.regval = 0x40,
		.group_base = PINCTRL_GRP_I2C1_0,
		.group_size = PINCTRL_GRP_I2C1_19 - PINCTRL_GRP_I2C1_0 + 1U,
	},
	[PINCTRL_FUNC_MDIO0] = {
		.name = "mdio0",
		.regval = 0x60,
		.group_base = PINCTRL_GRP_MDIO0_0,
		.group_size = PINCTRL_GRP_MDIO0_0 - PINCTRL_GRP_MDIO0_0 + 1U,
	},
	[PINCTRL_FUNC_MDIO1] = {
		.name = "mdio1",
		.regval = 0x80,
		.group_base = PINCTRL_GRP_MDIO1_0,
		.group_size = PINCTRL_GRP_MDIO1_1 - PINCTRL_GRP_MDIO1_0 + 1U,
	},
	[PINCTRL_FUNC_MDIO2] = {
		.name = "mdio2",
		.regval = 0xa0,
		.group_base = PINCTRL_GRP_MDIO2_0,
		.group_size = PINCTRL_GRP_MDIO2_0 - PINCTRL_GRP_MDIO2_0 + 1U,
	},
	[PINCTRL_FUNC_MDIO3] = {
		.name = "mdio3",
		.regval = 0xc0,
		.group_base = PINCTRL_GRP_MDIO3_0,
		.group_size = PINCTRL_GRP_MDIO3_0 - PINCTRL_GRP_MDIO3_0 + 1U,
	},
	[PINCTRL_FUNC_QSPI0] = {
		.name = "qspi0",
		.regval = 0x02,
		.group_base = PINCTRL_GRP_QSPI0_0,
		.group_size = PINCTRL_GRP_QSPI0_0 - PINCTRL_GRP_QSPI0_0 + 1U,
	},
	[PINCTRL_FUNC_QSPI_FBCLK] = {
		.name = "qspi_fbclk",
		.regval = 0x02,
		.group_base = PINCTRL_GRP_QSPI_FBCLK,
		.group_size = PINCTRL_GRP_QSPI_FBCLK - PINCTRL_GRP_QSPI_FBCLK + 1U,
	},
	[PINCTRL_FUNC_QSPI_SS] = {
		.name = "qspi_ss",
		.regval = 0x02,
		.group_base = PINCTRL_GRP_QSPI_SS,
		.group_size = PINCTRL_GRP_QSPI_SS - PINCTRL_GRP_QSPI_SS + 1U,
	},
	[PINCTRL_FUNC_SPI0] = {
		.name = "spi0",
		.regval = 0x80,
		.group_base = PINCTRL_GRP_SPI0_0,
		.group_size = PINCTRL_GRP_SPI0_5 - PINCTRL_GRP_SPI0_0 + 1U,
	},
	[PINCTRL_FUNC_SPI1] = {
		.name = "spi1",
		.regval = 0x80,
		.group_base = PINCTRL_GRP_SPI1_0,
		.group_size = PINCTRL_GRP_SPI1_5 - PINCTRL_GRP_SPI1_0 + 1U,
	},
	[PINCTRL_FUNC_SPI0_SS] = {
		.name = "spi0_ss",
		.regval = 0x80,
		.group_base = PINCTRL_GRP_SPI0_0_SS0,
		.group_size = PINCTRL_GRP_SPI0_5_SS2 - PINCTRL_GRP_SPI0_0_SS0 + 1U,
	},
	[PINCTRL_FUNC_SPI1_SS] = {
		.name = "spi1_ss",
		.regval = 0x80,
		.group_base = PINCTRL_GRP_SPI1_0_SS0,
		.group_size = PINCTRL_GRP_SPI1_5_SS2 - PINCTRL_GRP_SPI1_0_SS0 + 1U,
	},
	[PINCTRL_FUNC_SDIO0] = {
		.name = "sdio0",
		.regval = 0x08,
		.group_base = PINCTRL_GRP_SDIO0_0,
		.group_size = PINCTRL_GRP_SDIO0_1BIT_2_7 - PINCTRL_GRP_SDIO0_0 + 1U,
	},
	[PINCTRL_FUNC_SDIO0_PC] = {
		.name = "sdio0_pc",
		.regval = 0x08,
		.group_base = PINCTRL_GRP_SDIO0_0_PC,
		.group_size = PINCTRL_GRP_SDIO0_2_PC - PINCTRL_GRP_SDIO0_0_PC + 1U,
	},
	[PINCTRL_FUNC_SDIO0_CD] = {
		.name = "sdio0_cd",
		.regval = 0x08,
		.group_base = PINCTRL_GRP_SDIO0_0_CD,
		.group_size = PINCTRL_GRP_SDIO0_2_CD - PINCTRL_GRP_SDIO0_0_CD + 1U,
	},
	[PINCTRL_FUNC_SDIO0_WP] = {
		.name = "sdio0_wp",
		.regval = 0x08,
		.group_base = PINCTRL_GRP_SDIO0_0_WP,
		.group_size = PINCTRL_GRP_SDIO0_2_WP - PINCTRL_GRP_SDIO0_0_WP + 1U,
	},
	[PINCTRL_FUNC_SDIO1] = {
		.name = "sdio1",
		.regval = 0x10,
		.group_base = PINCTRL_GRP_SDIO1_0,
		.group_size = PINCTRL_GRP_SDIO1_1BIT_1_3 - PINCTRL_GRP_SDIO1_0 + 1U,
	},
	[PINCTRL_FUNC_SDIO1_PC] = {
		.name = "sdio1_pc",
		.regval = 0x10,
		.group_base = PINCTRL_GRP_SDIO1_0_PC,
		.group_size = PINCTRL_GRP_SDIO1_1_PC - PINCTRL_GRP_SDIO1_0_PC + 1U,
	},
	[PINCTRL_FUNC_SDIO1_CD] = {
		.name = "sdio1_cd",
		.regval = 0x10,
		.group_base = PINCTRL_GRP_SDIO1_0_CD,
		.group_size = PINCTRL_GRP_SDIO1_1_CD - PINCTRL_GRP_SDIO1_0_CD + 1U,
	},
	[PINCTRL_FUNC_SDIO1_WP] = {
		.name = "sdio1_wp",
		.regval = 0x10,
		.group_base = PINCTRL_GRP_SDIO1_0_WP,
		.group_size = PINCTRL_GRP_SDIO1_1_WP - PINCTRL_GRP_SDIO1_0_WP + 1U,
	},
	[PINCTRL_FUNC_NAND0] = {
		.name = "nand0",
		.regval = 0x04,
		.group_base = PINCTRL_GRP_NAND0_0,
		.group_size = PINCTRL_GRP_NAND0_0 - PINCTRL_GRP_NAND0_0 + 1U,
	},
	[PINCTRL_FUNC_NAND0_CE] = {
		.name = "nand0_ce",
		.regval = 0x04,
		.group_base = PINCTRL_GRP_NAND0_0_CE,
		.group_size = PINCTRL_GRP_NAND0_1_CE - PINCTRL_GRP_NAND0_0_CE + 1U,
	},
	[PINCTRL_FUNC_NAND0_RB] = {
		.name = "nand0_rb",
		.regval = 0x04,
		.group_base = PINCTRL_GRP_NAND0_0_RB,
		.group_size = PINCTRL_GRP_NAND0_1_RB - PINCTRL_GRP_NAND0_0_RB + 1U,
	},
	[PINCTRL_FUNC_NAND0_DQS] = {
		.name = "nand0_dqs",
		.regval = 0x04,
		.group_base = PINCTRL_GRP_NAND0_0_DQS,
		.group_size = PINCTRL_GRP_NAND0_1_DQS - PINCTRL_GRP_NAND0_0_DQS + 1U,
	},
	[PINCTRL_FUNC_TTC0_CLK] = {
		.name = "ttc0_clk",
		.regval = 0xa0,
		.group_base = PINCTRL_GRP_TTC0_0_CLK,
		.group_size = PINCTRL_GRP_TTC0_8_CLK - PINCTRL_GRP_TTC0_0_CLK + 1U,
	},
	[PINCTRL_FUNC_TTC0_WAV] = {
		.name = "ttc0_wav",
		.regval = 0xa0,
		.group_base = PINCTRL_GRP_TTC0_0_WAV,
		.group_size = PINCTRL_GRP_TTC0_8_WAV - PINCTRL_GRP_TTC0_0_WAV + 1U,
	},
	[PINCTRL_FUNC_TTC1_CLK] = {
		.name = "ttc1_clk",
		.regval = 0xa0,
		.group_base = PINCTRL_GRP_TTC1_0_CLK,
		.group_size = PINCTRL_GRP_TTC1_8_CLK - PINCTRL_GRP_TTC1_0_CLK + 1U,
	},
	[PINCTRL_FUNC_TTC1_WAV] = {
		.name = "ttc1_wav",
		.regval = 0xa0,
		.group_base = PINCTRL_GRP_TTC1_0_WAV,
		.group_size = PINCTRL_GRP_TTC1_8_WAV - PINCTRL_GRP_TTC1_0_WAV + 1U,
	},
	[PINCTRL_FUNC_TTC2_CLK] = {
		.name = "ttc2_clk",
		.regval = 0xa0,
		.group_base = PINCTRL_GRP_TTC2_0_CLK,
		.group_size = PINCTRL_GRP_TTC2_8_CLK - PINCTRL_GRP_TTC2_0_CLK + 1U,
	},
	[PINCTRL_FUNC_TTC2_WAV] = {
		.name = "ttc2_wav",
		.regval = 0xa0,
		.group_base = PINCTRL_GRP_TTC2_0_WAV,
		.group_size = PINCTRL_GRP_TTC2_8_WAV - PINCTRL_GRP_TTC2_0_WAV + 1U,
	},
	[PINCTRL_FUNC_TTC3_CLK] = {
		.name = "ttc3_clk",
		.regval = 0xa0,
		.group_base = PINCTRL_GRP_TTC3_0_CLK,
		.group_size = PINCTRL_GRP_TTC3_8_CLK - PINCTRL_GRP_TTC3_0_CLK + 1U,
	},
	[PINCTRL_FUNC_TTC3_WAV] = {
		.name = "ttc3_wav",
		.regval = 0xa0,
		.group_base = PINCTRL_GRP_TTC3_0_WAV,
		.group_size = PINCTRL_GRP_TTC3_8_WAV - PINCTRL_GRP_TTC3_0_WAV + 1U,
	},
	[PINCTRL_FUNC_UART0] = {
		.name = "uart0",
		.regval = 0xc0,
		.group_base = PINCTRL_GRP_UART0_0,
		.group_size = PINCTRL_GRP_UART0_18 - PINCTRL_GRP_UART0_0 + 1U,
	},
	[PINCTRL_FUNC_UART1] = {
		.name = "uart1",
		.regval = 0xc0,
		.group_base = PINCTRL_GRP_UART1_0,
		.group_size = PINCTRL_GRP_UART1_18 - PINCTRL_GRP_UART1_0 + 1U,
	},
	[PINCTRL_FUNC_USB0] = {
		.name = "usb0",
		.regval = 0x04,
		.group_base = PINCTRL_GRP_USB0_0,
		.group_size = PINCTRL_GRP_USB0_0 - PINCTRL_GRP_USB0_0 + 1U,
	},
	[PINCTRL_FUNC_USB1] = {
		.name = "usb1",
		.regval = 0x04,
		.group_base = PINCTRL_GRP_USB1_0,
		.group_size = PINCTRL_GRP_USB1_0 - PINCTRL_GRP_USB1_0 + 1U,
	},
	[PINCTRL_FUNC_SWDT0_CLK] = {
		.name = "swdt0_clk",
		.regval = 0x60,
		.group_base = PINCTRL_GRP_SWDT0_0_CLK,
		.group_size = PINCTRL_GRP_SWDT0_12_CLK - PINCTRL_GRP_SWDT0_0_CLK + 1U,
	},
	[PINCTRL_FUNC_SWDT0_RST] = {
		.name = "swdt0_rst",
		.regval = 0x60,
		.group_base = PINCTRL_GRP_SWDT0_0_RST,
		.group_size = PINCTRL_GRP_SWDT0_12_RST - PINCTRL_GRP_SWDT0_0_RST + 1U,
	},
	[PINCTRL_FUNC_SWDT1_CLK] = {
		.name = "swdt1_clk",
		.regval = 0x60,
		.group_base = PINCTRL_GRP_SWDT1_0_CLK,
		.group_size = PINCTRL_GRP_SWDT1_12_CLK - PINCTRL_GRP_SWDT1_0_CLK + 1U,
	},
	[PINCTRL_FUNC_SWDT1_RST] = {
		.name = "swdt1_rst",
		.regval = 0x60,
		.group_base = PINCTRL_GRP_SWDT1_0_RST,
		.group_size = PINCTRL_GRP_SWDT1_12_RST - PINCTRL_GRP_SWDT1_0_RST + 1U,
	},
	[PINCTRL_FUNC_PMU0] = {
		.name = "pmu0",
		.regval = 0x08,
		.group_base = PINCTRL_GRP_PMU0_0,
		.group_size = PINCTRL_GRP_PMU0_11 - PINCTRL_GRP_PMU0_0 + 1U,
	},
	[PINCTRL_FUNC_PCIE0] = {
		.name = "pcie0",
		.regval = 0x04,
		.group_base = PINCTRL_GRP_PCIE0_0,
		.group_size = PINCTRL_GRP_PCIE0_7 - PINCTRL_GRP_PCIE0_0 + 1U,
	},
	[PINCTRL_FUNC_CSU0] = {
		.name = "csu0",
		.regval = 0x18,
		.group_base = PINCTRL_GRP_CSU0_0,
		.group_size = PINCTRL_GRP_CSU0_11 - PINCTRL_GRP_CSU0_0 + 1U,
	},
	[PINCTRL_FUNC_DPAUX0] = {
		.name = "dpaux0",
		.regval = 0x18,
		.group_base = PINCTRL_GRP_DPAUX0_0,
		.group_size = PINCTRL_GRP_DPAUX0_3 - PINCTRL_GRP_DPAUX0_0 + 1U,
	},
	[PINCTRL_FUNC_PJTAG0] = {
		.name = "pjtag0",
		.regval = 0x60,
		.group_base = PINCTRL_GRP_PJTAG0_0,
		.group_size = PINCTRL_GRP_PJTAG0_5 - PINCTRL_GRP_PJTAG0_0 + 1U,
	},
	[PINCTRL_FUNC_TRACE0] = {
		.name = "trace0",
		.regval = 0xe0,
		.group_base = PINCTRL_GRP_TRACE0_0,
		.group_size = PINCTRL_GRP_TRACE0_2 - PINCTRL_GRP_TRACE0_0 + 1U,
	},
	[PINCTRL_FUNC_TRACE0_CLK] = {
		.name = "trace0_clk",
		.regval = 0xe0,
		.group_base = PINCTRL_GRP_TRACE0_0_CLK,
		.group_size = PINCTRL_GRP_TRACE0_2_CLK - PINCTRL_GRP_TRACE0_0_CLK + 1U,
	},
	[PINCTRL_FUNC_TESTSCAN0] = {
		.name = "testscan0",
		.regval = 0x10,
		.group_base = PINCTRL_GRP_TESTSCAN0_0,
		.group_size = PINCTRL_GRP_TESTSCAN0_0 - PINCTRL_GRP_TESTSCAN0_0 + 1U,
	},
};

static struct zynqmp_pin_group zynqmp_pin_groups[MAX_PIN] = {
	[PINCTRL_PIN_0] = {
		.groups = &((uint16_t []) {
			PINCTRL_GRP_QSPI0_0,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_TESTSCAN0_0,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_GPIO0_0,
			PINCTRL_GRP_CAN1_0,
			PINCTRL_GRP_I2C1_0,
			PINCTRL_GRP_PJTAG0_0,
			PINCTRL_GRP_SPI0_0,
			PINCTRL_GRP_TTC3_0_CLK,
			PINCTRL_GRP_UART1_0,
			PINCTRL_GRP_TRACE0_0_CLK,
			END_OF_GROUPS,
		}),
	},
	[PINCTRL_PIN_1] = {
		.groups = &((uint16_t []) {
			PINCTRL_GRP_QSPI0_0,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_TESTSCAN0_0,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_GPIO0_1,
			PINCTRL_GRP_CAN1_0,
			PINCTRL_GRP_I2C1_0,
			PINCTRL_GRP_PJTAG0_0,
			PINCTRL_GRP_SPI0_0_SS2,
			PINCTRL_GRP_TTC3_0_WAV,
			PINCTRL_GRP_UART1_0,
			PINCTRL_GRP_TRACE0_0_CLK,
			END_OF_GROUPS,
		}),
	},
	[PINCTRL_PIN_2] = {
		.groups = &((uint16_t []) {
			PINCTRL_GRP_QSPI0_0,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_TESTSCAN0_0,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_GPIO0_2,
			PINCTRL_GRP_CAN0_0,
			PINCTRL_GRP_I2C0_0,
			PINCTRL_GRP_PJTAG0_0,
			PINCTRL_GRP_SPI0_0_SS1,
			PINCTRL_GRP_TTC2_0_CLK,
			PINCTRL_GRP_UART0_0,
			PINCTRL_GRP_TRACE0_0,
			END_OF_GROUPS,
		}),
	},
	[PINCTRL_PIN_3] = {
		.groups = &((uint16_t []) {
			PINCTRL_GRP_QSPI0_0,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_TESTSCAN0_0,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_GPIO0_3,
			PINCTRL_GRP_CAN0_0,
			PINCTRL_GRP_I2C0_0,
			PINCTRL_GRP_PJTAG0_0,
			PINCTRL_GRP_SPI0_0_SS0,
			PINCTRL_GRP_TTC2_0_WAV,
			PINCTRL_GRP_UART0_0,
			PINCTRL_GRP_TRACE0_0,
			END_OF_GROUPS,
		}),
	},
	[PINCTRL_PIN_4] = {
		.groups = &((uint16_t []) {
			PINCTRL_GRP_QSPI0_0,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_TESTSCAN0_0,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_GPIO0_4,
			PINCTRL_GRP_CAN1_1,
			PINCTRL_GRP_I2C1_1,
			PINCTRL_GRP_SWDT1_0_CLK,
			PINCTRL_GRP_SPI0_0,
			PINCTRL_GRP_TTC1_0_CLK,
			PINCTRL_GRP_UART1_1,
			PINCTRL_GRP_TRACE0_0,
			END_OF_GROUPS,
		}),
	},
	[PINCTRL_PIN_5] = {
		.groups = &((uint16_t []) {
			PINCTRL_GRP_QSPI_SS,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_TESTSCAN0_0,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_GPIO0_5,
			PINCTRL_GRP_CAN1_1,
			PINCTRL_GRP_I2C1_1,
			PINCTRL_GRP_SWDT1_0_RST,
			PINCTRL_GRP_SPI0_0,
			PINCTRL_GRP_TTC1_0_WAV,
			PINCTRL_GRP_UART1_1,
			PINCTRL_GRP_TRACE0_0,
			END_OF_GROUPS,
		}),
	},
	[PINCTRL_PIN_6] = {
		.groups = &((uint16_t []) {
			PINCTRL_GRP_QSPI_FBCLK,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_TESTSCAN0_0,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_GPIO0_6,
			PINCTRL_GRP_CAN0_1,
			PINCTRL_GRP_I2C0_1,
			PINCTRL_GRP_SWDT0_0_CLK,
			PINCTRL_GRP_SPI1_0,
			PINCTRL_GRP_TTC0_0_CLK,
			PINCTRL_GRP_UART0_1,
			PINCTRL_GRP_TRACE0_0,
			END_OF_GROUPS,
		}),
	},
	[PINCTRL_PIN_7] = {
		.groups = &((uint16_t []) {
			PINCTRL_GRP_QSPI_SS,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_TESTSCAN0_0,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_GPIO0_7,
			PINCTRL_GRP_CAN0_1,
			PINCTRL_GRP_I2C0_1,
			PINCTRL_GRP_SWDT0_0_RST,
			PINCTRL_GRP_SPI1_0_SS2,
			PINCTRL_GRP_TTC0_0_WAV,
			PINCTRL_GRP_UART0_1,
			PINCTRL_GRP_TRACE0_0,
			END_OF_GROUPS,
		}),
	},
	[PINCTRL_PIN_8] = {
		.groups = &((uint16_t []) {
			PINCTRL_GRP_QSPI0_0,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_TESTSCAN0_0,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_GPIO0_8,
			PINCTRL_GRP_CAN1_2,
			PINCTRL_GRP_I2C1_2,
			PINCTRL_GRP_SWDT1_1_CLK,
			PINCTRL_GRP_SPI1_0_SS1,
			PINCTRL_GRP_TTC3_1_CLK,
			PINCTRL_GRP_UART1_2,
			PINCTRL_GRP_TRACE0_0,
			END_OF_GROUPS,
		}),
	},
	[PINCTRL_PIN_9] = {
		.groups = &((uint16_t []) {
			PINCTRL_GRP_QSPI0_0,
			PINCTRL_GRP_NAND0_0_CE,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_TESTSCAN0_0,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_GPIO0_9,
			PINCTRL_GRP_CAN1_2,
			PINCTRL_GRP_I2C1_2,
			PINCTRL_GRP_SWDT1_1_RST,
			PINCTRL_GRP_SPI1_0_SS0,
			PINCTRL_GRP_TTC3_1_WAV,
			PINCTRL_GRP_UART1_2,
			PINCTRL_GRP_TRACE0_0,
			END_OF_GROUPS,
		}),
	},
	[PINCTRL_PIN_10] = {
		.groups = &((uint16_t []) {
			PINCTRL_GRP_QSPI0_0,
			PINCTRL_GRP_NAND0_0_RB,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_TESTSCAN0_0,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_GPIO0_10,
			PINCTRL_GRP_CAN0_2,
			PINCTRL_GRP_I2C0_2,
			PINCTRL_GRP_SWDT0_1_CLK,
			PINCTRL_GRP_SPI1_0,
			PINCTRL_GRP_TTC2_1_CLK,
			PINCTRL_GRP_UART0_2,
			PINCTRL_GRP_TRACE0_0,
			END_OF_GROUPS,
		}),
	},
	[PINCTRL_PIN_11] = {
		.groups = &((uint16_t []) {
			PINCTRL_GRP_QSPI0_0,
			PINCTRL_GRP_NAND0_0_RB,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_TESTSCAN0_0,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_GPIO0_11,
			PINCTRL_GRP_CAN0_2,
			PINCTRL_GRP_I2C0_2,
			PINCTRL_GRP_SWDT0_1_RST,
			PINCTRL_GRP_SPI1_0,
			PINCTRL_GRP_TTC2_1_WAV,
			PINCTRL_GRP_UART0_2,
			PINCTRL_GRP_TRACE0_0,
			END_OF_GROUPS,
		}),
	},
	[PINCTRL_PIN_12] = {
		.groups = &((uint16_t []) {
			PINCTRL_GRP_QSPI0_0,
			PINCTRL_GRP_NAND0_0_DQS,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_TESTSCAN0_0,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_GPIO0_12,
			PINCTRL_GRP_CAN1_3,
			PINCTRL_GRP_I2C1_3,
			PINCTRL_GRP_PJTAG0_1,
			PINCTRL_GRP_SPI0_1,
			PINCTRL_GRP_TTC1_1_CLK,
			PINCTRL_GRP_UART1_3,
			PINCTRL_GRP_TRACE0_0,
			END_OF_GROUPS,
		}),
	},
	[PINCTRL_PIN_13] = {
		.groups = &((uint16_t []) {
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_NAND0_0,
			PINCTRL_GRP_SDIO0_0,
			PINCTRL_GRP_TESTSCAN0_0,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_GPIO0_13,
			PINCTRL_GRP_CAN1_3,
			PINCTRL_GRP_I2C1_3,
			PINCTRL_GRP_PJTAG0_1,
			PINCTRL_GRP_SPI0_1_SS2,
			PINCTRL_GRP_TTC1_1_WAV,
			PINCTRL_GRP_UART1_3,
			PINCTRL_GRP_TRACE0_0,
			PINCTRL_GRP_SDIO0_4BIT_0_0,
			PINCTRL_GRP_SDIO0_1BIT_0_0,
			END_OF_GROUPS,
		}),
	},
	[PINCTRL_PIN_14] = {
		.groups = &((uint16_t []) {
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_NAND0_0,
			PINCTRL_GRP_SDIO0_0,
			PINCTRL_GRP_TESTSCAN0_0,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_GPIO0_14,
			PINCTRL_GRP_CAN0_3,
			PINCTRL_GRP_I2C0_3,
			PINCTRL_GRP_PJTAG0_1,
			PINCTRL_GRP_SPI0_1_SS1,
			PINCTRL_GRP_TTC0_1_CLK,
			PINCTRL_GRP_UART0_3,
			PINCTRL_GRP_TRACE0_0,
			PINCTRL_GRP_SDIO0_4BIT_0_0,
			PINCTRL_GRP_SDIO0_1BIT_0_1,
			END_OF_GROUPS,
		}),
	},
	[PINCTRL_PIN_15] = {
		.groups = &((uint16_t []) {
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_NAND0_0,
			PINCTRL_GRP_SDIO0_0,
			PINCTRL_GRP_TESTSCAN0_0,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_GPIO0_15,
			PINCTRL_GRP_CAN0_3,
			PINCTRL_GRP_I2C0_3,
			PINCTRL_GRP_PJTAG0_1,
			PINCTRL_GRP_SPI0_1_SS0,
			PINCTRL_GRP_TTC0_1_WAV,
			PINCTRL_GRP_UART0_3,
			PINCTRL_GRP_TRACE0_0,
			PINCTRL_GRP_SDIO0_4BIT_0_0,
			PINCTRL_GRP_SDIO0_1BIT_0_2,
			END_OF_GROUPS,
		}),
	},
	[PINCTRL_PIN_16] = {
		.groups = &((uint16_t []) {
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_NAND0_0,
			PINCTRL_GRP_SDIO0_0,
			PINCTRL_GRP_TESTSCAN0_0,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_GPIO0_16,
			PINCTRL_GRP_CAN1_4,
			PINCTRL_GRP_I2C1_4,
			PINCTRL_GRP_SWDT1_2_CLK,
			PINCTRL_GRP_SPI0_1,
			PINCTRL_GRP_TTC3_2_CLK,
			PINCTRL_GRP_UART1_4,
			PINCTRL_GRP_TRACE0_0,
			PINCTRL_GRP_SDIO0_4BIT_0_0,
			PINCTRL_GRP_SDIO0_1BIT_0_3,
			END_OF_GROUPS,
		}),
	},
	[PINCTRL_PIN_17] = {
		.groups = &((uint16_t []) {
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_NAND0_0,
			PINCTRL_GRP_SDIO0_0,
			PINCTRL_GRP_TESTSCAN0_0,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_GPIO0_17,
			PINCTRL_GRP_CAN1_4,
			PINCTRL_GRP_I2C1_4,
			PINCTRL_GRP_SWDT1_2_RST,
			PINCTRL_GRP_SPI0_1,
			PINCTRL_GRP_TTC3_2_WAV,
			PINCTRL_GRP_UART1_4,
			PINCTRL_GRP_TRACE0_0,
			PINCTRL_GRP_SDIO0_4BIT_0_1,
			PINCTRL_GRP_SDIO0_1BIT_0_4,
			END_OF_GROUPS,
		}),
	},
	[PINCTRL_PIN_18] = {
		.groups = &((uint16_t []) {
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_NAND0_0,
			PINCTRL_GRP_SDIO0_0,
			PINCTRL_GRP_TESTSCAN0_0,
			PINCTRL_GRP_CSU0_0,
			PINCTRL_GRP_GPIO0_18,
			PINCTRL_GRP_CAN0_4,
			PINCTRL_GRP_I2C0_4,
			PINCTRL_GRP_SWDT0_2_CLK,
			PINCTRL_GRP_SPI1_1,
			PINCTRL_GRP_TTC2_2_CLK,
			PINCTRL_GRP_UART0_4,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_SDIO0_4BIT_0_1,
			PINCTRL_GRP_SDIO0_1BIT_0_5,
			END_OF_GROUPS,
		}),
	},
	[PINCTRL_PIN_19] = {
		.groups = &((uint16_t []) {
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_NAND0_0,
			PINCTRL_GRP_SDIO0_0,
			PINCTRL_GRP_TESTSCAN0_0,
			PINCTRL_GRP_CSU0_1,
			PINCTRL_GRP_GPIO0_19,
			PINCTRL_GRP_CAN0_4,
			PINCTRL_GRP_I2C0_4,
			PINCTRL_GRP_SWDT0_2_RST,
			PINCTRL_GRP_SPI1_1_SS2,
			PINCTRL_GRP_TTC2_2_WAV,
			PINCTRL_GRP_UART0_4,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_SDIO0_4BIT_0_1,
			PINCTRL_GRP_SDIO0_1BIT_0_6,
			END_OF_GROUPS,
		}),
	},
	[PINCTRL_PIN_20] = {
		.groups = &((uint16_t []) {
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_NAND0_0,
			PINCTRL_GRP_SDIO0_0,
			PINCTRL_GRP_TESTSCAN0_0,
			PINCTRL_GRP_CSU0_2,
			PINCTRL_GRP_GPIO0_20,
			PINCTRL_GRP_CAN1_5,
			PINCTRL_GRP_I2C1_5,
			PINCTRL_GRP_SWDT1_3_CLK,
			PINCTRL_GRP_SPI1_1_SS1,
			PINCTRL_GRP_TTC1_2_CLK,
			PINCTRL_GRP_UART1_5,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_SDIO0_4BIT_0_1,
			PINCTRL_GRP_SDIO0_1BIT_0_7,
			END_OF_GROUPS,
		}),
	},
	[PINCTRL_PIN_21] = {
		.groups = &((uint16_t []) {
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_NAND0_0,
			PINCTRL_GRP_SDIO0_0,
			PINCTRL_GRP_TESTSCAN0_0,
			PINCTRL_GRP_CSU0_3,
			PINCTRL_GRP_GPIO0_21,
			PINCTRL_GRP_CAN1_5,
			PINCTRL_GRP_I2C1_5,
			PINCTRL_GRP_SWDT1_3_RST,
			PINCTRL_GRP_SPI1_1_SS0,
			PINCTRL_GRP_TTC1_2_WAV,
			PINCTRL_GRP_UART1_5,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_SDIO0_4BIT_0_0,
			PINCTRL_GRP_SDIO0_4BIT_0_1,
			PINCTRL_GRP_SDIO0_1BIT_0_0,
			PINCTRL_GRP_SDIO0_1BIT_0_1,
			PINCTRL_GRP_SDIO0_1BIT_0_2,
			PINCTRL_GRP_SDIO0_1BIT_0_3,
			PINCTRL_GRP_SDIO0_1BIT_0_4,
			PINCTRL_GRP_SDIO0_1BIT_0_5,
			PINCTRL_GRP_SDIO0_1BIT_0_6,
			PINCTRL_GRP_SDIO0_1BIT_0_7,
			END_OF_GROUPS,
		}),
	},
	[PINCTRL_PIN_22] = {
		.groups = &((uint16_t []) {
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_NAND0_0,
			PINCTRL_GRP_SDIO0_0,
			PINCTRL_GRP_TESTSCAN0_0,
			PINCTRL_GRP_CSU0_4,
			PINCTRL_GRP_GPIO0_22,
			PINCTRL_GRP_CAN0_5,
			PINCTRL_GRP_I2C0_5,
			PINCTRL_GRP_SWDT0_3_CLK,
			PINCTRL_GRP_SPI1_1,
			PINCTRL_GRP_TTC0_2_CLK,
			PINCTRL_GRP_UART0_5,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_SDIO0_4BIT_0_0,
			PINCTRL_GRP_SDIO0_4BIT_0_1,
			PINCTRL_GRP_SDIO0_1BIT_0_0,
			PINCTRL_GRP_SDIO0_1BIT_0_1,
			PINCTRL_GRP_SDIO0_1BIT_0_2,
			PINCTRL_GRP_SDIO0_1BIT_0_3,
			PINCTRL_GRP_SDIO0_1BIT_0_4,
			PINCTRL_GRP_SDIO0_1BIT_0_5,
			PINCTRL_GRP_SDIO0_1BIT_0_6,
			PINCTRL_GRP_SDIO0_1BIT_0_7,
			END_OF_GROUPS,
		}),
	},
	[PINCTRL_PIN_23] = {
		.groups = &((uint16_t []) {
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_NAND0_0,
			PINCTRL_GRP_SDIO0_0_PC,
			PINCTRL_GRP_TESTSCAN0_0,
			PINCTRL_GRP_CSU0_5,
			PINCTRL_GRP_GPIO0_23,
			PINCTRL_GRP_CAN0_5,
			PINCTRL_GRP_I2C0_5,
			PINCTRL_GRP_SWDT0_3_RST,
			PINCTRL_GRP_SPI1_1,
			PINCTRL_GRP_TTC0_2_WAV,
			PINCTRL_GRP_UART0_5,
			PINCTRL_GRP_RESERVED,
			END_OF_GROUPS,
		}),
	},
	[PINCTRL_PIN_24] = {
		.groups = &((uint16_t []) {
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_NAND0_0,
			PINCTRL_GRP_SDIO0_0_CD,
			PINCTRL_GRP_TESTSCAN0_0,
			PINCTRL_GRP_CSU0_6,
			PINCTRL_GRP_GPIO0_24,
			PINCTRL_GRP_CAN1_6,
			PINCTRL_GRP_I2C1_6,
			PINCTRL_GRP_SWDT1_4_CLK,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_TTC3_3_CLK,
			PINCTRL_GRP_UART1_6,
			PINCTRL_GRP_RESERVED,
			END_OF_GROUPS,
		}),
	},
	[PINCTRL_PIN_25] = {
		.groups = &((uint16_t []) {
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_NAND0_0,
			PINCTRL_GRP_SDIO0_0_WP,
			PINCTRL_GRP_TESTSCAN0_0,
			PINCTRL_GRP_CSU0_7,
			PINCTRL_GRP_GPIO0_25,
			PINCTRL_GRP_CAN1_6,
			PINCTRL_GRP_I2C1_6,
			PINCTRL_GRP_SWDT1_4_RST,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_TTC3_3_WAV,
			PINCTRL_GRP_UART1_6,
			PINCTRL_GRP_RESERVED,
			END_OF_GROUPS,
		}),
	},
	[PINCTRL_PIN_26] = {
		.groups = &((uint16_t []) {
			PINCTRL_GRP_ETHERNET0_0,
			PINCTRL_GRP_GEMTSU0_0,
			PINCTRL_GRP_NAND0_1_CE,
			PINCTRL_GRP_PMU0_0,
			PINCTRL_GRP_TESTSCAN0_0,
			PINCTRL_GRP_CSU0_8,
			PINCTRL_GRP_GPIO0_26,
			PINCTRL_GRP_CAN0_6,
			PINCTRL_GRP_I2C0_6,
			PINCTRL_GRP_PJTAG0_2,
			PINCTRL_GRP_SPI0_2,
			PINCTRL_GRP_TTC2_3_CLK,
			PINCTRL_GRP_UART0_6,
			PINCTRL_GRP_TRACE0_1,
			END_OF_GROUPS,
		}),
	},
	[PINCTRL_PIN_27] = {
		.groups = &((uint16_t []) {
			PINCTRL_GRP_ETHERNET0_0,
			PINCTRL_GRP_NAND0_1_RB,
			PINCTRL_GRP_PMU0_1,
			PINCTRL_GRP_TESTSCAN0_0,
			PINCTRL_GRP_DPAUX0_0,
			PINCTRL_GRP_GPIO0_27,
			PINCTRL_GRP_CAN0_6,
			PINCTRL_GRP_I2C0_6,
			PINCTRL_GRP_PJTAG0_2,
			PINCTRL_GRP_SPI0_2_SS2,
			PINCTRL_GRP_TTC2_3_WAV,
			PINCTRL_GRP_UART0_6,
			PINCTRL_GRP_TRACE0_1,
			END_OF_GROUPS,
		}),
	},
	[PINCTRL_PIN_28] = {
		.groups = &((uint16_t []) {
			PINCTRL_GRP_ETHERNET0_0,
			PINCTRL_GRP_NAND0_1_RB,
			PINCTRL_GRP_PMU0_2,
			PINCTRL_GRP_TESTSCAN0_0,
			PINCTRL_GRP_DPAUX0_0,
			PINCTRL_GRP_GPIO0_28,
			PINCTRL_GRP_CAN1_7,
			PINCTRL_GRP_I2C1_7,
			PINCTRL_GRP_PJTAG0_2,
			PINCTRL_GRP_SPI0_2_SS1,
			PINCTRL_GRP_TTC1_3_CLK,
			PINCTRL_GRP_UART1_7,
			PINCTRL_GRP_TRACE0_1,
			END_OF_GROUPS,
		}),
	},
	[PINCTRL_PIN_29] = {
		.groups = &((uint16_t []) {
			PINCTRL_GRP_ETHERNET0_0,
			PINCTRL_GRP_PCIE0_0,
			PINCTRL_GRP_PMU0_3,
			PINCTRL_GRP_TESTSCAN0_0,
			PINCTRL_GRP_DPAUX0_1,
			PINCTRL_GRP_GPIO0_29,
			PINCTRL_GRP_CAN1_7,
			PINCTRL_GRP_I2C1_7,
			PINCTRL_GRP_PJTAG0_2,
			PINCTRL_GRP_SPI0_2_SS0,
			PINCTRL_GRP_TTC1_3_WAV,
			PINCTRL_GRP_UART1_7,
			PINCTRL_GRP_TRACE0_1,
			END_OF_GROUPS,
		}),
	},
	[PINCTRL_PIN_30] = {
		.groups = &((uint16_t []) {
			PINCTRL_GRP_ETHERNET0_0,
			PINCTRL_GRP_PCIE0_1,
			PINCTRL_GRP_PMU0_4,
			PINCTRL_GRP_TESTSCAN0_0,
			PINCTRL_GRP_DPAUX0_1,
			PINCTRL_GRP_GPIO0_30,
			PINCTRL_GRP_CAN0_7,
			PINCTRL_GRP_I2C0_7,
			PINCTRL_GRP_SWDT0_4_CLK,
			PINCTRL_GRP_SPI0_2,
			PINCTRL_GRP_TTC0_3_CLK,
			PINCTRL_GRP_UART0_7,
			PINCTRL_GRP_TRACE0_1,
			END_OF_GROUPS,
		}),
	},
	[PINCTRL_PIN_31] = {
		.groups = &((uint16_t []) {
			PINCTRL_GRP_ETHERNET0_0,
			PINCTRL_GRP_PCIE0_2,
			PINCTRL_GRP_PMU0_5,
			PINCTRL_GRP_TESTSCAN0_0,
			PINCTRL_GRP_CSU0_9,
			PINCTRL_GRP_GPIO0_31,
			PINCTRL_GRP_CAN0_7,
			PINCTRL_GRP_I2C0_7,
			PINCTRL_GRP_SWDT0_4_RST,
			PINCTRL_GRP_SPI0_2,
			PINCTRL_GRP_TTC0_3_WAV,
			PINCTRL_GRP_UART0_7,
			PINCTRL_GRP_TRACE0_1,
			END_OF_GROUPS,
		}),
	},
	[PINCTRL_PIN_32] = {
		.groups = &((uint16_t []) {
			PINCTRL_GRP_ETHERNET0_0,
			PINCTRL_GRP_NAND0_1_DQS,
			PINCTRL_GRP_PMU0_6,
			PINCTRL_GRP_TESTSCAN0_0,
			PINCTRL_GRP_CSU0_10,
			PINCTRL_GRP_GPIO0_32,
			PINCTRL_GRP_CAN1_8,
			PINCTRL_GRP_I2C1_8,
			PINCTRL_GRP_SWDT1_5_CLK,
			PINCTRL_GRP_SPI1_2,
			PINCTRL_GRP_TTC3_4_CLK,
			PINCTRL_GRP_UART1_8,
			PINCTRL_GRP_TRACE0_1,
			END_OF_GROUPS,
		}),
	},
	[PINCTRL_PIN_33] = {
		.groups = &((uint16_t []) {
			PINCTRL_GRP_ETHERNET0_0,
			PINCTRL_GRP_PCIE0_3,
			PINCTRL_GRP_PMU0_7,
			PINCTRL_GRP_TESTSCAN0_0,
			PINCTRL_GRP_CSU0_11,
			PINCTRL_GRP_GPIO0_33,
			PINCTRL_GRP_CAN1_8,
			PINCTRL_GRP_I2C1_8,
			PINCTRL_GRP_SWDT1_5_RST,
			PINCTRL_GRP_SPI1_2_SS2,
			PINCTRL_GRP_TTC3_4_WAV,
			PINCTRL_GRP_UART1_8,
			PINCTRL_GRP_TRACE0_1,
			END_OF_GROUPS,
		}),
	},
	[PINCTRL_PIN_34] = {
		.groups = &((uint16_t []) {
			PINCTRL_GRP_ETHERNET0_0,
			PINCTRL_GRP_PCIE0_4,
			PINCTRL_GRP_PMU0_8,
			PINCTRL_GRP_TESTSCAN0_0,
			PINCTRL_GRP_DPAUX0_2,
			PINCTRL_GRP_GPIO0_34,
			PINCTRL_GRP_CAN0_8,
			PINCTRL_GRP_I2C0_8,
			PINCTRL_GRP_SWDT0_5_CLK,
			PINCTRL_GRP_SPI1_2_SS1,
			PINCTRL_GRP_TTC2_4_CLK,
			PINCTRL_GRP_UART0_8,
			PINCTRL_GRP_TRACE0_1,
			END_OF_GROUPS,
		}),
	},
	[PINCTRL_PIN_35] = {
		.groups = &((uint16_t []) {
			PINCTRL_GRP_ETHERNET0_0,
			PINCTRL_GRP_PCIE0_5,
			PINCTRL_GRP_PMU0_9,
			PINCTRL_GRP_TESTSCAN0_0,
			PINCTRL_GRP_DPAUX0_2,
			PINCTRL_GRP_GPIO0_35,
			PINCTRL_GRP_CAN0_8,
			PINCTRL_GRP_I2C0_8,
			PINCTRL_GRP_SWDT0_5_RST,
			PINCTRL_GRP_SPI1_2_SS0,
			PINCTRL_GRP_TTC2_4_WAV,
			PINCTRL_GRP_UART0_8,
			PINCTRL_GRP_TRACE0_1,
			END_OF_GROUPS,
		}),
	},
	[PINCTRL_PIN_36] = {
		.groups = &((uint16_t []) {
			PINCTRL_GRP_ETHERNET0_0,
			PINCTRL_GRP_PCIE0_6,
			PINCTRL_GRP_PMU0_10,
			PINCTRL_GRP_TESTSCAN0_0,
			PINCTRL_GRP_DPAUX0_3,
			PINCTRL_GRP_GPIO0_36,
			PINCTRL_GRP_CAN1_9,
			PINCTRL_GRP_I2C1_9,
			PINCTRL_GRP_SWDT1_6_CLK,
			PINCTRL_GRP_SPI1_2,
			PINCTRL_GRP_TTC1_4_CLK,
			PINCTRL_GRP_UART1_9,
			PINCTRL_GRP_TRACE0_1,
			END_OF_GROUPS,
		}),
	},
	[PINCTRL_PIN_37] = {
		.groups = &((uint16_t []) {
			PINCTRL_GRP_ETHERNET0_0,
			PINCTRL_GRP_PCIE0_7,
			PINCTRL_GRP_PMU0_11,
			PINCTRL_GRP_TESTSCAN0_0,
			PINCTRL_GRP_DPAUX0_3,
			PINCTRL_GRP_GPIO0_37,
			PINCTRL_GRP_CAN1_9,
			PINCTRL_GRP_I2C1_9,
			PINCTRL_GRP_SWDT1_6_RST,
			PINCTRL_GRP_SPI1_2,
			PINCTRL_GRP_TTC1_4_WAV,
			PINCTRL_GRP_UART1_9,
			PINCTRL_GRP_TRACE0_1,
			END_OF_GROUPS,
		}),
	},
	[PINCTRL_PIN_38] = {
		.groups = &((uint16_t []) {
			PINCTRL_GRP_ETHERNET1_0,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_SDIO0_1,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_GPIO0_38,
			PINCTRL_GRP_CAN0_9,
			PINCTRL_GRP_I2C0_9,
			PINCTRL_GRP_PJTAG0_3,
			PINCTRL_GRP_SPI0_3,
			PINCTRL_GRP_TTC0_4_CLK,
			PINCTRL_GRP_UART0_9,
			PINCTRL_GRP_TRACE0_1_CLK,
			PINCTRL_GRP_SDIO0_4BIT_1_0,
			PINCTRL_GRP_SDIO0_4BIT_1_1,
			PINCTRL_GRP_SDIO0_1BIT_1_0,
			PINCTRL_GRP_SDIO0_1BIT_1_1,
			PINCTRL_GRP_SDIO0_1BIT_1_2,
			PINCTRL_GRP_SDIO0_1BIT_1_3,
			PINCTRL_GRP_SDIO0_1BIT_1_4,
			PINCTRL_GRP_SDIO0_1BIT_1_5,
			PINCTRL_GRP_SDIO0_1BIT_1_6,
			PINCTRL_GRP_SDIO0_1BIT_1_7,
			END_OF_GROUPS,
		}),
	},
	[PINCTRL_PIN_39] = {
		.groups = &((uint16_t []) {
			PINCTRL_GRP_ETHERNET1_0,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_SDIO0_1_CD,
			PINCTRL_GRP_SDIO1_0,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_GPIO0_39,
			PINCTRL_GRP_CAN0_9,
			PINCTRL_GRP_I2C0_9,
			PINCTRL_GRP_PJTAG0_3,
			PINCTRL_GRP_SPI0_3_SS2,
			PINCTRL_GRP_TTC0_4_WAV,
			PINCTRL_GRP_UART0_9,
			PINCTRL_GRP_TRACE0_1_CLK,
			PINCTRL_GRP_SDIO1_4BIT_0_0,
			PINCTRL_GRP_SDIO1_1BIT_0_0,
			END_OF_GROUPS,
		}),
	},
	[PINCTRL_PIN_40] = {
		.groups = &((uint16_t []) {
			PINCTRL_GRP_ETHERNET1_0,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_SDIO0_1,
			PINCTRL_GRP_SDIO1_0,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_GPIO0_40,
			PINCTRL_GRP_CAN1_10,
			PINCTRL_GRP_I2C1_10,
			PINCTRL_GRP_PJTAG0_3,
			PINCTRL_GRP_SPI0_3_SS1,
			PINCTRL_GRP_TTC3_5_CLK,
			PINCTRL_GRP_UART1_10,
			PINCTRL_GRP_TRACE0_1,
			PINCTRL_GRP_SDIO0_4BIT_1_0,
			PINCTRL_GRP_SDIO0_4BIT_1_1,
			PINCTRL_GRP_SDIO0_1BIT_1_0,
			PINCTRL_GRP_SDIO0_1BIT_1_1,
			PINCTRL_GRP_SDIO0_1BIT_1_2,
			PINCTRL_GRP_SDIO0_1BIT_1_3,
			PINCTRL_GRP_SDIO0_1BIT_1_4,
			PINCTRL_GRP_SDIO0_1BIT_1_5,
			PINCTRL_GRP_SDIO0_1BIT_1_6,
			PINCTRL_GRP_SDIO0_1BIT_1_7,
			PINCTRL_GRP_SDIO1_4BIT_0_0,
			PINCTRL_GRP_SDIO1_1BIT_0_1,
			END_OF_GROUPS,
		}),
	},
	[PINCTRL_PIN_41] = {
		.groups = &((uint16_t []) {
			PINCTRL_GRP_ETHERNET1_0,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_SDIO0_1,
			PINCTRL_GRP_SDIO1_0,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_GPIO0_41,
			PINCTRL_GRP_CAN1_10,
			PINCTRL_GRP_I2C1_10,
			PINCTRL_GRP_PJTAG0_3,
			PINCTRL_GRP_SPI0_3_SS0,
			PINCTRL_GRP_TTC3_5_WAV,
			PINCTRL_GRP_UART1_10,
			PINCTRL_GRP_TRACE0_1,
			PINCTRL_GRP_SDIO0_4BIT_1_0,
			PINCTRL_GRP_SDIO0_1BIT_1_0,
			PINCTRL_GRP_SDIO1_4BIT_0_0,
			PINCTRL_GRP_SDIO1_1BIT_0_2,
			END_OF_GROUPS,
		}),
	},
	[PINCTRL_PIN_42] = {
		.groups = &((uint16_t []) {
			PINCTRL_GRP_ETHERNET1_0,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_SDIO0_1,
			PINCTRL_GRP_SDIO1_0,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_GPIO0_42,
			PINCTRL_GRP_CAN0_10,
			PINCTRL_GRP_I2C0_10,
			PINCTRL_GRP_SWDT0_6_CLK,
			PINCTRL_GRP_SPI0_3,
			PINCTRL_GRP_TTC2_5_CLK,
			PINCTRL_GRP_UART0_10,
			PINCTRL_GRP_TRACE0_1,
			PINCTRL_GRP_SDIO0_1,
			PINCTRL_GRP_SDIO0_4BIT_1_0,
			PINCTRL_GRP_SDIO0_1BIT_1_1,
			PINCTRL_GRP_SDIO1_4BIT_0_0,
			PINCTRL_GRP_SDIO1_1BIT_0_3,
			END_OF_GROUPS,
		}),
	},
	[PINCTRL_PIN_43] = {
		.groups = &((uint16_t []) {
			PINCTRL_GRP_ETHERNET1_0,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_SDIO0_1,
			PINCTRL_GRP_SDIO1_0_PC,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_GPIO0_43,
			PINCTRL_GRP_CAN0_10,
			PINCTRL_GRP_I2C0_10,
			PINCTRL_GRP_SWDT0_6_RST,
			PINCTRL_GRP_SPI0_3,
			PINCTRL_GRP_TTC2_5_WAV,
			PINCTRL_GRP_UART0_10,
			PINCTRL_GRP_TRACE0_1,
			PINCTRL_GRP_SDIO0_4BIT_1_0,
			PINCTRL_GRP_SDIO0_1BIT_1_2,
			END_OF_GROUPS,
		}),
	},
	[PINCTRL_PIN_44] = {
		.groups = &((uint16_t []) {
			PINCTRL_GRP_ETHERNET1_0,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_SDIO0_1,
			PINCTRL_GRP_SDIO1_0_WP,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_GPIO0_44,
			PINCTRL_GRP_CAN1_11,
			PINCTRL_GRP_I2C1_11,
			PINCTRL_GRP_SWDT1_7_CLK,
			PINCTRL_GRP_SPI1_3,
			PINCTRL_GRP_TTC1_5_CLK,
			PINCTRL_GRP_UART1_11,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_SDIO0_4BIT_1_0,
			PINCTRL_GRP_SDIO0_1BIT_1_3,
			END_OF_GROUPS,
		}),
	},
	[PINCTRL_PIN_45] = {
		.groups = &((uint16_t []) {
			PINCTRL_GRP_ETHERNET1_0,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_SDIO0_1,
			PINCTRL_GRP_SDIO1_0_CD,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_GPIO0_45,
			PINCTRL_GRP_CAN1_11,
			PINCTRL_GRP_I2C1_11,
			PINCTRL_GRP_SWDT1_7_RST,
			PINCTRL_GRP_SPI1_3_SS2,
			PINCTRL_GRP_TTC1_5_WAV,
			PINCTRL_GRP_UART1_11,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_SDIO0_4BIT_1_1,
			PINCTRL_GRP_SDIO0_1BIT_1_4,
			END_OF_GROUPS,
		}),
	},
	[PINCTRL_PIN_46] = {
		.groups = &((uint16_t []) {
			PINCTRL_GRP_ETHERNET1_0,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_SDIO0_1,
			PINCTRL_GRP_SDIO1_0,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_GPIO0_46,
			PINCTRL_GRP_CAN0_11,
			PINCTRL_GRP_I2C0_11,
			PINCTRL_GRP_SWDT0_7_CLK,
			PINCTRL_GRP_SPI1_3_SS1,
			PINCTRL_GRP_TTC0_5_CLK,
			PINCTRL_GRP_UART0_11,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_SDIO0_4BIT_1_1,
			PINCTRL_GRP_SDIO0_1BIT_1_5,
			PINCTRL_GRP_SDIO1_4BIT_0_1,
			PINCTRL_GRP_SDIO1_1BIT_0_4,
			END_OF_GROUPS,
		}),
	},
	[PINCTRL_PIN_47] = {
		.groups = &((uint16_t []) {
			PINCTRL_GRP_ETHERNET1_0,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_SDIO0_1,
			PINCTRL_GRP_SDIO1_0,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_GPIO0_47,
			PINCTRL_GRP_CAN0_11,
			PINCTRL_GRP_I2C0_11,
			PINCTRL_GRP_SWDT0_7_RST,
			PINCTRL_GRP_SPI1_3_SS0,
			PINCTRL_GRP_TTC0_5_WAV,
			PINCTRL_GRP_UART0_11,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_SDIO0_4BIT_1_1,
			PINCTRL_GRP_SDIO0_1BIT_1_6,
			PINCTRL_GRP_SDIO1_4BIT_0_1,
			PINCTRL_GRP_SDIO1_1BIT_0_5,
			END_OF_GROUPS,
		}),
	},
	[PINCTRL_PIN_48] = {
		.groups = &((uint16_t []) {
			PINCTRL_GRP_ETHERNET1_0,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_SDIO0_1,
			PINCTRL_GRP_SDIO1_0,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_GPIO0_48,
			PINCTRL_GRP_CAN1_12,
			PINCTRL_GRP_I2C1_12,
			PINCTRL_GRP_SWDT1_8_CLK,
			PINCTRL_GRP_SPI1_3,
			PINCTRL_GRP_TTC3_6_CLK,
			PINCTRL_GRP_UART1_12,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_SDIO0_4BIT_1_1,
			PINCTRL_GRP_SDIO0_1BIT_1_7,
			PINCTRL_GRP_SDIO1_4BIT_0_1,
			PINCTRL_GRP_SDIO1_1BIT_0_6,
			END_OF_GROUPS,
		}),
	},
	[PINCTRL_PIN_49] = {
		.groups = &((uint16_t []) {
			PINCTRL_GRP_ETHERNET1_0,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_SDIO0_1_PC,
			PINCTRL_GRP_SDIO1_0,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_GPIO0_49,
			PINCTRL_GRP_CAN1_12,
			PINCTRL_GRP_I2C1_12,
			PINCTRL_GRP_SWDT1_8_RST,
			PINCTRL_GRP_SPI1_3,
			PINCTRL_GRP_TTC3_6_WAV,
			PINCTRL_GRP_UART1_12,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_SDIO1_4BIT_0_1,
			PINCTRL_GRP_SDIO1_1BIT_0_7,
			END_OF_GROUPS,
		}),
	},
	[PINCTRL_PIN_50] = {
		.groups = &((uint16_t []) {
			PINCTRL_GRP_GEMTSU0_1,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_SDIO0_1_WP,
			PINCTRL_GRP_SDIO1_0,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_GPIO0_50,
			PINCTRL_GRP_CAN0_12,
			PINCTRL_GRP_I2C0_12,
			PINCTRL_GRP_SWDT0_8_CLK,
			PINCTRL_GRP_MDIO1_0,
			PINCTRL_GRP_TTC2_6_CLK,
			PINCTRL_GRP_UART0_12,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_SDIO1_4BIT_0_0,
			PINCTRL_GRP_SDIO1_4BIT_0_1,
			PINCTRL_GRP_SDIO1_1BIT_0_0,
			PINCTRL_GRP_SDIO1_1BIT_0_1,
			PINCTRL_GRP_SDIO1_1BIT_0_2,
			PINCTRL_GRP_SDIO1_1BIT_0_3,
			PINCTRL_GRP_SDIO1_1BIT_0_4,
			PINCTRL_GRP_SDIO1_1BIT_0_5,
			PINCTRL_GRP_SDIO1_1BIT_0_6,
			PINCTRL_GRP_SDIO1_1BIT_0_7,
			END_OF_GROUPS,
		}),
	},
	[PINCTRL_PIN_51] = {
		.groups = &((uint16_t []) {
			PINCTRL_GRP_GEMTSU0_2,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_SDIO1_0,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_GPIO0_51,
			PINCTRL_GRP_CAN0_12,
			PINCTRL_GRP_I2C0_12,
			PINCTRL_GRP_SWDT0_8_RST,
			PINCTRL_GRP_MDIO1_0,
			PINCTRL_GRP_TTC2_6_WAV,
			PINCTRL_GRP_UART0_12,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_SDIO1_4BIT_0_0,
			PINCTRL_GRP_SDIO1_4BIT_0_1,
			PINCTRL_GRP_SDIO1_1BIT_0_0,
			PINCTRL_GRP_SDIO1_1BIT_0_1,
			PINCTRL_GRP_SDIO1_1BIT_0_2,
			PINCTRL_GRP_SDIO1_1BIT_0_3,
			PINCTRL_GRP_SDIO1_1BIT_0_4,
			PINCTRL_GRP_SDIO1_1BIT_0_5,
			PINCTRL_GRP_SDIO1_1BIT_0_6,
			PINCTRL_GRP_SDIO1_1BIT_0_7,
			END_OF_GROUPS,
		}),
	},
	[PINCTRL_PIN_52] = {
		.groups = &((uint16_t []) {
			PINCTRL_GRP_ETHERNET2_0,
			PINCTRL_GRP_USB0_0,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_GPIO0_52,
			PINCTRL_GRP_CAN1_13,
			PINCTRL_GRP_I2C1_13,
			PINCTRL_GRP_PJTAG0_4,
			PINCTRL_GRP_SPI0_4,
			PINCTRL_GRP_TTC1_6_CLK,
			PINCTRL_GRP_UART1_13,
			PINCTRL_GRP_TRACE0_2_CLK,
			END_OF_GROUPS,
		}),
	},
	[PINCTRL_PIN_53] = {
		.groups = &((uint16_t []) {
			PINCTRL_GRP_ETHERNET2_0,
			PINCTRL_GRP_USB0_0,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_GPIO0_53,
			PINCTRL_GRP_CAN1_13,
			PINCTRL_GRP_I2C1_13,
			PINCTRL_GRP_PJTAG0_4,
			PINCTRL_GRP_SPI0_4_SS2,
			PINCTRL_GRP_TTC1_6_WAV,
			PINCTRL_GRP_UART1_13,
			PINCTRL_GRP_TRACE0_2_CLK,
			END_OF_GROUPS,
		}),
	},
	[PINCTRL_PIN_54] = {
		.groups = &((uint16_t []) {
			PINCTRL_GRP_ETHERNET2_0,
			PINCTRL_GRP_USB0_0,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_GPIO0_54,
			PINCTRL_GRP_CAN0_13,
			PINCTRL_GRP_I2C0_13,
			PINCTRL_GRP_PJTAG0_4,
			PINCTRL_GRP_SPI0_4_SS1,
			PINCTRL_GRP_TTC0_6_CLK,
			PINCTRL_GRP_UART0_13,
			PINCTRL_GRP_TRACE0_2,
			END_OF_GROUPS,
		}),
	},
	[PINCTRL_PIN_55] = {
		.groups = &((uint16_t []) {
			PINCTRL_GRP_ETHERNET2_0,
			PINCTRL_GRP_USB0_0,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_GPIO0_55,
			PINCTRL_GRP_CAN0_13,
			PINCTRL_GRP_I2C0_13,
			PINCTRL_GRP_PJTAG0_4,
			PINCTRL_GRP_SPI0_4_SS0,
			PINCTRL_GRP_TTC0_6_WAV,
			PINCTRL_GRP_UART0_13,
			PINCTRL_GRP_TRACE0_2,
			END_OF_GROUPS,
		}),
	},
	[PINCTRL_PIN_56] = {
		.groups = &((uint16_t []) {
			PINCTRL_GRP_ETHERNET2_0,
			PINCTRL_GRP_USB0_0,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_GPIO0_56,
			PINCTRL_GRP_CAN1_14,
			PINCTRL_GRP_I2C1_14,
			PINCTRL_GRP_SWDT1_9_CLK,
			PINCTRL_GRP_SPI0_4,
			PINCTRL_GRP_TTC3_7_CLK,
			PINCTRL_GRP_UART1_14,
			PINCTRL_GRP_TRACE0_2,
			END_OF_GROUPS,
		}),
	},
	[PINCTRL_PIN_57] = {
		.groups = &((uint16_t []) {
			PINCTRL_GRP_ETHERNET2_0,
			PINCTRL_GRP_USB0_0,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_GPIO0_57,
			PINCTRL_GRP_CAN1_14,
			PINCTRL_GRP_I2C1_14,
			PINCTRL_GRP_SWDT1_9_RST,
			PINCTRL_GRP_SPI0_4,
			PINCTRL_GRP_TTC3_7_WAV,
			PINCTRL_GRP_UART1_14,
			PINCTRL_GRP_TRACE0_2,
			END_OF_GROUPS,
		}),
	},
	[PINCTRL_PIN_58] = {
		.groups = &((uint16_t []) {
			PINCTRL_GRP_ETHERNET2_0,
			PINCTRL_GRP_USB0_0,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_GPIO0_58,
			PINCTRL_GRP_CAN0_14,
			PINCTRL_GRP_I2C0_14,
			PINCTRL_GRP_PJTAG0_5,
			PINCTRL_GRP_SPI1_4,
			PINCTRL_GRP_TTC2_7_CLK,
			PINCTRL_GRP_UART0_14,
			PINCTRL_GRP_TRACE0_2,
			END_OF_GROUPS,
		}),
	},
	[PINCTRL_PIN_59] = {
		.groups = &((uint16_t []) {
			PINCTRL_GRP_ETHERNET2_0,
			PINCTRL_GRP_USB0_0,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_GPIO0_59,
			PINCTRL_GRP_CAN0_14,
			PINCTRL_GRP_I2C0_14,
			PINCTRL_GRP_PJTAG0_5,
			PINCTRL_GRP_SPI1_4_SS2,
			PINCTRL_GRP_TTC2_7_WAV,
			PINCTRL_GRP_UART0_14,
			PINCTRL_GRP_TRACE0_2,
			END_OF_GROUPS,
		}),
	},
	[PINCTRL_PIN_60] = {
		.groups = &((uint16_t []) {
			PINCTRL_GRP_ETHERNET2_0,
			PINCTRL_GRP_USB0_0,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_GPIO0_60,
			PINCTRL_GRP_CAN1_15,
			PINCTRL_GRP_I2C1_15,
			PINCTRL_GRP_PJTAG0_5,
			PINCTRL_GRP_SPI1_4_SS1,
			PINCTRL_GRP_TTC1_7_CLK,
			PINCTRL_GRP_UART1_15,
			PINCTRL_GRP_TRACE0_2,
			END_OF_GROUPS,
		}),
	},
	[PINCTRL_PIN_61] = {
		.groups = &((uint16_t []) {
			PINCTRL_GRP_ETHERNET2_0,
			PINCTRL_GRP_USB0_0,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_GPIO0_61,
			PINCTRL_GRP_CAN1_15,
			PINCTRL_GRP_I2C1_15,
			PINCTRL_GRP_PJTAG0_5,
			PINCTRL_GRP_SPI1_4_SS0,
			PINCTRL_GRP_TTC1_7_WAV,
			PINCTRL_GRP_UART1_15,
			PINCTRL_GRP_TRACE0_2,
			END_OF_GROUPS,
		}),
	},
	[PINCTRL_PIN_62] = {
		.groups = &((uint16_t []) {
			PINCTRL_GRP_ETHERNET2_0,
			PINCTRL_GRP_USB0_0,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_GPIO0_62,
			PINCTRL_GRP_CAN0_15,
			PINCTRL_GRP_I2C0_15,
			PINCTRL_GRP_SWDT0_9_CLK,
			PINCTRL_GRP_SPI1_4,
			PINCTRL_GRP_TTC0_7_CLK,
			PINCTRL_GRP_UART0_15,
			PINCTRL_GRP_TRACE0_2,
			END_OF_GROUPS,
		}),
	},
	[PINCTRL_PIN_63] = {
		.groups = &((uint16_t []) {
			PINCTRL_GRP_ETHERNET2_0,
			PINCTRL_GRP_USB0_0,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_GPIO0_63,
			PINCTRL_GRP_CAN0_15,
			PINCTRL_GRP_I2C0_15,
			PINCTRL_GRP_SWDT0_9_RST,
			PINCTRL_GRP_SPI1_4,
			PINCTRL_GRP_TTC0_7_WAV,
			PINCTRL_GRP_UART0_15,
			PINCTRL_GRP_TRACE0_2,
			END_OF_GROUPS,
		}),
	},
	[PINCTRL_PIN_64] = {
		.groups = &((uint16_t []) {
			PINCTRL_GRP_ETHERNET3_0,
			PINCTRL_GRP_USB1_0,
			PINCTRL_GRP_SDIO0_2,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_GPIO0_64,
			PINCTRL_GRP_CAN1_16,
			PINCTRL_GRP_I2C1_16,
			PINCTRL_GRP_SWDT1_10_CLK,
			PINCTRL_GRP_SPI0_5,
			PINCTRL_GRP_TTC3_8_CLK,
			PINCTRL_GRP_UART1_16,
			PINCTRL_GRP_TRACE0_2,
			PINCTRL_GRP_SDIO0_4BIT_2_0,
			PINCTRL_GRP_SDIO0_4BIT_2_1,
			PINCTRL_GRP_SDIO0_1BIT_2_0,
			PINCTRL_GRP_SDIO0_1BIT_2_1,
			PINCTRL_GRP_SDIO0_1BIT_2_2,
			PINCTRL_GRP_SDIO0_1BIT_2_3,
			PINCTRL_GRP_SDIO0_1BIT_2_4,
			PINCTRL_GRP_SDIO0_1BIT_2_5,
			PINCTRL_GRP_SDIO0_1BIT_2_6,
			PINCTRL_GRP_SDIO0_1BIT_2_7,
			END_OF_GROUPS,
		}),
	},
	[PINCTRL_PIN_65] = {
		.groups = &((uint16_t []) {
			PINCTRL_GRP_ETHERNET3_0,
			PINCTRL_GRP_USB1_0,
			PINCTRL_GRP_SDIO0_2_CD,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_GPIO0_65,
			PINCTRL_GRP_CAN1_16,
			PINCTRL_GRP_I2C1_16,
			PINCTRL_GRP_SWDT1_10_RST,
			PINCTRL_GRP_SPI0_5_SS2,
			PINCTRL_GRP_TTC3_8_WAV,
			PINCTRL_GRP_UART1_16,
			PINCTRL_GRP_TRACE0_2,
			END_OF_GROUPS,
		}),
	},
	[PINCTRL_PIN_66] = {
		.groups = &((uint16_t []) {
			PINCTRL_GRP_ETHERNET3_0,
			PINCTRL_GRP_USB1_0,
			PINCTRL_GRP_SDIO0_2,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_GPIO0_66,
			PINCTRL_GRP_CAN0_16,
			PINCTRL_GRP_I2C0_16,
			PINCTRL_GRP_SWDT0_10_CLK,
			PINCTRL_GRP_SPI0_5_SS1,
			PINCTRL_GRP_TTC2_8_CLK,
			PINCTRL_GRP_UART0_16,
			PINCTRL_GRP_TRACE0_2,
			PINCTRL_GRP_SDIO0_4BIT_2_0,
			PINCTRL_GRP_SDIO0_4BIT_2_1,
			PINCTRL_GRP_SDIO0_1BIT_2_0,
			PINCTRL_GRP_SDIO0_1BIT_2_1,
			PINCTRL_GRP_SDIO0_1BIT_2_2,
			PINCTRL_GRP_SDIO0_1BIT_2_3,
			PINCTRL_GRP_SDIO0_1BIT_2_4,
			PINCTRL_GRP_SDIO0_1BIT_2_5,
			PINCTRL_GRP_SDIO0_1BIT_2_6,
			PINCTRL_GRP_SDIO0_1BIT_2_7,
			END_OF_GROUPS,
		}),
	},
	[PINCTRL_PIN_67] = {
		.groups = &((uint16_t []) {
			PINCTRL_GRP_ETHERNET3_0,
			PINCTRL_GRP_USB1_0,
			PINCTRL_GRP_SDIO0_2,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_GPIO0_67,
			PINCTRL_GRP_CAN0_16,
			PINCTRL_GRP_I2C0_16,
			PINCTRL_GRP_SWDT0_10_RST,
			PINCTRL_GRP_SPI0_5_SS0,
			PINCTRL_GRP_TTC2_8_WAV,
			PINCTRL_GRP_UART0_16,
			PINCTRL_GRP_TRACE0_2,
			PINCTRL_GRP_SDIO0_4BIT_2_0,
			PINCTRL_GRP_SDIO0_1BIT_2_0,
			END_OF_GROUPS,
		}),
	},
	[PINCTRL_PIN_68] = {
		.groups = &((uint16_t []) {
			PINCTRL_GRP_ETHERNET3_0,
			PINCTRL_GRP_USB1_0,
			PINCTRL_GRP_SDIO0_2,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_GPIO0_68,
			PINCTRL_GRP_CAN1_17,
			PINCTRL_GRP_I2C1_17,
			PINCTRL_GRP_SWDT1_11_CLK,
			PINCTRL_GRP_SPI0_5,
			PINCTRL_GRP_TTC1_8_CLK,
			PINCTRL_GRP_UART1_17,
			PINCTRL_GRP_TRACE0_2,
			PINCTRL_GRP_SDIO0_4BIT_2_0,
			PINCTRL_GRP_SDIO0_1BIT_2_1,
			END_OF_GROUPS,
		}),
	},
	[PINCTRL_PIN_69] = {
		.groups = &((uint16_t []) {
			PINCTRL_GRP_ETHERNET3_0,
			PINCTRL_GRP_USB1_0,
			PINCTRL_GRP_SDIO0_2,
			PINCTRL_GRP_SDIO1_1_WP,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_GPIO0_69,
			PINCTRL_GRP_CAN1_17,
			PINCTRL_GRP_I2C1_17,
			PINCTRL_GRP_SWDT1_11_RST,
			PINCTRL_GRP_SPI0_5,
			PINCTRL_GRP_TTC1_8_WAV,
			PINCTRL_GRP_UART1_17,
			PINCTRL_GRP_TRACE0_2,
			PINCTRL_GRP_SDIO0_4BIT_2_0,
			PINCTRL_GRP_SDIO0_1BIT_2_2,
			END_OF_GROUPS,
		}),
	},
	[PINCTRL_PIN_70] = {
		.groups = &((uint16_t []) {
			PINCTRL_GRP_ETHERNET3_0,
			PINCTRL_GRP_USB1_0,
			PINCTRL_GRP_SDIO0_2,
			PINCTRL_GRP_SDIO1_1_PC,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_GPIO0_70,
			PINCTRL_GRP_CAN0_17,
			PINCTRL_GRP_I2C0_17,
			PINCTRL_GRP_SWDT0_11_CLK,
			PINCTRL_GRP_SPI1_5,
			PINCTRL_GRP_TTC0_8_CLK,
			PINCTRL_GRP_UART0_17,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_SDIO0_4BIT_2_0,
			PINCTRL_GRP_SDIO0_1BIT_2_3,
			END_OF_GROUPS,
		}),
	},
	[PINCTRL_PIN_71] = {
		.groups = &((uint16_t []) {
			PINCTRL_GRP_ETHERNET3_0,
			PINCTRL_GRP_USB1_0,
			PINCTRL_GRP_SDIO0_2,
			PINCTRL_GRP_SDIO1_4BIT_1_0,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_GPIO0_71,
			PINCTRL_GRP_CAN0_17,
			PINCTRL_GRP_I2C0_17,
			PINCTRL_GRP_SWDT0_11_RST,
			PINCTRL_GRP_SPI1_5_SS2,
			PINCTRL_GRP_TTC0_8_WAV,
			PINCTRL_GRP_UART0_17,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_SDIO0_2,
			PINCTRL_GRP_SDIO0_4BIT_2_1,
			PINCTRL_GRP_SDIO0_1BIT_2_4,
			PINCTRL_GRP_SDIO1_1BIT_1_0,
			END_OF_GROUPS,
		}),
	},
	[PINCTRL_PIN_72] = {
		.groups = &((uint16_t []) {
			PINCTRL_GRP_ETHERNET3_0,
			PINCTRL_GRP_USB1_0,
			PINCTRL_GRP_SDIO0_2,
			PINCTRL_GRP_SDIO1_4BIT_1_0,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_GPIO0_72,
			PINCTRL_GRP_CAN1_18,
			PINCTRL_GRP_I2C1_18,
			PINCTRL_GRP_SWDT1_12_CLK,
			PINCTRL_GRP_SPI1_5_SS1,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_UART1_18,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_SDIO0_4BIT_2_1,
			PINCTRL_GRP_SDIO0_1BIT_2_5,
			PINCTRL_GRP_SDIO1_1BIT_1_1,
			END_OF_GROUPS,
		}),
	},
	[PINCTRL_PIN_73] = {
		.groups = &((uint16_t []) {
			PINCTRL_GRP_ETHERNET3_0,
			PINCTRL_GRP_USB1_0,
			PINCTRL_GRP_SDIO0_2,
			PINCTRL_GRP_SDIO1_4BIT_1_0,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_GPIO0_73,
			PINCTRL_GRP_CAN1_18,
			PINCTRL_GRP_I2C1_18,
			PINCTRL_GRP_SWDT1_12_RST,
			PINCTRL_GRP_SPI1_5_SS0,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_UART1_18,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_SDIO0_4BIT_2_1,
			PINCTRL_GRP_SDIO0_1BIT_2_6,
			PINCTRL_GRP_SDIO1_1BIT_1_2,
			END_OF_GROUPS,
		}),
	},
	[PINCTRL_PIN_74] = {
		.groups = &((uint16_t []) {
			PINCTRL_GRP_ETHERNET3_0,
			PINCTRL_GRP_USB1_0,
			PINCTRL_GRP_SDIO0_2,
			PINCTRL_GRP_SDIO1_4BIT_1_0,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_GPIO0_74,
			PINCTRL_GRP_CAN0_18,
			PINCTRL_GRP_I2C0_18,
			PINCTRL_GRP_SWDT0_12_CLK,
			PINCTRL_GRP_SPI1_5,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_UART0_18,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_SDIO0_4BIT_2_1,
			PINCTRL_GRP_SDIO0_1BIT_2_7,
			PINCTRL_GRP_SDIO1_1BIT_1_3,
			END_OF_GROUPS,
		}),
	},
	[PINCTRL_PIN_75] = {
		.groups = &((uint16_t []) {
			PINCTRL_GRP_ETHERNET3_0,
			PINCTRL_GRP_USB1_0,
			PINCTRL_GRP_SDIO0_2_PC,
			PINCTRL_GRP_SDIO1_4BIT_1_0,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_GPIO0_75,
			PINCTRL_GRP_CAN0_18,
			PINCTRL_GRP_I2C0_18,
			PINCTRL_GRP_SWDT0_12_RST,
			PINCTRL_GRP_SPI1_5,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_UART0_18,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_SDIO1_1BIT_1_0,
			PINCTRL_GRP_SDIO1_1BIT_1_1,
			PINCTRL_GRP_SDIO1_1BIT_1_2,
			PINCTRL_GRP_SDIO1_1BIT_1_3,
			END_OF_GROUPS,
		}),
	},
	[PINCTRL_PIN_76] = {
		.groups = &((uint16_t []) {
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_SDIO0_2_WP,
			PINCTRL_GRP_SDIO1_4BIT_1_0,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_GPIO0_76,
			PINCTRL_GRP_CAN1_19,
			PINCTRL_GRP_I2C1_19,
			PINCTRL_GRP_MDIO0_0,
			PINCTRL_GRP_MDIO1_1,
			PINCTRL_GRP_MDIO2_0,
			PINCTRL_GRP_MDIO3_0,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_SDIO1_1BIT_1_0,
			PINCTRL_GRP_SDIO1_1BIT_1_1,
			PINCTRL_GRP_SDIO1_1BIT_1_2,
			PINCTRL_GRP_SDIO1_1BIT_1_3,
			END_OF_GROUPS,
		}),
	},
	[PINCTRL_PIN_77] = {
		.groups = &((uint16_t []) {
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_SDIO1_1_CD,
			PINCTRL_GRP_RESERVED,
			PINCTRL_GRP_GPIO0_77,
			PINCTRL_GRP_CAN1_19,
			PINCTRL_GRP_I2C1_19,
			PINCTRL_GRP_MDIO0_0,
			PINCTRL_GRP_MDIO1_1,
			PINCTRL_GRP_MDIO2_0,
			PINCTRL_GRP_MDIO3_0,
			PINCTRL_GRP_RESERVED,
			END_OF_GROUPS,
		}),
	},
};

/**
 * pm_api_pinctrl_get_num_pins() - PM call to request number of pins
 * @npins	Number of pins
 *
 * This function is used by master to get number of pins
 *
 * @return	Returns success.
 */
enum pm_ret_status pm_api_pinctrl_get_num_pins(uint32_t *npins)
{
	*npins = MAX_PIN;

	return PM_RET_SUCCESS;
}

/**
 * pm_api_pinctrl_get_num_functions() - PM call to request number of functions
 * @nfuncs	Number of functions
 *
 * This function is used by master to get number of functions
 *
 * @return	Returns success.
 */
enum pm_ret_status pm_api_pinctrl_get_num_functions(uint32_t *nfuncs)
{
	*nfuncs = MAX_FUNCTION;

	return PM_RET_SUCCESS;
}

/**
 * pm_api_pinctrl_get_num_func_groups() - PM call to request number of
 *					  function groups
 * @fid		Function Id
 * @ngroups	Number of function groups
 *
 * This function is used by master to get number of function groups
 *
 * @return	Returns success.
 */
enum pm_ret_status pm_api_pinctrl_get_num_func_groups(uint32_t fid,
						      uint32_t *ngroups)
{
	if (fid >= MAX_FUNCTION) {
		return PM_RET_ERROR_ARGS;
	}

	*ngroups = pinctrl_functions[fid].group_size;

	return PM_RET_SUCCESS;
}

/**
 * pm_api_pinctrl_get_function_name() - PM call to request a function name
 * @fid		Function ID
 * @name	Name of function (max 16 bytes)
 *
 * This function is used by master to get name of function specified
 * by given function ID.
 */
void pm_api_pinctrl_get_function_name(uint32_t fid, char *name)
{
	if (fid >= MAX_FUNCTION) {
		memcpy(name, END_OF_FUNCTION, FUNCTION_NAME_LEN);
	} else {
		memcpy(name, pinctrl_functions[fid].name, FUNCTION_NAME_LEN);
	}
}

/**
 * pm_api_pinctrl_get_function_groups() - PM call to request first 6 function
 *					  groups of function Id
 * @fid		Function ID
 * @index	Index of next function groups
 * @groups	Function groups
 *
 * This function is used by master to get function groups specified
 * by given function Id. This API will return 6 function groups with
 * a single response. To get other function groups, master should call
 * same API in loop with new function groups index till error is returned.
 *
 * E.g First call should have index 0 which will return function groups
 * 0, 1, 2, 3, 4 and 5. Next call, index should be 6 which will return
 * function groups 6, 7, 8, 9, 10 and 11 and so on.
 *
 * Return: Returns status, either success or error+reason.
 */
enum pm_ret_status pm_api_pinctrl_get_function_groups(uint32_t fid,
						      uint32_t index,
						      uint16_t *groups)
{
	uint16_t grps;
	uint16_t end_of_grp_offset;
	uint16_t i;

	if (fid >= MAX_FUNCTION) {
		return PM_RET_ERROR_ARGS;
	}

	memset(groups, END_OF_GROUPS, GROUPS_PAYLOAD_LEN);

	grps = pinctrl_functions[fid].group_base;
	end_of_grp_offset = grps + pinctrl_functions[fid].group_size;

	for (i = 0U; i < NUM_GROUPS_PER_RESP; i++) {
		if ((grps + index + i) >= end_of_grp_offset) {
			break;
		}
		groups[i] = (grps + index + i);
	}

	return PM_RET_SUCCESS;
}

/**
 * pm_api_pinctrl_get_pin_groups() - PM call to request first 6 pin
 *				     groups of pin
 * @pin		Pin
 * @index	Index of next pin groups
 * @groups	pin groups
 *
 * This function is used by master to get pin groups specified
 * by given pin Id. This API will return 6 pin groups with
 * a single response. To get other pin groups, master should call
 * same API in loop with new pin groups index till error is returned.
 *
 * E.g First call should have index 0 which will return pin groups
 * 0, 1, 2, 3, 4 and 5. Next call, index should be 6 which will return
 * pin groups 6, 7, 8, 9, 10 and 11 and so on.
 *
 * Return: Returns status, either success or error+reason.
 */
enum pm_ret_status pm_api_pinctrl_get_pin_groups(uint32_t pin,
						 uint32_t index,
						 uint16_t *groups)
{
	uint32_t i;
	uint16_t *grps;

	if (pin >= MAX_PIN) {
		return PM_RET_ERROR_ARGS;
	}

	memset(groups, END_OF_GROUPS, GROUPS_PAYLOAD_LEN);

	grps = *zynqmp_pin_groups[pin].groups;
	if (grps == NULL) {
		return PM_RET_SUCCESS;
	}

	/* Skip groups till index */
	for (i = 0; i < index; i++) {
		if (grps[i] == (uint16_t)END_OF_GROUPS) {
			return PM_RET_SUCCESS;
		}
	}

	for (i = 0; i < NUM_GROUPS_PER_RESP; i++) {
		groups[i] = grps[index + i];
		if (groups[i] == (uint16_t)END_OF_GROUPS) {
			break;
		}
	}

	return PM_RET_SUCCESS;
}
