//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      Base/Axis/MakeScale.cpp
//! @brief     Implements functions that create instances of class Scale.
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#include "Base/Axis/MakeScale.h"
#include "Base/Axis/Scale.h"
#include "Base/Util/Assert.h"

namespace {

std::vector<Bin1D> limits2bins(const std::vector<double>& limits)
{
    std::vector<Bin1D> result;
    ASSERT(!(limits.size() & 1)); // must be an even number
    for (size_t i = 0; i < limits.size(); i += 2)
        result.emplace_back(Bin1D::FromTo(limits[i], limits[i + 1]));
    return result;
}

std::vector<Bin1D> centers2bins(const std::vector<double>& centers)
{
    std::vector<Bin1D> result;
    for (double c : centers)
        result.emplace_back(Bin1D::At(c));
    return result;
}

std::vector<Bin1D> bounds2division(size_t N, double start, double end)
{
    ASSERT(N > 0);
    ASSERT(start <= end);
    std::vector<Bin1D> result;
    for (size_t i = 0; i < N; ++i)
        result.emplace_back(Bin1D::FromTo((N - i) * (start / N) + i * (end / N),
                                          (N - i - 1) * (start / N) + (i + 1) * (end / N)));
    ASSERT(result.size() == N);
    return result;
}

} // namespace

Scale GenericScale(const std::string& name, const std::vector<double>& limits)
{
    return Scale(name, limits2bins(limits));
}

Scale* newGenericScale(const std::string& name, const std::vector<double>& limits)
{
    return new Scale(name, limits2bins(limits));
}

Scale ListScan(const std::string& name, const std::vector<double>& points)
{
    return Scale(name, centers2bins(points));
}

Scale* newListScan(const std::string& name, const std::vector<double>& points)
{
    return new Scale(name, centers2bins(points));
}


Scale EquiDivision(const std::string& name, size_t nbins, double start, double end)
{
    return Scale(name, bounds2division(nbins, start, end));
}

Scale* newEquiDivision(const std::string& name, size_t nbins, double start, double end)
{
    return new Scale(name, bounds2division(nbins, start, end));
}

std::shared_ptr<Scale> sharedEquiDivision(const std::string& name, size_t nbins, double start,
                                          double end)
{
    return std::shared_ptr<Scale>(newEquiDivision(name, nbins, start, end));
}

std::unique_ptr<Scale> uniqueEquiDivision(const std::string& name, size_t nbins, double start,
                                          double end)
{
    return std::unique_ptr<Scale>(newEquiDivision(name, nbins, start, end));
}


Scale EquiScan(const std::string& name, size_t nbins, double start, double end)
{
    if (nbins < 2)
        throw std::runtime_error("Cannot create scan axis with less than two points");
    const double delta = (end - start) / (nbins - 1);
    return EquiDivision(name, nbins, start - delta / 2, end + delta / 2);
}

Scale* newEquiScan(const std::string& name, size_t nbins, double start, double end)
{
    if (nbins < 2)
        throw std::runtime_error("Cannot create scan axis with less than two points");
    const double delta = (end - start) / (nbins - 1);
    return newEquiDivision(name, nbins, start - delta / 2, end + delta / 2);
}

std::shared_ptr<Scale> sharedEquiScan(const std::string& name, size_t nbins, double start,
                                      double end)
{
    return std::shared_ptr<Scale>(newEquiDivision(name, nbins, start, end));
}

std::unique_ptr<Scale> uniqueEquiScan(const std::string& name, size_t nbins, double start,
                                      double end)
{
    return std::unique_ptr<Scale>(newEquiDivision(name, nbins, start, end));
}
