//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      Device/IO/ReadWriteNumpyTXT.cpp
//! @brief     Implements functions read|writeNumpyTxt
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#include "Device/IO/ReadWriteNumpyTXT.h"
#include "Base/Axis/Scale.h"
#include "Base/Util/Assert.h"
#include "Base/Util/StringUtil.h"
#include "Device/Data/ArrayUtil.h"
#include "Device/Data/Datafield.h"
#include "Device/IO/ParseUtil.h"
#include <string>
#include <vector>

namespace {

bool isDoubleStartChar(char c)
{
    return isdigit(c) || c == '-' || c == '+';
}

void write1DRepresentation(const Datafield& data, std::ostream& output_stream)
{
    output_stream << "# coordinates         intensities" << std::endl;
    output_stream.imbue(std::locale::classic());
    output_stream << std::scientific << std::setprecision(12);

    const std::vector<double> axis_values = data.axis(0).binCenters();

    // printing coordinate and associated intensity
    for (size_t i = 0, nrows = axis_values.size(); i < nrows; ++i)
        output_stream << axis_values[i] << "    " << Util::Parse::ignoreDenormalized(data[i])
                      << std::endl;
}

void write2DRepresentation(const Datafield& data, std::ostream& output_stream)
{
    const size_t nrows = data.axis(1).size();
    const size_t ncols = data.axis(0).size();

    output_stream << "# [nrows=" << nrows << ", ncols=" << ncols << "]" << std::endl;

    std::vector<std::vector<double>> dataArray = DataUtil::Array::createVector2D(data);
    output_stream.imbue(std::locale::classic());
    output_stream << std::scientific << std::setprecision(12);

    for (size_t i = 0; i < nrows; i++) {
        for (size_t j = 0; j < ncols; j++) {
            double z_value = dataArray[i][j];
            output_stream << Util::Parse::ignoreDenormalized(z_value) << "    ";
        }
        output_stream << std::endl;
    }
}

} // namespace

Datafield* Util::RW::readNumpyTxt(std::istream& input_stream)
{
    std::string line;
    std::vector<std::vector<double>> data;

    // Read numbers from input stream:
    while (std::getline(input_stream, line)) {
        line = Base::String::trim(line);
        if (line.empty() || !isDoubleStartChar(line[0]))
            continue;
        std::vector<double> dataInRow = Util::Parse::parse_doubles(line);
        data.push_back(dataInRow);
    }

    // validating
    size_t nrows = data.size();
    size_t ncols(0);
    if (nrows)
        ncols = data[0].size();

    if (ncols == 0)
        throw std::runtime_error("Numpy txt file has invalid content");

    for (size_t row = 0; row < nrows; row++)
        if (data[row].size() != ncols)
            throw std::runtime_error("Error in Numpy txt file: "
                                     "Number of elements is different from row to row.");

    if (nrows < 2)
        return DataUtil::Array::createPField1D(data[0]).release();

    if (ncols < 2) {
        const size_t size = data.size();
        std::vector<double> vector1d(size);
        for (size_t i = 0; i < size; ++i)
            vector1d[i] = data[i][0];
        return DataUtil::Array::createPField1D(vector1d).release();
    }

    return DataUtil::Array::createPField2D(data).release();
}

void Util::RW::writeNumpyTxt(const Datafield& data, std::ostream& output_stream)
{
    output_stream << "# BornAgain Intensity Data" << std::endl;
    output_stream << "# Simple array suitable for numpy, matlab etc." << std::endl;

    const size_t dim = data.rank();
    switch (dim) {
    case 1:
        write1DRepresentation(data, output_stream);
        break;
    case 2:
        write2DRepresentation(data, output_stream);
        break;
    default:
        ASSERT(false);
    }
}
