//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/Support/Util/Style.cpp
//! @brief     Defines GUI::StyleUtils namespace
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#include "GUI/Support/Util/Style.h"
#include "Base/Util/Assert.h"
#include "GUI/Support/Util/DesignerHelper.h"
#include <QApplication>
#include <QBoxLayout>
#include <QDialog>
#include <QTreeView>

namespace {

//! Calculates size of letter `M` for current system font settings.

QSize FindSizeOfLetterM(const QWidget* widget)
{
    QFontMetrics fontMetric(widget->font());
    auto em = fontMetric.horizontalAdvance('M');
    auto fontAscent = fontMetric.ascent();

    return QSize(em, fontAscent);
}

QSize DefaultSizeOfLetterM()
{
    QWidget widget;
    return FindSizeOfLetterM(&widget);
}

} // namespace


void GUI::Style::setPropertyStyle(QTreeView* tree)
{
    ASSERT(tree);
    tree->setStyleSheet(GUI::Style::propertyTreeStyle());
    tree->setAlternatingRowColors(true);
}

QString GUI::Style::propertyTreeStyle()
{
    QString result;

    // lines arount cell content
    result += "QTreeView::item {"
              "    border-bottom: 1px solid #c7c8c9; "
              "    border-right: 1px solid #c7c8c9;}"
              "QTreeView::branch {border-bottom: 1px solid #c7c8c9;}";

    // styling of branch to restore open/closed signs eliminated by previous styling
    result += "QTreeView::branch:has-children:!has-siblings:closed,"
              "QTreeView::branch:closed:has-children:has-siblings {"
              "padding:2px 2px 2px 2px;border-image: none;"
              "image: url(:/images/caret-right.svg);}"
              "QTreeView::branch:open:has-children:!has-siblings,"
              "QTreeView::branch:open:has-children:has-siblings  {"
              "padding:2px 2px 2px 2px;border-image: none; image: url(:/images/caret-down.svg);"
              "}";

    // background of selected rows restored
    result += "QTreeView::item:selected{background:#3daee9;}";

    return result;
}

QFont GUI::Style::sectionFont(bool bold)
{
    QFont result;
    result.setPointSize(DesignerHelper::getSectionFontSize());
    result.setBold(bold);

    return result;
}

QFont GUI::Style::labelFont(bool bold)
{
    QFont result;
    result.setPointSize(DesignerHelper::getLabelFontSize());
    result.setBold(bold);

    return result;
}

void GUI::Style::setResizable(QDialog*
#ifdef Q_OS_MAC
                                  dialog
#endif
)
{
#ifdef Q_OS_MAC
    dialog->setWindowFlags(Qt::WindowCloseButtonHint | Qt::CustomizeWindowHint
                           | Qt::WindowMinimizeButtonHint | Qt::WindowMaximizeButtonHint
                           | Qt::Window);
#endif
}

QSize GUI::Style::SizeOfLetterM(const QWidget* widget)
{
    static QSize default_size = DefaultSizeOfLetterM();
    return widget ? FindSizeOfLetterM(widget) : default_size;
}

int GUI::Style::SystemPointSize()
{
    return QApplication::font().pointSize();
}
