//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/View/Widget/ItemComboToolbar.cpp
//! @brief     Implements class ItemComboToolbar
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#include "GUI/View/Widget/ItemComboToolbar.h"
#include "Base/Util/Assert.h"
#include "GUI/Model/Project/ProjectDocument.h"
#include <QComboBox>
#include <QStandardItemModel>

ItemComboToolbar::ItemComboToolbar(QWidget* parent)
    : StyledToolbar(parent)
    , m_comboBox(new QComboBox)
    , m_comboBoxAction(nullptr)
{
    setToolButtonStyle(Qt::ToolButtonTextBesideIcon);

    m_comboBox->setToolTip("Select type of graphical presentation.");

    m_comboBoxAction = addWidget(m_comboBox);

    setComboConnected(true);
}

void ItemComboToolbar::setPresentation(const QString& name)
{
    setComboConnected(false);
    m_comboBox->setCurrentText(name);
    setComboConnected(true);
}

void ItemComboToolbar::setPresentationList(const QStringList& presentationList,
                                           const QStringList& activeList)
{
    QString previous = currentPresentation();

    setComboConnected(false);
    m_comboBox->clear();
    m_comboBox->addItems(presentationList);
    if (activeList.contains(previous))
        m_comboBox->setCurrentText(previous);

    if (!activeList.isEmpty())
        makeItemsEnabled(activeList);

    setComboConnected(true);
}

QString ItemComboToolbar::currentPresentation() const
{
    return m_comboBox->currentText();
}

//! Sets external actions to tool bar (previous actions will be removed).

void ItemComboToolbar::setActionList(const QList<QAction*>& actionList)
{
    for (auto* action : actions())
        removeAction(action);

    for (auto* action : actionList) {
        addAction(action);
        addSpacing();
    }
    addStyledExpand();
    addAction(m_comboBoxAction);
}

void ItemComboToolbar::setComboConnected(bool value)
{
    if (value)
        connect(m_comboBox, &QComboBox::currentIndexChanged, this, &ItemComboToolbar::_comboChanged,
                Qt::UniqueConnection);
    else
        disconnect(m_comboBox, &QComboBox::currentIndexChanged, this,
                   &ItemComboToolbar::_comboChanged);
}

void ItemComboToolbar::_comboChanged(const int index)
{
    QString txt{m_comboBox->itemText(index)};
    emit comboChanged(txt);
    gProjectDocument.value()->setModified();
}

//! All items in QComboBox which are not in given list, will be disabled (gray and unselectable).

void ItemComboToolbar::makeItemsEnabled(const QStringList& activePresentations)
{
    const auto* model = dynamic_cast<const QStandardItemModel*>(m_comboBox->model());
    ASSERT(model);

    for (int row = 0; row < m_comboBox->count(); ++row) {
        QString text = m_comboBox->itemText(row);
        model->item(row)->setEnabled(activePresentations.contains(text));
    }
}
