# Configure one component library.

function(MakeLib lib)

    # get target properties
    get_target_property(name ${lib} COMPONENT_NAME)
    get_target_property(swigtmpdir ${lib} SWIG_TMP_DIR)
    get_target_property(withPython ${lib} EMBED_PYTHON)
    get_target_property(source_files ${lib} SOURCE_FILES)
    get_target_property(include_files ${lib} INCLUDE_FILES)

    string(STRIP "${swigtmpdir}" swigtmpdir)

    if(withPython AND swigtmpdir)

        if((CLANG) OR (GCC))
            # suppress warnings from auto-generated code (last updated for Swig 4.0.1)
            set_source_files_properties(${AUTO_WRAP_DIR}/lib${lib}_wrap.cpp
                PROPERTIES COMPILE_OPTIONS
                "\
-Wno-unused-parameter;\
-Wno-missing-field-initializers;\
-Wno-sometimes-uninitialized;\
-Wno-deprecated-declarations")
        endif()

        list(APPEND source_files ${AUTO_WRAP_DIR}/lib${lib}_wrap.cpp)
    endif()

    set_target_properties(${lib} PROPERTIES
        SOURCES "${source_files}"
        PREFIX ${libprefix}
        SUFFIX ${libsuffix}
        OUTPUT_NAME ${lib}
        # eg., libBornAgainBase.so
        _BASEFILENAME ${libprefix}${lib}${libsuffix})

    # NOTE on run-time linking:
    # - Linux:
    # See <https://en.wikipedia.org/wiki/Rpath>
    # - MacOS:
    # See 'Run-Path Dependent Libraries' <https://apple.co/3HVbMWm> and `dlopen` manpage.
    # - Windows:
    # On Windows, run-time dependencies are located in the directory from which
    # the application is loaded and then the `PATH`;
    # see 'Dynamic-Link Library Search Order'
    # <https://docs.microsoft.com/en-us/windows/win32/dlls/dynamic-link-library-search-order>

    if(APPLE AND (BA_APPLE_BUNDLE OR BA_PY_PACKAGE))
        # Add proper RPATHs as hints to load the dependencies at run-time
        set(link_flags "-Wl,-rpath,@loader_path -Wl,-rpath,@loader_path/extra")
        set_target_properties(${lib} PROPERTIES LINK_FLAGS ${link_flags})
    endif()

    if(LINUX)
        # Add proper RPATHs as hints to load the dependencies at run-time
        # NOTE: Unlike `RUNPATH`, `RPATH` is searched for transitive dependencies;
        # ie. paths in RPATH will be considered for everything that is
        # dynamically loaded, even dependencies of dependencies.
        # see <https://bugs.debian.org/cgi-bin/bugreport.cgi?bug=847298>
        set(link_flags "-Wl,--disable-new-dtags,-rpath,\$ORIGIN:\$ORIGIN/extra")
        set_target_properties(${lib} PROPERTIES LINK_FLAGS ${link_flags})
    endif()

    if(BORNAGAIN_PYTHON)
        target_compile_definitions(${lib} PRIVATE -DBORNAGAIN_PYTHON)

        # SWIG-produced interface
        if(swigtmpdir)
            SwigLib(${name} ${lib} ${swigtmpdir})
        endif()

        # Python package
        if(BA_PY_PACKAGE)
            # add the BornAgain library to the Python wheel
            add_library_to_wheel(${lib})
        endif()

        if(WIN32)
            # python in windows required .pyd extension for the library name
            install(FILES ${CMAKE_LIBRARY_OUTPUT_DIRECTORY}/${libprefix}${lib}.pyd
                DESTINATION ${destination_python} COMPONENT Libraries)
            add_custom_command(
                TARGET ${lib}
                POST_BUILD
                COMMAND ${CMAKE_COMMAND} -E copy
                ${CMAKE_BINARY_DIR}/bin/${libprefix}${lib}${libsuffix}
                ${CMAKE_LIBRARY_OUTPUT_DIRECTORY}/${libprefix}${lib}${libsuffix}
            )
        endif()

    endif() # BORNAGAIN_PYTHON

    # installation
    install(TARGETS ${lib}
        LIBRARY DESTINATION ${destination_lib} COMPONENT Libraries
        RUNTIME DESTINATION ${destination_lib} COMPONENT Libraries)

    if(BA_CPP_API)
        foreach(file ${include_files})
            get_filename_component(dir ${file} DIRECTORY)
            install(FILES ${file} DESTINATION ${destination_include}/${name}/${dir})
        endforeach()
    endif()

endfunction()
