//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/Model/Sample/RoughnessItems.cpp
//! @brief     Implements classes RoughnessItems.
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#include "GUI/Model/Sample/RoughnessItems.h"
#include "GUI/Model/Util/UtilXML.h"

namespace {
namespace Tag {

const QString Sigma("Sigma");
const QString Hurst("Hurst");
const QString LateralCorrelationLength("LateralCorrelationLength");

} // namespace Tag
} // namespace

BasicRoughnessItem::BasicRoughnessItem(double sigma, double hurst, double corr_length)
{
    m_sigma.init("Sigma (nm)", "rms of the roughness", sigma, "sigma");
    m_hurst.init("Hurst",
                 "Hurst parameter which describes how jagged the interface,\n "
                 "dimensionless [0.0, 1.0], where 0.0 gives more spikes, \n1.0 more smoothness.",
                 hurst, 3, RealLimits::limited(0.0, 1.0), "hurst");
    m_lateral_correlation_length.init("Correlation length (nm)",
                                      "Lateral correlation length of the roughness", corr_length,
                                      "corrLen");
}

void BasicRoughnessItem::writeTo(QXmlStreamWriter* w) const
{
    m_sigma.writeTo2(w, Tag::Sigma);
    m_hurst.writeTo2(w, Tag::Hurst);
    m_lateral_correlation_length.writeTo2(w, Tag::LateralCorrelationLength);
}

void BasicRoughnessItem::readFrom(QXmlStreamReader* r)
{
    while (r->readNextStartElement()) {
        QString tag = r->name().toString();

        if (tag == Tag::Sigma)
            m_sigma.readFrom2(r, tag);
        else if (tag == Tag::Hurst)
            m_hurst.readFrom2(r, tag);
        else if (tag == Tag::LateralCorrelationLength)
            m_lateral_correlation_length.readFrom2(r, tag);
        else
            r->skipCurrentElement();
    }
}
