//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/View/Device/ScanRangeForm.cpp
//! @brief     Implement class ScanRangeForm
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#include "GUI/View/Device/ScanRangeForm.h"
#include "GUI/Model/Axis/BasicAxisItem.h"
#include "GUI/View/Numeric/SafeSpinBox.h"
#include "GUI/View/Numeric/ScientificSpinBox.h"

ScanRangeForm::ScanRangeForm(QFormLayout* form)
    : m_nbins_spin_box(new SafeSpinBox)
    , m_minimum_spin_box(new ScientificSpinBox)
    , m_maximum_spin_box(new ScientificSpinBox)
    , m_axis_item(nullptr)
{
    m_nbins_spin_box->setRange(1, 65536);
    connect(m_nbins_spin_box, &QSpinBox::valueChanged, this, &ScanRangeForm::onNbinsValueChanged);
    form->addRow("# scan points:", m_nbins_spin_box);

    m_minimum_spin_box->setMinimum(0);
    m_minimum_spin_box->setMaximum(90);
    m_minimum_spin_box->setDecimals(5);
    m_minimum_spin_box->setSingleStep(0.01);
    connect(m_minimum_spin_box, &ScientificSpinBox::valueChanged, this,
            &ScanRangeForm::onMinimumValueChanged);
    form->addRow("Initial angle [deg]:", m_minimum_spin_box);

    m_maximum_spin_box->setMinimum(0);
    m_maximum_spin_box->setMaximum(90);
    m_maximum_spin_box->setDecimals(5);
    m_maximum_spin_box->setSingleStep(0.01);
    connect(m_maximum_spin_box, &ScientificSpinBox::valueChanged, this,
            &ScanRangeForm::onMaximumValueChanged);
    form->addRow("Final angle [deg]:", m_maximum_spin_box);
}

void ScanRangeForm::setAxisItem(BasicAxisItem* axisItem)
{
    m_axis_item = axisItem;
    setEnabled(m_axis_item);
    updateData();
}

void ScanRangeForm::setEnabled(bool enabled)
{
    m_nbins_spin_box->setEnabled(enabled);
    m_minimum_spin_box->setEnabled(enabled);
    m_maximum_spin_box->setEnabled(enabled);
}

void ScanRangeForm::updateData()
{
    QSignalBlocker b1(m_nbins_spin_box);
    QSignalBlocker b2(m_minimum_spin_box);
    QSignalBlocker b3(m_maximum_spin_box);
    if (m_axis_item) {
        m_nbins_spin_box->setValue(m_axis_item->size());
        m_minimum_spin_box->setValue(m_axis_item->min());
        m_maximum_spin_box->setValue(m_axis_item->max());
    } else {
        m_nbins_spin_box->setValue(1);
        m_minimum_spin_box->setValue(0.0);
        m_maximum_spin_box->setValue(0.0);
    }
}

void ScanRangeForm::onNbinsValueChanged(size_t value)
{
    if (m_axis_item && m_axis_item->size() != value) {
        m_axis_item->resize(value);
        emit dataChanged();
    }
}

void ScanRangeForm::onMinimumValueChanged(double value)
{
    if (m_axis_item && m_axis_item->min() != value) {
        m_axis_item->setMin(value);
        emit dataChanged();
        if (m_axis_item->max() < value)
            m_maximum_spin_box->setValue(value);
    }
}

void ScanRangeForm::onMaximumValueChanged(double value)
{
    if (m_axis_item && m_axis_item->max() != value) {
        m_axis_item->setMax(value);
        emit dataChanged();
        if (m_axis_item->min() > value)
            m_minimum_spin_box->setValue(value);
    }
}
