//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/View/Frame/Fit2DFrame.cpp
//! @brief     Implements class Fit2DFrame.
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#include "GUI/View/Frame/Fit2DFrame.h"
#include "Base/Util/Assert.h"
#include "Device/Data/DataUtil.h"
#include "Device/Data/Datafield.h"
#include "GUI/Model/Data/Data1DItem.h"
#include "GUI/Model/Data/Data2DItem.h"
#include "GUI/Model/Data/RangeUtil.h"
#include "GUI/Model/Project/DataSource.h"
#include "GUI/Model/Project/ProjectDocument.h"
#include "GUI/View/Canvas/ColorMapCanvas.h"
#include "GUI/View/Canvas/ProgressCanvas.h"
#include "GUI/View/Plotter/ColorMap.h"
#include "GUI/View/Plotter/PlotStatusLabel.h"
#include "GUI/View/Setup/AxesPanel.h"
#include "GUI/View/Setup/FrameActions.h"
#include <QVBoxLayout>

Fit2DFrame::Fit2DFrame()
    : AnydataFrame(std::make_unique<DataFromJob>())
    , m_real_canvas(new ColorMapCanvas)
    , m_simu_canvas(new ColorMapCanvas)
    , m_diff_canvas(new ColorMapCanvas)
    , m_status_label(new PlotStatusLabel(
          {m_real_canvas->colorMap(), m_simu_canvas->colorMap(), m_diff_canvas->colorMap()}))
    , m_reset_view_action(new QAction(this))
{
    auto* vlayout = new QVBoxLayout;
    vlayout->setContentsMargins(0, 0, 0, 0);
    vlayout->setSpacing(0);

    auto* gridLayout = new QGridLayout;
    gridLayout->setContentsMargins(0, 0, 0, 0);
    gridLayout->setSpacing(0);

    gridLayout->addWidget(m_real_canvas, 0, 0);
    gridLayout->addWidget(m_simu_canvas, 0, 1);
    gridLayout->addWidget(m_diff_canvas, 1, 0);
    auto* progress_canvas = new ProgressCanvas;
    gridLayout->addWidget(progress_canvas, 1, 1);

    vlayout->addLayout(gridLayout);
    vlayout->addWidget(m_status_label);

    auto* hlayout = new QHBoxLayout(this);
    hlayout->setContentsMargins(0, 0, 0, 0);
    hlayout->setSpacing(0);
    hlayout->addLayout(vlayout);

    auto* axes_panel = new AxesPanel(dataSource());
    hlayout->addWidget(axes_panel);
    axes_panel->hide();
    // TODO restore when we have a toolbar here
    // QObject::connect(toolbar->actions()->toggle_properties_panel, &QAction::triggered,
    // axes_panel,
    //                 &QWidget::setVisible);

    m_reset_view_action->setText("Center view");
    m_reset_view_action->setIcon(QIcon(":/images/camera-metering-center.svg"));
    m_reset_view_action->setToolTip("Reset View");
    connect(m_reset_view_action, &QAction::triggered, this, &Fit2DFrame::onResetViewAction);

    updateFrame();
}

Fit2DFrame::~Fit2DFrame() = default;

void Fit2DFrame::updateFrame()
{
    if (dataSource()->simuData2DItem()) {
        GUI::Util::Ranges::setCommonRangeZ(dataSource()->mainData2DItems());
        updateDiffData();
        connectItems();

        m_simu_canvas->itemToCanvas(dataSource()->simuData2DItem());
        m_real_canvas->itemToCanvas(dataSource()->realData2DItem());
        m_diff_canvas->itemToCanvas(dataSource()->diffData2DItem());

        show();

    } else
        hide();
}

void Fit2DFrame::onResetViewAction()
{
    ASSERT(dataSource()->simuData2DItem() && dataSource()->diffData2DItem()
           && dataSource()->realData2DItem());
    dataSource()->simuData2DItem()->resetView();
    dataSource()->realData2DItem()->resetView();
    dataSource()->diffData2DItem()->resetView();

    // synchronize data range between simulated and real
    GUI::Util::Ranges::setCommonRangeZ(dataSource()->mainData2DItems());
    gDoc->setModified();
}

void Fit2DFrame::connectItems()
{
    // sync XY view area between simulated, real and difference plots
    for (auto* senderItem : dataSource()->allData2DItems())
        for (auto* receiverItem : dataSource()->allData2DItems())
            if (receiverItem != senderItem)
                connect(senderItem, &DataItem::updateOtherPlots, receiverItem,
                        &DataItem::checkXYranges, Qt::UniqueConnection);

    // sync Z range between simulated and real
    connect(dataSource()->simuData2DItem(), &Data2DItem::alignRanges,
            [this] { GUI::Util::Ranges::setCommonRangeZ(dataSource()->mainData2DItems()); });

    // sync Z range: simu --> real
    connect(dataSource()->simuData2DItem(), &DataItem::updateOtherPlots,
            dataSource()->realData2DItem(), &Data2DItem::copyZRangeFromItem, Qt::UniqueConnection);

    // sync Z range: real --> simu
    connect(dataSource()->realData2DItem(), &DataItem::updateOtherPlots,
            dataSource()->simuData2DItem(), &Data2DItem::copyZRangeFromItem, Qt::UniqueConnection);

    // update diff data if simulation data changes
    connect(dataSource()->simuData2DItem(), &Data1DItem::datafieldChanged, this,
            &Fit2DFrame::updateDiffData, Qt::UniqueConnection);
}

void Fit2DFrame::updateDiffData()
{
    ASSERT(dataSource()->simuData2DItem() && dataSource()->diffData2DItem()
           && dataSource()->realData2DItem());
    if (!dataSource()->simuData2DItem()->c_field() || !dataSource()->realData2DItem()->c_field())
        return;

    dataSource()->diffData2DItem()->setDatafield(DataUtil::relativeDifferenceField(
        *dataSource()->simuData2DItem()->c_field(), *dataSource()->realData2DItem()->c_field()));

    // keep Z axis range up with data range
    dataSource()->diffData2DItem()->computeDataRange();
}
