//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      Sample/Interface/LayerInterface.h
//! @brief     Defines class LayerInterface.
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#ifdef SWIG
#error no need to expose this header to Swig
#endif // SWIG
#ifndef BORNAGAIN_SAMPLE_INTERFACE_LAYERINTERFACE_H
#define BORNAGAIN_SAMPLE_INTERFACE_LAYERINTERFACE_H

#include "Sample/Scattering/ISampleNode.h"
#include <memory>

class Layer;
class LayerRoughness;

//! Interface between two layers, possibly with roughness.

class LayerInterface : public ISampleNode {
public:
    ~LayerInterface() override;

    LayerInterface* clone() const override;
    std::string className() const final { return "LayerInterface"; }

    //! Creates rough interface between two layers
    static LayerInterface* createInterface(const Layer* top_layer, const Layer* bottom_layer,
                                           const LayerRoughness* roughness);

    //! Sets roughness of the interface.
    void setRoughness(const LayerRoughness* roughness);

    //! Returns roughness of the interface.
    const LayerRoughness* roughness() const;

    const Layer* topLayer() const;

    const Layer* bottomLayer() const;

    std::vector<const INode*> nodeChildren() const override;

    std::string validate() const override;

private:
    void setLayersTopBottom(const Layer* top_layer, const Layer* bottom_layer);
    LayerInterface();

    const Layer* m_top_layer;                          //!< pointer to the layer above interface
    const Layer* m_bottom_layer;                       //!< pointer to the layer below interface
    std::unique_ptr<const LayerRoughness> m_roughness; //!< roughness of the interface
};

inline const LayerRoughness* LayerInterface::roughness() const
{
    return m_roughness.get();
}

inline const Layer* LayerInterface::topLayer() const
{
    return m_top_layer;
}

inline const Layer* LayerInterface::bottomLayer() const
{
    return m_bottom_layer;
}

#endif // BORNAGAIN_SAMPLE_INTERFACE_LAYERINTERFACE_H
