(**************************************************************************)
(*                                                                        *)
(*  Copyright (C) 2012 Johannes 'josch' Schauer <j.schauer@email.de>      *)
(*                                                                        *)
(*  This library is free software: you can redistribute it and/or modify  *)
(*  it under the terms of the GNU Lesser General Public License as        *)
(*  published by the Free Software Foundation, either version 3 of the    *)
(*  License, or (at your option) any later version.  A special linking    *)
(*  exception to the GNU Lesser General Public License applies to this    *)
(*  library, see the COPYING file for more information.                   *)
(**************************************************************************)

open ExtLib
open Common
open DoseparseNoRpm

include Util.Logging(struct let label = __FILE__ end) ;;

module Options = struct
  open OptParse

  let description = ("Given an acyclic srcgraph, calculate a partial order")
  let usage = "%prog srcgraph.xml Packages Soucres"

  let options = OptParser.make ~description ~usage
  include BootstrapCommon.MakeOptions(struct let options = options end)

  let noindep = StdOpt.store_false ()

  open OptParser

  let prog_group = add_group options "Program specific options" in
  add options ~group:prog_group ~long_name:"keep-indep" ~help:"Do not drop Build-Depends-Indep dependencies" noindep;

  include StdOptions.InputOptions;;
  let default = List.filter (fun e -> not (List.mem e ["checkonly"; "latest";"outfile"])) StdOptions.InputOptions.default_options in
  StdOptions.InputOptions.add_options ~default options ;;

  include StdOptions.DistribOptions;;
  let default = List.filter (fun e -> not (List.mem e ["deb-ignore-essential"; "inputtype"])) StdOptions.DistribOptions.default_options in
  StdOptions.DistribOptions.add_options ~default options ;;
end

let main () =
  let posargs = OptParse.OptParser.parse_argv Options.options in
  StdDebug.enable_debug (OptParse.Opt.get Options.verbose);
  Util.Debug.disable "Depsolver_int";
  StdDebug.all_quiet (OptParse.Opt.get Options.quiet);

  let noindep = OptParse.Opt.get Options.noindep in
  let options = Options.set_deb_options () in
  let buildarch = Option.get options.Debian.Debcudf.native in
  let hostarch = match options.Debian.Debcudf.host with None -> "" | Some s -> s in
  let foreignarchs = options.Debian.Debcudf.foreign in

  let sgf, posargs = match posargs with
   | sgf::posargs -> sgf,posargs
   | _ -> fatal "require srcgraph.xml Packages Sources"
  in

  let (binlist, (fgsrclist,bgsrclist), _) = BootstrapCommon.parse_packages ~noindep Options.parse_cmdline buildarch hostarch foreignarchs posargs in

  let tables = Debian.Debcudf.init_tables (fgsrclist@bgsrclist@binlist) in
  let fgsl = List.map (Debian.Debcudf.tocudf ~options tables) fgsrclist in
  let bgsl = List.map (Debian.Debcudf.tocudf ~options tables) bgsrclist in
  let pkglist = List.map (Debian.Debcudf.tocudf ~options tables) binlist in
  let universe = Cudf.load_universe (BootstrapCommon.unique [pkglist;fgsl;bgsl]) in

  let ic = open_in sgf in
  let sg = SrcGraph.from_ic universe ic in
  close_in ic;

  if SrcGraph.Dfs.has_cycle sg then fatal "graph is not a DAG";

  let order = SrcGraph.Utils.get_partial_order sg in

  let module SGE = SrcGraphExtras.Make(struct let univ = universe end) in

  List.iteri (fun i batch ->
    let batch = List.sort ~cmp:(fun v1 v2 ->
      CudfAdd.compare (SGE.package_of_vertex v1) (SGE.package_of_vertex v2)
    ) batch in
    let l = List.map SGE.string_of_vertex batch in
    Printf.printf "%d %s\n" (i+1) (String.concat " " l)
  ) order;
;;

main ();;
