/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2018 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

#include <cdi.h>

#include "cdo_int.h"
#include "grid.h"
#include "cdoDebugOutput.h"
#include "pstream_int.h"

/* read only the first data variable from input filename into a given double pointer */
static void
read_first_record(const char *filename, double *field)
{
  size_t nmiss;
  int varID, levelID;
  int streamID = streamOpenRead(filename);
  streamInqTimestep(streamID, 0);
  streamInqRecord(streamID, &varID, &levelID);
  streamReadRecord(streamID, field, &nmiss);
  streamClose(streamID);
}

/* count the number of locations, for which the mask is TRUE */
static int
countMask(const double *maskField, size_t gridSize, double falseVal)
{
  size_t counter = 0;

  for (size_t i = 0; i < gridSize; i++)
    {
      if (!DBL_IS_EQUAL(maskField[i], falseVal)) counter++;
    }

  return counter;
}

/*
 * the operators argument has to be a single horizontal field,
 * non-zero values are used to mark the relevant locations
 */
void *
MapReduce(void *process)
{
  int nrecs;
  int varID, levelID;
  size_t nmiss;

  cdoInitialize(process);

  if (operatorArgc() < 1) cdoAbort("Too few arguments!");

  /* check input grid type and size - this will be used for selecting relevant variables from the input file*/
  const char *maskfilename = operatorArgv()[0];
  int inputGridID = cdoDefineGrid(maskfilename);
  size_t inputGridSize = gridInqSize(inputGridID);
  int inputGridType = gridInqType(inputGridID);
  if (CdoDebug::cdoDebug) cdoPrint("MapReduce: input gridSize: %zu", inputGridSize);

  /* create an index list of the relevant locations */
  std::vector<int> maskIndexList;
  size_t maskSize;
  {
    std::vector<double> inputMaskField(inputGridSize);
    read_first_record(maskfilename, inputMaskField.data());

    /* non-zero values mark the relevant points */
    maskSize = countMask(inputMaskField.data(), inputGridSize, 0.0);
    if (CdoDebug::cdoDebug) cdoPrint("MapReduce: maskSize = %zu", maskSize);

    maskIndexList.resize(maskSize);

    for (size_t m = 0; m < maskSize; m++) maskIndexList[m] = -1;

    size_t k = 0;
    for (size_t i = 0; i < inputGridSize; i++)
      {
        if (!DBL_IS_EQUAL(inputMaskField[i], 0.0)) maskIndexList[k++] = i;
      }
  }

  /* check if coordinated bounds shound not be created */
  int nobounds = FALSE;
  int nocoords = FALSE;
  if (2 <= operatorArgc())
    {
      char *coordinatesLimitation = operatorArgv()[1];
      if (0 == strncmp("nobounds", coordinatesLimitation, 8)) nobounds = TRUE;
      if (0 == strncmp("nocoords", coordinatesLimitation, 8)) nocoords = TRUE;
    }
  /* create unstructured output grid including bounds*/
  int outputGridID = gridToUnstructuredSelecton(inputGridID, maskSize, maskIndexList.data(), nocoords, nobounds);

  /* create output vlist: Only variabes which have the same gridtype and
   * gridsize as the input mask should be proessed. Everything else is ignoreds
   * {{{ */
  int streamID1 = cdoStreamOpenRead(cdoStreamName(0));

  int vlistID1 = cdoStreamInqVlist(streamID1);
  int nvars = vlistNvars(vlistID1);
  std::vector<bool> vars(nvars, false);

  /* use vlist flags for marking the corresponding variables */
  vlistClearFlag(vlistID1);
  for (varID = 0; varID < nvars; varID++)
    {
      int gridID = vlistInqVarGrid(vlistID1, varID);
      if (inputGridType == gridInqType(gridID) && inputGridSize == gridInqSize(gridID))
        {
          vars[varID] = true;
          int zaxisID = vlistInqVarZaxis(vlistID1, varID);
          int nlevs = zaxisInqSize(zaxisID);
          for (int levID = 0; levID < nlevs; levID++)
            {
              vlistDefFlag(vlistID1, varID, levID, TRUE);
            }
        }
    }

  int vlistID2 = vlistCreate();
  cdoVlistCopyFlag(vlistID2, vlistID1);
  /* }}} */

  int taxisID1 = vlistInqTaxis(vlistID1);
  int taxisID2 = taxisDuplicate(taxisID1);
  vlistDefTaxis(vlistID2, taxisID2);

  /* use the new selection grid for all output variables */
  int ngrids = vlistNgrids(vlistID2);
  for (int index = 0; index < ngrids; index++) vlistChangeGridIndex(vlistID2, index, outputGridID);

  /* loop over input fields and mask the data values {{{ */
  int streamID2 = cdoStreamOpenWrite(cdoStreamName(1), cdoFiletype());
  pstreamDefVlist(streamID2, vlistID2);

  std::vector<double> arrayIn(inputGridSize);
  std::vector<double> arrayOut(maskSize);

  int tsID = 0;
  while ((nrecs = cdoStreamInqTimestep(streamID1, tsID)))
    {
      taxisCopyTimestep(taxisID2, taxisID1);
      pstreamDefTimestep(streamID2, tsID);

      for (int recID = 0; recID < nrecs; recID++)
        {
          pstreamInqRecord(streamID1, &varID, &levelID);
          if (vars[varID])
            {
              int varID2 = vlistFindVar(vlistID2, varID);
              int levelID2 = vlistFindLevel(vlistID2, varID, levelID);

              pstreamReadRecord(streamID1, arrayIn.data(), &nmiss);

              for (size_t i = 0; i < maskSize; i++) arrayOut[i] = arrayIn[maskIndexList[i]];

              pstreamDefRecord(streamID2, varID2, levelID2);
              pstreamWriteRecord(streamID2, arrayOut.data(), 0);
            }
        }
      tsID++;
    }
  /* }}} */

  pstreamClose(streamID2);
  pstreamClose(streamID1);

  cdoFinish();

  return 0;
}
