/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2019 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

/*
   This module contains the following operators:

      Intntime   intntime        Time interpolation
*/

#include <cdi.h>

#include "cdo_int.h"
#include "param_conversion.h"

#include "datetime.h"
#include "printinfo.h"

size_t
inttime(double fac1, double fac2, size_t gridsize, const double *single1, const double *single2, std::vector<double> &array,
        bool withMissval, double missval1, double missval2);

void *
Intntime(void *process)
{
  int varID, levelID;
  int64_t vdate;
  int vtime;

  cdoInitialize(process);

  operatorInputArg("number of timesteps between 2 timesteps");
  if (operatorArgc() < 1) cdoAbort("Too few arguments!");

  const int numts = parameter2int(operatorArgv()[0]);
  if (numts < 2) cdoAbort("parameter must be greater than 1!");

  CdoStreamID streamID1 = cdoOpenRead(0);

  const int vlistID1 = cdoStreamInqVlist(streamID1);
  const int vlistID2 = vlistDuplicate(vlistID1);

  VarList varList1, varList2;
  varListInit(varList1, vlistID1);
  varListInit(varList2, vlistID2);

  const int nvars = vlistNvars(vlistID1);

  const int maxrecs = vlistNrecs(vlistID1);
  std::vector<RecordInfo> recList(maxrecs);

  const size_t gridsizemax = vlistGridsizeMax(vlistID1);
  std::vector<double> array(gridsizemax);

  std::vector<std::vector<std::vector<size_t>>> nmiss(2);
  nmiss[0].resize(nvars);
  nmiss[1].resize(nvars);
  std::vector<std::vector<std::vector<double>>> vardata(2);
  vardata[0].resize(nvars);
  vardata[1].resize(nvars);

  for (varID = 0; varID < nvars; varID++)
    {
      const size_t gridsize = varList1[varID].gridsize;
      const int nlevel = varList1[varID].nlevels;
      nmiss[0][varID].resize(nlevel);
      nmiss[1][varID].resize(nlevel);
      vardata[0][varID].resize(gridsize * nlevel);
      vardata[1][varID].resize(gridsize * nlevel);
    }

  const int taxisID1 = vlistInqTaxis(vlistID1);
  const int taxisID2 = taxisDuplicate(taxisID1);
  if (taxisHasBounds(taxisID2)) taxisDeleteBounds(taxisID2);
  vlistDefTaxis(vlistID2, taxisID2);

  CdoStreamID streamID2 = cdoOpenWrite(1);

  cdoDefVlist(streamID2, vlistID2);

  int calendar = taxisInqCalendar(taxisID1);

  int curFirst = 0, curSecond = 1;

  int tsID = 0;
  int tsIDo = 0;
  int nrecs = cdoStreamInqTimestep(streamID1, tsID++);
  const int64_t vdate1 = taxisInqVdate(taxisID1);
  const int vtime1 = taxisInqVtime(taxisID1);
  JulianDate juldate1 = julianDateEncode(calendar, vdate1, vtime1);

  taxisCopyTimestep(taxisID2, taxisID1);
  cdoDefTimestep(streamID2, tsIDo++);
  for (int recID = 0; recID < nrecs; recID++)
    {
      cdoInqRecord(streamID1, &varID, &levelID);
      const size_t offset = varList1[varID].gridsize * levelID;
      double *single1 = &vardata[curFirst][varID][offset];
      cdoReadRecord(streamID1, single1, &nmiss[curFirst][varID][levelID]);

      cdoDefRecord(streamID2, varID, levelID);
      cdoWriteRecord(streamID2, single1, nmiss[curFirst][varID][levelID]);
    }

  while ((nrecs = cdoStreamInqTimestep(streamID1, tsID++)))
    {
      const int64_t vdate2 = taxisInqVdate(taxisID1);
      const int vtime2 = taxisInqVtime(taxisID1);
      JulianDate juldate2 = julianDateEncode(calendar, vdate2, vtime2);

      for (int recID = 0; recID < nrecs; recID++)
        {
          cdoInqRecord(streamID1, &varID, &levelID);

          recList[recID].varID = varID;
          recList[recID].levelID = levelID;

          const size_t offset = varList1[varID].gridsize * levelID;
          double *single2 = &vardata[curSecond][varID][offset];
          cdoReadRecord(streamID1, single2, &nmiss[curSecond][varID][levelID]);
        }

      for (int it = 1; it < numts; it++)
        {
          const double seconds = it * julianDateToSeconds(julianDateSub(juldate2, juldate1)) / numts;
          JulianDate juldate = julianDateAddSeconds(lround(seconds), juldate1);

          julianDateDecode(calendar, juldate, &vdate, &vtime);

          if (Options::cdoVerbose)
            {
              char vdatestr[32], vtimestr[32];
              date2str(vdate, vdatestr, sizeof(vdatestr));
              time2str(vtime, vtimestr, sizeof(vtimestr));
              cdoPrint("%s %s", vdatestr, vtimestr);
            }

          taxisDefVdate(taxisID2, vdate);
          taxisDefVtime(taxisID2, vtime);
          cdoDefTimestep(streamID2, tsIDo++);

          const double diff = julianDateToSeconds(julianDateSub(juldate2, juldate1));
          const double fac1 = julianDateToSeconds(julianDateSub(juldate2, juldate)) / diff;
          const double fac2 = julianDateToSeconds(julianDateSub(juldate, juldate1)) / diff;

          for (int recID = 0; recID < nrecs; recID++)
            {
              varID = recList[recID].varID;
              levelID = recList[recID].levelID;
              const size_t gridsize = varList1[varID].gridsize;
              const size_t offset = gridsize * levelID;
              const double *single1 = &vardata[curFirst][varID][offset];
              const double *single2 = &vardata[curSecond][varID][offset];

              const bool withMissval = nmiss[curFirst][varID][levelID] || nmiss[curSecond][varID][levelID];
              const size_t nmiss3 = inttime(fac1, fac2, gridsize, single1, single2, array,
                                            withMissval, varList1[varID].missval, varList2[varID].missval);

              cdoDefRecord(streamID2, varID, levelID);
              cdoWriteRecord(streamID2, array.data(), nmiss3);
            }
        }

      taxisDefVdate(taxisID2, vdate2);
      taxisDefVtime(taxisID2, vtime2);
      cdoDefTimestep(streamID2, tsIDo++);
      for (int recID = 0; recID < nrecs; recID++)
        {
          varID = recList[recID].varID;
          levelID = recList[recID].levelID;
          const size_t offset = varList2[varID].gridsize * levelID;
          double *single2 = &vardata[curSecond][varID][offset];

          cdoDefRecord(streamID2, varID, levelID);
          cdoWriteRecord(streamID2, single2, nmiss[curSecond][varID][levelID]);
        }

      juldate1 = juldate2;
      std::swap(curFirst, curSecond);
    }

  cdoStreamClose(streamID2);
  cdoStreamClose(streamID1);

  cdoFinish();

  return 0;
}
