/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2019 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <iostream>
#include <vector>

#ifdef HAVE_EXECINFO_H
#include <execinfo.h>
#endif

#include <fenv.h>
#include <sys/stat.h>
#ifdef HAVE_GETRLIMIT
#ifdef HAVE_SYS_RESOURCE_H
#include <sys/time.h>     /* getrlimit */
#include <sys/resource.h> /* getrlimit */
#endif
#endif
#include <unistd.h> /* sysconf, gethostname */
#include <csignal>
#include "timer.h"

#if defined(SX)
#define RLIM_T long long
#else
#define RLIM_T rlim_t
#endif

#include <cdi.h>
#include <mpim_grid.h>

#include "cdo_int.h"
#include "param_conversion.h"
#include "cdo_task.h"
#include "progress.h"

#include "cdo_getopt.h"

#ifdef HAVE_LIBPTHREAD
#include "pthread_debug.h"
#endif

#include "module_list.h"
#include "percentiles.h"
#include "util_wildcards.h"
#include "util_string.h"
#include "process_int.h"
#include "cdo_options.h"
#include "timer.h"
#include "commandline.h"
#include "mpmo_color.h"
#include "cdo_output.h"
#include "cdo_features.h"
#include "cdo_zaxis.h"
#include "compare.h"
#include "dmemory.h"
#include "datetime.h"
#include "remap_grid_cell_search.h"

#ifdef _OPENMP
#include <omp.h>
#endif

#ifndef VERSION
#define VERSION "0.0.1"
#endif

#define MAX_NUM_VARNAMES 256

#include <cstring>

void
cdoExit()
{
  cdoKillProcesses();
  cdoCloseAllStreams();
  exit(EXIT_FAILURE);
}

static int Debug = 0;
static int Version = 0;
static int Help = 0;
static int DebugLevel = 0;
static int numThreads = 0;
static int timer_total;
static int CDO_netcdf_hdr_pad = 0;
static int CDO_Rusage = 0;

extern "C" void streamGrbDefDataScanningMode(int scanmode);

void setPointSearchMethod(const char *methodstr);

#define PRINT_RLIMIT(resource)                                                           \
  {                                                                                      \
    struct rlimit rlim;                                                                  \
    const int status = getrlimit(resource, &rlim);                                       \
    if (status == 0)                                                                     \
      {                                                                                  \
        if (sizeof(RLIM_T) > sizeof(long))                                               \
          {                                                                              \
            fprintf(stderr, "CUR %-15s = %llu\n", #resource, (long long) rlim.rlim_cur); \
            fprintf(stderr, "MAX %-15s = %llu\n", #resource, (long long) rlim.rlim_max); \
          }                                                                              \
        else                                                                             \
          {                                                                              \
            fprintf(stderr, "CUR %-15s = %lu\n", #resource, (long) rlim.rlim_cur);       \
            fprintf(stderr, "MAX %-15s = %lu\n", #resource, (long) rlim.rlim_max);       \
          }                                                                              \
      }                                                                                  \
  }

#define ITSME (cstrIsEqual(Cdo::Username, "\x6d\x32\x31\x34\x30\x30\x33"))

static void
cdo_stackframe()
{
#if defined HAVE_EXECINFO_H && defined HAVE_BACKTRACE
  void *callstack[32];
  const int frames = backtrace(callstack, 32);
  char **messages = backtrace_symbols(callstack, frames);

  fprintf(stderr, "[bt] Execution path:\n");
  if (messages)
    {
      for (int i = 0; i < frames; ++i) fprintf(stderr, "[bt] %s\n", messages[i]);
      free(messages);
    }
#endif
}

static int
cdo_feenableexcept(int excepts)
{
#if defined HAVE_FEENABLEEXCEPT
  int feenableexcept(int);
  int old_excepts = feenableexcept(excepts);
  return old_excepts;
#else
  static fenv_t fenv;
  unsigned new_excepts = ((unsigned) excepts) & FE_ALL_EXCEPT;
  int old_excepts = -1;  // previous masks

  if (fegetenv(&fenv)) return -1;
#if defined(HAVE_FENV_T___CONTROL) && defined(HAVE_FENV_T___MXCSR)
  old_excepts = (int) (fenv.__control & FE_ALL_EXCEPT);

  // unmask
  fenv.__control &= ~new_excepts;
  fenv.__mxcsr &= ~(new_excepts << 7);
#endif

  return (fesetenv(&fenv) ? -1 : old_excepts);
#endif
}

static void
cdoSignalHandler(int signo)
{
  if (signo == SIGFPE)
    {
      cdo_stackframe();
      cdoAbort("floating-point exception!");
    }
}

static void
cdoSetDigits(const char *optarg)
{
  char *ptr1 = 0;
  if (optarg != 0 && (int) strlen(optarg) > 0 && optarg[0] != ',') Options::CDO_flt_digits = (int) strtol(optarg, &ptr1, 10);

  if (Options::CDO_flt_digits < 1 || Options::CDO_flt_digits > 20)
    cdoAbort("Unreasonable value for float significant digits: %d", Options::CDO_flt_digits);

  if (ptr1 && *ptr1 == ',')
    {
      char *ptr2 = 0;
      Options::CDO_dbl_digits = (int) strtol(ptr1 + 1, &ptr2, 10);
      if (ptr2 == ptr1 + 1 || Options::CDO_dbl_digits < 1 || Options::CDO_dbl_digits > 20)
        cdoAbort("Unreasonable value for double significant digits: %d", Options::CDO_dbl_digits);
    }
}

static void
cdo_version()
{
  const int filetypes[] = { CDI_FILETYPE_SRV, CDI_FILETYPE_EXT, CDI_FILETYPE_IEG, CDI_FILETYPE_GRB,  CDI_FILETYPE_GRB2,
                            CDI_FILETYPE_NC,  CDI_FILETYPE_NC2, CDI_FILETYPE_NC4, CDI_FILETYPE_NC4C, CDI_FILETYPE_NC5 };
  const char *typenames[] = { "srv", "ext", "ieg", "grb1", "grb2", "nc1", "nc2", "nc4", "nc4c", "nc5" };

  fprintf(stderr, "%s\n", Cdo::Version);
#ifdef SYSTEM_TYPE
  fprintf(stderr, "System: %s\n", SYSTEM_TYPE);
#endif
#ifdef CXX_COMPILER
  fprintf(stderr, "CXX Compiler: %s\n", CXX_COMPILER);
#ifdef CXX_VERSION
  fprintf(stderr, "CXX version : %s\n", CXX_VERSION);
#endif
#endif
#ifdef C_COMPILER
  fprintf(stderr, "C Compiler: %s\n", C_COMPILER);
#ifdef C_VERSION
  fprintf(stderr, "C version : %s\n", C_VERSION);
#endif
#endif
#ifdef F77_COMPILER
  fprintf(stderr, "F77 Compiler: %s\n", F77_COMPILER);
#ifdef F77_VERSION
  fprintf(stderr, "F77 version : %s\n", F77_VERSION);
#endif
#endif

  printFeatures();
  printLibraries();

  fprintf(stderr, "Filetypes: ");
  set_text_color(stderr, BRIGHT, GREEN);
  for (size_t i = 0; i < sizeof(filetypes) / sizeof(int); ++i)
    if (cdiHaveFiletype(filetypes[i])) fprintf(stderr, "%s ", typenames[i]);
  reset_text_color(stderr);
  fprintf(stderr, "\n");

  cdiPrintVersion();
  fprintf(stderr, "\n");
}

static void
cdo_usage()
{
  const char *name;

  /*  fprintf(stderr, "%s\n", CDO_version);*/
  /*  fprintf(stderr, "\n");*/
  fprintf(stderr, "usage : cdo  [Options]  Operator1  [-Operator2  [-OperatorN]]\n");
  fprintf(stderr, "\n");
  fprintf(stderr, "  Options:\n");
  set_text_color(stderr, BLUE);
  fprintf(stderr, "    -a             Generate an absolute time axis\n");
  fprintf(stderr, "    -b <nbits>     Set the number of bits for the output precision\n");
  fprintf(stderr, "                   (I8/I16/I32/F32/F64 for "
                  "nc1/nc2/nc4/nc4c/nc5; F32/F64 for grb2/srv/ext/ieg; P1 - P24 for grb1/grb2)\n");
  fprintf(stderr, "                   Add L or B to set the byteorder to Little or Big endian\n");
  fprintf(stderr, "    --cmor         CMOR conform NetCDF output\n");
  fprintf(stderr, "    -C, --no_color Disable colorized output messages\n");
  fprintf(stderr, "    --eccodes      Use ecCodes to decode/encode GRIB1 messages\n");
  fprintf(stderr, "    --enableexcept <except>\n");
  fprintf(stderr, "                   Set individual floating-point traps "
                  "(DIVBYZERO, INEXACT, INVALID, OVERFLOW, UNDERFLOW, ALL_EXCEPT)\n");
  fprintf(stderr, "    -f, --format <format>\n");
  fprintf(stderr, "                   Format of the output file. (grb1/grb2/nc1/nc2/nc4/nc4c/nc5/srv/ext/ieg)\n");
  fprintf(stderr, "    -g <grid>      Set default grid name or file. Available grids: \n");
  fprintf(stderr, "                   n<N>, t<RES>, tl<RES>, global_<DXY>, r<NX>x<NY>, g<NX>x<NY>, gme<NI>, lon=<LON>/lat=<LAT>\n");
  fprintf(stderr, "    -h, --help     Help information for the operators\n");
  fprintf(stderr, "    --history      Do not append to NetCDF \"history\" global attribute\n");
  fprintf(stderr, "    --netcdf_hdr_pad, --hdr_pad, --header_pad <nbr>\n");
  fprintf(stderr, "                   Pad NetCDF output header with nbr bytes\n");
  /*
  fprintf(stderr, "    -i <inst>      Institution name/file\n");
  fprintf(stderr, "                   Predefined instituts: ");
  for ( int id = 0; id < institutInqNumber; id++ )
    if ( (name = institutInqNamePtr(id)) )
      fprintf(stderr, " %s", name);
  fprintf(stderr, "\n");
  */
  /* fprintf(stderr, "    -l <level>     Level file\n"); */
  fprintf(stderr, "    -k <chunktype> NetCDF4 chunk type: auto, grid or lines\n");
  fprintf(stderr, "    -L             Lock IO (sequential access)\n");
  fprintf(stderr, "    -M             Switch to indicate that the I/O streams have missing values\n");
  fprintf(stderr, "    -m <missval>   Set the missing value of non NetCDF files (default: %g)\n", cdiInqMissval());
  fprintf(stderr, "    --no_warnings  Inhibit warning messages\n");
  fprintf(stderr, "    -O             Overwrite existing output file, if checked\n");
  fprintf(stderr, "    --operators    List of all operators\n");
  fprintf(stderr, "    --pedantic     Warnings count as errors\n");
#ifdef _OPENMP
  fprintf(stderr, "    -P <nthreads>  Set number of OpenMP threads\n");
#endif
  fprintf(stderr, "    --percentile <method>\n");
  fprintf(stderr, "                   Percentile method: nrank, nist, numpy, numpy_lower, numpy_higher, numpy_nearest\n");
  fprintf(stderr, "    --precision <float_digits[,double_digits]>\n");
  fprintf(stderr, "                   Precision to use in displaying floating-point data (default: 7,15)\n");
  if (ITSME) fprintf(stderr, "    --pointsearchmethod [full/kdtree/nanoflann/spherepart/latbins]\n");
  fprintf(stderr, "    --reduce_dim   Reduce NetCDF dimensions\n");
  if (ITSME) fprintf(stderr, "    --remap_genweights [0/1]\n");
  fprintf(stderr, "    -R, --regular  Convert GRIB1 data from global reduced to regular Gaussian grid (cgribex only)\n");
  fprintf(stderr, "    -r             Generate a relative time axis\n");
  fprintf(stderr, "    -S             Create an extra output stream for the module TIMSTAT. This stream\n");
  fprintf(stderr, "                   contains the number of non missing values for each output period.\n");
  fprintf(stderr, "    -s, --silent   Silent mode\n");
  fprintf(stderr, "    --sortname     Alphanumeric sorting of NetCDF parameter names\n");
  fprintf(stderr, "    -t <codetab>   Set GRIB1 default parameter code table name or file (cgribex only)\n");
  fprintf(stderr, "                   Predefined tables: ");
  for (int id = 0; id < tableInqNumber(); id++)
    if ((name = tableInqNamePtr(id))) fprintf(stderr, " %s", name);
  fprintf(stderr, "\n");

  fprintf(stderr, "    --timestat_date <srcdate>\n");
  fprintf(stderr, "                   Target timestamp (temporal statistics): "
                  "first, middle, midhigh or last source timestep.\n");
  fprintf(stderr, "    -V, --version  Print the version number\n");
  fprintf(stderr, "    -v, --verbose  Print extra details for some operators\n");
  fprintf(stderr, "    -W             Disable Warnings\n");
  fprintf(stderr, "    --worker <num> Number of worker to decode/decompress GRIB records\n");
  fprintf(stderr, "    -z szip        SZIP compression of GRIB1 records\n");
  fprintf(stderr, "       aec         AEC compression of GRIB2 records\n");
  fprintf(stderr, "       jpeg        JPEG compression of GRIB2 records\n");
  fprintf(stderr, "        zip[_1-9]  Deflate compression of NetCDF4 variables\n");
#ifdef HIRLAM_EXTENSIONS
  fprintf(stderr, "    --Dkext <debLev>   Setting debugLevel for extensions\n");
  fprintf(stderr, "    --outputGribDataScanningMode <mode>   Setting grib scanning mode for data in output file <0, 64, 96>; "
                  "Default is 64\n");
#endif  // HIRLAM_EXTENSIONS
  reset_text_color(stderr);
  fprintf(stderr, "\n");

  fprintf(stderr, "  Operators:\n");
  fprintf(stderr, "    Use option --operators for a list of all operators.\n");

  fprintf(stderr, "\n");
  fprintf(stderr, "  CDO version %s, Copyright (C) 2003-2019 Uwe Schulzweida\n", VERSION);
  fprintf(stderr, "  This is free software and comes with ABSOLUTELY NO WARRANTY\n");
  fprintf(stderr, "  Report bugs to <http://mpimet.mpg.de/cdo>\n");
}

static void
cdo_init_is_tty()
{
  struct stat statbuf;
  fstat(0, &statbuf);
  if (S_ISCHR(statbuf.st_mode)) stdin_is_tty = true;
  fstat(1, &statbuf);
  if (S_ISCHR(statbuf.st_mode))
    {
      stdout_is_tty = true;
      Progress::stdout_is_tty = true;
    }
  fstat(2, &statbuf);
  if (S_ISCHR(statbuf.st_mode)) stderr_is_tty = true;
}

static void
cdoPrintHelp(const std::vector<std::string> &help)
{
  if (help.empty())
    fprintf(stderr, "No help available for this operator!\n");
  else
    {
      for (unsigned long i = 0; i < help.size(); i++)
        {
          const bool lprint = !(help[i][0] == '\0' && help[i + 1][0] == ' ');
          if (lprint)
            {
              if (COLOR_STDOUT)
                {
                  if ((help[i] == "NAME") || (help[i] == "SYNOPSIS") || (help[i] == "DESCRIPTION") || (help[i] == "OPERATORS")
                      || (help[i] == "NAMELIST") || (help[i] == "PARAMETER") || (help[i] == "ENVIRONMENT") || (help[i] == "NOTE")
                      || (help[i] == "EXAMPLES"))
                    {
                      set_text_color(stdout, BRIGHT, BLACK);
                      fprintf(stdout, "%s", help[i].c_str());
                      reset_text_color(stdout);
                      fprintf(stdout, "\n");
                    }
                  else
                    fprintf(stdout, "%s\n", help[i].c_str());
                }
              else
                {
                  fprintf(stdout, "%s\n", help[i].c_str());
                }
            }
        }
    }
}

#undef IsBigendian
#define IsBigendian() (u_byteorder.c[sizeof(long) - 1])

static void
setDefaultDataType(const char *datatypestr)
{
  static union
  {
    unsigned long l;
    unsigned char c[sizeof(long)];
  } u_byteorder = { 1 };
  int nbits = -1;
  enum
  {
    D_UINT,
    D_INT,
    D_FLT,
    D_CPX
  };
  int dtype = -1;

  const int datatype = tolower(*datatypestr);
  // clang-format off
  if      (datatype == 'i') { dtype = D_INT;  datatypestr++; }
  else if (datatype == 'u') { dtype = D_UINT; datatypestr++; }
  else if (datatype == 'f') { dtype = D_FLT;  datatypestr++; }
  else if (datatype == 'c') { dtype = D_CPX;  datatypestr++; }
  else if (datatype == 'p') {                 datatypestr++; }
  // clang-format on

  if (isdigit((int) *datatypestr))
    {
      nbits = atoi(datatypestr);
      datatypestr += 1;
      if (nbits >= 10) datatypestr += 1;

      if (dtype == -1)
        {
          if (nbits > 0 && nbits < 32)
            CdoDefault::DataType = nbits;
          else if (nbits == 32)
            CdoDefault::DataType = (CdoDefault::FileType == CDI_FILETYPE_GRB) ? CDI_DATATYPE_PACK32 : CDI_DATATYPE_FLT32;
          else if (nbits == 64)
            CdoDefault::DataType = CDI_DATATYPE_FLT64;
          else
            {
              fprintf(stderr, "Unsupported number of bits %d!\n", nbits);
              fprintf(stderr, "Use I8/I16/I32/F32/F64 for nc1/nc2/nc4/nc4c/nc5; F32/F64 for "
                              "grb2/srv/ext/ieg; P1 - P24 for grb1/grb2.\n");
              exit(EXIT_FAILURE);
            }
        }
      else
        {
          // clang-format off
          if (dtype == D_INT)
            {
              if      (nbits ==  8) CdoDefault::DataType = CDI_DATATYPE_INT8;
              else if (nbits == 16) CdoDefault::DataType = CDI_DATATYPE_INT16;
              else if (nbits == 32) CdoDefault::DataType = CDI_DATATYPE_INT32;
              else cdoAbort("Unsupported number of bits = %d for datatype INT!", nbits);
            }
          else if (dtype == D_UINT)
            {
              if      (nbits ==  8) CdoDefault::DataType = CDI_DATATYPE_UINT8;
              else if (nbits == 16) CdoDefault::DataType = CDI_DATATYPE_UINT16;
              else if (nbits == 32) CdoDefault::DataType = CDI_DATATYPE_UINT32;
              else cdoAbort("Unsupported number of bits = %d for datatype UINT!", nbits);
            }
          else if (dtype == D_FLT)
            {
              if      (nbits == 32) CdoDefault::DataType = CDI_DATATYPE_FLT32;
              else if (nbits == 64) CdoDefault::DataType = CDI_DATATYPE_FLT64;
              else cdoAbort("Unsupported number of bits = %d for datatype FLT!", nbits);
            }
          else if (dtype == D_CPX)
            {
              if      (nbits == 32) CdoDefault::DataType = CDI_DATATYPE_CPX32;
              else if (nbits == 64) CdoDefault::DataType = CDI_DATATYPE_CPX64;
              else cdoAbort("Unsupported number of bits = %d for datatype CPX!", nbits);
            }
          // clang-format on
        }
    }

  if (*datatypestr != 0)
    {
      if (*datatypestr == 'l' || *datatypestr == 'L')
        {
          if (IsBigendian()) CdoDefault::Byteorder = CDI_LITTLEENDIAN;
          datatypestr++;
        }
      else if (*datatypestr == 'b' || *datatypestr == 'B')
        {
          if (!IsBigendian()) CdoDefault::Byteorder = CDI_BIGENDIAN;
          datatypestr++;
        }
      else
        {
          cdoAbort("Unsupported character in number of bytes: >%s< !", datatypestr);
        }
    }
}
/*
static
void setDefaultDataTypeByte(char *datatypestr)
{
  static union {unsigned long l; unsigned char c[sizeof(long)];} u_byteorder =
{1}; int datatype = -1;

  if ( isdigit((int) *datatypestr) )
    {
      datatype = atoi(datatypestr);
      datatypestr++;

      if      ( datatype == 1 ) CdoDefault::DataType = CDI_DATATYPE_PACK8;
      else if ( datatype == 2 ) CdoDefault::DataType = CDI_DATATYPE_PACK16;
      else if ( datatype == 3 ) CdoDefault::DataType = CDI_DATATYPE_PACK24;
      else if ( datatype == 4 ) CdoDefault::DataType = CDI_DATATYPE_FLT32;
      else if ( datatype == 8 ) CdoDefault::DataType = CDI_DATATYPE_FLT64;
      else
        {
          fprintf(stderr, "Unsupported datatype %d!\n", datatype);
          fprintf(stderr, "Use 4/8 for filetype nc/srv/ext/ieg and 1/2/3 for
grb1/grb2.\n"); exit(EXIT_FAILURE);
        }
    }

  if ( *datatypestr != 0 )
    {
      if ( *datatypestr == 'l' || *datatypestr == 'L' )
        {
          if ( IsBigendian() ) CdoDefault::Byteorder = CDI_LITTLEENDIAN;
          datatypestr++;
        }
      else if ( *datatypestr == 'b' || *datatypestr == 'B' )
        {
          if ( ! IsBigendian() ) CdoDefault::Byteorder = CDI_BIGENDIAN;
          datatypestr++;
        }
      else
        {
          cdoAbort("Unsupported character in number of bytes: %s!", datatypestr);
        }
    }
}
*/
static void
setDefaultFileType(const char *filetypestr)
{
  if (filetypestr)
    {
      const char *ftstr = filetypestr;
      size_t len;

      // clang-format off
      if      ( cmpstrlen(filetypestr, "grb2", len)  == 0 ) { ftstr += len; CdoDefault::FileType = CDI_FILETYPE_GRB2;}
      else if ( cmpstrlen(filetypestr, "grb1", len)  == 0 ) { ftstr += len; CdoDefault::FileType = CDI_FILETYPE_GRB; }
      else if ( cmpstrlen(filetypestr, "grb",  len)  == 0 ) { ftstr += len; CdoDefault::FileType = CDI_FILETYPE_GRB; }
      else if ( cmpstrlen(filetypestr, "nc2",  len)  == 0 ) { ftstr += len; CdoDefault::FileType = CDI_FILETYPE_NC2; }
      else if ( cmpstrlen(filetypestr, "nc4c", len)  == 0 ) { ftstr += len; CdoDefault::FileType = CDI_FILETYPE_NC4C;}
      else if ( cmpstrlen(filetypestr, "nc4",  len)  == 0 ) { ftstr += len; CdoDefault::FileType = CDI_FILETYPE_NC4; }
      else if ( cmpstrlen(filetypestr, "nc5",  len)  == 0 ) { ftstr += len; CdoDefault::FileType = CDI_FILETYPE_NC5; }
      else if ( cmpstrlen(filetypestr, "nc1",  len)  == 0 ) { ftstr += len; CdoDefault::FileType = CDI_FILETYPE_NC;  }
      else if ( cmpstrlen(filetypestr, "nc",   len)  == 0 ) { ftstr += len; CdoDefault::FileType = CDI_FILETYPE_NC2; }
      else if ( cmpstrlen(filetypestr, "srv",  len)  == 0 ) { ftstr += len; CdoDefault::FileType = CDI_FILETYPE_SRV; }
      else if ( cmpstrlen(filetypestr, "ext",  len)  == 0 ) { ftstr += len; CdoDefault::FileType = CDI_FILETYPE_EXT; }
      else if ( cmpstrlen(filetypestr, "ieg",  len)  == 0 ) { ftstr += len; CdoDefault::FileType = CDI_FILETYPE_IEG; }
      else
        {
          cdoWarning("Unsupported filetype %s!", filetypestr);
          cdoWarning("Available filetypes: grb1/grb2/nc1/nc2/nc4/nc4c/nc5/srv/ext/ieg");
          cdoAbort("Unsupported filetype %s!", filetypestr);
        }
      // clang-format on

      if (CdoDefault::FileType != CDI_UNDEFID && *ftstr != 0)
        {
          if (*ftstr == '_')
            {
              setDefaultDataType(++ftstr);
            }
          else
            {
              fprintf(stderr, "Unexpected character >%c< in file type >%s<!\n", *ftstr, filetypestr);
              fprintf(stderr, "Use format[_nbits] with:\n");
              fprintf(stderr, "    format = grb1, grb2, nc1, nc2, nc4, nc4c, nc5, srv, ext or ieg\n");
              fprintf(stderr, "    nbits  = 32/64 for grb2/nc1/nc2/nc4/nc4c/nc5/srv/ext/ieg; 1 - 24 for grb1/grb2\n");
              exit(EXIT_FAILURE);
            }
        }
    }
}

#define NTESTS 11
#include <inttypes.h>
static int
getMemAlignment()
{
  int ma = -1;
  double *ptr[NTESTS];
  int64_t iptr;
  const size_t tsize[NTESTS] = { 1, 3, 5, 9, 17, 33, 69, 121, 251, 510, 1025 };
  const size_t ma_check[4] = { 8, 16, 32, 64 };
  int ma_result[4] = { 1, 1, 1, 1 };

  for (int i = 0; i < NTESTS; ++i)
    {
      ptr[i] = (double *) malloc(tsize[i]);
      iptr = (int64_t) ptr[i];
      for (int k = 0; k < 4; ++k)
        if (iptr % ma_check[k]) ma_result[k] = 0;
    }
  for (auto &i : ptr) free(i);

  for (int i = NTESTS - 1; i >= 0; i--)
    {
      ptr[i] = (double *) malloc(tsize[i] + 5);
      iptr = (int64_t) ptr[i];
      for (int k = 0; k < 4; ++k)
        if (iptr % ma_check[k]) ma_result[k] = 0;
    }
  for (auto &i : ptr) free(i);

  for (int k = 0; k < 4; ++k)
    if (ma_result[k]) ma = ma_check[k];

  return ma;
}

static void
defineCompress(const char *arg)
{
  const size_t len = strlen(arg);

  if (strncmp(arg, "szip", len) == 0)
    {
      Options::cdoCompType = CDI_COMPRESS_SZIP;
      Options::cdoCompLevel = 0;
    }
  else if (strncmp(arg, "aec", len) == 0 || strncmp(arg, "ccsds", len) == 0)
    {
      Options::cdoCompType = CDI_COMPRESS_AEC;
      Options::cdoCompLevel = 0;
    }
  else if (strncmp(arg, "jpeg", len) == 0)
    {
      Options::cdoCompType = CDI_COMPRESS_JPEG;
      Options::cdoCompLevel = 0;
    }
  else if (strncmp(arg, "zip", 3) == 0)
    {
      Options::cdoCompType = CDI_COMPRESS_ZIP;
      Options::cdoCompLevel = (len == 5 && arg[3] == '_' && isdigit(arg[4])) ? atoi(&arg[4]) : 1;
    }
  else
    {
      cdoAbort("Compression type '%s' unsupported!", arg);
    }
}

static void
defineChunktype(const char *arg)
{
  // clang-format off
  if      (cstrIsEqual("auto",  arg)) Options::cdoChunkType = CDI_CHUNK_AUTO;
  else if (cstrIsEqual("grid",  arg)) Options::cdoChunkType = CDI_CHUNK_GRID;
  else if (cstrIsEqual("lines", arg)) Options::cdoChunkType = CDI_CHUNK_LINES;
  else cdoAbort("Chunk type '%s' unsupported!", arg);
  // clang-format on
}

static void
defineVarnames(const char *arg)
{
  size_t len = strlen(arg);
  size_t istart = 0;
  while (istart < len && (arg[istart] == ' ' || arg[istart] == ',')) istart++;

  len -= istart;

  if (len)
    {
      Options::cdoVarnames = (char **) Malloc(MAX_NUM_VARNAMES * sizeof(char *));

      char *pbuf = strdup(arg + istart);
      Options::cdoVarnames[Options::cdoNumVarnames++] = pbuf;

      char *commapos = pbuf;
      while ((commapos = strchr(commapos, ',')) != nullptr)
        {
          *commapos++ = '\0';
          if (strlen(commapos))
            {
              if (Options::cdoNumVarnames >= MAX_NUM_VARNAMES) cdoAbort("Too many variable names (limit=%d)!", MAX_NUM_VARNAMES);

              Options::cdoVarnames[Options::cdoNumVarnames++] = commapos;
            }
        }
      /*
      for ( int i = 0; i < Options::cdoNumVarnames; ++i )
        printf("varname %d: %s\n", i+1, Options::cdoVarnames[i]);
      */
    }
}

static void
get_env_vars()
{
  Cdo::Username = getenv("LOGNAME");
  if (Cdo::Username == nullptr)
    {
      Cdo::Username = getenv("USER");
      if (Cdo::Username == nullptr) Cdo::Username = "unknown";
    }

  char *envstr = getenv("CDO_GRID_SEARCH_DIR");
  if (envstr)
    {
      size_t len = strlen(envstr);
      if (len > 0)
        {
          len += 2;
          char *gridSearchDir = (char *) Malloc(len);
          memcpy(gridSearchDir, envstr, len - 1);
          if (gridSearchDir[len - 3] != '/')
            {
              gridSearchDir[len - 2] = '/';
              gridSearchDir[len - 1] = 0;
            }
        }
    }

  envstr = getenv("CDO_DISABLE_HISTORY");
  if (envstr)
    {
      if (parameter2bool(envstr) == true)
        {
          Options::CDO_Reset_History = true;
          if (Options::cdoVerbose) fprintf(stderr, "CDO_DISABLE_HISTORY = %s\n", envstr);
        }
    }

  envstr = getenv("CDO_RESET_HISTORY");
  if (envstr)
    {
      if (parameter2bool(envstr) == true)
        {
          Options::CDO_Reset_History = true;
          if (Options::cdoVerbose) fprintf(stderr, "CDO_RESET_HISTORY = %s\n", envstr);
        }
    }

  envstr = getenv("CDO_HISTORY_INFO");
  if (envstr)
    {
      const bool ival = parameter2bool(envstr);
      Options::CDO_Append_History = ival;
      if (Options::cdoVerbose) fprintf(stderr, "CDO_HISTORY_INFO = %s\n", envstr);
    }

  Cdo::File_Suffix[0] = 0;
  envstr = getenv("CDO_FILE_SUFFIX");
  if (envstr)
    {
      if (envstr[0])
        {
          strncat(Cdo::File_Suffix, envstr, sizeof(Cdo::File_Suffix) - 1);
          if (Options::cdoVerbose) fprintf(stderr, "CDO_FILE_SUFFIX = %s\n", envstr);
        }
    }

  envstr = getenv("CDO_DISABLE_FILESUFFIX");
  if (envstr)
    {
      if (parameter2bool(envstr) == true)
        {
          strcat(Cdo::File_Suffix, "nullptr");
          if (Options::cdoVerbose) fprintf(stderr, "CDO_DISABLE_FILESUFFIX = %s\n", envstr);
        }
    }

  envstr = getenv("CDO_DIAG");
  if (envstr)
    {
      if (parameter2bool(envstr) == true)
        {
          Options::cdoDiag = true;
          if (Options::cdoVerbose) fprintf(stderr, "CDO_DIAG = %s\n", envstr);
        }
    }

  envstr = getenv("CDO_USE_FFTW");
  if (envstr)
    {
      const bool ival = parameter2bool(envstr);
      Options::Use_FFTW = ival;
      if (Options::cdoVerbose) fprintf(stderr, "Options::Use_FFTW = %s\n", envstr);
    }

  envstr = getenv("CDO_VERSION_INFO");
  if (envstr)
    {
      const bool ival = parameter2bool(envstr);
      Options::VersionInfo = ival;
      if (Options::cdoVerbose) fprintf(stderr, "Options::VersionInfo = %s\n", envstr);
    }

  envstr = getenv("CDO_COLOR");
  if (envstr)
    {
      const bool ival = parameter2bool(envstr);
      CDO_Color = ival;
      if (Options::cdoVerbose) fprintf(stderr, "CDO_COLOR = %s\n", envstr);
    }
  else if (CDO_Color == false && ITSME)
    CDO_Color = true;
}

static void
print_system_info()
{
  fprintf(stderr, "\n");
  fprintf(stderr, "CDO_Color           = %d\n", CDO_Color);
  fprintf(stderr, "Options::CDO_Reset_History   = %d\n", Options::CDO_Reset_History);
  fprintf(stderr, "CDO_File_Suffix     = %s\n", Cdo::File_Suffix);
  fprintf(stderr, "CdoDefault::FileType  = %d\n", CdoDefault::FileType);
  fprintf(stderr, "CdoDefault::DataType  = %d\n", CdoDefault::DataType);
  fprintf(stderr, "CdoDefault::Byteorder = %d\n", CdoDefault::Byteorder);
  fprintf(stderr, "CdoDefault::TableID   = %d\n", CdoDefault::TableID);
  fprintf(stderr, "\n");

  const char *envstr;
  envstr = getenv("HOSTTYPE");
  if (envstr) fprintf(stderr, "HOSTTYPE            = %s\n", envstr);
  envstr = getenv("VENDOR");
  if (envstr) fprintf(stderr, "VENDOR              = %s\n", envstr);
  envstr = getenv("OSTYPE");
  if (envstr) fprintf(stderr, "OSTYPE              = %s\n", envstr);
  envstr = getenv("MACHTYPE");
  if (envstr) fprintf(stderr, "MACHTYPE            = %s\n", envstr);
  fprintf(stderr, "\n");

#if defined(_ARCH_PWR6)
  fprintf(stderr, "Predefined: _ARCH_PWR6\n");
#elif defined(_ARCH_PWR7)
  fprintf(stderr, "Predefined: _ARCH_PWR7\n");
#endif

#if defined(__AVX2__)
  fprintf(stderr, "Predefined: __AVX2__\n");
#elif defined(__AVX__)
  fprintf(stderr, "Predefined: __AVX__\n");
#elif defined(__SSE4_2__)
  fprintf(stderr, "Predefined: __SSE4_2__\n");
#elif defined(__SSE4_1__)
  fprintf(stderr, "Predefined: __SSE4_1__\n");
#elif defined(__SSE3__)
  fprintf(stderr, "Predefined: __SSE3__\n");
#elif defined(__SSE2__)
  fprintf(stderr, "Predefined: __SSE2__\n");
#endif
  fprintf(stderr, "\n");

  fprintf(stderr, "sizeof(size_t)      = %zu\n", sizeof(size_t));
  fprintf(stderr, "mem alignment       = %d\n\n", getMemAlignment());

#if defined(HAVE_MMAP)
  fprintf(stderr, "HAVE_MMAP\n");
#endif
#if defined(HAVE_MEMORY_H)
  fprintf(stderr, "HAVE_MEMORY_H\n");
#endif
  fprintf(stderr, "\n");

#if defined(_OPENACC)
  fprintf(stderr, "OPENACC VERSION     = %d\n", _OPENACC);
#endif
/* OPENMP 3:  201107 */
/* OPENMP 4:  201307 gcc 4.9 */
#ifdef _OPENMP
  fprintf(stderr, "OPENMP VERSION      = %d\n", _OPENMP);
#endif
  fprintf(stderr, "__cplusplus         = %ld\n", __cplusplus);
#if defined(__GNUC__)
  fprintf(stderr, "GNUC VERSION        = %d\n", __GNUC__);
#endif
#if defined(__GNUC_MINOR__)
  fprintf(stderr, "GNUC MINOR          = %d\n", __GNUC_MINOR__);
#endif
#if defined(__ICC)
  fprintf(stderr, "ICC VERSION         = %d\n", __ICC);
#endif
#if defined(__STDC__)
  fprintf(stderr, "STD ANSI C          = %d\n", __STDC__);
#endif
#if defined(__STD_VERSION__)
  fprintf(stderr, "STD VERSION         = %ld\n", __STD_VERSION__);
#endif
#if defined(__STDC_VERSION__)
  fprintf(stderr, "STDC VERSION        = %ld\n", __STDC_VERSION__);
#endif
#if defined(__STD_HOSTED__)
  fprintf(stderr, "STD HOSTED          = %d\n", __STD_HOSTED__);
#endif
#if defined(FLT_EVAL_METHOD)
  fprintf(stderr, "FLT_EVAL_METHOD     = %d\n", FLT_EVAL_METHOD);
#endif
#if defined(FP_FAST_FMA)
  fprintf(stderr, "FP_FAST_FMA         = defined\n");
#endif
#if defined(__FAST_MATH__)
  fprintf(stderr, "__FAST_MATH__       = defined\n");
#endif
  fprintf(stderr, "\n");

#if defined(_SC_VERSION)
  fprintf(stderr, "POSIX.1 VERSION     = %ld\n", sysconf(_SC_VERSION));
#endif
#if defined(_SC_ARG_MAX)
  fprintf(stderr, "POSIX.1 ARG_MAX     = %ld\n", sysconf(_SC_ARG_MAX));
#endif
#if defined(_SC_CHILD_MAX)
  fprintf(stderr, "POSIX.1 CHILD_MAX   = %ld\n", sysconf(_SC_CHILD_MAX));
#endif
#if defined(_SC_STREAM_MAX)
  fprintf(stderr, "POSIX.1 STREAM_MAX  = %ld\n", sysconf(_SC_STREAM_MAX));
#endif
#if defined(_SC_OPEN_MAX)
  fprintf(stderr, "POSIX.1 OPEN_MAX    = %ld\n", sysconf(_SC_OPEN_MAX));
#endif
#if defined(_SC_PAGESIZE)
  fprintf(stderr, "POSIX.1 PAGESIZE    = %ld\n", sysconf(_SC_PAGESIZE));
#endif

  fprintf(stderr, "\n");

#if defined(HAVE_GETRLIMIT)
#if defined(RLIMIT_FSIZE)
  PRINT_RLIMIT(RLIMIT_FSIZE);
#endif
#if defined(RLIMIT_NOFILE)
  PRINT_RLIMIT(RLIMIT_NOFILE);
#endif
#if defined(RLIMIT_STACK)
  PRINT_RLIMIT(RLIMIT_STACK);
#endif
#if defined(RLIMIT_RSS)
  PRINT_RLIMIT(RLIMIT_RSS);
#endif
#endif
  fprintf(stderr, "\n");
}

static void
check_stacksize()
{
#ifdef HAVE_GETRLIMIT
#ifdef RLIMIT_STACK
  {
    struct rlimit rlim;
    int status = getrlimit(RLIMIT_STACK, &rlim);
    if (status == 0)
      {
#define MIN_STACK_SIZE 67108864L  // 64MB
        RLIM_T min_stack_size = MIN_STACK_SIZE;
        if (min_stack_size > rlim.rlim_max) min_stack_size = rlim.rlim_max;
        if (rlim.rlim_cur < min_stack_size)
          {
            rlim.rlim_cur = min_stack_size;

            status = setrlimit(RLIMIT_STACK, &rlim);
            if (Debug)
              {
                if (status == 0)
                  {
                    fprintf(stderr, "Set stack size to %ld\n", (long) min_stack_size);
                    PRINT_RLIMIT(RLIMIT_STACK);
                  }
                else
                  fprintf(stderr, "Set stack size to %ld failed!\n", (long) min_stack_size);
                fprintf(stderr, "\n");
              }
          }
      }
  }
#endif
#endif
}

static void
cdo_set_options()
{
  if (Debug)
    {
      fprintf(stderr, "Options::CMOR_Mode       = %d\n", Options::CMOR_Mode);
      fprintf(stderr, "CDO_netcdf_hdr_pad  = %d\n", CDO_netcdf_hdr_pad);
      fprintf(stderr, "\n");
    }

  if (Options::CMOR_Mode) cdiDefGlobal("CMOR_MODE", Options::CMOR_Mode);
  if (Options::CDO_Reduce_Dim) cdiDefGlobal("REDUCE_DIM", Options::CDO_Reduce_Dim);
  if (CDO_netcdf_hdr_pad > 0) cdiDefGlobal("NETCDF_HDR_PAD", CDO_netcdf_hdr_pad);
}

static long
str_to_int(const char *intstring)
{
  long intval = -1;

  if (intstring)
    {
      long fact = 1;
      const int len = (int) strlen(intstring);
      for (int loop = 0; loop < len; loop++)
        {
          if (!isdigit((int) intstring[loop]))
            {
              switch (tolower((int) intstring[loop]))
                {
                case 'k': fact = 1024; break;
                case 'm': fact = 1048576; break;
                case 'g': fact = 1073741824; break;
                default: fact = 0; break;
                }
              break;
            }
        }

      if (fact) intval = fact * atol(intstring);
    }

  return intval;
}

static int
parse_options_long(int argc, char *argv[])
{
  int lasync_worker;
  int lnetcdf_hdr_pad;
  int luse_fftw;
  int leccodes;
  int lcellsearchmethod;
  int lpointsearchmethod;
  int lgridsearchradius;
  int lremap_genweights;
  int lprecision;
  int lpercentile;
  int lprintoperatorsno = 0;
  int lprintoperators = 0;
  int lpedantic = 0;
  int lenableexcept;
  int ltimestat_date;
  int ltimestat_bounds;
  int lsortname;
  int lsortparam;
  int ldebLevel;
  int lscmode;
  int lconfig;
  int verbose;

  char *token;
  // clang-format off
  const struct cdo_option opt_long[] =
    {
      { "worker"                     , required_argument , &lasync_worker               , 1             },
      { "precision"                  , required_argument , &lprecision                  , 1             },
      { "percentile"                 , required_argument , &lpercentile                 , 1             },
      { "netcdf_hdr_pad"             , required_argument , &lnetcdf_hdr_pad             , 1             },
      { "header_pad"                 , required_argument , &lnetcdf_hdr_pad             , 1             },
      { "hdr_pad"                    , required_argument , &lnetcdf_hdr_pad             , 1             },
      { "use_fftw"                   , required_argument , &luse_fftw                   , 1             },
      { "cellsearchmethod"           , required_argument , &lcellsearchmethod           , 1             },
      { "config"                     , required_argument , &lconfig                     , 1             },
      { "pointsearchmethod"          , required_argument , &lpointsearchmethod          , 1             },
      { "gridsearchradius"           , required_argument , &lgridsearchradius           , 1             },
      { "remap_genweights"           , required_argument , &lremap_genweights           , 1             },
      { "enableexcept"               , required_argument , &lenableexcept               , 1             },
      { "timestat_date"              , required_argument , &ltimestat_date              , 1             },
      { "timestat_bounds"            , no_argument       , &ltimestat_bounds            , 1             },
      { "cmor"                       , no_argument       , &Options::CMOR_Mode          , 1             },
      { "reduce_dim"                 , no_argument       , &Options::CDO_Reduce_Dim     , 1             },
      { "float"                      , no_argument       , &Options::CDO_Memtype        , MEMTYPE_FLOAT },
      { "rusage"                     , no_argument       , &CDO_Rusage                  , 1             },
      { "operators_no_output"        , no_argument       , &lprintoperatorsno           , 1             },
      { "operators"                  , no_argument       , &lprintoperators             , 1             },
      { "pedantic"                   , no_argument       , &lpedantic                   , 1             },
      { "no_warnings"                , no_argument       , &verbose                    , 0             },
      { "no_color"                   , no_argument       , nullptr                      , 'C'           },
      { "eccodes"                    , no_argument       , &leccodes                    , 1             },
      { "format"                     , required_argument , nullptr                      , 'f'           },
      { "help"                       , no_argument       , nullptr                      , 'h'           },
      { "history"                    , no_argument       , &Options::CDO_Append_History , 0             },
      { "no_history"                 , no_argument       , &Options::CDO_Append_History , 0             },
      { "regular"                    , no_argument       , nullptr                      , 'R'           },
      { "silent"                     , no_argument       , nullptr                      , 's'           },
      { "sort"                       , no_argument       , nullptr                      , 'Q'           },
      { "sortname"                   , no_argument       , &lsortname                   , 1             },
      { "sortparam"                  , no_argument       , &lsortparam                  , 1             },
      { "table"                      , required_argument , nullptr                      , 't'           },
      { "verbose"                    , no_argument       , nullptr                      , 'v'           },
      { "version"                    , no_argument       , nullptr                      , 'V'           },
      { "Dkext"                      , required_argument , &ldebLevel                   , 1             },
      { "outputGribDataScanningMode" , required_argument , &lscmode                     , 1             },
      { "seperateDebugFromLog"       , required_argument , nullptr                      , 2             },
      { nullptr                      , 0                 , nullptr                      , 0             }
    } ;
  // clang-format on

  CDO_opterr = 1;

  while (1)
    {
      // IMPORTANT: BY EVERY OPTION that takes arguments you MUST set its trigger variable to ZERO;
      // otherwise the parameters of other options get wrongly assigned.
      lasync_worker = 0;
      leccodes = 0;
      lprecision = 0;
      lpercentile = 0;
      lnetcdf_hdr_pad = 0;
      luse_fftw = 0;
      lcellsearchmethod = 0;
      lconfig = 0;
      lpointsearchmethod = 0;
      lgridsearchradius = 0;
      lremap_genweights = 0;
      lenableexcept = 0;
      ltimestat_date = 0;
      ltimestat_bounds = 0;
      lsortname = 0;
      lsortparam = 0;
      ldebLevel = 0;
      lscmode = 0;

      int c = cdo_getopt_long(argc, argv, "f:b:e:P:g:i:k:l:m:n:t:D:z:aCcdhLMOpQRrsSTuVvWXZ", opt_long, nullptr);
      if (c == -1) break;

      switch (c)
        {
        case '?':
          // cdo_usage();
          // fprintf(stderr, "Illegal option!\n");
          return -1;
        // break;
        case ':':
          // cdo_usage();
          // fprintf(stderr, "Option requires an argument!\n");
          return -1;
        // break;
        case 0:
          if (lnetcdf_hdr_pad)
            {
              const int netcdf_hdr_pad = str_to_int(CDO_optarg);
              if (netcdf_hdr_pad >= 0) CDO_netcdf_hdr_pad = netcdf_hdr_pad;
            }
          else if (lprecision)
            {
              cdoSetDigits(CDO_optarg);
            }
          else if (lpercentile)
            {
              percentile_set_method(CDO_optarg);
            }
          else if (lasync_worker)
            {
                Options::numStreamWorker = parameter2int(CDO_optarg);
            }
          else if (leccodes)
            {
              cdiDefGlobal("ECCODES_GRIB1", true);
            }
          else if (lenableexcept)
            {
              int except = -1;
              // clang-format off
              if      (cstrIsEqual(CDO_optarg, "DIVBYZERO")) except = FE_DIVBYZERO;
              else if (cstrIsEqual(CDO_optarg, "INEXACT")) except = FE_INEXACT;
              else if (cstrIsEqual(CDO_optarg, "INVALID")) except = FE_INVALID;
              else if (cstrIsEqual(CDO_optarg, "OVERFLOW")) except = FE_OVERFLOW;
              else if (cstrIsEqual(CDO_optarg, "UNDERFLOW")) except = FE_UNDERFLOW;
              else if (cstrIsEqual(CDO_optarg, "ALL_EXCEPT")) except = FE_ALL_EXCEPT;
              // clang-format on
              if (except < 0) cdoAbort("option --%s: unsupported argument: %s", "enableexcept", CDO_optarg);
              cdo_feenableexcept(except);
              if (signal(SIGFPE, cdoSignalHandler) == SIG_ERR) cdoWarning("can't catch SIGFPE!");
            }
          else if (ltimestat_date)
            {
              setTimestatDate(CDO_optarg);
            }
          else if (ltimestat_bounds)
            {
              extern bool CDO_Timestat_Bounds;
              CDO_Timestat_Bounds = true;
            }
          else if (luse_fftw)
            {
              const int intarg = parameter2int(CDO_optarg);
              if (intarg != 0 && intarg != 1) cdoAbort("Unsupported value for option --use_fftw=%d [range: 0-1]", intarg);
              Options::Use_FFTW = intarg;
            }
          else if (lcellsearchmethod)
            {
              setCellSearchMethod(CDO_optarg);
            }
          else if (lconfig)
            {
              cdoConfig(CDO_optarg);
            }
          else if (lpointsearchmethod)
            {
              setPointSearchMethod(CDO_optarg);
            }
          else if (lgridsearchradius)
            {
              extern double pointSearchRadius;
              const double fval = radius_str_to_deg(CDO_optarg);
              if (fval < 0 || fval > 180) cdoAbort("%s=%g out of bounds (0-180 deg)!", "gridsearchradius", fval);
              pointSearchRadius = fval;
            }
          else if (lremap_genweights)
            {
              const int intarg = parameter2int(CDO_optarg);
              if (intarg != 0 && intarg != 1) cdoAbort("Unsupported value for option --remap_genweights %d [0/1]", intarg);
              Options::REMAP_genweights = intarg;
            }
          else if (lsortname)
            {
              cdiDefGlobal("SORTNAME", true);
            }
          else if (lsortparam)
            {
              cdiDefGlobal("SORTPARAM", true);
            }
#ifdef HIRLAM_EXTENSIONS
          else if (ldebLevel)
            {
              const int newDebLevelVal = parameter2int(CDO_optarg);
              if (newDebLevelVal > 0)
                {
                  extern int cdiDebugExt;
                  cdoDebugExt = newDebLevelVal;
                  cdiDebugExt = newDebLevelVal;
                }
            }
          else if (lscmode)
            {
              const int scanningModeValue = parameter2int(CDO_optarg);
              if (cdoDebugExt) printf("scanningModeValue=%d\n", scanningModeValue);

              if ((scanningModeValue == 0) || (scanningModeValue == 64) || (scanningModeValue == 96))
                {
                  streamGrbDefDataScanningMode(scanningModeValue);  // -1: not used; allowed modes: <0,
                                                                    // 64, 96>; Default is 64
                }
              else
                {
                  cdoAbort("Warning: %d not in allowed modes: <0, 64, 96>; Using default: 64\n", scanningModeValue);
                  streamGrbDefDataScanningMode(64);
                }
            }
#endif
          break;
        case 'a': CdoDefault::TaxisType = TAXIS_ABSOLUTE; break;
        case 'b': setDefaultDataType(CDO_optarg); break;
        case 'C': CDO_Color = false; break;
        case 'c': Options::CheckDatarange = true; break;
        case 'd':
          Debug = 1;
          DebugLevel = 1;
          break;
        case 'D':
          Debug = 1;
          token = std::strtok((char *) CDO_optarg, ",");
          while (token != NULL)
            {
              DebugLevel = DebugLevel | parameter2int(token);
              token = std::strtok(NULL, ",");
            }
          break;
        case 'f': setDefaultFileType(CDO_optarg); break;
        case 'g': cdo_set_grids(CDO_optarg); break;
        case 'h': Help = 1; break;
        case 'i': defineInstitution(CDO_optarg); break;
        case 'k': defineChunktype(CDO_optarg); break;
        case 'L': Threading::cdoLockIO = true; break;
        case 'l': defineZaxis(CDO_optarg); break;
        case 'm': cdiDefMissval(atof(CDO_optarg)); break;
        case 'M': cdiDefGlobal("HAVE_MISSVAL", true); break;
        case 'n': defineVarnames(CDO_optarg); break;
        case 'O': Options::cdoOverwriteMode = true; break;
        case 'P': numThreads = parameter2int(CDO_optarg); break;
        case 'p':
          Options::CDO_Parallel_Read = true;
          CDO_task = true;
          break;
        case 'Q': cdiDefGlobal("SORTNAME", true); break;
        case 'R':
          Options::cdoRegulargrid = true;
          cdiDefGlobal("REGULARGRID", true);
          break;
        case 'r': CdoDefault::TaxisType = TAXIS_RELATIVE; break;
        case 'S': Options::cdoDiag = true; break;
        case 's':
          Options::silentMode = true;
          MpMO::enableSilentMode(Options::silentMode);
          Progress::silentMode = true;
          break;
        case 'T': Options::Timer = true; break;
        case 't': CdoDefault::TableID = defineTable(CDO_optarg); break;
        case 'u': Options::cdoInteractive = true; break;
        case 'V': Version = 1; break;
        case 'v':
          Options::cdoVerbose = true;
          MpMO::enableVerbose(true);
          gridEnableVerbose(Options::cdoVerbose);
          break;
        case 'W': /* disable Warning messages */ MpMO::enableWarnings(false); break;
        case 'X': /* multi threaded I/O */ Options::cdoParIO = true; break;
        case 'Z': Options::cdoCompress = true; break;
        case 'z': defineCompress(CDO_optarg); break;
        case 2: break;
        }
    }

  if (lpedantic)
    {
      MpMO::enablePedantic(true);
    }

  if (lprintoperators || lprintoperatorsno)
    {
      set_text_color(stderr, GREEN);
      const bool print_no_output = lprintoperatorsno > 0;
      operatorPrintList(print_no_output);
      // operatorPrintAll();
      reset_text_color(stderr);
      return 1;
    }

  return 0;
}

static void
cdo_rusage(void)
{
#if defined(HAVE_SYS_RESOURCE_H) && defined(RUSAGE_SELF)
  struct rusage ru;
  const int status = getrusage(RUSAGE_SELF, &ru);
  if (status == 0)
    {
      const double ut = ru.ru_utime.tv_sec + 0.000001 * ru.ru_utime.tv_usec;
      const double st = ru.ru_stime.tv_sec + 0.000001 * ru.ru_stime.tv_usec;

      fprintf(stderr, "  User time:     %.3f seconds\n", ut);
      fprintf(stderr, "  System time:   %.3f seconds\n", st);
      fprintf(stderr, "  Total time:    %.3f seconds\n", ut + st);
      fprintf(stderr, "  Memory usage:  %.2f MBytes\n", ru.ru_maxrss / (1024. * 1024.));
      fprintf(stderr, "  Page reclaims: %5ld page%s\n", ru.ru_minflt, ADD_PLURAL(ru.ru_minflt));
      fprintf(stderr, "  Page faults:   %5ld page%s\n", ru.ru_majflt, ADD_PLURAL(ru.ru_majflt));
      fprintf(stderr, "  Swaps:         %5ld\n", ru.ru_nswap);
      fprintf(stderr, "  Disk read:     %5ld block%s\n", ru.ru_inblock, ADD_PLURAL(ru.ru_inblock));
      fprintf(stderr, "  Disk Write:    %5ld block%s\n", ru.ru_oublock, ADD_PLURAL(ru.ru_oublock));
    }
#endif
}

/* clang-format off */
/**
 * Initializes all hardcoded aliases
 */
static void
init_aliases()
{
  add_alias("afterburner"     , "after");
  add_alias("anomaly"         , "ymonsub");
  add_alias("deltap_fl"       , "deltap");
  add_alias("diffv"           , "diffn");
  add_alias("covar0"          , "timcovar");
  add_alias("covar0r"         , "fldcovar");
  add_alias("geopotheight"    , "gheight");
  add_alias("globavg"         , "fldavg");
  add_alias("import_grads"    , "import_binary");
  add_alias("infos"           , "sinfo");
  add_alias("infov"           , "infon");
  add_alias("intgrid"         , "intgridbil");
  add_alias("log"             , "ln");
  add_alias("lmean"           , "ymonmean");
  add_alias("lmmean"          , "ymonmean");
  add_alias("lmavg"           , "ymonavg");
  add_alias("lmstd"           , "ymonstd");
  add_alias("lsmean"          , "yseasmean");
  add_alias("chvar"           , "chname");
  add_alias("nvar"            , "npar");
  add_alias("outputkey"       , "outputtab");
  add_alias("vardes"          , "codetab");
  add_alias("pardes"          , "codetab");
  add_alias("selvar"          , "selname");
  add_alias("delvar"          , "delname");
  add_alias("selindex"        , "selgridcell");
  add_alias("showvar"         , "showname");
  add_alias("selgridname"     , "selgrid");
  add_alias("setvar"          , "setname");
  add_alias("setpartabv"      , "setpartabn");
  add_alias("setpartab"       , "setcodetab");
  add_alias("sinfov"          , "sinfon");
  add_alias("sortvar"         , "sortname");
  add_alias("splitvar"        , "splitname");
  add_alias("sort"            , "timsort");
  add_alias("eca_r1mm"        , "eca_rr1");
  add_alias("fpressure"       , "pressure_fl");
  add_alias("hpressure"       , "pressure_hl");
  add_alias("outputcenter"    , "gmtxyz");
  add_alias("outputbounds"    , "gmtcells");
  add_alias("selseas"         , "selseason");
  add_alias("selmon"          , "selmonth");
  add_alias("remapycon"       , "remapcon");
  add_alias("genycon"         , "gencon");
  add_alias("for"             , "seq");
}

// stream in  -1 means: unlimited number of input streams
// stream out -1 means: usage of obase
/***
 * Initializes all hardcoded modules.
 */
void init_modules()
{
  //                           function        help function      operator names          mode number     num streams
  //                                                                                             type       in  out
  add_module("Adisit"        ,module_Adisit);
  add_module("Afterburner"   ,module_Afterburner);
  add_module("Arith"         ,module_Arith);
  add_module("Arithc"        ,module_Arithc);
  add_module("Arithdays"     ,module_Arithdays);
  add_module("Arithlat"      ,module_Arithlat);
  add_module("Cat"           ,module_Cat);
  add_module("CDItest"       ,module_CDItest);
  add_module("CDIread"       ,module_CDIread);
  add_module("CDIwrite"      ,module_CDIwrite);
  add_module("Change"        ,module_Change);
  add_module("Change_e5slm"  ,module_Change_e5slm);
  add_module("Cloudlayer"    ,module_Cloudlayer);
  add_module("CMOR"          ,module_CMOR);
  add_module("CMOR_lite"     ,module_CMOR_lite);
  add_module("CMOR_table"    ,module_CMOR_table);
  add_module("Collgrid"      ,module_Collgrid);
  add_module("Command"       ,module_Command);
  add_module("Comp"          ,module_Comp);
  add_module("Compc"         ,module_Compc);
  add_module("Complextorect" ,module_Complextorect);
  add_module("Cond"          ,module_Cond);
  add_module("Cond2"         ,module_Cond2);
  add_module("Condc"         ,module_Condc);
  add_module("Consecstat"    ,module_Consecstat);
  add_module("Copy"          ,module_Copy);
  add_module("Deltat"        ,module_Deltat);
  add_module("Deltime"       ,module_Deltime);
  add_module("Derivepar"     ,module_Derivepar);
  add_module("Detrend"       ,module_Detrend);
  add_module("Diff"          ,module_Diff);
  add_module("Distgrid"      ,module_Distgrid);
  add_module("Duplicate"     ,module_Duplicate);
  add_module("Echam5ini"     ,module_Echam5ini);
  add_module("Enlarge"       ,module_Enlarge);
  add_module("Enlargegrid"   ,module_Enlargegrid);
  add_module("Ensstat"       ,module_Ensstat);
  add_module("Ensstat3"      ,module_Ensstat3);
  add_module("Ensval"        ,module_Ensval);
  add_module("Eofcoeff"      ,module_Eofcoeff);
  add_module("Eofcoeff3d"    ,module_Eofcoeff3d);
  add_module("EOFs"          ,module_EOFs);
  add_module("EOF3d"         ,module_EOF3d);
  add_module("EstFreq"       ,module_EstFreq);
  add_module("Expr"          ,module_Expr);
  add_module("FC"            ,module_FC);
  add_module("Filedes"       ,module_Filedes);
  add_module("Fillmiss"      ,module_Fillmiss);
  add_module("Filter"        ,module_Filter);
  add_module("Fldrms"        ,module_Fldrms);
  add_module("Fldstat"       ,module_Fldstat);
  add_module("Fldstatcor"    ,module_Fldstatcor);
  add_module("Fldstatvar"    ,module_Fldstatvar);
  add_module("Fourier"       ,module_Fourier);
  add_module("Gengrid"       ,module_Gengrid);
  add_module("Gradsdes"      ,module_Gradsdes);
  add_module("Gridboxstat"   ,module_Gridboxstat);
  add_module("Gridcell"      ,module_Gridcell);
  add_module("Gridsearch"    ,module_Gridsearch);
  add_module("Harmonic"      ,module_Harmonic);
  add_module("Histogram"     ,module_Histogram);
  add_module("Importamsr"    ,module_Importamsr);
  add_module("Importbinary"  ,module_Importbinary);
  add_module("Importcmsaf"   ,module_Importcmsaf);
  add_module("Importobs"     ,module_Importobs);
  add_module("Importfv3grid" ,module_Importfv3grid);
  add_module("Info"          ,module_Info);
  add_module("Input"         ,module_Input);
  add_module("Intgrid"       ,module_Intgrid);
  add_module("Intgridtraj"   ,module_Intgridtraj);
  add_module("Intlevel"      ,module_Intlevel);
  add_module("Intlevel3d"    ,module_Intlevel3d);
  add_module("Inttime"       ,module_Inttime);
  add_module("Intntime"      ,module_Intntime);
  add_module("Intyear"       ,module_Intyear);
  add_module("Invert"        ,module_Invert);
  add_module("Invertlev"     ,module_Invertlev);
  add_module("Isosurface"    ,module_Isosurface);
  add_module("Lic"           ,module_Lic);
  add_module("MapReduce"     ,module_MapReduce);
  add_module("Maskbox"       ,module_Maskbox);
  add_module("Maskregion"    ,module_Maskregion);
  add_module("Mastrfu"       ,module_Mastrfu);
  add_module("Math"          ,module_Math);
  add_module("Merge"         ,module_Merge);
  add_module("Mergetime"     ,module_Mergetime);
  add_module("Mergegrid"     ,module_Mergegrid);
  add_module("Merstat"       ,module_Merstat);
  add_module("Monarith"      ,module_Monarith);
  add_module("Mrotuv"        ,module_Mrotuv);
  add_module("Mrotuvb"       ,module_Mrotuvb);
  add_module("NCL_wind"      ,module_NCL_wind);
  add_module("Ninfo"         ,module_Ninfo);
  add_module("Nmldump"       ,module_Nmldump);
  add_module("Output"        ,module_Output);
  add_module("Outputtab"     ,module_Outputtab);
  add_module("Outputgmt"     ,module_Outputgmt);
  add_module("Pack"          ,module_Pack);
  add_module("Pardup"        ,module_Pardup);
  add_module("Pinfo"         ,module_Pinfo);
  add_module("Pressure"      ,module_Pressure);
  add_module("Recttocomplex" ,module_Recttocomplex);
  add_module("Regres"        ,module_Regres);
  add_module("Remap"         ,module_Remap);
  add_module("Remapbil"      ,module_Remapbil);
  add_module("Remapbic"      ,module_Remapbic);
  add_module("Remapnn"       ,module_Remapnn);
  add_module("Remapdis"      ,module_Remapdis);
  add_module("Remapcon"      ,module_Remapcon);
  add_module("Remapycon2"    ,module_Remapycon2);
  add_module("Remapscon"     ,module_Remapscon);
  add_module("Remapcon2"     ,module_Remapcon2);
  add_module("Remaplaf"      ,module_Remaplaf);
  add_module("Remapgrid"     ,module_Remapgrid);
  add_module("Remapweights"  ,module_Remapweights);
  add_module("Remapeta"      ,module_Remapeta);
  add_module("Replace"       ,module_Replace);
  add_module("Replacevalues" ,module_Replacevalues);
  add_module("Rhopot"        ,module_Rhopot);
  add_module("Rotuv"         ,module_Rotuv);
  add_module("Runpctl"       ,module_Runpctl);
  add_module("Runstat"       ,module_Runstat);
  add_module("Samplegridicon",module_Samplegridicon);
  add_module("Seascount"     ,module_Seascount);
  add_module("Seaspctl"      ,module_Seaspctl);
  add_module("Seasstat"      ,module_Seasstat);
  add_module("Selbox"        ,module_Selbox);
  add_module("Selgridcell"   ,module_Selgridcell);
  add_module("Select"        ,module_Select);
  add_module("Selvar"        ,module_Selvar);
  add_module("Selrec"        ,module_Selrec);
  add_module("Seloperator"   ,module_Seloperator);
  add_module("Seltime"       ,module_Seltime);
  add_module("Selyearidx"    ,module_Selyearidx);
  add_module("Set"           ,module_Set);
  add_module("Setattribute"  ,module_Setattribute);
  add_module("Setbox"        ,module_Setbox);
  add_module("Setgatt"       ,module_Setgatt);
  add_module("Setgrid"       ,module_Setgrid);
  add_module("Sethalo"       ,module_Sethalo);
  add_module("Setmiss"       ,module_Setmiss);
  add_module("Setmisstonn"   ,module_Setmisstonn);
  add_module("Setcodetab"    ,module_Setcodetab);
  add_module("Setpartab"     ,module_Setpartab);
  add_module("Setrcaname"    ,module_Setrcaname);
  add_module("Settime"       ,module_Settime);
  add_module("Setzaxis"      ,module_Setzaxis);
  add_module("Shiftxy"       ,module_Shiftxy);
  add_module("Showinfo"      ,module_Showinfo);
  add_module("Showattribute" ,module_Showattribute);
  add_module("Sinfo"         ,module_Sinfo);
  add_module("Smooth"        ,module_Smooth);
  add_module("Sort"          ,module_Sort);
  add_module("Sorttimestamp" ,module_Sorttimestamp);
  add_module("Specinfo"      ,module_Specinfo);
  add_module("Spectral"      ,module_Spectral);
  add_module("Specconv"      ,module_Specconv);
  add_module("Spectrum"      ,module_Spectrum);
  add_module("Split"         ,module_Split);
  add_module("Splitrec"      ,module_Splitrec);
  add_module("Splitsel"      ,module_Splitsel);
  add_module("Splittime"     ,module_Splittime);
  add_module("Splityear"     ,module_Splityear);
  add_module("Subtrend"      ,module_Subtrend);
  add_module("Tee"           ,module_Tee);
  add_module("Template1"     ,module_Template1);
  add_module("Template2"     ,module_Template2);
  add_module("Test"          ,module_Test);
  add_module("Test2"         ,module_Test2);
  add_module("Testdata"      ,module_Testdata);
  add_module("Tests"         ,module_Tests);
  add_module("Timcount"      ,module_Timcount);
  add_module("Yearcount"     ,module_Yearcount);
  add_module("Moncount"      ,module_Moncount);
  add_module("Daycount"      ,module_Daycount);
  add_module("Hourcount"     ,module_Hourcount);
  add_module("Timcumsum"     ,module_Timcumsum);
  add_module("Timpctl"       ,module_Timpctl);
  add_module("Yearpctl"      ,module_Yearpctl);
  add_module("Monpctl"       ,module_Monpctl);
  add_module("Daypctl"       ,module_Daypctl);
  add_module("Hourpctl"      ,module_Hourpctl);
  add_module("Timselpctl"    ,module_Timselpctl);
  add_module("Timsort"       ,module_Timsort);
  add_module("Timselstat"    ,module_Timselstat);
  add_module("XTimstat"      ,module_XTimstat);
  add_module("Timstat"       ,module_Timstat);
  add_module("Yearstat"      ,module_Yearstat);
  add_module("Monstat"       ,module_Monstat);
  add_module("Daystat"       ,module_Daystat);
  add_module("Hourstat"      ,module_Hourstat);
  add_module("Timcor"        ,module_Timcor);
  add_module("Timscorvar"    ,module_Timscorvar);
  add_module("Timstat3"      ,module_Timstat3);
  add_module("Tinfo"         ,module_Tinfo);
  add_module("Tocomplex"     ,module_Tocomplex);
  add_module("Transpose"     ,module_Transpose);
  add_module("Trend"         ,module_Trend);
  add_module("Tstepcount"    ,module_Tstepcount);
  add_module("Unpack"        ,module_Unpack);
  add_module("Vargen"        ,module_Vargen);
  add_module("Varrms"        ,module_Varrms);
  add_module("Varsstat"      ,module_Varsstat);
  add_module("Vertintml"     ,module_Vertintml);
  add_module("Vertintap"     ,module_Vertintap);
  add_module("Vertstat"      ,module_Vertstat);
  add_module("Vertcum"       ,module_Vertcum);
  add_module("Vertwind"      ,module_Vertwind);
  add_module("Verifygrid"    ,module_Verifygrid);
  add_module("Wind"          ,module_Wind);
  add_module("Wind2"         ,module_Wind2);
  add_module("Writegrid"     ,module_Writegrid);
  add_module("Writerandom"   ,module_Writerandom);
  add_module("Yearmonstat"   ,module_Yearmonstat);
  add_module("Ydayarith"     ,module_Ydayarith);
  add_module("Ydaypctl"      ,module_Ydaypctl);
  add_module("Ydaystat"      ,module_Ydaystat);
  add_module("Ydrunpctl"     ,module_Ydrunpctl);
  add_module("Ydrunstat"     ,module_Ydrunstat);
  add_module("Yhourarith"    ,module_Yhourarith);
  add_module("Yhourstat"     ,module_Yhourstat);
  add_module("Dhourstat"     ,module_Dhourstat);
  add_module("Ymonarith"     ,module_Ymonarith);
  add_module("Yseasarith"    ,module_Yseasarith);
  add_module("Ymonpctl"      ,module_Ymonpctl);
  add_module("Ymonstat"      ,module_Ymonstat);
  add_module("Yseaspctl"     ,module_Yseaspctl);
  add_module("Yseasstat"     ,module_Yseasstat);
  add_module("Zonstat"       ,module_Zonstat);
  add_module("EcaCfd"        ,module_EcaCfd);
  add_module("EcaCsu"        ,module_EcaCsu);
  add_module("EcaCwdi"       ,module_EcaCwdi);
  add_module("EcaCwfi"       ,module_EcaCwfi);
  add_module("EcaEtr"        ,module_EcaEtr);
  add_module("EcaFd"         ,module_EcaFd);
  add_module("EcaGsl"        ,module_EcaGsl);
  add_module("EcaHd"         ,module_EcaHd);
  add_module("EcaHwdi"       ,module_EcaHwdi);
  add_module("EcaHwfi"       ,module_EcaHwfi);
  add_module("EcaId"         ,module_EcaId);
  add_module("EcaSu"         ,module_EcaSu);
  add_module("EcaTr"         ,module_EcaTr);
  add_module("EcaTg10p"      ,module_EcaTg10p);
  add_module("EcaTg90p"      ,module_EcaTg90p);
  add_module("EcaTn10p"      ,module_EcaTn10p);
  add_module("EcaTn90p"      ,module_EcaTn90p);
  add_module("EcaTx10p"      ,module_EcaTx10p);
  add_module("EcaTx90p"      ,module_EcaTx90p);
  add_module("EcaCdd"        ,module_EcaCdd);
  add_module("EcaCwd"        ,module_EcaCwd);
  add_module("EcaRr1"        ,module_EcaRr1);
  add_module("EcaPd"         ,module_EcaPd);
  add_module("EcaR75p"       ,module_EcaR75p);
  add_module("EcaR75ptot"    ,module_EcaR75ptot);
  add_module("EcaR90p"       ,module_EcaR90p);
  add_module("EcaR90ptot"    ,module_EcaR90ptot);
  add_module("EcaR95p"       ,module_EcaR95p);
  add_module("EcaR95ptot"    ,module_EcaR95ptot);
  add_module("EcaR99p"       ,module_EcaR99p);
  add_module("EcaR99ptot"    ,module_EcaR99ptot);
  add_module("EcaRx1day"     ,module_EcaRx1day);
  add_module("EcaRx5day"     ,module_EcaRx5day);
  add_module("EcaSdii"       ,module_EcaSdii);
  add_module("Fdns"          ,module_Fdns);
  add_module("Strwin"        ,module_Strwin);
  add_module("Strbre"        ,module_Strbre);
  add_module("Strgal"        ,module_Strgal);
  add_module("Hurr"          ,module_Hurr);
  // add_module("Hi"         ,module_Hi);
  add_module("Wct"           ,module_Wct);
  add_module("Magplot"       ,module_Magplot);
  add_module("Magvector"     ,module_Magvector);
  add_module("Maggraph"      ,module_Maggraph);
  // HIRLAM_EXTENSIONS
  add_module("Samplegrid"    ,module_Samplegrid);
  add_module("Selmulti "     ,module_Selmulti);
  add_module("WindTrans"     ,module_WindTrans);

  init_aliases();
}
/* clang-format on */

const char *
getProgname(char *string)
{
#ifdef _WIN32
  /*  progname = strrchr(string, '\\'); */
  char *progname = " cdo";
#else
  char *progname = strrchr(string, '/');
#endif

  if (progname == nullptr)
    progname = string;
  else
    progname++;

  return progname;
}

int
main(int argc, char *argv[])
{
  int status = 0;
  bool lstop = false;

  CdoOut::setExitFunction(cdoExit);
  CdoOut::setContextFunction(processInqPrompt);
  Progress::setContextFunction(processInqPrompt);
  cdo_init_is_tty();

  memExitOnError();

  Options::CDO_Reduce_Dim = 0;

  /* mallopt(M_MMAP_MAX, 0); */

  setCommandLine(argc, argv);

  Cdo::progname = getProgname(argv[0]);

  get_env_vars();
  init_modules();

  status = parse_options_long(argc, argv);
  if (status != 0) return 1;

  cdo_set_options();
  if (Debug || Version) cdo_version();

  if (Debug)
    {
      fprintf(stderr, "stdin_is_tty:   %d\n", stdin_is_tty);
      fprintf(stderr, "stdout_is_tty:  %d\n", stdout_is_tty);
      fprintf(stderr, "stderr_is_tty:  %d\n", stderr_is_tty);
      print_system_info();
    }

  check_stacksize();

  if (Debug)
    {
      print_pthread_info();
    }

#ifdef _OPENMP
  if (numThreads <= 0) numThreads = 1;
  omp_set_num_threads(numThreads);

  if (Debug)
    {
      fprintf(stderr, "OMP num procs       = %d\n", omp_get_num_procs());
      fprintf(stderr, "OMP max threads     = %d\n", omp_get_max_threads());
      fprintf(stderr, "OMP num threads     = %d\n", omp_get_num_threads());
#if defined(HAVE_OPENMP3)
      fprintf(stderr, "OMP thread limit    = %d\n", omp_get_thread_limit());
      omp_sched_t kind;
      int modifer;
      omp_get_schedule(&kind, &modifer);
      fprintf(stderr, "OMP schedule        = %d (1:static; 2:dynamic; 3:guided; 4:auto)\n", (int) kind);
#endif
#ifdef HAVE_OPENMP4
      fprintf(stderr, "OMP proc bind       = %d (0:false; 1:true; 2:master; 3:close; 4:spread)\n", (int) omp_get_proc_bind());
#if !defined(__ICC)
      fprintf(stderr, "OMP num devices     = %d\n", omp_get_num_devices());
#endif
#endif
    }

  Threading::ompNumThreads = omp_get_max_threads();
  if (omp_get_max_threads() > omp_get_num_procs())
    fprintf(stderr, "Warning: Number of OMP threads is greater than number of Cores=%d!\n", omp_get_num_procs());
  if (Threading::ompNumThreads < numThreads)
    fprintf(stderr, "Warning: omp_get_max_threads() returns %d!\n", Threading::ompNumThreads);
  if (Options::cdoVerbose)
    {
      fprintf(stderr, " OpenMP:  num_procs=%d  max_threads=%d", omp_get_num_procs(), omp_get_max_threads());
#ifdef HAVE_OPENMP4
#if !defined(__ICC)
      fprintf(stderr, "  num_devices=%d", omp_get_num_devices());
#endif
#endif
      fprintf(stderr, "\n");
    }
#else
  if (numThreads > 0)
    {
      fprintf(stderr, "Option -P failed, OpenMP support not compiled in!\n");
      return -1;
    }
#endif

  std::vector<std::string> new_argv(&argv[CDO_optind], argv + argc);
  new_argv = expandWildCards(new_argv);

  ///*TEMP*/ // should not be needed when std::string is standard string
  std::vector<char *> new_cargv(new_argv.size());
  for (unsigned long i = 0; i < new_argv.size(); i++)
    {
      new_cargv[i] = strdup(new_argv[i].c_str());
    }
  // temprorary end

  if (CDO_optind >= argc)
    {
      if (!Version && !Help)
        {
          fprintf(stderr, "\nNo operator given!\n\n");
          cdo_usage();
          status = 1;
        }

      if (Help) cdo_usage();
      lstop = true;
    }

  if (lstop) return status;

  if (CdoDefault::TableID != CDI_UNDEFID) cdiDefTableID(CdoDefault::TableID);

  extern int (*proj_lonlat_to_lcc_func)();
  extern int (*proj_lcc_to_lonlat_func)();
  proj_lonlat_to_lcc_func = (int (*)()) proj_lonlat_to_lcc;
  proj_lcc_to_lonlat_func = (int (*)()) proj_lcc_to_lonlat;

  extern int (*proj_lonlat_to_stere_func)();
  extern int (*proj_stere_to_lonlat_func)();
  proj_lonlat_to_stere_func = (int (*)()) proj_lonlat_to_stere;
  proj_stere_to_lonlat_func = (int (*)()) proj_stere_to_lonlat;

  const char *operatorName = get_original(argv[CDO_optind]);

  if (Help)
    {
      cdoPrintHelp(operatorHelp(operatorName));
    }
  else
    {
      if (Debug)
        {
          if (DebugLevel == 0)
            {
              std::cout << "No debug level given please choose: " << std::endl;
              printDebugOptions();
              exit(EXIT_SUCCESS);
            }
          if (DebugLevel > 1) cdiDebug(DebugLevel);
          SetDebug(DebugLevel);
        }

      timer_total = timer_new("total");
      timer_read = timer_new("read");
      timer_write = timer_new("write");

      timer_start(timer_total);

#ifdef CUSTOM_MODULES
      load_custom_modules("custom_modules");
      close_library_handles();
#endif
      cdoRun(new_argv.size(), (const char **) &new_cargv[0]);

      timer_stop(timer_total);

      if (Options::Timer) timer_report();
    }

  if (Options::cdoVarnames)
    {
      if (Options::cdoNumVarnames) Free(Options::cdoVarnames[0]);
      Free(Options::cdoVarnames);
    }

  if (gridSearchDir) Free(gridSearchDir);

  if (CDO_Rusage) cdo_rusage();

  if (!status) status = Options::cdoExitStatus;

  return status;
}
