/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2020 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

/*
   This module contains the following operators:

      Cond2      ifthenelse      If then else
*/

#include <cdi.h>

#include "cdo_options.h"
#include "functs.h"
#include "process_int.h"
#include "cdo_vlist.h"
#include "cdo_fill.h"

void *
Cond2(void *process)
{
  enum
  {
    FILL_NONE,
    FILL_TS,
    FILL_REC
  };
  int filltype = FILL_NONE;
  int nrecs = 0;
  int nrecs2 = 0;
  int varID, levelID;
  double missval1 = -9.E33;
  size_t nmiss1 = 0;
  std::vector<std::vector<size_t>> varnmiss1;
  Varray2D<double> vardata1;

  cdoInitialize(process);

  cdoOperatorAdd("ifthenelse", 0, 0, nullptr);

  operatorCheckArgc(0);

  const auto streamID1 = cdoOpenRead(0);
  const auto streamID2 = cdoOpenRead(1);
  const auto streamID3 = cdoOpenRead(2);

  const auto vlistID1 = cdoStreamInqVlist(streamID1);
  const auto vlistID2 = cdoStreamInqVlist(streamID2);
  const auto vlistID3 = cdoStreamInqVlist(streamID3);
  const auto vlistID4 = vlistDuplicate(vlistID2);

  const auto taxisID2 = vlistInqTaxis(vlistID2);
  const auto taxisID4 = taxisDuplicate(taxisID2);
  vlistDefTaxis(vlistID4, taxisID4);

  auto ntsteps1 = vlistNtsteps(vlistID1);
  auto ntsteps2 = vlistNtsteps(vlistID2);
  if (ntsteps1 == 0) ntsteps1 = 1;
  if (ntsteps2 == 0) ntsteps2 = 1;

  if (vlistNrecs(vlistID1) == 1 && vlistNrecs(vlistID2) != 1)
    {
      filltype = FILL_REC;
      cdoPrint("Filling up stream1 >%s< by copying the first record.", cdoGetStreamName(0));
    }

  if (filltype == FILL_NONE) vlistCompare(vlistID1, vlistID2, CMP_DIM);

  vlistCompare(vlistID2, vlistID3, CMP_DIM);

  nospec(vlistID1);
  nospec(vlistID2);
  nospec(vlistID3);

  const auto streamID4 = cdoOpenWrite(3);
  cdoDefVlist(streamID4, vlistID4);

  VarList varList1, varList2;
  varListInit(varList1, vlistID1);
  varListInit(varList2, vlistID2);

  const auto gridsizemax = vlistGridsizeMax(vlistID1);

  if (filltype == FILL_REC && gridsizemax != gridInqSize(vlistGrid(vlistID1, 0)))
    cdoAbort("Stream1 >%s< has wrong gridsize!", cdoGetStreamName(0));

  Varray<double> array1(gridsizemax), array2(gridsizemax), array3(gridsizemax), array4(gridsizemax);

  if (Options::cdoVerbose)
    cdoPrint("Number of timesteps: file1 %d, file2 %d, file3 %d", ntsteps1, ntsteps2, vlistNtsteps(vlistID3));

  if (filltype == FILL_NONE)
    {
      if (ntsteps1 == 1 && ntsteps2 != 1)
        {
          filltype = FILL_TS;
          cdoPrint("Filling up stream1 >%s< by copying the first timestep.", cdoGetStreamName(0));

          cdoFillTs(vlistID1, vardata1, varnmiss1);
        }
    }

  int tsID = 0;
  while ((nrecs = cdoStreamInqTimestep(streamID2, tsID)))
    {
      nrecs = cdoStreamInqTimestep(streamID3, tsID);
      if (nrecs == 0) cdoAbort("Input streams have different number of timesteps!");

      if (tsID == 0 || filltype == FILL_NONE)
        {
          nrecs2 = cdoStreamInqTimestep(streamID1, tsID);
          if (nrecs2 == 0) cdoAbort("Input streams have different number of timesteps!");
        }

      taxisCopyTimestep(taxisID4, taxisID2);
      cdoDefTimestep(streamID4, tsID);

      for (int recID = 0; recID < nrecs; recID++)
        {
          size_t nmiss;
          cdoInqRecord(streamID2, &varID, &levelID);
          cdoReadRecord(streamID2, &array2[0], &nmiss);

          cdoInqRecord(streamID3, &varID, &levelID);
          cdoReadRecord(streamID3, &array3[0], &nmiss);

          if (tsID == 0 || filltype == FILL_NONE)
            {
              if (recID == 0 || filltype != FILL_REC)
                {
                  cdoInqRecord(streamID1, &varID, &levelID);
                  cdoReadRecord(streamID1, &array1[0], &nmiss1);
                }

              if (filltype == FILL_TS)
                {
                  const auto gridsize = varList1[varID].gridsize;
                  const auto offset = gridsize * levelID;
                  arrayCopy(gridsize, &array1[0], &vardata1[varID][offset]);
                  varnmiss1[varID][levelID] = nmiss1;
                }
            }
          else if (filltype == FILL_TS)
            {
              const auto gridsize = varList1[varID].gridsize;
              const auto offset = gridsize * levelID;
              arrayCopy(gridsize, &vardata1[varID][offset], &array1[0]);
              nmiss1 = varnmiss1[varID][levelID];
            }

          const auto gridsize = varList2[varID].gridsize;
          const auto missval2 = varList2[varID].missval;
          if (recID == 0 || filltype != FILL_REC) missval1 = varList1[varID].missval;

          if (nmiss1 > 0) cdo_check_missval(missval1, varList1[varID].name);

          for (size_t i = 0; i < gridsize; i++)
            array4[i] = DBL_IS_EQUAL(array1[i], missval1) ? missval2 : !DBL_IS_EQUAL(array1[i], 0.) ? array2[i] : array3[i];

          nmiss = varrayNumMV(gridsize, array4, missval2);
          cdoDefRecord(streamID4, varID, levelID);
          cdoWriteRecord(streamID4, array4.data(), nmiss);
        }

      tsID++;
    }

  cdoStreamClose(streamID4);
  cdoStreamClose(streamID3);
  cdoStreamClose(streamID2);
  cdoStreamClose(streamID1);

  cdoFinish();

  return nullptr;
}
