/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2020 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

/*
   This module contains the following operators:

      Vertint    ah2hl           Model geometric height level to height level interpolation
*/

#include <cdi.h>

#include "cdo_options.h"
#include "process_int.h"
#include "cdo_vlist.h"
#include "field_vinterp.h"
#include "stdnametable.h"
#include "util_string.h"
#include "const.h"
#include "cdo_zaxis.h"
#include "param_conversion.h"


static bool
is_height_axis(int zaxisID)
{
  bool isHeight = false;
  if (zaxisInqType(zaxisID) == ZAXIS_REFERENCE)
    {
      char units[CDI_MAX_NAME], stdname[CDI_MAX_NAME];
      int length = CDI_MAX_NAME;
      cdiInqKeyString(zaxisID, CDI_GLOBAL, CDI_KEY_UNITS, units, &length);
      length = CDI_MAX_NAME;
      cdiInqKeyString(zaxisID, CDI_GLOBAL, CDI_KEY_STDNAME, stdname, &length);
      if (cstrIsEqual(stdname, "height") && *units == 0) isHeight = true;
    }
  return isHeight;
}

static void
change_height_zaxis(int nhlev, int vlistID1, int vlistID2, int zaxisID2)
{
  const auto nzaxis = vlistNzaxis(vlistID1);
  for (int iz = 0; iz < nzaxis; ++iz)
    {
      const auto zaxisID = vlistZaxis(vlistID1, iz);
      const auto nlevel = zaxisInqSize(zaxisID);
      if ((nlevel == nhlev || nlevel == (nhlev+1)) && is_height_axis(zaxisID)) vlistChangeZaxisIndex(vlistID2, iz, zaxisID2);
    }
}

void *
Vertintgh(void *process)
{
  int nrecs;
  int varID, levelID;
  int nhlev = 0, nhlevf = 0, nhlevh = 0;
  int f_heightID = -1, h_heightID = -1;;
  char stdname[CDI_MAX_NAME];
  bool extrapolate = false;

  cdoInitialize(process);

  // clang-format off
  const auto GH2HL   = cdoOperatorAdd("gh2hl",   0, 0, "height levels in meter");
  const auto GH2HLX  = cdoOperatorAdd("gh2hlx",  0, 0, "height levels in meter");
  // clang-format on

  const auto operatorID = cdoOperatorID();

  if (operatorID == GH2HL)
    {
      const auto envstr = getenv("EXTRAPOLATE");
      if (envstr && isdigit((int) envstr[0]))
        {
          if (atoi(envstr) == 1) extrapolate = true;
          if (extrapolate) cdoPrint("Extrapolation of missing values enabled!");
        }
    }
  else if (GH2HLX)
    {
      extrapolate = true;
    }

  operatorInputArg(cdoOperatorEnter(operatorID));

  int zaxisID2 = CDI_UNDEFID;
  const auto &arg1 = cdoOperatorArgv(0);
  Varray<double> plev;
  if (operatorArgc() == 1 && !isdigit(arg1[0]))
    {
      auto zfp = fopen(arg1.c_str(), "r");
      if (zfp)
        {
          zaxisID2 = zaxisFromFile(zfp, arg1.c_str());
          fclose(zfp);
          if (zaxisID2 == CDI_UNDEFID) cdoAbort("Invalid zaxis description file %s!", arg1.c_str());
          const auto nlevels = zaxisInqSize(zaxisID2);
          plev.resize(nlevels);
          zaxisInqLevels(zaxisID2, plev.data());
        }
      else if (arg1 == "default")
        plev = { 10, 50, 100, 500, 1000, 5000, 10000, 15000, 20000, 25000, 30000 };
      else
        cdoAbort("Open failed on %s", arg1.c_str());
    }
  else
    {
      plev = cdoArgvToFlt(cdoGetOperArgv());
    }

  int nplev = plev.size();

  const auto streamID1 = cdoOpenRead(0);

  const auto vlistID1 = cdoStreamInqVlist(streamID1);
  const auto vlistID2 = vlistDuplicate(vlistID1);

  const auto taxisID1 = vlistInqTaxis(vlistID1);
  const auto taxisID2 = taxisDuplicate(taxisID1);
  vlistDefTaxis(vlistID2, taxisID2);

  const auto gridsize = vlist_check_gridsize(vlistID1);

  if (zaxisID2 == CDI_UNDEFID)
    {
      zaxisID2 = zaxisCreate(ZAXIS_HEIGHT, nplev);
      zaxisDefLevels(zaxisID2, plev.data());
    }

  VarList varList1, varList2;
  varListInit(varList1, vlistID1);
  varListSetUniqueMemtype(varList1);
  const auto memtype = varList1[0].memType;

  const auto nvars = vlistNvars(vlistID1);

  for (varID = 0; varID < nvars; varID++)
    {
      int length = CDI_MAX_NAME;
      cdiInqKeyString(vlistID1, varID, CDI_KEY_STDNAME, stdname, &length);
      cstrToLowerCase(stdname);

      // clang-format off
      if (cstrIsEqual(stdname, var_stdname(geometric_height_at_full_level_center))) f_heightID = varID;
      if (cstrIsEqual(stdname, var_stdname(geometric_height_at_half_level_center))) h_heightID = varID;
      // clang-format on
    }

  if (Options::cdoVerbose)
    {
      cdoPrint("Found:");
      // clang-format off
      if (-1 != f_heightID) cdoPrint("  %s -> %s", var_stdname(geometric_height_at_full_level_center), varList1[f_heightID].name);
      if (-1 != h_heightID) cdoPrint("  %s -> %s", var_stdname(geometric_height_at_half_level_center), varList1[h_heightID].name);
      // clang-format on
    }

  if (f_heightID == -1) cdoAbort("%s not found!", var_stdname(geometric_height_at_full_level_center));

  int zaxisIDh = -1;
  const auto nzaxis = vlistNzaxis(vlistID1);
  for (int i = 0; i < nzaxis; i++)
    {
      const auto zaxisID = vlistZaxis(vlistID1, i);
      if (zaxisID == varList1[f_heightID].zaxisID)
        {
          bool mono_level = true;
          const auto nlevels = zaxisInqSize(zaxisID);

          if (nlevels > 1 && is_height_axis(zaxisID))
            {
              Varray<double> level(nlevels);
              cdoZaxisInqLevels(zaxisID, level.data());
              int l;
              for (l = 0; l < nlevels; l++)
                {
                  if ((l + 1) != (int) (level[l] + 0.5)) break;
                }
              if (l == nlevels) mono_level = true;
            }

          if (nlevels > 1 && is_height_axis(zaxisID) && mono_level)
            {
              zaxisIDh = zaxisID;
              nhlev = nlevels;
              nhlevf = nhlev;
              nhlevh = nhlevf + 1;

              break;
            }
        }
    }

  change_height_zaxis(nhlev, vlistID1, vlistID2, zaxisID2);
  varListInit(varList2, vlistID2);
  varListSetMemtype(varList2, memtype);

  std::vector<bool> vars(nvars), varinterp(nvars);
  std::vector<std::vector<size_t>> varnmiss(nvars);
  Field3DVector vardata1(nvars), vardata2(nvars);

  const auto maxlev = nhlevh > nplev ? nhlevh : nplev;

  std::vector<size_t> pnmiss;
  if (!extrapolate) pnmiss.resize(nplev);

  // check levels
  if (zaxisIDh != -1)
    {
      const auto nlev = zaxisInqSize(zaxisIDh);
      if (nlev != nhlev) cdoAbort("Internal error, wrong number of height level!");
    }

  std::vector<int> vert_index;
  Field level0;
  Field3D full_level, half_level;
  if (zaxisIDh != -1 && gridsize > 0)
    {
      vert_index.resize(gridsize * nplev);

      CdoVar var3Df, var3Dh;
      var3Df.gridsize = gridsize;
      var3Df.nlevels = nhlevf;
      var3Df.memType = memtype;
      full_level.init(var3Df);

      var3Dh.gridsize = gridsize;
      var3Dh.nlevels = nhlevh;
      var3Dh.memType = memtype;
      half_level.init(var3Dh);
    }
  else
    cdoWarning("No 3D variable with generalized height level found!");

  for (varID = 0; varID < nvars; varID++)
    {
      const auto gridID = varList1[varID].gridID;
      const auto zaxisID = varList1[varID].zaxisID;
      const auto nlevels = varList1[varID].nlevels;

      if (gridInqType(gridID) == GRID_SPECTRAL) cdoAbort("Spectral data unsupported!");

      vardata1[varID].init(varList1[varID]);

      varinterp[varID]
          = (zaxisID == zaxisIDh || (is_height_axis(zaxisID) && zaxisIDh != -1 && (nlevels == nhlevh || nlevels == nhlevf)));

      if (varinterp[varID])
        {
          varnmiss[varID].resize(maxlev, 0);
          vardata2[varID].init(varList2[varID]);
        }
      else
        {
          if (is_height_axis(zaxisID) && zaxisIDh != -1 && nlevels > 1)
            cdoWarning("Parameter %d has wrong number of levels, skipped! (name=%s nlevel=%d)", varID + 1, varList1[varID].name, nlevels);

          varnmiss[varID].resize(nlevels);
        }
    }

  for (varID = 0; varID < nvars; ++varID)
    {
      if (varinterp[varID] && varList1[varID].timetype == TIME_CONSTANT) vlistDefVarTimetype(vlistID2, varID, TIME_VARYING);
    }

  const auto streamID2 = cdoOpenWrite(1);

  cdoDefVlist(streamID2, vlistID2);

  int tsID = 0;
  while ((nrecs = cdoStreamInqTimestep(streamID1, tsID)))
    {
      for (varID = 0; varID < nvars; ++varID)
        {
          vars[varID] = false;
          const auto nlevels = varList1[varID].nlevels;
          for (levelID = 0; levelID < nlevels; levelID++) varnmiss[varID][levelID] = 0;
        }

      taxisCopyTimestep(taxisID2, taxisID1);
      cdoDefTimestep(streamID2, tsID);

      for (int recID = 0; recID < nrecs; recID++)
        {
          cdoInqRecord(streamID1, &varID, &levelID);
          cdoReadRecord(streamID1, vardata1[varID], levelID, &varnmiss[varID][levelID]);
          vars[varID] = true;
        }

      for (varID = 0; varID < nvars; varID++)
        if (varinterp[varID]) vars[varID] = true;

      if (zaxisIDh != -1)
        {
          level0.init(varList1[f_heightID]);
          fieldCopy(vardata1[f_heightID], nhlevf - 1, level0);

          fieldCopy(vardata1[f_heightID], full_level);

          if (-1 != h_heightID)
            fieldCopy(vardata1[h_heightID], half_level);

          bool lreverse = true;
          genind(vert_index, plev, full_level, gridsize, lreverse);
          if (!extrapolate) genindmiss(vert_index, plev, gridsize, level0, pnmiss, lreverse);
        }

      for (varID = 0; varID < nvars; varID++)
        {
          if (vars[varID])
            {
              if (tsID > 0 && varList1[varID].timetype == TIME_CONSTANT) continue;

              if (varinterp[varID])
                {
                  const auto nlevels = varList1[varID].nlevels;
                  if (nlevels != nhlevf && nlevels != nhlevh)
                    cdoAbort("Number of generalized height level differ from full/half level (param=%s)!", varList1[varID].name);

                  for (levelID = 0; levelID < nlevels; levelID++)
                    {
                      if (varnmiss[varID][levelID]) cdoAbort("Missing values unsupported for this operator!");
                    }

                  vertical_interp_X(nlevels, full_level, half_level, vardata1[varID], vardata2[varID], vert_index, plev, gridsize);
 
                  if (!extrapolate) varrayCopy(nplev, pnmiss, varnmiss[varID]);
                }

              for (levelID = 0; levelID < varList2[varID].nlevels; levelID++)
                {
                  cdoDefRecord(streamID2, varID, levelID);
                  cdoWriteRecord(streamID2, varinterp[varID] ? vardata2[varID] : vardata1[varID], levelID, varnmiss[varID][levelID]);
                }
            }
        }

      tsID++;
    }

  cdoStreamClose(streamID2);
  cdoStreamClose(streamID1);

  cdoFinish();

  return nullptr;
}
