/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2021 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

/*
   This module contains the following operators:

      Monarith  monadd         Add monthly time series
      Monarith  monsub         Subtract monthly time series
      Monarith  monmul         Multiply monthly time series
      Monarith  mondiv         Divide monthly time series
*/

#include <cdi.h>

#include "cdo_options.h"
#include "process_int.h"
#include "cdo_vlist.h"

void *
Monarith(void *process)
{
  cdo_initialize(process);

  cdo_operator_add("monadd", FieldFunc_Add, 0, nullptr);
  cdo_operator_add("monsub", FieldFunc_Sub, 0, nullptr);
  cdo_operator_add("monmul", FieldFunc_Mul, 0, nullptr);
  cdo_operator_add("mondiv", FieldFunc_Div, 0, nullptr);

  const auto operatorID = cdo_operator_id();
  const auto operfunc = cdo_operator_f1(operatorID);

  operator_check_argc(0);

  const auto streamID1 = cdo_open_read(0);
  const auto streamID2 = cdo_open_read(1);

  const auto vlistID1 = cdo_stream_inq_vlist(streamID1);
  const auto vlistID2 = cdo_stream_inq_vlist(streamID2);
  const auto vlistID3 = vlistDuplicate(vlistID1);

  vlist_compare(vlistID1, vlistID2, CMP_ALL);

  VarList varList1;
  varListInit(varList1, vlistID1);

  const auto gridsizemax = vlistGridsizeMax(vlistID1);

  Field field1, field2;
  field1.resize(gridsizemax);
  field2.resize(gridsizemax);

  const auto taxisID1 = vlistInqTaxis(vlistID1);
  const auto taxisID2 = vlistInqTaxis(vlistID2);
  const auto taxisID3 = taxisDuplicate(taxisID1);
  vlistDefTaxis(vlistID3, taxisID3);

  const auto streamID3 = cdo_open_write(2);
  cdo_def_vlist(streamID3, vlistID3);

  FieldVector2D vars2;
  fields_from_vlist(vlistID2, vars2, FIELD_VEC);

  int yearmon2 = -1;
  int tsID = 0;
  int tsID2 = 0;
  while (true)
    {
      const auto nrecs = cdo_stream_inq_timestep(streamID1, tsID);
      if (nrecs == 0) break;

      auto vdate = taxisInqVdate(taxisID1);
      const int yearmon1 = vdate / 100;

      if (yearmon1 != yearmon2)
        {
          const int year1 = yearmon1 / 100;
          const int mon1 = yearmon1 - (yearmon1 / 100) * 100;
          if (Options::cdoVerbose) cdo_print("Process: Year = %4d  Month = %2d", year1, mon1);

          const auto nrecs2 = cdo_stream_inq_timestep(streamID2, tsID2);
          if (nrecs2 == 0) cdo_abort("Missing year=%4d mon=%2d in %s!", year1, mon1, cdo_get_stream_name(1));

          vdate = taxisInqVdate(taxisID2);
          yearmon2 = vdate / 100;

          if (yearmon1 != yearmon2)
            {
              const int year2 = yearmon2 / 100;
              const int mon2 = yearmon2 - (yearmon2 / 100) * 100;
              cdo_abort("Timestep %d in %s has wrong date! Current year=%4d mon=%2d, expected year=%4d mon=%2d", tsID2 + 1,
                        cdo_get_stream_name(1), year2, mon2, year1, mon1);
            }

          for (int recID = 0; recID < nrecs2; recID++)
            {
              size_t nmiss;
              int varID, levelID;
              cdo_inq_record(streamID2, &varID, &levelID);
              cdo_read_record(streamID2, vars2[varID][levelID].vec_d.data(), &nmiss);
              vars2[varID][levelID].nmiss = nmiss;
            }

          tsID2++;
        }

      cdo_taxis_copy_timestep(taxisID3, taxisID1);
      cdo_def_timestep(streamID3, tsID);

      for (int recID = 0; recID < nrecs; recID++)
        {
          int varID, levelID;
          cdo_inq_record(streamID1, &varID, &levelID);
          cdo_read_record(streamID1, field1.vec_d.data(), &field1.nmiss);
          field1.grid = varList1[varID].gridID;
          field1.missval = varList1[varID].missval;

          field2_function(field1, vars2[varID][levelID], operfunc);

          cdo_def_record(streamID3, varID, levelID);
          cdo_write_record(streamID3, field1.vec_d.data(), field1.nmiss);
        }

      tsID++;
    }

  cdo_stream_close(streamID3);
  cdo_stream_close(streamID2);
  cdo_stream_close(streamID1);

  cdo_finish();

  return nullptr;
}
