#ifndef CDO_STEPSTAT_H
#define CDO_STEPSTAT_H

#include "process_int.h"
#include "field.h"
#include "field_functions.h"

namespace cdo
{
class StepStatBase
{
public:
  int operfunc{};
  bool lminmax{ false };
  bool lminidx{ false };
  bool lmaxidx{ false };
  bool lrange{ false };
  bool lmean{ false };
  bool lmeanavg{ false };
  bool lstd{ false };
  bool lvarstd{ false };
  double divisor{};

  void
  init(int _operfunc)
  {
    operfunc = _operfunc;
    lminmax = (operfunc == FieldFunc_Min || operfunc == FieldFunc_Max);
    lminidx = (operfunc == FieldFunc_Minidx);
    lmaxidx = (operfunc == FieldFunc_Maxidx);
    lrange = (operfunc == FieldFunc_Range);
    lmean = (operfunc == FieldFunc_Mean);
    lmeanavg = (operfunc == FieldFunc_Mean || operfunc == FieldFunc_Avg);
    lstd = (operfunc == FieldFunc_Std || operfunc == FieldFunc_Std1);
    lvarstd = (lstd || operfunc == FieldFunc_Var || operfunc == FieldFunc_Var1);
    divisor = (operfunc == FieldFunc_Std1 || operfunc == FieldFunc_Var1);
  }

  void
  add_field_kernel(const Field &field, Field &samp, Field &var1, Field &var2, int numSets)
  {
    if (numSets == 0)
      {
        if (lminidx || lmaxidx)
          field_fill(var1, 0.0);
        else
          field_copy(field, var1);

        if (lrange || lminidx || lmaxidx) field_copy(field, var2);

        if (lvarstd) field2_moq(var2, var1);

        if (field.numMissVals || !samp.empty())
          {
            if (samp.empty()) samp.resize(var1.size);
            field2_vinit(samp, field);
          }
      }
    else
      {
        if (field.numMissVals || !samp.empty())
          {
            if (samp.empty()) samp.resize(var1.size, numSets);
            field2_vincr(samp, field);
          }

        // clang-format off
        if      (lvarstd) field2_sumsumq(var1, var2, field);
        else if (lrange)  field2_maxmin(var1, var2, field);
        else if (lminidx) field2_minidx(var1, var2, field, numSets);
        else if (lmaxidx) field2_maxidx(var1, var2, field, numSets);
        else              field2_function(var1, field, operfunc);
        // clang-format on
      }
  }

  void
  process_kernel(const Field &samp, Field &var1, const Field &var2, int numSets)
  {
    auto field2_stdvar_func = lstd ? field2_std : field2_var;
    auto fieldc_stdvar_func = lstd ? fieldc_std : fieldc_var;

    if (lmeanavg)
      {
        if (!samp.empty())
          field2_div(var1, samp);
        else
          fieldc_div(var1, (double) numSets);
      }
    else if (lvarstd)
      {
        if (!samp.empty())
          field2_stdvar_func(var1, var2, samp, divisor);
        else
          fieldc_stdvar_func(var1, var2, numSets, divisor);
      }
    else if (lrange) { field2_sub(var1, var2); }
  }
};

class StepStat1Dvars : public StepStatBase
{
private:
  FieldVector sampData;
  FieldVector varsData1;
  FieldVector varsData2;

public:
  void
  alloc(const VarList &varList, int VARS_MEMTYPE)
  {
    auto var2needed = (lvarstd || lrange || lminidx || lmaxidx);
    field1Dvars_init(sampData, varList);
    field1Dvars_init(varsData1, varList, FIELD_VEC | VARS_MEMTYPE);
    field1Dvars_init(varsData2, varList, var2needed ? FIELD_VEC : 0);
  }

  Field &
  var1(int varID)
  {
    return varsData1[varID];
  }

  Field &
  var2(int varID)
  {
    return varsData2[varID];
  }

  Field &
  samp(int varID)
  {
    return sampData[varID];
  }

  void
  process(int varID, int numSets)
  {
    process_kernel(sampData[varID], varsData1[varID], varsData2[varID], numSets);
  }
};

class StepStat1Dlevels : public StepStatBase
{
private:
  FieldVector sampData;
  FieldVector varsData1;
  FieldVector varsData2;

public:
  void
  alloc(const VarList &varList, int VARS_MEMTYPE)
  {
    auto var2needed = (lvarstd || lrange || lminidx || lmaxidx);
    field1Dlevels_init(sampData, varList);
    field1Dlevels_init(varsData1, varList, FIELD_VEC | VARS_MEMTYPE);
    field1Dlevels_init(varsData2, varList, var2needed ? FIELD_VEC : 0);
  }

  Field &
  var1(int levelID)
  {
    return varsData1[levelID];
  }

  void
  add_field(const Field &field, int levelID, int numSets)
  {
    auto &samp = sampData[levelID];
    auto &var1 = varsData1[levelID];
    auto &var2 = varsData2[levelID];

    var1.nsamp++;
    if (lrange) var2.nsamp++;
    add_field_kernel(field, samp, var1, var2, numSets);
  }

  void
  moq(int levelID)
  {
    field2_moq(varsData2[levelID], varsData1[levelID]);
  }

  void
  process(int levelID, int numSets)
  {
    process_kernel(sampData[levelID], varsData1[levelID], varsData2[levelID], numSets);
  }
};

class StepStat2D : public StepStatBase
{
private:
  Varray<double> vsamp;
  FieldVector2D sampData;
  FieldVector2D varsData1;
  FieldVector2D varsData2;

  static void
  set_missval(Field &field, const Field &samp, int numSets, double vfraction)
  {
    auto fieldsize = field.size;
    auto missval = field.missval;

    size_t irun = 0;
    for (size_t i = 0; i < fieldsize; ++i)
      {
        if ((samp.vec_d[i] / numSets) < vfraction)
          {
            field.vec_d[i] = missval;
            irun++;
          }
      }

    if (irun) field.numMissVals = field_num_miss(field);
  }

public:
  void
  alloc(const VarList &varList, int VARS_MEMTYPE)
  {
    auto var2needed = (lvarstd || lrange || lminidx || lmaxidx);
    field2D_init(sampData, varList);
    field2D_init(varsData1, varList, FIELD_VEC | VARS_MEMTYPE);
    field2D_init(varsData2, varList, var2needed ? FIELD_VEC : 0);
  }

  Field &
  var1(int varID, int levelID)
  {
    return varsData1[varID][levelID];
  }

  Varray<double> &
  samp(int varID, int levelID, int numSets)
  {
    auto &samp = sampData[varID][levelID];
    auto &var1 = varsData1[varID][levelID];

    vsamp.resize(var1.size);
    if (!samp.empty())
      vsamp = samp.vec_d;
    else
      ranges::fill(vsamp, (double) numSets);

    return vsamp;
  }

  void
  add_field(const Field &field, int varID, int levelID, int numSets)
  {
    auto &samp = sampData[varID][levelID];
    auto &var1 = varsData1[varID][levelID];
    auto &var2 = varsData2[varID][levelID];

    add_field_kernel(field, samp, var1, var2, numSets);
  }

  void
  set_missval(int varID, int levelID, int numSets, double vfraction)
  {
    const auto &samp = sampData[varID][levelID];
    auto &var2 = varsData2[varID][levelID];
    if (!samp.empty()) set_missval(var2, samp, numSets, vfraction);
  }

  void
  process(int varID, int levelID, int numSets)
  {
    process_kernel(sampData[varID][levelID], varsData1[varID][levelID], varsData2[varID][levelID], numSets);
  }
};

class StepStat3D : public StepStatBase
{
private:
  FieldVector3D sampData;
  FieldVector3D varsData1;
  FieldVector3D varsData2;
  int m_dimlen0{ 0 };

public:
  void
  set_dimlen0(int dimlen0)
  {
    m_dimlen0 = dimlen0;
    sampData.resize(dimlen0);
    varsData1.resize(dimlen0);
    varsData2.resize(dimlen0);
  }

  void
  alloc(int dim0, const VarList &varList, int VARS_MEMTYPE)
  {
    auto var2needed = (lvarstd || lrange || lminidx || lmaxidx);
    field2D_init(sampData[dim0], varList);
    field2D_init(varsData1[dim0], varList, FIELD_VEC | VARS_MEMTYPE);
    field2D_init(varsData2[dim0], varList, var2needed ? FIELD_VEC : 0);
  }

  FieldVector2D &
  samp(int dim0)
  {
    return sampData[dim0];
  }

  Field &
  samp(int dim0, int varID, int levelID)
  {
    return sampData[dim0][varID][levelID];
  }

  FieldVector2D &
  var1(int dim0)
  {
    return varsData1[dim0];
  }

  Field &
  var1(int dim0, int varID, int levelID)
  {
    return varsData1[dim0][varID][levelID];
  }

  FieldVector2D &
  var2(int dim0)
  {
    return varsData2[dim0];
  }

  Field &
  var2(int dim0, int varID, int levelID)
  {
    return varsData2[dim0][varID][levelID];
  }

  void
  add_field(const Field &field, int dim0, int varID, int levelID, int numSets)
  {
    auto &samp = sampData[dim0][varID][levelID];
    auto &var1 = varsData1[dim0][varID][levelID];
    auto &var2 = varsData2[dim0][varID][levelID];

    add_field_kernel(field, samp, var1, var2, numSets);
  }

  void
  process(int dim0, int varID, int levelID, int numSets)
  {
    process_kernel(sampData[dim0][varID][levelID], varsData1[dim0][varID][levelID], varsData2[dim0][varID][levelID], numSets);
  }
};

const auto write_out_stream = [](CdoStreamID streamID2, const std::vector<RecordInfo> &recordList, const VarList &varList1,
                                 cdo::StepStat2D &stepStat, int otsID) noexcept {
  cdo_def_timestep(streamID2, otsID);

  for (const auto &record : recordList)
    {
      auto [varID, levelID] = record.get();
      if (otsID && varList1.vars[varID].isConstant) continue;

      cdo_def_record(streamID2, varID, levelID);
      cdo_write_record(streamID2, stepStat.var1(varID, levelID));
    }
};

const auto write_diag_stream = [](CdoStreamID streamID3, const std::vector<RecordInfo> &recordList, const VarList &varList1,
                                  cdo::StepStat2D &stepStat, int otsID, int numSets) noexcept {
  cdo_def_timestep(streamID3, otsID);

  for (const auto &record : recordList)
    {
      auto [varID, levelID] = record.get();
      if (otsID && varList1.vars[varID].isConstant) continue;

      auto &vsamp = stepStat.samp(varID, levelID, numSets);

      cdo_def_record(streamID3, varID, levelID);
      cdo_write_record(streamID3, vsamp.data(), 0);
    }
};

const auto records_process
    = [](const std::vector<RecordInfo> &recordList, const VarList &varList1, cdo::StepStat2D &stepStat, int numSets) noexcept {
        for (const auto &record : recordList)
          {
            auto [varID, levelID] = record.get();
            if (varList1.vars[varID].isConstant) continue;

            stepStat.process(varID, levelID, numSets);
          }
      };

const auto records_set_missval = [](const std::vector<RecordInfo> &recordList, const VarList &varList1, cdo::StepStat2D &stepStat,
                                    int numSets, double vfraction) noexcept {
  for (const auto &record : recordList)
    {
      auto [varID, levelID] = record.get();
      if (varList1.vars[varID].isConstant) continue;

      stepStat.set_missval(varID, levelID, numSets, vfraction);
    }
};

const auto records_process_3D = [](int dim0, const std::vector<RecordInfo> &recordList, const VarList &varList1,
                                   cdo::StepStat3D &stepStat, int numSets) noexcept {
  for (const auto &record : recordList)
    {
      auto [varID, levelID] = record.get();
      if (varList1.vars[varID].isConstant) continue;

      stepStat.process(dim0, varID, levelID, numSets);
    }
};

};  // namespace cdo

#endif
