// Copyright 2014 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "extensions/browser/api/cast_channel/cast_auth_ica.h"

#include "base/base64.h"
#include "base/lazy_instance.h"
#include "crypto/signature_verifier.h"
#include "extensions/browser/api/cast_channel/cast_auth_util.h"
#include "extensions/common/api/cast_channel/authority_keys.pb.h"
#include "net/base/hash_value.h"
#include "net/cert/x509_certificate.h"

namespace extensions {
namespace core_api {
namespace cast_channel {
namespace {

// Fingerprint of the default ICA.
static const net::SHA256HashValue kDefaultFingerprintICA = { {
    0x52, 0x9D, 0x9C, 0xD6, 0x7F, 0xE5, 0xEB, 0x69, 0x8E, 0x70, 0xDD, 0x26,
    0xD7, 0xD8, 0xF1, 0x26, 0x59, 0xF1, 0xE6, 0xE5, 0x23, 0x48, 0xBF, 0x6A,
    0x5C, 0xF7, 0x16, 0xE1, 0x3F, 0x41, 0x0E, 0x73
} };

// Built in public key for verifying trusted authorities data.
const uint8 kPublicKey[] = {
    0x30, 0x82, 0x01, 0x22, 0x30, 0x0D, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86,
    0xF7, 0x0D, 0x01, 0x01, 0x01, 0x05, 0x00, 0x03, 0x82, 0x01, 0x0F, 0x00,
    0x30, 0x82, 0x01, 0x0A, 0x02, 0x82, 0x01, 0x01, 0x00, 0xC3, 0x8D, 0xEC,
    0xF5, 0x94, 0x4C, 0x66, 0x90, 0xFC, 0x08, 0xED, 0x6E, 0x7D, 0xD0, 0xD6,
    0x93, 0x62, 0x94, 0x86, 0xFB, 0xBA, 0x6F, 0x06, 0xD5, 0x26, 0xAD, 0x52,
    0x84, 0x5A, 0x68, 0x52, 0xF1, 0xA9, 0x46, 0x04, 0x78, 0xC0, 0xA1, 0x0F,
    0x72, 0xCE, 0x1C, 0xB3, 0xD2, 0x34, 0xFA, 0x28, 0xD0, 0xA6, 0x2D, 0xDF,
    0xDF, 0x0C, 0x6C, 0x30, 0xCF, 0xF3, 0x67, 0x0E, 0xCE, 0xDD, 0x2D, 0x02,
    0x20, 0x05, 0x7D, 0x7F, 0x81, 0x19, 0xBC, 0x42, 0x74, 0xC0, 0x11, 0xC4,
    0x7F, 0x31, 0x0B, 0x63, 0xE7, 0x2E, 0xFE, 0xA4, 0x20, 0xE5, 0xD8, 0x19,
    0xE1, 0xBE, 0x25, 0x41, 0x88, 0xFA, 0xBB, 0x51, 0xFF, 0x84, 0x9E, 0x37,
    0x03, 0xAE, 0xAF, 0x41, 0x4E, 0x4B, 0x75, 0xF8, 0x73, 0x79, 0x20, 0x2B,
    0xA0, 0xE9, 0xFC, 0xD6, 0x03, 0x50, 0x1B, 0x2A, 0x6B, 0x31, 0xE8, 0x75,
    0x13, 0xB6, 0xEC, 0xE4, 0x5D, 0x2A, 0x3E, 0xE5, 0x49, 0x44, 0xBF, 0xE2,
    0xC2, 0x4A, 0x7B, 0xC8, 0xC2, 0xC9, 0x89, 0xEF, 0x1A, 0x67, 0x37, 0xB7,
    0x52, 0x93, 0x86, 0x69, 0xA4, 0x1A, 0x2C, 0x69, 0xD2, 0x03, 0x93, 0x15,
    0x02, 0x97, 0xB6, 0xF7, 0x55, 0x87, 0x5B, 0xAA, 0x73, 0x12, 0x3F, 0xDD,
    0x32, 0x16, 0x3B, 0x8E, 0xD9, 0xD9, 0x49, 0x06, 0x65, 0xAD, 0x28, 0x49,
    0xEA, 0x0F, 0x7E, 0x6A, 0x04, 0xF7, 0x89, 0x79, 0x8F, 0xCE, 0x3E, 0xD2,
    0x89, 0x6C, 0x43, 0x4B, 0x34, 0xBF, 0x42, 0x65, 0x2B, 0x09, 0x2F, 0x5C,
    0xCA, 0x46, 0x1C, 0x22, 0x36, 0x78, 0xEA, 0x98, 0x49, 0xA4, 0x0E, 0xAB,
    0x97, 0x0A, 0x76, 0xCD, 0x52, 0x99, 0x42, 0x44, 0xC2, 0x52, 0x40, 0x62,
    0x34, 0xC4, 0xCD, 0xE3, 0x3D, 0xCA, 0xE1, 0x33, 0x99, 0xE9, 0xF4, 0x6E,
    0xEB, 0xB0, 0x5D, 0xA5, 0x84, 0xA2, 0x5C, 0xC7, 0xBB, 0x14, 0x67, 0x0C,
    0x4F, 0x02, 0x03, 0x01, 0x00, 0x01};

// Verifies signature of the data using built-in public key.
bool VerifySignature(const std::string& signature, const std::string& data) {
  if (signature.size() != 256)
    return false;

  crypto::SignatureVerifier verifier;
  crypto::SignatureVerifier::HashAlgorithm hash_alg =
      crypto::SignatureVerifier::SHA256;
  crypto::SignatureVerifier::HashAlgorithm mask_hash_alg = hash_alg;
  unsigned int hash_len = 32;

  if (!verifier.VerifyInitRSAPSS(
          hash_alg,
          mask_hash_alg,
          hash_len,
          reinterpret_cast<const uint8*>(signature.data()),
          signature.size(),
          kPublicKey,
          sizeof(kPublicKey))) {
    return false;
  }

  verifier.VerifyUpdate(reinterpret_cast<const uint8*>(data.data()),
                        data.size());

  return verifier.VerifyFinal();
}

}  // namespace

AuthorityKeyStore::AuthorityKeyStore() {
}

AuthorityKeyStore::~AuthorityKeyStore() {
}

base::StringPiece AuthorityKeyStore::GetICAPublicKeyFromFingerprint(
    const net::SHA256HashValue& fingerprint) {
  AuthorityKeysMap::const_iterator it =
      certificate_authorities_.find(fingerprint);
  return (it == certificate_authorities_.end()) ? base::StringPiece()
                                                : (*it).second;
}

base::StringPiece AuthorityKeyStore::GetDefaultICAPublicKey() {
  return GetICAPublicKeyFromFingerprint(kDefaultFingerprintICA);
}

bool AuthorityKeyStore::Load(const std::string& keys) {
  certificate_authorities_.clear();
  authority_keys_.reset(new proto::AuthorityKeys());

  if (!authority_keys_->ParseFromString(keys))
    return false;

  for (int i = 0; i < authority_keys_->keys_size(); i++) {
    const std::string& fingerprint = authority_keys_->keys(i).fingerprint();
    net::SHA256HashValue hash;
    // Ensure it's the size of SHA-256 fingerprint data.
    if (fingerprint.size() != sizeof(hash.data)) {
      return false;
    }
    memcpy(hash.data, fingerprint.data(), sizeof(hash.data));
    const std::string& public_key = authority_keys_->keys(i).public_key();
    certificate_authorities_.insert(
        make_pair(hash, base::StringPiece(public_key)));
  }

  return true;
}

// Trusted certificate authorities.
static base::LazyInstance<AuthorityKeyStore>::Leaky g_authority_keys_store =
    LAZY_INSTANCE_INITIALIZER;

bool SetTrustedCertificateAuthorities(const std::string& keys,
                                      const std::string& signature) {
  std::string decoded_keys;
  if (!base::Base64Decode(keys, &decoded_keys))
    return false;

  std::string decoded_signature;
  if (!base::Base64Decode(signature, &decoded_signature))
    return false;

  if (!VerifySignature(decoded_signature, decoded_keys))
    return false;

  return g_authority_keys_store.Get().Load(decoded_keys);
}

base::StringPiece GetTrustedICAPublicKey(const base::StringPiece& data) {
  AuthorityKeyStore& authority_keys_store = g_authority_keys_store.Get();
  scoped_refptr<net::X509Certificate> ica_cert =
      net::X509Certificate::CreateFromBytes(data.data(), data.size());
  if (!ica_cert.get()) {
    LOG(ERROR) << "Invalid ICA certificate.";
    return "";
  }
  return authority_keys_store.GetICAPublicKeyFromFingerprint(
      net::X509Certificate::CalculateFingerprint256(
          ica_cert->os_cert_handle()));
}

base::StringPiece GetDefaultTrustedICAPublicKey() {
  AuthorityKeyStore& authority_keys_store = g_authority_keys_store.Get();
  return authority_keys_store.GetDefaultICAPublicKey();
}

}  // namespace cast_channel
}  // namespace core_api
}  // namespace extensions
