/*
//
//  Copyright 1997-2009 Torsten Rohlfing
//
//  Copyright 2004-2011 SRI International
//
//  This file is part of the Computational Morphometry Toolkit.
//
//  http://www.nitrc.org/projects/cmtk/
//
//  The Computational Morphometry Toolkit is free software: you can
//  redistribute it and/or modify it under the terms of the GNU General Public
//  License as published by the Free Software Foundation, either version 3 of
//  the License, or (at your option) any later version.
//
//  The Computational Morphometry Toolkit is distributed in the hope that it
//  will be useful, but WITHOUT ANY WARRANTY; without even the implied
//  warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License along
//  with the Computational Morphometry Toolkit.  If not, see
//  <http://www.gnu.org/licenses/>.
//
//  $Revision: 5436 $
//
//  $LastChangedDate: 2018-12-10 19:01:20 -0800 (Mon, 10 Dec 2018) $
//
//  $LastChangedBy: torstenrohlfing $
//
*/

#ifndef __cmtkProgress_h_included_
#define __cmtkProgress_h_included_

#include <cmtkconfig.h>

#include <string>
#include <queue>

namespace
cmtk
{

/** \addtogroup System */
//@{

/** Generic class for progress indication.
 */
class Progress 
{
public:
  /// Status code returned by SetPercentDone() method.
  typedef enum {
    /// Everything okay; continue as usual.
    OK,
    /// User requests interrupt of operation.
    INTERRUPT,
    /// Interrupt generated by timeout.
    TIMEOUT,
    /// Something went wrong.
    FAILED
  } ResultEnum;  

  /// This class.
  typedef Progress Self;

  /// Constructor.
  Progress()
  {
    Self::SetProgressInstance( this );
  }

  /// Virtual (dummy) destructor.
  virtual ~Progress() {};
  
  /// Set total number of steps to complete.
  static void Begin( const double start, const double end, const double increment, const std::string& taskName = std::string("") );

  /// Set number of tasks completed.
  static ResultEnum SetProgress( const double progress );

  /// Done with progress indicator.
  static void Done();

  /// Set number of tasks completed.
  virtual ResultEnum UpdateProgress() = 0;
  
  /// Set number of tasks completed.
  void SetProgressCurrent( const double progress );
  
  /// Set progress handler instance.
  static void SetProgressInstance( Self *const progressInstance ) 
  {
    Self::ProgressInstance = progressInstance;
  }
  
protected:
  /// Check if we're at the top level of the task hierarchy.
  bool IsTopLevel() const
  {
    return m_RangeStack.size() == 1;
  }

  /// Return the name of the current task (at the lowest level of nested ranges).
  const std::string GetCurrentTaskName() const;

  /// Compute current completion fraction from range stack.
  double GetFractionComplete() const;

  /// Set total number of steps to complete.
  virtual void BeginVirtual( const double start, const double end, const double increment, const std::string& taskName = std::string("") );

  /** Clean up progress output.
   * This member function can be overriden by derived classes.
   */
  virtual void DoneVirtual();

private:
  /// Instance of a derived class that handles GUI interaction etc.
  static Self* ProgressInstance;

  /// Class to current progress range, which can be nested.
  class Range
  {
  public:
    /// Constructor.
    Range( const double start, const double end, const double increment, const std::string& taskName = std::string("") )
      : m_Start( start ), m_End( end ), m_Increment( increment ), m_TaskName( taskName )
    {
      this->m_Current = this->m_Start;
    }

    /// Get complete fraction for this range.
    double GetFractionComplete( const double nestedFraction = 0 /*!< For nexted ranges, this is the fraction achieved in he next level down.*/ ) const;

    /// Start of this range.
    double m_Start;

    /// End of this range.
    double m_End;

    /// Increment between steps.
    double m_Increment;

    /// Current position in the range.
    double m_Current;

    /// Name of this task.
    const std::string m_TaskName;
  };

  /// Type for stack of nested progress ranges.
  typedef std::deque<Self::Range> RangeStackType;

  /// Stack of nested progress ranges.
  RangeStackType m_RangeStack;
};

//@}

} // namespace cmtk

#endif // #ifndef __cmtkProgress_h_included_
