/*
 * This file is part of the KMOS Pipeline
 * Copyright (C) 2002,2003 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
   								Includes
 -----------------------------------------------------------------------------*/

#include <cpl.h>
#include <string.h>

#include "kmos_pfits.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup kmos_pfits     FITS header protected access
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*-----------------------------------------------------------------------------
  							Function codes
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
  @brief    find out the EXPTIME
  @param    plist       property list to read from
  @return   the requested value
 */
/*----------------------------------------------------------------------------*/
double kmos_pfits_get_exptime(const cpl_propertylist * plist)
{
    return cpl_propertylist_get_double(plist, "EXPTIME") ;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    find out the position angle
  @param    plist       property list to read from
  @return   the requested value
 */
/*----------------------------------------------------------------------------*/
double kmos_pfits_get_rotangle(const cpl_propertylist * plist)
{
    return cpl_propertylist_get_double(plist, "ESO OCS ROT NAANGLE") ;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    find out the arcfile   
  @param    plist       property list to read from
  @return   pointer to statically allocated character string
 */
/*----------------------------------------------------------------------------*/
const char * kmos_pfits_get_arcfile(const cpl_propertylist * plist)
{
    return (const char *) cpl_propertylist_get_string(plist, "ARCFILE");
}

/*----------------------------------------------------------------------------*/
/**
  @brief    find out the TPL ID
  @param    plist       property list to read from
  @return   pointer to statically allocated character string
 */
/*----------------------------------------------------------------------------*/
const char * kmos_pfits_get_tplid(const cpl_propertylist * plist)
{
    return (const char *) cpl_propertylist_get_string(plist, 
            "ESO TPL ID");
}

/*----------------------------------------------------------------------------*/
/**
  @brief    find out the READMODE
  @param    plist       property list to read from
  @return   pointer to statically allocated character string
 */
/*----------------------------------------------------------------------------*/
const char * kmos_pfits_get_readmode(const cpl_propertylist * plist)
{
    return (const char *) cpl_propertylist_get_string(plist, 
            "ESO DET READ CURNAME");
}

/*----------------------------------------------------------------------------*/
/**
  @brief    find out the NAXIS1 value 
  @param    plist       property list to read from
  @return   the requested value
 */
/*----------------------------------------------------------------------------*/
int kmos_pfits_get_naxis1(const cpl_propertylist * plist)
{
    return cpl_propertylist_get_int(plist, "NAXIS1")  ;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    find out the NAXIS2 value 
  @param    plist       property list to read from
  @return   the requested value
 */
/*----------------------------------------------------------------------------*/
int kmos_pfits_get_naxis2(const cpl_propertylist * plist)
{
    return cpl_propertylist_get_int(plist, "NAXIS2")  ;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    find out the NAXIS3 value 
  @param    plist       property list to read from
  @return   the requested value
 */
/*----------------------------------------------------------------------------*/
int kmos_pfits_get_naxis3(const cpl_propertylist * plist)
{
    return cpl_propertylist_get_int(plist, "NAXIS3")  ;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    find out the NDIT value 
  @param    plist       property list to read from
  @return   the requested value
 */
/*----------------------------------------------------------------------------*/
int kmos_pfits_get_ndit(const cpl_propertylist * plist)
{
    return cpl_propertylist_get_int(plist, "ESO DET NDIT")  ;
}

/**@}*/
